
#  SomeUtilities.tcl ---
#  
#      This file is part of the whiteboard application. We collect some handy 
#      small utility procedures here.
#      
#  Copyright (c) 1999-2000  Mats Bengtsson
#  
#  See the README file for license, bugs etc.

# InvertArray ---
#
#    Inverts an array so that ...
#    No spaces allowed; no error checking made that the inverse is unique.

proc InvertArray { arrName invArrName }  {
    
    # Pretty tricky to make it work. Perhaps the new array should be unset?
    upvar $arrName locArr
    upvar $invArrName locInvArr
    foreach name [array names locArr] {
	set locInvArr($locArr($name)) $name
    }
}

# max, min ---
#
#    Finds max and min of two numerical values.

proc max  { a b }  {
    return [expr ($a >= $b) ? $a : $b]
}

proc min  { a b }  {
    return [expr ($a <= $b) ? $a : $b]
}

# lsort -unique
#
#    Removes duplicate list items (from the Wiki page)

proc luniq  { theList }  {
    
    set t {}
    foreach i $theList {
	if {[lsearch -exact $t $i] == -1} {
	    lappend t $i
	}
    }
    return $t
}

# getdirname ---
#
#    Returns the path from 'filePath' thus stripping of any file name.
#    This is a workaround for the strange [file dirname ...] which strips
#    off "the last thing."
#    We need actual files here, not fake ones.

proc getdirname { filePath } {
    
    if {[file isfile $filePath]} {
	return [file dirname $filePath]
    } else {
	return $filePath
    }
}

# GetRelativePath ---
#
#    Returns the relative path from fromPath to toPath. Both fromPath and toPath
#    must be complete paths.
#    
# Arguments:
#    fromPath       an absolute path which is the "original" path.
#    toPath         an absolute path which is the "destination" path.
#                   It may contain a file name at end.
# Results:
#    The relative path from 'fromPath' to 'toPath'.

proc GetRelativePath { fromPath toPath } {
    
    # Need real paths, not fake, for getdirname.
    set fromPath [getdirname $fromPath]
    if {[file pathtype $fromPath] != "absolute"} {
	error "both paths must be absolute paths"
    } elseif {[file pathtype $toPath] != "absolute"} {
	error "both paths must be absolute paths"
    }
    
    # This is the method to reach platform independence.
    # We must be sure that there are no path separators left.
    
    set fromP {}
    foreach elem [file split $fromPath] {
	lappend fromP [string trim $elem "/:\\"]
    }
    set toP {}
    foreach elem [file split $toPath] {
	lappend toP [string trim $elem "/:\\"]
    }
    set lenFrom [llength $fromP]
    set lenTo [llength $toP]
    #puts "lenFrom=$lenFrom, lenTo=$lenTo"
    set lenMin [min $lenFrom $lenTo]
    #puts "fromP=$fromP"
    #puts "toP=$toP"
    
    # Find first nonidentical dir; iid = index of lowest common directory.
    
    set iid 0
    while {([string compare [lindex $fromP $iid] [lindex $toP $iid]] == 0) && \
      ($iid < $lenMin)} {
	incr iid
    }
    incr iid -1
    #puts "iid=$iid"
    set up "../"
    set numUp [expr $lenFrom - 1 - $iid]
    #puts "numUp=$numUp"
    if {$numUp <= 0} {
	set relPath "./"
    } else {
	set relPath {}
	for {set i 1} {$i <= $numUp} {incr i} {
	    append relPath $up
	}
    }
    #puts "relPath=$relPath"
    
    # Append the remaining unique path from 'toPath'.
    set relPath "$relPath[join [lrange $toP [expr $iid + 1]  \
      [expr $lenTo - 1]] "/"]"
    set nat [file nativename $relPath]
    #puts "relPath=$relPath"
    return $relPath
}

# AddAbsolutePathWithRelative ---
#
#    Adds the second, relative path, to the first, absolute path.
#    
# Arguments:
#    absPath        an absolute path which is the "original" path.
#    toPath         a relative path which should be added.
# Results:
#    The absolute path by adding 'absPath' with 'relPath'.

proc AddAbsolutePathWithRelative { absPath relPath }  {
    
    # Be sure to strip off any filename.
    set absPath [getdirname $absPath]
    if {[file pathtype $absPath] != "absolute"} {
	error "first path must be an absolute path"
    } elseif {[file pathtype $relPath] != "relative"} {
	error "second path must be a relative path"
    }

    # This is the method to reach platform independence.
    # We must be sure that there are no path separators left.
    
    set absP {}
    foreach elem [file split $absPath] {
	lappend absP [string trim $elem "/:\\"]
    }
    #puts "absP=$absP, relPath=$relPath"
    
    # If any up dir (../ ::  ), find how many. Start with unix style.
    set numUp [regsub -all {\.\./} $relPath {} newRelPath]
    #puts "numUp=$numUp, newRelPath=$newRelPath"
   
    # Delete the same number of elements from the end of the absolute path
    # as there are up dirs in the relative path.
    
    if {$numUp > 0} {
	set iend [expr [llength $absP] - 1]
	set upAbsP [lreplace $absP [expr $iend - $numUp + 1] $iend]
    } else {
	set upAbsP $absP
    }
    #puts "upAbsP=$upAbsP"
    set relP {}
    foreach elem [file split $newRelPath] {
	lappend relP [string trim $elem "/:\\"]
    }
    #puts "relP=$relP"
    set completePath "$upAbsP $relP"
    set unixAbsPath "/[join $completePath "/"]"
    return $unixAbsPath
}

# FullFilePathToRelative --
#
#   Trims the 'basePath' from 'fullPath', and makes it a relative path.
#   This could perhaps be generalized. Yes!!!!!!!
#
#   OLD OLD OLD OLD OLD OLD OLD OLD OLD OLD OLD OLD OLD OLD OLD OLD 

proc FullFilePathToRelative { fullPath basePath }  {
    global  tcl_platform
    
    #puts "FullFilePathToRelative:: fullPath=$fullPath, basePath=$basePath"
    # Is already relative path?
    if {[file pathtype $fullPath] == "relative"}  {
	return $fullPath
    }
    set any_ {.*}
    if {![regexp "${basePath}($any_)" $fullPath match relPath]}  {
	puts stderr "FullFilePathToRelative:: path problems writing image."
	return $fullPath
    }
    #puts "match=$match, relPath=$relPath"
    
    # Be sure to make it a relative path (prepend ':', trim '/' etc.).
    if {[string compare [file pathtype $relPath] "absolute"] == 0}  {
	if {$tcl_platform(platform) == "macintosh"}  {
	    set relPath ":$relPath"
	} elseif {$tcl_platform(platform) == "unix"}  {
	    set relPath [string trimleft $relPath /]
	} elseif {$tcl_platform(platform) == "windows"}  {
	    # Don't know. This?
	    set relPath [string trimleft $relPath /]
	}
    }
    return $relPath
}

#------------------------------------------------------------------------
