###############################################################################
# antiklop.tcl   : inspector&editor for variables/procedures/windows/menus/bindings
# Date           : 19.02.2001
# Revision       : 1.0
# Tcl/Tk version : 8.2.3 (partially tested with 8.0.5)
# Author         : Victor Safronov <safronov@academ.kaluga.su>
#
# Remarks:
# - see antiklop.html for manual
# - sorry for very long lines and lack of comments
# - 'string equal' was replaced with 'string match' to conform Tcl 8.0
# - toplevel separation algorithm is stupid and slow (but works)
# - tab = 3 spaces
#
# Problems (-todo *bug @error):
# * multiline variables and options are not properly commented out (in fact only the 
#   first line is commented out), whatch what you execute !
# * auto-uncomment checks for '#' only the first symbol of the current line,
#   where antiklop always puts it
# - antiklop can not communicate with other interps
# - list non-existing upvars/variables prefixed with '#'
# - stack: use 'namespace code' instead of lots of qalifiers
# - stack: restore item position in listbox
# - redisplay array definition when an element added or unset
# - add support for specific window types: text(tags), graph(elements..) etc.
# - sort array names with -dictionary
# - display long lists in multiple lines
# - show code for WM_DELETE_WINDOW, etc
# - before start, check if window .antiklop was created by another app.
# - delete traces
###############################################################################

package provide antiklop 1.0

###########################> ::antiklop <######################################

namespace eval ::antiklop {

   namespace export popup trap configure echo
   
  #avoid multiple sourcings (sourced is set at the end of file):
   variable sourced
   if {[info exists sourced]} {return}
   
   variable cfg
  #defaults:
   #set cfg(debug)    yes; #writes all commands executed in stack to file antiklop.log
   set cfg(form)     .antiklop 
   set cfg(spacepat) *
   set cfg(itempat)  *
   set cfg(mode)     namespaces
   set cfg(type)     variables
   set cfg(handerr)  0
   set cfg(fontsize) 10
   set cfg(count)    0; #number of executed commands in stack
   
  #this trace (needed for configure) does not use interface:
   trace variable cfg(handerr) w {::antiklop::SetBgerror}

  #class bindings: 
   bind Antiklop <Destroy> {::antiklop::DestroyInterface}
  
   #################################> popup <##################################
  
   proc popup {} {
      variable cfg
      set f $cfg(form)
      if {![winfo exists $f]} {
         CreateInterface
      }
      wm deiconify $f
      raise $f
   }

   #################################> popup <##################################
  
   proc trap {{message "trap"}} {
      variable cfg
      variable cmd
      set f $cfg(form)
      
      popup

     #save grab:
      set oldgrab [grab current $f]
      if {[string length $oldgrab]} {
         set oldstatus [grab status $oldgrab]
         grab release $oldgrab
      }
            
     #check if already holding:
      if {[info exists cfg(holding)]} {
         Puts "$message\n" warn
         $cfg(form).def.mid.t see end
         return
      }

     #trap: 
      set cfg(holding) 1; #value does not matter, element existence is checked
      set blink        0; #for button color
      $f.def.btns.release configure -state normal

      if {[info level] > 1} {
        wm title $f "[lindex [info level -1] 0] \[$message\]"
        set cfg(mode) stack
      } else {
        wm title $f "\[$message\]"
        ListSpaces
      }
   
     #event loop:
      while {[info exists cfg(holding)]} {
         update
         if $blink {set blink 0; $f.def.btns.release configure -fg cyan} else {set blink 1; $f.def.btns.release configure -fg yellow}
         set ncmds [lsort -integer [array names cmd]]
         foreach ncmd $ncmds {
            if {[info exists cfg(debug)]} {
               append ::antiklop::log "$ncmd\n$cmd($ncmd)\n\n";
            }
            eval $cmd($ncmd)
            unset cmd($ncmd)
            update idletasks
         }
      }
      
     #release:
      if {[winfo exists $f]} {
         $f.def.btns.release configure -fg [$f.def.btns.exec cget -fg] -state disabled
         if {[string match stack $cfg(mode)]} {set cfg(mode) namespaces}
         wm title $f "Antiklop"
      }

     #restore old grab:
      if {[string length $oldgrab]} {
         if {$oldstatus == "global"} {
            grab -global $oldgrab
         } elseif {$oldstatus == "local"} {
            grab $oldgrab
         }
         focus $oldgrab
      }
   }
   
   ##############################> configure <#################################
   
   proc configure {args} {
      variable cfg
      foreach {tag value} $args {
         switch -- $tag {
            -bgerror {set cfg(handerr) $value}
            default {tk_messageBox -type ok -icon error -message "Skipped unknown option given for antiklop::configure: '$tag' with value '$value'"}
         }
      }
   }
   
   ################################> echo <###################################
   # write to antiklop console
   
   proc echo {message} {
      variable cfg
      if {![winfo exists $cfg(form)]} {
         CreateInterface
      }
      popup
      Puts "$message\n" msg
      $cfg(form).def.mid.t see end
   }
   
   ############################################################################
   # The rest of procedures is supposed to be called internally only.
   #
   
   #################################> Puts <###################################
   # Write to console; number of args: 1 or even (%text tag% pairs)
   
   proc Puts {args} {
      variable cfg
      if {[llength $args]==1} {
         $cfg(form).def.mid.t insert end [lindex $args 0]
      } else {
         foreach {text tag} $args {
            if {[string compare $tag warn] == 0} {
               bell
            }
            $cfg(form).def.mid.t insert end $text $tag
         }
      }
   }

   ############################> CreateInterface <#############################
   
   proc CreateInterface {} {
      variable cfg
      set f $cfg(form)

      toplevel $f -class Antiklop
      wm title $f "Antiklop"
      wm protocol $f WM_DELETE_WINDOW "wm withdraw $f"
      
       frame $f.spaces -borderwidth 2 -relief ridge
       tk_optionMenu $f.spaces.om ::antiklop::cfg(mode) namespaces toplevels menus bindtags stack
       trace variable cfg(mode) w {::antiklop::ListSpaces}
       frame $f.spaces.f
        label $f.spaces.f.l -text "Pattern:"
        entry $f.spaces.f.e -textvariable ::antiklop::cfg(spacepat) -width 5
        bind $f.spaces.f.e <Return> {::antiklop::ListSpaces}
        pack $f.spaces.f.l -side left -expand 0 -padx 2
        pack $f.spaces.f.e -side right -fill x -expand 1 -padx 2
       scrollbar $f.spaces.s -orient vertical -command "$f.spaces.lb yview"
       listbox $f.spaces.lb -yscrollcommand "$f.spaces.s set" -selectmode browse -exportselection 0
       bind $f.spaces.lb <ButtonRelease-1>  {::antiklop::DisplaySpaceInfo}
       pack $f.spaces.om $f.spaces.f -side top -anchor n -fill x -expand 0
       pack $f.spaces.s -side right -fill y -expand 0
       pack $f.spaces.lb -side left -fill both -expand 1

      frame $f.sep1 -width 3 -relief flat -cursor sb_h_double_arrow
       bind $f.sep1 <B1-Motion> "$f.spaces configure -width \[expr %X - \[winfo rootx $f.spaces\]\]"

      frame $f.items -borderwidth 2 -relief ridge
       tk_optionMenu $f.items.om ::antiklop::cfg(type) procs commands variables arrays windows entries bindings
       trace variable cfg(type) w {::antiklop::ListItems}
       frame $f.items.f
        label $f.items.f.l -text "Pattern:"
        entry $f.items.f.e -textvariable ::antiklop::cfg(itempat) -width 5
        bind $f.items.f.e <Return> {::antiklop::ListItems}
        pack $f.items.f.l -side left -expand 0 -padx 2
        pack $f.items.f.e -side right -fill x -expand 1 -padx 2
       scrollbar $f.items.s -orient vertical  -command "$f.items.lb yview"
       listbox $f.items.lb -yscrollcommand "$f.items.s set" -selectmode browse -exportselection 0
       bind $f.items.lb <ButtonRelease-1> {::antiklop::DisplayItemInfo}
       bind $f.items.lb <ButtonRelease-3> {::antiklop::ItemSpecialAction}
       pack $f.items.om $f.items.f -side top -anchor n -fill x -expand 0
       pack $f.items.s -side right -fill y -expand 0
       pack $f.items.lb -fill both -expand 1

      frame $f.sep2 -width 3 -relief flat -cursor sb_h_double_arrow
       bind $f.sep2 <B1-Motion> "$f.items configure -width \[expr %X - \[winfo rootx $f.items\]\]"

      frame $f.def -borderwidth 2 -relief ridge
       frame $f.def.mid
        text      $f.def.mid.t  -yscrollcommand "$f.def.mid.sv set" -xscrollcommand "$f.def.mid.sh set" -wrap none -height 10 -width 50
        bind $f.def.mid.t <Shift-Return>   {::antiklop::Execute line; break}
        bind $f.def.mid.t <Control-Delete> {::antiklop::EraseText; break}
        scrollbar $f.def.mid.sv -orient vertical   -command "$f.def.mid.t yview"
        scrollbar $f.def.mid.sh -orient horizontal -command "$f.def.mid.t xview"
        grid $f.def.mid.t $f.def.mid.sv -sticky news
        grid $f.def.mid.sh -sticky news
        grid columnconfigure $f.def.mid 0 -weight 1 -minsize 1
        grid rowconfigure    $f.def.mid 0 -weight 1 -minsize 1
       frame $f.def.btns;
        button $f.def.btns.release -text "Release" -command {catch {unset ::antiklop::cfg(holding)}} -state disabled
        checkbutton $f.def.btns.handerr -text "bgerror" -variable ::antiklop::cfg(handerr) -borderwidth 2 -relief ridge; #no command, because traced
        tk_optionMenu $f.def.btns.fs ::antiklop::cfg(fontsize) 6 7 8 9 10 11 12 13 14 15 16
        label $f.def.btns.fsl -text "Font size:"
        trace variable cfg(fontsize) w {::antiklop::SetFontSize}
        set cfg(fontsize) $cfg(fontsize); #invoke trace
        button $f.def.btns.exec    -text "Execute" -command {::antiklop::Execute all}
        pack $f.def.btns.release $f.def.btns.handerr -side left -padx 2 -fill y
        pack $f.def.btns.exec $f.def.btns.fs $f.def.btns.fsl -side right
       pack $f.def.btns -side top -fill x -pady 1
       pack $f.def.mid -fill both -expand 1
       
      pack $f.spaces $f.sep1 $f.items $f.sep2 -side left -fill y -expand 0
      pack $f.def -fill both -expand 1
      update
      $f.spaces configure -width [winfo width $f.spaces]; #reqwidth fails here w/o updating
      $f.items  configure -width [winfo width $f.items]
      pack propagate $f.spaces 0
      pack propagate $f.items  0

     #tags for syntax highlight:
      $f.def.mid.t tag configure msg     -foreground purple
      $f.def.mid.t tag configure warn    -foreground red
      $f.def.mid.t tag configure nspace  -foreground darkgreen
      $f.def.mid.t tag configure trace   -foreground magenta
      $f.def.mid.t tag configure name    -foreground blue
      $f.def.mid.t tag configure body    -foreground black
      $f.def.mid.t tag configure rem     -foreground gray30
      $f.def.mid.t tag configure cmd     -foreground gray60
      $f.def.mid.t tag configure index   -foreground darkred
      $f.def.mid.t tag configure arg     -foreground darkred
      $f.def.mid.t tag configure defarg  -foreground red
      $f.def.mid.t tag configure hang    -lmargin2 10
     #for scroll with 2-btn mice:
      bind $f.def.mid.t <Button-3>  [bind Text <Button-2>]
      bind $f.def.mid.t <B3-Motion> [bind Text <B2-Motion>]
     #about:
      if {[info exists cfg(debug)]} {
         Puts "# Debug mode.\n" trace
      }
      Puts "# Useful bindings:\n# <MouseButton-1> on item in a listbox to query it,\n# <MouseButton-3> on item in a listbox to add some special code to console (for procs, commands, vars and arrays),\n# <Return> on pattern entries to rescan,\n# <Shift-Return> on this text to execute the line under cursor,\n# <Control-Delete> on this text to clear it.\n"
      ListSpaces
   }

   ############################> DestroyInterface <############################

   proc DestroyInterface {} {
      variable cfg
      variable cmd
      set f $cfg(form)
     #save debug info:
      if {[info exists cfg(debug)]} {
         set log [open antiklop.log w]
         puts $log $::antiklop::log
         foreach ncmd [lsort -integer [array names cmd]] {
            puts $log "\n*** $ncmd (not executed) ***\n$cmd($ncmd)"
         }
         close $log
      }
   }

   #############################> SetFontSize <################################
   # dummy args are for trace calls

   proc SetFontSize {args} {
      variable cfg
      set f $cfg(form)
      catch {
         set font [$f.def.mid.t cget -font]
         if {[llength $font] >=2 } {
            $f.def.mid.t configure -font [lreplace $font 1 1 $cfg(fontsize)]
         }
      }
   }

   #########################> DisplaySpaceInfo <#########################
   
   proc DisplaySpaceInfo {} {
      variable cfg
      set f $cfg(form)
      $f.items.lb selection clear 0 end
      if ![$f.spaces.lb size] {return}
      switch -- $cfg(mode) {
         namespaces - bindtags {
            ListItems
         }
         toplevels - menus {
            ListItems
            EraseText
            InsertWindowInfo [Space]
         }
         stack {
            EraseText
            if {[$f.spaces.lb curselection] > 0} {
               if {[string index [Space] 0] == "<"} {
                  AddToQueue "
                     uplevel #[$f.spaces.lb curselection] \{
                        ::antiklop::Puts \"This is the \" rem \"namespace eval \" name \"\[namespace current\]\" nspace \" stack frame.\\nNamespace variables are listed together with globals/upvars.\\n\" rem
                        ::antiklop::ListItems
                     \}
                  "
               } else {
                  AddToQueue "
                     uplevel #[$f.spaces.lb curselection] \{
                        ::antiklop::Puts \"This is a procedure call.\\n\" rem \"Call line:\\n\" index \"\[info level [$f.spaces.lb curselection]\]\\n\" body \"Procedure definition:\\n\" index
                        ::antiklop::InsertProcInfo \[namespace which -command [Space]\] 1
                        ::antiklop::ListItems
                     \}
                  "
               }
            } else {
               ::antiklop::Puts "#This is global level.\n" rem
               ::antiklop::ListItems
            }
         }
         default {
            error "Wrong mode: $cfg(mode)"
         }
      }
   }

   ##################################> Space <################################

   proc Space {} {
      variable cfg
      set f $cfg(form)
      if {![llength [$f.spaces.lb curselection]] && [$f.spaces.lb size]} {
         if {$cfg(mode) == "stack"} {
            $f.spaces.lb selection set end
         } else {
            $f.spaces.lb selection set 0
         }
      }
      if [llength [$f.spaces.lb curselection]] {
         return [$f.spaces.lb get [$f.spaces.lb curselection]]
      } else {
         return ""
      }
   }

   #####################################> Item <##################################

   proc Item {} {
      variable cfg
      set f $cfg(form)
      if {![llength [$f.items.lb curselection]] && [$f.items.lb size]} {
         $f.items.lb selection set 0
      }
      if [llength [$f.items.lb curselection]] {
         return [$f.items.lb get [$f.items.lb curselection]]
      } else {
         return ""
      }
   }

   ################################> Execute <#################################
   
   proc Execute {what} {
      variable cfg
      set f $cfg(form)
      if {[$f.def.mid.t get "insert linestart"] == "#"} {
         if {[tk_messageBox -type yesno -icon question -message "The line under coursor is commented out !\nUncomment ?"] == "yes"} {
            $f.def.mid.t delete "insert linestart"
         }
      }
      if {$what == "all"} {
         set code [$f.def.mid.t get 1.0 end]
         set replcmd "
            ::antiklop::Puts \"\\n#\$::antiklop::repl\\n\" rem
            $f.def.mid.t see end
            unset ::antiklop::repl
         "
      } else {
         set code [$f.def.mid.t get "insert linestart" "insert lineend"]
         set replcmd "
            $f.def.mid.t insert {insert lineend} \"\\n#\$::antiklop::repl\" rem
            unset ::antiklop::repl
         "
      }
      set ::antiklop::repl {}
      switch -- $cfg(mode) {
         namespaces {
            catch {namespace eval [Space] $code} ::antiklop::repl
            eval $replcmd
            ListItems
         }
         stack {
            AddToQueue "
              #Execute
               uplevel #[$f.spaces.lb curselection] \{
                  catch \{$code\} ::antiklop::repl
                  $replcmd
                  ::antiklop::ListItems
               \}
            "
         }
         default {
            catch {namespace eval :: $code} ::antiklop::repl
            eval $replcmd
            ListItems
         }
      }
   }

   ###############################> Family <###################################

   proc Family {ns pat cmd} {
      set result ""
      if [string match $pat $ns] {lappend result $ns}
      foreach child [lsort [eval $cmd $ns]] {
         eval lappend result [Family $child $pat $cmd]
      }
      return $result
   }

   ############################> AddToQueue <##################################

   proc AddToQueue {script} {
      variable cfg
      variable cmd
      set cmd([incr cfg(count)]) $script
   }

   ###############################> EraseText <################################

   proc EraseText {} {
      variable cfg
      set f $cfg(form)
      $f.def.mid.t delete 1.0 end
      focus $f.def.mid.t
   }
   
   #########################> LookForItemInListBox <###########################
   
   proc LookForItemInListBox {lb item {defaultsel 0}} {
      set pos [lsearch [$lb get 0 end] $item]
      if {$pos < 0} {
         set pos $defaultsel
      }
      $lb selection clear 0 end
      $lb selection set $pos
      $lb see $pos
   }
   
   ###############################> ListSpaces <###############################
   # dummy args are for trace calls
   
   proc ListSpaces {args} {
      variable cfg
      set f $cfg(form)
     #save name of selected item:
      if [llength [$f.spaces.lb curselection]] {
         set previtem [$f.spaces.lb get [$f.spaces.lb curselection]]
      } else {
         set previtem {#}
      }
      $f.items.lb delete 0 end
      $f.spaces.lb delete 0 end
      switch -- $cfg(mode) {
         namespaces {
            eval $f.spaces.lb insert end [Family :: $cfg(spacepat) "namespace children"]
            LookForItemInListBox $f.spaces.lb $previtem 0
            if {[lsearch {procs commands variables arrays} $cfg(type)] == -1} {
               set cfg(type) procs
            } else {
               ListItems
            }
         }
         toplevels {
            set wnds [Family . $cfg(spacepat) "winfo children"]
            set tops ""
            foreach wnd $wnds {
               set top [winfo toplevel $wnd]
               if {[lsearch $tops $top] == -1 && [string compare Menu [winfo class $top]]} {lappend tops $top}
            }
            eval $f.spaces.lb insert end $tops
            LookForItemInListBox $f.spaces.lb $previtem
            set cfg(type) windows
         }
         menus {
            foreach wnd [lsort [Family . $cfg(spacepat) "winfo children"]] {
               if [string match Menu [winfo class $wnd]] {$f.spaces.lb insert end $wnd}
            }
            LookForItemInListBox $f.spaces.lb $previtem
            set cfg(type) entries
         }
         stack {
            if {[info exists cfg(holding)]} {
               set cfg(spacepat) *
               $f.spaces.lb insert end <global>
               AddToQueue "
                  set thislev \[info level\]
                  for \{set lev 1\} \{\$lev < \$thislev\} \{incr lev\} \{
                     set procnm \[lindex \[info level \$lev\] 0\]
                     if \{\[string length \$procnm\]\} \{
                        $f.spaces.lb insert end \$procnm
                     \} else \{
                        $f.spaces.lb insert end <\[uplevel #\$lev \{namespace current\}\]>
                     \}
                  \}
                  ::antiklop::LookForItemInListBox $f.spaces.lb $previtem end
                  if \{\[lsearch {variables arrays} \$cfg\(type\)\] == -1\} \{after 200 \"set ::antiklop::cfg\(type\) variables\"\} else \{after 200 \"::antiklop::ListItems\"\}
               "
            } else {
               #bell
               tk_messageBox -type ok -icon info -message "Antiklop is not in stack now !"
               after 200 [namespace code {set cfg(mode) namespaces}] 
               return
            }
         }
         bindtags {
            set wnds [Family . * "winfo children"]
            set tags ""
            foreach wnd $wnds {
               foreach tag [bindtags $wnd] {
                  if {[string match $cfg(spacepat) $tag] && [lsearch $tags $tag] == -1 && [llength [bind $tag]]} {
                     lappend tags $tag
                  }
               }
            }
            eval $f.spaces.lb insert end [lsort $tags]
            LookForItemInListBox $f.spaces.lb $previtem
            set cfg(type) bindings
         }
         interpreters {
            eval $f.spaces.lb insert end [winfo interps]
         }
         default {error "Wrong mode $cfg(mode)"}
      }
   }

   ############################> ListItems <############################
   # dummy args are for trace calls

   proc ListItems {args} {
      variable cfg
      set f $cfg(form)
     #save name of selected item:
      if [llength [$f.items.lb curselection]] {
         set previtem [$f.items.lb get [$f.items.lb curselection]]
      } 
      $f.items.lb  delete 0 end
      if ![$f.spaces.lb size] {return}

      switch -- $cfg(type) {
         procs {
            if ![string match namespaces $cfg(mode)] {after 200 [namespace code {set cfg(mode) namespaces}]; return}
            eval $f.items.lb insert end [lsort [namespace eval [Space] info procs $cfg(itempat)]]
         }
         commands {
            if ![string match namespaces $cfg(mode)] {after 200 [namespace code {set cfg(mode) namespaces}]; return}
            set items [lsort [info commands [Space]::$cfg(itempat)]]
            set p [namespace eval [Space] info procs $cfg(itempat)]
            foreach item $items {
               set tail [namespace tail $item]
               if {[lsearch -exact $p $tail]=="-1"} {$f.items.lb insert end $tail}
            }
         }
         variables {
            if [string match namespaces $cfg(mode)] {
               foreach item [lsort [info vars [Space]::$cfg(itempat)]] {
                  if {[info exists $item] && ![array exists $item]} {
                     $f.items.lb insert end [namespace tail $item]
                  }
               }
            } elseif [string match stack $cfg(mode)] {
               AddToQueue "
                  uplevel #[$f.spaces.lb curselection] \{
                     set ::antiklop::locs \[lsort \[info locals $cfg(itempat)\]\]
                     $f.items.lb insert end {#-------------locals:--------------}
                     foreach ::antiklop::item \$::antiklop::locs \{
                        if !\[array exists \$::antiklop::item\] \{
                           $f.items.lb insert end \$::antiklop::item
                        \}
                     \}
                     $f.items.lb insert end {#---------upvars/globals:---------}
                     foreach ::antiklop::item \[lsort \[info vars $cfg(itempat)\]\] \{
                        if \{\[info exists \$::antiklop::item\] && !\[array exists \$::antiklop::item\] && \[lsearch -exact \$::antiklop::locs \$::antiklop::item\] == -1\} \{
                           $f.items.lb insert end \$::antiklop::item
                        \}
                     \}
                  \}
                  if \[info exists ::antiklop::item\] {unset ::antiklop::item}
                  unset ::antiklop::locs
               "
            } else {after 200 [namespace code {set cfg(mode) namespaces}]; return}
         }
         arrays {
            if [string match namespaces $cfg(mode)] {
               foreach item [lsort [info vars [Space]::$cfg(itempat)]] {
                  if {[array exists $item]} {
                     $f.items.lb insert end [namespace tail $item]
                  }
               }
            } elseif [string match stack $cfg(mode)] {
               AddToQueue "
                  uplevel #[$f.spaces.lb curselection] \{
                     set ::antiklop::locs \[lsort \[info locals $cfg(itempat)\]\]
                     $f.items.lb insert end {#------------locals:------------}
                     foreach ::antiklop::item \$::antiklop::locs \{
                        if \[array exists \$::antiklop::item\] \{
                           $f.items.lb insert end \$::antiklop::item
                        \}
                     \}
                     $f.items.lb insert end {#---------upvars/globals:---------}
                     foreach ::antiklop::item \[lsort \[info vars $cfg(itempat)\]\] \{
                        if \{\[array exists \$::antiklop::item\] && \[lsearch -exact \$::antiklop::locs \$::antiklop::item\] == -1\} \{
                           $f.items.lb insert end \$::antiklop::item
                        \}
                     \}
                  \}
                  if \[info exists ::antiklop::item\] {unset ::antiklop::item}
                  unset ::antiklop::locs
               "
            } else {after 200 [namespace code {set cfg(mode) namespaces}]; return}
         }
         windows {
            if ![string match toplevels $cfg(mode)] {after 200 [namespace code {set cfg(mode) toplevels}]; return}
            set parent [Space]
            set wnds [Family $parent $cfg(itempat) "winfo children"]
            foreach wnd $wnds {
               if {![string match $parent $wnd] && [string compare Menu [winfo class $wnd]] && [string match $parent [winfo toplevel $wnd]]} {
                  eval $f.items.lb insert end $wnd
               }
            }
         }
         entries {
            if ![string match menus $cfg(mode)] {after 200 [namespace code {set cfg(mode) menus}]; return}
            set cfg(itempat) *
            set maxnum [[Space] index end]
            for {set num 0} {$num <= $maxnum} {incr num} {
               set typ [[Space] type $num]
               switch -- $typ {
                  separator - tearoff {
                     $f.items.lb insert end "------"
                  }
                  default {
                     $f.items.lb insert end [[Space] entrycget $num -label]
                  }
               }
            }
         }
         bindings {
            if ![string match bindtags $cfg(mode)] {after 200 [namespace code {set cfg(mode) bindtags}]; return}
            foreach seq [lsort [bind [Space]]] {
               if [string match $cfg(itempat) $seq] {$f.items.lb insert end $seq}
            }
         }
         vectors {
            if [catch {
                  foreach v [lsort [::blt::vector names]] {
                     if [string match $cfg(itempat) $v] {$f.items.lb insert end $v}
                  }}] {
               tk_messageBox -type ok -icon error -message "Failed calling '::blt::vector names' (BLT not loaded?)"
            }
         }
         default {error "Wrong type $cfg(type)"}
      }
     #restore position:
      if [info exists previtem] {
         LookForItemInListBox $f.items.lb $previtem
      }
   }
   
   #########################> DisplayItemInfo <##########################

   proc DisplayItemInfo {} {
      variable cfg
      variable InsertVarInfo
      variable InsertArrayInfo
      set f $cfg(form)
      EraseText
      if {![$f.items.lb size]} {return}
      
      switch $cfg(type) {
         procs {
            InsertProcInfo [Space]::[Item]
         }
         commands {
            Puts "# [Item] is a command\n" rem
         }
         variables {
            set ::antiklop::tmpvarname [Item]
            if [string match namespaces $cfg(mode)] {
               namespace eval [Space] $InsertVarInfo
            } elseif [string match stack $cfg(mode)] {
               if [string compare "#" [string index [Item] 0]] {
                  AddToQueue [list uplevel #[$f.spaces.lb curselection] $InsertVarInfo]
               }
            } else {
               error "Wrong mode: $cfg(mode)"
            }
         }
         arrays {
            set ::antiklop::tmparrname [Item]
            if [string match namespaces $cfg(mode)] {
               namespace eval [Space] $InsertArrayInfo
            } elseif [string match stack $cfg(mode)] {
               if [string compare "#" [string index [Item] 0]] {
                  AddToQueue [list uplevel #[$f.spaces.lb curselection] $InsertArrayInfo]
               }
            } else {
               error "Wrong mode: $cfg(mode)"
            }
         }
         windows {
            InsertWindowInfo [Item]
         }
         entries {
            InsertMenuEntryInfo [Space] [$f.items.lb curselection]
         }
         bindings {
            InsertBindInfo [Space] [Item]
         }
         vectors {
            InsertVectorInfo [Item]
         }
         default {error "Unnkown $cfg(type)"}
      }
   }

   ############################> InsertVectorInfo <############################
   
   proc InsertVectorInfo {v} {
      variable cfg
      set f $cfg(form)
      set len [$v length]
      Puts "# " rem  "$v " name "length " index "$len\n" body
      for {set num 0} {$num < $len} {incr num} {
         Puts "# set " cmd "$v" name "\(" cmd "$num" index "\)\t" cmd
         if [catch {Puts "[set ${v}($num)]\n" body}] {
            Puts "value not set\n" rem 
         }
      }
   }

   ##########################> InsertMenuEntryInfo <###########################
   
   proc InsertMenuEntryInfo {m index} {
      variable cfg
      set f $cfg(form)
      Puts "# " rem  "Type: " name "[$m type $index]\n" body
      set opts [$m entryconfigure $index]
      foreach opt $opts {
         if {[llength $opt] >=5} {
            if ![string match "-class" [lindex $opt 0]] {Puts "# $m entryconfigure " cmd "$index " body "[lindex $opt 0] " index "\{" cmd "[lindex $opt 4]" body "\}\n" cmd}
         }
      }
   }
   
   ##########################> InsertBindInfo <################################

   proc InsertBindInfo {tag seq} {
      variable cfg
      set f $cfg(form)
      Puts "bind " cmd "$tag " name "$seq " index "\{" cmd "[bind $tag $seq]" body "\}\n" cmd
   }

   #######################> InsertWindowInfo <###########################

   proc InsertWindowInfo {w} {
      variable cfg
      set f $cfg(form)
      Puts "# " rem  "Path:  " name "$w\n" body
      Puts "# " rem  "Class: " name "[winfo class $w]\n" body
      set manag [winfo manager $w]
      Puts "# " rem "Manager: " name " $manag\n" body
      Puts "# " rem "Geometry: " name "[winfo geometry $w]\n" body
      Puts "#------------ Bindings : ------------\n" rem
      set tags [bindtags $w]
      Puts "# " rem "bindtags " index "$w" name " \{" cmd "$tags" body "\}" cmd "; # \{ " rem
      foreach tag $tags {
         Puts "[llength [bind $tag]] " body
      }
      Puts "\}\n" rem
      Puts "#--------- Window options: ---------\n" rem
      set opts [$w configure]
      foreach opt $opts {
         if {[llength $opt] >=5} {
            if ![string match "-class" [lindex $opt 0]] {Puts "# " rem "$w " name "configure " cmd "[lindex $opt 0] " index "\{" cmd "[lindex $opt 4]" body "\}\n" cmd}
         }
      }
      switch -- [winfo class $w] {
         Graph {
            Puts "#--------- Specific [winfo class $w] options: ---------\n" rem
            Puts "# " rem "Axis: " index "[$w axis names]\n" body
            Puts "# " rem "Elements: " index "[$w element names]\n" body
            Puts "# " rem "Pens: " index "[$w pen names]\n" body
            Puts "# " rem "Markers: " index "[$w marker names]\n" body
         }
      }
      Puts "#------------ Manager \($manag\) details: ------------\n" rem
      switch -- $manag {
         wm {
            foreach opt {aspect client colormapwindows command focusmodel geometry grid group iconbitmap iconmask iconname iconposition iconwindow maxsize minsize overrideredirect positionfrom protocol resizable sizefrom state title transient} {
               Puts "# wm " rem "$opt " index "$w " name "[wm $opt $w]\n" body
            }
         }
         pack - grid {
            foreach {tag value} [$manag info $w] {
               Puts "# $manag configure " cmd "$w " name "$tag " index " \{" cmd "$value" body "\}\n" cmd
            }
            Puts "# $manag " cmd "propagate " index "$w " name "[$manag propagate $w]\n" body
         }
         place {
            foreach {tag value} [$manag info $w] {
               Puts "# $manag configure " cmd "$w " name "$tag " index " \{" cmd "$value" body "\}\n" cmd
            }
         }
         table {
            set p [$manag containers -slave $w]
            set opts [::blt::table configure $p $w]
            Puts "# Index: " rem "[lindex [::blt::table info $p $w] 0]\n" body
            foreach opt $opts {
               if {[llength $opt] >=5} {
                  Puts "# $manag configure $p " cmd "$w " name "[lindex $opt 0] " index " \{" cmd "[lindex $opt 4]" body "\}\n" cmd
               }
            }
         }
         htext {
            set p [winfo parent $w]
            set opts [$p configure $w]
            foreach opt $opts {
               if {[llength $opt] >=5} {
                  Puts "# $p configure " cmd "$w " name "[lindex $opt 0] " index "\{" cmd "[lindex $opt 4]" body "\}\n" cmd
               }
            }
         }
         default {
            Puts "# This manager is unknown by Antiklop, will try '$manag configure':\n" rem
            update idletasks
            if [catch {
                  foreach {tag value} [$manag info $w] {
                     Puts "# $manag configure " cmd "$w " name "$tag " index " \{" cmd "$value" body "\}\n" cmd
                  }}] {
               Puts "# Failed." rem
            }
         }
      }
     #try to flash window:
      if {[catch {set bg [$w cget -background]}]} {return}
      for {set i 1} {$i <= 3} {incr i} {
         $w configure -background white
         update idletasks
         after 50
         $w configure -background black
         update idletasks
         after 50
      }
      $w configure -background $bg
   }


   ##############################> ItemSpecialAction <###############################

   proc ItemSpecialAction {} {
      variable cfg
      set f $cfg(form)
      switch $cfg(type) {
         procs - commands {
            EraseText
            Puts "[Item] "
            focus $f.def.mid.t
         }
         variables - arrays {
            EraseText
            Puts "trace variable [Item] wu ::antiklop::Trace"
            focus $f.def.mid.t
         }
      }
   }

   #########################> InsertVarInfo <############################
   # this code will be executed in different scopes and stack levels

   set InsertVarInfo {
      set ::antiklop::tmptraces    [trace vinfo $::antiklop::tmpvarname]
      set ::antiklop::tmpnumtraces [llength $::antiklop::tmptraces]
      ::antiklop::Puts "set " cmd "$::antiklop::tmpvarname " name "\{" cmd "[set $::antiklop::tmpvarname]" body "\}" cmd
      if {$::antiklop::tmpnumtraces > 0} {
         ::antiklop::Puts "; # " rem "trace" trace
         if {$::antiklop::tmpnumtraces > 1} {
            ::antiklop::Puts "s \($::antiklop::tmpnumtraces\)" trace
         }
         ::antiklop::Puts ": $::antiklop::tmptraces" trace
      }
      ::antiklop::Puts "\n"
      unset ::antiklop::tmpvarname
      unset ::antiklop::tmptraces
      unset ::antiklop::tmpnumtraces
   }

   #########################> InsertArrayInfo <############################
   # this code will be executed in different scopes and stack levels   

   set InsertArrayInfo {
      set ::antiklop::tmptraces    [trace vinfo $::antiklop::tmparrname]
      set ::antiklop::tmpnumtraces [llength $::antiklop::tmptraces]
      set ::antiklop::tmpelems     [lsort [array names $::antiklop::tmparrname]]
      ::antiklop::Puts "# Array '$::antiklop::tmparrname' contains [llength $::antiklop::tmpelems] elements.\n" rem
      if {$::antiklop::tmpnumtraces > 0} {
         ::antiklop::Puts "# " rem "Whole array trace" trace
         if {$::antiklop::tmpnumtraces > 1} {
            ::antiklop::Puts "s \($::antiklop::tmpnumtraces\)" trace
         }
         ::antiklop::Puts ": $::antiklop::tmptraces \n" trace
      }
      foreach ::antiklop::tmpelem $::antiklop::tmpelems {
         ::antiklop::Puts "# set " cmd "$::antiklop::tmparrname" name "\(" cmd "$::antiklop::tmpelem" index "\)\t\{" cmd "[set ${::antiklop::tmparrname}($::antiklop::tmpelem)]" body "\}" cmd
         set ::antiklop::tmptraces    [trace vinfo $::antiklop::tmparrname\($::antiklop::tmpelem\)]
         set ::antiklop::tmpnumtraces [llength $::antiklop::tmptraces]
         if {$::antiklop::tmpnumtraces > 0} {
            ::antiklop::Puts "; # " rem "trace" trace
            if {$::antiklop::tmpnumtraces > 1} {
               ::antiklop::Puts "s \($::antiklop::tmpnumtraces\)" trace
            }
            ::antiklop::Puts ": $::antiklop::tmptraces" trace
         }
         ::antiklop::Puts "\n"
      }
      unset ::antiklop::tmparrname
      unset ::antiklop::tmptraces
      unset ::antiklop::tmpnumtraces
      unset ::antiklop::tmpelems
      catch {unset ::antiklop::tmpelem}
   }

   ############################> InsertProcInfo <##############################

   proc InsertProcInfo {proc {withnamespace 0}} {
      variable cfg
      set f $cfg(form)
      set needspace 0
      Puts "proc " cmd
      if {$withnamespace} {
         Puts [namespace qualifiers $proc]:: nspace
      }
      Puts "[namespace tail $proc] " name "\{" cmd
      foreach arg [info args $proc] {
         if {$needspace} {
            Puts " "
         } else {
            set needspace 1
         }
         if [info default $proc $arg value] {
            Puts "\{" cmd "[list $arg $value]" defarg "\}" cmd
         } else {
            Puts $arg arg
         }
      }
      Puts "\} \{" cmd "[info body $proc]" body "\}\n" cmd
   }
   
   #################################> BgError <################################
   # replacement for ::bgerror
   
   proc BgError {message} {
      global errorInfo errorCode
      ::antiklop::trap $message
   }
   
   ############################> SetBgerror <##################################
   # swaps bgerror; called by trace, interface is not neded to exist
   
   proc SetBgerror {args} {
      variable cfg
      if {$cfg(handerr) && [llength [info commands ::antiklop::BgError]]} {
         catch {rename ::bgerror ::antiklop::bgerror}
         rename ::antiklop::BgError ::bgerror
      } elseif {! $cfg(handerr) && ![llength [info commands ::antiklop::BgError]]} {
         rename ::bgerror ::antiklop::BgError
         catch {rename ::antiklop::bgerror ::bgerror}
      }
   }
   
   ############################################################################

   proc Trace {name index op} {
      if {$op == "w"} {
         if {[uplevel array exists $name]} {
            if {[string length $index]} {
               upvar ${name}($index) var
               Puts "# ${name}\(${index}\) was set to \{$var\}\n"
            }
         } else {
            upvar $name var
            Puts "# $name was set to \{$var\}\n"
         }
      }
      trap "$name $index $op"
   }

   ############################################################################

   set sourced ok

}; #namespace closed
###############################> the end <#####################################
