# copyright (C) 1997-98 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: diskstats.tcl,v 1.6 1998/12/06 21:37:24 jfontain Exp $}

package provide diskstats 1.0


namespace eval diskstats {

    set file [open /proc/version]                                                                      ;# first check kernel version
    set version [lindex [gets $file] 2]
    close $file

    if {[package vcompare $version 2]<0} {
        puts stderr {diskstats module needs at least a version 2 kernel}
        exit 1
    }
    set full [expr {[package vcompare $version 2.1]>=0}]                                 ;# 2.1 kernels and up support full features

    array set data {
        updates 0
        0,label disk 0,type integer 0,message {disk index}
        1,label {read IOs} 1,type integer 1,message {number of read operations}
        2,label {write IOs} 2,type integer 2,message {number of write operations}
        3,label {blocks read} 3,type integer 3,message {number of blocks read}
        4,label {blocks written} 4,type integer 4,message {number of blocks written}
        pollTimes {10 5 20 30 60 120 300}
        sort {0 increasing}
        helpText {
This is a view of disks statistics for the system (provided you have a 2.0 or later kernel with the proc filesystem compiled in).

Statistics are drawn from the /proc/stat file which supports up to 4 disks only, with SCSI having the highest priority. For example, if you have both a hda IDE and a sda SCSI disk, statistics will not be available for the IDE disk and only the SCSI sda disk will show at index 0.

If you have a 2.1 or above kernel, data is displayed in 2 tables: one with statistics and the other with static device, type, size and model data. Devices (sd*, hd*) are determined using the /proc/partitions file.

For 2.0 kernels, only the statistics table is displayed.

SCSI support is not very well tested, for lack of hardware: please report problems.
        }
    }

    if {$full} {
        array set data {
            5,label device 5,type ascii 5,message {device (sd*: SCSI, hd*: IDE)}
            6,label type 6,type ascii 6,message {type or media}
            7,label size 7,type integer 7,message {size in KiloBytes}
            8,label model 8,type ascii 8,message {model description}
            views {
                {visibleColumns {0 1 2 3 4} sort {0 increasing}}
                {visibleColumns {0 5 6 7 8} sort {0 increasing}}
            }
        }
    }

    # initialize disk index, device, type, size and model cells
    array set data {
        0,0 0 1,0 1 2,0 2 3,0 3 0,5 {} 1,5 {} 2,5 {} 3,5 {} 0,6 {} 1,6 {} 2,6 {} 3,6 {} 0,7 0 1,7 0 2,7 0 3,7 0
        0,8 {} 1,8 {} 2,8 {} 3,8 {}
    }

    set statistics [open /proc/stat]                                                   ;# keep the files open for better performance

    if {$full} {
        set partitions [open /proc/partitions]
        array set scsiFromCharacter {h 0 s 1}
        array set indexFromCharacter {a 0 b 1 c 2 d 3}
    }

    proc update {} {                                                                                       ;# gather disk statistics
        variable full
        variable statistics
        variable last
        variable data

        if {$full} {
            variable partitions
            variable scsiFromCharacter
            variable indexFromCharacter

            seek $partitions 0                                                               ;# always rewind before retrieving data
            gets $partitions; gets $partitions                                                                       ;# skip headers
            set nextSCSINumber 0
            while {[gets $partitions line]>=0} {
                if {![regexp {([hs])d([abcd])$} $line device type character]} continue                                 ;# not a disk
                set index $indexFromCharacter($character)
                set exists($index) {}                                                         ;# keep track of existing disk indices
                if {[string length $data($index,5)]>0} continue                         ;# known disk, no need to update static data
                # SCSI has highest priority. IDE entry is taken into account only if SCSI was not present.
                if {$scsiFromCharacter($type)||([string length $data($index,5)]==0)} {
                    set data($index,5) $device
                    scan $line {%*u %*u %u} blocks
                    set data($index,7) [expr {round(($blocks*[lindex [statfs /dev/$device] 0])/1024.0)}]
                    if {$scsiFromCharacter($type)} {                                                                         ;# SCSI
                        set file [open /proc/scsi/scsi]
                        gets $file                                                                               ;# skip header line
                        set diskNumber 0
                        while {[gets $file line]>=0} {                                             ;# each entry has 3 lines of data
                            regexp {Vendor:(.*)Model:(.*)Rev:(.*)$} [gets $file] dummy vendor model revision
                            regexp {Type:(.*)ANSI SCSI revision:} [gets $file] dummy media
                            set media [string trim $media]
                            if {[string compare $media Direct-Access]!=0} continue  ;# not a disk therefore not listed in partitions
                            if {$diskNumber==$nextSCSINumber} {            ;# assume that SCSI devices are listed in discovery order
                                set data($index,6) $media
                                set data($index,8) "[string trim $vendor], [string trim $model], [string trim $revision]"
                                incr nextSCSINumber
                                break                                                                          ;# done for this disk
                            }
                            incr diskNumber
                        }
                        close $file
                    } else {                                                                                                  ;# IDE
                        set file [open /proc/ide/$device/media]
                        gets $file data($index,6)
                        close $file
                        set file [open /proc/ide/$device/model]
                        gets $file data($index,8)
                        close $file
                    }
                }
            }
            for {set index 0} {$index<4} {incr index} {                                            ;# cleanup disappeared disks data
                if {[info exists exists($index)]} continue                                                      ;# no cleanup needed
                array set data "$index,5 {} $index,6 {} $index,7 0 $index,8 {}"                          ;# reset static table cells
            }
        }

        seek $statistics 0                                                                   ;# always rewind before retrieving data
        gets $statistics; gets $statistics                                                                     ;# skip first 2 lines
        scan [gets $statistics] {disk_rio %u %u %u %u} data(0,1) data(1,1) data(2,1) data(3,1)
        scan [gets $statistics] {disk_wio %u %u %u %u} data(0,2) data(1,2) data(2,2) data(3,2)
        scan [gets $statistics] {disk_rblk %u %u %u %u} data(0,3) data(1,3) data(2,3) data(3,3)
        scan [gets $statistics] {disk_wblk %u %u %u %u} data(0,4) data(1,4) data(2,4) data(3,4)
        if {[info exists last]} {
            for {set row 0} {$row<=3} {incr row} {
                for {set column 1} {$column<=4} {incr column} {
                    set value $data($row,$column)
                    incr data($row,$column) -$last($row,$column)
                    set last($row,$column) $value
                }
            }
        } else {                                                                                                     ;# first update
            for {set row 0} {$row<=3} {incr row} {
                for {set column 1} {$column<=4} {incr column} {
                    set last($row,$column) $data($row,$column)
                    set data($row,$column) 0
                }
            }
        }
        incr data(updates)
    }
}
