#!/bin/sh
# the next line restarts using the interpreter \
exec wish "$0" "$@"

wm command . [concat $argv0 $argv]                                   ;# for proper window manager (windowmaker for example) behavior
wm group . .

# copyright (C) 1997-98 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

source utility.tcl
startGatheringRCSIds

set rcsId {$Id: moodss.tcl,v 1.130 1998/12/12 20:49:01 jfontain Exp $}

set applicationVersion 5.5

source getopt.tcl

proc printUsage {exitCode} {
    puts stderr "Usage: $::argv0 \[OPTION\]... \[MODULE\] \[MODULE\]..."
    puts stderr {  -f, --file       configuration file name}
    puts stderr {  -h, --help       display this help and exit}
    puts stderr {  -p, --poll-time  poll time in seconds}
    puts stderr {  -r, --read-only  disable viewer creation, editing, ...}
    puts stderr {  -S, --static     disable internal window manager sizing and moving}
    puts stderr {  --version        output version information and exit}
    exit $exitCode
}

if {[catch\
    {\
        set argv [parseCommandLineArguments\
            {-f 1 --file 1 -h 0 --help 0 -p 1 --poll-time 1 -r 0 --read-only 0 -S 0 --static 0 --version 0} $argv arguments\
        ]\
    }\
    message\
]} {
    puts stderr $message
    printUsage 1
}
foreach {short long} {-f --file -h --help -p --poll-time -r --read-only -S --static} {
    catch {set arguments($short) $arguments($long)}                                          ;# long version if present has priority
}

if {[info exists arguments(--version)]} {
    puts "moodss (a Modular Object Oriented Dynamic SpreadSheet) version $applicationVersion"
    exit
}
if {(![info exists arguments(-f)]&&([llength $argv]==0))||[info exists arguments(-h)]} {
    printUsage 1
}

proc loadModules {modules} {
    foreach module $modules {
        if {[info exists loaded($module)]} {
            puts stderr "$::argv0: module \"$module\" was specified more than once"
            exit 1
        }
        package require $module                                                                             ;# module is loaded here
        set loaded($module) {}
        if {![info exists ${module}::data(indexColumns)]} {
            set ${module}::data(indexColumns) 0                                ;# use first column as single index column by default
        }
    }
}

proc setPollTimes {modules {commandLineTime {}}} {
    global pollTimes pollTime

    set default 0
    set minimum 0
    foreach module $modules {
        set times [set ${module}::data(pollTimes)]
        if {[llength $times]==0} {
            error "module $module poll times list is empty"
        }
        # for an asynchronous module, the sole time value would be negative and is used as graph interval, for example
        set time [lindex $times 0]
        if {$time<0} {                                              ;# asynchronous module, poll time is a viewer interval (negated)
            set intervals($time) {}
            continue
        }
        if {$time>$default} {                                                                  ;# default value is the first in list
            set default $time                                                        ;# keep the greater default time of all modules
        }
        set times [lsort -integer $times]                                                                         ;# sort poll times
        set time [lindex $times 0]
        if {$time>$minimum} {
            set minimum $time                                                        ;# keep the greater minimum time of all modules
            set minimumModule $module
        }
        foreach time $times {                                        ;# poll times list is the combination of all modules poll times
            set data($time) {}
        }
    }
    set pollTimes [lsort -integer [array names data]]         ;# sort and restrict poll times above maximum module minimum poll time
    set pollTimes [lrange $pollTimes [lsearch -exact $pollTimes $minimum] end]
    set pollTime $default
    if {[string length $commandLineTime]>0} {                                           ;# eventually validate command line override
        if {$commandLineTime<$minimum} {
            puts stderr "$::argv0: minimum poll time is $minimum seconds for module $minimumModule"
            exit 1
        }
        set pollTime $commandLineTime
    }
    if {$pollTime==0} { 
        # all modules are asynchronous, so use an average time as a viewer interval for viewers that need it, such as graphs.
        # the poll times list is empty at this point so the user cannot change the poll time.
        # note that the viewer interval can still be forced by the command line poll time option.
        set sum 0
        set number 0
        foreach interval [array names intervals] {
            incr sum $interval
            incr number
        }
        set pollTime [expr {round(double($sum)/-$number)}]
    }
}

proc commaSeparatedString {words} {
    for {set index 0} {$index<([llength $words]-1)} {incr index} {
        append string "[lindex $words $index], "
    }
    append string [lindex $words $index]
    return $string
}


if {[catch {package require stooop 3.7}]} {
    source stooop.tcl                                                                     ;# in case stooop package is not installed
}
namespace import stooop::*
if {[catch {package require switched 1.4}]} {                                           ;# in case switched package is not installed
    source switched.tcl
}

if {[catch {package require scwoop 2.6}]} {
    source scwoutil.tcl
    source scwoop.tcl                                                                     ;# in case scwoop package is not installed
    source bindings.tcl
    source widgetip.tcl
    source arrowbut.tcl
    source spinent.tcl
    source panner.tcl
}

source record.tcl

if {[info exists arguments(-f)]} {                                                              ;# configuration file name specified
    set saveFile $arguments(-f)
    set initializer [new record -file $saveFile]
    record::read $initializer
    set modules(initialized) [record::modules $initializer]
    set modules(all) [concat $argv $modules(initialized)] ;# load uninitialized modules 1st so that their tables are placed properly
} else {
    set saveFile {}
    set modules(all) $argv
}
set modules(all,string) [commaSeparatedString $modules(all)]
wm iconname . "moodss $modules(all)"                                                              ;# give the icon a meaningful name
set readOnly [info exists arguments(-r)]
set static [info exists arguments(-S)]

lappend auto_path /usr/lib/moodss .    ;# search in current directory sub-directories for development and rpm installation directory
loadModules $modules(all)

set modules(synchronous) {}                                                                                 ;# look at synchronicity
foreach module $modules(all) {
    if {[lindex [set ${module}::data(pollTimes)] 0]>0} {                                                 ;# if module is synchronous
        lappend modules(synchronous) $module
    }
}
set modules(synchronous,string) [commaSeparatedString $modules(synchronous)]

if {[info exists arguments(-p)]} {                                                     ;# command line argument has highest priority
    setPollTimes $modules(all) $arguments(-p)
} elseif {[info exists initializer]} {                                                                  ;# then stored configuration
    setPollTimes $modules(all) [record::pollTime $initializer]
} else {                                                                                                       ;# use modules values
    setPollTimes $modules(all)
}

if {[catch {package require Tktable 2}]} {                                               ;# in case tkTable package is not installed
    switch $tcl_platform(platform) {
        unix {
            load ./Tktable.so.2.4
        }
        windows {
            load ./tktable.dll
        }
    }
}

if {[catch {package require BLT}]} {                                                         ;# in case BLT package is not installed
    switch $tcl_platform(platform) {
        unix {
            load ./libBLT.so.2.4g
        }
        windows {
            load ./blt80.dll                                                  ;# so far, only unofficial 8.0 is supported on Windows
        }
    }
}
set officialBLT [expr {$blt_version!=8.0}]                                               ;# remember which BLT version is being used

if {[catch {package require tkpiechart 5.2.1}]} {                                     ;# in case tkpiechart package is not installed
    source pielabel.tcl
    source boxlabel.tcl
    source canlabel.tcl
    source labarray.tcl
    source perilabel.tcl
    source slice.tcl
    source selector.tcl
    source objselec.tcl
    source pie.tcl
}

source config.tcl
source font.tcl
source scroll.tcl
source lifolbl.tcl
source keyslink.tcl
source dialog.tcl
source viewer.tcl
source blt2d.tcl
source tablesel.tcl
source datatab.tcl
source datagraf.tcl
source databar.tcl
source datapie.tcl
source lastwish.tcl
source sumtable.tcl
source freetext.tcl
source htmllib.tcl                                                ;# Tcl HTML library from Sun, used for viewing HTML help documents
source htmldata.tcl
source html.tcl                                             ;# must be source after HTML library since some peocedures are redefined
source help.tcl
source drag.tcl
source drop.tcl
source canvhand.tcl
source canvaswm.tcl


proc updateTitle {} {                                                          ;# show modules and poll time or mode in window title
    global pollTimes pollTime modules

    if {[llength $pollTimes]==0} {
        wm title . "moodss: $modules(all,string) data (asynchronous)"
    } else {
        wm title . "moodss: $modules(all,string) data (every $pollTime seconds)"
    }
}

proc createMenuWidget {parentPath readOnly includePollTime} {
    global modules static

    set menu [menu $parentPath.menu -tearoff 0]
    $menu add cascade -label File -menu [menu $menu.file -tearoff 0] -underline 0
    if {!$readOnly} {
        $menu.file add command -label Save -command {save 0} -underline 1 -accelerator Alt+S
        bind $parentPath <Alt-s> {save 0}
        $menu.file add command -label {Save As...} -command {save 1} -underline 6 -accelerator Alt+A
        bind $parentPath <Alt-a> {save 1}
    }
    $menu.file add command -label Exit -command exit -underline 1 -accelerator Alt+X
    bind $parentPath <Alt-x> exit
    if {!$readOnly&&$includePollTime} {
        $menu add cascade -label Options -menu [menu $menu.options -tearoff 0] -underline 0
        $menu.options add command -label {Poll Time...} -command inquirePollTime -underline 0
    }
    if {!$readOnly} {
        $menu add cascade -label New -menu [menu $menu.new -tearoff 0] -underline 0
        $menu.new add command -label {Graph Chart...} -underline 0 -command "createCellsViewer dataGraph {} 1 $static \$pollTime"
if {$::officialBLT} {
        $menu.new add command -label {Overlap Bar Chart...} -underline 0\
            -command "createCellsViewer dataOverlapBarChart {} 1 $static"
}
        $menu.new add command -label {Side Bar Chart...} -underline 5 -command "createCellsViewer dataSideBarChart {} 1 $static"
        $menu.new add command -label {Stacked Bar Chart...} -underline 0\
            -command "createCellsViewer dataStackedBarChart {} 1 $static"
        $menu.new add command -label {2D Pie Chart...} -underline 0 -command "createCellsViewer data2DPieChart {} 1 $static"
        $menu.new add command -label {3D Pie Chart...} -underline 0 -command "createCellsViewer data3DPieChart {} 1 $static"
        $menu.new add command -label {Summary Table...} -underline 8 -command "createCellsViewer summaryTable {} 1 $static"
        $menu.new add command -label {Free Text...} -underline 0 -command "createCellsViewer freeText {} 1 $static"
    }
    $menu add cascade -label Help -menu [menu $menu.help -tearoff 0] -underline 0
    $menu.help add command -label Global... -underline 0 -accelerator F1 -command helpWindow
    bind $parentPath <F1> helpWindow
    $menu.help add command -label Modules... -underline 0 -command "modulesHelpDialogBox $modules(all)"
    $menu.help add command -label Copyright... -underline 0 -command {simpleTextDialogBox {moodss: Copyright} $help::copyright}
    $menu.help add command -label {Source Versions...} -underline 0 -command versionsDialogBox
    $menu.help add command -label About... -underline 0 -command aboutDialogBox

    $parentPath configure -menu $menu
}

proc createMessageWidget {parentPath} {
    global messenger

    set messenger [new lifoLabel $parentPath -headerfont $font::(mediumBold) -font $font::(mediumNormal)]
    return $widget::($messenger,path)                                                                     ;# return actual tk widget
}

proc createCellsViewer {class cells draggable static {pollTime {}}} {
    global canvas

    set viewer [new $class $canvas -draggable $draggable]
    if {[string length $pollTime]>0} {
        composite::configure $viewer -interval $pollTime
    }
    viewer::view $viewer $cells
    manageViewer $viewer 1 -static $static
    return $viewer
}

proc manageViewer {viewer destroyable args} {                         ;# arguments are window manager configuration switched options
    global windowManager

    set path $widget::($viewer,path)
    canvasWindowManager::manage $windowManager $path
    eval canvasWindowManager::configure $windowManager $path $args
    if {$destroyable} {
        composite::configure $viewer -deletecommand "canvasWindowManager::unmanage $windowManager $path"
    }
}

proc dragEcho {data format} {
    return $data
}

proc createDragAndDropZone {parentPath} {
    global canvas pollTime static

    set frame [frame $parentPath.drops]

    set label [label $frame.graph -image applicationIcon]
    pack $label -pady 1 -side left
    new dropSite -path $label -formats DATACELLS\
        -command "createCellsViewer dataGraph \$dragSite::data(DATACELLS) 1 $static \$pollTime"
    new widgetTip -path $label -text "graph chart\ndrag'n'drop site"
    set drag [new dragSite -path $label]
    dragSite::provide $drag VIEWER {dragEcho ::dataGraph}

if {$::officialBLT} {
    set label [label $frame.overlapBarChart -image [image create photo -data [dataOverlapBarChart::iconData]]]
    pack $label -pady 1 -side left
    new dropSite -path $label -formats DATACELLS\
        -command "createCellsViewer dataOverlapBarChart \$dragSite::data(DATACELLS) 1 $static"
    new widgetTip -path $label -text "overlap bar chart\ndrag'n'drop site"
    set drag [new dragSite -path $label]
    dragSite::provide $drag VIEWER {dragEcho ::dataOverlapBarChart}
}

    set label [label $frame.sideBarChart -image [image create photo -data [dataSideBarChart::iconData]]]
    pack $label -pady 1 -side left
    new dropSite -path $label -formats DATACELLS -command "createCellsViewer dataSideBarChart \$dragSite::data(DATACELLS) 1 $static"
    new widgetTip -path $label -text "side bar chart\ndrag'n'drop site"
    set drag [new dragSite -path $label]
    dragSite::provide $drag VIEWER {dragEcho ::dataSideBarChart}

    set label [label $frame.stackedBarChart -image [image create photo -data [dataStackedBarChart::iconData]]]
    pack $label -pady 1 -side left
    new dropSite -path $label -formats DATACELLS\
        -command "createCellsViewer dataStackedBarChart \$dragSite::data(DATACELLS) 1 $static"
    new widgetTip -path $label -text "stacked bar chart\ndrag'n'drop site"
    set drag [new dragSite -path $label]
    dragSite::provide $drag VIEWER {dragEcho ::dataStackedBarChart}

    set label [label $frame.2DPieChart -image [image create photo -data [data2DPieChart::iconData]]]
    pack $label -pady 1 -side left
    new dropSite -path $label -formats DATACELLS -command "createCellsViewer data2DPieChart \$dragSite::data(DATACELLS) 1 $static"
    new widgetTip -path $label -text "2D pie chart\ndrag'n'drop site"
    set drag [new dragSite -path $label]
    dragSite::provide $drag VIEWER {dragEcho ::data2DPieChart}

    set label [label $frame.3DPieChart -image [image create photo -data [data3DPieChart::iconData]]]
    pack $label -pady 1 -side left
    new dropSite -path $label -formats DATACELLS -command "createCellsViewer data3DPieChart \$dragSite::data(DATACELLS) 1 $static"
    new widgetTip -path $label -text "3D pie chart\ndrag'n'drop site"
    set drag [new dragSite -path $label]
    dragSite::provide $drag VIEWER {dragEcho ::data3DPieChart}

    set label [label $frame.summaryTable -image [image create photo -data [summaryTable::iconData]]]
    pack $label -pady 1 -side left
    new dropSite -path $label -formats DATACELLS -command "createCellsViewer summaryTable \$dragSite::data(DATACELLS) 1 $static"
    new widgetTip -path $label -text "summary table\ndrag'n'drop site"
    set drag [new dragSite -path $label]
    dragSite::provide $drag VIEWER {dragEcho ::summaryTable}

    set label [label $frame.freeText -image [image create photo -data [freeText::iconData]]]
    pack $label -pady 1 -side left
    new dropSite -path $label -formats DATACELLS -command "createCellsViewer freeText \$dragSite::data(DATACELLS) 1 $static"
    new widgetTip -path $label -text "free text\ndrag'n'drop site"
    set drag [new dragSite -path $label]
    dragSite::provide $drag VIEWER {dragEcho ::freeText}

    set trashData {
        R0lGODlhKAAoAKUAAP8A/8/Lz+/r71FRUZ6anpaSlv/7/wgICL66vkFBQYaChtfT17LA3LaytjAwMFlZWZaant/b3/fz96aipnFxcWlpaWFhYTAoKCAgICgo
        KOfj576yvklJSRAQEBAQCI6Slq6yrp6SnhgYGAAAAI6KjoaKhnl5ecfDx66ipiAoIOfr5ygwMJ6Slnlxce/j5zg4OL7Dx6aint/T38fLz4Z5hvf7/8/Tz66q
        tqairv/z939/f/fz79fLz0lRUXmChllRUSH5BAEAAAAALAAAAAAoACgAAAb+QIBwSBwGisikckkUDAhMYiEaNQwO06ggS00iDIXEQcFcLA5dpUDAaCgch8GD
        DIVEDJJJIp2URBQUFQoWCRcYGRkGihoBCnt8RQYbDxwZhx0eBwcYHyAMIRMDDpBEEwYPGBgiHSOtrh0PJCVhGaRDCw0JHq68vAkDJh0cfBIEHAoNGb3LvR0O
        DlBDJwsTSHcnBRaszNytAxMaa4oGJyRCAn4LBggMBAfd8BwQCxL1AiiPAOrjDRMU8PBSUDghIJyKFcOEBDAQIMKJCQqUAeQW54QEDREKjOACQIOGAhQ4iJgY
        0KIAFgcoFFlIgoKDXSS7UZAggIKIJAJCtFjxLib+NwcuNLwYkARGDAovevpchkFGhBFLEJhIupTbDA2jlBho8K/qsg400CzJ80Cp11Ydhi6JEIBEgm1nW2Hg
        kBDJFwQVHGCI6wqRWGs1ClRIMJLviAQmoCbBY4LDi1R8LUSwYSJJAAE3OhzogDBuBgISYNRCouFEhgHmAFCAWbUCugkvcKLCMYREA7MxNRjQMCFrkRMYog0x
        UCGFzwMcDORYMImYBhYJJHbTgUPGDow8eORKI2AHAkeqzB54QYG6ABkCFgSYESDACOFM0GGr8MCCBQo9FJgowAAHjjUCGODCAjxE0EAHFnSh2wkfkOADDSGw
        oIA/CTgwgH8MKKCAAC5S8MBAASLkw8QJBsgwAQEk0PBDYX4dgAMCrfg2AUqjUaGCAQsgQMIKRCHxT4K2EEEAOjZYIAJHRKQWJBEgCJCDkUtCgoJRHkQJCQkW
        bGTllrYEAQA7
    }
    set label [label $frame.trash -image [image create photo -data $trashData]]
    pack $label -pady 1 -side right
    new dropSite -path $label -formats OBJECTS -command "eval delete \$dragSite::data(OBJECTS)"
    new widgetTip -path $label -text "objects trashing\ndrag'n'drop site"
    set drag [new dragSite -path $label]
    dragSite::provide $drag KILL list                                           ;# drag data is unimportant, only the kill action is

    return $frame
}

proc inquirePollTime {} {
    global pollTimes pollTime

    set dialog\
        [new dialogBox . -buttons oc -default o -title {moodss: Poll Time} -die 0 -x [winfo pointerx .] -y [winfo pointery .]]
    set frame [frame $widget::($dialog,path).frame]
    set minimum [lindex $pollTimes 0]
    set message [message $frame.message\
        -width [winfo screenwidth .] -font $font::(mediumNormal) -justify center\
        -text "Please enter new poll time\n(greater than $minimum):"\
    ]
    pack $message
    set entry [new spinEntry $frame -width 4 -list $pollTimes -side right]
    spinEntry::set $entry $pollTime
    pack $widget::($entry,path) -anchor e -side left -expand 1 -padx 2       ;# evenly pack entry and label together near the center
    pack [label $frame.label -text seconds] -anchor w -side right -expand 1 -padx 2
    dialogBox::display $dialog $frame
    widget::configure $dialog -command "
        set time \[spinEntry::get $entry\]
        if {\$time<$minimum} {                                                                    ;# check against minimum poll time
            bell
            $message configure -text {Please enter new poll time\n(must be greater than $minimum):}
        } else {                                                                                                ;# new time is valid
            if {\$time!=\$pollTime} {                                             ;# but check that it actually differs from current
                set pollTime \$time
                viewer::updateInterval \$time
                updateTitle
                refresh                                      ;# update immediately in case poll time was set to a much greater value
            }
            delete $dialog                                                                                     ;# destroy dialog box
        }
    "
    bind $frame <Destroy> "delete $entry"                                            ;# delete objects not managed by the dialog box
}

proc save {ask} {                               ;# save current configuration in user defined file or currently defined storage file
    global saveFile messenger

    if {$ask||([string length $saveFile]==0)} {
        set file [tk_getSaveFile\
            -title {moodss: Save} -initialdir [pwd] -defaultextension .moo -filetypes {{{moodss data} .moo}} -initialfile $saveFile
        ]
        if {[string length $file]==0} return                                                                 ;# user canceled saving
        set saveFile $file
    }
    lifoLabel::flash $messenger "saving in $saveFile..."                ;# flash instead of pushing as that goes too fast to be seen
    set record [new record -file $saveFile]
    record::write $record
    delete $record
}

proc refresh {} {
    global modules updateEvent messenger pollTime

    catch {after cancel $updateEvent}                                                             ;# eventually cancel current event

    if {[llength $modules(synchronous)]==0} return                                                                  ;# nothing to do

    # we do not know when data will actually be updated
    lifoLabel::push $messenger "launching $modules(synchronous,string) data update..."
    update idletasks
    foreach module $modules(synchronous) {
        ${module}::update                                                                   ;# ask module to update its dynamic data
    }
    lifoLabel::pop $messenger
    set updateEvent [after [expr {1000*$pollTime}] refresh]                                               ;# convert to milliseconds
}

createMenuWidget . $readOnly [llength $pollTimes]
pack [createMessageWidget .] -side bottom -fill x
updateTitle

set scroll [new scroll canvas .]
set canvas $composite::($scroll,scrolled,path)
set width [winfo screenwidth .]
set height [winfo screenheight .]
$canvas configure -background white -width $width -height $height -scrollregion [list 0 0 $width $height] -highlightthickness 0\
    -takefocus 0

set windowManager [new canvasWindowManager $canvas]

if {[info exists initializer]} {
    foreach {width height} [record::sizes $initializer] {}                                   ;# used stored geometry for main window
} else {
    set width 400
    set height 300
}

image create photo applicationIcon -data [dataGraph::iconData]                                ;# use data graph icon for application
wm iconwindow . [toplevel .icon]
pack [label .icon.image -image applicationIcon]
wm geometry . ${width}x$height

if {!$readOnly} {
    pack [createDragAndDropZone .] -fill x
}
pack $widget::($scroll,path) -fill both -expand 1

set draggable [expr {!$readOnly}]

set x 0.0
set y 0.0
foreach module $modules(all) {
    if {[info exists ${module}::data(views)]} {                                                        ;# user overrode default view
        set viewMembers [set ${module}::data(views)]                                      ;# create and manage a table for each view
    } else {
        set viewMembers {{}}                                                           ;# there is a single table (the default view)
    }
    set index 0
    foreach members $viewMembers {
        if {[llength $members]>0} {                                                                                  ;# it is a view
            array set view $members
            set table [new dataTable $canvas -data ${module}::data -view ::view -draggable $draggable]
            unset view
        } else {                                                                                     ;# use single default data view
            set table [new dataTable $canvas -data ${module}::data -draggable $draggable]
        }
        if {[info exists initializer]&&([lsearch -exact $modules(initialized) $module]>=0)} {    ;# setup initialized modules tables
            eval composite::configure $table [record::tableOptions $initializer $module $index]
            # use stored window manager initialization data for table
            foreach {x y width height level} [record::tableWindowManagerData $initializer $module $index] {}
            manageViewer $table 0 -static $static -setx $x -sety $y -setwidth $width -setheight $height -level $level
        } else {
            manageViewer $table 0 -static $static -setx $x -sety $y
        }
        set x [expr {$x+$configuration::(xWindowManagerInitialOffset)}]                   ;# offset tables to achieve a nicer layout
        set y [expr {$y+$configuration::(yWindowManagerInitialOffset)}]
        incr index                                                                                      ;# next view for initializer
    }
}

refresh                                                                                                ;# initialize refresh process

if {[info exists initializer]} {                  ;# stored configuration, now that modules data is initialized, create some viewers
    foreach {class cells x y width height level switchedOptions} [record::viewersData $initializer] {
        set viewer [eval new $class $canvas $switchedOptions -draggable $draggable]
        foreach list [composite::configure $viewer] {
            if {[string compare [lindex $list 0] -interval]==0} {                                 ;# viewer supports interval option
                composite::configure $viewer -interval $pollTime                                          ;# so use current interval
                break                                                                                                        ;# done
            }
        }
        viewer::view $viewer $cells
        manageViewer $viewer 1 -static $static -setx $x -sety $y -setwidth $width -setheight $height -level $level
    }
}
