# copyright (C) 1997-98 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: arp.tcl,v 1.7 1998/12/24 22:45:08 jfontain Exp $}

package provide arp 1.0

package require network


namespace eval arp {

    array set data {
        updates 0
        0,label IP 0,type dictionary 0,message {IP address}
        1,label hardware 1,type dictionary 1,message {hardware address}
        2,label decoded 2,type dictionary 2,message {decoded hardware address}
        3,label type 3,type ascii 3,message {hardware type}
        4,label flags 4,type ascii 4,message {entry flags (C: complete, M: permanent, P: published)}
        5,label interface 5,type ascii 5,message {networking interface}
        pollTimes {10 5 20 30 60 120 300}
        sort {0 increasing}
        helpText {
This is a view of the network Address Resolution Protocol cache for the system (tested on IPv4 only).

Host addresses are looked up and converted to official names when possible.

Hardware addresses are decoded, with a full name (from the /etc/ethers file if present, read once when the application is started), or with the vendor name for the first 3 bytes (from the vendor.txt file in the arp module sub directory).
Note: you may update the vendor.txt file yourself by opening it from the http://pax.cavebear.com/CaveBear/Ethernet/ page and saving it as a text file under the vendor.txt name in the arp module sub directory (no editing needed).
        }
    }

    if {![catch {open /etc/ethers} file]} {                                      ;# save ethernet address to name mapping if present
        while {[gets $file line]>=0} {
            if {[string match #* $line]} continue                                                                   ;# skip comments
            if {[scan $line {%x:%x:%x:%x:%x:%x %s} byte(0) byte(1) byte(2) byte(3) byte(4) byte(5) name]==7} {
                set address [format {%02X:%02X:%02X:%02X:%02X:%02X} $byte(0) $byte(1) $byte(2) $byte(3) $byte(4) $byte(5)]
                set etherName($address) $name
            }
        }
        close $file
    }

    set file [open [file join $packageDirectory vendor.txt]]                                ;# directory is set in pkgIndex.tcl file
    while {[gets $file line]>=0} {                              ;# save ethernet address code (first 3 bytes) to vendor name mapping
        if {[scan $line {%2x%2x%2x %s} byte(0) byte(1) byte(2) word]==4} {                   ;# only keep the vendor name first word
            set code [format {%02X:%02X:%02X} $byte(0) $byte(1) $byte(2)]
            set vendor($code) $word
        }
    }
    close $file

    # initialize hardware type mapping (from the /usr/src/linux/include/linux/if_arp.h file)
    array set hardwareType {
        0 NETROM 1 ETHER 2 EETHER 3 AX25 4 PRONET 5 CHAOS 6 IEEE802 7 ARCNET 8 APPLETLK 15 DLCI 23 METRICOM 256 SLIP 257 CSLIP
        258 SLIP6 259 CSLIP6 260 RSRVD 264 ADAPT 270 ROSE 271 X25 512 PPP 513 HDLC 516 LAPB 768 TUNNEL 769 TUNNEL6 770 FRAD 771 SKIP
        772 LOOPBACK 773 LOCALTLK 774 FDDI 775 BIF 776 SIT 777 IPDDP 778 IPGRE 779 PIMREG 780 HIPPI 781 ASH 782 ECONET 783 IRDA
    }


    set file [open /proc/net/arp]                                                       ;# keep the file open for better performance

    set nextIndex 0

    proc update {} {
        variable file
        variable index
        variable nextIndex
        variable data
        variable hardwareType
        variable etherName
        variable vendor

        seek $file 0                                                                                ;# rewind before retrieving data
        gets $file                                                                                               ;# skip header line
        while {[gets $file line]>=0} {
            scan $line {%u.%u.%u.%u 0x%x 0x%x %s %*s %s} ip(0) ip(1) ip(2) ip(3) type flags address interface
            if {[catch {set index($address)} row]} {
                # new entry, use unique hardware address as index instead of IP address
                # (in which case duplicate addresses could not be displayed)
                set row [set index($address) $nextIndex]
                incr nextIndex
                set data($row,1) $address
                if {$type==1} {                                                                                          ;# ethernet
                    if {[info exists etherName($address)]} {                                                      ;# try name lookup
                        set data($row,2) $etherName($address)
                    } else {
                        regexp {^(..:..:..):(..:..:..)$} $address dummy code number
                        if {[info exists vendor($code)]} {                                                      ;# try vendor lookup
                            set data($row,2) $vendor($code):$number
                        } else {
                            set data($row,2) {}                                                                    ;# lookups failed
                        }
                    }
                }
            }
            set data($row,0) $ip(0).$ip(1).$ip(2).$ip(3)
            catch {set data($row,0) [network::hostfromaddress $data($row,0)]}                                     ;# try name lookup
            set data($row,3) $hardwareType($type)
            set data($row,4) {}                                         ;# flags from the /usr/src/linux/include/linux/if_arp.h file
            if {$flags&2} {append data($row,4) C}                                                                 ;# completed entry
            if {$flags&4} {append data($row,4) M}                                                                 ;# permanent entry
            if {$flags&8} {append data($row,4) P}                                                                 ;# published entry
            set data($row,5) $interface
            set current($address) {}
        }
        foreach address [array names index] {                                                         ;# cleanup disappeared entries
            if {[info exists current($address)]} continue
            set row $index($address)
            unset index($address) data($row,0) data($row,1) data($row,2) data($row,3)
        }
        incr data(updates)
    }

}
