# copyright (C) 1997-98 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: ps.tcl,v 1.24 1998/12/22 22:04:02 jfontain Exp $}

# system dependent process data access procedures. only linux is supported at this time (no wonder: it's the greatest operating
# system of them all :)

# corresponding pkgIndex.tcl file content:
# package ifneeded ps 1.2 "source [file join $dir ps.tcl]"

package provide ps 1.2

namespace eval ps {

    array set data {
        updates 0
        0,label PID 0,type integer 0,message {process identifier}
        1,label USER 1,type ascii 1,message {user name}
        2,label %CPU 2,type real 2,message {processor usage in percent}
        3,label %MEMORY 3,type real 3,message {memory usage in percent}
        4,label RSS 4,type integer 4,message {real memory size in kiloBytes}
        5,label TTY 5,type dictionary 5,message {terminal number}
        6,label STATUS 6,type ascii 6,message {state: Running, Sleeping (D: uninterruptible), Zombie, Traced (or stopped)}
        7,label COMMAND 7,type dictionary 7,message {filename of the executable}
        pollTimes {20 10 30 60 120 300}
        columns 8
        sort {2 decreasing}
        helpText {
This is a view of processor activity. Tasks running on the system are initially sorted, with the most CPU intensive first.
        }
    }

    array set pseudoTty {128 P 144 Q 160 R 176 S 192 p 208 q 224 r 240 s}

    proc updateUserNameArray {} {
        variable userName

        set file [open /etc/passwd]
        while {[gets $file line]>=0} {                                                    ;# build a user identifier to name mapping
            set list [split $line :]
            set userName([lindex $list 2]) [lindex $list 0]
        }
        close $file
    }
    updateUserNameArray                                                   ;# inititialize user identifier to user name mapping array

    proc ttyName {number} {                                                        ;# convert a terminal number into a terminal name
        variable pseudoTty

        if {($number>>8)!=4} {
            return {}
        }
        set minor [expr {$number&0xFF}]
        if {$minor<64} {
            return $minor
        } elseif {$minor<128} {
            return S[expr {$minor&0x3F}]
        } else {
            return $pseudoTty([expr {$minor&0xF0}])[format %x [expr {$minor&0x0F}]]
        }
    }

    proc nullToSpace {data} {                                                       ;# replace null characters with space characters
        set length [string length $data]
        set string {}
        for {set index 0} {$index<$length} {incr index} {                   ;# replace null characters by spaces (regsub fails here)
            set character [string index $data $index]
            if {[string compare $character \000]==0} {
                append string { }
            } else {
                append string $character
            }
        }
        return $string
    }

    # gather processes status information (based on the proc man page information and fs/proc/array.c kernel source)
    proc update {} {
        variable data
        variable userName
        variable uid                                                                                     ;# pid to uid mapping cache
        variable cmdline                                                                                ;# pid to command line cache
        variable pidActive                                                             ;# active binary flag array for cache cleanup
        variable last

        set file [open /proc/meminfo]
        while {[gets $file line]>=0} {
            if {[scan $line {MemTotal: %u} memoryTotal]>0} break
        }
        close $file

        set current [pwd]                                                                                  ;# save current directory
        cd /proc                                                                                     ;# and change to data directory

        foreach pid [array names pidActive] {                                                                  ;# reset active flags
            set pidActive($pid) 0
        }

        foreach pid [glob -nocomplain {[1-9]*}] {
            if {![info exists uid($pid)]} {                                                              ;# if uid is not yet cached
                if {[catch {open $pid/status} file]} continue                             ;# no valid data for this process, abandon
                while {[gets $file line]>=0} {
                    if {[scan $line {Uid: %u} uid($pid)]>0} break                                               ;# save uid in cache
                }
                close $file
                if {![info exists uid($pid)]} continue         ;# process may have disappeared while we were reading its status file
                if {![info exists userName($uid($pid))]} {                                                       ;# handle new users
                    updateUserNameArray
                }
            }

            if {![info exists cmdline($pid)]} {                                                 ;# if command line is not yet cached
                if {[catch {open $pid/cmdline} file]} {                                                      ;# account for failures
                    unset uid($pid)                                                                ;# undo what has been done so far
                    continue                                                              ;# no valid data for this process, abandon
                }
                set length [gets $file line]
                close $file
                if {$length==0} {                                                                            ;# account for failures
                    unset uid($pid)
                    continue
                }
                set cmdline($pid) [nullToSpace $line]                                   ;# command line arguments are null separated
            }

            if {[catch {open $pid/stat} file]} {
                unset uid($pid) cmdline($pid)
                continue
            }
            set length [gets $file line]
            set clock [expr {[clock clicks]/1e6}]                                      ;# immediately store current clock in seconds
            close $file
            if {$length==0} {                                                                                ;# account for failures
                unset uid($pid) cmdline($pid)
                continue
            }

            set pidActive($pid) 1                                                                        ;# pid is considered active

            regexp {\((.*)\)} $line dummy comm                                                       ;# remove parenthesized command
            regsub {\(.*\)} $line {} line
            # now scan some of the remaining fields among:
            # pid comm state ppid pgrp session tty tpgid flags minflt cminflt majflt cmajflt utime stime cutime cstime priority nice
            # timeout itrealvalue starttime vsize rss rlim startcode endcode startstack kstkesp kstkeip signal blocked sigignore
            # sigcatch wchan nswap cnswap
            scan $line {\
                %*d %s %d %*d %*d %d %*d %*u %*u %*u %*u %*u %u %u %*d %*d %*d %d %*d %*d %u %u %d %*u %*u %*u %*u %*u %*u %*u %*u\
                %*u %*u %u %*u %*u\
            } state ppid tty utime stime nice starttime vsize rss wchan

            # utime, stime and starttime are in hundredths of a second
            if {![info exists last($pid,utime)]} {                                                    ;# first occurence of this pid
                set file [open uptime]
                set uptime [lindex [gets $file] 0]                                                                     ;# in seconds
                close $file
                set cpu% [expr {($utime+$stime)/($uptime-($starttime/100.0))}]            ;# use average value since process started
            } else {                                                        ;# calculate cpu utilization during the last poll period
                set cpu% [expr {($utime-$last($pid,utime)+$stime-$last($pid,stime))/($clock-$last($pid,clock))}]
            }
            array set last "$pid,clock $clock $pid,utime $utime $pid,stime $stime"

            # set row data with pid (unique by definition) as row number. take into account page size (4 kBytes)
            array set data [list\
                $pid,0 $pid $pid,1 $userName($uid($pid)) $pid,2 [format %.1f ${cpu%}]\
                $pid,3 [format %.1f [expr {409.6*$rss/$memoryTotal}]] $pid,4 [expr {4*$rss}] $pid,5 [ttyName $tty] $pid,6 $state\
                $pid,7 $comm\
            ]
        }

        foreach pid [array names pidActive] {
            if {!$pidActive($pid)} {
                unset uid($pid) cmdline($pid) pidActive($pid)                                               ;# cleanup cache entries
                foreach name [array names data $pid,*] {                                                               
                    unset data($name)                                                                                ;# current data
                    catch {unset last($name)}                                                            ;# and eventually last data
                }
            }
        }

        cd $current                                                                                     ;# restore current directory
        incr data(updates)
    }
}
