# copyright (C) 1997-1999 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: mounts.tcl,v 1.2 1998/12/26 10:43:44 jfontain Exp $}

package provide mounts 1.0

package require filesystem

namespace eval mounts {

    set file [open /proc/version]                                                                      ;# first check kernel version
    set version [lindex [gets $file] 2]
    close $file

    if {[package vcompare $version 2]<0} {
        puts stderr {mounts module needs at least a version 2 kernel}
        exit 1
    }

    array set data {
        updates 0
        0,label directory 0,type ascii 0,message {mount point}
        1,label filesystem 1,type dictionary 1,message {filesystem device}
        2,label type 2,type ascii 2,message {filesystem type}
        3,label size 3,type integer 3,message {total size in KiloBytes}
        4,label used 4,type integer 4,message {used user space in KiloBytes}
        5,label available 5,type integer 5,message {available user space in KiloBytes}
        6,label capacity 6,type integer 6,message {used user space in percent}
        pollTimes {20 5 10 30 60 120 300}
        sort {0 increasing}
        helpText {
This is a view of mounted filesystems with sizes (provided you have a 2.0 or later kernel with the proc filesystem compiled in).


Filesystem types are (from the fstab manual page):


* minix:  a local filesystem, supporting filenames of length 14 or 30 characters.

* ext:  a local filesystem with longer filenames and larger inodes (has been replaced by the ext2 file system).

* ext2:  a local filesystem with longer filenames, larger inodes, and lots of other features.

* xiafs:  a local filesystem with longer filenames, larger inodes, and lots of other features.

* msdos:  a local filesystem for MS-DOS partitions.

* hpfs:  a local filesystem for HPFS partitions.

* iso9660:  a local filesystem used for CD-ROM drives.

* nfs:  a filesystem for mounting partitions from remote systems.
        }
    }

    set nextIndex 0

    proc update {} {
        variable table
        variable index
        variable nextIndex
        variable data

        set file [open /etc/mtab]                        ;# must be opened each poll otherwise unmounted filesystems never disappear
        while {[gets $file line]>=0} {
            foreach {device directory type flags dump order} $line {}
            if {[string compare $device none]==0} continue                                 ;# ignore irrelevant entries such as proc
            if {[catch {set index($directory)} row]} {                                                                  ;# new entry
                set row [set index($directory) $nextIndex]
                incr nextIndex
                set data($row,0) $directory                                                                ;# initialize static data
                set data($row,1) $device
                set data($row,2) $type
            }
            foreach {size total available used} [filesystem::statistics $directory] {}
            set ratio [expr {$size/1024.0}]                    ;# take filesystem block size into account to get values in kilobytes
            set data($row,3) [expr {round($total*$ratio)}]
            set data($row,4) [expr {round($used*$ratio)}]
            set data($row,5) [expr {round($available*$ratio)}]
            set data($row,6) [expr {round(($used*100.0)/($used+$available))}]                                           ;# used in %
            set current($directory) {}
        }
        close $file
        foreach directory [array names index] {                                                       ;# cleanup disappeared entries
            if {[info exists current($directory)]} continue
            set row $index($directory)
            unset index($directory) data($row,0) data($row,1) data($row,2) data($row,3) data($row,4) data($row,5) data($row,6)
        }
        incr data(updates)
    }
}
