# copyright (C) 1997-1999 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: modules.tcl,v 1.8 1999/03/07 18:52:01 jfontain Exp $}

class modules {

    set modules::(all) {}

    proc modules {this} error                                                                                   ;# object-less class

    proc printAvailable {} {            ;# using Tcl built-in package management facilities, seek and print available moodss modules
        catch {package require {}}                                              ;# make sure Tcl package auto loading search is done
        foreach package [package names] {
            if {[catch {package require $package}]||![info exists ::${package}::data(updates)]} continue  ;# ignore invalid packages
            puts -nonewline "$package: possibly in"
            set count 0
            foreach directory $::auto_path {
                if {[file readable [file join $directory $package pkgIndex.tcl]]} {
                    if {$count>0} {
                        puts -nonewline ,
                    }
                    puts -nonewline " [file join $directory $package]"
                    incr count
                }
            }
            puts {}
        }
    }

    # recursive procedure: eventually initialize next module and its eventual options in command line arguments
    proc parse {arguments} {         ;# arguments list format is: module [-option [value] -option ...] module [-option ...]
        if {[llength $arguments]==0} return                                                                         ;# nothing to do
        set module [lindex $arguments 0]
        set arguments [lrange $arguments 1 end]                                         ;# point to start of switches or next module
        if {[lsearch -exact $modules::(all) $module]>=0} {
            puts stderr "$::argv0: module \"$module\" was specified more than once"
            exit 1
        }
        lifoLabel::push $global::messenger "loading $module..."
        update idletasks
        package require $module                                                                             ;# module is loaded here
        lifoLabel::pop $global::messenger
        lappend modules::(all) $module                           ;# we never get here if there is an error when the module is loaded
        set initialize [expr {[string length [namespace eval ::$module {info proc initialize}]]>0}]
        set modules::($module,initialize) $initialize
        if {[catch {set ::${module}::data(switches)} switches]} {                                         ;# module takes no options
            set modules::($module,arguments) {}                                 ;# save module arguments for eventual saving in file
        } else {                                                                                             ;# module takes options
            if {[llength $switches]==0} {
                puts stderr "module \"$module\" switches are empty"
                exit 1
            }
            if {[catch {set next [parseCommandLineArguments $switches $arguments options]} message]}  {
                puts stderr "module \"$module\" options error: $message"
                exit 1
            }
            if {!$initialize} {
                puts stderr "module \"$module\" has no initialize procedure"
                exit 1
            }
            set modules::($module,options) [array get options]
            # save module arguments for eventual saving in file
            set modules::($module,arguments) [lrange $arguments 0 [expr {[llength $arguments]-[llength $next]-1}]]
            set arguments $next
        }
        parse $arguments                                                                                         ;# process the rest
        update idletasks                                       ;# make sure latest loading message is not left showing meaninglessly
    }

    proc helpHTMLData {module} {   ;# return HTML formatted module help no matter whether provided plain or preformatted from module
        if {[catch {set ${module}::data(helpText)} text]} {
            set text {no help available}
        }
        set header "<H6>module $module</H6>\n<I>version [package provide $module]"
        if {[string length $modules::($module,arguments)]>0} {
            append header ", invoked with arguments: $modules::($module,arguments)"
        }
        append header </I><BR><BR>\n
        if {[string first <BODY> $text]<0} {                                                                         ;# regular help
            regsub -all \n $text <BR> text                                                               ;# keep original formatting
            return ${header}$text
        } else {                                                                                              ;# HTML formatted help
            regsub <BODY> $text <BODY>\n$header text                                                                ;# insert header
            return $text
        }
    }

    proc initialize {} {          ;# eventually invoke modules initialization procedures. modules must be loaded first (see parse{})
        foreach module $modules::(all) {
            if {!$modules::($module,initialize)} continue
            lifoLabel::push $global::messenger "initializing $module..."
            update idletasks
            if {[info exists modules::($module,options)]} {
                array set options $modules::($module,options)
                ::${module}::initialize options                                                        ;# let module initialize self
                unset options
            } else {                                                                                      ;# module takes no options
                ::${module}::initialize                                                                ;# let module initialize self
            }
            lifoLabel::pop $global::messenger
        }
        update idletasks                                ;# make sure latest initialization message is not left showing meaninglessly
    }

    proc setPollTimes {{override {}}} {
        set default 0
        set minimum 0
        foreach module $modules::(all) {
            set times [set ${module}::data(pollTimes)]
            if {[llength $times]==0} {
                error "module $module poll times list is empty"
            }
            # for an asynchronous module, the sole time value would be negative and is used as graph interval, for example
            set time [lindex $times 0]
            if {$time<0} {                                          ;# asynchronous module, poll time is a viewer interval (negated)
                set intervals($time) {}
                continue
            }
            if {$time>$default} {                                                              ;# default value is the first in list
                set default $time                                                    ;# keep the greater default time of all modules
            }
            set times [lsort -integer $times]                                                                     ;# sort poll times
            set time [lindex $times 0]
            if {$time>$minimum} {
                set minimum $time                                                    ;# keep the greater minimum time of all modules
                set minimumModule $module
            }
            foreach time $times {                                    ;# poll times list is the combination of all modules poll times
                set data($time) {}
            }
        }
        # sort and restrict poll times above maximum module minimum poll time
        set global::pollTimes [lsort -integer [array names data]]
        set global::pollTimes [lrange $global::pollTimes [lsearch -exact $global::pollTimes $minimum] end]
        set global::pollTime $default
        if {[string length $override]>0} {                                       ;# eventually validate command line override
            if {$override<$minimum} {
                puts stderr "$::argv0: minimum poll time is $minimum seconds for module $minimumModule"
                exit 1
            }
            set global::pollTime $override
        }
        if {$global::pollTime==0} { 
            # all modules are asynchronous, so use an average time as a viewer interval for viewers that need it, such as graphs.
            # the poll times list is empty at this point so the user cannot change the poll time.
            # note that the viewer interval can still be forced by the command line poll time option.
            set sum 0
            set number 0
            foreach interval [array names intervals] {
                incr sum $interval
                incr number
            }
            set global::pollTime [expr {round(double($sum)/-$number)}]
        }
    }

}
