# copyright (C) 1997-1999 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: ps.tcl,v 1.34 1999/02/26 21:26:57 jfontain Exp $}


# System dependent process data access procedures. Only Linux is supported at this time (who needs another OS?).
# Corresponding pkgIndex.tcl file content: package ifneeded ps 1.4 "source [file join $dir ps.tcl]"

package provide ps 1.6
package require network 1.2

namespace eval ps {

    array set data {
        updates 0
        0,label PID 0,type integer 0,message {process identifier}
        1,label user 1,type ascii 1,message {user name}
        2,label %CPU 2,type real 2,message {processor usage in percent}
        3,label %memory 3,type real 3,message {memory usage in percent}
        4,label RSS 4,type integer 4,message {real memory size in kiloBytes}
        5,label TTY 5,type dictionary 5,message {terminal number}
        6,label status 6,type ascii 6,message {state: Running, Sleeping (D: uninterruptible), Zombie, Traced (or stopped)}
        7,label name 7,type dictionary 7,message {filename of the executable} 7,anchor left
        8,label {command line} 8,type dictionary 8,message {full command line} 8,anchor left
        helpText {
This is a view of processor activity, presented in 2 tables, one featuring the full command line.
In the first table, tasks running on the system are initially sorted, with the most CPU intensive first.

Module options:

-r (--remote) [user@]host: remote monitoring using "user" as logname on remote host "host" (rsh facilities must be properly setup). If user is not specified, current user is used as logname on remote host.

Examples:

$ moodss ps -r jdoe@foo.bar.com
$ moodss ps --remote foo.bar.com
        }
        views {
            {visibleColumns {0 1 2 3 4 5 6 7} sort {2 decreasing}}
            {visibleColumns {0 7 8} sort {7 increasing}}
        }
        switches {-r 1 --remote 1}
    }

    proc initialize {optionsName} {
        upvar $optionsName options
        variable remote
        variable data

        if {![catch {set locator $options(--remote)}]||![catch {set locator $options(-r)}]} {                   ;# remote monitoring
            set data(pollTimes) {30 20 60 120 300 600}                                   ;# poll less often when remotely monitoring
            set data(2,message) {processor usage in percent (approximated)}                                         ;# see help text
            foreach {remote(user) remote(host)} [network::parseRemoteLocator $locator] {}
        } else {
            set data(pollTimes) {20 10 30 60 120 300 600}
        }
        updateUserNameArray                                               ;# inititialize user identifier to user name mapping array
    }

    proc updateUserNameArray {} {
        variable remote
        variable userName

        if {[info exists remote]} {
            set file [open "| /usr/bin/rsh -nl $remote(user) $remote(host) cat /etc/passwd"]
            fileevent $file readable {set ::ps::remote(busy) 0}
            vwait ::ps::remote(busy)
        } else {
            set file [open /etc/passwd]
        }
        while {[gets $file line]>=0} {                                                    ;# build a user identifier to name mapping
            set list [split $line :]
            set userName([lindex $list 2]) [lindex $list 0]
        }
        close $file
    }

    array set pseudoTty {128 P 144 Q 160 R 176 S 192 p 208 q 224 r 240 s}

    proc ttyName {number} {                                                        ;# convert a terminal number into a terminal name
        variable pseudoTty

        if {($number>>8)!=4} {
            return {}
        }
        set minor [expr {$number&0xFF}]
        if {$minor<64} {
            return $minor
        } elseif {$minor<128} {
            return S[expr {$minor&0x3F}]
        } else {
            return $pseudoTty([expr {$minor&0xF0}])[format %x [expr {$minor&0x0F}]]
        }
    }

    proc nullToSpace {data} {                       ;# replace null characters with space characters ### use regsub with Tcl 8.1 ###
        set length [string length $data]
        set string {}
        for {set index 0} {$index<$length} {incr index} {                   ;# replace null characters by spaces (regsub fails here)
            set character [string index $data $index]
            if {[string compare $character \000]==0} {
                append string { }
            } else {
                append string $character
            }
        }
        return $string
    }

    proc update {} {
        variable remote

        if {[info exists remote]&&!$remote(busy)} {
            remoteUpdate
        } else {
            localUpdate
        }
    }

    # gather processes status information (based on the proc man page information and fs/proc/array.c kernel source)
    proc localUpdate {} {
        variable uid                                                                                     ;# pid to uid mapping cache
        variable cmdline                                                                                ;# pid to command line cache
        variable data

        set currentDirectory [pwd]
        cd /proc                                                                                         ;# change to data directory

        set file [open meminfo]
        while {[gets $file line]>=0} {
            if {[scan $line {MemTotal: %u} memoryTotal]>0} break
        }
        close $file

        set file [open uptime]
        set uptime [lindex [gets $file] 0]                                                                             ;# in seconds
        close $file

        set pids [glob -nocomplain {[1-9]*}]
        foreach pid $pids {
            if {![info exists uid($pid)]} {                                                              ;# if uid is not yet cached
                if {[catch {open $pid/status} file]} continue                             ;# no valid data for this process, abandon
                while {[gets $file line]>=0} {
                    if {[scan $line {Uid: %u} uid($pid)]>0} break                                               ;# save uid in cache
                }
                close $file
                if {![info exists uid($pid)]} continue         ;# process may have disappeared while we were reading its status file
            }

            if {![info exists cmdline($pid)]} {                                                 ;# if command line is not yet cached
                if {[catch {open $pid/cmdline} file]} {                                                      ;# account for failures
                    unset uid($pid)                                                                ;# undo what has been done so far
                    continue                                                              ;# no valid data for this process, abandon
                }
                set length [gets $file line]
                close $file
                if {$length==0} {                                                                            ;# account for failures
                    unset uid($pid)
                    continue
                }
                set cmdline($pid) [nullToSpace $line]                                   ;# command line arguments are null separated
            }

            if {[catch {open $pid/stat} file]} {                                                     ;# process may have disappeared
                unset uid($pid) cmdline($pid)
                continue
            }
            set length [gets $file line]
            set clock [expr {[clock clicks]/1e6}]                                      ;# immediately store current clock in seconds
            close $file
            if {$length==0} {                                                                                ;# account for failures
                unset uid($pid) cmdline($pid)
                continue
            }

            set current($pid) {}                                                                       ;# remember process existence

            updateProcessData $pid $uptime $clock $memoryTotal $line
        }

        cd $currentDirectory
        cleanupProcessesData current                                                          ;# clean up disappeared processes data
        incr data(updates)
    }

    proc remoteUpdate {} {                                    ;# modeled after localUpdate{}: comments there are not duplicated here
        variable remote

        # gather data at the other end in 1 shot to minimize network delays and remote processor usage, while also minimizing
        # local processing (data returned in array set command compatible list) and ignoring errors (empty data is then returned)

set command {\
exec 2>/dev/null
cd /proc
echo pid {$$}
pids=$(echo [0-9]*)
echo pids {$pids}
echo meminfo {$(fgrep MemTotal: meminfo)}
for id in $pids; do
    echo $id,status {$(fgrep Uid: $id/status)} $id,cmdline {$(tr "\0" " " < $id/cmdline)} $id,stat {$(cat $id/stat)}
done
echo uptime {$(cat uptime)}\
}
        set file [open [list | /usr/bin/rsh -nl $remote(user) $remote(host) $command]]
        fileevent $file readable "ps::remoteUpdated $file"                   ;# do not hang user interface and other modules updates
    }

    proc remoteUpdated {file} {                               ;# modeled after localUpdate{}: comments there are not duplicated here
        variable remote
        variable uid
        variable cmdline
        variable data

        array set there [read $file]
        close $file                                                                ;### errors and timeouts could be caught here ###

        # array sample extract:
        # there(pid)         = 2504
        # there(pids)        = 1 1163 1164 1239 142 151 162 173 184 2 215 216 218 233 235 237 238 239 2494 2502 2504 2509 260 3
        # there(meminfo)     = MemTotal: 63520 kB
        # there(uptime)      = 1611.47 1549.09
        # there(184,cmdline) = gpm -t ps/2
        # there(184,stat)    = 184 (gpm) S 1 184 184 0 -1 320 8 0 11 0 0 0 0 0 0 0 0 0 1757 761856 85 2147483647 134512640 ...
        # there(184,status)  = Uid: 0 0 0 0

        if {[scan $there(meminfo) {MemTotal: %u} memoryTotal]!=1} {                                         ;# invalid returned data
            cleanupProcessesData current                                                                ;# remove all processes data
            return
        }

        if {[llength $there(uptime)]==0} {                                                                  ;# invalid returned data
            cleanupProcessesData current                                                                ;# remove all processes data
            return
        }
        set uptime [lindex $there(uptime) 0]                                                ;# expected (example): "1611.47 1549.09"
        set clock [expr {[clock clicks]/1e6}]                ;# use the same local clock value for all remote processes calculations

        ldelete there(pids) $there(pid)                                                      ;# ignore data retrieval process itself

        foreach pid $there(pids) {
            if {![info exists uid($pid)]&&([scan $there($pid,status) {Uid: %u} uid($pid)]!=1)} continue
            if {![info exists cmdline($pid)]} {
                set cmdline($pid) $there($pid,cmdline)
            }
            if {[string length $there($pid,stat)]==0} {
                unset uid($pid) cmdline($pid)
                continue
            }
            set current($pid) {}
            updateProcessData $pid $uptime $clock $memoryTotal $there($pid,stat)
        }

        cleanupProcessesData current
        set remote(busy) 0
        incr data(updates)
    }

    proc updateProcessData {pid uptime clock memoryTotal statLine} {
        variable last
        variable data
        variable userName
        variable uid                                                                                     ;# pid to uid mapping cache
        variable cmdline                                                                                ;# pid to command line cache

        # scan some of the fields among:
        # pid comm state ppid pgrp session tty tpgid flags minflt cminflt majflt cmajflt utime stime cutime cstime priority nice
        # timeout itrealvalue starttime vsize rss rlim startcode endcode startstack kstkesp kstkeip signal blocked sigignore
        # sigcatch wchan nswap cnswap
        scan $statLine {\
            %*d (%[^)]) %s %d %*d %*d %d %*d %*u %*u %*u %*u %*u %u %u %*d %*d %*d %d %*d %*d %u %u %d %*u %*u %*u %*u %*u %*u %*u\
            %*u %*u %*u %u %*u %*u\
        } comm state ppid tty utime stime nice starttime vsize rss wchan

        # utime, stime and starttime are in hundredths of a second
        if {![info exists last($pid,utime)]} {                                                        ;# first occurence of this pid
            set cpu% [expr {($utime+$stime)/($uptime-($starttime/100.0))}]                ;# use average value since process started
        } else {                                                            ;# calculate cpu utilization during the last poll period
            set cpu% [expr {($utime-$last($pid,utime)+$stime-$last($pid,stime))/($clock-$last($pid,clock))}]
        }
        array set last "$pid,clock $clock $pid,utime $utime $pid,stime $stime"

        set user $uid($pid)                                                                  ;# user name defaults to its identifier
        if {[catch {set user $userName($user)}]} {
            updateUserNameArray                                                                                  ;# handle new users
            if {[catch {set user $userName($user)}]} {                                                               ;# unknown user
                set userName($user) $user                                                   ;# keep using its identifier from now on
            }
        }

        # set row data with pid (unique by definition) as row number. take into account page size (4 kBytes)
        array set data [list\
            $pid,0 $pid $pid,1 $user $pid,2 [format %.1f ${cpu%}] $pid,3 [format %.1f [expr {409.6*$rss/$memoryTotal}]]\
            $pid,4 [expr {4*$rss}] $pid,5 [ttyName $tty] $pid,6 $state $pid,7 $comm $pid,8 $cmdline($pid)\
        ]
    }

    proc cleanupProcessesData {currentPidsName} {
        upvar $currentPidsName current
        variable uid
        variable cmdline
        variable data
        variable last

        foreach pid [array names uid] {
            if {[info exists current($pid)]} continue
            unset uid($pid) cmdline($pid)                                                                   ;# cleanup cache entries
            foreach name [array names data $pid,\[0-9\]*] {                                                        ;# cell data only
                unset data($name)                                                                                    ;# current data
                catch {unset last($name)}                                                                ;# and eventually last data
            }
        }
    }

}
