# copyright (C) 1997-1999 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: memstats.tcl,v 1.9 1999/02/26 21:11:32 jfontain Exp $}


package provide memstats 1.3
package require network 1.2

namespace eval memstats {

    array set data {
        updates 0
        0,label type 0,type ascii 0,message {type}
        1,label available 1,type integer 1,message {total available}
        2,label used 2,type integer 2,message {used}
        3,label free 3,type integer 3,message {free}
        4,label shared 4,type integer 4,message {shared memory}
        5,label buffers 5,type integer 5,message {memory used for buffers}
        6,label cached 6,type integer 6,message {cached memory}
        columns 7
        sort {0 increasing}
        helpText {
This is a view of memory and swap statistics for the system as a whole.

Module options:

-r (--remote) [user@]host: remote monitoring using "user" as logname on remote host "host" (rsh facilities must be properly setup). If user is not specified, current user is used as logname on remote host.

Examples:

$ moodss ps -r jdoe@foo.bar.com
$ moodss ps --remote foo.bar.com
        }
        switches {-r 1 --remote 1}
    }

    proc initialize {optionsName} {
        upvar $optionsName options
        variable remote
        variable data
        variable memoryFile

        set lookup [expr {![info exists options(-n)]&&![info exists options(--numeric)]}]            ;# host or network names lookup
        if {![catch {set locator $options(--remote)}]||![catch {set locator $options(-r)}]} {                   ;# remote monitoring
            set data(pollTimes) {20 10 30 60 120 300 600}                                ;# poll less often when remotely monitoring
            foreach {remote(user) remote(host)} [network::parseRemoteLocator $locator] {}
            set file [open "| /usr/bin/rsh -nl $remote(user) $remote(host) cat /proc/meminfo"]
            fileevent $file readable {set ::memstats::remote(busy) 0}
            vwait ::memstats::remote(busy)                                                                  ;# allow GUI interaction
            if {[catch {close $file} message]} {
                puts stderr "on remote host $remote(host) as user $remote(user): $message"
                exit 1                                                                                        ;# detect errors early
            }
        } else {
            set data(pollTimes) {10 5 20 30 60 120 300 600}
            set memoryFile [open /proc/meminfo]                                       ;# keep local file open for better performance
        }
    }

    array set data {0,0 memory 0,6 {} 1,0 swap 1,4 {} 1,5 {}}                                             ;# initialize static cells

    proc update {} {                                               ;# gather cpu statistics (based on the proc man page information)
        variable remote
        variable memoryFile
        variable data

        if {[info exists remote]} {
            if {![info exists memoryFile]} {                               ;# start data gathering process in a non blocking fashion
                if {$remote(busy)} return                                           ;# core invocation while waiting for remote data
                set remote(busy) 1
                set file [open "| /usr/bin/rsh -nl $remote(user) $remote(host) cat /proc/meminfo"]
                # do not hang GUI, allow other modules updates
                fileevent $file readable "set ::memstats::memoryFile $file; ::memstats::update"
                return                                                                                       ;# wait for remote data
            }                                                                                 ;# else continue below to process data
        } else {
            seek $memoryFile 0                                                                      ;# rewind before retrieving data
        }
        gets $memoryFile; gets $memoryFile; gets $memoryFile                                         ;# skip redundant first 3 lines
        set data(0,1) [lindex [gets $memoryFile] 1]                                                                  ;# total memory
        set data(0,3) [lindex [gets $memoryFile] 1]                                                                   ;# free memory
        set data(0,2) [expr {$data(0,1)-$data(0,3)}]                                                                  ;# used memory
        set data(0,4) [lindex [gets $memoryFile] 1]                                                                 ;# shared memory
        set data(0,5) [lindex [gets $memoryFile] 1]                                                                       ;# buffers
        set data(1,6) [lindex [gets $memoryFile] 1]                                                                 ;# cached memory
        set data(1,1) [lindex [gets $memoryFile] 1]                                                                    ;# total swap
        set data(1,3) [lindex [gets $memoryFile] 1]                                                                     ;# free swap
        set data(1,2) [expr {$data(1,1)-$data(1,3)}]                                                                    ;# used swap
        if {[info exists remote]} {                                                 ;# closing is necessary since seek does not work
            close $memoryFile
            unset memoryFile
            set remote(busy) 0
        }
        incr data(updates)
    }
}
