# copyright (C) 1997-1999 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: modules.tcl,v 1.12 1999/07/14 15:48:46 jfontain Exp $}

class modules {

    set modules::(names) {}
    set modules::(indexed) {}
    set modules::(duplicates) 0                            ;# boolean to indicate whether there exists several instances of a module

    proc modules {this} error                                                                                   ;# object-less class

    proc printAvailable {} {            ;# using Tcl built-in package management facilities, seek and print available moodss modules
        catch {package require {}}                                              ;# make sure Tcl package auto loading search is done
        foreach package [package names] {
            if {[catch {package require $package}]||![info exists ::${package}::data(updates)]} continue  ;# ignore invalid packages
            puts -nonewline "$package: possibly in"
            set count 0
            foreach directory $::auto_path {
                if {[file readable [file join $directory $package pkgIndex.tcl]]} {
                    if {$count>0} {
                        puts -nonewline ,
                    }
                    puts -nonewline " [file join $directory $package]"
                    incr count
                }
            }
            puts {}
        }
    }

    # recursive procedure: eventually initialize next module and its eventual options in command line arguments
    proc parse {arguments} {         ;# arguments list format is: module [-option [value] -option ...] module [-option ...]
        if {[llength $arguments]==0} return                                                                         ;# nothing to do
        set module [lindex $arguments 0]
        set arguments [lrange $arguments 1 end]                                         ;# point to start of switches or next module

        scan $module {%[^<]<%u>} module index    ;# eventually split module into its name and its index (if coming from a save file)

        if {![info exists ::packageDirectory($module)]} {                             ;# not a valid module (usually a wrong switch)
            puts stderr "error: \"$module\" is not a valid moodss module name"
            exit 1
        }

        lifoLabel::push $global::messenger "loading $module..."
        update idletasks

        if {[info exists index]} {                                                                             ;# force module index
            set indexed [load $module $index]
        } else {
            set indexed [load $module]
        }

        lifoLabel::pop $global::messenger
        lappend modules::(indexed) $indexed                      ;# we never get here if there is an error when the module is loaded
        if {[lsearch -exact $modules::(names) $module]<0} {                                     ;# keep track of loaded module names
            lappend modules::(names) $module
        }
        if {[catch {set ::${indexed}::data(switches)} switches]} {                                        ;# module takes no options
            set modules::($indexed,arguments) {}                                ;# save module arguments for eventual saving in file
        } else {                                                                                             ;# module takes options
            if {[llength $switches]==0} {
                puts stderr "module \"$module\" switches are empty"
                exit 1
            }
            if {[catch {set next [parseCommandLineArguments $switches $arguments options]} message]}  {
                puts stderr "module \"$module\" options error: $message"
                exit 1
            }
            if {!$modules::($indexed,initialize)} {
                puts stderr "module \"$module\" has no initialize procedure"
                exit 1
            }
            set modules::($indexed,options) [array get options]
            # save module arguments for eventual saving in file
            set modules::($indexed,arguments) [lrange $arguments 0 [expr {[llength $arguments]-[llength $next]-1}]]
            set arguments $next
        }
        parse $arguments                                                                                         ;# process the rest
        update idletasks                                       ;# make sure latest loading message is not left showing meaninglessly
    }

    proc helpHTMLData {indexed} {  ;# return HTML formatted module help no matter whether provided plain or preformatted from module
        if {[catch {set ${indexed}::data(helpText)} text]} {
            set text {no help available}
        }
        set header "<H6>module $indexed</H6>\n<I>version $modules::($indexed,version)"
        if {[string length $modules::($indexed,arguments)]>0} {
            append header ", invoked with arguments: $modules::($indexed,arguments)"
        }
        append header </I><BR><BR>\n
        if {[string first <BODY> $text]<0} {                                                                         ;# regular help
            regsub -all \n $text <BR> text                                                               ;# keep original formatting
            return ${header}$text
        } else {                                                                                              ;# HTML formatted help
            regsub <BODY> $text <BODY>\n$header text                                                                ;# insert header
            return $text
        }
    }

    proc initialize {} {          ;# eventually invoke modules initialization procedures. modules must be loaded first (see parse{})
        foreach indexed $modules::(indexed) {
            if {!$modules::($indexed,initialize)} continue
            lifoLabel::push $global::messenger "initializing $indexed..."
            update idletasks
            if {[info exists modules::($indexed,options)]} {
                ::${indexed}::initialize $modules::($indexed,options)                                  ;# let module initialize self
            } else {                                                                                      ;# module takes no options
                ::${indexed}::initialize                                                               ;# let module initialize self
            }
            synchronize $indexed                                                                         ;# in case data was updated
            lifoLabel::pop $global::messenger
        }
        update idletasks                                ;# make sure latest initialization message is not left showing meaninglessly
    }

    proc setPollTimes {{override {}}} {
        set default 0
        set minimum 0
        foreach indexed $modules::(indexed) {
            set times [set ${indexed}::data(pollTimes)]
            if {[llength $times]==0} {
                error "module $indexed poll times list is empty"
            }
            # for an asynchronous module, the sole time value would be negative and is used as graph interval, for example
            set time [lindex $times 0]
            if {$time<0} {                                          ;# asynchronous module, poll time is a viewer interval (negated)
                set intervals($time) {}
                continue
            }
            if {$time>$default} {                                                              ;# default value is the first in list
                set default $time                                                    ;# keep the greater default time of all modules
            }
            set times [lsort -integer $times]                                                                     ;# sort poll times
            set time [lindex $times 0]
            if {$time>$minimum} {
                set minimum $time                                                    ;# keep the greater minimum time of all modules
                set minimumModule $indexed
            }
            foreach time $times {                                    ;# poll times list is the combination of all modules poll times
                set data($time) {}
            }
        }
        # sort and restrict poll times above maximum module minimum poll time
        set global::pollTimes [lsort -integer [array names data]]
        set global::pollTimes [lrange $global::pollTimes [lsearch -exact $global::pollTimes $minimum] end]
        set global::pollTime $default
        if {[string length $override]>0} {                                       ;# eventually validate command line override
            if {$override<$minimum} {
                puts stderr "$::argv0: minimum poll time is $minimum seconds for module $minimumModule"
                exit 1
            }
            set global::pollTime $override
        }
        if {$global::pollTime==0} { 
            # all modules are asynchronous, so use an average time as a viewer interval for viewers that need it, such as graphs.
            # the poll times list is empty at this point so the user cannot change the poll time.
            # note that the viewer interval can still be forced by the command line poll time option.
            set sum 0
            set number 0
            foreach interval [array names intervals] {
                incr sum $interval
                incr number
            }
            set global::pollTime [expr {round(double($sum)/-$number)}]
        }
    }

    # load a module in its own interpreter, in order to allow multiple instances of the same module
    proc load {module {index 0}} {                                             ;# index may be forced (when coming from a save file)
        variable nextIndex

        # maintain an index for each module in case it is loaded several times
        if {($index==0)&&[catch {set index $nextIndex($module)}]} {
            set index 1
        }
        if {$index>1} {                                                            ;# there are several instances of the same module
            set modules::(duplicates) 1
        }

        set indexed ${module}<$index>                                                                          ;# unique module name
        set modules::($indexed,name) $module

        set interpreter [interp create]                                                ;# use a separate interpreter for each module
        set modules::($indexed,interpreter) $interpreter
        set ::packageDirectory($indexed) $::packageDirectory($module)                        ;# set indexed module package directory
        $interpreter eval "
            set auto_path [list $::auto_path]                          ;# copy a few required global variables in module interpreter
            set ::packageDirectory($module) $::packageDirectory($module)
            package require $module                                                                         ;# module is loaded here
        "
        # we never get here if there is an error when the module is loaded
        # the indexed namespace, "interface' to the protected module namespace in its interpreter, is child of global namespace
        namespace eval ::$indexed [subst -nocommands {
            proc update {} {$interpreter eval ::${module}::update}
        }]
        set modules::($indexed,initialize)\
            [$interpreter eval [subst -nocommands {expr {[string length [namespace eval ::$module {info proc initialize}]]>0}}]]
        if {$modules::($indexed,initialize)} {                                                        ;# initialize procedure exists
            namespace eval ::$indexed [subst -nocommands {      ;# create an interface initialize procedure within indexed namespace
                proc initialize {arguments} {                     ;# arguments are a list of option / value (eventually empty) pairs
                    $interpreter eval "
                        array set _options [list \$arguments]
                        ::${module}::initialize _options
                        unset _options
                    "
                }
            }]
        }
        set modules::($indexed,version) [$interpreter eval "package provide $module"]
        synchronize $indexed                                        ;# initialize indexed module data from module in its interpreter
        # keep on eye on special module data array member "update"
        $interpreter alias _updated ::modules::updated $indexed
        $interpreter eval "trace variable ::${module}::data(updates) w _updated"

        set nextIndex($module) [incr index]

        return $indexed                                                                               ;# new indexed name for module
    }

    proc updated {indexed args} {                              ;# module data was just updated. ignore already known trace arguments
        synchronize $indexed {[0-9]*,[0-9]*}                                         ;# just copy all dynamic data from module array
        # and copy updates counter
        set ::${indexed}::data(updates) [$modules::($indexed,interpreter) eval "set ::$modules::($indexed,name)::data(updates)"]
    }

    proc synchronize {indexed {pattern *}} {                      ;# copy data from module in its interpreter to indexed module here
        array set ::${indexed}::data\
            [$modules::($indexed,interpreter) eval "array get ::$modules::($indexed,name)::data {$pattern}"]
    }

    proc identifier {array} {            ;# from a module data array name, deduce a unique module identifier (used in viewer labels)
        return [string trimleft [namespace qualifiers [namespace which -variable $array]] :]
    }

}
