#!/bin/sh
# the next line restarts using the interpreter \
exec wish "$0" "$@"

# copyright (C) 1997-1999 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu
proc echo {args} {foreach argument $args {puts -nonewline "$argument "}; puts {}}

proc recordRCSId {variableName empty operation} {
    foreach {header file version} $::rcsId break
    regsub {,v$} $file {} file
    set ::sourceVersion($file) $version
}
trace variable ::rcsId w recordRCSId                                                                ;# keep track of RCS identifiers

set rcsId {$Id: moodss.tcl,v 1.169 1999/07/14 14:28:13 jfontain Exp $}


source utility.tcl
startGatheringPackageDirectories

source global.tcl
source preferen.tcl
preferences::load                                                                     ;# initialize from rc file as soon as possible
source getopt.tcl

proc printUsage {exitCode} {
    puts stderr "Usage: $::argv0 \[OPTION\]... \[MODULE\] \[MODULE\]..."
    puts stderr {  -f, --file       configuration file name}
    puts stderr {  -h, --help       display this help and exit}
    puts stderr {  -p, --poll-time  poll time in seconds}
    puts stderr {  -r, --read-only  disable viewer creation, editing, ...}
    puts stderr {  -S, --static     disable internal window manager sizing and moving}
    puts stderr {  --show-modules   try to find available moodss modules}
    puts stderr {  --version        output version information and exit}
    exit $exitCode
}

if {[catch\
    {\
        set argv [parseCommandLineArguments\
            {-f 1 --file 1 -h 0 --help 0 -p 1 --poll-time 1 -r 0 --read-only 0 -S 0 --static 0 --show-modules 0 --version 0} $argv\
            arguments\
        ]\
    }\
    message\
]} {
    puts stderr $message
    printUsage 1
}

foreach {short long} {-f --file -h --help -p --poll-time -r --read-only -S --static} {
    catch {set arguments($short) $arguments($long)}                                          ;# long version if present has priority
}

if {[info exists arguments(--version)]} {
    puts "moodss (a Modular Object Oriented Dynamic SpreadSheet) version $global::applicationVersion"
    exit
}
if {
    [info exists arguments(-h)]||
    (([llength $argv]==0)&&![info exists arguments(-f)]&&![info exists arguments(--show-modules)])
} {
    printUsage 1
}

lappend auto_path /usr/lib/moodss .    ;# search in current directory sub-directories for development and rpm installation directory

if {[catch {package require stooop 3.7}]} {
    source stooop.tcl                                                                     ;# in case stooop package is not installed
}
namespace import stooop::*
if {[catch {package require switched 1.4}]} {                                           ;# in case switched package is not installed
    source switched.tcl
}

source modules.tcl

if {[info exists arguments(--show-modules)]} {
    modules::printAvailable
    exit                                                                                                                     ;# done
}

if {[catch {package require scwoop 2.6}]} {
    source scwoutil.tcl
    source scwoop.tcl                                                                     ;# in case scwoop package is not installed
    source bindings.tcl
    source widgetip.tcl
    source arrowbut.tcl
    source spinent.tcl
    source panner.tcl
    source scroll.tcl
}
if {[catch {package require tkpiechart 5.3}]} {                                       ;# in case tkpiechart package is not installed
    source pielabel.tcl
    source boxlabel.tcl
    source canlabel.tcl
    source labarray.tcl
    source perilabel.tcl
    source slice.tcl
    source selector.tcl
    source objselec.tcl
    source pie.tcl
}
if {[catch {package require Tktable 2}]} {                                               ;# in case tkTable package is not installed
    switch $tcl_platform(platform) {
        unix {
            load ./Tktable.so.2.5
        }
        windows {
            load ./tktable.dll
        }
    }
}
if {[catch {package require BLT 2.4}]} {                                                     ;# in case BLT package is not installed
    switch $tcl_platform(platform) {
        unix {
            load ./libBLT.so.2.4i
        }
        windows {
            load ./blt24.dll
        }
    }
}

wm protocol . WM_DELETE_WINDOW exit      ;# so that closing from window manager works when hung initializing a remote capable module
wm command . [concat $argv0 $argv]                                   ;# for proper window manager (windowmaker for example) behavior
wm group . .

source font.tcl
source lifolbl.tcl
source gui.tcl

pack [createMessageWidget .] -side bottom -fill x                              ;# so that modules can display informational messages
update

wm title . {moodss: loading modules...}                   ;# load uninitialized modules 1st so that their tables are placed properly
modules::parse $argv                                                                                                    ;# recursive

source record.tcl

if {[info exists arguments(-f)]} {                                                              ;# configuration file name specified
    set global::saveFile $arguments(-f)
    set initializer [new record -file $global::saveFile]
    record::read $initializer
    modules::parse [record::modulesWithArguments $initializer]
    set modules::(initialized) [record::modules $initializer]
} else {
    set global::saveFile {}
}

wm title . {moodss: initializing modules...}
modules::initialize                                                                            ;# with user feedback in message area

source canvhand.tcl
source canvaswm.tcl
source viewer.tcl
source blt2d.tcl
source databar.tcl
source datagraf.tcl
source datapie.tcl
source sumtable.tcl
source freetext.tcl
source drag.tcl
source drop.tcl
source menuhelp.tcl
source wcb.tcl
source wcbCommon.tcl
source wcbEntry.tcl

set readOnly [info exists arguments(-r)]
set global::static [info exists arguments(-S)]

set scroll [new scroll canvas .]
set global::canvas $composite::($scroll,scrolled,path)
$global::canvas configure -highlightthickness 0 -takefocus 0
trace variable global::canvasHeight w updateCanvasSize                                       ;# track changes for good user feedback
trace variable global::canvasWidth w updateCanvasSize
updateCanvasSize
trace variable global::canvasBackground w updateCanvasBackground                             ;# track changes for good user feedback
updateCanvasBackground

set global::windowManager [new canvasWindowManager $global::canvas]

if {[info exists ::geometry]} {                                                               ;# command line geometry was specified
    wm geometry . $::geometry
} elseif {[info exists initializer]} {
    foreach {width height} [record::sizes $initializer] {}                                   ;# used stored geometry for main window
    wm geometry . ${width}x$height
} else {
    wm geometry . 400x300
}

wm iconname . "moodss $modules::(names)"                                                          ;# give the icon a meaningful name
image create photo applicationIcon -data [dataGraph::iconData]                                ;# use data graph icon for application
if {[string compare $tcl_platform(platform) unix]==0} {
    wm iconwindow . [toplevel .icon]
    pack [label .icon.image -image applicationIcon]
}

if {!$readOnly} {
    pack [createDragAndDropZone .] -fill x
}
pack $widget::($scroll,path) -fill both -expand 1

set modules::(synchronous) {}                                                            ;# save list of synchronous indexed modules
set names {}
foreach indexed $modules::(indexed) {
    if {[lindex [set ${indexed}::data(pollTimes)] 0]>0} {                                                ;# if module is synchronous
        lappend modules::(synchronous) $indexed
        set module $modules::($indexed,name)
        if {[lsearch -exact $names $module]<0} {
            lappend names $module
        }
    }
}
set modules::(synchronous,string) [commaSeparatedString $names]                     ;# list on non indexed synchronous modules names

if {[info exists arguments(-p)]} {                                                     ;# command line argument has highest priority
    modules::setPollTimes $arguments(-p)
} elseif {[info exists initializer]} {                                                                  ;# then stored configuration
    modules::setPollTimes [record::pollTime $initializer]
} else {                                                                                                       ;# use modules values
    modules::setPollTimes
}

updateTitle                                                                              ;# now that modules and poll time are known
createMenuWidget . $readOnly [llength $global::pollTimes]

source keyslink.tcl
source dialog.tcl
source tablesel.tcl
source datatab.tcl
source lastwish.tcl
source htmllib.tcl                                                ;# Tcl HTML library from Sun, used for viewing HTML help documents
source htmldata.tcl
source html.tcl                                            ;# must be sourced after HTML library since some procedures are redefined
source help.tcl

proc createCellsViewer {class cells draggable static {pollTime {}}} {
    set viewer [new $class $global::canvas -draggable $draggable]
    if {[string length $pollTime]>0} {
        composite::configure $viewer -interval $pollTime
    }
    viewer::view $viewer $cells
    manageViewer $viewer 1 -static $static
    return $viewer
}

proc manageViewer {viewer destroyable args} {                         ;# arguments are window manager configuration switched options
    set path $widget::($viewer,path)
    canvasWindowManager::manage $global::windowManager $path
    eval canvasWindowManager::configure $global::windowManager $path $args
    if {$destroyable} {
        composite::configure $viewer -deletecommand "canvasWindowManager::unmanage $global::windowManager $path"
    }
}

proc save {ask} {                               ;# save current configuration in user defined file or currently defined storage file
    if {$ask||([string length $global::saveFile]==0)} {
        set file [tk_getSaveFile\
            -title {moodss: Save} -initialdir [pwd] -defaultextension .moo -filetypes {{{moodss data} .moo}}\
            -initialfile $global::saveFile
        ]
        if {[string length $file]==0} return                                                                 ;# user canceled saving
        updateFileSaveMenuHelp [set global::saveFile $file]
    }
    # flash instead of pushing as that goes too fast to be seen
    lifoLabel::flash $global::messenger "saving in $global::saveFile..."
    set record [new record -file $global::saveFile]
    record::write $record
    delete $record
}

proc refresh {} {
    static updateEvent

    catch {after cancel $updateEvent}                                                             ;# eventually cancel current event

    if {[llength $modules::(synchronous)]==0} return                                                                ;# nothing to do

    # we do not know when data will actually be updated
    lifoLabel::push $global::messenger "launching $modules::(synchronous,string) data update..."
    update idletasks
    foreach module $modules::(synchronous) {
        ${module}::update                                                                   ;# ask module to update its dynamic data
    }
    lifoLabel::pop $global::messenger
    set updateEvent [after [expr {1000*$global::pollTime}] refresh]                                       ;# convert to milliseconds
}

set draggable [expr {!$readOnly}]
set x 0.0
set y 0.0
foreach module $modules::(indexed) {
    if {[info exists ${module}::data(views)]} {                                                        ;# user overrode default view
        set viewMembers [set ${module}::data(views)]                                      ;# create and manage a table for each view
    } else {
        set viewMembers {{}}                                                           ;# there is a single table (the default view)
    }
    set index 0
    foreach members $viewMembers {
        if {[llength $members]>0} {                                                                                  ;# it is a view
            array set view $members
            set table [new dataTable $global::canvas -data ${module}::data -view ::view -draggable $draggable]
            unset view
        } else {                                                                                     ;# use single default data view
            set table [new dataTable $global::canvas -data ${module}::data -draggable $draggable]
        }
        if {[info exists initializer]&&([lsearch -exact $modules::(initialized) $module]>=0)} {  ;# setup initialized modules tables
            eval composite::configure $table [record::tableOptions $initializer $module $index]
            # use stored window manager initialization data for table
            foreach {x y width height level} [record::tableWindowManagerData $initializer $module $index] {}
            manageViewer $table 0 -static $global::static -setx $x -sety $y -setwidth $width -setheight $height -level $level
        } else {
            manageViewer $table 0 -static $global::static -setx $x -sety $y
        }
        set x [expr {$x+$global::xWindowManagerInitialOffset}]                            ;# offset tables to achieve a nicer layout
        set y [expr {$y+$global::yWindowManagerInitialOffset}]
        incr index                                                                                      ;# next view for initializer
    }
}

refresh                                                                                                ;# initialize refresh process

if {[info exists initializer]} {                  ;# stored configuration, now that modules data is initialized, create some viewers
    foreach {class cells x y width height level switchedOptions} [record::viewersData $initializer] {
        set viewer [eval new $class $global::canvas $switchedOptions -draggable $draggable]
        foreach list [composite::configure $viewer] {
            if {[string compare [lindex $list 0] -interval]==0} {                                 ;# viewer supports interval option
                composite::configure $viewer -interval $global::pollTime                                  ;# so use current interval
                break                                                                                                        ;# done
            }
        }
        viewer::view $viewer $cells
        manageViewer $viewer 1 -static $global::static -setx $x -sety $y -setwidth $width -setheight $height -level $level
    }
}
