# copyright (C) 1997-1999 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: preferen.tcl,v 1.5 1999/06/27 20:19:34 jfontain Exp $}

# general
#     main window size
#     save directory
# canvas
#     size
#     background
# viewers
#     color series
#     graphs
#         samples
#     pies
#         labeling
#     freetext
#         font


namespace eval preferences {

    variable rcFileName ~/.moodssrc
    variable container
    variable interface

    # list of: , initialize procedure, edit procedure, check procedure, apply procedure
### add configuration ok boolean ###
    variable hierarchy {
        {canvas canvas::initialize canvas::edit canvas::check canvas::apply canvas::saveList}
        {canvas.size canvas::size::initialize canvas::size::edit canvas::size::check canvas::size::apply canvas::size::saveList}
        {
            canvas.colors canvas::colors::initialize canvas::colors::edit canvas::colors::check canvas::colors::apply
            canvas::colors::saveList
        }
        {viewers viewers::initialize viewers::edit viewers::check viewers::apply viewers::saveList}
    }

    proc load {} {
        variable rcFileName

        if {![file readable $rcFileName]} return
        set file [::open $rcFileName] ;### detect errors and report failures to user ###
        foreach {name value} [read $file] {
            set $name $value
        }
        close $file
    }

    proc edit {} {
        variable hierarchy
        variable container
        variable interface
        variable tree

        set dialog [new dialogBox .\
            -buttons oc -default o -title {moodss: Preferences} -x [winfo pointerx .] -y [winfo pointery .]\
            -command preferences::done\
        ]
        set frame [frame $widget::($dialog,path).frame]

        set tree [blt::hierbox $frame.tree\
            -font $font::(mediumBold) -separator . -selectmode single -selectbackground lightgray -hideroot 1 -borderwidth 1\
            -highlightthickness 0 -takefocus 0 -width 150\
        ]
        set container [frame $frame.container -borderwidth 1 -relief sunken]

        set message [createMessage $container.message\
            -text {This is the preferences dialog box for
application-wide settings.

On UNIX systems, preferences data is saved
in each user home directory under the rc
file named .moodssrc.

On Windows, data is saved in C:\.moodssrc
(by default, but depends on the HOME variable).

After selection of the category from the tree
on the left, a related dialog interface
replaces this message.

Contextual help is provided for each interface,
which may or may not allow immediately applying
new data values.

Clicking on the OK button results in the
preferences data to be written to the rc file.}\
        ]
        pack $message -fill both -expand 1                                                   ;# initially display help message above

        bindtags $tree [list $tree [winfo toplevel $tree] all]                                              ;# ignore class bindtags
        $tree bind all <Double-ButtonPress-1> {}                                                           ;# and selection bindings
        $tree bind all <Shift-ButtonPress-1> {}
        $tree bind all <Control-ButtonPress-1> {}
        $tree bind all <B1-Motion> {}
        # only keep opening binding (no longer on double click)
### check before toggling ###
        $tree bind all <ButtonRelease-1> "$tree toggle current; $tree toggle current"

        catch {unset interface(current)}                                                               ;# currently opened interface

        foreach entry $hierarchy {
            foreach {name initialize edit check apply saveList} $entry {
                $initialize
                set index [$tree insert end $name]                                     ;# used generated tree index as unique index
                set interface($index,edit) $edit
                set interface($index,check) $check
                set interface($index,apply) $apply
                set interface($index,saveList) $saveList
### eventually use close command for checking ###
                $tree entry configure $index -opencommand "preferences::open $index"
            }
        }

        pack $tree -side left -fill y -padx 2
        pack $container -fill both -expand 1 -padx 2

        dialogBox::display $dialog $frame

        wm geometry $widget::($dialog,path) 500x400                                                        ;# maintain constant size
    }

    proc open {index} {
        variable container
        variable interface

        # check validity before moving to next interface
        if {[info exists interface(current)]&&![$interface($interface(current),check)]} return

        destroyChildren $container
        $interface($index,edit) $container
        set interface(current) $index
    }

    proc done {} {
        variable interface

        foreach name [array names interface *,apply] {
            $interface($name)                                                                                    ;# apply new values
        }
        save
    }

    proc save {} {                                                                                                ;# save to rc file
        variable rcFileName

        set file [::open $rcFileName w] ;### detect errors and report failures to user ###
        puts -nonewline $file [saveList]
        close $file
    }

    proc saveList {} {
        variable interface

        set list {}
        foreach name [array names interface *,saveList] {
            set list [concat $list [$interface($name)]]
        }
        return $list
    }

    proc createMessage {path args} {                  ;# create a generic message widget with eventual option / value pairs argument
        message $path -width [winfo screenwidth .] -font $font::(mediumNormal) -justify left
        if {[llength $args]>0} {
            eval $path configure $args
        }
        return $path
    }

    namespace eval canvas {

        proc initialize {} {}

        proc edit {parentPath} {
            set message [preferences::createMessage $parentPath.message\
                -text "The canvas is the data viewers background area.\nYou can set its characteristics from this node down."
            ]
            pack $message -fill both -expand 1                                               ;# initially display help message above
        }

        proc check {} {
            return 1
        }

        proc apply {} {}

        proc saveList {} {
            return {}
        }

        namespace eval size {

            proc initialize {} {
                variable height $global::canvasHeight width $global::canvasWidth
            }

            proc edit {parentPath} {
                variable height
                variable width

                set message [preferences::createMessage $parentPath.message -text {Enter size (in pixels):}]
                grid $message -sticky nsew -row 0 -column 0 -columnspan 100                                    ;# occupy whole width
                grid rowconfigure $parentPath 0 -weight 1

                grid columnconfigure $parentPath 0 -weight 1

                set widthEntry [new spinEntry $parentPath -font $font::(mediumBold) -width 4 -list {640 800 1024 1280 1600}]
                set path $composite::($widthEntry,entry,path)
                $path configure -textvariable preferences::canvas::size::width
                # filter on positive integers and limit entry length
                wcb::callback $path before insert wcb::checkStrForNum {wcb::checkEntryLen 4}
                spinEntry::set $widthEntry $width
                grid [label $parentPath.width -font $font::(mediumBold) -text width:] -row 1 -column 1 -padx 2
                grid $widget::($widthEntry,path) -row 1 -column 2

                grid columnconfigure $parentPath 3 -weight 1

                set heightEntry [new spinEntry $parentPath -font $font::(mediumBold) -width 4 -list {400 480 600 768 1024 1280}]
                set path $composite::($heightEntry,entry,path)
                $path configure -textvariable preferences::canvas::size::height
                # filter on positive integers and limit entry length
                wcb::callback $path before insert wcb::checkStrForNum {wcb::checkEntryLen 4}
                spinEntry::set $heightEntry $height
                grid [label $parentPath.height -font $font::(mediumBold) -text height:] -row 1 -column 4 -padx 2
                grid $widget::($heightEntry,path) -row 1 -column 5

                grid [button $parentPath.apply -text Apply -command preferences::canvas::size::apply] -row 1 -column 6
                grid columnconfigure $parentPath 6 -weight 2

                set message [preferences::createMessage $parentPath.help\
                    -text "The size is immediately updated when clicking\non the Apply button."
                ]
                grid $message -sticky nsew -row 2 -column 0 -columnspan 100
                grid rowconfigure $parentPath 2 -weight 1

                bind $message <Destroy> "delete $widthEntry $heightEntry"                   ;# delete inner objects upon destruction
            }

            proc check {} {
                return 1
            }

            proc apply {} {
                variable height
                variable width

                if {![check]} return
                if {$height!=$global::canvasHeight} {        ;# update only when necessary in order not to activate trace needlessly
                    set global::canvasHeight $height                                                      ;# use entry current value
                }
                if {$width!=$global::canvasWidth} {
                    set global::canvasWidth $width
                }
            }

            proc saveList {} {
                return [list global::canvasHeight $global::canvasHeight global::canvasWidth $global::canvasWidth]
            }

        }

        namespace eval colors {

            proc initialize {} {
                variable background $global::canvasBackground
            }

            proc edit {parentPath} {
                variable background
                variable colorViewer

                set message [preferences::createMessage $parentPath.message -text {Background color:}]
                grid $message -sticky nsew -row 0 -column 0 -columnspan 100                                    ;# occupy whole width
                grid rowconfigure $parentPath 0 -weight 1

                grid columnconfigure $parentPath 0 -weight 1

                set colorViewer\
                    [button $parentPath.choose -text Choose... -command "preferences::canvas::colors::choose $parentPath"]
                updateColorViewer
                grid $colorViewer -row 1 -column 1
                grid [button $parentPath.apply -text Apply -command preferences::canvas::colors::apply] -row 1 -column 2

                grid columnconfigure $parentPath 1 -weight 1
                grid columnconfigure $parentPath 2 -weight 1
                grid columnconfigure $parentPath 3 -weight 1

                set message [preferences::createMessage $parentPath.help\
                    -text "The color is immediately updated when clicking\non the Apply button."
                ]
                grid $message -sticky nsew -row 2 -column 0 -columnspan 100
                grid rowconfigure $parentPath 2 -weight 1
            }

            proc check {} {
                return 1
            }

            proc apply {} {
                variable background

                if {![check]} return
                # update only when necessary in order not to activate trace needlessly
                if {[string compare $background $global::canvasBackground]!=0} {
                    set global::canvasBackground $background
                }
            }

            proc saveList {} {
                return [list global::canvasBackground $global::canvasBackground]
            }

            proc updateColorViewer {} {
                variable colorViewer
                variable background

                foreach {red green blue} [winfo rgb $colorViewer $background] {}
                if {($red+$green+$blue)>=(32768*3)} {                                                            ;# light background
                    $colorViewer configure -foreground black
                } else {                                                                                          ;# dark background
                    $colorViewer configure -foreground white
                }
                $colorViewer configure -background $background
            }

            proc choose {parentPath} {
                variable background

                set choice [tk_chooseColor -initialcolor $background -title {Choose color:} -parent $parentPath]
                if {[string length $choice]>0} {
                    set background $choice
                    updateColorViewer
                }
            }

        }

    }


    namespace eval viewers {

        proc initialize {} {
            variable numberOfSamples $global::graphNumberOfIntervals
        }

        proc edit {parentPath} {
            variable numberOfSamples

            set message [preferences::createMessage $parentPath.message -text {Enter number of samples for data graphs:}]
            grid $message -sticky nsew -row 0 -column 0 -columnspan 100
            grid rowconfigure $parentPath 0 -weight 1

            grid columnconfigure $parentPath 0 -weight 1

            set entry [new spinEntry $parentPath -font $font::(mediumBold) -width 4 -list {20 50 100 150 200 300 500 1000}]
            set path $composite::($entry,entry,path)
            $path configure -textvariable preferences::viewers::numberOfSamples
            # filter on positive integers and limit entry length
            wcb::callback $path before insert wcb::checkStrForNum {wcb::checkEntryLen 4}
            spinEntry::set $entry $numberOfSamples
            grid [label $parentPath.width -font $font::(mediumBold) -text samples:] -row 1 -column 1 -padx 2
            grid $widget::($entry,path) -row 1 -column 2

            grid columnconfigure $parentPath 3 -weight 1

            set message [preferences::createMessage $parentPath.help\
                -text {This is the number of samples (on the X axis)
for data graph viewers. The specified value will
not be used for existing graphs but for newly
created ones.}
            ]
            grid $message -sticky nsew -row 2 -column 0 -columnspan 100
            grid rowconfigure $parentPath 2 -weight 1

            bind $message <Destroy> "delete $entry"                                         ;# delete inner objects upon destruction
        }

        proc check {} {
            return 1
        }

        proc apply {} {
            variable numberOfSamples

            if {![check]} return
            set global::graphNumberOfIntervals $numberOfSamples                                           ;# use entry current value
        }

        proc saveList {} {
            return [list global::graphNumberOfIntervals $global::graphNumberOfIntervals]
        }

    }

}
