# copyright (C) 1997-1999 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: cpustats.tcl,v 1.20 1999/07/23 21:26:40 jfontain Exp $}


package provide cpustats 1.5
package require network 1.3

namespace eval cpustats {

    # use an empty hidden column as index since there is only a single row
    array set data {
        updates 0
        0,label {} 0,type ascii 0,message {} 0,0 {}
        1,label user 1,type real 1,message {percentage spent in user mode}
        2,label system 2,type real 2,message {percentage spent in system mode}
        3,label nice 3,type real 3,message {percentage spent in nice mode}
        4,label idle 4,type real 4,message {percentage spent in idle mode}
        columns 5
        helpText {
This is a view of processor activity for the system as a whole.

Module options:

-r (--remote) [user@]host: remote monitoring using "user" as logname on remote host "host" (rsh facilities must be properly setup). If user is not specified, current user is used as logname on remote host. The module title is set to "cpustats(host)".

Examples:

$ moodss ps -r jdoe@foo.bar.com
$ moodss ps --remote foo.bar.com
        }
        views {
            {visibleColumns {1 2 3 4} sort {1 increasing}}
        }
        switches {-r 1 --remote 1}
    }

    proc initialize {optionsName} {
        upvar $optionsName options
        variable remote
        variable data
        variable statisticsFile

        set lookup [expr {![info exists options(-n)]&&![info exists options(--numeric)]}]            ;# host or network names lookup
        if {![catch {set locator $options(--remote)}]||![catch {set locator $options(-r)}]} {                   ;# remote monitoring
            set data(pollTimes) {20 10 30 60 120 300 600}                                ;# poll less often when remotely monitoring
            foreach {remote(user) remote(host)} [network::parseRemoteLocator $locator] {}
            network::checkRemoteOutputEmptiness $remote(host)
            set data(identifier) cpustats($remote(host))
            set file [open "| /usr/bin/rsh -nl $remote(user) $remote(host) cat /proc/stat"]
            fileevent $file readable {set ::cpustats::remote(busy) 0}
            vwait ::cpustats::remote(busy)
            if {[catch {close $file} message]} {
                puts stderr "on remote host $remote(host) as user $remote(user): $message"
                exit 1                                                                                        ;# detect errors early
            }
        } else {
            set data(pollTimes) {10 5 20 30 60 120 300 600}
            set statisticsFile [open /proc/stat]                                      ;# keep local file open for better performance
        }
    }

    array set last {user 0 nice 0 system 0 idle 0}                                                    ;# stored values for last poll

    proc update {} {                                               ;# gather cpu statistics (based on the proc man page information)
        variable remote
        variable statisticsFile
        variable last
        variable data

        if {[info exists remote]} {
            if {![info exists statisticsFile]} {                           ;# start data gathering process in a non blocking fashion
                if {$remote(busy)} return                                           ;# core invocation while waiting for remote data
                set remote(busy) 1
                set file [open "| /usr/bin/rsh -nl $remote(user) $remote(host) cat /proc/stat"]
                # do not hang GUI, allow other modules updates
                fileevent $file readable "set ::cpustats::statisticsFile $file; ::cpustats::update"
                return                                                                                       ;# wait for remote data
            }                                                                                 ;# else continue below to process data
            gets $statisticsFile line
            close $statisticsFile                                                   ;# closing is necessary since seek does not work
            unset statisticsFile
            set remote(busy) 0
        } else {
            seek $statisticsFile 0                                                                  ;# rewind before retrieving data
            gets $statisticsFile line
        }
        scan $line {cpu %d %d %d %d} user nice system idle                                              ;# in hundredths of a second
        foreach item {user nice system idle} {         ;# calculate statistics during the last poll period except for the first pass
            set current [set $item]
            set $item [expr {$current-$last($item)}]
            set last($item) $current
        }
        set multiplier [expr {100.0/($user+$nice+$system+$idle)}]
        array set data "
            0,1 [format %.1f [expr {$multiplier*$user}]] 0,2 [format %.1f [expr {$multiplier*$system}]]
            0,3 [format %.1f [expr {$multiplier*$nice}]] 0,4 [format %.1f [expr {$multiplier*$idle}]]
        "
        incr data(updates)
    }
}
