# copyright (C) 1997-1999 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: mounts.tcl,v 1.9 1999/08/01 12:59:55 jfontain Exp $}


package provide mounts 1.5
package require filesystem
package require network 1.3

namespace eval mounts {

    array set data {
        updates 0
        0,label directory 0,type ascii 0,message {mount point}
        1,label filesystem 1,type dictionary 1,message {filesystem device}
        2,label type 2,type ascii 2,message {filesystem type}
        3,label size 3,type integer 3,message {total size in KiloBytes}
        4,label used 4,type integer 4,message {used user space in KiloBytes}
        5,label available 5,type integer 5,message {available user space in KiloBytes}
        6,label capacity 6,type integer 6,message {used user space in percent}
        sort {0 increasing}
        switches {-r 1 --remote 1}
    }
    set file [open [file join $::packageDirectory(mounts) mounts.htm]]
    set data(helpText) [read $file]                                                           ;# initialize HTML help data from file
    close $file

    proc initialize {optionsName} {
        upvar $optionsName options
        variable remote
        variable data

        if {![catch {set locator $options(--remote)}]||![catch {set locator $options(-r)}]} {                   ;# remote monitoring
            set data(pollTimes) {30 10 20 60 120 300 600}                                ;# poll less often when remotely monitoring
            foreach {remote(user) remote(host)} [network::parseRemoteLocator $locator] {}
            network::checkRemoteOutputEmptiness $remote(host)
            set data(identifier) mounts($remote(host))
            set file [open "| /usr/bin/rsh -nl $remote(user) $remote(host) cat /proc/version"]
            fileevent $file readable {set ::mounts::remote(busy) 0}
            vwait ::mounts::remote(busy)                                                               ;# do not hang user interface
        } else {
            set data(pollTimes) {20 5 10 30 60 120 300 600}
            set file [open /proc/version]
        }
        regexp {^[0-9\.]+} [lindex [gets $file] 2] version                         ;# ignore extra characters, such as in 2.2.0-pre1
        close $file
        if {[package vcompare $version 2]<0} {                                                               ;# check kernel version
            puts stderr {mounts module needs at least a version 2 kernel}
            exit 1
        }
    }

    set nextIndex 0

    proc update {} {
        variable remote

        if {[info exists remote]&&!$remote(busy)} {
            remoteUpdate
        } else {
            localUpdate
        }
    }

    proc localUpdate {} {
        variable data

        set file [open /etc/mtab]                        ;# must be opened each poll otherwise unmounted filesystems never disappear
        while {[gets $file line]>=0} {
            foreach {device directory type flags dump order} $line {}
            if {[string compare $device none]==0} continue                                 ;# ignore irrelevant entries such as proc
            foreach {size total available used} [filesystem::statistics $directory] {}
            set ratio [expr {$size/1024.0}]                    ;# take filesystem block size into account to get values in kilobytes
            updateEntryData $directory $device $type [expr {round($total*$ratio)}] [expr {round($used*$ratio)}]\
                [expr {round($available*$ratio)}] [expr {round(($used*100.0)/($used+$available))}]                      ;# used in %
            set current($directory) {}
        }
        close $file
        cleanupEntriesData current
        incr data(updates)
    }

    proc remoteUpdate {} {
        variable remote

        set file [open "| /usr/bin/rsh -nl $remote(user) $remote(host) /bin/df -T"]
        fileevent $file readable "mounts::remoteUpdated $file"               ;# do not hang user interface and other modules updates
    }

    proc remoteUpdated {file} {
        variable remote
        variable data

        gets $file                                                                                               ;# skip header line
        while {[gets $file line]>=0} {
            foreach {device type size used available capacity directory} $line {}
            updateEntryData $directory $device $type $size $used $available [string trimright $capacity %]
            set current($directory) {}
        }
        close $file                                                                ;### errors and timeouts could be caught here ###
        cleanupEntriesData current
        set remote(busy) 0
        incr data(updates)
    }

    proc updateEntryData {directory filesystem type size used available capacity} {
        variable index
        variable nextIndex
        variable data

        if {[catch {set index($directory)} row]} {                                                                      ;# new entry
            set row [set index($directory) $nextIndex]
            incr nextIndex
            set data($row,0) $directory                                                                    ;# initialize static data
            set data($row,1) $filesystem
            set data($row,2) $type
        }
        set data($row,3) $size
        set data($row,4) $used
        set data($row,5) $available
        set data($row,6) $capacity
    }

    proc cleanupEntriesData {currentDirectoriesName} {
        upvar $currentDirectoriesName current
        variable index
        variable data

        foreach directory [array names index] {                                                       ;# cleanup disappeared entries
            if {[info exists current($directory)]} continue
            set row $index($directory)
            unset index($directory) data($row,0) data($row,1) data($row,2) data($row,3) data($row,4) data($row,5) data($row,6)
        }
    }

}
