# $Id: graph.tcl,v 1.8 2000/07/06 02:45:24 lduperva Exp $

# This module contains all the procedures needed to maintain and display
# intraday graph data


namespace eval tickergraph {
	namespace export showGraph addData createGraph tickFormat

	option add *Pixels 1
}

# Show daily graph
#
# symbol: symbol info to display
proc ::tickergraph::showGraph {symbol} {
	if {[string length [info procs ::tickergraph::${symbol}::showGraph]] == 0} {
		createGraph $symbol
	}
	${symbol}::showGraph
}	

# Add data to a graph
#
# symbol: Symbol of data to add
# time: time at which data was acquired
# value: well, the data's value, what else?
proc ::tickergraph::addData {symbol time value} {
	::tickergraph::${symbol}::addData $time $value
}

# Saves all data
proc ::tickergraph::saveAllData {} {
	foreach ns [namespace children] {
		${ns}::saveData 1
	}
}

# Save data for a symbol
#
# symbol: well....
proc ::tickergraph::saveData {symbol} {
	::tickergraph::${symbol}::saveData
}

# Clear data associated with all graphs
proc ::tickergraph::clearData {} {
	foreach ns [namespace children] {
		${ns}::clearData
	}
}

# Change configuration of all graphs
proc ::tickergraph::graphConfig {} {
	foreach ns [namespace children] {
		${ns}::graphConfig
	}
}

# Format time values for graph's time axis
#
# widget: argument added by graph command
# value:  argument added by graph command
proc ::tickergraph::tickFormat {widget value} {
	set value [expr {$value+[clock scan $::options(graphStart)]}]
	clock format $value -format "%H:%M"
}

# Create a graph for a symbol. This command will create a new namespace for
# the symbol's graph data. In this namespace, new procs are created to
# manipulate and display the graph data for the symbol.
#
# symbol: symbol to use
proc ::tickergraph::createGraph {symbol} {
	namespace eval ::tickergraph::$symbol {
		# data points
		variable graphXData
		variable graphYData
		# window containing the graph
		variable graphWindow
		# Da graph
		variable graphWidget
		variable startTime [clock scan $::options(graphStart)]
		variable endTime [clock scan $::options(graphStop)]
		variable graphHeight $::options(graphHeight)
		variable graphWidth $::options(graphWidth)
		
		namespace export addData showGraph

		set dataFile [file join $::options(stockwatcherDir) \
			[namespace tail [namespace current]].data ]

		if {[file exists $dataFile]} {
			source $dataFile
		} else {
			::blt::vector graphXData
			::blt::vector graphYData
		}
		set graphWindow ""
		set graphWidget ""
	} 
		
	# Initialis the graph for $symbol
	proc ::tickergraph::${symbol}::init {} {
		variable graphXData
		variable graphYData
		variable graphWidget
		variable graphWindow
		variable graphWidth
		variable graphHeight
		
		set symbol [namespace tail [namespace current]]
		regsub -all {\.} $symbol "_" wsymbol
		set graphWindow [Dialog .tickergraph$wsymbol -title "[mc "$symbol Variations"]" \
			-modal none -default 0]
		set subf [$graphWindow getframe] 
		set graphWidget [::blt::graph $subf.g -title "[mc "$symbol Variations"]" \
			-width $graphWidth -height $graphHeight -bg white]

		$graphWidget xaxis configure -min 0 -stepsize 1800 \
			-max [expr {[clock scan $::options(graphStop)] - [clock scan $::options(graphStart)]}] \
			-command ::tickergraph::tickFormat -rotate 90 
		$graphWidget element create line -xdata graphXData -ydata graphYData 
		$graphWidget legend configure -hide true
		pack $graphWidget -expand true -fill both
		$graphWindow add -text Close \
			-command [namespace code closeWindow] 
		$graphWindow add -text Clear \
			-command [namespace code clearData] 
	}

	# Configure the graph appearance for $symbol
	proc ::tickergraph::${symbol}::graphConfig {} {
		global options
		variable graphWidget
		variable graphWidth
		variable graphHeight

		set graphWidth $options(graphWidth)
		set graphHeight $options(graphHeight)

		if {[string length $graphWidget] > 0} {
			$graphWidget configure -width $graphWidth -height $graphHeight
			$graphWidget xaxis configure -min 0 -stepsize 1800 \
			-max [expr {[clock scan $options(graphStop)] - [clock scan $options(graphStart)]}] \
			-command ::tickergraph::tickFormat -rotate 90 
		}
	}

	# Clear graph data for $symbol
	proc ::tickergraph::${symbol}::clearData {} {
		graphXData set [list]
		graphYData set [list]
		file delete "$::options(stockwatcherDir)/[namespace tail [namespace current]].data"
	}

	# Close the graph window
	proc ::tickergraph::${symbol}::closeWindow {} {
		variable graphWindow
		
		$graphWindow withdraw
	}

	# Add new data for the graph
	#
	# time: Time at which data was added
	# value: data value
	proc ::tickergraph::${symbol}::addData {time value} {
		variable graphXData
		variable graphYData
		variable graphWidget
		# Recalculates every time to take care of date changes. Wastes CPU
		# cycles for nothing, though
		variable startTime [clock scan $::options(graphStart)]
		variable endTime [clock scan $::options(graphStop)]

		# Don't add data after hours
		if {$time > $startTime && $time < $endTime} {
			graphXData append [expr {$time-$startTime}]
			graphYData append $value
		} elseif {[expr {$time-$startTime}] < 0 && [expr {$time-$startTime}] > - [expr {$::options(graphReset)*60}]} {
			# Before the bell rings, reset all data
			clearData
			if {[string length $graphWidget]} {
				$graphWidget xaxis configure -min 0 -stepsize 1800 \
					-max [expr {$endTime - $startTime}]
			}
		}
	}

	# Save graph data
	#
	# force: if set to 1, will force saving of data. Otherwise, data is saved
	# at regular intervals only.
	proc ::tickergraph::${symbol}::saveData {{force 0}} {
		variable graphXData
		variable graphYData
		set symbol [namespace tail [namespace current]]
		if {$force == 1} {
			set fp [open [file join $::options(stockwatcherDir) $symbol.data] w]
			puts $fp "[list ::blt::vector ::tickergraph::${symbol}::graphXData]"
			puts $fp "[list ::blt::vector ::tickergraph::${symbol}::graphYData]"
			if {[graphXData length] >  0} {
			puts $fp "[list ::tickergraph::${symbol}::graphXData append [graphXData range 0 end]]"
			puts $fp "[list ::tickergraph::${symbol}::graphYData append [graphYData range 0 end]]"
			}
			close $fp
		} elseif {([graphXData length] > 0) &&[expr {[graphXData length] % $::options(graphSave)}] == 0} {
			set fp [open [file join $::options(stockwatcherDir) $symbol.data] w]
			puts $fp "[list ::blt::vector ::tickergraph::${symbol}::graphXData]"
			puts $fp "[list ::tickergraph::${symbol}::graphXData append [graphXData range 0 end]]"
			puts $fp "[list ::blt::vector ::tickergraph::${symbol}::graphYData]"
			puts $fp "[list ::tickergraph::${symbol}::graphYData append [graphYData range 0 end]]"
			close $fp
		}
	}

	# Show the graph
	proc ::tickergraph::${symbol}::showGraph {} {
		variable graphWindow
		
		if {[string length $graphWindow] == 0} {
			init
		}
		$graphWindow draw
	
	}
}
