class
	--
	-- A track identifier for a track in a song
	-- Might start index at 0 or 1 depending on compatibility mode
	--
	TRACK_NUMBER

creation
	make

feature -- Initialization
	make is
			-- create a way of identifying song tracks
		do
			!!error_description.copy("No track selected")
		end

feature -- Access
	context: COMMAND_CONTEXT
	song: SONG
	track_identifier: STRING
	track_number: INTEGER
	is_valid: BOOLEAN
	error_description: STRING

feature -- Element Change
	set_context(c: like context) is
			-- set command context
		do
			context := c
		ensure
			context_set: context = c
		end

	set_song(s: like song) is
			-- set song
		do
			song := s
		ensure
			song_set: song = s
		end

	set_track_identifier(t: STRING) is
			-- select track from tclmidi command identifier
		require
			valid_context: context /= Void
		do
			track_identifier := t
			if not track_identifier.is_integer then
				is_valid := False
				!!error_description.copy("Invalid track id '")
				error_description.append(track_identifier)
				error_description.append("'")
			else
				is_valid := True
				track_number := track_identifier.to_integer
				if off_by_one then
					-- earlier releases counted from 0, not 1
					track_number := track_number + 1
				end
			end
		end

	set_track_number(t: INTEGER) is
			-- select track by number
		require
			valid_context: context /= Void
		do
			track_number := t
			if off_by_one then
				!!track_identifier.copy((track_number - 1).to_string)
			else
				!!track_identifier.copy(track_number.to_string)
			end
		end

	check_existance is
			-- does track exist in the song
		require
			valid_song: song /= Void
		do
			if track_number >= 1 and then track_number <= song.number_of_tracks
				then
				is_valid := True
			else
				!!error_description.copy("track number '")
				error_description.append(track_identifier)
				error_description.append("' does not exist in the song")
				is_valid := False
			end
		end

	check_track_identifier(t: STRING) is
			-- convert and verify track id
		require
			valid_context: context /= Void
			valid_song: song /= Void
		do
			set_track_identifier(t)
			if is_valid then
				check_existance
			end
		end

feature {NONE} -- Implementation
	off_by_one: BOOLEAN is
			-- Is old style where counting began at 0, not 1?
		require
			valid_context: context /= Void
		do
			if context.compatibility_mode /= Void then
				if context.compatibility_mode.has_prefix("3.") or else
					context.compatibility_mode.is_equal("3") or else
					context.compatibility_mode.has_prefix("2.") or else
					context.compatibility_mode.is_equal("2") or else
					context.compatibility_mode.has_prefix("1.") or else
					context.compatibility_mode.is_equal("1") then
						Result := True
				end
			end
		end

end -- TRACK_NUMBER
