/* $Header: /cvsroot/osrs/libtiff/libtiff/tiffio.h,v 1.3 1999/09/08 12:21:13 warmerda Exp $ */

/*
 * Copyright (c) 1988-1997 Sam Leffler
 * Copyright (c) 1991-1997 Silicon Graphics, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the names of
 * Sam Leffler and Silicon Graphics may not be used in any advertising or
 * publicity relating to the software without the specific, prior written
 * permission of Sam Leffler and Silicon Graphics.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 * IN NO EVENT SHALL SAM LEFFLER OR SILICON GRAPHICS BE LIABLE FOR
 * ANY SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND,
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF 
 * LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 */

#ifndef _TIFFIO_
#define	_TIFFIO_

/*
 * TIFF I/O Library Definitions.
 */
#include "tiff.h"

/*
 * This define can be used in code that requires
 * compilation-related definitions specific to a
 * version or versions of the library.  Runtime
 * version checking should be done based on the
 * string returned by TIFFGetVersion.
 */
#define	TIFFLIB_VERSION	19970127	/* January 27, 1997 */

/*
 * TIFF is defined as an incomplete type to hide the
 * library's internal data structures from clients.
 */
typedef	struct tiff TIFF;

/*
 * The following typedefs define the intrinsic size of
 * data types used in the *exported* interfaces.  These
 * definitions depend on the proper definition of types
 * in tiff.h.  Note also that the varargs interface used
 * to pass tag types and values uses the types defined in
 * tiff.h directly.
 *
 * NB: ttag_t is unsigned int and not unsigned short because
 *     ANSI C requires that the type before the ellipsis be a
 *     promoted type (i.e. one of int, unsigned int, pointer,
 *     or double) and because we defined pseudo-tags that are
 *     outside the range of legal Aldus-assigned tags.
 * NB: tsize_t is int32 and not uint32 because some functions
 *     return -1.
 * NB: toff_t is not off_t for many reasons; TIFFs max out at
 *     32-bit file offsets being the most important
 */
typedef	uint32 ttag_t;		/* directory tag */
typedef	uint16 tdir_t;		/* directory index */
typedef	uint16 tsample_t;	/* sample number */
typedef	uint32 tstrip_t;	/* strip number */
typedef uint32 ttile_t;		/* tile number */
typedef	int32 tsize_t;		/* i/o size in bytes */
typedef	void* tdata_t;		/* image data ref */
typedef	int32 toff_t;		/* file offset */

#if !defined(__WIN32__) && (defined(_WIN32) || defined(WIN32))
#define __WIN32__
#endif
#if defined(_WINDOWS) || defined(__WIN32__) || defined(_Windows)
#include <windows.h>
#ifdef __WIN32__
DECLARE_HANDLE(thandle_t);	/* Win32 file handle */
#else
typedef	HFILE thandle_t;	/* client data handle */
#endif
#else
typedef	void* thandle_t;	/* client data handle */
#endif

#ifndef NULL
#define	NULL	0
#endif

/*
 * Flags to pass to TIFFPrintDirectory to control
 * printing of data structures that are potentially
 * very large.   Bit-or these flags to enable printing
 * multiple items.
 */
#define	TIFFPRINT_NONE		0x0		/* no extra info */
#define	TIFFPRINT_STRIPS	0x1		/* strips/tiles info */
#define	TIFFPRINT_CURVES	0x2		/* color/gray response curves */
#define	TIFFPRINT_COLORMAP	0x4		/* colormap */
#define	TIFFPRINT_JPEGQTABLES	0x100		/* JPEG Q matrices */
#define	TIFFPRINT_JPEGACTABLES	0x200		/* JPEG AC tables */
#define	TIFFPRINT_JPEGDCTABLES	0x200		/* JPEG DC tables */

/*
 * RGBA-style image support.
 */
typedef	unsigned char TIFFRGBValue;		/* 8-bit samples */
typedef struct _TIFFRGBAImage TIFFRGBAImage;
/*
 * The image reading and conversion routines invoke
 * ``put routines'' to copy/image/whatever tiles of
 * raw image data.  A default set of routines are 
 * provided to convert/copy raw image data to 8-bit
 * packed ABGR format rasters.  Applications can supply
 * alternate routines that unpack the data into a
 * different format or, for example, unpack the data
 * and draw the unpacked raster on the display.
 */
typedef void (*tileContigRoutine)
    (TIFFRGBAImage*, uint32*, uint32, uint32, uint32, uint32, int32, int32,
	unsigned char*);
typedef void (*tileSeparateRoutine)
    (TIFFRGBAImage*, uint32*, uint32, uint32, uint32, uint32, int32, int32,
	unsigned char*, unsigned char*, unsigned char*, unsigned char*);
/*
 * RGBA-reader state.
 */
typedef struct {				/* YCbCr->RGB support */
	TIFFRGBValue* clamptab;			/* range clamping table */
	int*	Cr_r_tab;
	int*	Cb_b_tab;
	int32*	Cr_g_tab;
	int32*	Cb_g_tab;
	float	coeffs[3];			/* cached for repeated use */
} TIFFYCbCrToRGB;

struct _TIFFRGBAImage {
	TIFF*	tif;				/* image handle */
	int	stoponerr;			/* stop on read error */
	int	isContig;			/* data is packed/separate */
	int	alpha;				/* type of alpha data present */
	uint32	width;				/* image width */
	uint32	height;				/* image height */
	uint16	bitspersample;			/* image bits/sample */
	uint16	samplesperpixel;		/* image samples/pixel */
	uint16	orientation;			/* image orientation */
	uint16	photometric;			/* image photometric interp */
	uint16*	redcmap;			/* colormap pallete */
	uint16*	greencmap;
	uint16*	bluecmap;
						/* get image data routine */
	int	(*get)(TIFFRGBAImage*, uint32*, uint32, uint32);
	union {
	    void (*any)(TIFFRGBAImage*);
	    tileContigRoutine	contig;
	    tileSeparateRoutine	separate;
	} put;					/* put decoded strip/tile */
	TIFFRGBValue* Map;			/* sample mapping array */
	uint32** BWmap;				/* black&white map */
	uint32** PALmap;			/* palette image map */
	TIFFYCbCrToRGB* ycbcr;			/* YCbCr conversion state */

        int	row_offset;
        int     col_offset;
};

/*
 * Macros for extracting components from the
 * packed ABGR form returned by TIFFReadRGBAImage.
 */
#define	TIFFGetR(abgr)	((abgr) & 0xff)
#define	TIFFGetG(abgr)	(((abgr) >> 8) & 0xff)
#define	TIFFGetB(abgr)	(((abgr) >> 16) & 0xff)
#define	TIFFGetA(abgr)	(((abgr) >> 24) & 0xff)

/*
 * A CODEC is a software package that implements decoding,
 * encoding, or decoding+encoding of a compression algorithm.
 * The library provides a collection of builtin codecs.
 * More codecs may be registered through calls to the library
 * and/or the builtin implementations may be overridden.
 */
typedef	int (*TIFFInitMethod)(TIFF*, int);
typedef struct {
	char*		name;
	uint16		scheme;
	TIFFInitMethod	init;
} TIFFCodec;

#include <stdio.h>
#include <stdarg.h>

#if defined(__cplusplus)
extern "C" {
#endif
typedef	void (*TIFFErrorHandler)(const char*, const char*, va_list);
typedef	tsize_t (*TIFFReadWriteProc)(thandle_t, tdata_t, tsize_t);
typedef	toff_t (*TIFFSeekProc)(thandle_t, toff_t, int);
typedef	int (*TIFFCloseProc)(thandle_t);
typedef	toff_t (*TIFFSizeProc)(thandle_t);
typedef	int (*TIFFMapFileProc)(thandle_t, tdata_t*, toff_t*);
typedef	void (*TIFFUnmapFileProc)(thandle_t, tdata_t, toff_t);
typedef	void (*TIFFExtendProc)(TIFF*); 

extern	EXPORT(const char*,TIFFGetVersion) (void);

extern	EXPORT(const TIFFCodec*,TIFFFindCODEC) (uint16);
extern	EXPORT(TIFFCodec*,TIFFRegisterCODEC) (uint16, const char*, TIFFInitMethod);
extern	EXPORT(void,TIFFUnRegisterCODEC) (TIFFCodec*);

extern	EXPORT(tdata_t,_TIFFmalloc) (tsize_t);
extern	EXPORT(tdata_t,_TIFFrealloc) (tdata_t, tsize_t);
extern	EXPORT(void,_TIFFmemset) (tdata_t, int, tsize_t);
extern	EXPORT(void,_TIFFmemcpy) (tdata_t, const tdata_t, tsize_t);
extern	EXPORT(int,_TIFFmemcmp) (const tdata_t, const tdata_t, tsize_t);
extern	EXPORT(void,_TIFFfree) (tdata_t);

extern	EXPORT(void,TIFFClose) (TIFF*);
extern	EXPORT(int,TIFFFlush) (TIFF*);
extern	EXPORT(int,TIFFFlushData) (TIFF*);
extern	EXPORT(int,TIFFGetField) (TIFF*, ttag_t, ...);
extern	EXPORT(int,TIFFVGetField) (TIFF*, ttag_t, va_list);
extern	EXPORT(int,TIFFGetFieldDefaulted) (TIFF*, ttag_t, ...);
extern	EXPORT(int,TIFFVGetFieldDefaulted) (TIFF*, ttag_t, va_list);
extern	EXPORT(int,TIFFReadDirectory) (TIFF*);
extern	EXPORT(tsize_t,TIFFScanlineSize) (TIFF*);
extern	EXPORT(tsize_t,TIFFRasterScanlineSize) (TIFF*);
extern	EXPORT(tsize_t,TIFFStripSize) (TIFF*);
extern	EXPORT(tsize_t,TIFFVStripSize) (TIFF*, uint32);
extern	EXPORT(tsize_t,TIFFTileRowSize) (TIFF*);
extern	EXPORT(tsize_t,TIFFTileSize) (TIFF*);
extern	EXPORT(tsize_t,TIFFVTileSize) (TIFF*, uint32);
extern	EXPORT(uint32,TIFFDefaultStripSize) (TIFF*, uint32);
extern	EXPORT(void,TIFFDefaultTileSize) (TIFF*, uint32*, uint32*);
extern	EXPORT(int,TIFFFileno) (TIFF*);
extern	EXPORT(int,TIFFGetMode) (TIFF*);
extern	EXPORT(int,TIFFIsTiled) (TIFF*);
extern	EXPORT(int,TIFFIsByteSwapped) (TIFF*);
extern	EXPORT(int,TIFFIsUpSampled) (TIFF*);
extern	EXPORT(int,TIFFIsMSB2LSB) (TIFF*);
extern	EXPORT(uint32,TIFFCurrentRow) (TIFF*);
extern	EXPORT(tdir_t,TIFFCurrentDirectory) (TIFF*);
extern	EXPORT(tdir_t,TIFFNumberOfDirectories) (TIFF*);
extern	EXPORT(uint32,TIFFCurrentDirOffset) (TIFF*);
extern	EXPORT(tstrip_t,TIFFCurrentStrip) (TIFF*);
extern	EXPORT(ttile_t,TIFFCurrentTile) (TIFF*);
extern	EXPORT(int,TIFFReadBufferSetup) (TIFF*, tdata_t, tsize_t);
extern	EXPORT(int,TIFFWriteBufferSetup) (TIFF*, tdata_t, tsize_t);
extern	EXPORT(int,TIFFLastDirectory) (TIFF*);
extern	EXPORT(int,TIFFSetDirectory) (TIFF*, tdir_t);
extern	EXPORT(int,TIFFSetSubDirectory) (TIFF*, uint32);
extern	EXPORT(int,TIFFUnlinkDirectory) (TIFF*, tdir_t);
extern	EXPORT(int,TIFFSetField) (TIFF*, ttag_t, ...);
extern	EXPORT(int,TIFFVSetField) (TIFF*, ttag_t, va_list);
extern	EXPORT(int,TIFFWriteDirectory) (TIFF *);
extern	EXPORT(int,TIFFReassignTagToIgnore) (enum TIFFIgnoreSense, int);

#if defined(c_plusplus) || defined(__cplusplus)
extern	EXPORT(void,TIFFPrintDirectory) (TIFF*, FILE*, long = 0);
extern	EXPORT(int,TIFFReadScanline) (TIFF*, tdata_t, uint32, tsample_t = 0);
extern	EXPORT(int,TIFFWriteScanline) (TIFF*, tdata_t, uint32, tsample_t = 0);
extern	EXPORT(int,TIFFReadRGBAImage) (TIFF*, uint32, uint32, uint32*, int = 0);
#else
extern	EXPORT(void,TIFFPrintDirectory) (TIFF*, FILE*, long);
extern	EXPORT(int,TIFFReadScanline) (TIFF*, tdata_t, uint32, tsample_t);
extern	EXPORT(int,TIFFWriteScanline) (TIFF*, tdata_t, uint32, tsample_t);
extern	EXPORT(int,TIFFReadRGBAImage) (TIFF*, uint32, uint32, uint32*, int);
#endif

extern	EXPORT(int,TIFFReadRGBAStrip) (TIFF*, tstrip_t, uint32 * );
extern	EXPORT(int,TIFFReadRGBATile) (TIFF*, uint32, uint32, uint32 * );
extern	EXPORT(int,TIFFRGBAImageOK) (TIFF*, char [1024]);
extern	EXPORT(int,TIFFRGBAImageBegin) (TIFFRGBAImage*, TIFF*, int, char [1024]);
extern	EXPORT(int,TIFFRGBAImageGet) (TIFFRGBAImage*, uint32*, uint32, uint32);
extern	EXPORT(void,TIFFRGBAImageEnd) (TIFFRGBAImage*);
extern	EXPORT(TIFF*,TIFFOpen) (const char*, const char*);
extern	EXPORT(TIFF*,TIFFFdOpen) (int, const char*, const char*);
extern	EXPORT(TIFF*,TIFFClientOpen) (const char*, const char*,
	    thandle_t,
	    TIFFReadWriteProc, TIFFReadWriteProc,
	    TIFFSeekProc, TIFFCloseProc,
	    TIFFSizeProc,
	    TIFFMapFileProc, TIFFUnmapFileProc);
extern	EXPORT(const char*,TIFFFileName) (TIFF*);
extern	EXPORT(void,TIFFError) (const char*, const char*, ...);
extern	EXPORT(void,TIFFWarning) (const char*, const char*, ...);
extern	EXPORT(TIFFErrorHandler,TIFFSetErrorHandler) (TIFFErrorHandler);
extern	EXPORT(TIFFErrorHandler,TIFFSetWarningHandler) (TIFFErrorHandler);
extern	EXPORT(TIFFExtendProc,TIFFSetTagExtender) (TIFFExtendProc);
extern	EXPORT(ttile_t,TIFFComputeTile) (TIFF*, uint32, uint32, uint32, tsample_t);
extern	EXPORT(int,TIFFCheckTile) (TIFF*, uint32, uint32, uint32, tsample_t);
extern	EXPORT(ttile_t,TIFFNumberOfTiles) (TIFF*);
extern	EXPORT(tsize_t,TIFFReadTile) (TIFF*,
	    tdata_t, uint32, uint32, uint32, tsample_t);
extern	EXPORT(tsize_t,TIFFWriteTile) (TIFF*,
	    tdata_t, uint32, uint32, uint32, tsample_t);
extern	EXPORT(tstrip_t,TIFFComputeStrip) (TIFF*, uint32, tsample_t);
extern	EXPORT(tstrip_t,TIFFNumberOfStrips) (TIFF*);
extern	EXPORT(tsize_t,TIFFReadEncodedStrip) (TIFF*, tstrip_t, tdata_t, tsize_t);
extern	EXPORT(tsize_t,TIFFReadRawStrip) (TIFF*, tstrip_t, tdata_t, tsize_t);
extern	EXPORT(tsize_t,TIFFReadEncodedTile) (TIFF*, ttile_t, tdata_t, tsize_t);
extern	EXPORT(tsize_t,TIFFReadRawTile) (TIFF*, ttile_t, tdata_t, tsize_t);
extern	EXPORT(tsize_t,TIFFWriteEncodedStrip) (TIFF*, tstrip_t, tdata_t, tsize_t);
extern	EXPORT(tsize_t,TIFFWriteRawStrip) (TIFF*, tstrip_t, tdata_t, tsize_t);
extern	EXPORT(tsize_t,TIFFWriteEncodedTile) (TIFF*, ttile_t, tdata_t, tsize_t);
extern	EXPORT(tsize_t,TIFFWriteRawTile) (TIFF*, ttile_t, tdata_t, tsize_t);
extern	EXPORT(void,TIFFSetWriteOffset) (TIFF*, toff_t);
extern	EXPORT(void,TIFFSwabShort) (uint16*);
extern	EXPORT(void,TIFFSwabLong) (uint32*);
extern	EXPORT(void,TIFFSwabDouble) (double*);
extern	EXPORT(void,TIFFSwabArrayOfShort) (uint16*, unsigned long);
extern	EXPORT(void,TIFFSwabArrayOfLong) (uint32*, unsigned long);
extern	EXPORT(void,TIFFSwabArrayOfDouble) (double*, unsigned long);
extern	EXPORT(void,TIFFReverseBits) (unsigned char *, unsigned long);
extern	EXPORT(const unsigned char*,TIFFGetBitRevTable) (int);
#if defined(__cplusplus)
}
#endif
#endif /* _TIFFIO_ */
