////////////////////////////////////////////////////////////////////////
//
// Copyright (C) 2011-2026 The Octave Project Developers
//
// See the file COPYRIGHT.md in the top-level directory of this
// distribution or <https://octave.org/copyright/>.
//
// This file is part of Octave.
//
// Octave is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Octave is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Octave; see the file COPYING.  If not, see
// <https://www.gnu.org/licenses/>.
//
////////////////////////////////////////////////////////////////////////

#if defined (HAVE_CONFIG_H)
#  include "config.h"
#endif

#include <QLayout>
#include <QTextBrowser>
#include <QThread>

#include "community-news.h"
#include "gui-utils.h"
#include "gui-preferences-global.h"
#include "gui-preferences-dw.h"
#include "gui-preferences-nr.h"
#include "gui-settings.h"
#include "news-reader.h"

OCTAVE_BEGIN_NAMESPACE(octave)

community_news::community_news (int serial)
  : QWidget (nullptr), m_browser (nullptr)
{
  construct ("https://octave.org", "community-news.html", serial);
}

community_news::community_news (QWidget *parent, const QString& base_url,
                                const QString& page, int serial)
  : QWidget (parent), m_browser (nullptr)
{
  construct (base_url, page, serial);
}

void
community_news::construct (const QString& base_url, const QString& page,
                           int serial)
{
  m_browser = new QTextBrowser (this);

  m_browser->setObjectName ("OctaveNews");
  m_browser->setOpenExternalLinks (true);

  QVBoxLayout *vlayout = new QVBoxLayout;

  vlayout->addWidget (m_browser);

  setLayout (vlayout);
  setWindowTitle (tr ("Octave Community News"));

  int win_x, win_y;
  get_screen_geometry (win_x, win_y);

  resize (win_x/2, win_y/2);
  move ((win_x - width ())/2, (win_y - height ())/2);

  gui_settings settings;

  QString icon;
  QString icon_set = settings.string_value (dw_icon_set);
  if (icon_set != "NONE")
    // No extra icon for Community news, take the one of the release notes
    icon = dw_icon_set_names[icon_set] + "ReleaseWidget" + global_icon_extension;
  else
    icon = dw_icon_set_names[icon_set];

  setWindowIcon (QIcon (icon));

  // FIXME: This is a news reader preference, so shouldn't it be used
  // in the news_reader object?

  bool connect_to_web = settings.bool_value (nr_allow_connection);

  QThread *worker_thread = new QThread;

  news_reader *reader
    = new news_reader (base_url, page, serial, connect_to_web);

  reader->moveToThread (worker_thread);

  connect (reader, &news_reader::display_news_signal,
           this, &community_news::set_news);

  connect (worker_thread, &QThread::started,
           reader, &news_reader::process);

  connect (reader, &news_reader::finished, worker_thread, &QThread::quit);

  connect (reader, &news_reader::finished, reader, &news_reader::deleteLater);

  connect (worker_thread, &QThread::finished,
           worker_thread, &QThread::deleteLater);

  worker_thread->start ();
}

void
community_news::set_news (const QString& news)
{
  m_browser->setHtml (news);
}

void
community_news::display ()
{
  if (! isVisible ())
    show ();
  else if (isMinimized ())
    showNormal ();

  raise ();
  activateWindow ();
}

OCTAVE_END_NAMESPACE(octave)
