/*
 * Copyright (C) 2004 Zack Rusin <zack@kde.org>
 * Copyright (C) 2004-2025 Apple Inc. All rights reserved.
 * Copyright (C) 2007 Alexey Proskuryakov <ap@webkit.org>
 * Copyright (C) 2007 Nicholas Shanks <webkit@nickshanks.com>
 * Copyright (C) 2011 Sencha, Inc. All rights reserved.
 * Copyright (C) 2013 Adobe Systems Incorporated. All rights reserved.
 * Copyright (C) 2025 Samuel Weinig <sam@webkit.org>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include "CSSBorderImage.h"
#include "CSSBorderImageSliceValue.h"
#include "CSSCounterValue.h"
#include "CSSEasingFunctionValue.h"
#include "CSSFontFeatureValue.h"
#include "CSSFontValue.h"
#include "CSSFontVariationValue.h"
#include "CSSFunctionValue.h"
#include "CSSPathValue.h"
#include "CSSPrimitiveValue.h"
#include "CSSPrimitiveValueMappings.h"
#include "CSSProperty.h"
#include "CSSPropertyParserConsumer+Anchor.h"
#include "CSSQuadValue.h"
#include "CSSRatioValue.h"
#include "CSSRayValue.h"
#include "CSSRectValue.h"
#include "CSSReflectValue.h"
#include "CSSRegisteredCustomProperty.h"
#include "CSSScrollValue.h"
#include "CSSSerializationContext.h"
#include "CSSTransformListValue.h"
#include "CSSURLValue.h"
#include "CSSValueList.h"
#include "CSSValuePair.h"
#include "CSSValuePool.h"
#include "CSSViewValue.h"
#include "ContainerNodeInlines.h"
#include "FontCascade.h"
#include "FontSelectionValueInlines.h"
#include "HTMLFrameOwnerElement.h"
#include "PathOperation.h"
#include "PerspectiveTransformOperation.h"
#include "RenderBlock.h"
#include "RenderBoxInlines.h"
#include "RenderElementInlines.h"
#include "RenderGrid.h"
#include "RenderInline.h"
#include "RenderStyleInlines.h"
#include "ScrollTimeline.h"
#include "SkewTransformOperation.h"
#include "StyleClipPath.h"
#include "StyleColor.h"
#include "StyleColorScheme.h"
#include "StyleCornerShapeValue.h"
#include "StyleDynamicRangeLimit.h"
#include "StyleEasingFunction.h"
#include "StyleExtractorState.h"
#include "StyleFlexBasis.h"
#include "StyleInset.h"
#include "StyleMargin.h"
#include "StyleMaximumSize.h"
#include "StyleMinimumSize.h"
#include "StyleOffsetPath.h"
#include "StylePadding.h"
#include "StylePerspective.h"
#include "StylePreferredSize.h"
#include "StylePrimitiveKeyword+CSSValueCreation.h"
#include "StylePrimitiveNumericTypes+CSSValueCreation.h"
#include "StylePrimitiveNumericTypes+Conversions.h"
#include "StyleRotate.h"
#include "StyleScale.h"
#include "StyleScrollMargin.h"
#include "StyleScrollPadding.h"
#include "StyleTranslate.h"
#include "TransformOperationData.h"
#include "ViewTimeline.h"
#include "WebAnimationUtilities.h"
#include <wtf/IteratorRange.h>

namespace WebCore {
namespace Style {

class ExtractorConverter {
public:
    // MARK: Strong value conversions

    template<typename T, typename... Rest> static Ref<CSSValue> convertStyleType(ExtractorState&, const T&, Rest&&...);

    // MARK: Primitive conversions

    template<typename ConvertibleType>
    static Ref<CSSPrimitiveValue> convert(ExtractorState&, const ConvertibleType&);
    static Ref<CSSPrimitiveValue> convert(ExtractorState&, double);
    static Ref<CSSPrimitiveValue> convert(ExtractorState&, float);
    static Ref<CSSPrimitiveValue> convert(ExtractorState&, unsigned);
    static Ref<CSSPrimitiveValue> convert(ExtractorState&, int);
    static Ref<CSSPrimitiveValue> convert(ExtractorState&, unsigned short);
    static Ref<CSSPrimitiveValue> convert(ExtractorState&, short);
    static Ref<CSSPrimitiveValue> convert(ExtractorState&, const ScopedName&);

    template<typename T> static Ref<CSSPrimitiveValue> convertNumberAsPixels(ExtractorState&, T);

    template<CSSValueID> static Ref<CSSPrimitiveValue> convertCustomIdentAtomOrKeyword(ExtractorState&, const AtomString&);

    // MARK: Transform conversions

    static Ref<CSSValue> convertTransformationMatrix(ExtractorState&, const TransformationMatrix&);
    static Ref<CSSValue> convertTransformationMatrix(const RenderStyle&, const TransformationMatrix&);

    // MARK: Shared conversions

    static Ref<CSSValue> convertPositionTryFallbacks(ExtractorState&, const FixedVector<PositionTryFallback>&);
    static Ref<CSSValue> convertTextUnderlinePosition(ExtractorState&, OptionSet<TextUnderlinePosition>);
    static Ref<CSSValue> convertTextEmphasisPosition(ExtractorState&, OptionSet<TextEmphasisPosition>);
    static Ref<CSSValue> convertSpeakAs(ExtractorState&, OptionSet<SpeakAs>);
    static Ref<CSSValue> convertPositionAnchor(ExtractorState&, const std::optional<ScopedName>&);
    static Ref<CSSValue> convertPositionArea(ExtractorState&, const PositionArea&);
    static Ref<CSSValue> convertPositionArea(ExtractorState&, const std::optional<PositionArea>&);
    static Ref<CSSValue> convertNameScope(ExtractorState&, const NameScope&);

    // MARK: MaskLayer property conversions

    static Ref<CSSValue> convertSingleMaskComposite(ExtractorState&, CompositeOperator);
    static Ref<CSSValue> convertSingleWebkitMaskComposite(ExtractorState&, CompositeOperator);
    static Ref<CSSValue> convertSingleMaskMode(ExtractorState&, MaskMode);
    static Ref<CSSValue> convertSingleWebkitMaskSourceType(ExtractorState&, MaskMode);
};

// MARK: - Strong value conversions

template<typename T, typename... Rest> Ref<CSSValue> ExtractorConverter::convertStyleType(ExtractorState& state, const T& value, Rest&&... rest)
{
    return createCSSValue(state.pool, state.style, value, std::forward<Rest>(rest)...);
}

// MARK: - Primitive conversions

template<typename ConvertibleType>
Ref<CSSPrimitiveValue> ExtractorConverter::convert(ExtractorState&, const ConvertibleType& value)
{
    return CSSPrimitiveValue::create(toCSSValueID(value));
}

inline Ref<CSSPrimitiveValue> ExtractorConverter::convert(ExtractorState&, double value)
{
    return CSSPrimitiveValue::create(value);
}

inline Ref<CSSPrimitiveValue> ExtractorConverter::convert(ExtractorState&, float value)
{
    return CSSPrimitiveValue::create(value);
}

inline Ref<CSSPrimitiveValue> ExtractorConverter::convert(ExtractorState&, unsigned value)
{
    return CSSPrimitiveValue::createInteger(value);
}

inline Ref<CSSPrimitiveValue> ExtractorConverter::convert(ExtractorState&, int value)
{
    return CSSPrimitiveValue::createInteger(value);
}

inline Ref<CSSPrimitiveValue> ExtractorConverter::convert(ExtractorState&, unsigned short value)
{
    return CSSPrimitiveValue::createInteger(value);
}

inline Ref<CSSPrimitiveValue> ExtractorConverter::convert(ExtractorState&, short value)
{
    return CSSPrimitiveValue::createInteger(value);
}

inline Ref<CSSPrimitiveValue> ExtractorConverter::convert(ExtractorState&, const ScopedName& scopedName)
{
    if (scopedName.isIdentifier)
        return CSSPrimitiveValue::createCustomIdent(scopedName.name);
    return CSSPrimitiveValue::create(scopedName.name);
}

template<typename T> Ref<CSSPrimitiveValue> ExtractorConverter::convertNumberAsPixels(ExtractorState& state, T number)
{
    return CSSPrimitiveValue::create(adjustFloatForAbsoluteZoom(number, state.style), CSSUnitType::CSS_PX);
}

template<CSSValueID keyword> Ref<CSSPrimitiveValue> ExtractorConverter::convertCustomIdentAtomOrKeyword(ExtractorState&, const AtomString& string)
{
    if (string.isNull())
        return CSSPrimitiveValue::create(keyword);
    return CSSPrimitiveValue::createCustomIdent(string);
}

// MARK: - Transform conversions

inline Ref<CSSValue> ExtractorConverter::convertTransformationMatrix(ExtractorState& state, const TransformationMatrix& transform)
{
    return convertTransformationMatrix(state.style, transform);
}

inline Ref<CSSValue> ExtractorConverter::convertTransformationMatrix(const RenderStyle& style, const TransformationMatrix& transform)
{
    auto zoom = style.usedZoom();
    if (transform.isAffine()) {
        double values[] = { transform.a(), transform.b(), transform.c(), transform.d(), transform.e() / zoom, transform.f() / zoom };
        CSSValueListBuilder arguments;
        for (auto value : values)
            arguments.append(CSSPrimitiveValue::create(value));
        return CSSFunctionValue::create(CSSValueMatrix, WTFMove(arguments));
    }

    double values[] = {
        transform.m11(), transform.m12(), transform.m13(), transform.m14() * zoom,
        transform.m21(), transform.m22(), transform.m23(), transform.m24() * zoom,
        transform.m31(), transform.m32(), transform.m33(), transform.m34() * zoom,
        transform.m41() / zoom, transform.m42() / zoom, transform.m43() / zoom, transform.m44()
    };
    CSSValueListBuilder arguments;
    for (auto value : values)
        arguments.append(CSSPrimitiveValue::create(value));
    return CSSFunctionValue::create(CSSValueMatrix3d, WTFMove(arguments));
}

// MARK: - Shared conversions

inline Ref<CSSValue> ExtractorConverter::convertPositionTryFallbacks(ExtractorState& state, const FixedVector<PositionTryFallback>& fallbacks)
{
    if (fallbacks.isEmpty())
        return CSSPrimitiveValue::create(CSSValueNone);

    CSSValueListBuilder list;
    for (auto& fallback : fallbacks) {
        if (RefPtr positionAreaProperties = fallback.positionAreaProperties) {
            auto areaValue = positionAreaProperties->getPropertyCSSValue(CSSPropertyPositionArea);
            if (areaValue)
                list.append(*areaValue);
            continue;
        }

        CSSValueListBuilder singleFallbackList;
        if (fallback.positionTryRuleName)
            singleFallbackList.append(convert(state, *fallback.positionTryRuleName));
        for (auto& tactic : fallback.tactics)
            singleFallbackList.append(convert(state, tactic));
        list.append(CSSValueList::createSpaceSeparated(singleFallbackList));
    }

    return CSSValueList::createCommaSeparated(WTFMove(list));
}

inline Ref<CSSValue> ExtractorConverter::convertTextUnderlinePosition(ExtractorState&, OptionSet<TextUnderlinePosition> textUnderlinePosition)
{
    ASSERT(!((textUnderlinePosition & TextUnderlinePosition::FromFont) && (textUnderlinePosition & TextUnderlinePosition::Under)));
    ASSERT(!((textUnderlinePosition & TextUnderlinePosition::Left) && (textUnderlinePosition & TextUnderlinePosition::Right)));

    if (textUnderlinePosition.isEmpty())
        return CSSPrimitiveValue::create(CSSValueAuto);
    bool isFromFont = textUnderlinePosition.contains(TextUnderlinePosition::FromFont);
    bool isUnder = textUnderlinePosition.contains(TextUnderlinePosition::Under);
    bool isLeft = textUnderlinePosition.contains(TextUnderlinePosition::Left);
    bool isRight = textUnderlinePosition.contains(TextUnderlinePosition::Right);

    auto metric = isUnder ? CSSValueUnder : CSSValueFromFont;
    auto side = isLeft ? CSSValueLeft : CSSValueRight;
    if (!isFromFont && !isUnder)
        return CSSPrimitiveValue::create(side);
    if (!isLeft && !isRight)
        return CSSPrimitiveValue::create(metric);
    return CSSValuePair::create(CSSPrimitiveValue::create(metric), CSSPrimitiveValue::create(side));
}

inline Ref<CSSValue> ExtractorConverter::convertTextEmphasisPosition(ExtractorState&, OptionSet<TextEmphasisPosition> textEmphasisPosition)
{
    ASSERT(!((textEmphasisPosition & TextEmphasisPosition::Over) && (textEmphasisPosition & TextEmphasisPosition::Under)));
    ASSERT(!((textEmphasisPosition & TextEmphasisPosition::Left) && (textEmphasisPosition & TextEmphasisPosition::Right)));
    ASSERT((textEmphasisPosition & TextEmphasisPosition::Over) || (textEmphasisPosition & TextEmphasisPosition::Under));

    CSSValueListBuilder list;
    if (textEmphasisPosition & TextEmphasisPosition::Over)
        list.append(CSSPrimitiveValue::create(CSSValueOver));
    if (textEmphasisPosition & TextEmphasisPosition::Under)
        list.append(CSSPrimitiveValue::create(CSSValueUnder));
    if (textEmphasisPosition & TextEmphasisPosition::Left)
        list.append(CSSPrimitiveValue::create(CSSValueLeft));
    return CSSValueList::createSpaceSeparated(WTFMove(list));
}

inline Ref<CSSValue> ExtractorConverter::convertSpeakAs(ExtractorState&, OptionSet<SpeakAs> speakAs)
{
    CSSValueListBuilder list;
    if (speakAs & SpeakAs::SpellOut)
        list.append(CSSPrimitiveValue::create(CSSValueSpellOut));
    if (speakAs & SpeakAs::Digits)
        list.append(CSSPrimitiveValue::create(CSSValueDigits));
    if (speakAs & SpeakAs::LiteralPunctuation)
        list.append(CSSPrimitiveValue::create(CSSValueLiteralPunctuation));
    if (speakAs & SpeakAs::NoPunctuation)
        list.append(CSSPrimitiveValue::create(CSSValueNoPunctuation));
    if (list.isEmpty())
        return CSSPrimitiveValue::create(CSSValueNormal);
    return CSSValueList::createSpaceSeparated(WTFMove(list));
}

inline Ref<CSSValue> ExtractorConverter::convertPositionAnchor(ExtractorState& state, const std::optional<ScopedName>& positionAnchor)
{
    if (!positionAnchor)
        return CSSPrimitiveValue::create(CSSValueAuto);
    return convert(state, *positionAnchor);
}

inline Ref<CSSValue> ExtractorConverter::convertPositionArea(ExtractorState&, const PositionArea& positionArea)
{
    auto keywordForPositionAreaSpan = [](const PositionAreaSpan span) -> CSSValueID {
        auto axis = span.axis();
        auto track = span.track();
        auto self = span.self();

        switch (axis) {
        case PositionAreaAxis::Horizontal:
            ASSERT(self == PositionAreaSelf::No);
            switch (track) {
            case PositionAreaTrack::Start:
                return CSSValueLeft;
            case PositionAreaTrack::SpanStart:
                return CSSValueSpanLeft;
            case PositionAreaTrack::End:
                return CSSValueRight;
            case PositionAreaTrack::SpanEnd:
                return CSSValueSpanRight;
            case PositionAreaTrack::Center:
                return CSSValueCenter;
            case PositionAreaTrack::SpanAll:
                return CSSValueSpanAll;
            default:
                ASSERT_NOT_REACHED();
                return CSSValueLeft;
            }

        case PositionAreaAxis::Vertical:
            ASSERT(self == PositionAreaSelf::No);
            switch (track) {
            case PositionAreaTrack::Start:
                return CSSValueTop;
            case PositionAreaTrack::SpanStart:
                return CSSValueSpanTop;
            case PositionAreaTrack::End:
                return CSSValueBottom;
            case PositionAreaTrack::SpanEnd:
                return CSSValueSpanBottom;
            case PositionAreaTrack::Center:
                return CSSValueCenter;
            case PositionAreaTrack::SpanAll:
                return CSSValueSpanAll;
            default:
                ASSERT_NOT_REACHED();
                return CSSValueTop;
            }

        case PositionAreaAxis::X:
            switch (track) {
            case PositionAreaTrack::Start:
                return self == PositionAreaSelf::No ? CSSValueXStart : CSSValueSelfXStart;
            case PositionAreaTrack::SpanStart:
                return self == PositionAreaSelf::No ? CSSValueSpanXStart : CSSValueSpanSelfXStart;
            case PositionAreaTrack::End:
                return self == PositionAreaSelf::No ? CSSValueXEnd : CSSValueSelfXEnd;
            case PositionAreaTrack::SpanEnd:
                return self == PositionAreaSelf::No ? CSSValueSpanXEnd : CSSValueSpanSelfXEnd;
            case PositionAreaTrack::Center:
                return CSSValueCenter;
            case PositionAreaTrack::SpanAll:
                return CSSValueSpanAll;
            default:
                ASSERT_NOT_REACHED();
                return CSSValueXStart;
            }

        case PositionAreaAxis::Y:
            switch (track) {
            case PositionAreaTrack::Start:
                return self == PositionAreaSelf::No ? CSSValueYStart : CSSValueSelfYStart;
            case PositionAreaTrack::SpanStart:
                return self == PositionAreaSelf::No ? CSSValueSpanYStart : CSSValueSpanSelfYStart;
            case PositionAreaTrack::End:
                return self == PositionAreaSelf::No ? CSSValueYEnd : CSSValueSelfYEnd;
            case PositionAreaTrack::SpanEnd:
                return self == PositionAreaSelf::No ? CSSValueSpanYEnd : CSSValueSpanSelfYEnd;
            case PositionAreaTrack::Center:
                return CSSValueCenter;
            case PositionAreaTrack::SpanAll:
                return CSSValueSpanAll;
            default:
                ASSERT_NOT_REACHED();
                return CSSValueYStart;
            }

        case PositionAreaAxis::Block:
            switch (track) {
            case PositionAreaTrack::Start:
                return self == PositionAreaSelf::No ? CSSValueBlockStart : CSSValueSelfBlockStart;
            case PositionAreaTrack::SpanStart:
                return self == PositionAreaSelf::No ? CSSValueSpanBlockStart : CSSValueSpanSelfBlockStart;
            case PositionAreaTrack::End:
                return self == PositionAreaSelf::No ? CSSValueBlockEnd : CSSValueSelfBlockEnd;
            case PositionAreaTrack::SpanEnd:
                return self == PositionAreaSelf::No ? CSSValueSpanBlockEnd : CSSValueSpanSelfBlockEnd;
            case PositionAreaTrack::Center:
                return CSSValueCenter;
            case PositionAreaTrack::SpanAll:
                return CSSValueSpanAll;
            default:
                ASSERT_NOT_REACHED();
                return CSSValueBlockStart;
            }

        case PositionAreaAxis::Inline:
            switch (track) {
            case PositionAreaTrack::Start:
                return self == PositionAreaSelf::No ? CSSValueInlineStart : CSSValueSelfInlineStart;
            case PositionAreaTrack::SpanStart:
                return self == PositionAreaSelf::No ? CSSValueSpanInlineStart : CSSValueSpanSelfInlineStart;
            case PositionAreaTrack::End:
                return self == PositionAreaSelf::No ? CSSValueInlineEnd : CSSValueSelfInlineEnd;
            case PositionAreaTrack::SpanEnd:
                return self == PositionAreaSelf::No ? CSSValueSpanInlineEnd : CSSValueSpanSelfInlineEnd;
            case PositionAreaTrack::Center:
                return CSSValueCenter;
            case PositionAreaTrack::SpanAll:
                return CSSValueSpanAll;
            default:
                ASSERT_NOT_REACHED();
                return CSSValueInlineStart;
            }
        }

        ASSERT_NOT_REACHED();
        return CSSValueLeft;
    };

    auto blockOrXAxisKeyword = keywordForPositionAreaSpan(positionArea.blockOrXAxis());
    auto inlineOrYAxisKeyword = keywordForPositionAreaSpan(positionArea.inlineOrYAxis());

    return CSSPropertyParserHelpers::valueForPositionArea(blockOrXAxisKeyword, inlineOrYAxisKeyword, CSSPropertyParserHelpers::ValueType::Computed).releaseNonNull();
}

inline Ref<CSSValue> ExtractorConverter::convertPositionArea(ExtractorState& state, const std::optional<PositionArea>& positionArea)
{
    if (!positionArea)
        return CSSPrimitiveValue::create(CSSValueNone);
    return convertPositionArea(state, *positionArea);
}

inline Ref<CSSValue> ExtractorConverter::convertNameScope(ExtractorState&, const NameScope& scope)
{
    switch (scope.type) {
    case NameScope::Type::None:
        return CSSPrimitiveValue::create(CSSValueNone);

    case NameScope::Type::All:
        return CSSPrimitiveValue::create(CSSValueAll);

    case NameScope::Type::Ident:
        if (scope.names.isEmpty())
            return CSSPrimitiveValue::create(CSSValueNone);

        CSSValueListBuilder list;
        for (auto& name : scope.names) {
            ASSERT(!name.isNull());
            list.append(CSSPrimitiveValue::createCustomIdent(name));
        }

        return CSSValueList::createCommaSeparated(WTFMove(list));
    }

    ASSERT_NOT_REACHED();
    return CSSPrimitiveValue::create(CSSValueNone);
}

// MARK: - MaskLayer property conversions

inline Ref<CSSValue> ExtractorConverter::convertSingleMaskComposite(ExtractorState&, CompositeOperator composite)
{
    return CSSPrimitiveValue::create(toCSSValueID(composite, CSSPropertyMaskComposite));
}

inline Ref<CSSValue> ExtractorConverter::convertSingleWebkitMaskComposite(ExtractorState&, CompositeOperator composite)
{
    return CSSPrimitiveValue::create(toCSSValueID(composite, CSSPropertyWebkitMaskComposite));
}

inline Ref<CSSValue> ExtractorConverter::convertSingleMaskMode(ExtractorState&, MaskMode maskMode)
{
    switch (maskMode) {
    case MaskMode::Alpha:
        return CSSPrimitiveValue::create(CSSValueAlpha);
    case MaskMode::Luminance:
        return CSSPrimitiveValue::create(CSSValueLuminance);
    case MaskMode::MatchSource:
        return CSSPrimitiveValue::create(CSSValueMatchSource);
    }
    ASSERT_NOT_REACHED();
    return CSSPrimitiveValue::create(CSSValueMatchSource);
}

inline Ref<CSSValue> ExtractorConverter::convertSingleWebkitMaskSourceType(ExtractorState&, MaskMode maskMode)
{
    switch (maskMode) {
    case MaskMode::Alpha:
        return CSSPrimitiveValue::create(CSSValueAlpha);
    case MaskMode::Luminance:
        return CSSPrimitiveValue::create(CSSValueLuminance);
    case MaskMode::MatchSource:
        // MatchSource is only available in the mask-mode property.
        return CSSPrimitiveValue::create(CSSValueAlpha);
    }
    ASSERT_NOT_REACHED();
    return CSSPrimitiveValue::create(CSSValueAlpha);
}

} // namespace Style
} // namespace WebCore
