/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
**/
#include "inputpwddialog.h"

#include <QDebug>
#include <QLineEdit>
#include <QSpacerItem>
#include <QLabel>
#include <QPushButton>
#include <QKeyEvent>
#include "fixlabel.h"


InputPwdDialog::InputPwdDialog(QWidget *parent)
{
    mVinoDBus = new QDBusInterface("org.ukui.ukcc.session",
                                    "/Vino",
                                    "org.ukui.ukcc.session.Vino",
                                    QDBusConnection::sessionBus(), this);
    initUi();
    initConnect();
}

InputPwdDialog::~InputPwdDialog()
{

}

void InputPwdDialog::initUi()
{
    setWindowTitle(tr("Set"));
    this->setFixedSize(480, 160);

    QVBoxLayout *mInputPwdLyt = new QVBoxLayout(this);
    mInputPwdLyt->setContentsMargins(24, 24, 24, 24);
    mInputPwdLyt->setSpacing(0);

    QFrame *mInputPwdFrame = new QFrame(this);
    mInputPwdFrame->setFixedSize(432, 36);
    mInputPwdFrame->setFrameShape(QFrame::NoFrame);

    QHBoxLayout *mLyt_1= new QHBoxLayout(mInputPwdFrame);
    mLyt_1->setContentsMargins(0, 0, 0, 0);
    mLyt_1->setSpacing(8);

    FixLabel *mSetPwdLabel = new FixLabel(mInputPwdFrame);
    mSetPwdLabel->setFixedSize(72, 36);
    mSetPwdLabel->setText(tr("Set Password"));

    mPwdLineEdit = new QLineEdit(mInputPwdFrame);
    mPwdLineEdit->setAttribute(Qt::WA_InputMethodEnabled, false);  //限制中文输入法
    mPwdLineEdit->setFixedSize(352, 36);
    mPwdLineEdit->installEventFilter(this);

    mLyt_1->addWidget(mSetPwdLabel);
    mLyt_1->addWidget(mPwdLineEdit);

    mFirstload = true;
    mStatus = false;

    mHintLabel = new QLabel(this);
    mHintLabel->setFixedSize(432,24);
    mHintLabel->setContentsMargins(84, 0, 0, 0);
    mHintLabel->setText(tr("Must be 1-8 characters long"));
    mHintLabel->setStyleSheet("QLabel{color:red; font-size : 14px}");

    QFrame *mInputPwdFrame_1 = new QFrame(this);
    mInputPwdFrame_1->setFixedSize(432, 48);
    mInputPwdFrame_1->setFrameShape(QFrame::NoFrame);

    QHBoxLayout *mLyt_2= new QHBoxLayout(mInputPwdFrame_1);
    mLyt_2->setContentsMargins(0, 0, 0, 0);
    mLyt_2->setSpacing(16);


    mLyt_1->addWidget(mSetPwdLabel);
    mLyt_1->addWidget(mPwdLineEdit);

    mCancelBtn = new QPushButton(mInputPwdFrame_1);
    mCancelBtn->setMinimumWidth(96);
    mCancelBtn->setText(tr("Cancel"));

    mConfirmBtn = new QPushButton(mInputPwdFrame_1);
    mConfirmBtn->setMinimumWidth(96);
    mConfirmBtn->setText(tr("Confirm"));

    mLyt_2->addStretch();
    mLyt_2->addWidget(mCancelBtn);
    mLyt_2->addWidget(mConfirmBtn);

    mInputPwdLyt->addWidget(mInputPwdFrame);
    mInputPwdLyt->addWidget(mHintLabel);
    mInputPwdLyt->addStretch();
    mInputPwdLyt->addWidget(mInputPwdFrame_1);


    if(QByteArray::fromBase64(mVinoDBus->property("password").toString().toLatin1()).length() <= 8) {
        if (mVinoDBus->property("password").toString() == "keyring") {
            mPwdLineEdit->setText("");
            mConfirmBtn->setEnabled(false);
            mHintLabel->setVisible(true);
        } else {
            mPwdLineEdit->setText(QByteArray::fromBase64(mVinoDBus->property("password").toString().toLatin1()));
            mHintLabel->setVisible(false);
        }
    }

}

void InputPwdDialog::mPwdLineEditInputSlot(const QString &pwd)
{
    Q_UNUSED(pwd);
    mStatus = true;
    mConfirmBtn->setEnabled(true);
    if (pwd.length() <= 8 && !pwd.isEmpty()) {
        QByteArray text = pwd.toLocal8Bit();
        secPwd = text.toBase64();
        mHintLabel->setVisible(false);
    } else if (pwd.isEmpty()) {
        mConfirmBtn->setEnabled(false);
        mHintLabel->setVisible(true);
        secPwd = NULL;
    } else {
        mPwdLineEdit->setText(pwd.mid(0, 8));
        QByteArray text = pwd.mid(0, 8).toLocal8Bit();
        mHintLabel->setVisible(true);
        secPwd = text.toBase64();
    }
}

void InputPwdDialog::initConnect() {

    connect(mCancelBtn, &QPushButton::clicked, [=](bool checked){
        Q_UNUSED(checked)
        this->close();
    });

    connect(mConfirmBtn, &QPushButton::clicked, [=](bool checked){
        Q_UNUSED(checked)
        if (mStatus && secPwd.length() == 0) {
            return;
        } else if (!mStatus){
            mVinoDBus->call("setVinoKey", kAuthenticationKey, "vnc");
            this->close();
        } else {
            mVinoDBus->call("setVinoKey", kVncPwdKey, secPwd);
            mVinoDBus->call("setVinoKey", kAuthenticationKey, "vnc");
            this->close();
        }
    });
    //使用textEdited信号是为了防止密码框setText时触发信号
    connect(mPwdLineEdit, &QLineEdit::textEdited, this, &InputPwdDialog::mPwdLineEditInputSlot);
}

bool InputPwdDialog::eventFilter(QObject *wcg, QEvent *event)
{
    //过滤
    if(wcg==mPwdLineEdit){
    if(event->type() == QEvent::MouseButtonPress){
       if(mPwdLineEdit->hasFocus()){
           if (mFirstload) {
               mPwdLineEdit->setText("");
               mFirstload = false;
           }
       }
    }
    }
    // 回车键触发确定按钮点击事件
    if (event->type() == QEvent::KeyPress) {
        QKeyEvent *mEvent = static_cast<QKeyEvent *>(event);
        if (mEvent->key() == Qt::Key_Enter || mEvent->key() == Qt::Key_Return) {
            if (mConfirmBtn->isEnabled()) {
                emit mConfirmBtn->clicked();
            } else {
                this->close();
            }
        }

    }
    return QWidget::eventFilter(wcg,event);
}
