// Copyright 2016-2018 Yubico AB
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// +build ignore

package main

import (
	"encoding/json"
	"io/ioutil"
	"log"
	"os"
	"runtime"
	"text/template"
	"time"
)

type VersionInput struct {
	Major int `json:"major"`
	Minor int `json:"minor"`
	Patch int `json:"patch"`
	Build int `json:"build"`
}

func die(err error) {
	if err != nil {
		log.Fatal(err)
	}
}

func main() {

	// Read JSON version file

	raw_version, err := ioutil.ReadFile("./VERSION")
	die(err)

	var vi VersionInput
	err = json.Unmarshal(raw_version, &vi)
	die(err)

	f, err := os.Create("version.go")
	die(err)
	defer f.Close()

	var funcMap = make(template.FuncMap)
	funcMap["now"] = time.Now
	funcMap["isWindows"] = func() bool { return runtime.GOOS == "windows" }

	var versionTemplate = template.Must(template.New("").Funcs(funcMap).Parse(`// go generate
// GENERATED BY THE COMMAND ABOVE; DO NOT EDIT
// This file was generated by robots at
// {{ now.UTC }}

package main

import (
	"fmt"
)

type VersionInfo struct {
	Major int
	Minor int
	Patch int
	Build int
}

func (v *VersionInfo) String() string {
	return fmt.Sprintf("%d.%d.%d", v.Major, v.Minor, v.Patch)
}

var Version = VersionInfo{ {{ .Major }}, {{ .Minor }}, {{ .Patch }}, {{ .Build }} }
	`))

	versionTemplate.Execute(f, vi)

	var versionInfoTemplate = template.Must(template.New("").Parse(`{
	"FixedFileInfo" : {
		"FileVersion": {
			"Major": {{ .Major }},
			"Minor": {{ .Minor }},
			"Patch": {{ .Patch }},
			"Build": {{ .Build }}
		},
		"ProductVersion": {
			"Major": {{ .Major }},
			"Minor": {{ .Minor }},
			"Patch": {{ .Patch }},
			"Build": {{ .Build }}
		},
		"FileFlagsMask": "3f",
		"FileFlags": "00",
		"FileOS": "040004",
		"FileType": "01",
		"FileSubType": "00"
	},
	"StringFileInfo": {
		"Comments": "",
		"CompanyName": "Yubico AB",
		"FileDescription": "YubiHSM Connector",
		"FileVersion": "{{ .Major }}.{{ .Minor }}.{{ .Patch }}.{{ .Build }}",
		"InternalName": "yubihsm-connector",
		"LegalCopyright": "© Yubico AB",
		"LegalTrademarks": "",
		"OriginalFilename": "yubihsm-connector",
		"PrivateBuild": "",
		"ProductName": "YubiHSM Connector",
		"ProductVersion": "{{ .Major }}.{{ .Minor }}.{{ .Patch }}.{{ .Build }}",
		"SpecialBuild": ""
	},
	"VarFileInfo": {
		"Translation": {
			"LangID": "0409",
			"CharsetID": "04B0"
		}
	}
}`))

	if runtime.GOOS == "windows" {
		f, err = os.Create("versioninfo.json")
		die(err)
		defer f.Close()

		versionInfoTemplate.Execute(f, vi)
	}
}
