// ****************************************************************************
//
//          Aevol - An in silico experimental evolution platform
//
// ****************************************************************************
//
// Copyright: See the AUTHORS file provided with the package or <www.aevol.fr>
// Web: http://www.aevol.fr/
// E-mail: See <http://www.aevol.fr/contact/>
// Original Authors : Guillaume Beslon, Carole Knibbe, David Parsons
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// ****************************************************************************

#include "FastaWriter.h"

namespace aevol {

FastaWriter::FastaWriter(std::filesystem::path filename): filename_(filename), out_(filename) {
  if (not out_) {
    throw std::runtime_error(std::string("failed to open file ") + filename_.string());
  }
}

void FastaWriter::write_individual(
    const std::string& seqid, const Individual& individual, const FastaModifiers& modifiers) {
  for (auto chrsm: chrsms) {
    // ----- Write fasta definition line
    auto chrsm_seqid = seqid + (exp_setup->diploid() ? "-" + to_string(chrsm) : "");
    out_ << '>' << chrsm_seqid;
    for (const auto& modifier: modifiers) {
      out_ << " [" << modifier.name << '=' << modifier.value << ']';
    }
    out_ << "\n";

    // ----- Write sequence
    out_ << individual.annotated_chromosome(chrsm).dna() << "\n";
  }
}

void FastaWriter::write_individual(const indivs_idxs_sorted_map_t::value_type& entry, bool clonal_population) {
  // Shorthand for genome_id and vector of indexes
  const auto& [genome_id, indexes] = entry.second;

  auto index_str = std::string{};
  if (clonal_population) {
    index_str = "all";
  } else {
    index_str = std::to_string(indexes[0]);
    for (size_t i = 1; i < indexes.size(); ++i) {
      index_str += " " + std::to_string(indexes[i]);
    }
  }

  auto modifiers = FastaModifiers();
  modifiers.emplace_back("organism", aevol::flavor);
  modifiers.emplace_back("indexes", index_str);

  write_individual(genome_id, *entry.first, modifiers);
}

}  // namespace aevol
