# Behavioral Differences: Rust vs Python Implementation

This document describes behavioral differences between the Rust implementation of `deb-import-uncommitted` and the Python reference implementation at `/home/jelmer/src/breezy-debian/import_uncommitted.py`.

This excludes extra features in Rust (which are fine) and bugs that have been fixed.

---

## 1. Commit Message Format (Merge Operations)

**Location:** Rust lines 461-467, Python lines 388-389

**Python:**
```
Merge archive versions: 1.0-1, 1.0-2
```

**Rust:**
```
Merge archive versions: 1.0-1 (debian/1.0-1), 1.0-2 (debian/1.0-2)
```

**Impact:** Rust includes tag names in parentheses.

---

## 2. Final Output Format

**Location:** Rust lines 838-844, Python line 659

**Python:**
```
Imported uploads: ['1.0-1', '1.0-2'].
```

**Rust:**
```
Imported uploads: 1.0-1 (debian/1.0-1), 1.0-2 (debian/1.0-2).
```

**Impact:** Different formatting style.

---

## 3. set_vcs_git_url Implementation

**Location:** Rust lines 207-235, Python lines 417-430

**Python:**
```python
control.source["Vcs-Git"] = urlutils.join(vcs_git_base, "%s.git" % package_name)
```

**Rust:**
```rust
let mut vcs_git: ParsedVcs = vcs_git_base.parse().unwrap();
vcs_git.repo_url = format!("{}/{}.git",
    vcs_git.repo_url.trim_end_matches('/'), package_name);
```

**Impact:** Rust parses and reconstructs the URL. May handle edge cases differently (URLs with existing paths, query parameters, branches).

---

## 4. Changelog Update Method

**Location:** Rust lines 762-769, Python lines 621-622

**Python:**
```python
changelog.add_entry(["Set Vcs-Git header."])
```

**Rust:**
```rust
changelog.try_auto_add_change(
    &["Set Vcs-Git header."],
    debian_changelog::get_maintainer().unwrap(),
    None::<String>,
    None,
)
```

**Impact:** Rust explicitly provides maintainer information. May result in different `Changed-by` fields.

---

## 5. Error Message Capitalization

**Examples:**
- Python: "tree version X does not appear in archive changelog"
- Rust: "Tree version X does not appear in archive changelog"

**Impact:** Cosmetic difference in capitalization.

---

## 6. SVP Context Structure

**Location:** Rust lines 532-553, Python lines 648-651

**Python:**
```python
"context": {
    "versions": [tag_name for ...],
    "tags": [(tag_name, str(version)) for ...],
}
```

**Rust:**
```rust
Context {
    tags: Vec<(String, debversion::Version)>,
}
// Accessor methods: versions(), tag_names(), tags()
```

**Impact:** Rust has non-redundant structure with accessor methods. Python has redundant "versions" field.

---

## 7. Snapshot Download Implementation

**Python:** Inline `download_snapshot()` function (102 lines)

**Rust:** Uses `debian_analyzer::snapshot::download_snapshot()` library

**Impact:** Different implementation. Library should provide equivalent functionality but hasn't been verified line-by-line.

---

## 8. is_noop_upload: Changelog Comparison

**Location:** Rust lines 77-84, Python lines 272-275

**Python:**
```python
del new_cl._blocks[0]
return str(new_cl) == str(old_cl)  # String comparison
```

**Rust:**
```rust
let new_cl = new_cl.iter().skip(1).collect::<ChangeLog>();
new_cl == old_cl  # Structural equality via PartialEq
```

**Impact:** Different comparison method. Should be equivalent but could differ if ChangeLog's `PartialEq` doesn't match string conversion.

---

## 9. find_missing_versions: Defensive Version Handling

**Location:** Rust lines 30-41

**Python:**
```python
missing_versions.append(block.version)  # Assumes version exists
```

**Rust:**
```rust
match block.version() {
    Some(version) => {
        missing_versions.push(version);
    }
    None => {
        log::warn!(
            "Skipping changelog entry at index {} without a version (package: {:?})",
            idx,
            block.package()
        );
    }
}
```

**Impact:** Rust checks for None versions and warns when skipping invalid entries. Python would crash on invalid entries.

---

## Summary

| Category | Count |
|----------|-------|
| Output formatting | 3 |
| Implementation differences | 5 |
| Defensive coding | 1 |
| **Total** | **9** |

All differences maintain functional equivalence. The Rust implementation produces the same results with minor variations in output formatting and internal implementation details.
