use layout;
use core:lang;
use lang:bs;
use lang:bs:macro;

// Create "classes" for the keyword "window".
Class createFrame(SrcPos pos, Scope env, SStr name, SWindowBody body) on Compiler {
	Class c = createClass(pos, env, name, body);
	c.super(name{ui:Frame});
	c;
}

Class createDialog(SrcPos pos, Scope env, SStr name, SWindowBody body) on Compiler {
	Class c = createClass(pos, env, name, body);
	c.super(name{ui:Dialog});
	c;
}

Class createWindow(SrcPos pos, Scope env, SStr name, SWindowBody body) on Compiler {
	Class c = createClass(pos, env, name, body);
	c.super(name{ui:Container});
	c;
}

Class createWindow(SrcPos pos, Scope env, SStr name, SrcName parent, SWindowBody body) on Compiler {
	// TODO: Check so that we inherit from a window. We can not necessarily do that right now, since
	// that particular class might not be loaded yet.
	Class c = createClass(pos, env, name, body);
	c.super(parent);
	c;
}

// Get all SLayoutMemberDecls inside a syntax node.
private SLayoutMemberDecl[] allDecls(SLayoutRoot root) on Compiler {
	SLayoutMemberDecl[] result;
	for (child in root.allChildren(named{SLayoutMemberDecl})) {
		if (child as SLayoutMemberDecl) {
			result << child;
		}
	}
	result;
}


/**
 * Placeholder for layout objects, adds some extra data.
 */
class WindowLayout on Compiler {
	SLayoutRoot layout;
	Bool border;

	init(SLayoutRoot layout, Bool border) {
		init {
			layout = layout;
			border = border;
		}
	}
}


/**
 * Custom contents for a window 'class'.
 */
class WindowBody extends ClassBody {
	private WindowLayout? layout;

	init(Class owner) {
		init(owner);
	}

	// Handle our special types.
	void add(TObject obj) {
		if (obj as WindowLayout) {
			if (layout) {
				throw SyntaxError(obj.layout.pos, "Multiple layout definitions are not allowed.");
			}

			layout = obj;
			return;
		}

		super:add(obj);
	}

	// Inject 'layout' now, so that we can account for any newly declared classes.
	void prepareWraps() {
		if (layout) {
			addLayoutVars(layout.layout);
			modifyCtors(layout);
			addResized(layout.layout.pos);
			addMinSize(layout.layout.pos);
		}
	}

	private void addLayoutVars(SLayoutRoot layout) {
		for (child in layout.allDecls()) {
			// Add member variables.
			owner.add(classVar(owner, child.type.transform, child.name.transform));
		}

		// Add a member for the layout itself.
		owner.add(MemberVar("layout", Value(named{Layout}), owner));
	}

	// Modify all constructors so that we can inject some code into them!
	private void modifyCtors(WindowLayout layout) {
		MemberWrap[] newWraps;
		for (item in wraps) {
			newWraps << CtorWrap(item.node, layout);
		}
		wraps = newWraps;
	}

	// Add the 'resized' overload.
	private void addResized(SrcPos pos) {
		Value sizeType(named{core:geometry:Size});
		BSTreeFn resized(Value(), SStr("resized", pos), [thisParam(owner), ValParam(sizeType, "s")], null);
		owner.add(resized);

		FnBody resizeBody(resized, owner.scope);
		resizeBody.add(pattern(resizeBody) {
						layout.pos = core:geometry:Rect(s);
					});
		resized.body = resizeBody;
	}

	// Add the 'minSize' overload.
	private void addMinSize(SrcPos pos) {
		Value size(named{core:geometry:Size});
		BSTreeFn minSize(size, SStr("minSize", pos), [thisParam(owner)], null);
		owner.add(minSize);

		FnBody minSizeBody(minSize, owner.scope);
		var layout = namedExpr(minSizeBody, pos, "layout", Actuals());
		minSizeBody.add(namedExpr(minSizeBody, pos, "minSize", layout));
		minSize.body = minSizeBody;
	}
}


/**
 * Custom wrap that alters constructors.
 */
class CtorWrap extends MemberWrap {
	private WindowLayout layout;

	// Create.
	init(lang:bnf:Node node, WindowLayout layout) {
		init(node) {
			layout = layout;
		}
	}

	// Transform the node.
	Named transform(Class owner) {
		Named r = super:transform(owner);
		if (r as BSCtor) {
			modifyCtor(r);
		}

		r;
	}

	// Modify a constructor to include additional initializers, and the setup of the actual layout.
	void modifyCtor(BSCtor ctor) {
		if (body = ctor.body as SCtorBody) {
			ctor.body = CtorBodyWrap(body, layout);
		} else {
			print("WARNING: Unable to wrap constructor body.");
		}
	}
}


/**
 * Wrap the constructor body.
 */
class CtorBodyWrap extends SCtorBody {
	private SCtorBody real;
	private WindowLayout layout;

	// Initialize.
	init(SCtorBody real, WindowLayout layout) {
		init() {
			real = real;
			layout = layout;
		}
	}

	// Wrap the transform function.
	CtorBody transform(BSCtor ctor) {
		CtorBody result = real.transform(ctor);

		for (Nat i = 0; i < result.count; i++) {
			Expr e = result[i];

			if (e as InitBlock) {
				i += fixSuper(result, i, e);
			}
		}

		result;
	}

	// Add things to the super call. Returns number of statements inserted.
	private Nat fixSuper(ExprBlock block, Nat pos, InitBlock s) {
		var decls = layout.layout.allDecls();
		Nat added = 0;

		SrcPos lPos = layout.layout.pos;

		for (member in decls) {
			Actuals params = if (x = member.params) { x.transform(block); } else { Actuals(); };
			SStr name = member.name.transform;
			SStr varName("layout:" + name.v, name.pos);

			// Create a local variable so that we can initialize the layout!
			Var v(block, member.type.transform, varName, params);
			block.insert(pos++, v);
			added++;

			// Initialize the variable as well.
			LocalVarAccess var(member.pos, v.var);
			s.init(Initializer(member.name.transform, var));
		}

		// Initialize the layout itself as well. We need an explicit cast here.
		SStr layoutName("layout", lPos);
		Expr layoutExpr = layout.layout.transform(block).block;
		if (layout.border) {
			layoutExpr = pattern(block) {
				layout:Border b;
				b.add(${layoutExpr});
				b;
			};
		}
		s.init(Initializer(layoutName, layoutExpr));


		// Call 'add' for each of the components after the constructor call itself.
		// We use "findAll" so that we are sure to find even the things added through
		// any other syntax.
		pos++;
		{
			var me = namedExpr(block, lPos, "this", Actuals());
			if (!Value(named{Container}).mayReferTo(me.result.type))
				throw SyntaxError(real.pos, "A class that uses the UI layout syntax must inherit from the class ui.Container.");

			CtorCall cb(lPos, block.scope, named{AddCallback:__init<AddCallback, Container>}, Actuals(me));
			FnPtr ptr(cb, named{AddCallback:add<AddCallback, Component>}, lPos);
			var find = namedExpr(block, lPos, "findAll", namedExpr(block, layoutName, Actuals()), Actuals(ptr));
			block.insert(pos++, find);
		}

		//  We add one statements for each declaration found.
		added;
	}
}

/**
 * Class used to hold the callback for the Window.
 */
class AddCallback on Ui {
	Container to;

	init(Container to) {
		init { to = to; }
	}

	void add(Component component) {
		if (component as UiComponent) {
			to.add(component.window);
		}
	}
}


/**
 * Custom layout block for variable declarations.
 */
class LayoutDeclBlock extends LayoutBlock {
	// Create.
	init(LayoutRoot parent, SStr member) {
		ExprBlock block(member.pos, parent.root);
		ExprBlock sub(member.pos, block);

		Expr? preWrap;
		Expr src = namedExpr(block, SStr("layout:" + member.v, member.pos), Actuals());
		if (!Value(named{layout:Component}).mayReferTo(src.result.type)) {
			preWrap = src;
			src = namedExpr(block, member.pos, "component", src);
		}
		Var result(block, SStr("@ result"), src);

		block.add(result);
		block.add(sub);
		LocalVarAccess resultAcc(member.pos, result.var);
		block.add(resultAcc);

		init(block, sub, resultAcc, preWrap);
	}
}


/**
 * Custom layout block for nested containers.
 */
package Expr nestedContainerBlock(SrcPos pos, Block parent, SLayout layout, Bool addBorder) on Compiler {

	LayoutRoot layoutRoot(ExprBlock(pos, parent));
	layoutRoot.add(layout.transform(layoutRoot));

	Expr layoutExpr = layoutRoot.block;

	if (addBorder) {
		layoutExpr = pattern(parent) {
			layout:Border border;
			border.add(${layoutExpr});
			border;
		};
	}

	pattern(parent) {
		ui:UiContainer container(${layoutExpr});
		container.layout.findAll(&AddCallback(container).add);
		container;
	};
}


/**
 * Special case of having nested containers last.
 */
package void addNested(LayoutBlock block, SStr name, Expr[] params, Expr last) on Compiler {
	Actuals actuals;
	for (p in params)
		actuals.add(p);
	actuals.add(last);
	block.add(name, actuals);
}


/**
 * Block for push:ing and pop:ing layers using layer(g, ...) {} syntax.
 *
 * Expands to:
 * try {
 *     layer.push(<params>)
 *     {...}
 *     layer.pop()
 * } catch (Exception x) {
 *     layer.pop()
 *     throw x;
 * }
 */
Expr graphicsLayer(SrcPos pos, Block context, Expr graphics, Expr[] params, Expr contents) {
	pattern(context) {
		var g = ${graphics};
		try {
			g.push(@{params});
			${contents};
			g.pop();
		} catch (Exception e) {
			g.pop();
			throw e;
		}
	};
}
