/*
 * Copyright 2021-2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.events.api;

import javax.xml.namespace.NamespaceContext;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import static org.opentest4j.reporting.events.api.Element.CDATA_WORKAROUND_COMMENT;

class EscapingXMLStreamWriter implements XMLStreamWriter {

	static final char ILLEGAL_CHARACTER_REPLACEMENT = '\uFFFD';

	private final AttributeValueEscapingWriter out;
	private final XMLStreamWriter delegate;

	EscapingXMLStreamWriter(AttributeValueEscapingWriter out, XMLStreamWriter delegate) {
		this.out = out;
		this.delegate = delegate;
	}

	@Override
	public void writeStartElement(String localName) throws XMLStreamException {
		delegate.writeStartElement(localName);
	}

	@Override
	public void writeStartElement(String namespaceURI, String localName) throws XMLStreamException {
		delegate.writeStartElement(namespaceURI, localName);
	}

	@Override
	public void writeStartElement(String prefix, String localName, String namespaceURI) throws XMLStreamException {
		delegate.writeStartElement(prefix, localName, namespaceURI);
	}

	@Override
	public void writeEmptyElement(String namespaceURI, String localName) throws XMLStreamException {
		delegate.writeEmptyElement(namespaceURI, localName);
	}

	@Override
	public void writeEmptyElement(String prefix, String localName, String namespaceURI) throws XMLStreamException {
		delegate.writeEmptyElement(prefix, localName, namespaceURI);
	}

	@Override
	public void writeEmptyElement(String localName) throws XMLStreamException {
		delegate.writeEmptyElement(localName);
	}

	@Override
	public void writeEndElement() throws XMLStreamException {
		delegate.writeEndElement();
	}

	@Override
	public void writeEndDocument() throws XMLStreamException {
		delegate.writeEndDocument();
	}

	@Override
	public void close() throws XMLStreamException {
		delegate.close();
	}

	@Override
	public void flush() throws XMLStreamException {
		delegate.flush();
	}

	@Override
	public void writeAttribute(String localName, String value) throws XMLStreamException {
		withWhitespaceReplacingEnabled(() -> delegate.writeAttribute(localName, replaceIllegalCharacters(value)));
	}

	@Override
	public void writeAttribute(String prefix, String namespaceURI, String localName, String value)
			throws XMLStreamException {
		withWhitespaceReplacingEnabled(
			() -> delegate.writeAttribute(prefix, namespaceURI, localName, replaceIllegalCharacters(value)));
	}

	@Override
	public void writeAttribute(String namespaceURI, String localName, String value) throws XMLStreamException {
		withWhitespaceReplacingEnabled(
			() -> delegate.writeAttribute(namespaceURI, localName, replaceIllegalCharacters(value)));
	}

	@Override
	public void writeNamespace(String prefix, String namespaceURI) throws XMLStreamException {
		delegate.writeNamespace(prefix, namespaceURI);
	}

	@Override
	public void writeDefaultNamespace(String namespaceURI) throws XMLStreamException {
		delegate.writeDefaultNamespace(namespaceURI);
	}

	@Override
	public void writeComment(String data) throws XMLStreamException {
		// Remove bogus comment that avoids CDATA sections from being merged
		if (!CDATA_WORKAROUND_COMMENT.equals(data)) {
			delegate.writeComment(replaceIllegalCharacters(data));
		}
	}

	@Override
	public void writeProcessingInstruction(String target) throws XMLStreamException {
		delegate.writeProcessingInstruction(target);
	}

	@Override
	public void writeProcessingInstruction(String target, String data) throws XMLStreamException {
		delegate.writeProcessingInstruction(target, data);
	}

	@Override
	public void writeCData(String data) throws XMLStreamException {
		delegate.writeCData(replaceIllegalCharacters(data));
	}

	@Override
	public void writeDTD(String dtd) throws XMLStreamException {
		delegate.writeDTD(dtd);
	}

	@Override
	public void writeEntityRef(String name) throws XMLStreamException {
		delegate.writeEntityRef(name);
	}

	@Override
	public void writeStartDocument() throws XMLStreamException {
		delegate.writeStartDocument();
	}

	@Override
	public void writeStartDocument(String version) throws XMLStreamException {
		delegate.writeStartDocument(version);
	}

	@Override
	public void writeStartDocument(String encoding, String version) throws XMLStreamException {
		delegate.writeStartDocument(encoding, version);
	}

	@Override
	public void writeCharacters(String text) throws XMLStreamException {
		delegate.writeCharacters(replaceIllegalCharacters(text));
	}

	@Override
	public void writeCharacters(char[] text, int start, int len) throws XMLStreamException {
		writeCharacters(new String(text, start, len));
	}

	@Override
	public String getPrefix(String uri) throws XMLStreamException {
		return delegate.getPrefix(uri);
	}

	@Override
	public void setPrefix(String prefix, String uri) throws XMLStreamException {
		delegate.setPrefix(prefix, uri);
	}

	@Override
	public void setDefaultNamespace(String uri) throws XMLStreamException {
		delegate.setDefaultNamespace(uri);
	}

	@Override
	public void setNamespaceContext(NamespaceContext context) throws XMLStreamException {
		delegate.setNamespaceContext(context);
	}

	@Override
	public NamespaceContext getNamespaceContext() {
		return delegate.getNamespaceContext();
	}

	@Override
	public Object getProperty(String name) throws IllegalArgumentException {
		return delegate.getProperty(name);
	}

	private static String replaceIllegalCharacters(String text) {
		if (text.codePoints().allMatch(EscapingXMLStreamWriter::isAllowedXmlCharacter)) {
			return text;
		}
		StringBuilder result = new StringBuilder(text.length() * 2);
		text.codePoints().forEach(codePoint -> {
			if (isAllowedXmlCharacter(codePoint)) {
				result.appendCodePoint(codePoint);
			}
			else {
				result.append(ILLEGAL_CHARACTER_REPLACEMENT);
			}
		});
		return result.toString();
	}

	private static boolean isAllowedXmlCharacter(int codePoint) {
		// source: https://www.w3.org/TR/xml/#charsets
		return codePoint == 0x9 //
				|| codePoint == 0xA //
				|| codePoint == 0xD //
				|| (codePoint >= 0x20 && codePoint <= 0xD7FF) //
				|| (codePoint >= 0xE000 && codePoint <= 0xFFFD) //
				|| (codePoint >= 0x10000 && codePoint <= 0x10FFFF);
	}

	private void withWhitespaceReplacingEnabled(XmlStreamOperation operation) throws XMLStreamException {
		delegate.flush();
		out.setWhitespaceReplacingEnabled(true);
		operation.execute();
		delegate.flush();
		out.setWhitespaceReplacingEnabled(false);
	}

	@FunctionalInterface
	private interface XmlStreamOperation {
		void execute() throws XMLStreamException;
	}
}
