use super::Element;

#[derive(Clone, Copy, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
#[cfg_attr(feature = "serde", derive(serde::Serialize, serde::Deserialize))]
/// The category of the oxidation state, to distinguish common and uncommon states
pub enum OxidationStateCategory {
    /// Most common oxidation states
    Main,
    /// Less common oxidation states that are observed under specific conditions
    Extended,
    /// All oxidation states, i.e., Main + Extended
    All,
}

impl Element {
    /// Returns the element's oxidation states for the given `category`.
    ///
    /// # Example
    /// ```rust
    /// use mendeleev::{Element, OxidationStateCategory::{Main, Extended, All}};
    ///
    /// assert_eq!(Element::N.oxidation_states(Main), [-3, 3, 5]);
    /// assert_eq!(Element::N.oxidation_states(Extended), [-2, -1, 0, 1, 2, 4]);
    /// assert_eq!(Element::N.oxidation_states(All), [-3, -2, -1, 0, 1, 2, 3, 4, 5]);
    /// ```
    pub const fn oxidation_states(&self, category: OxidationStateCategory) -> &'static [i8] {
        use Element as E;
        use OxidationStateCategory as O;
        match (self, category) {
            (E::H, O::Main) => &[-1, 1],
            (E::H, O::Extended) => &[],
            (E::H, O::All) => &[-1, 1],
            (E::He, O::Main) => &[],
            (E::He, O::Extended) => &[],
            (E::He, O::All) => &[],
            (E::Li, O::Main) => &[1],
            (E::Li, O::Extended) => &[0],
            (E::Li, O::All) => &[0, 1],
            (E::Be, O::Main) => &[2],
            (E::Be, O::Extended) => &[0, 1],
            (E::Be, O::All) => &[0, 1, 2],
            (E::B, O::Main) => &[3],
            (E::B, O::Extended) => &[-5, -1, 0, 1, 2],
            (E::B, O::All) => &[-5, -1, 0, 1, 2, 3],
            (E::C, O::Main) => &[-4, -3, -2, -1, 0, 1, 2, 3, 4],
            (E::C, O::Extended) => &[],
            (E::C, O::All) => &[-4, -3, -2, -1, 0, 1, 2, 3, 4],
            (E::N, O::Main) => &[-3, 3, 5],
            (E::N, O::Extended) => &[-2, -1, 0, 1, 2, 4],
            (E::N, O::All) => &[-3, -2, -1, 0, 1, 2, 3, 4, 5],
            (E::O, O::Main) => &[-2],
            (E::O, O::Extended) => &[-1, 0, 1, 2],
            (E::O, O::All) => &[-2, -1, 0, 1, 2],
            (E::F, O::Main) => &[-1],
            (E::F, O::Extended) => &[0],
            (E::F, O::All) => &[-1, 0],
            (E::Ne, O::Main) => &[],
            (E::Ne, O::Extended) => &[],
            (E::Ne, O::All) => &[],
            (E::Na, O::Main) => &[1],
            (E::Na, O::Extended) => &[-1, 0],
            (E::Na, O::All) => &[-1, 0, 1],
            (E::Mg, O::Main) => &[2],
            (E::Mg, O::Extended) => &[0, 1],
            (E::Mg, O::All) => &[0, 1, 2],
            (E::Al, O::Main) => &[3],
            (E::Al, O::Extended) => &[-2, -1, 0, 1, 2],
            (E::Al, O::All) => &[-2, -1, 0, 1, 2, 3],
            (E::Si, O::Main) => &[-4, 4],
            (E::Si, O::Extended) => &[-3, -2, -1, 0, 1, 2, 3],
            (E::Si, O::All) => &[-4, -3, -2, -1, 0, 1, 2, 3, 4],
            (E::P, O::Main) => &[-3, 3, 5],
            (E::P, O::Extended) => &[-2, -1, 0, 1, 2, 4],
            (E::P, O::All) => &[-3, -2, -1, 0, 1, 2, 3, 4, 5],
            (E::S, O::Main) => &[-2, 2, 4, 6],
            (E::S, O::Extended) => &[-1, 0, 1, 3, 5],
            (E::S, O::All) => &[-2, -1, 0, 1, 2, 3, 4, 5, 6],
            (E::Cl, O::Main) => &[-1, 1, 3, 5, 7],
            (E::Cl, O::Extended) => &[2, 4, 6],
            (E::Cl, O::All) => &[-1, 1, 2, 3, 4, 5, 6, 7],
            (E::Ar, O::Main) => &[0],
            (E::Ar, O::Extended) => &[],
            (E::Ar, O::All) => &[0],
            (E::K, O::Main) => &[1],
            (E::K, O::Extended) => &[-1],
            (E::K, O::All) => &[-1, 1],
            (E::Ca, O::Main) => &[2],
            (E::Ca, O::Extended) => &[1],
            (E::Ca, O::All) => &[1, 2],
            (E::Sc, O::Main) => &[3],
            (E::Sc, O::Extended) => &[0, 1, 2],
            (E::Sc, O::All) => &[0, 1, 2, 3],
            (E::Ti, O::Main) => &[2, 3, 4],
            (E::Ti, O::Extended) => &[-2, -1, 0, 1],
            (E::Ti, O::All) => &[-2, -1, 0, 1, 2, 3, 4],
            (E::V, O::Main) => &[2, 3, 4, 5],
            (E::V, O::Extended) => &[-3, -1, 0, 1],
            (E::V, O::All) => &[-3, -1, 0, 1, 2, 3, 4, 5],
            (E::Cr, O::Main) => &[2, 3, 6],
            (E::Cr, O::Extended) => &[-4, -2, -1, 0, 1, 4, 5],
            (E::Cr, O::All) => &[-4, -2, -1, 0, 1, 2, 3, 4, 5, 6],
            (E::Mn, O::Main) => &[2, 3, 4, 6, 7],
            (E::Mn, O::Extended) => &[-3, -2, -1, 0, 1, 5],
            (E::Mn, O::All) => &[-3, -2, -1, 0, 1, 2, 3, 4, 5, 6, 7],
            (E::Fe, O::Main) => &[2, 3],
            (E::Fe, O::Extended) => &[-4, -2, -1, 0, 1, 4, 5, 6, 7],
            (E::Fe, O::All) => &[-4, -2, -1, 0, 1, 2, 3, 4, 5, 6, 7],
            (E::Co, O::Main) => &[2, 3],
            (E::Co, O::Extended) => &[-3, -1, 0, 1, 4, 5],
            (E::Co, O::All) => &[-3, -1, 0, 1, 2, 3, 4, 5],
            (E::Ni, O::Main) => &[2],
            (E::Ni, O::Extended) => &[-2, -1, 0, 1, 3, 4],
            (E::Ni, O::All) => &[-2, -1, 0, 1, 2, 3, 4],
            (E::Cu, O::Main) => &[1, 2],
            (E::Cu, O::Extended) => &[-2, 0, 3, 4],
            (E::Cu, O::All) => &[-2, 0, 1, 2, 3, 4],
            (E::Zn, O::Main) => &[2],
            (E::Zn, O::Extended) => &[-2, 0, 1],
            (E::Zn, O::All) => &[-2, 0, 1, 2],
            (E::Ga, O::Main) => &[3],
            (E::Ga, O::Extended) => &[-5, -4, -3, -2, -1, 0, 1, 2],
            (E::Ga, O::All) => &[-5, -4, -3, -2, -1, 0, 1, 2, 3],
            (E::Ge, O::Main) => &[-4, 2, 4],
            (E::Ge, O::Extended) => &[-3, -2, -1, 0, 1, 3],
            (E::Ge, O::All) => &[-4, -3, -2, -1, 0, 1, 2, 3, 4],
            (E::As, O::Main) => &[-3, 3, 5],
            (E::As, O::Extended) => &[-2, -1, 0, 1, 2, 4],
            (E::As, O::All) => &[-3, -2, -1, 0, 1, 2, 3, 4, 5],
            (E::Se, O::Main) => &[-2, 2, 4, 6],
            (E::Se, O::Extended) => &[-1, 0, 1, 3, 5],
            (E::Se, O::All) => &[-2, -1, 0, 1, 2, 3, 4, 5, 6],
            (E::Br, O::Main) => &[-1, 1, 3, 5],
            (E::Br, O::Extended) => &[2, 4, 7],
            (E::Br, O::All) => &[-1, 1, 2, 3, 4, 5, 7],
            (E::Kr, O::Main) => &[0],
            (E::Kr, O::Extended) => &[1, 2],
            (E::Kr, O::All) => &[0, 1, 2],
            (E::Rb, O::Main) => &[1],
            (E::Rb, O::Extended) => &[-1],
            (E::Rb, O::All) => &[-1, 1],
            (E::Sr, O::Main) => &[2],
            (E::Sr, O::Extended) => &[1],
            (E::Sr, O::All) => &[1, 2],
            (E::Y, O::Main) => &[3],
            (E::Y, O::Extended) => &[0, 1, 2],
            (E::Y, O::All) => &[0, 1, 2, 3],
            (E::Zr, O::Main) => &[4],
            (E::Zr, O::Extended) => &[-2, 0, 1, 2, 3],
            (E::Zr, O::All) => &[-2, 0, 1, 2, 3, 4],
            (E::Nb, O::Main) => &[5],
            (E::Nb, O::Extended) => &[-3, -1, 0, 1, 2, 3, 4],
            (E::Nb, O::All) => &[-3, -1, 0, 1, 2, 3, 4, 5],
            (E::Mo, O::Main) => &[4, 6],
            (E::Mo, O::Extended) => &[-4, -2, -1, 0, 1, 2, 3, 5],
            (E::Mo, O::All) => &[-4, -2, -1, 0, 1, 2, 3, 4, 5, 6],
            (E::Tc, O::Main) => &[4, 7],
            (E::Tc, O::Extended) => &[-3, -1, 0, 1, 2, 3, 5, 6],
            (E::Tc, O::All) => &[-3, -1, 0, 1, 2, 3, 4, 5, 6, 7],
            (E::Ru, O::Main) => &[3, 4],
            (E::Ru, O::Extended) => &[-4, -2, 0, 1, 2, 5, 6, 7, 8],
            (E::Ru, O::All) => &[-4, -2, 0, 1, 2, 3, 4, 5, 6, 7, 8],
            (E::Rh, O::Main) => &[3],
            (E::Rh, O::Extended) => &[-3, -1, 0, 1, 2, 4, 5, 6, 7],
            (E::Rh, O::All) => &[-3, -1, 0, 1, 2, 3, 4, 5, 6, 7],
            (E::Pd, O::Main) => &[0, 2, 4],
            (E::Pd, O::Extended) => &[1, 3, 5],
            (E::Pd, O::All) => &[0, 1, 2, 3, 4, 5],
            (E::Ag, O::Main) => &[1],
            (E::Ag, O::Extended) => &[-2, -1, 0, 2, 3],
            (E::Ag, O::All) => &[-2, -1, 0, 1, 2, 3],
            (E::Cd, O::Main) => &[2],
            (E::Cd, O::Extended) => &[-2, 1],
            (E::Cd, O::All) => &[-2, 1, 2],
            (E::In, O::Main) => &[3],
            (E::In, O::Extended) => &[-5, -2, -1, 0, 1, 2],
            (E::In, O::All) => &[-5, -2, -1, 0, 1, 2, 3],
            (E::Sn, O::Main) => &[-4, 2, 4],
            (E::Sn, O::Extended) => &[-3, -2, -1, 0, 1, 3],
            (E::Sn, O::All) => &[-4, -3, -2, -1, 0, 1, 2, 3, 4],
            (E::Sb, O::Main) => &[-3, 3, 5],
            (E::Sb, O::Extended) => &[-2, -1, 0, 1, 2, 4],
            (E::Sb, O::All) => &[-3, -2, -1, 0, 1, 2, 3, 4, 5],
            (E::Te, O::Main) => &[-2, 2, 4, 6],
            (E::Te, O::Extended) => &[-1, 0, 1, 3, 5],
            (E::Te, O::All) => &[-2, -1, 0, 1, 2, 3, 4, 5, 6],
            (E::I, O::Main) => &[-1, 1, 3, 5, 7],
            (E::I, O::Extended) => &[2, 4, 6],
            (E::I, O::All) => &[-1, 1, 2, 3, 4, 5, 6, 7],
            (E::Xe, O::Main) => &[0],
            (E::Xe, O::Extended) => &[2, 4, 6, 8],
            (E::Xe, O::All) => &[0, 2, 4, 6, 8],
            (E::Cs, O::Main) => &[1],
            (E::Cs, O::Extended) => &[-1],
            (E::Cs, O::All) => &[-1, 1],
            (E::Ba, O::Main) => &[2],
            (E::Ba, O::Extended) => &[1],
            (E::Ba, O::All) => &[1, 2],
            (E::La, O::Main) => &[3],
            (E::La, O::Extended) => &[0, 1, 2],
            (E::La, O::All) => &[0, 1, 2, 3],
            (E::Ce, O::Main) => &[3, 4],
            (E::Ce, O::Extended) => &[2],
            (E::Ce, O::All) => &[2, 3, 4],
            (E::Pr, O::Main) => &[3],
            (E::Pr, O::Extended) => &[0, 1, 2, 4, 5],
            (E::Pr, O::All) => &[0, 1, 2, 3, 4, 5],
            (E::Nd, O::Main) => &[3],
            (E::Nd, O::Extended) => &[0, 2, 4],
            (E::Nd, O::All) => &[0, 2, 3, 4],
            (E::Pm, O::Main) => &[3],
            (E::Pm, O::Extended) => &[2],
            (E::Pm, O::All) => &[2, 3],
            (E::Sm, O::Main) => &[3],
            (E::Sm, O::Extended) => &[0, 1, 2],
            (E::Sm, O::All) => &[0, 1, 2, 3],
            (E::Eu, O::Main) => &[2, 3],
            (E::Eu, O::Extended) => &[0],
            (E::Eu, O::All) => &[0, 2, 3],
            (E::Gd, O::Main) => &[3],
            (E::Gd, O::Extended) => &[0, 1, 2],
            (E::Gd, O::All) => &[0, 1, 2, 3],
            (E::Tb, O::Main) => &[3],
            (E::Tb, O::Extended) => &[0, 1, 2, 4],
            (E::Tb, O::All) => &[0, 1, 2, 3, 4],
            (E::Dy, O::Main) => &[3],
            (E::Dy, O::Extended) => &[0, 2, 4],
            (E::Dy, O::All) => &[0, 2, 3, 4],
            (E::Ho, O::Main) => &[3],
            (E::Ho, O::Extended) => &[0, 2],
            (E::Ho, O::All) => &[0, 2, 3],
            (E::Er, O::Main) => &[3],
            (E::Er, O::Extended) => &[0, 2],
            (E::Er, O::All) => &[0, 2, 3],
            (E::Tm, O::Main) => &[3],
            (E::Tm, O::Extended) => &[0, 1, 2],
            (E::Tm, O::All) => &[0, 1, 2, 3],
            (E::Yb, O::Main) => &[3],
            (E::Yb, O::Extended) => &[0, 1, 2],
            (E::Yb, O::All) => &[0, 1, 2, 3],
            (E::Lu, O::Main) => &[3],
            (E::Lu, O::Extended) => &[0, 2],
            (E::Lu, O::All) => &[0, 2, 3],
            (E::Hf, O::Main) => &[4],
            (E::Hf, O::Extended) => &[-2, 0, 1, 2, 3],
            (E::Hf, O::All) => &[-2, 0, 1, 2, 3, 4],
            (E::Ta, O::Main) => &[5],
            (E::Ta, O::Extended) => &[-3, -1, 0, 1, 2, 3, 4],
            (E::Ta, O::All) => &[-3, -1, 0, 1, 2, 3, 4, 5],
            (E::W, O::Main) => &[4, 6],
            (E::W, O::Extended) => &[-4, -2, -1, 0, 1, 2, 3, 5],
            (E::W, O::All) => &[-4, -2, -1, 0, 1, 2, 3, 4, 5, 6],
            (E::Re, O::Main) => &[4, 7],
            (E::Re, O::Extended) => &[-3, -1, 0, 1, 2, 3, 5, 6],
            (E::Re, O::All) => &[-3, -1, 0, 1, 2, 3, 4, 5, 6, 7],
            (E::Os, O::Main) => &[4],
            (E::Os, O::Extended) => &[-4, -2, -1, 0, 1, 2, 3, 5, 6, 7, 8],
            (E::Os, O::All) => &[-4, -2, -1, 0, 1, 2, 3, 4, 5, 6, 7, 8],
            (E::Ir, O::Main) => &[3, 4],
            (E::Ir, O::Extended) => &[-3, -1, 0, 1, 2, 5, 6, 7, 8, 9],
            (E::Ir, O::All) => &[-3, -1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
            (E::Pt, O::Main) => &[2, 4],
            (E::Pt, O::Extended) => &[-3, -2, -1, 0, 1, 3, 5, 6],
            (E::Pt, O::All) => &[-3, -2, -1, 0, 1, 2, 3, 4, 5, 6],
            (E::Au, O::Main) => &[1, 3],
            (E::Au, O::Extended) => &[-3, -2, -1, 0, 2, 5],
            (E::Au, O::All) => &[-3, -2, -1, 0, 1, 2, 3, 5],
            (E::Hg, O::Main) => &[1, 2],
            (E::Hg, O::Extended) => &[-2],
            (E::Hg, O::All) => &[-2, 1, 2],
            (E::Tl, O::Main) => &[1, 3],
            (E::Tl, O::Extended) => &[-5, -2, -1, 2],
            (E::Tl, O::All) => &[-5, -2, -1, 1, 2, 3],
            (E::Pb, O::Main) => &[2, 4],
            (E::Pb, O::Extended) => &[-4, -2, -1, 0, 1, 3],
            (E::Pb, O::All) => &[-4, -2, -1, 0, 1, 2, 3, 4],
            (E::Bi, O::Main) => &[3],
            (E::Bi, O::Extended) => &[-3, -2, -1, 0, 1, 2, 4, 5],
            (E::Bi, O::All) => &[-3, -2, -1, 0, 1, 2, 3, 4, 5],
            (E::Po, O::Main) => &[-2, 2, 4],
            (E::Po, O::Extended) => &[5, 6],
            (E::Po, O::All) => &[-2, 2, 4, 5, 6],
            (E::At, O::Main) => &[-1, 1],
            (E::At, O::Extended) => &[3, 5, 7],
            (E::At, O::All) => &[-1, 1, 3, 5, 7],
            (E::Rn, O::Main) => &[2],
            (E::Rn, O::Extended) => &[6],
            (E::Rn, O::All) => &[2, 6],
            (E::Fr, O::Main) => &[1],
            (E::Fr, O::Extended) => &[],
            (E::Fr, O::All) => &[1],
            (E::Ra, O::Main) => &[2],
            (E::Ra, O::Extended) => &[],
            (E::Ra, O::All) => &[2],
            (E::Ac, O::Main) => &[3],
            (E::Ac, O::Extended) => &[2],
            (E::Ac, O::All) => &[2, 3],
            (E::Th, O::Main) => &[4],
            (E::Th, O::Extended) => &[-1, 1, 2, 3],
            (E::Th, O::All) => &[-1, 1, 2, 3, 4],
            (E::Pa, O::Main) => &[5],
            (E::Pa, O::Extended) => &[2, 3, 4],
            (E::Pa, O::All) => &[2, 3, 4, 5],
            (E::U, O::Main) => &[4, 6],
            (E::U, O::Extended) => &[-1, 1, 2, 3, 5],
            (E::U, O::All) => &[-1, 1, 2, 3, 4, 5, 6],
            (E::Np, O::Main) => &[5],
            (E::Np, O::Extended) => &[2, 3, 4, 6, 7],
            (E::Np, O::All) => &[2, 3, 4, 5, 6, 7],
            (E::Pu, O::Main) => &[4],
            (E::Pu, O::Extended) => &[2, 3, 5, 6, 7, 8],
            (E::Pu, O::All) => &[2, 3, 4, 5, 6, 7, 8],
            (E::Am, O::Main) => &[3],
            (E::Am, O::Extended) => &[2, 4, 5, 6, 7],
            (E::Am, O::All) => &[2, 3, 4, 5, 6, 7],
            (E::Cm, O::Main) => &[3],
            (E::Cm, O::Extended) => &[4, 5, 6],
            (E::Cm, O::All) => &[3, 4, 5, 6],
            (E::Bk, O::Main) => &[3],
            (E::Bk, O::Extended) => &[2, 4, 5],
            (E::Bk, O::All) => &[2, 3, 4, 5],
            (E::Cf, O::Main) => &[3],
            (E::Cf, O::Extended) => &[2, 4, 5],
            (E::Cf, O::All) => &[2, 3, 4, 5],
            (E::Es, O::Main) => &[3],
            (E::Es, O::Extended) => &[2, 4],
            (E::Es, O::All) => &[2, 3, 4],
            (E::Fm, O::Main) => &[3],
            (E::Fm, O::Extended) => &[2],
            (E::Fm, O::All) => &[2, 3],
            (E::Md, O::Main) => &[3],
            (E::Md, O::Extended) => &[2],
            (E::Md, O::All) => &[2, 3],
            (E::No, O::Main) => &[2],
            (E::No, O::Extended) => &[3],
            (E::No, O::All) => &[2, 3],
            (E::Lr, O::Main) => &[3],
            (E::Lr, O::Extended) => &[],
            (E::Lr, O::All) => &[3],
            (E::Rf, O::Main) => &[4],
            (E::Rf, O::Extended) => &[],
            (E::Rf, O::All) => &[4],
            (E::Db, O::Main) => &[5],
            (E::Db, O::Extended) => &[],
            (E::Db, O::All) => &[5],
            (E::Sg, O::Main) => &[6],
            (E::Sg, O::Extended) => &[0],
            (E::Sg, O::All) => &[0, 6],
            (E::Bh, O::Main) => &[7],
            (E::Bh, O::Extended) => &[],
            (E::Bh, O::All) => &[7],
            (E::Hs, O::Main) => &[8],
            (E::Hs, O::Extended) => &[],
            (E::Hs, O::All) => &[8],
            (E::Mt, O::Main) => &[],
            (E::Mt, O::Extended) => &[],
            (E::Mt, O::All) => &[],
            (E::Ds, O::Main) => &[],
            (E::Ds, O::Extended) => &[],
            (E::Ds, O::All) => &[],
            (E::Rg, O::Main) => &[],
            (E::Rg, O::Extended) => &[],
            (E::Rg, O::All) => &[],
            (E::Cn, O::Main) => &[2],
            (E::Cn, O::Extended) => &[],
            (E::Cn, O::All) => &[2],
            (E::Nh, O::Main) => &[],
            (E::Nh, O::Extended) => &[],
            (E::Nh, O::All) => &[],
            (E::Fl, O::Main) => &[],
            (E::Fl, O::Extended) => &[],
            (E::Fl, O::All) => &[],
            (E::Mc, O::Main) => &[],
            (E::Mc, O::Extended) => &[],
            (E::Mc, O::All) => &[],
            (E::Lv, O::Main) => &[],
            (E::Lv, O::Extended) => &[],
            (E::Lv, O::All) => &[],
            (E::Ts, O::Main) => &[],
            (E::Ts, O::Extended) => &[],
            (E::Ts, O::All) => &[],
            (E::Og, O::Main) => &[],
            (E::Og, O::Extended) => &[],
            (E::Og, O::All) => &[],
        }
    }
}
