<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Symfony\Component\TypeInfo\Tests\TypeResolver;

use PHPUnit\Framework\TestCase;
use Symfony\Component\TypeInfo\Exception\UnsupportedException;
use Symfony\Component\TypeInfo\Tests\Fixtures\ReflectionExtractableDummy;
use Symfony\Component\TypeInfo\Tests\Fixtures\ReflectionExtractableDummyUsingTrait;
use Symfony\Component\TypeInfo\Tests\Fixtures\ReflectionExtractableTrait;
use Symfony\Component\TypeInfo\Type;
use Symfony\Component\TypeInfo\TypeContext\TypeContextFactory;
use Symfony\Component\TypeInfo\TypeResolver\ReflectionReturnTypeResolver;
use Symfony\Component\TypeInfo\TypeResolver\ReflectionTypeResolver;

class ReflectionReturnTypeResolverTest extends TestCase
{
    private ReflectionReturnTypeResolver $resolver;

    protected function setUp(): void
    {
        $this->resolver = new ReflectionReturnTypeResolver(new ReflectionTypeResolver(), new TypeContextFactory());
    }

    public function testCannotResolveNonReflectionFunctionAbstract()
    {
        $this->expectException(UnsupportedException::class);
        $this->resolver->resolve(123);
    }

    public function testCannotResolveReflectionFunctionAbstractWithoutType()
    {
        $this->expectException(UnsupportedException::class);

        $reflectionClass = new \ReflectionClass(ReflectionExtractableDummy::class);
        $reflectionFunction = $reflectionClass->getMethod('getNothing');

        $this->resolver->resolve($reflectionFunction);
    }

    public function testCannotResolveReflectionFunctionWithoutType()
    {
        $reflectionFunction = new \ReflectionFunction('fopen');

        $this->expectException(UnsupportedException::class);
        $this->expectExceptionMessage('Cannot resolve type for "fopen()".');

        $this->resolver->resolve($reflectionFunction);
    }

    public function testResolve()
    {
        $reflectionClass = new \ReflectionClass(ReflectionExtractableDummy::class);
        $reflectionFunction = $reflectionClass->getMethod('getBuiltin');

        $this->assertEquals(Type::int(), $this->resolver->resolve($reflectionFunction));
    }

    public function testResolveSelfFromClassWithoutContext()
    {
        $reflectionClass = new \ReflectionClass(ReflectionExtractableDummy::class);
        $reflectionFunction = $reflectionClass->getMethod('getSelf');

        $this->assertEquals(Type::object(ReflectionExtractableDummy::class), $this->resolver->resolve($reflectionFunction));
    }

    public function testResolveSelfFromTraitWithoutContext()
    {
        $reflectionClass = new \ReflectionClass(ReflectionExtractableTrait::class);
        $reflectionFunction = $reflectionClass->getMethod('getSelf');

        $this->assertEquals(Type::object(ReflectionExtractableTrait::class), $this->resolver->resolve($reflectionFunction));
    }

    public function testResolveSelfFromTraitWithClassContext()
    {
        $reflectionClass = new \ReflectionClass(ReflectionExtractableTrait::class);
        $reflectionFunction = $reflectionClass->getMethod('getSelf');

        $typeContext = (new TypeContextFactory())->createFromClassName(ReflectionExtractableDummyUsingTrait::class);

        $this->assertEquals(Type::object(ReflectionExtractableDummyUsingTrait::class), $this->resolver->resolve($reflectionFunction, $typeContext));
    }
}
