/* SPDX-License-Identifier: GPL-2.0-only */
/****************************************************************************
 * Driver for Solarflare network controllers and boards
 * Copyright 2009-2018 Solarflare Communications Inc.
 * Copyright 2019-2020 Xilinx Inc.
 */


#ifndef MCDI_PCOL_H
#define MCDI_PCOL_H

/* Values to be written into FMCR_CZ_RESET_STATE_REG to control boot. */
/* Power-on reset state */
#define MC_FW_STATE_POR (1)
/* If this is set in MC_RESET_STATE_REG then it should be
 * possible to jump into IMEM without loading code from flash. */
#define MC_FW_WARM_BOOT_OK (2)
/* The MC main image has started to boot. */
#define MC_FW_STATE_BOOTING (4)
/* The Scheduler has started. */
#define MC_FW_STATE_SCHED (8)
/* If this is set in MC_RESET_STATE_REG then it should be
 * possible to jump into IMEM without loading code from flash.
 * Unlike a warm boot, assume DMEM has been reloaded, so that
 * the MC persistent data must be reinitialised. */
#define MC_FW_TEPID_BOOT_OK (16)
/* We have entered the main firmware via recovery mode.  This
 * means that MC persistent data must be reinitialised, but that
 * we shouldn't touch PCIe config. */
#define MC_FW_RECOVERY_MODE_PCIE_INIT_OK (32)
/* BIST state has been initialized */
#define MC_FW_BIST_INIT_OK (128)

/* Siena MC shared memmory offsets */
/* The 'doorbell' addresses are hard-wired to alert the MC when written */
#define	MC_SMEM_P0_DOORBELL_OFST	0x000
#define	MC_SMEM_P1_DOORBELL_OFST	0x004
/* The rest of these are firmware-defined */
#define	MC_SMEM_P0_PDU_OFST		0x008
#define	MC_SMEM_P1_PDU_OFST		0x108
#define	MC_SMEM_PDU_LEN			0x100
#define	MC_SMEM_P0_PTP_TIME_OFST	0x7f0
#define	MC_SMEM_P0_STATUS_OFST		0x7f8
#define	MC_SMEM_P1_STATUS_OFST		0x7fc

/* Values to be written to the per-port status dword in shared
 * memory on reboot and assert */
#define MC_STATUS_DWORD_REBOOT (0xb007b007)
#define MC_STATUS_DWORD_ASSERT (0xdeaddead)

/* Check whether an mcfw version (in host order) belongs to a bootloader */
#define MC_FW_VERSION_IS_BOOTLOADER(_v) (((_v) >> 16) == 0xb007)

/* The current version of the MCDI protocol.
 *
 * Note that the ROM burnt into the card only talks V0, so at the very
 * least every driver must support version 0 and MCDI_PCOL_VERSION
 */
#define MCDI_PCOL_VERSION 2

/* Unused commands: 0x23, 0x27, 0x30, 0x31 */

/* MCDI version 1
 *
 * Each MCDI request starts with an MCDI_HEADER, which is a 32bit
 * structure, filled in by the client.
 *
 *       0       7  8     16    20     22  23  24    31
 *      | CODE | R | LEN | SEQ | Rsvd | E | R | XFLAGS |
 *               |                      |   |
 *               |                      |   \--- Response
 *               |                      \------- Error
 *               \------------------------------ Resync (always set)
 *
 * The client writes its request into MC shared memory, and rings the
 * doorbell. Each request is completed either by the MC writing
 * back into shared memory, or by writing out an event.
 *
 * All MCDI commands support completion by shared memory response. Each
 * request may also contain additional data (accounted for by HEADER.LEN),
 * and some responses may also contain additional data (again, accounted
 * for by HEADER.LEN).
 *
 * Some MCDI commands support completion by event, in which any associated
 * response data is included in the event.
 *
 * The protocol requires one response to be delivered for every request; a
 * request should not be sent unless the response for the previous request
 * has been received (either by polling shared memory, or by receiving
 * an event).
 */

/** Request/Response structure */
#define MCDI_HEADER_OFST 0
#define MCDI_HEADER_CODE_LBN 0
#define MCDI_HEADER_CODE_WIDTH 7
#define MCDI_HEADER_RESYNC_LBN 7
#define MCDI_HEADER_RESYNC_WIDTH 1
#define MCDI_HEADER_DATALEN_LBN 8
#define MCDI_HEADER_DATALEN_WIDTH 8
#define MCDI_HEADER_SEQ_LBN 16
#define MCDI_HEADER_SEQ_WIDTH 4
#define MCDI_HEADER_RSVD_LBN 20
#define MCDI_HEADER_RSVD_WIDTH 1
#define MCDI_HEADER_NOT_EPOCH_LBN 21
#define MCDI_HEADER_NOT_EPOCH_WIDTH 1
#define MCDI_HEADER_ERROR_LBN 22
#define MCDI_HEADER_ERROR_WIDTH 1
#define MCDI_HEADER_RESPONSE_LBN 23
#define MCDI_HEADER_RESPONSE_WIDTH 1
#define MCDI_HEADER_XFLAGS_LBN 24
#define MCDI_HEADER_XFLAGS_WIDTH 8
/* Request response using event */
#define MCDI_HEADER_XFLAGS_EVREQ 0x01
/* Request (and signal) early doorbell return */
#define MCDI_HEADER_XFLAGS_DBRET 0x02

/* Maximum number of payload bytes */
#define MCDI_CTL_SDU_LEN_MAX_V1 0xfc
#define MCDI_CTL_SDU_LEN_MAX_V2 0x400

#define MCDI_CTL_SDU_LEN_MAX MCDI_CTL_SDU_LEN_MAX_V2


/* The MC can generate events for two reasons:
 *   - To advance a shared memory request if XFLAGS_EVREQ was set
 *   - As a notification (link state, i2c event), controlled
 *     via MC_CMD_LOG_CTRL
 *
 * Both events share a common structure:
 *
 *  0      32     33      36    44     52     60
 * | Data | Cont | Level | Src | Code | Rsvd |
 *           |
 *           \ There is another event pending in this notification
 *
 * If Code==CMDDONE, then the fields are further interpreted as:
 *
 *   - LEVEL==INFO    Command succeeded
 *   - LEVEL==ERR     Command failed
 *
 *    0     8         16      24     32
 *   | Seq | Datalen | Errno | Rsvd |
 *
 *   These fields are taken directly out of the standard MCDI header, i.e.,
 *   LEVEL==ERR, Datalen == 0 => Reboot
 *
 * Events can be squirted out of the UART (using LOG_CTRL) without a
 * MCDI header.  An event can be distinguished from a MCDI response by
 * examining the first byte which is 0xc0.  This corresponds to the
 * non-existent MCDI command MC_CMD_DEBUG_LOG.
 *
 *      0         7        8
 *     | command | Resync |     = 0xc0
 *
 * Since the event is written in big-endian byte order, this works
 * providing bits 56-63 of the event are 0xc0.
 *
 *      56     60  63
 *     | Rsvd | Code |    = 0xc0
 *
 * Which means for convenience the event code is 0xc for all MC
 * generated events.
 */
#define FSE_AZ_EV_CODE_MCDI_EVRESPONSE 0xc



#define MC_CMD_ERR_CODE_OFST 0
#define MC_CMD_ERR_PROXY_PENDING_HANDLE_OFST 4

/* We define 8 "escape" commands to allow
   for command number space extension */

#define MC_CMD_CMD_SPACE_ESCAPE_0	      0x78
#define MC_CMD_CMD_SPACE_ESCAPE_1	      0x79
#define MC_CMD_CMD_SPACE_ESCAPE_2	      0x7A
#define MC_CMD_CMD_SPACE_ESCAPE_3	      0x7B
#define MC_CMD_CMD_SPACE_ESCAPE_4	      0x7C
#define MC_CMD_CMD_SPACE_ESCAPE_5	      0x7D
#define MC_CMD_CMD_SPACE_ESCAPE_6	      0x7E
#define MC_CMD_CMD_SPACE_ESCAPE_7	      0x7F

/* Vectors in the boot ROM */
/* Point to the copycode entry point. */
#define SIENA_MC_BOOTROM_COPYCODE_VEC (0x800 - 3 * 0x4)
#define HUNT_MC_BOOTROM_COPYCODE_VEC (0x8000 - 3 * 0x4)
#define MEDFORD_MC_BOOTROM_COPYCODE_VEC (0x10000 - 3 * 0x4)
/* Points to the recovery mode entry point. Misnamed but kept for compatibility. */
#define SIENA_MC_BOOTROM_NOFLASH_VEC (0x800 - 2 * 0x4)
#define HUNT_MC_BOOTROM_NOFLASH_VEC (0x8000 - 2 * 0x4)
#define MEDFORD_MC_BOOTROM_NOFLASH_VEC (0x10000 - 2 * 0x4)
/* Points to the recovery mode entry point. Same as above, but the right name. */
#define SIENA_MC_BOOTROM_RECOVERY_VEC (0x800 - 2 * 0x4)
#define HUNT_MC_BOOTROM_RECOVERY_VEC (0x8000 - 2 * 0x4)
#define MEDFORD_MC_BOOTROM_RECOVERY_VEC (0x10000 - 2 * 0x4)

/* Points to noflash mode entry point. */
#define MEDFORD_MC_BOOTROM_REAL_NOFLASH_VEC (0x10000 - 4 * 0x4)

/* The command set exported by the boot ROM (MCDI v0) */
#define MC_CMD_GET_VERSION_V0_SUPPORTED_FUNCS {		\
	(1 << MC_CMD_READ32)	|			\
	(1 << MC_CMD_WRITE32)	|			\
	(1 << MC_CMD_COPYCODE)	|			\
	(1 << MC_CMD_GET_VERSION),			\
	0, 0, 0 }

#define MC_CMD_SENSOR_INFO_OUT_OFFSET_OFST(_x)		\
	(MC_CMD_SENSOR_ENTRY_OFST + (_x))

#define MC_CMD_DBI_WRITE_IN_ADDRESS_OFST(n)		\
	(MC_CMD_DBI_WRITE_IN_DBIWROP_OFST +		\
	 MC_CMD_DBIWROP_TYPEDEF_ADDRESS_OFST +		\
	 (n) * MC_CMD_DBIWROP_TYPEDEF_LEN)

#define MC_CMD_DBI_WRITE_IN_BYTE_MASK_OFST(n)		\
	(MC_CMD_DBI_WRITE_IN_DBIWROP_OFST +		\
	 MC_CMD_DBIWROP_TYPEDEF_BYTE_MASK_OFST +	\
	 (n) * MC_CMD_DBIWROP_TYPEDEF_LEN)

#define MC_CMD_DBI_WRITE_IN_VALUE_OFST(n)		\
	(MC_CMD_DBI_WRITE_IN_DBIWROP_OFST +		\
	 MC_CMD_DBIWROP_TYPEDEF_VALUE_OFST +		\
	 (n) * MC_CMD_DBIWROP_TYPEDEF_LEN)

/* This may be ORed with an EVB_PORT_ID_xxx constant to pass a non-default
 * stack ID (which must be in the range 1-255) along with an EVB port ID.
 */
#define EVB_STACK_ID(n)  (((n) & 0xff) << 16)


/* Version 2 adds an optional argument to error returns: the errno value
 * may be followed by the (0-based) number of the first argument that
 * could not be processed.
 */
#define MC_CMD_ERR_ARG_OFST 4

/* MC_CMD_ERR enum: Public MCDI error codes. Error codes that correspond to
 * POSIX errnos should use the same numeric values that linux does. Error codes
 * specific to Solarflare firmware should use values in the range 0x1000 -
 * 0x10ff. The range 0x2000 - 0x20ff is reserved for private error codes (see
 * MC_CMD_ERR_PRIV below).
 */
/* enum: Operation not permitted. */
#define          MC_CMD_ERR_EPERM 0x1
/* enum: Non-existent command target */
#define          MC_CMD_ERR_ENOENT 0x2
/* enum: assert() has killed the MC */
#define          MC_CMD_ERR_EINTR 0x4
/* enum: I/O failure */
#define          MC_CMD_ERR_EIO 0x5
/* enum: Already exists */
#define          MC_CMD_ERR_EEXIST 0x6
/* enum: Try again */
#define          MC_CMD_ERR_EAGAIN 0xb
/* enum: Out of memory */
#define          MC_CMD_ERR_ENOMEM 0xc
/* enum: Caller does not hold required locks */
#define          MC_CMD_ERR_EACCES 0xd
/* enum: Resource is currently unavailable (e.g. lock contention) */
#define          MC_CMD_ERR_EBUSY 0x10
/* enum: No such device */
#define          MC_CMD_ERR_ENODEV 0x13
/* enum: Invalid argument to target */
#define          MC_CMD_ERR_EINVAL 0x16
/* enum: No space */
#define          MC_CMD_ERR_ENOSPC 0x1c
/* enum: Read-only */
#define          MC_CMD_ERR_EROFS 0x1e
/* enum: Broken pipe */
#define          MC_CMD_ERR_EPIPE 0x20
/* enum: Out of range */
#define          MC_CMD_ERR_ERANGE 0x22
/* enum: Non-recursive resource is already acquired */
#define          MC_CMD_ERR_EDEADLK 0x23
/* enum: Operation not implemented */
#define          MC_CMD_ERR_ENOSYS 0x26
/* enum: Operation timed out */
#define          MC_CMD_ERR_ETIME 0x3e
/* enum: Link has been severed */
#define          MC_CMD_ERR_ENOLINK 0x43
/* enum: Protocol error */
#define          MC_CMD_ERR_EPROTO 0x47
/* enum: Bad message */
#define          MC_CMD_ERR_EBADMSG 0x4a
/* enum: Operation not supported */
#define          MC_CMD_ERR_ENOTSUP 0x5f
/* enum: Address not available */
#define          MC_CMD_ERR_EADDRNOTAVAIL 0x63
/* enum: Not connected */
#define          MC_CMD_ERR_ENOTCONN 0x6b
/* enum: Operation already in progress */
#define          MC_CMD_ERR_EALREADY 0x72
/* enum: Stale handle. The handle references a resource that no longer exists.
 */
#define          MC_CMD_ERR_ESTALE 0x74
/* enum: Resource allocation failed. */
#define          MC_CMD_ERR_ALLOC_FAIL 0x1000
/* enum: V-adaptor not found. */
#define          MC_CMD_ERR_NO_VADAPTOR 0x1001
/* enum: EVB port not found. */
#define          MC_CMD_ERR_NO_EVB_PORT 0x1002
/* enum: V-switch not found. */
#define          MC_CMD_ERR_NO_VSWITCH 0x1003
/* enum: Too many VLAN tags. */
#define          MC_CMD_ERR_VLAN_LIMIT 0x1004
/* enum: Bad PCI function number. */
#define          MC_CMD_ERR_BAD_PCI_FUNC 0x1005
/* enum: Invalid VLAN mode. */
#define          MC_CMD_ERR_BAD_VLAN_MODE 0x1006
/* enum: Invalid v-switch type. */
#define          MC_CMD_ERR_BAD_VSWITCH_TYPE 0x1007
/* enum: Invalid v-port type. */
#define          MC_CMD_ERR_BAD_VPORT_TYPE 0x1008
/* enum: MAC address exists. */
#define          MC_CMD_ERR_MAC_EXIST 0x1009
/* enum: Slave core not present */
#define          MC_CMD_ERR_SLAVE_NOT_PRESENT 0x100a
/* enum: The datapath is disabled. */
#define          MC_CMD_ERR_DATAPATH_DISABLED 0x100b
/* enum: The requesting client is not a function */
#define          MC_CMD_ERR_CLIENT_NOT_FN 0x100c
/* enum: The requested operation might require the command to be passed between
 * MCs, and the transport doesn't support that. Should only ever been seen over
 * the UART.
 */
#define          MC_CMD_ERR_TRANSPORT_NOPROXY 0x100d
/* enum: VLAN tag(s) exists */
#define          MC_CMD_ERR_VLAN_EXIST 0x100e
/* enum: No MAC address assigned to an EVB port */
#define          MC_CMD_ERR_NO_MAC_ADDR 0x100f
/* enum: Notifies the driver that the request has been relayed to an admin
 * function for authorization. The driver should wait for a PROXY_RESPONSE
 * event and then resend its request. This error code is followed by a 32-bit
 * handle that helps matching it with the respective PROXY_RESPONSE event.
 */
#define          MC_CMD_ERR_PROXY_PENDING 0x1010
/* enum: The request cannot be passed for authorization because another request
 * from the same function is currently being authorized. The drvier should try
 * again later.
 */
#define          MC_CMD_ERR_PROXY_INPROGRESS 0x1011
/* enum: Returned by MC_CMD_PROXY_COMPLETE if the caller is not the function
 * that has enabled proxying or BLOCK_INDEX points to a function that doesn't
 * await an authorization.
 */
#define          MC_CMD_ERR_PROXY_UNEXPECTED 0x1012
/* enum: This code is currently only used internally in FW. Its meaning is that
 * an operation failed due to lack of SR-IOV privilege. Normally it is
 * translated to EPERM by send_cmd_err(), but it may also be used to trigger
 * some special mechanism for handling such case, e.g. to relay the failed
 * request to a designated admin function for authorization.
 */
#define          MC_CMD_ERR_NO_PRIVILEGE 0x1013
/* enum: Workaround 26807 could not be turned on/off because some functions
 * have already installed filters. See the comment at
 * MC_CMD_WORKAROUND_BUG26807. May also returned for other operations such as
 * sub-variant switching.
 */
#define          MC_CMD_ERR_FILTERS_PRESENT 0x1014
/* enum: The clock whose frequency you've attempted to set doesn't exist on
 * this NIC
 */
#define          MC_CMD_ERR_NO_CLOCK 0x1015
/* enum: Returned by MC_CMD_TESTASSERT if the action that should have caused an
 * assertion failed to do so.
 */
#define          MC_CMD_ERR_UNREACHABLE 0x1016
/* enum: This command needs to be processed in the background but there were no
 * resources to do so. Send it again after a command has completed.
 */
#define          MC_CMD_ERR_QUEUE_FULL 0x1017
/* enum: The operation could not be completed because the PCIe link has gone
 * away. This error code is never expected to be returned over the TLP
 * transport.
 */
#define          MC_CMD_ERR_NO_PCIE 0x1018
/* enum: The operation could not be completed because the datapath has gone
 * away. This is distinct from MC_CMD_ERR_DATAPATH_DISABLED in that the
 * datapath absence may be temporary
 */
#define          MC_CMD_ERR_NO_DATAPATH 0x1019
/* enum: The operation could not complete because some VIs are allocated */
#define          MC_CMD_ERR_VIS_PRESENT 0x101a
/* enum: The operation could not complete because some PIO buffers are
 * allocated
 */
#define          MC_CMD_ERR_PIOBUFS_PRESENT 0x101b

/* PCIE_INTERFACE enum: From EF100 onwards, SFC products can have multiple PCIe
 * interfaces. There is a need to refer to interfaces explicitly from drivers
 * (for example, a management driver on one interface administering a function
 * on another interface). This enumeration provides stable identifiers to all
 * interfaces present on a product. Product documentation will specify which
 * interfaces exist and their associated identifier. In general, drivers,
 * should not assign special meanings to specific values. Instead, behaviour
 * should be determined by NIC configuration, which will identify interfaces
 * where appropriate.
 */
/* enum: Primary host interfaces. Typically (i.e. for all known SFC products)
 * the interface exposed on the edge connector (or form factor equivalent).
 */
#define          PCIE_INTERFACE_HOST_PRIMARY 0x0
/* enum: Riverhead and keystone products have a second PCIe interface to which
 * an on-NIC ARM module is expected to be connected.
 */
#define          PCIE_INTERFACE_NIC_EMBEDDED 0x1
/* enum: The PCIe logical interface 0. It is an alias for HOST_PRIMARY. */
#define          PCIE_INTERFACE_PCIE_HOST_INTF_0 0x0
/* enum: The PCIe logical interface 1. */
#define          PCIE_INTERFACE_PCIE_HOST_INTF_1 0x2
/* enum: The PCIe logical interface 2. */
#define          PCIE_INTERFACE_PCIE_HOST_INTF_2 0x3
/* enum: The PCIe logical interface 3. */
#define          PCIE_INTERFACE_PCIE_HOST_INTF_3 0x4
/* enum: For MCDI commands issued over a PCIe interface, this value is
 * translated into the interface over which the command was issued. Not
 * meaningful for other MCDI transports.
 */
#define          PCIE_INTERFACE_CALLER 0xffffffff

/* MC_CLIENT_ID_SPECIFIER enum */
/* enum: Equivalent to the caller's client ID */
#define          MC_CMD_CLIENT_ID_SELF 0xffffffff

/* MAE_FIELD_SUPPORT_STATUS enum */
/* enum: The NIC does not support this field. The driver must ensure that any
 * mask associated with this field in a match rule is zeroed. The NIC may
 * either reject requests with an invalid mask for such a field, or may assume
 * that the mask is zero. (This category only exists to describe behaviour for
 * fields that a newer driver might know about but that older firmware does
 * not. It is recommended that firmware report MAE_FIELD_FIELD_MATCH_NEVER for
 * all match fields defined at the time of its compilation. If a driver see a
 * field support status value that it does not recognise, it must treat that
 * field as thought the field was reported as MAE_FIELD_SUPPORTED_MATCH_NEVER,
 * and must never set a non-zero mask value for this field.
 */
#define          MAE_FIELD_UNSUPPORTED 0x0
/* enum: The NIC supports this field, but cannot use it in a match rule. The
 * driver must ensure that any mask for such a field in a match rule is zeroed.
 * The NIC will reject requests with an invalid mask for such a field.
 */
#define          MAE_FIELD_SUPPORTED_MATCH_NEVER 0x1
/* enum: The NIC supports this field, and must use it in all match rules. The
 * driver must ensure that any mask for such a field is all ones. The NIC will
 * reject requests with an invalid mask for such a field.
 */
#define          MAE_FIELD_SUPPORTED_MATCH_ALWAYS 0x2
/* enum: The NIC supports this field, and may optionally use it in match rules.
 * The driver must ensure that any mask for such a field is either all zeroes
 * or all ones. The NIC will reject requests with an invalid mask for such a
 * field.
 */
#define          MAE_FIELD_SUPPORTED_MATCH_OPTIONAL 0x3
/* enum: The NIC supports this field, and may optionally use it in match rules.
 * The driver must ensure that any mask for such a field is either all zeroes
 * or a consecutive set of ones following by all zeroes (starting from MSB).
 * The NIC will reject requests with an invalid mask for such a field.
 */
#define          MAE_FIELD_SUPPORTED_MATCH_PREFIX 0x4
/* enum: The NIC supports this field, and may optionally use it in match rules.
 * The driver may provide an arbitrary mask for such a field.
 */
#define          MAE_FIELD_SUPPORTED_MATCH_MASK 0x5

/* MAE_CT_VNI_MODE enum: Controls the layout of the VNI input to the conntrack
 * lookup. (Values are not arbitrary - constrained by table access ABI.)
 */
/* enum: The VNI input to the conntrack lookup will be zero. */
#define          MAE_CT_VNI_MODE_ZERO 0x0
/* enum: The VNI input to the conntrack lookup will be the VNI (VXLAN/Geneve)
 * or VSID (NVGRE) field from the packet.
 */
#define          MAE_CT_VNI_MODE_VNI 0x1
/* enum: The VNI input to the conntrack lookup will be the VLAN ID from the
 * outermost VLAN tag (in bottom 12 bits; top 12 bits zero).
 */
#define          MAE_CT_VNI_MODE_1VLAN 0x2
/* enum: The VNI input to the conntrack lookup will be the VLAN IDs from both
 * VLAN tags (outermost in bottom 12 bits, innermost in top 12 bits).
 */
#define          MAE_CT_VNI_MODE_2VLAN 0x3

/* MAE_FIELD enum: NB: this enum shares namespace with the support status enum.
 */
/* enum: Source mport upon entering the MAE. */
#define          MAE_FIELD_INGRESS_PORT 0x0
#define          MAE_FIELD_MARK 0x1 /* enum */
/* enum: Table ID used in action rule. Initially zero, can be changed in action
 * rule response.
 */
#define          MAE_FIELD_RECIRC_ID 0x2
#define          MAE_FIELD_IS_IP_FRAG 0x3 /* enum */
#define          MAE_FIELD_DO_CT 0x4 /* enum */
#define          MAE_FIELD_CT_HIT 0x5 /* enum */
/* enum: Undefined unless CT_HIT=1. */
#define          MAE_FIELD_CT_MARK 0x6
/* enum: Undefined unless DO_CT=1. */
#define          MAE_FIELD_CT_DOMAIN 0x7
/* enum: Undefined unless CT_HIT=1. */
#define          MAE_FIELD_CT_PRIVATE_FLAGS 0x8
/* enum: 1 if the packet ingressed the NIC from one of the MACs, else 0. */
#define          MAE_FIELD_IS_FROM_NETWORK 0x9
/* enum: 1 if the packet has 1 or more VLAN tags, else 0. */
#define          MAE_FIELD_HAS_OVLAN 0xa
/* enum: 1 if the packet has 2 or more VLAN tags, else 0. */
#define          MAE_FIELD_HAS_IVLAN 0xb
/* enum: 1 if the outer packet has 1 or more VLAN tags, else 0; only present
 * when encap
 */
#define          MAE_FIELD_ENC_HAS_OVLAN 0xc
/* enum: 1 if the outer packet has 2 or more VLAN tags, else 0; only present
 * when encap
 */
#define          MAE_FIELD_ENC_HAS_IVLAN 0xd
/* enum: Packet is IP fragment */
#define          MAE_FIELD_ENC_IP_FRAG 0xe
#define          MAE_FIELD_ETHER_TYPE 0x21 /* enum */
#define          MAE_FIELD_VLAN0_TCI 0x22 /* enum */
#define          MAE_FIELD_VLAN0_PROTO 0x23 /* enum */
#define          MAE_FIELD_VLAN1_TCI 0x24 /* enum */
#define          MAE_FIELD_VLAN1_PROTO 0x25 /* enum */
/* enum: Inner when encap */
#define          MAE_FIELD_ETH_SADDR 0x28
/* enum: Inner when encap */
#define          MAE_FIELD_ETH_DADDR 0x29
/* enum: Inner when encap. NB: IPv4 and IPv6 fields are mutually exclusive. */
#define          MAE_FIELD_SRC_IP4 0x2a
/* enum: Inner when encap */
#define          MAE_FIELD_SRC_IP6 0x2b
/* enum: Inner when encap */
#define          MAE_FIELD_DST_IP4 0x2c
/* enum: Inner when encap */
#define          MAE_FIELD_DST_IP6 0x2d
/* enum: Inner when encap */
#define          MAE_FIELD_IP_PROTO 0x2e
/* enum: Inner when encap */
#define          MAE_FIELD_IP_TOS 0x2f
/* enum: Inner when encap */
#define          MAE_FIELD_IP_TTL 0x30
/* enum: Inner when encap TODO: how this is defined? The raw flags +
 * frag_offset from the packet, or some derived value more amenable to ternary
 * matching? TODO: there was a proposal for driver-allocation fields. The
 * driver would provide some instruction for how to extract given field values,
 * and would be given a field id in return. It could then use that field id in
 * its matches. This feels like it would be extremely hard to implement in
 * hardware, but I mention it for completeness.
 */
#define          MAE_FIELD_IP_FLAGS 0x31
/* enum: Ports (UDP, TCP) Inner when encap */
#define          MAE_FIELD_L4_SPORT 0x32
/* enum: Ports (UDP, TCP) Inner when encap */
#define          MAE_FIELD_L4_DPORT 0x33
/* enum: Inner when encap */
#define          MAE_FIELD_TCP_FLAGS 0x34
/* enum: TCP packet with any of SYN, FIN or RST flag set */
#define          MAE_FIELD_TCP_SYN_FIN_RST 0x35
/* enum: Packet is IP fragment with fragment offset 0 */
#define          MAE_FIELD_IP_FIRST_FRAG 0x36
/* enum: The type of encapsulated used for this packet. Value as per
 * ENCAP_TYPE_*.
 */
#define          MAE_FIELD_ENCAP_TYPE 0x3f
/* enum: The ID of the outer rule that marked this packet as encapsulated.
 * Useful for implicitly matching on outer fields.
 */
#define          MAE_FIELD_OUTER_RULE_ID 0x40
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_ETHER_TYPE 0x41
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_VLAN0_TCI 0x42
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_VLAN0_PROTO 0x43
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_VLAN1_TCI 0x44
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_VLAN1_PROTO 0x45
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_ETH_SADDR 0x48
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_ETH_DADDR 0x49
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_SRC_IP4 0x4a
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_SRC_IP6 0x4b
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_DST_IP4 0x4c
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_DST_IP6 0x4d
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_IP_PROTO 0x4e
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_IP_TOS 0x4f
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_IP_TTL 0x50
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_IP_FLAGS 0x51
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_L4_SPORT 0x52
/* enum: Outer; only present when encap */
#define          MAE_FIELD_ENC_L4_DPORT 0x53
/* enum: VNI (when VXLAN or GENEVE) VSID (when NVGRE) Bottom 24 bits of Key
 * (when L2GRE) Outer; only present when encap
 */
#define          MAE_FIELD_ENC_VNET_ID 0x54

/* MAE_MCDI_ENCAP_TYPE enum: Encapsulation type. Defines how the payload will
 * be parsed to an inner frame. Other values are reserved. Unknown values
 * should be treated same as NONE. (Values are not arbitrary - constrained by
 * table access ABI.)
 */
#define          MAE_MCDI_ENCAP_TYPE_NONE 0x0 /* enum */
/* enum: Don't assume enum aligns with support bitmask... */
#define          MAE_MCDI_ENCAP_TYPE_VXLAN 0x1
#define          MAE_MCDI_ENCAP_TYPE_NVGRE 0x2 /* enum */
#define          MAE_MCDI_ENCAP_TYPE_GENEVE 0x3 /* enum */
#define          MAE_MCDI_ENCAP_TYPE_L2GRE 0x4 /* enum */

/* MAE_MPORT_END enum: Selects which end of the logical link identified by an
 * MPORT_SELECTOR is targeted by an operation.
 */
/* enum: Selects the port on the MAE virtual switch */
#define          MAE_MPORT_END_MAE 0x1
/* enum: Selects the virtual NIC plugged into the MAE switch */
#define          MAE_MPORT_END_VNIC 0x2

/* MAE_COUNTER_TYPE enum: The datapath maintains several sets of counters, each
 * being associated with a different table. Note that the same counter ID may
 * be allocated by different counter blocks, so e.g. AR counter 42 is different
 * from CT counter 42. Generation counts are also type-specific. This value is
 * also present in the header of streaming counter packets, in the IDENTIFIER
 * field (see packetiser packet format definitions). Also note that LACP
 * counter IDs are not allocated individually, instead the counter IDs are
 * directly tied to the LACP balance table indices. These in turn are allocated
 * in large contiguous blocks as a LAG config. Calling MAE_COUNTER_ALLOC/FREE
 * with an LACP counter type will return EPERM.
 */
/* enum: Action Rule counters - can be referenced in AR response. */
#define          MAE_COUNTER_TYPE_AR 0x0
/* enum: Conntrack counters - can be referenced in CT response. */
#define          MAE_COUNTER_TYPE_CT 0x1
/* enum: Outer Rule counters - can be referenced in OR response. */
#define          MAE_COUNTER_TYPE_OR 0x2
/* enum: LACP counters - linked to LACP balance table entries. */
#define          MAE_COUNTER_TYPE_LACP 0x3

/* MAE_COUNTER_ID enum: ID of allocated counter or counter list. */
/* enum: A counter ID that is guaranteed never to represent a real counter or
 * counter list.
 */
#define          MAE_COUNTER_ID_NULL 0xffffffff

/* TABLE_ID enum: Unique IDs for tables. The 32-bit ID values have been
 * structured with bits [31:24] reserved (0), [23:16] indicating which major
 * block the tables belongs to (0=VNIC TX, none currently; 1=MAE; 2=VNIC RX),
 * [15:8] a unique ID within the block, and [7:0] reserved for future
 * variations of the same table. (All of the tables currently defined within
 * the streaming engines are listed here, but this does not imply that they are
 * all supported - MC_CMD_TABLE_LIST returns the list of actually supported
 * tables.) The DPU offload engines' enumerators follow a deliberate pattern:
 * 0x01010000 + is_dpu_net * 0x10000 + is_wr_or_tx * 0x8000 + is_lite_pipe *
 * 0x1000 + oe_engine_type * 0x100 + oe_instance_within_pipe * 0x10
 */
/* enum: Outer_Rule_Table in the MAE - refer to SF-123102-TC. */
#define          TABLE_ID_OUTER_RULE_TABLE 0x10000
/* enum: Outer_Rule_No_CT_Table in the MAE - refer to SF-123102-TC. */
#define          TABLE_ID_OUTER_RULE_NO_CT_TABLE 0x10100
/* enum: Mgmt_Filter_Table in the MAE - refer to SF-123102-TC. */
#define          TABLE_ID_MGMT_FILTER_TABLE 0x10200
/* enum: Conntrack_Table in the MAE - refer to SF-123102-TC. */
#define          TABLE_ID_CONNTRACK_TABLE 0x10300
/* enum: Action_Rule_Table in the MAE - refer to SF-123102-TC. */
#define          TABLE_ID_ACTION_RULE_TABLE 0x10400
/* enum: Mgroup_Default_Action_Set_Table in the MAE - refer to SF-123102-TC. */
#define          TABLE_ID_MGROUP_DEFAULT_ACTION_SET_TABLE 0x10500
/* enum: Encap_Hdr_Part1_Table in the MAE - refer to SF-123102-TC. */
#define          TABLE_ID_ENCAP_HDR_PART1_TABLE 0x10600
/* enum: Encap_Hdr_Part2_Table in the MAE - refer to SF-123102-TC. */
#define          TABLE_ID_ENCAP_HDR_PART2_TABLE 0x10700
/* enum: Replace_Src_MAC_Table in the MAE - refer to SF-123102-TC. */
#define          TABLE_ID_REPLACE_SRC_MAC_TABLE 0x10800
/* enum: Replace_Dst_MAC_Table in the MAE - refer to SF-123102-TC. */
#define          TABLE_ID_REPLACE_DST_MAC_TABLE 0x10900
/* enum: Dst_Mport_VC_Table in the MAE - refer to SF-123102-TC. */
#define          TABLE_ID_DST_MPORT_VC_TABLE 0x10a00
/* enum: LACP_LAG_Config_Table in the MAE - refer to SF-123102-TC. */
#define          TABLE_ID_LACP_LAG_CONFIG_TABLE 0x10b00
/* enum: LACP_Balance_Table in the MAE - refer to SF-123102-TC. */
#define          TABLE_ID_LACP_BALANCE_TABLE 0x10c00
/* enum: Dst_Mport_Host_Chan_Table in the MAE - refer to SF-123102-TC. */
#define          TABLE_ID_DST_MPORT_HOST_CHAN_TABLE 0x10d00
/* enum: VNIC_Rx_Encap_Table in VNIC Rx - refer to SF-123102-TC. */
#define          TABLE_ID_VNIC_RX_ENCAP_TABLE 0x20000
/* enum: Steering_Table in VNIC Rx - refer to SF-123102-TC. */
#define          TABLE_ID_STEERING_TABLE 0x20100
/* enum: RSS_Context_Table in VNIC Rx - refer to SF-123102-TC. */
#define          TABLE_ID_RSS_CONTEXT_TABLE 0x20200
/* enum: Indirection_Table in VNIC Rx - refer to SF-123102-TC. */
#define          TABLE_ID_INDIRECTION_TABLE 0x20300
/* enum: DPU.host read pipe first CRC offload engine profiles - refer to
 * XN-200147-AN.
 */
#define          TABLE_ID_DPU_HOST_RD_CRC0_OE_PROFILE 0x1010000
/* enum: DPU.host read pipe second CRC offload engine profiles - refer to
 * XN-200147-AN.
 */
#define          TABLE_ID_DPU_HOST_RD_CRC1_OE_PROFILE 0x1010010
/* enum: DPU.host write pipe first CRC offload engine profiles - refer to
 * XN-200147-AN.
 */
#define          TABLE_ID_DPU_HOST_WR_CRC0_OE_PROFILE 0x1018000
/* enum: DPU.host write pipe second CRC offload engine profiles - refer to
 * XN-200147-AN.
 */
#define          TABLE_ID_DPU_HOST_WR_CRC1_OE_PROFILE 0x1018010
/* enum: DPU.net 'full' receive pipe CRC offload engine profiles - refer to
 * XN-200147-AN.
 */
#define          TABLE_ID_DPU_NET_RX_CRC0_OE_PROFILE 0x1020000
/* enum: DPU.net 'full' receive pipe first checksum offload engine profiles -
 * refer to XN-200147-AN.
 */
#define          TABLE_ID_DPU_NET_RX_CSUM0_OE_PROFILE 0x1020100
/* enum: DPU.net 'full' receive pipe second checksum offload engine profiles -
 * refer to XN-200147-AN.
 */
#define          TABLE_ID_DPU_NET_RX_CSUM1_OE_PROFILE 0x1020110
/* enum: DPU.net 'full' receive pipe AES-GCM offload engine profiles - refer to
 * XN-200147-AN.
 */
#define          TABLE_ID_DPU_NET_RX_AES_GCM0_OE_PROFILE 0x1020200
/* enum: DPU.net 'lite' receive pipe CRC offload engine profiles - refer to
 * XN-200147-AN.
 */
#define          TABLE_ID_DPU_NET_RXLITE_CRC0_OE_PROFILE 0x1021000
/* enum: DPU.net 'lite' receive pipe checksum offload engine profiles - refer
 * to XN-200147-AN.
 */
#define          TABLE_ID_DPU_NET_RXLITE_CSUM0_OE_PROFILE 0x1021100
/* enum: DPU.net 'full' transmit pipe CRC offload engine profiles - refer to
 * XN-200147-AN.
 */
#define          TABLE_ID_DPU_NET_TX_CRC0_OE_PROFILE 0x1028000
/* enum: DPU.net 'full' transmit pipe first checksum offload engine profiles -
 * refer to XN-200147-AN.
 */
#define          TABLE_ID_DPU_NET_TX_CSUM0_OE_PROFILE 0x1028100
/* enum: DPU.net 'full' transmit pipe second checksum offload engine profiles -
 * refer to XN-200147-AN.
 */
#define          TABLE_ID_DPU_NET_TX_CSUM1_OE_PROFILE 0x1028110
/* enum: DPU.net 'full' transmit pipe AES-GCM offload engine profiles - refer
 * to XN-200147-AN.
 */
#define          TABLE_ID_DPU_NET_TX_AES_GCM0_OE_PROFILE 0x1028200
/* enum: DPU.net 'lite' transmit pipe CRC offload engine profiles - refer to
 * XN-200147-AN.
 */
#define          TABLE_ID_DPU_NET_TXLITE_CRC0_OE_PROFILE 0x1029000
/* enum: DPU.net 'lite' transmit pipe checksum offload engine profiles - refer
 * to XN-200147-AN.
 */
#define          TABLE_ID_DPU_NET_TXLITE_CSUM0_OE_PROFILE 0x1029100

/* TABLE_FIELD_ID enum: Unique IDs for fields. Related concepts have been
 * loosely grouped together into blocks with gaps for expansion, but the values
 * are arbitrary. Field IDs are not specific to particular tables, and in some
 * cases this sharing means that they are not used with the exact names of the
 * corresponding table definitions in SF-123102-TC; however, the mapping should
 * still be clear. The intent is that a list of fields, with their associated
 * bit widths and semantics version code, unambiguously defines the semantics
 * of the fields in a key or response. (Again, this list includes all of the
 * fields currently defined within the streaming engines, but only a subset may
 * actually be used by the supported list of tables.)
 */
/* enum: May appear multiple times within a key or response, and indicates that
 * the field is unused and should be set to 0 (or masked out if permitted by
 * the MASK_VALUE for this field).
 */
#define          TABLE_FIELD_ID_UNUSED 0x0
/* enum: Source m-port (a full m-port label). */
#define          TABLE_FIELD_ID_SRC_MPORT 0x1
/* enum: Destination m-port (a full m-port label). */
#define          TABLE_FIELD_ID_DST_MPORT 0x2
/* enum: Source m-group ID. */
#define          TABLE_FIELD_ID_SRC_MGROUP_ID 0x3
/* enum: Physical network port ID (or m-port ID; same thing, for physical
 * network ports).
 */
#define          TABLE_FIELD_ID_NETWORK_PORT_ID 0x4
/* enum: True if packet arrived via network port, false if it arrived via host.
 */
#define          TABLE_FIELD_ID_IS_FROM_NETWORK 0x5
/* enum: Full virtual channel from capsule header. */
#define          TABLE_FIELD_ID_CH_VC 0x6
/* enum: Low bits of virtual channel from capsule header. */
#define          TABLE_FIELD_ID_CH_VC_LOW 0x7
/* enum: User mark value in metadata and packet prefix. */
#define          TABLE_FIELD_ID_USER_MARK 0x8
/* enum: User flag value in metadata and packet prefix. */
#define          TABLE_FIELD_ID_USER_FLAG 0x9
/* enum: Counter ID associated with a response. All-bits-1 is a null value to
 * suppress counting.
 */
#define          TABLE_FIELD_ID_COUNTER_ID 0xa
/* enum: Discriminator which may be set by plugins in some lookup keys; this
 * allows plugins to make a reinterpretation of packet fields in these keys
 * without clashing with the normal interpretation.
 */
#define          TABLE_FIELD_ID_DISCRIM 0xb
/* enum: Destination MAC address. The mapping from bytes in a frame to the
 * 48-bit value for this field is in network order, i.e. a MAC address of
 * AA:BB:CC:DD:EE:FF becomes a 48-bit value of 0xAABBCCDDEEFF.
 */
#define          TABLE_FIELD_ID_DST_MAC 0x14
/* enum: Source MAC address (see notes for DST_MAC). */
#define          TABLE_FIELD_ID_SRC_MAC 0x15
/* enum: Outer VLAN tag TPID, compressed to an enumeration. */
#define          TABLE_FIELD_ID_OVLAN_TPID_COMPRESSED 0x16
/* enum: Full outer VLAN tag TCI (16 bits). */
#define          TABLE_FIELD_ID_OVLAN 0x17
/* enum: Outer VLAN ID (least significant 12 bits of full 16-bit TCI) only. */
#define          TABLE_FIELD_ID_OVLAN_VID 0x18
/* enum: Inner VLAN tag TPID, compressed to an enumeration. */
#define          TABLE_FIELD_ID_IVLAN_TPID_COMPRESSED 0x19
/* enum: Full inner VLAN tag TCI (16 bits). */
#define          TABLE_FIELD_ID_IVLAN 0x1a
/* enum: Inner VLAN ID (least significant 12 bits of full 16-bit TCI) only. */
#define          TABLE_FIELD_ID_IVLAN_VID 0x1b
/* enum: Ethertype. */
#define          TABLE_FIELD_ID_ETHER_TYPE 0x1c
/* enum: Source IP address, either IPv4 or IPv6. The mapping from bytes in a
 * frame to the 128-bit value for this field is in network order, with IPv4
 * addresses assumed to have 12 bytes of trailing zeroes. i.e. the IPv6 address
 * [2345::6789:ABCD] is 0x2345000000000000000000006789ABCD; the IPv4 address
 * 192.168.1.2 is 0xC0A80102000000000000000000000000.
 */
#define          TABLE_FIELD_ID_SRC_IP 0x1d
/* enum: Destination IP address (see notes for SRC_IP). */
#define          TABLE_FIELD_ID_DST_IP 0x1e
/* enum: IPv4 Type-of-Service or IPv6 Traffic Class field. */
#define          TABLE_FIELD_ID_IP_TOS 0x1f
/* enum: IP Protocol. */
#define          TABLE_FIELD_ID_IP_PROTO 0x20
/* enum: Layer 4 source port. */
#define          TABLE_FIELD_ID_SRC_PORT 0x21
/* enum: Layer 4 destination port. */
#define          TABLE_FIELD_ID_DST_PORT 0x22
/* enum: TCP flags. */
#define          TABLE_FIELD_ID_TCP_FLAGS 0x23
/* enum: Virtual Network Identifier (VXLAN) or Virtual Session ID (NVGRE). */
#define          TABLE_FIELD_ID_VNI 0x24
/* enum: True if packet has any tunnel encapsulation header. */
#define          TABLE_FIELD_ID_HAS_ENCAP 0x32
/* enum: True if encap header has an outer VLAN tag. */
#define          TABLE_FIELD_ID_HAS_ENC_OVLAN 0x33
/* enum: True if encap header has an inner VLAN tag. */
#define          TABLE_FIELD_ID_HAS_ENC_IVLAN 0x34
/* enum: True if encap header is some sort of IP. */
#define          TABLE_FIELD_ID_HAS_ENC_IP 0x35
/* enum: True if encap header is specifically IPv4. */
#define          TABLE_FIELD_ID_HAS_ENC_IP4 0x36
/* enum: True if encap header is UDP. */
#define          TABLE_FIELD_ID_HAS_ENC_UDP 0x37
/* enum: True if only/inner frame has an outer VLAN tag. */
#define          TABLE_FIELD_ID_HAS_OVLAN 0x38
/* enum: True if only/inner frame has an inner VLAN tag. */
#define          TABLE_FIELD_ID_HAS_IVLAN 0x39
/* enum: True if only/inner frame is some sort of IP. */
#define          TABLE_FIELD_ID_HAS_IP 0x3a
/* enum: True if only/inner frame has a recognised L4 IP protocol (TCP or UDP).
 */
#define          TABLE_FIELD_ID_HAS_L4 0x3b
/* enum: True if only/inner frame is an IP fragment. */
#define          TABLE_FIELD_ID_IP_FRAG 0x3c
/* enum: True if only/inner frame is the first IP fragment (fragment offset 0).
 */
#define          TABLE_FIELD_ID_IP_FIRST_FRAG 0x3d
/* enum: True if only/inner frame has an IP Time-To-Live of <= 1. (Note: the
 * implementation calls this "ip_ttl_is_one" but does in fact match packets
 * with TTL=0 - which we shouldn't be seeing! - as well.)
 */
#define          TABLE_FIELD_ID_IP_TTL_LE_ONE 0x3e
/* enum: True if only/inner frame has any of TCP SYN, FIN or RST flags set. */
#define          TABLE_FIELD_ID_TCP_INTERESTING_FLAGS 0x3f
/* enum: Plugin channel selection. */
#define          TABLE_FIELD_ID_RDP_PL_CHAN 0x50
/* enum: Enable update of CH_ROUTE_RDP_C_PL route bit. */
#define          TABLE_FIELD_ID_RDP_C_PL_EN 0x51
/* enum: New value of CH_ROUTE_RDP_C_PL route bit. */
#define          TABLE_FIELD_ID_RDP_C_PL 0x52
/* enum: Enable update of CH_ROUTE_RDP_D_PL route bit. */
#define          TABLE_FIELD_ID_RDP_D_PL_EN 0x53
/* enum: New value of CH_ROUTE_RDP_D_PL route bit. */
#define          TABLE_FIELD_ID_RDP_D_PL 0x54
/* enum: Enable update of CH_ROUTE_RDP_OUT_HOST_CHAN route bit. */
#define          TABLE_FIELD_ID_RDP_OUT_HOST_CHAN_EN 0x55
/* enum: New value of CH_ROUTE_RDP_OUT_HOST_CHAN route bit. */
#define          TABLE_FIELD_ID_RDP_OUT_HOST_CHAN 0x56
/* enum: Recirculation ID for lookup sequences with two action rule lookups. */
#define          TABLE_FIELD_ID_RECIRC_ID 0x64
/* enum: Domain ID passed to conntrack and action rule lookups. */
#define          TABLE_FIELD_ID_DOMAIN 0x65
/* enum: Construction mode for encap_tunnel_id - see MAE_CT_VNI_MODE enum. */
#define          TABLE_FIELD_ID_CT_VNI_MODE 0x66
/* enum: True to inhibit conntrack lookup if TCP SYN, FIN or RST flag is set.
 */
#define          TABLE_FIELD_ID_CT_TCP_FLAGS_INHIBIT 0x67
/* enum: True to do conntrack lookups for IPv4 TCP packets. */
#define          TABLE_FIELD_ID_DO_CT_IP4_TCP 0x68
/* enum: True to do conntrack lookups for IPv4 UDP packets. */
#define          TABLE_FIELD_ID_DO_CT_IP4_UDP 0x69
/* enum: True to do conntrack lookups for IPv6 TCP packets. */
#define          TABLE_FIELD_ID_DO_CT_IP6_TCP 0x6a
/* enum: True to do conntrack lookups for IPv6 UDP packets. */
#define          TABLE_FIELD_ID_DO_CT_IP6_UDP 0x6b
/* enum: Outer rule identifier. */
#define          TABLE_FIELD_ID_OUTER_RULE_ID 0x6c
/* enum: Encapsulation type - see MAE_MCDI_ENCAP_TYPE enum. */
#define          TABLE_FIELD_ID_ENCAP_TYPE 0x6d
/* enum: Encap tunnel ID for conntrack lookups from VNI, VLAN tag(s), or 0,
 * depending on CT_VNI_MODE.
 */
#define          TABLE_FIELD_ID_ENCAP_TUNNEL_ID 0x78
/* enum: A conntrack entry identifier, passed to plugins. */
#define          TABLE_FIELD_ID_CT_ENTRY_ID 0x79
/* enum: Either source or destination NAT replacement port. */
#define          TABLE_FIELD_ID_NAT_PORT 0x7a
/* enum: Either source or destination NAT replacement IPv4 address. Note that
 * this is specifically an IPv4 address (IPv6 is not supported for NAT), with
 * byte mapped to a 32-bit value in network order, i.e. the IPv4 address
 * 192.168.1.2 is the value 0xC0A80102.
 */
#define          TABLE_FIELD_ID_NAT_IP 0x7b
/* enum: NAT direction: 0=>source, 1=>destination. */
#define          TABLE_FIELD_ID_NAT_DIR 0x7c
/* enum: Conntrack mark value, passed to action rule lookup. Note that this is
 * not related to the "user mark" in the metadata / packet prefix.
 */
#define          TABLE_FIELD_ID_CT_MARK 0x7d
/* enum: Private flags for conntrack, passed to action rule lookup. */
#define          TABLE_FIELD_ID_CT_PRIV_FLAGS 0x7e
/* enum: True if the conntrack lookup resulted in a hit. */
#define          TABLE_FIELD_ID_CT_HIT 0x7f
/* enum: True to suppress delivery when source and destination m-ports match.
 */
#define          TABLE_FIELD_ID_SUPPRESS_SELF_DELIVERY 0x8c
/* enum: True to perform tunnel decapsulation. */
#define          TABLE_FIELD_ID_DO_DECAP 0x8d
/* enum: True to copy outer frame DSCP to inner on decap. */
#define          TABLE_FIELD_ID_DECAP_DSCP_COPY 0x8e
/* enum: True to map outer frame ECN to inner on decap, by RFC 6040 rules. */
#define          TABLE_FIELD_ID_DECAP_ECN_RFC6040 0x8f
/* enum: True to replace DSCP field. */
#define          TABLE_FIELD_ID_DO_REPLACE_DSCP 0x90
/* enum: True to replace ECN field. */
#define          TABLE_FIELD_ID_DO_REPLACE_ECN 0x91
/* enum: True to decrement IP Time-To-Live. */
#define          TABLE_FIELD_ID_DO_DECR_IP_TTL 0x92
/* enum: True to replace source MAC address. */
#define          TABLE_FIELD_ID_DO_SRC_MAC 0x93
/* enum: True to replace destination MAC address. */
#define          TABLE_FIELD_ID_DO_DST_MAC 0x94
/* enum: Number of VLAN tags to pop. Valid values are 0, 1, or 2. */
#define          TABLE_FIELD_ID_DO_VLAN_POP 0x95
/* enum: Number of VLANs tags to push. Valid values are 0, 1, or 2. */
#define          TABLE_FIELD_ID_DO_VLAN_PUSH 0x96
/* enum: True to count this packet. */
#define          TABLE_FIELD_ID_DO_COUNT 0x97
/* enum: True to perform tunnel encapsulation. */
#define          TABLE_FIELD_ID_DO_ENCAP 0x98
/* enum: True to copy inner frame DSCP to outer on encap. */
#define          TABLE_FIELD_ID_ENCAP_DSCP_COPY 0x99
/* enum: True to copy inner frame ECN to outer on encap. */
#define          TABLE_FIELD_ID_ENCAP_ECN_COPY 0x9a
/* enum: True to deliver the packet (otherwise it is dropped). */
#define          TABLE_FIELD_ID_DO_DELIVER 0x9b
/* enum: True to set the user flag in the metadata. */
#define          TABLE_FIELD_ID_DO_FLAG 0x9c
/* enum: True to update the user mark in the metadata. */
#define          TABLE_FIELD_ID_DO_MARK 0x9d
/* enum: True to override the capsule virtual channel for network deliveries.
 */
#define          TABLE_FIELD_ID_DO_SET_NET_CHAN 0x9e
/* enum: True to override the reported source m-port for host deliveries. */
#define          TABLE_FIELD_ID_DO_SET_SRC_MPORT 0x9f
/* enum: Encap header ID for DO_ENCAP, indexing Encap_Hdr_Part1/2_Table. */
#define          TABLE_FIELD_ID_ENCAP_HDR_ID 0xaa
/* enum: New DSCP value for DO_REPLACE_DSCP. */
#define          TABLE_FIELD_ID_DSCP_VALUE 0xab
/* enum: If DO_REPLACE_ECN is set, the new value for the ECN field. If
 * DO_REPLACE_ECN is not set, ECN_CONTROL[0] and ECN_CONTROL[1] are set to
 * request remapping of ECT0 and ECT1 ECN codepoints respectively to CE.
 */
#define          TABLE_FIELD_ID_ECN_CONTROL 0xac
/* enum: Source MAC ID for DO_SRC_MAC, indexing Replace_Src_MAC_Table. */
#define          TABLE_FIELD_ID_SRC_MAC_ID 0xad
/* enum: Destination MAC ID for DO_DST_MAC, indexing Replace_Dst_MAC_Table. */
#define          TABLE_FIELD_ID_DST_MAC_ID 0xae
/* enum: Parameter for either DO_SET_NET_CHAN (only bottom 6 bits used in this
 * case) or DO_SET_SRC_MPORT.
 */
#define          TABLE_FIELD_ID_REPORTED_SRC_MPORT_OR_NET_CHAN 0xaf
/* enum: 64-byte chunk of added encapsulation header. */
#define          TABLE_FIELD_ID_CHUNK64 0xb4
/* enum: 32-byte chunk of added encapsulation header. */
#define          TABLE_FIELD_ID_CHUNK32 0xb5
/* enum: 16-byte chunk of added encapsulation header. */
#define          TABLE_FIELD_ID_CHUNK16 0xb6
/* enum: 8-byte chunk of added encapsulation header. */
#define          TABLE_FIELD_ID_CHUNK8 0xb7
/* enum: 4-byte chunk of added encapsulation header. */
#define          TABLE_FIELD_ID_CHUNK4 0xb8
/* enum: 2-byte chunk of added encapsulation header. */
#define          TABLE_FIELD_ID_CHUNK2 0xb9
/* enum: Added encapsulation header length in words. */
#define          TABLE_FIELD_ID_HDR_LEN_W 0xba
/* enum: Static value for layer 2/3 LACP hash of the encapsulation header. */
#define          TABLE_FIELD_ID_ENC_LACP_HASH_L23 0xbb
/* enum: Static value for layer 4 LACP hash of the encapsulation header. */
#define          TABLE_FIELD_ID_ENC_LACP_HASH_L4 0xbc
/* enum: True to use the static ENC_LACP_HASH values for the encap header
 * instead of the calculated values for the inner frame when delivering a newly
 * encapsulated packet to a LAG m-port.
 */
#define          TABLE_FIELD_ID_USE_ENC_LACP_HASHES 0xbd
/* enum: True to trigger conntrack from first action rule lookup (AR=>CT=>AR
 * sequence).
 */
#define          TABLE_FIELD_ID_DO_CT 0xc8
/* enum: True to perform NAT using parameters from conntrack lookup response.
 */
#define          TABLE_FIELD_ID_DO_NAT 0xc9
/* enum: True to trigger recirculated action rule lookup (AR=>AR sequence). */
#define          TABLE_FIELD_ID_DO_RECIRC 0xca
/* enum: Next action set payload ID for replay. The null value is all-1-bits.
 */
#define          TABLE_FIELD_ID_NEXT_ACTION_SET_PAYLOAD 0xcb
/* enum: Next action set row ID for replay. The null value is all-1-bits. */
#define          TABLE_FIELD_ID_NEXT_ACTION_SET_ROW 0xcc
/* enum: Action set payload ID for additional delivery to management CPU. The
 * null value is all-1-bits.
 */
#define          TABLE_FIELD_ID_MC_ACTION_SET_PAYLOAD 0xcd
/* enum: Action set row ID for additional delivery to management CPU. The null
 * value is all-1-bits.
 */
#define          TABLE_FIELD_ID_MC_ACTION_SET_ROW 0xce
/* enum: True to include layer 4 in LACP hash on delivery to a LAG m-port. */
#define          TABLE_FIELD_ID_LACP_INC_L4 0xdc
/* enum: True to request that LACP is performed by a plugin. */
#define          TABLE_FIELD_ID_LACP_PLUGIN 0xdd
/* enum: LACP_Balance_Table base address divided by 64. */
#define          TABLE_FIELD_ID_BAL_TBL_BASE_DIV64 0xde
/* enum: Length of balance table region: 0=>64, 1=>128, 2=>256. */
#define          TABLE_FIELD_ID_BAL_TBL_LEN_ID 0xdf
/* enum: LACP LAG ID (i.e. the low 3 bits of LACP LAG mport ID), indexing
 * LACP_LAG_Config_Table. Refer to SF-123102-TC.
 */
#define          TABLE_FIELD_ID_LACP_LAG_ID 0xe0
/* enum: Address in LACP_Balance_Table. The balance table is partitioned
 * between LAGs according to the settings in LACP_LAG_Config_Table and then
 * indexed by the LACP hash, providing the mapping to destination mports. Refer
 * to SF-123102-TC.
 */
#define          TABLE_FIELD_ID_BAL_TBL_ADDR 0xe1
/* enum: UDP port to match for UDP-based encapsulations; required to be 0 for
 * other encapsulation types.
 */
#define          TABLE_FIELD_ID_UDP_PORT 0xe6
/* enum: True to perform RSS based on outer fields rather than inner fields. */
#define          TABLE_FIELD_ID_RSS_ON_OUTER 0xe7
/* enum: True to perform steering table lookup on outer fields rather than
 * inner fields.
 */
#define          TABLE_FIELD_ID_STEER_ON_OUTER 0xe8
/* enum: Destination queue ID for host delivery. */
#define          TABLE_FIELD_ID_DST_QID 0xf0
/* enum: True to drop this packet. */
#define          TABLE_FIELD_ID_DROP 0xf1
/* enum: True to strip outer VLAN tag from this packet. */
#define          TABLE_FIELD_ID_VLAN_STRIP 0xf2
/* enum: True to override the user mark field with the supplied USER_MARK, or
 * false to bitwise-OR the USER_MARK into it.
 */
#define          TABLE_FIELD_ID_MARK_OVERRIDE 0xf3
/* enum: True to override the user flag field with the supplied USER_FLAG, or
 * false to bitwise-OR the USER_FLAG into it.
 */
#define          TABLE_FIELD_ID_FLAG_OVERRIDE 0xf4
/* enum: RSS context ID, indexing the RSS_Context_Table. */
#define          TABLE_FIELD_ID_RSS_CTX_ID 0xfa
/* enum: True to enable RSS. */
#define          TABLE_FIELD_ID_RSS_EN 0xfb
/* enum: Toeplitz hash key. */
#define          TABLE_FIELD_ID_KEY 0xfc
/* enum: Key mode for IPv4 TCP packets - see TABLE_RSS_KEY_MODE enum. */
#define          TABLE_FIELD_ID_TCP_V4_KEY_MODE 0xfd
/* enum: Key mode for IPv6 TCP packets - see TABLE_RSS_KEY_MODE enum. */
#define          TABLE_FIELD_ID_TCP_V6_KEY_MODE 0xfe
/* enum: Key mode for IPv4 UDP packets - see TABLE_RSS_KEY_MODE enum. */
#define          TABLE_FIELD_ID_UDP_V4_KEY_MODE 0xff
/* enum: Key mode for IPv6 UDP packets - see TABLE_RSS_KEY_MODE enum. */
#define          TABLE_FIELD_ID_UDP_V6_KEY_MODE 0x100
/* enum: Key mode for other IPv4 packets - see TABLE_RSS_KEY_MODE enum. */
#define          TABLE_FIELD_ID_OTHER_V4_KEY_MODE 0x101
/* enum: Key mode for other IPv6 packets - see TABLE_RSS_KEY_MODE enum. */
#define          TABLE_FIELD_ID_OTHER_V6_KEY_MODE 0x102
/* enum: Spreading mode - 0=>indirection; 1=>even. */
#define          TABLE_FIELD_ID_SPREAD_MODE 0x103
/* enum: For indirection spreading mode, the base address of a region within
 * the Indirection_Table. For even spreading mode, the number of queues to
 * spread across (only values 1-255 are valid for this mode).
 */
#define          TABLE_FIELD_ID_INDIR_TBL_BASE 0x104
/* enum: For indirection spreading mode, identifies the length of a region
 * within the Indirection_Table, where length = 32 << len_id. Must be set to 0
 * for even spreading mode.
 */
#define          TABLE_FIELD_ID_INDIR_TBL_LEN_ID 0x105
/* enum: An offset to be applied to the base destination queue ID. */
#define          TABLE_FIELD_ID_INDIR_OFFSET 0x106
/* enum: DPU offload engine profile ID to address. */
#define          TABLE_FIELD_ID_OE_PROFILE 0x3e8
/* enum: Width of the CRC to calculate - see CRC_VARIANT enum. */
#define          TABLE_FIELD_ID_CRC_VARIANT 0x3f2
/* enum: If set, reflect the bits of each input byte, bit 7 is LSB, bit 0 is
 * MSB. If clear, bit 7 is MSB, bit 0 is LSB.
 */
#define          TABLE_FIELD_ID_CRC_REFIN 0x3f3
/* enum: If set, reflect the bits of each output byte, bit 7 is LSB, bit 0 is
 * MSB. If clear, bit 7 is MSB, bit 0 is LSB.
 */
#define          TABLE_FIELD_ID_CRC_REFOUT 0x3f4
/* enum: If set, invert every bit of the output value. */
#define          TABLE_FIELD_ID_CRC_INVOUT 0x3f5
/* enum: The CRC polynomial to use for checksumming, in normal form. */
#define          TABLE_FIELD_ID_CRC_POLY 0x3f6
/* enum: Operation for the checksum engine to perform - see DPU_CSUM_OP enum.
 */
#define          TABLE_FIELD_ID_CSUM_OP 0x410
/* enum: Byte offset of checksum relative to region_start (for VALIDATE_*
 * operations only).
 */
#define          TABLE_FIELD_ID_CSUM_OFFSET 0x411
/* enum: Indicates there is additional data on OPR bus that needs to be
 * incorporated into the payload checksum.
 */
#define          TABLE_FIELD_ID_CSUM_OPR_ADDITIONAL_DATA 0x412
/* enum: Log2 data size of additional data on OPR bus. */
#define          TABLE_FIELD_ID_CSUM_OPR_DATA_SIZE_LOG2 0x413
/* enum: 4 byte offset of where to find the additional data on the OPR bus. */
#define          TABLE_FIELD_ID_CSUM_OPR_4B_OFF 0x414
/* enum: Operation type for the AES-GCM core - see GCM_OP_CODE enum. */
#define          TABLE_FIELD_ID_GCM_OP_CODE 0x41a
/* enum: Key length - AES_KEY_LEN enum. */
#define          TABLE_FIELD_ID_GCM_KEY_LEN 0x41b
/* enum: OPR 4 byte offset for ICV or GHASH output (only in BULK_* mode) or
 * IPSEC descrypt output.
 */
#define          TABLE_FIELD_ID_GCM_OPR_4B_OFFSET 0x41c
/* enum: If OP_CODE is BULK_*, indicates Emit GHASH (Fragment mode). Else,
 * indicates IPSEC-ESN mode.
 */
#define          TABLE_FIELD_ID_GCM_EMIT_GHASH_ISESN 0x41d
/* enum: Replay Protection Enable. */
#define          TABLE_FIELD_ID_GCM_REPLAY_PROTECT_EN 0x41e
/* enum: IPSEC Encrypt ESP trailer NEXT_HEADER byte. */
#define          TABLE_FIELD_ID_GCM_NEXT_HDR 0x41f
/* enum: Replay Window Size. */
#define          TABLE_FIELD_ID_GCM_REPLAY_WIN_SIZE 0x420

/* MCDI_EVENT structuredef: The structure of an MCDI_EVENT on Siena/EF10/EF100
 * platforms
 */
#define    MCDI_EVENT_LEN 8
#define       MCDI_EVENT_CONT_LBN 32
#define       MCDI_EVENT_CONT_WIDTH 1
#define       MCDI_EVENT_LEVEL_LBN 33
#define       MCDI_EVENT_LEVEL_WIDTH 3
/* enum: Info. */
#define          MCDI_EVENT_LEVEL_INFO 0x0
/* enum: Warning. */
#define          MCDI_EVENT_LEVEL_WARN 0x1
/* enum: Error. */
#define          MCDI_EVENT_LEVEL_ERR 0x2
/* enum: Fatal. */
#define          MCDI_EVENT_LEVEL_FATAL 0x3
#define       MCDI_EVENT_DATA_OFST 0
#define       MCDI_EVENT_DATA_LEN 4
#define        MCDI_EVENT_CMDDONE_SEQ_OFST 0
#define        MCDI_EVENT_CMDDONE_SEQ_LBN 0
#define        MCDI_EVENT_CMDDONE_SEQ_WIDTH 8
#define        MCDI_EVENT_CMDDONE_DATALEN_OFST 0
#define        MCDI_EVENT_CMDDONE_DATALEN_LBN 8
#define        MCDI_EVENT_CMDDONE_DATALEN_WIDTH 8
#define        MCDI_EVENT_CMDDONE_ERRNO_OFST 0
#define        MCDI_EVENT_CMDDONE_ERRNO_LBN 16
#define        MCDI_EVENT_CMDDONE_ERRNO_WIDTH 8
#define        MCDI_EVENT_LINKCHANGE_LP_CAP_OFST 0
#define        MCDI_EVENT_LINKCHANGE_LP_CAP_LBN 0
#define        MCDI_EVENT_LINKCHANGE_LP_CAP_WIDTH 16
#define        MCDI_EVENT_LINKCHANGE_SPEED_OFST 0
#define        MCDI_EVENT_LINKCHANGE_SPEED_LBN 16
#define        MCDI_EVENT_LINKCHANGE_SPEED_WIDTH 4
/* enum: Link is down or link speed could not be determined */
#define          MCDI_EVENT_LINKCHANGE_SPEED_UNKNOWN 0x0
/* enum: 100Mbs */
#define          MCDI_EVENT_LINKCHANGE_SPEED_100M 0x1
/* enum: 1Gbs */
#define          MCDI_EVENT_LINKCHANGE_SPEED_1G 0x2
/* enum: 10Gbs */
#define          MCDI_EVENT_LINKCHANGE_SPEED_10G 0x3
/* enum: 40Gbs */
#define          MCDI_EVENT_LINKCHANGE_SPEED_40G 0x4
/* enum: 25Gbs */
#define          MCDI_EVENT_LINKCHANGE_SPEED_25G 0x5
/* enum: 50Gbs */
#define          MCDI_EVENT_LINKCHANGE_SPEED_50G 0x6
/* enum: 100Gbs */
#define          MCDI_EVENT_LINKCHANGE_SPEED_100G 0x7
#define        MCDI_EVENT_LINKCHANGE_FCNTL_OFST 0
#define        MCDI_EVENT_LINKCHANGE_FCNTL_LBN 20
#define        MCDI_EVENT_LINKCHANGE_FCNTL_WIDTH 4
#define        MCDI_EVENT_LINKCHANGE_LINK_FLAGS_OFST 0
#define        MCDI_EVENT_LINKCHANGE_LINK_FLAGS_LBN 24
#define        MCDI_EVENT_LINKCHANGE_LINK_FLAGS_WIDTH 8
#define        MCDI_EVENT_PORT_LINKCHANGE_PORT_HANDLE_OFST 0
#define        MCDI_EVENT_PORT_LINKCHANGE_PORT_HANDLE_LBN 0
#define        MCDI_EVENT_PORT_LINKCHANGE_PORT_HANDLE_WIDTH 24
#define        MCDI_EVENT_PORT_LINKCHANGE_SEQ_NUM_OFST 0
#define        MCDI_EVENT_PORT_LINKCHANGE_SEQ_NUM_LBN 24
#define        MCDI_EVENT_PORT_LINKCHANGE_SEQ_NUM_WIDTH 7
#define        MCDI_EVENT_PORT_LINKCHANGE_LINK_UP_OFST 0
#define        MCDI_EVENT_PORT_LINKCHANGE_LINK_UP_LBN 31
#define        MCDI_EVENT_PORT_LINKCHANGE_LINK_UP_WIDTH 1
#define        MCDI_EVENT_PORT_MODULECHANGE_PORT_HANDLE_OFST 0
#define        MCDI_EVENT_PORT_MODULECHANGE_PORT_HANDLE_LBN 0
#define        MCDI_EVENT_PORT_MODULECHANGE_PORT_HANDLE_WIDTH 24
#define        MCDI_EVENT_PORT_MODULECHANGE_SEQ_NUM_OFST 0
#define        MCDI_EVENT_PORT_MODULECHANGE_SEQ_NUM_LBN 24
#define        MCDI_EVENT_PORT_MODULECHANGE_SEQ_NUM_WIDTH 7
#define        MCDI_EVENT_PORT_MODULECHANGE_MDI_CONNECTED_OFST 0
#define        MCDI_EVENT_PORT_MODULECHANGE_MDI_CONNECTED_LBN 31
#define        MCDI_EVENT_PORT_MODULECHANGE_MDI_CONNECTED_WIDTH 1
#define        MCDI_EVENT_SENSOREVT_MONITOR_OFST 0
#define        MCDI_EVENT_SENSOREVT_MONITOR_LBN 0
#define        MCDI_EVENT_SENSOREVT_MONITOR_WIDTH 8
#define        MCDI_EVENT_SENSOREVT_STATE_OFST 0
#define        MCDI_EVENT_SENSOREVT_STATE_LBN 8
#define        MCDI_EVENT_SENSOREVT_STATE_WIDTH 8
#define        MCDI_EVENT_SENSOREVT_VALUE_OFST 0
#define        MCDI_EVENT_SENSOREVT_VALUE_LBN 16
#define        MCDI_EVENT_SENSOREVT_VALUE_WIDTH 16
#define        MCDI_EVENT_FWALERT_DATA_OFST 0
#define        MCDI_EVENT_FWALERT_DATA_LBN 8
#define        MCDI_EVENT_FWALERT_DATA_WIDTH 24
#define        MCDI_EVENT_FWALERT_REASON_OFST 0
#define        MCDI_EVENT_FWALERT_REASON_LBN 0
#define        MCDI_EVENT_FWALERT_REASON_WIDTH 8
/* enum: SRAM Access. */
#define          MCDI_EVENT_FWALERT_REASON_SRAM_ACCESS 0x1
#define        MCDI_EVENT_FLR_VF_OFST 0
#define        MCDI_EVENT_FLR_VF_LBN 0
#define        MCDI_EVENT_FLR_VF_WIDTH 8
#define        MCDI_EVENT_TX_ERR_TXQ_OFST 0
#define        MCDI_EVENT_TX_ERR_TXQ_LBN 0
#define        MCDI_EVENT_TX_ERR_TXQ_WIDTH 12
#define        MCDI_EVENT_TX_ERR_TYPE_OFST 0
#define        MCDI_EVENT_TX_ERR_TYPE_LBN 12
#define        MCDI_EVENT_TX_ERR_TYPE_WIDTH 4
/* enum: Descriptor loader reported failure. Specific to EF10-family NICs. */
#define          MCDI_EVENT_TX_ERR_DL_FAIL 0x1
/* enum: Descriptor ring empty and no EOP seen for packet. Specific to
 * EF10-family NICs
 */
#define          MCDI_EVENT_TX_ERR_NO_EOP 0x2
/* enum: Overlength packet. Specific to EF10-family NICs. */
#define          MCDI_EVENT_TX_ERR_2BIG 0x3
/* enum: Malformed option descriptor. Specific to EF10-family NICs. */
#define          MCDI_EVENT_TX_BAD_OPTDESC 0x5
/* enum: Option descriptor part way through a packet. Specific to EF10-family
 * NICs.
 */
#define          MCDI_EVENT_TX_OPT_IN_PKT 0x8
/* enum: DMA or PIO data access error. Specific to EF10-family NICs */
#define          MCDI_EVENT_TX_ERR_BAD_DMA_OR_PIO 0x9
#define        MCDI_EVENT_TX_ERR_INFO_OFST 0
#define        MCDI_EVENT_TX_ERR_INFO_LBN 16
#define        MCDI_EVENT_TX_ERR_INFO_WIDTH 16
#define        MCDI_EVENT_TX_FLUSH_TO_DRIVER_OFST 0
#define        MCDI_EVENT_TX_FLUSH_TO_DRIVER_LBN 12
#define        MCDI_EVENT_TX_FLUSH_TO_DRIVER_WIDTH 1
#define        MCDI_EVENT_TX_FLUSH_TXQ_OFST 0
#define        MCDI_EVENT_TX_FLUSH_TXQ_LBN 0
#define        MCDI_EVENT_TX_FLUSH_TXQ_WIDTH 12
#define        MCDI_EVENT_PTP_ERR_TYPE_OFST 0
#define        MCDI_EVENT_PTP_ERR_TYPE_LBN 0
#define        MCDI_EVENT_PTP_ERR_TYPE_WIDTH 8
/* enum: PLL lost lock */
#define          MCDI_EVENT_PTP_ERR_PLL_LOST 0x1
/* enum: Filter overflow (PDMA) */
#define          MCDI_EVENT_PTP_ERR_FILTER 0x2
/* enum: FIFO overflow (FPGA) */
#define          MCDI_EVENT_PTP_ERR_FIFO 0x3
/* enum: Merge queue overflow */
#define          MCDI_EVENT_PTP_ERR_QUEUE 0x4
#define        MCDI_EVENT_AOE_ERR_TYPE_OFST 0
#define        MCDI_EVENT_AOE_ERR_TYPE_LBN 0
#define        MCDI_EVENT_AOE_ERR_TYPE_WIDTH 8
/* enum: AOE failed to load - no valid image? */
#define          MCDI_EVENT_AOE_NO_LOAD 0x1
/* enum: AOE FC reported an exception */
#define          MCDI_EVENT_AOE_FC_ASSERT 0x2
/* enum: AOE FC watchdogged */
#define          MCDI_EVENT_AOE_FC_WATCHDOG 0x3
/* enum: AOE FC failed to start */
#define          MCDI_EVENT_AOE_FC_NO_START 0x4
/* enum: Generic AOE fault - likely to have been reported via other means too
 * but intended for use by aoex driver.
 */
#define          MCDI_EVENT_AOE_FAULT 0x5
/* enum: Results of reprogramming the CPLD (status in AOE_ERR_DATA) */
#define          MCDI_EVENT_AOE_CPLD_REPROGRAMMED 0x6
/* enum: AOE loaded successfully */
#define          MCDI_EVENT_AOE_LOAD 0x7
/* enum: AOE DMA operation completed (LSB of HOST_HANDLE in AOE_ERR_DATA) */
#define          MCDI_EVENT_AOE_DMA 0x8
/* enum: AOE byteblaster connected/disconnected (Connection status in
 * AOE_ERR_DATA)
 */
#define          MCDI_EVENT_AOE_BYTEBLASTER 0x9
/* enum: DDR ECC status update */
#define          MCDI_EVENT_AOE_DDR_ECC_STATUS 0xa
/* enum: PTP status update */
#define          MCDI_EVENT_AOE_PTP_STATUS 0xb
/* enum: FPGA header incorrect */
#define          MCDI_EVENT_AOE_FPGA_LOAD_HEADER_ERR 0xc
/* enum: FPGA Powered Off due to error in powering up FPGA */
#define          MCDI_EVENT_AOE_FPGA_POWER_OFF 0xd
/* enum: AOE FPGA load failed due to MC to MUM communication failure */
#define          MCDI_EVENT_AOE_FPGA_LOAD_FAILED 0xe
/* enum: Notify that invalid flash type detected */
#define          MCDI_EVENT_AOE_INVALID_FPGA_FLASH_TYPE 0xf
/* enum: Notify that the attempt to run FPGA Controller firmware timed out */
#define          MCDI_EVENT_AOE_FC_RUN_TIMEDOUT 0x10
/* enum: Failure to probe one or more FPGA boot flash chips */
#define          MCDI_EVENT_AOE_FPGA_BOOT_FLASH_INVALID 0x11
/* enum: FPGA boot-flash contains an invalid image header */
#define          MCDI_EVENT_AOE_FPGA_BOOT_FLASH_HDR_INVALID 0x12
/* enum: Failed to program clocks required by the FPGA */
#define          MCDI_EVENT_AOE_FPGA_CLOCKS_PROGRAM_FAILED 0x13
/* enum: Notify that FPGA Controller is alive to serve MCDI requests */
#define          MCDI_EVENT_AOE_FC_RUNNING 0x14
#define        MCDI_EVENT_AOE_ERR_DATA_OFST 0
#define        MCDI_EVENT_AOE_ERR_DATA_LBN 8
#define        MCDI_EVENT_AOE_ERR_DATA_WIDTH 8
#define        MCDI_EVENT_AOE_ERR_FC_ASSERT_INFO_OFST 0
#define        MCDI_EVENT_AOE_ERR_FC_ASSERT_INFO_LBN 8
#define        MCDI_EVENT_AOE_ERR_FC_ASSERT_INFO_WIDTH 8
/* enum: FC Assert happened, but the register information is not available */
#define          MCDI_EVENT_AOE_ERR_FC_ASSERT_SEEN 0x0
/* enum: The register information for FC Assert is ready for reading by driver
 */
#define          MCDI_EVENT_AOE_ERR_FC_ASSERT_DATA_READY 0x1
#define        MCDI_EVENT_AOE_ERR_CODE_FPGA_HEADER_VERIFY_FAILED_OFST 0
#define        MCDI_EVENT_AOE_ERR_CODE_FPGA_HEADER_VERIFY_FAILED_LBN 8
#define        MCDI_EVENT_AOE_ERR_CODE_FPGA_HEADER_VERIFY_FAILED_WIDTH 8
/* enum: Reading from NV failed */
#define          MCDI_EVENT_AOE_ERR_FPGA_HEADER_NV_READ_FAIL 0x0
/* enum: Invalid Magic Number if FPGA header */
#define          MCDI_EVENT_AOE_ERR_FPGA_HEADER_MAGIC_FAIL 0x1
/* enum: Invalid Silicon type detected in header */
#define          MCDI_EVENT_AOE_ERR_FPGA_HEADER_SILICON_TYPE 0x2
/* enum: Unsupported VRatio */
#define          MCDI_EVENT_AOE_ERR_FPGA_HEADER_VRATIO 0x3
/* enum: Unsupported DDR Type */
#define          MCDI_EVENT_AOE_ERR_FPGA_HEADER_DDR_TYPE 0x4
/* enum: DDR Voltage out of supported range */
#define          MCDI_EVENT_AOE_ERR_FPGA_HEADER_DDR_VOLTAGE 0x5
/* enum: Unsupported DDR speed */
#define          MCDI_EVENT_AOE_ERR_FPGA_HEADER_DDR_SPEED 0x6
/* enum: Unsupported DDR size */
#define          MCDI_EVENT_AOE_ERR_FPGA_HEADER_DDR_SIZE 0x7
/* enum: Unsupported DDR rank */
#define          MCDI_EVENT_AOE_ERR_FPGA_HEADER_DDR_RANK 0x8
#define        MCDI_EVENT_AOE_ERR_CODE_INVALID_FPGA_FLASH_TYPE_INFO_OFST 0
#define        MCDI_EVENT_AOE_ERR_CODE_INVALID_FPGA_FLASH_TYPE_INFO_LBN 8
#define        MCDI_EVENT_AOE_ERR_CODE_INVALID_FPGA_FLASH_TYPE_INFO_WIDTH 8
/* enum: Primary boot flash */
#define          MCDI_EVENT_AOE_FLASH_TYPE_BOOT_PRIMARY 0x0
/* enum: Secondary boot flash */
#define          MCDI_EVENT_AOE_FLASH_TYPE_BOOT_SECONDARY 0x1
#define        MCDI_EVENT_AOE_ERR_CODE_FPGA_POWER_OFF_OFST 0
#define        MCDI_EVENT_AOE_ERR_CODE_FPGA_POWER_OFF_LBN 8
#define        MCDI_EVENT_AOE_ERR_CODE_FPGA_POWER_OFF_WIDTH 8
#define        MCDI_EVENT_AOE_ERR_CODE_FPGA_LOAD_FAILED_OFST 0
#define        MCDI_EVENT_AOE_ERR_CODE_FPGA_LOAD_FAILED_LBN 8
#define        MCDI_EVENT_AOE_ERR_CODE_FPGA_LOAD_FAILED_WIDTH 8
#define        MCDI_EVENT_RX_ERR_RXQ_OFST 0
#define        MCDI_EVENT_RX_ERR_RXQ_LBN 0
#define        MCDI_EVENT_RX_ERR_RXQ_WIDTH 12
#define        MCDI_EVENT_RX_ERR_TYPE_OFST 0
#define        MCDI_EVENT_RX_ERR_TYPE_LBN 12
#define        MCDI_EVENT_RX_ERR_TYPE_WIDTH 4
#define        MCDI_EVENT_RX_ERR_INFO_OFST 0
#define        MCDI_EVENT_RX_ERR_INFO_LBN 16
#define        MCDI_EVENT_RX_ERR_INFO_WIDTH 16
#define        MCDI_EVENT_RX_FLUSH_TO_DRIVER_OFST 0
#define        MCDI_EVENT_RX_FLUSH_TO_DRIVER_LBN 12
#define        MCDI_EVENT_RX_FLUSH_TO_DRIVER_WIDTH 1
#define        MCDI_EVENT_RX_FLUSH_RXQ_OFST 0
#define        MCDI_EVENT_RX_FLUSH_RXQ_LBN 0
#define        MCDI_EVENT_RX_FLUSH_RXQ_WIDTH 12
#define        MCDI_EVENT_MC_REBOOT_COUNT_OFST 0
#define        MCDI_EVENT_MC_REBOOT_COUNT_LBN 0
#define        MCDI_EVENT_MC_REBOOT_COUNT_WIDTH 16
#define        MCDI_EVENT_MUM_ERR_TYPE_OFST 0
#define        MCDI_EVENT_MUM_ERR_TYPE_LBN 0
#define        MCDI_EVENT_MUM_ERR_TYPE_WIDTH 8
/* enum: MUM failed to load - no valid image? */
#define          MCDI_EVENT_MUM_NO_LOAD 0x1
/* enum: MUM f/w reported an exception */
#define          MCDI_EVENT_MUM_ASSERT 0x2
/* enum: MUM not kicking watchdog */
#define          MCDI_EVENT_MUM_WATCHDOG 0x3
#define        MCDI_EVENT_MUM_ERR_DATA_OFST 0
#define        MCDI_EVENT_MUM_ERR_DATA_LBN 8
#define        MCDI_EVENT_MUM_ERR_DATA_WIDTH 8
#define        MCDI_EVENT_DBRET_SEQ_OFST 0
#define        MCDI_EVENT_DBRET_SEQ_LBN 0
#define        MCDI_EVENT_DBRET_SEQ_WIDTH 8
#define        MCDI_EVENT_SUC_ERR_TYPE_OFST 0
#define        MCDI_EVENT_SUC_ERR_TYPE_LBN 0
#define        MCDI_EVENT_SUC_ERR_TYPE_WIDTH 8
/* enum: Corrupted or bad SUC application. */
#define          MCDI_EVENT_SUC_BAD_APP 0x1
/* enum: SUC application reported an assert. */
#define          MCDI_EVENT_SUC_ASSERT 0x2
/* enum: SUC application reported an exception. */
#define          MCDI_EVENT_SUC_EXCEPTION 0x3
/* enum: SUC watchdog timer expired. */
#define          MCDI_EVENT_SUC_WATCHDOG 0x4
#define        MCDI_EVENT_SUC_ERR_ADDRESS_OFST 0
#define        MCDI_EVENT_SUC_ERR_ADDRESS_LBN 8
#define        MCDI_EVENT_SUC_ERR_ADDRESS_WIDTH 24
#define        MCDI_EVENT_SUC_ERR_DATA_OFST 0
#define        MCDI_EVENT_SUC_ERR_DATA_LBN 8
#define        MCDI_EVENT_SUC_ERR_DATA_WIDTH 24
#define        MCDI_EVENT_LINKCHANGE_V2_LP_CAP_OFST 0
#define        MCDI_EVENT_LINKCHANGE_V2_LP_CAP_LBN 0
#define        MCDI_EVENT_LINKCHANGE_V2_LP_CAP_WIDTH 24
#define        MCDI_EVENT_LINKCHANGE_V2_SPEED_OFST 0
#define        MCDI_EVENT_LINKCHANGE_V2_SPEED_LBN 24
#define        MCDI_EVENT_LINKCHANGE_V2_SPEED_WIDTH 4
/*             Enum values, see field(s): */
/*                MCDI_EVENT/LINKCHANGE_SPEED */
#define        MCDI_EVENT_LINKCHANGE_V2_FLAGS_LINK_UP_OFST 0
#define        MCDI_EVENT_LINKCHANGE_V2_FLAGS_LINK_UP_LBN 28
#define        MCDI_EVENT_LINKCHANGE_V2_FLAGS_LINK_UP_WIDTH 1
#define        MCDI_EVENT_LINKCHANGE_V2_FCNTL_OFST 0
#define        MCDI_EVENT_LINKCHANGE_V2_FCNTL_LBN 29
#define        MCDI_EVENT_LINKCHANGE_V2_FCNTL_WIDTH 3
/*             Enum values, see field(s): */
/*                MC_CMD_SET_MAC/MC_CMD_SET_MAC_IN/FCNTL */
#define        MCDI_EVENT_MODULECHANGE_LD_CAP_OFST 0
#define        MCDI_EVENT_MODULECHANGE_LD_CAP_LBN 0
#define        MCDI_EVENT_MODULECHANGE_LD_CAP_WIDTH 30
#define        MCDI_EVENT_MODULECHANGE_SEQ_OFST 0
#define        MCDI_EVENT_MODULECHANGE_SEQ_LBN 30
#define        MCDI_EVENT_MODULECHANGE_SEQ_WIDTH 2
#define        MCDI_EVENT_DESC_PROXY_VIRTQ_VI_ID_OFST 0
#define        MCDI_EVENT_DESC_PROXY_VIRTQ_VI_ID_LBN 0
#define        MCDI_EVENT_DESC_PROXY_VIRTQ_VI_ID_WIDTH 16
#define        MCDI_EVENT_DESC_PROXY_VIRTQ_ID_OFST 0
#define        MCDI_EVENT_DESC_PROXY_VIRTQ_ID_LBN 16
#define        MCDI_EVENT_DESC_PROXY_VIRTQ_ID_WIDTH 16
#define       MCDI_EVENT_DATA_LBN 0
#define       MCDI_EVENT_DATA_WIDTH 32
/* Alias for PTP_DATA. */
#define       MCDI_EVENT_SRC_LBN 36
#define       MCDI_EVENT_SRC_WIDTH 8
/* Data associated with PTP events which doesn't fit into the main DATA field
 */
#define       MCDI_EVENT_PTP_DATA_LBN 36
#define       MCDI_EVENT_PTP_DATA_WIDTH 8
/* EF100 specific. Defined by QDMA. The phase bit, changes each time round the
 * event ring
 */
#define       MCDI_EVENT_EV_EVQ_PHASE_LBN 59
#define       MCDI_EVENT_EV_EVQ_PHASE_WIDTH 1
#define       MCDI_EVENT_EV_CODE_LBN 60
#define       MCDI_EVENT_EV_CODE_WIDTH 4
#define       MCDI_EVENT_CODE_LBN 44
#define       MCDI_EVENT_CODE_WIDTH 8
/* enum: Event generated by host software */
#define          MCDI_EVENT_SW_EVENT 0x0
/* enum: Bad assert. */
#define          MCDI_EVENT_CODE_BADSSERT 0x1
/* enum: PM Notice. */
#define          MCDI_EVENT_CODE_PMNOTICE 0x2
/* enum: Command done. */
#define          MCDI_EVENT_CODE_CMDDONE 0x3
/* enum: Link change. */
#define          MCDI_EVENT_CODE_LINKCHANGE 0x4
/* enum: Sensor Event. */
#define          MCDI_EVENT_CODE_SENSOREVT 0x5
/* enum: Schedule error. */
#define          MCDI_EVENT_CODE_SCHEDERR 0x6
/* enum: Reboot. */
#define          MCDI_EVENT_CODE_REBOOT 0x7
/* enum: Mac stats DMA. */
#define          MCDI_EVENT_CODE_MAC_STATS_DMA 0x8
/* enum: Firmware alert. */
#define          MCDI_EVENT_CODE_FWALERT 0x9
/* enum: Function level reset. */
#define          MCDI_EVENT_CODE_FLR 0xa
/* enum: Transmit error */
#define          MCDI_EVENT_CODE_TX_ERR 0xb
/* enum: Tx flush has completed */
#define          MCDI_EVENT_CODE_TX_FLUSH 0xc
/* enum: PTP packet received timestamp */
#define          MCDI_EVENT_CODE_PTP_RX 0xd
/* enum: PTP NIC failure */
#define          MCDI_EVENT_CODE_PTP_FAULT 0xe
/* enum: PTP PPS event */
#define          MCDI_EVENT_CODE_PTP_PPS 0xf
/* enum: Rx flush has completed */
#define          MCDI_EVENT_CODE_RX_FLUSH 0x10
/* enum: Receive error */
#define          MCDI_EVENT_CODE_RX_ERR 0x11
/* enum: AOE fault */
#define          MCDI_EVENT_CODE_AOE 0x12
/* enum: Network port calibration failed (VCAL). */
#define          MCDI_EVENT_CODE_VCAL_FAIL 0x13
/* enum: HW PPS event */
#define          MCDI_EVENT_CODE_HW_PPS 0x14
/* enum: The MC has rebooted (huntington and later, siena uses CODE_REBOOT and
 * a different format)
 */
#define          MCDI_EVENT_CODE_MC_REBOOT 0x15
/* enum: the MC has detected a parity error */
#define          MCDI_EVENT_CODE_PAR_ERR 0x16
/* enum: the MC has detected a correctable error */
#define          MCDI_EVENT_CODE_ECC_CORR_ERR 0x17
/* enum: the MC has detected an uncorrectable error */
#define          MCDI_EVENT_CODE_ECC_FATAL_ERR 0x18
/* enum: The MC has entered offline BIST mode */
#define          MCDI_EVENT_CODE_MC_BIST 0x19
/* enum: PTP tick event providing current NIC time */
#define          MCDI_EVENT_CODE_PTP_TIME 0x1a
/* enum: MUM fault */
#define          MCDI_EVENT_CODE_MUM 0x1b
/* enum: notify the designated PF of a new authorization request */
#define          MCDI_EVENT_CODE_PROXY_REQUEST 0x1c
/* enum: notify a function that awaits an authorization that its request has
 * been processed and it may now resend the command
 */
#define          MCDI_EVENT_CODE_PROXY_RESPONSE 0x1d
/* enum: MCDI command accepted. New commands can be issued but this command is
 * not done yet.
 */
#define          MCDI_EVENT_CODE_DBRET 0x1e
/* enum: The MC has detected a fault on the SUC */
#define          MCDI_EVENT_CODE_SUC 0x1f
/* enum: Link change. This event is sent instead of LINKCHANGE if
 * WANT_V2_LINKCHANGES was set on driver attach.
 */
#define          MCDI_EVENT_CODE_LINKCHANGE_V2 0x20
/* enum: This event is sent if WANT_V2_LINKCHANGES was set on driver attach
 * when the local device capabilities changes. This will usually correspond to
 * a module change.
 */
#define          MCDI_EVENT_CODE_MODULECHANGE 0x21
/* enum: Notification that the sensors have been added and/or removed from the
 * sensor table. This event includes the new sensor table generation count, if
 * this does not match the driver's local copy it is expected to call
 * DYNAMIC_SENSORS_LIST to refresh it.
 */
#define          MCDI_EVENT_CODE_DYNAMIC_SENSORS_CHANGE 0x22
/* enum: Notification that a sensor has changed state as a result of a reading
 * crossing a threshold. This is sent as two events, the first event contains
 * the handle and the sensor's state (in the SRC field), and the second
 * contains the value.
 */
#define          MCDI_EVENT_CODE_DYNAMIC_SENSORS_STATE_CHANGE 0x23
/* enum: Notification that a descriptor proxy function configuration has been
 * pushed to "live" status (visible to host). SRC field contains the handle of
 * the affected descriptor proxy function. DATA field contains the generation
 * count of configuration set applied. See MC_CMD_DESC_PROXY_FUNC_CONFIG_SET /
 * MC_CMD_DESC_PROXY_FUNC_CONFIG_COMMIT and SF-122927-TC for details.
 */
#define          MCDI_EVENT_CODE_DESC_PROXY_FUNC_CONFIG_COMMITTED 0x24
/* enum: Notification that a descriptor proxy function has been reset. SRC
 * field contains the handle of the affected descriptor proxy function. See
 * SF-122927-TC for details.
 */
#define          MCDI_EVENT_CODE_DESC_PROXY_FUNC_RESET 0x25
/* enum: Notification that a driver attached to a descriptor proxy function.
 * SRC field contains the handle of the affected descriptor proxy function. For
 * Virtio proxy functions this message consists of two MCDI events, where the
 * first event's (CONT=1) DATA field carries negotiated virtio feature bits 0
 * to 31 and the second (CONT=0) carries bits 32 to 63. For EF100 proxy
 * functions event length and meaning of DATA field is not yet defined. See
 * SF-122927-TC for details.
 */
#define          MCDI_EVENT_CODE_DESC_PROXY_FUNC_DRIVER_ATTACH 0x26
/* enum: Notification that the mport journal has changed since it was last read
 * and updates can be read using the MC_CMD_MAE_MPORT_READ_JOURNAL command. The
 * firmware may moderate the events so that an event is not sent for every
 * change to the journal.
 */
#define          MCDI_EVENT_CODE_MPORT_JOURNAL_CHANGE 0x27
/* enum: Notification that a source queue is enabled and attached to its proxy
 * sink queue. SRC field contains the handle of the affected descriptor proxy
 * function. DATA field contains the relative source queue number and absolute
 * VI ID.
 */
#define          MCDI_EVENT_CODE_DESC_PROXY_FUNC_QUEUE_START 0x28
/* enum: Notification of a change in link state and/or link speed of a network
 * port link. This event applies to a network port identified by a handle,
 * PORT_HANDLE, which is discovered by the driver using the MC_CMD_ENUM_PORTS
 * command.
 */
#define          MCDI_EVENT_CODE_PORT_LINKCHANGE 0x29
/* enum: Notification of a change in the state of an MDI (external connector)
 * of a network port. This typically corresponds to module plug/unplug for
 * modular interfaces (e.g., SFP/QSFP and similar) or cable connect/disconnect.
 * This event applies to a network port identified by a handle, PORT_HANDLE,
 * which is discovered by the driver using the MC_CMD_ENUM_PORTS command.
 */
#define          MCDI_EVENT_CODE_PORT_MODULECHANGE 0x2a
/* enum: Notification that the port enumeration journal has changed since it
 * was last read and updates can be read using the MC_CMD_ENUM_PORTS command.
 * The firmware may moderate the events so that an event is not sent for every
 * change to the journal.
 */
#define          MCDI_EVENT_CODE_ENUM_PORTS_CHANGE 0x2b
/* enum: Artificial event generated by host and posted via MC for test
 * purposes.
 */
#define          MCDI_EVENT_CODE_TESTGEN 0xfa
#define       MCDI_EVENT_CMDDONE_DATA_OFST 0
#define       MCDI_EVENT_CMDDONE_DATA_LEN 4
#define       MCDI_EVENT_CMDDONE_DATA_LBN 0
#define       MCDI_EVENT_CMDDONE_DATA_WIDTH 32
#define       MCDI_EVENT_LINKCHANGE_DATA_OFST 0
#define       MCDI_EVENT_LINKCHANGE_DATA_LEN 4
#define       MCDI_EVENT_LINKCHANGE_DATA_LBN 0
#define       MCDI_EVENT_LINKCHANGE_DATA_WIDTH 32
#define       MCDI_EVENT_PORT_LINKCHANGE_DATA_OFST 0
#define       MCDI_EVENT_PORT_LINKCHANGE_DATA_LEN 4
#define       MCDI_EVENT_PORT_LINKCHANGE_DATA_LBN 0
#define       MCDI_EVENT_PORT_LINKCHANGE_DATA_WIDTH 32
#define       MCDI_EVENT_PORT_MODULECHANGE_DATA_OFST 0
#define       MCDI_EVENT_PORT_MODULECHANGE_DATA_LEN 4
#define       MCDI_EVENT_PORT_MODULECHANGE_DATA_LBN 0
#define       MCDI_EVENT_PORT_MODULECHANGE_DATA_WIDTH 32
#define       MCDI_EVENT_SENSOREVT_DATA_OFST 0
#define       MCDI_EVENT_SENSOREVT_DATA_LEN 4
#define       MCDI_EVENT_SENSOREVT_DATA_LBN 0
#define       MCDI_EVENT_SENSOREVT_DATA_WIDTH 32
#define       MCDI_EVENT_MAC_STATS_DMA_GENERATION_OFST 0
#define       MCDI_EVENT_MAC_STATS_DMA_GENERATION_LEN 4
#define       MCDI_EVENT_MAC_STATS_DMA_GENERATION_LBN 0
#define       MCDI_EVENT_MAC_STATS_DMA_GENERATION_WIDTH 32
#define       MCDI_EVENT_TX_ERR_DATA_OFST 0
#define       MCDI_EVENT_TX_ERR_DATA_LEN 4
#define       MCDI_EVENT_TX_ERR_DATA_LBN 0
#define       MCDI_EVENT_TX_ERR_DATA_WIDTH 32
/* For CODE_PTP_RX, CODE_PTP_PPS and CODE_HW_PPS events the seconds field of
 * timestamp
 */
#define       MCDI_EVENT_PTP_SECONDS_OFST 0
#define       MCDI_EVENT_PTP_SECONDS_LEN 4
#define       MCDI_EVENT_PTP_SECONDS_LBN 0
#define       MCDI_EVENT_PTP_SECONDS_WIDTH 32
/* For CODE_PTP_RX, CODE_PTP_PPS and CODE_HW_PPS events the major field of
 * timestamp
 */
#define       MCDI_EVENT_PTP_MAJOR_OFST 0
#define       MCDI_EVENT_PTP_MAJOR_LEN 4
#define       MCDI_EVENT_PTP_MAJOR_LBN 0
#define       MCDI_EVENT_PTP_MAJOR_WIDTH 32
/* For CODE_PTP_RX, CODE_PTP_PPS and CODE_HW_PPS events the nanoseconds field
 * of timestamp
 */
#define       MCDI_EVENT_PTP_NANOSECONDS_OFST 0
#define       MCDI_EVENT_PTP_NANOSECONDS_LEN 4
#define       MCDI_EVENT_PTP_NANOSECONDS_LBN 0
#define       MCDI_EVENT_PTP_NANOSECONDS_WIDTH 32
/* For CODE_PTP_RX, CODE_PTP_PPS and CODE_HW_PPS events the minor field of
 * timestamp
 */
#define       MCDI_EVENT_PTP_MINOR_OFST 0
#define       MCDI_EVENT_PTP_MINOR_LEN 4
#define       MCDI_EVENT_PTP_MINOR_LBN 0
#define       MCDI_EVENT_PTP_MINOR_WIDTH 32
/* For CODE_PTP_RX events, the lowest four bytes of sourceUUID from PTP packet
 */
#define       MCDI_EVENT_PTP_UUID_OFST 0
#define       MCDI_EVENT_PTP_UUID_LEN 4
#define       MCDI_EVENT_PTP_UUID_LBN 0
#define       MCDI_EVENT_PTP_UUID_WIDTH 32
#define       MCDI_EVENT_RX_ERR_DATA_OFST 0
#define       MCDI_EVENT_RX_ERR_DATA_LEN 4
#define       MCDI_EVENT_RX_ERR_DATA_LBN 0
#define       MCDI_EVENT_RX_ERR_DATA_WIDTH 32
#define       MCDI_EVENT_PAR_ERR_DATA_OFST 0
#define       MCDI_EVENT_PAR_ERR_DATA_LEN 4
#define       MCDI_EVENT_PAR_ERR_DATA_LBN 0
#define       MCDI_EVENT_PAR_ERR_DATA_WIDTH 32
#define       MCDI_EVENT_ECC_CORR_ERR_DATA_OFST 0
#define       MCDI_EVENT_ECC_CORR_ERR_DATA_LEN 4
#define       MCDI_EVENT_ECC_CORR_ERR_DATA_LBN 0
#define       MCDI_EVENT_ECC_CORR_ERR_DATA_WIDTH 32
#define       MCDI_EVENT_ECC_FATAL_ERR_DATA_OFST 0
#define       MCDI_EVENT_ECC_FATAL_ERR_DATA_LEN 4
#define       MCDI_EVENT_ECC_FATAL_ERR_DATA_LBN 0
#define       MCDI_EVENT_ECC_FATAL_ERR_DATA_WIDTH 32
/* For CODE_PTP_TIME events, the major value of the PTP clock */
#define       MCDI_EVENT_PTP_TIME_MAJOR_OFST 0
#define       MCDI_EVENT_PTP_TIME_MAJOR_LEN 4
#define       MCDI_EVENT_PTP_TIME_MAJOR_LBN 0
#define       MCDI_EVENT_PTP_TIME_MAJOR_WIDTH 32
/* For CODE_PTP_TIME events, bits 19-26 of the minor value of the PTP clock */
#define       MCDI_EVENT_PTP_TIME_MINOR_26_19_LBN 36
#define       MCDI_EVENT_PTP_TIME_MINOR_26_19_WIDTH 8
/* For CODE_PTP_TIME events, most significant bits of the minor value of the
 * PTP clock. This is a more generic equivalent of PTP_TIME_MINOR_26_19.
 */
#define       MCDI_EVENT_PTP_TIME_MINOR_MS_8BITS_LBN 36
#define       MCDI_EVENT_PTP_TIME_MINOR_MS_8BITS_WIDTH 8
/* For CODE_PTP_TIME events where report sync status is enabled, indicates
 * whether the NIC clock has ever been set
 */
#define       MCDI_EVENT_PTP_TIME_NIC_CLOCK_VALID_LBN 36
#define       MCDI_EVENT_PTP_TIME_NIC_CLOCK_VALID_WIDTH 1
/* For CODE_PTP_TIME events where report sync status is enabled, indicates
 * whether the NIC and System clocks are in sync
 */
#define       MCDI_EVENT_PTP_TIME_HOST_NIC_IN_SYNC_LBN 37
#define       MCDI_EVENT_PTP_TIME_HOST_NIC_IN_SYNC_WIDTH 1
/* For CODE_PTP_TIME events where report sync status is enabled, bits 21-26 of
 * the minor value of the PTP clock
 */
#define       MCDI_EVENT_PTP_TIME_MINOR_26_21_LBN 38
#define       MCDI_EVENT_PTP_TIME_MINOR_26_21_WIDTH 6
/* For CODE_PTP_TIME events, most significant bits of the minor value of the
 * PTP clock. This is a more generic equivalent of PTP_TIME_MINOR_26_21.
 */
#define       MCDI_EVENT_PTP_TIME_MINOR_MS_6BITS_LBN 38
#define       MCDI_EVENT_PTP_TIME_MINOR_MS_6BITS_WIDTH 6
#define       MCDI_EVENT_PROXY_REQUEST_BUFF_INDEX_OFST 0
#define       MCDI_EVENT_PROXY_REQUEST_BUFF_INDEX_LEN 4
#define       MCDI_EVENT_PROXY_REQUEST_BUFF_INDEX_LBN 0
#define       MCDI_EVENT_PROXY_REQUEST_BUFF_INDEX_WIDTH 32
#define       MCDI_EVENT_PROXY_RESPONSE_HANDLE_OFST 0
#define       MCDI_EVENT_PROXY_RESPONSE_HANDLE_LEN 4
#define       MCDI_EVENT_PROXY_RESPONSE_HANDLE_LBN 0
#define       MCDI_EVENT_PROXY_RESPONSE_HANDLE_WIDTH 32
/* Zero means that the request has been completed or authorized, and the driver
 * should resend it. A non-zero value means that the authorization has been
 * denied, and gives the reason. Typically it will be EPERM.
 */
#define       MCDI_EVENT_PROXY_RESPONSE_RC_LBN 36
#define       MCDI_EVENT_PROXY_RESPONSE_RC_WIDTH 8
#define       MCDI_EVENT_DBRET_DATA_OFST 0
#define       MCDI_EVENT_DBRET_DATA_LEN 4
#define       MCDI_EVENT_DBRET_DATA_LBN 0
#define       MCDI_EVENT_DBRET_DATA_WIDTH 32
#define       MCDI_EVENT_LINKCHANGE_V2_DATA_OFST 0
#define       MCDI_EVENT_LINKCHANGE_V2_DATA_LEN 4
#define       MCDI_EVENT_LINKCHANGE_V2_DATA_LBN 0
#define       MCDI_EVENT_LINKCHANGE_V2_DATA_WIDTH 32
#define       MCDI_EVENT_MODULECHANGE_DATA_OFST 0
#define       MCDI_EVENT_MODULECHANGE_DATA_LEN 4
#define       MCDI_EVENT_MODULECHANGE_DATA_LBN 0
#define       MCDI_EVENT_MODULECHANGE_DATA_WIDTH 32
/* The new generation count after a sensor has been added or deleted. */
#define       MCDI_EVENT_DYNAMIC_SENSORS_GENERATION_OFST 0
#define       MCDI_EVENT_DYNAMIC_SENSORS_GENERATION_LEN 4
#define       MCDI_EVENT_DYNAMIC_SENSORS_GENERATION_LBN 0
#define       MCDI_EVENT_DYNAMIC_SENSORS_GENERATION_WIDTH 32
/* The handle of a dynamic sensor. */
#define       MCDI_EVENT_DYNAMIC_SENSORS_HANDLE_OFST 0
#define       MCDI_EVENT_DYNAMIC_SENSORS_HANDLE_LEN 4
#define       MCDI_EVENT_DYNAMIC_SENSORS_HANDLE_LBN 0
#define       MCDI_EVENT_DYNAMIC_SENSORS_HANDLE_WIDTH 32
/* The current values of a sensor. */
#define       MCDI_EVENT_DYNAMIC_SENSORS_VALUE_OFST 0
#define       MCDI_EVENT_DYNAMIC_SENSORS_VALUE_LEN 4
#define       MCDI_EVENT_DYNAMIC_SENSORS_VALUE_LBN 0
#define       MCDI_EVENT_DYNAMIC_SENSORS_VALUE_WIDTH 32
/* The current state of a sensor. */
#define       MCDI_EVENT_DYNAMIC_SENSORS_STATE_LBN 36
#define       MCDI_EVENT_DYNAMIC_SENSORS_STATE_WIDTH 8
#define       MCDI_EVENT_DESC_PROXY_DATA_OFST 0
#define       MCDI_EVENT_DESC_PROXY_DATA_LEN 4
#define       MCDI_EVENT_DESC_PROXY_DATA_LBN 0
#define       MCDI_EVENT_DESC_PROXY_DATA_WIDTH 32
/* Generation count of applied configuration set */
#define       MCDI_EVENT_DESC_PROXY_GENERATION_OFST 0
#define       MCDI_EVENT_DESC_PROXY_GENERATION_LEN 4
#define       MCDI_EVENT_DESC_PROXY_GENERATION_LBN 0
#define       MCDI_EVENT_DESC_PROXY_GENERATION_WIDTH 32
/* Virtio features negotiated with the host driver. First event (CONT=1)
 * carries bits 0 to 31. Second event (CONT=0) carries bits 32 to 63.
 */
#define       MCDI_EVENT_DESC_PROXY_VIRTIO_FEATURES_OFST 0
#define       MCDI_EVENT_DESC_PROXY_VIRTIO_FEATURES_LEN 4
#define       MCDI_EVENT_DESC_PROXY_VIRTIO_FEATURES_LBN 0
#define       MCDI_EVENT_DESC_PROXY_VIRTIO_FEATURES_WIDTH 32


/***********************************/
/* MC_CMD_READ32
 * Read multiple 32byte words from MC memory. Note - this command really
 * belongs to INSECURE category but is required by shmboot. The command handler
 * has additional checks to reject insecure calls.
 */
#define MC_CMD_READ32 0x1
#undef MC_CMD_0x1_PRIVILEGE_CTG

#define MC_CMD_0x1_PRIVILEGE_CTG SRIOV_CTG_ADMIN

/* MC_CMD_READ32_IN msgrequest */
#define    MC_CMD_READ32_IN_LEN 8
#define       MC_CMD_READ32_IN_ADDR_OFST 0
#define       MC_CMD_READ32_IN_ADDR_LEN 4
#define       MC_CMD_READ32_IN_NUMWORDS_OFST 4
#define       MC_CMD_READ32_IN_NUMWORDS_LEN 4

/* MC_CMD_READ32_OUT msgresponse */
#define    MC_CMD_READ32_OUT_LENMIN 4
#define    MC_CMD_READ32_OUT_LENMAX 252
#define    MC_CMD_READ32_OUT_LENMAX_MCDI2 1020
#define    MC_CMD_READ32_OUT_LEN(num) (0+4*(num))
#define    MC_CMD_READ32_OUT_BUFFER_NUM(len) (((len)-0)/4)
#define       MC_CMD_READ32_OUT_BUFFER_OFST 0
#define       MC_CMD_READ32_OUT_BUFFER_LEN 4
#define       MC_CMD_READ32_OUT_BUFFER_MINNUM 1
#define       MC_CMD_READ32_OUT_BUFFER_MAXNUM 63
#define       MC_CMD_READ32_OUT_BUFFER_MAXNUM_MCDI2 255


/***********************************/
/* MC_CMD_WRITE32
 * Write multiple 32byte words to MC memory.
 */
#define MC_CMD_WRITE32 0x2
#undef MC_CMD_0x2_PRIVILEGE_CTG

#define MC_CMD_0x2_PRIVILEGE_CTG SRIOV_CTG_INSECURE

/* MC_CMD_WRITE32_IN msgrequest */
#define    MC_CMD_WRITE32_IN_LENMIN 8
#define    MC_CMD_WRITE32_IN_LENMAX 252
#define    MC_CMD_WRITE32_IN_LENMAX_MCDI2 1020
#define    MC_CMD_WRITE32_IN_LEN(num) (4+4*(num))
#define    MC_CMD_WRITE32_IN_BUFFER_NUM(len) (((len)-4)/4)
#define       MC_CMD_WRITE32_IN_ADDR_OFST 0
#define       MC_CMD_WRITE32_IN_ADDR_LEN 4
#define       MC_CMD_WRITE32_IN_BUFFER_OFST 4
#define       MC_CMD_WRITE32_IN_BUFFER_LEN 4
#define       MC_CMD_WRITE32_IN_BUFFER_MINNUM 1
#define       MC_CMD_WRITE32_IN_BUFFER_MAXNUM 62
#define       MC_CMD_WRITE32_IN_BUFFER_MAXNUM_MCDI2 254

/* MC_CMD_WRITE32_OUT msgresponse */
#define    MC_CMD_WRITE32_OUT_LEN 0


/***********************************/
/* MC_CMD_GET_BOOT_STATUS
 * Get the instruction address from which the MC booted.
 */
#define MC_CMD_GET_BOOT_STATUS 0x5
#undef MC_CMD_0x5_PRIVILEGE_CTG

#define MC_CMD_0x5_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_GET_BOOT_STATUS_IN msgrequest */
#define    MC_CMD_GET_BOOT_STATUS_IN_LEN 0

/* MC_CMD_GET_BOOT_STATUS_OUT msgresponse */
#define    MC_CMD_GET_BOOT_STATUS_OUT_LEN 8
/* ?? */
#define       MC_CMD_GET_BOOT_STATUS_OUT_BOOT_OFFSET_OFST 0
#define       MC_CMD_GET_BOOT_STATUS_OUT_BOOT_OFFSET_LEN 4
/* enum: indicates that the MC wasn't flash booted */
#define          MC_CMD_GET_BOOT_STATUS_OUT_BOOT_OFFSET_NULL 0xdeadbeef
#define       MC_CMD_GET_BOOT_STATUS_OUT_FLAGS_OFST 4
#define       MC_CMD_GET_BOOT_STATUS_OUT_FLAGS_LEN 4
#define        MC_CMD_GET_BOOT_STATUS_OUT_FLAGS_WATCHDOG_OFST 4
#define        MC_CMD_GET_BOOT_STATUS_OUT_FLAGS_WATCHDOG_LBN 0
#define        MC_CMD_GET_BOOT_STATUS_OUT_FLAGS_WATCHDOG_WIDTH 1
#define        MC_CMD_GET_BOOT_STATUS_OUT_FLAGS_PRIMARY_OFST 4
#define        MC_CMD_GET_BOOT_STATUS_OUT_FLAGS_PRIMARY_LBN 1
#define        MC_CMD_GET_BOOT_STATUS_OUT_FLAGS_PRIMARY_WIDTH 1
#define        MC_CMD_GET_BOOT_STATUS_OUT_FLAGS_BACKUP_OFST 4
#define        MC_CMD_GET_BOOT_STATUS_OUT_FLAGS_BACKUP_LBN 2
#define        MC_CMD_GET_BOOT_STATUS_OUT_FLAGS_BACKUP_WIDTH 1


/***********************************/
/* MC_CMD_GET_ASSERTS
 * Get (and optionally clear) the current assertion status. Only
 * OUT.GLOBAL_FLAGS is guaranteed to exist in the completion payload. The other
 * fields will only be present if OUT.GLOBAL_FLAGS != NO_FAILS
 */
#define MC_CMD_GET_ASSERTS 0x6
#undef MC_CMD_0x6_PRIVILEGE_CTG

#define MC_CMD_0x6_PRIVILEGE_CTG SRIOV_CTG_ADMIN

/* MC_CMD_GET_ASSERTS_IN msgrequest */
#define    MC_CMD_GET_ASSERTS_IN_LEN 4
/* Set to clear assertion */
#define       MC_CMD_GET_ASSERTS_IN_CLEAR_OFST 0
#define       MC_CMD_GET_ASSERTS_IN_CLEAR_LEN 4

/* MC_CMD_GET_ASSERTS_OUT msgresponse */
#define    MC_CMD_GET_ASSERTS_OUT_LEN 140
/* Assertion status flag. */
#define       MC_CMD_GET_ASSERTS_OUT_GLOBAL_FLAGS_OFST 0
#define       MC_CMD_GET_ASSERTS_OUT_GLOBAL_FLAGS_LEN 4
/* enum: No assertions have failed. */
#define          MC_CMD_GET_ASSERTS_FLAGS_NO_FAILS 0x1
/* enum: A system-level assertion has failed. */
#define          MC_CMD_GET_ASSERTS_FLAGS_SYS_FAIL 0x2
/* enum: A thread-level assertion has failed. */
#define          MC_CMD_GET_ASSERTS_FLAGS_THR_FAIL 0x3
/* enum: The system was reset by the watchdog. */
#define          MC_CMD_GET_ASSERTS_FLAGS_WDOG_FIRED 0x4
/* enum: An illegal address trap stopped the system (huntington and later) */
#define          MC_CMD_GET_ASSERTS_FLAGS_ADDR_TRAP 0x5
/* Failing PC value */
#define       MC_CMD_GET_ASSERTS_OUT_SAVED_PC_OFFS_OFST 4
#define       MC_CMD_GET_ASSERTS_OUT_SAVED_PC_OFFS_LEN 4
/* Saved GP regs */
#define       MC_CMD_GET_ASSERTS_OUT_GP_REGS_OFFS_OFST 8
#define       MC_CMD_GET_ASSERTS_OUT_GP_REGS_OFFS_LEN 4
#define       MC_CMD_GET_ASSERTS_OUT_GP_REGS_OFFS_NUM 31
/* enum: A magic value hinting that the value in this register at the time of
 * the failure has likely been lost.
 */
#define          MC_CMD_GET_ASSERTS_REG_NO_DATA 0xda7a1057
/* Failing thread address */
#define       MC_CMD_GET_ASSERTS_OUT_THREAD_OFFS_OFST 132
#define       MC_CMD_GET_ASSERTS_OUT_THREAD_OFFS_LEN 4
#define       MC_CMD_GET_ASSERTS_OUT_RESERVED_OFST 136
#define       MC_CMD_GET_ASSERTS_OUT_RESERVED_LEN 4

/* MC_CMD_GET_ASSERTS_OUT_V2 msgresponse: Extended response for MicroBlaze CPUs
 * found on Riverhead designs
 */
#define    MC_CMD_GET_ASSERTS_OUT_V2_LEN 240
/* Assertion status flag. */
#define       MC_CMD_GET_ASSERTS_OUT_V2_GLOBAL_FLAGS_OFST 0
#define       MC_CMD_GET_ASSERTS_OUT_V2_GLOBAL_FLAGS_LEN 4
/* enum: No assertions have failed. */
/*               MC_CMD_GET_ASSERTS_FLAGS_NO_FAILS 0x1 */
/* enum: A system-level assertion has failed. */
/*               MC_CMD_GET_ASSERTS_FLAGS_SYS_FAIL 0x2 */
/* enum: A thread-level assertion has failed. */
/*               MC_CMD_GET_ASSERTS_FLAGS_THR_FAIL 0x3 */
/* enum: The system was reset by the watchdog. */
/*               MC_CMD_GET_ASSERTS_FLAGS_WDOG_FIRED 0x4 */
/* enum: An illegal address trap stopped the system (huntington and later) */
/*               MC_CMD_GET_ASSERTS_FLAGS_ADDR_TRAP 0x5 */
/* Failing PC value */
#define       MC_CMD_GET_ASSERTS_OUT_V2_SAVED_PC_OFFS_OFST 4
#define       MC_CMD_GET_ASSERTS_OUT_V2_SAVED_PC_OFFS_LEN 4
/* Saved GP regs */
#define       MC_CMD_GET_ASSERTS_OUT_V2_GP_REGS_OFFS_OFST 8
#define       MC_CMD_GET_ASSERTS_OUT_V2_GP_REGS_OFFS_LEN 4
#define       MC_CMD_GET_ASSERTS_OUT_V2_GP_REGS_OFFS_NUM 31
/* enum: A magic value hinting that the value in this register at the time of
 * the failure has likely been lost.
 */
/*               MC_CMD_GET_ASSERTS_REG_NO_DATA 0xda7a1057 */
/* Failing thread address */
#define       MC_CMD_GET_ASSERTS_OUT_V2_THREAD_OFFS_OFST 132
#define       MC_CMD_GET_ASSERTS_OUT_V2_THREAD_OFFS_LEN 4
#define       MC_CMD_GET_ASSERTS_OUT_V2_RESERVED_OFST 136
#define       MC_CMD_GET_ASSERTS_OUT_V2_RESERVED_LEN 4
/* Saved Special Function Registers */
#define       MC_CMD_GET_ASSERTS_OUT_V2_SF_REGS_OFFS_OFST 136
#define       MC_CMD_GET_ASSERTS_OUT_V2_SF_REGS_OFFS_LEN 4
#define       MC_CMD_GET_ASSERTS_OUT_V2_SF_REGS_OFFS_NUM 26

/* MC_CMD_GET_ASSERTS_OUT_V3 msgresponse: Extended response with asserted
 * firmware version information
 */
#define    MC_CMD_GET_ASSERTS_OUT_V3_LEN 360
/* Assertion status flag. */
#define       MC_CMD_GET_ASSERTS_OUT_V3_GLOBAL_FLAGS_OFST 0
#define       MC_CMD_GET_ASSERTS_OUT_V3_GLOBAL_FLAGS_LEN 4
/* enum: No assertions have failed. */
/*               MC_CMD_GET_ASSERTS_FLAGS_NO_FAILS 0x1 */
/* enum: A system-level assertion has failed. */
/*               MC_CMD_GET_ASSERTS_FLAGS_SYS_FAIL 0x2 */
/* enum: A thread-level assertion has failed. */
/*               MC_CMD_GET_ASSERTS_FLAGS_THR_FAIL 0x3 */
/* enum: The system was reset by the watchdog. */
/*               MC_CMD_GET_ASSERTS_FLAGS_WDOG_FIRED 0x4 */
/* enum: An illegal address trap stopped the system (huntington and later) */
/*               MC_CMD_GET_ASSERTS_FLAGS_ADDR_TRAP 0x5 */
/* Failing PC value */
#define       MC_CMD_GET_ASSERTS_OUT_V3_SAVED_PC_OFFS_OFST 4
#define       MC_CMD_GET_ASSERTS_OUT_V3_SAVED_PC_OFFS_LEN 4
/* Saved GP regs */
#define       MC_CMD_GET_ASSERTS_OUT_V3_GP_REGS_OFFS_OFST 8
#define       MC_CMD_GET_ASSERTS_OUT_V3_GP_REGS_OFFS_LEN 4
#define       MC_CMD_GET_ASSERTS_OUT_V3_GP_REGS_OFFS_NUM 31
/* enum: A magic value hinting that the value in this register at the time of
 * the failure has likely been lost.
 */
/*               MC_CMD_GET_ASSERTS_REG_NO_DATA 0xda7a1057 */
/* Failing thread address */
#define       MC_CMD_GET_ASSERTS_OUT_V3_THREAD_OFFS_OFST 132
#define       MC_CMD_GET_ASSERTS_OUT_V3_THREAD_OFFS_LEN 4
#define       MC_CMD_GET_ASSERTS_OUT_V3_RESERVED_OFST 136
#define       MC_CMD_GET_ASSERTS_OUT_V3_RESERVED_LEN 4
/* Saved Special Function Registers */
#define       MC_CMD_GET_ASSERTS_OUT_V3_SF_REGS_OFFS_OFST 136
#define       MC_CMD_GET_ASSERTS_OUT_V3_SF_REGS_OFFS_LEN 4
#define       MC_CMD_GET_ASSERTS_OUT_V3_SF_REGS_OFFS_NUM 26
/* MC firmware unique build ID (as binary SHA-1 value) */
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_BUILD_ID_OFST 240
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_BUILD_ID_LEN 20
/* MC firmware build date (as Unix timestamp) */
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_BUILD_TIMESTAMP_OFST 260
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_BUILD_TIMESTAMP_LEN 8
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_BUILD_TIMESTAMP_LO_OFST 260
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_BUILD_TIMESTAMP_LO_LEN 4
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_BUILD_TIMESTAMP_LO_LBN 2080
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_BUILD_TIMESTAMP_LO_WIDTH 32
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_BUILD_TIMESTAMP_HI_OFST 264
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_BUILD_TIMESTAMP_HI_LEN 4
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_BUILD_TIMESTAMP_HI_LBN 2112
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_BUILD_TIMESTAMP_HI_WIDTH 32
/* MC firmware version number */
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_VERSION_OFST 268
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_VERSION_LEN 8
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_VERSION_LO_OFST 268
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_VERSION_LO_LEN 4
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_VERSION_LO_LBN 2144
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_VERSION_LO_WIDTH 32
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_VERSION_HI_OFST 272
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_VERSION_HI_LEN 4
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_VERSION_HI_LBN 2176
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_VERSION_HI_WIDTH 32
/* MC firmware security level */
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_SECURITY_LEVEL_OFST 276
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_SECURITY_LEVEL_LEN 4
/* MC firmware extra version info (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_EXTRA_INFO_OFST 280
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_EXTRA_INFO_LEN 16
/* MC firmware build name (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_BUILD_NAME_OFST 296
#define       MC_CMD_GET_ASSERTS_OUT_V3_MC_FW_BUILD_NAME_LEN 64


/***********************************/
/* MC_CMD_LOG_CTRL
 * Configure the output stream for log events such as link state changes,
 * sensor notifications and MCDI completions
 */
#define MC_CMD_LOG_CTRL 0x7
#undef MC_CMD_0x7_PRIVILEGE_CTG

#define MC_CMD_0x7_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_LOG_CTRL_IN msgrequest */
#define    MC_CMD_LOG_CTRL_IN_LEN 8
/* Log destination */
#define       MC_CMD_LOG_CTRL_IN_LOG_DEST_OFST 0
#define       MC_CMD_LOG_CTRL_IN_LOG_DEST_LEN 4
/* enum property: bitmask */
/* enum: UART. */
#define          MC_CMD_LOG_CTRL_IN_LOG_DEST_UART 0x1
/* enum: Event queue. */
#define          MC_CMD_LOG_CTRL_IN_LOG_DEST_EVQ 0x2
/* Legacy argument. Must be zero. */
#define       MC_CMD_LOG_CTRL_IN_LOG_DEST_EVQ_OFST 4
#define       MC_CMD_LOG_CTRL_IN_LOG_DEST_EVQ_LEN 4

/* MC_CMD_LOG_CTRL_OUT msgresponse */
#define    MC_CMD_LOG_CTRL_OUT_LEN 0


/***********************************/
/* MC_CMD_GET_VERSION
 * Get version information about adapter components.
 */
#define MC_CMD_GET_VERSION 0x8
#undef MC_CMD_0x8_PRIVILEGE_CTG

#define MC_CMD_0x8_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_GET_VERSION_IN msgrequest */
#define    MC_CMD_GET_VERSION_IN_LEN 0

/* MC_CMD_GET_VERSION_EXT_IN msgrequest: Asks for the extended version */
#define    MC_CMD_GET_VERSION_EXT_IN_LEN 4
/* placeholder, set to 0 */
#define       MC_CMD_GET_VERSION_EXT_IN_EXT_FLAGS_OFST 0
#define       MC_CMD_GET_VERSION_EXT_IN_EXT_FLAGS_LEN 4

/* MC_CMD_GET_VERSION_V0_OUT msgresponse: deprecated version format */
#define    MC_CMD_GET_VERSION_V0_OUT_LEN 4
#define       MC_CMD_GET_VERSION_OUT_FIRMWARE_OFST 0
#define       MC_CMD_GET_VERSION_OUT_FIRMWARE_LEN 4
/* enum: Reserved version number to indicate "any" version. */
#define          MC_CMD_GET_VERSION_OUT_FIRMWARE_ANY 0xffffffff
/* enum: Bootrom version value for Siena. */
#define          MC_CMD_GET_VERSION_OUT_FIRMWARE_SIENA_BOOTROM 0xb0070000
/* enum: Bootrom version value for Huntington. */
#define          MC_CMD_GET_VERSION_OUT_FIRMWARE_HUNT_BOOTROM 0xb0070001
/* enum: Bootrom version value for Medford2. */
#define          MC_CMD_GET_VERSION_OUT_FIRMWARE_MEDFORD2_BOOTROM 0xb0070002

/* MC_CMD_GET_VERSION_OUT msgresponse */
#define    MC_CMD_GET_VERSION_OUT_LEN 32
/* This is normally the UTC build time in seconds since epoch or one of the
 * special values listed
 */
/*            MC_CMD_GET_VERSION_OUT_FIRMWARE_OFST 0 */
/*            MC_CMD_GET_VERSION_OUT_FIRMWARE_LEN 4 */
/*            Enum values, see field(s): */
/*               MC_CMD_GET_VERSION_V0_OUT/MC_CMD_GET_VERSION_OUT_FIRMWARE */
#define       MC_CMD_GET_VERSION_OUT_PCOL_OFST 4
#define       MC_CMD_GET_VERSION_OUT_PCOL_LEN 4
/* 128bit mask of functions supported by the current firmware */
#define       MC_CMD_GET_VERSION_OUT_SUPPORTED_FUNCS_OFST 8
#define       MC_CMD_GET_VERSION_OUT_SUPPORTED_FUNCS_LEN 16
#define       MC_CMD_GET_VERSION_OUT_VERSION_OFST 24
#define       MC_CMD_GET_VERSION_OUT_VERSION_LEN 8
#define       MC_CMD_GET_VERSION_OUT_VERSION_LO_OFST 24
#define       MC_CMD_GET_VERSION_OUT_VERSION_LO_LEN 4
#define       MC_CMD_GET_VERSION_OUT_VERSION_LO_LBN 192
#define       MC_CMD_GET_VERSION_OUT_VERSION_LO_WIDTH 32
#define       MC_CMD_GET_VERSION_OUT_VERSION_HI_OFST 28
#define       MC_CMD_GET_VERSION_OUT_VERSION_HI_LEN 4
#define       MC_CMD_GET_VERSION_OUT_VERSION_HI_LBN 224
#define       MC_CMD_GET_VERSION_OUT_VERSION_HI_WIDTH 32

/* MC_CMD_GET_VERSION_EXT_OUT msgresponse */
#define    MC_CMD_GET_VERSION_EXT_OUT_LEN 48
/* This is normally the UTC build time in seconds since epoch or one of the
 * special values listed
 */
/*            MC_CMD_GET_VERSION_OUT_FIRMWARE_OFST 0 */
/*            MC_CMD_GET_VERSION_OUT_FIRMWARE_LEN 4 */
/*            Enum values, see field(s): */
/*               MC_CMD_GET_VERSION_V0_OUT/MC_CMD_GET_VERSION_OUT_FIRMWARE */
#define       MC_CMD_GET_VERSION_EXT_OUT_PCOL_OFST 4
#define       MC_CMD_GET_VERSION_EXT_OUT_PCOL_LEN 4
/* 128bit mask of functions supported by the current firmware */
#define       MC_CMD_GET_VERSION_EXT_OUT_SUPPORTED_FUNCS_OFST 8
#define       MC_CMD_GET_VERSION_EXT_OUT_SUPPORTED_FUNCS_LEN 16
#define       MC_CMD_GET_VERSION_EXT_OUT_VERSION_OFST 24
#define       MC_CMD_GET_VERSION_EXT_OUT_VERSION_LEN 8
#define       MC_CMD_GET_VERSION_EXT_OUT_VERSION_LO_OFST 24
#define       MC_CMD_GET_VERSION_EXT_OUT_VERSION_LO_LEN 4
#define       MC_CMD_GET_VERSION_EXT_OUT_VERSION_LO_LBN 192
#define       MC_CMD_GET_VERSION_EXT_OUT_VERSION_LO_WIDTH 32
#define       MC_CMD_GET_VERSION_EXT_OUT_VERSION_HI_OFST 28
#define       MC_CMD_GET_VERSION_EXT_OUT_VERSION_HI_LEN 4
#define       MC_CMD_GET_VERSION_EXT_OUT_VERSION_HI_LBN 224
#define       MC_CMD_GET_VERSION_EXT_OUT_VERSION_HI_WIDTH 32
/* extra info */
#define       MC_CMD_GET_VERSION_EXT_OUT_EXTRA_OFST 32
#define       MC_CMD_GET_VERSION_EXT_OUT_EXTRA_LEN 16

/* MC_CMD_GET_VERSION_V2_OUT msgresponse: Extended response providing version
 * information for all adapter components. For Riverhead based designs, base MC
 * firmware version fields refer to NMC firmware, while CMC firmware data is in
 * dedicated CMC fields. Flags indicate which data is present in the response
 * (depending on which components exist on a particular adapter)
 */
#define    MC_CMD_GET_VERSION_V2_OUT_LEN 304
/* This is normally the UTC build time in seconds since epoch or one of the
 * special values listed
 */
/*            MC_CMD_GET_VERSION_OUT_FIRMWARE_OFST 0 */
/*            MC_CMD_GET_VERSION_OUT_FIRMWARE_LEN 4 */
/*            Enum values, see field(s): */
/*               MC_CMD_GET_VERSION_V0_OUT/MC_CMD_GET_VERSION_OUT_FIRMWARE */
#define       MC_CMD_GET_VERSION_V2_OUT_PCOL_OFST 4
#define       MC_CMD_GET_VERSION_V2_OUT_PCOL_LEN 4
/* 128bit mask of functions supported by the current firmware */
#define       MC_CMD_GET_VERSION_V2_OUT_SUPPORTED_FUNCS_OFST 8
#define       MC_CMD_GET_VERSION_V2_OUT_SUPPORTED_FUNCS_LEN 16
#define       MC_CMD_GET_VERSION_V2_OUT_VERSION_OFST 24
#define       MC_CMD_GET_VERSION_V2_OUT_VERSION_LEN 8
#define       MC_CMD_GET_VERSION_V2_OUT_VERSION_LO_OFST 24
#define       MC_CMD_GET_VERSION_V2_OUT_VERSION_LO_LEN 4
#define       MC_CMD_GET_VERSION_V2_OUT_VERSION_LO_LBN 192
#define       MC_CMD_GET_VERSION_V2_OUT_VERSION_LO_WIDTH 32
#define       MC_CMD_GET_VERSION_V2_OUT_VERSION_HI_OFST 28
#define       MC_CMD_GET_VERSION_V2_OUT_VERSION_HI_LEN 4
#define       MC_CMD_GET_VERSION_V2_OUT_VERSION_HI_LBN 224
#define       MC_CMD_GET_VERSION_V2_OUT_VERSION_HI_WIDTH 32
/* extra info */
#define       MC_CMD_GET_VERSION_V2_OUT_EXTRA_OFST 32
#define       MC_CMD_GET_VERSION_V2_OUT_EXTRA_LEN 16
/* Flags indicating which extended fields are valid */
#define       MC_CMD_GET_VERSION_V2_OUT_FLAGS_OFST 48
#define       MC_CMD_GET_VERSION_V2_OUT_FLAGS_LEN 4
#define        MC_CMD_GET_VERSION_V2_OUT_MCFW_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V2_OUT_MCFW_EXT_INFO_PRESENT_LBN 0
#define        MC_CMD_GET_VERSION_V2_OUT_MCFW_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V2_OUT_SUCFW_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V2_OUT_SUCFW_EXT_INFO_PRESENT_LBN 1
#define        MC_CMD_GET_VERSION_V2_OUT_SUCFW_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V2_OUT_CMC_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V2_OUT_CMC_EXT_INFO_PRESENT_LBN 2
#define        MC_CMD_GET_VERSION_V2_OUT_CMC_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V2_OUT_FPGA_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V2_OUT_FPGA_EXT_INFO_PRESENT_LBN 3
#define        MC_CMD_GET_VERSION_V2_OUT_FPGA_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V2_OUT_BOARD_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V2_OUT_BOARD_EXT_INFO_PRESENT_LBN 4
#define        MC_CMD_GET_VERSION_V2_OUT_BOARD_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V2_OUT_DATAPATH_HW_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V2_OUT_DATAPATH_HW_VERSION_PRESENT_LBN 5
#define        MC_CMD_GET_VERSION_V2_OUT_DATAPATH_HW_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V2_OUT_DATAPATH_FW_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V2_OUT_DATAPATH_FW_VERSION_PRESENT_LBN 6
#define        MC_CMD_GET_VERSION_V2_OUT_DATAPATH_FW_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V2_OUT_SOC_BOOT_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V2_OUT_SOC_BOOT_VERSION_PRESENT_LBN 7
#define        MC_CMD_GET_VERSION_V2_OUT_SOC_BOOT_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V2_OUT_SOC_UBOOT_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V2_OUT_SOC_UBOOT_VERSION_PRESENT_LBN 8
#define        MC_CMD_GET_VERSION_V2_OUT_SOC_UBOOT_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V2_OUT_SOC_MAIN_ROOTFS_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V2_OUT_SOC_MAIN_ROOTFS_VERSION_PRESENT_LBN 9
#define        MC_CMD_GET_VERSION_V2_OUT_SOC_MAIN_ROOTFS_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V2_OUT_SOC_RECOVERY_BUILDROOT_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V2_OUT_SOC_RECOVERY_BUILDROOT_VERSION_PRESENT_LBN 10
#define        MC_CMD_GET_VERSION_V2_OUT_SOC_RECOVERY_BUILDROOT_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V2_OUT_SUCFW_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V2_OUT_SUCFW_VERSION_PRESENT_LBN 11
#define        MC_CMD_GET_VERSION_V2_OUT_SUCFW_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V2_OUT_BOARD_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V2_OUT_BOARD_VERSION_PRESENT_LBN 12
#define        MC_CMD_GET_VERSION_V2_OUT_BOARD_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V2_OUT_BUNDLE_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V2_OUT_BUNDLE_VERSION_PRESENT_LBN 13
#define        MC_CMD_GET_VERSION_V2_OUT_BUNDLE_VERSION_PRESENT_WIDTH 1
/* MC firmware unique build ID (as binary SHA-1 value) */
#define       MC_CMD_GET_VERSION_V2_OUT_MCFW_BUILD_ID_OFST 52
#define       MC_CMD_GET_VERSION_V2_OUT_MCFW_BUILD_ID_LEN 20
/* MC firmware security level */
#define       MC_CMD_GET_VERSION_V2_OUT_MCFW_SECURITY_LEVEL_OFST 72
#define       MC_CMD_GET_VERSION_V2_OUT_MCFW_SECURITY_LEVEL_LEN 4
/* MC firmware build name (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_VERSION_V2_OUT_MCFW_BUILD_NAME_OFST 76
#define       MC_CMD_GET_VERSION_V2_OUT_MCFW_BUILD_NAME_LEN 64
/* The SUC firmware version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V2_OUT_SUCFW_VERSION_OFST 140
#define       MC_CMD_GET_VERSION_V2_OUT_SUCFW_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V2_OUT_SUCFW_VERSION_NUM 4
/* SUC firmware build date (as 64-bit Unix timestamp) */
#define       MC_CMD_GET_VERSION_V2_OUT_SUCFW_BUILD_DATE_OFST 156
#define       MC_CMD_GET_VERSION_V2_OUT_SUCFW_BUILD_DATE_LEN 8
#define       MC_CMD_GET_VERSION_V2_OUT_SUCFW_BUILD_DATE_LO_OFST 156
#define       MC_CMD_GET_VERSION_V2_OUT_SUCFW_BUILD_DATE_LO_LEN 4
#define       MC_CMD_GET_VERSION_V2_OUT_SUCFW_BUILD_DATE_LO_LBN 1248
#define       MC_CMD_GET_VERSION_V2_OUT_SUCFW_BUILD_DATE_LO_WIDTH 32
#define       MC_CMD_GET_VERSION_V2_OUT_SUCFW_BUILD_DATE_HI_OFST 160
#define       MC_CMD_GET_VERSION_V2_OUT_SUCFW_BUILD_DATE_HI_LEN 4
#define       MC_CMD_GET_VERSION_V2_OUT_SUCFW_BUILD_DATE_HI_LBN 1280
#define       MC_CMD_GET_VERSION_V2_OUT_SUCFW_BUILD_DATE_HI_WIDTH 32
/* The ID of the SUC chip. This is specific to the platform but typically
 * indicates family, memory sizes etc. See SF-116728-SW for further details.
 */
#define       MC_CMD_GET_VERSION_V2_OUT_SUCFW_CHIP_ID_OFST 164
#define       MC_CMD_GET_VERSION_V2_OUT_SUCFW_CHIP_ID_LEN 4
/* The CMC firmware version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V2_OUT_CMCFW_VERSION_OFST 168
#define       MC_CMD_GET_VERSION_V2_OUT_CMCFW_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V2_OUT_CMCFW_VERSION_NUM 4
/* CMC firmware build date (as 64-bit Unix timestamp) */
#define       MC_CMD_GET_VERSION_V2_OUT_CMCFW_BUILD_DATE_OFST 184
#define       MC_CMD_GET_VERSION_V2_OUT_CMCFW_BUILD_DATE_LEN 8
#define       MC_CMD_GET_VERSION_V2_OUT_CMCFW_BUILD_DATE_LO_OFST 184
#define       MC_CMD_GET_VERSION_V2_OUT_CMCFW_BUILD_DATE_LO_LEN 4
#define       MC_CMD_GET_VERSION_V2_OUT_CMCFW_BUILD_DATE_LO_LBN 1472
#define       MC_CMD_GET_VERSION_V2_OUT_CMCFW_BUILD_DATE_LO_WIDTH 32
#define       MC_CMD_GET_VERSION_V2_OUT_CMCFW_BUILD_DATE_HI_OFST 188
#define       MC_CMD_GET_VERSION_V2_OUT_CMCFW_BUILD_DATE_HI_LEN 4
#define       MC_CMD_GET_VERSION_V2_OUT_CMCFW_BUILD_DATE_HI_LBN 1504
#define       MC_CMD_GET_VERSION_V2_OUT_CMCFW_BUILD_DATE_HI_WIDTH 32
/* FPGA version as three numbers. On Riverhead based systems this field uses
 * the same encoding as hardware version ID registers (MC_FPGA_BUILD_HWRD_REG):
 * FPGA_VERSION[0]: x => Image H{x} FPGA_VERSION[1]: Revision letter (0 => A, 1
 * => B, ...) FPGA_VERSION[2]: Sub-revision number
 */
#define       MC_CMD_GET_VERSION_V2_OUT_FPGA_VERSION_OFST 192
#define       MC_CMD_GET_VERSION_V2_OUT_FPGA_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V2_OUT_FPGA_VERSION_NUM 3
/* Extra FPGA revision information (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_VERSION_V2_OUT_FPGA_EXTRA_OFST 204
#define       MC_CMD_GET_VERSION_V2_OUT_FPGA_EXTRA_LEN 16
/* Board name / adapter model (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_VERSION_V2_OUT_BOARD_NAME_OFST 220
#define       MC_CMD_GET_VERSION_V2_OUT_BOARD_NAME_LEN 16
/* Board revision number */
#define       MC_CMD_GET_VERSION_V2_OUT_BOARD_REVISION_OFST 236
#define       MC_CMD_GET_VERSION_V2_OUT_BOARD_REVISION_LEN 4
/* Board serial number (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_VERSION_V2_OUT_BOARD_SERIAL_OFST 240
#define       MC_CMD_GET_VERSION_V2_OUT_BOARD_SERIAL_LEN 64

/* MC_CMD_GET_VERSION_V3_OUT msgresponse: Extended response providing version
 * information for all adapter components. For Riverhead based designs, base MC
 * firmware version fields refer to NMC firmware, while CMC firmware data is in
 * dedicated CMC fields. Flags indicate which data is present in the response
 * (depending on which components exist on a particular adapter)
 */
#define    MC_CMD_GET_VERSION_V3_OUT_LEN 328
/* This is normally the UTC build time in seconds since epoch or one of the
 * special values listed
 */
/*            MC_CMD_GET_VERSION_OUT_FIRMWARE_OFST 0 */
/*            MC_CMD_GET_VERSION_OUT_FIRMWARE_LEN 4 */
/*            Enum values, see field(s): */
/*               MC_CMD_GET_VERSION_V0_OUT/MC_CMD_GET_VERSION_OUT_FIRMWARE */
#define       MC_CMD_GET_VERSION_V3_OUT_PCOL_OFST 4
#define       MC_CMD_GET_VERSION_V3_OUT_PCOL_LEN 4
/* 128bit mask of functions supported by the current firmware */
#define       MC_CMD_GET_VERSION_V3_OUT_SUPPORTED_FUNCS_OFST 8
#define       MC_CMD_GET_VERSION_V3_OUT_SUPPORTED_FUNCS_LEN 16
#define       MC_CMD_GET_VERSION_V3_OUT_VERSION_OFST 24
#define       MC_CMD_GET_VERSION_V3_OUT_VERSION_LEN 8
#define       MC_CMD_GET_VERSION_V3_OUT_VERSION_LO_OFST 24
#define       MC_CMD_GET_VERSION_V3_OUT_VERSION_LO_LEN 4
#define       MC_CMD_GET_VERSION_V3_OUT_VERSION_LO_LBN 192
#define       MC_CMD_GET_VERSION_V3_OUT_VERSION_LO_WIDTH 32
#define       MC_CMD_GET_VERSION_V3_OUT_VERSION_HI_OFST 28
#define       MC_CMD_GET_VERSION_V3_OUT_VERSION_HI_LEN 4
#define       MC_CMD_GET_VERSION_V3_OUT_VERSION_HI_LBN 224
#define       MC_CMD_GET_VERSION_V3_OUT_VERSION_HI_WIDTH 32
/* extra info */
#define       MC_CMD_GET_VERSION_V3_OUT_EXTRA_OFST 32
#define       MC_CMD_GET_VERSION_V3_OUT_EXTRA_LEN 16
/* Flags indicating which extended fields are valid */
#define       MC_CMD_GET_VERSION_V3_OUT_FLAGS_OFST 48
#define       MC_CMD_GET_VERSION_V3_OUT_FLAGS_LEN 4
#define        MC_CMD_GET_VERSION_V3_OUT_MCFW_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V3_OUT_MCFW_EXT_INFO_PRESENT_LBN 0
#define        MC_CMD_GET_VERSION_V3_OUT_MCFW_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V3_OUT_SUCFW_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V3_OUT_SUCFW_EXT_INFO_PRESENT_LBN 1
#define        MC_CMD_GET_VERSION_V3_OUT_SUCFW_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V3_OUT_CMC_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V3_OUT_CMC_EXT_INFO_PRESENT_LBN 2
#define        MC_CMD_GET_VERSION_V3_OUT_CMC_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V3_OUT_FPGA_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V3_OUT_FPGA_EXT_INFO_PRESENT_LBN 3
#define        MC_CMD_GET_VERSION_V3_OUT_FPGA_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V3_OUT_BOARD_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V3_OUT_BOARD_EXT_INFO_PRESENT_LBN 4
#define        MC_CMD_GET_VERSION_V3_OUT_BOARD_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V3_OUT_DATAPATH_HW_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V3_OUT_DATAPATH_HW_VERSION_PRESENT_LBN 5
#define        MC_CMD_GET_VERSION_V3_OUT_DATAPATH_HW_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V3_OUT_DATAPATH_FW_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V3_OUT_DATAPATH_FW_VERSION_PRESENT_LBN 6
#define        MC_CMD_GET_VERSION_V3_OUT_DATAPATH_FW_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V3_OUT_SOC_BOOT_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V3_OUT_SOC_BOOT_VERSION_PRESENT_LBN 7
#define        MC_CMD_GET_VERSION_V3_OUT_SOC_BOOT_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V3_OUT_SOC_UBOOT_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V3_OUT_SOC_UBOOT_VERSION_PRESENT_LBN 8
#define        MC_CMD_GET_VERSION_V3_OUT_SOC_UBOOT_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V3_OUT_SOC_MAIN_ROOTFS_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V3_OUT_SOC_MAIN_ROOTFS_VERSION_PRESENT_LBN 9
#define        MC_CMD_GET_VERSION_V3_OUT_SOC_MAIN_ROOTFS_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V3_OUT_SOC_RECOVERY_BUILDROOT_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V3_OUT_SOC_RECOVERY_BUILDROOT_VERSION_PRESENT_LBN 10
#define        MC_CMD_GET_VERSION_V3_OUT_SOC_RECOVERY_BUILDROOT_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V3_OUT_SUCFW_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V3_OUT_SUCFW_VERSION_PRESENT_LBN 11
#define        MC_CMD_GET_VERSION_V3_OUT_SUCFW_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V3_OUT_BOARD_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V3_OUT_BOARD_VERSION_PRESENT_LBN 12
#define        MC_CMD_GET_VERSION_V3_OUT_BOARD_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V3_OUT_BUNDLE_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V3_OUT_BUNDLE_VERSION_PRESENT_LBN 13
#define        MC_CMD_GET_VERSION_V3_OUT_BUNDLE_VERSION_PRESENT_WIDTH 1
/* MC firmware unique build ID (as binary SHA-1 value) */
#define       MC_CMD_GET_VERSION_V3_OUT_MCFW_BUILD_ID_OFST 52
#define       MC_CMD_GET_VERSION_V3_OUT_MCFW_BUILD_ID_LEN 20
/* MC firmware security level */
#define       MC_CMD_GET_VERSION_V3_OUT_MCFW_SECURITY_LEVEL_OFST 72
#define       MC_CMD_GET_VERSION_V3_OUT_MCFW_SECURITY_LEVEL_LEN 4
/* MC firmware build name (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_VERSION_V3_OUT_MCFW_BUILD_NAME_OFST 76
#define       MC_CMD_GET_VERSION_V3_OUT_MCFW_BUILD_NAME_LEN 64
/* The SUC firmware version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V3_OUT_SUCFW_VERSION_OFST 140
#define       MC_CMD_GET_VERSION_V3_OUT_SUCFW_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V3_OUT_SUCFW_VERSION_NUM 4
/* SUC firmware build date (as 64-bit Unix timestamp) */
#define       MC_CMD_GET_VERSION_V3_OUT_SUCFW_BUILD_DATE_OFST 156
#define       MC_CMD_GET_VERSION_V3_OUT_SUCFW_BUILD_DATE_LEN 8
#define       MC_CMD_GET_VERSION_V3_OUT_SUCFW_BUILD_DATE_LO_OFST 156
#define       MC_CMD_GET_VERSION_V3_OUT_SUCFW_BUILD_DATE_LO_LEN 4
#define       MC_CMD_GET_VERSION_V3_OUT_SUCFW_BUILD_DATE_LO_LBN 1248
#define       MC_CMD_GET_VERSION_V3_OUT_SUCFW_BUILD_DATE_LO_WIDTH 32
#define       MC_CMD_GET_VERSION_V3_OUT_SUCFW_BUILD_DATE_HI_OFST 160
#define       MC_CMD_GET_VERSION_V3_OUT_SUCFW_BUILD_DATE_HI_LEN 4
#define       MC_CMD_GET_VERSION_V3_OUT_SUCFW_BUILD_DATE_HI_LBN 1280
#define       MC_CMD_GET_VERSION_V3_OUT_SUCFW_BUILD_DATE_HI_WIDTH 32
/* The ID of the SUC chip. This is specific to the platform but typically
 * indicates family, memory sizes etc. See SF-116728-SW for further details.
 */
#define       MC_CMD_GET_VERSION_V3_OUT_SUCFW_CHIP_ID_OFST 164
#define       MC_CMD_GET_VERSION_V3_OUT_SUCFW_CHIP_ID_LEN 4
/* The CMC firmware version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V3_OUT_CMCFW_VERSION_OFST 168
#define       MC_CMD_GET_VERSION_V3_OUT_CMCFW_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V3_OUT_CMCFW_VERSION_NUM 4
/* CMC firmware build date (as 64-bit Unix timestamp) */
#define       MC_CMD_GET_VERSION_V3_OUT_CMCFW_BUILD_DATE_OFST 184
#define       MC_CMD_GET_VERSION_V3_OUT_CMCFW_BUILD_DATE_LEN 8
#define       MC_CMD_GET_VERSION_V3_OUT_CMCFW_BUILD_DATE_LO_OFST 184
#define       MC_CMD_GET_VERSION_V3_OUT_CMCFW_BUILD_DATE_LO_LEN 4
#define       MC_CMD_GET_VERSION_V3_OUT_CMCFW_BUILD_DATE_LO_LBN 1472
#define       MC_CMD_GET_VERSION_V3_OUT_CMCFW_BUILD_DATE_LO_WIDTH 32
#define       MC_CMD_GET_VERSION_V3_OUT_CMCFW_BUILD_DATE_HI_OFST 188
#define       MC_CMD_GET_VERSION_V3_OUT_CMCFW_BUILD_DATE_HI_LEN 4
#define       MC_CMD_GET_VERSION_V3_OUT_CMCFW_BUILD_DATE_HI_LBN 1504
#define       MC_CMD_GET_VERSION_V3_OUT_CMCFW_BUILD_DATE_HI_WIDTH 32
/* FPGA version as three numbers. On Riverhead based systems this field uses
 * the same encoding as hardware version ID registers (MC_FPGA_BUILD_HWRD_REG):
 * FPGA_VERSION[0]: x => Image H{x} FPGA_VERSION[1]: Revision letter (0 => A, 1
 * => B, ...) FPGA_VERSION[2]: Sub-revision number
 */
#define       MC_CMD_GET_VERSION_V3_OUT_FPGA_VERSION_OFST 192
#define       MC_CMD_GET_VERSION_V3_OUT_FPGA_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V3_OUT_FPGA_VERSION_NUM 3
/* Extra FPGA revision information (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_VERSION_V3_OUT_FPGA_EXTRA_OFST 204
#define       MC_CMD_GET_VERSION_V3_OUT_FPGA_EXTRA_LEN 16
/* Board name / adapter model (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_VERSION_V3_OUT_BOARD_NAME_OFST 220
#define       MC_CMD_GET_VERSION_V3_OUT_BOARD_NAME_LEN 16
/* Board revision number */
#define       MC_CMD_GET_VERSION_V3_OUT_BOARD_REVISION_OFST 236
#define       MC_CMD_GET_VERSION_V3_OUT_BOARD_REVISION_LEN 4
/* Board serial number (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_VERSION_V3_OUT_BOARD_SERIAL_OFST 240
#define       MC_CMD_GET_VERSION_V3_OUT_BOARD_SERIAL_LEN 64
/* The version of the datapath hardware design as three number - a.b.c */
#define       MC_CMD_GET_VERSION_V3_OUT_DATAPATH_HW_VERSION_OFST 304
#define       MC_CMD_GET_VERSION_V3_OUT_DATAPATH_HW_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V3_OUT_DATAPATH_HW_VERSION_NUM 3
/* The version of the firmware library used to control the datapath as three
 * number - a.b.c
 */
#define       MC_CMD_GET_VERSION_V3_OUT_DATAPATH_FW_VERSION_OFST 316
#define       MC_CMD_GET_VERSION_V3_OUT_DATAPATH_FW_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V3_OUT_DATAPATH_FW_VERSION_NUM 3

/* MC_CMD_GET_VERSION_V4_OUT msgresponse: Extended response providing SoC
 * version information
 */
#define    MC_CMD_GET_VERSION_V4_OUT_LEN 392
/* This is normally the UTC build time in seconds since epoch or one of the
 * special values listed
 */
/*            MC_CMD_GET_VERSION_OUT_FIRMWARE_OFST 0 */
/*            MC_CMD_GET_VERSION_OUT_FIRMWARE_LEN 4 */
/*            Enum values, see field(s): */
/*               MC_CMD_GET_VERSION_V0_OUT/MC_CMD_GET_VERSION_OUT_FIRMWARE */
#define       MC_CMD_GET_VERSION_V4_OUT_PCOL_OFST 4
#define       MC_CMD_GET_VERSION_V4_OUT_PCOL_LEN 4
/* 128bit mask of functions supported by the current firmware */
#define       MC_CMD_GET_VERSION_V4_OUT_SUPPORTED_FUNCS_OFST 8
#define       MC_CMD_GET_VERSION_V4_OUT_SUPPORTED_FUNCS_LEN 16
#define       MC_CMD_GET_VERSION_V4_OUT_VERSION_OFST 24
#define       MC_CMD_GET_VERSION_V4_OUT_VERSION_LEN 8
#define       MC_CMD_GET_VERSION_V4_OUT_VERSION_LO_OFST 24
#define       MC_CMD_GET_VERSION_V4_OUT_VERSION_LO_LEN 4
#define       MC_CMD_GET_VERSION_V4_OUT_VERSION_LO_LBN 192
#define       MC_CMD_GET_VERSION_V4_OUT_VERSION_LO_WIDTH 32
#define       MC_CMD_GET_VERSION_V4_OUT_VERSION_HI_OFST 28
#define       MC_CMD_GET_VERSION_V4_OUT_VERSION_HI_LEN 4
#define       MC_CMD_GET_VERSION_V4_OUT_VERSION_HI_LBN 224
#define       MC_CMD_GET_VERSION_V4_OUT_VERSION_HI_WIDTH 32
/* extra info */
#define       MC_CMD_GET_VERSION_V4_OUT_EXTRA_OFST 32
#define       MC_CMD_GET_VERSION_V4_OUT_EXTRA_LEN 16
/* Flags indicating which extended fields are valid */
#define       MC_CMD_GET_VERSION_V4_OUT_FLAGS_OFST 48
#define       MC_CMD_GET_VERSION_V4_OUT_FLAGS_LEN 4
#define        MC_CMD_GET_VERSION_V4_OUT_MCFW_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V4_OUT_MCFW_EXT_INFO_PRESENT_LBN 0
#define        MC_CMD_GET_VERSION_V4_OUT_MCFW_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V4_OUT_SUCFW_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V4_OUT_SUCFW_EXT_INFO_PRESENT_LBN 1
#define        MC_CMD_GET_VERSION_V4_OUT_SUCFW_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V4_OUT_CMC_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V4_OUT_CMC_EXT_INFO_PRESENT_LBN 2
#define        MC_CMD_GET_VERSION_V4_OUT_CMC_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V4_OUT_FPGA_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V4_OUT_FPGA_EXT_INFO_PRESENT_LBN 3
#define        MC_CMD_GET_VERSION_V4_OUT_FPGA_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V4_OUT_BOARD_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V4_OUT_BOARD_EXT_INFO_PRESENT_LBN 4
#define        MC_CMD_GET_VERSION_V4_OUT_BOARD_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V4_OUT_DATAPATH_HW_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V4_OUT_DATAPATH_HW_VERSION_PRESENT_LBN 5
#define        MC_CMD_GET_VERSION_V4_OUT_DATAPATH_HW_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V4_OUT_DATAPATH_FW_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V4_OUT_DATAPATH_FW_VERSION_PRESENT_LBN 6
#define        MC_CMD_GET_VERSION_V4_OUT_DATAPATH_FW_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V4_OUT_SOC_BOOT_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V4_OUT_SOC_BOOT_VERSION_PRESENT_LBN 7
#define        MC_CMD_GET_VERSION_V4_OUT_SOC_BOOT_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V4_OUT_SOC_UBOOT_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V4_OUT_SOC_UBOOT_VERSION_PRESENT_LBN 8
#define        MC_CMD_GET_VERSION_V4_OUT_SOC_UBOOT_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V4_OUT_SOC_MAIN_ROOTFS_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V4_OUT_SOC_MAIN_ROOTFS_VERSION_PRESENT_LBN 9
#define        MC_CMD_GET_VERSION_V4_OUT_SOC_MAIN_ROOTFS_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V4_OUT_SOC_RECOVERY_BUILDROOT_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V4_OUT_SOC_RECOVERY_BUILDROOT_VERSION_PRESENT_LBN 10
#define        MC_CMD_GET_VERSION_V4_OUT_SOC_RECOVERY_BUILDROOT_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V4_OUT_SUCFW_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V4_OUT_SUCFW_VERSION_PRESENT_LBN 11
#define        MC_CMD_GET_VERSION_V4_OUT_SUCFW_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V4_OUT_BOARD_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V4_OUT_BOARD_VERSION_PRESENT_LBN 12
#define        MC_CMD_GET_VERSION_V4_OUT_BOARD_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V4_OUT_BUNDLE_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V4_OUT_BUNDLE_VERSION_PRESENT_LBN 13
#define        MC_CMD_GET_VERSION_V4_OUT_BUNDLE_VERSION_PRESENT_WIDTH 1
/* MC firmware unique build ID (as binary SHA-1 value) */
#define       MC_CMD_GET_VERSION_V4_OUT_MCFW_BUILD_ID_OFST 52
#define       MC_CMD_GET_VERSION_V4_OUT_MCFW_BUILD_ID_LEN 20
/* MC firmware security level */
#define       MC_CMD_GET_VERSION_V4_OUT_MCFW_SECURITY_LEVEL_OFST 72
#define       MC_CMD_GET_VERSION_V4_OUT_MCFW_SECURITY_LEVEL_LEN 4
/* MC firmware build name (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_VERSION_V4_OUT_MCFW_BUILD_NAME_OFST 76
#define       MC_CMD_GET_VERSION_V4_OUT_MCFW_BUILD_NAME_LEN 64
/* The SUC firmware version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V4_OUT_SUCFW_VERSION_OFST 140
#define       MC_CMD_GET_VERSION_V4_OUT_SUCFW_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V4_OUT_SUCFW_VERSION_NUM 4
/* SUC firmware build date (as 64-bit Unix timestamp) */
#define       MC_CMD_GET_VERSION_V4_OUT_SUCFW_BUILD_DATE_OFST 156
#define       MC_CMD_GET_VERSION_V4_OUT_SUCFW_BUILD_DATE_LEN 8
#define       MC_CMD_GET_VERSION_V4_OUT_SUCFW_BUILD_DATE_LO_OFST 156
#define       MC_CMD_GET_VERSION_V4_OUT_SUCFW_BUILD_DATE_LO_LEN 4
#define       MC_CMD_GET_VERSION_V4_OUT_SUCFW_BUILD_DATE_LO_LBN 1248
#define       MC_CMD_GET_VERSION_V4_OUT_SUCFW_BUILD_DATE_LO_WIDTH 32
#define       MC_CMD_GET_VERSION_V4_OUT_SUCFW_BUILD_DATE_HI_OFST 160
#define       MC_CMD_GET_VERSION_V4_OUT_SUCFW_BUILD_DATE_HI_LEN 4
#define       MC_CMD_GET_VERSION_V4_OUT_SUCFW_BUILD_DATE_HI_LBN 1280
#define       MC_CMD_GET_VERSION_V4_OUT_SUCFW_BUILD_DATE_HI_WIDTH 32
/* The ID of the SUC chip. This is specific to the platform but typically
 * indicates family, memory sizes etc. See SF-116728-SW for further details.
 */
#define       MC_CMD_GET_VERSION_V4_OUT_SUCFW_CHIP_ID_OFST 164
#define       MC_CMD_GET_VERSION_V4_OUT_SUCFW_CHIP_ID_LEN 4
/* The CMC firmware version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V4_OUT_CMCFW_VERSION_OFST 168
#define       MC_CMD_GET_VERSION_V4_OUT_CMCFW_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V4_OUT_CMCFW_VERSION_NUM 4
/* CMC firmware build date (as 64-bit Unix timestamp) */
#define       MC_CMD_GET_VERSION_V4_OUT_CMCFW_BUILD_DATE_OFST 184
#define       MC_CMD_GET_VERSION_V4_OUT_CMCFW_BUILD_DATE_LEN 8
#define       MC_CMD_GET_VERSION_V4_OUT_CMCFW_BUILD_DATE_LO_OFST 184
#define       MC_CMD_GET_VERSION_V4_OUT_CMCFW_BUILD_DATE_LO_LEN 4
#define       MC_CMD_GET_VERSION_V4_OUT_CMCFW_BUILD_DATE_LO_LBN 1472
#define       MC_CMD_GET_VERSION_V4_OUT_CMCFW_BUILD_DATE_LO_WIDTH 32
#define       MC_CMD_GET_VERSION_V4_OUT_CMCFW_BUILD_DATE_HI_OFST 188
#define       MC_CMD_GET_VERSION_V4_OUT_CMCFW_BUILD_DATE_HI_LEN 4
#define       MC_CMD_GET_VERSION_V4_OUT_CMCFW_BUILD_DATE_HI_LBN 1504
#define       MC_CMD_GET_VERSION_V4_OUT_CMCFW_BUILD_DATE_HI_WIDTH 32
/* FPGA version as three numbers. On Riverhead based systems this field uses
 * the same encoding as hardware version ID registers (MC_FPGA_BUILD_HWRD_REG):
 * FPGA_VERSION[0]: x => Image H{x} FPGA_VERSION[1]: Revision letter (0 => A, 1
 * => B, ...) FPGA_VERSION[2]: Sub-revision number
 */
#define       MC_CMD_GET_VERSION_V4_OUT_FPGA_VERSION_OFST 192
#define       MC_CMD_GET_VERSION_V4_OUT_FPGA_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V4_OUT_FPGA_VERSION_NUM 3
/* Extra FPGA revision information (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_VERSION_V4_OUT_FPGA_EXTRA_OFST 204
#define       MC_CMD_GET_VERSION_V4_OUT_FPGA_EXTRA_LEN 16
/* Board name / adapter model (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_VERSION_V4_OUT_BOARD_NAME_OFST 220
#define       MC_CMD_GET_VERSION_V4_OUT_BOARD_NAME_LEN 16
/* Board revision number */
#define       MC_CMD_GET_VERSION_V4_OUT_BOARD_REVISION_OFST 236
#define       MC_CMD_GET_VERSION_V4_OUT_BOARD_REVISION_LEN 4
/* Board serial number (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_VERSION_V4_OUT_BOARD_SERIAL_OFST 240
#define       MC_CMD_GET_VERSION_V4_OUT_BOARD_SERIAL_LEN 64
/* The version of the datapath hardware design as three number - a.b.c */
#define       MC_CMD_GET_VERSION_V4_OUT_DATAPATH_HW_VERSION_OFST 304
#define       MC_CMD_GET_VERSION_V4_OUT_DATAPATH_HW_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V4_OUT_DATAPATH_HW_VERSION_NUM 3
/* The version of the firmware library used to control the datapath as three
 * number - a.b.c
 */
#define       MC_CMD_GET_VERSION_V4_OUT_DATAPATH_FW_VERSION_OFST 316
#define       MC_CMD_GET_VERSION_V4_OUT_DATAPATH_FW_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V4_OUT_DATAPATH_FW_VERSION_NUM 3
/* The SOC boot version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V4_OUT_SOC_BOOT_VERSION_OFST 328
#define       MC_CMD_GET_VERSION_V4_OUT_SOC_BOOT_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V4_OUT_SOC_BOOT_VERSION_NUM 4
/* The SOC uboot version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V4_OUT_SOC_UBOOT_VERSION_OFST 344
#define       MC_CMD_GET_VERSION_V4_OUT_SOC_UBOOT_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V4_OUT_SOC_UBOOT_VERSION_NUM 4
/* The SOC main rootfs version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V4_OUT_SOC_MAIN_ROOTFS_VERSION_OFST 360
#define       MC_CMD_GET_VERSION_V4_OUT_SOC_MAIN_ROOTFS_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V4_OUT_SOC_MAIN_ROOTFS_VERSION_NUM 4
/* The SOC recovery buildroot version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V4_OUT_SOC_RECOVERY_BUILDROOT_VERSION_OFST 376
#define       MC_CMD_GET_VERSION_V4_OUT_SOC_RECOVERY_BUILDROOT_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V4_OUT_SOC_RECOVERY_BUILDROOT_VERSION_NUM 4

/* MC_CMD_GET_VERSION_V5_OUT msgresponse: Extended response providing bundle
 * and board version information
 */
#define    MC_CMD_GET_VERSION_V5_OUT_LEN 424
/* This is normally the UTC build time in seconds since epoch or one of the
 * special values listed
 */
/*            MC_CMD_GET_VERSION_OUT_FIRMWARE_OFST 0 */
/*            MC_CMD_GET_VERSION_OUT_FIRMWARE_LEN 4 */
/*            Enum values, see field(s): */
/*               MC_CMD_GET_VERSION_V0_OUT/MC_CMD_GET_VERSION_OUT_FIRMWARE */
#define       MC_CMD_GET_VERSION_V5_OUT_PCOL_OFST 4
#define       MC_CMD_GET_VERSION_V5_OUT_PCOL_LEN 4
/* 128bit mask of functions supported by the current firmware */
#define       MC_CMD_GET_VERSION_V5_OUT_SUPPORTED_FUNCS_OFST 8
#define       MC_CMD_GET_VERSION_V5_OUT_SUPPORTED_FUNCS_LEN 16
#define       MC_CMD_GET_VERSION_V5_OUT_VERSION_OFST 24
#define       MC_CMD_GET_VERSION_V5_OUT_VERSION_LEN 8
#define       MC_CMD_GET_VERSION_V5_OUT_VERSION_LO_OFST 24
#define       MC_CMD_GET_VERSION_V5_OUT_VERSION_LO_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_VERSION_LO_LBN 192
#define       MC_CMD_GET_VERSION_V5_OUT_VERSION_LO_WIDTH 32
#define       MC_CMD_GET_VERSION_V5_OUT_VERSION_HI_OFST 28
#define       MC_CMD_GET_VERSION_V5_OUT_VERSION_HI_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_VERSION_HI_LBN 224
#define       MC_CMD_GET_VERSION_V5_OUT_VERSION_HI_WIDTH 32
/* extra info */
#define       MC_CMD_GET_VERSION_V5_OUT_EXTRA_OFST 32
#define       MC_CMD_GET_VERSION_V5_OUT_EXTRA_LEN 16
/* Flags indicating which extended fields are valid */
#define       MC_CMD_GET_VERSION_V5_OUT_FLAGS_OFST 48
#define       MC_CMD_GET_VERSION_V5_OUT_FLAGS_LEN 4
#define        MC_CMD_GET_VERSION_V5_OUT_MCFW_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V5_OUT_MCFW_EXT_INFO_PRESENT_LBN 0
#define        MC_CMD_GET_VERSION_V5_OUT_MCFW_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V5_OUT_SUCFW_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V5_OUT_SUCFW_EXT_INFO_PRESENT_LBN 1
#define        MC_CMD_GET_VERSION_V5_OUT_SUCFW_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V5_OUT_CMC_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V5_OUT_CMC_EXT_INFO_PRESENT_LBN 2
#define        MC_CMD_GET_VERSION_V5_OUT_CMC_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V5_OUT_FPGA_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V5_OUT_FPGA_EXT_INFO_PRESENT_LBN 3
#define        MC_CMD_GET_VERSION_V5_OUT_FPGA_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V5_OUT_BOARD_EXT_INFO_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V5_OUT_BOARD_EXT_INFO_PRESENT_LBN 4
#define        MC_CMD_GET_VERSION_V5_OUT_BOARD_EXT_INFO_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V5_OUT_DATAPATH_HW_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V5_OUT_DATAPATH_HW_VERSION_PRESENT_LBN 5
#define        MC_CMD_GET_VERSION_V5_OUT_DATAPATH_HW_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V5_OUT_DATAPATH_FW_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V5_OUT_DATAPATH_FW_VERSION_PRESENT_LBN 6
#define        MC_CMD_GET_VERSION_V5_OUT_DATAPATH_FW_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V5_OUT_SOC_BOOT_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V5_OUT_SOC_BOOT_VERSION_PRESENT_LBN 7
#define        MC_CMD_GET_VERSION_V5_OUT_SOC_BOOT_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V5_OUT_SOC_UBOOT_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V5_OUT_SOC_UBOOT_VERSION_PRESENT_LBN 8
#define        MC_CMD_GET_VERSION_V5_OUT_SOC_UBOOT_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V5_OUT_SOC_MAIN_ROOTFS_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V5_OUT_SOC_MAIN_ROOTFS_VERSION_PRESENT_LBN 9
#define        MC_CMD_GET_VERSION_V5_OUT_SOC_MAIN_ROOTFS_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V5_OUT_SOC_RECOVERY_BUILDROOT_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V5_OUT_SOC_RECOVERY_BUILDROOT_VERSION_PRESENT_LBN 10
#define        MC_CMD_GET_VERSION_V5_OUT_SOC_RECOVERY_BUILDROOT_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V5_OUT_SUCFW_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V5_OUT_SUCFW_VERSION_PRESENT_LBN 11
#define        MC_CMD_GET_VERSION_V5_OUT_SUCFW_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V5_OUT_BOARD_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V5_OUT_BOARD_VERSION_PRESENT_LBN 12
#define        MC_CMD_GET_VERSION_V5_OUT_BOARD_VERSION_PRESENT_WIDTH 1
#define        MC_CMD_GET_VERSION_V5_OUT_BUNDLE_VERSION_PRESENT_OFST 48
#define        MC_CMD_GET_VERSION_V5_OUT_BUNDLE_VERSION_PRESENT_LBN 13
#define        MC_CMD_GET_VERSION_V5_OUT_BUNDLE_VERSION_PRESENT_WIDTH 1
/* MC firmware unique build ID (as binary SHA-1 value) */
#define       MC_CMD_GET_VERSION_V5_OUT_MCFW_BUILD_ID_OFST 52
#define       MC_CMD_GET_VERSION_V5_OUT_MCFW_BUILD_ID_LEN 20
/* MC firmware security level */
#define       MC_CMD_GET_VERSION_V5_OUT_MCFW_SECURITY_LEVEL_OFST 72
#define       MC_CMD_GET_VERSION_V5_OUT_MCFW_SECURITY_LEVEL_LEN 4
/* MC firmware build name (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_VERSION_V5_OUT_MCFW_BUILD_NAME_OFST 76
#define       MC_CMD_GET_VERSION_V5_OUT_MCFW_BUILD_NAME_LEN 64
/* The SUC firmware version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V5_OUT_SUCFW_VERSION_OFST 140
#define       MC_CMD_GET_VERSION_V5_OUT_SUCFW_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_SUCFW_VERSION_NUM 4
/* SUC firmware build date (as 64-bit Unix timestamp) */
#define       MC_CMD_GET_VERSION_V5_OUT_SUCFW_BUILD_DATE_OFST 156
#define       MC_CMD_GET_VERSION_V5_OUT_SUCFW_BUILD_DATE_LEN 8
#define       MC_CMD_GET_VERSION_V5_OUT_SUCFW_BUILD_DATE_LO_OFST 156
#define       MC_CMD_GET_VERSION_V5_OUT_SUCFW_BUILD_DATE_LO_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_SUCFW_BUILD_DATE_LO_LBN 1248
#define       MC_CMD_GET_VERSION_V5_OUT_SUCFW_BUILD_DATE_LO_WIDTH 32
#define       MC_CMD_GET_VERSION_V5_OUT_SUCFW_BUILD_DATE_HI_OFST 160
#define       MC_CMD_GET_VERSION_V5_OUT_SUCFW_BUILD_DATE_HI_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_SUCFW_BUILD_DATE_HI_LBN 1280
#define       MC_CMD_GET_VERSION_V5_OUT_SUCFW_BUILD_DATE_HI_WIDTH 32
/* The ID of the SUC chip. This is specific to the platform but typically
 * indicates family, memory sizes etc. See SF-116728-SW for further details.
 */
#define       MC_CMD_GET_VERSION_V5_OUT_SUCFW_CHIP_ID_OFST 164
#define       MC_CMD_GET_VERSION_V5_OUT_SUCFW_CHIP_ID_LEN 4
/* The CMC firmware version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V5_OUT_CMCFW_VERSION_OFST 168
#define       MC_CMD_GET_VERSION_V5_OUT_CMCFW_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_CMCFW_VERSION_NUM 4
/* CMC firmware build date (as 64-bit Unix timestamp) */
#define       MC_CMD_GET_VERSION_V5_OUT_CMCFW_BUILD_DATE_OFST 184
#define       MC_CMD_GET_VERSION_V5_OUT_CMCFW_BUILD_DATE_LEN 8
#define       MC_CMD_GET_VERSION_V5_OUT_CMCFW_BUILD_DATE_LO_OFST 184
#define       MC_CMD_GET_VERSION_V5_OUT_CMCFW_BUILD_DATE_LO_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_CMCFW_BUILD_DATE_LO_LBN 1472
#define       MC_CMD_GET_VERSION_V5_OUT_CMCFW_BUILD_DATE_LO_WIDTH 32
#define       MC_CMD_GET_VERSION_V5_OUT_CMCFW_BUILD_DATE_HI_OFST 188
#define       MC_CMD_GET_VERSION_V5_OUT_CMCFW_BUILD_DATE_HI_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_CMCFW_BUILD_DATE_HI_LBN 1504
#define       MC_CMD_GET_VERSION_V5_OUT_CMCFW_BUILD_DATE_HI_WIDTH 32
/* FPGA version as three numbers. On Riverhead based systems this field uses
 * the same encoding as hardware version ID registers (MC_FPGA_BUILD_HWRD_REG):
 * FPGA_VERSION[0]: x => Image H{x} FPGA_VERSION[1]: Revision letter (0 => A, 1
 * => B, ...) FPGA_VERSION[2]: Sub-revision number
 */
#define       MC_CMD_GET_VERSION_V5_OUT_FPGA_VERSION_OFST 192
#define       MC_CMD_GET_VERSION_V5_OUT_FPGA_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_FPGA_VERSION_NUM 3
/* Extra FPGA revision information (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_VERSION_V5_OUT_FPGA_EXTRA_OFST 204
#define       MC_CMD_GET_VERSION_V5_OUT_FPGA_EXTRA_LEN 16
/* Board name / adapter model (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_VERSION_V5_OUT_BOARD_NAME_OFST 220
#define       MC_CMD_GET_VERSION_V5_OUT_BOARD_NAME_LEN 16
/* Board revision number */
#define       MC_CMD_GET_VERSION_V5_OUT_BOARD_REVISION_OFST 236
#define       MC_CMD_GET_VERSION_V5_OUT_BOARD_REVISION_LEN 4
/* Board serial number (as null-terminated US-ASCII string) */
#define       MC_CMD_GET_VERSION_V5_OUT_BOARD_SERIAL_OFST 240
#define       MC_CMD_GET_VERSION_V5_OUT_BOARD_SERIAL_LEN 64
/* The version of the datapath hardware design as three number - a.b.c */
#define       MC_CMD_GET_VERSION_V5_OUT_DATAPATH_HW_VERSION_OFST 304
#define       MC_CMD_GET_VERSION_V5_OUT_DATAPATH_HW_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_DATAPATH_HW_VERSION_NUM 3
/* The version of the firmware library used to control the datapath as three
 * number - a.b.c
 */
#define       MC_CMD_GET_VERSION_V5_OUT_DATAPATH_FW_VERSION_OFST 316
#define       MC_CMD_GET_VERSION_V5_OUT_DATAPATH_FW_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_DATAPATH_FW_VERSION_NUM 3
/* The SOC boot version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V5_OUT_SOC_BOOT_VERSION_OFST 328
#define       MC_CMD_GET_VERSION_V5_OUT_SOC_BOOT_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_SOC_BOOT_VERSION_NUM 4
/* The SOC uboot version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V5_OUT_SOC_UBOOT_VERSION_OFST 344
#define       MC_CMD_GET_VERSION_V5_OUT_SOC_UBOOT_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_SOC_UBOOT_VERSION_NUM 4
/* The SOC main rootfs version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V5_OUT_SOC_MAIN_ROOTFS_VERSION_OFST 360
#define       MC_CMD_GET_VERSION_V5_OUT_SOC_MAIN_ROOTFS_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_SOC_MAIN_ROOTFS_VERSION_NUM 4
/* The SOC recovery buildroot version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V5_OUT_SOC_RECOVERY_BUILDROOT_VERSION_OFST 376
#define       MC_CMD_GET_VERSION_V5_OUT_SOC_RECOVERY_BUILDROOT_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_SOC_RECOVERY_BUILDROOT_VERSION_NUM 4
/* Board version as four numbers - a.b.c.d. BOARD_VERSION[0] duplicates the
 * BOARD_REVISION field
 */
#define       MC_CMD_GET_VERSION_V5_OUT_BOARD_VERSION_OFST 392
#define       MC_CMD_GET_VERSION_V5_OUT_BOARD_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_BOARD_VERSION_NUM 4
/* Bundle version as four numbers - a.b.c.d */
#define       MC_CMD_GET_VERSION_V5_OUT_BUNDLE_VERSION_OFST 408
#define       MC_CMD_GET_VERSION_V5_OUT_BUNDLE_VERSION_LEN 4
#define       MC_CMD_GET_VERSION_V5_OUT_BUNDLE_VERSION_NUM 4


/***********************************/
/* MC_CMD_PTP
 * Perform PTP operation
 */
#define MC_CMD_PTP 0xb
#undef MC_CMD_0xb_PRIVILEGE_CTG

#define MC_CMD_0xb_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_PTP_IN msgrequest */
#define    MC_CMD_PTP_IN_LEN 1
/* PTP operation code */
#define       MC_CMD_PTP_IN_OP_OFST 0
#define       MC_CMD_PTP_IN_OP_LEN 1
/* enum: Enable PTP packet timestamping operation. */
#define          MC_CMD_PTP_OP_ENABLE 0x1
/* enum: Disable PTP packet timestamping operation. */
#define          MC_CMD_PTP_OP_DISABLE 0x2
/* enum: Send a PTP packet. This operation is used on Siena and Huntington.
 * From Medford onwards it is not supported: on those platforms PTP transmit
 * timestamping is done using the fast path.
 */
#define          MC_CMD_PTP_OP_TRANSMIT 0x3
/* enum: Read the current NIC time. */
#define          MC_CMD_PTP_OP_READ_NIC_TIME 0x4
/* enum: Get the current PTP status. Note that the clock frequency returned (in
 * Hz) is rounded to the nearest MHz (e.g. 666000000 for 666666666).
 */
#define          MC_CMD_PTP_OP_STATUS 0x5
/* enum: Adjust the PTP NIC's time. */
#define          MC_CMD_PTP_OP_ADJUST 0x6
/* enum: Synchronize host and NIC time. */
#define          MC_CMD_PTP_OP_SYNCHRONIZE 0x7
/* enum: Basic manufacturing tests. Siena PTP adapters only. */
#define          MC_CMD_PTP_OP_MANFTEST_BASIC 0x8
/* enum: Packet based manufacturing tests. Siena PTP adapters only. */
#define          MC_CMD_PTP_OP_MANFTEST_PACKET 0x9
/* enum: Reset some of the PTP related statistics */
#define          MC_CMD_PTP_OP_RESET_STATS 0xa
/* enum: Debug operations to MC. */
#define          MC_CMD_PTP_OP_DEBUG 0xb
/* enum: Read an FPGA register. Siena PTP adapters only. */
#define          MC_CMD_PTP_OP_FPGAREAD 0xc
/* enum: Write an FPGA register. Siena PTP adapters only. */
#define          MC_CMD_PTP_OP_FPGAWRITE 0xd
/* enum: Apply an offset to the NIC clock */
#define          MC_CMD_PTP_OP_CLOCK_OFFSET_ADJUST 0xe
/* enum: Change the frequency correction applied to the NIC clock */
#define          MC_CMD_PTP_OP_CLOCK_FREQ_ADJUST 0xf
/* enum: Set the MC packet filter VLAN tags for received PTP packets.
 * Deprecated for Huntington onwards.
 */
#define          MC_CMD_PTP_OP_RX_SET_VLAN_FILTER 0x10
/* enum: Set the MC packet filter UUID for received PTP packets. Deprecated for
 * Huntington onwards.
 */
#define          MC_CMD_PTP_OP_RX_SET_UUID_FILTER 0x11
/* enum: Set the MC packet filter Domain for received PTP packets. Deprecated
 * for Huntington onwards.
 */
#define          MC_CMD_PTP_OP_RX_SET_DOMAIN_FILTER 0x12
/* enum: Set the clock source. Required for snapper tests on Huntington and
 * Medford. Not implemented for Siena or Medford2.
 */
#define          MC_CMD_PTP_OP_SET_CLK_SRC 0x13
/* enum: Reset value of Timer Reg. Not implemented. */
#define          MC_CMD_PTP_OP_RST_CLK 0x14
/* enum: Enable the forwarding of PPS events to the host */
#define          MC_CMD_PTP_OP_PPS_ENABLE 0x15
/* enum: Get the time format used by this NIC for PTP operations */
#define          MC_CMD_PTP_OP_GET_TIME_FORMAT 0x16
/* enum: Get the clock attributes. NOTE- extended version of
 * MC_CMD_PTP_OP_GET_TIME_FORMAT
 */
#define          MC_CMD_PTP_OP_GET_ATTRIBUTES 0x16
/* enum: Get corrections that should be applied to the various different
 * timestamps
 */
#define          MC_CMD_PTP_OP_GET_TIMESTAMP_CORRECTIONS 0x17
/* enum: Subscribe to receive periodic time events indicating the current NIC
 * time
 */
#define          MC_CMD_PTP_OP_TIME_EVENT_SUBSCRIBE 0x18
/* enum: Unsubscribe to stop receiving time events */
#define          MC_CMD_PTP_OP_TIME_EVENT_UNSUBSCRIBE 0x19
/* enum: PPS based manfacturing tests. Requires PPS output to be looped to PPS
 * input on the same NIC. Siena PTP adapters only.
 */
#define          MC_CMD_PTP_OP_MANFTEST_PPS 0x1a
/* enum: Set the PTP sync status. Status is used by firmware to report to event
 * subscribers.
 */
#define          MC_CMD_PTP_OP_SET_SYNC_STATUS 0x1b
/* enum: X4 and later adapters should use this instead of
 * PTP_OP_TIME_EVENT_SUBSCRIBE. Subscribe to receive periodic time events
 * indicating the current NIC time
 */
#define          MC_CMD_PTP_OP_TIME_EVENT_SUBSCRIBE_V2 0x1c
/* enum: For X4 and later NICs. Packet timestamps and time sync events have
 * IS_SET and IN_SYNC flags, that indicates whether time is updated and if it
 * is in sync with host. Once set, IN_SYNC flag is cleared by hardware after a
 * software configurable time out. Host driver need to query what is set and
 * what is maximum supported interval, this MCDI can be used to query these.
 */
#define          MC_CMD_PTP_OP_GET_SYNC_TIMEOUT 0x1d

/* MC_CMD_PTP_IN_ENABLE msgrequest */
#define    MC_CMD_PTP_IN_ENABLE_LEN 16
#define       MC_CMD_PTP_IN_CMD_OFST 0
#define       MC_CMD_PTP_IN_CMD_LEN 4
#define       MC_CMD_PTP_IN_PERIPH_ID_OFST 4
#define       MC_CMD_PTP_IN_PERIPH_ID_LEN 4
/* Not used, initialize to 0. Events are always sent to function relative queue
 * 0.
 */
#define       MC_CMD_PTP_IN_ENABLE_QUEUE_OFST 8
#define       MC_CMD_PTP_IN_ENABLE_QUEUE_LEN 4
/* PTP timestamping mode. Not used from Huntington onwards. */
#define       MC_CMD_PTP_IN_ENABLE_MODE_OFST 12
#define       MC_CMD_PTP_IN_ENABLE_MODE_LEN 4
/* enum: PTP, version 1 */
#define          MC_CMD_PTP_MODE_V1 0x0
/* enum: PTP, version 1, with VLAN headers - deprecated */
#define          MC_CMD_PTP_MODE_V1_VLAN 0x1
/* enum: PTP, version 2 */
#define          MC_CMD_PTP_MODE_V2 0x2
/* enum: PTP, version 2, with VLAN headers - deprecated */
#define          MC_CMD_PTP_MODE_V2_VLAN 0x3
/* enum: PTP, version 2, with improved UUID filtering */
#define          MC_CMD_PTP_MODE_V2_ENHANCED 0x4
/* enum: FCoE (seconds and microseconds) */
#define          MC_CMD_PTP_MODE_FCOE 0x5

/* MC_CMD_PTP_IN_DISABLE msgrequest */
#define    MC_CMD_PTP_IN_DISABLE_LEN 8
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */

/* MC_CMD_PTP_IN_TRANSMIT msgrequest */
#define    MC_CMD_PTP_IN_TRANSMIT_LENMIN 13
#define    MC_CMD_PTP_IN_TRANSMIT_LENMAX 252
#define    MC_CMD_PTP_IN_TRANSMIT_LENMAX_MCDI2 1020
#define    MC_CMD_PTP_IN_TRANSMIT_LEN(num) (12+1*(num))
#define    MC_CMD_PTP_IN_TRANSMIT_PACKET_NUM(len) (((len)-12)/1)
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* Transmit packet length */
#define       MC_CMD_PTP_IN_TRANSMIT_LENGTH_OFST 8
#define       MC_CMD_PTP_IN_TRANSMIT_LENGTH_LEN 4
/* Transmit packet data */
#define       MC_CMD_PTP_IN_TRANSMIT_PACKET_OFST 12
#define       MC_CMD_PTP_IN_TRANSMIT_PACKET_LEN 1
#define       MC_CMD_PTP_IN_TRANSMIT_PACKET_MINNUM 1
#define       MC_CMD_PTP_IN_TRANSMIT_PACKET_MAXNUM 240
#define       MC_CMD_PTP_IN_TRANSMIT_PACKET_MAXNUM_MCDI2 1008

/* MC_CMD_PTP_IN_READ_NIC_TIME msgrequest */
#define    MC_CMD_PTP_IN_READ_NIC_TIME_LEN 8
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */

/* MC_CMD_PTP_IN_READ_NIC_TIME_V2 msgrequest */
#define    MC_CMD_PTP_IN_READ_NIC_TIME_V2_LEN 8
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */

/* MC_CMD_PTP_IN_STATUS msgrequest */
#define    MC_CMD_PTP_IN_STATUS_LEN 8
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */

/* MC_CMD_PTP_IN_ADJUST msgrequest */
#define    MC_CMD_PTP_IN_ADJUST_LEN 24
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* Frequency adjustment 40 bit fixed point ns */
#define       MC_CMD_PTP_IN_ADJUST_FREQ_OFST 8
#define       MC_CMD_PTP_IN_ADJUST_FREQ_LEN 8
#define       MC_CMD_PTP_IN_ADJUST_FREQ_LO_OFST 8
#define       MC_CMD_PTP_IN_ADJUST_FREQ_LO_LEN 4
#define       MC_CMD_PTP_IN_ADJUST_FREQ_LO_LBN 64
#define       MC_CMD_PTP_IN_ADJUST_FREQ_LO_WIDTH 32
#define       MC_CMD_PTP_IN_ADJUST_FREQ_HI_OFST 12
#define       MC_CMD_PTP_IN_ADJUST_FREQ_HI_LEN 4
#define       MC_CMD_PTP_IN_ADJUST_FREQ_HI_LBN 96
#define       MC_CMD_PTP_IN_ADJUST_FREQ_HI_WIDTH 32
/* enum: Number of fractional bits in frequency adjustment */
#define          MC_CMD_PTP_IN_ADJUST_BITS 0x28
/* enum: Number of fractional bits in frequency adjustment when FP44_FREQ_ADJ
 * is indicated in the MC_CMD_PTP_OUT_GET_ATTRIBUTES command CAPABILITIES
 * field.
 */
#define          MC_CMD_PTP_IN_ADJUST_BITS_FP44 0x2c
/* Time adjustment in seconds */
#define       MC_CMD_PTP_IN_ADJUST_SECONDS_OFST 16
#define       MC_CMD_PTP_IN_ADJUST_SECONDS_LEN 4
/* Time adjustment major value */
#define       MC_CMD_PTP_IN_ADJUST_MAJOR_OFST 16
#define       MC_CMD_PTP_IN_ADJUST_MAJOR_LEN 4
/* Time adjustment in nanoseconds */
#define       MC_CMD_PTP_IN_ADJUST_NANOSECONDS_OFST 20
#define       MC_CMD_PTP_IN_ADJUST_NANOSECONDS_LEN 4
/* Time adjustment minor value */
#define       MC_CMD_PTP_IN_ADJUST_MINOR_OFST 20
#define       MC_CMD_PTP_IN_ADJUST_MINOR_LEN 4

/* MC_CMD_PTP_IN_ADJUST_V2 msgrequest */
#define    MC_CMD_PTP_IN_ADJUST_V2_LEN 28
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* Frequency adjustment 40 bit fixed point ns */
#define       MC_CMD_PTP_IN_ADJUST_V2_FREQ_OFST 8
#define       MC_CMD_PTP_IN_ADJUST_V2_FREQ_LEN 8
#define       MC_CMD_PTP_IN_ADJUST_V2_FREQ_LO_OFST 8
#define       MC_CMD_PTP_IN_ADJUST_V2_FREQ_LO_LEN 4
#define       MC_CMD_PTP_IN_ADJUST_V2_FREQ_LO_LBN 64
#define       MC_CMD_PTP_IN_ADJUST_V2_FREQ_LO_WIDTH 32
#define       MC_CMD_PTP_IN_ADJUST_V2_FREQ_HI_OFST 12
#define       MC_CMD_PTP_IN_ADJUST_V2_FREQ_HI_LEN 4
#define       MC_CMD_PTP_IN_ADJUST_V2_FREQ_HI_LBN 96
#define       MC_CMD_PTP_IN_ADJUST_V2_FREQ_HI_WIDTH 32
/* enum: Number of fractional bits in frequency adjustment */
/*               MC_CMD_PTP_IN_ADJUST_BITS 0x28 */
/* enum: Number of fractional bits in frequency adjustment when FP44_FREQ_ADJ
 * is indicated in the MC_CMD_PTP_OUT_GET_ATTRIBUTES command CAPABILITIES
 * field.
 */
/*               MC_CMD_PTP_IN_ADJUST_BITS_FP44 0x2c */
/* Time adjustment in seconds */
#define       MC_CMD_PTP_IN_ADJUST_V2_SECONDS_OFST 16
#define       MC_CMD_PTP_IN_ADJUST_V2_SECONDS_LEN 4
/* Time adjustment major value */
#define       MC_CMD_PTP_IN_ADJUST_V2_MAJOR_OFST 16
#define       MC_CMD_PTP_IN_ADJUST_V2_MAJOR_LEN 4
/* Time adjustment in nanoseconds */
#define       MC_CMD_PTP_IN_ADJUST_V2_NANOSECONDS_OFST 20
#define       MC_CMD_PTP_IN_ADJUST_V2_NANOSECONDS_LEN 4
/* Time adjustment minor value */
#define       MC_CMD_PTP_IN_ADJUST_V2_MINOR_OFST 20
#define       MC_CMD_PTP_IN_ADJUST_V2_MINOR_LEN 4
/* Upper 32bits of major time offset adjustment */
#define       MC_CMD_PTP_IN_ADJUST_V2_MAJOR_HI_OFST 24
#define       MC_CMD_PTP_IN_ADJUST_V2_MAJOR_HI_LEN 4

/* MC_CMD_PTP_IN_SYNCHRONIZE msgrequest */
#define    MC_CMD_PTP_IN_SYNCHRONIZE_LEN 20
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* Number of time readings to capture */
#define       MC_CMD_PTP_IN_SYNCHRONIZE_NUMTIMESETS_OFST 8
#define       MC_CMD_PTP_IN_SYNCHRONIZE_NUMTIMESETS_LEN 4
/* Host address in which to write "synchronization started" indication (64
 * bits)
 */
#define       MC_CMD_PTP_IN_SYNCHRONIZE_START_ADDR_OFST 12
#define       MC_CMD_PTP_IN_SYNCHRONIZE_START_ADDR_LEN 8
#define       MC_CMD_PTP_IN_SYNCHRONIZE_START_ADDR_LO_OFST 12
#define       MC_CMD_PTP_IN_SYNCHRONIZE_START_ADDR_LO_LEN 4
#define       MC_CMD_PTP_IN_SYNCHRONIZE_START_ADDR_LO_LBN 96
#define       MC_CMD_PTP_IN_SYNCHRONIZE_START_ADDR_LO_WIDTH 32
#define       MC_CMD_PTP_IN_SYNCHRONIZE_START_ADDR_HI_OFST 16
#define       MC_CMD_PTP_IN_SYNCHRONIZE_START_ADDR_HI_LEN 4
#define       MC_CMD_PTP_IN_SYNCHRONIZE_START_ADDR_HI_LBN 128
#define       MC_CMD_PTP_IN_SYNCHRONIZE_START_ADDR_HI_WIDTH 32

/* MC_CMD_PTP_IN_MANFTEST_BASIC msgrequest */
#define    MC_CMD_PTP_IN_MANFTEST_BASIC_LEN 8
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */

/* MC_CMD_PTP_IN_MANFTEST_PACKET msgrequest */
#define    MC_CMD_PTP_IN_MANFTEST_PACKET_LEN 12
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* Enable or disable packet testing */
#define       MC_CMD_PTP_IN_MANFTEST_PACKET_TEST_ENABLE_OFST 8
#define       MC_CMD_PTP_IN_MANFTEST_PACKET_TEST_ENABLE_LEN 4

/* MC_CMD_PTP_IN_RESET_STATS msgrequest: Reset PTP statistics */
#define    MC_CMD_PTP_IN_RESET_STATS_LEN 8
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */

/* MC_CMD_PTP_IN_DEBUG msgrequest */
#define    MC_CMD_PTP_IN_DEBUG_LEN 12
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* Debug operations */
#define       MC_CMD_PTP_IN_DEBUG_DEBUG_PARAM_OFST 8
#define       MC_CMD_PTP_IN_DEBUG_DEBUG_PARAM_LEN 4

/* MC_CMD_PTP_IN_FPGAREAD msgrequest */
#define    MC_CMD_PTP_IN_FPGAREAD_LEN 16
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
#define       MC_CMD_PTP_IN_FPGAREAD_ADDR_OFST 8
#define       MC_CMD_PTP_IN_FPGAREAD_ADDR_LEN 4
#define       MC_CMD_PTP_IN_FPGAREAD_NUMBYTES_OFST 12
#define       MC_CMD_PTP_IN_FPGAREAD_NUMBYTES_LEN 4

/* MC_CMD_PTP_IN_FPGAWRITE msgrequest */
#define    MC_CMD_PTP_IN_FPGAWRITE_LENMIN 13
#define    MC_CMD_PTP_IN_FPGAWRITE_LENMAX 252
#define    MC_CMD_PTP_IN_FPGAWRITE_LENMAX_MCDI2 1020
#define    MC_CMD_PTP_IN_FPGAWRITE_LEN(num) (12+1*(num))
#define    MC_CMD_PTP_IN_FPGAWRITE_BUFFER_NUM(len) (((len)-12)/1)
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
#define       MC_CMD_PTP_IN_FPGAWRITE_ADDR_OFST 8
#define       MC_CMD_PTP_IN_FPGAWRITE_ADDR_LEN 4
#define       MC_CMD_PTP_IN_FPGAWRITE_BUFFER_OFST 12
#define       MC_CMD_PTP_IN_FPGAWRITE_BUFFER_LEN 1
#define       MC_CMD_PTP_IN_FPGAWRITE_BUFFER_MINNUM 1
#define       MC_CMD_PTP_IN_FPGAWRITE_BUFFER_MAXNUM 240
#define       MC_CMD_PTP_IN_FPGAWRITE_BUFFER_MAXNUM_MCDI2 1008

/* MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST msgrequest */
#define    MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_LEN 16
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* Time adjustment in seconds */
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_SECONDS_OFST 8
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_SECONDS_LEN 4
/* Time adjustment major value */
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_MAJOR_OFST 8
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_MAJOR_LEN 4
/* Time adjustment in nanoseconds */
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_NANOSECONDS_OFST 12
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_NANOSECONDS_LEN 4
/* Time adjustment minor value */
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_MINOR_OFST 12
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_MINOR_LEN 4

/* MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_V2 msgrequest */
#define    MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_V2_LEN 20
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* Time adjustment in seconds */
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_V2_SECONDS_OFST 8
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_V2_SECONDS_LEN 4
/* Time adjustment major value */
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_V2_MAJOR_OFST 8
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_V2_MAJOR_LEN 4
/* Time adjustment in nanoseconds */
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_V2_NANOSECONDS_OFST 12
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_V2_NANOSECONDS_LEN 4
/* Time adjustment minor value */
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_V2_MINOR_OFST 12
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_V2_MINOR_LEN 4
/* Upper 32bits of major time offset adjustment */
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_V2_MAJOR_HI_OFST 16
#define       MC_CMD_PTP_IN_CLOCK_OFFSET_ADJUST_V2_MAJOR_HI_LEN 4

/* MC_CMD_PTP_IN_CLOCK_FREQ_ADJUST msgrequest */
#define    MC_CMD_PTP_IN_CLOCK_FREQ_ADJUST_LEN 16
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* Frequency adjustment 40 bit fixed point ns */
#define       MC_CMD_PTP_IN_CLOCK_FREQ_ADJUST_FREQ_OFST 8
#define       MC_CMD_PTP_IN_CLOCK_FREQ_ADJUST_FREQ_LEN 8
#define       MC_CMD_PTP_IN_CLOCK_FREQ_ADJUST_FREQ_LO_OFST 8
#define       MC_CMD_PTP_IN_CLOCK_FREQ_ADJUST_FREQ_LO_LEN 4
#define       MC_CMD_PTP_IN_CLOCK_FREQ_ADJUST_FREQ_LO_LBN 64
#define       MC_CMD_PTP_IN_CLOCK_FREQ_ADJUST_FREQ_LO_WIDTH 32
#define       MC_CMD_PTP_IN_CLOCK_FREQ_ADJUST_FREQ_HI_OFST 12
#define       MC_CMD_PTP_IN_CLOCK_FREQ_ADJUST_FREQ_HI_LEN 4
#define       MC_CMD_PTP_IN_CLOCK_FREQ_ADJUST_FREQ_HI_LBN 96
#define       MC_CMD_PTP_IN_CLOCK_FREQ_ADJUST_FREQ_HI_WIDTH 32
/*            Enum values, see field(s): */
/*               MC_CMD_PTP/MC_CMD_PTP_IN_ADJUST/FREQ */

/* MC_CMD_PTP_IN_RX_SET_VLAN_FILTER msgrequest */
#define    MC_CMD_PTP_IN_RX_SET_VLAN_FILTER_LEN 24
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* Number of VLAN tags, 0 if not VLAN */
#define       MC_CMD_PTP_IN_RX_SET_VLAN_FILTER_NUM_VLAN_TAGS_OFST 8
#define       MC_CMD_PTP_IN_RX_SET_VLAN_FILTER_NUM_VLAN_TAGS_LEN 4
/* Set of VLAN tags to filter against */
#define       MC_CMD_PTP_IN_RX_SET_VLAN_FILTER_VLAN_TAG_OFST 12
#define       MC_CMD_PTP_IN_RX_SET_VLAN_FILTER_VLAN_TAG_LEN 4
#define       MC_CMD_PTP_IN_RX_SET_VLAN_FILTER_VLAN_TAG_NUM 3

/* MC_CMD_PTP_IN_RX_SET_UUID_FILTER msgrequest */
#define    MC_CMD_PTP_IN_RX_SET_UUID_FILTER_LEN 20
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* 1 to enable UUID filtering, 0 to disable */
#define       MC_CMD_PTP_IN_RX_SET_UUID_FILTER_ENABLE_OFST 8
#define       MC_CMD_PTP_IN_RX_SET_UUID_FILTER_ENABLE_LEN 4
/* UUID to filter against */
#define       MC_CMD_PTP_IN_RX_SET_UUID_FILTER_UUID_OFST 12
#define       MC_CMD_PTP_IN_RX_SET_UUID_FILTER_UUID_LEN 8
#define       MC_CMD_PTP_IN_RX_SET_UUID_FILTER_UUID_LO_OFST 12
#define       MC_CMD_PTP_IN_RX_SET_UUID_FILTER_UUID_LO_LEN 4
#define       MC_CMD_PTP_IN_RX_SET_UUID_FILTER_UUID_LO_LBN 96
#define       MC_CMD_PTP_IN_RX_SET_UUID_FILTER_UUID_LO_WIDTH 32
#define       MC_CMD_PTP_IN_RX_SET_UUID_FILTER_UUID_HI_OFST 16
#define       MC_CMD_PTP_IN_RX_SET_UUID_FILTER_UUID_HI_LEN 4
#define       MC_CMD_PTP_IN_RX_SET_UUID_FILTER_UUID_HI_LBN 128
#define       MC_CMD_PTP_IN_RX_SET_UUID_FILTER_UUID_HI_WIDTH 32

/* MC_CMD_PTP_IN_RX_SET_DOMAIN_FILTER msgrequest */
#define    MC_CMD_PTP_IN_RX_SET_DOMAIN_FILTER_LEN 16
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* 1 to enable Domain filtering, 0 to disable */
#define       MC_CMD_PTP_IN_RX_SET_DOMAIN_FILTER_ENABLE_OFST 8
#define       MC_CMD_PTP_IN_RX_SET_DOMAIN_FILTER_ENABLE_LEN 4
/* Domain number to filter against */
#define       MC_CMD_PTP_IN_RX_SET_DOMAIN_FILTER_DOMAIN_OFST 12
#define       MC_CMD_PTP_IN_RX_SET_DOMAIN_FILTER_DOMAIN_LEN 4

/* MC_CMD_PTP_IN_SET_CLK_SRC msgrequest */
#define    MC_CMD_PTP_IN_SET_CLK_SRC_LEN 12
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* Set the clock source. */
#define       MC_CMD_PTP_IN_SET_CLK_SRC_CLK_OFST 8
#define       MC_CMD_PTP_IN_SET_CLK_SRC_CLK_LEN 4
/* enum: Internal. */
#define          MC_CMD_PTP_CLK_SRC_INTERNAL 0x0
/* enum: External. */
#define          MC_CMD_PTP_CLK_SRC_EXTERNAL 0x1

/* MC_CMD_PTP_IN_RST_CLK msgrequest: Reset value of Timer Reg. */
#define    MC_CMD_PTP_IN_RST_CLK_LEN 8
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */

/* MC_CMD_PTP_IN_PPS_ENABLE msgrequest */
#define    MC_CMD_PTP_IN_PPS_ENABLE_LEN 12
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/* Enable or disable */
#define       MC_CMD_PTP_IN_PPS_ENABLE_OP_OFST 4
#define       MC_CMD_PTP_IN_PPS_ENABLE_OP_LEN 4
/* enum: Enable */
#define          MC_CMD_PTP_ENABLE_PPS 0x0
/* enum: Disable */
#define          MC_CMD_PTP_DISABLE_PPS 0x1
/* Not used, initialize to 0. Events are always sent to function relative queue
 * 0.
 */
#define       MC_CMD_PTP_IN_PPS_ENABLE_QUEUE_ID_OFST 8
#define       MC_CMD_PTP_IN_PPS_ENABLE_QUEUE_ID_LEN 4

/* MC_CMD_PTP_IN_GET_TIME_FORMAT msgrequest */
#define    MC_CMD_PTP_IN_GET_TIME_FORMAT_LEN 8
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */

/* MC_CMD_PTP_IN_GET_ATTRIBUTES msgrequest */
#define    MC_CMD_PTP_IN_GET_ATTRIBUTES_LEN 8
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */

/* MC_CMD_PTP_IN_GET_TIMESTAMP_CORRECTIONS msgrequest */
#define    MC_CMD_PTP_IN_GET_TIMESTAMP_CORRECTIONS_LEN 8
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */

/* MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE msgrequest */
#define    MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_LEN 12
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* Original field containing queue ID. Now extended to include flags. */
#define       MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_QUEUE_OFST 8
#define       MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_QUEUE_LEN 4
#define        MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_QUEUE_ID_OFST 8
#define        MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_QUEUE_ID_LBN 0
#define        MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_QUEUE_ID_WIDTH 16
#define        MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_REPORT_SYNC_STATUS_OFST 8
#define        MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_REPORT_SYNC_STATUS_LBN 31
#define        MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_REPORT_SYNC_STATUS_WIDTH 1

/* MC_CMD_PTP_IN_TIME_EVENT_UNSUBSCRIBE msgrequest */
#define    MC_CMD_PTP_IN_TIME_EVENT_UNSUBSCRIBE_LEN 16
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* Unsubscribe options */
#define       MC_CMD_PTP_IN_TIME_EVENT_UNSUBSCRIBE_CONTROL_OFST 8
#define       MC_CMD_PTP_IN_TIME_EVENT_UNSUBSCRIBE_CONTROL_LEN 4
/* enum: Unsubscribe a single queue */
#define          MC_CMD_PTP_IN_TIME_EVENT_UNSUBSCRIBE_SINGLE 0x0
/* enum: Unsubscribe all queues */
#define          MC_CMD_PTP_IN_TIME_EVENT_UNSUBSCRIBE_ALL 0x1
/* Event queue ID */
#define       MC_CMD_PTP_IN_TIME_EVENT_UNSUBSCRIBE_QUEUE_OFST 12
#define       MC_CMD_PTP_IN_TIME_EVENT_UNSUBSCRIBE_QUEUE_LEN 4

/* MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_V2 msgrequest */
#define    MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_V2_LEN 16
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* Event queue ID */
#define       MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_V2_QUEUE_ID_OFST 8
#define       MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_V2_QUEUE_ID_LEN 4
/* Space for flags. */
#define       MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_V2_FLAGS_OFST 12
#define       MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_V2_FLAGS_LEN 4
#define        MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_V2_REPORT_SYNC_STATUS_OFST 12
#define        MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_V2_REPORT_SYNC_STATUS_LBN 31
#define        MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_V2_REPORT_SYNC_STATUS_WIDTH 1

/* MC_CMD_PTP_IN_MANFTEST_PPS msgrequest */
#define    MC_CMD_PTP_IN_MANFTEST_PPS_LEN 12
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* 1 to enable PPS test mode, 0 to disable and return result. */
#define       MC_CMD_PTP_IN_MANFTEST_PPS_TEST_ENABLE_OFST 8
#define       MC_CMD_PTP_IN_MANFTEST_PPS_TEST_ENABLE_LEN 4

/* MC_CMD_PTP_IN_SET_SYNC_STATUS msgrequest */
#define    MC_CMD_PTP_IN_SET_SYNC_STATUS_LEN 24
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */
/* NIC - Host System Clock Synchronization status */
#define       MC_CMD_PTP_IN_SET_SYNC_STATUS_STATUS_OFST 8
#define       MC_CMD_PTP_IN_SET_SYNC_STATUS_STATUS_LEN 4
/* enum: Host System clock and NIC clock are not in sync */
#define          MC_CMD_PTP_IN_SET_SYNC_STATUS_NOT_IN_SYNC 0x0
/* enum: Host System clock and NIC clock are synchronized */
#define          MC_CMD_PTP_IN_SET_SYNC_STATUS_IN_SYNC 0x1
/* If synchronized, number of seconds until clocks should be considered to be
 * no longer in sync.
 */
#define       MC_CMD_PTP_IN_SET_SYNC_STATUS_TIMEOUT_OFST 12
#define       MC_CMD_PTP_IN_SET_SYNC_STATUS_TIMEOUT_LEN 4
#define       MC_CMD_PTP_IN_SET_SYNC_STATUS_RESERVED0_OFST 16
#define       MC_CMD_PTP_IN_SET_SYNC_STATUS_RESERVED0_LEN 4
#define       MC_CMD_PTP_IN_SET_SYNC_STATUS_RESERVED1_OFST 20
#define       MC_CMD_PTP_IN_SET_SYNC_STATUS_RESERVED1_LEN 4

/* MC_CMD_PTP_IN_GET_SYNC_TIMEOUT msgrequest */
#define    MC_CMD_PTP_IN_GET_SYNC_TIMEOUT_LEN 8
/*            MC_CMD_PTP_IN_CMD_OFST 0 */
/*            MC_CMD_PTP_IN_CMD_LEN 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_OFST 4 */
/*            MC_CMD_PTP_IN_PERIPH_ID_LEN 4 */

/* MC_CMD_PTP_OUT msgresponse */
#define    MC_CMD_PTP_OUT_LEN 0

/* MC_CMD_PTP_OUT_TRANSMIT msgresponse */
#define    MC_CMD_PTP_OUT_TRANSMIT_LEN 8
/* Value of seconds timestamp */
#define       MC_CMD_PTP_OUT_TRANSMIT_SECONDS_OFST 0
#define       MC_CMD_PTP_OUT_TRANSMIT_SECONDS_LEN 4
/* Timestamp major value */
#define       MC_CMD_PTP_OUT_TRANSMIT_MAJOR_OFST 0
#define       MC_CMD_PTP_OUT_TRANSMIT_MAJOR_LEN 4
/* Value of nanoseconds timestamp */
#define       MC_CMD_PTP_OUT_TRANSMIT_NANOSECONDS_OFST 4
#define       MC_CMD_PTP_OUT_TRANSMIT_NANOSECONDS_LEN 4
/* Timestamp minor value */
#define       MC_CMD_PTP_OUT_TRANSMIT_MINOR_OFST 4
#define       MC_CMD_PTP_OUT_TRANSMIT_MINOR_LEN 4

/* MC_CMD_PTP_OUT_TIME_EVENT_SUBSCRIBE msgresponse */
#define    MC_CMD_PTP_OUT_TIME_EVENT_SUBSCRIBE_LEN 0

/* MC_CMD_PTP_OUT_TIME_EVENT_UNSUBSCRIBE msgresponse */
#define    MC_CMD_PTP_OUT_TIME_EVENT_UNSUBSCRIBE_LEN 0

/* MC_CMD_PTP_OUT_READ_NIC_TIME msgresponse */
#define    MC_CMD_PTP_OUT_READ_NIC_TIME_LEN 8
/* Value of seconds timestamp */
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_SECONDS_OFST 0
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_SECONDS_LEN 4
/* Timestamp major value */
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_MAJOR_OFST 0
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_MAJOR_LEN 4
/* Value of nanoseconds timestamp */
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_NANOSECONDS_OFST 4
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_NANOSECONDS_LEN 4
/* Timestamp minor value */
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_MINOR_OFST 4
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_MINOR_LEN 4

/* MC_CMD_PTP_OUT_READ_NIC_TIME_V2 msgresponse */
#define    MC_CMD_PTP_OUT_READ_NIC_TIME_V2_LEN 12
/* Value of seconds timestamp */
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_V2_SECONDS_OFST 0
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_V2_SECONDS_LEN 4
/* Timestamp major value */
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_V2_MAJOR_OFST 0
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_V2_MAJOR_LEN 4
/* Value of nanoseconds timestamp */
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_V2_NANOSECONDS_OFST 4
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_V2_NANOSECONDS_LEN 4
/* Timestamp minor value */
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_V2_MINOR_OFST 4
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_V2_MINOR_LEN 4
/* Upper 32bits of major timestamp value */
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_V2_MAJOR_HI_OFST 8
#define       MC_CMD_PTP_OUT_READ_NIC_TIME_V2_MAJOR_HI_LEN 4

/* MC_CMD_PTP_OUT_STATUS msgresponse */
#define    MC_CMD_PTP_OUT_STATUS_LEN 64
/* Frequency of NIC's hardware clock */
#define       MC_CMD_PTP_OUT_STATUS_CLOCK_FREQ_OFST 0
#define       MC_CMD_PTP_OUT_STATUS_CLOCK_FREQ_LEN 4
/* Number of packets transmitted and timestamped */
#define       MC_CMD_PTP_OUT_STATUS_STATS_TX_OFST 4
#define       MC_CMD_PTP_OUT_STATUS_STATS_TX_LEN 4
/* Number of packets received and timestamped */
#define       MC_CMD_PTP_OUT_STATUS_STATS_RX_OFST 8
#define       MC_CMD_PTP_OUT_STATUS_STATS_RX_LEN 4
/* Number of packets timestamped by the FPGA */
#define       MC_CMD_PTP_OUT_STATUS_STATS_TS_OFST 12
#define       MC_CMD_PTP_OUT_STATUS_STATS_TS_LEN 4
/* Number of packets filter matched */
#define       MC_CMD_PTP_OUT_STATUS_STATS_FM_OFST 16
#define       MC_CMD_PTP_OUT_STATUS_STATS_FM_LEN 4
/* Number of packets not filter matched */
#define       MC_CMD_PTP_OUT_STATUS_STATS_NFM_OFST 20
#define       MC_CMD_PTP_OUT_STATUS_STATS_NFM_LEN 4
/* Number of PPS overflows (noise on input?) */
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_OFLOW_OFST 24
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_OFLOW_LEN 4
/* Number of PPS bad periods */
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_BAD_OFST 28
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_BAD_LEN 4
/* Minimum period of PPS pulse in nanoseconds */
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_PER_MIN_OFST 32
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_PER_MIN_LEN 4
/* Maximum period of PPS pulse in nanoseconds */
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_PER_MAX_OFST 36
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_PER_MAX_LEN 4
/* Last period of PPS pulse in nanoseconds */
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_PER_LAST_OFST 40
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_PER_LAST_LEN 4
/* Mean period of PPS pulse in nanoseconds */
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_PER_MEAN_OFST 44
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_PER_MEAN_LEN 4
/* Minimum offset of PPS pulse in nanoseconds (signed) */
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_OFF_MIN_OFST 48
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_OFF_MIN_LEN 4
/* Maximum offset of PPS pulse in nanoseconds (signed) */
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_OFF_MAX_OFST 52
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_OFF_MAX_LEN 4
/* Last offset of PPS pulse in nanoseconds (signed) */
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_OFF_LAST_OFST 56
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_OFF_LAST_LEN 4
/* Mean offset of PPS pulse in nanoseconds (signed) */
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_OFF_MEAN_OFST 60
#define       MC_CMD_PTP_OUT_STATUS_STATS_PPS_OFF_MEAN_LEN 4

/* MC_CMD_PTP_OUT_SYNCHRONIZE msgresponse */
#define    MC_CMD_PTP_OUT_SYNCHRONIZE_LENMIN 20
#define    MC_CMD_PTP_OUT_SYNCHRONIZE_LENMAX 240
#define    MC_CMD_PTP_OUT_SYNCHRONIZE_LENMAX_MCDI2 1020
#define    MC_CMD_PTP_OUT_SYNCHRONIZE_LEN(num) (0+20*(num))
#define    MC_CMD_PTP_OUT_SYNCHRONIZE_TIMESET_NUM(len) (((len)-0)/20)
/* A set of host and NIC times */
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_TIMESET_OFST 0
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_TIMESET_LEN 20
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_TIMESET_MINNUM 1
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_TIMESET_MAXNUM 12
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_TIMESET_MAXNUM_MCDI2 51
/* Host time immediately before NIC's hardware clock read */
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_HOSTSTART_OFST 0
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_HOSTSTART_LEN 4
/* Value of seconds timestamp */
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_SECONDS_OFST 4
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_SECONDS_LEN 4
/* Timestamp major value */
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_MAJOR_OFST 4
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_MAJOR_LEN 4
/* Value of nanoseconds timestamp */
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_NANOSECONDS_OFST 8
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_NANOSECONDS_LEN 4
/* Timestamp minor value */
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_MINOR_OFST 8
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_MINOR_LEN 4
/* Host time immediately after NIC's hardware clock read */
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_HOSTEND_OFST 12
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_HOSTEND_LEN 4
/* Number of nanoseconds waited after reading NIC's hardware clock */
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_WAITNS_OFST 16
#define       MC_CMD_PTP_OUT_SYNCHRONIZE_WAITNS_LEN 4

/* MC_CMD_PTP_OUT_MANFTEST_BASIC msgresponse */
#define    MC_CMD_PTP_OUT_MANFTEST_BASIC_LEN 8
/* Results of testing */
#define       MC_CMD_PTP_OUT_MANFTEST_BASIC_TEST_RESULT_OFST 0
#define       MC_CMD_PTP_OUT_MANFTEST_BASIC_TEST_RESULT_LEN 4
/* enum: Successful test */
#define          MC_CMD_PTP_MANF_SUCCESS 0x0
/* enum: FPGA load failed */
#define          MC_CMD_PTP_MANF_FPGA_LOAD 0x1
/* enum: FPGA version invalid */
#define          MC_CMD_PTP_MANF_FPGA_VERSION 0x2
/* enum: FPGA registers incorrect */
#define          MC_CMD_PTP_MANF_FPGA_REGISTERS 0x3
/* enum: Oscillator possibly not working? */
#define          MC_CMD_PTP_MANF_OSCILLATOR 0x4
/* enum: Timestamps not increasing */
#define          MC_CMD_PTP_MANF_TIMESTAMPS 0x5
/* enum: Mismatched packet count */
#define          MC_CMD_PTP_MANF_PACKET_COUNT 0x6
/* enum: Mismatched packet count (Siena filter and FPGA) */
#define          MC_CMD_PTP_MANF_FILTER_COUNT 0x7
/* enum: Not enough packets to perform timestamp check */
#define          MC_CMD_PTP_MANF_PACKET_ENOUGH 0x8
/* enum: Timestamp trigger GPIO not working */
#define          MC_CMD_PTP_MANF_GPIO_TRIGGER 0x9
/* enum: Insufficient PPS events to perform checks */
#define          MC_CMD_PTP_MANF_PPS_ENOUGH 0xa
/* enum: PPS time event period not sufficiently close to 1s. */
#define          MC_CMD_PTP_MANF_PPS_PERIOD 0xb
/* enum: PPS time event nS reading not sufficiently close to zero. */
#define          MC_CMD_PTP_MANF_PPS_NS 0xc
/* enum: PTP peripheral registers incorrect */
#define          MC_CMD_PTP_MANF_REGISTERS 0xd
/* enum: Failed to read time from PTP peripheral */
#define          MC_CMD_PTP_MANF_CLOCK_READ 0xe
/* Presence of external oscillator */
#define       MC_CMD_PTP_OUT_MANFTEST_BASIC_TEST_EXTOSC_OFST 4
#define       MC_CMD_PTP_OUT_MANFTEST_BASIC_TEST_EXTOSC_LEN 4

/* MC_CMD_PTP_OUT_MANFTEST_PACKET msgresponse */
#define    MC_CMD_PTP_OUT_MANFTEST_PACKET_LEN 12
/* Results of testing */
#define       MC_CMD_PTP_OUT_MANFTEST_PACKET_TEST_RESULT_OFST 0
#define       MC_CMD_PTP_OUT_MANFTEST_PACKET_TEST_RESULT_LEN 4
/* Number of packets received by FPGA */
#define       MC_CMD_PTP_OUT_MANFTEST_PACKET_TEST_FPGACOUNT_OFST 4
#define       MC_CMD_PTP_OUT_MANFTEST_PACKET_TEST_FPGACOUNT_LEN 4
/* Number of packets received by Siena filters */
#define       MC_CMD_PTP_OUT_MANFTEST_PACKET_TEST_FILTERCOUNT_OFST 8
#define       MC_CMD_PTP_OUT_MANFTEST_PACKET_TEST_FILTERCOUNT_LEN 4

/* MC_CMD_PTP_OUT_FPGAREAD msgresponse */
#define    MC_CMD_PTP_OUT_FPGAREAD_LENMIN 1
#define    MC_CMD_PTP_OUT_FPGAREAD_LENMAX 252
#define    MC_CMD_PTP_OUT_FPGAREAD_LENMAX_MCDI2 1020
#define    MC_CMD_PTP_OUT_FPGAREAD_LEN(num) (0+1*(num))
#define    MC_CMD_PTP_OUT_FPGAREAD_BUFFER_NUM(len) (((len)-0)/1)
#define       MC_CMD_PTP_OUT_FPGAREAD_BUFFER_OFST 0
#define       MC_CMD_PTP_OUT_FPGAREAD_BUFFER_LEN 1
#define       MC_CMD_PTP_OUT_FPGAREAD_BUFFER_MINNUM 1
#define       MC_CMD_PTP_OUT_FPGAREAD_BUFFER_MAXNUM 252
#define       MC_CMD_PTP_OUT_FPGAREAD_BUFFER_MAXNUM_MCDI2 1020

/* MC_CMD_PTP_OUT_GET_TIME_FORMAT msgresponse */
#define    MC_CMD_PTP_OUT_GET_TIME_FORMAT_LEN 4
/* Time format required/used by for this NIC. Applies to all PTP MCDI
 * operations that pass times between the host and firmware. If this operation
 * is not supported (older firmware) a format of seconds and nanoseconds should
 * be assumed. Note this enum is deprecated. Do not add to it- use the
 * TIME_FORMAT field in MC_CMD_PTP_OUT_GET_ATTRIBUTES instead.
 */
#define       MC_CMD_PTP_OUT_GET_TIME_FORMAT_FORMAT_OFST 0
#define       MC_CMD_PTP_OUT_GET_TIME_FORMAT_FORMAT_LEN 4
/* enum: Times are in seconds and nanoseconds */
#define          MC_CMD_PTP_OUT_GET_TIME_FORMAT_SECONDS_NANOSECONDS 0x0
/* enum: Major register has units of 16 second per tick, minor 8 ns per tick */
#define          MC_CMD_PTP_OUT_GET_TIME_FORMAT_16SECONDS_8NANOSECONDS 0x1
/* enum: Major register has units of seconds, minor 2^-27s per tick */
#define          MC_CMD_PTP_OUT_GET_TIME_FORMAT_SECONDS_27FRACTION 0x2

/* MC_CMD_PTP_OUT_GET_ATTRIBUTES msgresponse */
#define    MC_CMD_PTP_OUT_GET_ATTRIBUTES_LEN 24
/* Time format required/used by for this NIC. Applies to all PTP MCDI
 * operations that pass times between the host and firmware. If this operation
 * is not supported (older firmware) a format of seconds and nanoseconds should
 * be assumed.
 */
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_TIME_FORMAT_OFST 0
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_TIME_FORMAT_LEN 4
/* enum: Times are in seconds and nanoseconds */
#define          MC_CMD_PTP_OUT_GET_ATTRIBUTES_SECONDS_NANOSECONDS 0x0
/* enum: Major register has units of 16 second per tick, minor 8 ns per tick */
#define          MC_CMD_PTP_OUT_GET_ATTRIBUTES_16SECONDS_8NANOSECONDS 0x1
/* enum: Major register has units of seconds, minor 2^-27s per tick */
#define          MC_CMD_PTP_OUT_GET_ATTRIBUTES_SECONDS_27FRACTION 0x2
/* enum: Major register units are seconds, minor units are quarter nanoseconds
 */
#define          MC_CMD_PTP_OUT_GET_ATTRIBUTES_SECONDS_QTR_NANOSECONDS 0x3
/* Minimum acceptable value for a corrected synchronization timeset. When
 * comparing host and NIC clock times, the MC returns a set of samples that
 * contain the host start and end time, the MC time when the host start was
 * detected and the time the MC waited between reading the time and detecting
 * the host end. The corrected sync window is the difference between the host
 * end and start times minus the time that the MC waited for host end.
 */
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_SYNC_WINDOW_MIN_OFST 4
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_SYNC_WINDOW_MIN_LEN 4
/* Various PTP capabilities */
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_CAPABILITIES_OFST 8
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_CAPABILITIES_LEN 4
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_REPORT_SYNC_STATUS_OFST 8
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_REPORT_SYNC_STATUS_LBN 0
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_REPORT_SYNC_STATUS_WIDTH 1
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_RX_TSTAMP_OOB_OFST 8
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_RX_TSTAMP_OOB_LBN 1
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_RX_TSTAMP_OOB_WIDTH 1
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_64BIT_SECONDS_OFST 8
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_64BIT_SECONDS_LBN 2
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_64BIT_SECONDS_WIDTH 1
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_FP44_FREQ_ADJ_OFST 8
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_FP44_FREQ_ADJ_LBN 3
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_FP44_FREQ_ADJ_WIDTH 1
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_RESERVED0_OFST 12
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_RESERVED0_LEN 4
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_RESERVED1_OFST 16
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_RESERVED1_LEN 4
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_RESERVED2_OFST 20
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_RESERVED2_LEN 4

/* MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2 msgresponse */
#define    MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_LEN 40
/* Time format required/used by for this NIC. Applies to all PTP MCDI
 * operations that pass times between the host and firmware. If this operation
 * is not supported (older firmware) a format of seconds and nanoseconds should
 * be assumed.
 */
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_TIME_FORMAT_OFST 0
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_TIME_FORMAT_LEN 4
/* enum: Times are in seconds and nanoseconds */
#define          MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_SECONDS_NANOSECONDS 0x0
/* enum: Major register has units of 16 second per tick, minor 8 ns per tick */
#define          MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_16SECONDS_8NANOSECONDS 0x1
/* enum: Major register has units of seconds, minor 2^-27s per tick */
#define          MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_SECONDS_27FRACTION 0x2
/* enum: Major register units are seconds, minor units are quarter nanoseconds
 */
#define          MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_SECONDS_QTR_NANOSECONDS 0x3
/* Minimum acceptable value for a corrected synchronization timeset. When
 * comparing host and NIC clock times, the MC returns a set of samples that
 * contain the host start and end time, the MC time when the host start was
 * detected and the time the MC waited between reading the time and detecting
 * the host end. The corrected sync window is the difference between the host
 * end and start times minus the time that the MC waited for host end.
 */
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_SYNC_WINDOW_MIN_OFST 4
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_SYNC_WINDOW_MIN_LEN 4
/* Various PTP capabilities */
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_CAPABILITIES_OFST 8
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_CAPABILITIES_LEN 4
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_REPORT_SYNC_STATUS_OFST 8
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_REPORT_SYNC_STATUS_LBN 0
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_REPORT_SYNC_STATUS_WIDTH 1
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_RX_TSTAMP_OOB_OFST 8
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_RX_TSTAMP_OOB_LBN 1
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_RX_TSTAMP_OOB_WIDTH 1
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_64BIT_SECONDS_OFST 8
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_64BIT_SECONDS_LBN 2
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_64BIT_SECONDS_WIDTH 1
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FP44_FREQ_ADJ_OFST 8
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FP44_FREQ_ADJ_LBN 3
#define        MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FP44_FREQ_ADJ_WIDTH 1
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_RESERVED0_OFST 12
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_RESERVED0_LEN 4
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_RESERVED1_OFST 16
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_RESERVED1_LEN 4
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_RESERVED2_OFST 20
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_RESERVED2_LEN 4
/* Minimum supported value for the FREQ field in
 * MC_CMD_PTP/MC_CMD_PTP_IN_ADJUST and
 * MC_CMD_PTP/MC_CMD_PTP_IN_CLOCK_FREQ_ADJUST message requests. If this message
 * response is not supported a value of -0.1 ns should be assumed, which is
 * equivalent to a -10% adjustment.
 */
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MIN_OFST 24
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MIN_LEN 8
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MIN_LO_OFST 24
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MIN_LO_LEN 4
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MIN_LO_LBN 192
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MIN_LO_WIDTH 32
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MIN_HI_OFST 28
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MIN_HI_LEN 4
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MIN_HI_LBN 224
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MIN_HI_WIDTH 32
/* Maximum supported value for the FREQ field in
 * MC_CMD_PTP/MC_CMD_PTP_IN_ADJUST and
 * MC_CMD_PTP/MC_CMD_PTP_IN_CLOCK_FREQ_ADJUST message requests. If this message
 * response is not supported a value of 0.1 ns should be assumed, which is
 * equivalent to a +10% adjustment.
 */
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MAX_OFST 32
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MAX_LEN 8
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MAX_LO_OFST 32
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MAX_LO_LEN 4
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MAX_LO_LBN 256
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MAX_LO_WIDTH 32
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MAX_HI_OFST 36
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MAX_HI_LEN 4
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MAX_HI_LBN 288
#define       MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MAX_HI_WIDTH 32

/* MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS msgresponse */
#define    MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_LEN 16
/* Uncorrected error on PTP transmit timestamps in NIC clock format */
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_TRANSMIT_OFST 0
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_TRANSMIT_LEN 4
/* Uncorrected error on PTP receive timestamps in NIC clock format */
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_RECEIVE_OFST 4
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_RECEIVE_LEN 4
/* Uncorrected error on PPS output in NIC clock format */
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_PPS_OUT_OFST 8
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_PPS_OUT_LEN 4
/* Uncorrected error on PPS input in NIC clock format */
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_PPS_IN_OFST 12
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_PPS_IN_LEN 4

/* MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2 msgresponse */
#define    MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_LEN 24
/* Uncorrected error on PTP transmit timestamps in NIC clock format */
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_PTP_TX_OFST 0
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_PTP_TX_LEN 4
/* Uncorrected error on PTP receive timestamps in NIC clock format */
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_PTP_RX_OFST 4
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_PTP_RX_LEN 4
/* Uncorrected error on PPS output in NIC clock format */
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_PPS_OUT_OFST 8
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_PPS_OUT_LEN 4
/* Uncorrected error on PPS input in NIC clock format */
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_PPS_IN_OFST 12
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_PPS_IN_LEN 4
/* Uncorrected error on non-PTP transmit timestamps in NIC clock format */
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_GENERAL_TX_OFST 16
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_GENERAL_TX_LEN 4
/* Uncorrected error on non-PTP receive timestamps in NIC clock format */
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_GENERAL_RX_OFST 20
#define       MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_GENERAL_RX_LEN 4

/* MC_CMD_PTP_OUT_MANFTEST_PPS msgresponse */
#define    MC_CMD_PTP_OUT_MANFTEST_PPS_LEN 4
/* Results of testing */
#define       MC_CMD_PTP_OUT_MANFTEST_PPS_TEST_RESULT_OFST 0
#define       MC_CMD_PTP_OUT_MANFTEST_PPS_TEST_RESULT_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_PTP_OUT_MANFTEST_BASIC/TEST_RESULT */

/* MC_CMD_PTP_OUT_SET_SYNC_STATUS msgresponse */
#define    MC_CMD_PTP_OUT_SET_SYNC_STATUS_LEN 0

/* MC_CMD_PTP_OUT_GET_SYNC_TIMEOUT msgresponse */
#define    MC_CMD_PTP_OUT_GET_SYNC_TIMEOUT_LEN 8
/* Current value set in NIC, in seconds */
#define       MC_CMD_PTP_OUT_GET_SYNC_TIMEOUT_CURRENT_OFST 0
#define       MC_CMD_PTP_OUT_GET_SYNC_TIMEOUT_CURRENT_LEN 4
/* Maximum supported by NIC, in seconds */
#define       MC_CMD_PTP_OUT_GET_SYNC_TIMEOUT_MAXIMUM_OFST 4
#define       MC_CMD_PTP_OUT_GET_SYNC_TIMEOUT_MAXIMUM_LEN 4


/***********************************/
/* MC_CMD_GET_BOARD_CFG
 * Returns the MC firmware configuration structure.
 */
#define MC_CMD_GET_BOARD_CFG 0x18
#undef MC_CMD_0x18_PRIVILEGE_CTG

#define MC_CMD_0x18_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_GET_BOARD_CFG_IN msgrequest */
#define    MC_CMD_GET_BOARD_CFG_IN_LEN 0

/* MC_CMD_GET_BOARD_CFG_OUT msgresponse */
#define    MC_CMD_GET_BOARD_CFG_OUT_LENMIN 96
#define    MC_CMD_GET_BOARD_CFG_OUT_LENMAX 136
#define    MC_CMD_GET_BOARD_CFG_OUT_LENMAX_MCDI2 136
#define    MC_CMD_GET_BOARD_CFG_OUT_LEN(num) (72+2*(num))
#define    MC_CMD_GET_BOARD_CFG_OUT_FW_SUBTYPE_LIST_NUM(len) (((len)-72)/2)
#define       MC_CMD_GET_BOARD_CFG_OUT_BOARD_TYPE_OFST 0
#define       MC_CMD_GET_BOARD_CFG_OUT_BOARD_TYPE_LEN 4
#define       MC_CMD_GET_BOARD_CFG_OUT_BOARD_NAME_OFST 4
#define       MC_CMD_GET_BOARD_CFG_OUT_BOARD_NAME_LEN 32
/* Capabilities for Siena Port0 (see struct MC_CMD_CAPABILITIES). Unused on
 * EF10 and later (use MC_CMD_GET_CAPABILITIES).
 */
#define       MC_CMD_GET_BOARD_CFG_OUT_CAPABILITIES_PORT0_OFST 36
#define       MC_CMD_GET_BOARD_CFG_OUT_CAPABILITIES_PORT0_LEN 4
/* Capabilities for Siena Port1 (see struct MC_CMD_CAPABILITIES). Unused on
 * EF10 and later (use MC_CMD_GET_CAPABILITIES).
 */
#define       MC_CMD_GET_BOARD_CFG_OUT_CAPABILITIES_PORT1_OFST 40
#define       MC_CMD_GET_BOARD_CFG_OUT_CAPABILITIES_PORT1_LEN 4
/* Base MAC address for Siena Port0. Unused on EF10 and later (use
 * MC_CMD_GET_MAC_ADDRESSES).
 */
#define       MC_CMD_GET_BOARD_CFG_OUT_MAC_ADDR_BASE_PORT0_OFST 44
#define       MC_CMD_GET_BOARD_CFG_OUT_MAC_ADDR_BASE_PORT0_LEN 6
/* Base MAC address for Siena Port1. Unused on EF10 and later (use
 * MC_CMD_GET_MAC_ADDRESSES).
 */
#define       MC_CMD_GET_BOARD_CFG_OUT_MAC_ADDR_BASE_PORT1_OFST 50
#define       MC_CMD_GET_BOARD_CFG_OUT_MAC_ADDR_BASE_PORT1_LEN 6
/* Size of MAC address pool for Siena Port0. Unused on EF10 and later (use
 * MC_CMD_GET_MAC_ADDRESSES).
 */
#define       MC_CMD_GET_BOARD_CFG_OUT_MAC_COUNT_PORT0_OFST 56
#define       MC_CMD_GET_BOARD_CFG_OUT_MAC_COUNT_PORT0_LEN 4
/* Size of MAC address pool for Siena Port1. Unused on EF10 and later (use
 * MC_CMD_GET_MAC_ADDRESSES).
 */
#define       MC_CMD_GET_BOARD_CFG_OUT_MAC_COUNT_PORT1_OFST 60
#define       MC_CMD_GET_BOARD_CFG_OUT_MAC_COUNT_PORT1_LEN 4
/* Increment between addresses in MAC address pool for Siena Port0. Unused on
 * EF10 and later (use MC_CMD_GET_MAC_ADDRESSES).
 */
#define       MC_CMD_GET_BOARD_CFG_OUT_MAC_STRIDE_PORT0_OFST 64
#define       MC_CMD_GET_BOARD_CFG_OUT_MAC_STRIDE_PORT0_LEN 4
/* Increment between addresses in MAC address pool for Siena Port1. Unused on
 * EF10 and later (use MC_CMD_GET_MAC_ADDRESSES).
 */
#define       MC_CMD_GET_BOARD_CFG_OUT_MAC_STRIDE_PORT1_OFST 68
#define       MC_CMD_GET_BOARD_CFG_OUT_MAC_STRIDE_PORT1_LEN 4
/* Siena only. This field contains a 16-bit value for each of the types of
 * NVRAM area. The values are defined in the firmware/mc/platform/.c file for a
 * specific board type, but otherwise have no meaning to the MC; they are used
 * by the driver to manage selection of appropriate firmware updates. Unused on
 * EF10 and later (use MC_CMD_NVRAM_METADATA).
 */
#define       MC_CMD_GET_BOARD_CFG_OUT_FW_SUBTYPE_LIST_OFST 72
#define       MC_CMD_GET_BOARD_CFG_OUT_FW_SUBTYPE_LIST_LEN 2
#define       MC_CMD_GET_BOARD_CFG_OUT_FW_SUBTYPE_LIST_MINNUM 12
#define       MC_CMD_GET_BOARD_CFG_OUT_FW_SUBTYPE_LIST_MAXNUM 32
#define       MC_CMD_GET_BOARD_CFG_OUT_FW_SUBTYPE_LIST_MAXNUM_MCDI2 32


/***********************************/
/* MC_CMD_DRV_ATTACH
 * Inform MCPU that this port is managed on the host (i.e. driver active). For
 * Huntington, also request the preferred datapath firmware to use if possible
 * (it may not be possible for this request to be fulfilled; the driver must
 * issue a subsequent MC_CMD_GET_CAPABILITIES command to determine which
 * features are actually available). The FIRMWARE_ID field is ignored by older
 * platforms.
 */
#define MC_CMD_DRV_ATTACH 0x1c
#undef MC_CMD_0x1c_PRIVILEGE_CTG

#define MC_CMD_0x1c_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_DRV_ATTACH_IN msgrequest */
#define    MC_CMD_DRV_ATTACH_IN_LEN 12
/* new state to set if UPDATE=1 */
#define       MC_CMD_DRV_ATTACH_IN_NEW_STATE_OFST 0
#define       MC_CMD_DRV_ATTACH_IN_NEW_STATE_LEN 4
#define        MC_CMD_DRV_ATTACH_OFST 0
#define        MC_CMD_DRV_ATTACH_LBN 0
#define        MC_CMD_DRV_ATTACH_WIDTH 1
#define        MC_CMD_DRV_ATTACH_IN_ATTACH_OFST 0
#define        MC_CMD_DRV_ATTACH_IN_ATTACH_LBN 0
#define        MC_CMD_DRV_ATTACH_IN_ATTACH_WIDTH 1
#define        MC_CMD_DRV_PREBOOT_OFST 0
#define        MC_CMD_DRV_PREBOOT_LBN 1
#define        MC_CMD_DRV_PREBOOT_WIDTH 1
#define        MC_CMD_DRV_ATTACH_IN_PREBOOT_OFST 0
#define        MC_CMD_DRV_ATTACH_IN_PREBOOT_LBN 1
#define        MC_CMD_DRV_ATTACH_IN_PREBOOT_WIDTH 1
#define        MC_CMD_DRV_ATTACH_IN_SUBVARIANT_AWARE_OFST 0
#define        MC_CMD_DRV_ATTACH_IN_SUBVARIANT_AWARE_LBN 2
#define        MC_CMD_DRV_ATTACH_IN_SUBVARIANT_AWARE_WIDTH 1
#define        MC_CMD_DRV_ATTACH_IN_WANT_VI_SPREADING_OFST 0
#define        MC_CMD_DRV_ATTACH_IN_WANT_VI_SPREADING_LBN 3
#define        MC_CMD_DRV_ATTACH_IN_WANT_VI_SPREADING_WIDTH 1
#define        MC_CMD_DRV_ATTACH_IN_WANT_V2_LINKCHANGES_OFST 0
#define        MC_CMD_DRV_ATTACH_IN_WANT_V2_LINKCHANGES_LBN 4
#define        MC_CMD_DRV_ATTACH_IN_WANT_V2_LINKCHANGES_WIDTH 1
#define        MC_CMD_DRV_ATTACH_IN_WANT_RX_VI_SPREADING_INHIBIT_OFST 0
#define        MC_CMD_DRV_ATTACH_IN_WANT_RX_VI_SPREADING_INHIBIT_LBN 5
#define        MC_CMD_DRV_ATTACH_IN_WANT_RX_VI_SPREADING_INHIBIT_WIDTH 1
#define        MC_CMD_DRV_ATTACH_IN_WANT_TX_ONLY_SPREADING_OFST 0
#define        MC_CMD_DRV_ATTACH_IN_WANT_TX_ONLY_SPREADING_LBN 5
#define        MC_CMD_DRV_ATTACH_IN_WANT_TX_ONLY_SPREADING_WIDTH 1
/* 1 to set new state, or 0 to just report the existing state */
#define       MC_CMD_DRV_ATTACH_IN_UPDATE_OFST 4
#define       MC_CMD_DRV_ATTACH_IN_UPDATE_LEN 4
/* preferred datapath firmware (for Huntington; ignored for Siena) */
#define       MC_CMD_DRV_ATTACH_IN_FIRMWARE_ID_OFST 8
#define       MC_CMD_DRV_ATTACH_IN_FIRMWARE_ID_LEN 4
/* enum: Prefer to use full featured firmware */
#define          MC_CMD_FW_FULL_FEATURED 0x0
/* enum: Prefer to use firmware with fewer features but lower latency */
#define          MC_CMD_FW_LOW_LATENCY 0x1
/* enum: Prefer to use firmware for SolarCapture packed stream mode */
#define          MC_CMD_FW_PACKED_STREAM 0x2
/* enum: Prefer to use firmware with fewer features and simpler TX event
 * batching but higher TX packet rate
 */
#define          MC_CMD_FW_HIGH_TX_RATE 0x3
/* enum: Reserved value */
#define          MC_CMD_FW_PACKED_STREAM_HASH_MODE_1 0x4
/* enum: Prefer to use firmware with additional "rules engine" filtering
 * support
 */
#define          MC_CMD_FW_RULES_ENGINE 0x5
/* enum: Prefer to use firmware with additional DPDK support */
#define          MC_CMD_FW_DPDK 0x6
/* enum: Prefer to use "l3xudp" custom datapath firmware (see SF-119495-PD and
 * bug69716)
 */
#define          MC_CMD_FW_L3XUDP 0x7
/* enum: Requests that the MC keep whatever datapath firmware is currently
 * running. It's used for test purposes, where we want to be able to shmboot
 * special test firmware variants. This option is only recognised in eftest
 * (i.e. non-production) builds.
 */
#define          MC_CMD_FW_KEEP_CURRENT_EFTEST_ONLY 0xfffffffe
/* enum: Only this option is allowed for non-admin functions */
#define          MC_CMD_FW_DONT_CARE 0xffffffff

/* MC_CMD_DRV_ATTACH_IN_V2 msgrequest: Updated DRV_ATTACH to include driver
 * version
 */
#define    MC_CMD_DRV_ATTACH_IN_V2_LEN 32
/* new state to set if UPDATE=1 */
#define       MC_CMD_DRV_ATTACH_IN_V2_NEW_STATE_OFST 0
#define       MC_CMD_DRV_ATTACH_IN_V2_NEW_STATE_LEN 4
/*             MC_CMD_DRV_ATTACH_OFST 0 */
/*             MC_CMD_DRV_ATTACH_LBN 0 */
/*             MC_CMD_DRV_ATTACH_WIDTH 1 */
#define        MC_CMD_DRV_ATTACH_IN_V2_ATTACH_OFST 0
#define        MC_CMD_DRV_ATTACH_IN_V2_ATTACH_LBN 0
#define        MC_CMD_DRV_ATTACH_IN_V2_ATTACH_WIDTH 1
/*             MC_CMD_DRV_PREBOOT_OFST 0 */
/*             MC_CMD_DRV_PREBOOT_LBN 1 */
/*             MC_CMD_DRV_PREBOOT_WIDTH 1 */
#define        MC_CMD_DRV_ATTACH_IN_V2_PREBOOT_OFST 0
#define        MC_CMD_DRV_ATTACH_IN_V2_PREBOOT_LBN 1
#define        MC_CMD_DRV_ATTACH_IN_V2_PREBOOT_WIDTH 1
#define        MC_CMD_DRV_ATTACH_IN_V2_SUBVARIANT_AWARE_OFST 0
#define        MC_CMD_DRV_ATTACH_IN_V2_SUBVARIANT_AWARE_LBN 2
#define        MC_CMD_DRV_ATTACH_IN_V2_SUBVARIANT_AWARE_WIDTH 1
#define        MC_CMD_DRV_ATTACH_IN_V2_WANT_VI_SPREADING_OFST 0
#define        MC_CMD_DRV_ATTACH_IN_V2_WANT_VI_SPREADING_LBN 3
#define        MC_CMD_DRV_ATTACH_IN_V2_WANT_VI_SPREADING_WIDTH 1
#define        MC_CMD_DRV_ATTACH_IN_V2_WANT_V2_LINKCHANGES_OFST 0
#define        MC_CMD_DRV_ATTACH_IN_V2_WANT_V2_LINKCHANGES_LBN 4
#define        MC_CMD_DRV_ATTACH_IN_V2_WANT_V2_LINKCHANGES_WIDTH 1
#define        MC_CMD_DRV_ATTACH_IN_V2_WANT_RX_VI_SPREADING_INHIBIT_OFST 0
#define        MC_CMD_DRV_ATTACH_IN_V2_WANT_RX_VI_SPREADING_INHIBIT_LBN 5
#define        MC_CMD_DRV_ATTACH_IN_V2_WANT_RX_VI_SPREADING_INHIBIT_WIDTH 1
#define        MC_CMD_DRV_ATTACH_IN_V2_WANT_TX_ONLY_SPREADING_OFST 0
#define        MC_CMD_DRV_ATTACH_IN_V2_WANT_TX_ONLY_SPREADING_LBN 5
#define        MC_CMD_DRV_ATTACH_IN_V2_WANT_TX_ONLY_SPREADING_WIDTH 1
/* 1 to set new state, or 0 to just report the existing state */
#define       MC_CMD_DRV_ATTACH_IN_V2_UPDATE_OFST 4
#define       MC_CMD_DRV_ATTACH_IN_V2_UPDATE_LEN 4
/* preferred datapath firmware (for Huntington; ignored for Siena) */
#define       MC_CMD_DRV_ATTACH_IN_V2_FIRMWARE_ID_OFST 8
#define       MC_CMD_DRV_ATTACH_IN_V2_FIRMWARE_ID_LEN 4
/* enum: Prefer to use full featured firmware */
/*               MC_CMD_FW_FULL_FEATURED 0x0 */
/* enum: Prefer to use firmware with fewer features but lower latency */
/*               MC_CMD_FW_LOW_LATENCY 0x1 */
/* enum: Prefer to use firmware for SolarCapture packed stream mode */
/*               MC_CMD_FW_PACKED_STREAM 0x2 */
/* enum: Prefer to use firmware with fewer features and simpler TX event
 * batching but higher TX packet rate
 */
/*               MC_CMD_FW_HIGH_TX_RATE 0x3 */
/* enum: Reserved value */
/*               MC_CMD_FW_PACKED_STREAM_HASH_MODE_1 0x4 */
/* enum: Prefer to use firmware with additional "rules engine" filtering
 * support
 */
/*               MC_CMD_FW_RULES_ENGINE 0x5 */
/* enum: Prefer to use firmware with additional DPDK support */
/*               MC_CMD_FW_DPDK 0x6 */
/* enum: Prefer to use "l3xudp" custom datapath firmware (see SF-119495-PD and
 * bug69716)
 */
/*               MC_CMD_FW_L3XUDP 0x7 */
/* enum: Requests that the MC keep whatever datapath firmware is currently
 * running. It's used for test purposes, where we want to be able to shmboot
 * special test firmware variants. This option is only recognised in eftest
 * (i.e. non-production) builds.
 */
/*               MC_CMD_FW_KEEP_CURRENT_EFTEST_ONLY 0xfffffffe */
/* enum: Only this option is allowed for non-admin functions */
/*               MC_CMD_FW_DONT_CARE 0xffffffff */
/* Version of the driver to be reported by management protocols (e.g. NC-SI)
 * handled by the NIC. This is a zero-terminated ASCII string.
 */
#define       MC_CMD_DRV_ATTACH_IN_V2_DRIVER_VERSION_OFST 12
#define       MC_CMD_DRV_ATTACH_IN_V2_DRIVER_VERSION_LEN 20

/* MC_CMD_DRV_ATTACH_OUT msgresponse */
#define    MC_CMD_DRV_ATTACH_OUT_LEN 4
/* previous or existing state, see the bitmask at NEW_STATE */
#define       MC_CMD_DRV_ATTACH_OUT_OLD_STATE_OFST 0
#define       MC_CMD_DRV_ATTACH_OUT_OLD_STATE_LEN 4

/* MC_CMD_DRV_ATTACH_EXT_OUT msgresponse */
#define    MC_CMD_DRV_ATTACH_EXT_OUT_LEN 8
/* previous or existing state, see the bitmask at NEW_STATE */
#define       MC_CMD_DRV_ATTACH_EXT_OUT_OLD_STATE_OFST 0
#define       MC_CMD_DRV_ATTACH_EXT_OUT_OLD_STATE_LEN 4
/* Flags associated with this function */
#define       MC_CMD_DRV_ATTACH_EXT_OUT_FUNC_FLAGS_OFST 4
#define       MC_CMD_DRV_ATTACH_EXT_OUT_FUNC_FLAGS_LEN 4
/* enum property: bitshift */
/* enum: Labels the lowest-numbered function visible to the OS */
#define          MC_CMD_DRV_ATTACH_EXT_OUT_FLAG_PRIMARY 0x0
/* enum: The function can control the link state of the physical port it is
 * bound to.
 */
#define          MC_CMD_DRV_ATTACH_EXT_OUT_FLAG_LINKCTRL 0x1
/* enum: The function can perform privileged operations */
#define          MC_CMD_DRV_ATTACH_EXT_OUT_FLAG_TRUSTED 0x2
/* enum: The function does not have an active port associated with it. The port
 * refers to the Sorrento external FPGA port.
 */
#define          MC_CMD_DRV_ATTACH_EXT_OUT_FLAG_NO_ACTIVE_PORT 0x3
/* enum: If set, indicates that VI spreading is currently enabled. Will always
 * indicate the current state, regardless of the value in the WANT_VI_SPREADING
 * input.
 */
#define          MC_CMD_DRV_ATTACH_EXT_OUT_FLAG_VI_SPREADING_ENABLED 0x4
/* enum: Used during development only. Should no longer be used. */
#define          MC_CMD_DRV_ATTACH_EXT_OUT_FLAG_RX_VI_SPREADING_INHIBITED 0x5
/* enum: If set, indicates that TX only spreading is enabled. Even-numbered
 * TXQs will use one engine, and odd-numbered TXQs will use the other. This
 * also has the effect that only even-numbered RXQs will receive traffic.
 */
#define          MC_CMD_DRV_ATTACH_EXT_OUT_FLAG_TX_ONLY_VI_SPREADING_ENABLED 0x5


/***********************************/
/* MC_CMD_PORT_RESET
 * Generic per-port reset. There is no equivalent for per-board reset. Locks
 * required: None; Return code: 0, ETIME. NOTE: This command is deprecated -
 * use MC_CMD_ENTITY_RESET instead.
 */
#define MC_CMD_PORT_RESET 0x20
#undef MC_CMD_0x20_PRIVILEGE_CTG

#define MC_CMD_0x20_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_PORT_RESET_IN msgrequest */
#define    MC_CMD_PORT_RESET_IN_LEN 0

/* MC_CMD_PORT_RESET_OUT msgresponse */
#define    MC_CMD_PORT_RESET_OUT_LEN 0


/***********************************/
/* MC_CMD_ENTITY_RESET
 * Generic per-resource reset. There is no equivalent for per-board reset.
 * Locks required: None; Return code: 0, ETIME. NOTE: This command is an
 * extended version of the deprecated MC_CMD_PORT_RESET with added fields.
 */
#define MC_CMD_ENTITY_RESET 0x20
/*      MC_CMD_0x20_PRIVILEGE_CTG SRIOV_CTG_GENERAL */

/* MC_CMD_ENTITY_RESET_IN msgrequest */
#define    MC_CMD_ENTITY_RESET_IN_LEN 4
/* Optional flags field. Omitting this will perform a "legacy" reset action
 * (TBD).
 */
#define       MC_CMD_ENTITY_RESET_IN_FLAG_OFST 0
#define       MC_CMD_ENTITY_RESET_IN_FLAG_LEN 4
#define        MC_CMD_ENTITY_RESET_IN_FUNCTION_RESOURCE_RESET_OFST 0
#define        MC_CMD_ENTITY_RESET_IN_FUNCTION_RESOURCE_RESET_LBN 0
#define        MC_CMD_ENTITY_RESET_IN_FUNCTION_RESOURCE_RESET_WIDTH 1

/* MC_CMD_ENTITY_RESET_OUT msgresponse */
#define    MC_CMD_ENTITY_RESET_OUT_LEN 0


/***********************************/
/* MC_CMD_PUTS
 * Copy the given ASCII string out onto UART and/or out of the network port.
 */
#define MC_CMD_PUTS 0x23
#undef MC_CMD_0x23_PRIVILEGE_CTG

#define MC_CMD_0x23_PRIVILEGE_CTG SRIOV_CTG_INSECURE

/* MC_CMD_PUTS_IN msgrequest */
#define    MC_CMD_PUTS_IN_LENMIN 13
#define    MC_CMD_PUTS_IN_LENMAX 252
#define    MC_CMD_PUTS_IN_LENMAX_MCDI2 1020
#define    MC_CMD_PUTS_IN_LEN(num) (12+1*(num))
#define    MC_CMD_PUTS_IN_STRING_NUM(len) (((len)-12)/1)
#define       MC_CMD_PUTS_IN_DEST_OFST 0
#define       MC_CMD_PUTS_IN_DEST_LEN 4
#define        MC_CMD_PUTS_IN_UART_OFST 0
#define        MC_CMD_PUTS_IN_UART_LBN 0
#define        MC_CMD_PUTS_IN_UART_WIDTH 1
#define        MC_CMD_PUTS_IN_PORT_OFST 0
#define        MC_CMD_PUTS_IN_PORT_LBN 1
#define        MC_CMD_PUTS_IN_PORT_WIDTH 1
#define       MC_CMD_PUTS_IN_DHOST_OFST 4
#define       MC_CMD_PUTS_IN_DHOST_LEN 6
#define       MC_CMD_PUTS_IN_STRING_OFST 12
#define       MC_CMD_PUTS_IN_STRING_LEN 1
#define       MC_CMD_PUTS_IN_STRING_MINNUM 1
#define       MC_CMD_PUTS_IN_STRING_MAXNUM 240
#define       MC_CMD_PUTS_IN_STRING_MAXNUM_MCDI2 1008

/* MC_CMD_PUTS_OUT msgresponse */
#define    MC_CMD_PUTS_OUT_LEN 0


/***********************************/
/* MC_CMD_GET_PHY_CFG
 * Report PHY configuration. This guarantees to succeed even if the PHY is in a
 * 'zombie' state. Locks required: None
 */
#define MC_CMD_GET_PHY_CFG 0x24
#undef MC_CMD_0x24_PRIVILEGE_CTG

#define MC_CMD_0x24_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_GET_PHY_CFG_IN msgrequest */
#define    MC_CMD_GET_PHY_CFG_IN_LEN 0

/* MC_CMD_GET_PHY_CFG_IN_V2 msgrequest */
#define    MC_CMD_GET_PHY_CFG_IN_V2_LEN 8
/* Target port to request PHY state for. Uses MAE_LINK_ENDPOINT_SELECTOR which
 * identifies a real or virtual network port by MAE port and link end. See the
 * structure definition for more details
 */
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_OFST 0
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_LEN 8
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_LO_OFST 0
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_LO_LEN 4
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_LO_LBN 0
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_LO_WIDTH 32
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_HI_OFST 4
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_HI_LEN 4
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_HI_LBN 32
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_HI_WIDTH 32
/* See structuredef: MAE_LINK_ENDPOINT_SELECTOR */
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_OFST 0
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_LEN 4
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_FLAT_OFST 0
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_FLAT_LEN 4
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_TYPE_OFST 3
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_TYPE_LEN 1
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_MPORT_ID_OFST 0
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_MPORT_ID_LEN 3
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_PPORT_ID_LBN 0
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_PPORT_ID_WIDTH 4
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_FUNC_INTF_ID_LBN 20
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_FUNC_INTF_ID_WIDTH 4
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_FUNC_MH_PF_ID_LBN 16
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_FUNC_MH_PF_ID_WIDTH 4
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_FUNC_PF_ID_OFST 2
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_FUNC_PF_ID_LEN 1
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_FUNC_VF_ID_OFST 0
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_MPORT_SELECTOR_FUNC_VF_ID_LEN 2
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_LINK_END_OFST 4
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_LINK_END_LEN 4
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_FLAT_OFST 0
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_FLAT_LEN 8
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_FLAT_LO_OFST 0
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_FLAT_LO_LEN 4
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_FLAT_LO_LBN 0
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_FLAT_LO_WIDTH 32
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_FLAT_HI_OFST 4
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_FLAT_HI_LEN 4
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_FLAT_HI_LBN 32
#define       MC_CMD_GET_PHY_CFG_IN_V2_TARGET_FLAT_HI_WIDTH 32

/* MC_CMD_GET_PHY_CFG_OUT msgresponse */
#define    MC_CMD_GET_PHY_CFG_OUT_LEN 72
/* flags */
#define       MC_CMD_GET_PHY_CFG_OUT_FLAGS_OFST 0
#define       MC_CMD_GET_PHY_CFG_OUT_FLAGS_LEN 4
#define        MC_CMD_GET_PHY_CFG_OUT_PRESENT_OFST 0
#define        MC_CMD_GET_PHY_CFG_OUT_PRESENT_LBN 0
#define        MC_CMD_GET_PHY_CFG_OUT_PRESENT_WIDTH 1
#define        MC_CMD_GET_PHY_CFG_OUT_BIST_CABLE_SHORT_OFST 0
#define        MC_CMD_GET_PHY_CFG_OUT_BIST_CABLE_SHORT_LBN 1
#define        MC_CMD_GET_PHY_CFG_OUT_BIST_CABLE_SHORT_WIDTH 1
#define        MC_CMD_GET_PHY_CFG_OUT_BIST_CABLE_LONG_OFST 0
#define        MC_CMD_GET_PHY_CFG_OUT_BIST_CABLE_LONG_LBN 2
#define        MC_CMD_GET_PHY_CFG_OUT_BIST_CABLE_LONG_WIDTH 1
#define        MC_CMD_GET_PHY_CFG_OUT_LOWPOWER_OFST 0
#define        MC_CMD_GET_PHY_CFG_OUT_LOWPOWER_LBN 3
#define        MC_CMD_GET_PHY_CFG_OUT_LOWPOWER_WIDTH 1
#define        MC_CMD_GET_PHY_CFG_OUT_POWEROFF_OFST 0
#define        MC_CMD_GET_PHY_CFG_OUT_POWEROFF_LBN 4
#define        MC_CMD_GET_PHY_CFG_OUT_POWEROFF_WIDTH 1
#define        MC_CMD_GET_PHY_CFG_OUT_TXDIS_OFST 0
#define        MC_CMD_GET_PHY_CFG_OUT_TXDIS_LBN 5
#define        MC_CMD_GET_PHY_CFG_OUT_TXDIS_WIDTH 1
#define        MC_CMD_GET_PHY_CFG_OUT_BIST_OFST 0
#define        MC_CMD_GET_PHY_CFG_OUT_BIST_LBN 6
#define        MC_CMD_GET_PHY_CFG_OUT_BIST_WIDTH 1
/* ?? */
#define       MC_CMD_GET_PHY_CFG_OUT_TYPE_OFST 4
#define       MC_CMD_GET_PHY_CFG_OUT_TYPE_LEN 4
/* Bitmask of supported capabilities */
#define       MC_CMD_GET_PHY_CFG_OUT_SUPPORTED_CAP_OFST 8
#define       MC_CMD_GET_PHY_CFG_OUT_SUPPORTED_CAP_LEN 4
#define        MC_CMD_PHY_CAP_10HDX_OFST 8
#define        MC_CMD_PHY_CAP_10HDX_LBN 1
#define        MC_CMD_PHY_CAP_10HDX_WIDTH 1
#define        MC_CMD_PHY_CAP_10FDX_OFST 8
#define        MC_CMD_PHY_CAP_10FDX_LBN 2
#define        MC_CMD_PHY_CAP_10FDX_WIDTH 1
#define        MC_CMD_PHY_CAP_100HDX_OFST 8
#define        MC_CMD_PHY_CAP_100HDX_LBN 3
#define        MC_CMD_PHY_CAP_100HDX_WIDTH 1
#define        MC_CMD_PHY_CAP_100FDX_OFST 8
#define        MC_CMD_PHY_CAP_100FDX_LBN 4
#define        MC_CMD_PHY_CAP_100FDX_WIDTH 1
#define        MC_CMD_PHY_CAP_1000HDX_OFST 8
#define        MC_CMD_PHY_CAP_1000HDX_LBN 5
#define        MC_CMD_PHY_CAP_1000HDX_WIDTH 1
#define        MC_CMD_PHY_CAP_1000FDX_OFST 8
#define        MC_CMD_PHY_CAP_1000FDX_LBN 6
#define        MC_CMD_PHY_CAP_1000FDX_WIDTH 1
#define        MC_CMD_PHY_CAP_10000FDX_OFST 8
#define        MC_CMD_PHY_CAP_10000FDX_LBN 7
#define        MC_CMD_PHY_CAP_10000FDX_WIDTH 1
#define        MC_CMD_PHY_CAP_PAUSE_OFST 8
#define        MC_CMD_PHY_CAP_PAUSE_LBN 8
#define        MC_CMD_PHY_CAP_PAUSE_WIDTH 1
#define        MC_CMD_PHY_CAP_ASYM_OFST 8
#define        MC_CMD_PHY_CAP_ASYM_LBN 9
#define        MC_CMD_PHY_CAP_ASYM_WIDTH 1
#define        MC_CMD_PHY_CAP_AN_OFST 8
#define        MC_CMD_PHY_CAP_AN_LBN 10
#define        MC_CMD_PHY_CAP_AN_WIDTH 1
#define        MC_CMD_PHY_CAP_40000FDX_OFST 8
#define        MC_CMD_PHY_CAP_40000FDX_LBN 11
#define        MC_CMD_PHY_CAP_40000FDX_WIDTH 1
#define        MC_CMD_PHY_CAP_DDM_OFST 8
#define        MC_CMD_PHY_CAP_DDM_LBN 12
#define        MC_CMD_PHY_CAP_DDM_WIDTH 1
#define        MC_CMD_PHY_CAP_100000FDX_OFST 8
#define        MC_CMD_PHY_CAP_100000FDX_LBN 13
#define        MC_CMD_PHY_CAP_100000FDX_WIDTH 1
#define        MC_CMD_PHY_CAP_25000FDX_OFST 8
#define        MC_CMD_PHY_CAP_25000FDX_LBN 14
#define        MC_CMD_PHY_CAP_25000FDX_WIDTH 1
#define        MC_CMD_PHY_CAP_50000FDX_OFST 8
#define        MC_CMD_PHY_CAP_50000FDX_LBN 15
#define        MC_CMD_PHY_CAP_50000FDX_WIDTH 1
#define        MC_CMD_PHY_CAP_BASER_FEC_OFST 8
#define        MC_CMD_PHY_CAP_BASER_FEC_LBN 16
#define        MC_CMD_PHY_CAP_BASER_FEC_WIDTH 1
#define        MC_CMD_PHY_CAP_BASER_FEC_REQUESTED_OFST 8
#define        MC_CMD_PHY_CAP_BASER_FEC_REQUESTED_LBN 17
#define        MC_CMD_PHY_CAP_BASER_FEC_REQUESTED_WIDTH 1
#define        MC_CMD_PHY_CAP_RS_FEC_OFST 8
#define        MC_CMD_PHY_CAP_RS_FEC_LBN 18
#define        MC_CMD_PHY_CAP_RS_FEC_WIDTH 1
#define        MC_CMD_PHY_CAP_RS_FEC_REQUESTED_OFST 8
#define        MC_CMD_PHY_CAP_RS_FEC_REQUESTED_LBN 19
#define        MC_CMD_PHY_CAP_RS_FEC_REQUESTED_WIDTH 1
#define        MC_CMD_PHY_CAP_25G_BASER_FEC_OFST 8
#define        MC_CMD_PHY_CAP_25G_BASER_FEC_LBN 20
#define        MC_CMD_PHY_CAP_25G_BASER_FEC_WIDTH 1
#define        MC_CMD_PHY_CAP_25G_BASER_FEC_REQUESTED_OFST 8
#define        MC_CMD_PHY_CAP_25G_BASER_FEC_REQUESTED_LBN 21
#define        MC_CMD_PHY_CAP_25G_BASER_FEC_REQUESTED_WIDTH 1
#define        MC_CMD_PHY_CAP_200000FDX_OFST 8
#define        MC_CMD_PHY_CAP_200000FDX_LBN 22
#define        MC_CMD_PHY_CAP_200000FDX_WIDTH 1
/* ?? */
#define       MC_CMD_GET_PHY_CFG_OUT_CHANNEL_OFST 12
#define       MC_CMD_GET_PHY_CFG_OUT_CHANNEL_LEN 4
/* ?? */
#define       MC_CMD_GET_PHY_CFG_OUT_PRT_OFST 16
#define       MC_CMD_GET_PHY_CFG_OUT_PRT_LEN 4
/* ?? */
#define       MC_CMD_GET_PHY_CFG_OUT_STATS_MASK_OFST 20
#define       MC_CMD_GET_PHY_CFG_OUT_STATS_MASK_LEN 4
/* ?? */
#define       MC_CMD_GET_PHY_CFG_OUT_NAME_OFST 24
#define       MC_CMD_GET_PHY_CFG_OUT_NAME_LEN 20
/* ?? */
#define       MC_CMD_GET_PHY_CFG_OUT_MEDIA_TYPE_OFST 44
#define       MC_CMD_GET_PHY_CFG_OUT_MEDIA_TYPE_LEN 4
/* enum: Xaui. */
#define          MC_CMD_MEDIA_XAUI 0x1
/* enum: CX4. */
#define          MC_CMD_MEDIA_CX4 0x2
/* enum: KX4. */
#define          MC_CMD_MEDIA_KX4 0x3
/* enum: XFP Far. */
#define          MC_CMD_MEDIA_XFP 0x4
/* enum: SFP+. */
#define          MC_CMD_MEDIA_SFP_PLUS 0x5
/* enum: 10GBaseT. */
#define          MC_CMD_MEDIA_BASE_T 0x6
/* enum: QSFP+. */
#define          MC_CMD_MEDIA_QSFP_PLUS 0x7
/* enum: DSFP. */
#define          MC_CMD_MEDIA_DSFP 0x8
#define       MC_CMD_GET_PHY_CFG_OUT_MMD_MASK_OFST 48
#define       MC_CMD_GET_PHY_CFG_OUT_MMD_MASK_LEN 4
/* enum property: bitshift */
/* enum: Native clause 22 */
#define          MC_CMD_MMD_CLAUSE22 0x0
#define          MC_CMD_MMD_CLAUSE45_PMAPMD 0x1 /* enum */
#define          MC_CMD_MMD_CLAUSE45_WIS 0x2 /* enum */
#define          MC_CMD_MMD_CLAUSE45_PCS 0x3 /* enum */
#define          MC_CMD_MMD_CLAUSE45_PHYXS 0x4 /* enum */
#define          MC_CMD_MMD_CLAUSE45_DTEXS 0x5 /* enum */
#define          MC_CMD_MMD_CLAUSE45_TC 0x6 /* enum */
#define          MC_CMD_MMD_CLAUSE45_AN 0x7 /* enum */
/* enum: Clause22 proxied over clause45 by PHY. */
#define          MC_CMD_MMD_CLAUSE45_C22EXT 0x1d
#define          MC_CMD_MMD_CLAUSE45_VEND1 0x1e /* enum */
#define          MC_CMD_MMD_CLAUSE45_VEND2 0x1f /* enum */
#define       MC_CMD_GET_PHY_CFG_OUT_REVISION_OFST 52
#define       MC_CMD_GET_PHY_CFG_OUT_REVISION_LEN 20


/***********************************/
/* MC_CMD_START_BIST
 * Start a BIST test on the PHY. Locks required: PHY_LOCK if doing a PHY BIST
 * Return code: 0, EINVAL, EACCES (if PHY_LOCK is not held)
 */
#define MC_CMD_START_BIST 0x25
#undef MC_CMD_0x25_PRIVILEGE_CTG

#define MC_CMD_0x25_PRIVILEGE_CTG SRIOV_CTG_ADMIN_TSA_UNBOUND

/* MC_CMD_START_BIST_IN msgrequest */
#define    MC_CMD_START_BIST_IN_LEN 4
/* Type of test. */
#define       MC_CMD_START_BIST_IN_TYPE_OFST 0
#define       MC_CMD_START_BIST_IN_TYPE_LEN 4
/* enum: Run the PHY's short cable BIST. */
#define          MC_CMD_PHY_BIST_CABLE_SHORT 0x1
/* enum: Run the PHY's long cable BIST. */
#define          MC_CMD_PHY_BIST_CABLE_LONG 0x2
/* enum: Run BIST on the currently selected BPX Serdes (XAUI or XFI) . */
#define          MC_CMD_BPX_SERDES_BIST 0x3
/* enum: Run the MC loopback tests. */
#define          MC_CMD_MC_LOOPBACK_BIST 0x4
/* enum: Run the PHY's standard BIST. */
#define          MC_CMD_PHY_BIST 0x5
/* enum: Run MC RAM test. */
#define          MC_CMD_MC_MEM_BIST 0x6
/* enum: Run Port RAM test. */
#define          MC_CMD_PORT_MEM_BIST 0x7
/* enum: Run register test. */
#define          MC_CMD_REG_BIST 0x8

/* MC_CMD_START_BIST_OUT msgresponse */
#define    MC_CMD_START_BIST_OUT_LEN 0


/***********************************/
/* MC_CMD_POLL_BIST
 * Poll for BIST completion. Returns a single status code, and optionally some
 * PHY specific bist output. The driver should only consume the BIST output
 * after validating OUTLEN and MC_CMD_GET_PHY_CFG.TYPE. If a driver can't
 * successfully parse the BIST output, it should still respect the pass/Fail in
 * OUT.RESULT. Locks required: PHY_LOCK if doing a PHY BIST. Return code: 0,
 * EACCES (if PHY_LOCK is not held).
 */
#define MC_CMD_POLL_BIST 0x26
#undef MC_CMD_0x26_PRIVILEGE_CTG

#define MC_CMD_0x26_PRIVILEGE_CTG SRIOV_CTG_ADMIN_TSA_UNBOUND

/* MC_CMD_POLL_BIST_IN msgrequest */
#define    MC_CMD_POLL_BIST_IN_LEN 0

/* MC_CMD_POLL_BIST_OUT msgresponse */
#define    MC_CMD_POLL_BIST_OUT_LEN 8
/* result */
#define       MC_CMD_POLL_BIST_OUT_RESULT_OFST 0
#define       MC_CMD_POLL_BIST_OUT_RESULT_LEN 4
/* enum: Running. */
#define          MC_CMD_POLL_BIST_RUNNING 0x1
/* enum: Passed. */
#define          MC_CMD_POLL_BIST_PASSED 0x2
/* enum: Failed. */
#define          MC_CMD_POLL_BIST_FAILED 0x3
/* enum: Timed-out. */
#define          MC_CMD_POLL_BIST_TIMEOUT 0x4
#define       MC_CMD_POLL_BIST_OUT_PRIVATE_OFST 4
#define       MC_CMD_POLL_BIST_OUT_PRIVATE_LEN 4

/* MC_CMD_POLL_BIST_OUT_SFT9001 msgresponse */
#define    MC_CMD_POLL_BIST_OUT_SFT9001_LEN 36
/* result */
/*            MC_CMD_POLL_BIST_OUT_RESULT_OFST 0 */
/*            MC_CMD_POLL_BIST_OUT_RESULT_LEN 4 */
/*            Enum values, see field(s): */
/*               MC_CMD_POLL_BIST_OUT/MC_CMD_POLL_BIST_OUT_RESULT */
#define       MC_CMD_POLL_BIST_OUT_SFT9001_CABLE_LENGTH_A_OFST 4
#define       MC_CMD_POLL_BIST_OUT_SFT9001_CABLE_LENGTH_A_LEN 4
#define       MC_CMD_POLL_BIST_OUT_SFT9001_CABLE_LENGTH_B_OFST 8
#define       MC_CMD_POLL_BIST_OUT_SFT9001_CABLE_LENGTH_B_LEN 4
#define       MC_CMD_POLL_BIST_OUT_SFT9001_CABLE_LENGTH_C_OFST 12
#define       MC_CMD_POLL_BIST_OUT_SFT9001_CABLE_LENGTH_C_LEN 4
#define       MC_CMD_POLL_BIST_OUT_SFT9001_CABLE_LENGTH_D_OFST 16
#define       MC_CMD_POLL_BIST_OUT_SFT9001_CABLE_LENGTH_D_LEN 4
/* Status of each channel A */
#define       MC_CMD_POLL_BIST_OUT_SFT9001_CABLE_STATUS_A_OFST 20
#define       MC_CMD_POLL_BIST_OUT_SFT9001_CABLE_STATUS_A_LEN 4
/* enum: Ok. */
#define          MC_CMD_POLL_BIST_SFT9001_PAIR_OK 0x1
/* enum: Open. */
#define          MC_CMD_POLL_BIST_SFT9001_PAIR_OPEN 0x2
/* enum: Intra-pair short. */
#define          MC_CMD_POLL_BIST_SFT9001_INTRA_PAIR_SHORT 0x3
/* enum: Inter-pair short. */
#define          MC_CMD_POLL_BIST_SFT9001_INTER_PAIR_SHORT 0x4
/* enum: Busy. */
#define          MC_CMD_POLL_BIST_SFT9001_PAIR_BUSY 0x9
/* Status of each channel B */
#define       MC_CMD_POLL_BIST_OUT_SFT9001_CABLE_STATUS_B_OFST 24
#define       MC_CMD_POLL_BIST_OUT_SFT9001_CABLE_STATUS_B_LEN 4
/*            Enum values, see field(s): */
/*               CABLE_STATUS_A */
/* Status of each channel C */
#define       MC_CMD_POLL_BIST_OUT_SFT9001_CABLE_STATUS_C_OFST 28
#define       MC_CMD_POLL_BIST_OUT_SFT9001_CABLE_STATUS_C_LEN 4
/*            Enum values, see field(s): */
/*               CABLE_STATUS_A */
/* Status of each channel D */
#define       MC_CMD_POLL_BIST_OUT_SFT9001_CABLE_STATUS_D_OFST 32
#define       MC_CMD_POLL_BIST_OUT_SFT9001_CABLE_STATUS_D_LEN 4
/*            Enum values, see field(s): */
/*               CABLE_STATUS_A */

/* MC_CMD_POLL_BIST_OUT_MRSFP msgresponse */
#define    MC_CMD_POLL_BIST_OUT_MRSFP_LEN 8
/* result */
/*            MC_CMD_POLL_BIST_OUT_RESULT_OFST 0 */
/*            MC_CMD_POLL_BIST_OUT_RESULT_LEN 4 */
/*            Enum values, see field(s): */
/*               MC_CMD_POLL_BIST_OUT/MC_CMD_POLL_BIST_OUT_RESULT */
#define       MC_CMD_POLL_BIST_OUT_MRSFP_TEST_OFST 4
#define       MC_CMD_POLL_BIST_OUT_MRSFP_TEST_LEN 4
/* enum: Complete. */
#define          MC_CMD_POLL_BIST_MRSFP_TEST_COMPLETE 0x0
/* enum: Bus switch off I2C write. */
#define          MC_CMD_POLL_BIST_MRSFP_TEST_BUS_SWITCH_OFF_I2C_WRITE 0x1
/* enum: Bus switch off I2C no access IO exp. */
#define          MC_CMD_POLL_BIST_MRSFP_TEST_BUS_SWITCH_OFF_I2C_NO_ACCESS_IO_EXP 0x2
/* enum: Bus switch off I2C no access module. */
#define          MC_CMD_POLL_BIST_MRSFP_TEST_BUS_SWITCH_OFF_I2C_NO_ACCESS_MODULE 0x3
/* enum: IO exp I2C configure. */
#define          MC_CMD_POLL_BIST_MRSFP_TEST_IO_EXP_I2C_CONFIGURE 0x4
/* enum: Bus switch I2C no cross talk. */
#define          MC_CMD_POLL_BIST_MRSFP_TEST_BUS_SWITCH_I2C_NO_CROSSTALK 0x5
/* enum: Module presence. */
#define          MC_CMD_POLL_BIST_MRSFP_TEST_MODULE_PRESENCE 0x6
/* enum: Module ID I2C access. */
#define          MC_CMD_POLL_BIST_MRSFP_TEST_MODULE_ID_I2C_ACCESS 0x7
/* enum: Module ID sane value. */
#define          MC_CMD_POLL_BIST_MRSFP_TEST_MODULE_ID_SANE_VALUE 0x8

/* MC_CMD_POLL_BIST_OUT_MEM msgresponse */
#define    MC_CMD_POLL_BIST_OUT_MEM_LEN 36
/* result */
/*            MC_CMD_POLL_BIST_OUT_RESULT_OFST 0 */
/*            MC_CMD_POLL_BIST_OUT_RESULT_LEN 4 */
/*            Enum values, see field(s): */
/*               MC_CMD_POLL_BIST_OUT/MC_CMD_POLL_BIST_OUT_RESULT */
#define       MC_CMD_POLL_BIST_OUT_MEM_TEST_OFST 4
#define       MC_CMD_POLL_BIST_OUT_MEM_TEST_LEN 4
/* enum: Test has completed. */
#define          MC_CMD_POLL_BIST_MEM_COMPLETE 0x0
/* enum: RAM test - walk ones. */
#define          MC_CMD_POLL_BIST_MEM_MEM_WALK_ONES 0x1
/* enum: RAM test - walk zeros. */
#define          MC_CMD_POLL_BIST_MEM_MEM_WALK_ZEROS 0x2
/* enum: RAM test - walking inversions zeros/ones. */
#define          MC_CMD_POLL_BIST_MEM_MEM_INV_ZERO_ONE 0x3
/* enum: RAM test - walking inversions checkerboard. */
#define          MC_CMD_POLL_BIST_MEM_MEM_INV_CHKBOARD 0x4
/* enum: Register test - set / clear individual bits. */
#define          MC_CMD_POLL_BIST_MEM_REG 0x5
/* enum: ECC error detected. */
#define          MC_CMD_POLL_BIST_MEM_ECC 0x6
/* Failure address, only valid if result is POLL_BIST_FAILED */
#define       MC_CMD_POLL_BIST_OUT_MEM_ADDR_OFST 8
#define       MC_CMD_POLL_BIST_OUT_MEM_ADDR_LEN 4
/* Bus or address space to which the failure address corresponds */
#define       MC_CMD_POLL_BIST_OUT_MEM_BUS_OFST 12
#define       MC_CMD_POLL_BIST_OUT_MEM_BUS_LEN 4
/* enum: MC MIPS bus. */
#define          MC_CMD_POLL_BIST_MEM_BUS_MC 0x0
/* enum: CSR IREG bus. */
#define          MC_CMD_POLL_BIST_MEM_BUS_CSR 0x1
/* enum: RX0 DPCPU bus. */
#define          MC_CMD_POLL_BIST_MEM_BUS_DPCPU_RX 0x2
/* enum: TX0 DPCPU bus. */
#define          MC_CMD_POLL_BIST_MEM_BUS_DPCPU_TX0 0x3
/* enum: TX1 DPCPU bus. */
#define          MC_CMD_POLL_BIST_MEM_BUS_DPCPU_TX1 0x4
/* enum: RX0 DICPU bus. */
#define          MC_CMD_POLL_BIST_MEM_BUS_DICPU_RX 0x5
/* enum: TX DICPU bus. */
#define          MC_CMD_POLL_BIST_MEM_BUS_DICPU_TX 0x6
/* enum: RX1 DPCPU bus. */
#define          MC_CMD_POLL_BIST_MEM_BUS_DPCPU_RX1 0x7
/* enum: RX1 DICPU bus. */
#define          MC_CMD_POLL_BIST_MEM_BUS_DICPU_RX1 0x8
/* Pattern written to RAM / register */
#define       MC_CMD_POLL_BIST_OUT_MEM_EXPECT_OFST 16
#define       MC_CMD_POLL_BIST_OUT_MEM_EXPECT_LEN 4
/* Actual value read from RAM / register */
#define       MC_CMD_POLL_BIST_OUT_MEM_ACTUAL_OFST 20
#define       MC_CMD_POLL_BIST_OUT_MEM_ACTUAL_LEN 4
/* ECC error mask */
#define       MC_CMD_POLL_BIST_OUT_MEM_ECC_OFST 24
#define       MC_CMD_POLL_BIST_OUT_MEM_ECC_LEN 4
/* ECC parity error mask */
#define       MC_CMD_POLL_BIST_OUT_MEM_ECC_PARITY_OFST 28
#define       MC_CMD_POLL_BIST_OUT_MEM_ECC_PARITY_LEN 4
/* ECC fatal error mask */
#define       MC_CMD_POLL_BIST_OUT_MEM_ECC_FATAL_OFST 32
#define       MC_CMD_POLL_BIST_OUT_MEM_ECC_FATAL_LEN 4


/***********************************/
/* MC_CMD_GET_LOOPBACK_MODES
 * Returns a bitmask of loopback modes available at each speed.
 */
#define MC_CMD_GET_LOOPBACK_MODES 0x28
#undef MC_CMD_0x28_PRIVILEGE_CTG

#define MC_CMD_0x28_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_GET_LOOPBACK_MODES_IN msgrequest */
#define    MC_CMD_GET_LOOPBACK_MODES_IN_LEN 0

/* MC_CMD_GET_LOOPBACK_MODES_IN_V2 msgrequest */
#define    MC_CMD_GET_LOOPBACK_MODES_IN_V2_LEN 8
/* Target port to request loopback modes for. Uses MAE_LINK_ENDPOINT_SELECTOR
 * which identifies a real or virtual network port by MAE port and link end.
 * See the structure definition for more details
 */
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_OFST 0
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_LO_OFST 0
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_LO_LBN 0
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_HI_OFST 4
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_HI_LBN 32
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_HI_WIDTH 32
/* See structuredef: MAE_LINK_ENDPOINT_SELECTOR */
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_OFST 0
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_FLAT_OFST 0
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_FLAT_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_TYPE_OFST 3
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_TYPE_LEN 1
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_MPORT_ID_OFST 0
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_MPORT_ID_LEN 3
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_PPORT_ID_LBN 0
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_PPORT_ID_WIDTH 4
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_FUNC_INTF_ID_LBN 20
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_FUNC_INTF_ID_WIDTH 4
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_FUNC_MH_PF_ID_LBN 16
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_FUNC_MH_PF_ID_WIDTH 4
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_FUNC_PF_ID_OFST 2
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_FUNC_PF_ID_LEN 1
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_FUNC_VF_ID_OFST 0
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_MPORT_SELECTOR_FUNC_VF_ID_LEN 2
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_LINK_END_OFST 4
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_LINK_END_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_FLAT_OFST 0
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_FLAT_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_FLAT_LO_OFST 0
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_FLAT_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_FLAT_LO_LBN 0
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_FLAT_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_FLAT_HI_OFST 4
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_FLAT_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_FLAT_HI_LBN 32
#define       MC_CMD_GET_LOOPBACK_MODES_IN_V2_TARGET_FLAT_HI_WIDTH 32

/* MC_CMD_GET_LOOPBACK_MODES_OUT msgresponse */
#define    MC_CMD_GET_LOOPBACK_MODES_OUT_LEN 40
/* Supported loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_100M_OFST 0
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_100M_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_100M_LO_OFST 0
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_100M_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_100M_LO_LBN 0
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_100M_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_100M_HI_OFST 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_100M_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_100M_HI_LBN 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_100M_HI_WIDTH 32
/* enum property: bitshift */
/* enum: None. */
#define          MC_CMD_LOOPBACK_NONE 0x0
/* enum: Data. */
#define          MC_CMD_LOOPBACK_DATA 0x1
/* enum: GMAC. */
#define          MC_CMD_LOOPBACK_GMAC 0x2
/* enum: XGMII. */
#define          MC_CMD_LOOPBACK_XGMII 0x3
/* enum: XGXS. */
#define          MC_CMD_LOOPBACK_XGXS 0x4
/* enum: XAUI. */
#define          MC_CMD_LOOPBACK_XAUI 0x5
/* enum: GMII. */
#define          MC_CMD_LOOPBACK_GMII 0x6
/* enum: SGMII. */
#define          MC_CMD_LOOPBACK_SGMII 0x7
/* enum: XGBR. */
#define          MC_CMD_LOOPBACK_XGBR 0x8
/* enum: XFI. */
#define          MC_CMD_LOOPBACK_XFI 0x9
/* enum: XAUI Far. */
#define          MC_CMD_LOOPBACK_XAUI_FAR 0xa
/* enum: GMII Far. */
#define          MC_CMD_LOOPBACK_GMII_FAR 0xb
/* enum: SGMII Far. */
#define          MC_CMD_LOOPBACK_SGMII_FAR 0xc
/* enum: XFI Far. */
#define          MC_CMD_LOOPBACK_XFI_FAR 0xd
/* enum: GPhy. */
#define          MC_CMD_LOOPBACK_GPHY 0xe
/* enum: PhyXS. */
#define          MC_CMD_LOOPBACK_PHYXS 0xf
/* enum: PCS. */
#define          MC_CMD_LOOPBACK_PCS 0x10
/* enum: PMA-PMD. */
#define          MC_CMD_LOOPBACK_PMAPMD 0x11
/* enum: Cross-Port. */
#define          MC_CMD_LOOPBACK_XPORT 0x12
/* enum: XGMII-Wireside. */
#define          MC_CMD_LOOPBACK_XGMII_WS 0x13
/* enum: XAUI Wireside. */
#define          MC_CMD_LOOPBACK_XAUI_WS 0x14
/* enum: XAUI Wireside Far. */
#define          MC_CMD_LOOPBACK_XAUI_WS_FAR 0x15
/* enum: XAUI Wireside near. */
#define          MC_CMD_LOOPBACK_XAUI_WS_NEAR 0x16
/* enum: GMII Wireside. */
#define          MC_CMD_LOOPBACK_GMII_WS 0x17
/* enum: XFI Wireside. */
#define          MC_CMD_LOOPBACK_XFI_WS 0x18
/* enum: XFI Wireside Far. */
#define          MC_CMD_LOOPBACK_XFI_WS_FAR 0x19
/* enum: PhyXS Wireside. */
#define          MC_CMD_LOOPBACK_PHYXS_WS 0x1a
/* enum: PMA lanes MAC-Serdes. */
#define          MC_CMD_LOOPBACK_PMA_INT 0x1b
/* enum: KR Serdes Parallel (Encoder). */
#define          MC_CMD_LOOPBACK_SD_NEAR 0x1c
/* enum: KR Serdes Serial. */
#define          MC_CMD_LOOPBACK_SD_FAR 0x1d
/* enum: PMA lanes MAC-Serdes Wireside. */
#define          MC_CMD_LOOPBACK_PMA_INT_WS 0x1e
/* enum: KR Serdes Parallel Wireside (Full PCS). */
#define          MC_CMD_LOOPBACK_SD_FEP2_WS 0x1f
/* enum: KR Serdes Parallel Wireside (Sym Aligner to TX). */
#define          MC_CMD_LOOPBACK_SD_FEP1_5_WS 0x20
/* enum: KR Serdes Parallel Wireside (Deserializer to Serializer). */
#define          MC_CMD_LOOPBACK_SD_FEP_WS 0x21
/* enum: KR Serdes Serial Wireside. */
#define          MC_CMD_LOOPBACK_SD_FES_WS 0x22
/* enum: Near side of AOE Siena side port */
#define          MC_CMD_LOOPBACK_AOE_INT_NEAR 0x23
/* enum: Medford Wireside datapath loopback */
#define          MC_CMD_LOOPBACK_DATA_WS 0x24
/* enum: Force link up without setting up any physical loopback (snapper use
 * only)
 */
#define          MC_CMD_LOOPBACK_FORCE_EXT_LINK 0x25
/* Supported loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_1G_OFST 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_1G_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_1G_LO_OFST 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_1G_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_1G_LO_LBN 64
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_1G_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_1G_HI_OFST 12
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_1G_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_1G_HI_LBN 96
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_1G_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */
/* Supported loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_10G_OFST 16
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_10G_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_10G_LO_OFST 16
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_10G_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_10G_LO_LBN 128
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_10G_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_10G_HI_OFST 20
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_10G_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_10G_HI_LBN 160
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_10G_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */
/* Supported loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_SUGGESTED_OFST 24
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_SUGGESTED_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_SUGGESTED_LO_OFST 24
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_SUGGESTED_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_SUGGESTED_LO_LBN 192
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_SUGGESTED_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_SUGGESTED_HI_OFST 28
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_SUGGESTED_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_SUGGESTED_HI_LBN 224
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_SUGGESTED_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */
/* Supported loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_40G_OFST 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_40G_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_40G_LO_OFST 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_40G_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_40G_LO_LBN 256
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_40G_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_40G_HI_OFST 36
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_40G_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_40G_HI_LBN 288
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_40G_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */

/* MC_CMD_GET_LOOPBACK_MODES_OUT_V2 msgresponse: Supported loopback modes for
 * newer NICs with 25G/50G/100G support
 */
#define    MC_CMD_GET_LOOPBACK_MODES_OUT_V2_LEN 64
/* Supported loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100M_OFST 0
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100M_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100M_LO_OFST 0
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100M_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100M_LO_LBN 0
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100M_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100M_HI_OFST 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100M_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100M_HI_LBN 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100M_HI_WIDTH 32
/* enum property: bitshift */
/* enum: None. */
/*               MC_CMD_LOOPBACK_NONE 0x0 */
/* enum: Data. */
/*               MC_CMD_LOOPBACK_DATA 0x1 */
/* enum: GMAC. */
/*               MC_CMD_LOOPBACK_GMAC 0x2 */
/* enum: XGMII. */
/*               MC_CMD_LOOPBACK_XGMII 0x3 */
/* enum: XGXS. */
/*               MC_CMD_LOOPBACK_XGXS 0x4 */
/* enum: XAUI. */
/*               MC_CMD_LOOPBACK_XAUI 0x5 */
/* enum: GMII. */
/*               MC_CMD_LOOPBACK_GMII 0x6 */
/* enum: SGMII. */
/*               MC_CMD_LOOPBACK_SGMII 0x7 */
/* enum: XGBR. */
/*               MC_CMD_LOOPBACK_XGBR 0x8 */
/* enum: XFI. */
/*               MC_CMD_LOOPBACK_XFI 0x9 */
/* enum: XAUI Far. */
/*               MC_CMD_LOOPBACK_XAUI_FAR 0xa */
/* enum: GMII Far. */
/*               MC_CMD_LOOPBACK_GMII_FAR 0xb */
/* enum: SGMII Far. */
/*               MC_CMD_LOOPBACK_SGMII_FAR 0xc */
/* enum: XFI Far. */
/*               MC_CMD_LOOPBACK_XFI_FAR 0xd */
/* enum: GPhy. */
/*               MC_CMD_LOOPBACK_GPHY 0xe */
/* enum: PhyXS. */
/*               MC_CMD_LOOPBACK_PHYXS 0xf */
/* enum: PCS. */
/*               MC_CMD_LOOPBACK_PCS 0x10 */
/* enum: PMA-PMD. */
/*               MC_CMD_LOOPBACK_PMAPMD 0x11 */
/* enum: Cross-Port. */
/*               MC_CMD_LOOPBACK_XPORT 0x12 */
/* enum: XGMII-Wireside. */
/*               MC_CMD_LOOPBACK_XGMII_WS 0x13 */
/* enum: XAUI Wireside. */
/*               MC_CMD_LOOPBACK_XAUI_WS 0x14 */
/* enum: XAUI Wireside Far. */
/*               MC_CMD_LOOPBACK_XAUI_WS_FAR 0x15 */
/* enum: XAUI Wireside near. */
/*               MC_CMD_LOOPBACK_XAUI_WS_NEAR 0x16 */
/* enum: GMII Wireside. */
/*               MC_CMD_LOOPBACK_GMII_WS 0x17 */
/* enum: XFI Wireside. */
/*               MC_CMD_LOOPBACK_XFI_WS 0x18 */
/* enum: XFI Wireside Far. */
/*               MC_CMD_LOOPBACK_XFI_WS_FAR 0x19 */
/* enum: PhyXS Wireside. */
/*               MC_CMD_LOOPBACK_PHYXS_WS 0x1a */
/* enum: PMA lanes MAC-Serdes. */
/*               MC_CMD_LOOPBACK_PMA_INT 0x1b */
/* enum: KR Serdes Parallel (Encoder). */
/*               MC_CMD_LOOPBACK_SD_NEAR 0x1c */
/* enum: KR Serdes Serial. */
/*               MC_CMD_LOOPBACK_SD_FAR 0x1d */
/* enum: PMA lanes MAC-Serdes Wireside. */
/*               MC_CMD_LOOPBACK_PMA_INT_WS 0x1e */
/* enum: KR Serdes Parallel Wireside (Full PCS). */
/*               MC_CMD_LOOPBACK_SD_FEP2_WS 0x1f */
/* enum: KR Serdes Parallel Wireside (Sym Aligner to TX). */
/*               MC_CMD_LOOPBACK_SD_FEP1_5_WS 0x20 */
/* enum: KR Serdes Parallel Wireside (Deserializer to Serializer). */
/*               MC_CMD_LOOPBACK_SD_FEP_WS 0x21 */
/* enum: KR Serdes Serial Wireside. */
/*               MC_CMD_LOOPBACK_SD_FES_WS 0x22 */
/* enum: Near side of AOE Siena side port */
/*               MC_CMD_LOOPBACK_AOE_INT_NEAR 0x23 */
/* enum: Medford Wireside datapath loopback */
/*               MC_CMD_LOOPBACK_DATA_WS 0x24 */
/* enum: Force link up without setting up any physical loopback (snapper use
 * only)
 */
/*               MC_CMD_LOOPBACK_FORCE_EXT_LINK 0x25 */
/* Supported loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_1G_OFST 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_1G_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_1G_LO_OFST 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_1G_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_1G_LO_LBN 64
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_1G_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_1G_HI_OFST 12
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_1G_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_1G_HI_LBN 96
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_1G_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */
/* Supported loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_10G_OFST 16
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_10G_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_10G_LO_OFST 16
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_10G_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_10G_LO_LBN 128
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_10G_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_10G_HI_OFST 20
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_10G_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_10G_HI_LBN 160
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_10G_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */
/* Supported loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_SUGGESTED_OFST 24
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_SUGGESTED_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_SUGGESTED_LO_OFST 24
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_SUGGESTED_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_SUGGESTED_LO_LBN 192
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_SUGGESTED_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_SUGGESTED_HI_OFST 28
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_SUGGESTED_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_SUGGESTED_HI_LBN 224
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_SUGGESTED_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */
/* Supported loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_40G_OFST 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_40G_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_40G_LO_OFST 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_40G_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_40G_LO_LBN 256
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_40G_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_40G_HI_OFST 36
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_40G_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_40G_HI_LBN 288
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_40G_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */
/* Supported 25G loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_25G_OFST 40
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_25G_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_25G_LO_OFST 40
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_25G_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_25G_LO_LBN 320
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_25G_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_25G_HI_OFST 44
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_25G_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_25G_HI_LBN 352
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_25G_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */
/* Supported 50 loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_50G_OFST 48
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_50G_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_50G_LO_OFST 48
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_50G_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_50G_LO_LBN 384
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_50G_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_50G_HI_OFST 52
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_50G_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_50G_HI_LBN 416
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_50G_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */
/* Supported 100G loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100G_OFST 56
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100G_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100G_LO_OFST 56
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100G_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100G_LO_LBN 448
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100G_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100G_HI_OFST 60
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100G_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100G_HI_LBN 480
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V2_100G_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */

/* MC_CMD_GET_LOOPBACK_MODES_OUT_V3 msgresponse: Supported loopback modes for
 * newer NICs with 200G support
 */
#define    MC_CMD_GET_LOOPBACK_MODES_OUT_V3_LEN 72
/* Supported loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100M_OFST 0
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100M_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100M_LO_OFST 0
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100M_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100M_LO_LBN 0
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100M_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100M_HI_OFST 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100M_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100M_HI_LBN 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100M_HI_WIDTH 32
/* enum property: bitshift */
/* enum: None. */
/*               MC_CMD_LOOPBACK_NONE 0x0 */
/* enum: Data. */
/*               MC_CMD_LOOPBACK_DATA 0x1 */
/* enum: GMAC. */
/*               MC_CMD_LOOPBACK_GMAC 0x2 */
/* enum: XGMII. */
/*               MC_CMD_LOOPBACK_XGMII 0x3 */
/* enum: XGXS. */
/*               MC_CMD_LOOPBACK_XGXS 0x4 */
/* enum: XAUI. */
/*               MC_CMD_LOOPBACK_XAUI 0x5 */
/* enum: GMII. */
/*               MC_CMD_LOOPBACK_GMII 0x6 */
/* enum: SGMII. */
/*               MC_CMD_LOOPBACK_SGMII 0x7 */
/* enum: XGBR. */
/*               MC_CMD_LOOPBACK_XGBR 0x8 */
/* enum: XFI. */
/*               MC_CMD_LOOPBACK_XFI 0x9 */
/* enum: XAUI Far. */
/*               MC_CMD_LOOPBACK_XAUI_FAR 0xa */
/* enum: GMII Far. */
/*               MC_CMD_LOOPBACK_GMII_FAR 0xb */
/* enum: SGMII Far. */
/*               MC_CMD_LOOPBACK_SGMII_FAR 0xc */
/* enum: XFI Far. */
/*               MC_CMD_LOOPBACK_XFI_FAR 0xd */
/* enum: GPhy. */
/*               MC_CMD_LOOPBACK_GPHY 0xe */
/* enum: PhyXS. */
/*               MC_CMD_LOOPBACK_PHYXS 0xf */
/* enum: PCS. */
/*               MC_CMD_LOOPBACK_PCS 0x10 */
/* enum: PMA-PMD. */
/*               MC_CMD_LOOPBACK_PMAPMD 0x11 */
/* enum: Cross-Port. */
/*               MC_CMD_LOOPBACK_XPORT 0x12 */
/* enum: XGMII-Wireside. */
/*               MC_CMD_LOOPBACK_XGMII_WS 0x13 */
/* enum: XAUI Wireside. */
/*               MC_CMD_LOOPBACK_XAUI_WS 0x14 */
/* enum: XAUI Wireside Far. */
/*               MC_CMD_LOOPBACK_XAUI_WS_FAR 0x15 */
/* enum: XAUI Wireside near. */
/*               MC_CMD_LOOPBACK_XAUI_WS_NEAR 0x16 */
/* enum: GMII Wireside. */
/*               MC_CMD_LOOPBACK_GMII_WS 0x17 */
/* enum: XFI Wireside. */
/*               MC_CMD_LOOPBACK_XFI_WS 0x18 */
/* enum: XFI Wireside Far. */
/*               MC_CMD_LOOPBACK_XFI_WS_FAR 0x19 */
/* enum: PhyXS Wireside. */
/*               MC_CMD_LOOPBACK_PHYXS_WS 0x1a */
/* enum: PMA lanes MAC-Serdes. */
/*               MC_CMD_LOOPBACK_PMA_INT 0x1b */
/* enum: KR Serdes Parallel (Encoder). */
/*               MC_CMD_LOOPBACK_SD_NEAR 0x1c */
/* enum: KR Serdes Serial. */
/*               MC_CMD_LOOPBACK_SD_FAR 0x1d */
/* enum: PMA lanes MAC-Serdes Wireside. */
/*               MC_CMD_LOOPBACK_PMA_INT_WS 0x1e */
/* enum: KR Serdes Parallel Wireside (Full PCS). */
/*               MC_CMD_LOOPBACK_SD_FEP2_WS 0x1f */
/* enum: KR Serdes Parallel Wireside (Sym Aligner to TX). */
/*               MC_CMD_LOOPBACK_SD_FEP1_5_WS 0x20 */
/* enum: KR Serdes Parallel Wireside (Deserializer to Serializer). */
/*               MC_CMD_LOOPBACK_SD_FEP_WS 0x21 */
/* enum: KR Serdes Serial Wireside. */
/*               MC_CMD_LOOPBACK_SD_FES_WS 0x22 */
/* enum: Near side of AOE Siena side port */
/*               MC_CMD_LOOPBACK_AOE_INT_NEAR 0x23 */
/* enum: Medford Wireside datapath loopback */
/*               MC_CMD_LOOPBACK_DATA_WS 0x24 */
/* enum: Force link up without setting up any physical loopback (snapper use
 * only)
 */
/*               MC_CMD_LOOPBACK_FORCE_EXT_LINK 0x25 */
/* Supported loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_1G_OFST 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_1G_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_1G_LO_OFST 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_1G_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_1G_LO_LBN 64
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_1G_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_1G_HI_OFST 12
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_1G_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_1G_HI_LBN 96
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_1G_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */
/* Supported loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_10G_OFST 16
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_10G_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_10G_LO_OFST 16
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_10G_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_10G_LO_LBN 128
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_10G_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_10G_HI_OFST 20
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_10G_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_10G_HI_LBN 160
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_10G_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */
/* Supported loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_SUGGESTED_OFST 24
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_SUGGESTED_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_SUGGESTED_LO_OFST 24
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_SUGGESTED_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_SUGGESTED_LO_LBN 192
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_SUGGESTED_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_SUGGESTED_HI_OFST 28
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_SUGGESTED_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_SUGGESTED_HI_LBN 224
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_SUGGESTED_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */
/* Supported loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_40G_OFST 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_40G_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_40G_LO_OFST 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_40G_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_40G_LO_LBN 256
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_40G_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_40G_HI_OFST 36
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_40G_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_40G_HI_LBN 288
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_40G_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */
/* Supported 25G loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_25G_OFST 40
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_25G_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_25G_LO_OFST 40
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_25G_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_25G_LO_LBN 320
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_25G_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_25G_HI_OFST 44
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_25G_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_25G_HI_LBN 352
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_25G_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */
/* Supported 50 loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_50G_OFST 48
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_50G_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_50G_LO_OFST 48
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_50G_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_50G_LO_LBN 384
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_50G_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_50G_HI_OFST 52
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_50G_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_50G_HI_LBN 416
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_50G_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */
/* Supported 100G loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100G_OFST 56
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100G_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100G_LO_OFST 56
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100G_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100G_LO_LBN 448
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100G_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100G_HI_OFST 60
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100G_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100G_HI_LBN 480
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_100G_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */
/* Supported 200G loopbacks. */
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_200G_OFST 64
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_200G_LEN 8
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_200G_LO_OFST 64
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_200G_LO_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_200G_LO_LBN 512
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_200G_LO_WIDTH 32
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_200G_HI_OFST 68
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_200G_HI_LEN 4
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_200G_HI_LBN 544
#define       MC_CMD_GET_LOOPBACK_MODES_OUT_V3_200G_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               100M */

/* AN_TYPE structuredef: Auto-negotiation types defined in IEEE802.3 */
#define    AN_TYPE_LEN 4
#define       AN_TYPE_TYPE_OFST 0
#define       AN_TYPE_TYPE_LEN 4
/* enum: None, AN disabled or not supported */
#define          MC_CMD_AN_NONE 0x0
/* enum: Clause 28 - BASE-T */
#define          MC_CMD_AN_CLAUSE28 0x1
/* enum: Clause 37 - BASE-X */
#define          MC_CMD_AN_CLAUSE37 0x2
/* enum: Clause 73 - BASE-R startup protocol for backplane and copper cable
 * assemblies. Includes Clause 72/Clause 92 link-training.
 */
#define          MC_CMD_AN_CLAUSE73 0x3
#define       AN_TYPE_TYPE_LBN 0
#define       AN_TYPE_TYPE_WIDTH 32

/* FEC_TYPE structuredef: Forward error correction types defined in IEEE802.3
 */
#define    FEC_TYPE_LEN 4
#define       FEC_TYPE_TYPE_OFST 0
#define       FEC_TYPE_TYPE_LEN 4
/* enum: No FEC */
#define          MC_CMD_FEC_NONE 0x0
/* enum: IEEE 802.3, Clause 74 BASE-R FEC (a.k.a Firecode) */
#define          MC_CMD_FEC_BASER 0x1
/* enum: IEEE 802.3, Clause 91/Clause 108 Reed-Solomon FEC */
#define          MC_CMD_FEC_RS 0x2
/* enum: IEEE 802.3, Clause 161, interleaved RS-FEC sublayer for 100GBASE-R
 * PHYs
 */
#define          MC_CMD_FEC_IEEE_RS_INT 0x3
/* enum: Ethernet Consortium, Low Latency RS-FEC. RS(272, 258). Replaces FEC
 * specified in Clause 119 for 100/200G PHY. Replaces FEC specified in Clause
 * 134 for 50G PHY.
 */
#define          MC_CMD_FEC_ETCS_RS_LL 0x4
/* enum: FEC mode selected automatically */
#define          MC_CMD_FEC_AUTO 0x5
#define       FEC_TYPE_TYPE_LBN 0
#define       FEC_TYPE_TYPE_WIDTH 32

/* MC_CMD_ETH_TECH structuredef: Ethernet technology as defined by IEEE802.3,
 * Ethernet Technology Consortium, proprietary technologies. The driver must
 * not use technologies labelled NONE and AUTO.
 */
#define    MC_CMD_ETH_TECH_LEN 16
/* The enums in this field can be used either as bitwise indices into a tech
 * mask (e.g. see MC_CMD_ETH_AN_FIELDS/TECH_MASK for example) or as regular
 * enums (e.g. see MC_CMD_LINK_CTRL_IN/ADVERTISED_TECH_ABILITIES_MASK). This
 * structure must be updated to add new technologies when projects that need
 * them arise. An incomplete list of possible expansion in the future include:
 * 100GBASE_KP4, 800GBASE_CR8, 800GBASE_KR8, 800GBASE_DR8, 800GBASE_SR8
 * 800GBASE_VR8
 */
#define       MC_CMD_ETH_TECH_TECH_OFST 0
#define       MC_CMD_ETH_TECH_TECH_LEN 16
/* enum: 1000BASE-KX - 1000BASE-X PCS/PMA over an electrical backplane PMD. See
 * IEEE 802.3 Clause 70
 */
#define          MC_CMD_ETH_TECH_1000BASEKX 0x0
/* enum: 10GBASE-R - PCS/PMA over an electrical backplane PMD. Refer to IEEE
 * 802.3 Clause 72
 */
#define          MC_CMD_ETH_TECH_10GBASE_KR 0x1
/* enum: 40GBASE-R PCS/PMA over an electrical backplane PMD. See IEEE 802.3
 * Clause 84.
 */
#define          MC_CMD_ETH_TECH_40GBASE_KR4 0x2
/* enum: 40GBASE-R PCS/PMA over 4 lane shielded copper balanced cable PMD. See
 * IEEE 802.3 Clause 85
 */
#define          MC_CMD_ETH_TECH_40GBASE_CR4 0x3
/* enum: 40GBASE-R PCS/PMA over 4 lane multimode fiber PMD as specified in
 * Clause 86
 */
#define          MC_CMD_ETH_TECH_40GBASE_SR4 0x4
/* enum: 40GBASE-R PCS/PMA over 4 WDM lane single mode fiber PMD with long
 * reach. See IEEE 802.3 Clause 87
 */
#define          MC_CMD_ETH_TECH_40GBASE_LR4 0x5
/* enum: 25GBASE-R PCS/PMA over shielded balanced copper cable PMD. See IEEE
 * 802.3 Clause 110
 */
#define          MC_CMD_ETH_TECH_25GBASE_CR 0x6
/* enum: 25GBASE-R PCS/PMA over an electrical backplane PMD. See IEEE 802.3
 * Clause 111
 */
#define          MC_CMD_ETH_TECH_25GBASE_KR 0x7
/* enum: 25GBASE-R PCS/PMA over multimode fiber PMD. Refer to IEEE 802.3 Clause
 * 112
 */
#define          MC_CMD_ETH_TECH_25GBASE_SR 0x8
/* enum: An Ethernet Physical layer operating at 50 Gb/s on twin-axial copper
 * cable. Refer to Ethernet Technology Consortium 25/50G Ethernet Spec.
 */
#define          MC_CMD_ETH_TECH_50GBASE_CR2 0x9
/* enum: An Ethernet Physical layer operating at 50 Gb/s on copper backplane.
 * Refer to Ethernet Technology Consortium 25/50G Ethernet Spec.
 */
#define          MC_CMD_ETH_TECH_50GBASE_KR2 0xa
/* enum: 100GBASE-R PCS/PMA over an electrical backplane PMD. See IEEE 802.3
 * Clause 93
 */
#define          MC_CMD_ETH_TECH_100GBASE_KR4 0xb
/* enum: 100GBASE-R PCS/PMA over 4 lane multimode fiber PMD. See IEEE 802.3
 * Clause 95
 */
#define          MC_CMD_ETH_TECH_100GBASE_SR4 0xc
/* enum: 100GBASE-R PCS/PMA over 4 lane shielded copper balanced cable PMD. See
 * IEEE 802.3 Clause 92
 */
#define          MC_CMD_ETH_TECH_100GBASE_CR4 0xd
/* enum: 100GBASE-R PCS/PMA over 4 WDM lane single mode fiber PMD, with
 * long/extended reach,. See IEEE 802.3 Clause 88
 */
#define          MC_CMD_ETH_TECH_100GBASE_LR4_ER4 0xe
/* enum: An Ethernet Physical layer operating at 50 Gb/s on short reach fiber.
 * Refer to Ethernet Technology Consortium 25/50G Ethernet Spec.
 */
#define          MC_CMD_ETH_TECH_50GBASE_SR2 0xf
/* enum: 1000BASEX PCS/PMA. See IEEE 802.3 Clause 36 over undefined PMD, duplex
 * mode unknown
 */
#define          MC_CMD_ETH_TECH_1000BASEX 0x10
/* enum: Non-standardised. 10G direct attach */
#define          MC_CMD_ETH_TECH_10GBASE_CR 0x11
/* enum: 10GBASE-SR fiber over 850nm optics. See IEEE 802.3 Clause 52 */
#define          MC_CMD_ETH_TECH_10GBASE_SR 0x12
/* enum: 10GBASE-LR fiber over 1310nm optics. See IEEE 802.3 Clause 52 */
#define          MC_CMD_ETH_TECH_10GBASE_LR 0x13
/* enum: 10GBASE-LRM fiber over 1310 nm optics. See IEEE 802.3 Clause 68 */
#define          MC_CMD_ETH_TECH_10GBASE_LRM 0x14
/* enum: 10GBASE-ER fiber over 1550nm optics. See IEEE 802.3 Clause 52 */
#define          MC_CMD_ETH_TECH_10GBASE_ER 0x15
/* enum: 50GBASE-R PCS/PMA over an electrical backplane PMD. See IEEE 802.3
 * Clause 137
 */
#define          MC_CMD_ETH_TECH_50GBASE_KR 0x16
/* enum: 50GBASE-SR PCS/PMA over multimode fiber PMD as specified in Clause 138
 */
#define          MC_CMD_ETH_TECH_50GBASE_SR 0x17
/* enum: 50GBASE-CR PCS/PMA over shielded copper balanced cable PMD. See IEEE
 * 802.3 Clause 136
 */
#define          MC_CMD_ETH_TECH_50GBASE_CR 0x18
/* enum: 50GBASE-R PCS/PMA over single mode fiber PMD as specified in Clause
 * 139.
 */
#define          MC_CMD_ETH_TECH_50GBASE_LR_ER_FR 0x19
/* enum: 100 Gb/s PHY using 100GBASE-R encoding over single-mode fiber with
 * reach up to at least 500 m (see IEEE 802.3 Clause 140)
 */
#define          MC_CMD_ETH_TECH_50GBASE_DR 0x1a
/* enum: 100GBASE-R PCS/PMA over an electrical backplane PMD. See IEEE 802.3
 * Clause 137
 */
#define          MC_CMD_ETH_TECH_100GBASE_KR2 0x1b
/* enum: 100GBASE-R PCS/PMA over 2 lane multimode fiber PMD. See IEEE 802.3
 * Clause 138
 */
#define          MC_CMD_ETH_TECH_100GBASE_SR2 0x1c
/* enum: 100GBASE-R PCS/PMA over 2 lane shielded copper balanced cable PMD. See
 * IEEE 802.3 Clause 136
 */
#define          MC_CMD_ETH_TECH_100GBASE_CR2 0x1d
/* enum: Unknown source */
#define          MC_CMD_ETH_TECH_100GBASE_LR2_ER2_FR2 0x1e
/* enum: Unknown source */
#define          MC_CMD_ETH_TECH_100GBASE_DR2 0x1f
/* enum: 200GBASE-R PCS/PMA over an electrical backplane PMD. See IEEE 802.3
 * Clause 137
 */
#define          MC_CMD_ETH_TECH_200GBASE_KR4 0x20
/* enum: 200GBASE-R PCS/PMA over 4 lane multimode fiber PMD. See IEEE 802.3
 * Clause 138
 */
#define          MC_CMD_ETH_TECH_200GBASE_SR4 0x21
/* enum: 200GBASE-R PCS/PMA over 4 WDM lane single-mode fiber PMD as specified
 * in Clause 122
 */
#define          MC_CMD_ETH_TECH_200GBASE_LR4_ER4_FR4 0x22
/* enum: 200GBASE-R PCS/PMA over 4-lane single-mode fiber PMD. See IEEE 802.3
 * Clause 121
 */
#define          MC_CMD_ETH_TECH_200GBASE_DR4 0x23
/* enum: 200GBASE-R PCS/PMA over 4 lane shielded copper balanced cable PMD as
 * specified in Clause 136
 */
#define          MC_CMD_ETH_TECH_200GBASE_CR4 0x24
/* enum: Ethernet Technology Consortium 400G AN Spec. 400GBASE-KR8 PMD uses
 * 802.3 Clause 137, but the number PMD lanes is 8.
 */
#define          MC_CMD_ETH_TECH_400GBASE_KR8 0x25
/* enum: 400GBASE-R PCS/PMA over 8-lane multimode fiber PMD. See IEEE 802.3
 * Clause 138
 */
#define          MC_CMD_ETH_TECH_400GBASE_SR8 0x26
/* enum: 400GBASE-R PCS/PMA over 8 WDM lane single-mode fiber PMD. See IEEE
 * 802.3 Clause 122
 */
#define          MC_CMD_ETH_TECH_400GBASE_LR8_ER8_FR8 0x27
/* enum: Unknown source */
#define          MC_CMD_ETH_TECH_400GBASE_DR8 0x28
/* enum: Ethernet Technology Consortium 400G AN Spec. 400GBASE-CR8 PMD uses
 * IEEE 802.3 Clause 136, but the number PMD lanes is 8.
 */
#define          MC_CMD_ETH_TECH_400GBASE_CR8 0x29
/* enum: 100GBASE-R PCS/PMA over an electrical backplane PMD. See IEEE 802.3ck
 * Clause 163.
 */
#define          MC_CMD_ETH_TECH_100GBASE_KR 0x2a
/* enum: IEEE 802.3ck. 100G PHY with PMD as specified in Clause 167 over short
 * reach fiber
 */
#define          MC_CMD_ETH_TECH_100GBASE_SR 0x2b
/* enum: 100G PMD together with single-mode fiber medium. See IEEE 802.3 Clause
 * 140
 */
#define          MC_CMD_ETH_TECH_100GBASE_LR_ER_FR 0x2c
/* enum: 100GBASE-R PCS/PMA over shielded balanced copper cable PMD. See IEEE
 * 802.3 in Clause 162 IEEE 802.3ck.
 */
#define          MC_CMD_ETH_TECH_100GBASE_CR 0x2d
/* enum: 100G PMD together with single-mode fiber medium. See IEEE 802.3 Clause
 * 140
 */
#define          MC_CMD_ETH_TECH_100GBASE_DR 0x2e
/* enum: 200GBASE-R PCS/PMA over an electrical backplane PMD as specified in
 * Clause 163 IEEE 802.3ck
 */
#define          MC_CMD_ETH_TECH_200GBASE_KR2 0x2f
/* enum: 200G PHY with PMD as specified in Clause 167 over short reach fiber
 * IEEE 802.3ck
 */
#define          MC_CMD_ETH_TECH_200GBASE_SR2 0x30
/* enum: Unknown source */
#define          MC_CMD_ETH_TECH_200GBASE_LR2_ER2_FR2 0x31
/* enum: Unknown source */
#define          MC_CMD_ETH_TECH_200GBASE_DR2 0x32
/* enum: 200GBASE-R PCS/PMA over 2 lane shielded balanced copper cable PMD as
 * specified in Clause 162 IEEE 802.3ck.
 */
#define          MC_CMD_ETH_TECH_200GBASE_CR2 0x33
/* enum: 400GBASE-R PCS/PMA over an electrical backplane PMD. See IEEE 802.3
 * Clause 163 IEEE 802.3ck.
 */
#define          MC_CMD_ETH_TECH_400GBASE_KR4 0x34
/* enum: 400G PHY with PMD over short reach fiber. See Clause 167 of IEEE
 * 802.3ck.
 */
#define          MC_CMD_ETH_TECH_400GBASE_SR4 0x35
/* enum: 400GBASE-R PCS/PMA over 4 WDM lane single-mode fiber PMD. See IEEE
 * 802.3 Clause 151
 */
#define          MC_CMD_ETH_TECH_400GBASE_LR4_ER4_FR4 0x36
/* enum: 400GBASE-R PCS/PMA over 4-lane single-mode fiber PMD as specified in
 * Clause 124
 */
#define          MC_CMD_ETH_TECH_400GBASE_DR4 0x37
/* enum: 400GBASE-R PCS/PMA over 4 lane shielded balanced copper cable PMD as
 * specified in Clause 162 of IEEE 802.3ck.
 */
#define          MC_CMD_ETH_TECH_400GBASE_CR4 0x38
/* enum: Automatic tech mode. The driver must not use this. */
#define          MC_CMD_ETH_TECH_AUTO 0x39
/* enum: See IEEE 802.3cc-2017 Clause 114 */
#define          MC_CMD_ETH_TECH_25GBASE_LR_ER 0x3a
/* enum: Up to 7 m over twinaxial copper cable assembly (10 lanes, 10 Gbit/s
 * each) See IEEE 802.3ba-2010 Clause 85
 */
#define          MC_CMD_ETH_TECH_100GBASE_CR10 0x3b
/* enum: Invalid tech mode. The driver must not use this. */
#define          MC_CMD_ETH_TECH_NONE 0x7f
#define       MC_CMD_ETH_TECH_TECH_LBN 0
#define       MC_CMD_ETH_TECH_TECH_WIDTH 128

/* MC_CMD_LINK_STATUS_FLAGS structuredef */
#define    MC_CMD_LINK_STATUS_FLAGS_LEN 8
/* Flags used to report the current configuration/state of the link. */
#define       MC_CMD_LINK_STATUS_FLAGS_STATUS_FLAGS_OFST 0
#define       MC_CMD_LINK_STATUS_FLAGS_STATUS_FLAGS_LEN 8
#define       MC_CMD_LINK_STATUS_FLAGS_STATUS_FLAGS_LO_OFST 0
#define       MC_CMD_LINK_STATUS_FLAGS_STATUS_FLAGS_LO_LEN 4
#define       MC_CMD_LINK_STATUS_FLAGS_STATUS_FLAGS_LO_LBN 0
#define       MC_CMD_LINK_STATUS_FLAGS_STATUS_FLAGS_LO_WIDTH 32
#define       MC_CMD_LINK_STATUS_FLAGS_STATUS_FLAGS_HI_OFST 4
#define       MC_CMD_LINK_STATUS_FLAGS_STATUS_FLAGS_HI_LEN 4
#define       MC_CMD_LINK_STATUS_FLAGS_STATUS_FLAGS_HI_LBN 32
#define       MC_CMD_LINK_STATUS_FLAGS_STATUS_FLAGS_HI_WIDTH 32
/* enum property: bitshift */
/* enum: Whether we have overall link up */
#define          MC_CMD_LINK_STATUS_FLAGS_LINK_UP 0x0
/* enum: If set, the PHY has no external RX link synchronisation */
#define          MC_CMD_LINK_STATUS_FLAGS_NO_PHY_LINK 0x1
/* enum: If set, PMD/MDI is not connected (e.g. cable disconnected, module cage
 * empty)
 */
#define          MC_CMD_LINK_STATUS_FLAGS_PMD_MDI_DISCONNECTED 0x2
/* enum: Set on error while decoding module data (e.g. module EEPROM does not
 * contain valid values, has checksum errors, etc.)
 */
#define          MC_CMD_LINK_STATUS_FLAGS_PMD_BAD 0x3
/* enum: Set when module unsupported (e.g. unsupported link rate or link
 * technology)
 */
#define          MC_CMD_LINK_STATUS_FLAGS_PMD_UNSUPPORTED 0x4
/* enum: Set on error while communicating with the module (e.g. I2C errors
 * while reading EEPROM)
 */
#define          MC_CMD_LINK_STATUS_FLAGS_PMD_COMMS_FAULT 0x5
/* enum: Set on module overcurrent/overvoltage condition */
#define          MC_CMD_LINK_STATUS_FLAGS_PMD_POWER_FAULT 0x6
/* enum: Set on module overtemperature condition */
#define          MC_CMD_LINK_STATUS_FLAGS_PMD_THERMAL_FAULT 0x7
/* enum: If set, the module is indicating Loss of Signal */
#define          MC_CMD_LINK_STATUS_FLAGS_PMD_LOS 0x8
/* enum: If set, PMA is indicating loss of CDR lock (clock sync) */
#define          MC_CMD_LINK_STATUS_FLAGS_PMA_NO_CDR_LOCK 0x9
/* enum: If set, PMA is indicating loss of analog signal */
#define          MC_CMD_LINK_STATUS_FLAGS_PMA_LOS 0xa
/* enum: If set, PCS is indicating loss of block lock */
#define          MC_CMD_LINK_STATUS_FLAGS_PCS_NO_BLOCK_LOCK 0xb
/* enum: If set, PCS is indicating loss of alignment marker lock on one or more
 * lanes
 */
#define          MC_CMD_LINK_STATUS_FLAGS_PCS_NO_AM_LOCK 0xc
/* enum: If set, PCS is indicating loss of overall alignment lock */
#define          MC_CMD_LINK_STATUS_FLAGS_PCS_NO_ALIGN_LOCK 0xd
/* enum: If set, PCS is indicating high bit error rate condition. */
#define          MC_CMD_LINK_STATUS_FLAGS_PCS_HI_BER 0xe
/* enum: If set, FEC is indicating loss of FEC lock */
#define          MC_CMD_LINK_STATUS_FLAGS_FEC_NO_LOCK 0xf
/* enum: If set, indicates that the number of symbol errors in a 8192-codeword
 * window has exceeded the threshold K (417).
 */
#define          MC_CMD_LINK_STATUS_FLAGS_FEC_HI_SER 0x10
/* enum: If set, the receiver has detected the local FEC has degraded. */
#define          MC_CMD_LINK_STATUS_FLAGS_FEC_LOCAL_DEGRADED 0x11
/* enum: If set, the receiver has detected the remote FEC has degraded. */
#define          MC_CMD_LINK_STATUS_FLAGS_FEC_RM_DEGRADED 0x12
/* enum: If set, the number of symbol errors is over an internal threshold. */
#define          MC_CMD_LINK_STATUS_FLAGS_FEC_DEGRADED_SER 0x13
/* enum: If set, autonegotiation has detected an auto-negotiation capable link
 * partner
 */
#define          MC_CMD_LINK_STATUS_FLAGS_AN_ABLE 0x14
/* enum: If set, autonegotiation base page exchange has failed */
#define          MC_CMD_LINK_STATUS_FLAGS_AN_BP_FAILED 0x15
/* enum: If set, autonegotiation next page exchange has failed */
#define          MC_CMD_LINK_STATUS_FLAGS_AN_NP_FAILED 0x16
/* enum: If set, autonegotiation has failed to negotiate a common set of
 * capabilities
 */
#define          MC_CMD_LINK_STATUS_FLAGS_AN_NO_HCD 0x17
/* enum: If set, local end link training has failed to establish link training
 * frame lock on one or more lanes
 */
#define          MC_CMD_LINK_STATUS_FLAGS_LT_NO_LOCAL_FRAME_LOCK 0x18
/* enum: If set, remote end link training has failed to establish link training
 * frame lock on one or more lanes
 */
#define          MC_CMD_LINK_STATUS_FLAGS_LT_NO_RM_FRAME_LOCK 0x19
/* enum: If set, remote end has failed to assert Receiver Ready (link training
 * success) within the designated timeout
 */
#define          MC_CMD_LINK_STATUS_FLAGS_LT_NO_RX_READY 0x1a
#define       MC_CMD_LINK_STATUS_FLAGS_STATUS_FLAGS_LBN 0
#define       MC_CMD_LINK_STATUS_FLAGS_STATUS_FLAGS_WIDTH 64

/* MC_CMD_PAUSE_MODE structuredef */
#define    MC_CMD_PAUSE_MODE_LEN 1
#define       MC_CMD_PAUSE_MODE_TYPE_OFST 0
#define       MC_CMD_PAUSE_MODE_TYPE_LEN 1
/* enum: See IEEE 802.3 Clause 73.6.6 */
#define          MC_CMD_PAUSE_MODE_AN_PAUSE 0x0
/* enum: See IEEE 802.3 Clause 73.6.6 */
#define          MC_CMD_PAUSE_MODE_AN_ASYM_DIR 0x1
#define       MC_CMD_PAUSE_MODE_TYPE_LBN 0
#define       MC_CMD_PAUSE_MODE_TYPE_WIDTH 8

/* MC_CMD_ETH_AN_FIELDS structuredef: Fields used for IEEE 802.3 Clause 73
 * Auto-Negotiation. Warning - This is fixed size and cannot be extended. This
 * structure is used to define autonegotiable abilities (advertised, link
 * partner and supported abilities).
 */
#define    MC_CMD_ETH_AN_FIELDS_LEN 25
/* Mask of Ethernet technologies. The bit indices in this mask are taken from
 * the TECH field in the MC_CMD_ETH_TECH structure.
 */
#define       MC_CMD_ETH_AN_FIELDS_TECH_MASK_OFST 0
#define       MC_CMD_ETH_AN_FIELDS_TECH_MASK_LEN 16
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_ETH_TECH/TECH */
#define       MC_CMD_ETH_AN_FIELDS_TECH_MASK_LBN 0
#define       MC_CMD_ETH_AN_FIELDS_TECH_MASK_WIDTH 128
/* Mask of supported FEC modes */
#define       MC_CMD_ETH_AN_FIELDS_FEC_MASK_OFST 16
#define       MC_CMD_ETH_AN_FIELDS_FEC_MASK_LEN 4
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               FEC_TYPE/TYPE */
#define       MC_CMD_ETH_AN_FIELDS_FEC_MASK_LBN 128
#define       MC_CMD_ETH_AN_FIELDS_FEC_MASK_WIDTH 32
/* Mask of requested FEC modes */
#define       MC_CMD_ETH_AN_FIELDS_FEC_REQ_OFST 20
#define       MC_CMD_ETH_AN_FIELDS_FEC_REQ_LEN 4
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               FEC_TYPE/TYPE */
#define       MC_CMD_ETH_AN_FIELDS_FEC_REQ_LBN 160
#define       MC_CMD_ETH_AN_FIELDS_FEC_REQ_WIDTH 32
/* Bitmask of negotiated pause modes */
#define       MC_CMD_ETH_AN_FIELDS_PAUSE_MASK_OFST 24
#define       MC_CMD_ETH_AN_FIELDS_PAUSE_MASK_LEN 1
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_PAUSE_MODE/TYPE */
#define       MC_CMD_ETH_AN_FIELDS_PAUSE_MASK_LBN 192
#define       MC_CMD_ETH_AN_FIELDS_PAUSE_MASK_WIDTH 8

/* MC_CMD_LOOPBACK_V2 structuredef: Loopback modes for use with the new
 * MC_CMD_LINK_CTRL and MC_CMD_LINK_STATE. These loopback modes are not
 * supported in other getlink/setlink commands.
 */
#define    MC_CMD_LOOPBACK_V2_LEN 4
#define       MC_CMD_LOOPBACK_V2_MODE_OFST 0
#define       MC_CMD_LOOPBACK_V2_MODE_LEN 4
/* enum: No loopback */
#define          MC_CMD_LOOPBACK_V2_NONE 0x0
/* enum: Let firmware choose a supported loopback mode */
#define          MC_CMD_LOOPBACK_V2_AUTO 0x1
/* enum: Loopback after the MAC */
#define          MC_CMD_LOOPBACK_V2_POST_MAC 0x2
/* enum: Loopback after the PCS */
#define          MC_CMD_LOOPBACK_V2_POST_PCS 0x3
/* enum: Loopback after the PMA */
#define          MC_CMD_LOOPBACK_V2_POST_PMA 0x4
/* enum: Loopback after the MDI Wireside */
#define          MC_CMD_LOOPBACK_V2_POST_MDI_WS 0x5
/* enum: Loopback after the PMA Wireside */
#define          MC_CMD_LOOPBACK_V2_POST_PMA_WS 0x6
/* enum: Loopback after the PCS Wireside */
#define          MC_CMD_LOOPBACK_V2_POST_PCS_WS 0x7
/* enum: Loopback after the MAC Wireside */
#define          MC_CMD_LOOPBACK_V2_POST_MAC_WS 0x8
/* enum: Loopback after the MAC FIFOs (before the MAC) */
#define          MC_CMD_LOOPBACK_V2_PRE_MAC 0x9
#define       MC_CMD_LOOPBACK_V2_MODE_LBN 0
#define       MC_CMD_LOOPBACK_V2_MODE_WIDTH 32

/* MC_CMD_FCNTL structuredef */
#define    MC_CMD_FCNTL_LEN 4
#define       MC_CMD_FCNTL_MASK_OFST 0
#define       MC_CMD_FCNTL_MASK_LEN 4
/* enum: Flow control is off. */
#define          MC_CMD_FCNTL_OFF 0x0
/* enum: Respond to flow control. */
#define          MC_CMD_FCNTL_RESPOND 0x1
/* enum: Respond to and Issue flow control. */
#define          MC_CMD_FCNTL_BIDIR 0x2
/* enum: Auto negotiate flow control. */
#define          MC_CMD_FCNTL_AUTO 0x3
/* enum: Priority flow control. This is only supported on KSB. */
#define          MC_CMD_FCNTL_QBB 0x4
/* enum: Issue flow control. */
#define          MC_CMD_FCNTL_GENERATE 0x5
#define       MC_CMD_FCNTL_MASK_LBN 0
#define       MC_CMD_FCNTL_MASK_WIDTH 32

/* MC_CMD_LINK_FLAGS structuredef */
#define    MC_CMD_LINK_FLAGS_LEN 4
/* The enums defined in this field are used as indices into the
 * MC_CMD_LINK_FLAGS bitmask.
 */
#define       MC_CMD_LINK_FLAGS_MASK_OFST 0
#define       MC_CMD_LINK_FLAGS_MASK_LEN 4
/* enum property: bitshift */
/* enum: Enable auto-negotiation. If AN is enabled, link technology and FEC
 * mode are determined by advertised capabilities and requested FEC modes,
 * combined with link partner capabilities. If AN is disabled, link technology
 * is forced to LINK_TECHNOLOGY and FEC mode is forced to FEC_MODE. Not valid
 * if loopback is enabled
 */
#define          MC_CMD_LINK_FLAGS_AUTONEG_EN 0x0
/* enum: Enable parallel detect. In addition to AN, try to sense partner forced
 * speed/FEC mode (when partner AN disabled). Only valid if AN is enabled.
 */
#define          MC_CMD_LINK_FLAGS_PARALLEL_DETECT_EN 0x1
/* enum: Force link down, in electrical idle. */
#define          MC_CMD_LINK_FLAGS_LINK_DISABLE 0x2
/* enum: Ignore the sequence number and always apply. */
#define          MC_CMD_LINK_FLAGS_IGNORE_MODULE_SEQ 0x3
#define       MC_CMD_LINK_FLAGS_MASK_LBN 0
#define       MC_CMD_LINK_FLAGS_MASK_WIDTH 32


/***********************************/
/* MC_CMD_LINK_CTRL
 * Write the unified MAC/PHY link configuration. Locks required: None. Return
 * code: 0, EINVAL, ETIME, EAGAIN
 */
#define MC_CMD_LINK_CTRL 0x6b
#undef MC_CMD_0x6b_PRIVILEGE_CTG

#define MC_CMD_0x6b_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_LINK_CTRL_IN msgrequest */
#define    MC_CMD_LINK_CTRL_IN_LEN 40
/* Handle to the port to set link state for. */
#define       MC_CMD_LINK_CTRL_IN_PORT_HANDLE_OFST 0
#define       MC_CMD_LINK_CTRL_IN_PORT_HANDLE_LEN 4
/* Control flags */
#define       MC_CMD_LINK_CTRL_IN_CONTROL_FLAGS_OFST 4
#define       MC_CMD_LINK_CTRL_IN_CONTROL_FLAGS_LEN 4
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_LINK_FLAGS/MASK */
/* Reserved for future expansion, and included to provide padding for alignment
 * purposes.
 */
#define       MC_CMD_LINK_CTRL_IN_RESERVED_OFST 8
#define       MC_CMD_LINK_CTRL_IN_RESERVED_LEN 8
#define       MC_CMD_LINK_CTRL_IN_RESERVED_LO_OFST 8
#define       MC_CMD_LINK_CTRL_IN_RESERVED_LO_LEN 4
#define       MC_CMD_LINK_CTRL_IN_RESERVED_LO_LBN 64
#define       MC_CMD_LINK_CTRL_IN_RESERVED_LO_WIDTH 32
#define       MC_CMD_LINK_CTRL_IN_RESERVED_HI_OFST 12
#define       MC_CMD_LINK_CTRL_IN_RESERVED_HI_LEN 4
#define       MC_CMD_LINK_CTRL_IN_RESERVED_HI_LBN 96
#define       MC_CMD_LINK_CTRL_IN_RESERVED_HI_WIDTH 32
/* Technology abilities to advertise during auto-negotiation */
#define       MC_CMD_LINK_CTRL_IN_ADVERTISED_TECH_ABILITIES_MASK_OFST 16
#define       MC_CMD_LINK_CTRL_IN_ADVERTISED_TECH_ABILITIES_MASK_LEN 16
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_ETH_TECH/TECH */
/* Pause abilities to advertise during auto-negotiation. Valid when auto-
 * negotation is enabled and MC_CMD_SET_MAC_IN/FCTL is set to
 * MC_CMD_FCNTL_AUTO. If auto-negotiation is disabled the driver must
 * explicitly configure pause mode with MC_CMD_SET_MAC.
 */
#define       MC_CMD_LINK_CTRL_IN_ADVERTISED_PAUSE_ABILITIES_MASK_OFST 32
#define       MC_CMD_LINK_CTRL_IN_ADVERTISED_PAUSE_ABILITIES_MASK_LEN 1
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_PAUSE_MODE/TYPE */
/* When auto-negotiation is enabled, this is the FEC mode to request. Note that
 * a weaker FEC mode may get negotiated, depending on what the link partner
 * supports. The driver should subsequently use MC_CMD_GET_LINK to check the
 * actual negotiated FEC mode. When auto-negotiation is disabled, this is the
 * forced FEC mode.
 */
#define       MC_CMD_LINK_CTRL_IN_FEC_MODE_OFST 33
#define       MC_CMD_LINK_CTRL_IN_FEC_MODE_LEN 1
/* enum property: value */
/*            Enum values, see field(s): */
/*               FEC_TYPE/TYPE */
/* This is only to be used when auto-negotiation is disabled (forced speed or
 * loopback mode). If the specified value does not align with the values
 * defined in the enum MC_CMD_ETH_TECH/TECH, it is considered invalid.
 */
#define       MC_CMD_LINK_CTRL_IN_LINK_TECHNOLOGY_OFST 36
#define       MC_CMD_LINK_CTRL_IN_LINK_TECHNOLOGY_LEN 2
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_ETH_TECH/TECH */
/* The sequence number of the last MODULECHANGE event. If this doesn't match,
 * fail with EAGAIN.
 */
#define       MC_CMD_LINK_CTRL_IN_MODULE_SEQ_OFST 38
#define       MC_CMD_LINK_CTRL_IN_MODULE_SEQ_LEN 1
/* Loopback Mode. Only valid when auto-negotiation is disabled. */
#define       MC_CMD_LINK_CTRL_IN_LOOPBACK_OFST 39
#define       MC_CMD_LINK_CTRL_IN_LOOPBACK_LEN 1
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_LOOPBACK_V2/MODE */

/* MC_CMD_LINK_CTRL_OUT msgresponse */
#define    MC_CMD_LINK_CTRL_OUT_LEN 0


/***********************************/
/* MC_CMD_LINK_STATE
 */
#define MC_CMD_LINK_STATE 0x6c
#undef MC_CMD_0x6c_PRIVILEGE_CTG

#define MC_CMD_0x6c_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_LINK_STATE_IN msgrequest */
#define    MC_CMD_LINK_STATE_IN_LEN 4
/* Handle to the port to get link state for. */
#define       MC_CMD_LINK_STATE_IN_PORT_HANDLE_OFST 0
#define       MC_CMD_LINK_STATE_IN_PORT_HANDLE_LEN 4

/* MC_CMD_LINK_STATE_OUT msgresponse */
#define    MC_CMD_LINK_STATE_OUT_LEN 114
/* Flags used to report the current configuration/state of the link. */
#define       MC_CMD_LINK_STATE_OUT_STATUS_FLAGS_OFST 0
#define       MC_CMD_LINK_STATE_OUT_STATUS_FLAGS_LEN 8
#define       MC_CMD_LINK_STATE_OUT_STATUS_FLAGS_LO_OFST 0
#define       MC_CMD_LINK_STATE_OUT_STATUS_FLAGS_LO_LEN 4
#define       MC_CMD_LINK_STATE_OUT_STATUS_FLAGS_LO_LBN 0
#define       MC_CMD_LINK_STATE_OUT_STATUS_FLAGS_LO_WIDTH 32
#define       MC_CMD_LINK_STATE_OUT_STATUS_FLAGS_HI_OFST 4
#define       MC_CMD_LINK_STATE_OUT_STATUS_FLAGS_HI_LEN 4
#define       MC_CMD_LINK_STATE_OUT_STATUS_FLAGS_HI_LBN 32
#define       MC_CMD_LINK_STATE_OUT_STATUS_FLAGS_HI_WIDTH 32
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_LINK_STATUS_FLAGS/STATUS_FLAGS */
/* Configured technology. If the specified value does not align with the values
 * defined in the enum MC_CMD_ETH_TECH/TECH, it is considered invalid.
 */
#define       MC_CMD_LINK_STATE_OUT_LINK_TECHNOLOGY_OFST 8
#define       MC_CMD_LINK_STATE_OUT_LINK_TECHNOLOGY_LEN 2
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_ETH_TECH/TECH */
/* Configured FEC mode */
#define       MC_CMD_LINK_STATE_OUT_FEC_MODE_OFST 10
#define       MC_CMD_LINK_STATE_OUT_FEC_MODE_LEN 1
/* enum property: value */
/*            Enum values, see field(s): */
/*               FEC_TYPE/TYPE */
/* Bitmask of auto-negotiated pause modes */
#define       MC_CMD_LINK_STATE_OUT_PAUSE_MASK_OFST 11
#define       MC_CMD_LINK_STATE_OUT_PAUSE_MASK_LEN 1
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_PAUSE_MODE/TYPE */
/* Configured loopback mode */
#define       MC_CMD_LINK_STATE_OUT_LOOPBACK_OFST 12
#define       MC_CMD_LINK_STATE_OUT_LOOPBACK_LEN 1
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_LOOPBACK_V2/MODE */
/* Abilities requested by the driver to advertise during auto-negotiation */
#define       MC_CMD_LINK_STATE_OUT_ADVERTISED_ABILITIES_OFST 16
#define       MC_CMD_LINK_STATE_OUT_ADVERTISED_ABILITIES_LEN 32
/* See structuredef: MC_CMD_ETH_AN_FIELDS */
#define       MC_CMD_LINK_STATE_OUT_ADVERTISED_ABILITIES_TECH_MASK_OFST 16
#define       MC_CMD_LINK_STATE_OUT_ADVERTISED_ABILITIES_TECH_MASK_LEN 16
#define       MC_CMD_LINK_STATE_OUT_ADVERTISED_ABILITIES_FEC_MASK_OFST 32
#define       MC_CMD_LINK_STATE_OUT_ADVERTISED_ABILITIES_FEC_MASK_LEN 4
#define       MC_CMD_LINK_STATE_OUT_ADVERTISED_ABILITIES_FEC_REQ_OFST 36
#define       MC_CMD_LINK_STATE_OUT_ADVERTISED_ABILITIES_FEC_REQ_LEN 4
#define       MC_CMD_LINK_STATE_OUT_ADVERTISED_ABILITIES_PAUSE_MASK_OFST 40
#define       MC_CMD_LINK_STATE_OUT_ADVERTISED_ABILITIES_PAUSE_MASK_LEN 1
/* Abilities advertised by the link partner during auto-negotiation */
#define       MC_CMD_LINK_STATE_OUT_LINK_PARTNER_ABILITIES_OFST 48
#define       MC_CMD_LINK_STATE_OUT_LINK_PARTNER_ABILITIES_LEN 32
/* See structuredef: MC_CMD_ETH_AN_FIELDS */
#define       MC_CMD_LINK_STATE_OUT_LINK_PARTNER_ABILITIES_TECH_MASK_OFST 48
#define       MC_CMD_LINK_STATE_OUT_LINK_PARTNER_ABILITIES_TECH_MASK_LEN 16
#define       MC_CMD_LINK_STATE_OUT_LINK_PARTNER_ABILITIES_FEC_MASK_OFST 64
#define       MC_CMD_LINK_STATE_OUT_LINK_PARTNER_ABILITIES_FEC_MASK_LEN 4
#define       MC_CMD_LINK_STATE_OUT_LINK_PARTNER_ABILITIES_FEC_REQ_OFST 68
#define       MC_CMD_LINK_STATE_OUT_LINK_PARTNER_ABILITIES_FEC_REQ_LEN 4
#define       MC_CMD_LINK_STATE_OUT_LINK_PARTNER_ABILITIES_PAUSE_MASK_OFST 72
#define       MC_CMD_LINK_STATE_OUT_LINK_PARTNER_ABILITIES_PAUSE_MASK_LEN 1
/* Abilities supported by the local device (including cable abilities) For
 * fixed local device capbilities see MC_CMD_GET_LOCAL_DEVICE_INFO
 */
#define       MC_CMD_LINK_STATE_OUT_SUPPORTED_ABILITIES_OFST 80
#define       MC_CMD_LINK_STATE_OUT_SUPPORTED_ABILITIES_LEN 28
/* See structuredef: MC_CMD_ETH_AN_FIELDS */
#define       MC_CMD_LINK_STATE_OUT_SUPPORTED_ABILITIES_TECH_MASK_OFST 80
#define       MC_CMD_LINK_STATE_OUT_SUPPORTED_ABILITIES_TECH_MASK_LEN 16
#define       MC_CMD_LINK_STATE_OUT_SUPPORTED_ABILITIES_FEC_MASK_OFST 96
#define       MC_CMD_LINK_STATE_OUT_SUPPORTED_ABILITIES_FEC_MASK_LEN 4
#define       MC_CMD_LINK_STATE_OUT_SUPPORTED_ABILITIES_FEC_REQ_OFST 100
#define       MC_CMD_LINK_STATE_OUT_SUPPORTED_ABILITIES_FEC_REQ_LEN 4
#define       MC_CMD_LINK_STATE_OUT_SUPPORTED_ABILITIES_PAUSE_MASK_OFST 104
#define       MC_CMD_LINK_STATE_OUT_SUPPORTED_ABILITIES_PAUSE_MASK_LEN 1
/* Control flags */
#define       MC_CMD_LINK_STATE_OUT_CONTROL_FLAGS_OFST 108
#define       MC_CMD_LINK_STATE_OUT_CONTROL_FLAGS_LEN 4
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_LINK_FLAGS/MASK */
/* Sequence number to synchronize link change events */
#define       MC_CMD_LINK_STATE_OUT_PORT_LINKCHANGE_SEQ_NUM_OFST 112
#define       MC_CMD_LINK_STATE_OUT_PORT_LINKCHANGE_SEQ_NUM_LEN 1
/* Sequence number to synchronize module change events */
#define       MC_CMD_LINK_STATE_OUT_PORT_MODULECHANGE_SEQ_NUM_OFST 113
#define       MC_CMD_LINK_STATE_OUT_PORT_MODULECHANGE_SEQ_NUM_LEN 1

/* MC_CMD_LINK_STATE_OUT_V2 msgresponse: Updated LINK_STATE_OUT with
 * LOCAL_AN_SUPPORT
 */
#define    MC_CMD_LINK_STATE_OUT_V2_LEN 120
/* Flags used to report the current configuration/state of the link. */
#define       MC_CMD_LINK_STATE_OUT_V2_STATUS_FLAGS_OFST 0
#define       MC_CMD_LINK_STATE_OUT_V2_STATUS_FLAGS_LEN 8
#define       MC_CMD_LINK_STATE_OUT_V2_STATUS_FLAGS_LO_OFST 0
#define       MC_CMD_LINK_STATE_OUT_V2_STATUS_FLAGS_LO_LEN 4
#define       MC_CMD_LINK_STATE_OUT_V2_STATUS_FLAGS_LO_LBN 0
#define       MC_CMD_LINK_STATE_OUT_V2_STATUS_FLAGS_LO_WIDTH 32
#define       MC_CMD_LINK_STATE_OUT_V2_STATUS_FLAGS_HI_OFST 4
#define       MC_CMD_LINK_STATE_OUT_V2_STATUS_FLAGS_HI_LEN 4
#define       MC_CMD_LINK_STATE_OUT_V2_STATUS_FLAGS_HI_LBN 32
#define       MC_CMD_LINK_STATE_OUT_V2_STATUS_FLAGS_HI_WIDTH 32
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_LINK_STATUS_FLAGS/STATUS_FLAGS */
/* Configured technology. If the specified value does not align with the values
 * defined in the enum MC_CMD_ETH_TECH/TECH, it is considered invalid.
 */
#define       MC_CMD_LINK_STATE_OUT_V2_LINK_TECHNOLOGY_OFST 8
#define       MC_CMD_LINK_STATE_OUT_V2_LINK_TECHNOLOGY_LEN 2
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_ETH_TECH/TECH */
/* Configured FEC mode */
#define       MC_CMD_LINK_STATE_OUT_V2_FEC_MODE_OFST 10
#define       MC_CMD_LINK_STATE_OUT_V2_FEC_MODE_LEN 1
/* enum property: value */
/*            Enum values, see field(s): */
/*               FEC_TYPE/TYPE */
/* Bitmask of auto-negotiated pause modes */
#define       MC_CMD_LINK_STATE_OUT_V2_PAUSE_MASK_OFST 11
#define       MC_CMD_LINK_STATE_OUT_V2_PAUSE_MASK_LEN 1
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_PAUSE_MODE/TYPE */
/* Configured loopback mode */
#define       MC_CMD_LINK_STATE_OUT_V2_LOOPBACK_OFST 12
#define       MC_CMD_LINK_STATE_OUT_V2_LOOPBACK_LEN 1
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_LOOPBACK_V2/MODE */
/* Abilities requested by the driver to advertise during auto-negotiation */
#define       MC_CMD_LINK_STATE_OUT_V2_ADVERTISED_ABILITIES_OFST 16
#define       MC_CMD_LINK_STATE_OUT_V2_ADVERTISED_ABILITIES_LEN 32
/* Abilities advertised by the link partner during auto-negotiation */
#define       MC_CMD_LINK_STATE_OUT_V2_LINK_PARTNER_ABILITIES_OFST 48
#define       MC_CMD_LINK_STATE_OUT_V2_LINK_PARTNER_ABILITIES_LEN 32
/* Abilities supported by the local device (including cable abilities) For
 * fixed local device capbilities see MC_CMD_GET_LOCAL_DEVICE_INFO
 */
#define       MC_CMD_LINK_STATE_OUT_V2_SUPPORTED_ABILITIES_OFST 80
#define       MC_CMD_LINK_STATE_OUT_V2_SUPPORTED_ABILITIES_LEN 28
/* Control flags */
#define       MC_CMD_LINK_STATE_OUT_V2_CONTROL_FLAGS_OFST 108
#define       MC_CMD_LINK_STATE_OUT_V2_CONTROL_FLAGS_LEN 4
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_LINK_FLAGS/MASK */
/* Sequence number to synchronize link change events */
#define       MC_CMD_LINK_STATE_OUT_V2_PORT_LINKCHANGE_SEQ_NUM_OFST 112
#define       MC_CMD_LINK_STATE_OUT_V2_PORT_LINKCHANGE_SEQ_NUM_LEN 1
/* Sequence number to synchronize module change events */
#define       MC_CMD_LINK_STATE_OUT_V2_PORT_MODULECHANGE_SEQ_NUM_OFST 113
#define       MC_CMD_LINK_STATE_OUT_V2_PORT_MODULECHANGE_SEQ_NUM_LEN 1
/* Reports the auto-negotiation supported by the local device. This depends on
 * the port and module properties.
 */
#define       MC_CMD_LINK_STATE_OUT_V2_LOCAL_AN_SUPPORT_OFST 116
#define       MC_CMD_LINK_STATE_OUT_V2_LOCAL_AN_SUPPORT_LEN 4
/*            Enum values, see field(s): */
/*               AN_TYPE/TYPE */

/* MC_CMD_LINK_STATE_OUT_V3 msgresponse: Updated LINK_STATE_OUT_V2 for explicit
 * reporting of the link speed and duplex mode.
 */
#define    MC_CMD_LINK_STATE_OUT_V3_LEN 128
/* Flags used to report the current configuration/state of the link. */
#define       MC_CMD_LINK_STATE_OUT_V3_STATUS_FLAGS_OFST 0
#define       MC_CMD_LINK_STATE_OUT_V3_STATUS_FLAGS_LEN 8
#define       MC_CMD_LINK_STATE_OUT_V3_STATUS_FLAGS_LO_OFST 0
#define       MC_CMD_LINK_STATE_OUT_V3_STATUS_FLAGS_LO_LEN 4
#define       MC_CMD_LINK_STATE_OUT_V3_STATUS_FLAGS_LO_LBN 0
#define       MC_CMD_LINK_STATE_OUT_V3_STATUS_FLAGS_LO_WIDTH 32
#define       MC_CMD_LINK_STATE_OUT_V3_STATUS_FLAGS_HI_OFST 4
#define       MC_CMD_LINK_STATE_OUT_V3_STATUS_FLAGS_HI_LEN 4
#define       MC_CMD_LINK_STATE_OUT_V3_STATUS_FLAGS_HI_LBN 32
#define       MC_CMD_LINK_STATE_OUT_V3_STATUS_FLAGS_HI_WIDTH 32
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_LINK_STATUS_FLAGS/STATUS_FLAGS */
/* Configured technology. If the specified value does not align with the values
 * defined in the enum MC_CMD_ETH_TECH/TECH, it is considered invalid.
 */
#define       MC_CMD_LINK_STATE_OUT_V3_LINK_TECHNOLOGY_OFST 8
#define       MC_CMD_LINK_STATE_OUT_V3_LINK_TECHNOLOGY_LEN 2
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_ETH_TECH/TECH */
/* Configured FEC mode */
#define       MC_CMD_LINK_STATE_OUT_V3_FEC_MODE_OFST 10
#define       MC_CMD_LINK_STATE_OUT_V3_FEC_MODE_LEN 1
/* enum property: value */
/*            Enum values, see field(s): */
/*               FEC_TYPE/TYPE */
/* Bitmask of auto-negotiated pause modes */
#define       MC_CMD_LINK_STATE_OUT_V3_PAUSE_MASK_OFST 11
#define       MC_CMD_LINK_STATE_OUT_V3_PAUSE_MASK_LEN 1
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_PAUSE_MODE/TYPE */
/* Configured loopback mode */
#define       MC_CMD_LINK_STATE_OUT_V3_LOOPBACK_OFST 12
#define       MC_CMD_LINK_STATE_OUT_V3_LOOPBACK_LEN 1
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_LOOPBACK_V2/MODE */
/* Abilities requested by the driver to advertise during auto-negotiation */
#define       MC_CMD_LINK_STATE_OUT_V3_ADVERTISED_ABILITIES_OFST 16
#define       MC_CMD_LINK_STATE_OUT_V3_ADVERTISED_ABILITIES_LEN 32
/* Abilities advertised by the link partner during auto-negotiation */
#define       MC_CMD_LINK_STATE_OUT_V3_LINK_PARTNER_ABILITIES_OFST 48
#define       MC_CMD_LINK_STATE_OUT_V3_LINK_PARTNER_ABILITIES_LEN 32
/* Abilities supported by the local device (including cable abilities) For
 * fixed local device capbilities see MC_CMD_GET_LOCAL_DEVICE_INFO
 */
#define       MC_CMD_LINK_STATE_OUT_V3_SUPPORTED_ABILITIES_OFST 80
#define       MC_CMD_LINK_STATE_OUT_V3_SUPPORTED_ABILITIES_LEN 28
/* Control flags */
#define       MC_CMD_LINK_STATE_OUT_V3_CONTROL_FLAGS_OFST 108
#define       MC_CMD_LINK_STATE_OUT_V3_CONTROL_FLAGS_LEN 4
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_LINK_FLAGS/MASK */
/* Sequence number to synchronize link change events */
#define       MC_CMD_LINK_STATE_OUT_V3_PORT_LINKCHANGE_SEQ_NUM_OFST 112
#define       MC_CMD_LINK_STATE_OUT_V3_PORT_LINKCHANGE_SEQ_NUM_LEN 1
/* Sequence number to synchronize module change events */
#define       MC_CMD_LINK_STATE_OUT_V3_PORT_MODULECHANGE_SEQ_NUM_OFST 113
#define       MC_CMD_LINK_STATE_OUT_V3_PORT_MODULECHANGE_SEQ_NUM_LEN 1
/* Reports the auto-negotiation supported by the local device. This depends on
 * the port and module properties.
 */
#define       MC_CMD_LINK_STATE_OUT_V3_LOCAL_AN_SUPPORT_OFST 116
#define       MC_CMD_LINK_STATE_OUT_V3_LOCAL_AN_SUPPORT_LEN 4
/*            Enum values, see field(s): */
/*               AN_TYPE/TYPE */
/* Autonegotiated speed in mbit/s. The link may still be down even if this
 * reads non-zero. LINK_SPEED field is intended to be used by drivers without
 * the most up-to-date MCDI definitions, unable to deduce the link speed from
 * the reported LINK_TECHNOLOGY field.
 */
#define       MC_CMD_LINK_STATE_OUT_V3_LINK_SPEED_OFST 120
#define       MC_CMD_LINK_STATE_OUT_V3_LINK_SPEED_LEN 4
#define       MC_CMD_LINK_STATE_OUT_V3_FLAGS_OFST 124
#define       MC_CMD_LINK_STATE_OUT_V3_FLAGS_LEN 4
#define        MC_CMD_LINK_STATE_OUT_V3_FULL_DUPLEX_OFST 124
#define        MC_CMD_LINK_STATE_OUT_V3_FULL_DUPLEX_LBN 0
#define        MC_CMD_LINK_STATE_OUT_V3_FULL_DUPLEX_WIDTH 1


/***********************************/
/* MC_CMD_GET_LINK
 * Read the unified MAC/PHY link state. Locks required: None Return code: 0,
 * ETIME.
 */
#define MC_CMD_GET_LINK 0x29
#undef MC_CMD_0x29_PRIVILEGE_CTG

#define MC_CMD_0x29_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_GET_LINK_IN msgrequest */
#define    MC_CMD_GET_LINK_IN_LEN 0

/* MC_CMD_GET_LINK_IN_V2 msgrequest */
#define    MC_CMD_GET_LINK_IN_V2_LEN 8
/* Target port to request link state for. Uses MAE_LINK_ENDPOINT_SELECTOR which
 * identifies a real or virtual network port by MAE port and link end. See the
 * structure definition for more details.
 */
#define       MC_CMD_GET_LINK_IN_V2_TARGET_OFST 0
#define       MC_CMD_GET_LINK_IN_V2_TARGET_LEN 8
#define       MC_CMD_GET_LINK_IN_V2_TARGET_LO_OFST 0
#define       MC_CMD_GET_LINK_IN_V2_TARGET_LO_LEN 4
#define       MC_CMD_GET_LINK_IN_V2_TARGET_LO_LBN 0
#define       MC_CMD_GET_LINK_IN_V2_TARGET_LO_WIDTH 32
#define       MC_CMD_GET_LINK_IN_V2_TARGET_HI_OFST 4
#define       MC_CMD_GET_LINK_IN_V2_TARGET_HI_LEN 4
#define       MC_CMD_GET_LINK_IN_V2_TARGET_HI_LBN 32
#define       MC_CMD_GET_LINK_IN_V2_TARGET_HI_WIDTH 32
/* See structuredef: MAE_LINK_ENDPOINT_SELECTOR */
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_OFST 0
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_LEN 4
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_FLAT_OFST 0
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_FLAT_LEN 4
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_TYPE_OFST 3
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_TYPE_LEN 1
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_MPORT_ID_OFST 0
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_MPORT_ID_LEN 3
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_PPORT_ID_LBN 0
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_PPORT_ID_WIDTH 4
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_FUNC_INTF_ID_LBN 20
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_FUNC_INTF_ID_WIDTH 4
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_FUNC_MH_PF_ID_LBN 16
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_FUNC_MH_PF_ID_WIDTH 4
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_FUNC_PF_ID_OFST 2
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_FUNC_PF_ID_LEN 1
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_FUNC_VF_ID_OFST 0
#define       MC_CMD_GET_LINK_IN_V2_TARGET_MPORT_SELECTOR_FUNC_VF_ID_LEN 2
#define       MC_CMD_GET_LINK_IN_V2_TARGET_LINK_END_OFST 4
#define       MC_CMD_GET_LINK_IN_V2_TARGET_LINK_END_LEN 4
#define       MC_CMD_GET_LINK_IN_V2_TARGET_FLAT_OFST 0
#define       MC_CMD_GET_LINK_IN_V2_TARGET_FLAT_LEN 8
#define       MC_CMD_GET_LINK_IN_V2_TARGET_FLAT_LO_OFST 0
#define       MC_CMD_GET_LINK_IN_V2_TARGET_FLAT_LO_LEN 4
#define       MC_CMD_GET_LINK_IN_V2_TARGET_FLAT_LO_LBN 0
#define       MC_CMD_GET_LINK_IN_V2_TARGET_FLAT_LO_WIDTH 32
#define       MC_CMD_GET_LINK_IN_V2_TARGET_FLAT_HI_OFST 4
#define       MC_CMD_GET_LINK_IN_V2_TARGET_FLAT_HI_LEN 4
#define       MC_CMD_GET_LINK_IN_V2_TARGET_FLAT_HI_LBN 32
#define       MC_CMD_GET_LINK_IN_V2_TARGET_FLAT_HI_WIDTH 32

/* MC_CMD_GET_LINK_OUT msgresponse */
#define    MC_CMD_GET_LINK_OUT_LEN 28
/* Near-side advertised capabilities. Refer to
 * MC_CMD_GET_PHY_CFG_OUT/SUPPORTED_CAP for bit definitions.
 */
#define       MC_CMD_GET_LINK_OUT_CAP_OFST 0
#define       MC_CMD_GET_LINK_OUT_CAP_LEN 4
/* Link-partner advertised capabilities. Refer to
 * MC_CMD_GET_PHY_CFG_OUT/SUPPORTED_CAP for bit definitions.
 */
#define       MC_CMD_GET_LINK_OUT_LP_CAP_OFST 4
#define       MC_CMD_GET_LINK_OUT_LP_CAP_LEN 4
/* Autonegotiated speed in mbit/s. The link may still be down even if this
 * reads non-zero.
 */
#define       MC_CMD_GET_LINK_OUT_LINK_SPEED_OFST 8
#define       MC_CMD_GET_LINK_OUT_LINK_SPEED_LEN 4
/* Current loopback setting. */
#define       MC_CMD_GET_LINK_OUT_LOOPBACK_MODE_OFST 12
#define       MC_CMD_GET_LINK_OUT_LOOPBACK_MODE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_GET_LOOPBACK_MODES/MC_CMD_GET_LOOPBACK_MODES_OUT/100M */
#define       MC_CMD_GET_LINK_OUT_FLAGS_OFST 16
#define       MC_CMD_GET_LINK_OUT_FLAGS_LEN 4
#define        MC_CMD_GET_LINK_OUT_LINK_UP_OFST 16
#define        MC_CMD_GET_LINK_OUT_LINK_UP_LBN 0
#define        MC_CMD_GET_LINK_OUT_LINK_UP_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_FULL_DUPLEX_OFST 16
#define        MC_CMD_GET_LINK_OUT_FULL_DUPLEX_LBN 1
#define        MC_CMD_GET_LINK_OUT_FULL_DUPLEX_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_BPX_LINK_OFST 16
#define        MC_CMD_GET_LINK_OUT_BPX_LINK_LBN 2
#define        MC_CMD_GET_LINK_OUT_BPX_LINK_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_PHY_LINK_OFST 16
#define        MC_CMD_GET_LINK_OUT_PHY_LINK_LBN 3
#define        MC_CMD_GET_LINK_OUT_PHY_LINK_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_LINK_FAULT_RX_OFST 16
#define        MC_CMD_GET_LINK_OUT_LINK_FAULT_RX_LBN 6
#define        MC_CMD_GET_LINK_OUT_LINK_FAULT_RX_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_LINK_FAULT_TX_OFST 16
#define        MC_CMD_GET_LINK_OUT_LINK_FAULT_TX_LBN 7
#define        MC_CMD_GET_LINK_OUT_LINK_FAULT_TX_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_MODULE_UP_VALID_OFST 16
#define        MC_CMD_GET_LINK_OUT_MODULE_UP_VALID_LBN 8
#define        MC_CMD_GET_LINK_OUT_MODULE_UP_VALID_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_MODULE_UP_OFST 16
#define        MC_CMD_GET_LINK_OUT_MODULE_UP_LBN 9
#define        MC_CMD_GET_LINK_OUT_MODULE_UP_WIDTH 1
/* This returns the negotiated flow control value. */
#define       MC_CMD_GET_LINK_OUT_FCNTL_OFST 20
#define       MC_CMD_GET_LINK_OUT_FCNTL_LEN 4
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_SET_MAC/MC_CMD_SET_MAC_IN/FCNTL */
#define       MC_CMD_GET_LINK_OUT_MAC_FAULT_OFST 24
#define       MC_CMD_GET_LINK_OUT_MAC_FAULT_LEN 4
#define        MC_CMD_MAC_FAULT_XGMII_LOCAL_OFST 24
#define        MC_CMD_MAC_FAULT_XGMII_LOCAL_LBN 0
#define        MC_CMD_MAC_FAULT_XGMII_LOCAL_WIDTH 1
#define        MC_CMD_MAC_FAULT_XGMII_REMOTE_OFST 24
#define        MC_CMD_MAC_FAULT_XGMII_REMOTE_LBN 1
#define        MC_CMD_MAC_FAULT_XGMII_REMOTE_WIDTH 1
#define        MC_CMD_MAC_FAULT_SGMII_REMOTE_OFST 24
#define        MC_CMD_MAC_FAULT_SGMII_REMOTE_LBN 2
#define        MC_CMD_MAC_FAULT_SGMII_REMOTE_WIDTH 1
#define        MC_CMD_MAC_FAULT_PENDING_RECONFIG_OFST 24
#define        MC_CMD_MAC_FAULT_PENDING_RECONFIG_LBN 3
#define        MC_CMD_MAC_FAULT_PENDING_RECONFIG_WIDTH 1

/* MC_CMD_GET_LINK_OUT_V2 msgresponse: Extended link state information */
#define    MC_CMD_GET_LINK_OUT_V2_LEN 44
/* Near-side advertised capabilities. Refer to
 * MC_CMD_GET_PHY_CFG_OUT/SUPPORTED_CAP for bit definitions.
 */
#define       MC_CMD_GET_LINK_OUT_V2_CAP_OFST 0
#define       MC_CMD_GET_LINK_OUT_V2_CAP_LEN 4
/* Link-partner advertised capabilities. Refer to
 * MC_CMD_GET_PHY_CFG_OUT/SUPPORTED_CAP for bit definitions.
 */
#define       MC_CMD_GET_LINK_OUT_V2_LP_CAP_OFST 4
#define       MC_CMD_GET_LINK_OUT_V2_LP_CAP_LEN 4
/* Autonegotiated speed in mbit/s. The link may still be down even if this
 * reads non-zero.
 */
#define       MC_CMD_GET_LINK_OUT_V2_LINK_SPEED_OFST 8
#define       MC_CMD_GET_LINK_OUT_V2_LINK_SPEED_LEN 4
/* Current loopback setting. */
#define       MC_CMD_GET_LINK_OUT_V2_LOOPBACK_MODE_OFST 12
#define       MC_CMD_GET_LINK_OUT_V2_LOOPBACK_MODE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_GET_LOOPBACK_MODES/MC_CMD_GET_LOOPBACK_MODES_OUT/100M */
#define       MC_CMD_GET_LINK_OUT_V2_FLAGS_OFST 16
#define       MC_CMD_GET_LINK_OUT_V2_FLAGS_LEN 4
#define        MC_CMD_GET_LINK_OUT_V2_LINK_UP_OFST 16
#define        MC_CMD_GET_LINK_OUT_V2_LINK_UP_LBN 0
#define        MC_CMD_GET_LINK_OUT_V2_LINK_UP_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_V2_FULL_DUPLEX_OFST 16
#define        MC_CMD_GET_LINK_OUT_V2_FULL_DUPLEX_LBN 1
#define        MC_CMD_GET_LINK_OUT_V2_FULL_DUPLEX_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_V2_BPX_LINK_OFST 16
#define        MC_CMD_GET_LINK_OUT_V2_BPX_LINK_LBN 2
#define        MC_CMD_GET_LINK_OUT_V2_BPX_LINK_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_V2_PHY_LINK_OFST 16
#define        MC_CMD_GET_LINK_OUT_V2_PHY_LINK_LBN 3
#define        MC_CMD_GET_LINK_OUT_V2_PHY_LINK_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_V2_LINK_FAULT_RX_OFST 16
#define        MC_CMD_GET_LINK_OUT_V2_LINK_FAULT_RX_LBN 6
#define        MC_CMD_GET_LINK_OUT_V2_LINK_FAULT_RX_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_V2_LINK_FAULT_TX_OFST 16
#define        MC_CMD_GET_LINK_OUT_V2_LINK_FAULT_TX_LBN 7
#define        MC_CMD_GET_LINK_OUT_V2_LINK_FAULT_TX_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_V2_MODULE_UP_VALID_OFST 16
#define        MC_CMD_GET_LINK_OUT_V2_MODULE_UP_VALID_LBN 8
#define        MC_CMD_GET_LINK_OUT_V2_MODULE_UP_VALID_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_V2_MODULE_UP_OFST 16
#define        MC_CMD_GET_LINK_OUT_V2_MODULE_UP_LBN 9
#define        MC_CMD_GET_LINK_OUT_V2_MODULE_UP_WIDTH 1
/* This returns the negotiated flow control value. */
#define       MC_CMD_GET_LINK_OUT_V2_FCNTL_OFST 20
#define       MC_CMD_GET_LINK_OUT_V2_FCNTL_LEN 4
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_SET_MAC/MC_CMD_SET_MAC_IN/FCNTL */
#define       MC_CMD_GET_LINK_OUT_V2_MAC_FAULT_OFST 24
#define       MC_CMD_GET_LINK_OUT_V2_MAC_FAULT_LEN 4
/*             MC_CMD_MAC_FAULT_XGMII_LOCAL_OFST 24 */
/*             MC_CMD_MAC_FAULT_XGMII_LOCAL_LBN 0 */
/*             MC_CMD_MAC_FAULT_XGMII_LOCAL_WIDTH 1 */
/*             MC_CMD_MAC_FAULT_XGMII_REMOTE_OFST 24 */
/*             MC_CMD_MAC_FAULT_XGMII_REMOTE_LBN 1 */
/*             MC_CMD_MAC_FAULT_XGMII_REMOTE_WIDTH 1 */
/*             MC_CMD_MAC_FAULT_SGMII_REMOTE_OFST 24 */
/*             MC_CMD_MAC_FAULT_SGMII_REMOTE_LBN 2 */
/*             MC_CMD_MAC_FAULT_SGMII_REMOTE_WIDTH 1 */
/*             MC_CMD_MAC_FAULT_PENDING_RECONFIG_OFST 24 */
/*             MC_CMD_MAC_FAULT_PENDING_RECONFIG_LBN 3 */
/*             MC_CMD_MAC_FAULT_PENDING_RECONFIG_WIDTH 1 */
/* True local device capabilities (taking into account currently used PMD/MDI,
 * e.g. plugged-in module). In general, subset of
 * MC_CMD_GET_PHY_CFG_OUT/SUPPORTED_CAP, but may include extra _FEC_REQUEST
 * bits, if the PMD requires FEC. 0 if unknown (e.g. module unplugged). Equal
 * to SUPPORTED_CAP for non-pluggable PMDs. Refer to
 * MC_CMD_GET_PHY_CFG_OUT/SUPPORTED_CAP for bit definitions.
 */
#define       MC_CMD_GET_LINK_OUT_V2_LD_CAP_OFST 28
#define       MC_CMD_GET_LINK_OUT_V2_LD_CAP_LEN 4
/* Auto-negotiation type used on the link */
#define       MC_CMD_GET_LINK_OUT_V2_AN_TYPE_OFST 32
#define       MC_CMD_GET_LINK_OUT_V2_AN_TYPE_LEN 4
/*            Enum values, see field(s): */
/*               AN_TYPE/TYPE */
/* Forward error correction used on the link */
#define       MC_CMD_GET_LINK_OUT_V2_FEC_TYPE_OFST 36
#define       MC_CMD_GET_LINK_OUT_V2_FEC_TYPE_LEN 4
/*            Enum values, see field(s): */
/*               FEC_TYPE/TYPE */
#define       MC_CMD_GET_LINK_OUT_V2_EXT_FLAGS_OFST 40
#define       MC_CMD_GET_LINK_OUT_V2_EXT_FLAGS_LEN 4
#define        MC_CMD_GET_LINK_OUT_V2_PMD_MDI_CONNECTED_OFST 40
#define        MC_CMD_GET_LINK_OUT_V2_PMD_MDI_CONNECTED_LBN 0
#define        MC_CMD_GET_LINK_OUT_V2_PMD_MDI_CONNECTED_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_V2_PMD_READY_OFST 40
#define        MC_CMD_GET_LINK_OUT_V2_PMD_READY_LBN 1
#define        MC_CMD_GET_LINK_OUT_V2_PMD_READY_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_V2_PMD_LINK_UP_OFST 40
#define        MC_CMD_GET_LINK_OUT_V2_PMD_LINK_UP_LBN 2
#define        MC_CMD_GET_LINK_OUT_V2_PMD_LINK_UP_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_V2_PMA_LINK_UP_OFST 40
#define        MC_CMD_GET_LINK_OUT_V2_PMA_LINK_UP_LBN 3
#define        MC_CMD_GET_LINK_OUT_V2_PMA_LINK_UP_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_V2_PCS_LOCK_OFST 40
#define        MC_CMD_GET_LINK_OUT_V2_PCS_LOCK_LBN 4
#define        MC_CMD_GET_LINK_OUT_V2_PCS_LOCK_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_V2_ALIGN_LOCK_OFST 40
#define        MC_CMD_GET_LINK_OUT_V2_ALIGN_LOCK_LBN 5
#define        MC_CMD_GET_LINK_OUT_V2_ALIGN_LOCK_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_V2_HI_BER_OFST 40
#define        MC_CMD_GET_LINK_OUT_V2_HI_BER_LBN 6
#define        MC_CMD_GET_LINK_OUT_V2_HI_BER_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_V2_FEC_LOCK_OFST 40
#define        MC_CMD_GET_LINK_OUT_V2_FEC_LOCK_LBN 7
#define        MC_CMD_GET_LINK_OUT_V2_FEC_LOCK_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_V2_AN_DONE_OFST 40
#define        MC_CMD_GET_LINK_OUT_V2_AN_DONE_LBN 8
#define        MC_CMD_GET_LINK_OUT_V2_AN_DONE_WIDTH 1
#define        MC_CMD_GET_LINK_OUT_V2_PORT_SHUTDOWN_OFST 40
#define        MC_CMD_GET_LINK_OUT_V2_PORT_SHUTDOWN_LBN 9
#define        MC_CMD_GET_LINK_OUT_V2_PORT_SHUTDOWN_WIDTH 1


/***********************************/
/* MC_CMD_SET_LINK
 * Write the unified MAC/PHY link configuration. Locks required: None. Return
 * code: 0, EINVAL, ETIME, EAGAIN
 */
#define MC_CMD_SET_LINK 0x2a
#undef MC_CMD_0x2a_PRIVILEGE_CTG

#define MC_CMD_0x2a_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_SET_LINK_IN msgrequest */
#define    MC_CMD_SET_LINK_IN_LEN 16
/* Near-side advertised capabilities. Refer to
 * MC_CMD_GET_PHY_CFG_OUT/SUPPORTED_CAP for bit definitions.
 */
#define       MC_CMD_SET_LINK_IN_CAP_OFST 0
#define       MC_CMD_SET_LINK_IN_CAP_LEN 4
/* Flags */
#define       MC_CMD_SET_LINK_IN_FLAGS_OFST 4
#define       MC_CMD_SET_LINK_IN_FLAGS_LEN 4
#define        MC_CMD_SET_LINK_IN_LOWPOWER_OFST 4
#define        MC_CMD_SET_LINK_IN_LOWPOWER_LBN 0
#define        MC_CMD_SET_LINK_IN_LOWPOWER_WIDTH 1
#define        MC_CMD_SET_LINK_IN_POWEROFF_OFST 4
#define        MC_CMD_SET_LINK_IN_POWEROFF_LBN 1
#define        MC_CMD_SET_LINK_IN_POWEROFF_WIDTH 1
#define        MC_CMD_SET_LINK_IN_TXDIS_OFST 4
#define        MC_CMD_SET_LINK_IN_TXDIS_LBN 2
#define        MC_CMD_SET_LINK_IN_TXDIS_WIDTH 1
#define        MC_CMD_SET_LINK_IN_LINKDOWN_OFST 4
#define        MC_CMD_SET_LINK_IN_LINKDOWN_LBN 3
#define        MC_CMD_SET_LINK_IN_LINKDOWN_WIDTH 1
/* Loopback mode. */
#define       MC_CMD_SET_LINK_IN_LOOPBACK_MODE_OFST 8
#define       MC_CMD_SET_LINK_IN_LOOPBACK_MODE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_GET_LOOPBACK_MODES/MC_CMD_GET_LOOPBACK_MODES_OUT/100M */
/* A loopback speed of "0" is supported, and means (choose any available
 * speed).
 */
#define       MC_CMD_SET_LINK_IN_LOOPBACK_SPEED_OFST 12
#define       MC_CMD_SET_LINK_IN_LOOPBACK_SPEED_LEN 4

/* MC_CMD_SET_LINK_IN_V2 msgrequest: Updated SET_LINK to include sequence
 * number to ensure this SET_LINK command corresponds to the latest
 * MODULECHANGE event.
 */
#define    MC_CMD_SET_LINK_IN_V2_LEN 17
/* Near-side advertised capabilities. Refer to
 * MC_CMD_GET_PHY_CFG_OUT/SUPPORTED_CAP for bit definitions.
 */
#define       MC_CMD_SET_LINK_IN_V2_CAP_OFST 0
#define       MC_CMD_SET_LINK_IN_V2_CAP_LEN 4
/* Flags */
#define       MC_CMD_SET_LINK_IN_V2_FLAGS_OFST 4
#define       MC_CMD_SET_LINK_IN_V2_FLAGS_LEN 4
#define        MC_CMD_SET_LINK_IN_V2_LOWPOWER_OFST 4
#define        MC_CMD_SET_LINK_IN_V2_LOWPOWER_LBN 0
#define        MC_CMD_SET_LINK_IN_V2_LOWPOWER_WIDTH 1
#define        MC_CMD_SET_LINK_IN_V2_POWEROFF_OFST 4
#define        MC_CMD_SET_LINK_IN_V2_POWEROFF_LBN 1
#define        MC_CMD_SET_LINK_IN_V2_POWEROFF_WIDTH 1
#define        MC_CMD_SET_LINK_IN_V2_TXDIS_OFST 4
#define        MC_CMD_SET_LINK_IN_V2_TXDIS_LBN 2
#define        MC_CMD_SET_LINK_IN_V2_TXDIS_WIDTH 1
#define        MC_CMD_SET_LINK_IN_V2_LINKDOWN_OFST 4
#define        MC_CMD_SET_LINK_IN_V2_LINKDOWN_LBN 3
#define        MC_CMD_SET_LINK_IN_V2_LINKDOWN_WIDTH 1
/* Loopback mode. */
#define       MC_CMD_SET_LINK_IN_V2_LOOPBACK_MODE_OFST 8
#define       MC_CMD_SET_LINK_IN_V2_LOOPBACK_MODE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_GET_LOOPBACK_MODES/MC_CMD_GET_LOOPBACK_MODES_OUT/100M */
/* A loopback speed of "0" is supported, and means (choose any available
 * speed).
 */
#define       MC_CMD_SET_LINK_IN_V2_LOOPBACK_SPEED_OFST 12
#define       MC_CMD_SET_LINK_IN_V2_LOOPBACK_SPEED_LEN 4
#define       MC_CMD_SET_LINK_IN_V2_MODULE_SEQ_OFST 16
#define       MC_CMD_SET_LINK_IN_V2_MODULE_SEQ_LEN 1
#define        MC_CMD_SET_LINK_IN_V2_MODULE_SEQ_NUMBER_OFST 16
#define        MC_CMD_SET_LINK_IN_V2_MODULE_SEQ_NUMBER_LBN 0
#define        MC_CMD_SET_LINK_IN_V2_MODULE_SEQ_NUMBER_WIDTH 7
#define        MC_CMD_SET_LINK_IN_V2_MODULE_SEQ_IGNORE_OFST 16
#define        MC_CMD_SET_LINK_IN_V2_MODULE_SEQ_IGNORE_LBN 7
#define        MC_CMD_SET_LINK_IN_V2_MODULE_SEQ_IGNORE_WIDTH 1

/* MC_CMD_SET_LINK_IN_V3 msgrequest */
#define    MC_CMD_SET_LINK_IN_V3_LEN 28
/* Near-side advertised capabilities. Refer to
 * MC_CMD_GET_PHY_CFG_OUT/SUPPORTED_CAP for bit definitions.
 */
#define       MC_CMD_SET_LINK_IN_V3_CAP_OFST 0
#define       MC_CMD_SET_LINK_IN_V3_CAP_LEN 4
/* Flags */
#define       MC_CMD_SET_LINK_IN_V3_FLAGS_OFST 4
#define       MC_CMD_SET_LINK_IN_V3_FLAGS_LEN 4
#define        MC_CMD_SET_LINK_IN_V3_LOWPOWER_OFST 4
#define        MC_CMD_SET_LINK_IN_V3_LOWPOWER_LBN 0
#define        MC_CMD_SET_LINK_IN_V3_LOWPOWER_WIDTH 1
#define        MC_CMD_SET_LINK_IN_V3_POWEROFF_OFST 4
#define        MC_CMD_SET_LINK_IN_V3_POWEROFF_LBN 1
#define        MC_CMD_SET_LINK_IN_V3_POWEROFF_WIDTH 1
#define        MC_CMD_SET_LINK_IN_V3_TXDIS_OFST 4
#define        MC_CMD_SET_LINK_IN_V3_TXDIS_LBN 2
#define        MC_CMD_SET_LINK_IN_V3_TXDIS_WIDTH 1
#define        MC_CMD_SET_LINK_IN_V3_LINKDOWN_OFST 4
#define        MC_CMD_SET_LINK_IN_V3_LINKDOWN_LBN 3
#define        MC_CMD_SET_LINK_IN_V3_LINKDOWN_WIDTH 1
/* Loopback mode. */
#define       MC_CMD_SET_LINK_IN_V3_LOOPBACK_MODE_OFST 8
#define       MC_CMD_SET_LINK_IN_V3_LOOPBACK_MODE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_GET_LOOPBACK_MODES/MC_CMD_GET_LOOPBACK_MODES_OUT/100M */
/* A loopback speed of "0" is supported, and means (choose any available
 * speed).
 */
#define       MC_CMD_SET_LINK_IN_V3_LOOPBACK_SPEED_OFST 12
#define       MC_CMD_SET_LINK_IN_V3_LOOPBACK_SPEED_LEN 4
#define       MC_CMD_SET_LINK_IN_V3_MODULE_SEQ_OFST 16
#define       MC_CMD_SET_LINK_IN_V3_MODULE_SEQ_LEN 1
#define        MC_CMD_SET_LINK_IN_V3_MODULE_SEQ_NUMBER_OFST 16
#define        MC_CMD_SET_LINK_IN_V3_MODULE_SEQ_NUMBER_LBN 0
#define        MC_CMD_SET_LINK_IN_V3_MODULE_SEQ_NUMBER_WIDTH 7
#define        MC_CMD_SET_LINK_IN_V3_MODULE_SEQ_IGNORE_OFST 16
#define        MC_CMD_SET_LINK_IN_V3_MODULE_SEQ_IGNORE_LBN 7
#define        MC_CMD_SET_LINK_IN_V3_MODULE_SEQ_IGNORE_WIDTH 1
/* Padding */
#define       MC_CMD_SET_LINK_IN_V3_RESERVED_OFST 17
#define       MC_CMD_SET_LINK_IN_V3_RESERVED_LEN 3
/* Target port to set link state for. Uses MAE_LINK_ENDPOINT_SELECTOR which
 * identifies a real or virtual network port by MAE port and link end. See the
 * structure definition for more details
 */
#define       MC_CMD_SET_LINK_IN_V3_TARGET_OFST 20
#define       MC_CMD_SET_LINK_IN_V3_TARGET_LEN 8
#define       MC_CMD_SET_LINK_IN_V3_TARGET_LO_OFST 20
#define       MC_CMD_SET_LINK_IN_V3_TARGET_LO_LEN 4
#define       MC_CMD_SET_LINK_IN_V3_TARGET_LO_LBN 160
#define       MC_CMD_SET_LINK_IN_V3_TARGET_LO_WIDTH 32
#define       MC_CMD_SET_LINK_IN_V3_TARGET_HI_OFST 24
#define       MC_CMD_SET_LINK_IN_V3_TARGET_HI_LEN 4
#define       MC_CMD_SET_LINK_IN_V3_TARGET_HI_LBN 192
#define       MC_CMD_SET_LINK_IN_V3_TARGET_HI_WIDTH 32
/* See structuredef: MAE_LINK_ENDPOINT_SELECTOR */
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_OFST 20
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_LEN 4
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_FLAT_OFST 20
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_FLAT_LEN 4
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_TYPE_OFST 23
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_TYPE_LEN 1
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_MPORT_ID_OFST 20
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_MPORT_ID_LEN 3
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_PPORT_ID_LBN 160
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_PPORT_ID_WIDTH 4
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_FUNC_INTF_ID_LBN 180
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_FUNC_INTF_ID_WIDTH 4
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_FUNC_MH_PF_ID_LBN 176
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_FUNC_MH_PF_ID_WIDTH 4
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_FUNC_PF_ID_OFST 22
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_FUNC_PF_ID_LEN 1
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_FUNC_VF_ID_OFST 20
#define       MC_CMD_SET_LINK_IN_V3_TARGET_MPORT_SELECTOR_FUNC_VF_ID_LEN 2
#define       MC_CMD_SET_LINK_IN_V3_TARGET_LINK_END_OFST 24
#define       MC_CMD_SET_LINK_IN_V3_TARGET_LINK_END_LEN 4
#define       MC_CMD_SET_LINK_IN_V3_TARGET_FLAT_OFST 20
#define       MC_CMD_SET_LINK_IN_V3_TARGET_FLAT_LEN 8
#define       MC_CMD_SET_LINK_IN_V3_TARGET_FLAT_LO_OFST 20
#define       MC_CMD_SET_LINK_IN_V3_TARGET_FLAT_LO_LEN 4
#define       MC_CMD_SET_LINK_IN_V3_TARGET_FLAT_LO_LBN 160
#define       MC_CMD_SET_LINK_IN_V3_TARGET_FLAT_LO_WIDTH 32
#define       MC_CMD_SET_LINK_IN_V3_TARGET_FLAT_HI_OFST 24
#define       MC_CMD_SET_LINK_IN_V3_TARGET_FLAT_HI_LEN 4
#define       MC_CMD_SET_LINK_IN_V3_TARGET_FLAT_HI_LBN 192
#define       MC_CMD_SET_LINK_IN_V3_TARGET_FLAT_HI_WIDTH 32

/* MC_CMD_SET_LINK_OUT msgresponse */
#define    MC_CMD_SET_LINK_OUT_LEN 0


/***********************************/
/* MC_CMD_SET_ID_LED
 * Set identification LED state. Locks required: None. Return code: 0, EINVAL
 */
#define MC_CMD_SET_ID_LED 0x2b
#undef MC_CMD_0x2b_PRIVILEGE_CTG

#define MC_CMD_0x2b_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_SET_ID_LED_IN msgrequest */
#define    MC_CMD_SET_ID_LED_IN_LEN 4
/* Set LED state. */
#define       MC_CMD_SET_ID_LED_IN_STATE_OFST 0
#define       MC_CMD_SET_ID_LED_IN_STATE_LEN 4
#define          MC_CMD_LED_OFF 0x0 /* enum */
#define          MC_CMD_LED_ON 0x1 /* enum */
#define          MC_CMD_LED_DEFAULT 0x2 /* enum */

/* MC_CMD_SET_ID_LED_OUT msgresponse */
#define    MC_CMD_SET_ID_LED_OUT_LEN 0


/***********************************/
/* MC_CMD_SET_MAC
 * Set MAC configuration. Locks required: None. Return code: 0, EINVAL
 */
#define MC_CMD_SET_MAC 0x2c
#undef MC_CMD_0x2c_PRIVILEGE_CTG

#define MC_CMD_0x2c_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_SET_MAC_IN msgrequest */
#define    MC_CMD_SET_MAC_IN_LEN 28
/* The MTU is the MTU programmed directly into the XMAC/GMAC (inclusive of
 * EtherII, VLAN, bug16011 padding).
 */
#define       MC_CMD_SET_MAC_IN_MTU_OFST 0
#define       MC_CMD_SET_MAC_IN_MTU_LEN 4
#define       MC_CMD_SET_MAC_IN_DRAIN_OFST 4
#define       MC_CMD_SET_MAC_IN_DRAIN_LEN 4
#define       MC_CMD_SET_MAC_IN_ADDR_OFST 8
#define       MC_CMD_SET_MAC_IN_ADDR_LEN 8
#define       MC_CMD_SET_MAC_IN_ADDR_LO_OFST 8
#define       MC_CMD_SET_MAC_IN_ADDR_LO_LEN 4
#define       MC_CMD_SET_MAC_IN_ADDR_LO_LBN 64
#define       MC_CMD_SET_MAC_IN_ADDR_LO_WIDTH 32
#define       MC_CMD_SET_MAC_IN_ADDR_HI_OFST 12
#define       MC_CMD_SET_MAC_IN_ADDR_HI_LEN 4
#define       MC_CMD_SET_MAC_IN_ADDR_HI_LBN 96
#define       MC_CMD_SET_MAC_IN_ADDR_HI_WIDTH 32
#define       MC_CMD_SET_MAC_IN_REJECT_OFST 16
#define       MC_CMD_SET_MAC_IN_REJECT_LEN 4
#define        MC_CMD_SET_MAC_IN_REJECT_UNCST_OFST 16
#define        MC_CMD_SET_MAC_IN_REJECT_UNCST_LBN 0
#define        MC_CMD_SET_MAC_IN_REJECT_UNCST_WIDTH 1
#define        MC_CMD_SET_MAC_IN_REJECT_BRDCST_OFST 16
#define        MC_CMD_SET_MAC_IN_REJECT_BRDCST_LBN 1
#define        MC_CMD_SET_MAC_IN_REJECT_BRDCST_WIDTH 1
#define       MC_CMD_SET_MAC_IN_FCNTL_OFST 20
#define       MC_CMD_SET_MAC_IN_FCNTL_LEN 4
/* enum: Flow control is off. */
/*               MC_CMD_FCNTL_OFF 0x0 */
/* enum: Respond to flow control. */
/*               MC_CMD_FCNTL_RESPOND 0x1 */
/* enum: Respond to and Issue flow control. */
/*               MC_CMD_FCNTL_BIDIR 0x2 */
/* enum: Auto negotiate flow control. */
/*               MC_CMD_FCNTL_AUTO 0x3 */
/* enum: Priority flow control. This is only supported on KSB. */
/*               MC_CMD_FCNTL_QBB 0x4 */
/* enum: Issue flow control. */
/*               MC_CMD_FCNTL_GENERATE 0x5 */
#define       MC_CMD_SET_MAC_IN_FLAGS_OFST 24
#define       MC_CMD_SET_MAC_IN_FLAGS_LEN 4
#define        MC_CMD_SET_MAC_IN_FLAG_INCLUDE_FCS_OFST 24
#define        MC_CMD_SET_MAC_IN_FLAG_INCLUDE_FCS_LBN 0
#define        MC_CMD_SET_MAC_IN_FLAG_INCLUDE_FCS_WIDTH 1

/* MC_CMD_SET_MAC_EXT_IN msgrequest */
#define    MC_CMD_SET_MAC_EXT_IN_LEN 32
/* The MTU is the MTU programmed directly into the XMAC/GMAC (inclusive of
 * EtherII, VLAN, bug16011 padding).
 */
#define       MC_CMD_SET_MAC_EXT_IN_MTU_OFST 0
#define       MC_CMD_SET_MAC_EXT_IN_MTU_LEN 4
#define       MC_CMD_SET_MAC_EXT_IN_DRAIN_OFST 4
#define       MC_CMD_SET_MAC_EXT_IN_DRAIN_LEN 4
#define       MC_CMD_SET_MAC_EXT_IN_ADDR_OFST 8
#define       MC_CMD_SET_MAC_EXT_IN_ADDR_LEN 8
#define       MC_CMD_SET_MAC_EXT_IN_ADDR_LO_OFST 8
#define       MC_CMD_SET_MAC_EXT_IN_ADDR_LO_LEN 4
#define       MC_CMD_SET_MAC_EXT_IN_ADDR_LO_LBN 64
#define       MC_CMD_SET_MAC_EXT_IN_ADDR_LO_WIDTH 32
#define       MC_CMD_SET_MAC_EXT_IN_ADDR_HI_OFST 12
#define       MC_CMD_SET_MAC_EXT_IN_ADDR_HI_LEN 4
#define       MC_CMD_SET_MAC_EXT_IN_ADDR_HI_LBN 96
#define       MC_CMD_SET_MAC_EXT_IN_ADDR_HI_WIDTH 32
#define       MC_CMD_SET_MAC_EXT_IN_REJECT_OFST 16
#define       MC_CMD_SET_MAC_EXT_IN_REJECT_LEN 4
#define        MC_CMD_SET_MAC_EXT_IN_REJECT_UNCST_OFST 16
#define        MC_CMD_SET_MAC_EXT_IN_REJECT_UNCST_LBN 0
#define        MC_CMD_SET_MAC_EXT_IN_REJECT_UNCST_WIDTH 1
#define        MC_CMD_SET_MAC_EXT_IN_REJECT_BRDCST_OFST 16
#define        MC_CMD_SET_MAC_EXT_IN_REJECT_BRDCST_LBN 1
#define        MC_CMD_SET_MAC_EXT_IN_REJECT_BRDCST_WIDTH 1
#define       MC_CMD_SET_MAC_EXT_IN_FCNTL_OFST 20
#define       MC_CMD_SET_MAC_EXT_IN_FCNTL_LEN 4
/* enum: Flow control is off. */
/*               MC_CMD_FCNTL_OFF 0x0 */
/* enum: Respond to flow control. */
/*               MC_CMD_FCNTL_RESPOND 0x1 */
/* enum: Respond to and Issue flow control. */
/*               MC_CMD_FCNTL_BIDIR 0x2 */
/* enum: Auto negotiate flow control. */
/*               MC_CMD_FCNTL_AUTO 0x3 */
/* enum: Priority flow control. This is only supported on KSB. */
/*               MC_CMD_FCNTL_QBB 0x4 */
/* enum: Issue flow control. */
/*               MC_CMD_FCNTL_GENERATE 0x5 */
#define       MC_CMD_SET_MAC_EXT_IN_FLAGS_OFST 24
#define       MC_CMD_SET_MAC_EXT_IN_FLAGS_LEN 4
#define        MC_CMD_SET_MAC_EXT_IN_FLAG_INCLUDE_FCS_OFST 24
#define        MC_CMD_SET_MAC_EXT_IN_FLAG_INCLUDE_FCS_LBN 0
#define        MC_CMD_SET_MAC_EXT_IN_FLAG_INCLUDE_FCS_WIDTH 1
/* Select which parameters to configure. A parameter will only be modified if
 * the corresponding control flag is set. If SET_MAC_ENHANCED is not set in
 * capabilities then this field is ignored (and all flags are assumed to be
 * set).
 */
#define       MC_CMD_SET_MAC_EXT_IN_CONTROL_OFST 28
#define       MC_CMD_SET_MAC_EXT_IN_CONTROL_LEN 4
#define        MC_CMD_SET_MAC_EXT_IN_CFG_MTU_OFST 28
#define        MC_CMD_SET_MAC_EXT_IN_CFG_MTU_LBN 0
#define        MC_CMD_SET_MAC_EXT_IN_CFG_MTU_WIDTH 1
#define        MC_CMD_SET_MAC_EXT_IN_CFG_DRAIN_OFST 28
#define        MC_CMD_SET_MAC_EXT_IN_CFG_DRAIN_LBN 1
#define        MC_CMD_SET_MAC_EXT_IN_CFG_DRAIN_WIDTH 1
#define        MC_CMD_SET_MAC_EXT_IN_CFG_REJECT_OFST 28
#define        MC_CMD_SET_MAC_EXT_IN_CFG_REJECT_LBN 2
#define        MC_CMD_SET_MAC_EXT_IN_CFG_REJECT_WIDTH 1
#define        MC_CMD_SET_MAC_EXT_IN_CFG_FCNTL_OFST 28
#define        MC_CMD_SET_MAC_EXT_IN_CFG_FCNTL_LBN 3
#define        MC_CMD_SET_MAC_EXT_IN_CFG_FCNTL_WIDTH 1
#define        MC_CMD_SET_MAC_EXT_IN_CFG_FCS_OFST 28
#define        MC_CMD_SET_MAC_EXT_IN_CFG_FCS_LBN 4
#define        MC_CMD_SET_MAC_EXT_IN_CFG_FCS_WIDTH 1

/* MC_CMD_SET_MAC_V3_IN msgrequest */
#define    MC_CMD_SET_MAC_V3_IN_LEN 40
/* The MTU is the MTU programmed directly into the XMAC/GMAC (inclusive of
 * EtherII, VLAN, bug16011 padding).
 */
#define       MC_CMD_SET_MAC_V3_IN_MTU_OFST 0
#define       MC_CMD_SET_MAC_V3_IN_MTU_LEN 4
#define       MC_CMD_SET_MAC_V3_IN_DRAIN_OFST 4
#define       MC_CMD_SET_MAC_V3_IN_DRAIN_LEN 4
#define       MC_CMD_SET_MAC_V3_IN_ADDR_OFST 8
#define       MC_CMD_SET_MAC_V3_IN_ADDR_LEN 8
#define       MC_CMD_SET_MAC_V3_IN_ADDR_LO_OFST 8
#define       MC_CMD_SET_MAC_V3_IN_ADDR_LO_LEN 4
#define       MC_CMD_SET_MAC_V3_IN_ADDR_LO_LBN 64
#define       MC_CMD_SET_MAC_V3_IN_ADDR_LO_WIDTH 32
#define       MC_CMD_SET_MAC_V3_IN_ADDR_HI_OFST 12
#define       MC_CMD_SET_MAC_V3_IN_ADDR_HI_LEN 4
#define       MC_CMD_SET_MAC_V3_IN_ADDR_HI_LBN 96
#define       MC_CMD_SET_MAC_V3_IN_ADDR_HI_WIDTH 32
#define       MC_CMD_SET_MAC_V3_IN_REJECT_OFST 16
#define       MC_CMD_SET_MAC_V3_IN_REJECT_LEN 4
#define        MC_CMD_SET_MAC_V3_IN_REJECT_UNCST_OFST 16
#define        MC_CMD_SET_MAC_V3_IN_REJECT_UNCST_LBN 0
#define        MC_CMD_SET_MAC_V3_IN_REJECT_UNCST_WIDTH 1
#define        MC_CMD_SET_MAC_V3_IN_REJECT_BRDCST_OFST 16
#define        MC_CMD_SET_MAC_V3_IN_REJECT_BRDCST_LBN 1
#define        MC_CMD_SET_MAC_V3_IN_REJECT_BRDCST_WIDTH 1
#define       MC_CMD_SET_MAC_V3_IN_FCNTL_OFST 20
#define       MC_CMD_SET_MAC_V3_IN_FCNTL_LEN 4
/* enum: Flow control is off. */
/*               MC_CMD_FCNTL_OFF 0x0 */
/* enum: Respond to flow control. */
/*               MC_CMD_FCNTL_RESPOND 0x1 */
/* enum: Respond to and Issue flow control. */
/*               MC_CMD_FCNTL_BIDIR 0x2 */
/* enum: Auto negotiate flow control. */
/*               MC_CMD_FCNTL_AUTO 0x3 */
/* enum: Priority flow control. This is only supported on KSB. */
/*               MC_CMD_FCNTL_QBB 0x4 */
/* enum: Issue flow control. */
/*               MC_CMD_FCNTL_GENERATE 0x5 */
#define       MC_CMD_SET_MAC_V3_IN_FLAGS_OFST 24
#define       MC_CMD_SET_MAC_V3_IN_FLAGS_LEN 4
#define        MC_CMD_SET_MAC_V3_IN_FLAG_INCLUDE_FCS_OFST 24
#define        MC_CMD_SET_MAC_V3_IN_FLAG_INCLUDE_FCS_LBN 0
#define        MC_CMD_SET_MAC_V3_IN_FLAG_INCLUDE_FCS_WIDTH 1
/* Select which parameters to configure. A parameter will only be modified if
 * the corresponding control flag is set. If SET_MAC_ENHANCED is not set in
 * capabilities then this field is ignored (and all flags are assumed to be
 * set).
 */
#define       MC_CMD_SET_MAC_V3_IN_CONTROL_OFST 28
#define       MC_CMD_SET_MAC_V3_IN_CONTROL_LEN 4
#define        MC_CMD_SET_MAC_V3_IN_CFG_MTU_OFST 28
#define        MC_CMD_SET_MAC_V3_IN_CFG_MTU_LBN 0
#define        MC_CMD_SET_MAC_V3_IN_CFG_MTU_WIDTH 1
#define        MC_CMD_SET_MAC_V3_IN_CFG_DRAIN_OFST 28
#define        MC_CMD_SET_MAC_V3_IN_CFG_DRAIN_LBN 1
#define        MC_CMD_SET_MAC_V3_IN_CFG_DRAIN_WIDTH 1
#define        MC_CMD_SET_MAC_V3_IN_CFG_REJECT_OFST 28
#define        MC_CMD_SET_MAC_V3_IN_CFG_REJECT_LBN 2
#define        MC_CMD_SET_MAC_V3_IN_CFG_REJECT_WIDTH 1
#define        MC_CMD_SET_MAC_V3_IN_CFG_FCNTL_OFST 28
#define        MC_CMD_SET_MAC_V3_IN_CFG_FCNTL_LBN 3
#define        MC_CMD_SET_MAC_V3_IN_CFG_FCNTL_WIDTH 1
#define        MC_CMD_SET_MAC_V3_IN_CFG_FCS_OFST 28
#define        MC_CMD_SET_MAC_V3_IN_CFG_FCS_LBN 4
#define        MC_CMD_SET_MAC_V3_IN_CFG_FCS_WIDTH 1
/* Target port to set mac state for. Uses MAE_LINK_ENDPOINT_SELECTOR which
 * identifies a real or virtual network port by MAE port and link end. See the
 * structure definition for more details
 */
#define       MC_CMD_SET_MAC_V3_IN_TARGET_OFST 32
#define       MC_CMD_SET_MAC_V3_IN_TARGET_LEN 8
#define       MC_CMD_SET_MAC_V3_IN_TARGET_LO_OFST 32
#define       MC_CMD_SET_MAC_V3_IN_TARGET_LO_LEN 4
#define       MC_CMD_SET_MAC_V3_IN_TARGET_LO_LBN 256
#define       MC_CMD_SET_MAC_V3_IN_TARGET_LO_WIDTH 32
#define       MC_CMD_SET_MAC_V3_IN_TARGET_HI_OFST 36
#define       MC_CMD_SET_MAC_V3_IN_TARGET_HI_LEN 4
#define       MC_CMD_SET_MAC_V3_IN_TARGET_HI_LBN 288
#define       MC_CMD_SET_MAC_V3_IN_TARGET_HI_WIDTH 32
/* See structuredef: MAE_LINK_ENDPOINT_SELECTOR */
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_OFST 32
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_LEN 4
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_FLAT_OFST 32
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_FLAT_LEN 4
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_TYPE_OFST 35
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_TYPE_LEN 1
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_MPORT_ID_OFST 32
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_MPORT_ID_LEN 3
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_PPORT_ID_LBN 256
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_PPORT_ID_WIDTH 4
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_FUNC_INTF_ID_LBN 276
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_FUNC_INTF_ID_WIDTH 4
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_FUNC_MH_PF_ID_LBN 272
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_FUNC_MH_PF_ID_WIDTH 4
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_FUNC_PF_ID_OFST 34
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_FUNC_PF_ID_LEN 1
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_FUNC_VF_ID_OFST 32
#define       MC_CMD_SET_MAC_V3_IN_TARGET_MPORT_SELECTOR_FUNC_VF_ID_LEN 2
#define       MC_CMD_SET_MAC_V3_IN_TARGET_LINK_END_OFST 36
#define       MC_CMD_SET_MAC_V3_IN_TARGET_LINK_END_LEN 4
#define       MC_CMD_SET_MAC_V3_IN_TARGET_FLAT_OFST 32
#define       MC_CMD_SET_MAC_V3_IN_TARGET_FLAT_LEN 8
#define       MC_CMD_SET_MAC_V3_IN_TARGET_FLAT_LO_OFST 32
#define       MC_CMD_SET_MAC_V3_IN_TARGET_FLAT_LO_LEN 4
#define       MC_CMD_SET_MAC_V3_IN_TARGET_FLAT_LO_LBN 256
#define       MC_CMD_SET_MAC_V3_IN_TARGET_FLAT_LO_WIDTH 32
#define       MC_CMD_SET_MAC_V3_IN_TARGET_FLAT_HI_OFST 36
#define       MC_CMD_SET_MAC_V3_IN_TARGET_FLAT_HI_LEN 4
#define       MC_CMD_SET_MAC_V3_IN_TARGET_FLAT_HI_LBN 288
#define       MC_CMD_SET_MAC_V3_IN_TARGET_FLAT_HI_WIDTH 32

/* MC_CMD_SET_MAC_OUT msgresponse */
#define    MC_CMD_SET_MAC_OUT_LEN 0

/* MC_CMD_SET_MAC_V2_OUT msgresponse */
#define    MC_CMD_SET_MAC_V2_OUT_LEN 4
/* MTU as configured after processing the request. See comment at
 * MC_CMD_SET_MAC_IN/MTU. To query MTU without doing any changes, set CONTROL
 * to 0.
 */
#define       MC_CMD_SET_MAC_V2_OUT_MTU_OFST 0
#define       MC_CMD_SET_MAC_V2_OUT_MTU_LEN 4


/***********************************/
/* MC_CMD_PHY_STATS
 * Get generic PHY statistics. This call returns the statistics for a generic
 * PHY in a sparse array (indexed by the enumerate). Each value is represented
 * by a 32bit number. If the DMA_ADDR is 0, then no DMA is performed, and the
 * statistics may be read from the message response. If DMA_ADDR != 0, then the
 * statistics are dmad to that (page-aligned location). Locks required: None.
 * Returns: 0, ETIME
 */
#define MC_CMD_PHY_STATS 0x2d
#undef MC_CMD_0x2d_PRIVILEGE_CTG

#define MC_CMD_0x2d_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_PHY_STATS_IN msgrequest */
#define    MC_CMD_PHY_STATS_IN_LEN 8
/* ??? */
#define       MC_CMD_PHY_STATS_IN_DMA_ADDR_OFST 0
#define       MC_CMD_PHY_STATS_IN_DMA_ADDR_LEN 8
#define       MC_CMD_PHY_STATS_IN_DMA_ADDR_LO_OFST 0
#define       MC_CMD_PHY_STATS_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_PHY_STATS_IN_DMA_ADDR_LO_LBN 0
#define       MC_CMD_PHY_STATS_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_PHY_STATS_IN_DMA_ADDR_HI_OFST 4
#define       MC_CMD_PHY_STATS_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_PHY_STATS_IN_DMA_ADDR_HI_LBN 32
#define       MC_CMD_PHY_STATS_IN_DMA_ADDR_HI_WIDTH 32

/* MC_CMD_PHY_STATS_OUT_DMA msgresponse */
#define    MC_CMD_PHY_STATS_OUT_DMA_LEN 0

/* MC_CMD_PHY_STATS_OUT_NO_DMA msgresponse */
#define    MC_CMD_PHY_STATS_OUT_NO_DMA_LEN (((MC_CMD_PHY_NSTATS*32))>>3)
#define       MC_CMD_PHY_STATS_OUT_NO_DMA_STATISTICS_OFST 0
#define       MC_CMD_PHY_STATS_OUT_NO_DMA_STATISTICS_LEN 4
#define       MC_CMD_PHY_STATS_OUT_NO_DMA_STATISTICS_NUM MC_CMD_PHY_NSTATS
/* enum: OUI. */
#define          MC_CMD_OUI 0x0
/* enum: PMA-PMD Link Up. */
#define          MC_CMD_PMA_PMD_LINK_UP 0x1
/* enum: PMA-PMD RX Fault. */
#define          MC_CMD_PMA_PMD_RX_FAULT 0x2
/* enum: PMA-PMD TX Fault. */
#define          MC_CMD_PMA_PMD_TX_FAULT 0x3
/* enum: PMA-PMD Signal */
#define          MC_CMD_PMA_PMD_SIGNAL 0x4
/* enum: PMA-PMD SNR A. */
#define          MC_CMD_PMA_PMD_SNR_A 0x5
/* enum: PMA-PMD SNR B. */
#define          MC_CMD_PMA_PMD_SNR_B 0x6
/* enum: PMA-PMD SNR C. */
#define          MC_CMD_PMA_PMD_SNR_C 0x7
/* enum: PMA-PMD SNR D. */
#define          MC_CMD_PMA_PMD_SNR_D 0x8
/* enum: PCS Link Up. */
#define          MC_CMD_PCS_LINK_UP 0x9
/* enum: PCS RX Fault. */
#define          MC_CMD_PCS_RX_FAULT 0xa
/* enum: PCS TX Fault. */
#define          MC_CMD_PCS_TX_FAULT 0xb
/* enum: PCS BER. */
#define          MC_CMD_PCS_BER 0xc
/* enum: PCS Block Errors. */
#define          MC_CMD_PCS_BLOCK_ERRORS 0xd
/* enum: PhyXS Link Up. */
#define          MC_CMD_PHYXS_LINK_UP 0xe
/* enum: PhyXS RX Fault. */
#define          MC_CMD_PHYXS_RX_FAULT 0xf
/* enum: PhyXS TX Fault. */
#define          MC_CMD_PHYXS_TX_FAULT 0x10
/* enum: PhyXS Align. */
#define          MC_CMD_PHYXS_ALIGN 0x11
/* enum: PhyXS Sync. */
#define          MC_CMD_PHYXS_SYNC 0x12
/* enum: AN link-up. */
#define          MC_CMD_AN_LINK_UP 0x13
/* enum: AN Complete. */
#define          MC_CMD_AN_COMPLETE 0x14
/* enum: AN 10GBaseT Status. */
#define          MC_CMD_AN_10GBT_STATUS 0x15
/* enum: Clause 22 Link-Up. */
#define          MC_CMD_CL22_LINK_UP 0x16
/* enum: (Last entry) */
#define          MC_CMD_PHY_NSTATS 0x17


/***********************************/
/* MC_CMD_MAC_STATS
 * Get generic MAC statistics. This call returns unified statistics maintained
 * by the MC as it switches between the GMAC and XMAC. The MC will write out
 * all supported stats. The driver should zero initialise the buffer to
 * guarantee consistent results. If the DMA_ADDR is 0, then no DMA is
 * performed, and the statistics may be read from the message response. If
 * DMA_ADDR != 0, then the statistics are dmad to that (page-aligned location).
 * Locks required: None. The PERIODIC_CLEAR option is not used and now has no
 * effect. Returns: 0, ETIME
 */
#define MC_CMD_MAC_STATS 0x2e
#undef MC_CMD_0x2e_PRIVILEGE_CTG

#define MC_CMD_0x2e_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_MAC_STATS_IN msgrequest */
#define    MC_CMD_MAC_STATS_IN_LEN 20
/* ??? */
#define       MC_CMD_MAC_STATS_IN_DMA_ADDR_OFST 0
#define       MC_CMD_MAC_STATS_IN_DMA_ADDR_LEN 8
#define       MC_CMD_MAC_STATS_IN_DMA_ADDR_LO_OFST 0
#define       MC_CMD_MAC_STATS_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_MAC_STATS_IN_DMA_ADDR_LO_LBN 0
#define       MC_CMD_MAC_STATS_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_MAC_STATS_IN_DMA_ADDR_HI_OFST 4
#define       MC_CMD_MAC_STATS_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_MAC_STATS_IN_DMA_ADDR_HI_LBN 32
#define       MC_CMD_MAC_STATS_IN_DMA_ADDR_HI_WIDTH 32
#define       MC_CMD_MAC_STATS_IN_CMD_OFST 8
#define       MC_CMD_MAC_STATS_IN_CMD_LEN 4
#define        MC_CMD_MAC_STATS_IN_DMA_OFST 8
#define        MC_CMD_MAC_STATS_IN_DMA_LBN 0
#define        MC_CMD_MAC_STATS_IN_DMA_WIDTH 1
#define        MC_CMD_MAC_STATS_IN_CLEAR_OFST 8
#define        MC_CMD_MAC_STATS_IN_CLEAR_LBN 1
#define        MC_CMD_MAC_STATS_IN_CLEAR_WIDTH 1
#define        MC_CMD_MAC_STATS_IN_PERIODIC_CHANGE_OFST 8
#define        MC_CMD_MAC_STATS_IN_PERIODIC_CHANGE_LBN 2
#define        MC_CMD_MAC_STATS_IN_PERIODIC_CHANGE_WIDTH 1
#define        MC_CMD_MAC_STATS_IN_PERIODIC_ENABLE_OFST 8
#define        MC_CMD_MAC_STATS_IN_PERIODIC_ENABLE_LBN 3
#define        MC_CMD_MAC_STATS_IN_PERIODIC_ENABLE_WIDTH 1
#define        MC_CMD_MAC_STATS_IN_PERIODIC_CLEAR_OFST 8
#define        MC_CMD_MAC_STATS_IN_PERIODIC_CLEAR_LBN 4
#define        MC_CMD_MAC_STATS_IN_PERIODIC_CLEAR_WIDTH 1
#define        MC_CMD_MAC_STATS_IN_PERIODIC_NOEVENT_OFST 8
#define        MC_CMD_MAC_STATS_IN_PERIODIC_NOEVENT_LBN 5
#define        MC_CMD_MAC_STATS_IN_PERIODIC_NOEVENT_WIDTH 1
#define        MC_CMD_MAC_STATS_IN_PERIOD_MS_OFST 8
#define        MC_CMD_MAC_STATS_IN_PERIOD_MS_LBN 16
#define        MC_CMD_MAC_STATS_IN_PERIOD_MS_WIDTH 16
/* DMA length. Should be set to MAC_STATS_NUM_STATS * sizeof(uint64_t), as
 * returned by MC_CMD_GET_CAPABILITIES_V4_OUT. For legacy firmware not
 * supporting MC_CMD_GET_CAPABILITIES_V4_OUT, DMA_LEN should be set to
 * MC_CMD_MAC_NSTATS * sizeof(uint64_t)
 */
#define       MC_CMD_MAC_STATS_IN_DMA_LEN_OFST 12
#define       MC_CMD_MAC_STATS_IN_DMA_LEN_LEN 4
/* port id so vadapter stats can be provided */
#define       MC_CMD_MAC_STATS_IN_PORT_ID_OFST 16
#define       MC_CMD_MAC_STATS_IN_PORT_ID_LEN 4

/* MC_CMD_MAC_STATS_V2_IN msgrequest */
#define    MC_CMD_MAC_STATS_V2_IN_LEN 28
/* ??? */
#define       MC_CMD_MAC_STATS_V2_IN_DMA_ADDR_OFST 0
#define       MC_CMD_MAC_STATS_V2_IN_DMA_ADDR_LEN 8
#define       MC_CMD_MAC_STATS_V2_IN_DMA_ADDR_LO_OFST 0
#define       MC_CMD_MAC_STATS_V2_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_MAC_STATS_V2_IN_DMA_ADDR_LO_LBN 0
#define       MC_CMD_MAC_STATS_V2_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_MAC_STATS_V2_IN_DMA_ADDR_HI_OFST 4
#define       MC_CMD_MAC_STATS_V2_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_MAC_STATS_V2_IN_DMA_ADDR_HI_LBN 32
#define       MC_CMD_MAC_STATS_V2_IN_DMA_ADDR_HI_WIDTH 32
#define       MC_CMD_MAC_STATS_V2_IN_CMD_OFST 8
#define       MC_CMD_MAC_STATS_V2_IN_CMD_LEN 4
#define        MC_CMD_MAC_STATS_V2_IN_DMA_OFST 8
#define        MC_CMD_MAC_STATS_V2_IN_DMA_LBN 0
#define        MC_CMD_MAC_STATS_V2_IN_DMA_WIDTH 1
#define        MC_CMD_MAC_STATS_V2_IN_CLEAR_OFST 8
#define        MC_CMD_MAC_STATS_V2_IN_CLEAR_LBN 1
#define        MC_CMD_MAC_STATS_V2_IN_CLEAR_WIDTH 1
#define        MC_CMD_MAC_STATS_V2_IN_PERIODIC_CHANGE_OFST 8
#define        MC_CMD_MAC_STATS_V2_IN_PERIODIC_CHANGE_LBN 2
#define        MC_CMD_MAC_STATS_V2_IN_PERIODIC_CHANGE_WIDTH 1
#define        MC_CMD_MAC_STATS_V2_IN_PERIODIC_ENABLE_OFST 8
#define        MC_CMD_MAC_STATS_V2_IN_PERIODIC_ENABLE_LBN 3
#define        MC_CMD_MAC_STATS_V2_IN_PERIODIC_ENABLE_WIDTH 1
#define        MC_CMD_MAC_STATS_V2_IN_PERIODIC_CLEAR_OFST 8
#define        MC_CMD_MAC_STATS_V2_IN_PERIODIC_CLEAR_LBN 4
#define        MC_CMD_MAC_STATS_V2_IN_PERIODIC_CLEAR_WIDTH 1
#define        MC_CMD_MAC_STATS_V2_IN_PERIODIC_NOEVENT_OFST 8
#define        MC_CMD_MAC_STATS_V2_IN_PERIODIC_NOEVENT_LBN 5
#define        MC_CMD_MAC_STATS_V2_IN_PERIODIC_NOEVENT_WIDTH 1
#define        MC_CMD_MAC_STATS_V2_IN_PERIOD_MS_OFST 8
#define        MC_CMD_MAC_STATS_V2_IN_PERIOD_MS_LBN 16
#define        MC_CMD_MAC_STATS_V2_IN_PERIOD_MS_WIDTH 16
/* DMA length. Should be set to MAC_STATS_NUM_STATS * sizeof(uint64_t), as
 * returned by MC_CMD_GET_CAPABILITIES_V4_OUT. For legacy firmware not
 * supporting MC_CMD_GET_CAPABILITIES_V4_OUT, DMA_LEN should be set to
 * MC_CMD_MAC_NSTATS * sizeof(uint64_t)
 */
#define       MC_CMD_MAC_STATS_V2_IN_DMA_LEN_OFST 12
#define       MC_CMD_MAC_STATS_V2_IN_DMA_LEN_LEN 4
/* port id so vadapter stats can be provided */
#define       MC_CMD_MAC_STATS_V2_IN_PORT_ID_OFST 16
#define       MC_CMD_MAC_STATS_V2_IN_PORT_ID_LEN 4
/* Target port to request statistics for. Uses MAE_LINK_ENDPOINT_SELECTOR which
 * identifies a real or virtual network port by MAE port and link end. See the
 * structure definition for more details
 */
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_OFST 20
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_LEN 8
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_LO_OFST 20
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_LO_LEN 4
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_LO_LBN 160
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_LO_WIDTH 32
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_HI_OFST 24
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_HI_LEN 4
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_HI_LBN 192
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_HI_WIDTH 32
/* See structuredef: MAE_LINK_ENDPOINT_SELECTOR */
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_OFST 20
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_LEN 4
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_FLAT_OFST 20
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_FLAT_LEN 4
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_TYPE_OFST 23
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_TYPE_LEN 1
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_MPORT_ID_OFST 20
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_MPORT_ID_LEN 3
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_PPORT_ID_LBN 160
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_PPORT_ID_WIDTH 4
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_FUNC_INTF_ID_LBN 180
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_FUNC_INTF_ID_WIDTH 4
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_FUNC_MH_PF_ID_LBN 176
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_FUNC_MH_PF_ID_WIDTH 4
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_FUNC_PF_ID_OFST 22
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_FUNC_PF_ID_LEN 1
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_FUNC_VF_ID_OFST 20
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_MPORT_SELECTOR_FUNC_VF_ID_LEN 2
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_LINK_END_OFST 24
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_LINK_END_LEN 4
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_FLAT_OFST 20
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_FLAT_LEN 8
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_FLAT_LO_OFST 20
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_FLAT_LO_LEN 4
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_FLAT_LO_LBN 160
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_FLAT_LO_WIDTH 32
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_FLAT_HI_OFST 24
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_FLAT_HI_LEN 4
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_FLAT_HI_LBN 192
#define       MC_CMD_MAC_STATS_V2_IN_TARGET_FLAT_HI_WIDTH 32

/* MC_CMD_MAC_STATS_OUT_DMA msgresponse */
#define    MC_CMD_MAC_STATS_OUT_DMA_LEN 0

/* MC_CMD_MAC_STATS_OUT_NO_DMA msgresponse */
#define    MC_CMD_MAC_STATS_OUT_NO_DMA_LEN (((MC_CMD_MAC_NSTATS*64))>>3)
#define       MC_CMD_MAC_STATS_OUT_NO_DMA_STATISTICS_OFST 0
#define       MC_CMD_MAC_STATS_OUT_NO_DMA_STATISTICS_LEN 8
#define       MC_CMD_MAC_STATS_OUT_NO_DMA_STATISTICS_LO_OFST 0
#define       MC_CMD_MAC_STATS_OUT_NO_DMA_STATISTICS_LO_LEN 4
#define       MC_CMD_MAC_STATS_OUT_NO_DMA_STATISTICS_LO_LBN 0
#define       MC_CMD_MAC_STATS_OUT_NO_DMA_STATISTICS_LO_WIDTH 32
#define       MC_CMD_MAC_STATS_OUT_NO_DMA_STATISTICS_HI_OFST 4
#define       MC_CMD_MAC_STATS_OUT_NO_DMA_STATISTICS_HI_LEN 4
#define       MC_CMD_MAC_STATS_OUT_NO_DMA_STATISTICS_HI_LBN 32
#define       MC_CMD_MAC_STATS_OUT_NO_DMA_STATISTICS_HI_WIDTH 32
#define       MC_CMD_MAC_STATS_OUT_NO_DMA_STATISTICS_NUM MC_CMD_MAC_NSTATS
/* enum property: index */
#define          MC_CMD_MAC_GENERATION_START 0x0 /* enum */
#define          MC_CMD_MAC_DMABUF_START 0x1 /* enum */
#define          MC_CMD_MAC_TX_PKTS 0x1 /* enum */
#define          MC_CMD_MAC_TX_PAUSE_PKTS 0x2 /* enum */
#define          MC_CMD_MAC_TX_CONTROL_PKTS 0x3 /* enum */
#define          MC_CMD_MAC_TX_UNICAST_PKTS 0x4 /* enum */
#define          MC_CMD_MAC_TX_MULTICAST_PKTS 0x5 /* enum */
#define          MC_CMD_MAC_TX_BROADCAST_PKTS 0x6 /* enum */
#define          MC_CMD_MAC_TX_BYTES 0x7 /* enum */
#define          MC_CMD_MAC_TX_BAD_BYTES 0x8 /* enum */
#define          MC_CMD_MAC_TX_LT64_PKTS 0x9 /* enum */
#define          MC_CMD_MAC_TX_64_PKTS 0xa /* enum */
#define          MC_CMD_MAC_TX_65_TO_127_PKTS 0xb /* enum */
#define          MC_CMD_MAC_TX_128_TO_255_PKTS 0xc /* enum */
#define          MC_CMD_MAC_TX_256_TO_511_PKTS 0xd /* enum */
#define          MC_CMD_MAC_TX_512_TO_1023_PKTS 0xe /* enum */
#define          MC_CMD_MAC_TX_1024_TO_15XX_PKTS 0xf /* enum */
#define          MC_CMD_MAC_TX_15XX_TO_JUMBO_PKTS 0x10 /* enum */
#define          MC_CMD_MAC_TX_GTJUMBO_PKTS 0x11 /* enum */
#define          MC_CMD_MAC_TX_BAD_FCS_PKTS 0x12 /* enum */
#define          MC_CMD_MAC_TX_SINGLE_COLLISION_PKTS 0x13 /* enum */
#define          MC_CMD_MAC_TX_MULTIPLE_COLLISION_PKTS 0x14 /* enum */
#define          MC_CMD_MAC_TX_EXCESSIVE_COLLISION_PKTS 0x15 /* enum */
#define          MC_CMD_MAC_TX_LATE_COLLISION_PKTS 0x16 /* enum */
#define          MC_CMD_MAC_TX_DEFERRED_PKTS 0x17 /* enum */
#define          MC_CMD_MAC_TX_EXCESSIVE_DEFERRED_PKTS 0x18 /* enum */
#define          MC_CMD_MAC_TX_NON_TCPUDP_PKTS 0x19 /* enum */
#define          MC_CMD_MAC_TX_MAC_SRC_ERR_PKTS 0x1a /* enum */
#define          MC_CMD_MAC_TX_IP_SRC_ERR_PKTS 0x1b /* enum */
#define          MC_CMD_MAC_RX_PKTS 0x1c /* enum */
#define          MC_CMD_MAC_RX_PAUSE_PKTS 0x1d /* enum */
#define          MC_CMD_MAC_RX_GOOD_PKTS 0x1e /* enum */
#define          MC_CMD_MAC_RX_CONTROL_PKTS 0x1f /* enum */
#define          MC_CMD_MAC_RX_UNICAST_PKTS 0x20 /* enum */
#define          MC_CMD_MAC_RX_MULTICAST_PKTS 0x21 /* enum */
#define          MC_CMD_MAC_RX_BROADCAST_PKTS 0x22 /* enum */
#define          MC_CMD_MAC_RX_BYTES 0x23 /* enum */
#define          MC_CMD_MAC_RX_BAD_BYTES 0x24 /* enum */
#define          MC_CMD_MAC_RX_64_PKTS 0x25 /* enum */
#define          MC_CMD_MAC_RX_65_TO_127_PKTS 0x26 /* enum */
#define          MC_CMD_MAC_RX_128_TO_255_PKTS 0x27 /* enum */
#define          MC_CMD_MAC_RX_256_TO_511_PKTS 0x28 /* enum */
#define          MC_CMD_MAC_RX_512_TO_1023_PKTS 0x29 /* enum */
#define          MC_CMD_MAC_RX_1024_TO_15XX_PKTS 0x2a /* enum */
#define          MC_CMD_MAC_RX_15XX_TO_JUMBO_PKTS 0x2b /* enum */
#define          MC_CMD_MAC_RX_GTJUMBO_PKTS 0x2c /* enum */
#define          MC_CMD_MAC_RX_UNDERSIZE_PKTS 0x2d /* enum */
#define          MC_CMD_MAC_RX_BAD_FCS_PKTS 0x2e /* enum */
#define          MC_CMD_MAC_RX_OVERFLOW_PKTS 0x2f /* enum */
#define          MC_CMD_MAC_RX_FALSE_CARRIER_PKTS 0x30 /* enum */
#define          MC_CMD_MAC_RX_SYMBOL_ERROR_PKTS 0x31 /* enum */
#define          MC_CMD_MAC_RX_ALIGN_ERROR_PKTS 0x32 /* enum */
#define          MC_CMD_MAC_RX_LENGTH_ERROR_PKTS 0x33 /* enum */
#define          MC_CMD_MAC_RX_INTERNAL_ERROR_PKTS 0x34 /* enum */
#define          MC_CMD_MAC_RX_JABBER_PKTS 0x35 /* enum */
#define          MC_CMD_MAC_RX_NODESC_DROPS 0x36 /* enum */
#define          MC_CMD_MAC_RX_LANES01_CHAR_ERR 0x37 /* enum */
#define          MC_CMD_MAC_RX_LANES23_CHAR_ERR 0x38 /* enum */
#define          MC_CMD_MAC_RX_LANES01_DISP_ERR 0x39 /* enum */
#define          MC_CMD_MAC_RX_LANES23_DISP_ERR 0x3a /* enum */
#define          MC_CMD_MAC_RX_MATCH_FAULT 0x3b /* enum */
/* enum: PM trunc_bb_overflow counter. Valid for EF10 with PM_AND_RXDP_COUNTERS
 * capability only.
 */
#define          MC_CMD_MAC_PM_TRUNC_BB_OVERFLOW 0x3c
/* enum: PM discard_bb_overflow counter. Valid for EF10 with
 * PM_AND_RXDP_COUNTERS capability only.
 */
#define          MC_CMD_MAC_PM_DISCARD_BB_OVERFLOW 0x3d
/* enum: PM trunc_vfifo_full counter. Valid for EF10 with PM_AND_RXDP_COUNTERS
 * capability only.
 */
#define          MC_CMD_MAC_PM_TRUNC_VFIFO_FULL 0x3e
/* enum: PM discard_vfifo_full counter. Valid for EF10 with
 * PM_AND_RXDP_COUNTERS capability only.
 */
#define          MC_CMD_MAC_PM_DISCARD_VFIFO_FULL 0x3f
/* enum: PM trunc_qbb counter. Valid for EF10 with PM_AND_RXDP_COUNTERS
 * capability only.
 */
#define          MC_CMD_MAC_PM_TRUNC_QBB 0x40
/* enum: PM discard_qbb counter. Valid for EF10 with PM_AND_RXDP_COUNTERS
 * capability only.
 */
#define          MC_CMD_MAC_PM_DISCARD_QBB 0x41
/* enum: PM discard_mapping counter. Valid for EF10 with PM_AND_RXDP_COUNTERS
 * capability only.
 */
#define          MC_CMD_MAC_PM_DISCARD_MAPPING 0x42
/* enum: RXDP counter: Number of packets dropped due to the queue being
 * disabled. Valid for EF10 with PM_AND_RXDP_COUNTERS capability only.
 */
#define          MC_CMD_MAC_RXDP_Q_DISABLED_PKTS 0x43
/* enum: RXDP counter: Number of packets dropped by the DICPU. Valid for EF10
 * with PM_AND_RXDP_COUNTERS capability only.
 */
#define          MC_CMD_MAC_RXDP_DI_DROPPED_PKTS 0x45
/* enum: RXDP counter: Number of non-host packets. Valid for EF10 with
 * PM_AND_RXDP_COUNTERS capability only.
 */
#define          MC_CMD_MAC_RXDP_STREAMING_PKTS 0x46
/* enum: RXDP counter: Number of times an hlb descriptor fetch was performed.
 * Valid for EF10 with PM_AND_RXDP_COUNTERS capability only.
 */
#define          MC_CMD_MAC_RXDP_HLB_FETCH_CONDITIONS 0x47
/* enum: RXDP counter: Number of times the DPCPU waited for an existing
 * descriptor fetch. Valid for EF10 with PM_AND_RXDP_COUNTERS capability only.
 */
#define          MC_CMD_MAC_RXDP_HLB_WAIT_CONDITIONS 0x48
#define          MC_CMD_MAC_VADAPTER_RX_DMABUF_START 0x4c /* enum */
#define          MC_CMD_MAC_VADAPTER_RX_UNICAST_PACKETS 0x4c /* enum */
#define          MC_CMD_MAC_VADAPTER_RX_UNICAST_BYTES 0x4d /* enum */
#define          MC_CMD_MAC_VADAPTER_RX_MULTICAST_PACKETS 0x4e /* enum */
#define          MC_CMD_MAC_VADAPTER_RX_MULTICAST_BYTES 0x4f /* enum */
#define          MC_CMD_MAC_VADAPTER_RX_BROADCAST_PACKETS 0x50 /* enum */
#define          MC_CMD_MAC_VADAPTER_RX_BROADCAST_BYTES 0x51 /* enum */
#define          MC_CMD_MAC_VADAPTER_RX_BAD_PACKETS 0x52 /* enum */
#define          MC_CMD_MAC_VADAPTER_RX_BAD_BYTES 0x53 /* enum */
#define          MC_CMD_MAC_VADAPTER_RX_OVERFLOW 0x54 /* enum */
#define          MC_CMD_MAC_VADAPTER_TX_DMABUF_START 0x57 /* enum */
#define          MC_CMD_MAC_VADAPTER_TX_UNICAST_PACKETS 0x57 /* enum */
#define          MC_CMD_MAC_VADAPTER_TX_UNICAST_BYTES 0x58 /* enum */
#define          MC_CMD_MAC_VADAPTER_TX_MULTICAST_PACKETS 0x59 /* enum */
#define          MC_CMD_MAC_VADAPTER_TX_MULTICAST_BYTES 0x5a /* enum */
#define          MC_CMD_MAC_VADAPTER_TX_BROADCAST_PACKETS 0x5b /* enum */
#define          MC_CMD_MAC_VADAPTER_TX_BROADCAST_BYTES 0x5c /* enum */
#define          MC_CMD_MAC_VADAPTER_TX_BAD_PACKETS 0x5d /* enum */
#define          MC_CMD_MAC_VADAPTER_TX_BAD_BYTES 0x5e /* enum */
#define          MC_CMD_MAC_VADAPTER_TX_OVERFLOW 0x5f /* enum */
/* enum: Start of GMAC stats buffer space, for Siena only. */
#define          MC_CMD_GMAC_DMABUF_START 0x40
/* enum: End of GMAC stats buffer space, for Siena only. */
#define          MC_CMD_GMAC_DMABUF_END 0x5f
/* enum: GENERATION_END value, used together with GENERATION_START to verify
 * consistency of DMAd data. For legacy firmware / drivers without extended
 * stats (more precisely, when DMA_LEN == MC_CMD_MAC_NSTATS *
 * sizeof(uint64_t)), this entry holds the GENERATION_END value. Otherwise,
 * this value is invalid/ reserved and GENERATION_END is written as the last
 * 64-bit word of the DMA buffer (at DMA_LEN - sizeof(uint64_t)). Note that
 * this is consistent with the legacy behaviour, in the sense that entry 96 is
 * the last 64-bit word in the buffer when DMA_LEN == MC_CMD_MAC_NSTATS *
 * sizeof(uint64_t). See SF-109306-TC, Section 9.2 for details.
 */
#define          MC_CMD_MAC_GENERATION_END 0x60
#define          MC_CMD_MAC_NSTATS 0x61 /* enum */

/* MC_CMD_MAC_STATS_V2_OUT_DMA msgresponse */
#define    MC_CMD_MAC_STATS_V2_OUT_DMA_LEN 0

/* MC_CMD_MAC_STATS_V2_OUT_NO_DMA msgresponse */
#define    MC_CMD_MAC_STATS_V2_OUT_NO_DMA_LEN (((MC_CMD_MAC_NSTATS_V2*64))>>3)
#define       MC_CMD_MAC_STATS_V2_OUT_NO_DMA_STATISTICS_OFST 0
#define       MC_CMD_MAC_STATS_V2_OUT_NO_DMA_STATISTICS_LEN 8
#define       MC_CMD_MAC_STATS_V2_OUT_NO_DMA_STATISTICS_LO_OFST 0
#define       MC_CMD_MAC_STATS_V2_OUT_NO_DMA_STATISTICS_LO_LEN 4
#define       MC_CMD_MAC_STATS_V2_OUT_NO_DMA_STATISTICS_LO_LBN 0
#define       MC_CMD_MAC_STATS_V2_OUT_NO_DMA_STATISTICS_LO_WIDTH 32
#define       MC_CMD_MAC_STATS_V2_OUT_NO_DMA_STATISTICS_HI_OFST 4
#define       MC_CMD_MAC_STATS_V2_OUT_NO_DMA_STATISTICS_HI_LEN 4
#define       MC_CMD_MAC_STATS_V2_OUT_NO_DMA_STATISTICS_HI_LBN 32
#define       MC_CMD_MAC_STATS_V2_OUT_NO_DMA_STATISTICS_HI_WIDTH 32
#define       MC_CMD_MAC_STATS_V2_OUT_NO_DMA_STATISTICS_NUM MC_CMD_MAC_NSTATS_V2
/* enum property: index */
/* enum: Start of FEC stats buffer space, Medford2 and up */
#define          MC_CMD_MAC_FEC_DMABUF_START 0x61
/* enum: Number of uncorrected FEC codewords on link (RS-FEC only for Medford2)
 */
#define          MC_CMD_MAC_FEC_UNCORRECTED_ERRORS 0x61
/* enum: Number of corrected FEC codewords on link (RS-FEC only for Medford2)
 */
#define          MC_CMD_MAC_FEC_CORRECTED_ERRORS 0x62
/* enum: Number of corrected 10-bit symbol errors, lane 0 (RS-FEC only) */
#define          MC_CMD_MAC_FEC_CORRECTED_SYMBOLS_LANE0 0x63
/* enum: Number of corrected 10-bit symbol errors, lane 1 (RS-FEC only) */
#define          MC_CMD_MAC_FEC_CORRECTED_SYMBOLS_LANE1 0x64
/* enum: Number of corrected 10-bit symbol errors, lane 2 (RS-FEC only) */
#define          MC_CMD_MAC_FEC_CORRECTED_SYMBOLS_LANE2 0x65
/* enum: Number of corrected 10-bit symbol errors, lane 3 (RS-FEC only) */
#define          MC_CMD_MAC_FEC_CORRECTED_SYMBOLS_LANE3 0x66
/* enum: This includes the space at offset 103 which is the final
 * GENERATION_END in a MAC_STATS_V2 response and otherwise unused.
 */
#define          MC_CMD_MAC_NSTATS_V2 0x68
/*            Other enum values, see field(s): */
/*               MC_CMD_MAC_STATS_OUT_NO_DMA/STATISTICS */

/* MC_CMD_MAC_STATS_V3_OUT_DMA msgresponse */
#define    MC_CMD_MAC_STATS_V3_OUT_DMA_LEN 0

/* MC_CMD_MAC_STATS_V3_OUT_NO_DMA msgresponse */
#define    MC_CMD_MAC_STATS_V3_OUT_NO_DMA_LEN (((MC_CMD_MAC_NSTATS_V3*64))>>3)
#define       MC_CMD_MAC_STATS_V3_OUT_NO_DMA_STATISTICS_OFST 0
#define       MC_CMD_MAC_STATS_V3_OUT_NO_DMA_STATISTICS_LEN 8
#define       MC_CMD_MAC_STATS_V3_OUT_NO_DMA_STATISTICS_LO_OFST 0
#define       MC_CMD_MAC_STATS_V3_OUT_NO_DMA_STATISTICS_LO_LEN 4
#define       MC_CMD_MAC_STATS_V3_OUT_NO_DMA_STATISTICS_LO_LBN 0
#define       MC_CMD_MAC_STATS_V3_OUT_NO_DMA_STATISTICS_LO_WIDTH 32
#define       MC_CMD_MAC_STATS_V3_OUT_NO_DMA_STATISTICS_HI_OFST 4
#define       MC_CMD_MAC_STATS_V3_OUT_NO_DMA_STATISTICS_HI_LEN 4
#define       MC_CMD_MAC_STATS_V3_OUT_NO_DMA_STATISTICS_HI_LBN 32
#define       MC_CMD_MAC_STATS_V3_OUT_NO_DMA_STATISTICS_HI_WIDTH 32
#define       MC_CMD_MAC_STATS_V3_OUT_NO_DMA_STATISTICS_NUM MC_CMD_MAC_NSTATS_V3
/* enum property: index */
/* enum: Start of CTPIO stats buffer space, Medford2 and up */
#define          MC_CMD_MAC_CTPIO_DMABUF_START 0x68
/* enum: Number of CTPIO fallbacks because a DMA packet was in progress on the
 * target VI
 */
#define          MC_CMD_MAC_CTPIO_VI_BUSY_FALLBACK 0x68
/* enum: Number of times a CTPIO send wrote beyond frame end (informational
 * only)
 */
#define          MC_CMD_MAC_CTPIO_LONG_WRITE_SUCCESS 0x69
/* enum: Number of CTPIO failures because the TX doorbell was written before
 * the end of the frame data
 */
#define          MC_CMD_MAC_CTPIO_MISSING_DBELL_FAIL 0x6a
/* enum: Number of CTPIO failures because the internal FIFO overflowed */
#define          MC_CMD_MAC_CTPIO_OVERFLOW_FAIL 0x6b
/* enum: Number of CTPIO failures because the host did not deliver data fast
 * enough to avoid MAC underflow
 */
#define          MC_CMD_MAC_CTPIO_UNDERFLOW_FAIL 0x6c
/* enum: Number of CTPIO failures because the host did not deliver all the
 * frame data within the timeout
 */
#define          MC_CMD_MAC_CTPIO_TIMEOUT_FAIL 0x6d
/* enum: Number of CTPIO failures because the frame data arrived out of order
 * or with gaps
 */
#define          MC_CMD_MAC_CTPIO_NONCONTIG_WR_FAIL 0x6e
/* enum: Number of CTPIO failures because the host started a new frame before
 * completing the previous one
 */
#define          MC_CMD_MAC_CTPIO_FRM_CLOBBER_FAIL 0x6f
/* enum: Number of CTPIO failures because a write was not a multiple of 32 bits
 * or not 32-bit aligned
 */
#define          MC_CMD_MAC_CTPIO_INVALID_WR_FAIL 0x70
/* enum: Number of CTPIO fallbacks because another VI on the same port was
 * sending a CTPIO frame
 */
#define          MC_CMD_MAC_CTPIO_VI_CLOBBER_FALLBACK 0x71
/* enum: Number of CTPIO fallbacks because target VI did not have CTPIO enabled
 */
#define          MC_CMD_MAC_CTPIO_UNQUALIFIED_FALLBACK 0x72
/* enum: Number of CTPIO fallbacks because length in header was less than 29
 * bytes
 */
#define          MC_CMD_MAC_CTPIO_RUNT_FALLBACK 0x73
/* enum: Total number of successful CTPIO sends on this port */
#define          MC_CMD_MAC_CTPIO_SUCCESS 0x74
/* enum: Total number of CTPIO fallbacks on this port */
#define          MC_CMD_MAC_CTPIO_FALLBACK 0x75
/* enum: Total number of CTPIO poisoned frames on this port, whether erased or
 * not
 */
#define          MC_CMD_MAC_CTPIO_POISON 0x76
/* enum: Total number of CTPIO erased frames on this port */
#define          MC_CMD_MAC_CTPIO_ERASE 0x77
/* enum: This includes the space at offset 120 which is the final
 * GENERATION_END in a MAC_STATS_V3 response and otherwise unused.
 */
#define          MC_CMD_MAC_NSTATS_V3 0x79
/*            Other enum values, see field(s): */
/*               MC_CMD_MAC_STATS_V2_OUT_NO_DMA/STATISTICS */

/* MC_CMD_MAC_STATS_V4_OUT_DMA msgresponse */
#define    MC_CMD_MAC_STATS_V4_OUT_DMA_LEN 0

/* MC_CMD_MAC_STATS_V4_OUT_NO_DMA msgresponse */
#define    MC_CMD_MAC_STATS_V4_OUT_NO_DMA_LEN (((MC_CMD_MAC_NSTATS_V4*64))>>3)
#define       MC_CMD_MAC_STATS_V4_OUT_NO_DMA_STATISTICS_OFST 0
#define       MC_CMD_MAC_STATS_V4_OUT_NO_DMA_STATISTICS_LEN 8
#define       MC_CMD_MAC_STATS_V4_OUT_NO_DMA_STATISTICS_LO_OFST 0
#define       MC_CMD_MAC_STATS_V4_OUT_NO_DMA_STATISTICS_LO_LEN 4
#define       MC_CMD_MAC_STATS_V4_OUT_NO_DMA_STATISTICS_LO_LBN 0
#define       MC_CMD_MAC_STATS_V4_OUT_NO_DMA_STATISTICS_LO_WIDTH 32
#define       MC_CMD_MAC_STATS_V4_OUT_NO_DMA_STATISTICS_HI_OFST 4
#define       MC_CMD_MAC_STATS_V4_OUT_NO_DMA_STATISTICS_HI_LEN 4
#define       MC_CMD_MAC_STATS_V4_OUT_NO_DMA_STATISTICS_HI_LBN 32
#define       MC_CMD_MAC_STATS_V4_OUT_NO_DMA_STATISTICS_HI_WIDTH 32
#define       MC_CMD_MAC_STATS_V4_OUT_NO_DMA_STATISTICS_NUM MC_CMD_MAC_NSTATS_V4
/* enum property: index */
/* enum: Start of V4 stats buffer space */
#define          MC_CMD_MAC_V4_DMABUF_START 0x79
/* enum: RXDP counter: Number of packets truncated because scattering was
 * disabled.
 */
#define          MC_CMD_MAC_RXDP_SCATTER_DISABLED_TRUNC 0x79
/* enum: RXDP counter: Number of times the RXDP head of line blocked waiting
 * for descriptors. Will be zero unless RXDP_HLB_IDLE capability is set.
 */
#define          MC_CMD_MAC_RXDP_HLB_IDLE 0x7a
/* enum: RXDP counter: Number of times the RXDP timed out while head of line
 * blocking. Will be zero unless RXDP_HLB_IDLE capability is set.
 */
#define          MC_CMD_MAC_RXDP_HLB_TIMEOUT 0x7b
/* enum: This includes the space at offset 124 which is the final
 * GENERATION_END in a MAC_STATS_V4 response and otherwise unused.
 */
#define          MC_CMD_MAC_NSTATS_V4 0x7d
/*            Other enum values, see field(s): */
/*               MC_CMD_MAC_STATS_V3_OUT_NO_DMA/STATISTICS */

/* MC_CMD_MAC_STATS_V5_OUT_DMA msgresponse */
#define    MC_CMD_MAC_STATS_V5_OUT_DMA_LEN 0

/* MC_CMD_MAC_STATS_V5_OUT_NO_DMA msgresponse */
#define    MC_CMD_MAC_STATS_V5_OUT_NO_DMA_LEN (((MC_CMD_MAC_NSTATS_V5*64))>>3)
#define       MC_CMD_MAC_STATS_V5_OUT_NO_DMA_STATISTICS_OFST 0
#define       MC_CMD_MAC_STATS_V5_OUT_NO_DMA_STATISTICS_LEN 8
#define       MC_CMD_MAC_STATS_V5_OUT_NO_DMA_STATISTICS_LO_OFST 0
#define       MC_CMD_MAC_STATS_V5_OUT_NO_DMA_STATISTICS_LO_LEN 4
#define       MC_CMD_MAC_STATS_V5_OUT_NO_DMA_STATISTICS_LO_LBN 0
#define       MC_CMD_MAC_STATS_V5_OUT_NO_DMA_STATISTICS_LO_WIDTH 32
#define       MC_CMD_MAC_STATS_V5_OUT_NO_DMA_STATISTICS_HI_OFST 4
#define       MC_CMD_MAC_STATS_V5_OUT_NO_DMA_STATISTICS_HI_LEN 4
#define       MC_CMD_MAC_STATS_V5_OUT_NO_DMA_STATISTICS_HI_LBN 32
#define       MC_CMD_MAC_STATS_V5_OUT_NO_DMA_STATISTICS_HI_WIDTH 32
#define       MC_CMD_MAC_STATS_V5_OUT_NO_DMA_STATISTICS_NUM MC_CMD_MAC_NSTATS_V5
/* enum property: index */
/* enum: Start of V5 stats buffer space */
#define          MC_CMD_MAC_V5_DMABUF_START 0x7c
/* enum: Link toggle counter: Number of times the link has toggled between
 * up/down and down/up
 */
#define          MC_CMD_MAC_LINK_TOGGLES 0x7c
/* enum: This includes the space at offset 125 which is the final
 * GENERATION_END in a MAC_STATS_V5 response and otherwise unused.
 */
#define          MC_CMD_MAC_NSTATS_V5 0x7e
/*            Other enum values, see field(s): */
/*               MC_CMD_MAC_STATS_V4_OUT_NO_DMA/STATISTICS */


/***********************************/
/* MC_CMD_WOL_FILTER_SET
 * Set a WoL filter.
 */
#define MC_CMD_WOL_FILTER_SET 0x32
#undef MC_CMD_0x32_PRIVILEGE_CTG

#define MC_CMD_0x32_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_WOL_FILTER_SET_IN msgrequest */
#define    MC_CMD_WOL_FILTER_SET_IN_LEN 192
#define       MC_CMD_WOL_FILTER_SET_IN_FILTER_MODE_OFST 0
#define       MC_CMD_WOL_FILTER_SET_IN_FILTER_MODE_LEN 4
#define          MC_CMD_FILTER_MODE_SIMPLE 0x0 /* enum */
#define          MC_CMD_FILTER_MODE_STRUCTURED 0xffffffff /* enum */
/* A type value of 1 is unused. */
#define       MC_CMD_WOL_FILTER_SET_IN_WOL_TYPE_OFST 4
#define       MC_CMD_WOL_FILTER_SET_IN_WOL_TYPE_LEN 4
/* enum: Magic */
#define          MC_CMD_WOL_TYPE_MAGIC 0x0
/* enum: MS Windows Magic */
#define          MC_CMD_WOL_TYPE_WIN_MAGIC 0x2
/* enum: IPv4 Syn */
#define          MC_CMD_WOL_TYPE_IPV4_SYN 0x3
/* enum: IPv6 Syn */
#define          MC_CMD_WOL_TYPE_IPV6_SYN 0x4
/* enum: Bitmap */
#define          MC_CMD_WOL_TYPE_BITMAP 0x5
/* enum: Link */
#define          MC_CMD_WOL_TYPE_LINK 0x6
/* enum: (Above this for future use) */
#define          MC_CMD_WOL_TYPE_MAX 0x7
#define       MC_CMD_WOL_FILTER_SET_IN_DATA_OFST 8
#define       MC_CMD_WOL_FILTER_SET_IN_DATA_LEN 4
#define       MC_CMD_WOL_FILTER_SET_IN_DATA_NUM 46

/* MC_CMD_WOL_FILTER_SET_IN_MAGIC msgrequest */
#define    MC_CMD_WOL_FILTER_SET_IN_MAGIC_LEN 16
/*            MC_CMD_WOL_FILTER_SET_IN_FILTER_MODE_OFST 0 */
/*            MC_CMD_WOL_FILTER_SET_IN_FILTER_MODE_LEN 4 */
/*            MC_CMD_WOL_FILTER_SET_IN_WOL_TYPE_OFST 4 */
/*            MC_CMD_WOL_FILTER_SET_IN_WOL_TYPE_LEN 4 */
#define       MC_CMD_WOL_FILTER_SET_IN_MAGIC_MAC_OFST 8
#define       MC_CMD_WOL_FILTER_SET_IN_MAGIC_MAC_LEN 8
#define       MC_CMD_WOL_FILTER_SET_IN_MAGIC_MAC_LO_OFST 8
#define       MC_CMD_WOL_FILTER_SET_IN_MAGIC_MAC_LO_LEN 4
#define       MC_CMD_WOL_FILTER_SET_IN_MAGIC_MAC_LO_LBN 64
#define       MC_CMD_WOL_FILTER_SET_IN_MAGIC_MAC_LO_WIDTH 32
#define       MC_CMD_WOL_FILTER_SET_IN_MAGIC_MAC_HI_OFST 12
#define       MC_CMD_WOL_FILTER_SET_IN_MAGIC_MAC_HI_LEN 4
#define       MC_CMD_WOL_FILTER_SET_IN_MAGIC_MAC_HI_LBN 96
#define       MC_CMD_WOL_FILTER_SET_IN_MAGIC_MAC_HI_WIDTH 32

/* MC_CMD_WOL_FILTER_SET_IN_IPV4_SYN msgrequest */
#define    MC_CMD_WOL_FILTER_SET_IN_IPV4_SYN_LEN 20
/*            MC_CMD_WOL_FILTER_SET_IN_FILTER_MODE_OFST 0 */
/*            MC_CMD_WOL_FILTER_SET_IN_FILTER_MODE_LEN 4 */
/*            MC_CMD_WOL_FILTER_SET_IN_WOL_TYPE_OFST 4 */
/*            MC_CMD_WOL_FILTER_SET_IN_WOL_TYPE_LEN 4 */
#define       MC_CMD_WOL_FILTER_SET_IN_IPV4_SYN_SRC_IP_OFST 8
#define       MC_CMD_WOL_FILTER_SET_IN_IPV4_SYN_SRC_IP_LEN 4
#define       MC_CMD_WOL_FILTER_SET_IN_IPV4_SYN_DST_IP_OFST 12
#define       MC_CMD_WOL_FILTER_SET_IN_IPV4_SYN_DST_IP_LEN 4
#define       MC_CMD_WOL_FILTER_SET_IN_IPV4_SYN_SRC_PORT_OFST 16
#define       MC_CMD_WOL_FILTER_SET_IN_IPV4_SYN_SRC_PORT_LEN 2
#define       MC_CMD_WOL_FILTER_SET_IN_IPV4_SYN_DST_PORT_OFST 18
#define       MC_CMD_WOL_FILTER_SET_IN_IPV4_SYN_DST_PORT_LEN 2

/* MC_CMD_WOL_FILTER_SET_IN_IPV6_SYN msgrequest */
#define    MC_CMD_WOL_FILTER_SET_IN_IPV6_SYN_LEN 44
/*            MC_CMD_WOL_FILTER_SET_IN_FILTER_MODE_OFST 0 */
/*            MC_CMD_WOL_FILTER_SET_IN_FILTER_MODE_LEN 4 */
/*            MC_CMD_WOL_FILTER_SET_IN_WOL_TYPE_OFST 4 */
/*            MC_CMD_WOL_FILTER_SET_IN_WOL_TYPE_LEN 4 */
#define       MC_CMD_WOL_FILTER_SET_IN_IPV6_SYN_SRC_IP_OFST 8
#define       MC_CMD_WOL_FILTER_SET_IN_IPV6_SYN_SRC_IP_LEN 16
#define       MC_CMD_WOL_FILTER_SET_IN_IPV6_SYN_DST_IP_OFST 24
#define       MC_CMD_WOL_FILTER_SET_IN_IPV6_SYN_DST_IP_LEN 16
#define       MC_CMD_WOL_FILTER_SET_IN_IPV6_SYN_SRC_PORT_OFST 40
#define       MC_CMD_WOL_FILTER_SET_IN_IPV6_SYN_SRC_PORT_LEN 2
#define       MC_CMD_WOL_FILTER_SET_IN_IPV6_SYN_DST_PORT_OFST 42
#define       MC_CMD_WOL_FILTER_SET_IN_IPV6_SYN_DST_PORT_LEN 2

/* MC_CMD_WOL_FILTER_SET_IN_BITMAP msgrequest */
#define    MC_CMD_WOL_FILTER_SET_IN_BITMAP_LEN 187
/*            MC_CMD_WOL_FILTER_SET_IN_FILTER_MODE_OFST 0 */
/*            MC_CMD_WOL_FILTER_SET_IN_FILTER_MODE_LEN 4 */
/*            MC_CMD_WOL_FILTER_SET_IN_WOL_TYPE_OFST 4 */
/*            MC_CMD_WOL_FILTER_SET_IN_WOL_TYPE_LEN 4 */
#define       MC_CMD_WOL_FILTER_SET_IN_BITMAP_MASK_OFST 8
#define       MC_CMD_WOL_FILTER_SET_IN_BITMAP_MASK_LEN 48
#define       MC_CMD_WOL_FILTER_SET_IN_BITMAP_BITMAP_OFST 56
#define       MC_CMD_WOL_FILTER_SET_IN_BITMAP_BITMAP_LEN 128
#define       MC_CMD_WOL_FILTER_SET_IN_BITMAP_LEN_OFST 184
#define       MC_CMD_WOL_FILTER_SET_IN_BITMAP_LEN_LEN 1
#define       MC_CMD_WOL_FILTER_SET_IN_BITMAP_LAYER3_OFST 185
#define       MC_CMD_WOL_FILTER_SET_IN_BITMAP_LAYER3_LEN 1
#define       MC_CMD_WOL_FILTER_SET_IN_BITMAP_LAYER4_OFST 186
#define       MC_CMD_WOL_FILTER_SET_IN_BITMAP_LAYER4_LEN 1

/* MC_CMD_WOL_FILTER_SET_IN_LINK msgrequest */
#define    MC_CMD_WOL_FILTER_SET_IN_LINK_LEN 12
/*            MC_CMD_WOL_FILTER_SET_IN_FILTER_MODE_OFST 0 */
/*            MC_CMD_WOL_FILTER_SET_IN_FILTER_MODE_LEN 4 */
/*            MC_CMD_WOL_FILTER_SET_IN_WOL_TYPE_OFST 4 */
/*            MC_CMD_WOL_FILTER_SET_IN_WOL_TYPE_LEN 4 */
#define       MC_CMD_WOL_FILTER_SET_IN_LINK_MASK_OFST 8
#define       MC_CMD_WOL_FILTER_SET_IN_LINK_MASK_LEN 4
#define        MC_CMD_WOL_FILTER_SET_IN_LINK_UP_OFST 8
#define        MC_CMD_WOL_FILTER_SET_IN_LINK_UP_LBN 0
#define        MC_CMD_WOL_FILTER_SET_IN_LINK_UP_WIDTH 1
#define        MC_CMD_WOL_FILTER_SET_IN_LINK_DOWN_OFST 8
#define        MC_CMD_WOL_FILTER_SET_IN_LINK_DOWN_LBN 1
#define        MC_CMD_WOL_FILTER_SET_IN_LINK_DOWN_WIDTH 1

/* MC_CMD_WOL_FILTER_SET_OUT msgresponse */
#define    MC_CMD_WOL_FILTER_SET_OUT_LEN 4
#define       MC_CMD_WOL_FILTER_SET_OUT_FILTER_ID_OFST 0
#define       MC_CMD_WOL_FILTER_SET_OUT_FILTER_ID_LEN 4


/***********************************/
/* MC_CMD_WOL_FILTER_REMOVE
 * Remove a WoL filter. Locks required: None. Returns: 0, EINVAL, ENOSYS
 */
#define MC_CMD_WOL_FILTER_REMOVE 0x33
#undef MC_CMD_0x33_PRIVILEGE_CTG

#define MC_CMD_0x33_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_WOL_FILTER_REMOVE_IN msgrequest */
#define    MC_CMD_WOL_FILTER_REMOVE_IN_LEN 4
#define       MC_CMD_WOL_FILTER_REMOVE_IN_FILTER_ID_OFST 0
#define       MC_CMD_WOL_FILTER_REMOVE_IN_FILTER_ID_LEN 4

/* MC_CMD_WOL_FILTER_REMOVE_OUT msgresponse */
#define    MC_CMD_WOL_FILTER_REMOVE_OUT_LEN 0


/***********************************/
/* MC_CMD_WOL_FILTER_RESET
 * Reset (i.e. remove all) WoL filters. Locks required: None. Returns: 0,
 * ENOSYS
 */
#define MC_CMD_WOL_FILTER_RESET 0x34
#undef MC_CMD_0x34_PRIVILEGE_CTG

#define MC_CMD_0x34_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_WOL_FILTER_RESET_IN msgrequest */
#define    MC_CMD_WOL_FILTER_RESET_IN_LEN 4
#define       MC_CMD_WOL_FILTER_RESET_IN_MASK_OFST 0
#define       MC_CMD_WOL_FILTER_RESET_IN_MASK_LEN 4
/* enum property: bitmask */
#define          MC_CMD_WOL_FILTER_RESET_IN_WAKE_FILTERS 0x1 /* enum */
#define          MC_CMD_WOL_FILTER_RESET_IN_LIGHTSOUT_OFFLOADS 0x2 /* enum */

/* MC_CMD_WOL_FILTER_RESET_OUT msgresponse */
#define    MC_CMD_WOL_FILTER_RESET_OUT_LEN 0


/***********************************/
/* MC_CMD_NVRAM_TYPES
 * Return bitfield indicating available types of virtual NVRAM partitions.
 * Locks required: none. Returns: 0
 */
#define MC_CMD_NVRAM_TYPES 0x36
#undef MC_CMD_0x36_PRIVILEGE_CTG

#define MC_CMD_0x36_PRIVILEGE_CTG SRIOV_CTG_ADMIN

/* MC_CMD_NVRAM_TYPES_IN msgrequest */
#define    MC_CMD_NVRAM_TYPES_IN_LEN 0

/* MC_CMD_NVRAM_TYPES_OUT msgresponse */
#define    MC_CMD_NVRAM_TYPES_OUT_LEN 4
/* Bit mask of supported types. */
#define       MC_CMD_NVRAM_TYPES_OUT_TYPES_OFST 0
#define       MC_CMD_NVRAM_TYPES_OUT_TYPES_LEN 4
/* enum property: bitshift */
/* enum: Disabled callisto. */
#define          MC_CMD_NVRAM_TYPE_DISABLED_CALLISTO 0x0
/* enum: MC firmware. */
#define          MC_CMD_NVRAM_TYPE_MC_FW 0x1
/* enum: MC backup firmware. */
#define          MC_CMD_NVRAM_TYPE_MC_FW_BACKUP 0x2
/* enum: Static configuration Port0. */
#define          MC_CMD_NVRAM_TYPE_STATIC_CFG_PORT0 0x3
/* enum: Static configuration Port1. */
#define          MC_CMD_NVRAM_TYPE_STATIC_CFG_PORT1 0x4
/* enum: Dynamic configuration Port0. */
#define          MC_CMD_NVRAM_TYPE_DYNAMIC_CFG_PORT0 0x5
/* enum: Dynamic configuration Port1. */
#define          MC_CMD_NVRAM_TYPE_DYNAMIC_CFG_PORT1 0x6
/* enum: Expansion Rom. */
#define          MC_CMD_NVRAM_TYPE_EXP_ROM 0x7
/* enum: Expansion Rom Configuration Port0. */
#define          MC_CMD_NVRAM_TYPE_EXP_ROM_CFG_PORT0 0x8
/* enum: Expansion Rom Configuration Port1. */
#define          MC_CMD_NVRAM_TYPE_EXP_ROM_CFG_PORT1 0x9
/* enum: Phy Configuration Port0. */
#define          MC_CMD_NVRAM_TYPE_PHY_PORT0 0xa
/* enum: Phy Configuration Port1. */
#define          MC_CMD_NVRAM_TYPE_PHY_PORT1 0xb
/* enum: Log. */
#define          MC_CMD_NVRAM_TYPE_LOG 0xc
/* enum: FPGA image. */
#define          MC_CMD_NVRAM_TYPE_FPGA 0xd
/* enum: FPGA backup image */
#define          MC_CMD_NVRAM_TYPE_FPGA_BACKUP 0xe
/* enum: FC firmware. */
#define          MC_CMD_NVRAM_TYPE_FC_FW 0xf
/* enum: FC backup firmware. */
#define          MC_CMD_NVRAM_TYPE_FC_FW_BACKUP 0x10
/* enum: CPLD image. */
#define          MC_CMD_NVRAM_TYPE_CPLD 0x11
/* enum: Licensing information. */
#define          MC_CMD_NVRAM_TYPE_LICENSE 0x12
/* enum: FC Log. */
#define          MC_CMD_NVRAM_TYPE_FC_LOG 0x13
/* enum: Additional flash on FPGA. */
#define          MC_CMD_NVRAM_TYPE_FC_EXTRA 0x14


/***********************************/
/* MC_CMD_NVRAM_INFO
 * Read info about a virtual NVRAM partition. Locks required: none. Returns: 0,
 * EINVAL (bad type).
 */
#define MC_CMD_NVRAM_INFO 0x37
#undef MC_CMD_0x37_PRIVILEGE_CTG

#define MC_CMD_0x37_PRIVILEGE_CTG SRIOV_CTG_ADMIN

/* MC_CMD_NVRAM_INFO_IN msgrequest */
#define    MC_CMD_NVRAM_INFO_IN_LEN 4
#define       MC_CMD_NVRAM_INFO_IN_TYPE_OFST 0
#define       MC_CMD_NVRAM_INFO_IN_TYPE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_NVRAM_TYPES/MC_CMD_NVRAM_TYPES_OUT/TYPES */

/* MC_CMD_NVRAM_INFO_OUT msgresponse */
#define    MC_CMD_NVRAM_INFO_OUT_LEN 24
#define       MC_CMD_NVRAM_INFO_OUT_TYPE_OFST 0
#define       MC_CMD_NVRAM_INFO_OUT_TYPE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_NVRAM_TYPES/MC_CMD_NVRAM_TYPES_OUT/TYPES */
#define       MC_CMD_NVRAM_INFO_OUT_SIZE_OFST 4
#define       MC_CMD_NVRAM_INFO_OUT_SIZE_LEN 4
#define       MC_CMD_NVRAM_INFO_OUT_ERASESIZE_OFST 8
#define       MC_CMD_NVRAM_INFO_OUT_ERASESIZE_LEN 4
#define       MC_CMD_NVRAM_INFO_OUT_FLAGS_OFST 12
#define       MC_CMD_NVRAM_INFO_OUT_FLAGS_LEN 4
#define        MC_CMD_NVRAM_INFO_OUT_PROTECTED_OFST 12
#define        MC_CMD_NVRAM_INFO_OUT_PROTECTED_LBN 0
#define        MC_CMD_NVRAM_INFO_OUT_PROTECTED_WIDTH 1
#define        MC_CMD_NVRAM_INFO_OUT_TLV_OFST 12
#define        MC_CMD_NVRAM_INFO_OUT_TLV_LBN 1
#define        MC_CMD_NVRAM_INFO_OUT_TLV_WIDTH 1
#define        MC_CMD_NVRAM_INFO_OUT_READ_ONLY_IF_TSA_BOUND_OFST 12
#define        MC_CMD_NVRAM_INFO_OUT_READ_ONLY_IF_TSA_BOUND_LBN 2
#define        MC_CMD_NVRAM_INFO_OUT_READ_ONLY_IF_TSA_BOUND_WIDTH 1
#define        MC_CMD_NVRAM_INFO_OUT_CRC_OFST 12
#define        MC_CMD_NVRAM_INFO_OUT_CRC_LBN 3
#define        MC_CMD_NVRAM_INFO_OUT_CRC_WIDTH 1
#define        MC_CMD_NVRAM_INFO_OUT_READ_ONLY_OFST 12
#define        MC_CMD_NVRAM_INFO_OUT_READ_ONLY_LBN 5
#define        MC_CMD_NVRAM_INFO_OUT_READ_ONLY_WIDTH 1
#define        MC_CMD_NVRAM_INFO_OUT_CMAC_OFST 12
#define        MC_CMD_NVRAM_INFO_OUT_CMAC_LBN 6
#define        MC_CMD_NVRAM_INFO_OUT_CMAC_WIDTH 1
#define        MC_CMD_NVRAM_INFO_OUT_A_B_OFST 12
#define        MC_CMD_NVRAM_INFO_OUT_A_B_LBN 7
#define        MC_CMD_NVRAM_INFO_OUT_A_B_WIDTH 1
#define       MC_CMD_NVRAM_INFO_OUT_PHYSDEV_OFST 16
#define       MC_CMD_NVRAM_INFO_OUT_PHYSDEV_LEN 4
#define       MC_CMD_NVRAM_INFO_OUT_PHYSADDR_OFST 20
#define       MC_CMD_NVRAM_INFO_OUT_PHYSADDR_LEN 4

/* MC_CMD_NVRAM_INFO_V2_OUT msgresponse */
#define    MC_CMD_NVRAM_INFO_V2_OUT_LEN 28
#define       MC_CMD_NVRAM_INFO_V2_OUT_TYPE_OFST 0
#define       MC_CMD_NVRAM_INFO_V2_OUT_TYPE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_NVRAM_TYPES/MC_CMD_NVRAM_TYPES_OUT/TYPES */
#define       MC_CMD_NVRAM_INFO_V2_OUT_SIZE_OFST 4
#define       MC_CMD_NVRAM_INFO_V2_OUT_SIZE_LEN 4
#define       MC_CMD_NVRAM_INFO_V2_OUT_ERASESIZE_OFST 8
#define       MC_CMD_NVRAM_INFO_V2_OUT_ERASESIZE_LEN 4
#define       MC_CMD_NVRAM_INFO_V2_OUT_FLAGS_OFST 12
#define       MC_CMD_NVRAM_INFO_V2_OUT_FLAGS_LEN 4
#define        MC_CMD_NVRAM_INFO_V2_OUT_PROTECTED_OFST 12
#define        MC_CMD_NVRAM_INFO_V2_OUT_PROTECTED_LBN 0
#define        MC_CMD_NVRAM_INFO_V2_OUT_PROTECTED_WIDTH 1
#define        MC_CMD_NVRAM_INFO_V2_OUT_TLV_OFST 12
#define        MC_CMD_NVRAM_INFO_V2_OUT_TLV_LBN 1
#define        MC_CMD_NVRAM_INFO_V2_OUT_TLV_WIDTH 1
#define        MC_CMD_NVRAM_INFO_V2_OUT_READ_ONLY_IF_TSA_BOUND_OFST 12
#define        MC_CMD_NVRAM_INFO_V2_OUT_READ_ONLY_IF_TSA_BOUND_LBN 2
#define        MC_CMD_NVRAM_INFO_V2_OUT_READ_ONLY_IF_TSA_BOUND_WIDTH 1
#define        MC_CMD_NVRAM_INFO_V2_OUT_READ_ONLY_OFST 12
#define        MC_CMD_NVRAM_INFO_V2_OUT_READ_ONLY_LBN 5
#define        MC_CMD_NVRAM_INFO_V2_OUT_READ_ONLY_WIDTH 1
#define        MC_CMD_NVRAM_INFO_V2_OUT_A_B_OFST 12
#define        MC_CMD_NVRAM_INFO_V2_OUT_A_B_LBN 7
#define        MC_CMD_NVRAM_INFO_V2_OUT_A_B_WIDTH 1
#define        MC_CMD_NVRAM_INFO_V2_OUT_WRITE_ONLY_OFST 12
#define        MC_CMD_NVRAM_INFO_V2_OUT_WRITE_ONLY_LBN 8
#define        MC_CMD_NVRAM_INFO_V2_OUT_WRITE_ONLY_WIDTH 1
#define        MC_CMD_NVRAM_INFO_V2_OUT_SEQUENTIAL_WRITE_OFST 12
#define        MC_CMD_NVRAM_INFO_V2_OUT_SEQUENTIAL_WRITE_LBN 9
#define        MC_CMD_NVRAM_INFO_V2_OUT_SEQUENTIAL_WRITE_WIDTH 1
#define       MC_CMD_NVRAM_INFO_V2_OUT_PHYSDEV_OFST 16
#define       MC_CMD_NVRAM_INFO_V2_OUT_PHYSDEV_LEN 4
#define       MC_CMD_NVRAM_INFO_V2_OUT_PHYSADDR_OFST 20
#define       MC_CMD_NVRAM_INFO_V2_OUT_PHYSADDR_LEN 4
/* Writes must be multiples of this size. Added to support the MUM on Sorrento.
 */
#define       MC_CMD_NVRAM_INFO_V2_OUT_WRITESIZE_OFST 24
#define       MC_CMD_NVRAM_INFO_V2_OUT_WRITESIZE_LEN 4


/***********************************/
/* MC_CMD_NVRAM_UPDATE_START
 * Start a group of update operations on a virtual NVRAM partition. Locks
 * required: PHY_LOCK if type==*PHY*. Returns: 0, EINVAL (bad type), EACCES (if
 * PHY_LOCK required and not held). In an adapter bound to a TSA controller,
 * MC_CMD_NVRAM_UPDATE_START can only be used on a subset of partition types
 * i.e. static config, dynamic config and expansion ROM config. Attempting to
 * perform this operation on a restricted partition will return the error
 * EPERM.
 */
#define MC_CMD_NVRAM_UPDATE_START 0x38
#undef MC_CMD_0x38_PRIVILEGE_CTG

#define MC_CMD_0x38_PRIVILEGE_CTG SRIOV_CTG_ADMIN

/* MC_CMD_NVRAM_UPDATE_START_IN msgrequest: Legacy NVRAM_UPDATE_START request.
 * Use NVRAM_UPDATE_START_V2_IN in new code
 */
#define    MC_CMD_NVRAM_UPDATE_START_IN_LEN 4
#define       MC_CMD_NVRAM_UPDATE_START_IN_TYPE_OFST 0
#define       MC_CMD_NVRAM_UPDATE_START_IN_TYPE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_NVRAM_TYPES/MC_CMD_NVRAM_TYPES_OUT/TYPES */

/* MC_CMD_NVRAM_UPDATE_START_V2_IN msgrequest: Extended NVRAM_UPDATE_START
 * request with additional flags indicating version of command in use. See
 * MC_CMD_NVRAM_UPDATE_FINISH_V2_OUT for details of extended functionality. Use
 * paired up with NVRAM_UPDATE_FINISH_V2_IN.
 */
#define    MC_CMD_NVRAM_UPDATE_START_V2_IN_LEN 8
#define       MC_CMD_NVRAM_UPDATE_START_V2_IN_TYPE_OFST 0
#define       MC_CMD_NVRAM_UPDATE_START_V2_IN_TYPE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_NVRAM_TYPES/MC_CMD_NVRAM_TYPES_OUT/TYPES */
#define       MC_CMD_NVRAM_UPDATE_START_V2_IN_FLAGS_OFST 4
#define       MC_CMD_NVRAM_UPDATE_START_V2_IN_FLAGS_LEN 4
#define        MC_CMD_NVRAM_UPDATE_START_V2_IN_FLAG_REPORT_VERIFY_RESULT_OFST 4
#define        MC_CMD_NVRAM_UPDATE_START_V2_IN_FLAG_REPORT_VERIFY_RESULT_LBN 0
#define        MC_CMD_NVRAM_UPDATE_START_V2_IN_FLAG_REPORT_VERIFY_RESULT_WIDTH 1

/* MC_CMD_NVRAM_UPDATE_START_OUT msgresponse */
#define    MC_CMD_NVRAM_UPDATE_START_OUT_LEN 0


/***********************************/
/* MC_CMD_NVRAM_READ
 * Read data from a virtual NVRAM partition. Locks required: PHY_LOCK if
 * type==*PHY*. Returns: 0, EINVAL (bad type/offset/length), EACCES (if
 * PHY_LOCK required and not held)
 */
#define MC_CMD_NVRAM_READ 0x39
#undef MC_CMD_0x39_PRIVILEGE_CTG

#define MC_CMD_0x39_PRIVILEGE_CTG SRIOV_CTG_ADMIN

/* MC_CMD_NVRAM_READ_IN msgrequest */
#define    MC_CMD_NVRAM_READ_IN_LEN 12
#define       MC_CMD_NVRAM_READ_IN_TYPE_OFST 0
#define       MC_CMD_NVRAM_READ_IN_TYPE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_NVRAM_TYPES/MC_CMD_NVRAM_TYPES_OUT/TYPES */
#define       MC_CMD_NVRAM_READ_IN_OFFSET_OFST 4
#define       MC_CMD_NVRAM_READ_IN_OFFSET_LEN 4
/* amount to read in bytes */
#define       MC_CMD_NVRAM_READ_IN_LENGTH_OFST 8
#define       MC_CMD_NVRAM_READ_IN_LENGTH_LEN 4

/* MC_CMD_NVRAM_READ_IN_V2 msgrequest */
#define    MC_CMD_NVRAM_READ_IN_V2_LEN 16
#define       MC_CMD_NVRAM_READ_IN_V2_TYPE_OFST 0
#define       MC_CMD_NVRAM_READ_IN_V2_TYPE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_NVRAM_TYPES/MC_CMD_NVRAM_TYPES_OUT/TYPES */
#define       MC_CMD_NVRAM_READ_IN_V2_OFFSET_OFST 4
#define       MC_CMD_NVRAM_READ_IN_V2_OFFSET_LEN 4
/* amount to read in bytes */
#define       MC_CMD_NVRAM_READ_IN_V2_LENGTH_OFST 8
#define       MC_CMD_NVRAM_READ_IN_V2_LENGTH_LEN 4
/* Optional control info. If a partition is stored with an A/B versioning
 * scheme (i.e. in more than one physical partition in NVRAM) the host can set
 * this to control which underlying physical partition is used to read data
 * from. This allows it to perform a read-modify-write-verify with the write
 * lock continuously held by calling NVRAM_UPDATE_START, reading the old
 * contents using MODE=TARGET_CURRENT, overwriting the old partition and then
 * verifying by reading with MODE=TARGET_BACKUP.
 */
#define       MC_CMD_NVRAM_READ_IN_V2_MODE_OFST 12
#define       MC_CMD_NVRAM_READ_IN_V2_MODE_LEN 4
/* enum: Same as omitting MODE: caller sees data in current partition unless it
 * holds the write lock in which case it sees data in the partition it is
 * updating.
 */
#define          MC_CMD_NVRAM_READ_IN_V2_DEFAULT 0x0
/* enum: Read from the current partition of an A/B pair, even if holding the
 * write lock.
 */
#define          MC_CMD_NVRAM_READ_IN_V2_TARGET_CURRENT 0x1
/* enum: Read from the non-current (i.e. to be updated) partition of an A/B
 * pair
 */
#define          MC_CMD_NVRAM_READ_IN_V2_TARGET_BACKUP 0x2

/* MC_CMD_NVRAM_READ_OUT msgresponse */
#define    MC_CMD_NVRAM_READ_OUT_LENMIN 1
#define    MC_CMD_NVRAM_READ_OUT_LENMAX 252
#define    MC_CMD_NVRAM_READ_OUT_LENMAX_MCDI2 1020
#define    MC_CMD_NVRAM_READ_OUT_LEN(num) (0+1*(num))
#define    MC_CMD_NVRAM_READ_OUT_READ_BUFFER_NUM(len) (((len)-0)/1)
#define       MC_CMD_NVRAM_READ_OUT_READ_BUFFER_OFST 0
#define       MC_CMD_NVRAM_READ_OUT_READ_BUFFER_LEN 1
#define       MC_CMD_NVRAM_READ_OUT_READ_BUFFER_MINNUM 1
#define       MC_CMD_NVRAM_READ_OUT_READ_BUFFER_MAXNUM 252
#define       MC_CMD_NVRAM_READ_OUT_READ_BUFFER_MAXNUM_MCDI2 1020


/***********************************/
/* MC_CMD_NVRAM_WRITE
 * Write data to a virtual NVRAM partition. Locks required: PHY_LOCK if
 * type==*PHY*. Returns: 0, EINVAL (bad type/offset/length), EACCES (if
 * PHY_LOCK required and not held)
 */
#define MC_CMD_NVRAM_WRITE 0x3a
#undef MC_CMD_0x3a_PRIVILEGE_CTG

#define MC_CMD_0x3a_PRIVILEGE_CTG SRIOV_CTG_ADMIN

/* MC_CMD_NVRAM_WRITE_IN msgrequest */
#define    MC_CMD_NVRAM_WRITE_IN_LENMIN 13
#define    MC_CMD_NVRAM_WRITE_IN_LENMAX 252
#define    MC_CMD_NVRAM_WRITE_IN_LENMAX_MCDI2 1020
#define    MC_CMD_NVRAM_WRITE_IN_LEN(num) (12+1*(num))
#define    MC_CMD_NVRAM_WRITE_IN_WRITE_BUFFER_NUM(len) (((len)-12)/1)
#define       MC_CMD_NVRAM_WRITE_IN_TYPE_OFST 0
#define       MC_CMD_NVRAM_WRITE_IN_TYPE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_NVRAM_TYPES/MC_CMD_NVRAM_TYPES_OUT/TYPES */
#define       MC_CMD_NVRAM_WRITE_IN_OFFSET_OFST 4
#define       MC_CMD_NVRAM_WRITE_IN_OFFSET_LEN 4
#define       MC_CMD_NVRAM_WRITE_IN_LENGTH_OFST 8
#define       MC_CMD_NVRAM_WRITE_IN_LENGTH_LEN 4
#define       MC_CMD_NVRAM_WRITE_IN_WRITE_BUFFER_OFST 12
#define       MC_CMD_NVRAM_WRITE_IN_WRITE_BUFFER_LEN 1
#define       MC_CMD_NVRAM_WRITE_IN_WRITE_BUFFER_MINNUM 1
#define       MC_CMD_NVRAM_WRITE_IN_WRITE_BUFFER_MAXNUM 240
#define       MC_CMD_NVRAM_WRITE_IN_WRITE_BUFFER_MAXNUM_MCDI2 1008

/* MC_CMD_NVRAM_WRITE_OUT msgresponse */
#define    MC_CMD_NVRAM_WRITE_OUT_LEN 0


/***********************************/
/* MC_CMD_NVRAM_ERASE
 * Erase sector(s) from a virtual NVRAM partition. Locks required: PHY_LOCK if
 * type==*PHY*. Returns: 0, EINVAL (bad type/offset/length), EACCES (if
 * PHY_LOCK required and not held)
 */
#define MC_CMD_NVRAM_ERASE 0x3b
#undef MC_CMD_0x3b_PRIVILEGE_CTG

#define MC_CMD_0x3b_PRIVILEGE_CTG SRIOV_CTG_ADMIN

/* MC_CMD_NVRAM_ERASE_IN msgrequest */
#define    MC_CMD_NVRAM_ERASE_IN_LEN 12
#define       MC_CMD_NVRAM_ERASE_IN_TYPE_OFST 0
#define       MC_CMD_NVRAM_ERASE_IN_TYPE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_NVRAM_TYPES/MC_CMD_NVRAM_TYPES_OUT/TYPES */
#define       MC_CMD_NVRAM_ERASE_IN_OFFSET_OFST 4
#define       MC_CMD_NVRAM_ERASE_IN_OFFSET_LEN 4
#define       MC_CMD_NVRAM_ERASE_IN_LENGTH_OFST 8
#define       MC_CMD_NVRAM_ERASE_IN_LENGTH_LEN 4

/* MC_CMD_NVRAM_ERASE_OUT msgresponse */
#define    MC_CMD_NVRAM_ERASE_OUT_LEN 0


/***********************************/
/* MC_CMD_NVRAM_UPDATE_FINISH
 * Finish a group of update operations on a virtual NVRAM partition. Locks
 * required: PHY_LOCK if type==*PHY*. Returns: 0, EINVAL (bad type/offset/
 * length), EACCES (if PHY_LOCK required and not held). In an adapter bound to
 * a TSA controller, MC_CMD_NVRAM_UPDATE_FINISH can only be used on a subset of
 * partition types i.e. static config, dynamic config and expansion ROM config.
 * Attempting to perform this operation on a restricted partition will return
 * the error EPERM.
 */
#define MC_CMD_NVRAM_UPDATE_FINISH 0x3c
#undef MC_CMD_0x3c_PRIVILEGE_CTG

#define MC_CMD_0x3c_PRIVILEGE_CTG SRIOV_CTG_ADMIN

/* MC_CMD_NVRAM_UPDATE_FINISH_IN msgrequest: Legacy NVRAM_UPDATE_FINISH
 * request. Use NVRAM_UPDATE_FINISH_V2_IN in new code
 */
#define    MC_CMD_NVRAM_UPDATE_FINISH_IN_LEN 8
#define       MC_CMD_NVRAM_UPDATE_FINISH_IN_TYPE_OFST 0
#define       MC_CMD_NVRAM_UPDATE_FINISH_IN_TYPE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_NVRAM_TYPES/MC_CMD_NVRAM_TYPES_OUT/TYPES */
#define       MC_CMD_NVRAM_UPDATE_FINISH_IN_REBOOT_OFST 4
#define       MC_CMD_NVRAM_UPDATE_FINISH_IN_REBOOT_LEN 4

/* MC_CMD_NVRAM_UPDATE_FINISH_V2_IN msgrequest: Extended NVRAM_UPDATE_FINISH
 * request with additional flags indicating version of NVRAM_UPDATE commands in
 * use. See MC_CMD_NVRAM_UPDATE_FINISH_V2_OUT for details of extended
 * functionality. Use paired up with NVRAM_UPDATE_START_V2_IN.
 */
#define    MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_LEN 12
#define       MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_TYPE_OFST 0
#define       MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_TYPE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_NVRAM_TYPES/MC_CMD_NVRAM_TYPES_OUT/TYPES */
#define       MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_REBOOT_OFST 4
#define       MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_REBOOT_LEN 4
#define       MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_FLAGS_OFST 8
#define       MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_FLAGS_LEN 4
#define        MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_FLAG_REPORT_VERIFY_RESULT_OFST 8
#define        MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_FLAG_REPORT_VERIFY_RESULT_LBN 0
#define        MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_FLAG_REPORT_VERIFY_RESULT_WIDTH 1
#define        MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_FLAG_RUN_IN_BACKGROUND_OFST 8
#define        MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_FLAG_RUN_IN_BACKGROUND_LBN 1
#define        MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_FLAG_RUN_IN_BACKGROUND_WIDTH 1
#define        MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_FLAG_POLL_VERIFY_RESULT_OFST 8
#define        MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_FLAG_POLL_VERIFY_RESULT_LBN 2
#define        MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_FLAG_POLL_VERIFY_RESULT_WIDTH 1
#define        MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_FLAG_ABORT_OFST 8
#define        MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_FLAG_ABORT_LBN 3
#define        MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_FLAG_ABORT_WIDTH 1

/* MC_CMD_NVRAM_UPDATE_FINISH_OUT msgresponse: Legacy NVRAM_UPDATE_FINISH
 * response. Use NVRAM_UPDATE_FINISH_V2_OUT in new code
 */
#define    MC_CMD_NVRAM_UPDATE_FINISH_OUT_LEN 0

/* MC_CMD_NVRAM_UPDATE_FINISH_V2_OUT msgresponse:
 *
 * Extended NVRAM_UPDATE_FINISH response that communicates the result of secure
 * firmware validation where applicable back to the host.
 *
 * Medford only: For signed firmware images, such as those for medford, the MC
 * firmware verifies the signature before marking the firmware image as valid.
 * This process takes a few seconds to complete. So is likely to take more than
 * the MCDI timeout. Hence signature verification is initiated when
 * MC_CMD_NVRAM_UPDATE_FINISH_V2_IN is received by the firmware, however, the
 * MCDI command is run in a background MCDI processing thread. This response
 * payload includes the results of the signature verification. Note that the
 * per-partition nvram lock in firmware is only released after the verification
 * has completed.
 */
#define    MC_CMD_NVRAM_UPDATE_FINISH_V2_OUT_LEN 4
/* Result of nvram update completion processing. Result codes that indicate an
 * internal build failure and therefore not expected to be seen by customers in
 * the field are marked with a prefix 'Internal-error'.
 */
#define       MC_CMD_NVRAM_UPDATE_FINISH_V2_OUT_RESULT_CODE_OFST 0
#define       MC_CMD_NVRAM_UPDATE_FINISH_V2_OUT_RESULT_CODE_LEN 4
/* enum: Invalid return code; only non-zero values are defined. Defined as
 * unknown for backwards compatibility with NVRAM_UPDATE_FINISH_OUT.
 */
#define          MC_CMD_NVRAM_VERIFY_RC_UNKNOWN 0x0
/* enum: Verify succeeded without any errors. */
#define          MC_CMD_NVRAM_VERIFY_RC_SUCCESS 0x1
/* enum: CMS format verification failed due to an internal error. */
#define          MC_CMD_NVRAM_VERIFY_RC_CMS_CHECK_FAILED 0x2
/* enum: Invalid CMS format in image metadata. */
#define          MC_CMD_NVRAM_VERIFY_RC_INVALID_CMS_FORMAT 0x3
/* enum: Message digest verification failed due to an internal error. */
#define          MC_CMD_NVRAM_VERIFY_RC_MESSAGE_DIGEST_CHECK_FAILED 0x4
/* enum: Error in message digest calculated over the reflash-header, payload
 * and reflash-trailer.
 */
#define          MC_CMD_NVRAM_VERIFY_RC_BAD_MESSAGE_DIGEST 0x5
/* enum: Signature verification failed due to an internal error. */
#define          MC_CMD_NVRAM_VERIFY_RC_SIGNATURE_CHECK_FAILED 0x6
/* enum: There are no valid signatures in the image. */
#define          MC_CMD_NVRAM_VERIFY_RC_NO_VALID_SIGNATURES 0x7
/* enum: Trusted approvers verification failed due to an internal error. */
#define          MC_CMD_NVRAM_VERIFY_RC_TRUSTED_APPROVERS_CHECK_FAILED 0x8
/* enum: The Trusted approver's list is empty. */
#define          MC_CMD_NVRAM_VERIFY_RC_NO_TRUSTED_APPROVERS 0x9
/* enum: Signature chain verification failed due to an internal error. */
#define          MC_CMD_NVRAM_VERIFY_RC_SIGNATURE_CHAIN_CHECK_FAILED 0xa
/* enum: The signers of the signatures in the image are not listed in the
 * Trusted approver's list.
 */
#define          MC_CMD_NVRAM_VERIFY_RC_NO_SIGNATURE_MATCH 0xb
/* enum: The image contains a test-signed certificate, but the adapter accepts
 * only production signed images.
 */
#define          MC_CMD_NVRAM_VERIFY_RC_REJECT_TEST_SIGNED 0xc
/* enum: The image has a lower security level than the current firmware. */
#define          MC_CMD_NVRAM_VERIFY_RC_SECURITY_LEVEL_DOWNGRADE 0xd
/* enum: Internal-error. The signed image is missing the 'contents' section,
 * where the 'contents' section holds the actual image payload to be applied.
 */
#define          MC_CMD_NVRAM_VERIFY_RC_CONTENT_NOT_FOUND 0xe
/* enum: Internal-error. The bundle header is invalid. */
#define          MC_CMD_NVRAM_VERIFY_RC_BUNDLE_CONTENT_HEADER_INVALID 0xf
/* enum: Internal-error. The bundle does not have a valid reflash image layout.
 */
#define          MC_CMD_NVRAM_VERIFY_RC_BUNDLE_REFLASH_IMAGE_INVALID 0x10
/* enum: Internal-error. The bundle has an inconsistent layout of components or
 * incorrect checksum.
 */
#define          MC_CMD_NVRAM_VERIFY_RC_BUNDLE_IMAGE_LAYOUT_INVALID 0x11
/* enum: Internal-error. The bundle manifest is inconsistent with components in
 * the bundle.
 */
#define          MC_CMD_NVRAM_VERIFY_RC_BUNDLE_MANIFEST_INVALID 0x12
/* enum: Internal-error. The number of components in a bundle do not match the
 * number of components advertised by the bundle manifest.
 */
#define          MC_CMD_NVRAM_VERIFY_RC_BUNDLE_MANIFEST_NUM_COMPONENTS_MISMATCH 0x13
/* enum: Internal-error. The bundle contains too many components for the MC
 * firmware to process
 */
#define          MC_CMD_NVRAM_VERIFY_RC_BUNDLE_MANIFEST_TOO_MANY_COMPONENTS 0x14
/* enum: Internal-error. The bundle manifest has an invalid/inconsistent
 * component.
 */
#define          MC_CMD_NVRAM_VERIFY_RC_BUNDLE_MANIFEST_COMPONENT_INVALID 0x15
/* enum: Internal-error. The hash of a component does not match the hash stored
 * in the bundle manifest.
 */
#define          MC_CMD_NVRAM_VERIFY_RC_BUNDLE_MANIFEST_COMPONENT_HASH_MISMATCH 0x16
/* enum: Internal-error. Component hash calculation failed. */
#define          MC_CMD_NVRAM_VERIFY_RC_BUNDLE_MANIFEST_COMPONENT_HASH_FAILED 0x17
/* enum: Internal-error. The component does not have a valid reflash image
 * layout.
 */
#define          MC_CMD_NVRAM_VERIFY_RC_BUNDLE_COMPONENT_REFLASH_IMAGE_INVALID 0x18
/* enum: The bundle processing code failed to copy a component to its target
 * partition.
 */
#define          MC_CMD_NVRAM_VERIFY_RC_BUNDLE_COMPONENT_COPY_FAILED 0x19
/* enum: The update operation is in-progress. */
#define          MC_CMD_NVRAM_VERIFY_RC_PENDING 0x1a
/* enum: The update was an invalid user configuration file. */
#define          MC_CMD_NVRAM_VERIFY_RC_BAD_CONFIG 0x1b
/* enum: The write was to the AUTO partition but the data was not recognised as
 * a valid partition.
 */
#define          MC_CMD_NVRAM_VERIFY_RC_UNKNOWN_TYPE 0x1c

/* MC_CMD_NVRAM_UPDATE_FINISH_V3_OUT msgresponse */
#define    MC_CMD_NVRAM_UPDATE_FINISH_V3_OUT_LEN 88
/* Result of nvram update completion processing. Result codes that indicate an
 * internal build failure and therefore not expected to be seen by customers in
 * the field are marked with a prefix 'Internal-error'.
 */
#define       MC_CMD_NVRAM_UPDATE_FINISH_V3_OUT_RESULT_CODE_OFST 0
#define       MC_CMD_NVRAM_UPDATE_FINISH_V3_OUT_RESULT_CODE_LEN 4
/* enum: Invalid return code; only non-zero values are defined. Defined as
 * unknown for backwards compatibility with NVRAM_UPDATE_FINISH_OUT.
 */
/*               MC_CMD_NVRAM_VERIFY_RC_UNKNOWN 0x0 */
/* enum: Verify succeeded without any errors. */
/*               MC_CMD_NVRAM_VERIFY_RC_SUCCESS 0x1 */
/* enum: CMS format verification failed due to an internal error. */
/*               MC_CMD_NVRAM_VERIFY_RC_CMS_CHECK_FAILED 0x2 */
/* enum: Invalid CMS format in image metadata. */
/*               MC_CMD_NVRAM_VERIFY_RC_INVALID_CMS_FORMAT 0x3 */
/* enum: Message digest verification failed due to an internal error. */
/*               MC_CMD_NVRAM_VERIFY_RC_MESSAGE_DIGEST_CHECK_FAILED 0x4 */
/* enum: Error in message digest calculated over the reflash-header, payload
 * and reflash-trailer.
 */
/*               MC_CMD_NVRAM_VERIFY_RC_BAD_MESSAGE_DIGEST 0x5 */
/* enum: Signature verification failed due to an internal error. */
/*               MC_CMD_NVRAM_VERIFY_RC_SIGNATURE_CHECK_FAILED 0x6 */
/* enum: There are no valid signatures in the image. */
/*               MC_CMD_NVRAM_VERIFY_RC_NO_VALID_SIGNATURES 0x7 */
/* enum: Trusted approvers verification failed due to an internal error. */
/*               MC_CMD_NVRAM_VERIFY_RC_TRUSTED_APPROVERS_CHECK_FAILED 0x8 */
/* enum: The Trusted approver's list is empty. */
/*               MC_CMD_NVRAM_VERIFY_RC_NO_TRUSTED_APPROVERS 0x9 */
/* enum: Signature chain verification failed due to an internal error. */
/*               MC_CMD_NVRAM_VERIFY_RC_SIGNATURE_CHAIN_CHECK_FAILED 0xa */
/* enum: The signers of the signatures in the image are not listed in the
 * Trusted approver's list.
 */
/*               MC_CMD_NVRAM_VERIFY_RC_NO_SIGNATURE_MATCH 0xb */
/* enum: The image contains a test-signed certificate, but the adapter accepts
 * only production signed images.
 */
/*               MC_CMD_NVRAM_VERIFY_RC_REJECT_TEST_SIGNED 0xc */
/* enum: The image has a lower security level than the current firmware. */
/*               MC_CMD_NVRAM_VERIFY_RC_SECURITY_LEVEL_DOWNGRADE 0xd */
/* enum: Internal-error. The signed image is missing the 'contents' section,
 * where the 'contents' section holds the actual image payload to be applied.
 */
/*               MC_CMD_NVRAM_VERIFY_RC_CONTENT_NOT_FOUND 0xe */
/* enum: Internal-error. The bundle header is invalid. */
/*               MC_CMD_NVRAM_VERIFY_RC_BUNDLE_CONTENT_HEADER_INVALID 0xf */
/* enum: Internal-error. The bundle does not have a valid reflash image layout.
 */
/*               MC_CMD_NVRAM_VERIFY_RC_BUNDLE_REFLASH_IMAGE_INVALID 0x10 */
/* enum: Internal-error. The bundle has an inconsistent layout of components or
 * incorrect checksum.
 */
/*               MC_CMD_NVRAM_VERIFY_RC_BUNDLE_IMAGE_LAYOUT_INVALID 0x11 */
/* enum: Internal-error. The bundle manifest is inconsistent with components in
 * the bundle.
 */
/*               MC_CMD_NVRAM_VERIFY_RC_BUNDLE_MANIFEST_INVALID 0x12 */
/* enum: Internal-error. The number of components in a bundle do not match the
 * number of components advertised by the bundle manifest.
 */
/*               MC_CMD_NVRAM_VERIFY_RC_BUNDLE_MANIFEST_NUM_COMPONENTS_MISMATCH 0x13 */
/* enum: Internal-error. The bundle contains too many components for the MC
 * firmware to process
 */
/*               MC_CMD_NVRAM_VERIFY_RC_BUNDLE_MANIFEST_TOO_MANY_COMPONENTS 0x14 */
/* enum: Internal-error. The bundle manifest has an invalid/inconsistent
 * component.
 */
/*               MC_CMD_NVRAM_VERIFY_RC_BUNDLE_MANIFEST_COMPONENT_INVALID 0x15 */
/* enum: Internal-error. The hash of a component does not match the hash stored
 * in the bundle manifest.
 */
/*               MC_CMD_NVRAM_VERIFY_RC_BUNDLE_MANIFEST_COMPONENT_HASH_MISMATCH 0x16 */
/* enum: Internal-error. Component hash calculation failed. */
/*               MC_CMD_NVRAM_VERIFY_RC_BUNDLE_MANIFEST_COMPONENT_HASH_FAILED 0x17 */
/* enum: Internal-error. The component does not have a valid reflash image
 * layout.
 */
/*               MC_CMD_NVRAM_VERIFY_RC_BUNDLE_COMPONENT_REFLASH_IMAGE_INVALID 0x18 */
/* enum: The bundle processing code failed to copy a component to its target
 * partition.
 */
/*               MC_CMD_NVRAM_VERIFY_RC_BUNDLE_COMPONENT_COPY_FAILED 0x19 */
/* enum: The update operation is in-progress. */
/*               MC_CMD_NVRAM_VERIFY_RC_PENDING 0x1a */
/* enum: The update was an invalid user configuration file. */
/*               MC_CMD_NVRAM_VERIFY_RC_BAD_CONFIG 0x1b */
/* enum: The write was to the AUTO partition but the data was not recognised as
 * a valid partition.
 */
/*               MC_CMD_NVRAM_VERIFY_RC_UNKNOWN_TYPE 0x1c */
/* If the update was a user configuration, what action(s) the user must take to
 * apply the new configuration.
 */
#define       MC_CMD_NVRAM_UPDATE_FINISH_V3_OUT_ACTIONS_REQUIRED_OFST 4
#define       MC_CMD_NVRAM_UPDATE_FINISH_V3_OUT_ACTIONS_REQUIRED_LEN 4
/* enum: No action required. */
#define          MC_CMD_NVRAM_UPDATE_FINISH_V3_OUT_NONE 0x0
/* enum: The MC firmware must be rebooted (eg with MC_CMD_REBOOT). */
#define          MC_CMD_NVRAM_UPDATE_FINISH_V3_OUT_FIRMWARE_REBOOT 0x1
/* enum: The host must be rebooted. */
#define          MC_CMD_NVRAM_UPDATE_FINISH_V3_OUT_HOST_REBOOT 0x2
/* enum: The firmware and host must be rebooted (in either order). */
#define          MC_CMD_NVRAM_UPDATE_FINISH_V3_OUT_FIRMWARE_AND_HOST_REBOOT 0x3
/* enum: The host must be fully powered off. */
#define          MC_CMD_NVRAM_UPDATE_FINISH_V3_OUT_HOST_POWERCYCLE 0x4
/* If the update failed with MC_CMD_NVRAM_VERIFY_RC_BAD_CONFIG, a null-
 * terminated US-ASCII string suitable for showing to the user.
 */
#define       MC_CMD_NVRAM_UPDATE_FINISH_V3_OUT_ERROR_STRING_OFST 8
#define       MC_CMD_NVRAM_UPDATE_FINISH_V3_OUT_ERROR_STRING_LEN 80


/***********************************/
/* MC_CMD_REBOOT
 * Reboot the MC.
 *
 * The AFTER_ASSERTION flag is intended to be used when the driver notices an
 * assertion failure (at which point it is expected to perform a complete tear
 * down and reinitialise), to allow both ports to reset the MC once in an
 * atomic fashion.
 *
 * Production mc firmwares are generally compiled with REBOOT_ON_ASSERT=1,
 * which means that they will automatically reboot out of the assertion
 * handler, so this is in practise an optional operation. It is still
 * recommended that drivers execute this to support custom firmwares with
 * REBOOT_ON_ASSERT=0.
 *
 * Locks required: NONE Returns: Nothing. You get back a response with ERR=1,
 * DATALEN=0
 */
#define MC_CMD_REBOOT 0x3d
#undef MC_CMD_0x3d_PRIVILEGE_CTG

#define MC_CMD_0x3d_PRIVILEGE_CTG SRIOV_CTG_ADMIN_TSA_UNBOUND

/* MC_CMD_REBOOT_IN msgrequest */
#define    MC_CMD_REBOOT_IN_LEN 4
#define       MC_CMD_REBOOT_IN_FLAGS_OFST 0
#define       MC_CMD_REBOOT_IN_FLAGS_LEN 4
#define          MC_CMD_REBOOT_FLAGS_AFTER_ASSERTION 0x1 /* enum */

/* MC_CMD_REBOOT_OUT msgresponse */
#define    MC_CMD_REBOOT_OUT_LEN 0


/***********************************/
/* MC_CMD_SENSOR_INFO
 * Returns information about every available sensor.
 *
 * Each sensor has a single (16bit) value, and a corresponding state. The
 * mapping between value and state is nominally determined by the MC, but may
 * be implemented using up to 2 ranges per sensor.
 *
 * This call returns a mask (32bit) of the sensors that are supported by this
 * platform, then an array of sensor information structures, in order of sensor
 * type (but without gaps for unimplemented sensors). Each structure defines
 * the ranges for the corresponding sensor. An unused range is indicated by
 * equal limit values. If one range is used, a value outside that range results
 * in STATE_FATAL. If two ranges are used, a value outside the second range
 * results in STATE_FATAL while a value outside the first and inside the second
 * range results in STATE_WARNING.
 *
 * Sensor masks and sensor information arrays are organised into pages. For
 * backward compatibility, older host software can only use sensors in page 0.
 * Bit 32 in the sensor mask was previously unused, and is no reserved for use
 * as the next page flag.
 *
 * If the request does not contain a PAGE value then firmware will only return
 * page 0 of sensor information, with bit 31 in the sensor mask cleared.
 *
 * If the request contains a PAGE value then firmware responds with the sensor
 * mask and sensor information array for that page of sensors. In this case bit
 * 31 in the mask is set if another page exists.
 *
 * Locks required: None Returns: 0
 */
#define MC_CMD_SENSOR_INFO 0x41
#undef MC_CMD_0x41_PRIVILEGE_CTG

#define MC_CMD_0x41_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_SENSOR_INFO_IN msgrequest */
#define    MC_CMD_SENSOR_INFO_IN_LEN 0

/* MC_CMD_SENSOR_INFO_EXT_IN msgrequest */
#define    MC_CMD_SENSOR_INFO_EXT_IN_LEN 4
/* Which page of sensors to report.
 *
 * Page 0 contains sensors 0 to 30 (sensor 31 is the next page bit).
 *
 * Page 1 contains sensors 32 to 62 (sensor 63 is the next page bit). etc.
 */
#define       MC_CMD_SENSOR_INFO_EXT_IN_PAGE_OFST 0
#define       MC_CMD_SENSOR_INFO_EXT_IN_PAGE_LEN 4

/* MC_CMD_SENSOR_INFO_EXT_IN_V2 msgrequest */
#define    MC_CMD_SENSOR_INFO_EXT_IN_V2_LEN 8
/* Which page of sensors to report.
 *
 * Page 0 contains sensors 0 to 30 (sensor 31 is the next page bit).
 *
 * Page 1 contains sensors 32 to 62 (sensor 63 is the next page bit). etc.
 */
#define       MC_CMD_SENSOR_INFO_EXT_IN_V2_PAGE_OFST 0
#define       MC_CMD_SENSOR_INFO_EXT_IN_V2_PAGE_LEN 4
/* Flags controlling information retrieved */
#define       MC_CMD_SENSOR_INFO_EXT_IN_V2_FLAGS_OFST 4
#define       MC_CMD_SENSOR_INFO_EXT_IN_V2_FLAGS_LEN 4
#define        MC_CMD_SENSOR_INFO_EXT_IN_V2_ENGINEERING_OFST 4
#define        MC_CMD_SENSOR_INFO_EXT_IN_V2_ENGINEERING_LBN 0
#define        MC_CMD_SENSOR_INFO_EXT_IN_V2_ENGINEERING_WIDTH 1

/* MC_CMD_SENSOR_INFO_OUT msgresponse */
#define    MC_CMD_SENSOR_INFO_OUT_LENMIN 4
#define    MC_CMD_SENSOR_INFO_OUT_LENMAX 252
#define    MC_CMD_SENSOR_INFO_OUT_LENMAX_MCDI2 1020
#define    MC_CMD_SENSOR_INFO_OUT_LEN(num) (4+8*(num))
#define    MC_CMD_SENSOR_INFO_OUT_MC_CMD_SENSOR_ENTRY_NUM(len) (((len)-4)/8)
#define       MC_CMD_SENSOR_INFO_OUT_MASK_OFST 0
#define       MC_CMD_SENSOR_INFO_OUT_MASK_LEN 4
/* enum: Controller temperature: degC */
#define          MC_CMD_SENSOR_CONTROLLER_TEMP 0x0
/* enum: Phy common temperature: degC */
#define          MC_CMD_SENSOR_PHY_COMMON_TEMP 0x1
/* enum: Controller cooling: bool */
#define          MC_CMD_SENSOR_CONTROLLER_COOLING 0x2
/* enum: Phy 0 temperature: degC */
#define          MC_CMD_SENSOR_PHY0_TEMP 0x3
/* enum: Phy 0 cooling: bool */
#define          MC_CMD_SENSOR_PHY0_COOLING 0x4
/* enum: Phy 1 temperature: degC */
#define          MC_CMD_SENSOR_PHY1_TEMP 0x5
/* enum: Phy 1 cooling: bool */
#define          MC_CMD_SENSOR_PHY1_COOLING 0x6
/* enum: 1.0v power: mV */
#define          MC_CMD_SENSOR_IN_1V0 0x7
/* enum: 1.2v power: mV */
#define          MC_CMD_SENSOR_IN_1V2 0x8
/* enum: 1.8v power: mV */
#define          MC_CMD_SENSOR_IN_1V8 0x9
/* enum: 2.5v power: mV */
#define          MC_CMD_SENSOR_IN_2V5 0xa
/* enum: 3.3v power: mV */
#define          MC_CMD_SENSOR_IN_3V3 0xb
/* enum: 12v power: mV */
#define          MC_CMD_SENSOR_IN_12V0 0xc
/* enum: 1.2v analogue power: mV */
#define          MC_CMD_SENSOR_IN_1V2A 0xd
/* enum: reference voltage: mV */
#define          MC_CMD_SENSOR_IN_VREF 0xe
/* enum: AOE FPGA power: mV */
#define          MC_CMD_SENSOR_OUT_VAOE 0xf
/* enum: AOE FPGA temperature: degC */
#define          MC_CMD_SENSOR_AOE_TEMP 0x10
/* enum: AOE FPGA PSU temperature: degC */
#define          MC_CMD_SENSOR_PSU_AOE_TEMP 0x11
/* enum: AOE PSU temperature: degC */
#define          MC_CMD_SENSOR_PSU_TEMP 0x12
/* enum: Fan 0 speed: RPM */
#define          MC_CMD_SENSOR_FAN_0 0x13
/* enum: Fan 1 speed: RPM */
#define          MC_CMD_SENSOR_FAN_1 0x14
/* enum: Fan 2 speed: RPM */
#define          MC_CMD_SENSOR_FAN_2 0x15
/* enum: Fan 3 speed: RPM */
#define          MC_CMD_SENSOR_FAN_3 0x16
/* enum: Fan 4 speed: RPM */
#define          MC_CMD_SENSOR_FAN_4 0x17
/* enum: AOE FPGA input power: mV */
#define          MC_CMD_SENSOR_IN_VAOE 0x18
/* enum: AOE FPGA current: mA */
#define          MC_CMD_SENSOR_OUT_IAOE 0x19
/* enum: AOE FPGA input current: mA */
#define          MC_CMD_SENSOR_IN_IAOE 0x1a
/* enum: NIC power consumption: W */
#define          MC_CMD_SENSOR_NIC_POWER 0x1b
/* enum: 0.9v power voltage: mV */
#define          MC_CMD_SENSOR_IN_0V9 0x1c
/* enum: 0.9v power current: mA */
#define          MC_CMD_SENSOR_IN_I0V9 0x1d
/* enum: 1.2v power current: mA */
#define          MC_CMD_SENSOR_IN_I1V2 0x1e
/* enum: Not a sensor: reserved for the next page flag */
#define          MC_CMD_SENSOR_PAGE0_NEXT 0x1f
/* enum: 0.9v power voltage (at ADC): mV */
#define          MC_CMD_SENSOR_IN_0V9_ADC 0x20
/* enum: Controller temperature 2: degC */
#define          MC_CMD_SENSOR_CONTROLLER_2_TEMP 0x21
/* enum: Voltage regulator internal temperature: degC */
#define          MC_CMD_SENSOR_VREG_INTERNAL_TEMP 0x22
/* enum: 0.9V voltage regulator temperature: degC */
#define          MC_CMD_SENSOR_VREG_0V9_TEMP 0x23
/* enum: 1.2V voltage regulator temperature: degC */
#define          MC_CMD_SENSOR_VREG_1V2_TEMP 0x24
/* enum: controller internal temperature sensor voltage (internal ADC): mV */
#define          MC_CMD_SENSOR_CONTROLLER_VPTAT 0x25
/* enum: controller internal temperature (internal ADC): degC */
#define          MC_CMD_SENSOR_CONTROLLER_INTERNAL_TEMP 0x26
/* enum: controller internal temperature sensor voltage (external ADC): mV */
#define          MC_CMD_SENSOR_CONTROLLER_VPTAT_EXTADC 0x27
/* enum: controller internal temperature (external ADC): degC */
#define          MC_CMD_SENSOR_CONTROLLER_INTERNAL_TEMP_EXTADC 0x28
/* enum: ambient temperature: degC */
#define          MC_CMD_SENSOR_AMBIENT_TEMP 0x29
/* enum: air flow: bool */
#define          MC_CMD_SENSOR_AIRFLOW 0x2a
/* enum: voltage between VSS08D and VSS08D at CSR: mV */
#define          MC_CMD_SENSOR_VDD08D_VSS08D_CSR 0x2b
/* enum: voltage between VSS08D and VSS08D at CSR (external ADC): mV */
#define          MC_CMD_SENSOR_VDD08D_VSS08D_CSR_EXTADC 0x2c
/* enum: Hotpoint temperature: degC */
#define          MC_CMD_SENSOR_HOTPOINT_TEMP 0x2d
/* enum: Port 0 PHY power switch over-current: bool */
#define          MC_CMD_SENSOR_PHY_POWER_PORT0 0x2e
/* enum: Port 1 PHY power switch over-current: bool */
#define          MC_CMD_SENSOR_PHY_POWER_PORT1 0x2f
/* enum: Mop-up microcontroller reference voltage: mV */
#define          MC_CMD_SENSOR_MUM_VCC 0x30
/* enum: 0.9v power phase A voltage: mV */
#define          MC_CMD_SENSOR_IN_0V9_A 0x31
/* enum: 0.9v power phase A current: mA */
#define          MC_CMD_SENSOR_IN_I0V9_A 0x32
/* enum: 0.9V voltage regulator phase A temperature: degC */
#define          MC_CMD_SENSOR_VREG_0V9_A_TEMP 0x33
/* enum: 0.9v power phase B voltage: mV */
#define          MC_CMD_SENSOR_IN_0V9_B 0x34
/* enum: 0.9v power phase B current: mA */
#define          MC_CMD_SENSOR_IN_I0V9_B 0x35
/* enum: 0.9V voltage regulator phase B temperature: degC */
#define          MC_CMD_SENSOR_VREG_0V9_B_TEMP 0x36
/* enum: CCOM AVREG 1v2 supply (interval ADC): mV */
#define          MC_CMD_SENSOR_CCOM_AVREG_1V2_SUPPLY 0x37
/* enum: CCOM AVREG 1v2 supply (external ADC): mV */
#define          MC_CMD_SENSOR_CCOM_AVREG_1V2_SUPPLY_EXTADC 0x38
/* enum: CCOM AVREG 1v8 supply (interval ADC): mV */
#define          MC_CMD_SENSOR_CCOM_AVREG_1V8_SUPPLY 0x39
/* enum: CCOM AVREG 1v8 supply (external ADC): mV */
#define          MC_CMD_SENSOR_CCOM_AVREG_1V8_SUPPLY_EXTADC 0x3a
/* enum: CCOM RTS temperature: degC */
#define          MC_CMD_SENSOR_CONTROLLER_RTS 0x3b
/* enum: Not a sensor: reserved for the next page flag */
#define          MC_CMD_SENSOR_PAGE1_NEXT 0x3f
/* enum: controller internal temperature sensor voltage on master core
 * (internal ADC): mV
 */
#define          MC_CMD_SENSOR_CONTROLLER_MASTER_VPTAT 0x40
/* enum: controller internal temperature on master core (internal ADC): degC */
#define          MC_CMD_SENSOR_CONTROLLER_MASTER_INTERNAL_TEMP 0x41
/* enum: controller internal temperature sensor voltage on master core
 * (external ADC): mV
 */
#define          MC_CMD_SENSOR_CONTROLLER_MASTER_VPTAT_EXTADC 0x42
/* enum: controller internal temperature on master core (external ADC): degC */
#define          MC_CMD_SENSOR_CONTROLLER_MASTER_INTERNAL_TEMP_EXTADC 0x43
/* enum: controller internal temperature on slave core sensor voltage (internal
 * ADC): mV
 */
#define          MC_CMD_SENSOR_CONTROLLER_SLAVE_VPTAT 0x44
/* enum: controller internal temperature on slave core (internal ADC): degC */
#define          MC_CMD_SENSOR_CONTROLLER_SLAVE_INTERNAL_TEMP 0x45
/* enum: controller internal temperature on slave core sensor voltage (external
 * ADC): mV
 */
#define          MC_CMD_SENSOR_CONTROLLER_SLAVE_VPTAT_EXTADC 0x46
/* enum: controller internal temperature on slave core (external ADC): degC */
#define          MC_CMD_SENSOR_CONTROLLER_SLAVE_INTERNAL_TEMP_EXTADC 0x47
/* enum: Voltage supplied to the SODIMMs from their power supply: mV */
#define          MC_CMD_SENSOR_SODIMM_VOUT 0x49
/* enum: Temperature of SODIMM 0 (if installed): degC */
#define          MC_CMD_SENSOR_SODIMM_0_TEMP 0x4a
/* enum: Temperature of SODIMM 1 (if installed): degC */
#define          MC_CMD_SENSOR_SODIMM_1_TEMP 0x4b
/* enum: Voltage supplied to the QSFP #0 from their power supply: mV */
#define          MC_CMD_SENSOR_PHY0_VCC 0x4c
/* enum: Voltage supplied to the QSFP #1 from their power supply: mV */
#define          MC_CMD_SENSOR_PHY1_VCC 0x4d
/* enum: Controller die temperature (TDIODE): degC */
#define          MC_CMD_SENSOR_CONTROLLER_TDIODE_TEMP 0x4e
/* enum: Board temperature (front): degC */
#define          MC_CMD_SENSOR_BOARD_FRONT_TEMP 0x4f
/* enum: Board temperature (back): degC */
#define          MC_CMD_SENSOR_BOARD_BACK_TEMP 0x50
/* enum: 1.8v power current: mA */
#define          MC_CMD_SENSOR_IN_I1V8 0x51
/* enum: 2.5v power current: mA */
#define          MC_CMD_SENSOR_IN_I2V5 0x52
/* enum: 3.3v power current: mA */
#define          MC_CMD_SENSOR_IN_I3V3 0x53
/* enum: 12v power current: mA */
#define          MC_CMD_SENSOR_IN_I12V0 0x54
/* enum: 1.3v power: mV */
#define          MC_CMD_SENSOR_IN_1V3 0x55
/* enum: 1.3v power current: mA */
#define          MC_CMD_SENSOR_IN_I1V3 0x56
/* enum: Engineering sensor 1 */
#define          MC_CMD_SENSOR_ENGINEERING_1 0x57
/* enum: Engineering sensor 2 */
#define          MC_CMD_SENSOR_ENGINEERING_2 0x58
/* enum: Engineering sensor 3 */
#define          MC_CMD_SENSOR_ENGINEERING_3 0x59
/* enum: Engineering sensor 4 */
#define          MC_CMD_SENSOR_ENGINEERING_4 0x5a
/* enum: Engineering sensor 5 */
#define          MC_CMD_SENSOR_ENGINEERING_5 0x5b
/* enum: Engineering sensor 6 */
#define          MC_CMD_SENSOR_ENGINEERING_6 0x5c
/* enum: Engineering sensor 7 */
#define          MC_CMD_SENSOR_ENGINEERING_7 0x5d
/* enum: Engineering sensor 8 */
#define          MC_CMD_SENSOR_ENGINEERING_8 0x5e
/* enum: Not a sensor: reserved for the next page flag */
#define          MC_CMD_SENSOR_PAGE2_NEXT 0x5f
/* MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF */
#define       MC_CMD_SENSOR_ENTRY_OFST 4
#define       MC_CMD_SENSOR_ENTRY_LEN 8
#define       MC_CMD_SENSOR_ENTRY_LO_OFST 4
#define       MC_CMD_SENSOR_ENTRY_LO_LEN 4
#define       MC_CMD_SENSOR_ENTRY_LO_LBN 32
#define       MC_CMD_SENSOR_ENTRY_LO_WIDTH 32
#define       MC_CMD_SENSOR_ENTRY_HI_OFST 8
#define       MC_CMD_SENSOR_ENTRY_HI_LEN 4
#define       MC_CMD_SENSOR_ENTRY_HI_LBN 64
#define       MC_CMD_SENSOR_ENTRY_HI_WIDTH 32
#define       MC_CMD_SENSOR_ENTRY_MINNUM 0
#define       MC_CMD_SENSOR_ENTRY_MAXNUM 31
#define       MC_CMD_SENSOR_ENTRY_MAXNUM_MCDI2 127

/* MC_CMD_SENSOR_INFO_EXT_OUT msgresponse */
#define    MC_CMD_SENSOR_INFO_EXT_OUT_LENMIN 4
#define    MC_CMD_SENSOR_INFO_EXT_OUT_LENMAX 252
#define    MC_CMD_SENSOR_INFO_EXT_OUT_LENMAX_MCDI2 1020
#define    MC_CMD_SENSOR_INFO_EXT_OUT_LEN(num) (4+8*(num))
#define    MC_CMD_SENSOR_INFO_EXT_OUT_MC_CMD_SENSOR_ENTRY_NUM(len) (((len)-4)/8)
#define       MC_CMD_SENSOR_INFO_EXT_OUT_MASK_OFST 0
#define       MC_CMD_SENSOR_INFO_EXT_OUT_MASK_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_SENSOR_INFO_OUT */
#define        MC_CMD_SENSOR_INFO_EXT_OUT_NEXT_PAGE_OFST 0
#define        MC_CMD_SENSOR_INFO_EXT_OUT_NEXT_PAGE_LBN 31
#define        MC_CMD_SENSOR_INFO_EXT_OUT_NEXT_PAGE_WIDTH 1
/* MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF */
/*            MC_CMD_SENSOR_ENTRY_OFST 4 */
/*            MC_CMD_SENSOR_ENTRY_LEN 8 */
/*            MC_CMD_SENSOR_ENTRY_LO_OFST 4 */
/*            MC_CMD_SENSOR_ENTRY_LO_LEN 4 */
/*            MC_CMD_SENSOR_ENTRY_LO_LBN 32 */
/*            MC_CMD_SENSOR_ENTRY_LO_WIDTH 32 */
/*            MC_CMD_SENSOR_ENTRY_HI_OFST 8 */
/*            MC_CMD_SENSOR_ENTRY_HI_LEN 4 */
/*            MC_CMD_SENSOR_ENTRY_HI_LBN 64 */
/*            MC_CMD_SENSOR_ENTRY_HI_WIDTH 32 */
/*            MC_CMD_SENSOR_ENTRY_MINNUM 0 */
/*            MC_CMD_SENSOR_ENTRY_MAXNUM 31 */
/*            MC_CMD_SENSOR_ENTRY_MAXNUM_MCDI2 127 */

/* MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF structuredef */
#define    MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_LEN 8
#define       MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_MIN1_OFST 0
#define       MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_MIN1_LEN 2
#define       MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_MIN1_LBN 0
#define       MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_MIN1_WIDTH 16
#define       MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_MAX1_OFST 2
#define       MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_MAX1_LEN 2
#define       MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_MAX1_LBN 16
#define       MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_MAX1_WIDTH 16
#define       MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_MIN2_OFST 4
#define       MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_MIN2_LEN 2
#define       MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_MIN2_LBN 32
#define       MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_MIN2_WIDTH 16
#define       MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_MAX2_OFST 6
#define       MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_MAX2_LEN 2
#define       MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_MAX2_LBN 48
#define       MC_CMD_SENSOR_INFO_ENTRY_TYPEDEF_MAX2_WIDTH 16


/***********************************/
/* MC_CMD_READ_SENSORS
 * Returns the current reading from each sensor. DMAs an array of sensor
 * readings, in order of sensor type (but without gaps for unimplemented
 * sensors), into host memory. Each array element is a
 * MC_CMD_SENSOR_VALUE_ENTRY_TYPEDEF dword.
 *
 * If the request does not contain the LENGTH field then only sensors 0 to 30
 * are reported, to avoid DMA buffer overflow in older host software. If the
 * sensor reading require more space than the LENGTH allows, then return
 * EINVAL.
 *
 * The MC will send a SENSOREVT event every time any sensor changes state. The
 * driver is responsible for ensuring that it doesn't miss any events. The
 * board will function normally if all sensors are in STATE_OK or
 * STATE_WARNING. Otherwise the board should not be expected to function.
 */
#define MC_CMD_READ_SENSORS 0x42
#undef MC_CMD_0x42_PRIVILEGE_CTG

#define MC_CMD_0x42_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_READ_SENSORS_IN msgrequest */
#define    MC_CMD_READ_SENSORS_IN_LEN 8
/* DMA address of host buffer for sensor readings (must be 4Kbyte aligned).
 *
 * If the address is 0xffffffffffffffff send the readings in the response (used
 * by cmdclient).
 */
#define       MC_CMD_READ_SENSORS_IN_DMA_ADDR_OFST 0
#define       MC_CMD_READ_SENSORS_IN_DMA_ADDR_LEN 8
#define       MC_CMD_READ_SENSORS_IN_DMA_ADDR_LO_OFST 0
#define       MC_CMD_READ_SENSORS_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_READ_SENSORS_IN_DMA_ADDR_LO_LBN 0
#define       MC_CMD_READ_SENSORS_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_READ_SENSORS_IN_DMA_ADDR_HI_OFST 4
#define       MC_CMD_READ_SENSORS_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_READ_SENSORS_IN_DMA_ADDR_HI_LBN 32
#define       MC_CMD_READ_SENSORS_IN_DMA_ADDR_HI_WIDTH 32

/* MC_CMD_READ_SENSORS_EXT_IN msgrequest */
#define    MC_CMD_READ_SENSORS_EXT_IN_LEN 12
/* DMA address of host buffer for sensor readings (must be 4Kbyte aligned).
 *
 * If the address is 0xffffffffffffffff send the readings in the response (used
 * by cmdclient).
 */
#define       MC_CMD_READ_SENSORS_EXT_IN_DMA_ADDR_OFST 0
#define       MC_CMD_READ_SENSORS_EXT_IN_DMA_ADDR_LEN 8
#define       MC_CMD_READ_SENSORS_EXT_IN_DMA_ADDR_LO_OFST 0
#define       MC_CMD_READ_SENSORS_EXT_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_READ_SENSORS_EXT_IN_DMA_ADDR_LO_LBN 0
#define       MC_CMD_READ_SENSORS_EXT_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_READ_SENSORS_EXT_IN_DMA_ADDR_HI_OFST 4
#define       MC_CMD_READ_SENSORS_EXT_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_READ_SENSORS_EXT_IN_DMA_ADDR_HI_LBN 32
#define       MC_CMD_READ_SENSORS_EXT_IN_DMA_ADDR_HI_WIDTH 32
/* Size in bytes of host buffer. */
#define       MC_CMD_READ_SENSORS_EXT_IN_LENGTH_OFST 8
#define       MC_CMD_READ_SENSORS_EXT_IN_LENGTH_LEN 4

/* MC_CMD_READ_SENSORS_EXT_IN_V2 msgrequest */
#define    MC_CMD_READ_SENSORS_EXT_IN_V2_LEN 16
/* DMA address of host buffer for sensor readings (must be 4Kbyte aligned).
 *
 * If the address is 0xffffffffffffffff send the readings in the response (used
 * by cmdclient).
 */
#define       MC_CMD_READ_SENSORS_EXT_IN_V2_DMA_ADDR_OFST 0
#define       MC_CMD_READ_SENSORS_EXT_IN_V2_DMA_ADDR_LEN 8
#define       MC_CMD_READ_SENSORS_EXT_IN_V2_DMA_ADDR_LO_OFST 0
#define       MC_CMD_READ_SENSORS_EXT_IN_V2_DMA_ADDR_LO_LEN 4
#define       MC_CMD_READ_SENSORS_EXT_IN_V2_DMA_ADDR_LO_LBN 0
#define       MC_CMD_READ_SENSORS_EXT_IN_V2_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_READ_SENSORS_EXT_IN_V2_DMA_ADDR_HI_OFST 4
#define       MC_CMD_READ_SENSORS_EXT_IN_V2_DMA_ADDR_HI_LEN 4
#define       MC_CMD_READ_SENSORS_EXT_IN_V2_DMA_ADDR_HI_LBN 32
#define       MC_CMD_READ_SENSORS_EXT_IN_V2_DMA_ADDR_HI_WIDTH 32
/* Size in bytes of host buffer. */
#define       MC_CMD_READ_SENSORS_EXT_IN_V2_LENGTH_OFST 8
#define       MC_CMD_READ_SENSORS_EXT_IN_V2_LENGTH_LEN 4
/* Flags controlling information retrieved */
#define       MC_CMD_READ_SENSORS_EXT_IN_V2_FLAGS_OFST 12
#define       MC_CMD_READ_SENSORS_EXT_IN_V2_FLAGS_LEN 4
#define        MC_CMD_READ_SENSORS_EXT_IN_V2_ENGINEERING_OFST 12
#define        MC_CMD_READ_SENSORS_EXT_IN_V2_ENGINEERING_LBN 0
#define        MC_CMD_READ_SENSORS_EXT_IN_V2_ENGINEERING_WIDTH 1

/* MC_CMD_READ_SENSORS_OUT msgresponse */
#define    MC_CMD_READ_SENSORS_OUT_LEN 0

/* MC_CMD_READ_SENSORS_EXT_OUT msgresponse */
#define    MC_CMD_READ_SENSORS_EXT_OUT_LEN 0

/* MC_CMD_SENSOR_VALUE_ENTRY_TYPEDEF structuredef */
#define    MC_CMD_SENSOR_VALUE_ENTRY_TYPEDEF_LEN 4
#define       MC_CMD_SENSOR_VALUE_ENTRY_TYPEDEF_VALUE_OFST 0
#define       MC_CMD_SENSOR_VALUE_ENTRY_TYPEDEF_VALUE_LEN 2
#define       MC_CMD_SENSOR_VALUE_ENTRY_TYPEDEF_VALUE_LBN 0
#define       MC_CMD_SENSOR_VALUE_ENTRY_TYPEDEF_VALUE_WIDTH 16
#define       MC_CMD_SENSOR_VALUE_ENTRY_TYPEDEF_STATE_OFST 2
#define       MC_CMD_SENSOR_VALUE_ENTRY_TYPEDEF_STATE_LEN 1
/* enum: Ok. */
#define          MC_CMD_SENSOR_STATE_OK 0x0
/* enum: Breached warning threshold. */
#define          MC_CMD_SENSOR_STATE_WARNING 0x1
/* enum: Breached fatal threshold. */
#define          MC_CMD_SENSOR_STATE_FATAL 0x2
/* enum: Fault with sensor. */
#define          MC_CMD_SENSOR_STATE_BROKEN 0x3
/* enum: Sensor is working but does not currently have a reading. */
#define          MC_CMD_SENSOR_STATE_NO_READING 0x4
/* enum: Sensor initialisation failed. */
#define          MC_CMD_SENSOR_STATE_INIT_FAILED 0x5
#define       MC_CMD_SENSOR_VALUE_ENTRY_TYPEDEF_STATE_LBN 16
#define       MC_CMD_SENSOR_VALUE_ENTRY_TYPEDEF_STATE_WIDTH 8
#define       MC_CMD_SENSOR_VALUE_ENTRY_TYPEDEF_TYPE_OFST 3
#define       MC_CMD_SENSOR_VALUE_ENTRY_TYPEDEF_TYPE_LEN 1
/*            Enum values, see field(s): */
/*               MC_CMD_SENSOR_INFO/MC_CMD_SENSOR_INFO_OUT/MASK */
#define       MC_CMD_SENSOR_VALUE_ENTRY_TYPEDEF_TYPE_LBN 24
#define       MC_CMD_SENSOR_VALUE_ENTRY_TYPEDEF_TYPE_WIDTH 8


/***********************************/
/* MC_CMD_GET_PHY_STATE
 * Report current state of PHY. A 'zombie' PHY is a PHY that has failed to boot
 * (e.g. due to missing or corrupted firmware). Locks required: None. Return
 * code: 0
 */
#define MC_CMD_GET_PHY_STATE 0x43
#undef MC_CMD_0x43_PRIVILEGE_CTG

#define MC_CMD_0x43_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_GET_PHY_STATE_IN msgrequest */
#define    MC_CMD_GET_PHY_STATE_IN_LEN 0

/* MC_CMD_GET_PHY_STATE_IN_V2 msgrequest */
#define    MC_CMD_GET_PHY_STATE_IN_V2_LEN 8
/* Target port to request PHY state for. Uses MAE_LINK_ENDPOINT_SELECTOR which
 * identifies a real or virtual network port by MAE port and link end. See the
 * structure definition for more details.
 */
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_OFST 0
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_LEN 8
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_LO_OFST 0
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_LO_LEN 4
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_LO_LBN 0
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_LO_WIDTH 32
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_HI_OFST 4
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_HI_LEN 4
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_HI_LBN 32
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_HI_WIDTH 32
/* See structuredef: MAE_LINK_ENDPOINT_SELECTOR */
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_OFST 0
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_LEN 4
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_FLAT_OFST 0
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_FLAT_LEN 4
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_TYPE_OFST 3
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_TYPE_LEN 1
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_MPORT_ID_OFST 0
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_MPORT_ID_LEN 3
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_PPORT_ID_LBN 0
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_PPORT_ID_WIDTH 4
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_FUNC_INTF_ID_LBN 20
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_FUNC_INTF_ID_WIDTH 4
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_FUNC_MH_PF_ID_LBN 16
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_FUNC_MH_PF_ID_WIDTH 4
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_FUNC_PF_ID_OFST 2
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_FUNC_PF_ID_LEN 1
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_FUNC_VF_ID_OFST 0
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_MPORT_SELECTOR_FUNC_VF_ID_LEN 2
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_LINK_END_OFST 4
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_LINK_END_LEN 4
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_FLAT_OFST 0
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_FLAT_LEN 8
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_FLAT_LO_OFST 0
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_FLAT_LO_LEN 4
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_FLAT_LO_LBN 0
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_FLAT_LO_WIDTH 32
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_FLAT_HI_OFST 4
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_FLAT_HI_LEN 4
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_FLAT_HI_LBN 32
#define       MC_CMD_GET_PHY_STATE_IN_V2_TARGET_FLAT_HI_WIDTH 32

/* MC_CMD_GET_PHY_STATE_OUT msgresponse */
#define    MC_CMD_GET_PHY_STATE_OUT_LEN 4
#define       MC_CMD_GET_PHY_STATE_OUT_STATE_OFST 0
#define       MC_CMD_GET_PHY_STATE_OUT_STATE_LEN 4
/* enum: Ok. */
#define          MC_CMD_PHY_STATE_OK 0x1
/* enum: Faulty. */
#define          MC_CMD_PHY_STATE_ZOMBIE 0x2


/***********************************/
/* MC_CMD_WOL_FILTER_GET
 * Retrieve ID of any WoL filters. Locks required: None. Returns: 0, ENOSYS
 */
#define MC_CMD_WOL_FILTER_GET 0x45
#undef MC_CMD_0x45_PRIVILEGE_CTG

#define MC_CMD_0x45_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_WOL_FILTER_GET_IN msgrequest */
#define    MC_CMD_WOL_FILTER_GET_IN_LEN 0

/* MC_CMD_WOL_FILTER_GET_OUT msgresponse */
#define    MC_CMD_WOL_FILTER_GET_OUT_LEN 4
#define       MC_CMD_WOL_FILTER_GET_OUT_FILTER_ID_OFST 0
#define       MC_CMD_WOL_FILTER_GET_OUT_FILTER_ID_LEN 4


/***********************************/
/* MC_CMD_WORKAROUND
 * Enable/Disable a given workaround. The mcfw will return EINVAL if it doesn't
 * understand the given workaround number - which should not be treated as a
 * hard error by client code. This op does not imply any semantics about each
 * workaround, that's between the driver and the mcfw on a per-workaround
 * basis. Locks required: None. Returns: 0, EINVAL .
 */
#define MC_CMD_WORKAROUND 0x4a
#undef MC_CMD_0x4a_PRIVILEGE_CTG

#define MC_CMD_0x4a_PRIVILEGE_CTG SRIOV_CTG_ADMIN

/* MC_CMD_WORKAROUND_IN msgrequest */
#define    MC_CMD_WORKAROUND_IN_LEN 8
/* The enums here must correspond with those in MC_CMD_GET_WORKAROUND. */
#define       MC_CMD_WORKAROUND_IN_TYPE_OFST 0
#define       MC_CMD_WORKAROUND_IN_TYPE_LEN 4
/* enum: Bug 17230 work around. */
#define          MC_CMD_WORKAROUND_BUG17230 0x1
/* enum: Bug 35388 work around (unsafe EVQ writes). */
#define          MC_CMD_WORKAROUND_BUG35388 0x2
/* enum: Bug35017 workaround (A64 tables must be identity map) */
#define          MC_CMD_WORKAROUND_BUG35017 0x3
/* enum: Bug 41750 present (MC_CMD_TRIGGER_INTERRUPT won't work) */
#define          MC_CMD_WORKAROUND_BUG41750 0x4
/* enum: Bug 42008 present (Interrupts can overtake associated events). Caution
 * - before adding code that queries this workaround, remember that there's
 * released Monza firmware that doesn't understand MC_CMD_WORKAROUND_BUG42008,
 * and will hence (incorrectly) report that the bug doesn't exist.
 */
#define          MC_CMD_WORKAROUND_BUG42008 0x5
/* enum: Bug 26807 features present in firmware (multicast filter chaining)
 * This feature cannot be turned on/off while there are any filters already
 * present. The behaviour in such case depends on the acting client's privilege
 * level. If the client has the admin privilege, then all functions that have
 * filters installed will be FLRed and the FLR_DONE flag will be set. Otherwise
 * the command will fail with MC_CMD_ERR_FILTERS_PRESENT.
 */
#define          MC_CMD_WORKAROUND_BUG26807 0x6
/* enum: Bug 61265 work around (broken EVQ TMR writes). */
#define          MC_CMD_WORKAROUND_BUG61265 0x7
/* 0 = disable the workaround indicated by TYPE; any non-zero value = enable
 * the workaround
 */
#define       MC_CMD_WORKAROUND_IN_ENABLED_OFST 4
#define       MC_CMD_WORKAROUND_IN_ENABLED_LEN 4

/* MC_CMD_WORKAROUND_OUT msgresponse */
#define    MC_CMD_WORKAROUND_OUT_LEN 0

/* MC_CMD_WORKAROUND_EXT_OUT msgresponse: This response format will be used
 * when (TYPE == MC_CMD_WORKAROUND_BUG26807)
 */
#define    MC_CMD_WORKAROUND_EXT_OUT_LEN 4
#define       MC_CMD_WORKAROUND_EXT_OUT_FLAGS_OFST 0
#define       MC_CMD_WORKAROUND_EXT_OUT_FLAGS_LEN 4
#define        MC_CMD_WORKAROUND_EXT_OUT_FLR_DONE_OFST 0
#define        MC_CMD_WORKAROUND_EXT_OUT_FLR_DONE_LBN 0
#define        MC_CMD_WORKAROUND_EXT_OUT_FLR_DONE_WIDTH 1


/***********************************/
/* MC_CMD_GET_PHY_MEDIA_INFO
 * Read media-specific data from PHY (e.g. SFP/SFP+ module ID information for
 * SFP+ PHYs). The "media type" can be found via GET_PHY_CFG
 * (GET_PHY_CFG_OUT_MEDIA_TYPE); the valid "page number" input values, and the
 * output data, are interpreted on a per-type basis. For SFP+, PAGE=0 or 1
 * returns a 128-byte block read from module I2C address 0xA0 offset 0 or 0x80.
 * For QSFP, PAGE=-1 is the lower (unbanked) page. PAGE=2 is the EEPROM and
 * PAGE=3 is the module limits. For DSFP, module addressing requires a
 * "BANK:PAGE". Not every bank has the same number of pages. See the Common
 * Management Interface Specification (CMIS) for further details. A BANK:PAGE
 * of "0xffff:0xffff" retrieves the lower (unbanked) page. Locks required -
 * None. Return code - 0.
 */
#define MC_CMD_GET_PHY_MEDIA_INFO 0x4b
#undef MC_CMD_0x4b_PRIVILEGE_CTG

#define MC_CMD_0x4b_PRIVILEGE_CTG SRIOV_CTG_ADMIN

/* MC_CMD_GET_PHY_MEDIA_INFO_IN msgrequest */
#define    MC_CMD_GET_PHY_MEDIA_INFO_IN_LEN 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_PAGE_OFST 0
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_PAGE_LEN 4
#define        MC_CMD_GET_PHY_MEDIA_INFO_IN_DSFP_PAGE_OFST 0
#define        MC_CMD_GET_PHY_MEDIA_INFO_IN_DSFP_PAGE_LBN 0
#define        MC_CMD_GET_PHY_MEDIA_INFO_IN_DSFP_PAGE_WIDTH 16
#define        MC_CMD_GET_PHY_MEDIA_INFO_IN_DSFP_BANK_OFST 0
#define        MC_CMD_GET_PHY_MEDIA_INFO_IN_DSFP_BANK_LBN 16
#define        MC_CMD_GET_PHY_MEDIA_INFO_IN_DSFP_BANK_WIDTH 16

/* MC_CMD_GET_PHY_MEDIA_INFO_IN_V2 msgrequest */
#define    MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_LEN 12
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_PAGE_OFST 0
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_PAGE_LEN 4
#define        MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_DSFP_PAGE_OFST 0
#define        MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_DSFP_PAGE_LBN 0
#define        MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_DSFP_PAGE_WIDTH 16
#define        MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_DSFP_BANK_OFST 0
#define        MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_DSFP_BANK_LBN 16
#define        MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_DSFP_BANK_WIDTH 16
/* Target port to request PHY state for. Uses MAE_LINK_ENDPOINT_SELECTOR which
 * identifies a real or virtual network port by MAE port and link end. See the
 * structure definition for more details
 */
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_OFST 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_LEN 8
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_LO_OFST 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_LO_LEN 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_LO_LBN 32
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_LO_WIDTH 32
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_HI_OFST 8
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_HI_LEN 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_HI_LBN 64
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_HI_WIDTH 32
/* See structuredef: MAE_LINK_ENDPOINT_SELECTOR */
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_OFST 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_LEN 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_FLAT_OFST 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_FLAT_LEN 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_TYPE_OFST 7
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_TYPE_LEN 1
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_MPORT_ID_OFST 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_MPORT_ID_LEN 3
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_PPORT_ID_LBN 32
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_PPORT_ID_WIDTH 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_FUNC_INTF_ID_LBN 52
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_FUNC_INTF_ID_WIDTH 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_FUNC_MH_PF_ID_LBN 48
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_FUNC_MH_PF_ID_WIDTH 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_FUNC_PF_ID_OFST 6
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_FUNC_PF_ID_LEN 1
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_FUNC_VF_ID_OFST 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_MPORT_SELECTOR_FUNC_VF_ID_LEN 2
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_LINK_END_OFST 8
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_LINK_END_LEN 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_FLAT_OFST 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_FLAT_LEN 8
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_FLAT_LO_OFST 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_FLAT_LO_LEN 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_FLAT_LO_LBN 32
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_FLAT_LO_WIDTH 32
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_FLAT_HI_OFST 8
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_FLAT_HI_LEN 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_FLAT_HI_LBN 64
#define       MC_CMD_GET_PHY_MEDIA_INFO_IN_V2_TARGET_FLAT_HI_WIDTH 32

/* MC_CMD_GET_PHY_MEDIA_INFO_OUT msgresponse */
#define    MC_CMD_GET_PHY_MEDIA_INFO_OUT_LENMIN 5
#define    MC_CMD_GET_PHY_MEDIA_INFO_OUT_LENMAX 252
#define    MC_CMD_GET_PHY_MEDIA_INFO_OUT_LENMAX_MCDI2 1020
#define    MC_CMD_GET_PHY_MEDIA_INFO_OUT_LEN(num) (4+1*(num))
#define    MC_CMD_GET_PHY_MEDIA_INFO_OUT_DATA_NUM(len) (((len)-4)/1)
/* in bytes */
#define       MC_CMD_GET_PHY_MEDIA_INFO_OUT_DATALEN_OFST 0
#define       MC_CMD_GET_PHY_MEDIA_INFO_OUT_DATALEN_LEN 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_OUT_DATA_OFST 4
#define       MC_CMD_GET_PHY_MEDIA_INFO_OUT_DATA_LEN 1
#define       MC_CMD_GET_PHY_MEDIA_INFO_OUT_DATA_MINNUM 1
#define       MC_CMD_GET_PHY_MEDIA_INFO_OUT_DATA_MAXNUM 248
#define       MC_CMD_GET_PHY_MEDIA_INFO_OUT_DATA_MAXNUM_MCDI2 1016


/***********************************/
/* MC_CMD_NVRAM_TEST
 * Test a particular NVRAM partition for valid contents (where "valid" depends
 * on the type of partition).
 */
#define MC_CMD_NVRAM_TEST 0x4c
#undef MC_CMD_0x4c_PRIVILEGE_CTG

#define MC_CMD_0x4c_PRIVILEGE_CTG SRIOV_CTG_ADMIN_TSA_UNBOUND

/* MC_CMD_NVRAM_TEST_IN msgrequest */
#define    MC_CMD_NVRAM_TEST_IN_LEN 4
#define       MC_CMD_NVRAM_TEST_IN_TYPE_OFST 0
#define       MC_CMD_NVRAM_TEST_IN_TYPE_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_NVRAM_TYPES/MC_CMD_NVRAM_TYPES_OUT/TYPES */

/* MC_CMD_NVRAM_TEST_OUT msgresponse */
#define    MC_CMD_NVRAM_TEST_OUT_LEN 4
#define       MC_CMD_NVRAM_TEST_OUT_RESULT_OFST 0
#define       MC_CMD_NVRAM_TEST_OUT_RESULT_LEN 4
/* enum: Passed. */
#define          MC_CMD_NVRAM_TEST_PASS 0x0
/* enum: Failed. */
#define          MC_CMD_NVRAM_TEST_FAIL 0x1
/* enum: Not supported. */
#define          MC_CMD_NVRAM_TEST_NOTSUPP 0x2


/***********************************/
/* MC_CMD_NVRAM_PARTITIONS
 * Reads the list of available virtual NVRAM partition types. Locks required:
 * none. Returns: 0, EINVAL (bad type).
 */
#define MC_CMD_NVRAM_PARTITIONS 0x51
#undef MC_CMD_0x51_PRIVILEGE_CTG

#define MC_CMD_0x51_PRIVILEGE_CTG SRIOV_CTG_ADMIN

/* MC_CMD_NVRAM_PARTITIONS_IN msgrequest */
#define    MC_CMD_NVRAM_PARTITIONS_IN_LEN 0

/* MC_CMD_NVRAM_PARTITIONS_OUT msgresponse */
#define    MC_CMD_NVRAM_PARTITIONS_OUT_LENMIN 4
#define    MC_CMD_NVRAM_PARTITIONS_OUT_LENMAX 252
#define    MC_CMD_NVRAM_PARTITIONS_OUT_LENMAX_MCDI2 1020
#define    MC_CMD_NVRAM_PARTITIONS_OUT_LEN(num) (4+4*(num))
#define    MC_CMD_NVRAM_PARTITIONS_OUT_TYPE_ID_NUM(len) (((len)-4)/4)
/* total number of partitions */
#define       MC_CMD_NVRAM_PARTITIONS_OUT_NUM_PARTITIONS_OFST 0
#define       MC_CMD_NVRAM_PARTITIONS_OUT_NUM_PARTITIONS_LEN 4
/* type ID code for each of NUM_PARTITIONS partitions */
#define       MC_CMD_NVRAM_PARTITIONS_OUT_TYPE_ID_OFST 4
#define       MC_CMD_NVRAM_PARTITIONS_OUT_TYPE_ID_LEN 4
#define       MC_CMD_NVRAM_PARTITIONS_OUT_TYPE_ID_MINNUM 0
#define       MC_CMD_NVRAM_PARTITIONS_OUT_TYPE_ID_MAXNUM 62
#define       MC_CMD_NVRAM_PARTITIONS_OUT_TYPE_ID_MAXNUM_MCDI2 254


/***********************************/
/* MC_CMD_NVRAM_METADATA
 * Reads soft metadata for a virtual NVRAM partition type. Locks required:
 * none. Returns: 0, EINVAL (bad type).
 */
#define MC_CMD_NVRAM_METADATA 0x52
#undef MC_CMD_0x52_PRIVILEGE_CTG

#define MC_CMD_0x52_PRIVILEGE_CTG SRIOV_CTG_ADMIN

/* MC_CMD_NVRAM_METADATA_IN msgrequest */
#define    MC_CMD_NVRAM_METADATA_IN_LEN 4
/* Partition type ID code */
#define       MC_CMD_NVRAM_METADATA_IN_TYPE_OFST 0
#define       MC_CMD_NVRAM_METADATA_IN_TYPE_LEN 4

/* MC_CMD_NVRAM_METADATA_OUT msgresponse */
#define    MC_CMD_NVRAM_METADATA_OUT_LENMIN 20
#define    MC_CMD_NVRAM_METADATA_OUT_LENMAX 252
#define    MC_CMD_NVRAM_METADATA_OUT_LENMAX_MCDI2 1020
#define    MC_CMD_NVRAM_METADATA_OUT_LEN(num) (20+1*(num))
#define    MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_NUM(len) (((len)-20)/1)
/* Partition type ID code */
#define       MC_CMD_NVRAM_METADATA_OUT_TYPE_OFST 0
#define       MC_CMD_NVRAM_METADATA_OUT_TYPE_LEN 4
#define       MC_CMD_NVRAM_METADATA_OUT_FLAGS_OFST 4
#define       MC_CMD_NVRAM_METADATA_OUT_FLAGS_LEN 4
#define        MC_CMD_NVRAM_METADATA_OUT_SUBTYPE_VALID_OFST 4
#define        MC_CMD_NVRAM_METADATA_OUT_SUBTYPE_VALID_LBN 0
#define        MC_CMD_NVRAM_METADATA_OUT_SUBTYPE_VALID_WIDTH 1
#define        MC_CMD_NVRAM_METADATA_OUT_VERSION_VALID_OFST 4
#define        MC_CMD_NVRAM_METADATA_OUT_VERSION_VALID_LBN 1
#define        MC_CMD_NVRAM_METADATA_OUT_VERSION_VALID_WIDTH 1
#define        MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_VALID_OFST 4
#define        MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_VALID_LBN 2
#define        MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_VALID_WIDTH 1
/* Subtype ID code for content of this partition */
#define       MC_CMD_NVRAM_METADATA_OUT_SUBTYPE_OFST 8
#define       MC_CMD_NVRAM_METADATA_OUT_SUBTYPE_LEN 4
/* 1st component of W.X.Y.Z version number for content of this partition */
#define       MC_CMD_NVRAM_METADATA_OUT_VERSION_W_OFST 12
#define       MC_CMD_NVRAM_METADATA_OUT_VERSION_W_LEN 2
/* 2nd component of W.X.Y.Z version number for content of this partition */
#define       MC_CMD_NVRAM_METADATA_OUT_VERSION_X_OFST 14
#define       MC_CMD_NVRAM_METADATA_OUT_VERSION_X_LEN 2
/* 3rd component of W.X.Y.Z version number for content of this partition */
#define       MC_CMD_NVRAM_METADATA_OUT_VERSION_Y_OFST 16
#define       MC_CMD_NVRAM_METADATA_OUT_VERSION_Y_LEN 2
/* 4th component of W.X.Y.Z version number for content of this partition */
#define       MC_CMD_NVRAM_METADATA_OUT_VERSION_Z_OFST 18
#define       MC_CMD_NVRAM_METADATA_OUT_VERSION_Z_LEN 2
/* Zero-terminated string describing the content of this partition */
#define       MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_OFST 20
#define       MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_LEN 1
#define       MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_MINNUM 0
#define       MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_MAXNUM 232
#define       MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_MAXNUM_MCDI2 1000


/***********************************/
/* MC_CMD_GET_MAC_ADDRESSES
 * Returns the base MAC, count and stride for the requesting function
 */
#define MC_CMD_GET_MAC_ADDRESSES 0x55
#undef MC_CMD_0x55_PRIVILEGE_CTG

#define MC_CMD_0x55_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_GET_MAC_ADDRESSES_IN msgrequest */
#define    MC_CMD_GET_MAC_ADDRESSES_IN_LEN 0

/* MC_CMD_GET_MAC_ADDRESSES_OUT msgresponse */
#define    MC_CMD_GET_MAC_ADDRESSES_OUT_LEN 16
/* Base MAC address */
#define       MC_CMD_GET_MAC_ADDRESSES_OUT_MAC_ADDR_BASE_OFST 0
#define       MC_CMD_GET_MAC_ADDRESSES_OUT_MAC_ADDR_BASE_LEN 6
/* Padding */
#define       MC_CMD_GET_MAC_ADDRESSES_OUT_RESERVED_OFST 6
#define       MC_CMD_GET_MAC_ADDRESSES_OUT_RESERVED_LEN 2
/* Number of allocated MAC addresses */
#define       MC_CMD_GET_MAC_ADDRESSES_OUT_MAC_COUNT_OFST 8
#define       MC_CMD_GET_MAC_ADDRESSES_OUT_MAC_COUNT_LEN 4
/* Spacing of allocated MAC addresses */
#define       MC_CMD_GET_MAC_ADDRESSES_OUT_MAC_STRIDE_OFST 12
#define       MC_CMD_GET_MAC_ADDRESSES_OUT_MAC_STRIDE_LEN 4

/* MC_CMD_DYNAMIC_SENSORS_LIMITS structuredef: Set of sensor limits. This
 * should match the equivalent structure in the sensor_query SPHINX service.
 */
#define    MC_CMD_DYNAMIC_SENSORS_LIMITS_LEN 24
/* A value below this will trigger a warning event. */
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_LO_WARNING_OFST 0
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_LO_WARNING_LEN 4
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_LO_WARNING_LBN 0
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_LO_WARNING_WIDTH 32
/* A value below this will trigger a critical event. */
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_LO_CRITICAL_OFST 4
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_LO_CRITICAL_LEN 4
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_LO_CRITICAL_LBN 32
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_LO_CRITICAL_WIDTH 32
/* A value below this will shut down the card. */
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_LO_FATAL_OFST 8
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_LO_FATAL_LEN 4
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_LO_FATAL_LBN 64
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_LO_FATAL_WIDTH 32
/* A value above this will trigger a warning event. */
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_HI_WARNING_OFST 12
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_HI_WARNING_LEN 4
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_HI_WARNING_LBN 96
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_HI_WARNING_WIDTH 32
/* A value above this will trigger a critical event. */
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_HI_CRITICAL_OFST 16
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_HI_CRITICAL_LEN 4
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_HI_CRITICAL_LBN 128
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_HI_CRITICAL_WIDTH 32
/* A value above this will shut down the card. */
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_HI_FATAL_OFST 20
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_HI_FATAL_LEN 4
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_HI_FATAL_LBN 160
#define       MC_CMD_DYNAMIC_SENSORS_LIMITS_HI_FATAL_WIDTH 32

/* MC_CMD_DYNAMIC_SENSORS_DESCRIPTION structuredef: Description of a sensor.
 * This should match the equivalent structure in the sensor_query SPHINX
 * service.
 */
#define    MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_LEN 64
/* The handle used to identify the sensor in calls to
 * MC_CMD_DYNAMIC_SENSORS_GET_VALUES
 */
#define       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_HANDLE_OFST 0
#define       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_HANDLE_LEN 4
#define       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_HANDLE_LBN 0
#define       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_HANDLE_WIDTH 32
/* A human-readable name for the sensor (zero terminated string, max 32 bytes)
 */
#define       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_NAME_OFST 4
#define       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_NAME_LEN 32
#define       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_NAME_LBN 32
#define       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_NAME_WIDTH 256
/* The type of the sensor device, and by implication the unit of that the
 * values will be reported in
 */
#define       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_TYPE_OFST 36
#define       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_TYPE_LEN 4
/* enum: A voltage sensor. Unit is mV */
#define          MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_VOLTAGE 0x0
/* enum: A current sensor. Unit is mA */
#define          MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_CURRENT 0x1
/* enum: A power sensor. Unit is mW */
#define          MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_POWER 0x2
/* enum: A temperature sensor. Unit is Celsius */
#define          MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_TEMPERATURE 0x3
/* enum: A cooling fan sensor. Unit is RPM */
#define          MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_FAN 0x4
#define       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_TYPE_LBN 288
#define       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_TYPE_WIDTH 32
/* A single MC_CMD_DYNAMIC_SENSORS_LIMITS structure */
#define       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_LIMITS_OFST 40
#define       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_LIMITS_LEN 24
#define       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_LIMITS_LBN 320
#define       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_LIMITS_WIDTH 192

/* MC_CMD_DYNAMIC_SENSORS_READING structuredef: State and value of a sensor.
 * This should match the equivalent structure in the sensor_query SPHINX
 * service.
 */
#define    MC_CMD_DYNAMIC_SENSORS_READING_LEN 12
/* The handle used to identify the sensor */
#define       MC_CMD_DYNAMIC_SENSORS_READING_HANDLE_OFST 0
#define       MC_CMD_DYNAMIC_SENSORS_READING_HANDLE_LEN 4
#define       MC_CMD_DYNAMIC_SENSORS_READING_HANDLE_LBN 0
#define       MC_CMD_DYNAMIC_SENSORS_READING_HANDLE_WIDTH 32
/* The current value of the sensor */
#define       MC_CMD_DYNAMIC_SENSORS_READING_VALUE_OFST 4
#define       MC_CMD_DYNAMIC_SENSORS_READING_VALUE_LEN 4
#define       MC_CMD_DYNAMIC_SENSORS_READING_VALUE_LBN 32
#define       MC_CMD_DYNAMIC_SENSORS_READING_VALUE_WIDTH 32
/* The sensor's condition, e.g. good, broken or removed */
#define       MC_CMD_DYNAMIC_SENSORS_READING_STATE_OFST 8
#define       MC_CMD_DYNAMIC_SENSORS_READING_STATE_LEN 4
/* enum: Sensor working normally within limits */
#define          MC_CMD_DYNAMIC_SENSORS_READING_OK 0x0
/* enum: Warning threshold breached */
#define          MC_CMD_DYNAMIC_SENSORS_READING_WARNING 0x1
/* enum: Critical threshold breached */
#define          MC_CMD_DYNAMIC_SENSORS_READING_CRITICAL 0x2
/* enum: Fatal threshold breached */
#define          MC_CMD_DYNAMIC_SENSORS_READING_FATAL 0x3
/* enum: Sensor not working */
#define          MC_CMD_DYNAMIC_SENSORS_READING_BROKEN 0x4
/* enum: Sensor working but no reading available */
#define          MC_CMD_DYNAMIC_SENSORS_READING_NO_READING 0x5
/* enum: Sensor initialization failed */
#define          MC_CMD_DYNAMIC_SENSORS_READING_INIT_FAILED 0x6
#define       MC_CMD_DYNAMIC_SENSORS_READING_STATE_LBN 64
#define       MC_CMD_DYNAMIC_SENSORS_READING_STATE_WIDTH 32


/***********************************/
/* MC_CMD_DYNAMIC_SENSORS_LIST
 * Return a complete list of handles for sensors currently managed by the MC,
 * and a generation count for this version of the sensor table. On systems
 * advertising the DYNAMIC_SENSORS capability bit, this replaces the
 * MC_CMD_READ_SENSORS command. On multi-MC systems this may include sensors
 * added by the NMC. Sensor handles are persistent for the lifetime of the
 * sensor and are used to identify sensors in
 * MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS and
 * MC_CMD_DYNAMIC_SENSORS_GET_VALUES. The generation count is maintained by the
 * MC, is persistent across reboots and will be incremented each time the
 * sensor table is modified. When the table is modified, a
 * CODE_DYNAMIC_SENSORS_CHANGE event will be generated containing the new
 * generation count. The driver should compare this against the current
 * generation count, and if it is different, call MC_CMD_DYNAMIC_SENSORS_LIST
 * again to update it's copy of the sensor table. The sensor count is provided
 * to allow a future path to supporting more than
 * MC_CMD_DYNAMIC_SENSORS_GET_READINGS_IN_HANDLES_MAXNUM_MCDI2 sensors, i.e.
 * the maximum number that will fit in a single response. As this is a fairly
 * large number (253) it is not anticipated that this will be needed in the
 * near future, so can currently be ignored. On Riverhead this command is
 * implemented as a wrapper for `list` in the sensor_query SPHINX service.
 */
#define MC_CMD_DYNAMIC_SENSORS_LIST 0x66
#undef MC_CMD_0x66_PRIVILEGE_CTG

#define MC_CMD_0x66_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_DYNAMIC_SENSORS_LIST_IN msgrequest */
#define    MC_CMD_DYNAMIC_SENSORS_LIST_IN_LEN 0

/* MC_CMD_DYNAMIC_SENSORS_LIST_OUT msgresponse */
#define    MC_CMD_DYNAMIC_SENSORS_LIST_OUT_LENMIN 8
#define    MC_CMD_DYNAMIC_SENSORS_LIST_OUT_LENMAX 252
#define    MC_CMD_DYNAMIC_SENSORS_LIST_OUT_LENMAX_MCDI2 1020
#define    MC_CMD_DYNAMIC_SENSORS_LIST_OUT_LEN(num) (8+4*(num))
#define    MC_CMD_DYNAMIC_SENSORS_LIST_OUT_HANDLES_NUM(len) (((len)-8)/4)
/* Generation count, which will be updated each time a sensor is added to or
 * removed from the MC sensor table.
 */
#define       MC_CMD_DYNAMIC_SENSORS_LIST_OUT_GENERATION_OFST 0
#define       MC_CMD_DYNAMIC_SENSORS_LIST_OUT_GENERATION_LEN 4
/* Number of sensors managed by the MC. Note that in principle, this can be
 * larger than the size of the HANDLES array.
 */
#define       MC_CMD_DYNAMIC_SENSORS_LIST_OUT_COUNT_OFST 4
#define       MC_CMD_DYNAMIC_SENSORS_LIST_OUT_COUNT_LEN 4
/* Array of sensor handles */
#define       MC_CMD_DYNAMIC_SENSORS_LIST_OUT_HANDLES_OFST 8
#define       MC_CMD_DYNAMIC_SENSORS_LIST_OUT_HANDLES_LEN 4
#define       MC_CMD_DYNAMIC_SENSORS_LIST_OUT_HANDLES_MINNUM 0
#define       MC_CMD_DYNAMIC_SENSORS_LIST_OUT_HANDLES_MAXNUM 61
#define       MC_CMD_DYNAMIC_SENSORS_LIST_OUT_HANDLES_MAXNUM_MCDI2 253


/***********************************/
/* MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS
 * Get descriptions for a set of sensors, specified as an array of sensor
 * handles as returned by MC_CMD_DYNAMIC_SENSORS_LIST. Any handles which do not
 * correspond to a sensor currently managed by the MC will be dropped from from
 * the response. This may happen when a sensor table update is in progress, and
 * effectively means the set of usable sensors is the intersection between the
 * sets of sensors known to the driver and the MC. On Riverhead this command is
 * implemented as a wrapper for `get_descriptions` in the sensor_query SPHINX
 * service.
 */
#define MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS 0x67
#undef MC_CMD_0x67_PRIVILEGE_CTG

#define MC_CMD_0x67_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_IN msgrequest */
#define    MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_IN_LENMIN 0
#define    MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_IN_LENMAX 252
#define    MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_IN_LENMAX_MCDI2 1020
#define    MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_IN_LEN(num) (0+4*(num))
#define    MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_IN_HANDLES_NUM(len) (((len)-0)/4)
/* Array of sensor handles */
#define       MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_IN_HANDLES_OFST 0
#define       MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_IN_HANDLES_LEN 4
#define       MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_IN_HANDLES_MINNUM 0
#define       MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_IN_HANDLES_MAXNUM 63
#define       MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_IN_HANDLES_MAXNUM_MCDI2 255

/* MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_OUT msgresponse */
#define    MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_OUT_LENMIN 0
#define    MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_OUT_LENMAX 192
#define    MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_OUT_LENMAX_MCDI2 960
#define    MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_OUT_LEN(num) (0+64*(num))
#define    MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_OUT_SENSORS_NUM(len) (((len)-0)/64)
/* Array of MC_CMD_DYNAMIC_SENSORS_DESCRIPTION structures */
#define       MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_OUT_SENSORS_OFST 0
#define       MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_OUT_SENSORS_LEN 64
#define       MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_OUT_SENSORS_MINNUM 0
#define       MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_OUT_SENSORS_MAXNUM 3
#define       MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_OUT_SENSORS_MAXNUM_MCDI2 15


/***********************************/
/* MC_CMD_DYNAMIC_SENSORS_GET_READINGS
 * Read the state and value for a set of sensors, specified as an array of
 * sensor handles as returned by MC_CMD_DYNAMIC_SENSORS_LIST. In the case of a
 * broken sensor, then the state of the response's MC_CMD_DYNAMIC_SENSORS_VALUE
 * entry will be set to BROKEN, and any value provided should be treated as
 * erroneous. Any handles which do not correspond to a sensor currently managed
 * by the MC will be dropped from from the response. This may happen when a
 * sensor table update is in progress, and effectively means the set of usable
 * sensors is the intersection between the sets of sensors known to the driver
 * and the MC. On Riverhead this command is implemented as a wrapper for
 * `get_readings` in the sensor_query SPHINX service.
 */
#define MC_CMD_DYNAMIC_SENSORS_GET_READINGS 0x68
#undef MC_CMD_0x68_PRIVILEGE_CTG

#define MC_CMD_0x68_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_DYNAMIC_SENSORS_GET_READINGS_IN msgrequest */
#define    MC_CMD_DYNAMIC_SENSORS_GET_READINGS_IN_LENMIN 0
#define    MC_CMD_DYNAMIC_SENSORS_GET_READINGS_IN_LENMAX 252
#define    MC_CMD_DYNAMIC_SENSORS_GET_READINGS_IN_LENMAX_MCDI2 1020
#define    MC_CMD_DYNAMIC_SENSORS_GET_READINGS_IN_LEN(num) (0+4*(num))
#define    MC_CMD_DYNAMIC_SENSORS_GET_READINGS_IN_HANDLES_NUM(len) (((len)-0)/4)
/* Array of sensor handles */
#define       MC_CMD_DYNAMIC_SENSORS_GET_READINGS_IN_HANDLES_OFST 0
#define       MC_CMD_DYNAMIC_SENSORS_GET_READINGS_IN_HANDLES_LEN 4
#define       MC_CMD_DYNAMIC_SENSORS_GET_READINGS_IN_HANDLES_MINNUM 0
#define       MC_CMD_DYNAMIC_SENSORS_GET_READINGS_IN_HANDLES_MAXNUM 63
#define       MC_CMD_DYNAMIC_SENSORS_GET_READINGS_IN_HANDLES_MAXNUM_MCDI2 255

/* MC_CMD_DYNAMIC_SENSORS_GET_READINGS_OUT msgresponse */
#define    MC_CMD_DYNAMIC_SENSORS_GET_READINGS_OUT_LENMIN 0
#define    MC_CMD_DYNAMIC_SENSORS_GET_READINGS_OUT_LENMAX 252
#define    MC_CMD_DYNAMIC_SENSORS_GET_READINGS_OUT_LENMAX_MCDI2 1020
#define    MC_CMD_DYNAMIC_SENSORS_GET_READINGS_OUT_LEN(num) (0+12*(num))
#define    MC_CMD_DYNAMIC_SENSORS_GET_READINGS_OUT_VALUES_NUM(len) (((len)-0)/12)
/* Array of MC_CMD_DYNAMIC_SENSORS_READING structures */
#define       MC_CMD_DYNAMIC_SENSORS_GET_READINGS_OUT_VALUES_OFST 0
#define       MC_CMD_DYNAMIC_SENSORS_GET_READINGS_OUT_VALUES_LEN 12
#define       MC_CMD_DYNAMIC_SENSORS_GET_READINGS_OUT_VALUES_MINNUM 0
#define       MC_CMD_DYNAMIC_SENSORS_GET_READINGS_OUT_VALUES_MAXNUM 21
#define       MC_CMD_DYNAMIC_SENSORS_GET_READINGS_OUT_VALUES_MAXNUM_MCDI2 85

/* MC_CMD_MAC_FLAGS structuredef */
#define    MC_CMD_MAC_FLAGS_LEN 4
/* The enums defined in this field are used as indices into the
 * MC_CMD_MAC_FLAGS bitmask.
 */
#define       MC_CMD_MAC_FLAGS_MASK_OFST 0
#define       MC_CMD_MAC_FLAGS_MASK_LEN 4
/* enum property: bitshift */
/* enum: Include the FCS in the packet data delivered to the host. Ignored if
 * RX_INCLUDE_FCS not set in capabilities.
 */
#define          MC_CMD_MAC_FLAGS_FLAG_INCLUDE_FCS 0x0
#define       MC_CMD_MAC_FLAGS_MASK_LBN 0
#define       MC_CMD_MAC_FLAGS_MASK_WIDTH 32

/* MC_CMD_TRANSMISSION_MODE structuredef */
#define    MC_CMD_TRANSMISSION_MODE_LEN 4
#define       MC_CMD_TRANSMISSION_MODE_MASK_OFST 0
#define       MC_CMD_TRANSMISSION_MODE_MASK_LEN 4
/* enum property: value */
#define          MC_CMD_TRANSMISSION_MODE_PROMSC_MODE 0x0 /* enum */
#define          MC_CMD_TRANSMISSION_MODE_UNCST_MODE 0x1 /* enum */
#define          MC_CMD_TRANSMISSION_MODE_BRDCST_MODE 0x2 /* enum */
#define       MC_CMD_TRANSMISSION_MODE_MASK_LBN 0
#define       MC_CMD_TRANSMISSION_MODE_MASK_WIDTH 32

/* MC_CMD_MAC_CONFIG_OPTIONS structuredef */
#define    MC_CMD_MAC_CONFIG_OPTIONS_LEN 4
#define       MC_CMD_MAC_CONFIG_OPTIONS_MASK_OFST 0
#define       MC_CMD_MAC_CONFIG_OPTIONS_MASK_LEN 4
/* enum property: bitmask */
/* enum: Configure the MAC address. */
#define          MC_CMD_MAC_CONFIG_OPTIONS_CFG_ADDR 0x0
/* enum: Configure the maximum frame length. */
#define          MC_CMD_MAC_CONFIG_OPTIONS_CFG_MAX_FRAME_LEN 0x1
/* enum: Configure flow control. */
#define          MC_CMD_MAC_CONFIG_OPTIONS_CFG_FCNTL 0x2
/* enum: Configure the transmission mode. */
#define          MC_CMD_MAC_CONFIG_OPTIONS_CFG_TRANSMISSION_MODE 0x3
/* enum: Configure FCS. */
#define          MC_CMD_MAC_CONFIG_OPTIONS_CFG_INCLUDE_FCS 0x4
#define       MC_CMD_MAC_CONFIG_OPTIONS_MASK_LBN 0
#define       MC_CMD_MAC_CONFIG_OPTIONS_MASK_WIDTH 32


/***********************************/
/* MC_CMD_MAC_CTRL
 * Set MAC configuration. Return code: 0, EINVAL, ENOTSUP
 */
#define MC_CMD_MAC_CTRL 0x1df
#undef MC_CMD_0x1df_PRIVILEGE_CTG

#define MC_CMD_0x1df_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_MAC_CTRL_IN msgrequest */
#define    MC_CMD_MAC_CTRL_IN_LEN 32
/* Handle for selected network port. */
#define       MC_CMD_MAC_CTRL_IN_PORT_HANDLE_OFST 0
#define       MC_CMD_MAC_CTRL_IN_PORT_HANDLE_LEN 4
/* Select which parameters to configure. A parameter will only be modified if
 * the corresponding control flag is set.
 */
#define       MC_CMD_MAC_CTRL_IN_CONTROL_FLAGS_OFST 4
#define       MC_CMD_MAC_CTRL_IN_CONTROL_FLAGS_LEN 4
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_MAC_CONFIG_OPTIONS/MASK */
/* MAC address of the device. */
#define       MC_CMD_MAC_CTRL_IN_ADDR_OFST 8
#define       MC_CMD_MAC_CTRL_IN_ADDR_LEN 8
#define       MC_CMD_MAC_CTRL_IN_ADDR_LO_OFST 8
#define       MC_CMD_MAC_CTRL_IN_ADDR_LO_LEN 4
#define       MC_CMD_MAC_CTRL_IN_ADDR_LO_LBN 64
#define       MC_CMD_MAC_CTRL_IN_ADDR_LO_WIDTH 32
#define       MC_CMD_MAC_CTRL_IN_ADDR_HI_OFST 12
#define       MC_CMD_MAC_CTRL_IN_ADDR_HI_LEN 4
#define       MC_CMD_MAC_CTRL_IN_ADDR_HI_LBN 96
#define       MC_CMD_MAC_CTRL_IN_ADDR_HI_WIDTH 32
/* Includes the ethernet header, optional VLAN tags, payload and FCS. */
#define       MC_CMD_MAC_CTRL_IN_MAX_FRAME_LEN_OFST 16
#define       MC_CMD_MAC_CTRL_IN_MAX_FRAME_LEN_LEN 4
/* Settings for flow control. */
#define       MC_CMD_MAC_CTRL_IN_FCNTL_OFST 20
#define       MC_CMD_MAC_CTRL_IN_FCNTL_LEN 4
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_FCNTL/MASK */
/* Configure the MAC to transmit in one of promiscuous, unicast or broadcast
 * mode.
 */
#define       MC_CMD_MAC_CTRL_IN_TRANSMISSION_MODE_OFST 24
#define       MC_CMD_MAC_CTRL_IN_TRANSMISSION_MODE_LEN 4
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_TRANSMISSION_MODE/MASK */
/* Flags to control and expand the configuration of the MAC. */
#define       MC_CMD_MAC_CTRL_IN_FLAGS_OFST 28
#define       MC_CMD_MAC_CTRL_IN_FLAGS_LEN 4
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_MAC_FLAGS/MASK */

/* MC_CMD_MAC_CTRL_IN_V2 msgrequest: Updated MAC_CTRL with QBB mask */
#define    MC_CMD_MAC_CTRL_IN_V2_LEN 33
/* Handle for selected network port. */
#define       MC_CMD_MAC_CTRL_IN_V2_PORT_HANDLE_OFST 0
#define       MC_CMD_MAC_CTRL_IN_V2_PORT_HANDLE_LEN 4
/* Select which parameters to configure. A parameter will only be modified if
 * the corresponding control flag is set.
 */
#define       MC_CMD_MAC_CTRL_IN_V2_CONTROL_FLAGS_OFST 4
#define       MC_CMD_MAC_CTRL_IN_V2_CONTROL_FLAGS_LEN 4
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_MAC_CONFIG_OPTIONS/MASK */
/* MAC address of the device. */
#define       MC_CMD_MAC_CTRL_IN_V2_ADDR_OFST 8
#define       MC_CMD_MAC_CTRL_IN_V2_ADDR_LEN 8
#define       MC_CMD_MAC_CTRL_IN_V2_ADDR_LO_OFST 8
#define       MC_CMD_MAC_CTRL_IN_V2_ADDR_LO_LEN 4
#define       MC_CMD_MAC_CTRL_IN_V2_ADDR_LO_LBN 64
#define       MC_CMD_MAC_CTRL_IN_V2_ADDR_LO_WIDTH 32
#define       MC_CMD_MAC_CTRL_IN_V2_ADDR_HI_OFST 12
#define       MC_CMD_MAC_CTRL_IN_V2_ADDR_HI_LEN 4
#define       MC_CMD_MAC_CTRL_IN_V2_ADDR_HI_LBN 96
#define       MC_CMD_MAC_CTRL_IN_V2_ADDR_HI_WIDTH 32
/* Includes the ethernet header, optional VLAN tags, payload and FCS. */
#define       MC_CMD_MAC_CTRL_IN_V2_MAX_FRAME_LEN_OFST 16
#define       MC_CMD_MAC_CTRL_IN_V2_MAX_FRAME_LEN_LEN 4
/* Settings for flow control. */
#define       MC_CMD_MAC_CTRL_IN_V2_FCNTL_OFST 20
#define       MC_CMD_MAC_CTRL_IN_V2_FCNTL_LEN 4
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_FCNTL/MASK */
/* Configure the MAC to transmit in one of promiscuous, unicast or broadcast
 * mode.
 */
#define       MC_CMD_MAC_CTRL_IN_V2_TRANSMISSION_MODE_OFST 24
#define       MC_CMD_MAC_CTRL_IN_V2_TRANSMISSION_MODE_LEN 4
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_TRANSMISSION_MODE/MASK */
/* Flags to control and expand the configuration of the MAC. */
#define       MC_CMD_MAC_CTRL_IN_V2_FLAGS_OFST 28
#define       MC_CMD_MAC_CTRL_IN_V2_FLAGS_LEN 4
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_MAC_FLAGS/MASK */
/* Priority-based flow control mask (QBB). PRIO7 corresponds to the highest
 * priority, and PRIO0 to the lowest. This field is only used when CFG_FCNTL is
 * set and FCNTL is QBB
 */
#define       MC_CMD_MAC_CTRL_IN_V2_PRIO_FCNTL_MASK_OFST 32
#define       MC_CMD_MAC_CTRL_IN_V2_PRIO_FCNTL_MASK_LEN 1
/* enum property: bitmask */
#define          MC_CMD_MAC_CTRL_IN_V2_QBB_PRIO0 0x0 /* enum */
#define          MC_CMD_MAC_CTRL_IN_V2_QBB_PRIO1 0x1 /* enum */
#define          MC_CMD_MAC_CTRL_IN_V2_QBB_PRIO2 0x2 /* enum */
#define          MC_CMD_MAC_CTRL_IN_V2_QBB_PRIO3 0x3 /* enum */
#define          MC_CMD_MAC_CTRL_IN_V2_QBB_PRIO4 0x4 /* enum */
#define          MC_CMD_MAC_CTRL_IN_V2_QBB_PRIO5 0x5 /* enum */
#define          MC_CMD_MAC_CTRL_IN_V2_QBB_PRIO6 0x6 /* enum */
#define          MC_CMD_MAC_CTRL_IN_V2_QBB_PRIO7 0x7 /* enum */

/* MC_CMD_MAC_CTRL_OUT msgresponse */
#define    MC_CMD_MAC_CTRL_OUT_LEN 0


/***********************************/
/* MC_CMD_MAC_STATE
 * Read the MAC state. Return code: 0, ETIME.
 */
#define MC_CMD_MAC_STATE 0x1e0
#undef MC_CMD_0x1e0_PRIVILEGE_CTG

#define MC_CMD_0x1e0_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_MAC_STATE_IN msgrequest */
#define    MC_CMD_MAC_STATE_IN_LEN 4
/* Handle for selected network port. */
#define       MC_CMD_MAC_STATE_IN_PORT_HANDLE_OFST 0
#define       MC_CMD_MAC_STATE_IN_PORT_HANDLE_LEN 4

/* MC_CMD_MAC_STATE_OUT msgresponse */
#define    MC_CMD_MAC_STATE_OUT_LEN 32
/* The configured maximum frame length of the MAC. */
#define       MC_CMD_MAC_STATE_OUT_MAX_FRAME_LEN_OFST 0
#define       MC_CMD_MAC_STATE_OUT_MAX_FRAME_LEN_LEN 4
/* This returns the negotiated flow control value. */
#define       MC_CMD_MAC_STATE_OUT_FCNTL_OFST 4
#define       MC_CMD_MAC_STATE_OUT_FCNTL_LEN 4
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_FCNTL/MASK */
/* MAC address of the device. */
#define       MC_CMD_MAC_STATE_OUT_ADDR_OFST 8
#define       MC_CMD_MAC_STATE_OUT_ADDR_LEN 8
#define       MC_CMD_MAC_STATE_OUT_ADDR_LO_OFST 8
#define       MC_CMD_MAC_STATE_OUT_ADDR_LO_LEN 4
#define       MC_CMD_MAC_STATE_OUT_ADDR_LO_LBN 64
#define       MC_CMD_MAC_STATE_OUT_ADDR_LO_WIDTH 32
#define       MC_CMD_MAC_STATE_OUT_ADDR_HI_OFST 12
#define       MC_CMD_MAC_STATE_OUT_ADDR_HI_LEN 4
#define       MC_CMD_MAC_STATE_OUT_ADDR_HI_LBN 96
#define       MC_CMD_MAC_STATE_OUT_ADDR_HI_WIDTH 32
/* Flags indicating MAC faults. */
#define       MC_CMD_MAC_STATE_OUT_MAC_FAULT_FLAGS_OFST 16
#define       MC_CMD_MAC_STATE_OUT_MAC_FAULT_FLAGS_LEN 4
/* enum property: bitshift */
/* enum: Indicates a local MAC fault. */
#define          MC_CMD_MAC_STATE_OUT_LOCAL 0x0
/* enum: Indicates a remote MAC fault. */
#define          MC_CMD_MAC_STATE_OUT_REMOTE 0x1
/* enum: Indicates a pending reconfiguration of the MAC. */
#define          MC_CMD_MAC_STATE_OUT_PENDING_RECONFIG 0x2
/* The flags that were used to configure the MAC. This is a copy of the FLAGS
 * field in the MC_CMD_MAC_CTRL_IN command.
 */
#define       MC_CMD_MAC_STATE_OUT_FLAGS_OFST 20
#define       MC_CMD_MAC_STATE_OUT_FLAGS_LEN 4
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_MAC_FLAGS/MASK */
/* The transmission mode that was used to configure the MAC. This is a copy of
 * the TRANSMISSION_MODE field in the MC_CMD_MAC_CTRL_IN command.
 */
#define       MC_CMD_MAC_STATE_OUT_TRANSMISSION_MODE_OFST 24
#define       MC_CMD_MAC_STATE_OUT_TRANSMISSION_MODE_LEN 4
/* enum property: value */
/*            Enum values, see field(s): */
/*               MC_CMD_TRANSMISSION_MODE/MASK */
/* The control flags that were used to configure the MAC. This is a copy of the
 * CONTROL field in the MC_CMD_MAC_CTRL_IN command.
 */
#define       MC_CMD_MAC_STATE_OUT_CONTROL_FLAGS_OFST 28
#define       MC_CMD_MAC_STATE_OUT_CONTROL_FLAGS_LEN 4
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_MAC_CONFIG_OPTIONS/MASK */


/***********************************/
/* MC_CMD_GET_ASSIGNED_PORT_HANDLE
 * Obtain a handle that can be operated on to configure and query the status of
 * the link. ENOENT is returned when no port is assigned to the client. Return
 * code: 0, ENOENT
 */
#define MC_CMD_GET_ASSIGNED_PORT_HANDLE 0x1e2
#undef MC_CMD_0x1e2_PRIVILEGE_CTG

#define MC_CMD_0x1e2_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_GET_ASSIGNED_PORT_HANDLE_IN msgrequest */
#define    MC_CMD_GET_ASSIGNED_PORT_HANDLE_IN_LEN 0

/* MC_CMD_GET_ASSIGNED_PORT_HANDLE_OUT msgresponse */
#define    MC_CMD_GET_ASSIGNED_PORT_HANDLE_OUT_LEN 4
/* Handle for assigned port. */
#define       MC_CMD_GET_ASSIGNED_PORT_HANDLE_OUT_PORT_HANDLE_OFST 0
#define       MC_CMD_GET_ASSIGNED_PORT_HANDLE_OUT_PORT_HANDLE_LEN 4

/* MC_CMD_STAT_ID structuredef */
#define    MC_CMD_STAT_ID_LEN 4
#define       MC_CMD_STAT_ID_SOURCE_ID_OFST 0
#define       MC_CMD_STAT_ID_SOURCE_ID_LEN 2
/* enum property: index */
/* enum: Internal markers (generation start and end markers) */
#define          MC_CMD_STAT_ID_MARKER 0x1
/* enum: Network port MAC statistics. */
#define          MC_CMD_STAT_ID_MAC 0x2
/* enum: Network port PHY statistics. */
#define          MC_CMD_STAT_ID_PHY 0x3
#define       MC_CMD_STAT_ID_SOURCE_ID_LBN 0
#define       MC_CMD_STAT_ID_SOURCE_ID_WIDTH 16
#define       MC_CMD_STAT_ID_MARKER_STAT_ID_OFST 2
#define       MC_CMD_STAT_ID_MARKER_STAT_ID_LEN 2
/* enum property: index */
/* enum: This value is used to mark the start of a generation of statistics for
 * DMA synchronization. It is incremented whenever a new set of statistics is
 * transferred. Always the first entry in the DMA buffer.
 */
#define          MC_CMD_STAT_ID_GENERATION_START 0x1
/* enum: This value is used to mark the end of a generation of statistics for
 * DMA synchronizaion. Always the last entry in the DMA buffer and set to the
 * same value as GENERATION_START. The host driver must compare the
 * GENERATION_START and GENERATION_END values to verify that the DMA buffer is
 * consistent upon copying the the DMA buffer. If they do not match, it means
 * that new DMA transfer has started while the host driver was copying the DMA
 * buffer. In this case, the host driver must repeat the copy operation.
 */
#define          MC_CMD_STAT_ID_GENERATION_END 0x2
#define       MC_CMD_STAT_ID_MARKER_STAT_ID_LBN 16
#define       MC_CMD_STAT_ID_MARKER_STAT_ID_WIDTH 16
#define       MC_CMD_STAT_ID_MAC_STAT_ID_OFST 2
#define       MC_CMD_STAT_ID_MAC_STAT_ID_LEN 2
/* enum property: index */
/* enum: Total number of packets transmitted (includes pause frames). */
#define          MC_CMD_STAT_ID_TX_PKTS 0x1
/* enum: Pause frames transmitted. */
#define          MC_CMD_STAT_ID_TX_PAUSE_PKTS 0x2
/* enum: Control frames transmitted. */
#define          MC_CMD_STAT_ID_TX_CONTROL_PKTS 0x3
/* enum: Unicast packets transmitted (includes pause frames). */
#define          MC_CMD_STAT_ID_TX_UNICAST_PKTS 0x4
/* enum: Multicast packets transmitted (includes pause frames). */
#define          MC_CMD_STAT_ID_TX_MULTICAST_PKTS 0x5
/* enum: Broadcast packets transmitted (includes pause frames). */
#define          MC_CMD_STAT_ID_TX_BROADCAST_PKTS 0x6
/* enum: Bytes transmitted (includes pause frames). */
#define          MC_CMD_STAT_ID_TX_BYTES 0x7
/* enum: Bytes transmitted with bad CRC. */
#define          MC_CMD_STAT_ID_TX_BAD_BYTES 0x8
/* enum: Bytes transmitted with good CRC. */
#define          MC_CMD_STAT_ID_TX_GOOD_BYTES 0x9
/* enum: Packets transmitted with length less than 64 bytes. */
#define          MC_CMD_STAT_ID_TX_LT64_PKTS 0xa
/* enum: Packets transmitted with length equal to 64 bytes. */
#define          MC_CMD_STAT_ID_TX_64_PKTS 0xb
/* enum: Packets transmitted with length between 65 and 127 bytes. */
#define          MC_CMD_STAT_ID_TX_65_TO_127_PKTS 0xc
/* enum: Packets transmitted with length between 128 and 255 bytes. */
#define          MC_CMD_STAT_ID_TX_128_TO_255_PKTS 0xd
/* enum: Packets transmitted with length between 256 and 511 bytes. */
#define          MC_CMD_STAT_ID_TX_256_TO_511_PKTS 0xe
/* enum: Packets transmitted with length between 512 and 1023 bytes. */
#define          MC_CMD_STAT_ID_TX_512_TO_1023_PKTS 0xf
/* enum: Packets transmitted with length between 1024 and 1518 bytes. */
#define          MC_CMD_STAT_ID_TX_1024_TO_15XX_PKTS 0x10
/* enum: Packets transmitted with length between 1519 and 9216 bytes. */
#define          MC_CMD_STAT_ID_TX_15XX_TO_JUMBO_PKTS 0x11
/* enum: Packets transmitted with length greater than 9216 bytes. */
#define          MC_CMD_STAT_ID_TX_GTJUMBO_PKTS 0x12
/* enum: Packets transmitted with bad FCS. */
#define          MC_CMD_STAT_ID_TX_BAD_FCS_PKTS 0x13
/* enum: Packets transmitted with good FCS. */
#define          MC_CMD_STAT_ID_TX_GOOD_FCS_PKTS 0x14
/* enum: Packets received. */
#define          MC_CMD_STAT_ID_RX_PKTS 0x15
/* enum: Pause frames received. */
#define          MC_CMD_STAT_ID_RX_PAUSE_PKTS 0x16
/* enum: Total number of good packets received. */
#define          MC_CMD_STAT_ID_RX_GOOD_PKTS 0x17
/* enum: Total number of BAD packets received. */
#define          MC_CMD_STAT_ID_RX_BAD_PKTS 0x18
/* enum: Total number of control frames received. */
#define          MC_CMD_STAT_ID_RX_CONTROL_PKTS 0x19
/* enum: Total number of unicast packets received. */
#define          MC_CMD_STAT_ID_RX_UNICAST_PKTS 0x1a
/* enum: Total number of multicast packets received. */
#define          MC_CMD_STAT_ID_RX_MULTICAST_PKTS 0x1b
/* enum: Total number of broadcast packets received. */
#define          MC_CMD_STAT_ID_RX_BROADCAST_PKTS 0x1c
/* enum: Total number of bytes received. */
#define          MC_CMD_STAT_ID_RX_BYTES 0x1d
/* enum: Total number of bytes received with bad CRC. */
#define          MC_CMD_STAT_ID_RX_BAD_BYTES 0x1e
/* enum: Total number of bytes received with GOOD CRC. */
#define          MC_CMD_STAT_ID_RX_GOOD_BYTES 0x1f
/* enum: Packets received with length equal to 64 bytes. */
#define          MC_CMD_STAT_ID_RX_64_PKTS 0x20
/* enum: Packets received with length between 65 and 127 bytes. */
#define          MC_CMD_STAT_ID_RX_65_TO_127_PKTS 0x21
/* enum: Packets received with length between 128 and 255 bytes. */
#define          MC_CMD_STAT_ID_RX_128_TO_255_PKTS 0x22
/* enum: Packets received with length between 256 and 511 bytes. */
#define          MC_CMD_STAT_ID_RX_256_TO_511_PKTS 0x23
/* enum: Packets received with length between 512 and 1023 bytes. */
#define          MC_CMD_STAT_ID_RX_512_TO_1023_PKTS 0x24
/* enum: Packets received with length between 1024 and 1518 bytes. */
#define          MC_CMD_STAT_ID_RX_1024_TO_15XX_PKTS 0x25
/* enum: Packets received with length between 1519 and 9216 bytes. */
#define          MC_CMD_STAT_ID_RX_15XX_TO_JUMBO_PKTS 0x26
/* enum: Packets received with length greater than 9216 bytes. */
#define          MC_CMD_STAT_ID_RX_GTJUMBO_PKTS 0x27
/* enum: Packets received with length less than 64 bytes. */
#define          MC_CMD_STAT_ID_RX_UNDERSIZE_PKTS 0x28
/* enum: Packets received with bad FCS. */
#define          MC_CMD_STAT_ID_RX_BAD_FCS_PKTS 0x29
/* enum: Packets received with GOOD FCS. */
#define          MC_CMD_STAT_ID_RX_GOOD_FCS_PKTS 0x2a
/* enum: Packets received with overflow. */
#define          MC_CMD_STAT_ID_RX_OVERFLOW_PKTS 0x2b
/* enum: Packets received with symbol error. */
#define          MC_CMD_STAT_ID_RX_SYMBOL_ERROR_PKTS 0x2c
/* enum: Packets received with alignment error. */
#define          MC_CMD_STAT_ID_RX_ALIGN_ERROR_PKTS 0x2d
/* enum: Packets received with length error. */
#define          MC_CMD_STAT_ID_RX_LENGTH_ERROR_PKTS 0x2e
/* enum: Packets received with internal error. */
#define          MC_CMD_STAT_ID_RX_INTERNAL_ERROR_PKTS 0x2f
/* enum: Packets received with jabber. These packets are larger than the
 * allowed maximum receive unit (MRU). This indicates that a packet either has
 * a bad CRC or has an RX error.
 */
#define          MC_CMD_STAT_ID_RX_JABBER_PKTS 0x30
/* enum: Packets dropped due to having no descriptor. This is a datapath stat
 */
#define          MC_CMD_STAT_ID_RX_NODESC_DROPS 0x31
/* enum: Packets received with lanes 0 and 1 character error. */
#define          MC_CMD_STAT_ID_RX_LANES01_CHAR_ERR 0x32
/* enum: Packets received with lanes 2 and 3 character error. */
#define          MC_CMD_STAT_ID_RX_LANES23_CHAR_ERR 0x33
/* enum: Packets received with lanes 0 and 1 disparity error. */
#define          MC_CMD_STAT_ID_RX_LANES01_DISP_ERR 0x34
/* enum: Packets received with lanes 2 and 3 disparity error. */
#define          MC_CMD_STAT_ID_RX_LANES23_DISP_ERR 0x35
/* enum: Packets received with match fault. */
#define          MC_CMD_STAT_ID_RX_MATCH_FAULT 0x36
#define       MC_CMD_STAT_ID_MAC_STAT_ID_LBN 16
#define       MC_CMD_STAT_ID_MAC_STAT_ID_WIDTH 16
/* Include FEC stats. */
#define       MC_CMD_STAT_ID_PHY_STAT_ID_OFST 2
#define       MC_CMD_STAT_ID_PHY_STAT_ID_LEN 2
/* enum property: index */
/* enum: Number of uncorrected FEC codewords on link (RS-FEC only from Medford2
 * onwards)
 */
#define          MC_CMD_STAT_ID_FEC_UNCORRECTED_ERRORS 0x1
/* enum: Number of corrected FEC codewords on link (RS-FEC only from Medford2
 * onwards)
 */
#define          MC_CMD_STAT_ID_FEC_CORRECTED_ERRORS 0x2
/* enum: Number of corrected 10-bit symbol errors, lane 0 (RS-FEC only) */
#define          MC_CMD_STAT_ID_FEC_CORRECTED_SYMBOLS_LANE0 0x3
/* enum: Number of corrected 10-bit symbol errors, lane 1 (RS-FEC only) */
#define          MC_CMD_STAT_ID_FEC_CORRECTED_SYMBOLS_LANE1 0x4
/* enum: Number of corrected 10-bit symbol errors, lane 2 (RS-FEC only) */
#define          MC_CMD_STAT_ID_FEC_CORRECTED_SYMBOLS_LANE2 0x5
/* enum: Number of corrected 10-bit symbol errors, lane 3 (RS-FEC only) */
#define          MC_CMD_STAT_ID_FEC_CORRECTED_SYMBOLS_LANE3 0x6
#define       MC_CMD_STAT_ID_PHY_STAT_ID_LBN 16
#define       MC_CMD_STAT_ID_PHY_STAT_ID_WIDTH 16

/* MC_CMD_STAT_DESC structuredef: Structure describing the layout and size of
 * the stats DMA buffer descriptor.
 */
#define    MC_CMD_STAT_DESC_LEN 8
/* Unique identifier of the statistic. Formatted as MC_CMD_STAT_ID */
#define       MC_CMD_STAT_DESC_STAT_ID_OFST 0
#define       MC_CMD_STAT_DESC_STAT_ID_LEN 4
#define       MC_CMD_STAT_DESC_STAT_ID_LBN 0
#define       MC_CMD_STAT_DESC_STAT_ID_WIDTH 32
/* See structuredef: MC_CMD_STAT_ID */
#define       MC_CMD_STAT_DESC_STAT_ID_SOURCE_ID_OFST 0
#define       MC_CMD_STAT_DESC_STAT_ID_SOURCE_ID_LEN 2
#define       MC_CMD_STAT_DESC_STAT_ID_SOURCE_ID_LBN 0
#define       MC_CMD_STAT_DESC_STAT_ID_SOURCE_ID_WIDTH 16
#define       MC_CMD_STAT_DESC_STAT_ID_MARKER_STAT_ID_OFST 2
#define       MC_CMD_STAT_DESC_STAT_ID_MARKER_STAT_ID_LEN 2
#define       MC_CMD_STAT_DESC_STAT_ID_MARKER_STAT_ID_LBN 16
#define       MC_CMD_STAT_DESC_STAT_ID_MARKER_STAT_ID_WIDTH 16
#define       MC_CMD_STAT_DESC_STAT_ID_MAC_STAT_ID_OFST 2
#define       MC_CMD_STAT_DESC_STAT_ID_MAC_STAT_ID_LEN 2
#define       MC_CMD_STAT_DESC_STAT_ID_MAC_STAT_ID_LBN 16
#define       MC_CMD_STAT_DESC_STAT_ID_MAC_STAT_ID_WIDTH 16
#define       MC_CMD_STAT_DESC_STAT_ID_PHY_STAT_ID_OFST 2
#define       MC_CMD_STAT_DESC_STAT_ID_PHY_STAT_ID_LEN 2
#define       MC_CMD_STAT_DESC_STAT_ID_PHY_STAT_ID_LBN 16
#define       MC_CMD_STAT_DESC_STAT_ID_PHY_STAT_ID_WIDTH 16
/* Index of the statistic in the DMA buffer. */
#define       MC_CMD_STAT_DESC_STAT_INDEX_OFST 4
#define       MC_CMD_STAT_DESC_STAT_INDEX_LEN 2
#define       MC_CMD_STAT_DESC_STAT_INDEX_LBN 32
#define       MC_CMD_STAT_DESC_STAT_INDEX_WIDTH 16
/* Reserved for future extension (e.g. flags field) - currently always 0. */
#define       MC_CMD_STAT_DESC_RESERVED_OFST 6
#define       MC_CMD_STAT_DESC_RESERVED_LEN 2
#define       MC_CMD_STAT_DESC_RESERVED_LBN 48
#define       MC_CMD_STAT_DESC_RESERVED_WIDTH 16


/***********************************/
/* MC_CMD_MAC_STATISTICS_DESCRIPTOR
 * Get a list of descriptors that describe the layout and size of the stats
 * buffer required for retrieving statistics for a given port. Each entry in
 * the list is formatted as MC_CMD_STAT_DESC and provides the ID of each stat
 * and its location and size in the buffer. It also gives the overall minimum
 * size of the DMA buffer required when DMA mode is used. Note that the first
 * and last entries in the list are reserved for the generation start
 * (MC_CMD_MARKER_STAT_GENERATION_START) and end
 * (MC_CMD_MARKER_STAT_GENERATION_END) markers respectively, to be used for DMA
 * synchronisation as described in the documentation for the relevant enum
 * entries. The entries are present in the buffer even if DMA mode is not used.
 * Provisions are made (but currently unused) for extending the size of the
 * descriptors, extending the size of the list beyond the maximum MCDI response
 * size, as well as the dynamic runtime updates of the list. Returns: 0 on
 * success, ENOENT on non-existent port handle
 */
#define MC_CMD_MAC_STATISTICS_DESCRIPTOR 0x1e3
#undef MC_CMD_0x1e3_PRIVILEGE_CTG

#define MC_CMD_0x1e3_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_MAC_STATISTICS_DESCRIPTOR_IN msgrequest */
#define    MC_CMD_MAC_STATISTICS_DESCRIPTOR_IN_LEN 8
/* Handle of port to get MAC statitstics descriptors for. */
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_IN_PORT_HANDLE_OFST 0
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_IN_PORT_HANDLE_LEN 4
/* Offset of the first entry to return, for cases where not all entries fit in
 * the MCDI response. Should be set to 0 on initial request, and on subsequent
 * requests updated by the number of entries already returned, as long as the
 * MORE_ENTRIES flag is set.
 */
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_IN_OFFSET_OFST 4
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_IN_OFFSET_LEN 4

/* MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT msgresponse */
#define    MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_LENMIN 28
#define    MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_LENMAX 252
#define    MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_LENMAX_MCDI2 1020
#define    MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_LEN(num) (20+8*(num))
#define    MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRIES_NUM(len) (((len)-20)/8)
/* Generation number of the stats buffer. This is incremented each time the
 * buffer is updated, and is used to verify the consistency of the buffer
 * contents. Reserved for future extension (dynamic list updates). Currently
 * always set to 0.
 */
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_GENERATION_OFST 0
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_GENERATION_LEN 4
/* Minimum size of the DMA buffer required to retrieve all statistics for the
 * port. This is the sum of the sizes of all the statistics, plus the size of
 * the generation markers. Minimum buffer size in bytes required to fit all
 * statistics. Drivers will typically round up this value to the granularity of
 * the host DMA allocation units.
 */
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_DMA_BUFFER_SIZE_OFST 4
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_DMA_BUFFER_SIZE_LEN 4
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_FLAGS_OFST 8
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_FLAGS_LEN 4
#define        MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_MORE_ENTRIES_OFST 8
#define        MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_MORE_ENTRIES_LBN 0
#define        MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_MORE_ENTRIES_WIDTH 1
/* Size of the individual descriptor entry in the list. Determines the entry
 * stride in the list. Currently always set to size of MC_CMD_STAT_DESC, larger
 * values can be used in the future for extending the descriptor, by appending
 * new data to the end of the existing structure.
 */
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRY_SIZE_OFST 12
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRY_SIZE_LEN 4
/* Number of entries returned in the descriptor list. */
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRY_COUNT_OFST 16
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRY_COUNT_LEN 4
/* List of descriptors. Each entry is formatted as MC_CMD_STAT_DESC and
 * provides the ID of each stat and its location and size in the buffer. The
 * first and last entries are reserved for the generation start and end markers
 * respectively.
 */
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRIES_OFST 20
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRIES_LEN 8
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRIES_LO_OFST 20
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRIES_LO_LEN 4
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRIES_LO_LBN 160
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRIES_LO_WIDTH 32
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRIES_HI_OFST 24
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRIES_HI_LEN 4
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRIES_HI_LBN 192
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRIES_HI_WIDTH 32
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRIES_MINNUM 1
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRIES_MAXNUM 29
#define       MC_CMD_MAC_STATISTICS_DESCRIPTOR_OUT_ENTRIES_MAXNUM_MCDI2 125


/***********************************/
/* MC_CMD_MAC_STATISTICS
 * Get generic MAC statistics. This call retrieves unified statistics managed
 * by the MC. The MC will populate and provide all supported statistics in the
 * format as returned by MC_CMD_MAC_STATISTICS_DESCRIPTOR. Refer to the
 * aforementioned command for the format and contents of the stats DMA buffer.
 * To ensure consistent and accurate results, it is essential for the driver to
 * initialize the DMA buffer with zeros when DMA mode is used. Returns: 0 on
 * success, ETIME if the DMA buffer is not ready, ENOENT on non-existent port
 * handle, and EINVAL on invalid parameters (DMA buffer too small)
 */
#define MC_CMD_MAC_STATISTICS 0x1e4
#undef MC_CMD_0x1e4_PRIVILEGE_CTG

#define MC_CMD_0x1e4_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_MAC_STATISTICS_IN msgrequest */
#define    MC_CMD_MAC_STATISTICS_IN_LEN 20
/* Handle of port to get MAC statistics for. */
#define       MC_CMD_MAC_STATISTICS_IN_PORT_HANDLE_OFST 0
#define       MC_CMD_MAC_STATISTICS_IN_PORT_HANDLE_LEN 4
/* Contains options for querying the MAC statistics. */
#define       MC_CMD_MAC_STATISTICS_IN_CMD_OFST 4
#define       MC_CMD_MAC_STATISTICS_IN_CMD_LEN 4
#define        MC_CMD_MAC_STATISTICS_IN_DMA_OFST 4
#define        MC_CMD_MAC_STATISTICS_IN_DMA_LBN 0
#define        MC_CMD_MAC_STATISTICS_IN_DMA_WIDTH 1
#define        MC_CMD_MAC_STATISTICS_IN_CLEAR_OFST 4
#define        MC_CMD_MAC_STATISTICS_IN_CLEAR_LBN 1
#define        MC_CMD_MAC_STATISTICS_IN_CLEAR_WIDTH 1
#define        MC_CMD_MAC_STATISTICS_IN_PERIODIC_CHANGE_OFST 4
#define        MC_CMD_MAC_STATISTICS_IN_PERIODIC_CHANGE_LBN 2
#define        MC_CMD_MAC_STATISTICS_IN_PERIODIC_CHANGE_WIDTH 1
#define        MC_CMD_MAC_STATISTICS_IN_PERIODIC_ENABLE_OFST 4
#define        MC_CMD_MAC_STATISTICS_IN_PERIODIC_ENABLE_LBN 3
#define        MC_CMD_MAC_STATISTICS_IN_PERIODIC_ENABLE_WIDTH 1
#define        MC_CMD_MAC_STATISTICS_IN_PERIODIC_NOEVENT_OFST 4
#define        MC_CMD_MAC_STATISTICS_IN_PERIODIC_NOEVENT_LBN 4
#define        MC_CMD_MAC_STATISTICS_IN_PERIODIC_NOEVENT_WIDTH 1
#define        MC_CMD_MAC_STATISTICS_IN_PERIOD_MS_OFST 4
#define        MC_CMD_MAC_STATISTICS_IN_PERIOD_MS_LBN 16
#define        MC_CMD_MAC_STATISTICS_IN_PERIOD_MS_WIDTH 16
/* This is the address of the DMA buffer to use for transfer of the statistics.
 * Only valid if the DMA flag is set to 1.
 */
#define       MC_CMD_MAC_STATISTICS_IN_DMA_ADDR_OFST 8
#define       MC_CMD_MAC_STATISTICS_IN_DMA_ADDR_LEN 8
#define       MC_CMD_MAC_STATISTICS_IN_DMA_ADDR_LO_OFST 8
#define       MC_CMD_MAC_STATISTICS_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_MAC_STATISTICS_IN_DMA_ADDR_LO_LBN 64
#define       MC_CMD_MAC_STATISTICS_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_MAC_STATISTICS_IN_DMA_ADDR_HI_OFST 12
#define       MC_CMD_MAC_STATISTICS_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_MAC_STATISTICS_IN_DMA_ADDR_HI_LBN 96
#define       MC_CMD_MAC_STATISTICS_IN_DMA_ADDR_HI_WIDTH 32
/* This is the length of the DMA buffer to use for the transfer of the
 * statistics. The buffer should be at least DMA_BUFFER_SIZE long, as returned
 * by MC_CMD_MAC_STATISTICS_DESCRIPTOR. If the supplied buffer is too small,
 * the command will fail with EINVAL. Only valid if the DMA flag is set to 1.
 */
#define       MC_CMD_MAC_STATISTICS_IN_DMA_LEN_OFST 16
#define       MC_CMD_MAC_STATISTICS_IN_DMA_LEN_LEN 4

/* MC_CMD_MAC_STATISTICS_OUT msgresponse */
#define    MC_CMD_MAC_STATISTICS_OUT_LENMIN 5
#define    MC_CMD_MAC_STATISTICS_OUT_LENMAX 252
#define    MC_CMD_MAC_STATISTICS_OUT_LENMAX_MCDI2 1020
#define    MC_CMD_MAC_STATISTICS_OUT_LEN(num) (4+1*(num))
#define    MC_CMD_MAC_STATISTICS_OUT_DATA_NUM(len) (((len)-4)/1)
/* length of the data in bytes */
#define       MC_CMD_MAC_STATISTICS_OUT_DATALEN_OFST 0
#define       MC_CMD_MAC_STATISTICS_OUT_DATALEN_LEN 4
#define       MC_CMD_MAC_STATISTICS_OUT_DATA_OFST 4
#define       MC_CMD_MAC_STATISTICS_OUT_DATA_LEN 1
#define       MC_CMD_MAC_STATISTICS_OUT_DATA_MINNUM 1
#define       MC_CMD_MAC_STATISTICS_OUT_DATA_MAXNUM 248
#define       MC_CMD_MAC_STATISTICS_OUT_DATA_MAXNUM_MCDI2 1016

/* NET_PORT_HANDLE_DESC structuredef: Network port descriptor containing a port
 * handle and attributes used, for example, in MC_CMD_ENUM_PORTS.
 */
#define    NET_PORT_HANDLE_DESC_LEN 53
/* The handle to identify the port */
#define       NET_PORT_HANDLE_DESC_PORT_HANDLE_OFST 0
#define       NET_PORT_HANDLE_DESC_PORT_HANDLE_LEN 4
#define       NET_PORT_HANDLE_DESC_PORT_HANDLE_LBN 0
#define       NET_PORT_HANDLE_DESC_PORT_HANDLE_WIDTH 32
/* Includes the type of port e.g. physical, virtual or MAE MPORT and other
 * properties relevant to the port.
 */
#define       NET_PORT_HANDLE_DESC_PORT_PROPERTIES_OFST 4
#define       NET_PORT_HANDLE_DESC_PORT_PROPERTIES_LEN 8
#define       NET_PORT_HANDLE_DESC_PORT_PROPERTIES_LO_OFST 4
#define       NET_PORT_HANDLE_DESC_PORT_PROPERTIES_LO_LEN 4
#define       NET_PORT_HANDLE_DESC_PORT_PROPERTIES_LO_LBN 32
#define       NET_PORT_HANDLE_DESC_PORT_PROPERTIES_LO_WIDTH 32
#define       NET_PORT_HANDLE_DESC_PORT_PROPERTIES_HI_OFST 8
#define       NET_PORT_HANDLE_DESC_PORT_PROPERTIES_HI_LEN 4
#define       NET_PORT_HANDLE_DESC_PORT_PROPERTIES_HI_LBN 64
#define       NET_PORT_HANDLE_DESC_PORT_PROPERTIES_HI_WIDTH 32
#define        NET_PORT_HANDLE_DESC_PORT_TYPE_OFST 4
#define        NET_PORT_HANDLE_DESC_PORT_TYPE_LBN 0
#define        NET_PORT_HANDLE_DESC_PORT_TYPE_WIDTH 3
#define          NET_PORT_HANDLE_DESC_PHYSICAL 0x0 /* enum */
#define          NET_PORT_HANDLE_DESC_VIRTUAL 0x1 /* enum */
#define          NET_PORT_HANDLE_DESC_MPORT 0x2 /* enum */
#define        NET_PORT_HANDLE_DESC_IS_ZOMBIE_OFST 4
#define        NET_PORT_HANDLE_DESC_IS_ZOMBIE_LBN 8
#define        NET_PORT_HANDLE_DESC_IS_ZOMBIE_WIDTH 1
#define       NET_PORT_HANDLE_DESC_PORT_PROPERTIES_LBN 32
#define       NET_PORT_HANDLE_DESC_PORT_PROPERTIES_WIDTH 64
/* The dynamic change that led to the port enumeration */
#define       NET_PORT_HANDLE_DESC_ENTRY_SRC_OFST 12
#define       NET_PORT_HANDLE_DESC_ENTRY_SRC_LEN 1
/* enum: Indicates that the ENTRY_SRC field has not been initialized. */
#define          NET_PORT_HANDLE_DESC_UNKNOWN 0x0
/* enum: The port was enumerated at start of day. */
#define          NET_PORT_HANDLE_DESC_PRESENT 0x1
/* enum: The port was dynamically added. */
#define          NET_PORT_HANDLE_DESC_ADDED 0x2
/* enum: The port was dynamically deleted. */
#define          NET_PORT_HANDLE_DESC_DELETED 0x3
#define       NET_PORT_HANDLE_DESC_ENTRY_SRC_LBN 96
#define       NET_PORT_HANDLE_DESC_ENTRY_SRC_WIDTH 8
/* This is an opaque 40 byte label exposed to users as a unique identifier of
 * the port. It is represented as a zero-terminated ASCII string and assigned
 * by the port administrator which is typically either the firmware for a
 * physical port or the host software responsible for creating the virtual
 * port. The label is conveyed to the driver after assignment, which, unlike
 * the port administrator, does not need to know how to interpret the label.
 */
#define       NET_PORT_HANDLE_DESC_PORT_LABEL_OFST 13
#define       NET_PORT_HANDLE_DESC_PORT_LABEL_LEN 40
#define       NET_PORT_HANDLE_DESC_PORT_LABEL_LBN 104
#define       NET_PORT_HANDLE_DESC_PORT_LABEL_WIDTH 320


/***********************************/
/* MC_CMD_ENUM_PORTS
 * This command returns handles for all ports present in the system. The PCIe
 * function type of each port (either physical or virtual) is also reported.
 * After a start-of-day port enumeration, firmware keeps track of all available
 * ports upon creation or deletion and updates the ports if there is a change.
 * This command is cleared after a control interface reset (e.g. FLR,
 * ENTITY_RESET), in which case it must be called again to reenumerate the
 * ports. The command is also clear-on-read and repeated calls will drain the
 * buffer.
 */
#define MC_CMD_ENUM_PORTS 0x1e5
#undef MC_CMD_0x1e5_PRIVILEGE_CTG

#define MC_CMD_0x1e5_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_ENUM_PORTS_IN msgrequest */
#define    MC_CMD_ENUM_PORTS_IN_LEN 0

/* MC_CMD_ENUM_PORTS_OUT msgresponse */
#define    MC_CMD_ENUM_PORTS_OUT_LENMIN 12
#define    MC_CMD_ENUM_PORTS_OUT_LENMAX 252
#define    MC_CMD_ENUM_PORTS_OUT_LENMAX_MCDI2 1020
#define    MC_CMD_ENUM_PORTS_OUT_LEN(num) (12+1*(num))
#define    MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_NUM(len) (((len)-12)/1)
/* Any unused flags are reserved and must be ignored. */
#define       MC_CMD_ENUM_PORTS_OUT_FLAGS_OFST 0
#define       MC_CMD_ENUM_PORTS_OUT_FLAGS_LEN 4
#define        MC_CMD_ENUM_PORTS_OUT_MORE_OFST 0
#define        MC_CMD_ENUM_PORTS_OUT_MORE_LBN 0
#define        MC_CMD_ENUM_PORTS_OUT_MORE_WIDTH 1
/* The number of NET_PORT_HANDLE_DESC structures in PORT_HANDLES. */
#define       MC_CMD_ENUM_PORTS_OUT_PORT_COUNT_OFST 4
#define       MC_CMD_ENUM_PORTS_OUT_PORT_COUNT_LEN 4
#define       MC_CMD_ENUM_PORTS_OUT_SIZEOF_NET_PORT_HANDLE_DESC_OFST 8
#define       MC_CMD_ENUM_PORTS_OUT_SIZEOF_NET_PORT_HANDLE_DESC_LEN 4
/* Array of NET_PORT_HANDLE_DESC structures. Callers must use must use the
 * SIZEOF_NET_PORT_HANDLE_DESC field field as the array stride between entries.
 * This may also allow for tail padding for alignment. Fields beyond
 * SIZEOF_NET_PORT_HANDLE_DESC are not present.
 */
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_OFST 12
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_LEN 1
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_MINNUM 0
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_MAXNUM 240
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_MAXNUM_MCDI2 1008
/* See structuredef: NET_PORT_HANDLE_DESC */
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_PORT_HANDLE_OFST 12
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_PORT_HANDLE_LEN 4
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_PORT_PROPERTIES_OFST 16
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_PORT_PROPERTIES_LEN 8
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_PORT_PROPERTIES_LO_OFST 16
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_PORT_PROPERTIES_LO_LEN 4
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_PORT_PROPERTIES_LO_LBN 128
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_PORT_PROPERTIES_LO_WIDTH 32
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_PORT_PROPERTIES_HI_OFST 20
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_PORT_PROPERTIES_HI_LEN 4
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_PORT_PROPERTIES_HI_LBN 160
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_PORT_PROPERTIES_HI_WIDTH 32
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_PORT_TYPE_LBN 128
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_PORT_TYPE_WIDTH 3
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_IS_ZOMBIE_LBN 136
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_IS_ZOMBIE_WIDTH 1
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_ENTRY_SRC_OFST 24
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_ENTRY_SRC_LEN 1
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_PORT_LABEL_OFST 25
#define       MC_CMD_ENUM_PORTS_OUT_PORT_HANDLES_PORT_LABEL_LEN 40


/***********************************/
/* MC_CMD_GET_TRANSCEIVER_PROPERTIES
 * Read properties of the transceiver associated with the port. Can be either
 * for a fixed onboard transceiver or an inserted module. The returned data is
 * interpreted from the transceiver hardware and may be fixed up by the
 * firmware. Use MC_CMD_GET_MODULE_DATA to get raw undecoded data.
 */
#define MC_CMD_GET_TRANSCEIVER_PROPERTIES 0x1e6
#undef MC_CMD_0x1e6_PRIVILEGE_CTG

#define MC_CMD_0x1e6_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_GET_TRANSCEIVER_PROPERTIES_IN msgrequest */
#define    MC_CMD_GET_TRANSCEIVER_PROPERTIES_IN_LEN 4
/* Handle to port to get transceiver properties from. */
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_IN_PORT_HANDLE_OFST 0
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_IN_PORT_HANDLE_LEN 4

/* MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT msgresponse */
#define    MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_LEN 89
/* Supported technology abilities. */
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_TECH_ABILITIES_MASK_OFST 0
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_TECH_ABILITIES_MASK_LEN 16
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_ETH_TECH/TECH */
/* Reserved for future expansion to accommodate future Ethernet technology
 * expansion.
 */
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_RESERVED_OFST 16
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_RESERVED_LEN 16
/* Preferred FEC modes. This is a function of the cable type and length. */
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_PREFERRED_FEC_MASK_OFST 32
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_PREFERRED_FEC_MASK_LEN 4
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               FEC_TYPE/TYPE */
/* SFF-8042 code reported by the module. */
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_CODE_OFST 36
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_CODE_LEN 2
/* Medium. */
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_MEDIUM_OFST 38
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_MEDIUM_LEN 1
/* enum property: value */
#define          MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_UNKNOWN 0x0 /* enum */
#define          MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_COPPER 0x1 /* enum */
#define          MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_OPTICAL 0x2 /* enum */
#define          MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_BACKPLANE 0x3 /* enum */
/* Identifies the tech */
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_MEDIA_SUBTYPE_OFST 39
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_MEDIA_SUBTYPE_LEN 1
/* enum property: value */
/*               MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_UNKNOWN 0x0 */
/* enum: Ethernet over twisted-pair copper cables for distances up to 100
 * meters.
 */
#define          MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_BASET 0x1
/* enum: Ethernet over twin-axial, balanced copper cable. */
#define          MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_CR 0x2
/* enum: Ethernet over backplane for connections on the same board. */
#define          MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_KX 0x3
/* enum: Ethernet over a single backplane lane for connections between
 * different boards.
 */
#define          MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_KR 0x4
/* enum: Ethernet over copper backplane. */
#define          MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_KP 0x5
/* enum: Ethernet over fiber optic. */
#define          MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_BASEX 0x6
/* enum: Short range ethernet over multimode fiber optic (See IEEE 802.3 Clause
 * 49 and 52).
 */
#define          MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_SR 0x7
/* enum: Long range, extended range or far reach ethernet used with single mode
 * fiber optics.
 */
#define          MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_LR_ER_FR 0x8
/* enum: Long reach multimode ethernet over multimode optical fiber. */
#define          MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_LRM 0x9
/* enum: Very short reach PAM4 ethernet over multimode optical fiber (see IEEE
 * 802.3db).
 */
#define          MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_VR 0xa
/* enum: BASE-R encoding and PAM4 over single-mode fiber with reach up to at
 * least 500 meters (803.2 Clause 121 and 124)
 */
#define          MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_DR 0xb
/* String of the vendor name as intepreted by NMC firmware. NMC firmware
 * applies workarounds for known buggy transceivers. The vendor name is
 * presented as 16 bytes of ASCII characters padded with spaces. It can also be
 * represented as 16 bytes of zeros if the field is unspecified for the
 * connected module. See SFF-8472/CMIS specifications for details.
 */
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_VENDOR_NAME_OFST 40
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_VENDOR_NAME_LEN 1
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_VENDOR_NAME_NUM 16
/* The vendor part number as intepreted by NMC firmware. The field is presented
 * as 16 bytes of ASCII chars padded with spaces. It can also be 16 bytes of
 * zeros if the field is unspecified for the connected module.
 */
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_VENDOR_PN_OFST 56
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_VENDOR_PN_LEN 1
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_VENDOR_PN_NUM 16
/* Serial number of the module presented as 16 bytes of ASCII characters padded
 * with spaces. It can also be 16 bytes of zeros if the field is unspecified
 * for the connected module. See SFF-8472/CMIS specifications for details.
 */
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_SERIAL_NUMBER_OFST 72
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_SERIAL_NUMBER_LEN 1
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_SERIAL_NUMBER_NUM 16
/* This reports the number of module changes detected by the NMC firmware. */
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_PORT_MODULECHANGE_SEQ_NUM_OFST 88
#define       MC_CMD_GET_TRANSCEIVER_PROPERTIES_OUT_PORT_MODULECHANGE_SEQ_NUM_LEN 1


/***********************************/
/* MC_CMD_GET_FIXED_PORT_PROPERTIES
 */
#define MC_CMD_GET_FIXED_PORT_PROPERTIES 0x1e7
#undef MC_CMD_0x1e7_PRIVILEGE_CTG

#define MC_CMD_0x1e7_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_GET_FIXED_PORT_PROPERTIES_IN msgrequest: In this context, the port
 * consists of the MAC and the PHY, and excludes any modules inserted into the
 * cage. This information is fixed for a given board but not for a given ASIC.
 * This command reports properties for the port as it is currently configured,
 * and not its hardware capabilities, which can be better than the current
 * configuration.
 */
#define    MC_CMD_GET_FIXED_PORT_PROPERTIES_IN_LEN 4
/* Handle to the port to from which to retreive properties */
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_IN_PORT_HANDLE_OFST 0
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_IN_PORT_HANDLE_LEN 4

/* MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT msgresponse */
#define    MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_LEN 36
/* Supported capabilities of the port in its current configuration. */
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_ABILITIES_OFST 0
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_ABILITIES_LEN 25
/* See structuredef: MC_CMD_ETH_AN_FIELDS */
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_ABILITIES_TECH_MASK_OFST 0
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_ABILITIES_TECH_MASK_LEN 16
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_ABILITIES_FEC_MASK_OFST 16
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_ABILITIES_FEC_MASK_LEN 4
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_ABILITIES_FEC_REQ_OFST 20
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_ABILITIES_FEC_REQ_LEN 4
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_ABILITIES_PAUSE_MASK_OFST 24
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_ABILITIES_PAUSE_MASK_LEN 1
/* Number of lanes supported by the port in its current configuration. */
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_NUM_LANES_OFST 25
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_NUM_LANES_LEN 1
/* Bitmask of supported loopback modes. Where the response to this command
 * includes the LOOPBACK_MODES_MASK_V2 field, that field should be used in
 * preference to ensure that all available loopback modes are seen.
 */
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_LOOPBACK_MODES_MASK_OFST 26
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_LOOPBACK_MODES_MASK_LEN 1
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_LOOPBACK_V2/MODE */
/* This field serves as a cage index that uniquely identifies the cage to which
 * the module is connected. This is useful when splitter cables that have
 * multiple ports on a single cage are used.
 */
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_MDI_INDEX_OFST 27
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_MDI_INDEX_LEN 1
/* This bitmask is used to specify the lanes within the cage identified by
 * MDI_INDEX that are allocated to the port.
 */
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_MDI_LANE_MASK_OFST 28
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_MDI_LANE_MASK_LEN 1
/* Maximum frame length supported by the port in its current configuration. */
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_MAX_FRAME_LEN_OFST 32
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_MAX_FRAME_LEN_LEN 4

/* MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2 msgresponse */
#define    MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_LEN 48
/* Supported capabilities of the port in its current configuration. */
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_ABILITIES_OFST 0
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_ABILITIES_LEN 25
/* Number of lanes supported by the port in its current configuration. */
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_NUM_LANES_OFST 25
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_NUM_LANES_LEN 1
/* Bitmask of supported loopback modes. Where the response to this command
 * includes the LOOPBACK_MODES_MASK_V2 field, that field should be used in
 * preference to ensure that all available loopback modes are seen.
 */
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_LOOPBACK_MODES_MASK_OFST 26
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_LOOPBACK_MODES_MASK_LEN 1
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_LOOPBACK_V2/MODE */
/* This field serves as a cage index that uniquely identifies the cage to which
 * the module is connected. This is useful when splitter cables that have
 * multiple ports on a single cage are used.
 */
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_MDI_INDEX_OFST 27
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_MDI_INDEX_LEN 1
/* This bitmask is used to specify the lanes within the cage identified by
 * MDI_INDEX that are allocated to the port.
 */
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_MDI_LANE_MASK_OFST 28
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_MDI_LANE_MASK_LEN 1
/* Maximum frame length supported by the port in its current configuration. */
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_MAX_FRAME_LEN_OFST 32
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_MAX_FRAME_LEN_LEN 4
/* Bitmask of supported loopback modes. This field replaces the
 * LOOPBACK_MODES_MASK field which is defined under version 1 of this command.
 */
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_LOOPBACK_MODES_MASK_V2_OFST 40
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_LOOPBACK_MODES_MASK_V2_LEN 8
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_LOOPBACK_MODES_MASK_V2_LO_OFST 40
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_LOOPBACK_MODES_MASK_V2_LO_LEN 4
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_LOOPBACK_MODES_MASK_V2_LO_LBN 320
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_LOOPBACK_MODES_MASK_V2_LO_WIDTH 32
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_LOOPBACK_MODES_MASK_V2_HI_OFST 44
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_LOOPBACK_MODES_MASK_V2_HI_LEN 4
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_LOOPBACK_MODES_MASK_V2_HI_LBN 352
#define       MC_CMD_GET_FIXED_PORT_PROPERTIES_OUT_V2_LOOPBACK_MODES_MASK_V2_HI_WIDTH 32
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               MC_CMD_LOOPBACK_V2/MODE */


/***********************************/
/* MC_CMD_GET_MODULE_DATA
 * Read media-specific data from the PHY (e.g. SFP/SFP+ module ID information
 * for SFP+ PHYs). This command returns raw data from the module's EEPROM and
 * it is not interpreted by the MC. Use MC_CMD_GET_TRANSCEIVER_PROPERTIES to
 * get interpreted data. Return code: 0, ENOENT
 */
#define MC_CMD_GET_MODULE_DATA 0x1e8
#undef MC_CMD_0x1e8_PRIVILEGE_CTG

#define MC_CMD_0x1e8_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_GET_MODULE_DATA_IN msgrequest */
#define    MC_CMD_GET_MODULE_DATA_IN_LEN 16
/* Handle to identify the port from which to request module properties. */
#define       MC_CMD_GET_MODULE_DATA_IN_PORT_HANDLE_OFST 0
#define       MC_CMD_GET_MODULE_DATA_IN_PORT_HANDLE_LEN 4
/* 7 bit I2C address of the device. DEPRECATED: This field is replaced by
 * MODULE_ADDR in V2. Use V2 of this command for proper alignment and easier
 * access.
 */
#define       MC_CMD_GET_MODULE_DATA_IN_DEVADDR_LBN 32
#define       MC_CMD_GET_MODULE_DATA_IN_DEVADDR_WIDTH 7
/* 0 if the page does not support banked access, non-zero otherwise. Non-zero
 * BANK is valid if OFFSET is in the range 80h - ffh, i.e. in the Upper Memory
 * region.
 */
#define       MC_CMD_GET_MODULE_DATA_IN_BANK_OFST 6
#define       MC_CMD_GET_MODULE_DATA_IN_BANK_LEN 2
/* 0 if paged access is not supported, non-zero otherwise. Non-zero PAGE is
 * valid if OFFSET is in the range 80h - ffh.
 */
#define       MC_CMD_GET_MODULE_DATA_IN_PAGE_OFST 8
#define       MC_CMD_GET_MODULE_DATA_IN_PAGE_LEN 2
/* Offset in the range 00h - 7fh to access lower memory. Offset in the range
 * 80h - ffh to access upper memory
 */
#define       MC_CMD_GET_MODULE_DATA_IN_OFFSET_OFST 10
#define       MC_CMD_GET_MODULE_DATA_IN_OFFSET_LEN 1
#define       MC_CMD_GET_MODULE_DATA_IN_LENGTH_OFST 12
#define       MC_CMD_GET_MODULE_DATA_IN_LENGTH_LEN 4

/* MC_CMD_GET_MODULE_DATA_IN_V2 msgrequest: Updated MC_CMD_GET_MODULE_DATA with
 * 8-bit wide ADDRESSING field. This new field provides a correctly aligned
 * container for the 7-bit DEVADDR field from V1, now renamed MODULE_ADDR, to
 * ensure proper alignment.
 */
#define    MC_CMD_GET_MODULE_DATA_IN_V2_LEN 16
/* Handle to identify the port from which to request module properties. */
#define       MC_CMD_GET_MODULE_DATA_IN_V2_PORT_HANDLE_OFST 0
#define       MC_CMD_GET_MODULE_DATA_IN_V2_PORT_HANDLE_LEN 4
/* 7 bit I2C address of the device. DEPRECATED: This field is replaced by
 * MODULE_ADDR in V2. Use V2 of this command for proper alignment and easier
 * access.
 */
#define       MC_CMD_GET_MODULE_DATA_IN_V2_DEVADDR_LBN 32
#define       MC_CMD_GET_MODULE_DATA_IN_V2_DEVADDR_WIDTH 7
/* 0 if the page does not support banked access, non-zero otherwise. Non-zero
 * BANK is valid if OFFSET is in the range 80h - ffh, i.e. in the Upper Memory
 * region.
 */
#define       MC_CMD_GET_MODULE_DATA_IN_V2_BANK_OFST 6
#define       MC_CMD_GET_MODULE_DATA_IN_V2_BANK_LEN 2
/* 0 if paged access is not supported, non-zero otherwise. Non-zero PAGE is
 * valid if OFFSET is in the range 80h - ffh.
 */
#define       MC_CMD_GET_MODULE_DATA_IN_V2_PAGE_OFST 8
#define       MC_CMD_GET_MODULE_DATA_IN_V2_PAGE_LEN 2
/* Offset in the range 00h - 7fh to access lower memory. Offset in the range
 * 80h - ffh to access upper memory
 */
#define       MC_CMD_GET_MODULE_DATA_IN_V2_OFFSET_OFST 10
#define       MC_CMD_GET_MODULE_DATA_IN_V2_OFFSET_LEN 1
#define       MC_CMD_GET_MODULE_DATA_IN_V2_LENGTH_OFST 12
#define       MC_CMD_GET_MODULE_DATA_IN_V2_LENGTH_LEN 4
/* Container for 7 bit I2C addresses. */
#define       MC_CMD_GET_MODULE_DATA_IN_V2_ADDRESSING_OFST 4
#define       MC_CMD_GET_MODULE_DATA_IN_V2_ADDRESSING_LEN 1
#define        MC_CMD_GET_MODULE_DATA_IN_V2_MODULE_ADDR_OFST 4
#define        MC_CMD_GET_MODULE_DATA_IN_V2_MODULE_ADDR_LBN 0
#define        MC_CMD_GET_MODULE_DATA_IN_V2_MODULE_ADDR_WIDTH 7

/* MC_CMD_GET_MODULE_DATA_OUT msgresponse */
#define    MC_CMD_GET_MODULE_DATA_OUT_LENMIN 5
#define    MC_CMD_GET_MODULE_DATA_OUT_LENMAX 252
#define    MC_CMD_GET_MODULE_DATA_OUT_LENMAX_MCDI2 1020
#define    MC_CMD_GET_MODULE_DATA_OUT_LEN(num) (4+1*(num))
#define    MC_CMD_GET_MODULE_DATA_OUT_DATA_NUM(len) (((len)-4)/1)
/* length of the data in bytes */
#define       MC_CMD_GET_MODULE_DATA_OUT_DATALEN_OFST 0
#define       MC_CMD_GET_MODULE_DATA_OUT_DATALEN_LEN 4
#define       MC_CMD_GET_MODULE_DATA_OUT_DATA_OFST 4
#define       MC_CMD_GET_MODULE_DATA_OUT_DATA_LEN 1
#define       MC_CMD_GET_MODULE_DATA_OUT_DATA_MINNUM 1
#define       MC_CMD_GET_MODULE_DATA_OUT_DATA_MAXNUM 248
#define       MC_CMD_GET_MODULE_DATA_OUT_DATA_MAXNUM_MCDI2 1016

/* EVENT_MASK structuredef */
#define    EVENT_MASK_LEN 4
#define       EVENT_MASK_TYPE_OFST 0
#define       EVENT_MASK_TYPE_LEN 4
/* enum: PORT_LINKCHANGE event is enabled */
#define          EVENT_MASK_PORT_LINKCHANGE 0x0
/* enum: PORT_MODULECHANGE event is enabled */
#define          EVENT_MASK_PORT_MODULECHANGE 0x1
#define       EVENT_MASK_TYPE_LBN 0
#define       EVENT_MASK_TYPE_WIDTH 32


/***********************************/
/* MC_CMD_SET_NETPORT_EVENTS_MASK
 */
#define MC_CMD_SET_NETPORT_EVENTS_MASK 0x1e9
#undef MC_CMD_0x1e9_PRIVILEGE_CTG

#define MC_CMD_0x1e9_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_SET_NETPORT_EVENTS_MASK_IN msgrequest: Enable or disable delivery of
 * specified network port events for a given port identified by PORT_HANDLE. At
 * start of day, or after any control interface reset (FLR, ENTITY_RESET,
 * etc.), all event delivery is disabled for all ports associated with the
 * control interface.
 */
#define    MC_CMD_SET_NETPORT_EVENTS_MASK_IN_LEN 8
/* Handle to port to set event delivery mask. */
#define       MC_CMD_SET_NETPORT_EVENTS_MASK_IN_PORT_HANDLE_OFST 0
#define       MC_CMD_SET_NETPORT_EVENTS_MASK_IN_PORT_HANDLE_LEN 4
/* Bitmask of events to enable. Event delivery is enabled when corresponding
 * bit is 1, disabled when 0.
 */
#define       MC_CMD_SET_NETPORT_EVENTS_MASK_IN_EVENT_MASK_OFST 4
#define       MC_CMD_SET_NETPORT_EVENTS_MASK_IN_EVENT_MASK_LEN 4
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               EVENT_MASK/TYPE */

/* MC_CMD_SET_NETPORT_EVENTS_MASK_OUT msgresponse */
#define    MC_CMD_SET_NETPORT_EVENTS_MASK_OUT_LEN 0


/***********************************/
/* MC_CMD_GET_NETPORT_EVENTS_MASK
 */
#define MC_CMD_GET_NETPORT_EVENTS_MASK 0x1ea
#undef MC_CMD_0x1ea_PRIVILEGE_CTG

#define MC_CMD_0x1ea_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_GET_NETPORT_EVENTS_MASK_IN msgrequest: Get event delivery mask a
 * given port identified by PORT_HANDLE.
 */
#define    MC_CMD_GET_NETPORT_EVENTS_MASK_IN_LEN 4
/* Handle to port to get event deliver mask for. */
#define       MC_CMD_GET_NETPORT_EVENTS_MASK_IN_PORT_HANDLE_OFST 0
#define       MC_CMD_GET_NETPORT_EVENTS_MASK_IN_PORT_HANDLE_LEN 4

/* MC_CMD_GET_NETPORT_EVENTS_MASK_OUT msgresponse */
#define    MC_CMD_GET_NETPORT_EVENTS_MASK_OUT_LEN 4
/* Bitmask of events enabled. Event delivery is enabled when corresponding bit
 * is 1, disabled when 0.
 */
#define       MC_CMD_GET_NETPORT_EVENTS_MASK_OUT_EVENT_MASK_OFST 0
#define       MC_CMD_GET_NETPORT_EVENTS_MASK_OUT_EVENT_MASK_LEN 4
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               EVENT_MASK/TYPE */


/***********************************/
/* MC_CMD_GET_SUPPORTED_NETPORT_EVENTS
 */
#define MC_CMD_GET_SUPPORTED_NETPORT_EVENTS 0x1eb
#undef MC_CMD_0x1eb_PRIVILEGE_CTG

#define MC_CMD_0x1eb_PRIVILEGE_CTG SRIOV_CTG_LINK

/* MC_CMD_GET_SUPPORTED_NETPORT_EVENTS_IN msgrequest: Get network port events
 * supported by the platform. Information returned is fixed for a given NIC
 * platform.
 */
#define    MC_CMD_GET_SUPPORTED_NETPORT_EVENTS_IN_LEN 0

/* MC_CMD_GET_SUPPORTED_NETPORT_EVENTS_OUT msgresponse */
#define    MC_CMD_GET_SUPPORTED_NETPORT_EVENTS_OUT_LEN 4
/* Bitmask of events enabled. Event delivery is enabled when corresponding bit
 * is 1, disabled when 0.
 */
#define       MC_CMD_GET_SUPPORTED_NETPORT_EVENTS_OUT_EVENT_MASK_OFST 0
#define       MC_CMD_GET_SUPPORTED_NETPORT_EVENTS_OUT_EVENT_MASK_LEN 4
/* enum property: bitshift */
/*            Enum values, see field(s): */
/*               EVENT_MASK/TYPE */


/***********************************/
/* MC_CMD_GET_NETPORT_STATISTICS
 * Get generic MAC statistics. This call retrieves unified statistics managed
 * by the MC. The MC will populate and provide all supported statistics in the
 * format as returned by MC_CMD_MAC_STATISTICS_DESCRIPTOR. Refer to the
 * aforementioned command for the format and contents of the stats DMA buffer.
 * To ensure consistent and accurate results, it is essential for the driver to
 * initialize the DMA buffer with zeros when DMA mode is used. Returns: 0 on
 * success, ETIME if the DMA buffer is not ready, ENOENT on non-existent port
 * handle, and EINVAL on invalid parameters (DMA buffer too small)
 */
#define MC_CMD_GET_NETPORT_STATISTICS 0x1fa
#undef MC_CMD_0x1fa_PRIVILEGE_CTG

#define MC_CMD_0x1fa_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_GET_NETPORT_STATISTICS_IN msgrequest */
#define    MC_CMD_GET_NETPORT_STATISTICS_IN_LEN 20
/* Handle of port to get MAC statistics for. */
#define       MC_CMD_GET_NETPORT_STATISTICS_IN_PORT_HANDLE_OFST 0
#define       MC_CMD_GET_NETPORT_STATISTICS_IN_PORT_HANDLE_LEN 4
/* Contains options for querying the MAC statistics. */
#define       MC_CMD_GET_NETPORT_STATISTICS_IN_CMD_OFST 4
#define       MC_CMD_GET_NETPORT_STATISTICS_IN_CMD_LEN 4
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_DMA_OFST 4
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_DMA_LBN 0
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_DMA_WIDTH 1
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_CLEAR_OFST 4
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_CLEAR_LBN 1
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_CLEAR_WIDTH 1
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_PERIODIC_CHANGE_OFST 4
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_PERIODIC_CHANGE_LBN 2
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_PERIODIC_CHANGE_WIDTH 1
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_PERIODIC_ENABLE_OFST 4
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_PERIODIC_ENABLE_LBN 3
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_PERIODIC_ENABLE_WIDTH 1
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_PERIODIC_NOEVENT_OFST 4
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_PERIODIC_NOEVENT_LBN 4
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_PERIODIC_NOEVENT_WIDTH 1
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_PERIOD_MS_OFST 4
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_PERIOD_MS_LBN 15
#define        MC_CMD_GET_NETPORT_STATISTICS_IN_PERIOD_MS_WIDTH 17
/* Specifies the physical address of the DMA buffer to use for statistics
 * transfer. This field must contain a valid address under either of these
 * conditions: 1. DMA flag is set (immediate DMA requested) 2. Both
 * PERIODIC_CHANGE and PERIODIC_ENABLE are set (periodic DMA configured)
 */
#define       MC_CMD_GET_NETPORT_STATISTICS_IN_DMA_ADDR_OFST 8
#define       MC_CMD_GET_NETPORT_STATISTICS_IN_DMA_ADDR_LEN 8
#define       MC_CMD_GET_NETPORT_STATISTICS_IN_DMA_ADDR_LO_OFST 8
#define       MC_CMD_GET_NETPORT_STATISTICS_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_GET_NETPORT_STATISTICS_IN_DMA_ADDR_LO_LBN 64
#define       MC_CMD_GET_NETPORT_STATISTICS_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_GET_NETPORT_STATISTICS_IN_DMA_ADDR_HI_OFST 12
#define       MC_CMD_GET_NETPORT_STATISTICS_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_GET_NETPORT_STATISTICS_IN_DMA_ADDR_HI_LBN 96
#define       MC_CMD_GET_NETPORT_STATISTICS_IN_DMA_ADDR_HI_WIDTH 32
/* Specifies the length of the DMA buffer in bytes for statistics transfer. The
 * buffer size must be at least DMA_BUFFER_SIZE bytes (as returned by
 * MC_CMD_MAC_STATISTICS_DESCRIPTOR). Providing an insufficient buffer size
 * will result in an EINVAL error. This field must contain a valid length under
 * either of these conditions: 1. DMA flag is set (immediate DMA requested) 2.
 * Both PERIODIC_CHANGE and PERIODIC_ENABLE are set (periodic DMA configured)
 */
#define       MC_CMD_GET_NETPORT_STATISTICS_IN_DMA_LEN_OFST 16
#define       MC_CMD_GET_NETPORT_STATISTICS_IN_DMA_LEN_LEN 4

/* MC_CMD_GET_NETPORT_STATISTICS_OUT msgresponse */
#define    MC_CMD_GET_NETPORT_STATISTICS_OUT_LENMIN 0
#define    MC_CMD_GET_NETPORT_STATISTICS_OUT_LENMAX 248
#define    MC_CMD_GET_NETPORT_STATISTICS_OUT_LENMAX_MCDI2 1016
#define    MC_CMD_GET_NETPORT_STATISTICS_OUT_LEN(num) (0+8*(num))
#define    MC_CMD_GET_NETPORT_STATISTICS_OUT_STATS_NUM(len) (((len)-0)/8)
/* Statistics buffer. Zero-length if DMA mode is used. The statistics buffer is
 * an array of 8-byte counter values, containing the generation start marker,
 * stats counters, and generation end marker. The index of each counter in the
 * array is reported by the MAC_STATISTICS_DESCRIPTOR command. The same layout
 * is used for the DMA buffer for DMA mode stats.
 */
#define       MC_CMD_GET_NETPORT_STATISTICS_OUT_STATS_OFST 0
#define       MC_CMD_GET_NETPORT_STATISTICS_OUT_STATS_LEN 8
#define       MC_CMD_GET_NETPORT_STATISTICS_OUT_STATS_LO_OFST 0
#define       MC_CMD_GET_NETPORT_STATISTICS_OUT_STATS_LO_LEN 4
#define       MC_CMD_GET_NETPORT_STATISTICS_OUT_STATS_LO_LBN 0
#define       MC_CMD_GET_NETPORT_STATISTICS_OUT_STATS_LO_WIDTH 32
#define       MC_CMD_GET_NETPORT_STATISTICS_OUT_STATS_HI_OFST 4
#define       MC_CMD_GET_NETPORT_STATISTICS_OUT_STATS_HI_LEN 4
#define       MC_CMD_GET_NETPORT_STATISTICS_OUT_STATS_HI_LBN 32
#define       MC_CMD_GET_NETPORT_STATISTICS_OUT_STATS_HI_WIDTH 32
#define       MC_CMD_GET_NETPORT_STATISTICS_OUT_STATS_MINNUM 0
#define       MC_CMD_GET_NETPORT_STATISTICS_OUT_STATS_MAXNUM 31
#define       MC_CMD_GET_NETPORT_STATISTICS_OUT_STATS_MAXNUM_MCDI2 127

/* EVB_PORT_ID structuredef */
#define    EVB_PORT_ID_LEN 4
#define       EVB_PORT_ID_PORT_ID_OFST 0
#define       EVB_PORT_ID_PORT_ID_LEN 4
/* enum: An invalid port handle. */
#define          EVB_PORT_ID_NULL 0x0
/* enum: The port assigned to this function.. */
#define          EVB_PORT_ID_ASSIGNED 0x1000000
/* enum: External network port 0 */
#define          EVB_PORT_ID_MAC0 0x2000000
/* enum: External network port 1 */
#define          EVB_PORT_ID_MAC1 0x2000001
/* enum: External network port 2 */
#define          EVB_PORT_ID_MAC2 0x2000002
/* enum: External network port 3 */
#define          EVB_PORT_ID_MAC3 0x2000003
#define       EVB_PORT_ID_PORT_ID_LBN 0
#define       EVB_PORT_ID_PORT_ID_WIDTH 32

/* NVRAM_PARTITION_TYPE structuredef */
#define    NVRAM_PARTITION_TYPE_LEN 2
#define       NVRAM_PARTITION_TYPE_ID_OFST 0
#define       NVRAM_PARTITION_TYPE_ID_LEN 2
/* enum: Primary MC firmware partition */
#define          NVRAM_PARTITION_TYPE_MC_FIRMWARE 0x100
/* enum: NMC firmware partition (this is intentionally an alias of MC_FIRMWARE)
 */
#define          NVRAM_PARTITION_TYPE_NMC_FIRMWARE 0x100
/* enum: Secondary MC firmware partition */
#define          NVRAM_PARTITION_TYPE_MC_FIRMWARE_BACKUP 0x200
/* enum: Expansion ROM partition */
#define          NVRAM_PARTITION_TYPE_EXPANSION_ROM 0x300
/* enum: Static configuration TLV partition */
#define          NVRAM_PARTITION_TYPE_STATIC_CONFIG 0x400
/* enum: Factory configuration TLV partition (this is intentionally an alias of
 * STATIC_CONFIG)
 */
#define          NVRAM_PARTITION_TYPE_FACTORY_CONFIG 0x400
/* enum: Dynamic configuration TLV partition */
#define          NVRAM_PARTITION_TYPE_DYNAMIC_CONFIG 0x500
/* enum: User configuration TLV partition (this is intentionally an alias of
 * DYNAMIC_CONFIG)
 */
#define          NVRAM_PARTITION_TYPE_USER_CONFIG 0x500
/* enum: Expansion ROM configuration data for port 0 */
#define          NVRAM_PARTITION_TYPE_EXPROM_CONFIG_PORT0 0x600
/* enum: Synonym for EXPROM_CONFIG_PORT0 as used in pmap files */
#define          NVRAM_PARTITION_TYPE_EXPROM_CONFIG 0x600
/* enum: Expansion ROM configuration data for port 1 */
#define          NVRAM_PARTITION_TYPE_EXPROM_CONFIG_PORT1 0x601
/* enum: Expansion ROM configuration data for port 2 */
#define          NVRAM_PARTITION_TYPE_EXPROM_CONFIG_PORT2 0x602
/* enum: Expansion ROM configuration data for port 3 */
#define          NVRAM_PARTITION_TYPE_EXPROM_CONFIG_PORT3 0x603
/* enum: Non-volatile log output partition */
#define          NVRAM_PARTITION_TYPE_LOG 0x700
/* enum: Non-volatile log output partition for NMC firmware (this is
 * intentionally an alias of LOG)
 */
#define          NVRAM_PARTITION_TYPE_NMC_LOG 0x700
/* enum: Non-volatile log output of second core on dual-core device */
#define          NVRAM_PARTITION_TYPE_LOG_SLAVE 0x701
/* enum: RAM (volatile) log output partition */
#define          NVRAM_PARTITION_TYPE_RAM_LOG 0x702
/* enum: Device state dump output partition */
#define          NVRAM_PARTITION_TYPE_DUMP 0x800
/* enum: Crash log partition for NMC firmware */
#define          NVRAM_PARTITION_TYPE_NMC_CRASH_LOG 0x801
/* enum: Application license key storage partition */
#define          NVRAM_PARTITION_TYPE_LICENSE 0x900
/* enum: Start of range used for PHY partitions (low 8 bits are the PHY ID) */
#define          NVRAM_PARTITION_TYPE_PHY_MIN 0xa00
/* enum: End of range used for PHY partitions (low 8 bits are the PHY ID) */
#define          NVRAM_PARTITION_TYPE_PHY_MAX 0xaff
/* enum: Primary FPGA partition */
#define          NVRAM_PARTITION_TYPE_FPGA 0xb00
/* enum: Secondary FPGA partition */
#define          NVRAM_PARTITION_TYPE_FPGA_BACKUP 0xb01
/* enum: FC firmware partition */
#define          NVRAM_PARTITION_TYPE_FC_FIRMWARE 0xb02
/* enum: FC License partition */
#define          NVRAM_PARTITION_TYPE_FC_LICENSE 0xb03
/* enum: Non-volatile log output partition for FC */
#define          NVRAM_PARTITION_TYPE_FC_LOG 0xb04
/* enum: FPGA Stage 1 bitstream */
#define          NVRAM_PARTITION_TYPE_FPGA_STAGE1 0xb05
/* enum: FPGA Stage 2 bitstream */
#define          NVRAM_PARTITION_TYPE_FPGA_STAGE2 0xb06
/* enum: FPGA User XCLBIN / Programmable Region 0 bitstream */
#define          NVRAM_PARTITION_TYPE_FPGA_REGION0 0xb07
/* enum: FPGA User XCLBIN (this is intentionally an alias of FPGA_REGION0) */
#define          NVRAM_PARTITION_TYPE_FPGA_XCLBIN_USER 0xb07
/* enum: FPGA jump instruction (a.k.a. boot) partition to select Stage1
 * bitstream
 */
#define          NVRAM_PARTITION_TYPE_FPGA_JUMP 0xb08
/* enum: FPGA Validate XCLBIN */
#define          NVRAM_PARTITION_TYPE_FPGA_XCLBIN_VALIDATE 0xb09
/* enum: FPGA XOCL Configuration information */
#define          NVRAM_PARTITION_TYPE_FPGA_XOCL_CONFIG 0xb0a
/* enum: MUM firmware partition */
#define          NVRAM_PARTITION_TYPE_MUM_FIRMWARE 0xc00
/* enum: SUC firmware partition (this is intentionally an alias of
 * MUM_FIRMWARE)
 */
#define          NVRAM_PARTITION_TYPE_SUC_FIRMWARE 0xc00
/* enum: MUM Non-volatile log output partition. */
#define          NVRAM_PARTITION_TYPE_MUM_LOG 0xc01
/* enum: SUC Non-volatile log output partition (this is intentionally an alias
 * of MUM_LOG).
 */
#define          NVRAM_PARTITION_TYPE_SUC_LOG 0xc01
/* enum: MUM Application table partition. */
#define          NVRAM_PARTITION_TYPE_MUM_APPTABLE 0xc02
/* enum: MUM boot rom partition. */
#define          NVRAM_PARTITION_TYPE_MUM_BOOT_ROM 0xc03
/* enum: MUM production signatures & calibration rom partition. */
#define          NVRAM_PARTITION_TYPE_MUM_PROD_ROM 0xc04
/* enum: MUM user signatures & calibration rom partition. */
#define          NVRAM_PARTITION_TYPE_MUM_USER_ROM 0xc05
/* enum: MUM fuses and lockbits partition. */
#define          NVRAM_PARTITION_TYPE_MUM_FUSELOCK 0xc06
/* enum: UEFI expansion ROM if separate from PXE */
#define          NVRAM_PARTITION_TYPE_EXPANSION_UEFI 0xd00
/* enum: Used by the expansion ROM for logging */
#define          NVRAM_PARTITION_TYPE_PXE_LOG 0x1000
/* enum: Non-volatile log output partition for Expansion ROM (this is
 * intentionally an alias of PXE_LOG).
 */
#define          NVRAM_PARTITION_TYPE_EXPROM_LOG 0x1000
/* enum: Used for XIP code of shmbooted images */
#define          NVRAM_PARTITION_TYPE_XIP_SCRATCH 0x1100
/* enum: Spare partition 2 */
#define          NVRAM_PARTITION_TYPE_SPARE_2 0x1200
/* enum: Manufacturing partition. Used during manufacture to pass information
 * between XJTAG and Manftest.
 */
#define          NVRAM_PARTITION_TYPE_MANUFACTURING 0x1300
/* enum: Deployment configuration TLV partition (this is intentionally an alias
 * of MANUFACTURING)
 */
#define          NVRAM_PARTITION_TYPE_DEPLOYMENT_CONFIG 0x1300
/* enum: Spare partition 4 */
#define          NVRAM_PARTITION_TYPE_SPARE_4 0x1400
/* enum: Spare partition 5 */
#define          NVRAM_PARTITION_TYPE_SPARE_5 0x1500
/* enum: Partition for reporting MC status. See mc_flash_layout.h
 * medford_mc_status_hdr_t for layout on Medford.
 */
#define          NVRAM_PARTITION_TYPE_STATUS 0x1600
/* enum: Spare partition 13 */
#define          NVRAM_PARTITION_TYPE_SPARE_13 0x1700
/* enum: Spare partition 14 */
#define          NVRAM_PARTITION_TYPE_SPARE_14 0x1800
/* enum: Spare partition 15 */
#define          NVRAM_PARTITION_TYPE_SPARE_15 0x1900
/* enum: Spare partition 16 */
#define          NVRAM_PARTITION_TYPE_SPARE_16 0x1a00
/* enum: Factory defaults for dynamic configuration */
#define          NVRAM_PARTITION_TYPE_DYNCONFIG_DEFAULTS 0x1b00
/* enum: Factory defaults for expansion ROM configuration */
#define          NVRAM_PARTITION_TYPE_ROMCONFIG_DEFAULTS 0x1c00
/* enum: Field Replaceable Unit inventory information for use on IPMI
 * platforms. See SF-119124-PS. The STATIC_CONFIG partition may contain a
 * subset of the information stored in this partition.
 */
#define          NVRAM_PARTITION_TYPE_FRU_INFORMATION 0x1d00
/* enum: Bundle image partition */
#define          NVRAM_PARTITION_TYPE_BUNDLE 0x1e00
/* enum: Bundle metadata partition that holds additional information related to
 * a bundle update in TLV format
 */
#define          NVRAM_PARTITION_TYPE_BUNDLE_METADATA 0x1e01
/* enum: Bundle update non-volatile log output partition */
#define          NVRAM_PARTITION_TYPE_BUNDLE_LOG 0x1e02
/* enum: Partition for Solarflare gPXE bootrom installed via Bundle update. */
#define          NVRAM_PARTITION_TYPE_EXPANSION_ROM_INTERNAL 0x1e03
/* enum: Partition to store ASN.1 format Bundle Signature for checking. */
#define          NVRAM_PARTITION_TYPE_BUNDLE_SIGNATURE 0x1e04
/* enum: Test partition on SmartNIC system microcontroller (SUC) */
#define          NVRAM_PARTITION_TYPE_SUC_TEST 0x1f00
/* enum: System microcontroller access to primary FPGA flash. */
#define          NVRAM_PARTITION_TYPE_SUC_FPGA_PRIMARY 0x1f01
/* enum: System microcontroller access to secondary FPGA flash (if present) */
#define          NVRAM_PARTITION_TYPE_SUC_FPGA_SECONDARY 0x1f02
/* enum: System microcontroller access to primary System-on-Chip flash */
#define          NVRAM_PARTITION_TYPE_SUC_SOC_PRIMARY 0x1f03
/* enum: System microcontroller access to secondary System-on-Chip flash (if
 * present)
 */
#define          NVRAM_PARTITION_TYPE_SUC_SOC_SECONDARY 0x1f04
/* enum: System microcontroller critical failure logs. Contains structured
 * details of sensors leading up to a critical failure (where the board is shut
 * down).
 */
#define          NVRAM_PARTITION_TYPE_SUC_FAILURE_LOG 0x1f05
/* enum: System-on-Chip configuration information (see XN-200467-PS). */
#define          NVRAM_PARTITION_TYPE_SUC_SOC_CONFIG 0x1f07
/* enum: System-on-Chip update information. */
#define          NVRAM_PARTITION_TYPE_SOC_UPDATE 0x2003
/* enum: Virtual partition. Write-only. Writes will actually be sent to an
 * appropriate partition (for instance BUNDLE if the data starts with the magic
 * number for a bundle update), or discarded with an error if not recognised as
 * a supported type.
 */
#define          NVRAM_PARTITION_TYPE_AUTO 0x2100
/* enum: MC/NMC (first stage) bootloader firmware. (For X4, see XN-202072-PS
 * and XN-202084-SW section 3.1).
 */
#define          NVRAM_PARTITION_TYPE_BOOTLOADER 0x2200
/* enum: Start of reserved value range (firmware may use for any purpose) */
#define          NVRAM_PARTITION_TYPE_RESERVED_VALUES_MIN 0xff00
/* enum: End of reserved value range (firmware may use for any purpose) */
#define          NVRAM_PARTITION_TYPE_RESERVED_VALUES_MAX 0xfffd
/* enum: Recovery partition map (provided if real map is missing or corrupt) */
#define          NVRAM_PARTITION_TYPE_RECOVERY_MAP 0xfffe
/* enum: Recovery Flash Partition Table, see SF-122606-TC. (this is
 * intentionally an alias of RECOVERY_MAP)
 */
#define          NVRAM_PARTITION_TYPE_RECOVERY_FPT 0xfffe
/* enum: Partition map (real map as stored in flash) */
#define          NVRAM_PARTITION_TYPE_PARTITION_MAP 0xffff
/* enum: Flash Partition Table, see SF-122606-TC. (this is intentionally an
 * alias of PARTITION_MAP)
 */
#define          NVRAM_PARTITION_TYPE_FPT 0xffff
#define       NVRAM_PARTITION_TYPE_ID_LBN 0
#define       NVRAM_PARTITION_TYPE_ID_WIDTH 16

/* LICENSED_APP_ID structuredef */
#define    LICENSED_APP_ID_LEN 4
#define       LICENSED_APP_ID_ID_OFST 0
#define       LICENSED_APP_ID_ID_LEN 4
/* enum: OpenOnload */
#define          LICENSED_APP_ID_ONLOAD 0x1
/* enum: PTP timestamping */
#define          LICENSED_APP_ID_PTP 0x2
/* enum: SolarCapture Pro */
#define          LICENSED_APP_ID_SOLARCAPTURE_PRO 0x4
/* enum: SolarSecure filter engine */
#define          LICENSED_APP_ID_SOLARSECURE 0x8
/* enum: Performance monitor */
#define          LICENSED_APP_ID_PERF_MONITOR 0x10
/* enum: SolarCapture Live */
#define          LICENSED_APP_ID_SOLARCAPTURE_LIVE 0x20
/* enum: Capture SolarSystem */
#define          LICENSED_APP_ID_CAPTURE_SOLARSYSTEM 0x40
/* enum: Network Access Control */
#define          LICENSED_APP_ID_NETWORK_ACCESS_CONTROL 0x80
/* enum: TCP Direct */
#define          LICENSED_APP_ID_TCP_DIRECT 0x100
/* enum: Low Latency */
#define          LICENSED_APP_ID_LOW_LATENCY 0x200
/* enum: SolarCapture Tap */
#define          LICENSED_APP_ID_SOLARCAPTURE_TAP 0x400
/* enum: Capture SolarSystem 40G */
#define          LICENSED_APP_ID_CAPTURE_SOLARSYSTEM_40G 0x800
/* enum: Capture SolarSystem 1G */
#define          LICENSED_APP_ID_CAPTURE_SOLARSYSTEM_1G 0x1000
/* enum: ScaleOut Onload */
#define          LICENSED_APP_ID_SCALEOUT_ONLOAD 0x2000
/* enum: SCS Network Analytics Dashboard */
#define          LICENSED_APP_ID_DSHBRD 0x4000
/* enum: SolarCapture Trading Analytics */
#define          LICENSED_APP_ID_SCATRD 0x8000
#define       LICENSED_APP_ID_ID_LBN 0
#define       LICENSED_APP_ID_ID_WIDTH 32

/* LICENSED_V3_FEATURES structuredef */
#define    LICENSED_V3_FEATURES_LEN 8
/* Bitmask of licensed firmware features */
#define       LICENSED_V3_FEATURES_MASK_OFST 0
#define       LICENSED_V3_FEATURES_MASK_LEN 8
#define       LICENSED_V3_FEATURES_MASK_LO_OFST 0
#define       LICENSED_V3_FEATURES_MASK_LO_LEN 4
#define       LICENSED_V3_FEATURES_MASK_LO_LBN 0
#define       LICENSED_V3_FEATURES_MASK_LO_WIDTH 32
#define       LICENSED_V3_FEATURES_MASK_HI_OFST 4
#define       LICENSED_V3_FEATURES_MASK_HI_LEN 4
#define       LICENSED_V3_FEATURES_MASK_HI_LBN 32
#define       LICENSED_V3_FEATURES_MASK_HI_WIDTH 32
#define        LICENSED_V3_FEATURES_RX_CUT_THROUGH_OFST 0
#define        LICENSED_V3_FEATURES_RX_CUT_THROUGH_LBN 0
#define        LICENSED_V3_FEATURES_RX_CUT_THROUGH_WIDTH 1
#define        LICENSED_V3_FEATURES_PIO_OFST 0
#define        LICENSED_V3_FEATURES_PIO_LBN 1
#define        LICENSED_V3_FEATURES_PIO_WIDTH 1
#define        LICENSED_V3_FEATURES_EVQ_TIMER_OFST 0
#define        LICENSED_V3_FEATURES_EVQ_TIMER_LBN 2
#define        LICENSED_V3_FEATURES_EVQ_TIMER_WIDTH 1
#define        LICENSED_V3_FEATURES_CLOCK_OFST 0
#define        LICENSED_V3_FEATURES_CLOCK_LBN 3
#define        LICENSED_V3_FEATURES_CLOCK_WIDTH 1
#define        LICENSED_V3_FEATURES_RX_TIMESTAMPS_OFST 0
#define        LICENSED_V3_FEATURES_RX_TIMESTAMPS_LBN 4
#define        LICENSED_V3_FEATURES_RX_TIMESTAMPS_WIDTH 1
#define        LICENSED_V3_FEATURES_TX_TIMESTAMPS_OFST 0
#define        LICENSED_V3_FEATURES_TX_TIMESTAMPS_LBN 5
#define        LICENSED_V3_FEATURES_TX_TIMESTAMPS_WIDTH 1
#define        LICENSED_V3_FEATURES_RX_SNIFF_OFST 0
#define        LICENSED_V3_FEATURES_RX_SNIFF_LBN 6
#define        LICENSED_V3_FEATURES_RX_SNIFF_WIDTH 1
#define        LICENSED_V3_FEATURES_TX_SNIFF_OFST 0
#define        LICENSED_V3_FEATURES_TX_SNIFF_LBN 7
#define        LICENSED_V3_FEATURES_TX_SNIFF_WIDTH 1
#define        LICENSED_V3_FEATURES_PROXY_FILTER_OPS_OFST 0
#define        LICENSED_V3_FEATURES_PROXY_FILTER_OPS_LBN 8
#define        LICENSED_V3_FEATURES_PROXY_FILTER_OPS_WIDTH 1
#define        LICENSED_V3_FEATURES_EVENT_CUT_THROUGH_OFST 0
#define        LICENSED_V3_FEATURES_EVENT_CUT_THROUGH_LBN 9
#define        LICENSED_V3_FEATURES_EVENT_CUT_THROUGH_WIDTH 1
#define       LICENSED_V3_FEATURES_MASK_LBN 0
#define       LICENSED_V3_FEATURES_MASK_WIDTH 64

/* TX_TIMESTAMP_EVENT structuredef */
#define    TX_TIMESTAMP_EVENT_LEN 6
/* lower 16 bits of timestamp data */
#define       TX_TIMESTAMP_EVENT_TSTAMP_DATA_LO_OFST 0
#define       TX_TIMESTAMP_EVENT_TSTAMP_DATA_LO_LEN 2
#define       TX_TIMESTAMP_EVENT_TSTAMP_DATA_LO_LBN 0
#define       TX_TIMESTAMP_EVENT_TSTAMP_DATA_LO_WIDTH 16
/* Type of TX event, ordinary TX completion, low or high part of TX timestamp
 */
#define       TX_TIMESTAMP_EVENT_TX_EV_TYPE_OFST 3
#define       TX_TIMESTAMP_EVENT_TX_EV_TYPE_LEN 1
/* enum: This is a TX completion event, not a timestamp */
#define          TX_TIMESTAMP_EVENT_TX_EV_COMPLETION 0x0
/* enum: This is a TX completion event for a CTPIO transmit. The event format
 * is the same as for TX_EV_COMPLETION.
 */
#define          TX_TIMESTAMP_EVENT_TX_EV_CTPIO_COMPLETION 0x11
/* enum: This is the low part of a TX timestamp for a CTPIO transmission. The
 * event format is the same as for TX_EV_TSTAMP_LO
 */
#define          TX_TIMESTAMP_EVENT_TX_EV_CTPIO_TS_LO 0x12
/* enum: This is the high part of a TX timestamp for a CTPIO transmission. The
 * event format is the same as for TX_EV_TSTAMP_HI
 */
#define          TX_TIMESTAMP_EVENT_TX_EV_CTPIO_TS_HI 0x13
/* enum: This is the low part of a TX timestamp event */
#define          TX_TIMESTAMP_EVENT_TX_EV_TSTAMP_LO 0x51
/* enum: This is the high part of a TX timestamp event */
#define          TX_TIMESTAMP_EVENT_TX_EV_TSTAMP_HI 0x52
#define       TX_TIMESTAMP_EVENT_TX_EV_TYPE_LBN 24
#define       TX_TIMESTAMP_EVENT_TX_EV_TYPE_WIDTH 8
/* upper 16 bits of timestamp data */
#define       TX_TIMESTAMP_EVENT_TSTAMP_DATA_HI_OFST 4
#define       TX_TIMESTAMP_EVENT_TSTAMP_DATA_HI_LEN 2
#define       TX_TIMESTAMP_EVENT_TSTAMP_DATA_HI_LBN 32
#define       TX_TIMESTAMP_EVENT_TSTAMP_DATA_HI_WIDTH 16

/* RSS_MODE structuredef */
#define    RSS_MODE_LEN 1
/* The RSS mode for a particular packet type is a value from 0 - 15 which can
 * be considered as 4 bits selecting which fields are included in the hash. (A
 * value 0 effectively disables RSS spreading for the packet type.) The YAML
 * generation tools require this structure to be a whole number of bytes wide,
 * but only 4 bits are relevant.
 */
#define       RSS_MODE_HASH_SELECTOR_OFST 0
#define       RSS_MODE_HASH_SELECTOR_LEN 1
#define        RSS_MODE_HASH_SRC_ADDR_OFST 0
#define        RSS_MODE_HASH_SRC_ADDR_LBN 0
#define        RSS_MODE_HASH_SRC_ADDR_WIDTH 1
#define        RSS_MODE_HASH_DST_ADDR_OFST 0
#define        RSS_MODE_HASH_DST_ADDR_LBN 1
#define        RSS_MODE_HASH_DST_ADDR_WIDTH 1
#define        RSS_MODE_HASH_SRC_PORT_OFST 0
#define        RSS_MODE_HASH_SRC_PORT_LBN 2
#define        RSS_MODE_HASH_SRC_PORT_WIDTH 1
#define        RSS_MODE_HASH_DST_PORT_OFST 0
#define        RSS_MODE_HASH_DST_PORT_LBN 3
#define        RSS_MODE_HASH_DST_PORT_WIDTH 1
#define       RSS_MODE_HASH_SELECTOR_LBN 0
#define       RSS_MODE_HASH_SELECTOR_WIDTH 8


/***********************************/
/* MC_CMD_INIT_EVQ
 * Set up an event queue according to the supplied parameters. The IN arguments
 * end with an address for each 4k of host memory required to back the EVQ.
 */
#define MC_CMD_INIT_EVQ 0x80
#undef MC_CMD_0x80_PRIVILEGE_CTG

#define MC_CMD_0x80_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_INIT_EVQ_IN msgrequest */
#define    MC_CMD_INIT_EVQ_IN_LENMIN 44
#define    MC_CMD_INIT_EVQ_IN_LENMAX 548
#define    MC_CMD_INIT_EVQ_IN_LENMAX_MCDI2 548
#define    MC_CMD_INIT_EVQ_IN_LEN(num) (36+8*(num))
#define    MC_CMD_INIT_EVQ_IN_DMA_ADDR_NUM(len) (((len)-36)/8)
/* Size, in entries */
#define       MC_CMD_INIT_EVQ_IN_SIZE_OFST 0
#define       MC_CMD_INIT_EVQ_IN_SIZE_LEN 4
/* Desired instance. Must be set to a specific instance, which is a function
 * local queue index. The calling client must be the currently-assigned user of
 * this VI (see MC_CMD_SET_VI_USER).
 */
#define       MC_CMD_INIT_EVQ_IN_INSTANCE_OFST 4
#define       MC_CMD_INIT_EVQ_IN_INSTANCE_LEN 4
/* The initial timer value. The load value is ignored if the timer mode is DIS.
 */
#define       MC_CMD_INIT_EVQ_IN_TMR_LOAD_OFST 8
#define       MC_CMD_INIT_EVQ_IN_TMR_LOAD_LEN 4
/* The reload value is ignored in one-shot modes */
#define       MC_CMD_INIT_EVQ_IN_TMR_RELOAD_OFST 12
#define       MC_CMD_INIT_EVQ_IN_TMR_RELOAD_LEN 4
/* tbd */
#define       MC_CMD_INIT_EVQ_IN_FLAGS_OFST 16
#define       MC_CMD_INIT_EVQ_IN_FLAGS_LEN 4
#define        MC_CMD_INIT_EVQ_IN_FLAG_INTERRUPTING_OFST 16
#define        MC_CMD_INIT_EVQ_IN_FLAG_INTERRUPTING_LBN 0
#define        MC_CMD_INIT_EVQ_IN_FLAG_INTERRUPTING_WIDTH 1
#define        MC_CMD_INIT_EVQ_IN_FLAG_RPTR_DOS_OFST 16
#define        MC_CMD_INIT_EVQ_IN_FLAG_RPTR_DOS_LBN 1
#define        MC_CMD_INIT_EVQ_IN_FLAG_RPTR_DOS_WIDTH 1
#define        MC_CMD_INIT_EVQ_IN_FLAG_INT_ARMD_OFST 16
#define        MC_CMD_INIT_EVQ_IN_FLAG_INT_ARMD_LBN 2
#define        MC_CMD_INIT_EVQ_IN_FLAG_INT_ARMD_WIDTH 1
#define        MC_CMD_INIT_EVQ_IN_FLAG_CUT_THRU_OFST 16
#define        MC_CMD_INIT_EVQ_IN_FLAG_CUT_THRU_LBN 3
#define        MC_CMD_INIT_EVQ_IN_FLAG_CUT_THRU_WIDTH 1
#define        MC_CMD_INIT_EVQ_IN_FLAG_RX_MERGE_OFST 16
#define        MC_CMD_INIT_EVQ_IN_FLAG_RX_MERGE_LBN 4
#define        MC_CMD_INIT_EVQ_IN_FLAG_RX_MERGE_WIDTH 1
#define        MC_CMD_INIT_EVQ_IN_FLAG_TX_MERGE_OFST 16
#define        MC_CMD_INIT_EVQ_IN_FLAG_TX_MERGE_LBN 5
#define        MC_CMD_INIT_EVQ_IN_FLAG_TX_MERGE_WIDTH 1
#define        MC_CMD_INIT_EVQ_IN_FLAG_USE_TIMER_OFST 16
#define        MC_CMD_INIT_EVQ_IN_FLAG_USE_TIMER_LBN 6
#define        MC_CMD_INIT_EVQ_IN_FLAG_USE_TIMER_WIDTH 1
#define       MC_CMD_INIT_EVQ_IN_TMR_MODE_OFST 20
#define       MC_CMD_INIT_EVQ_IN_TMR_MODE_LEN 4
/* enum: Disabled */
#define          MC_CMD_INIT_EVQ_IN_TMR_MODE_DIS 0x0
/* enum: Immediate */
#define          MC_CMD_INIT_EVQ_IN_TMR_IMMED_START 0x1
/* enum: Triggered */
#define          MC_CMD_INIT_EVQ_IN_TMR_TRIG_START 0x2
/* enum: Hold-off */
#define          MC_CMD_INIT_EVQ_IN_TMR_INT_HLDOFF 0x3
/* Target EVQ for wakeups if in wakeup mode. */
#define       MC_CMD_INIT_EVQ_IN_TARGET_EVQ_OFST 24
#define       MC_CMD_INIT_EVQ_IN_TARGET_EVQ_LEN 4
/* Target interrupt if in interrupting mode (note union with target EVQ). Use
 * MC_CMD_RESOURCE_INSTANCE_ANY unless a specific one required for test
 * purposes.
 */
#define       MC_CMD_INIT_EVQ_IN_IRQ_NUM_OFST 24
#define       MC_CMD_INIT_EVQ_IN_IRQ_NUM_LEN 4
/* Event Counter Mode. */
#define       MC_CMD_INIT_EVQ_IN_COUNT_MODE_OFST 28
#define       MC_CMD_INIT_EVQ_IN_COUNT_MODE_LEN 4
/* enum: Disabled */
#define          MC_CMD_INIT_EVQ_IN_COUNT_MODE_DIS 0x0
/* enum: Disabled */
#define          MC_CMD_INIT_EVQ_IN_COUNT_MODE_RX 0x1
/* enum: Disabled */
#define          MC_CMD_INIT_EVQ_IN_COUNT_MODE_TX 0x2
/* enum: Disabled */
#define          MC_CMD_INIT_EVQ_IN_COUNT_MODE_RXTX 0x3
/* Event queue packet count threshold. */
#define       MC_CMD_INIT_EVQ_IN_COUNT_THRSHLD_OFST 32
#define       MC_CMD_INIT_EVQ_IN_COUNT_THRSHLD_LEN 4
/* 64-bit address of 4k of 4k-aligned host memory buffer */
#define       MC_CMD_INIT_EVQ_IN_DMA_ADDR_OFST 36
#define       MC_CMD_INIT_EVQ_IN_DMA_ADDR_LEN 8
#define       MC_CMD_INIT_EVQ_IN_DMA_ADDR_LO_OFST 36
#define       MC_CMD_INIT_EVQ_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_INIT_EVQ_IN_DMA_ADDR_LO_LBN 288
#define       MC_CMD_INIT_EVQ_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_INIT_EVQ_IN_DMA_ADDR_HI_OFST 40
#define       MC_CMD_INIT_EVQ_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_INIT_EVQ_IN_DMA_ADDR_HI_LBN 320
#define       MC_CMD_INIT_EVQ_IN_DMA_ADDR_HI_WIDTH 32
#define       MC_CMD_INIT_EVQ_IN_DMA_ADDR_MINNUM 1
#define       MC_CMD_INIT_EVQ_IN_DMA_ADDR_MAXNUM 64
#define       MC_CMD_INIT_EVQ_IN_DMA_ADDR_MAXNUM_MCDI2 64

/* MC_CMD_INIT_EVQ_OUT msgresponse */
#define    MC_CMD_INIT_EVQ_OUT_LEN 4
/* Only valid if INTRFLAG was true */
#define       MC_CMD_INIT_EVQ_OUT_IRQ_OFST 0
#define       MC_CMD_INIT_EVQ_OUT_IRQ_LEN 4

/* MC_CMD_INIT_EVQ_V2_IN msgrequest */
#define    MC_CMD_INIT_EVQ_V2_IN_LENMIN 44
#define    MC_CMD_INIT_EVQ_V2_IN_LENMAX 548
#define    MC_CMD_INIT_EVQ_V2_IN_LENMAX_MCDI2 548
#define    MC_CMD_INIT_EVQ_V2_IN_LEN(num) (36+8*(num))
#define    MC_CMD_INIT_EVQ_V2_IN_DMA_ADDR_NUM(len) (((len)-36)/8)
/* Size, in entries */
#define       MC_CMD_INIT_EVQ_V2_IN_SIZE_OFST 0
#define       MC_CMD_INIT_EVQ_V2_IN_SIZE_LEN 4
/* Desired instance. Must be set to a specific instance, which is a function
 * local queue index. The calling client must be the currently-assigned user of
 * this VI (see MC_CMD_SET_VI_USER).
 */
#define       MC_CMD_INIT_EVQ_V2_IN_INSTANCE_OFST 4
#define       MC_CMD_INIT_EVQ_V2_IN_INSTANCE_LEN 4
/* The initial timer value. The load value is ignored if the timer mode is DIS.
 */
#define       MC_CMD_INIT_EVQ_V2_IN_TMR_LOAD_OFST 8
#define       MC_CMD_INIT_EVQ_V2_IN_TMR_LOAD_LEN 4
/* The reload value is ignored in one-shot modes */
#define       MC_CMD_INIT_EVQ_V2_IN_TMR_RELOAD_OFST 12
#define       MC_CMD_INIT_EVQ_V2_IN_TMR_RELOAD_LEN 4
/* tbd */
#define       MC_CMD_INIT_EVQ_V2_IN_FLAGS_OFST 16
#define       MC_CMD_INIT_EVQ_V2_IN_FLAGS_LEN 4
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_INTERRUPTING_OFST 16
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_INTERRUPTING_LBN 0
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_INTERRUPTING_WIDTH 1
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_RPTR_DOS_OFST 16
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_RPTR_DOS_LBN 1
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_RPTR_DOS_WIDTH 1
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_INT_ARMD_OFST 16
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_INT_ARMD_LBN 2
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_INT_ARMD_WIDTH 1
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_CUT_THRU_OFST 16
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_CUT_THRU_LBN 3
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_CUT_THRU_WIDTH 1
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_RX_MERGE_OFST 16
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_RX_MERGE_LBN 4
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_RX_MERGE_WIDTH 1
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_TX_MERGE_OFST 16
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_TX_MERGE_LBN 5
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_TX_MERGE_WIDTH 1
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_USE_TIMER_OFST 16
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_USE_TIMER_LBN 6
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_USE_TIMER_WIDTH 1
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_TYPE_OFST 16
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_TYPE_LBN 7
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_TYPE_WIDTH 4
/* enum: All initialisation flags specified by host. */
#define          MC_CMD_INIT_EVQ_V2_IN_FLAG_TYPE_MANUAL 0x0
/* enum: MEDFORD only. Certain initialisation flags specified by host may be
 * over-ridden by firmware based on licenses and firmware variant in order to
 * provide the lowest latency achievable. See
 * MC_CMD_INIT_EVQ_V2/MC_CMD_INIT_EVQ_V2_OUT/FLAGS for list of affected flags.
 */
#define          MC_CMD_INIT_EVQ_V2_IN_FLAG_TYPE_LOW_LATENCY 0x1
/* enum: MEDFORD only. Certain initialisation flags specified by host may be
 * over-ridden by firmware based on licenses and firmware variant in order to
 * provide the best throughput achievable. See
 * MC_CMD_INIT_EVQ_V2/MC_CMD_INIT_EVQ_V2_OUT/FLAGS for list of affected flags.
 */
#define          MC_CMD_INIT_EVQ_V2_IN_FLAG_TYPE_THROUGHPUT 0x2
/* enum: MEDFORD only. Certain initialisation flags may be over-ridden by
 * firmware based on licenses and firmware variant. See
 * MC_CMD_INIT_EVQ_V2/MC_CMD_INIT_EVQ_V2_OUT/FLAGS for list of affected flags.
 */
#define          MC_CMD_INIT_EVQ_V2_IN_FLAG_TYPE_AUTO 0x3
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_EXT_WIDTH_OFST 16
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_EXT_WIDTH_LBN 11
#define        MC_CMD_INIT_EVQ_V2_IN_FLAG_EXT_WIDTH_WIDTH 1
#define       MC_CMD_INIT_EVQ_V2_IN_TMR_MODE_OFST 20
#define       MC_CMD_INIT_EVQ_V2_IN_TMR_MODE_LEN 4
/* enum: Disabled */
#define          MC_CMD_INIT_EVQ_V2_IN_TMR_MODE_DIS 0x0
/* enum: Immediate */
#define          MC_CMD_INIT_EVQ_V2_IN_TMR_IMMED_START 0x1
/* enum: Triggered */
#define          MC_CMD_INIT_EVQ_V2_IN_TMR_TRIG_START 0x2
/* enum: Hold-off */
#define          MC_CMD_INIT_EVQ_V2_IN_TMR_INT_HLDOFF 0x3
/* Target EVQ for wakeups if in wakeup mode. */
#define       MC_CMD_INIT_EVQ_V2_IN_TARGET_EVQ_OFST 24
#define       MC_CMD_INIT_EVQ_V2_IN_TARGET_EVQ_LEN 4
/* Target interrupt if in interrupting mode (note union with target EVQ). Use
 * MC_CMD_RESOURCE_INSTANCE_ANY unless a specific one required for test
 * purposes.
 */
#define       MC_CMD_INIT_EVQ_V2_IN_IRQ_NUM_OFST 24
#define       MC_CMD_INIT_EVQ_V2_IN_IRQ_NUM_LEN 4
/* Event Counter Mode. */
#define       MC_CMD_INIT_EVQ_V2_IN_COUNT_MODE_OFST 28
#define       MC_CMD_INIT_EVQ_V2_IN_COUNT_MODE_LEN 4
/* enum: Disabled */
#define          MC_CMD_INIT_EVQ_V2_IN_COUNT_MODE_DIS 0x0
/* enum: Disabled */
#define          MC_CMD_INIT_EVQ_V2_IN_COUNT_MODE_RX 0x1
/* enum: Disabled */
#define          MC_CMD_INIT_EVQ_V2_IN_COUNT_MODE_TX 0x2
/* enum: Disabled */
#define          MC_CMD_INIT_EVQ_V2_IN_COUNT_MODE_RXTX 0x3
/* Event queue packet count threshold. */
#define       MC_CMD_INIT_EVQ_V2_IN_COUNT_THRSHLD_OFST 32
#define       MC_CMD_INIT_EVQ_V2_IN_COUNT_THRSHLD_LEN 4
/* 64-bit address of 4k of 4k-aligned host memory buffer */
#define       MC_CMD_INIT_EVQ_V2_IN_DMA_ADDR_OFST 36
#define       MC_CMD_INIT_EVQ_V2_IN_DMA_ADDR_LEN 8
#define       MC_CMD_INIT_EVQ_V2_IN_DMA_ADDR_LO_OFST 36
#define       MC_CMD_INIT_EVQ_V2_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_INIT_EVQ_V2_IN_DMA_ADDR_LO_LBN 288
#define       MC_CMD_INIT_EVQ_V2_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_INIT_EVQ_V2_IN_DMA_ADDR_HI_OFST 40
#define       MC_CMD_INIT_EVQ_V2_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_INIT_EVQ_V2_IN_DMA_ADDR_HI_LBN 320
#define       MC_CMD_INIT_EVQ_V2_IN_DMA_ADDR_HI_WIDTH 32
#define       MC_CMD_INIT_EVQ_V2_IN_DMA_ADDR_MINNUM 1
#define       MC_CMD_INIT_EVQ_V2_IN_DMA_ADDR_MAXNUM 64
#define       MC_CMD_INIT_EVQ_V2_IN_DMA_ADDR_MAXNUM_MCDI2 64

/* MC_CMD_INIT_EVQ_V2_OUT msgresponse */
#define    MC_CMD_INIT_EVQ_V2_OUT_LEN 8
/* Only valid if INTRFLAG was true */
#define       MC_CMD_INIT_EVQ_V2_OUT_IRQ_OFST 0
#define       MC_CMD_INIT_EVQ_V2_OUT_IRQ_LEN 4
/* Actual configuration applied on the card */
#define       MC_CMD_INIT_EVQ_V2_OUT_FLAGS_OFST 4
#define       MC_CMD_INIT_EVQ_V2_OUT_FLAGS_LEN 4
#define        MC_CMD_INIT_EVQ_V2_OUT_FLAG_CUT_THRU_OFST 4
#define        MC_CMD_INIT_EVQ_V2_OUT_FLAG_CUT_THRU_LBN 0
#define        MC_CMD_INIT_EVQ_V2_OUT_FLAG_CUT_THRU_WIDTH 1
#define        MC_CMD_INIT_EVQ_V2_OUT_FLAG_RX_MERGE_OFST 4
#define        MC_CMD_INIT_EVQ_V2_OUT_FLAG_RX_MERGE_LBN 1
#define        MC_CMD_INIT_EVQ_V2_OUT_FLAG_RX_MERGE_WIDTH 1
#define        MC_CMD_INIT_EVQ_V2_OUT_FLAG_TX_MERGE_OFST 4
#define        MC_CMD_INIT_EVQ_V2_OUT_FLAG_TX_MERGE_LBN 2
#define        MC_CMD_INIT_EVQ_V2_OUT_FLAG_TX_MERGE_WIDTH 1
#define        MC_CMD_INIT_EVQ_V2_OUT_FLAG_RXQ_FORCE_EV_MERGING_OFST 4
#define        MC_CMD_INIT_EVQ_V2_OUT_FLAG_RXQ_FORCE_EV_MERGING_LBN 3
#define        MC_CMD_INIT_EVQ_V2_OUT_FLAG_RXQ_FORCE_EV_MERGING_WIDTH 1

/* MC_CMD_INIT_EVQ_V3_IN msgrequest: Extended request to specify per-queue
 * event merge timeouts.
 */
#define    MC_CMD_INIT_EVQ_V3_IN_LEN 556
/* Size, in entries */
#define       MC_CMD_INIT_EVQ_V3_IN_SIZE_OFST 0
#define       MC_CMD_INIT_EVQ_V3_IN_SIZE_LEN 4
/* Desired instance. Must be set to a specific instance, which is a function
 * local queue index. The calling client must be the currently-assigned user of
 * this VI (see MC_CMD_SET_VI_USER).
 */
#define       MC_CMD_INIT_EVQ_V3_IN_INSTANCE_OFST 4
#define       MC_CMD_INIT_EVQ_V3_IN_INSTANCE_LEN 4
/* The initial timer value. The load value is ignored if the timer mode is DIS.
 */
#define       MC_CMD_INIT_EVQ_V3_IN_TMR_LOAD_OFST 8
#define       MC_CMD_INIT_EVQ_V3_IN_TMR_LOAD_LEN 4
/* The reload value is ignored in one-shot modes */
#define       MC_CMD_INIT_EVQ_V3_IN_TMR_RELOAD_OFST 12
#define       MC_CMD_INIT_EVQ_V3_IN_TMR_RELOAD_LEN 4
/* tbd */
#define       MC_CMD_INIT_EVQ_V3_IN_FLAGS_OFST 16
#define       MC_CMD_INIT_EVQ_V3_IN_FLAGS_LEN 4
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_INTERRUPTING_OFST 16
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_INTERRUPTING_LBN 0
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_INTERRUPTING_WIDTH 1
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_RPTR_DOS_OFST 16
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_RPTR_DOS_LBN 1
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_RPTR_DOS_WIDTH 1
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_INT_ARMD_OFST 16
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_INT_ARMD_LBN 2
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_INT_ARMD_WIDTH 1
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_CUT_THRU_OFST 16
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_CUT_THRU_LBN 3
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_CUT_THRU_WIDTH 1
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_RX_MERGE_OFST 16
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_RX_MERGE_LBN 4
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_RX_MERGE_WIDTH 1
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_TX_MERGE_OFST 16
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_TX_MERGE_LBN 5
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_TX_MERGE_WIDTH 1
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_USE_TIMER_OFST 16
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_USE_TIMER_LBN 6
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_USE_TIMER_WIDTH 1
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_TYPE_OFST 16
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_TYPE_LBN 7
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_TYPE_WIDTH 4
/* enum: All initialisation flags specified by host. */
#define          MC_CMD_INIT_EVQ_V3_IN_FLAG_TYPE_MANUAL 0x0
/* enum: MEDFORD only. Certain initialisation flags specified by host may be
 * over-ridden by firmware based on licenses and firmware variant in order to
 * provide the lowest latency achievable. See
 * MC_CMD_INIT_EVQ_V2/MC_CMD_INIT_EVQ_V2_OUT/FLAGS for list of affected flags.
 */
#define          MC_CMD_INIT_EVQ_V3_IN_FLAG_TYPE_LOW_LATENCY 0x1
/* enum: MEDFORD only. Certain initialisation flags specified by host may be
 * over-ridden by firmware based on licenses and firmware variant in order to
 * provide the best throughput achievable. See
 * MC_CMD_INIT_EVQ_V2/MC_CMD_INIT_EVQ_V2_OUT/FLAGS for list of affected flags.
 */
#define          MC_CMD_INIT_EVQ_V3_IN_FLAG_TYPE_THROUGHPUT 0x2
/* enum: MEDFORD only. Certain initialisation flags may be over-ridden by
 * firmware based on licenses and firmware variant. See
 * MC_CMD_INIT_EVQ_V2/MC_CMD_INIT_EVQ_V2_OUT/FLAGS for list of affected flags.
 */
#define          MC_CMD_INIT_EVQ_V3_IN_FLAG_TYPE_AUTO 0x3
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_EXT_WIDTH_OFST 16
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_EXT_WIDTH_LBN 11
#define        MC_CMD_INIT_EVQ_V3_IN_FLAG_EXT_WIDTH_WIDTH 1
#define       MC_CMD_INIT_EVQ_V3_IN_TMR_MODE_OFST 20
#define       MC_CMD_INIT_EVQ_V3_IN_TMR_MODE_LEN 4
/* enum: Disabled */
#define          MC_CMD_INIT_EVQ_V3_IN_TMR_MODE_DIS 0x0
/* enum: Immediate */
#define          MC_CMD_INIT_EVQ_V3_IN_TMR_IMMED_START 0x1
/* enum: Triggered */
#define          MC_CMD_INIT_EVQ_V3_IN_TMR_TRIG_START 0x2
/* enum: Hold-off */
#define          MC_CMD_INIT_EVQ_V3_IN_TMR_INT_HLDOFF 0x3
/* Target EVQ for wakeups if in wakeup mode. */
#define       MC_CMD_INIT_EVQ_V3_IN_TARGET_EVQ_OFST 24
#define       MC_CMD_INIT_EVQ_V3_IN_TARGET_EVQ_LEN 4
/* Target interrupt if in interrupting mode (note union with target EVQ). Use
 * MC_CMD_RESOURCE_INSTANCE_ANY unless a specific one required for test
 * purposes.
 */
#define       MC_CMD_INIT_EVQ_V3_IN_IRQ_NUM_OFST 24
#define       MC_CMD_INIT_EVQ_V3_IN_IRQ_NUM_LEN 4
/* Event Counter Mode. */
#define       MC_CMD_INIT_EVQ_V3_IN_COUNT_MODE_OFST 28
#define       MC_CMD_INIT_EVQ_V3_IN_COUNT_MODE_LEN 4
/* enum: Disabled */
#define          MC_CMD_INIT_EVQ_V3_IN_COUNT_MODE_DIS 0x0
/* enum: Disabled */
#define          MC_CMD_INIT_EVQ_V3_IN_COUNT_MODE_RX 0x1
/* enum: Disabled */
#define          MC_CMD_INIT_EVQ_V3_IN_COUNT_MODE_TX 0x2
/* enum: Disabled */
#define          MC_CMD_INIT_EVQ_V3_IN_COUNT_MODE_RXTX 0x3
/* Event queue packet count threshold. */
#define       MC_CMD_INIT_EVQ_V3_IN_COUNT_THRSHLD_OFST 32
#define       MC_CMD_INIT_EVQ_V3_IN_COUNT_THRSHLD_LEN 4
/* 64-bit address of 4k of 4k-aligned host memory buffer */
#define       MC_CMD_INIT_EVQ_V3_IN_DMA_ADDR_OFST 36
#define       MC_CMD_INIT_EVQ_V3_IN_DMA_ADDR_LEN 8
#define       MC_CMD_INIT_EVQ_V3_IN_DMA_ADDR_LO_OFST 36
#define       MC_CMD_INIT_EVQ_V3_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_INIT_EVQ_V3_IN_DMA_ADDR_LO_LBN 288
#define       MC_CMD_INIT_EVQ_V3_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_INIT_EVQ_V3_IN_DMA_ADDR_HI_OFST 40
#define       MC_CMD_INIT_EVQ_V3_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_INIT_EVQ_V3_IN_DMA_ADDR_HI_LBN 320
#define       MC_CMD_INIT_EVQ_V3_IN_DMA_ADDR_HI_WIDTH 32
#define       MC_CMD_INIT_EVQ_V3_IN_DMA_ADDR_MINNUM 1
#define       MC_CMD_INIT_EVQ_V3_IN_DMA_ADDR_MAXNUM 64
#define       MC_CMD_INIT_EVQ_V3_IN_DMA_ADDR_MAXNUM_MCDI2 64
/* Receive event merge timeout to configure, in nanoseconds. The valid range
 * and granularity are device specific. Specify 0 to use the firmware's default
 * value. This field is ignored and per-queue merging is disabled if
 * MC_CMD_INIT_EVQ/MC_CMD_INIT_EVQ_IN/FLAG_RX_MERGE is not set.
 */
#define       MC_CMD_INIT_EVQ_V3_IN_RX_MERGE_TIMEOUT_NS_OFST 548
#define       MC_CMD_INIT_EVQ_V3_IN_RX_MERGE_TIMEOUT_NS_LEN 4
/* Transmit event merge timeout to configure, in nanoseconds. The valid range
 * and granularity are device specific. Specify 0 to use the firmware's default
 * value. This field is ignored and per-queue merging is disabled if
 * MC_CMD_INIT_EVQ/MC_CMD_INIT_EVQ_IN/FLAG_TX_MERGE is not set.
 */
#define       MC_CMD_INIT_EVQ_V3_IN_TX_MERGE_TIMEOUT_NS_OFST 552
#define       MC_CMD_INIT_EVQ_V3_IN_TX_MERGE_TIMEOUT_NS_LEN 4

/* MC_CMD_INIT_EVQ_V3_OUT msgresponse */
#define    MC_CMD_INIT_EVQ_V3_OUT_LEN 8
/* Only valid if INTRFLAG was true */
#define       MC_CMD_INIT_EVQ_V3_OUT_IRQ_OFST 0
#define       MC_CMD_INIT_EVQ_V3_OUT_IRQ_LEN 4
/* Actual configuration applied on the card */
#define       MC_CMD_INIT_EVQ_V3_OUT_FLAGS_OFST 4
#define       MC_CMD_INIT_EVQ_V3_OUT_FLAGS_LEN 4
#define        MC_CMD_INIT_EVQ_V3_OUT_FLAG_CUT_THRU_OFST 4
#define        MC_CMD_INIT_EVQ_V3_OUT_FLAG_CUT_THRU_LBN 0
#define        MC_CMD_INIT_EVQ_V3_OUT_FLAG_CUT_THRU_WIDTH 1
#define        MC_CMD_INIT_EVQ_V3_OUT_FLAG_RX_MERGE_OFST 4
#define        MC_CMD_INIT_EVQ_V3_OUT_FLAG_RX_MERGE_LBN 1
#define        MC_CMD_INIT_EVQ_V3_OUT_FLAG_RX_MERGE_WIDTH 1
#define        MC_CMD_INIT_EVQ_V3_OUT_FLAG_TX_MERGE_OFST 4
#define        MC_CMD_INIT_EVQ_V3_OUT_FLAG_TX_MERGE_LBN 2
#define        MC_CMD_INIT_EVQ_V3_OUT_FLAG_TX_MERGE_WIDTH 1
#define        MC_CMD_INIT_EVQ_V3_OUT_FLAG_RXQ_FORCE_EV_MERGING_OFST 4
#define        MC_CMD_INIT_EVQ_V3_OUT_FLAG_RXQ_FORCE_EV_MERGING_LBN 3
#define        MC_CMD_INIT_EVQ_V3_OUT_FLAG_RXQ_FORCE_EV_MERGING_WIDTH 1


/***********************************/
/* MC_CMD_INIT_RXQ
 * set up a receive queue according to the supplied parameters. The IN
 * arguments end with an address for each 4k of host memory required to back
 * the RXQ.
 */
#define MC_CMD_INIT_RXQ 0x81
#undef MC_CMD_0x81_PRIVILEGE_CTG

#define MC_CMD_0x81_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_INIT_RXQ_IN msgrequest: Legacy RXQ_INIT request. Use extended version
 * in new code.
 */
#define    MC_CMD_INIT_RXQ_IN_LENMIN 36
#define    MC_CMD_INIT_RXQ_IN_LENMAX 252
#define    MC_CMD_INIT_RXQ_IN_LENMAX_MCDI2 1020
#define    MC_CMD_INIT_RXQ_IN_LEN(num) (28+8*(num))
#define    MC_CMD_INIT_RXQ_IN_DMA_ADDR_NUM(len) (((len)-28)/8)
/* Size, in entries */
#define       MC_CMD_INIT_RXQ_IN_SIZE_OFST 0
#define       MC_CMD_INIT_RXQ_IN_SIZE_LEN 4
/* The EVQ to send events to. This is an index originally specified to INIT_EVQ
 */
#define       MC_CMD_INIT_RXQ_IN_TARGET_EVQ_OFST 4
#define       MC_CMD_INIT_RXQ_IN_TARGET_EVQ_LEN 4
/* The value to put in the event data. Check hardware spec. for valid range. */
#define       MC_CMD_INIT_RXQ_IN_LABEL_OFST 8
#define       MC_CMD_INIT_RXQ_IN_LABEL_LEN 4
/* Desired instance. Must be set to a specific instance, which is a function
 * local queue index. The calling client must be the currently-assigned user of
 * this VI (see MC_CMD_SET_VI_USER).
 */
#define       MC_CMD_INIT_RXQ_IN_INSTANCE_OFST 12
#define       MC_CMD_INIT_RXQ_IN_INSTANCE_LEN 4
/* There will be more flags here. */
#define       MC_CMD_INIT_RXQ_IN_FLAGS_OFST 16
#define       MC_CMD_INIT_RXQ_IN_FLAGS_LEN 4
#define        MC_CMD_INIT_RXQ_IN_FLAG_BUFF_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_IN_FLAG_BUFF_MODE_LBN 0
#define        MC_CMD_INIT_RXQ_IN_FLAG_BUFF_MODE_WIDTH 1
#define        MC_CMD_INIT_RXQ_IN_FLAG_HDR_SPLIT_OFST 16
#define        MC_CMD_INIT_RXQ_IN_FLAG_HDR_SPLIT_LBN 1
#define        MC_CMD_INIT_RXQ_IN_FLAG_HDR_SPLIT_WIDTH 1
#define        MC_CMD_INIT_RXQ_IN_FLAG_TIMESTAMP_OFST 16
#define        MC_CMD_INIT_RXQ_IN_FLAG_TIMESTAMP_LBN 2
#define        MC_CMD_INIT_RXQ_IN_FLAG_TIMESTAMP_WIDTH 1
#define        MC_CMD_INIT_RXQ_IN_CRC_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_IN_CRC_MODE_LBN 3
#define        MC_CMD_INIT_RXQ_IN_CRC_MODE_WIDTH 4
#define        MC_CMD_INIT_RXQ_IN_FLAG_CHAIN_OFST 16
#define        MC_CMD_INIT_RXQ_IN_FLAG_CHAIN_LBN 7
#define        MC_CMD_INIT_RXQ_IN_FLAG_CHAIN_WIDTH 1
#define        MC_CMD_INIT_RXQ_IN_FLAG_PREFIX_OFST 16
#define        MC_CMD_INIT_RXQ_IN_FLAG_PREFIX_LBN 8
#define        MC_CMD_INIT_RXQ_IN_FLAG_PREFIX_WIDTH 1
#define        MC_CMD_INIT_RXQ_IN_FLAG_DISABLE_SCATTER_OFST 16
#define        MC_CMD_INIT_RXQ_IN_FLAG_DISABLE_SCATTER_LBN 9
#define        MC_CMD_INIT_RXQ_IN_FLAG_DISABLE_SCATTER_WIDTH 1
#define        MC_CMD_INIT_RXQ_IN_UNUSED_OFST 16
#define        MC_CMD_INIT_RXQ_IN_UNUSED_LBN 10
#define        MC_CMD_INIT_RXQ_IN_UNUSED_WIDTH 1
/* Owner ID to use if in buffer mode (zero if physical) */
#define       MC_CMD_INIT_RXQ_IN_OWNER_ID_OFST 20
#define       MC_CMD_INIT_RXQ_IN_OWNER_ID_LEN 4
/* The port ID associated with the v-adaptor which should contain this DMAQ. */
#define       MC_CMD_INIT_RXQ_IN_PORT_ID_OFST 24
#define       MC_CMD_INIT_RXQ_IN_PORT_ID_LEN 4
/* 64-bit address of 4k of 4k-aligned host memory buffer */
#define       MC_CMD_INIT_RXQ_IN_DMA_ADDR_OFST 28
#define       MC_CMD_INIT_RXQ_IN_DMA_ADDR_LEN 8
#define       MC_CMD_INIT_RXQ_IN_DMA_ADDR_LO_OFST 28
#define       MC_CMD_INIT_RXQ_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_INIT_RXQ_IN_DMA_ADDR_LO_LBN 224
#define       MC_CMD_INIT_RXQ_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_INIT_RXQ_IN_DMA_ADDR_HI_OFST 32
#define       MC_CMD_INIT_RXQ_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_INIT_RXQ_IN_DMA_ADDR_HI_LBN 256
#define       MC_CMD_INIT_RXQ_IN_DMA_ADDR_HI_WIDTH 32
#define       MC_CMD_INIT_RXQ_IN_DMA_ADDR_MINNUM 1
#define       MC_CMD_INIT_RXQ_IN_DMA_ADDR_MAXNUM 28
#define       MC_CMD_INIT_RXQ_IN_DMA_ADDR_MAXNUM_MCDI2 124

/* MC_CMD_INIT_RXQ_EXT_IN msgrequest: Extended RXQ_INIT with additional mode
 * flags
 */
#define    MC_CMD_INIT_RXQ_EXT_IN_LEN 544
/* Size, in entries */
#define       MC_CMD_INIT_RXQ_EXT_IN_SIZE_OFST 0
#define       MC_CMD_INIT_RXQ_EXT_IN_SIZE_LEN 4
/* The EVQ to send events to. This is an index originally specified to
 * INIT_EVQ. If DMA_MODE == PACKED_STREAM this must be equal to INSTANCE.
 */
#define       MC_CMD_INIT_RXQ_EXT_IN_TARGET_EVQ_OFST 4
#define       MC_CMD_INIT_RXQ_EXT_IN_TARGET_EVQ_LEN 4
/* The value to put in the event data. Check hardware spec. for valid range.
 * This field is ignored if DMA_MODE == EQUAL_STRIDE_SUPER_BUFFER or DMA_MODE
 * == PACKED_STREAM.
 */
#define       MC_CMD_INIT_RXQ_EXT_IN_LABEL_OFST 8
#define       MC_CMD_INIT_RXQ_EXT_IN_LABEL_LEN 4
/* Desired instance. Must be set to a specific instance, which is a function
 * local queue index. The calling client must be the currently-assigned user of
 * this VI (see MC_CMD_SET_VI_USER).
 */
#define       MC_CMD_INIT_RXQ_EXT_IN_INSTANCE_OFST 12
#define       MC_CMD_INIT_RXQ_EXT_IN_INSTANCE_LEN 4
/* There will be more flags here. */
#define       MC_CMD_INIT_RXQ_EXT_IN_FLAGS_OFST 16
#define       MC_CMD_INIT_RXQ_EXT_IN_FLAGS_LEN 4
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_BUFF_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_BUFF_MODE_LBN 0
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_BUFF_MODE_WIDTH 1
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_HDR_SPLIT_OFST 16
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_HDR_SPLIT_LBN 1
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_HDR_SPLIT_WIDTH 1
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_TIMESTAMP_OFST 16
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_TIMESTAMP_LBN 2
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_TIMESTAMP_WIDTH 1
#define        MC_CMD_INIT_RXQ_EXT_IN_CRC_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_EXT_IN_CRC_MODE_LBN 3
#define        MC_CMD_INIT_RXQ_EXT_IN_CRC_MODE_WIDTH 4
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_CHAIN_OFST 16
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_CHAIN_LBN 7
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_CHAIN_WIDTH 1
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_PREFIX_OFST 16
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_PREFIX_LBN 8
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_PREFIX_WIDTH 1
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_DISABLE_SCATTER_OFST 16
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_DISABLE_SCATTER_LBN 9
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_DISABLE_SCATTER_WIDTH 1
#define        MC_CMD_INIT_RXQ_EXT_IN_DMA_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_EXT_IN_DMA_MODE_LBN 10
#define        MC_CMD_INIT_RXQ_EXT_IN_DMA_MODE_WIDTH 4
/* enum: One packet per descriptor (for normal networking) */
#define          MC_CMD_INIT_RXQ_EXT_IN_SINGLE_PACKET 0x0
/* enum: Pack multiple packets into large descriptors (for SolarCapture) */
#define          MC_CMD_INIT_RXQ_EXT_IN_PACKED_STREAM 0x1
/* enum: Pack multiple packets into large descriptors using the format designed
 * to maximise packet rate. This mode uses 1 "bucket" per descriptor with
 * multiple fixed-size packet buffers within each bucket. For a full
 * description see SF-119419-TC. This mode is only supported by "dpdk" datapath
 * firmware.
 */
#define          MC_CMD_INIT_RXQ_EXT_IN_EQUAL_STRIDE_SUPER_BUFFER 0x2
/* enum: Deprecated name for EQUAL_STRIDE_SUPER_BUFFER. */
#define          MC_CMD_INIT_RXQ_EXT_IN_EQUAL_STRIDE_PACKED_STREAM 0x2
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_SNAPSHOT_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_SNAPSHOT_MODE_LBN 14
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_SNAPSHOT_MODE_WIDTH 1
#define        MC_CMD_INIT_RXQ_EXT_IN_PACKED_STREAM_BUFF_SIZE_OFST 16
#define        MC_CMD_INIT_RXQ_EXT_IN_PACKED_STREAM_BUFF_SIZE_LBN 15
#define        MC_CMD_INIT_RXQ_EXT_IN_PACKED_STREAM_BUFF_SIZE_WIDTH 3
#define          MC_CMD_INIT_RXQ_EXT_IN_PS_BUFF_1M 0x0 /* enum */
#define          MC_CMD_INIT_RXQ_EXT_IN_PS_BUFF_512K 0x1 /* enum */
#define          MC_CMD_INIT_RXQ_EXT_IN_PS_BUFF_256K 0x2 /* enum */
#define          MC_CMD_INIT_RXQ_EXT_IN_PS_BUFF_128K 0x3 /* enum */
#define          MC_CMD_INIT_RXQ_EXT_IN_PS_BUFF_64K 0x4 /* enum */
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_WANT_OUTER_CLASSES_OFST 16
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_WANT_OUTER_CLASSES_LBN 18
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_WANT_OUTER_CLASSES_WIDTH 1
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_FORCE_EV_MERGING_OFST 16
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_FORCE_EV_MERGING_LBN 19
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_FORCE_EV_MERGING_WIDTH 1
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_NO_CONT_EV_OFST 16
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_NO_CONT_EV_LBN 20
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_NO_CONT_EV_WIDTH 1
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_SUPPRESS_RX_EVENTS_OFST 16
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_SUPPRESS_RX_EVENTS_LBN 21
#define        MC_CMD_INIT_RXQ_EXT_IN_FLAG_SUPPRESS_RX_EVENTS_WIDTH 1
/* Owner ID to use if in buffer mode (zero if physical) */
#define       MC_CMD_INIT_RXQ_EXT_IN_OWNER_ID_OFST 20
#define       MC_CMD_INIT_RXQ_EXT_IN_OWNER_ID_LEN 4
/* The port ID associated with the v-adaptor which should contain this DMAQ. */
#define       MC_CMD_INIT_RXQ_EXT_IN_PORT_ID_OFST 24
#define       MC_CMD_INIT_RXQ_EXT_IN_PORT_ID_LEN 4
/* 64-bit address of 4k of 4k-aligned host memory buffer */
#define       MC_CMD_INIT_RXQ_EXT_IN_DMA_ADDR_OFST 28
#define       MC_CMD_INIT_RXQ_EXT_IN_DMA_ADDR_LEN 8
#define       MC_CMD_INIT_RXQ_EXT_IN_DMA_ADDR_LO_OFST 28
#define       MC_CMD_INIT_RXQ_EXT_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_INIT_RXQ_EXT_IN_DMA_ADDR_LO_LBN 224
#define       MC_CMD_INIT_RXQ_EXT_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_INIT_RXQ_EXT_IN_DMA_ADDR_HI_OFST 32
#define       MC_CMD_INIT_RXQ_EXT_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_INIT_RXQ_EXT_IN_DMA_ADDR_HI_LBN 256
#define       MC_CMD_INIT_RXQ_EXT_IN_DMA_ADDR_HI_WIDTH 32
#define       MC_CMD_INIT_RXQ_EXT_IN_DMA_ADDR_MINNUM 0
#define       MC_CMD_INIT_RXQ_EXT_IN_DMA_ADDR_MAXNUM 64
#define       MC_CMD_INIT_RXQ_EXT_IN_DMA_ADDR_MAXNUM_MCDI2 64
/* Maximum length of packet to receive, if SNAPSHOT_MODE flag is set */
#define       MC_CMD_INIT_RXQ_EXT_IN_SNAPSHOT_LENGTH_OFST 540
#define       MC_CMD_INIT_RXQ_EXT_IN_SNAPSHOT_LENGTH_LEN 4

/* MC_CMD_INIT_RXQ_V3_IN msgrequest */
#define    MC_CMD_INIT_RXQ_V3_IN_LEN 560
/* Size, in entries */
#define       MC_CMD_INIT_RXQ_V3_IN_SIZE_OFST 0
#define       MC_CMD_INIT_RXQ_V3_IN_SIZE_LEN 4
/* The EVQ to send events to. This is an index originally specified to
 * INIT_EVQ. If DMA_MODE == PACKED_STREAM this must be equal to INSTANCE.
 */
#define       MC_CMD_INIT_RXQ_V3_IN_TARGET_EVQ_OFST 4
#define       MC_CMD_INIT_RXQ_V3_IN_TARGET_EVQ_LEN 4
/* The value to put in the event data. Check hardware spec. for valid range.
 * This field is ignored if DMA_MODE == EQUAL_STRIDE_SUPER_BUFFER or DMA_MODE
 * == PACKED_STREAM.
 */
#define       MC_CMD_INIT_RXQ_V3_IN_LABEL_OFST 8
#define       MC_CMD_INIT_RXQ_V3_IN_LABEL_LEN 4
/* Desired instance. Must be set to a specific instance, which is a function
 * local queue index. The calling client must be the currently-assigned user of
 * this VI (see MC_CMD_SET_VI_USER).
 */
#define       MC_CMD_INIT_RXQ_V3_IN_INSTANCE_OFST 12
#define       MC_CMD_INIT_RXQ_V3_IN_INSTANCE_LEN 4
/* There will be more flags here. */
#define       MC_CMD_INIT_RXQ_V3_IN_FLAGS_OFST 16
#define       MC_CMD_INIT_RXQ_V3_IN_FLAGS_LEN 4
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_BUFF_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_BUFF_MODE_LBN 0
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_BUFF_MODE_WIDTH 1
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_HDR_SPLIT_OFST 16
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_HDR_SPLIT_LBN 1
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_HDR_SPLIT_WIDTH 1
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_TIMESTAMP_OFST 16
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_TIMESTAMP_LBN 2
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_TIMESTAMP_WIDTH 1
#define        MC_CMD_INIT_RXQ_V3_IN_CRC_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_V3_IN_CRC_MODE_LBN 3
#define        MC_CMD_INIT_RXQ_V3_IN_CRC_MODE_WIDTH 4
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_CHAIN_OFST 16
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_CHAIN_LBN 7
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_CHAIN_WIDTH 1
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_PREFIX_OFST 16
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_PREFIX_LBN 8
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_PREFIX_WIDTH 1
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_DISABLE_SCATTER_OFST 16
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_DISABLE_SCATTER_LBN 9
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_DISABLE_SCATTER_WIDTH 1
#define        MC_CMD_INIT_RXQ_V3_IN_DMA_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_V3_IN_DMA_MODE_LBN 10
#define        MC_CMD_INIT_RXQ_V3_IN_DMA_MODE_WIDTH 4
/* enum: One packet per descriptor (for normal networking) */
#define          MC_CMD_INIT_RXQ_V3_IN_SINGLE_PACKET 0x0
/* enum: Pack multiple packets into large descriptors (for SolarCapture) */
#define          MC_CMD_INIT_RXQ_V3_IN_PACKED_STREAM 0x1
/* enum: Pack multiple packets into large descriptors using the format designed
 * to maximise packet rate. This mode uses 1 "bucket" per descriptor with
 * multiple fixed-size packet buffers within each bucket. For a full
 * description see SF-119419-TC. This mode is only supported by "dpdk" datapath
 * firmware.
 */
#define          MC_CMD_INIT_RXQ_V3_IN_EQUAL_STRIDE_SUPER_BUFFER 0x2
/* enum: Deprecated name for EQUAL_STRIDE_SUPER_BUFFER. */
#define          MC_CMD_INIT_RXQ_V3_IN_EQUAL_STRIDE_PACKED_STREAM 0x2
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_SNAPSHOT_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_SNAPSHOT_MODE_LBN 14
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_SNAPSHOT_MODE_WIDTH 1
#define        MC_CMD_INIT_RXQ_V3_IN_PACKED_STREAM_BUFF_SIZE_OFST 16
#define        MC_CMD_INIT_RXQ_V3_IN_PACKED_STREAM_BUFF_SIZE_LBN 15
#define        MC_CMD_INIT_RXQ_V3_IN_PACKED_STREAM_BUFF_SIZE_WIDTH 3
#define          MC_CMD_INIT_RXQ_V3_IN_PS_BUFF_1M 0x0 /* enum */
#define          MC_CMD_INIT_RXQ_V3_IN_PS_BUFF_512K 0x1 /* enum */
#define          MC_CMD_INIT_RXQ_V3_IN_PS_BUFF_256K 0x2 /* enum */
#define          MC_CMD_INIT_RXQ_V3_IN_PS_BUFF_128K 0x3 /* enum */
#define          MC_CMD_INIT_RXQ_V3_IN_PS_BUFF_64K 0x4 /* enum */
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_WANT_OUTER_CLASSES_OFST 16
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_WANT_OUTER_CLASSES_LBN 18
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_WANT_OUTER_CLASSES_WIDTH 1
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_FORCE_EV_MERGING_OFST 16
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_FORCE_EV_MERGING_LBN 19
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_FORCE_EV_MERGING_WIDTH 1
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_NO_CONT_EV_OFST 16
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_NO_CONT_EV_LBN 20
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_NO_CONT_EV_WIDTH 1
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_SUPPRESS_RX_EVENTS_OFST 16
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_SUPPRESS_RX_EVENTS_LBN 21
#define        MC_CMD_INIT_RXQ_V3_IN_FLAG_SUPPRESS_RX_EVENTS_WIDTH 1
/* Owner ID to use if in buffer mode (zero if physical) */
#define       MC_CMD_INIT_RXQ_V3_IN_OWNER_ID_OFST 20
#define       MC_CMD_INIT_RXQ_V3_IN_OWNER_ID_LEN 4
/* The port ID associated with the v-adaptor which should contain this DMAQ. */
#define       MC_CMD_INIT_RXQ_V3_IN_PORT_ID_OFST 24
#define       MC_CMD_INIT_RXQ_V3_IN_PORT_ID_LEN 4
/* 64-bit address of 4k of 4k-aligned host memory buffer */
#define       MC_CMD_INIT_RXQ_V3_IN_DMA_ADDR_OFST 28
#define       MC_CMD_INIT_RXQ_V3_IN_DMA_ADDR_LEN 8
#define       MC_CMD_INIT_RXQ_V3_IN_DMA_ADDR_LO_OFST 28
#define       MC_CMD_INIT_RXQ_V3_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_INIT_RXQ_V3_IN_DMA_ADDR_LO_LBN 224
#define       MC_CMD_INIT_RXQ_V3_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_INIT_RXQ_V3_IN_DMA_ADDR_HI_OFST 32
#define       MC_CMD_INIT_RXQ_V3_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_INIT_RXQ_V3_IN_DMA_ADDR_HI_LBN 256
#define       MC_CMD_INIT_RXQ_V3_IN_DMA_ADDR_HI_WIDTH 32
#define       MC_CMD_INIT_RXQ_V3_IN_DMA_ADDR_MINNUM 0
#define       MC_CMD_INIT_RXQ_V3_IN_DMA_ADDR_MAXNUM 64
#define       MC_CMD_INIT_RXQ_V3_IN_DMA_ADDR_MAXNUM_MCDI2 64
/* Maximum length of packet to receive, if SNAPSHOT_MODE flag is set */
#define       MC_CMD_INIT_RXQ_V3_IN_SNAPSHOT_LENGTH_OFST 540
#define       MC_CMD_INIT_RXQ_V3_IN_SNAPSHOT_LENGTH_LEN 4
/* The number of packet buffers that will be contained within each
 * EQUAL_STRIDE_SUPER_BUFFER format bucket supplied by the driver. This field
 * is ignored unless DMA_MODE == EQUAL_STRIDE_SUPER_BUFFER.
 */
#define       MC_CMD_INIT_RXQ_V3_IN_ES_PACKET_BUFFERS_PER_BUCKET_OFST 544
#define       MC_CMD_INIT_RXQ_V3_IN_ES_PACKET_BUFFERS_PER_BUCKET_LEN 4
/* The length in bytes of the area in each packet buffer that can be written to
 * by the adapter. This is used to store the packet prefix and the packet
 * payload. This length does not include any end padding added by the driver.
 * This field is ignored unless DMA_MODE == EQUAL_STRIDE_SUPER_BUFFER.
 */
#define       MC_CMD_INIT_RXQ_V3_IN_ES_MAX_DMA_LEN_OFST 548
#define       MC_CMD_INIT_RXQ_V3_IN_ES_MAX_DMA_LEN_LEN 4
/* The length in bytes of a single packet buffer within a
 * EQUAL_STRIDE_SUPER_BUFFER format bucket. This field is ignored unless
 * DMA_MODE == EQUAL_STRIDE_SUPER_BUFFER.
 */
#define       MC_CMD_INIT_RXQ_V3_IN_ES_PACKET_STRIDE_OFST 552
#define       MC_CMD_INIT_RXQ_V3_IN_ES_PACKET_STRIDE_LEN 4
/* The maximum time in nanoseconds that the datapath will be backpressured if
 * there are no RX descriptors available. If the timeout is reached and there
 * are still no descriptors then the packet will be dropped. A timeout of 0
 * means the datapath will never be blocked. This field is ignored unless
 * DMA_MODE == EQUAL_STRIDE_SUPER_BUFFER.
 */
#define       MC_CMD_INIT_RXQ_V3_IN_ES_HEAD_OF_LINE_BLOCK_TIMEOUT_OFST 556
#define       MC_CMD_INIT_RXQ_V3_IN_ES_HEAD_OF_LINE_BLOCK_TIMEOUT_LEN 4

/* MC_CMD_INIT_RXQ_V4_IN msgrequest: INIT_RXQ request with new field required
 * for systems with a QDMA (currently, Riverhead)
 */
#define    MC_CMD_INIT_RXQ_V4_IN_LEN 564
/* Size, in entries */
#define       MC_CMD_INIT_RXQ_V4_IN_SIZE_OFST 0
#define       MC_CMD_INIT_RXQ_V4_IN_SIZE_LEN 4
/* The EVQ to send events to. This is an index originally specified to
 * INIT_EVQ. If DMA_MODE == PACKED_STREAM this must be equal to INSTANCE.
 */
#define       MC_CMD_INIT_RXQ_V4_IN_TARGET_EVQ_OFST 4
#define       MC_CMD_INIT_RXQ_V4_IN_TARGET_EVQ_LEN 4
/* The value to put in the event data. Check hardware spec. for valid range.
 * This field is ignored if DMA_MODE == EQUAL_STRIDE_SUPER_BUFFER or DMA_MODE
 * == PACKED_STREAM.
 */
#define       MC_CMD_INIT_RXQ_V4_IN_LABEL_OFST 8
#define       MC_CMD_INIT_RXQ_V4_IN_LABEL_LEN 4
/* Desired instance. Must be set to a specific instance, which is a function
 * local queue index. The calling client must be the currently-assigned user of
 * this VI (see MC_CMD_SET_VI_USER).
 */
#define       MC_CMD_INIT_RXQ_V4_IN_INSTANCE_OFST 12
#define       MC_CMD_INIT_RXQ_V4_IN_INSTANCE_LEN 4
/* There will be more flags here. */
#define       MC_CMD_INIT_RXQ_V4_IN_FLAGS_OFST 16
#define       MC_CMD_INIT_RXQ_V4_IN_FLAGS_LEN 4
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_BUFF_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_BUFF_MODE_LBN 0
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_BUFF_MODE_WIDTH 1
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_HDR_SPLIT_OFST 16
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_HDR_SPLIT_LBN 1
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_HDR_SPLIT_WIDTH 1
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_TIMESTAMP_OFST 16
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_TIMESTAMP_LBN 2
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_TIMESTAMP_WIDTH 1
#define        MC_CMD_INIT_RXQ_V4_IN_CRC_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_V4_IN_CRC_MODE_LBN 3
#define        MC_CMD_INIT_RXQ_V4_IN_CRC_MODE_WIDTH 4
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_CHAIN_OFST 16
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_CHAIN_LBN 7
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_CHAIN_WIDTH 1
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_PREFIX_OFST 16
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_PREFIX_LBN 8
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_PREFIX_WIDTH 1
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_DISABLE_SCATTER_OFST 16
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_DISABLE_SCATTER_LBN 9
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_DISABLE_SCATTER_WIDTH 1
#define        MC_CMD_INIT_RXQ_V4_IN_DMA_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_V4_IN_DMA_MODE_LBN 10
#define        MC_CMD_INIT_RXQ_V4_IN_DMA_MODE_WIDTH 4
/* enum: One packet per descriptor (for normal networking) */
#define          MC_CMD_INIT_RXQ_V4_IN_SINGLE_PACKET 0x0
/* enum: Pack multiple packets into large descriptors (for SolarCapture) */
#define          MC_CMD_INIT_RXQ_V4_IN_PACKED_STREAM 0x1
/* enum: Pack multiple packets into large descriptors using the format designed
 * to maximise packet rate. This mode uses 1 "bucket" per descriptor with
 * multiple fixed-size packet buffers within each bucket. For a full
 * description see SF-119419-TC. This mode is only supported by "dpdk" datapath
 * firmware.
 */
#define          MC_CMD_INIT_RXQ_V4_IN_EQUAL_STRIDE_SUPER_BUFFER 0x2
/* enum: Deprecated name for EQUAL_STRIDE_SUPER_BUFFER. */
#define          MC_CMD_INIT_RXQ_V4_IN_EQUAL_STRIDE_PACKED_STREAM 0x2
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_SNAPSHOT_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_SNAPSHOT_MODE_LBN 14
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_SNAPSHOT_MODE_WIDTH 1
#define        MC_CMD_INIT_RXQ_V4_IN_PACKED_STREAM_BUFF_SIZE_OFST 16
#define        MC_CMD_INIT_RXQ_V4_IN_PACKED_STREAM_BUFF_SIZE_LBN 15
#define        MC_CMD_INIT_RXQ_V4_IN_PACKED_STREAM_BUFF_SIZE_WIDTH 3
#define          MC_CMD_INIT_RXQ_V4_IN_PS_BUFF_1M 0x0 /* enum */
#define          MC_CMD_INIT_RXQ_V4_IN_PS_BUFF_512K 0x1 /* enum */
#define          MC_CMD_INIT_RXQ_V4_IN_PS_BUFF_256K 0x2 /* enum */
#define          MC_CMD_INIT_RXQ_V4_IN_PS_BUFF_128K 0x3 /* enum */
#define          MC_CMD_INIT_RXQ_V4_IN_PS_BUFF_64K 0x4 /* enum */
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_WANT_OUTER_CLASSES_OFST 16
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_WANT_OUTER_CLASSES_LBN 18
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_WANT_OUTER_CLASSES_WIDTH 1
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_FORCE_EV_MERGING_OFST 16
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_FORCE_EV_MERGING_LBN 19
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_FORCE_EV_MERGING_WIDTH 1
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_NO_CONT_EV_OFST 16
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_NO_CONT_EV_LBN 20
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_NO_CONT_EV_WIDTH 1
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_SUPPRESS_RX_EVENTS_OFST 16
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_SUPPRESS_RX_EVENTS_LBN 21
#define        MC_CMD_INIT_RXQ_V4_IN_FLAG_SUPPRESS_RX_EVENTS_WIDTH 1
/* Owner ID to use if in buffer mode (zero if physical) */
#define       MC_CMD_INIT_RXQ_V4_IN_OWNER_ID_OFST 20
#define       MC_CMD_INIT_RXQ_V4_IN_OWNER_ID_LEN 4
/* The port ID associated with the v-adaptor which should contain this DMAQ. */
#define       MC_CMD_INIT_RXQ_V4_IN_PORT_ID_OFST 24
#define       MC_CMD_INIT_RXQ_V4_IN_PORT_ID_LEN 4
/* 64-bit address of 4k of 4k-aligned host memory buffer */
#define       MC_CMD_INIT_RXQ_V4_IN_DMA_ADDR_OFST 28
#define       MC_CMD_INIT_RXQ_V4_IN_DMA_ADDR_LEN 8
#define       MC_CMD_INIT_RXQ_V4_IN_DMA_ADDR_LO_OFST 28
#define       MC_CMD_INIT_RXQ_V4_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_INIT_RXQ_V4_IN_DMA_ADDR_LO_LBN 224
#define       MC_CMD_INIT_RXQ_V4_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_INIT_RXQ_V4_IN_DMA_ADDR_HI_OFST 32
#define       MC_CMD_INIT_RXQ_V4_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_INIT_RXQ_V4_IN_DMA_ADDR_HI_LBN 256
#define       MC_CMD_INIT_RXQ_V4_IN_DMA_ADDR_HI_WIDTH 32
#define       MC_CMD_INIT_RXQ_V4_IN_DMA_ADDR_MINNUM 0
#define       MC_CMD_INIT_RXQ_V4_IN_DMA_ADDR_MAXNUM 64
#define       MC_CMD_INIT_RXQ_V4_IN_DMA_ADDR_MAXNUM_MCDI2 64
/* Maximum length of packet to receive, if SNAPSHOT_MODE flag is set */
#define       MC_CMD_INIT_RXQ_V4_IN_SNAPSHOT_LENGTH_OFST 540
#define       MC_CMD_INIT_RXQ_V4_IN_SNAPSHOT_LENGTH_LEN 4
/* The number of packet buffers that will be contained within each
 * EQUAL_STRIDE_SUPER_BUFFER format bucket supplied by the driver. This field
 * is ignored unless DMA_MODE == EQUAL_STRIDE_SUPER_BUFFER.
 */
#define       MC_CMD_INIT_RXQ_V4_IN_ES_PACKET_BUFFERS_PER_BUCKET_OFST 544
#define       MC_CMD_INIT_RXQ_V4_IN_ES_PACKET_BUFFERS_PER_BUCKET_LEN 4
/* The length in bytes of the area in each packet buffer that can be written to
 * by the adapter. This is used to store the packet prefix and the packet
 * payload. This length does not include any end padding added by the driver.
 * This field is ignored unless DMA_MODE == EQUAL_STRIDE_SUPER_BUFFER.
 */
#define       MC_CMD_INIT_RXQ_V4_IN_ES_MAX_DMA_LEN_OFST 548
#define       MC_CMD_INIT_RXQ_V4_IN_ES_MAX_DMA_LEN_LEN 4
/* The length in bytes of a single packet buffer within a
 * EQUAL_STRIDE_SUPER_BUFFER format bucket. This field is ignored unless
 * DMA_MODE == EQUAL_STRIDE_SUPER_BUFFER.
 */
#define       MC_CMD_INIT_RXQ_V4_IN_ES_PACKET_STRIDE_OFST 552
#define       MC_CMD_INIT_RXQ_V4_IN_ES_PACKET_STRIDE_LEN 4
/* The maximum time in nanoseconds that the datapath will be backpressured if
 * there are no RX descriptors available. If the timeout is reached and there
 * are still no descriptors then the packet will be dropped. A timeout of 0
 * means the datapath will never be blocked. This field is ignored unless
 * DMA_MODE == EQUAL_STRIDE_SUPER_BUFFER.
 */
#define       MC_CMD_INIT_RXQ_V4_IN_ES_HEAD_OF_LINE_BLOCK_TIMEOUT_OFST 556
#define       MC_CMD_INIT_RXQ_V4_IN_ES_HEAD_OF_LINE_BLOCK_TIMEOUT_LEN 4
/* V4 message data */
#define       MC_CMD_INIT_RXQ_V4_IN_V4_DATA_OFST 560
#define       MC_CMD_INIT_RXQ_V4_IN_V4_DATA_LEN 4
/* Size in bytes of buffers attached to descriptors posted to this queue. Set
 * to zero if using this message on non-QDMA based platforms. Currently in
 * Riverhead there is a global limit of eight different buffer sizes across all
 * active queues. A 2KB and 4KB buffer is guaranteed to be available, but a
 * request for a different buffer size will fail if there are already eight
 * other buffer sizes in use. In future Riverhead this limit will go away and
 * any size will be accepted.
 */
#define       MC_CMD_INIT_RXQ_V4_IN_BUFFER_SIZE_BYTES_OFST 560
#define       MC_CMD_INIT_RXQ_V4_IN_BUFFER_SIZE_BYTES_LEN 4

/* MC_CMD_INIT_RXQ_V5_IN msgrequest: INIT_RXQ request with ability to request a
 * different RX packet prefix
 */
#define    MC_CMD_INIT_RXQ_V5_IN_LEN 568
/* Size, in entries */
#define       MC_CMD_INIT_RXQ_V5_IN_SIZE_OFST 0
#define       MC_CMD_INIT_RXQ_V5_IN_SIZE_LEN 4
/* The EVQ to send events to. This is an index originally specified to
 * INIT_EVQ. If DMA_MODE == PACKED_STREAM this must be equal to INSTANCE.
 */
#define       MC_CMD_INIT_RXQ_V5_IN_TARGET_EVQ_OFST 4
#define       MC_CMD_INIT_RXQ_V5_IN_TARGET_EVQ_LEN 4
/* The value to put in the event data. Check hardware spec. for valid range.
 * This field is ignored if DMA_MODE == EQUAL_STRIDE_SUPER_BUFFER or DMA_MODE
 * == PACKED_STREAM.
 */
#define       MC_CMD_INIT_RXQ_V5_IN_LABEL_OFST 8
#define       MC_CMD_INIT_RXQ_V5_IN_LABEL_LEN 4
/* Desired instance. Must be set to a specific instance, which is a function
 * local queue index. The calling client must be the currently-assigned user of
 * this VI (see MC_CMD_SET_VI_USER).
 */
#define       MC_CMD_INIT_RXQ_V5_IN_INSTANCE_OFST 12
#define       MC_CMD_INIT_RXQ_V5_IN_INSTANCE_LEN 4
/* There will be more flags here. */
#define       MC_CMD_INIT_RXQ_V5_IN_FLAGS_OFST 16
#define       MC_CMD_INIT_RXQ_V5_IN_FLAGS_LEN 4
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_BUFF_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_BUFF_MODE_LBN 0
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_BUFF_MODE_WIDTH 1
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_HDR_SPLIT_OFST 16
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_HDR_SPLIT_LBN 1
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_HDR_SPLIT_WIDTH 1
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_TIMESTAMP_OFST 16
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_TIMESTAMP_LBN 2
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_TIMESTAMP_WIDTH 1
#define        MC_CMD_INIT_RXQ_V5_IN_CRC_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_V5_IN_CRC_MODE_LBN 3
#define        MC_CMD_INIT_RXQ_V5_IN_CRC_MODE_WIDTH 4
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_CHAIN_OFST 16
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_CHAIN_LBN 7
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_CHAIN_WIDTH 1
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_PREFIX_OFST 16
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_PREFIX_LBN 8
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_PREFIX_WIDTH 1
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_DISABLE_SCATTER_OFST 16
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_DISABLE_SCATTER_LBN 9
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_DISABLE_SCATTER_WIDTH 1
#define        MC_CMD_INIT_RXQ_V5_IN_DMA_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_V5_IN_DMA_MODE_LBN 10
#define        MC_CMD_INIT_RXQ_V5_IN_DMA_MODE_WIDTH 4
/* enum: One packet per descriptor (for normal networking) */
#define          MC_CMD_INIT_RXQ_V5_IN_SINGLE_PACKET 0x0
/* enum: Pack multiple packets into large descriptors (for SolarCapture) */
#define          MC_CMD_INIT_RXQ_V5_IN_PACKED_STREAM 0x1
/* enum: Pack multiple packets into large descriptors using the format designed
 * to maximise packet rate. This mode uses 1 "bucket" per descriptor with
 * multiple fixed-size packet buffers within each bucket. For a full
 * description see SF-119419-TC. This mode is only supported by "dpdk" datapath
 * firmware.
 */
#define          MC_CMD_INIT_RXQ_V5_IN_EQUAL_STRIDE_SUPER_BUFFER 0x2
/* enum: Deprecated name for EQUAL_STRIDE_SUPER_BUFFER. */
#define          MC_CMD_INIT_RXQ_V5_IN_EQUAL_STRIDE_PACKED_STREAM 0x2
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_SNAPSHOT_MODE_OFST 16
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_SNAPSHOT_MODE_LBN 14
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_SNAPSHOT_MODE_WIDTH 1
#define        MC_CMD_INIT_RXQ_V5_IN_PACKED_STREAM_BUFF_SIZE_OFST 16
#define        MC_CMD_INIT_RXQ_V5_IN_PACKED_STREAM_BUFF_SIZE_LBN 15
#define        MC_CMD_INIT_RXQ_V5_IN_PACKED_STREAM_BUFF_SIZE_WIDTH 3
#define          MC_CMD_INIT_RXQ_V5_IN_PS_BUFF_1M 0x0 /* enum */
#define          MC_CMD_INIT_RXQ_V5_IN_PS_BUFF_512K 0x1 /* enum */
#define          MC_CMD_INIT_RXQ_V5_IN_PS_BUFF_256K 0x2 /* enum */
#define          MC_CMD_INIT_RXQ_V5_IN_PS_BUFF_128K 0x3 /* enum */
#define          MC_CMD_INIT_RXQ_V5_IN_PS_BUFF_64K 0x4 /* enum */
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_WANT_OUTER_CLASSES_OFST 16
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_WANT_OUTER_CLASSES_LBN 18
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_WANT_OUTER_CLASSES_WIDTH 1
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_FORCE_EV_MERGING_OFST 16
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_FORCE_EV_MERGING_LBN 19
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_FORCE_EV_MERGING_WIDTH 1
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_NO_CONT_EV_OFST 16
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_NO_CONT_EV_LBN 20
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_NO_CONT_EV_WIDTH 1
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_SUPPRESS_RX_EVENTS_OFST 16
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_SUPPRESS_RX_EVENTS_LBN 21
#define        MC_CMD_INIT_RXQ_V5_IN_FLAG_SUPPRESS_RX_EVENTS_WIDTH 1
/* Owner ID to use if in buffer mode (zero if physical) */
#define       MC_CMD_INIT_RXQ_V5_IN_OWNER_ID_OFST 20
#define       MC_CMD_INIT_RXQ_V5_IN_OWNER_ID_LEN 4
/* The port ID associated with the v-adaptor which should contain this DMAQ. */
#define       MC_CMD_INIT_RXQ_V5_IN_PORT_ID_OFST 24
#define       MC_CMD_INIT_RXQ_V5_IN_PORT_ID_LEN 4
/* 64-bit address of 4k of 4k-aligned host memory buffer */
#define       MC_CMD_INIT_RXQ_V5_IN_DMA_ADDR_OFST 28
#define       MC_CMD_INIT_RXQ_V5_IN_DMA_ADDR_LEN 8
#define       MC_CMD_INIT_RXQ_V5_IN_DMA_ADDR_LO_OFST 28
#define       MC_CMD_INIT_RXQ_V5_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_INIT_RXQ_V5_IN_DMA_ADDR_LO_LBN 224
#define       MC_CMD_INIT_RXQ_V5_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_INIT_RXQ_V5_IN_DMA_ADDR_HI_OFST 32
#define       MC_CMD_INIT_RXQ_V5_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_INIT_RXQ_V5_IN_DMA_ADDR_HI_LBN 256
#define       MC_CMD_INIT_RXQ_V5_IN_DMA_ADDR_HI_WIDTH 32
#define       MC_CMD_INIT_RXQ_V5_IN_DMA_ADDR_MINNUM 0
#define       MC_CMD_INIT_RXQ_V5_IN_DMA_ADDR_MAXNUM 64
#define       MC_CMD_INIT_RXQ_V5_IN_DMA_ADDR_MAXNUM_MCDI2 64
/* Maximum length of packet to receive, if SNAPSHOT_MODE flag is set */
#define       MC_CMD_INIT_RXQ_V5_IN_SNAPSHOT_LENGTH_OFST 540
#define       MC_CMD_INIT_RXQ_V5_IN_SNAPSHOT_LENGTH_LEN 4
/* The number of packet buffers that will be contained within each
 * EQUAL_STRIDE_SUPER_BUFFER format bucket supplied by the driver. This field
 * is ignored unless DMA_MODE == EQUAL_STRIDE_SUPER_BUFFER.
 */
#define       MC_CMD_INIT_RXQ_V5_IN_ES_PACKET_BUFFERS_PER_BUCKET_OFST 544
#define       MC_CMD_INIT_RXQ_V5_IN_ES_PACKET_BUFFERS_PER_BUCKET_LEN 4
/* The length in bytes of the area in each packet buffer that can be written to
 * by the adapter. This is used to store the packet prefix and the packet
 * payload. This length does not include any end padding added by the driver.
 * This field is ignored unless DMA_MODE == EQUAL_STRIDE_SUPER_BUFFER.
 */
#define       MC_CMD_INIT_RXQ_V5_IN_ES_MAX_DMA_LEN_OFST 548
#define       MC_CMD_INIT_RXQ_V5_IN_ES_MAX_DMA_LEN_LEN 4
/* The length in bytes of a single packet buffer within a
 * EQUAL_STRIDE_SUPER_BUFFER format bucket. This field is ignored unless
 * DMA_MODE == EQUAL_STRIDE_SUPER_BUFFER.
 */
#define       MC_CMD_INIT_RXQ_V5_IN_ES_PACKET_STRIDE_OFST 552
#define       MC_CMD_INIT_RXQ_V5_IN_ES_PACKET_STRIDE_LEN 4
/* The maximum time in nanoseconds that the datapath will be backpressured if
 * there are no RX descriptors available. If the timeout is reached and there
 * are still no descriptors then the packet will be dropped. A timeout of 0
 * means the datapath will never be blocked. This field is ignored unless
 * DMA_MODE == EQUAL_STRIDE_SUPER_BUFFER.
 */
#define       MC_CMD_INIT_RXQ_V5_IN_ES_HEAD_OF_LINE_BLOCK_TIMEOUT_OFST 556
#define       MC_CMD_INIT_RXQ_V5_IN_ES_HEAD_OF_LINE_BLOCK_TIMEOUT_LEN 4
/* V4 message data */
#define       MC_CMD_INIT_RXQ_V5_IN_V4_DATA_OFST 560
#define       MC_CMD_INIT_RXQ_V5_IN_V4_DATA_LEN 4
/* Size in bytes of buffers attached to descriptors posted to this queue. Set
 * to zero if using this message on non-QDMA based platforms. Currently in
 * Riverhead there is a global limit of eight different buffer sizes across all
 * active queues. A 2KB and 4KB buffer is guaranteed to be available, but a
 * request for a different buffer size will fail if there are already eight
 * other buffer sizes in use. In future Riverhead this limit will go away and
 * any size will be accepted.
 */
#define       MC_CMD_INIT_RXQ_V5_IN_BUFFER_SIZE_BYTES_OFST 560
#define       MC_CMD_INIT_RXQ_V5_IN_BUFFER_SIZE_BYTES_LEN 4
/* Prefix id for the RX prefix format to use on packets delivered this queue.
 * Zero is always a valid prefix id and means the default prefix format
 * documented for the platform. Other prefix ids can be obtained by calling
 * MC_CMD_GET_RX_PREFIX_ID with a requested set of prefix fields.
 */
#define       MC_CMD_INIT_RXQ_V5_IN_RX_PREFIX_ID_OFST 564
#define       MC_CMD_INIT_RXQ_V5_IN_RX_PREFIX_ID_LEN 4

/* MC_CMD_INIT_RXQ_OUT msgresponse */
#define    MC_CMD_INIT_RXQ_OUT_LEN 0

/* MC_CMD_INIT_RXQ_EXT_OUT msgresponse */
#define    MC_CMD_INIT_RXQ_EXT_OUT_LEN 0

/* MC_CMD_INIT_RXQ_V3_OUT msgresponse */
#define    MC_CMD_INIT_RXQ_V3_OUT_LEN 0

/* MC_CMD_INIT_RXQ_V4_OUT msgresponse */
#define    MC_CMD_INIT_RXQ_V4_OUT_LEN 0

/* MC_CMD_INIT_RXQ_V5_OUT msgresponse */
#define    MC_CMD_INIT_RXQ_V5_OUT_LEN 0


/***********************************/
/* MC_CMD_INIT_TXQ
 */
#define MC_CMD_INIT_TXQ 0x82
#undef MC_CMD_0x82_PRIVILEGE_CTG

#define MC_CMD_0x82_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_INIT_TXQ_IN msgrequest: Legacy INIT_TXQ request. Use extended version
 * in new code.
 */
#define    MC_CMD_INIT_TXQ_IN_LENMIN 36
#define    MC_CMD_INIT_TXQ_IN_LENMAX 252
#define    MC_CMD_INIT_TXQ_IN_LENMAX_MCDI2 1020
#define    MC_CMD_INIT_TXQ_IN_LEN(num) (28+8*(num))
#define    MC_CMD_INIT_TXQ_IN_DMA_ADDR_NUM(len) (((len)-28)/8)
/* Size, in entries */
#define       MC_CMD_INIT_TXQ_IN_SIZE_OFST 0
#define       MC_CMD_INIT_TXQ_IN_SIZE_LEN 4
/* The EVQ to send events to. This is an index originally specified to
 * INIT_EVQ.
 */
#define       MC_CMD_INIT_TXQ_IN_TARGET_EVQ_OFST 4
#define       MC_CMD_INIT_TXQ_IN_TARGET_EVQ_LEN 4
/* The value to put in the event data. Check hardware spec. for valid range. */
#define       MC_CMD_INIT_TXQ_IN_LABEL_OFST 8
#define       MC_CMD_INIT_TXQ_IN_LABEL_LEN 4
/* Desired instance. Must be set to a specific instance, which is a function
 * local queue index. The calling client must be the currently-assigned user of
 * this VI (see MC_CMD_SET_VI_USER).
 */
#define       MC_CMD_INIT_TXQ_IN_INSTANCE_OFST 12
#define       MC_CMD_INIT_TXQ_IN_INSTANCE_LEN 4
/* There will be more flags here. */
#define       MC_CMD_INIT_TXQ_IN_FLAGS_OFST 16
#define       MC_CMD_INIT_TXQ_IN_FLAGS_LEN 4
#define        MC_CMD_INIT_TXQ_IN_FLAG_BUFF_MODE_OFST 16
#define        MC_CMD_INIT_TXQ_IN_FLAG_BUFF_MODE_LBN 0
#define        MC_CMD_INIT_TXQ_IN_FLAG_BUFF_MODE_WIDTH 1
#define        MC_CMD_INIT_TXQ_IN_FLAG_IP_CSUM_DIS_OFST 16
#define        MC_CMD_INIT_TXQ_IN_FLAG_IP_CSUM_DIS_LBN 1
#define        MC_CMD_INIT_TXQ_IN_FLAG_IP_CSUM_DIS_WIDTH 1
#define        MC_CMD_INIT_TXQ_IN_FLAG_TCP_CSUM_DIS_OFST 16
#define        MC_CMD_INIT_TXQ_IN_FLAG_TCP_CSUM_DIS_LBN 2
#define        MC_CMD_INIT_TXQ_IN_FLAG_TCP_CSUM_DIS_WIDTH 1
#define        MC_CMD_INIT_TXQ_IN_FLAG_TCP_UDP_ONLY_OFST 16
#define        MC_CMD_INIT_TXQ_IN_FLAG_TCP_UDP_ONLY_LBN 3
#define        MC_CMD_INIT_TXQ_IN_FLAG_TCP_UDP_ONLY_WIDTH 1
#define        MC_CMD_INIT_TXQ_IN_CRC_MODE_OFST 16
#define        MC_CMD_INIT_TXQ_IN_CRC_MODE_LBN 4
#define        MC_CMD_INIT_TXQ_IN_CRC_MODE_WIDTH 4
#define        MC_CMD_INIT_TXQ_IN_FLAG_TIMESTAMP_OFST 16
#define        MC_CMD_INIT_TXQ_IN_FLAG_TIMESTAMP_LBN 8
#define        MC_CMD_INIT_TXQ_IN_FLAG_TIMESTAMP_WIDTH 1
#define        MC_CMD_INIT_TXQ_IN_FLAG_PACER_BYPASS_OFST 16
#define        MC_CMD_INIT_TXQ_IN_FLAG_PACER_BYPASS_LBN 9
#define        MC_CMD_INIT_TXQ_IN_FLAG_PACER_BYPASS_WIDTH 1
#define        MC_CMD_INIT_TXQ_IN_FLAG_INNER_IP_CSUM_EN_OFST 16
#define        MC_CMD_INIT_TXQ_IN_FLAG_INNER_IP_CSUM_EN_LBN 10
#define        MC_CMD_INIT_TXQ_IN_FLAG_INNER_IP_CSUM_EN_WIDTH 1
#define        MC_CMD_INIT_TXQ_IN_FLAG_INNER_TCP_CSUM_EN_OFST 16
#define        MC_CMD_INIT_TXQ_IN_FLAG_INNER_TCP_CSUM_EN_LBN 11
#define        MC_CMD_INIT_TXQ_IN_FLAG_INNER_TCP_CSUM_EN_WIDTH 1
/* Owner ID to use if in buffer mode (zero if physical) */
#define       MC_CMD_INIT_TXQ_IN_OWNER_ID_OFST 20
#define       MC_CMD_INIT_TXQ_IN_OWNER_ID_LEN 4
/* The port ID associated with the v-adaptor which should contain this DMAQ. */
#define       MC_CMD_INIT_TXQ_IN_PORT_ID_OFST 24
#define       MC_CMD_INIT_TXQ_IN_PORT_ID_LEN 4
/* 64-bit address of 4k of 4k-aligned host memory buffer */
#define       MC_CMD_INIT_TXQ_IN_DMA_ADDR_OFST 28
#define       MC_CMD_INIT_TXQ_IN_DMA_ADDR_LEN 8
#define       MC_CMD_INIT_TXQ_IN_DMA_ADDR_LO_OFST 28
#define       MC_CMD_INIT_TXQ_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_INIT_TXQ_IN_DMA_ADDR_LO_LBN 224
#define       MC_CMD_INIT_TXQ_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_INIT_TXQ_IN_DMA_ADDR_HI_OFST 32
#define       MC_CMD_INIT_TXQ_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_INIT_TXQ_IN_DMA_ADDR_HI_LBN 256
#define       MC_CMD_INIT_TXQ_IN_DMA_ADDR_HI_WIDTH 32
#define       MC_CMD_INIT_TXQ_IN_DMA_ADDR_MINNUM 1
#define       MC_CMD_INIT_TXQ_IN_DMA_ADDR_MAXNUM 28
#define       MC_CMD_INIT_TXQ_IN_DMA_ADDR_MAXNUM_MCDI2 124

/* MC_CMD_INIT_TXQ_EXT_IN msgrequest: Extended INIT_TXQ with additional mode
 * flags
 */
#define    MC_CMD_INIT_TXQ_EXT_IN_LEN 544
/* Size, in entries */
#define       MC_CMD_INIT_TXQ_EXT_IN_SIZE_OFST 0
#define       MC_CMD_INIT_TXQ_EXT_IN_SIZE_LEN 4
/* The EVQ to send events to. This is an index originally specified to
 * INIT_EVQ.
 */
#define       MC_CMD_INIT_TXQ_EXT_IN_TARGET_EVQ_OFST 4
#define       MC_CMD_INIT_TXQ_EXT_IN_TARGET_EVQ_LEN 4
/* The value to put in the event data. Check hardware spec. for valid range. */
#define       MC_CMD_INIT_TXQ_EXT_IN_LABEL_OFST 8
#define       MC_CMD_INIT_TXQ_EXT_IN_LABEL_LEN 4
/* Desired instance. Must be set to a specific instance, which is a function
 * local queue index. The calling client must be the currently-assigned user of
 * this VI (see MC_CMD_SET_VI_USER).
 */
#define       MC_CMD_INIT_TXQ_EXT_IN_INSTANCE_OFST 12
#define       MC_CMD_INIT_TXQ_EXT_IN_INSTANCE_LEN 4
/* There will be more flags here. */
#define       MC_CMD_INIT_TXQ_EXT_IN_FLAGS_OFST 16
#define       MC_CMD_INIT_TXQ_EXT_IN_FLAGS_LEN 4
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_BUFF_MODE_OFST 16
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_BUFF_MODE_LBN 0
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_BUFF_MODE_WIDTH 1
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_IP_CSUM_DIS_OFST 16
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_IP_CSUM_DIS_LBN 1
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_IP_CSUM_DIS_WIDTH 1
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_TCP_CSUM_DIS_OFST 16
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_TCP_CSUM_DIS_LBN 2
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_TCP_CSUM_DIS_WIDTH 1
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_TCP_UDP_ONLY_OFST 16
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_TCP_UDP_ONLY_LBN 3
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_TCP_UDP_ONLY_WIDTH 1
#define        MC_CMD_INIT_TXQ_EXT_IN_CRC_MODE_OFST 16
#define        MC_CMD_INIT_TXQ_EXT_IN_CRC_MODE_LBN 4
#define        MC_CMD_INIT_TXQ_EXT_IN_CRC_MODE_WIDTH 4
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_TIMESTAMP_OFST 16
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_TIMESTAMP_LBN 8
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_TIMESTAMP_WIDTH 1
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_PACER_BYPASS_OFST 16
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_PACER_BYPASS_LBN 9
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_PACER_BYPASS_WIDTH 1
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_INNER_IP_CSUM_EN_OFST 16
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_INNER_IP_CSUM_EN_LBN 10
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_INNER_IP_CSUM_EN_WIDTH 1
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_INNER_TCP_CSUM_EN_OFST 16
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_INNER_TCP_CSUM_EN_LBN 11
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_INNER_TCP_CSUM_EN_WIDTH 1
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_TSOV2_EN_OFST 16
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_TSOV2_EN_LBN 12
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_TSOV2_EN_WIDTH 1
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_CTPIO_OFST 16
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_CTPIO_LBN 13
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_CTPIO_WIDTH 1
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_CTPIO_UTHRESH_OFST 16
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_CTPIO_UTHRESH_LBN 14
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_CTPIO_UTHRESH_WIDTH 1
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_M2M_D2C_OFST 16
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_M2M_D2C_LBN 15
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_M2M_D2C_WIDTH 1
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_DESC_PROXY_OFST 16
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_DESC_PROXY_LBN 16
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_DESC_PROXY_WIDTH 1
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_ABS_TARGET_EVQ_OFST 16
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_ABS_TARGET_EVQ_LBN 17
#define        MC_CMD_INIT_TXQ_EXT_IN_FLAG_ABS_TARGET_EVQ_WIDTH 1
/* Owner ID to use if in buffer mode (zero if physical) */
#define       MC_CMD_INIT_TXQ_EXT_IN_OWNER_ID_OFST 20
#define       MC_CMD_INIT_TXQ_EXT_IN_OWNER_ID_LEN 4
/* The port ID associated with the v-adaptor which should contain this DMAQ. */
#define       MC_CMD_INIT_TXQ_EXT_IN_PORT_ID_OFST 24
#define       MC_CMD_INIT_TXQ_EXT_IN_PORT_ID_LEN 4
/* 64-bit address of 4k of 4k-aligned host memory buffer */
#define       MC_CMD_INIT_TXQ_EXT_IN_DMA_ADDR_OFST 28
#define       MC_CMD_INIT_TXQ_EXT_IN_DMA_ADDR_LEN 8
#define       MC_CMD_INIT_TXQ_EXT_IN_DMA_ADDR_LO_OFST 28
#define       MC_CMD_INIT_TXQ_EXT_IN_DMA_ADDR_LO_LEN 4
#define       MC_CMD_INIT_TXQ_EXT_IN_DMA_ADDR_LO_LBN 224
#define       MC_CMD_INIT_TXQ_EXT_IN_DMA_ADDR_LO_WIDTH 32
#define       MC_CMD_INIT_TXQ_EXT_IN_DMA_ADDR_HI_OFST 32
#define       MC_CMD_INIT_TXQ_EXT_IN_DMA_ADDR_HI_LEN 4
#define       MC_CMD_INIT_TXQ_EXT_IN_DMA_ADDR_HI_LBN 256
#define       MC_CMD_INIT_TXQ_EXT_IN_DMA_ADDR_HI_WIDTH 32
#define       MC_CMD_INIT_TXQ_EXT_IN_DMA_ADDR_MINNUM 0
#define       MC_CMD_INIT_TXQ_EXT_IN_DMA_ADDR_MAXNUM 64
#define       MC_CMD_INIT_TXQ_EXT_IN_DMA_ADDR_MAXNUM_MCDI2 64
/* Flags related to Qbb flow control mode. */
#define       MC_CMD_INIT_TXQ_EXT_IN_QBB_FLAGS_OFST 540
#define       MC_CMD_INIT_TXQ_EXT_IN_QBB_FLAGS_LEN 4
#define        MC_CMD_INIT_TXQ_EXT_IN_QBB_ENABLE_OFST 540
#define        MC_CMD_INIT_TXQ_EXT_IN_QBB_ENABLE_LBN 0
#define        MC_CMD_INIT_TXQ_EXT_IN_QBB_ENABLE_WIDTH 1
#define        MC_CMD_INIT_TXQ_EXT_IN_QBB_PRIORITY_OFST 540
#define        MC_CMD_INIT_TXQ_EXT_IN_QBB_PRIORITY_LBN 1
#define        MC_CMD_INIT_TXQ_EXT_IN_QBB_PRIORITY_WIDTH 3

/* MC_CMD_INIT_TXQ_OUT msgresponse */
#define    MC_CMD_INIT_TXQ_OUT_LEN 0


/***********************************/
/* MC_CMD_FINI_EVQ
 * Teardown an EVQ.
 *
 * All DMAQs or EVQs that point to the EVQ to tear down must be torn down first
 * or the operation will fail with EBUSY
 */
#define MC_CMD_FINI_EVQ 0x83
#undef MC_CMD_0x83_PRIVILEGE_CTG

#define MC_CMD_0x83_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_FINI_EVQ_IN msgrequest */
#define    MC_CMD_FINI_EVQ_IN_LEN 4
/* Instance of EVQ to destroy. Should be the same instance as that previously
 * passed to INIT_EVQ
 */
#define       MC_CMD_FINI_EVQ_IN_INSTANCE_OFST 0
#define       MC_CMD_FINI_EVQ_IN_INSTANCE_LEN 4

/* MC_CMD_FINI_EVQ_OUT msgresponse */
#define    MC_CMD_FINI_EVQ_OUT_LEN 0


/***********************************/
/* MC_CMD_FINI_RXQ
 * Teardown a RXQ.
 */
#define MC_CMD_FINI_RXQ 0x84
#undef MC_CMD_0x84_PRIVILEGE_CTG

#define MC_CMD_0x84_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_FINI_RXQ_IN msgrequest */
#define    MC_CMD_FINI_RXQ_IN_LEN 4
/* Instance of RXQ to destroy */
#define       MC_CMD_FINI_RXQ_IN_INSTANCE_OFST 0
#define       MC_CMD_FINI_RXQ_IN_INSTANCE_LEN 4

/* MC_CMD_FINI_RXQ_OUT msgresponse */
#define    MC_CMD_FINI_RXQ_OUT_LEN 0


/***********************************/
/* MC_CMD_FINI_TXQ
 * Teardown a TXQ.
 */
#define MC_CMD_FINI_TXQ 0x85
#undef MC_CMD_0x85_PRIVILEGE_CTG

#define MC_CMD_0x85_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_FINI_TXQ_IN msgrequest */
#define    MC_CMD_FINI_TXQ_IN_LEN 4
/* Instance of TXQ to destroy */
#define       MC_CMD_FINI_TXQ_IN_INSTANCE_OFST 0
#define       MC_CMD_FINI_TXQ_IN_INSTANCE_LEN 4

/* MC_CMD_FINI_TXQ_OUT msgresponse */
#define    MC_CMD_FINI_TXQ_OUT_LEN 0


/***********************************/
/* MC_CMD_DRIVER_EVENT
 * Generate an event on an EVQ belonging to the function issuing the command.
 */
#define MC_CMD_DRIVER_EVENT 0x86
#undef MC_CMD_0x86_PRIVILEGE_CTG

#define MC_CMD_0x86_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_DRIVER_EVENT_IN msgrequest */
#define    MC_CMD_DRIVER_EVENT_IN_LEN 12
/* Handle of target EVQ */
#define       MC_CMD_DRIVER_EVENT_IN_EVQ_OFST 0
#define       MC_CMD_DRIVER_EVENT_IN_EVQ_LEN 4
/* Bits 0 - 63 of event */
#define       MC_CMD_DRIVER_EVENT_IN_DATA_OFST 4
#define       MC_CMD_DRIVER_EVENT_IN_DATA_LEN 8
#define       MC_CMD_DRIVER_EVENT_IN_DATA_LO_OFST 4
#define       MC_CMD_DRIVER_EVENT_IN_DATA_LO_LEN 4
#define       MC_CMD_DRIVER_EVENT_IN_DATA_LO_LBN 32
#define       MC_CMD_DRIVER_EVENT_IN_DATA_LO_WIDTH 32
#define       MC_CMD_DRIVER_EVENT_IN_DATA_HI_OFST 8
#define       MC_CMD_DRIVER_EVENT_IN_DATA_HI_LEN 4
#define       MC_CMD_DRIVER_EVENT_IN_DATA_HI_LBN 64
#define       MC_CMD_DRIVER_EVENT_IN_DATA_HI_WIDTH 32

/* MC_CMD_DRIVER_EVENT_OUT msgresponse */
#define    MC_CMD_DRIVER_EVENT_OUT_LEN 0


/***********************************/
/* MC_CMD_PROXY_CMD
 * Execute an arbitrary MCDI command on behalf of a different function, subject
 * to security restrictions. The command to be proxied follows immediately
 * afterward in the host buffer (or on the UART). This command supercedes
 * MC_CMD_SET_FUNC, which remains available for Siena but now deprecated.
 */
#define MC_CMD_PROXY_CMD 0x5b
#undef MC_CMD_0x5b_PRIVILEGE_CTG

#define MC_CMD_0x5b_PRIVILEGE_CTG SRIOV_CTG_ADMIN

/* MC_CMD_PROXY_CMD_IN msgrequest */
#define    MC_CMD_PROXY_CMD_IN_LEN 4
/* The handle of the target function. */
#define       MC_CMD_PROXY_CMD_IN_TARGET_OFST 0
#define       MC_CMD_PROXY_CMD_IN_TARGET_LEN 4
#define        MC_CMD_PROXY_CMD_IN_TARGET_PF_OFST 0
#define        MC_CMD_PROXY_CMD_IN_TARGET_PF_LBN 0
#define        MC_CMD_PROXY_CMD_IN_TARGET_PF_WIDTH 16
#define        MC_CMD_PROXY_CMD_IN_TARGET_VF_OFST 0
#define        MC_CMD_PROXY_CMD_IN_TARGET_VF_LBN 16
#define        MC_CMD_PROXY_CMD_IN_TARGET_VF_WIDTH 16
#define          MC_CMD_PROXY_CMD_IN_VF_NULL 0xffff /* enum */

/* MC_CMD_PROXY_CMD_OUT msgresponse */
#define    MC_CMD_PROXY_CMD_OUT_LEN 0


/***********************************/
/* MC_CMD_FILTER_OP
 * Multiplexed MCDI call for filter operations
 */
#define MC_CMD_FILTER_OP 0x8a
#undef MC_CMD_0x8a_PRIVILEGE_CTG

#define MC_CMD_0x8a_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_FILTER_OP_IN msgrequest */
#define    MC_CMD_FILTER_OP_IN_LEN 108
/* identifies the type of operation requested */
#define       MC_CMD_FILTER_OP_IN_OP_OFST 0
#define       MC_CMD_FILTER_OP_IN_OP_LEN 4
/* enum: single-recipient filter insert */
#define          MC_CMD_FILTER_OP_IN_OP_INSERT 0x0
/* enum: single-recipient filter remove */
#define          MC_CMD_FILTER_OP_IN_OP_REMOVE 0x1
/* enum: multi-recipient filter subscribe */
#define          MC_CMD_FILTER_OP_IN_OP_SUBSCRIBE 0x2
/* enum: multi-recipient filter unsubscribe */
#define          MC_CMD_FILTER_OP_IN_OP_UNSUBSCRIBE 0x3
/* enum: replace one recipient with another (warning - the filter handle may
 * change)
 */
#define          MC_CMD_FILTER_OP_IN_OP_REPLACE 0x4
/* filter handle (for remove / unsubscribe operations) */
#define       MC_CMD_FILTER_OP_IN_HANDLE_OFST 4
#define       MC_CMD_FILTER_OP_IN_HANDLE_LEN 8
#define       MC_CMD_FILTER_OP_IN_HANDLE_LO_OFST 4
#define       MC_CMD_FILTER_OP_IN_HANDLE_LO_LEN 4
#define       MC_CMD_FILTER_OP_IN_HANDLE_LO_LBN 32
#define       MC_CMD_FILTER_OP_IN_HANDLE_LO_WIDTH 32
#define       MC_CMD_FILTER_OP_IN_HANDLE_HI_OFST 8
#define       MC_CMD_FILTER_OP_IN_HANDLE_HI_LEN 4
#define       MC_CMD_FILTER_OP_IN_HANDLE_HI_LBN 64
#define       MC_CMD_FILTER_OP_IN_HANDLE_HI_WIDTH 32
/* The port ID associated with the v-adaptor which should contain this filter.
 */
#define       MC_CMD_FILTER_OP_IN_PORT_ID_OFST 12
#define       MC_CMD_FILTER_OP_IN_PORT_ID_LEN 4
/* fields to include in match criteria */
#define       MC_CMD_FILTER_OP_IN_MATCH_FIELDS_OFST 16
#define       MC_CMD_FILTER_OP_IN_MATCH_FIELDS_LEN 4
#define        MC_CMD_FILTER_OP_IN_MATCH_SRC_IP_OFST 16
#define        MC_CMD_FILTER_OP_IN_MATCH_SRC_IP_LBN 0
#define        MC_CMD_FILTER_OP_IN_MATCH_SRC_IP_WIDTH 1
#define        MC_CMD_FILTER_OP_IN_MATCH_DST_IP_OFST 16
#define        MC_CMD_FILTER_OP_IN_MATCH_DST_IP_LBN 1
#define        MC_CMD_FILTER_OP_IN_MATCH_DST_IP_WIDTH 1
#define        MC_CMD_FILTER_OP_IN_MATCH_SRC_MAC_OFST 16
#define        MC_CMD_FILTER_OP_IN_MATCH_SRC_MAC_LBN 2
#define        MC_CMD_FILTER_OP_IN_MATCH_SRC_MAC_WIDTH 1
#define        MC_CMD_FILTER_OP_IN_MATCH_SRC_PORT_OFST 16
#define        MC_CMD_FILTER_OP_IN_MATCH_SRC_PORT_LBN 3
#define        MC_CMD_FILTER_OP_IN_MATCH_SRC_PORT_WIDTH 1
#define        MC_CMD_FILTER_OP_IN_MATCH_DST_MAC_OFST 16
#define        MC_CMD_FILTER_OP_IN_MATCH_DST_MAC_LBN 4
#define        MC_CMD_FILTER_OP_IN_MATCH_DST_MAC_WIDTH 1
#define        MC_CMD_FILTER_OP_IN_MATCH_DST_PORT_OFST 16
#define        MC_CMD_FILTER_OP_IN_MATCH_DST_PORT_LBN 5
#define        MC_CMD_FILTER_OP_IN_MATCH_DST_PORT_WIDTH 1
#define        MC_CMD_FILTER_OP_IN_MATCH_ETHER_TYPE_OFST 16
#define        MC_CMD_FILTER_OP_IN_MATCH_ETHER_TYPE_LBN 6
#define        MC_CMD_FILTER_OP_IN_MATCH_ETHER_TYPE_WIDTH 1
#define        MC_CMD_FILTER_OP_IN_MATCH_INNER_VLAN_OFST 16
#define        MC_CMD_FILTER_OP_IN_MATCH_INNER_VLAN_LBN 7
#define        MC_CMD_FILTER_OP_IN_MATCH_INNER_VLAN_WIDTH 1
#define        MC_CMD_FILTER_OP_IN_MATCH_OUTER_VLAN_OFST 16
#define        MC_CMD_FILTER_OP_IN_MATCH_OUTER_VLAN_LBN 8
#define        MC_CMD_FILTER_OP_IN_MATCH_OUTER_VLAN_WIDTH 1
#define        MC_CMD_FILTER_OP_IN_MATCH_IP_PROTO_OFST 16
#define        MC_CMD_FILTER_OP_IN_MATCH_IP_PROTO_LBN 9
#define        MC_CMD_FILTER_OP_IN_MATCH_IP_PROTO_WIDTH 1
#define        MC_CMD_FILTER_OP_IN_MATCH_FWDEF0_OFST 16
#define        MC_CMD_FILTER_OP_IN_MATCH_FWDEF0_LBN 10
#define        MC_CMD_FILTER_OP_IN_MATCH_FWDEF0_WIDTH 1
#define        MC_CMD_FILTER_OP_IN_MATCH_FWDEF1_OFST 16
#define        MC_CMD_FILTER_OP_IN_MATCH_FWDEF1_LBN 11
#define        MC_CMD_FILTER_OP_IN_MATCH_FWDEF1_WIDTH 1
#define        MC_CMD_FILTER_OP_IN_MATCH_UNKNOWN_IPV4_MCAST_DST_OFST 16
#define        MC_CMD_FILTER_OP_IN_MATCH_UNKNOWN_IPV4_MCAST_DST_LBN 29
#define        MC_CMD_FILTER_OP_IN_MATCH_UNKNOWN_IPV4_MCAST_DST_WIDTH 1
#define        MC_CMD_FILTER_OP_IN_MATCH_UNKNOWN_MCAST_DST_OFST 16
#define        MC_CMD_FILTER_OP_IN_MATCH_UNKNOWN_MCAST_DST_LBN 30
#define        MC_CMD_FILTER_OP_IN_MATCH_UNKNOWN_MCAST_DST_WIDTH 1
#define        MC_CMD_FILTER_OP_IN_MATCH_UNKNOWN_UCAST_DST_OFST 16
#define        MC_CMD_FILTER_OP_IN_MATCH_UNKNOWN_UCAST_DST_LBN 31
#define        MC_CMD_FILTER_OP_IN_MATCH_UNKNOWN_UCAST_DST_WIDTH 1
/* receive destination */
#define       MC_CMD_FILTER_OP_IN_RX_DEST_OFST 20
#define       MC_CMD_FILTER_OP_IN_RX_DEST_LEN 4
/* enum: drop packets */
#define          MC_CMD_FILTER_OP_IN_RX_DEST_DROP 0x0
/* enum: receive to host */
#define          MC_CMD_FILTER_OP_IN_RX_DEST_HOST 0x1
/* enum: receive to MC */
#define          MC_CMD_FILTER_OP_IN_RX_DEST_MC 0x2
/* enum: loop back to TXDP 0 */
#define          MC_CMD_FILTER_OP_IN_RX_DEST_TX0 0x3
/* enum: loop back to TXDP 1 */
#define          MC_CMD_FILTER_OP_IN_RX_DEST_TX1 0x4
/* receive queue handle (for multiple queue modes, this is the base queue) */
#define       MC_CMD_FILTER_OP_IN_RX_QUEUE_OFST 24
#define       MC_CMD_FILTER_OP_IN_RX_QUEUE_LEN 4
/* receive mode */
#define       MC_CMD_FILTER_OP_IN_RX_MODE_OFST 28
#define       MC_CMD_FILTER_OP_IN_RX_MODE_LEN 4
/* enum: receive to just the specified queue */
#define          MC_CMD_FILTER_OP_IN_RX_MODE_SIMPLE 0x0
/* enum: receive to multiple queues using RSS context */
#define          MC_CMD_FILTER_OP_IN_RX_MODE_RSS 0x1
/* enum: receive to multiple queues using .1p mapping */
#define          MC_CMD_FILTER_OP_IN_RX_MODE_DOT1P_MAPPING 0x2
/* enum: install a filter entry that will never match; for test purposes only
 */
#define          MC_CMD_FILTER_OP_IN_RX_MODE_TEST_NEVER_MATCH 0x80000000
/* RSS context (for RX_MODE_RSS) or .1p mapping handle (for
 * RX_MODE_DOT1P_MAPPING), as returned by MC_CMD_RSS_CONTEXT_ALLOC or
 * MC_CMD_DOT1P_MAPPING_ALLOC.
 */
#define       MC_CMD_FILTER_OP_IN_RX_CONTEXT_OFST 32
#define       MC_CMD_FILTER_OP_IN_RX_CONTEXT_LEN 4
/* transmit domain (reserved; set to 0) */
#define       MC_CMD_FILTER_OP_IN_TX_DOMAIN_OFST 36
#define       MC_CMD_FILTER_OP_IN_TX_DOMAIN_LEN 4
/* transmit destination (either set the MAC and/or PM bits for explicit
 * control, or set this field to TX_DEST_DEFAULT for sensible default
 * behaviour)
 */
#define       MC_CMD_FILTER_OP_IN_TX_DEST_OFST 40
#define       MC_CMD_FILTER_OP_IN_TX_DEST_LEN 4
/* enum: request default behaviour (based on filter type) */
#define          MC_CMD_FILTER_OP_IN_TX_DEST_DEFAULT 0xffffffff
#define        MC_CMD_FILTER_OP_IN_TX_DEST_MAC_OFST 40
#define        MC_CMD_FILTER_OP_IN_TX_DEST_MAC_LBN 0
#define        MC_CMD_FILTER_OP_IN_TX_DEST_MAC_WIDTH 1
#define        MC_CMD_FILTER_OP_IN_TX_DEST_PM_OFST 40
#define        MC_CMD_FILTER_OP_IN_TX_DEST_PM_LBN 1
#define        MC_CMD_FILTER_OP_IN_TX_DEST_PM_WIDTH 1
/* source MAC address to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_IN_SRC_MAC_OFST 44
#define       MC_CMD_FILTER_OP_IN_SRC_MAC_LEN 6
/* source port to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_IN_SRC_PORT_OFST 50
#define       MC_CMD_FILTER_OP_IN_SRC_PORT_LEN 2
/* destination MAC address to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_IN_DST_MAC_OFST 52
#define       MC_CMD_FILTER_OP_IN_DST_MAC_LEN 6
/* destination port to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_IN_DST_PORT_OFST 58
#define       MC_CMD_FILTER_OP_IN_DST_PORT_LEN 2
/* Ethernet type to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_IN_ETHER_TYPE_OFST 60
#define       MC_CMD_FILTER_OP_IN_ETHER_TYPE_LEN 2
/* Inner VLAN tag to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_IN_INNER_VLAN_OFST 62
#define       MC_CMD_FILTER_OP_IN_INNER_VLAN_LEN 2
/* Outer VLAN tag to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_IN_OUTER_VLAN_OFST 64
#define       MC_CMD_FILTER_OP_IN_OUTER_VLAN_LEN 2
/* IP protocol to match (in low byte; set high byte to 0) */
#define       MC_CMD_FILTER_OP_IN_IP_PROTO_OFST 66
#define       MC_CMD_FILTER_OP_IN_IP_PROTO_LEN 2
/* Firmware defined register 0 to match (reserved; set to 0) */
#define       MC_CMD_FILTER_OP_IN_FWDEF0_OFST 68
#define       MC_CMD_FILTER_OP_IN_FWDEF0_LEN 4
/* Firmware defined register 1 to match (reserved; set to 0) */
#define       MC_CMD_FILTER_OP_IN_FWDEF1_OFST 72
#define       MC_CMD_FILTER_OP_IN_FWDEF1_LEN 4
/* source IP address to match (as bytes in network order; set last 12 bytes to
 * 0 for IPv4 address)
 */
#define       MC_CMD_FILTER_OP_IN_SRC_IP_OFST 76
#define       MC_CMD_FILTER_OP_IN_SRC_IP_LEN 16
/* destination IP address to match (as bytes in network order; set last 12
 * bytes to 0 for IPv4 address)
 */
#define       MC_CMD_FILTER_OP_IN_DST_IP_OFST 92
#define       MC_CMD_FILTER_OP_IN_DST_IP_LEN 16

/* MC_CMD_FILTER_OP_EXT_IN msgrequest: Extension to MC_CMD_FILTER_OP_IN to
 * include handling of VXLAN/NVGRE encapsulated frame filtering (which is
 * supported on Medford only).
 */
#define    MC_CMD_FILTER_OP_EXT_IN_LEN 172
/* identifies the type of operation requested */
#define       MC_CMD_FILTER_OP_EXT_IN_OP_OFST 0
#define       MC_CMD_FILTER_OP_EXT_IN_OP_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_FILTER_OP_IN/OP */
/* filter handle (for remove / unsubscribe operations) */
#define       MC_CMD_FILTER_OP_EXT_IN_HANDLE_OFST 4
#define       MC_CMD_FILTER_OP_EXT_IN_HANDLE_LEN 8
#define       MC_CMD_FILTER_OP_EXT_IN_HANDLE_LO_OFST 4
#define       MC_CMD_FILTER_OP_EXT_IN_HANDLE_LO_LEN 4
#define       MC_CMD_FILTER_OP_EXT_IN_HANDLE_LO_LBN 32
#define       MC_CMD_FILTER_OP_EXT_IN_HANDLE_LO_WIDTH 32
#define       MC_CMD_FILTER_OP_EXT_IN_HANDLE_HI_OFST 8
#define       MC_CMD_FILTER_OP_EXT_IN_HANDLE_HI_LEN 4
#define       MC_CMD_FILTER_OP_EXT_IN_HANDLE_HI_LBN 64
#define       MC_CMD_FILTER_OP_EXT_IN_HANDLE_HI_WIDTH 32
/* The port ID associated with the v-adaptor which should contain this filter.
 */
#define       MC_CMD_FILTER_OP_EXT_IN_PORT_ID_OFST 12
#define       MC_CMD_FILTER_OP_EXT_IN_PORT_ID_LEN 4
/* fields to include in match criteria */
#define       MC_CMD_FILTER_OP_EXT_IN_MATCH_FIELDS_OFST 16
#define       MC_CMD_FILTER_OP_EXT_IN_MATCH_FIELDS_LEN 4
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_SRC_IP_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_SRC_IP_LBN 0
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_SRC_IP_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_DST_IP_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_DST_IP_LBN 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_DST_IP_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_SRC_MAC_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_SRC_MAC_LBN 2
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_SRC_MAC_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_SRC_PORT_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_SRC_PORT_LBN 3
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_SRC_PORT_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_DST_MAC_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_DST_MAC_LBN 4
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_DST_MAC_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_DST_PORT_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_DST_PORT_LBN 5
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_DST_PORT_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_ETHER_TYPE_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_ETHER_TYPE_LBN 6
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_ETHER_TYPE_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_INNER_VLAN_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_INNER_VLAN_LBN 7
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_INNER_VLAN_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_OUTER_VLAN_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_OUTER_VLAN_LBN 8
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_OUTER_VLAN_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IP_PROTO_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IP_PROTO_LBN 9
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IP_PROTO_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_FWDEF0_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_FWDEF0_LBN 10
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_FWDEF0_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_VNI_OR_VSID_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_VNI_OR_VSID_LBN 11
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_VNI_OR_VSID_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_SRC_IP_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_SRC_IP_LBN 12
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_SRC_IP_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_DST_IP_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_DST_IP_LBN 13
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_DST_IP_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_SRC_MAC_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_SRC_MAC_LBN 14
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_SRC_MAC_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_SRC_PORT_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_SRC_PORT_LBN 15
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_SRC_PORT_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_DST_MAC_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_DST_MAC_LBN 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_DST_MAC_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_DST_PORT_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_DST_PORT_LBN 17
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_DST_PORT_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_ETHER_TYPE_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_ETHER_TYPE_LBN 18
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_ETHER_TYPE_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_INNER_VLAN_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_INNER_VLAN_LBN 19
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_INNER_VLAN_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_OUTER_VLAN_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_OUTER_VLAN_LBN 20
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_OUTER_VLAN_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_IP_PROTO_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_IP_PROTO_LBN 21
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_IP_PROTO_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_FWDEF0_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_FWDEF0_LBN 22
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_FWDEF0_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_FWDEF1_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_FWDEF1_LBN 23
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_FWDEF1_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_UNKNOWN_MCAST_DST_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_UNKNOWN_MCAST_DST_LBN 24
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_UNKNOWN_MCAST_DST_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_UNKNOWN_UCAST_DST_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_UNKNOWN_UCAST_DST_LBN 25
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_IFRM_UNKNOWN_UCAST_DST_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_UNKNOWN_IPV4_MCAST_DST_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_UNKNOWN_IPV4_MCAST_DST_LBN 29
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_UNKNOWN_IPV4_MCAST_DST_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_UNKNOWN_MCAST_DST_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_UNKNOWN_MCAST_DST_LBN 30
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_UNKNOWN_MCAST_DST_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_UNKNOWN_UCAST_DST_OFST 16
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_UNKNOWN_UCAST_DST_LBN 31
#define        MC_CMD_FILTER_OP_EXT_IN_MATCH_UNKNOWN_UCAST_DST_WIDTH 1
/* receive destination */
#define       MC_CMD_FILTER_OP_EXT_IN_RX_DEST_OFST 20
#define       MC_CMD_FILTER_OP_EXT_IN_RX_DEST_LEN 4
/* enum: drop packets */
#define          MC_CMD_FILTER_OP_EXT_IN_RX_DEST_DROP 0x0
/* enum: receive to host */
#define          MC_CMD_FILTER_OP_EXT_IN_RX_DEST_HOST 0x1
/* enum: receive to MC */
#define          MC_CMD_FILTER_OP_EXT_IN_RX_DEST_MC 0x2
/* enum: loop back to TXDP 0 */
#define          MC_CMD_FILTER_OP_EXT_IN_RX_DEST_TX0 0x3
/* enum: loop back to TXDP 1 */
#define          MC_CMD_FILTER_OP_EXT_IN_RX_DEST_TX1 0x4
/* receive queue handle (for multiple queue modes, this is the base queue) */
#define       MC_CMD_FILTER_OP_EXT_IN_RX_QUEUE_OFST 24
#define       MC_CMD_FILTER_OP_EXT_IN_RX_QUEUE_LEN 4
/* receive mode */
#define       MC_CMD_FILTER_OP_EXT_IN_RX_MODE_OFST 28
#define       MC_CMD_FILTER_OP_EXT_IN_RX_MODE_LEN 4
/* enum: receive to just the specified queue */
#define          MC_CMD_FILTER_OP_EXT_IN_RX_MODE_SIMPLE 0x0
/* enum: receive to multiple queues using RSS context */
#define          MC_CMD_FILTER_OP_EXT_IN_RX_MODE_RSS 0x1
/* enum: receive to multiple queues using .1p mapping */
#define          MC_CMD_FILTER_OP_EXT_IN_RX_MODE_DOT1P_MAPPING 0x2
/* enum: install a filter entry that will never match; for test purposes only
 */
#define          MC_CMD_FILTER_OP_EXT_IN_RX_MODE_TEST_NEVER_MATCH 0x80000000
/* RSS context (for RX_MODE_RSS) or .1p mapping handle (for
 * RX_MODE_DOT1P_MAPPING), as returned by MC_CMD_RSS_CONTEXT_ALLOC or
 * MC_CMD_DOT1P_MAPPING_ALLOC.
 */
#define       MC_CMD_FILTER_OP_EXT_IN_RX_CONTEXT_OFST 32
#define       MC_CMD_FILTER_OP_EXT_IN_RX_CONTEXT_LEN 4
/* transmit domain (reserved; set to 0) */
#define       MC_CMD_FILTER_OP_EXT_IN_TX_DOMAIN_OFST 36
#define       MC_CMD_FILTER_OP_EXT_IN_TX_DOMAIN_LEN 4
/* transmit destination (either set the MAC and/or PM bits for explicit
 * control, or set this field to TX_DEST_DEFAULT for sensible default
 * behaviour)
 */
#define       MC_CMD_FILTER_OP_EXT_IN_TX_DEST_OFST 40
#define       MC_CMD_FILTER_OP_EXT_IN_TX_DEST_LEN 4
/* enum: request default behaviour (based on filter type) */
#define          MC_CMD_FILTER_OP_EXT_IN_TX_DEST_DEFAULT 0xffffffff
#define        MC_CMD_FILTER_OP_EXT_IN_TX_DEST_MAC_OFST 40
#define        MC_CMD_FILTER_OP_EXT_IN_TX_DEST_MAC_LBN 0
#define        MC_CMD_FILTER_OP_EXT_IN_TX_DEST_MAC_WIDTH 1
#define        MC_CMD_FILTER_OP_EXT_IN_TX_DEST_PM_OFST 40
#define        MC_CMD_FILTER_OP_EXT_IN_TX_DEST_PM_LBN 1
#define        MC_CMD_FILTER_OP_EXT_IN_TX_DEST_PM_WIDTH 1
/* source MAC address to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_EXT_IN_SRC_MAC_OFST 44
#define       MC_CMD_FILTER_OP_EXT_IN_SRC_MAC_LEN 6
/* source port to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_EXT_IN_SRC_PORT_OFST 50
#define       MC_CMD_FILTER_OP_EXT_IN_SRC_PORT_LEN 2
/* destination MAC address to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_EXT_IN_DST_MAC_OFST 52
#define       MC_CMD_FILTER_OP_EXT_IN_DST_MAC_LEN 6
/* destination port to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_EXT_IN_DST_PORT_OFST 58
#define       MC_CMD_FILTER_OP_EXT_IN_DST_PORT_LEN 2
/* Ethernet type to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_EXT_IN_ETHER_TYPE_OFST 60
#define       MC_CMD_FILTER_OP_EXT_IN_ETHER_TYPE_LEN 2
/* Inner VLAN tag to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_EXT_IN_INNER_VLAN_OFST 62
#define       MC_CMD_FILTER_OP_EXT_IN_INNER_VLAN_LEN 2
/* Outer VLAN tag to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_EXT_IN_OUTER_VLAN_OFST 64
#define       MC_CMD_FILTER_OP_EXT_IN_OUTER_VLAN_LEN 2
/* IP protocol to match (in low byte; set high byte to 0) */
#define       MC_CMD_FILTER_OP_EXT_IN_IP_PROTO_OFST 66
#define       MC_CMD_FILTER_OP_EXT_IN_IP_PROTO_LEN 2
/* Firmware defined register 0 to match (reserved; set to 0) */
#define       MC_CMD_FILTER_OP_EXT_IN_FWDEF0_OFST 68
#define       MC_CMD_FILTER_OP_EXT_IN_FWDEF0_LEN 4
/* VNI (for VXLAN/Geneve, when IP protocol is UDP) or VSID (for NVGRE, when IP
 * protocol is GRE) to match (as bytes in network order; set last byte to 0 for
 * VXLAN/NVGRE, or 1 for Geneve)
 */
#define       MC_CMD_FILTER_OP_EXT_IN_VNI_OR_VSID_OFST 72
#define       MC_CMD_FILTER_OP_EXT_IN_VNI_OR_VSID_LEN 4
#define        MC_CMD_FILTER_OP_EXT_IN_VNI_VALUE_OFST 72
#define        MC_CMD_FILTER_OP_EXT_IN_VNI_VALUE_LBN 0
#define        MC_CMD_FILTER_OP_EXT_IN_VNI_VALUE_WIDTH 24
#define        MC_CMD_FILTER_OP_EXT_IN_VNI_TYPE_OFST 72
#define        MC_CMD_FILTER_OP_EXT_IN_VNI_TYPE_LBN 24
#define        MC_CMD_FILTER_OP_EXT_IN_VNI_TYPE_WIDTH 8
/* enum: Match VXLAN traffic with this VNI */
#define          MC_CMD_FILTER_OP_EXT_IN_VNI_TYPE_VXLAN 0x0
/* enum: Match Geneve traffic with this VNI */
#define          MC_CMD_FILTER_OP_EXT_IN_VNI_TYPE_GENEVE 0x1
/* enum: Reserved for experimental development use */
#define          MC_CMD_FILTER_OP_EXT_IN_VNI_TYPE_EXPERIMENTAL 0xfe
#define        MC_CMD_FILTER_OP_EXT_IN_VSID_VALUE_OFST 72
#define        MC_CMD_FILTER_OP_EXT_IN_VSID_VALUE_LBN 0
#define        MC_CMD_FILTER_OP_EXT_IN_VSID_VALUE_WIDTH 24
#define        MC_CMD_FILTER_OP_EXT_IN_VSID_TYPE_OFST 72
#define        MC_CMD_FILTER_OP_EXT_IN_VSID_TYPE_LBN 24
#define        MC_CMD_FILTER_OP_EXT_IN_VSID_TYPE_WIDTH 8
/* enum: Match NVGRE traffic with this VSID */
#define          MC_CMD_FILTER_OP_EXT_IN_VSID_TYPE_NVGRE 0x0
/* source IP address to match (as bytes in network order; set last 12 bytes to
 * 0 for IPv4 address)
 */
#define       MC_CMD_FILTER_OP_EXT_IN_SRC_IP_OFST 76
#define       MC_CMD_FILTER_OP_EXT_IN_SRC_IP_LEN 16
/* destination IP address to match (as bytes in network order; set last 12
 * bytes to 0 for IPv4 address)
 */
#define       MC_CMD_FILTER_OP_EXT_IN_DST_IP_OFST 92
#define       MC_CMD_FILTER_OP_EXT_IN_DST_IP_LEN 16
/* VXLAN/NVGRE inner frame source MAC address to match (as bytes in network
 * order)
 */
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_SRC_MAC_OFST 108
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_SRC_MAC_LEN 6
/* VXLAN/NVGRE inner frame source port to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_SRC_PORT_OFST 114
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_SRC_PORT_LEN 2
/* VXLAN/NVGRE inner frame destination MAC address to match (as bytes in
 * network order)
 */
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_DST_MAC_OFST 116
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_DST_MAC_LEN 6
/* VXLAN/NVGRE inner frame destination port to match (as bytes in network
 * order)
 */
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_DST_PORT_OFST 122
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_DST_PORT_LEN 2
/* VXLAN/NVGRE inner frame Ethernet type to match (as bytes in network order)
 */
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_ETHER_TYPE_OFST 124
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_ETHER_TYPE_LEN 2
/* VXLAN/NVGRE inner frame Inner VLAN tag to match (as bytes in network order)
 */
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_INNER_VLAN_OFST 126
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_INNER_VLAN_LEN 2
/* VXLAN/NVGRE inner frame Outer VLAN tag to match (as bytes in network order)
 */
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_OUTER_VLAN_OFST 128
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_OUTER_VLAN_LEN 2
/* VXLAN/NVGRE inner frame IP protocol to match (in low byte; set high byte to
 * 0)
 */
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_IP_PROTO_OFST 130
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_IP_PROTO_LEN 2
/* VXLAN/NVGRE inner frame Firmware defined register 0 to match (reserved; set
 * to 0)
 */
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_FWDEF0_OFST 132
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_FWDEF0_LEN 4
/* VXLAN/NVGRE inner frame Firmware defined register 1 to match (reserved; set
 * to 0)
 */
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_FWDEF1_OFST 136
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_FWDEF1_LEN 4
/* VXLAN/NVGRE inner frame source IP address to match (as bytes in network
 * order; set last 12 bytes to 0 for IPv4 address)
 */
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_SRC_IP_OFST 140
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_SRC_IP_LEN 16
/* VXLAN/NVGRE inner frame destination IP address to match (as bytes in network
 * order; set last 12 bytes to 0 for IPv4 address)
 */
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_DST_IP_OFST 156
#define       MC_CMD_FILTER_OP_EXT_IN_IFRM_DST_IP_LEN 16

/* MC_CMD_FILTER_OP_V3_IN msgrequest: FILTER_OP extension to support additional
 * filter actions for EF100. Some of these actions are also supported on EF10,
 * for Intel's DPDK (Data Plane Development Kit, dpdk.org) via its rte_flow
 * API. In the latter case, this extension is only useful with the sfc_efx
 * driver included as part of DPDK, used in conjunction with the dpdk datapath
 * firmware variant.
 */
#define    MC_CMD_FILTER_OP_V3_IN_LEN 180
/* identifies the type of operation requested */
#define       MC_CMD_FILTER_OP_V3_IN_OP_OFST 0
#define       MC_CMD_FILTER_OP_V3_IN_OP_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_FILTER_OP_IN/OP */
/* filter handle (for remove / unsubscribe operations) */
#define       MC_CMD_FILTER_OP_V3_IN_HANDLE_OFST 4
#define       MC_CMD_FILTER_OP_V3_IN_HANDLE_LEN 8
#define       MC_CMD_FILTER_OP_V3_IN_HANDLE_LO_OFST 4
#define       MC_CMD_FILTER_OP_V3_IN_HANDLE_LO_LEN 4
#define       MC_CMD_FILTER_OP_V3_IN_HANDLE_LO_LBN 32
#define       MC_CMD_FILTER_OP_V3_IN_HANDLE_LO_WIDTH 32
#define       MC_CMD_FILTER_OP_V3_IN_HANDLE_HI_OFST 8
#define       MC_CMD_FILTER_OP_V3_IN_HANDLE_HI_LEN 4
#define       MC_CMD_FILTER_OP_V3_IN_HANDLE_HI_LBN 64
#define       MC_CMD_FILTER_OP_V3_IN_HANDLE_HI_WIDTH 32
/* The port ID associated with the v-adaptor which should contain this filter.
 */
#define       MC_CMD_FILTER_OP_V3_IN_PORT_ID_OFST 12
#define       MC_CMD_FILTER_OP_V3_IN_PORT_ID_LEN 4
/* fields to include in match criteria */
#define       MC_CMD_FILTER_OP_V3_IN_MATCH_FIELDS_OFST 16
#define       MC_CMD_FILTER_OP_V3_IN_MATCH_FIELDS_LEN 4
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_SRC_IP_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_SRC_IP_LBN 0
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_SRC_IP_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_DST_IP_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_DST_IP_LBN 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_DST_IP_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_SRC_MAC_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_SRC_MAC_LBN 2
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_SRC_MAC_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_SRC_PORT_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_SRC_PORT_LBN 3
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_SRC_PORT_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_DST_MAC_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_DST_MAC_LBN 4
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_DST_MAC_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_DST_PORT_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_DST_PORT_LBN 5
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_DST_PORT_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_ETHER_TYPE_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_ETHER_TYPE_LBN 6
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_ETHER_TYPE_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_INNER_VLAN_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_INNER_VLAN_LBN 7
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_INNER_VLAN_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_OUTER_VLAN_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_OUTER_VLAN_LBN 8
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_OUTER_VLAN_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IP_PROTO_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IP_PROTO_LBN 9
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IP_PROTO_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_FWDEF0_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_FWDEF0_LBN 10
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_FWDEF0_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_VNI_OR_VSID_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_VNI_OR_VSID_LBN 11
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_VNI_OR_VSID_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_SRC_IP_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_SRC_IP_LBN 12
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_SRC_IP_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_DST_IP_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_DST_IP_LBN 13
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_DST_IP_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_SRC_MAC_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_SRC_MAC_LBN 14
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_SRC_MAC_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_SRC_PORT_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_SRC_PORT_LBN 15
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_SRC_PORT_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_DST_MAC_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_DST_MAC_LBN 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_DST_MAC_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_DST_PORT_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_DST_PORT_LBN 17
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_DST_PORT_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_ETHER_TYPE_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_ETHER_TYPE_LBN 18
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_ETHER_TYPE_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_INNER_VLAN_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_INNER_VLAN_LBN 19
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_INNER_VLAN_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_OUTER_VLAN_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_OUTER_VLAN_LBN 20
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_OUTER_VLAN_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_IP_PROTO_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_IP_PROTO_LBN 21
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_IP_PROTO_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_FWDEF0_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_FWDEF0_LBN 22
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_FWDEF0_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_FWDEF1_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_FWDEF1_LBN 23
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_FWDEF1_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_UNKNOWN_MCAST_DST_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_UNKNOWN_MCAST_DST_LBN 24
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_UNKNOWN_MCAST_DST_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_UNKNOWN_UCAST_DST_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_UNKNOWN_UCAST_DST_LBN 25
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_IFRM_UNKNOWN_UCAST_DST_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_UNKNOWN_IPV4_MCAST_DST_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_UNKNOWN_IPV4_MCAST_DST_LBN 29
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_UNKNOWN_IPV4_MCAST_DST_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_UNKNOWN_MCAST_DST_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_UNKNOWN_MCAST_DST_LBN 30
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_UNKNOWN_MCAST_DST_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_UNKNOWN_UCAST_DST_OFST 16
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_UNKNOWN_UCAST_DST_LBN 31
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_UNKNOWN_UCAST_DST_WIDTH 1
/* receive destination */
#define       MC_CMD_FILTER_OP_V3_IN_RX_DEST_OFST 20
#define       MC_CMD_FILTER_OP_V3_IN_RX_DEST_LEN 4
/* enum: drop packets */
#define          MC_CMD_FILTER_OP_V3_IN_RX_DEST_DROP 0x0
/* enum: receive to host */
#define          MC_CMD_FILTER_OP_V3_IN_RX_DEST_HOST 0x1
/* enum: receive to MC */
#define          MC_CMD_FILTER_OP_V3_IN_RX_DEST_MC 0x2
/* enum: loop back to TXDP 0 */
#define          MC_CMD_FILTER_OP_V3_IN_RX_DEST_TX0 0x3
/* enum: loop back to TXDP 1 */
#define          MC_CMD_FILTER_OP_V3_IN_RX_DEST_TX1 0x4
/* receive queue handle (for multiple queue modes, this is the base queue) */
#define       MC_CMD_FILTER_OP_V3_IN_RX_QUEUE_OFST 24
#define       MC_CMD_FILTER_OP_V3_IN_RX_QUEUE_LEN 4
/* receive mode */
#define       MC_CMD_FILTER_OP_V3_IN_RX_MODE_OFST 28
#define       MC_CMD_FILTER_OP_V3_IN_RX_MODE_LEN 4
/* enum: receive to just the specified queue */
#define          MC_CMD_FILTER_OP_V3_IN_RX_MODE_SIMPLE 0x0
/* enum: receive to multiple queues using RSS context */
#define          MC_CMD_FILTER_OP_V3_IN_RX_MODE_RSS 0x1
/* enum: receive to multiple queues using .1p mapping */
#define          MC_CMD_FILTER_OP_V3_IN_RX_MODE_DOT1P_MAPPING 0x2
/* enum: install a filter entry that will never match; for test purposes only
 */
#define          MC_CMD_FILTER_OP_V3_IN_RX_MODE_TEST_NEVER_MATCH 0x80000000
/* RSS context (for RX_MODE_RSS) or .1p mapping handle (for
 * RX_MODE_DOT1P_MAPPING), as returned by MC_CMD_RSS_CONTEXT_ALLOC or
 * MC_CMD_DOT1P_MAPPING_ALLOC.
 */
#define       MC_CMD_FILTER_OP_V3_IN_RX_CONTEXT_OFST 32
#define       MC_CMD_FILTER_OP_V3_IN_RX_CONTEXT_LEN 4
/* transmit domain (reserved; set to 0) */
#define       MC_CMD_FILTER_OP_V3_IN_TX_DOMAIN_OFST 36
#define       MC_CMD_FILTER_OP_V3_IN_TX_DOMAIN_LEN 4
/* transmit destination (either set the MAC and/or PM bits for explicit
 * control, or set this field to TX_DEST_DEFAULT for sensible default
 * behaviour)
 */
#define       MC_CMD_FILTER_OP_V3_IN_TX_DEST_OFST 40
#define       MC_CMD_FILTER_OP_V3_IN_TX_DEST_LEN 4
/* enum: request default behaviour (based on filter type) */
#define          MC_CMD_FILTER_OP_V3_IN_TX_DEST_DEFAULT 0xffffffff
#define        MC_CMD_FILTER_OP_V3_IN_TX_DEST_MAC_OFST 40
#define        MC_CMD_FILTER_OP_V3_IN_TX_DEST_MAC_LBN 0
#define        MC_CMD_FILTER_OP_V3_IN_TX_DEST_MAC_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_TX_DEST_PM_OFST 40
#define        MC_CMD_FILTER_OP_V3_IN_TX_DEST_PM_LBN 1
#define        MC_CMD_FILTER_OP_V3_IN_TX_DEST_PM_WIDTH 1
/* source MAC address to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_V3_IN_SRC_MAC_OFST 44
#define       MC_CMD_FILTER_OP_V3_IN_SRC_MAC_LEN 6
/* source port to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_V3_IN_SRC_PORT_OFST 50
#define       MC_CMD_FILTER_OP_V3_IN_SRC_PORT_LEN 2
/* destination MAC address to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_V3_IN_DST_MAC_OFST 52
#define       MC_CMD_FILTER_OP_V3_IN_DST_MAC_LEN 6
/* destination port to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_V3_IN_DST_PORT_OFST 58
#define       MC_CMD_FILTER_OP_V3_IN_DST_PORT_LEN 2
/* Ethernet type to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_V3_IN_ETHER_TYPE_OFST 60
#define       MC_CMD_FILTER_OP_V3_IN_ETHER_TYPE_LEN 2
/* Inner VLAN tag to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_V3_IN_INNER_VLAN_OFST 62
#define       MC_CMD_FILTER_OP_V3_IN_INNER_VLAN_LEN 2
/* Outer VLAN tag to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_V3_IN_OUTER_VLAN_OFST 64
#define       MC_CMD_FILTER_OP_V3_IN_OUTER_VLAN_LEN 2
/* IP protocol to match (in low byte; set high byte to 0) */
#define       MC_CMD_FILTER_OP_V3_IN_IP_PROTO_OFST 66
#define       MC_CMD_FILTER_OP_V3_IN_IP_PROTO_LEN 2
/* Firmware defined register 0 to match (reserved; set to 0) */
#define       MC_CMD_FILTER_OP_V3_IN_FWDEF0_OFST 68
#define       MC_CMD_FILTER_OP_V3_IN_FWDEF0_LEN 4
/* VNI (for VXLAN/Geneve, when IP protocol is UDP) or VSID (for NVGRE, when IP
 * protocol is GRE) to match (as bytes in network order; set last byte to 0 for
 * VXLAN/NVGRE, or 1 for Geneve)
 */
#define       MC_CMD_FILTER_OP_V3_IN_VNI_OR_VSID_OFST 72
#define       MC_CMD_FILTER_OP_V3_IN_VNI_OR_VSID_LEN 4
#define        MC_CMD_FILTER_OP_V3_IN_VNI_VALUE_OFST 72
#define        MC_CMD_FILTER_OP_V3_IN_VNI_VALUE_LBN 0
#define        MC_CMD_FILTER_OP_V3_IN_VNI_VALUE_WIDTH 24
#define        MC_CMD_FILTER_OP_V3_IN_VNI_TYPE_OFST 72
#define        MC_CMD_FILTER_OP_V3_IN_VNI_TYPE_LBN 24
#define        MC_CMD_FILTER_OP_V3_IN_VNI_TYPE_WIDTH 8
/* enum: Match VXLAN traffic with this VNI */
#define          MC_CMD_FILTER_OP_V3_IN_VNI_TYPE_VXLAN 0x0
/* enum: Match Geneve traffic with this VNI */
#define          MC_CMD_FILTER_OP_V3_IN_VNI_TYPE_GENEVE 0x1
/* enum: Reserved for experimental development use */
#define          MC_CMD_FILTER_OP_V3_IN_VNI_TYPE_EXPERIMENTAL 0xfe
#define        MC_CMD_FILTER_OP_V3_IN_VSID_VALUE_OFST 72
#define        MC_CMD_FILTER_OP_V3_IN_VSID_VALUE_LBN 0
#define        MC_CMD_FILTER_OP_V3_IN_VSID_VALUE_WIDTH 24
#define        MC_CMD_FILTER_OP_V3_IN_VSID_TYPE_OFST 72
#define        MC_CMD_FILTER_OP_V3_IN_VSID_TYPE_LBN 24
#define        MC_CMD_FILTER_OP_V3_IN_VSID_TYPE_WIDTH 8
/* enum: Match NVGRE traffic with this VSID */
#define          MC_CMD_FILTER_OP_V3_IN_VSID_TYPE_NVGRE 0x0
/* source IP address to match (as bytes in network order; set last 12 bytes to
 * 0 for IPv4 address)
 */
#define       MC_CMD_FILTER_OP_V3_IN_SRC_IP_OFST 76
#define       MC_CMD_FILTER_OP_V3_IN_SRC_IP_LEN 16
/* destination IP address to match (as bytes in network order; set last 12
 * bytes to 0 for IPv4 address)
 */
#define       MC_CMD_FILTER_OP_V3_IN_DST_IP_OFST 92
#define       MC_CMD_FILTER_OP_V3_IN_DST_IP_LEN 16
/* VXLAN/NVGRE inner frame source MAC address to match (as bytes in network
 * order)
 */
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_SRC_MAC_OFST 108
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_SRC_MAC_LEN 6
/* VXLAN/NVGRE inner frame source port to match (as bytes in network order) */
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_SRC_PORT_OFST 114
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_SRC_PORT_LEN 2
/* VXLAN/NVGRE inner frame destination MAC address to match (as bytes in
 * network order)
 */
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_DST_MAC_OFST 116
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_DST_MAC_LEN 6
/* VXLAN/NVGRE inner frame destination port to match (as bytes in network
 * order)
 */
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_DST_PORT_OFST 122
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_DST_PORT_LEN 2
/* VXLAN/NVGRE inner frame Ethernet type to match (as bytes in network order)
 */
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_ETHER_TYPE_OFST 124
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_ETHER_TYPE_LEN 2
/* VXLAN/NVGRE inner frame Inner VLAN tag to match (as bytes in network order)
 */
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_INNER_VLAN_OFST 126
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_INNER_VLAN_LEN 2
/* VXLAN/NVGRE inner frame Outer VLAN tag to match (as bytes in network order)
 */
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_OUTER_VLAN_OFST 128
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_OUTER_VLAN_LEN 2
/* VXLAN/NVGRE inner frame IP protocol to match (in low byte; set high byte to
 * 0)
 */
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_IP_PROTO_OFST 130
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_IP_PROTO_LEN 2
/* VXLAN/NVGRE inner frame Firmware defined register 0 to match (reserved; set
 * to 0)
 */
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_FWDEF0_OFST 132
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_FWDEF0_LEN 4
/* VXLAN/NVGRE inner frame Firmware defined register 1 to match (reserved; set
 * to 0)
 */
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_FWDEF1_OFST 136
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_FWDEF1_LEN 4
/* VXLAN/NVGRE inner frame source IP address to match (as bytes in network
 * order; set last 12 bytes to 0 for IPv4 address)
 */
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_SRC_IP_OFST 140
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_SRC_IP_LEN 16
/* VXLAN/NVGRE inner frame destination IP address to match (as bytes in network
 * order; set last 12 bytes to 0 for IPv4 address)
 */
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_DST_IP_OFST 156
#define       MC_CMD_FILTER_OP_V3_IN_IFRM_DST_IP_LEN 16
/* Flags controlling mutations of the packet and/or metadata when the filter is
 * matched. The user_mark and user_flag fields' logic is as follows: if
 * (req.MATCH_BITOR_FLAG == 1) user_flag = req.MATCH_SET_FLAG bit_or user_flag;
 * else user_flag = req.MATCH_SET_FLAG; if (req.MATCH_SET_MARK == 0) user_mark
 * = 0; else if (req.MATCH_BITOR_MARK == 1) user_mark = req.MATCH_SET_MARK
 * bit_or user_mark; else user_mark = req.MATCH_SET_MARK; N.B. These flags
 * overlap with the MATCH_ACTION field, which is deprecated in favour of this
 * field. For the cases where these flags induce a valid encoding of the
 * MATCH_ACTION field, the semantics agree.
 */
#define       MC_CMD_FILTER_OP_V3_IN_MATCH_ACTION_FLAGS_OFST 172
#define       MC_CMD_FILTER_OP_V3_IN_MATCH_ACTION_FLAGS_LEN 4
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_SET_FLAG_OFST 172
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_SET_FLAG_LBN 0
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_SET_FLAG_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_SET_MARK_OFST 172
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_SET_MARK_LBN 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_SET_MARK_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_BITOR_FLAG_OFST 172
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_BITOR_FLAG_LBN 2
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_BITOR_FLAG_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_BITOR_MARK_OFST 172
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_BITOR_MARK_LBN 3
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_BITOR_MARK_WIDTH 1
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_STRIP_VLAN_OFST 172
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_STRIP_VLAN_LBN 4
#define        MC_CMD_FILTER_OP_V3_IN_MATCH_STRIP_VLAN_WIDTH 1
/* Deprecated: the overlapping MATCH_ACTION_FLAGS field exposes all of the
 * functionality of this field in an ABI-backwards-compatible manner, and
 * should be used instead. Any future extensions should be made to the
 * MATCH_ACTION_FLAGS field, and not to this field. Set an action for all
 * packets matching this filter. The DPDK driver and (on EF10) dpdk f/w variant
 * use their own specific delivery structures, which are documented in the DPDK
 * Firmware Driver Interface (SF-119419-TC). Requesting anything other than
 * MATCH_ACTION_NONE on an EF10 NIC running another f/w variant will cause the
 * filter insertion to fail with ENOTSUP.
 */
#define       MC_CMD_FILTER_OP_V3_IN_MATCH_ACTION_OFST 172
#define       MC_CMD_FILTER_OP_V3_IN_MATCH_ACTION_LEN 4
/* enum: do nothing extra */
#define          MC_CMD_FILTER_OP_V3_IN_MATCH_ACTION_NONE 0x0
/* enum: Set the match flag in the packet prefix for packets matching the
 * filter (only with dpdk firmware, otherwise fails with ENOTSUP). Used to
 * support the DPDK rte_flow "FLAG" action.
 */
#define          MC_CMD_FILTER_OP_V3_IN_MATCH_ACTION_FLAG 0x1
/* enum: Insert MATCH_MARK_VALUE into the packet prefix for packets matching
 * the filter (only with dpdk firmware, otherwise fails with ENOTSUP). Used to
 * support the DPDK rte_flow "MARK" action.
 */
#define          MC_CMD_FILTER_OP_V3_IN_MATCH_ACTION_MARK 0x2
/* the mark value for MATCH_ACTION_MARK. Requesting a value larger than the
 * maximum (obtained from MC_CMD_GET_CAPABILITIES_V5/FILTER_ACTION_MARK_MAX)
 * will cause the filter insertion to fail with EINVAL.
 */
#define       MC_CMD_FILTER_OP_V3_IN_MATCH_MARK_VALUE_OFST 176
#define       MC_CMD_FILTER_OP_V3_IN_MATCH_MARK_VALUE_LEN 4

/* MC_CMD_FILTER_OP_OUT msgresponse */
#define    MC_CMD_FILTER_OP_OUT_LEN 12
/* identifies the type of operation requested */
#define       MC_CMD_FILTER_OP_OUT_OP_OFST 0
#define       MC_CMD_FILTER_OP_OUT_OP_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_FILTER_OP_IN/OP */
/* Returned filter handle (for insert / subscribe operations). Note that these
 * handles should be considered opaque to the host, although a value of
 * 0xFFFFFFFF_FFFFFFFF is guaranteed never to be a valid handle.
 */
#define       MC_CMD_FILTER_OP_OUT_HANDLE_OFST 4
#define       MC_CMD_FILTER_OP_OUT_HANDLE_LEN 8
#define       MC_CMD_FILTER_OP_OUT_HANDLE_LO_OFST 4
#define       MC_CMD_FILTER_OP_OUT_HANDLE_LO_LEN 4
#define       MC_CMD_FILTER_OP_OUT_HANDLE_LO_LBN 32
#define       MC_CMD_FILTER_OP_OUT_HANDLE_LO_WIDTH 32
#define       MC_CMD_FILTER_OP_OUT_HANDLE_HI_OFST 8
#define       MC_CMD_FILTER_OP_OUT_HANDLE_HI_LEN 4
#define       MC_CMD_FILTER_OP_OUT_HANDLE_HI_LBN 64
#define       MC_CMD_FILTER_OP_OUT_HANDLE_HI_WIDTH 32
/* enum: guaranteed invalid filter handle (low 32 bits) */
#define          MC_CMD_FILTER_OP_OUT_HANDLE_LO_INVALID 0xffffffff
/* enum: guaranteed invalid filter handle (high 32 bits) */
#define          MC_CMD_FILTER_OP_OUT_HANDLE_HI_INVALID 0xffffffff

/* MC_CMD_FILTER_OP_EXT_OUT msgresponse */
#define    MC_CMD_FILTER_OP_EXT_OUT_LEN 12
/* identifies the type of operation requested */
#define       MC_CMD_FILTER_OP_EXT_OUT_OP_OFST 0
#define       MC_CMD_FILTER_OP_EXT_OUT_OP_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_FILTER_OP_EXT_IN/OP */
/* Returned filter handle (for insert / subscribe operations). Note that these
 * handles should be considered opaque to the host, although a value of
 * 0xFFFFFFFF_FFFFFFFF is guaranteed never to be a valid handle.
 */
#define       MC_CMD_FILTER_OP_EXT_OUT_HANDLE_OFST 4
#define       MC_CMD_FILTER_OP_EXT_OUT_HANDLE_LEN 8
#define       MC_CMD_FILTER_OP_EXT_OUT_HANDLE_LO_OFST 4
#define       MC_CMD_FILTER_OP_EXT_OUT_HANDLE_LO_LEN 4
#define       MC_CMD_FILTER_OP_EXT_OUT_HANDLE_LO_LBN 32
#define       MC_CMD_FILTER_OP_EXT_OUT_HANDLE_LO_WIDTH 32
#define       MC_CMD_FILTER_OP_EXT_OUT_HANDLE_HI_OFST 8
#define       MC_CMD_FILTER_OP_EXT_OUT_HANDLE_HI_LEN 4
#define       MC_CMD_FILTER_OP_EXT_OUT_HANDLE_HI_LBN 64
#define       MC_CMD_FILTER_OP_EXT_OUT_HANDLE_HI_WIDTH 32
/*            Enum values, see field(s): */
/*               MC_CMD_FILTER_OP_OUT/HANDLE */


/***********************************/
/* MC_CMD_GET_PARSER_DISP_INFO
 * Get information related to the parser-dispatcher subsystem
 */
#define MC_CMD_GET_PARSER_DISP_INFO 0xe4
#undef MC_CMD_0xe4_PRIVILEGE_CTG

#define MC_CMD_0xe4_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_GET_PARSER_DISP_INFO_IN msgrequest */
#define    MC_CMD_GET_PARSER_DISP_INFO_IN_LEN 4
/* identifies the type of operation requested */
#define       MC_CMD_GET_PARSER_DISP_INFO_IN_OP_OFST 0
#define       MC_CMD_GET_PARSER_DISP_INFO_IN_OP_LEN 4
/* enum: read the list of supported RX filter matches */
#define          MC_CMD_GET_PARSER_DISP_INFO_IN_OP_GET_SUPPORTED_RX_MATCHES 0x1
/* enum: read flags indicating restrictions on filter insertion for the calling
 * client
 */
#define          MC_CMD_GET_PARSER_DISP_INFO_IN_OP_GET_RESTRICTIONS 0x2
/* enum: read properties relating to security rules (Medford-only; for use by
 * SolarSecure apps, not directly by drivers. See SF-114946-SW.)
 */
#define          MC_CMD_GET_PARSER_DISP_INFO_IN_OP_GET_SECURITY_RULE_INFO 0x3
/* enum: read the list of supported RX filter matches for VXLAN/NVGRE
 * encapsulated frames, which follow a different match sequence to normal
 * frames (Medford only)
 */
#define          MC_CMD_GET_PARSER_DISP_INFO_IN_OP_GET_SUPPORTED_ENCAP_RX_MATCHES 0x4
/* enum: read the list of supported matches for the encapsulation detection
 * rules inserted by MC_CMD_VNIC_ENCAP_RULE_ADD. (ef100 and later)
 */
#define          MC_CMD_GET_PARSER_DISP_INFO_IN_OP_GET_SUPPORTED_VNIC_ENCAP_MATCHES 0x5
/* enum: read the supported encapsulation types for the VNIC */
#define          MC_CMD_GET_PARSER_DISP_INFO_IN_OP_GET_SUPPORTED_VNIC_ENCAP_TYPES 0x6
/* enum: read the supported RX filter matches for low-latency queues (as
 * allocated by MC_CMD_ALLOC_LL_QUEUES)
 */
#define          MC_CMD_GET_PARSER_DISP_INFO_IN_OP_GET_SUPPORTED_LL_RX_MATCHES 0x7

/* MC_CMD_GET_PARSER_DISP_INFO_OUT msgresponse */
#define    MC_CMD_GET_PARSER_DISP_INFO_OUT_LENMIN 8
#define    MC_CMD_GET_PARSER_DISP_INFO_OUT_LENMAX 252
#define    MC_CMD_GET_PARSER_DISP_INFO_OUT_LENMAX_MCDI2 1020
#define    MC_CMD_GET_PARSER_DISP_INFO_OUT_LEN(num) (8+4*(num))
#define    MC_CMD_GET_PARSER_DISP_INFO_OUT_SUPPORTED_MATCHES_NUM(len) (((len)-8)/4)
/* identifies the type of operation requested */
#define       MC_CMD_GET_PARSER_DISP_INFO_OUT_OP_OFST 0
#define       MC_CMD_GET_PARSER_DISP_INFO_OUT_OP_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_GET_PARSER_DISP_INFO_IN/OP */
/* number of supported match types */
#define       MC_CMD_GET_PARSER_DISP_INFO_OUT_NUM_SUPPORTED_MATCHES_OFST 4
#define       MC_CMD_GET_PARSER_DISP_INFO_OUT_NUM_SUPPORTED_MATCHES_LEN 4
/* array of supported match types (valid MATCH_FIELDS values for
 * MC_CMD_FILTER_OP) sorted in decreasing priority order
 */
#define       MC_CMD_GET_PARSER_DISP_INFO_OUT_SUPPORTED_MATCHES_OFST 8
#define       MC_CMD_GET_PARSER_DISP_INFO_OUT_SUPPORTED_MATCHES_LEN 4
#define       MC_CMD_GET_PARSER_DISP_INFO_OUT_SUPPORTED_MATCHES_MINNUM 0
#define       MC_CMD_GET_PARSER_DISP_INFO_OUT_SUPPORTED_MATCHES_MAXNUM 61
#define       MC_CMD_GET_PARSER_DISP_INFO_OUT_SUPPORTED_MATCHES_MAXNUM_MCDI2 253

/* MC_CMD_GET_PARSER_DISP_RESTRICTIONS_OUT msgresponse */
#define    MC_CMD_GET_PARSER_DISP_RESTRICTIONS_OUT_LEN 8
/* identifies the type of operation requested */
#define       MC_CMD_GET_PARSER_DISP_RESTRICTIONS_OUT_OP_OFST 0
#define       MC_CMD_GET_PARSER_DISP_RESTRICTIONS_OUT_OP_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_GET_PARSER_DISP_INFO_IN/OP */
/* bitfield of filter insertion restrictions */
#define       MC_CMD_GET_PARSER_DISP_RESTRICTIONS_OUT_RESTRICTION_FLAGS_OFST 4
#define       MC_CMD_GET_PARSER_DISP_RESTRICTIONS_OUT_RESTRICTION_FLAGS_LEN 4
#define        MC_CMD_GET_PARSER_DISP_RESTRICTIONS_OUT_DST_IP_MCAST_ONLY_OFST 4
#define        MC_CMD_GET_PARSER_DISP_RESTRICTIONS_OUT_DST_IP_MCAST_ONLY_LBN 0
#define        MC_CMD_GET_PARSER_DISP_RESTRICTIONS_OUT_DST_IP_MCAST_ONLY_WIDTH 1

/* MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT msgresponse:
 * GET_PARSER_DISP_INFO response format for OP_GET_SECURITY_RULE_INFO.
 * (Medford-only; for use by SolarSecure apps, not directly by drivers. See
 * SF-114946-SW.) NOTE - this message definition is provisional. It has not yet
 * been used in any released code and may change during development. This note
 * will be removed once it is regarded as stable.
 */
#define    MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_LEN 36
/* identifies the type of operation requested */
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_OP_OFST 0
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_OP_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_GET_PARSER_DISP_INFO_IN/OP */
/* a version number representing the set of rule lookups that are implemented
 * by the currently running firmware
 */
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_RULES_VERSION_OFST 4
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_RULES_VERSION_LEN 4
/* enum: implements lookup sequences described in SF-114946-SW draft C */
#define          MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_RULES_VERSION_SF_114946_SW_C 0x0
/* the number of nodes in the subnet map */
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_SUBNET_MAP_NUM_NODES_OFST 8
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_SUBNET_MAP_NUM_NODES_LEN 4
/* the number of entries in one subnet map node */
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_SUBNET_MAP_NUM_ENTRIES_PER_NODE_OFST 12
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_SUBNET_MAP_NUM_ENTRIES_PER_NODE_LEN 4
/* minimum valid value for a subnet ID in a subnet map leaf */
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_SUBNET_ID_MIN_OFST 16
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_SUBNET_ID_MIN_LEN 4
/* maximum valid value for a subnet ID in a subnet map leaf */
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_SUBNET_ID_MAX_OFST 20
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_SUBNET_ID_MAX_LEN 4
/* the number of entries in the local and remote port range maps */
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_PORTRANGE_TREE_NUM_ENTRIES_OFST 24
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_PORTRANGE_TREE_NUM_ENTRIES_LEN 4
/* minimum valid value for a portrange ID in a port range map leaf */
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_PORTRANGE_ID_MIN_OFST 28
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_PORTRANGE_ID_MIN_LEN 4
/* maximum valid value for a portrange ID in a port range map leaf */
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_PORTRANGE_ID_MAX_OFST 32
#define       MC_CMD_GET_PARSER_DISP_SECURITY_RULE_INFO_OUT_PORTRANGE_ID_MAX_LEN 4

/* MC_CMD_GET_PARSER_DISP_VNIC_ENCAP_MATCHES_OUT msgresponse: This response is
 * returned if a MC_CMD_GET_PARSER_DISP_INFO_IN request is sent with OP value
 * OP_GET_SUPPORTED_VNIC_ENCAP_MATCHES. It contains information about the
 * supported match types that can be used in the encapsulation detection rules
 * inserted by MC_CMD_VNIC_ENCAP_RULE_ADD.
 */
#define    MC_CMD_GET_PARSER_DISP_VNIC_ENCAP_MATCHES_OUT_LENMIN 8
#define    MC_CMD_GET_PARSER_DISP_VNIC_ENCAP_MATCHES_OUT_LENMAX 252
#define    MC_CMD_GET_PARSER_DISP_VNIC_ENCAP_MATCHES_OUT_LENMAX_MCDI2 1020
#define    MC_CMD_GET_PARSER_DISP_VNIC_ENCAP_MATCHES_OUT_LEN(num) (8+4*(num))
#define    MC_CMD_GET_PARSER_DISP_VNIC_ENCAP_MATCHES_OUT_SUPPORTED_MATCHES_NUM(len) (((len)-8)/4)
/* The op code OP_GET_SUPPORTED_VNIC_ENCAP_MATCHES is returned. */
#define       MC_CMD_GET_PARSER_DISP_VNIC_ENCAP_MATCHES_OUT_OP_OFST 0
#define       MC_CMD_GET_PARSER_DISP_VNIC_ENCAP_MATCHES_OUT_OP_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_GET_PARSER_DISP_INFO_IN/OP */
/* number of supported match types */
#define       MC_CMD_GET_PARSER_DISP_VNIC_ENCAP_MATCHES_OUT_NUM_SUPPORTED_MATCHES_OFST 4
#define       MC_CMD_GET_PARSER_DISP_VNIC_ENCAP_MATCHES_OUT_NUM_SUPPORTED_MATCHES_LEN 4
/* array of supported match types (valid MATCH_FLAGS values for
 * MC_CMD_VNIC_ENCAP_RULE_ADD) sorted in decreasing priority order
 */
#define       MC_CMD_GET_PARSER_DISP_VNIC_ENCAP_MATCHES_OUT_SUPPORTED_MATCHES_OFST 8
#define       MC_CMD_GET_PARSER_DISP_VNIC_ENCAP_MATCHES_OUT_SUPPORTED_MATCHES_LEN 4
#define       MC_CMD_GET_PARSER_DISP_VNIC_ENCAP_MATCHES_OUT_SUPPORTED_MATCHES_MINNUM 0
#define       MC_CMD_GET_PARSER_DISP_VNIC_ENCAP_MATCHES_OUT_SUPPORTED_MATCHES_MAXNUM 61
#define       MC_CMD_GET_PARSER_DISP_VNIC_ENCAP_MATCHES_OUT_SUPPORTED_MATCHES_MAXNUM_MCDI2 253

/* MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT msgresponse: Returns
 * the supported encapsulation types for the VNIC
 */
#define    MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_LEN 8
/* The op code OP_GET_SUPPORTED_VNIC_ENCAP_TYPES is returned */
#define       MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_OP_OFST 0
#define       MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_OP_LEN 4
/*            Enum values, see field(s): */
/*               MC_CMD_GET_PARSER_DISP_INFO_IN/OP */
#define       MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_ENCAP_TYPES_SUPPORTED_OFST 4
#define       MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_ENCAP_TYPES_SUPPORTED_LEN 4
#define        MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_ENCAP_TYPE_VXLAN_OFST 4
#define        MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_ENCAP_TYPE_VXLAN_LBN 0
#define        MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_ENCAP_TYPE_VXLAN_WIDTH 1
#define        MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_ENCAP_TYPE_NVGRE_OFST 4
#define        MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_ENCAP_TYPE_NVGRE_LBN 1
#define        MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_ENCAP_TYPE_NVGRE_WIDTH 1
#define        MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_ENCAP_TYPE_GENEVE_OFST 4
#define        MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_ENCAP_TYPE_GENEVE_LBN 2
#define        MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_ENCAP_TYPE_GENEVE_WIDTH 1
#define        MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_ENCAP_TYPE_L2GRE_OFST 4
#define        MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_ENCAP_TYPE_L2GRE_LBN 3
#define        MC_CMD_GET_PARSER_DISP_SUPPORTED_VNIC_ENCAP_TYPES_OUT_ENCAP_TYPE_L2GRE_WIDTH 1


/***********************************/
/* MC_CMD_GET_PORT_ASSIGNMENT
 * Get port assignment for current PCI function.
 */
#define MC_CMD_GET_PORT_ASSIGNMENT 0xb8
#undef MC_CMD_0xb8_PRIVILEGE_CTG

#define MC_CMD_0xb8_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_GET_PORT_ASSIGNMENT_IN msgrequest */
#define    MC_CMD_GET_PORT_ASSIGNMENT_IN_LEN 0

/* MC_CMD_GET_PORT_ASSIGNMENT_OUT msgresponse */
#define    MC_CMD_GET_PORT_ASSIGNMENT_OUT_LEN 4
/* Identifies the port assignment for this function. On EF100, it is possible
 * for the function to have no network port assigned (either because it is not
 * yet configured, or assigning a port to a given function personality makes no
 * sense - e.g. virtio-blk), in which case the return value is NULL_PORT.
 */
#define       MC_CMD_GET_PORT_ASSIGNMENT_OUT_PORT_OFST 0
#define       MC_CMD_GET_PORT_ASSIGNMENT_OUT_PORT_LEN 4
/* enum: Special value to indicate no port is assigned to a function. */
#define          MC_CMD_GET_PORT_ASSIGNMENT_OUT_NULL_PORT 0xffffffff


/***********************************/
/* MC_CMD_ALLOC_VIS
 * Allocate VIs for current PCI function.
 */
#define MC_CMD_ALLOC_VIS 0x8b
#undef MC_CMD_0x8b_PRIVILEGE_CTG

#define MC_CMD_0x8b_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_ALLOC_VIS_IN msgrequest */
#define    MC_CMD_ALLOC_VIS_IN_LEN 8
/* The minimum number of VIs that is acceptable */
#define       MC_CMD_ALLOC_VIS_IN_MIN_VI_COUNT_OFST 0
#define       MC_CMD_ALLOC_VIS_IN_MIN_VI_COUNT_LEN 4
/* The maximum number of VIs that would be useful */
#define       MC_CMD_ALLOC_VIS_IN_MAX_VI_COUNT_OFST 4
#define       MC_CMD_ALLOC_VIS_IN_MAX_VI_COUNT_LEN 4

/* MC_CMD_ALLOC_VIS_OUT msgresponse: Huntington-compatible VI_ALLOC request.
 * Use extended version in new code.
 */
#define    MC_CMD_ALLOC_VIS_OUT_LEN 8
/* The number of VIs allocated on this function */
#define       MC_CMD_ALLOC_VIS_OUT_VI_COUNT_OFST 0
#define       MC_CMD_ALLOC_VIS_OUT_VI_COUNT_LEN 4
/* The base absolute VI number allocated to this function. Required to
 * correctly interpret wakeup events.
 */
#define       MC_CMD_ALLOC_VIS_OUT_VI_BASE_OFST 4
#define       MC_CMD_ALLOC_VIS_OUT_VI_BASE_LEN 4

/* MC_CMD_ALLOC_VIS_EXT_OUT msgresponse */
#define    MC_CMD_ALLOC_VIS_EXT_OUT_LEN 12
/* The number of VIs allocated on this function */
#define       MC_CMD_ALLOC_VIS_EXT_OUT_VI_COUNT_OFST 0
#define       MC_CMD_ALLOC_VIS_EXT_OUT_VI_COUNT_LEN 4
/* The base absolute VI number allocated to this function. Required to
 * correctly interpret wakeup events.
 */
#define       MC_CMD_ALLOC_VIS_EXT_OUT_VI_BASE_OFST 4
#define       MC_CMD_ALLOC_VIS_EXT_OUT_VI_BASE_LEN 4
/* Function's port vi_shift value (always 0 on Huntington) */
#define       MC_CMD_ALLOC_VIS_EXT_OUT_VI_SHIFT_OFST 8
#define       MC_CMD_ALLOC_VIS_EXT_OUT_VI_SHIFT_LEN 4


/***********************************/
/* MC_CMD_FREE_VIS
 * Free VIs for current PCI function. Any linked PIO buffers will be unlinked,
 * but not freed.
 */
#define MC_CMD_FREE_VIS 0x8c
#undef MC_CMD_0x8c_PRIVILEGE_CTG

#define MC_CMD_0x8c_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_FREE_VIS_IN msgrequest */
#define    MC_CMD_FREE_VIS_IN_LEN 0

/* MC_CMD_FREE_VIS_OUT msgresponse */
#define    MC_CMD_FREE_VIS_OUT_LEN 0


/***********************************/
/* MC_CMD_GET_SRIOV_CFG
 * Get SRIOV config for this PF.
 */
#define MC_CMD_GET_SRIOV_CFG 0xba
#undef MC_CMD_0xba_PRIVILEGE_CTG

#define MC_CMD_0xba_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_GET_SRIOV_CFG_IN msgrequest */
#define    MC_CMD_GET_SRIOV_CFG_IN_LEN 0

/* MC_CMD_GET_SRIOV_CFG_OUT msgresponse */
#define    MC_CMD_GET_SRIOV_CFG_OUT_LEN 20
/* Number of VFs currently enabled. */
#define       MC_CMD_GET_SRIOV_CFG_OUT_VF_CURRENT_OFST 0
#define       MC_CMD_GET_SRIOV_CFG_OUT_VF_CURRENT_LEN 4
/* Max number of VFs before sriov stride and offset may need to be changed. */
#define       MC_CMD_GET_SRIOV_CFG_OUT_VF_MAX_OFST 4
#define       MC_CMD_GET_SRIOV_CFG_OUT_VF_MAX_LEN 4
#define       MC_CMD_GET_SRIOV_CFG_OUT_FLAGS_OFST 8
#define       MC_CMD_GET_SRIOV_CFG_OUT_FLAGS_LEN 4
#define        MC_CMD_GET_SRIOV_CFG_OUT_VF_ENABLED_OFST 8
#define        MC_CMD_GET_SRIOV_CFG_OUT_VF_ENABLED_LBN 0
#define        MC_CMD_GET_SRIOV_CFG_OUT_VF_ENABLED_WIDTH 1
/* RID offset of first VF from PF. */
#define       MC_CMD_GET_SRIOV_CFG_OUT_VF_OFFSET_OFST 12
#define       MC_CMD_GET_SRIOV_CFG_OUT_VF_OFFSET_LEN 4
/* RID offset of each subsequent VF from the previous. */
#define       MC_CMD_GET_SRIOV_CFG_OUT_VF_STRIDE_OFST 16
#define       MC_CMD_GET_SRIOV_CFG_OUT_VF_STRIDE_LEN 4


/***********************************/
/* MC_CMD_ALLOC_PIOBUF
 * Allocate a push I/O buffer for later use with a tx queue.
 */
#define MC_CMD_ALLOC_PIOBUF 0x8f
#undef MC_CMD_0x8f_PRIVILEGE_CTG

#define MC_CMD_0x8f_PRIVILEGE_CTG SRIOV_CTG_ONLOAD

/* MC_CMD_ALLOC_PIOBUF_IN msgrequest */
#define    MC_CMD_ALLOC_PIOBUF_IN_LEN 0

/* MC_CMD_ALLOC_PIOBUF_OUT msgresponse */
#define    MC_CMD_ALLOC_PIOBUF_OUT_LEN 4
/* Handle for allocated push I/O buffer. */
#define       MC_CMD_ALLOC_PIOBUF_OUT_PIOBUF_HANDLE_OFST 0
#define       MC_CMD_ALLOC_PIOBUF_OUT_PIOBUF_HANDLE_LEN 4


/***********************************/
/* MC_CMD_FREE_PIOBUF
 * Free a push I/O buffer.
 */
#define MC_CMD_FREE_PIOBUF 0x90
#undef MC_CMD_0x90_PRIVILEGE_CTG

#define MC_CMD_0x90_PRIVILEGE_CTG SRIOV_CTG_ONLOAD

/* MC_CMD_FREE_PIOBUF_IN msgrequest */
#define    MC_CMD_FREE_PIOBUF_IN_LEN 4
/* Handle for allocated push I/O buffer. */
#define       MC_CMD_FREE_PIOBUF_IN_PIOBUF_HANDLE_OFST 0
#define       MC_CMD_FREE_PIOBUF_IN_PIOBUF_HANDLE_LEN 4

/* MC_CMD_FREE_PIOBUF_OUT msgresponse */
#define    MC_CMD_FREE_PIOBUF_OUT_LEN 0


/***********************************/
/* MC_CMD_GET_VI_TLP_PROCESSING
 * Get TLP steering and ordering information for a VI. The caller must have the
 * GRP_FUNC_DMA privilege and must be the currently-assigned user of this VI or
 * an ancestor of the current user (see MC_CMD_SET_VI_USER).
 */
#define MC_CMD_GET_VI_TLP_PROCESSING 0xb0
#undef MC_CMD_0xb0_PRIVILEGE_CTG

#define MC_CMD_0xb0_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_GET_VI_TLP_PROCESSING_IN msgrequest */
#define    MC_CMD_GET_VI_TLP_PROCESSING_IN_LEN 4
/* Queue handle, encodes queue type and VI number to get information for. */
#define       MC_CMD_GET_VI_TLP_PROCESSING_IN_INSTANCE_OFST 0
#define       MC_CMD_GET_VI_TLP_PROCESSING_IN_INSTANCE_LEN 4

/* MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT msgresponse: This message has the same
 * layout as GET_VI_TLP_PROCESSING_OUT, but with corrected field ordering to
 * simplify use in drivers
 */
#define    MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_LEN 4
#define       MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_DATA_OFST 0
#define       MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_DATA_LEN 4
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_TPH_TAG1_RX_OFST 0
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_TPH_TAG1_RX_LBN 0
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_TPH_TAG1_RX_WIDTH 8
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_TPH_TAG2_EV_OFST 0
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_TPH_TAG2_EV_LBN 8
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_TPH_TAG2_EV_WIDTH 8
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_RELAXED_ORDERING_OFST 0
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_RELAXED_ORDERING_LBN 16
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_RELAXED_ORDERING_WIDTH 1
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_RELAXED_ORDERING_PACKET_DATA_OFST 0
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_RELAXED_ORDERING_PACKET_DATA_LBN 16
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_RELAXED_ORDERING_PACKET_DATA_WIDTH 1
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_ID_BASED_ORDERING_OFST 0
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_ID_BASED_ORDERING_LBN 17
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_ID_BASED_ORDERING_WIDTH 1
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_NO_SNOOP_OFST 0
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_NO_SNOOP_LBN 18
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_NO_SNOOP_WIDTH 1
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_TPH_ON_OFST 0
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_TPH_ON_LBN 19
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_TPH_ON_WIDTH 1
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_RELAXED_ORDERING_SYNC_DATA_OFST 0
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_RELAXED_ORDERING_SYNC_DATA_LBN 20
#define        MC_CMD_GET_VI_TLP_PROCESSING_V2_OUT_RELAXED_ORDERING_SYNC_DATA_WIDTH 1


/***********************************/
/* MC_CMD_SET_VI_TLP_PROCESSING
 * Set TLP steering and ordering information for a VI. The caller must have the
 * GRP_FUNC_DMA privilege and must be the currently-assigned user of this VI or
 * an ancestor of the current user (see MC_CMD_SET_VI_USER). Note that LL
 * queues require this to be called after allocation but before initialisation
 * of the queue. TLP options of a queue are fixed after queue is initialised,
 * with the values set to current global value or they can be overriden using
 * this command. At LL queue allocation, all overrides are cleared.
 */
#define MC_CMD_SET_VI_TLP_PROCESSING 0xb1
#undef MC_CMD_0xb1_PRIVILEGE_CTG

#define MC_CMD_0xb1_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_SET_VI_TLP_PROCESSING_V2_IN msgrequest: This message has the same
 * layout as SET_VI_TLP_PROCESSING_OUT, but with corrected field ordering to
 * simplify use in drivers.
 */
#define    MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_LEN 8
/* Queue handle, encodes queue type and VI number to set information for. */
#define       MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_INSTANCE_OFST 0
#define       MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_INSTANCE_LEN 4
#define       MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_DATA_OFST 4
#define       MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_DATA_LEN 4
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_TPH_TAG1_RX_OFST 4
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_TPH_TAG1_RX_LBN 0
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_TPH_TAG1_RX_WIDTH 8
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_TPH_TAG2_EV_OFST 4
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_TPH_TAG2_EV_LBN 8
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_TPH_TAG2_EV_WIDTH 8
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_RELAXED_ORDERING_OFST 4
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_RELAXED_ORDERING_LBN 16
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_RELAXED_ORDERING_WIDTH 1
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_RELAXED_ORDERING_PACKET_DATA_OFST 4
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_RELAXED_ORDERING_PACKET_DATA_LBN 16
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_RELAXED_ORDERING_PACKET_DATA_WIDTH 1
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_ID_BASED_ORDERING_OFST 4
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_ID_BASED_ORDERING_LBN 17
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_ID_BASED_ORDERING_WIDTH 1
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_NO_SNOOP_OFST 4
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_NO_SNOOP_LBN 18
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_NO_SNOOP_WIDTH 1
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_TPH_ON_OFST 4
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_TPH_ON_LBN 19
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_TPH_ON_WIDTH 1
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_RELAXED_ORDERING_SYNC_DATA_OFST 4
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_RELAXED_ORDERING_SYNC_DATA_LBN 20
#define        MC_CMD_SET_VI_TLP_PROCESSING_V2_IN_RELAXED_ORDERING_SYNC_DATA_WIDTH 1

/* MC_CMD_SET_VI_TLP_PROCESSING_OUT msgresponse */
#define    MC_CMD_SET_VI_TLP_PROCESSING_OUT_LEN 0


/***********************************/
/* MC_CMD_GET_CAPABILITIES
 * Get device capabilities. This is supplementary to the MC_CMD_GET_BOARD_CFG
 * command, and intended to reference inherent device capabilities as opposed
 * to current NVRAM config.
 */
#define MC_CMD_GET_CAPABILITIES 0xbe
#undef MC_CMD_0xbe_PRIVILEGE_CTG

#define MC_CMD_0xbe_PRIVILEGE_CTG SRIOV_CTG_GENERAL

/* MC_CMD_GET_CAPABILITIES_IN msgrequest */
#define    MC_CMD_GET_CAPABILITIES_IN_LEN 0

/* MC_CMD_GET_CAPABILITIES_OUT msgresponse */
#define    MC_CMD_GET_CAPABILITIES_OUT_LEN 20
/* First word of flags. */
#define       MC_CMD_GET_CAPABILITIES_OUT_FLAGS1_OFST 0
#define       MC_CMD_GET_CAPABILITIES_OUT_FLAGS1_LEN 4
#define        MC_CMD_GET_CAPABILITIES_OUT_VPORT_RECONFIGURE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_VPORT_RECONFIGURE_LBN 3
#define        MC_CMD_GET_CAPABILITIES_OUT_VPORT_RECONFIGURE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_TX_STRIPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_TX_STRIPING_LBN 4
#define        MC_CMD_GET_CAPABILITIES_OUT_TX_STRIPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_VADAPTOR_QUERY_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_VADAPTOR_QUERY_LBN 5
#define        MC_CMD_GET_CAPABILITIES_OUT_VADAPTOR_QUERY_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_EVB_PORT_VLAN_RESTRICT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_EVB_PORT_VLAN_RESTRICT_LBN 6
#define        MC_CMD_GET_CAPABILITIES_OUT_EVB_PORT_VLAN_RESTRICT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_DRV_ATTACH_PREBOOT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_DRV_ATTACH_PREBOOT_LBN 7
#define        MC_CMD_GET_CAPABILITIES_OUT_DRV_ATTACH_PREBOOT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_FORCE_EVENT_MERGING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_FORCE_EVENT_MERGING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_FORCE_EVENT_MERGING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_SET_MAC_ENHANCED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_SET_MAC_ENHANCED_LBN 9
#define        MC_CMD_GET_CAPABILITIES_OUT_SET_MAC_ENHANCED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_LBN 10
#define        MC_CMD_GET_CAPABILITIES_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_LBN 11
#define        MC_CMD_GET_CAPABILITIES_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_TX_MAC_SECURITY_FILTERING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_TX_MAC_SECURITY_FILTERING_LBN 12
#define        MC_CMD_GET_CAPABILITIES_OUT_TX_MAC_SECURITY_FILTERING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_ADDITIONAL_RSS_MODES_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_ADDITIONAL_RSS_MODES_LBN 13
#define        MC_CMD_GET_CAPABILITIES_OUT_ADDITIONAL_RSS_MODES_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_QBB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_QBB_LBN 14
#define        MC_CMD_GET_CAPABILITIES_OUT_QBB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_PACKED_STREAM_VAR_BUFFERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_PACKED_STREAM_VAR_BUFFERS_LBN 15
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_PACKED_STREAM_VAR_BUFFERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_RSS_LIMITED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_RSS_LIMITED_LBN 16
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_RSS_LIMITED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_PACKED_STREAM_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_PACKED_STREAM_LBN 17
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_INCLUDE_FCS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_INCLUDE_FCS_LBN 18
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_INCLUDE_FCS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_TX_VLAN_INSERTION_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_TX_VLAN_INSERTION_LBN 19
#define        MC_CMD_GET_CAPABILITIES_OUT_TX_VLAN_INSERTION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_VLAN_STRIPPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_VLAN_STRIPPING_LBN 20
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_VLAN_STRIPPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_TX_TSO_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_TX_TSO_LBN 21
#define        MC_CMD_GET_CAPABILITIES_OUT_TX_TSO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_PREFIX_LEN_0_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_PREFIX_LEN_0_LBN 22
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_PREFIX_LEN_0_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_PREFIX_LEN_14_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_PREFIX_LEN_14_LBN 23
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_PREFIX_LEN_14_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_TIMESTAMP_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_TIMESTAMP_LBN 24
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_BATCHING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_BATCHING_LBN 25
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_BATCHING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_MCAST_FILTER_CHAINING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_MCAST_FILTER_CHAINING_LBN 26
#define        MC_CMD_GET_CAPABILITIES_OUT_MCAST_FILTER_CHAINING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_PM_AND_RXDP_COUNTERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_PM_AND_RXDP_COUNTERS_LBN 27
#define        MC_CMD_GET_CAPABILITIES_OUT_PM_AND_RXDP_COUNTERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_DISABLE_SCATTER_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_DISABLE_SCATTER_LBN 28
#define        MC_CMD_GET_CAPABILITIES_OUT_RX_DISABLE_SCATTER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_TX_MCAST_UDP_LOOPBACK_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_TX_MCAST_UDP_LOOPBACK_LBN 29
#define        MC_CMD_GET_CAPABILITIES_OUT_TX_MCAST_UDP_LOOPBACK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_EVB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_EVB_LBN 30
#define        MC_CMD_GET_CAPABILITIES_OUT_EVB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_OUT_VXLAN_NVGRE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_OUT_VXLAN_NVGRE_LBN 31
#define        MC_CMD_GET_CAPABILITIES_OUT_VXLAN_NVGRE_WIDTH 1
/* RxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_OUT_RX_DPCPU_FW_ID_OFST 4
#define       MC_CMD_GET_CAPABILITIES_OUT_RX_DPCPU_FW_ID_LEN 2
/* enum: Standard RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXDP 0x0
/* enum: Low latency RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXDP_LOW_LATENCY 0x1
/* enum: Packed stream RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXDP_PACKED_STREAM 0x2
/* enum: Rules engine RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXDP_RULES_ENGINE 0x5
/* enum: DPDK RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXDP_DPDK 0x6
/* enum: BIST RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXDP_BIST 0x10a
/* enum: RXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXDP_TEST_FW_TO_MC_CUT_THROUGH 0x101
/* enum: RXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD 0x102
/* enum: RXDP Test firmware image 3 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD_FIRST 0x103
/* enum: RXDP Test firmware image 4 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXDP_TEST_EVERY_EVENT_BATCHABLE 0x104
/* enum: RXDP Test firmware image 5 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXDP_TEST_BACKPRESSURE 0x105
/* enum: RXDP Test firmware image 6 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXDP_TEST_FW_PACKET_EDITS 0x106
/* enum: RXDP Test firmware image 7 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXDP_TEST_FW_RX_HDR_SPLIT 0x107
/* enum: RXDP Test firmware image 8 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXDP_TEST_FW_DISABLE_DL 0x108
/* enum: RXDP Test firmware image 9 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXDP_TEST_FW_DOORBELL_DELAY 0x10b
/* enum: RXDP Test firmware image 10 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXDP_TEST_FW_SLOW 0x10c
/* TxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_OUT_TX_DPCPU_FW_ID_OFST 6
#define       MC_CMD_GET_CAPABILITIES_OUT_TX_DPCPU_FW_ID_LEN 2
/* enum: Standard TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXDP 0x0
/* enum: Low latency TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXDP_LOW_LATENCY 0x1
/* enum: High packet rate TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXDP_HIGH_PACKET_RATE 0x3
/* enum: Rules engine TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXDP_RULES_ENGINE 0x5
/* enum: DPDK TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXDP_DPDK 0x6
/* enum: BIST TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXDP_BIST 0x12d
/* enum: TXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXDP_TEST_FW_TSO_EDIT 0x101
/* enum: TXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXDP_TEST_FW_PACKET_EDITS 0x102
/* enum: TXDP CSR bus test firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXDP_TEST_FW_CSR 0x103
#define       MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_VERSION_OFST 8
#define       MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_VERSION_REV_OFST 8
#define        MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_VERSION_TYPE_OFST 8
#define        MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial RX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: RX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: RX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant RX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
/* enum: Low latency RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_TYPE_LOW_LATENCY 0x5
/* enum: Packed stream RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_TYPE_PACKED_STREAM 0x6
/* enum: RX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* enum: RX PD firmware parsing but not filtering network overlay tunnel
 * encapsulations (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_OUT_RXPD_FW_TYPE_TESTFW_ENCAP_PARSING_ONLY 0xf
#define       MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_VERSION_OFST 10
#define       MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_VERSION_REV_OFST 10
#define        MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_VERSION_TYPE_OFST 10
#define        MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial TX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: TX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: TX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant TX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
#define          MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_TYPE_LOW_LATENCY 0x5 /* enum */
/* enum: TX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_OUT_TXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* Hardware capabilities of NIC */
#define       MC_CMD_GET_CAPABILITIES_OUT_HW_CAPABILITIES_OFST 12
#define       MC_CMD_GET_CAPABILITIES_OUT_HW_CAPABILITIES_LEN 4
/* Licensed capabilities */
#define       MC_CMD_GET_CAPABILITIES_OUT_LICENSE_CAPABILITIES_OFST 16
#define       MC_CMD_GET_CAPABILITIES_OUT_LICENSE_CAPABILITIES_LEN 4

/* MC_CMD_GET_CAPABILITIES_V2_IN msgrequest */
#define    MC_CMD_GET_CAPABILITIES_V2_IN_LEN 0

/* MC_CMD_GET_CAPABILITIES_V2_OUT msgresponse */
#define    MC_CMD_GET_CAPABILITIES_V2_OUT_LEN 72
/* First word of flags. */
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_FLAGS1_OFST 0
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_FLAGS1_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_VPORT_RECONFIGURE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_VPORT_RECONFIGURE_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_VPORT_RECONFIGURE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_STRIPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_STRIPING_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_STRIPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_VADAPTOR_QUERY_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_VADAPTOR_QUERY_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_VADAPTOR_QUERY_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EVB_PORT_VLAN_RESTRICT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EVB_PORT_VLAN_RESTRICT_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EVB_PORT_VLAN_RESTRICT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_DRV_ATTACH_PREBOOT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_DRV_ATTACH_PREBOOT_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_DRV_ATTACH_PREBOOT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_FORCE_EVENT_MERGING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_FORCE_EVENT_MERGING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_FORCE_EVENT_MERGING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_SET_MAC_ENHANCED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_SET_MAC_ENHANCED_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_SET_MAC_ENHANCED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_MAC_SECURITY_FILTERING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_MAC_SECURITY_FILTERING_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_MAC_SECURITY_FILTERING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_ADDITIONAL_RSS_MODES_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_ADDITIONAL_RSS_MODES_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_ADDITIONAL_RSS_MODES_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_QBB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_QBB_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_QBB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_PACKED_STREAM_VAR_BUFFERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_PACKED_STREAM_VAR_BUFFERS_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_PACKED_STREAM_VAR_BUFFERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_RSS_LIMITED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_RSS_LIMITED_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_RSS_LIMITED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_PACKED_STREAM_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_PACKED_STREAM_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_INCLUDE_FCS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_INCLUDE_FCS_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_INCLUDE_FCS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_VLAN_INSERTION_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_VLAN_INSERTION_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_VLAN_INSERTION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_VLAN_STRIPPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_VLAN_STRIPPING_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_VLAN_STRIPPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TSO_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TSO_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TSO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_PREFIX_LEN_0_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_PREFIX_LEN_0_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_PREFIX_LEN_0_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_PREFIX_LEN_14_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_PREFIX_LEN_14_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_PREFIX_LEN_14_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_TIMESTAMP_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_TIMESTAMP_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_BATCHING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_BATCHING_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_BATCHING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_MCAST_FILTER_CHAINING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_MCAST_FILTER_CHAINING_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_MCAST_FILTER_CHAINING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_PM_AND_RXDP_COUNTERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_PM_AND_RXDP_COUNTERS_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_PM_AND_RXDP_COUNTERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_DISABLE_SCATTER_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_DISABLE_SCATTER_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_DISABLE_SCATTER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_MCAST_UDP_LOOPBACK_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_MCAST_UDP_LOOPBACK_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_MCAST_UDP_LOOPBACK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EVB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EVB_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EVB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_VXLAN_NVGRE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_VXLAN_NVGRE_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_VXLAN_NVGRE_WIDTH 1
/* RxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_RX_DPCPU_FW_ID_OFST 4
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_RX_DPCPU_FW_ID_LEN 2
/* enum: Standard RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP 0x0
/* enum: Low latency RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_LOW_LATENCY 0x1
/* enum: Packed stream RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_PACKED_STREAM 0x2
/* enum: Rules engine RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_RULES_ENGINE 0x5
/* enum: DPDK RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_DPDK 0x6
/* enum: BIST RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_BIST 0x10a
/* enum: RXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_TEST_FW_TO_MC_CUT_THROUGH 0x101
/* enum: RXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD 0x102
/* enum: RXDP Test firmware image 3 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD_FIRST 0x103
/* enum: RXDP Test firmware image 4 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_TEST_EVERY_EVENT_BATCHABLE 0x104
/* enum: RXDP Test firmware image 5 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_TEST_BACKPRESSURE 0x105
/* enum: RXDP Test firmware image 6 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_TEST_FW_PACKET_EDITS 0x106
/* enum: RXDP Test firmware image 7 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_TEST_FW_RX_HDR_SPLIT 0x107
/* enum: RXDP Test firmware image 8 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_TEST_FW_DISABLE_DL 0x108
/* enum: RXDP Test firmware image 9 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_TEST_FW_DOORBELL_DELAY 0x10b
/* enum: RXDP Test firmware image 10 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_TEST_FW_SLOW 0x10c
/* TxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_TX_DPCPU_FW_ID_OFST 6
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_TX_DPCPU_FW_ID_LEN 2
/* enum: Standard TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXDP 0x0
/* enum: Low latency TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXDP_LOW_LATENCY 0x1
/* enum: High packet rate TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXDP_HIGH_PACKET_RATE 0x3
/* enum: Rules engine TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXDP_RULES_ENGINE 0x5
/* enum: DPDK TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXDP_DPDK 0x6
/* enum: BIST TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXDP_BIST 0x12d
/* enum: TXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXDP_TEST_FW_TSO_EDIT 0x101
/* enum: TXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXDP_TEST_FW_PACKET_EDITS 0x102
/* enum: TXDP CSR bus test firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXDP_TEST_FW_CSR 0x103
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_VERSION_OFST 8
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_VERSION_REV_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_VERSION_TYPE_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial RX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: RX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: RX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant RX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
/* enum: Low latency RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_TYPE_LOW_LATENCY 0x5
/* enum: Packed stream RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_TYPE_PACKED_STREAM 0x6
/* enum: RX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* enum: RX PD firmware parsing but not filtering network overlay tunnel
 * encapsulations (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_RXPD_FW_TYPE_TESTFW_ENCAP_PARSING_ONLY 0xf
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_VERSION_OFST 10
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_VERSION_REV_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_VERSION_TYPE_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial TX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: TX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: TX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant TX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_TYPE_LOW_LATENCY 0x5 /* enum */
/* enum: TX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_TXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* Hardware capabilities of NIC */
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_HW_CAPABILITIES_OFST 12
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_HW_CAPABILITIES_LEN 4
/* Licensed capabilities */
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_LICENSE_CAPABILITIES_OFST 16
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_LICENSE_CAPABILITIES_LEN 4
/* Second word of flags. Not present on older firmware (check the length). */
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_FLAGS2_OFST 20
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_FLAGS2_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TSO_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TSO_V2_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TSO_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TSO_V2_ENCAP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TSO_V2_ENCAP_LBN 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TSO_V2_ENCAP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EVQ_TIMER_CTRL_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EVQ_TIMER_CTRL_LBN 2
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EVQ_TIMER_CTRL_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EVENT_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EVENT_CUT_THROUGH_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EVENT_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_CUT_THROUGH_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_VFIFO_ULL_MODE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_VFIFO_ULL_MODE_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_VFIFO_ULL_MODE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_MAC_STATS_40G_TX_SIZE_BINS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_MAC_STATS_40G_TX_SIZE_BINS_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_MAC_STATS_40G_TX_SIZE_BINS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_INIT_EVQ_TYPE_SUPPORTED_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_INIT_EVQ_TYPE_SUPPORTED_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_INIT_EVQ_TYPE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_INIT_EVQ_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_INIT_EVQ_V2_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_INIT_EVQ_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_MAC_TIMESTAMPING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_MAC_TIMESTAMPING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_MAC_TIMESTAMPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TIMESTAMP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TIMESTAMP_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_SNIFF_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_SNIFF_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_MCDI_BACKGROUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_MCDI_BACKGROUND_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_MCDI_BACKGROUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_MCDI_DB_RETURN_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_MCDI_DB_RETURN_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_MCDI_DB_RETURN_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_CTPIO_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_CTPIO_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_CTPIO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TSA_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TSA_SUPPORT_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TSA_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TSA_BOUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TSA_BOUND_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TSA_BOUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_SF_ADAPTER_AUTHENTICATION_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_SF_ADAPTER_AUTHENTICATION_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_SF_ADAPTER_AUTHENTICATION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_FILTER_ACTION_FLAG_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_FILTER_ACTION_FLAG_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_FILTER_ACTION_FLAG_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_FILTER_ACTION_MARK_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_FILTER_ACTION_MARK_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_FILTER_ACTION_MARK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EQUAL_STRIDE_SUPER_BUFFER_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EQUAL_STRIDE_SUPER_BUFFER_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EQUAL_STRIDE_SUPER_BUFFER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EQUAL_STRIDE_PACKED_STREAM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EQUAL_STRIDE_PACKED_STREAM_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_EQUAL_STRIDE_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_L3XUDP_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_L3XUDP_SUPPORT_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_L3XUDP_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_FW_SUBVARIANT_NO_TX_CSUM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_FW_SUBVARIANT_NO_TX_CSUM_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_FW_SUBVARIANT_NO_TX_CSUM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_VI_SPREADING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_VI_SPREADING_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_VI_SPREADING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_HLB_IDLE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_HLB_IDLE_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_RXDP_HLB_IDLE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_INIT_RXQ_NO_CONT_EV_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_INIT_RXQ_NO_CONT_EV_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_INIT_RXQ_NO_CONT_EV_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_INIT_RXQ_WITH_BUFFER_SIZE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_INIT_RXQ_WITH_BUFFER_SIZE_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_INIT_RXQ_WITH_BUFFER_SIZE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_BUNDLE_UPDATE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_BUNDLE_UPDATE_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_BUNDLE_UPDATE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TSO_V3_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TSO_V3_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TSO_V3_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_DYNAMIC_SENSORS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_DYNAMIC_SENSORS_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_DYNAMIC_SENSORS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V2_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_WIDTH 1
/* Number of FATSOv2 contexts per datapath supported by this NIC (when
 * TX_TSO_V2 == 1). Not present on older firmware (check the length).
 */
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TSO_V2_N_CONTEXTS_OFST 24
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_TX_TSO_V2_N_CONTEXTS_LEN 2
/* One byte per PF containing the number of the external port assigned to this
 * PF, indexed by PF number. Special values indicate that a PF is either not
 * present or not assigned.
 */
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_PFS_TO_PORTS_ASSIGNMENT_OFST 26
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_PFS_TO_PORTS_ASSIGNMENT_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_PFS_TO_PORTS_ASSIGNMENT_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_ACCESS_NOT_PERMITTED 0xff
/* enum: PF does not exist. */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_PF_NOT_PRESENT 0xfe
/* enum: PF does exist but is not assigned to any external port. */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_PF_NOT_ASSIGNED 0xfd
/* enum: This value indicates that PF is assigned, but it cannot be expressed
 * in this field. It is intended for a possible future situation where a more
 * complex scheme of PFs to ports mapping is being used. The future driver
 * should look for a new field supporting the new scheme. The current/old
 * driver should treat this value as PF_NOT_ASSIGNED.
 */
#define          MC_CMD_GET_CAPABILITIES_V2_OUT_INCOMPATIBLE_ASSIGNMENT 0xfc
/* One byte per PF containing the number of its VFs, indexed by PF number. A
 * special value indicates that a PF is not present.
 */
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_NUM_VFS_PER_PF_OFST 42
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_NUM_VFS_PER_PF_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_NUM_VFS_PER_PF_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
/*               MC_CMD_GET_CAPABILITIES_V2_OUT_ACCESS_NOT_PERMITTED 0xff */
/* enum: PF does not exist. */
/*               MC_CMD_GET_CAPABILITIES_V2_OUT_PF_NOT_PRESENT 0xfe */
/* Number of VIs available for external ports 0-3. For devices with more than
 * four ports, the remainder are in NUM_VIS_PER_PORT2 in
 * GET_CAPABILITIES_V12_OUT.
 */
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_NUM_VIS_PER_PORT_OFST 58
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_NUM_VIS_PER_PORT_LEN 2
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_NUM_VIS_PER_PORT_NUM 4
/* Size of RX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ RX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_RX_DESC_CACHE_SIZE_OFST 66
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_RX_DESC_CACHE_SIZE_LEN 1
/* Size of TX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ TX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_TX_DESC_CACHE_SIZE_OFST 67
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_TX_DESC_CACHE_SIZE_LEN 1
/* Total number of available PIO buffers */
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_NUM_PIO_BUFFS_OFST 68
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_NUM_PIO_BUFFS_LEN 2
/* Size of a single PIO buffer */
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_SIZE_PIO_BUFF_OFST 70
#define       MC_CMD_GET_CAPABILITIES_V2_OUT_SIZE_PIO_BUFF_LEN 2

/* MC_CMD_GET_CAPABILITIES_V3_OUT msgresponse */
#define    MC_CMD_GET_CAPABILITIES_V3_OUT_LEN 76
/* First word of flags. */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_FLAGS1_OFST 0
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_FLAGS1_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_VPORT_RECONFIGURE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_VPORT_RECONFIGURE_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_VPORT_RECONFIGURE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_STRIPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_STRIPING_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_STRIPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_VADAPTOR_QUERY_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_VADAPTOR_QUERY_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_VADAPTOR_QUERY_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EVB_PORT_VLAN_RESTRICT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EVB_PORT_VLAN_RESTRICT_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EVB_PORT_VLAN_RESTRICT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_DRV_ATTACH_PREBOOT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_DRV_ATTACH_PREBOOT_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_DRV_ATTACH_PREBOOT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_FORCE_EVENT_MERGING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_FORCE_EVENT_MERGING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_FORCE_EVENT_MERGING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_SET_MAC_ENHANCED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_SET_MAC_ENHANCED_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_SET_MAC_ENHANCED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_MAC_SECURITY_FILTERING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_MAC_SECURITY_FILTERING_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_MAC_SECURITY_FILTERING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_ADDITIONAL_RSS_MODES_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_ADDITIONAL_RSS_MODES_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_ADDITIONAL_RSS_MODES_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_QBB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_QBB_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_QBB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_PACKED_STREAM_VAR_BUFFERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_PACKED_STREAM_VAR_BUFFERS_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_PACKED_STREAM_VAR_BUFFERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_RSS_LIMITED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_RSS_LIMITED_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_RSS_LIMITED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_PACKED_STREAM_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_PACKED_STREAM_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_INCLUDE_FCS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_INCLUDE_FCS_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_INCLUDE_FCS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_VLAN_INSERTION_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_VLAN_INSERTION_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_VLAN_INSERTION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_VLAN_STRIPPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_VLAN_STRIPPING_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_VLAN_STRIPPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TSO_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TSO_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TSO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_PREFIX_LEN_0_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_PREFIX_LEN_0_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_PREFIX_LEN_0_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_PREFIX_LEN_14_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_PREFIX_LEN_14_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_PREFIX_LEN_14_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_TIMESTAMP_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_TIMESTAMP_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_BATCHING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_BATCHING_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_BATCHING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_MCAST_FILTER_CHAINING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_MCAST_FILTER_CHAINING_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_MCAST_FILTER_CHAINING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_PM_AND_RXDP_COUNTERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_PM_AND_RXDP_COUNTERS_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_PM_AND_RXDP_COUNTERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_DISABLE_SCATTER_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_DISABLE_SCATTER_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_DISABLE_SCATTER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_MCAST_UDP_LOOPBACK_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_MCAST_UDP_LOOPBACK_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_MCAST_UDP_LOOPBACK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EVB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EVB_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EVB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_VXLAN_NVGRE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_VXLAN_NVGRE_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_VXLAN_NVGRE_WIDTH 1
/* RxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_RX_DPCPU_FW_ID_OFST 4
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_RX_DPCPU_FW_ID_LEN 2
/* enum: Standard RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP 0x0
/* enum: Low latency RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_LOW_LATENCY 0x1
/* enum: Packed stream RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_PACKED_STREAM 0x2
/* enum: Rules engine RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_RULES_ENGINE 0x5
/* enum: DPDK RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_DPDK 0x6
/* enum: BIST RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_BIST 0x10a
/* enum: RXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_TEST_FW_TO_MC_CUT_THROUGH 0x101
/* enum: RXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD 0x102
/* enum: RXDP Test firmware image 3 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD_FIRST 0x103
/* enum: RXDP Test firmware image 4 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_TEST_EVERY_EVENT_BATCHABLE 0x104
/* enum: RXDP Test firmware image 5 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_TEST_BACKPRESSURE 0x105
/* enum: RXDP Test firmware image 6 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_TEST_FW_PACKET_EDITS 0x106
/* enum: RXDP Test firmware image 7 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_TEST_FW_RX_HDR_SPLIT 0x107
/* enum: RXDP Test firmware image 8 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_TEST_FW_DISABLE_DL 0x108
/* enum: RXDP Test firmware image 9 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_TEST_FW_DOORBELL_DELAY 0x10b
/* enum: RXDP Test firmware image 10 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_TEST_FW_SLOW 0x10c
/* TxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_TX_DPCPU_FW_ID_OFST 6
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_TX_DPCPU_FW_ID_LEN 2
/* enum: Standard TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXDP 0x0
/* enum: Low latency TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXDP_LOW_LATENCY 0x1
/* enum: High packet rate TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXDP_HIGH_PACKET_RATE 0x3
/* enum: Rules engine TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXDP_RULES_ENGINE 0x5
/* enum: DPDK TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXDP_DPDK 0x6
/* enum: BIST TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXDP_BIST 0x12d
/* enum: TXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXDP_TEST_FW_TSO_EDIT 0x101
/* enum: TXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXDP_TEST_FW_PACKET_EDITS 0x102
/* enum: TXDP CSR bus test firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXDP_TEST_FW_CSR 0x103
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_VERSION_OFST 8
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_VERSION_REV_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_VERSION_TYPE_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial RX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: RX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: RX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant RX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
/* enum: Low latency RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_TYPE_LOW_LATENCY 0x5
/* enum: Packed stream RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_TYPE_PACKED_STREAM 0x6
/* enum: RX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* enum: RX PD firmware parsing but not filtering network overlay tunnel
 * encapsulations (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_RXPD_FW_TYPE_TESTFW_ENCAP_PARSING_ONLY 0xf
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_VERSION_OFST 10
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_VERSION_REV_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_VERSION_TYPE_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial TX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: TX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: TX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant TX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_TYPE_LOW_LATENCY 0x5 /* enum */
/* enum: TX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_TXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* Hardware capabilities of NIC */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_HW_CAPABILITIES_OFST 12
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_HW_CAPABILITIES_LEN 4
/* Licensed capabilities */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_LICENSE_CAPABILITIES_OFST 16
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_LICENSE_CAPABILITIES_LEN 4
/* Second word of flags. Not present on older firmware (check the length). */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_FLAGS2_OFST 20
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_FLAGS2_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TSO_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TSO_V2_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TSO_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TSO_V2_ENCAP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TSO_V2_ENCAP_LBN 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TSO_V2_ENCAP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EVQ_TIMER_CTRL_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EVQ_TIMER_CTRL_LBN 2
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EVQ_TIMER_CTRL_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EVENT_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EVENT_CUT_THROUGH_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EVENT_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_CUT_THROUGH_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_VFIFO_ULL_MODE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_VFIFO_ULL_MODE_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_VFIFO_ULL_MODE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_MAC_STATS_40G_TX_SIZE_BINS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_MAC_STATS_40G_TX_SIZE_BINS_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_MAC_STATS_40G_TX_SIZE_BINS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_INIT_EVQ_TYPE_SUPPORTED_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_INIT_EVQ_TYPE_SUPPORTED_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_INIT_EVQ_TYPE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_INIT_EVQ_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_INIT_EVQ_V2_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_INIT_EVQ_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_MAC_TIMESTAMPING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_MAC_TIMESTAMPING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_MAC_TIMESTAMPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TIMESTAMP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TIMESTAMP_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_SNIFF_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_SNIFF_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_MCDI_BACKGROUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_MCDI_BACKGROUND_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_MCDI_BACKGROUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_MCDI_DB_RETURN_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_MCDI_DB_RETURN_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_MCDI_DB_RETURN_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_CTPIO_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_CTPIO_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_CTPIO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TSA_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TSA_SUPPORT_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TSA_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TSA_BOUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TSA_BOUND_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TSA_BOUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_SF_ADAPTER_AUTHENTICATION_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_SF_ADAPTER_AUTHENTICATION_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_SF_ADAPTER_AUTHENTICATION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_FILTER_ACTION_FLAG_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_FILTER_ACTION_FLAG_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_FILTER_ACTION_FLAG_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_FILTER_ACTION_MARK_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_FILTER_ACTION_MARK_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_FILTER_ACTION_MARK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EQUAL_STRIDE_SUPER_BUFFER_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EQUAL_STRIDE_SUPER_BUFFER_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EQUAL_STRIDE_SUPER_BUFFER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EQUAL_STRIDE_PACKED_STREAM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EQUAL_STRIDE_PACKED_STREAM_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_EQUAL_STRIDE_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_L3XUDP_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_L3XUDP_SUPPORT_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_L3XUDP_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_FW_SUBVARIANT_NO_TX_CSUM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_FW_SUBVARIANT_NO_TX_CSUM_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_FW_SUBVARIANT_NO_TX_CSUM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_VI_SPREADING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_VI_SPREADING_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_VI_SPREADING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_HLB_IDLE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_HLB_IDLE_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_RXDP_HLB_IDLE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_INIT_RXQ_NO_CONT_EV_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_INIT_RXQ_NO_CONT_EV_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_INIT_RXQ_NO_CONT_EV_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_INIT_RXQ_WITH_BUFFER_SIZE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_INIT_RXQ_WITH_BUFFER_SIZE_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_INIT_RXQ_WITH_BUFFER_SIZE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_BUNDLE_UPDATE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_BUNDLE_UPDATE_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_BUNDLE_UPDATE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TSO_V3_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TSO_V3_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TSO_V3_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_DYNAMIC_SENSORS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_DYNAMIC_SENSORS_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_DYNAMIC_SENSORS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V3_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_WIDTH 1
/* Number of FATSOv2 contexts per datapath supported by this NIC (when
 * TX_TSO_V2 == 1). Not present on older firmware (check the length).
 */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TSO_V2_N_CONTEXTS_OFST 24
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_TX_TSO_V2_N_CONTEXTS_LEN 2
/* One byte per PF containing the number of the external port assigned to this
 * PF, indexed by PF number. Special values indicate that a PF is either not
 * present or not assigned.
 */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_PFS_TO_PORTS_ASSIGNMENT_OFST 26
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_PFS_TO_PORTS_ASSIGNMENT_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_PFS_TO_PORTS_ASSIGNMENT_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_ACCESS_NOT_PERMITTED 0xff
/* enum: PF does not exist. */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_PF_NOT_PRESENT 0xfe
/* enum: PF does exist but is not assigned to any external port. */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_PF_NOT_ASSIGNED 0xfd
/* enum: This value indicates that PF is assigned, but it cannot be expressed
 * in this field. It is intended for a possible future situation where a more
 * complex scheme of PFs to ports mapping is being used. The future driver
 * should look for a new field supporting the new scheme. The current/old
 * driver should treat this value as PF_NOT_ASSIGNED.
 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_INCOMPATIBLE_ASSIGNMENT 0xfc
/* One byte per PF containing the number of its VFs, indexed by PF number. A
 * special value indicates that a PF is not present.
 */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_NUM_VFS_PER_PF_OFST 42
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_NUM_VFS_PER_PF_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_NUM_VFS_PER_PF_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
/*               MC_CMD_GET_CAPABILITIES_V3_OUT_ACCESS_NOT_PERMITTED 0xff */
/* enum: PF does not exist. */
/*               MC_CMD_GET_CAPABILITIES_V3_OUT_PF_NOT_PRESENT 0xfe */
/* Number of VIs available for external ports 0-3. For devices with more than
 * four ports, the remainder are in NUM_VIS_PER_PORT2 in
 * GET_CAPABILITIES_V12_OUT.
 */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_NUM_VIS_PER_PORT_OFST 58
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_NUM_VIS_PER_PORT_LEN 2
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_NUM_VIS_PER_PORT_NUM 4
/* Size of RX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ RX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_RX_DESC_CACHE_SIZE_OFST 66
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_RX_DESC_CACHE_SIZE_LEN 1
/* Size of TX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ TX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_TX_DESC_CACHE_SIZE_OFST 67
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_TX_DESC_CACHE_SIZE_LEN 1
/* Total number of available PIO buffers */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_NUM_PIO_BUFFS_OFST 68
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_NUM_PIO_BUFFS_LEN 2
/* Size of a single PIO buffer */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_SIZE_PIO_BUFF_OFST 70
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_SIZE_PIO_BUFF_LEN 2
/* On chips later than Medford the amount of address space assigned to each VI
 * is configurable. This is a global setting that the driver must query to
 * discover the VI to address mapping. Cut-through PIO (CTPIO) is not available
 * with 8k VI windows.
 */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_VI_WINDOW_MODE_OFST 72
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_VI_WINDOW_MODE_LEN 1
/* enum: Each VI occupies 8k as on Huntington and Medford. PIO is at offset 4k.
 * CTPIO is not mapped.
 */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_VI_WINDOW_MODE_8K 0x0
/* enum: Each VI occupies 16k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_VI_WINDOW_MODE_16K 0x1
/* enum: Each VI occupies 64k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V3_OUT_VI_WINDOW_MODE_64K 0x2
/* Number of vFIFOs per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_VFIFO_STUFFING_NUM_VFIFOS_OFST 73
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_VFIFO_STUFFING_NUM_VFIFOS_LEN 1
/* Number of buffers per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_OFST 74
#define       MC_CMD_GET_CAPABILITIES_V3_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_LEN 2

/* MC_CMD_GET_CAPABILITIES_V4_OUT msgresponse */
#define    MC_CMD_GET_CAPABILITIES_V4_OUT_LEN 78
/* First word of flags. */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_FLAGS1_OFST 0
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_FLAGS1_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_VPORT_RECONFIGURE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_VPORT_RECONFIGURE_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_VPORT_RECONFIGURE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_STRIPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_STRIPING_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_STRIPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_VADAPTOR_QUERY_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_VADAPTOR_QUERY_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_VADAPTOR_QUERY_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EVB_PORT_VLAN_RESTRICT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EVB_PORT_VLAN_RESTRICT_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EVB_PORT_VLAN_RESTRICT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_DRV_ATTACH_PREBOOT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_DRV_ATTACH_PREBOOT_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_DRV_ATTACH_PREBOOT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_FORCE_EVENT_MERGING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_FORCE_EVENT_MERGING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_FORCE_EVENT_MERGING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_SET_MAC_ENHANCED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_SET_MAC_ENHANCED_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_SET_MAC_ENHANCED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_MAC_SECURITY_FILTERING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_MAC_SECURITY_FILTERING_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_MAC_SECURITY_FILTERING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_ADDITIONAL_RSS_MODES_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_ADDITIONAL_RSS_MODES_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_ADDITIONAL_RSS_MODES_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_QBB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_QBB_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_QBB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_PACKED_STREAM_VAR_BUFFERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_PACKED_STREAM_VAR_BUFFERS_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_PACKED_STREAM_VAR_BUFFERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_RSS_LIMITED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_RSS_LIMITED_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_RSS_LIMITED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_PACKED_STREAM_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_PACKED_STREAM_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_INCLUDE_FCS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_INCLUDE_FCS_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_INCLUDE_FCS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_VLAN_INSERTION_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_VLAN_INSERTION_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_VLAN_INSERTION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_VLAN_STRIPPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_VLAN_STRIPPING_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_VLAN_STRIPPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TSO_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TSO_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TSO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_PREFIX_LEN_0_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_PREFIX_LEN_0_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_PREFIX_LEN_0_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_PREFIX_LEN_14_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_PREFIX_LEN_14_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_PREFIX_LEN_14_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_TIMESTAMP_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_TIMESTAMP_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_BATCHING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_BATCHING_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_BATCHING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_MCAST_FILTER_CHAINING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_MCAST_FILTER_CHAINING_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_MCAST_FILTER_CHAINING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_PM_AND_RXDP_COUNTERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_PM_AND_RXDP_COUNTERS_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_PM_AND_RXDP_COUNTERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_DISABLE_SCATTER_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_DISABLE_SCATTER_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_DISABLE_SCATTER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_MCAST_UDP_LOOPBACK_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_MCAST_UDP_LOOPBACK_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_MCAST_UDP_LOOPBACK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EVB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EVB_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EVB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_VXLAN_NVGRE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_VXLAN_NVGRE_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_VXLAN_NVGRE_WIDTH 1
/* RxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_RX_DPCPU_FW_ID_OFST 4
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_RX_DPCPU_FW_ID_LEN 2
/* enum: Standard RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP 0x0
/* enum: Low latency RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_LOW_LATENCY 0x1
/* enum: Packed stream RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_PACKED_STREAM 0x2
/* enum: Rules engine RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_RULES_ENGINE 0x5
/* enum: DPDK RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_DPDK 0x6
/* enum: BIST RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_BIST 0x10a
/* enum: RXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_TEST_FW_TO_MC_CUT_THROUGH 0x101
/* enum: RXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD 0x102
/* enum: RXDP Test firmware image 3 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD_FIRST 0x103
/* enum: RXDP Test firmware image 4 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_TEST_EVERY_EVENT_BATCHABLE 0x104
/* enum: RXDP Test firmware image 5 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_TEST_BACKPRESSURE 0x105
/* enum: RXDP Test firmware image 6 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_TEST_FW_PACKET_EDITS 0x106
/* enum: RXDP Test firmware image 7 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_TEST_FW_RX_HDR_SPLIT 0x107
/* enum: RXDP Test firmware image 8 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_TEST_FW_DISABLE_DL 0x108
/* enum: RXDP Test firmware image 9 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_TEST_FW_DOORBELL_DELAY 0x10b
/* enum: RXDP Test firmware image 10 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_TEST_FW_SLOW 0x10c
/* TxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_TX_DPCPU_FW_ID_OFST 6
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_TX_DPCPU_FW_ID_LEN 2
/* enum: Standard TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXDP 0x0
/* enum: Low latency TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXDP_LOW_LATENCY 0x1
/* enum: High packet rate TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXDP_HIGH_PACKET_RATE 0x3
/* enum: Rules engine TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXDP_RULES_ENGINE 0x5
/* enum: DPDK TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXDP_DPDK 0x6
/* enum: BIST TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXDP_BIST 0x12d
/* enum: TXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXDP_TEST_FW_TSO_EDIT 0x101
/* enum: TXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXDP_TEST_FW_PACKET_EDITS 0x102
/* enum: TXDP CSR bus test firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXDP_TEST_FW_CSR 0x103
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_VERSION_OFST 8
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_VERSION_REV_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_VERSION_TYPE_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial RX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: RX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: RX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant RX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
/* enum: Low latency RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_TYPE_LOW_LATENCY 0x5
/* enum: Packed stream RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_TYPE_PACKED_STREAM 0x6
/* enum: RX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* enum: RX PD firmware parsing but not filtering network overlay tunnel
 * encapsulations (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_RXPD_FW_TYPE_TESTFW_ENCAP_PARSING_ONLY 0xf
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_VERSION_OFST 10
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_VERSION_REV_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_VERSION_TYPE_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial TX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: TX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: TX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant TX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_TYPE_LOW_LATENCY 0x5 /* enum */
/* enum: TX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_TXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* Hardware capabilities of NIC */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_HW_CAPABILITIES_OFST 12
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_HW_CAPABILITIES_LEN 4
/* Licensed capabilities */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_LICENSE_CAPABILITIES_OFST 16
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_LICENSE_CAPABILITIES_LEN 4
/* Second word of flags. Not present on older firmware (check the length). */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_FLAGS2_OFST 20
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_FLAGS2_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TSO_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TSO_V2_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TSO_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TSO_V2_ENCAP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TSO_V2_ENCAP_LBN 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TSO_V2_ENCAP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EVQ_TIMER_CTRL_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EVQ_TIMER_CTRL_LBN 2
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EVQ_TIMER_CTRL_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EVENT_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EVENT_CUT_THROUGH_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EVENT_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_CUT_THROUGH_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_VFIFO_ULL_MODE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_VFIFO_ULL_MODE_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_VFIFO_ULL_MODE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_MAC_STATS_40G_TX_SIZE_BINS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_MAC_STATS_40G_TX_SIZE_BINS_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_MAC_STATS_40G_TX_SIZE_BINS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_INIT_EVQ_TYPE_SUPPORTED_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_INIT_EVQ_TYPE_SUPPORTED_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_INIT_EVQ_TYPE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_INIT_EVQ_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_INIT_EVQ_V2_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_INIT_EVQ_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_MAC_TIMESTAMPING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_MAC_TIMESTAMPING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_MAC_TIMESTAMPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TIMESTAMP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TIMESTAMP_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_SNIFF_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_SNIFF_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_MCDI_BACKGROUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_MCDI_BACKGROUND_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_MCDI_BACKGROUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_MCDI_DB_RETURN_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_MCDI_DB_RETURN_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_MCDI_DB_RETURN_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_CTPIO_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_CTPIO_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_CTPIO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TSA_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TSA_SUPPORT_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TSA_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TSA_BOUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TSA_BOUND_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TSA_BOUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_SF_ADAPTER_AUTHENTICATION_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_SF_ADAPTER_AUTHENTICATION_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_SF_ADAPTER_AUTHENTICATION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_FILTER_ACTION_FLAG_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_FILTER_ACTION_FLAG_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_FILTER_ACTION_FLAG_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_FILTER_ACTION_MARK_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_FILTER_ACTION_MARK_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_FILTER_ACTION_MARK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EQUAL_STRIDE_SUPER_BUFFER_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EQUAL_STRIDE_SUPER_BUFFER_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EQUAL_STRIDE_SUPER_BUFFER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EQUAL_STRIDE_PACKED_STREAM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EQUAL_STRIDE_PACKED_STREAM_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_EQUAL_STRIDE_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_L3XUDP_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_L3XUDP_SUPPORT_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_L3XUDP_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_FW_SUBVARIANT_NO_TX_CSUM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_FW_SUBVARIANT_NO_TX_CSUM_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_FW_SUBVARIANT_NO_TX_CSUM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_VI_SPREADING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_VI_SPREADING_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_VI_SPREADING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_HLB_IDLE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_HLB_IDLE_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_RXDP_HLB_IDLE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_INIT_RXQ_NO_CONT_EV_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_INIT_RXQ_NO_CONT_EV_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_INIT_RXQ_NO_CONT_EV_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_INIT_RXQ_WITH_BUFFER_SIZE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_INIT_RXQ_WITH_BUFFER_SIZE_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_INIT_RXQ_WITH_BUFFER_SIZE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_BUNDLE_UPDATE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_BUNDLE_UPDATE_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_BUNDLE_UPDATE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TSO_V3_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TSO_V3_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TSO_V3_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_DYNAMIC_SENSORS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_DYNAMIC_SENSORS_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_DYNAMIC_SENSORS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V4_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_WIDTH 1
/* Number of FATSOv2 contexts per datapath supported by this NIC (when
 * TX_TSO_V2 == 1). Not present on older firmware (check the length).
 */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TSO_V2_N_CONTEXTS_OFST 24
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_TX_TSO_V2_N_CONTEXTS_LEN 2
/* One byte per PF containing the number of the external port assigned to this
 * PF, indexed by PF number. Special values indicate that a PF is either not
 * present or not assigned.
 */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_PFS_TO_PORTS_ASSIGNMENT_OFST 26
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_PFS_TO_PORTS_ASSIGNMENT_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_PFS_TO_PORTS_ASSIGNMENT_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_ACCESS_NOT_PERMITTED 0xff
/* enum: PF does not exist. */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_PF_NOT_PRESENT 0xfe
/* enum: PF does exist but is not assigned to any external port. */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_PF_NOT_ASSIGNED 0xfd
/* enum: This value indicates that PF is assigned, but it cannot be expressed
 * in this field. It is intended for a possible future situation where a more
 * complex scheme of PFs to ports mapping is being used. The future driver
 * should look for a new field supporting the new scheme. The current/old
 * driver should treat this value as PF_NOT_ASSIGNED.
 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_INCOMPATIBLE_ASSIGNMENT 0xfc
/* One byte per PF containing the number of its VFs, indexed by PF number. A
 * special value indicates that a PF is not present.
 */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_NUM_VFS_PER_PF_OFST 42
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_NUM_VFS_PER_PF_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_NUM_VFS_PER_PF_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
/*               MC_CMD_GET_CAPABILITIES_V4_OUT_ACCESS_NOT_PERMITTED 0xff */
/* enum: PF does not exist. */
/*               MC_CMD_GET_CAPABILITIES_V4_OUT_PF_NOT_PRESENT 0xfe */
/* Number of VIs available for external ports 0-3. For devices with more than
 * four ports, the remainder are in NUM_VIS_PER_PORT2 in
 * GET_CAPABILITIES_V12_OUT.
 */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_NUM_VIS_PER_PORT_OFST 58
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_NUM_VIS_PER_PORT_LEN 2
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_NUM_VIS_PER_PORT_NUM 4
/* Size of RX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ RX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_RX_DESC_CACHE_SIZE_OFST 66
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_RX_DESC_CACHE_SIZE_LEN 1
/* Size of TX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ TX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_TX_DESC_CACHE_SIZE_OFST 67
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_TX_DESC_CACHE_SIZE_LEN 1
/* Total number of available PIO buffers */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_NUM_PIO_BUFFS_OFST 68
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_NUM_PIO_BUFFS_LEN 2
/* Size of a single PIO buffer */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_SIZE_PIO_BUFF_OFST 70
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_SIZE_PIO_BUFF_LEN 2
/* On chips later than Medford the amount of address space assigned to each VI
 * is configurable. This is a global setting that the driver must query to
 * discover the VI to address mapping. Cut-through PIO (CTPIO) is not available
 * with 8k VI windows.
 */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_VI_WINDOW_MODE_OFST 72
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_VI_WINDOW_MODE_LEN 1
/* enum: Each VI occupies 8k as on Huntington and Medford. PIO is at offset 4k.
 * CTPIO is not mapped.
 */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_VI_WINDOW_MODE_8K 0x0
/* enum: Each VI occupies 16k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_VI_WINDOW_MODE_16K 0x1
/* enum: Each VI occupies 64k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V4_OUT_VI_WINDOW_MODE_64K 0x2
/* Number of vFIFOs per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_VFIFO_STUFFING_NUM_VFIFOS_OFST 73
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_VFIFO_STUFFING_NUM_VFIFOS_LEN 1
/* Number of buffers per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_OFST 74
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_LEN 2
/* Entry count in the MAC stats array, including the final GENERATION_END
 * entry. For MAC stats DMA, drivers should allocate a buffer large enough to
 * hold at least this many 64-bit stats values, if they wish to receive all
 * available stats. If the buffer is shorter than MAC_STATS_NUM_STATS * 8, the
 * stats array returned will be truncated.
 */
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_MAC_STATS_NUM_STATS_OFST 76
#define       MC_CMD_GET_CAPABILITIES_V4_OUT_MAC_STATS_NUM_STATS_LEN 2

/* MC_CMD_GET_CAPABILITIES_V5_OUT msgresponse */
#define    MC_CMD_GET_CAPABILITIES_V5_OUT_LEN 84
/* First word of flags. */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_FLAGS1_OFST 0
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_FLAGS1_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_VPORT_RECONFIGURE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_VPORT_RECONFIGURE_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_VPORT_RECONFIGURE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_STRIPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_STRIPING_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_STRIPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_VADAPTOR_QUERY_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_VADAPTOR_QUERY_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_VADAPTOR_QUERY_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EVB_PORT_VLAN_RESTRICT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EVB_PORT_VLAN_RESTRICT_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EVB_PORT_VLAN_RESTRICT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_DRV_ATTACH_PREBOOT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_DRV_ATTACH_PREBOOT_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_DRV_ATTACH_PREBOOT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_FORCE_EVENT_MERGING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_FORCE_EVENT_MERGING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_FORCE_EVENT_MERGING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_SET_MAC_ENHANCED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_SET_MAC_ENHANCED_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_SET_MAC_ENHANCED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_MAC_SECURITY_FILTERING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_MAC_SECURITY_FILTERING_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_MAC_SECURITY_FILTERING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_ADDITIONAL_RSS_MODES_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_ADDITIONAL_RSS_MODES_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_ADDITIONAL_RSS_MODES_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_QBB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_QBB_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_QBB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_PACKED_STREAM_VAR_BUFFERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_PACKED_STREAM_VAR_BUFFERS_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_PACKED_STREAM_VAR_BUFFERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_RSS_LIMITED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_RSS_LIMITED_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_RSS_LIMITED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_PACKED_STREAM_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_PACKED_STREAM_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_INCLUDE_FCS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_INCLUDE_FCS_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_INCLUDE_FCS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_VLAN_INSERTION_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_VLAN_INSERTION_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_VLAN_INSERTION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_VLAN_STRIPPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_VLAN_STRIPPING_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_VLAN_STRIPPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TSO_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TSO_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TSO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_PREFIX_LEN_0_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_PREFIX_LEN_0_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_PREFIX_LEN_0_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_PREFIX_LEN_14_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_PREFIX_LEN_14_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_PREFIX_LEN_14_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_TIMESTAMP_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_TIMESTAMP_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_BATCHING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_BATCHING_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_BATCHING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_MCAST_FILTER_CHAINING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_MCAST_FILTER_CHAINING_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_MCAST_FILTER_CHAINING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_PM_AND_RXDP_COUNTERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_PM_AND_RXDP_COUNTERS_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_PM_AND_RXDP_COUNTERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_DISABLE_SCATTER_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_DISABLE_SCATTER_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_DISABLE_SCATTER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_MCAST_UDP_LOOPBACK_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_MCAST_UDP_LOOPBACK_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_MCAST_UDP_LOOPBACK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EVB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EVB_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EVB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_VXLAN_NVGRE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_VXLAN_NVGRE_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_VXLAN_NVGRE_WIDTH 1
/* RxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_RX_DPCPU_FW_ID_OFST 4
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_RX_DPCPU_FW_ID_LEN 2
/* enum: Standard RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP 0x0
/* enum: Low latency RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_LOW_LATENCY 0x1
/* enum: Packed stream RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_PACKED_STREAM 0x2
/* enum: Rules engine RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_RULES_ENGINE 0x5
/* enum: DPDK RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_DPDK 0x6
/* enum: BIST RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_BIST 0x10a
/* enum: RXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_TEST_FW_TO_MC_CUT_THROUGH 0x101
/* enum: RXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD 0x102
/* enum: RXDP Test firmware image 3 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD_FIRST 0x103
/* enum: RXDP Test firmware image 4 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_TEST_EVERY_EVENT_BATCHABLE 0x104
/* enum: RXDP Test firmware image 5 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_TEST_BACKPRESSURE 0x105
/* enum: RXDP Test firmware image 6 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_TEST_FW_PACKET_EDITS 0x106
/* enum: RXDP Test firmware image 7 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_TEST_FW_RX_HDR_SPLIT 0x107
/* enum: RXDP Test firmware image 8 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_TEST_FW_DISABLE_DL 0x108
/* enum: RXDP Test firmware image 9 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_TEST_FW_DOORBELL_DELAY 0x10b
/* enum: RXDP Test firmware image 10 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_TEST_FW_SLOW 0x10c
/* TxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_TX_DPCPU_FW_ID_OFST 6
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_TX_DPCPU_FW_ID_LEN 2
/* enum: Standard TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXDP 0x0
/* enum: Low latency TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXDP_LOW_LATENCY 0x1
/* enum: High packet rate TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXDP_HIGH_PACKET_RATE 0x3
/* enum: Rules engine TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXDP_RULES_ENGINE 0x5
/* enum: DPDK TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXDP_DPDK 0x6
/* enum: BIST TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXDP_BIST 0x12d
/* enum: TXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXDP_TEST_FW_TSO_EDIT 0x101
/* enum: TXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXDP_TEST_FW_PACKET_EDITS 0x102
/* enum: TXDP CSR bus test firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXDP_TEST_FW_CSR 0x103
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_VERSION_OFST 8
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_VERSION_REV_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_VERSION_TYPE_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial RX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: RX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: RX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant RX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
/* enum: Low latency RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_TYPE_LOW_LATENCY 0x5
/* enum: Packed stream RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_TYPE_PACKED_STREAM 0x6
/* enum: RX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* enum: RX PD firmware parsing but not filtering network overlay tunnel
 * encapsulations (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_RXPD_FW_TYPE_TESTFW_ENCAP_PARSING_ONLY 0xf
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_VERSION_OFST 10
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_VERSION_REV_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_VERSION_TYPE_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial TX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: TX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: TX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant TX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_TYPE_LOW_LATENCY 0x5 /* enum */
/* enum: TX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_TXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* Hardware capabilities of NIC */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_HW_CAPABILITIES_OFST 12
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_HW_CAPABILITIES_LEN 4
/* Licensed capabilities */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_LICENSE_CAPABILITIES_OFST 16
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_LICENSE_CAPABILITIES_LEN 4
/* Second word of flags. Not present on older firmware (check the length). */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_FLAGS2_OFST 20
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_FLAGS2_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TSO_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TSO_V2_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TSO_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TSO_V2_ENCAP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TSO_V2_ENCAP_LBN 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TSO_V2_ENCAP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EVQ_TIMER_CTRL_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EVQ_TIMER_CTRL_LBN 2
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EVQ_TIMER_CTRL_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EVENT_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EVENT_CUT_THROUGH_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EVENT_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_CUT_THROUGH_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_VFIFO_ULL_MODE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_VFIFO_ULL_MODE_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_VFIFO_ULL_MODE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_MAC_STATS_40G_TX_SIZE_BINS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_MAC_STATS_40G_TX_SIZE_BINS_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_MAC_STATS_40G_TX_SIZE_BINS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_INIT_EVQ_TYPE_SUPPORTED_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_INIT_EVQ_TYPE_SUPPORTED_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_INIT_EVQ_TYPE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_INIT_EVQ_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_INIT_EVQ_V2_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_INIT_EVQ_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_MAC_TIMESTAMPING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_MAC_TIMESTAMPING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_MAC_TIMESTAMPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TIMESTAMP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TIMESTAMP_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_SNIFF_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_SNIFF_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_MCDI_BACKGROUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_MCDI_BACKGROUND_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_MCDI_BACKGROUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_MCDI_DB_RETURN_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_MCDI_DB_RETURN_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_MCDI_DB_RETURN_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_CTPIO_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_CTPIO_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_CTPIO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TSA_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TSA_SUPPORT_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TSA_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TSA_BOUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TSA_BOUND_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TSA_BOUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_SF_ADAPTER_AUTHENTICATION_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_SF_ADAPTER_AUTHENTICATION_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_SF_ADAPTER_AUTHENTICATION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_FILTER_ACTION_FLAG_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_FILTER_ACTION_FLAG_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_FILTER_ACTION_FLAG_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_FILTER_ACTION_MARK_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_FILTER_ACTION_MARK_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_FILTER_ACTION_MARK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EQUAL_STRIDE_SUPER_BUFFER_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EQUAL_STRIDE_SUPER_BUFFER_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EQUAL_STRIDE_SUPER_BUFFER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EQUAL_STRIDE_PACKED_STREAM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EQUAL_STRIDE_PACKED_STREAM_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_EQUAL_STRIDE_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_L3XUDP_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_L3XUDP_SUPPORT_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_L3XUDP_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_FW_SUBVARIANT_NO_TX_CSUM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_FW_SUBVARIANT_NO_TX_CSUM_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_FW_SUBVARIANT_NO_TX_CSUM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_VI_SPREADING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_VI_SPREADING_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_VI_SPREADING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_HLB_IDLE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_HLB_IDLE_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_RXDP_HLB_IDLE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_INIT_RXQ_NO_CONT_EV_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_INIT_RXQ_NO_CONT_EV_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_INIT_RXQ_NO_CONT_EV_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_INIT_RXQ_WITH_BUFFER_SIZE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_INIT_RXQ_WITH_BUFFER_SIZE_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_INIT_RXQ_WITH_BUFFER_SIZE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_BUNDLE_UPDATE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_BUNDLE_UPDATE_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_BUNDLE_UPDATE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TSO_V3_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TSO_V3_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TSO_V3_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_DYNAMIC_SENSORS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_DYNAMIC_SENSORS_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_DYNAMIC_SENSORS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V5_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_WIDTH 1
/* Number of FATSOv2 contexts per datapath supported by this NIC (when
 * TX_TSO_V2 == 1). Not present on older firmware (check the length).
 */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TSO_V2_N_CONTEXTS_OFST 24
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_TX_TSO_V2_N_CONTEXTS_LEN 2
/* One byte per PF containing the number of the external port assigned to this
 * PF, indexed by PF number. Special values indicate that a PF is either not
 * present or not assigned.
 */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_PFS_TO_PORTS_ASSIGNMENT_OFST 26
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_PFS_TO_PORTS_ASSIGNMENT_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_PFS_TO_PORTS_ASSIGNMENT_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_ACCESS_NOT_PERMITTED 0xff
/* enum: PF does not exist. */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_PF_NOT_PRESENT 0xfe
/* enum: PF does exist but is not assigned to any external port. */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_PF_NOT_ASSIGNED 0xfd
/* enum: This value indicates that PF is assigned, but it cannot be expressed
 * in this field. It is intended for a possible future situation where a more
 * complex scheme of PFs to ports mapping is being used. The future driver
 * should look for a new field supporting the new scheme. The current/old
 * driver should treat this value as PF_NOT_ASSIGNED.
 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_INCOMPATIBLE_ASSIGNMENT 0xfc
/* One byte per PF containing the number of its VFs, indexed by PF number. A
 * special value indicates that a PF is not present.
 */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_NUM_VFS_PER_PF_OFST 42
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_NUM_VFS_PER_PF_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_NUM_VFS_PER_PF_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
/*               MC_CMD_GET_CAPABILITIES_V5_OUT_ACCESS_NOT_PERMITTED 0xff */
/* enum: PF does not exist. */
/*               MC_CMD_GET_CAPABILITIES_V5_OUT_PF_NOT_PRESENT 0xfe */
/* Number of VIs available for external ports 0-3. For devices with more than
 * four ports, the remainder are in NUM_VIS_PER_PORT2 in
 * GET_CAPABILITIES_V12_OUT.
 */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_NUM_VIS_PER_PORT_OFST 58
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_NUM_VIS_PER_PORT_LEN 2
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_NUM_VIS_PER_PORT_NUM 4
/* Size of RX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ RX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_RX_DESC_CACHE_SIZE_OFST 66
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_RX_DESC_CACHE_SIZE_LEN 1
/* Size of TX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ TX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_TX_DESC_CACHE_SIZE_OFST 67
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_TX_DESC_CACHE_SIZE_LEN 1
/* Total number of available PIO buffers */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_NUM_PIO_BUFFS_OFST 68
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_NUM_PIO_BUFFS_LEN 2
/* Size of a single PIO buffer */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_SIZE_PIO_BUFF_OFST 70
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_SIZE_PIO_BUFF_LEN 2
/* On chips later than Medford the amount of address space assigned to each VI
 * is configurable. This is a global setting that the driver must query to
 * discover the VI to address mapping. Cut-through PIO (CTPIO) is not available
 * with 8k VI windows.
 */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_VI_WINDOW_MODE_OFST 72
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_VI_WINDOW_MODE_LEN 1
/* enum: Each VI occupies 8k as on Huntington and Medford. PIO is at offset 4k.
 * CTPIO is not mapped.
 */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_VI_WINDOW_MODE_8K 0x0
/* enum: Each VI occupies 16k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_VI_WINDOW_MODE_16K 0x1
/* enum: Each VI occupies 64k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V5_OUT_VI_WINDOW_MODE_64K 0x2
/* Number of vFIFOs per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_VFIFO_STUFFING_NUM_VFIFOS_OFST 73
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_VFIFO_STUFFING_NUM_VFIFOS_LEN 1
/* Number of buffers per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_OFST 74
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_LEN 2
/* Entry count in the MAC stats array, including the final GENERATION_END
 * entry. For MAC stats DMA, drivers should allocate a buffer large enough to
 * hold at least this many 64-bit stats values, if they wish to receive all
 * available stats. If the buffer is shorter than MAC_STATS_NUM_STATS * 8, the
 * stats array returned will be truncated.
 */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_MAC_STATS_NUM_STATS_OFST 76
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_MAC_STATS_NUM_STATS_LEN 2
/* Maximum supported value for MC_CMD_FILTER_OP_V3/MATCH_MARK_VALUE. This field
 * will only be non-zero if MC_CMD_GET_CAPABILITIES/FILTER_ACTION_MARK is set.
 */
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_FILTER_ACTION_MARK_MAX_OFST 80
#define       MC_CMD_GET_CAPABILITIES_V5_OUT_FILTER_ACTION_MARK_MAX_LEN 4

/* MC_CMD_GET_CAPABILITIES_V6_OUT msgresponse */
#define    MC_CMD_GET_CAPABILITIES_V6_OUT_LEN 148
/* First word of flags. */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_FLAGS1_OFST 0
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_FLAGS1_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_VPORT_RECONFIGURE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_VPORT_RECONFIGURE_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_VPORT_RECONFIGURE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_STRIPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_STRIPING_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_STRIPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_VADAPTOR_QUERY_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_VADAPTOR_QUERY_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_VADAPTOR_QUERY_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EVB_PORT_VLAN_RESTRICT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EVB_PORT_VLAN_RESTRICT_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EVB_PORT_VLAN_RESTRICT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_DRV_ATTACH_PREBOOT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_DRV_ATTACH_PREBOOT_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_DRV_ATTACH_PREBOOT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_FORCE_EVENT_MERGING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_FORCE_EVENT_MERGING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_FORCE_EVENT_MERGING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_SET_MAC_ENHANCED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_SET_MAC_ENHANCED_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_SET_MAC_ENHANCED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_MAC_SECURITY_FILTERING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_MAC_SECURITY_FILTERING_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_MAC_SECURITY_FILTERING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_ADDITIONAL_RSS_MODES_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_ADDITIONAL_RSS_MODES_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_ADDITIONAL_RSS_MODES_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_QBB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_QBB_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_QBB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_PACKED_STREAM_VAR_BUFFERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_PACKED_STREAM_VAR_BUFFERS_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_PACKED_STREAM_VAR_BUFFERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_RSS_LIMITED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_RSS_LIMITED_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_RSS_LIMITED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_PACKED_STREAM_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_PACKED_STREAM_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_INCLUDE_FCS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_INCLUDE_FCS_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_INCLUDE_FCS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_VLAN_INSERTION_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_VLAN_INSERTION_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_VLAN_INSERTION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_VLAN_STRIPPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_VLAN_STRIPPING_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_VLAN_STRIPPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TSO_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TSO_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TSO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_PREFIX_LEN_0_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_PREFIX_LEN_0_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_PREFIX_LEN_0_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_PREFIX_LEN_14_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_PREFIX_LEN_14_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_PREFIX_LEN_14_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_TIMESTAMP_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_TIMESTAMP_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_BATCHING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_BATCHING_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_BATCHING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_MCAST_FILTER_CHAINING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_MCAST_FILTER_CHAINING_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_MCAST_FILTER_CHAINING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_PM_AND_RXDP_COUNTERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_PM_AND_RXDP_COUNTERS_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_PM_AND_RXDP_COUNTERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_DISABLE_SCATTER_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_DISABLE_SCATTER_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_DISABLE_SCATTER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_MCAST_UDP_LOOPBACK_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_MCAST_UDP_LOOPBACK_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_MCAST_UDP_LOOPBACK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EVB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EVB_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EVB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_VXLAN_NVGRE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_VXLAN_NVGRE_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_VXLAN_NVGRE_WIDTH 1
/* RxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_RX_DPCPU_FW_ID_OFST 4
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_RX_DPCPU_FW_ID_LEN 2
/* enum: Standard RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP 0x0
/* enum: Low latency RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_LOW_LATENCY 0x1
/* enum: Packed stream RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_PACKED_STREAM 0x2
/* enum: Rules engine RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_RULES_ENGINE 0x5
/* enum: DPDK RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_DPDK 0x6
/* enum: BIST RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_BIST 0x10a
/* enum: RXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_TEST_FW_TO_MC_CUT_THROUGH 0x101
/* enum: RXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD 0x102
/* enum: RXDP Test firmware image 3 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD_FIRST 0x103
/* enum: RXDP Test firmware image 4 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_TEST_EVERY_EVENT_BATCHABLE 0x104
/* enum: RXDP Test firmware image 5 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_TEST_BACKPRESSURE 0x105
/* enum: RXDP Test firmware image 6 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_TEST_FW_PACKET_EDITS 0x106
/* enum: RXDP Test firmware image 7 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_TEST_FW_RX_HDR_SPLIT 0x107
/* enum: RXDP Test firmware image 8 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_TEST_FW_DISABLE_DL 0x108
/* enum: RXDP Test firmware image 9 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_TEST_FW_DOORBELL_DELAY 0x10b
/* enum: RXDP Test firmware image 10 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_TEST_FW_SLOW 0x10c
/* TxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_TX_DPCPU_FW_ID_OFST 6
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_TX_DPCPU_FW_ID_LEN 2
/* enum: Standard TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXDP 0x0
/* enum: Low latency TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXDP_LOW_LATENCY 0x1
/* enum: High packet rate TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXDP_HIGH_PACKET_RATE 0x3
/* enum: Rules engine TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXDP_RULES_ENGINE 0x5
/* enum: DPDK TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXDP_DPDK 0x6
/* enum: BIST TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXDP_BIST 0x12d
/* enum: TXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXDP_TEST_FW_TSO_EDIT 0x101
/* enum: TXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXDP_TEST_FW_PACKET_EDITS 0x102
/* enum: TXDP CSR bus test firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXDP_TEST_FW_CSR 0x103
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_VERSION_OFST 8
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_VERSION_REV_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_VERSION_TYPE_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial RX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: RX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: RX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant RX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
/* enum: Low latency RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_TYPE_LOW_LATENCY 0x5
/* enum: Packed stream RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_TYPE_PACKED_STREAM 0x6
/* enum: RX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* enum: RX PD firmware parsing but not filtering network overlay tunnel
 * encapsulations (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_RXPD_FW_TYPE_TESTFW_ENCAP_PARSING_ONLY 0xf
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_VERSION_OFST 10
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_VERSION_REV_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_VERSION_TYPE_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial TX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: TX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: TX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant TX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_TYPE_LOW_LATENCY 0x5 /* enum */
/* enum: TX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_TXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* Hardware capabilities of NIC */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_HW_CAPABILITIES_OFST 12
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_HW_CAPABILITIES_LEN 4
/* Licensed capabilities */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_LICENSE_CAPABILITIES_OFST 16
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_LICENSE_CAPABILITIES_LEN 4
/* Second word of flags. Not present on older firmware (check the length). */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_FLAGS2_OFST 20
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_FLAGS2_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TSO_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TSO_V2_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TSO_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TSO_V2_ENCAP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TSO_V2_ENCAP_LBN 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TSO_V2_ENCAP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EVQ_TIMER_CTRL_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EVQ_TIMER_CTRL_LBN 2
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EVQ_TIMER_CTRL_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EVENT_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EVENT_CUT_THROUGH_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EVENT_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_CUT_THROUGH_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_VFIFO_ULL_MODE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_VFIFO_ULL_MODE_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_VFIFO_ULL_MODE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_MAC_STATS_40G_TX_SIZE_BINS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_MAC_STATS_40G_TX_SIZE_BINS_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_MAC_STATS_40G_TX_SIZE_BINS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_INIT_EVQ_TYPE_SUPPORTED_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_INIT_EVQ_TYPE_SUPPORTED_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_INIT_EVQ_TYPE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_INIT_EVQ_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_INIT_EVQ_V2_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_INIT_EVQ_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_MAC_TIMESTAMPING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_MAC_TIMESTAMPING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_MAC_TIMESTAMPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TIMESTAMP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TIMESTAMP_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_SNIFF_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_SNIFF_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_MCDI_BACKGROUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_MCDI_BACKGROUND_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_MCDI_BACKGROUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_MCDI_DB_RETURN_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_MCDI_DB_RETURN_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_MCDI_DB_RETURN_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_CTPIO_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_CTPIO_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_CTPIO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TSA_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TSA_SUPPORT_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TSA_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TSA_BOUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TSA_BOUND_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TSA_BOUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_SF_ADAPTER_AUTHENTICATION_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_SF_ADAPTER_AUTHENTICATION_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_SF_ADAPTER_AUTHENTICATION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_FILTER_ACTION_FLAG_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_FILTER_ACTION_FLAG_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_FILTER_ACTION_FLAG_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_FILTER_ACTION_MARK_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_FILTER_ACTION_MARK_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_FILTER_ACTION_MARK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EQUAL_STRIDE_SUPER_BUFFER_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EQUAL_STRIDE_SUPER_BUFFER_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EQUAL_STRIDE_SUPER_BUFFER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EQUAL_STRIDE_PACKED_STREAM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EQUAL_STRIDE_PACKED_STREAM_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_EQUAL_STRIDE_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_L3XUDP_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_L3XUDP_SUPPORT_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_L3XUDP_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_FW_SUBVARIANT_NO_TX_CSUM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_FW_SUBVARIANT_NO_TX_CSUM_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_FW_SUBVARIANT_NO_TX_CSUM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_VI_SPREADING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_VI_SPREADING_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_VI_SPREADING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_HLB_IDLE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_HLB_IDLE_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_RXDP_HLB_IDLE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_INIT_RXQ_NO_CONT_EV_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_INIT_RXQ_NO_CONT_EV_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_INIT_RXQ_NO_CONT_EV_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_INIT_RXQ_WITH_BUFFER_SIZE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_INIT_RXQ_WITH_BUFFER_SIZE_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_INIT_RXQ_WITH_BUFFER_SIZE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_BUNDLE_UPDATE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_BUNDLE_UPDATE_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_BUNDLE_UPDATE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TSO_V3_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TSO_V3_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TSO_V3_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_DYNAMIC_SENSORS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_DYNAMIC_SENSORS_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_DYNAMIC_SENSORS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V6_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_WIDTH 1
/* Number of FATSOv2 contexts per datapath supported by this NIC (when
 * TX_TSO_V2 == 1). Not present on older firmware (check the length).
 */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TSO_V2_N_CONTEXTS_OFST 24
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_TX_TSO_V2_N_CONTEXTS_LEN 2
/* One byte per PF containing the number of the external port assigned to this
 * PF, indexed by PF number. Special values indicate that a PF is either not
 * present or not assigned.
 */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_PFS_TO_PORTS_ASSIGNMENT_OFST 26
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_PFS_TO_PORTS_ASSIGNMENT_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_PFS_TO_PORTS_ASSIGNMENT_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_ACCESS_NOT_PERMITTED 0xff
/* enum: PF does not exist. */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_PF_NOT_PRESENT 0xfe
/* enum: PF does exist but is not assigned to any external port. */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_PF_NOT_ASSIGNED 0xfd
/* enum: This value indicates that PF is assigned, but it cannot be expressed
 * in this field. It is intended for a possible future situation where a more
 * complex scheme of PFs to ports mapping is being used. The future driver
 * should look for a new field supporting the new scheme. The current/old
 * driver should treat this value as PF_NOT_ASSIGNED.
 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_INCOMPATIBLE_ASSIGNMENT 0xfc
/* One byte per PF containing the number of its VFs, indexed by PF number. A
 * special value indicates that a PF is not present.
 */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_NUM_VFS_PER_PF_OFST 42
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_NUM_VFS_PER_PF_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_NUM_VFS_PER_PF_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
/*               MC_CMD_GET_CAPABILITIES_V6_OUT_ACCESS_NOT_PERMITTED 0xff */
/* enum: PF does not exist. */
/*               MC_CMD_GET_CAPABILITIES_V6_OUT_PF_NOT_PRESENT 0xfe */
/* Number of VIs available for external ports 0-3. For devices with more than
 * four ports, the remainder are in NUM_VIS_PER_PORT2 in
 * GET_CAPABILITIES_V12_OUT.
 */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_NUM_VIS_PER_PORT_OFST 58
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_NUM_VIS_PER_PORT_LEN 2
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_NUM_VIS_PER_PORT_NUM 4
/* Size of RX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ RX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_RX_DESC_CACHE_SIZE_OFST 66
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_RX_DESC_CACHE_SIZE_LEN 1
/* Size of TX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ TX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_TX_DESC_CACHE_SIZE_OFST 67
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_TX_DESC_CACHE_SIZE_LEN 1
/* Total number of available PIO buffers */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_NUM_PIO_BUFFS_OFST 68
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_NUM_PIO_BUFFS_LEN 2
/* Size of a single PIO buffer */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_SIZE_PIO_BUFF_OFST 70
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_SIZE_PIO_BUFF_LEN 2
/* On chips later than Medford the amount of address space assigned to each VI
 * is configurable. This is a global setting that the driver must query to
 * discover the VI to address mapping. Cut-through PIO (CTPIO) is not available
 * with 8k VI windows.
 */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_VI_WINDOW_MODE_OFST 72
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_VI_WINDOW_MODE_LEN 1
/* enum: Each VI occupies 8k as on Huntington and Medford. PIO is at offset 4k.
 * CTPIO is not mapped.
 */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_VI_WINDOW_MODE_8K 0x0
/* enum: Each VI occupies 16k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_VI_WINDOW_MODE_16K 0x1
/* enum: Each VI occupies 64k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V6_OUT_VI_WINDOW_MODE_64K 0x2
/* Number of vFIFOs per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_VFIFO_STUFFING_NUM_VFIFOS_OFST 73
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_VFIFO_STUFFING_NUM_VFIFOS_LEN 1
/* Number of buffers per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_OFST 74
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_LEN 2
/* Entry count in the MAC stats array, including the final GENERATION_END
 * entry. For MAC stats DMA, drivers should allocate a buffer large enough to
 * hold at least this many 64-bit stats values, if they wish to receive all
 * available stats. If the buffer is shorter than MAC_STATS_NUM_STATS * 8, the
 * stats array returned will be truncated.
 */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_MAC_STATS_NUM_STATS_OFST 76
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_MAC_STATS_NUM_STATS_LEN 2
/* Maximum supported value for MC_CMD_FILTER_OP_V3/MATCH_MARK_VALUE. This field
 * will only be non-zero if MC_CMD_GET_CAPABILITIES/FILTER_ACTION_MARK is set.
 */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_FILTER_ACTION_MARK_MAX_OFST 80
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_FILTER_ACTION_MARK_MAX_LEN 4
/* On devices where the INIT_RXQ_WITH_BUFFER_SIZE flag (in
 * GET_CAPABILITIES_OUT_V2) is set, drivers have to specify a buffer size when
 * they create an RX queue. Due to hardware limitations, only a small number of
 * different buffer sizes may be available concurrently. Nonzero entries in
 * this array are the sizes of buffers which the system guarantees will be
 * available for use. If the list is empty, there are no limitations on
 * concurrent buffer sizes.
 */
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_GUARANTEED_RX_BUFFER_SIZES_OFST 84
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_GUARANTEED_RX_BUFFER_SIZES_LEN 4
#define       MC_CMD_GET_CAPABILITIES_V6_OUT_GUARANTEED_RX_BUFFER_SIZES_NUM 16

/* MC_CMD_GET_CAPABILITIES_V7_OUT msgresponse */
#define    MC_CMD_GET_CAPABILITIES_V7_OUT_LEN 152
/* First word of flags. */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_FLAGS1_OFST 0
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_FLAGS1_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VPORT_RECONFIGURE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VPORT_RECONFIGURE_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VPORT_RECONFIGURE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_STRIPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_STRIPING_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_STRIPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VADAPTOR_QUERY_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VADAPTOR_QUERY_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VADAPTOR_QUERY_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EVB_PORT_VLAN_RESTRICT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EVB_PORT_VLAN_RESTRICT_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EVB_PORT_VLAN_RESTRICT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_DRV_ATTACH_PREBOOT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_DRV_ATTACH_PREBOOT_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_DRV_ATTACH_PREBOOT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_FORCE_EVENT_MERGING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_FORCE_EVENT_MERGING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_FORCE_EVENT_MERGING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_SET_MAC_ENHANCED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_SET_MAC_ENHANCED_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_SET_MAC_ENHANCED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_MAC_SECURITY_FILTERING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_MAC_SECURITY_FILTERING_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_MAC_SECURITY_FILTERING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_ADDITIONAL_RSS_MODES_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_ADDITIONAL_RSS_MODES_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_ADDITIONAL_RSS_MODES_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_QBB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_QBB_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_QBB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_PACKED_STREAM_VAR_BUFFERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_PACKED_STREAM_VAR_BUFFERS_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_PACKED_STREAM_VAR_BUFFERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_RSS_LIMITED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_RSS_LIMITED_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_RSS_LIMITED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_PACKED_STREAM_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_PACKED_STREAM_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_INCLUDE_FCS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_INCLUDE_FCS_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_INCLUDE_FCS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_VLAN_INSERTION_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_VLAN_INSERTION_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_VLAN_INSERTION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_VLAN_STRIPPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_VLAN_STRIPPING_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_VLAN_STRIPPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TSO_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TSO_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TSO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_PREFIX_LEN_0_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_PREFIX_LEN_0_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_PREFIX_LEN_0_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_PREFIX_LEN_14_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_PREFIX_LEN_14_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_PREFIX_LEN_14_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_TIMESTAMP_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_TIMESTAMP_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_BATCHING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_BATCHING_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_BATCHING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MCAST_FILTER_CHAINING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MCAST_FILTER_CHAINING_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MCAST_FILTER_CHAINING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_PM_AND_RXDP_COUNTERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_PM_AND_RXDP_COUNTERS_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_PM_AND_RXDP_COUNTERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_DISABLE_SCATTER_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_DISABLE_SCATTER_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_DISABLE_SCATTER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_MCAST_UDP_LOOPBACK_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_MCAST_UDP_LOOPBACK_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_MCAST_UDP_LOOPBACK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EVB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EVB_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EVB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VXLAN_NVGRE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VXLAN_NVGRE_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VXLAN_NVGRE_WIDTH 1
/* RxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_RX_DPCPU_FW_ID_OFST 4
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_RX_DPCPU_FW_ID_LEN 2
/* enum: Standard RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP 0x0
/* enum: Low latency RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_LOW_LATENCY 0x1
/* enum: Packed stream RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_PACKED_STREAM 0x2
/* enum: Rules engine RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_RULES_ENGINE 0x5
/* enum: DPDK RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_DPDK 0x6
/* enum: BIST RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_BIST 0x10a
/* enum: RXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_TEST_FW_TO_MC_CUT_THROUGH 0x101
/* enum: RXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD 0x102
/* enum: RXDP Test firmware image 3 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD_FIRST 0x103
/* enum: RXDP Test firmware image 4 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_TEST_EVERY_EVENT_BATCHABLE 0x104
/* enum: RXDP Test firmware image 5 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_TEST_BACKPRESSURE 0x105
/* enum: RXDP Test firmware image 6 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_TEST_FW_PACKET_EDITS 0x106
/* enum: RXDP Test firmware image 7 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_TEST_FW_RX_HDR_SPLIT 0x107
/* enum: RXDP Test firmware image 8 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_TEST_FW_DISABLE_DL 0x108
/* enum: RXDP Test firmware image 9 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_TEST_FW_DOORBELL_DELAY 0x10b
/* enum: RXDP Test firmware image 10 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_TEST_FW_SLOW 0x10c
/* TxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_TX_DPCPU_FW_ID_OFST 6
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_TX_DPCPU_FW_ID_LEN 2
/* enum: Standard TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXDP 0x0
/* enum: Low latency TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXDP_LOW_LATENCY 0x1
/* enum: High packet rate TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXDP_HIGH_PACKET_RATE 0x3
/* enum: Rules engine TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXDP_RULES_ENGINE 0x5
/* enum: DPDK TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXDP_DPDK 0x6
/* enum: BIST TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXDP_BIST 0x12d
/* enum: TXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXDP_TEST_FW_TSO_EDIT 0x101
/* enum: TXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXDP_TEST_FW_PACKET_EDITS 0x102
/* enum: TXDP CSR bus test firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXDP_TEST_FW_CSR 0x103
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_VERSION_OFST 8
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_VERSION_REV_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_VERSION_TYPE_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial RX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: RX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: RX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant RX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
/* enum: Low latency RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_TYPE_LOW_LATENCY 0x5
/* enum: Packed stream RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_TYPE_PACKED_STREAM 0x6
/* enum: RX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* enum: RX PD firmware parsing but not filtering network overlay tunnel
 * encapsulations (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_RXPD_FW_TYPE_TESTFW_ENCAP_PARSING_ONLY 0xf
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_VERSION_OFST 10
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_VERSION_REV_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_VERSION_TYPE_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial TX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: TX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: TX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant TX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_TYPE_LOW_LATENCY 0x5 /* enum */
/* enum: TX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_TXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* Hardware capabilities of NIC */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_HW_CAPABILITIES_OFST 12
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_HW_CAPABILITIES_LEN 4
/* Licensed capabilities */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_LICENSE_CAPABILITIES_OFST 16
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_LICENSE_CAPABILITIES_LEN 4
/* Second word of flags. Not present on older firmware (check the length). */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_FLAGS2_OFST 20
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_FLAGS2_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TSO_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TSO_V2_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TSO_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TSO_V2_ENCAP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TSO_V2_ENCAP_LBN 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TSO_V2_ENCAP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EVQ_TIMER_CTRL_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EVQ_TIMER_CTRL_LBN 2
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EVQ_TIMER_CTRL_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EVENT_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EVENT_CUT_THROUGH_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EVENT_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_CUT_THROUGH_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_VFIFO_ULL_MODE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_VFIFO_ULL_MODE_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_VFIFO_ULL_MODE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MAC_STATS_40G_TX_SIZE_BINS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MAC_STATS_40G_TX_SIZE_BINS_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MAC_STATS_40G_TX_SIZE_BINS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_INIT_EVQ_TYPE_SUPPORTED_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_INIT_EVQ_TYPE_SUPPORTED_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_INIT_EVQ_TYPE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_INIT_EVQ_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_INIT_EVQ_V2_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_INIT_EVQ_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_MAC_TIMESTAMPING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_MAC_TIMESTAMPING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_MAC_TIMESTAMPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TIMESTAMP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TIMESTAMP_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_SNIFF_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_SNIFF_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MCDI_BACKGROUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MCDI_BACKGROUND_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MCDI_BACKGROUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MCDI_DB_RETURN_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MCDI_DB_RETURN_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MCDI_DB_RETURN_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_CTPIO_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_CTPIO_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_CTPIO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TSA_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TSA_SUPPORT_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TSA_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TSA_BOUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TSA_BOUND_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TSA_BOUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_SF_ADAPTER_AUTHENTICATION_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_SF_ADAPTER_AUTHENTICATION_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_SF_ADAPTER_AUTHENTICATION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_FILTER_ACTION_FLAG_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_FILTER_ACTION_FLAG_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_FILTER_ACTION_FLAG_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_FILTER_ACTION_MARK_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_FILTER_ACTION_MARK_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_FILTER_ACTION_MARK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EQUAL_STRIDE_SUPER_BUFFER_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EQUAL_STRIDE_SUPER_BUFFER_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EQUAL_STRIDE_SUPER_BUFFER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EQUAL_STRIDE_PACKED_STREAM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EQUAL_STRIDE_PACKED_STREAM_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EQUAL_STRIDE_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_L3XUDP_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_L3XUDP_SUPPORT_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_L3XUDP_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_FW_SUBVARIANT_NO_TX_CSUM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_FW_SUBVARIANT_NO_TX_CSUM_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_FW_SUBVARIANT_NO_TX_CSUM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VI_SPREADING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VI_SPREADING_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VI_SPREADING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_HLB_IDLE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_HLB_IDLE_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RXDP_HLB_IDLE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_INIT_RXQ_NO_CONT_EV_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_INIT_RXQ_NO_CONT_EV_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_INIT_RXQ_NO_CONT_EV_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_INIT_RXQ_WITH_BUFFER_SIZE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_INIT_RXQ_WITH_BUFFER_SIZE_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_INIT_RXQ_WITH_BUFFER_SIZE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_BUNDLE_UPDATE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_BUNDLE_UPDATE_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_BUNDLE_UPDATE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TSO_V3_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TSO_V3_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TSO_V3_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_DYNAMIC_SENSORS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_DYNAMIC_SENSORS_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_DYNAMIC_SENSORS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_WIDTH 1
/* Number of FATSOv2 contexts per datapath supported by this NIC (when
 * TX_TSO_V2 == 1). Not present on older firmware (check the length).
 */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TSO_V2_N_CONTEXTS_OFST 24
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_TX_TSO_V2_N_CONTEXTS_LEN 2
/* One byte per PF containing the number of the external port assigned to this
 * PF, indexed by PF number. Special values indicate that a PF is either not
 * present or not assigned.
 */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_PFS_TO_PORTS_ASSIGNMENT_OFST 26
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_PFS_TO_PORTS_ASSIGNMENT_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_PFS_TO_PORTS_ASSIGNMENT_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_ACCESS_NOT_PERMITTED 0xff
/* enum: PF does not exist. */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_PF_NOT_PRESENT 0xfe
/* enum: PF does exist but is not assigned to any external port. */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_PF_NOT_ASSIGNED 0xfd
/* enum: This value indicates that PF is assigned, but it cannot be expressed
 * in this field. It is intended for a possible future situation where a more
 * complex scheme of PFs to ports mapping is being used. The future driver
 * should look for a new field supporting the new scheme. The current/old
 * driver should treat this value as PF_NOT_ASSIGNED.
 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_INCOMPATIBLE_ASSIGNMENT 0xfc
/* One byte per PF containing the number of its VFs, indexed by PF number. A
 * special value indicates that a PF is not present.
 */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_NUM_VFS_PER_PF_OFST 42
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_NUM_VFS_PER_PF_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_NUM_VFS_PER_PF_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
/*               MC_CMD_GET_CAPABILITIES_V7_OUT_ACCESS_NOT_PERMITTED 0xff */
/* enum: PF does not exist. */
/*               MC_CMD_GET_CAPABILITIES_V7_OUT_PF_NOT_PRESENT 0xfe */
/* Number of VIs available for external ports 0-3. For devices with more than
 * four ports, the remainder are in NUM_VIS_PER_PORT2 in
 * GET_CAPABILITIES_V12_OUT.
 */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_NUM_VIS_PER_PORT_OFST 58
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_NUM_VIS_PER_PORT_LEN 2
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_NUM_VIS_PER_PORT_NUM 4
/* Size of RX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ RX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_RX_DESC_CACHE_SIZE_OFST 66
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_RX_DESC_CACHE_SIZE_LEN 1
/* Size of TX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ TX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_TX_DESC_CACHE_SIZE_OFST 67
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_TX_DESC_CACHE_SIZE_LEN 1
/* Total number of available PIO buffers */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_NUM_PIO_BUFFS_OFST 68
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_NUM_PIO_BUFFS_LEN 2
/* Size of a single PIO buffer */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_SIZE_PIO_BUFF_OFST 70
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_SIZE_PIO_BUFF_LEN 2
/* On chips later than Medford the amount of address space assigned to each VI
 * is configurable. This is a global setting that the driver must query to
 * discover the VI to address mapping. Cut-through PIO (CTPIO) is not available
 * with 8k VI windows.
 */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_VI_WINDOW_MODE_OFST 72
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_VI_WINDOW_MODE_LEN 1
/* enum: Each VI occupies 8k as on Huntington and Medford. PIO is at offset 4k.
 * CTPIO is not mapped.
 */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_VI_WINDOW_MODE_8K 0x0
/* enum: Each VI occupies 16k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_VI_WINDOW_MODE_16K 0x1
/* enum: Each VI occupies 64k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V7_OUT_VI_WINDOW_MODE_64K 0x2
/* Number of vFIFOs per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_VFIFO_STUFFING_NUM_VFIFOS_OFST 73
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_VFIFO_STUFFING_NUM_VFIFOS_LEN 1
/* Number of buffers per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_OFST 74
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_LEN 2
/* Entry count in the MAC stats array, including the final GENERATION_END
 * entry. For MAC stats DMA, drivers should allocate a buffer large enough to
 * hold at least this many 64-bit stats values, if they wish to receive all
 * available stats. If the buffer is shorter than MAC_STATS_NUM_STATS * 8, the
 * stats array returned will be truncated.
 */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_MAC_STATS_NUM_STATS_OFST 76
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_MAC_STATS_NUM_STATS_LEN 2
/* Maximum supported value for MC_CMD_FILTER_OP_V3/MATCH_MARK_VALUE. This field
 * will only be non-zero if MC_CMD_GET_CAPABILITIES/FILTER_ACTION_MARK is set.
 */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_FILTER_ACTION_MARK_MAX_OFST 80
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_FILTER_ACTION_MARK_MAX_LEN 4
/* On devices where the INIT_RXQ_WITH_BUFFER_SIZE flag (in
 * GET_CAPABILITIES_OUT_V2) is set, drivers have to specify a buffer size when
 * they create an RX queue. Due to hardware limitations, only a small number of
 * different buffer sizes may be available concurrently. Nonzero entries in
 * this array are the sizes of buffers which the system guarantees will be
 * available for use. If the list is empty, there are no limitations on
 * concurrent buffer sizes.
 */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_GUARANTEED_RX_BUFFER_SIZES_OFST 84
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_GUARANTEED_RX_BUFFER_SIZES_LEN 4
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_GUARANTEED_RX_BUFFER_SIZES_NUM 16
/* Third word of flags. Not present on older firmware (check the length). */
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_FLAGS3_OFST 148
#define       MC_CMD_GET_CAPABILITIES_V7_OUT_FLAGS3_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_WOL_ETHERWAKE_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_WOL_ETHERWAKE_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_WOL_ETHERWAKE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RSS_EVEN_SPREADING_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RSS_EVEN_SPREADING_LBN 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RSS_EVEN_SPREADING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RSS_SELECTABLE_TABLE_SIZE_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RSS_SELECTABLE_TABLE_SIZE_LBN 2
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RSS_SELECTABLE_TABLE_SIZE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MAE_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MAE_SUPPORTED_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MAE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VDPA_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VDPA_SUPPORTED_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_VDPA_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_VLAN_STRIPPING_PER_ENCAP_RULE_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_VLAN_STRIPPING_PER_ENCAP_RULE_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RX_VLAN_STRIPPING_PER_ENCAP_RULE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EXTENDED_WIDTH_EVQS_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EXTENDED_WIDTH_EVQS_SUPPORTED_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EXTENDED_WIDTH_EVQS_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_UNSOL_EV_CREDIT_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_UNSOL_EV_CREDIT_SUPPORTED_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_UNSOL_EV_CREDIT_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_ENCAPSULATED_MCDI_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_ENCAPSULATED_MCDI_SUPPORTED_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_ENCAPSULATED_MCDI_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EXTERNAL_MAE_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EXTERNAL_MAE_SUPPORTED_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_EXTERNAL_MAE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_NVRAM_UPDATE_ABORT_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_NVRAM_UPDATE_ABORT_SUPPORTED_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_NVRAM_UPDATE_ABORT_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MAE_ACTION_SET_ALLOC_V2_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MAE_ACTION_SET_ALLOC_V2_SUPPORTED_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MAE_ACTION_SET_ALLOC_V2_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RSS_STEER_ON_OUTER_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RSS_STEER_ON_OUTER_SUPPORTED_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_RSS_STEER_ON_OUTER_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MAE_ACTION_SET_ALLOC_V3_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MAE_ACTION_SET_ALLOC_V3_SUPPORTED_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_MAE_ACTION_SET_ALLOC_V3_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_DYNAMIC_MPORT_JOURNAL_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_DYNAMIC_MPORT_JOURNAL_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_DYNAMIC_MPORT_JOURNAL_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_CLIENT_CMD_VF_PROXY_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_CLIENT_CMD_VF_PROXY_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_CLIENT_CMD_VF_PROXY_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_LL_RX_EVENT_SUPPRESSION_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_LL_RX_EVENT_SUPPRESSION_SUPPORTED_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_LL_RX_EVENT_SUPPRESSION_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_CXL_CONFIG_ENABLE_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_CXL_CONFIG_ENABLE_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V7_OUT_CXL_CONFIG_ENABLE_WIDTH 1

/* MC_CMD_GET_CAPABILITIES_V8_OUT msgresponse */
#define    MC_CMD_GET_CAPABILITIES_V8_OUT_LEN 160
/* First word of flags. */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_FLAGS1_OFST 0
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_FLAGS1_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VPORT_RECONFIGURE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VPORT_RECONFIGURE_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VPORT_RECONFIGURE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_STRIPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_STRIPING_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_STRIPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VADAPTOR_QUERY_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VADAPTOR_QUERY_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VADAPTOR_QUERY_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EVB_PORT_VLAN_RESTRICT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EVB_PORT_VLAN_RESTRICT_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EVB_PORT_VLAN_RESTRICT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_DRV_ATTACH_PREBOOT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_DRV_ATTACH_PREBOOT_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_DRV_ATTACH_PREBOOT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_FORCE_EVENT_MERGING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_FORCE_EVENT_MERGING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_FORCE_EVENT_MERGING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_SET_MAC_ENHANCED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_SET_MAC_ENHANCED_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_SET_MAC_ENHANCED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_MAC_SECURITY_FILTERING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_MAC_SECURITY_FILTERING_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_MAC_SECURITY_FILTERING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_ADDITIONAL_RSS_MODES_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_ADDITIONAL_RSS_MODES_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_ADDITIONAL_RSS_MODES_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_QBB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_QBB_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_QBB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_PACKED_STREAM_VAR_BUFFERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_PACKED_STREAM_VAR_BUFFERS_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_PACKED_STREAM_VAR_BUFFERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_RSS_LIMITED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_RSS_LIMITED_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_RSS_LIMITED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_PACKED_STREAM_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_PACKED_STREAM_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_INCLUDE_FCS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_INCLUDE_FCS_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_INCLUDE_FCS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_VLAN_INSERTION_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_VLAN_INSERTION_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_VLAN_INSERTION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_VLAN_STRIPPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_VLAN_STRIPPING_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_VLAN_STRIPPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TSO_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TSO_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TSO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_PREFIX_LEN_0_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_PREFIX_LEN_0_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_PREFIX_LEN_0_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_PREFIX_LEN_14_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_PREFIX_LEN_14_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_PREFIX_LEN_14_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_TIMESTAMP_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_TIMESTAMP_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_BATCHING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_BATCHING_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_BATCHING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MCAST_FILTER_CHAINING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MCAST_FILTER_CHAINING_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MCAST_FILTER_CHAINING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_PM_AND_RXDP_COUNTERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_PM_AND_RXDP_COUNTERS_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_PM_AND_RXDP_COUNTERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_DISABLE_SCATTER_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_DISABLE_SCATTER_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_DISABLE_SCATTER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_MCAST_UDP_LOOPBACK_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_MCAST_UDP_LOOPBACK_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_MCAST_UDP_LOOPBACK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EVB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EVB_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EVB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VXLAN_NVGRE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VXLAN_NVGRE_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VXLAN_NVGRE_WIDTH 1
/* RxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_RX_DPCPU_FW_ID_OFST 4
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_RX_DPCPU_FW_ID_LEN 2
/* enum: Standard RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP 0x0
/* enum: Low latency RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_LOW_LATENCY 0x1
/* enum: Packed stream RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_PACKED_STREAM 0x2
/* enum: Rules engine RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_RULES_ENGINE 0x5
/* enum: DPDK RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_DPDK 0x6
/* enum: BIST RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_BIST 0x10a
/* enum: RXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_TEST_FW_TO_MC_CUT_THROUGH 0x101
/* enum: RXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD 0x102
/* enum: RXDP Test firmware image 3 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD_FIRST 0x103
/* enum: RXDP Test firmware image 4 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_TEST_EVERY_EVENT_BATCHABLE 0x104
/* enum: RXDP Test firmware image 5 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_TEST_BACKPRESSURE 0x105
/* enum: RXDP Test firmware image 6 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_TEST_FW_PACKET_EDITS 0x106
/* enum: RXDP Test firmware image 7 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_TEST_FW_RX_HDR_SPLIT 0x107
/* enum: RXDP Test firmware image 8 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_TEST_FW_DISABLE_DL 0x108
/* enum: RXDP Test firmware image 9 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_TEST_FW_DOORBELL_DELAY 0x10b
/* enum: RXDP Test firmware image 10 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_TEST_FW_SLOW 0x10c
/* TxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TX_DPCPU_FW_ID_OFST 6
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TX_DPCPU_FW_ID_LEN 2
/* enum: Standard TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXDP 0x0
/* enum: Low latency TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXDP_LOW_LATENCY 0x1
/* enum: High packet rate TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXDP_HIGH_PACKET_RATE 0x3
/* enum: Rules engine TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXDP_RULES_ENGINE 0x5
/* enum: DPDK TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXDP_DPDK 0x6
/* enum: BIST TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXDP_BIST 0x12d
/* enum: TXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXDP_TEST_FW_TSO_EDIT 0x101
/* enum: TXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXDP_TEST_FW_PACKET_EDITS 0x102
/* enum: TXDP CSR bus test firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXDP_TEST_FW_CSR 0x103
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_VERSION_OFST 8
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_VERSION_REV_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_VERSION_TYPE_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial RX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: RX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: RX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant RX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
/* enum: Low latency RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_TYPE_LOW_LATENCY 0x5
/* enum: Packed stream RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_TYPE_PACKED_STREAM 0x6
/* enum: RX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* enum: RX PD firmware parsing but not filtering network overlay tunnel
 * encapsulations (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_RXPD_FW_TYPE_TESTFW_ENCAP_PARSING_ONLY 0xf
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_VERSION_OFST 10
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_VERSION_REV_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_VERSION_TYPE_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial TX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: TX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: TX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant TX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_TYPE_LOW_LATENCY 0x5 /* enum */
/* enum: TX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_TXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* Hardware capabilities of NIC */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_HW_CAPABILITIES_OFST 12
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_HW_CAPABILITIES_LEN 4
/* Licensed capabilities */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_LICENSE_CAPABILITIES_OFST 16
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_LICENSE_CAPABILITIES_LEN 4
/* Second word of flags. Not present on older firmware (check the length). */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_FLAGS2_OFST 20
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_FLAGS2_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TSO_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TSO_V2_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TSO_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TSO_V2_ENCAP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TSO_V2_ENCAP_LBN 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TSO_V2_ENCAP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EVQ_TIMER_CTRL_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EVQ_TIMER_CTRL_LBN 2
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EVQ_TIMER_CTRL_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EVENT_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EVENT_CUT_THROUGH_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EVENT_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_CUT_THROUGH_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_VFIFO_ULL_MODE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_VFIFO_ULL_MODE_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_VFIFO_ULL_MODE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MAC_STATS_40G_TX_SIZE_BINS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MAC_STATS_40G_TX_SIZE_BINS_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MAC_STATS_40G_TX_SIZE_BINS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_INIT_EVQ_TYPE_SUPPORTED_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_INIT_EVQ_TYPE_SUPPORTED_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_INIT_EVQ_TYPE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_INIT_EVQ_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_INIT_EVQ_V2_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_INIT_EVQ_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_MAC_TIMESTAMPING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_MAC_TIMESTAMPING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_MAC_TIMESTAMPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TIMESTAMP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TIMESTAMP_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_SNIFF_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_SNIFF_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MCDI_BACKGROUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MCDI_BACKGROUND_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MCDI_BACKGROUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MCDI_DB_RETURN_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MCDI_DB_RETURN_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MCDI_DB_RETURN_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_CTPIO_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_CTPIO_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_CTPIO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TSA_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TSA_SUPPORT_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TSA_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TSA_BOUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TSA_BOUND_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TSA_BOUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_SF_ADAPTER_AUTHENTICATION_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_SF_ADAPTER_AUTHENTICATION_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_SF_ADAPTER_AUTHENTICATION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_FILTER_ACTION_FLAG_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_FILTER_ACTION_FLAG_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_FILTER_ACTION_FLAG_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_FILTER_ACTION_MARK_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_FILTER_ACTION_MARK_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_FILTER_ACTION_MARK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EQUAL_STRIDE_SUPER_BUFFER_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EQUAL_STRIDE_SUPER_BUFFER_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EQUAL_STRIDE_SUPER_BUFFER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EQUAL_STRIDE_PACKED_STREAM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EQUAL_STRIDE_PACKED_STREAM_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EQUAL_STRIDE_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_L3XUDP_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_L3XUDP_SUPPORT_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_L3XUDP_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_FW_SUBVARIANT_NO_TX_CSUM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_FW_SUBVARIANT_NO_TX_CSUM_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_FW_SUBVARIANT_NO_TX_CSUM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VI_SPREADING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VI_SPREADING_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VI_SPREADING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_HLB_IDLE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_HLB_IDLE_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RXDP_HLB_IDLE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_INIT_RXQ_NO_CONT_EV_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_INIT_RXQ_NO_CONT_EV_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_INIT_RXQ_NO_CONT_EV_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_INIT_RXQ_WITH_BUFFER_SIZE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_INIT_RXQ_WITH_BUFFER_SIZE_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_INIT_RXQ_WITH_BUFFER_SIZE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_BUNDLE_UPDATE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_BUNDLE_UPDATE_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_BUNDLE_UPDATE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TSO_V3_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TSO_V3_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TSO_V3_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_DYNAMIC_SENSORS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_DYNAMIC_SENSORS_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_DYNAMIC_SENSORS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_WIDTH 1
/* Number of FATSOv2 contexts per datapath supported by this NIC (when
 * TX_TSO_V2 == 1). Not present on older firmware (check the length).
 */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TSO_V2_N_CONTEXTS_OFST 24
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TX_TSO_V2_N_CONTEXTS_LEN 2
/* One byte per PF containing the number of the external port assigned to this
 * PF, indexed by PF number. Special values indicate that a PF is either not
 * present or not assigned.
 */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_PFS_TO_PORTS_ASSIGNMENT_OFST 26
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_PFS_TO_PORTS_ASSIGNMENT_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_PFS_TO_PORTS_ASSIGNMENT_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_ACCESS_NOT_PERMITTED 0xff
/* enum: PF does not exist. */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_PF_NOT_PRESENT 0xfe
/* enum: PF does exist but is not assigned to any external port. */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_PF_NOT_ASSIGNED 0xfd
/* enum: This value indicates that PF is assigned, but it cannot be expressed
 * in this field. It is intended for a possible future situation where a more
 * complex scheme of PFs to ports mapping is being used. The future driver
 * should look for a new field supporting the new scheme. The current/old
 * driver should treat this value as PF_NOT_ASSIGNED.
 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_INCOMPATIBLE_ASSIGNMENT 0xfc
/* One byte per PF containing the number of its VFs, indexed by PF number. A
 * special value indicates that a PF is not present.
 */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_NUM_VFS_PER_PF_OFST 42
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_NUM_VFS_PER_PF_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_NUM_VFS_PER_PF_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
/*               MC_CMD_GET_CAPABILITIES_V8_OUT_ACCESS_NOT_PERMITTED 0xff */
/* enum: PF does not exist. */
/*               MC_CMD_GET_CAPABILITIES_V8_OUT_PF_NOT_PRESENT 0xfe */
/* Number of VIs available for external ports 0-3. For devices with more than
 * four ports, the remainder are in NUM_VIS_PER_PORT2 in
 * GET_CAPABILITIES_V12_OUT.
 */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_NUM_VIS_PER_PORT_OFST 58
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_NUM_VIS_PER_PORT_LEN 2
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_NUM_VIS_PER_PORT_NUM 4
/* Size of RX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ RX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_RX_DESC_CACHE_SIZE_OFST 66
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_RX_DESC_CACHE_SIZE_LEN 1
/* Size of TX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ TX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TX_DESC_CACHE_SIZE_OFST 67
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TX_DESC_CACHE_SIZE_LEN 1
/* Total number of available PIO buffers */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_NUM_PIO_BUFFS_OFST 68
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_NUM_PIO_BUFFS_LEN 2
/* Size of a single PIO buffer */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_SIZE_PIO_BUFF_OFST 70
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_SIZE_PIO_BUFF_LEN 2
/* On chips later than Medford the amount of address space assigned to each VI
 * is configurable. This is a global setting that the driver must query to
 * discover the VI to address mapping. Cut-through PIO (CTPIO) is not available
 * with 8k VI windows.
 */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_VI_WINDOW_MODE_OFST 72
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_VI_WINDOW_MODE_LEN 1
/* enum: Each VI occupies 8k as on Huntington and Medford. PIO is at offset 4k.
 * CTPIO is not mapped.
 */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_VI_WINDOW_MODE_8K 0x0
/* enum: Each VI occupies 16k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_VI_WINDOW_MODE_16K 0x1
/* enum: Each VI occupies 64k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V8_OUT_VI_WINDOW_MODE_64K 0x2
/* Number of vFIFOs per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_VFIFO_STUFFING_NUM_VFIFOS_OFST 73
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_VFIFO_STUFFING_NUM_VFIFOS_LEN 1
/* Number of buffers per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_OFST 74
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_LEN 2
/* Entry count in the MAC stats array, including the final GENERATION_END
 * entry. For MAC stats DMA, drivers should allocate a buffer large enough to
 * hold at least this many 64-bit stats values, if they wish to receive all
 * available stats. If the buffer is shorter than MAC_STATS_NUM_STATS * 8, the
 * stats array returned will be truncated.
 */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_MAC_STATS_NUM_STATS_OFST 76
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_MAC_STATS_NUM_STATS_LEN 2
/* Maximum supported value for MC_CMD_FILTER_OP_V3/MATCH_MARK_VALUE. This field
 * will only be non-zero if MC_CMD_GET_CAPABILITIES/FILTER_ACTION_MARK is set.
 */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_FILTER_ACTION_MARK_MAX_OFST 80
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_FILTER_ACTION_MARK_MAX_LEN 4
/* On devices where the INIT_RXQ_WITH_BUFFER_SIZE flag (in
 * GET_CAPABILITIES_OUT_V2) is set, drivers have to specify a buffer size when
 * they create an RX queue. Due to hardware limitations, only a small number of
 * different buffer sizes may be available concurrently. Nonzero entries in
 * this array are the sizes of buffers which the system guarantees will be
 * available for use. If the list is empty, there are no limitations on
 * concurrent buffer sizes.
 */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_GUARANTEED_RX_BUFFER_SIZES_OFST 84
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_GUARANTEED_RX_BUFFER_SIZES_LEN 4
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_GUARANTEED_RX_BUFFER_SIZES_NUM 16
/* Third word of flags. Not present on older firmware (check the length). */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_FLAGS3_OFST 148
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_FLAGS3_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_WOL_ETHERWAKE_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_WOL_ETHERWAKE_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_WOL_ETHERWAKE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RSS_EVEN_SPREADING_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RSS_EVEN_SPREADING_LBN 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RSS_EVEN_SPREADING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RSS_SELECTABLE_TABLE_SIZE_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RSS_SELECTABLE_TABLE_SIZE_LBN 2
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RSS_SELECTABLE_TABLE_SIZE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MAE_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MAE_SUPPORTED_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MAE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VDPA_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VDPA_SUPPORTED_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_VDPA_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_VLAN_STRIPPING_PER_ENCAP_RULE_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_VLAN_STRIPPING_PER_ENCAP_RULE_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RX_VLAN_STRIPPING_PER_ENCAP_RULE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EXTENDED_WIDTH_EVQS_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EXTENDED_WIDTH_EVQS_SUPPORTED_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EXTENDED_WIDTH_EVQS_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_UNSOL_EV_CREDIT_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_UNSOL_EV_CREDIT_SUPPORTED_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_UNSOL_EV_CREDIT_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_ENCAPSULATED_MCDI_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_ENCAPSULATED_MCDI_SUPPORTED_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_ENCAPSULATED_MCDI_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EXTERNAL_MAE_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EXTERNAL_MAE_SUPPORTED_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_EXTERNAL_MAE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_NVRAM_UPDATE_ABORT_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_NVRAM_UPDATE_ABORT_SUPPORTED_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_NVRAM_UPDATE_ABORT_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MAE_ACTION_SET_ALLOC_V2_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MAE_ACTION_SET_ALLOC_V2_SUPPORTED_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MAE_ACTION_SET_ALLOC_V2_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RSS_STEER_ON_OUTER_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RSS_STEER_ON_OUTER_SUPPORTED_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_RSS_STEER_ON_OUTER_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MAE_ACTION_SET_ALLOC_V3_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MAE_ACTION_SET_ALLOC_V3_SUPPORTED_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_MAE_ACTION_SET_ALLOC_V3_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_DYNAMIC_MPORT_JOURNAL_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_DYNAMIC_MPORT_JOURNAL_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_DYNAMIC_MPORT_JOURNAL_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_CLIENT_CMD_VF_PROXY_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_CLIENT_CMD_VF_PROXY_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_CLIENT_CMD_VF_PROXY_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_LL_RX_EVENT_SUPPRESSION_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_LL_RX_EVENT_SUPPRESSION_SUPPORTED_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_LL_RX_EVENT_SUPPRESSION_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_CXL_CONFIG_ENABLE_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_CXL_CONFIG_ENABLE_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V8_OUT_CXL_CONFIG_ENABLE_WIDTH 1
/* These bits are reserved for communicating test-specific capabilities to
 * host-side test software. All production drivers should treat this field as
 * opaque.
 */
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TEST_RESERVED_OFST 152
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TEST_RESERVED_LEN 8
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TEST_RESERVED_LO_OFST 152
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TEST_RESERVED_LO_LEN 4
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TEST_RESERVED_LO_LBN 1216
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TEST_RESERVED_LO_WIDTH 32
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TEST_RESERVED_HI_OFST 156
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TEST_RESERVED_HI_LEN 4
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TEST_RESERVED_HI_LBN 1248
#define       MC_CMD_GET_CAPABILITIES_V8_OUT_TEST_RESERVED_HI_WIDTH 32

/* MC_CMD_GET_CAPABILITIES_V9_OUT msgresponse */
#define    MC_CMD_GET_CAPABILITIES_V9_OUT_LEN 184
/* First word of flags. */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_FLAGS1_OFST 0
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_FLAGS1_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VPORT_RECONFIGURE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VPORT_RECONFIGURE_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VPORT_RECONFIGURE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_STRIPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_STRIPING_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_STRIPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VADAPTOR_QUERY_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VADAPTOR_QUERY_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VADAPTOR_QUERY_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EVB_PORT_VLAN_RESTRICT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EVB_PORT_VLAN_RESTRICT_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EVB_PORT_VLAN_RESTRICT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_DRV_ATTACH_PREBOOT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_DRV_ATTACH_PREBOOT_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_DRV_ATTACH_PREBOOT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_FORCE_EVENT_MERGING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_FORCE_EVENT_MERGING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_FORCE_EVENT_MERGING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_SET_MAC_ENHANCED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_SET_MAC_ENHANCED_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_SET_MAC_ENHANCED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_MAC_SECURITY_FILTERING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_MAC_SECURITY_FILTERING_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_MAC_SECURITY_FILTERING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_ADDITIONAL_RSS_MODES_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_ADDITIONAL_RSS_MODES_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_ADDITIONAL_RSS_MODES_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_QBB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_QBB_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_QBB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_PACKED_STREAM_VAR_BUFFERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_PACKED_STREAM_VAR_BUFFERS_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_PACKED_STREAM_VAR_BUFFERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_RSS_LIMITED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_RSS_LIMITED_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_RSS_LIMITED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_PACKED_STREAM_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_PACKED_STREAM_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_INCLUDE_FCS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_INCLUDE_FCS_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_INCLUDE_FCS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_VLAN_INSERTION_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_VLAN_INSERTION_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_VLAN_INSERTION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_VLAN_STRIPPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_VLAN_STRIPPING_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_VLAN_STRIPPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TSO_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TSO_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TSO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_PREFIX_LEN_0_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_PREFIX_LEN_0_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_PREFIX_LEN_0_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_PREFIX_LEN_14_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_PREFIX_LEN_14_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_PREFIX_LEN_14_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_TIMESTAMP_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_TIMESTAMP_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_BATCHING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_BATCHING_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_BATCHING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MCAST_FILTER_CHAINING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MCAST_FILTER_CHAINING_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MCAST_FILTER_CHAINING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_PM_AND_RXDP_COUNTERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_PM_AND_RXDP_COUNTERS_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_PM_AND_RXDP_COUNTERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_DISABLE_SCATTER_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_DISABLE_SCATTER_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_DISABLE_SCATTER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_MCAST_UDP_LOOPBACK_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_MCAST_UDP_LOOPBACK_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_MCAST_UDP_LOOPBACK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EVB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EVB_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EVB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VXLAN_NVGRE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VXLAN_NVGRE_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VXLAN_NVGRE_WIDTH 1
/* RxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RX_DPCPU_FW_ID_OFST 4
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RX_DPCPU_FW_ID_LEN 2
/* enum: Standard RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP 0x0
/* enum: Low latency RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_LOW_LATENCY 0x1
/* enum: Packed stream RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_PACKED_STREAM 0x2
/* enum: Rules engine RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_RULES_ENGINE 0x5
/* enum: DPDK RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_DPDK 0x6
/* enum: BIST RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_BIST 0x10a
/* enum: RXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_TEST_FW_TO_MC_CUT_THROUGH 0x101
/* enum: RXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD 0x102
/* enum: RXDP Test firmware image 3 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD_FIRST 0x103
/* enum: RXDP Test firmware image 4 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_TEST_EVERY_EVENT_BATCHABLE 0x104
/* enum: RXDP Test firmware image 5 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_TEST_BACKPRESSURE 0x105
/* enum: RXDP Test firmware image 6 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_TEST_FW_PACKET_EDITS 0x106
/* enum: RXDP Test firmware image 7 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_TEST_FW_RX_HDR_SPLIT 0x107
/* enum: RXDP Test firmware image 8 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_TEST_FW_DISABLE_DL 0x108
/* enum: RXDP Test firmware image 9 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_TEST_FW_DOORBELL_DELAY 0x10b
/* enum: RXDP Test firmware image 10 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_TEST_FW_SLOW 0x10c
/* TxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TX_DPCPU_FW_ID_OFST 6
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TX_DPCPU_FW_ID_LEN 2
/* enum: Standard TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXDP 0x0
/* enum: Low latency TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXDP_LOW_LATENCY 0x1
/* enum: High packet rate TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXDP_HIGH_PACKET_RATE 0x3
/* enum: Rules engine TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXDP_RULES_ENGINE 0x5
/* enum: DPDK TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXDP_DPDK 0x6
/* enum: BIST TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXDP_BIST 0x12d
/* enum: TXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXDP_TEST_FW_TSO_EDIT 0x101
/* enum: TXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXDP_TEST_FW_PACKET_EDITS 0x102
/* enum: TXDP CSR bus test firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXDP_TEST_FW_CSR 0x103
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_VERSION_OFST 8
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_VERSION_REV_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_VERSION_TYPE_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial RX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: RX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: RX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant RX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
/* enum: Low latency RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_TYPE_LOW_LATENCY 0x5
/* enum: Packed stream RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_TYPE_PACKED_STREAM 0x6
/* enum: RX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* enum: RX PD firmware parsing but not filtering network overlay tunnel
 * encapsulations (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_RXPD_FW_TYPE_TESTFW_ENCAP_PARSING_ONLY 0xf
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_VERSION_OFST 10
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_VERSION_REV_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_VERSION_TYPE_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial TX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: TX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: TX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant TX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_TYPE_LOW_LATENCY 0x5 /* enum */
/* enum: TX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_TXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* Hardware capabilities of NIC */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_HW_CAPABILITIES_OFST 12
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_HW_CAPABILITIES_LEN 4
/* Licensed capabilities */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_LICENSE_CAPABILITIES_OFST 16
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_LICENSE_CAPABILITIES_LEN 4
/* Second word of flags. Not present on older firmware (check the length). */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_FLAGS2_OFST 20
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_FLAGS2_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TSO_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TSO_V2_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TSO_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TSO_V2_ENCAP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TSO_V2_ENCAP_LBN 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TSO_V2_ENCAP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EVQ_TIMER_CTRL_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EVQ_TIMER_CTRL_LBN 2
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EVQ_TIMER_CTRL_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EVENT_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EVENT_CUT_THROUGH_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EVENT_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_CUT_THROUGH_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_VFIFO_ULL_MODE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_VFIFO_ULL_MODE_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_VFIFO_ULL_MODE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MAC_STATS_40G_TX_SIZE_BINS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MAC_STATS_40G_TX_SIZE_BINS_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MAC_STATS_40G_TX_SIZE_BINS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_INIT_EVQ_TYPE_SUPPORTED_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_INIT_EVQ_TYPE_SUPPORTED_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_INIT_EVQ_TYPE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_INIT_EVQ_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_INIT_EVQ_V2_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_INIT_EVQ_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_MAC_TIMESTAMPING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_MAC_TIMESTAMPING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_MAC_TIMESTAMPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TIMESTAMP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TIMESTAMP_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_SNIFF_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_SNIFF_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MCDI_BACKGROUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MCDI_BACKGROUND_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MCDI_BACKGROUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MCDI_DB_RETURN_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MCDI_DB_RETURN_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MCDI_DB_RETURN_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_CTPIO_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_CTPIO_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_CTPIO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TSA_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TSA_SUPPORT_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TSA_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TSA_BOUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TSA_BOUND_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TSA_BOUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_SF_ADAPTER_AUTHENTICATION_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_SF_ADAPTER_AUTHENTICATION_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_SF_ADAPTER_AUTHENTICATION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_FILTER_ACTION_FLAG_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_FILTER_ACTION_FLAG_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_FILTER_ACTION_FLAG_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_FILTER_ACTION_MARK_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_FILTER_ACTION_MARK_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_FILTER_ACTION_MARK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EQUAL_STRIDE_SUPER_BUFFER_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EQUAL_STRIDE_SUPER_BUFFER_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EQUAL_STRIDE_SUPER_BUFFER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EQUAL_STRIDE_PACKED_STREAM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EQUAL_STRIDE_PACKED_STREAM_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EQUAL_STRIDE_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_L3XUDP_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_L3XUDP_SUPPORT_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_L3XUDP_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_FW_SUBVARIANT_NO_TX_CSUM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_FW_SUBVARIANT_NO_TX_CSUM_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_FW_SUBVARIANT_NO_TX_CSUM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VI_SPREADING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VI_SPREADING_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VI_SPREADING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_HLB_IDLE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_HLB_IDLE_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RXDP_HLB_IDLE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_INIT_RXQ_NO_CONT_EV_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_INIT_RXQ_NO_CONT_EV_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_INIT_RXQ_NO_CONT_EV_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_INIT_RXQ_WITH_BUFFER_SIZE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_INIT_RXQ_WITH_BUFFER_SIZE_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_INIT_RXQ_WITH_BUFFER_SIZE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_BUNDLE_UPDATE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_BUNDLE_UPDATE_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_BUNDLE_UPDATE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TSO_V3_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TSO_V3_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TSO_V3_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_DYNAMIC_SENSORS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_DYNAMIC_SENSORS_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_DYNAMIC_SENSORS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_WIDTH 1
/* Number of FATSOv2 contexts per datapath supported by this NIC (when
 * TX_TSO_V2 == 1). Not present on older firmware (check the length).
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TSO_V2_N_CONTEXTS_OFST 24
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TX_TSO_V2_N_CONTEXTS_LEN 2
/* One byte per PF containing the number of the external port assigned to this
 * PF, indexed by PF number. Special values indicate that a PF is either not
 * present or not assigned.
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_PFS_TO_PORTS_ASSIGNMENT_OFST 26
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_PFS_TO_PORTS_ASSIGNMENT_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_PFS_TO_PORTS_ASSIGNMENT_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_ACCESS_NOT_PERMITTED 0xff
/* enum: PF does not exist. */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_PF_NOT_PRESENT 0xfe
/* enum: PF does exist but is not assigned to any external port. */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_PF_NOT_ASSIGNED 0xfd
/* enum: This value indicates that PF is assigned, but it cannot be expressed
 * in this field. It is intended for a possible future situation where a more
 * complex scheme of PFs to ports mapping is being used. The future driver
 * should look for a new field supporting the new scheme. The current/old
 * driver should treat this value as PF_NOT_ASSIGNED.
 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_INCOMPATIBLE_ASSIGNMENT 0xfc
/* One byte per PF containing the number of its VFs, indexed by PF number. A
 * special value indicates that a PF is not present.
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_NUM_VFS_PER_PF_OFST 42
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_NUM_VFS_PER_PF_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_NUM_VFS_PER_PF_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
/*               MC_CMD_GET_CAPABILITIES_V9_OUT_ACCESS_NOT_PERMITTED 0xff */
/* enum: PF does not exist. */
/*               MC_CMD_GET_CAPABILITIES_V9_OUT_PF_NOT_PRESENT 0xfe */
/* Number of VIs available for external ports 0-3. For devices with more than
 * four ports, the remainder are in NUM_VIS_PER_PORT2 in
 * GET_CAPABILITIES_V12_OUT.
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_NUM_VIS_PER_PORT_OFST 58
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_NUM_VIS_PER_PORT_LEN 2
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_NUM_VIS_PER_PORT_NUM 4
/* Size of RX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ RX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RX_DESC_CACHE_SIZE_OFST 66
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RX_DESC_CACHE_SIZE_LEN 1
/* Size of TX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ TX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TX_DESC_CACHE_SIZE_OFST 67
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TX_DESC_CACHE_SIZE_LEN 1
/* Total number of available PIO buffers */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_NUM_PIO_BUFFS_OFST 68
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_NUM_PIO_BUFFS_LEN 2
/* Size of a single PIO buffer */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_SIZE_PIO_BUFF_OFST 70
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_SIZE_PIO_BUFF_LEN 2
/* On chips later than Medford the amount of address space assigned to each VI
 * is configurable. This is a global setting that the driver must query to
 * discover the VI to address mapping. Cut-through PIO (CTPIO) is not available
 * with 8k VI windows.
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_VI_WINDOW_MODE_OFST 72
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_VI_WINDOW_MODE_LEN 1
/* enum: Each VI occupies 8k as on Huntington and Medford. PIO is at offset 4k.
 * CTPIO is not mapped.
 */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_VI_WINDOW_MODE_8K 0x0
/* enum: Each VI occupies 16k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_VI_WINDOW_MODE_16K 0x1
/* enum: Each VI occupies 64k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V9_OUT_VI_WINDOW_MODE_64K 0x2
/* Number of vFIFOs per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_VFIFO_STUFFING_NUM_VFIFOS_OFST 73
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_VFIFO_STUFFING_NUM_VFIFOS_LEN 1
/* Number of buffers per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_OFST 74
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_LEN 2
/* Entry count in the MAC stats array, including the final GENERATION_END
 * entry. For MAC stats DMA, drivers should allocate a buffer large enough to
 * hold at least this many 64-bit stats values, if they wish to receive all
 * available stats. If the buffer is shorter than MAC_STATS_NUM_STATS * 8, the
 * stats array returned will be truncated.
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_MAC_STATS_NUM_STATS_OFST 76
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_MAC_STATS_NUM_STATS_LEN 2
/* Maximum supported value for MC_CMD_FILTER_OP_V3/MATCH_MARK_VALUE. This field
 * will only be non-zero if MC_CMD_GET_CAPABILITIES/FILTER_ACTION_MARK is set.
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_FILTER_ACTION_MARK_MAX_OFST 80
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_FILTER_ACTION_MARK_MAX_LEN 4
/* On devices where the INIT_RXQ_WITH_BUFFER_SIZE flag (in
 * GET_CAPABILITIES_OUT_V2) is set, drivers have to specify a buffer size when
 * they create an RX queue. Due to hardware limitations, only a small number of
 * different buffer sizes may be available concurrently. Nonzero entries in
 * this array are the sizes of buffers which the system guarantees will be
 * available for use. If the list is empty, there are no limitations on
 * concurrent buffer sizes.
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_GUARANTEED_RX_BUFFER_SIZES_OFST 84
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_GUARANTEED_RX_BUFFER_SIZES_LEN 4
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_GUARANTEED_RX_BUFFER_SIZES_NUM 16
/* Third word of flags. Not present on older firmware (check the length). */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_FLAGS3_OFST 148
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_FLAGS3_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_WOL_ETHERWAKE_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_WOL_ETHERWAKE_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_WOL_ETHERWAKE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_EVEN_SPREADING_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_EVEN_SPREADING_LBN 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_EVEN_SPREADING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_SELECTABLE_TABLE_SIZE_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_SELECTABLE_TABLE_SIZE_LBN 2
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_SELECTABLE_TABLE_SIZE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MAE_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MAE_SUPPORTED_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MAE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VDPA_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VDPA_SUPPORTED_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_VDPA_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_VLAN_STRIPPING_PER_ENCAP_RULE_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_VLAN_STRIPPING_PER_ENCAP_RULE_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RX_VLAN_STRIPPING_PER_ENCAP_RULE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EXTENDED_WIDTH_EVQS_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EXTENDED_WIDTH_EVQS_SUPPORTED_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EXTENDED_WIDTH_EVQS_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_UNSOL_EV_CREDIT_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_UNSOL_EV_CREDIT_SUPPORTED_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_UNSOL_EV_CREDIT_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_ENCAPSULATED_MCDI_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_ENCAPSULATED_MCDI_SUPPORTED_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_ENCAPSULATED_MCDI_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EXTERNAL_MAE_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EXTERNAL_MAE_SUPPORTED_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_EXTERNAL_MAE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_NVRAM_UPDATE_ABORT_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_NVRAM_UPDATE_ABORT_SUPPORTED_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_NVRAM_UPDATE_ABORT_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MAE_ACTION_SET_ALLOC_V2_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MAE_ACTION_SET_ALLOC_V2_SUPPORTED_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MAE_ACTION_SET_ALLOC_V2_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_STEER_ON_OUTER_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_STEER_ON_OUTER_SUPPORTED_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_STEER_ON_OUTER_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MAE_ACTION_SET_ALLOC_V3_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MAE_ACTION_SET_ALLOC_V3_SUPPORTED_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_MAE_ACTION_SET_ALLOC_V3_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_DYNAMIC_MPORT_JOURNAL_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_DYNAMIC_MPORT_JOURNAL_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_DYNAMIC_MPORT_JOURNAL_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_CLIENT_CMD_VF_PROXY_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_CLIENT_CMD_VF_PROXY_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_CLIENT_CMD_VF_PROXY_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_LL_RX_EVENT_SUPPRESSION_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_LL_RX_EVENT_SUPPRESSION_SUPPORTED_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_LL_RX_EVENT_SUPPRESSION_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_CXL_CONFIG_ENABLE_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_CXL_CONFIG_ENABLE_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V9_OUT_CXL_CONFIG_ENABLE_WIDTH 1
/* These bits are reserved for communicating test-specific capabilities to
 * host-side test software. All production drivers should treat this field as
 * opaque.
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TEST_RESERVED_OFST 152
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TEST_RESERVED_LEN 8
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TEST_RESERVED_LO_OFST 152
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TEST_RESERVED_LO_LEN 4
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TEST_RESERVED_LO_LBN 1216
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TEST_RESERVED_LO_WIDTH 32
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TEST_RESERVED_HI_OFST 156
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TEST_RESERVED_HI_LEN 4
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TEST_RESERVED_HI_LBN 1248
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_TEST_RESERVED_HI_WIDTH 32
/* The minimum size (in table entries) of indirection table to be allocated
 * from the pool for an RSS context. Note that the table size used must be a
 * power of 2.
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_MIN_INDIRECTION_TABLE_SIZE_OFST 160
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_MIN_INDIRECTION_TABLE_SIZE_LEN 4
/* The maximum size (in table entries) of indirection table to be allocated
 * from the pool for an RSS context. Note that the table size used must be a
 * power of 2.
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_MAX_INDIRECTION_TABLE_SIZE_OFST 164
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_MAX_INDIRECTION_TABLE_SIZE_LEN 4
/* The maximum number of queues that can be used by an RSS context in exclusive
 * mode. In exclusive mode the context has a configurable indirection table and
 * a configurable RSS key.
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_MAX_INDIRECTION_QUEUES_OFST 168
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_MAX_INDIRECTION_QUEUES_LEN 4
/* The maximum number of queues that can be used by an RSS context in even-
 * spreading mode. In even-spreading mode the context has no indirection table
 * but it does have a configurable RSS key.
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_MAX_EVEN_SPREADING_QUEUES_OFST 172
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_MAX_EVEN_SPREADING_QUEUES_LEN 4
/* The total number of RSS contexts supported. Note that the number of
 * available contexts using indirection tables is also limited by the
 * availability of indirection table space allocated from a common pool.
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_NUM_CONTEXTS_OFST 176
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_NUM_CONTEXTS_LEN 4
/* The total amount of indirection table space that can be shared between RSS
 * contexts.
 */
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_TABLE_POOL_SIZE_OFST 180
#define       MC_CMD_GET_CAPABILITIES_V9_OUT_RSS_TABLE_POOL_SIZE_LEN 4

/* MC_CMD_GET_CAPABILITIES_V10_OUT msgresponse */
#define    MC_CMD_GET_CAPABILITIES_V10_OUT_LEN 192
/* First word of flags. */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_FLAGS1_OFST 0
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_FLAGS1_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VPORT_RECONFIGURE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VPORT_RECONFIGURE_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VPORT_RECONFIGURE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_STRIPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_STRIPING_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_STRIPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VADAPTOR_QUERY_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VADAPTOR_QUERY_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VADAPTOR_QUERY_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EVB_PORT_VLAN_RESTRICT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EVB_PORT_VLAN_RESTRICT_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EVB_PORT_VLAN_RESTRICT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_DRV_ATTACH_PREBOOT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_DRV_ATTACH_PREBOOT_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_DRV_ATTACH_PREBOOT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_FORCE_EVENT_MERGING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_FORCE_EVENT_MERGING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_FORCE_EVENT_MERGING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_SET_MAC_ENHANCED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_SET_MAC_ENHANCED_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_SET_MAC_ENHANCED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_MAC_SECURITY_FILTERING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_MAC_SECURITY_FILTERING_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_MAC_SECURITY_FILTERING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_ADDITIONAL_RSS_MODES_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_ADDITIONAL_RSS_MODES_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_ADDITIONAL_RSS_MODES_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_QBB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_QBB_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_QBB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_PACKED_STREAM_VAR_BUFFERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_PACKED_STREAM_VAR_BUFFERS_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_PACKED_STREAM_VAR_BUFFERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_RSS_LIMITED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_RSS_LIMITED_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_RSS_LIMITED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_PACKED_STREAM_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_PACKED_STREAM_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_INCLUDE_FCS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_INCLUDE_FCS_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_INCLUDE_FCS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_VLAN_INSERTION_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_VLAN_INSERTION_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_VLAN_INSERTION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_VLAN_STRIPPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_VLAN_STRIPPING_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_VLAN_STRIPPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TSO_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TSO_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TSO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_PREFIX_LEN_0_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_PREFIX_LEN_0_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_PREFIX_LEN_0_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_PREFIX_LEN_14_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_PREFIX_LEN_14_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_PREFIX_LEN_14_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_TIMESTAMP_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_TIMESTAMP_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_BATCHING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_BATCHING_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_BATCHING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MCAST_FILTER_CHAINING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MCAST_FILTER_CHAINING_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MCAST_FILTER_CHAINING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_PM_AND_RXDP_COUNTERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_PM_AND_RXDP_COUNTERS_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_PM_AND_RXDP_COUNTERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_DISABLE_SCATTER_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_DISABLE_SCATTER_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_DISABLE_SCATTER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_MCAST_UDP_LOOPBACK_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_MCAST_UDP_LOOPBACK_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_MCAST_UDP_LOOPBACK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EVB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EVB_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EVB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VXLAN_NVGRE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VXLAN_NVGRE_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VXLAN_NVGRE_WIDTH 1
/* RxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RX_DPCPU_FW_ID_OFST 4
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RX_DPCPU_FW_ID_LEN 2
/* enum: Standard RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP 0x0
/* enum: Low latency RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_LOW_LATENCY 0x1
/* enum: Packed stream RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_PACKED_STREAM 0x2
/* enum: Rules engine RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_RULES_ENGINE 0x5
/* enum: DPDK RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_DPDK 0x6
/* enum: BIST RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_BIST 0x10a
/* enum: RXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_TEST_FW_TO_MC_CUT_THROUGH 0x101
/* enum: RXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD 0x102
/* enum: RXDP Test firmware image 3 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD_FIRST 0x103
/* enum: RXDP Test firmware image 4 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_TEST_EVERY_EVENT_BATCHABLE 0x104
/* enum: RXDP Test firmware image 5 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_TEST_BACKPRESSURE 0x105
/* enum: RXDP Test firmware image 6 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_TEST_FW_PACKET_EDITS 0x106
/* enum: RXDP Test firmware image 7 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_TEST_FW_RX_HDR_SPLIT 0x107
/* enum: RXDP Test firmware image 8 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_TEST_FW_DISABLE_DL 0x108
/* enum: RXDP Test firmware image 9 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_TEST_FW_DOORBELL_DELAY 0x10b
/* enum: RXDP Test firmware image 10 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_TEST_FW_SLOW 0x10c
/* TxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TX_DPCPU_FW_ID_OFST 6
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TX_DPCPU_FW_ID_LEN 2
/* enum: Standard TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXDP 0x0
/* enum: Low latency TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXDP_LOW_LATENCY 0x1
/* enum: High packet rate TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXDP_HIGH_PACKET_RATE 0x3
/* enum: Rules engine TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXDP_RULES_ENGINE 0x5
/* enum: DPDK TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXDP_DPDK 0x6
/* enum: BIST TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXDP_BIST 0x12d
/* enum: TXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXDP_TEST_FW_TSO_EDIT 0x101
/* enum: TXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXDP_TEST_FW_PACKET_EDITS 0x102
/* enum: TXDP CSR bus test firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXDP_TEST_FW_CSR 0x103
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_VERSION_OFST 8
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_VERSION_REV_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_VERSION_TYPE_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial RX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: RX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: RX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant RX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
/* enum: Low latency RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_TYPE_LOW_LATENCY 0x5
/* enum: Packed stream RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_TYPE_PACKED_STREAM 0x6
/* enum: RX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* enum: RX PD firmware parsing but not filtering network overlay tunnel
 * encapsulations (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_RXPD_FW_TYPE_TESTFW_ENCAP_PARSING_ONLY 0xf
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_VERSION_OFST 10
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_VERSION_REV_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_VERSION_TYPE_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial TX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: TX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: TX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant TX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_TYPE_LOW_LATENCY 0x5 /* enum */
/* enum: TX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_TXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* Hardware capabilities of NIC */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_HW_CAPABILITIES_OFST 12
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_HW_CAPABILITIES_LEN 4
/* Licensed capabilities */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_LICENSE_CAPABILITIES_OFST 16
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_LICENSE_CAPABILITIES_LEN 4
/* Second word of flags. Not present on older firmware (check the length). */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_FLAGS2_OFST 20
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_FLAGS2_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TSO_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TSO_V2_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TSO_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TSO_V2_ENCAP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TSO_V2_ENCAP_LBN 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TSO_V2_ENCAP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EVQ_TIMER_CTRL_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EVQ_TIMER_CTRL_LBN 2
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EVQ_TIMER_CTRL_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EVENT_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EVENT_CUT_THROUGH_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EVENT_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_CUT_THROUGH_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_VFIFO_ULL_MODE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_VFIFO_ULL_MODE_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_VFIFO_ULL_MODE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MAC_STATS_40G_TX_SIZE_BINS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MAC_STATS_40G_TX_SIZE_BINS_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MAC_STATS_40G_TX_SIZE_BINS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_INIT_EVQ_TYPE_SUPPORTED_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_INIT_EVQ_TYPE_SUPPORTED_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_INIT_EVQ_TYPE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_INIT_EVQ_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_INIT_EVQ_V2_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_INIT_EVQ_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_MAC_TIMESTAMPING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_MAC_TIMESTAMPING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_MAC_TIMESTAMPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TIMESTAMP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TIMESTAMP_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_SNIFF_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_SNIFF_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MCDI_BACKGROUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MCDI_BACKGROUND_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MCDI_BACKGROUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MCDI_DB_RETURN_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MCDI_DB_RETURN_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MCDI_DB_RETURN_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_CTPIO_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_CTPIO_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_CTPIO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TSA_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TSA_SUPPORT_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TSA_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TSA_BOUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TSA_BOUND_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TSA_BOUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_SF_ADAPTER_AUTHENTICATION_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_SF_ADAPTER_AUTHENTICATION_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_SF_ADAPTER_AUTHENTICATION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_FILTER_ACTION_FLAG_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_FILTER_ACTION_FLAG_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_FILTER_ACTION_FLAG_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_FILTER_ACTION_MARK_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_FILTER_ACTION_MARK_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_FILTER_ACTION_MARK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EQUAL_STRIDE_SUPER_BUFFER_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EQUAL_STRIDE_SUPER_BUFFER_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EQUAL_STRIDE_SUPER_BUFFER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EQUAL_STRIDE_PACKED_STREAM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EQUAL_STRIDE_PACKED_STREAM_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EQUAL_STRIDE_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_L3XUDP_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_L3XUDP_SUPPORT_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_L3XUDP_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_FW_SUBVARIANT_NO_TX_CSUM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_FW_SUBVARIANT_NO_TX_CSUM_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_FW_SUBVARIANT_NO_TX_CSUM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VI_SPREADING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VI_SPREADING_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VI_SPREADING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_HLB_IDLE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_HLB_IDLE_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RXDP_HLB_IDLE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_INIT_RXQ_NO_CONT_EV_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_INIT_RXQ_NO_CONT_EV_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_INIT_RXQ_NO_CONT_EV_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_INIT_RXQ_WITH_BUFFER_SIZE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_INIT_RXQ_WITH_BUFFER_SIZE_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_INIT_RXQ_WITH_BUFFER_SIZE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_BUNDLE_UPDATE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_BUNDLE_UPDATE_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_BUNDLE_UPDATE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TSO_V3_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TSO_V3_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TSO_V3_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_DYNAMIC_SENSORS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_DYNAMIC_SENSORS_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_DYNAMIC_SENSORS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_WIDTH 1
/* Number of FATSOv2 contexts per datapath supported by this NIC (when
 * TX_TSO_V2 == 1). Not present on older firmware (check the length).
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TSO_V2_N_CONTEXTS_OFST 24
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TX_TSO_V2_N_CONTEXTS_LEN 2
/* One byte per PF containing the number of the external port assigned to this
 * PF, indexed by PF number. Special values indicate that a PF is either not
 * present or not assigned.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_PFS_TO_PORTS_ASSIGNMENT_OFST 26
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_PFS_TO_PORTS_ASSIGNMENT_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_PFS_TO_PORTS_ASSIGNMENT_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_ACCESS_NOT_PERMITTED 0xff
/* enum: PF does not exist. */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_PF_NOT_PRESENT 0xfe
/* enum: PF does exist but is not assigned to any external port. */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_PF_NOT_ASSIGNED 0xfd
/* enum: This value indicates that PF is assigned, but it cannot be expressed
 * in this field. It is intended for a possible future situation where a more
 * complex scheme of PFs to ports mapping is being used. The future driver
 * should look for a new field supporting the new scheme. The current/old
 * driver should treat this value as PF_NOT_ASSIGNED.
 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_INCOMPATIBLE_ASSIGNMENT 0xfc
/* One byte per PF containing the number of its VFs, indexed by PF number. A
 * special value indicates that a PF is not present.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_NUM_VFS_PER_PF_OFST 42
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_NUM_VFS_PER_PF_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_NUM_VFS_PER_PF_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
/*               MC_CMD_GET_CAPABILITIES_V10_OUT_ACCESS_NOT_PERMITTED 0xff */
/* enum: PF does not exist. */
/*               MC_CMD_GET_CAPABILITIES_V10_OUT_PF_NOT_PRESENT 0xfe */
/* Number of VIs available for external ports 0-3. For devices with more than
 * four ports, the remainder are in NUM_VIS_PER_PORT2 in
 * GET_CAPABILITIES_V12_OUT.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_NUM_VIS_PER_PORT_OFST 58
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_NUM_VIS_PER_PORT_LEN 2
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_NUM_VIS_PER_PORT_NUM 4
/* Size of RX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ RX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RX_DESC_CACHE_SIZE_OFST 66
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RX_DESC_CACHE_SIZE_LEN 1
/* Size of TX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ TX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TX_DESC_CACHE_SIZE_OFST 67
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TX_DESC_CACHE_SIZE_LEN 1
/* Total number of available PIO buffers */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_NUM_PIO_BUFFS_OFST 68
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_NUM_PIO_BUFFS_LEN 2
/* Size of a single PIO buffer */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_SIZE_PIO_BUFF_OFST 70
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_SIZE_PIO_BUFF_LEN 2
/* On chips later than Medford the amount of address space assigned to each VI
 * is configurable. This is a global setting that the driver must query to
 * discover the VI to address mapping. Cut-through PIO (CTPIO) is not available
 * with 8k VI windows.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_VI_WINDOW_MODE_OFST 72
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_VI_WINDOW_MODE_LEN 1
/* enum: Each VI occupies 8k as on Huntington and Medford. PIO is at offset 4k.
 * CTPIO is not mapped.
 */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_VI_WINDOW_MODE_8K 0x0
/* enum: Each VI occupies 16k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_VI_WINDOW_MODE_16K 0x1
/* enum: Each VI occupies 64k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V10_OUT_VI_WINDOW_MODE_64K 0x2
/* Number of vFIFOs per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_VFIFO_STUFFING_NUM_VFIFOS_OFST 73
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_VFIFO_STUFFING_NUM_VFIFOS_LEN 1
/* Number of buffers per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_OFST 74
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_LEN 2
/* Entry count in the MAC stats array, including the final GENERATION_END
 * entry. For MAC stats DMA, drivers should allocate a buffer large enough to
 * hold at least this many 64-bit stats values, if they wish to receive all
 * available stats. If the buffer is shorter than MAC_STATS_NUM_STATS * 8, the
 * stats array returned will be truncated.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_MAC_STATS_NUM_STATS_OFST 76
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_MAC_STATS_NUM_STATS_LEN 2
/* Maximum supported value for MC_CMD_FILTER_OP_V3/MATCH_MARK_VALUE. This field
 * will only be non-zero if MC_CMD_GET_CAPABILITIES/FILTER_ACTION_MARK is set.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_FILTER_ACTION_MARK_MAX_OFST 80
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_FILTER_ACTION_MARK_MAX_LEN 4
/* On devices where the INIT_RXQ_WITH_BUFFER_SIZE flag (in
 * GET_CAPABILITIES_OUT_V2) is set, drivers have to specify a buffer size when
 * they create an RX queue. Due to hardware limitations, only a small number of
 * different buffer sizes may be available concurrently. Nonzero entries in
 * this array are the sizes of buffers which the system guarantees will be
 * available for use. If the list is empty, there are no limitations on
 * concurrent buffer sizes.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_GUARANTEED_RX_BUFFER_SIZES_OFST 84
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_GUARANTEED_RX_BUFFER_SIZES_LEN 4
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_GUARANTEED_RX_BUFFER_SIZES_NUM 16
/* Third word of flags. Not present on older firmware (check the length). */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_FLAGS3_OFST 148
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_FLAGS3_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_WOL_ETHERWAKE_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_WOL_ETHERWAKE_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_WOL_ETHERWAKE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_EVEN_SPREADING_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_EVEN_SPREADING_LBN 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_EVEN_SPREADING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_SELECTABLE_TABLE_SIZE_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_SELECTABLE_TABLE_SIZE_LBN 2
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_SELECTABLE_TABLE_SIZE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MAE_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MAE_SUPPORTED_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MAE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VDPA_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VDPA_SUPPORTED_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_VDPA_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_VLAN_STRIPPING_PER_ENCAP_RULE_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_VLAN_STRIPPING_PER_ENCAP_RULE_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RX_VLAN_STRIPPING_PER_ENCAP_RULE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EXTENDED_WIDTH_EVQS_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EXTENDED_WIDTH_EVQS_SUPPORTED_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EXTENDED_WIDTH_EVQS_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_UNSOL_EV_CREDIT_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_UNSOL_EV_CREDIT_SUPPORTED_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_UNSOL_EV_CREDIT_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_ENCAPSULATED_MCDI_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_ENCAPSULATED_MCDI_SUPPORTED_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_ENCAPSULATED_MCDI_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EXTERNAL_MAE_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EXTERNAL_MAE_SUPPORTED_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_EXTERNAL_MAE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_NVRAM_UPDATE_ABORT_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_NVRAM_UPDATE_ABORT_SUPPORTED_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_NVRAM_UPDATE_ABORT_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MAE_ACTION_SET_ALLOC_V2_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MAE_ACTION_SET_ALLOC_V2_SUPPORTED_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MAE_ACTION_SET_ALLOC_V2_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_STEER_ON_OUTER_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_STEER_ON_OUTER_SUPPORTED_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_STEER_ON_OUTER_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MAE_ACTION_SET_ALLOC_V3_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MAE_ACTION_SET_ALLOC_V3_SUPPORTED_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_MAE_ACTION_SET_ALLOC_V3_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_DYNAMIC_MPORT_JOURNAL_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_DYNAMIC_MPORT_JOURNAL_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_DYNAMIC_MPORT_JOURNAL_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_CLIENT_CMD_VF_PROXY_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_CLIENT_CMD_VF_PROXY_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_CLIENT_CMD_VF_PROXY_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_LL_RX_EVENT_SUPPRESSION_SUPPORTED_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_LL_RX_EVENT_SUPPRESSION_SUPPORTED_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_LL_RX_EVENT_SUPPRESSION_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_CXL_CONFIG_ENABLE_OFST 148
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_CXL_CONFIG_ENABLE_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V10_OUT_CXL_CONFIG_ENABLE_WIDTH 1
/* These bits are reserved for communicating test-specific capabilities to
 * host-side test software. All production drivers should treat this field as
 * opaque.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TEST_RESERVED_OFST 152
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TEST_RESERVED_LEN 8
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TEST_RESERVED_LO_OFST 152
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TEST_RESERVED_LO_LEN 4
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TEST_RESERVED_LO_LBN 1216
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TEST_RESERVED_LO_WIDTH 32
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TEST_RESERVED_HI_OFST 156
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TEST_RESERVED_HI_LEN 4
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TEST_RESERVED_HI_LBN 1248
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_TEST_RESERVED_HI_WIDTH 32
/* The minimum size (in table entries) of indirection table to be allocated
 * from the pool for an RSS context. Note that the table size used must be a
 * power of 2.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_MIN_INDIRECTION_TABLE_SIZE_OFST 160
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_MIN_INDIRECTION_TABLE_SIZE_LEN 4
/* The maximum size (in table entries) of indirection table to be allocated
 * from the pool for an RSS context. Note that the table size used must be a
 * power of 2.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_MAX_INDIRECTION_TABLE_SIZE_OFST 164
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_MAX_INDIRECTION_TABLE_SIZE_LEN 4
/* The maximum number of queues that can be used by an RSS context in exclusive
 * mode. In exclusive mode the context has a configurable indirection table and
 * a configurable RSS key.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_MAX_INDIRECTION_QUEUES_OFST 168
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_MAX_INDIRECTION_QUEUES_LEN 4
/* The maximum number of queues that can be used by an RSS context in even-
 * spreading mode. In even-spreading mode the context has no indirection table
 * but it does have a configurable RSS key.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_MAX_EVEN_SPREADING_QUEUES_OFST 172
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_MAX_EVEN_SPREADING_QUEUES_LEN 4
/* The total number of RSS contexts supported. Note that the number of
 * available contexts using indirection tables is also limited by the
 * availability of indirection table space allocated from a common pool.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_NUM_CONTEXTS_OFST 176
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_NUM_CONTEXTS_LEN 4
/* The total amount of indirection table space that can be shared between RSS
 * contexts.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_TABLE_POOL_SIZE_OFST 180
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_RSS_TABLE_POOL_SIZE_LEN 4
/* A bitmap of the queue sizes the device can provide, where bit N being set
 * indicates that 2**N is a valid size. The device may be limited in the number
 * of different queue sizes that can exist simultaneously, so a bit being set
 * here does not guarantee that an attempt to create a queue of that size will
 * succeed.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_SUPPORTED_QUEUE_SIZES_OFST 184
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_SUPPORTED_QUEUE_SIZES_LEN 4
/* A bitmap of queue sizes that are always available, in the same format as
 * SUPPORTED_QUEUE_SIZES. Attempting to create a queue with one of these sizes
 * will never fail due to unavailability of the requested size.
 */
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_GUARANTEED_QUEUE_SIZES_OFST 188
#define       MC_CMD_GET_CAPABILITIES_V10_OUT_GUARANTEED_QUEUE_SIZES_LEN 4

/* MC_CMD_GET_CAPABILITIES_V11_OUT msgresponse */
#define    MC_CMD_GET_CAPABILITIES_V11_OUT_LEN 196
/* First word of flags. */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_FLAGS1_OFST 0
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_FLAGS1_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_VPORT_RECONFIGURE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_VPORT_RECONFIGURE_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_VPORT_RECONFIGURE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_STRIPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_STRIPING_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_STRIPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_VADAPTOR_QUERY_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_VADAPTOR_QUERY_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_VADAPTOR_QUERY_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EVB_PORT_VLAN_RESTRICT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EVB_PORT_VLAN_RESTRICT_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EVB_PORT_VLAN_RESTRICT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_DRV_ATTACH_PREBOOT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_DRV_ATTACH_PREBOOT_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_DRV_ATTACH_PREBOOT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_FORCE_EVENT_MERGING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_FORCE_EVENT_MERGING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_FORCE_EVENT_MERGING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_SET_MAC_ENHANCED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_SET_MAC_ENHANCED_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_SET_MAC_ENHANCED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_UNKNOWN_UCAST_DST_FILTER_ALWAYS_MULTI_RECIPIENT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_VADAPTOR_PERMIT_SET_MAC_WHEN_FILTERS_INSTALLED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_MAC_SECURITY_FILTERING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_MAC_SECURITY_FILTERING_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_MAC_SECURITY_FILTERING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_ADDITIONAL_RSS_MODES_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_ADDITIONAL_RSS_MODES_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_ADDITIONAL_RSS_MODES_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_QBB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_QBB_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_QBB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_PACKED_STREAM_VAR_BUFFERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_PACKED_STREAM_VAR_BUFFERS_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_PACKED_STREAM_VAR_BUFFERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_RSS_LIMITED_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_RSS_LIMITED_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_RSS_LIMITED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_PACKED_STREAM_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_PACKED_STREAM_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_INCLUDE_FCS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_INCLUDE_FCS_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_INCLUDE_FCS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_VLAN_INSERTION_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_VLAN_INSERTION_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_VLAN_INSERTION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_VLAN_STRIPPING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_VLAN_STRIPPING_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_VLAN_STRIPPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TSO_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TSO_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TSO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_PREFIX_LEN_0_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_PREFIX_LEN_0_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_PREFIX_LEN_0_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_PREFIX_LEN_14_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_PREFIX_LEN_14_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_PREFIX_LEN_14_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_TIMESTAMP_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_TIMESTAMP_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_BATCHING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_BATCHING_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_BATCHING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_MCAST_FILTER_CHAINING_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_MCAST_FILTER_CHAINING_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_MCAST_FILTER_CHAINING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_PM_AND_RXDP_COUNTERS_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_PM_AND_RXDP_COUNTERS_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_PM_AND_RXDP_COUNTERS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_DISABLE_SCATTER_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_DISABLE_SCATTER_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_DISABLE_SCATTER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_MCAST_UDP_LOOPBACK_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_MCAST_UDP_LOOPBACK_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_MCAST_UDP_LOOPBACK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EVB_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EVB_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EVB_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_VXLAN_NVGRE_OFST 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_VXLAN_NVGRE_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_VXLAN_NVGRE_WIDTH 1
/* RxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_RX_DPCPU_FW_ID_OFST 4
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_RX_DPCPU_FW_ID_LEN 2
/* enum: Standard RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP 0x0
/* enum: Low latency RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_LOW_LATENCY 0x1
/* enum: Packed stream RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_PACKED_STREAM 0x2
/* enum: Rules engine RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_RULES_ENGINE 0x5
/* enum: DPDK RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_DPDK 0x6
/* enum: BIST RXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_BIST 0x10a
/* enum: RXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_TEST_FW_TO_MC_CUT_THROUGH 0x101
/* enum: RXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD 0x102
/* enum: RXDP Test firmware image 3 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_TEST_FW_TO_MC_STORE_FORWARD_FIRST 0x103
/* enum: RXDP Test firmware image 4 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_TEST_EVERY_EVENT_BATCHABLE 0x104
/* enum: RXDP Test firmware image 5 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_TEST_BACKPRESSURE 0x105
/* enum: RXDP Test firmware image 6 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_TEST_FW_PACKET_EDITS 0x106
/* enum: RXDP Test firmware image 7 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_TEST_FW_RX_HDR_SPLIT 0x107
/* enum: RXDP Test firmware image 8 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_TEST_FW_DISABLE_DL 0x108
/* enum: RXDP Test firmware image 9 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_TEST_FW_DOORBELL_DELAY 0x10b
/* enum: RXDP Test firmware image 10 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_TEST_FW_SLOW 0x10c
/* TxDPCPU firmware id. */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_TX_DPCPU_FW_ID_OFST 6
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_TX_DPCPU_FW_ID_LEN 2
/* enum: Standard TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXDP 0x0
/* enum: Low latency TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXDP_LOW_LATENCY 0x1
/* enum: High packet rate TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXDP_HIGH_PACKET_RATE 0x3
/* enum: Rules engine TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXDP_RULES_ENGINE 0x5
/* enum: DPDK TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXDP_DPDK 0x6
/* enum: BIST TXDP firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXDP_BIST 0x12d
/* enum: TXDP Test firmware image 1 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXDP_TEST_FW_TSO_EDIT 0x101
/* enum: TXDP Test firmware image 2 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXDP_TEST_FW_PACKET_EDITS 0x102
/* enum: TXDP CSR bus test firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXDP_TEST_FW_CSR 0x103
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_VERSION_OFST 8
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_VERSION_REV_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_VERSION_TYPE_OFST 8
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial RX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: RX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: RX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant RX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
/* enum: Low latency RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_TYPE_LOW_LATENCY 0x5
/* enum: Packed stream RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_TYPE_PACKED_STREAM 0x6
/* enum: RX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK RX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* enum: RX PD firmware parsing but not filtering network overlay tunnel
 * encapsulations (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_RXPD_FW_TYPE_TESTFW_ENCAP_PARSING_ONLY 0xf
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_VERSION_OFST 10
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_VERSION_LEN 2
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_VERSION_REV_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_VERSION_REV_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_VERSION_REV_WIDTH 12
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_VERSION_TYPE_OFST 10
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_VERSION_TYPE_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_VERSION_TYPE_WIDTH 4
/* enum: reserved value - do not use (may indicate alternative interpretation
 * of REV field in future)
 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_TYPE_RESERVED 0x0
/* enum: Trivial TX PD firmware for early Huntington development (Huntington
 * development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_TYPE_FIRST_PKT 0x1
/* enum: TX PD firmware for telemetry prototyping (Medford2 development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_TYPE_TESTFW_TELEMETRY 0x1
/* enum: TX PD firmware with approximately Siena-compatible behaviour
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_TYPE_SIENA_COMPAT 0x2
/* enum: Full featured TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_TYPE_FULL_FEATURED 0x3
/* enum: (deprecated original name for the FULL_FEATURED variant) */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_TYPE_VSWITCH 0x3
/* enum: siena_compat variant TX PD firmware using PM rather than MAC
 * (Huntington development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_TYPE_SIENA_COMPAT_PM 0x4
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_TYPE_LOW_LATENCY 0x5 /* enum */
/* enum: TX PD firmware handling layer 2 only for high packet rate performance
 * tests (Medford development only)
 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_TYPE_LAYER2_PERF 0x7
/* enum: Rules engine TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_TYPE_RULES_ENGINE 0x8
/* enum: Custom firmware variant (see SF-119495-PD and bug69716) */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_TYPE_L3XUDP 0x9
/* enum: DPDK TX PD production firmware */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_TYPE_DPDK 0xa
/* enum: RX PD firmware for GUE parsing prototype (Medford development only) */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_TXPD_FW_TYPE_TESTFW_GUE_PROTOTYPE 0xe
/* Hardware capabilities of NIC */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_HW_CAPABILITIES_OFST 12
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_HW_CAPABILITIES_LEN 4
/* Licensed capabilities */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_LICENSE_CAPABILITIES_OFST 16
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_LICENSE_CAPABILITIES_LEN 4
/* Second word of flags. Not present on older firmware (check the length). */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_FLAGS2_OFST 20
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_FLAGS2_LEN 4
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TSO_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TSO_V2_LBN 0
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TSO_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TSO_V2_ENCAP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TSO_V2_ENCAP_LBN 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TSO_V2_ENCAP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EVQ_TIMER_CTRL_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EVQ_TIMER_CTRL_LBN 2
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EVQ_TIMER_CTRL_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EVENT_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EVENT_CUT_THROUGH_LBN 3
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EVENT_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_CUT_THROUGH_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_CUT_THROUGH_LBN 4
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_CUT_THROUGH_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_VFIFO_ULL_MODE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_VFIFO_ULL_MODE_LBN 5
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_VFIFO_ULL_MODE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_MAC_STATS_40G_TX_SIZE_BINS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_MAC_STATS_40G_TX_SIZE_BINS_LBN 6
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_MAC_STATS_40G_TX_SIZE_BINS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_INIT_EVQ_TYPE_SUPPORTED_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_INIT_EVQ_TYPE_SUPPORTED_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_INIT_EVQ_TYPE_SUPPORTED_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_INIT_EVQ_V2_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_INIT_EVQ_V2_LBN 7
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_INIT_EVQ_V2_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_MAC_TIMESTAMPING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_MAC_TIMESTAMPING_LBN 8
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_MAC_TIMESTAMPING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TIMESTAMP_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TIMESTAMP_LBN 9
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TIMESTAMP_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_SNIFF_LBN 10
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_SNIFF_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_SNIFF_LBN 11
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_SNIFF_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_LBN 12
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_NVRAM_UPDATE_REPORT_VERIFY_RESULT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_MCDI_BACKGROUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_MCDI_BACKGROUND_LBN 13
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_MCDI_BACKGROUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_MCDI_DB_RETURN_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_MCDI_DB_RETURN_LBN 14
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_MCDI_DB_RETURN_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_CTPIO_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_CTPIO_LBN 15
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_CTPIO_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TSA_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TSA_SUPPORT_LBN 16
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TSA_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TSA_BOUND_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TSA_BOUND_LBN 17
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TSA_BOUND_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_SF_ADAPTER_AUTHENTICATION_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_SF_ADAPTER_AUTHENTICATION_LBN 18
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_SF_ADAPTER_AUTHENTICATION_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_FILTER_ACTION_FLAG_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_FILTER_ACTION_FLAG_LBN 19
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_FILTER_ACTION_FLAG_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_FILTER_ACTION_MARK_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_FILTER_ACTION_MARK_LBN 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_FILTER_ACTION_MARK_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EQUAL_STRIDE_SUPER_BUFFER_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EQUAL_STRIDE_SUPER_BUFFER_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EQUAL_STRIDE_SUPER_BUFFER_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EQUAL_STRIDE_PACKED_STREAM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EQUAL_STRIDE_PACKED_STREAM_LBN 21
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_EQUAL_STRIDE_PACKED_STREAM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_L3XUDP_SUPPORT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_L3XUDP_SUPPORT_LBN 22
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_L3XUDP_SUPPORT_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_FW_SUBVARIANT_NO_TX_CSUM_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_FW_SUBVARIANT_NO_TX_CSUM_LBN 23
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_FW_SUBVARIANT_NO_TX_CSUM_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_VI_SPREADING_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_VI_SPREADING_LBN 24
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_VI_SPREADING_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_HLB_IDLE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_HLB_IDLE_LBN 25
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_RXDP_HLB_IDLE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_INIT_RXQ_NO_CONT_EV_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_INIT_RXQ_NO_CONT_EV_LBN 26
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_INIT_RXQ_NO_CONT_EV_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_INIT_RXQ_WITH_BUFFER_SIZE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_INIT_RXQ_WITH_BUFFER_SIZE_LBN 27
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_INIT_RXQ_WITH_BUFFER_SIZE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_BUNDLE_UPDATE_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_BUNDLE_UPDATE_LBN 28
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_BUNDLE_UPDATE_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TSO_V3_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TSO_V3_LBN 29
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TSO_V3_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_DYNAMIC_SENSORS_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_DYNAMIC_SENSORS_LBN 30
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_DYNAMIC_SENSORS_WIDTH 1
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_OFST 20
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_LBN 31
#define        MC_CMD_GET_CAPABILITIES_V11_OUT_NVRAM_UPDATE_POLL_VERIFY_RESULT_WIDTH 1
/* Number of FATSOv2 contexts per datapath supported by this NIC (when
 * TX_TSO_V2 == 1). Not present on older firmware (check the length).
 */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TSO_V2_N_CONTEXTS_OFST 24
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_TX_TSO_V2_N_CONTEXTS_LEN 2
/* One byte per PF containing the number of the external port assigned to this
 * PF, indexed by PF number. Special values indicate that a PF is either not
 * present or not assigned.
 */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_PFS_TO_PORTS_ASSIGNMENT_OFST 26
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_PFS_TO_PORTS_ASSIGNMENT_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_PFS_TO_PORTS_ASSIGNMENT_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_ACCESS_NOT_PERMITTED 0xff
/* enum: PF does not exist. */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_PF_NOT_PRESENT 0xfe
/* enum: PF does exist but is not assigned to any external port. */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_PF_NOT_ASSIGNED 0xfd
/* enum: This value indicates that PF is assigned, but it cannot be expressed
 * in this field. It is intended for a possible future situation where a more
 * complex scheme of PFs to ports mapping is being used. The future driver
 * should look for a new field supporting the new scheme. The current/old
 * driver should treat this value as PF_NOT_ASSIGNED.
 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_INCOMPATIBLE_ASSIGNMENT 0xfc
/* One byte per PF containing the number of its VFs, indexed by PF number. A
 * special value indicates that a PF is not present.
 */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_NUM_VFS_PER_PF_OFST 42
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_NUM_VFS_PER_PF_LEN 1
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_NUM_VFS_PER_PF_NUM 16
/* enum: The caller is not permitted to access information on this PF. */
/*               MC_CMD_GET_CAPABILITIES_V11_OUT_ACCESS_NOT_PERMITTED 0xff */
/* enum: PF does not exist. */
/*               MC_CMD_GET_CAPABILITIES_V11_OUT_PF_NOT_PRESENT 0xfe */
/* Number of VIs available for external ports 0-3. For devices with more than
 * four ports, the remainder are in NUM_VIS_PER_PORT2 in
 * GET_CAPABILITIES_V12_OUT.
 */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_NUM_VIS_PER_PORT_OFST 58
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_NUM_VIS_PER_PORT_LEN 2
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_NUM_VIS_PER_PORT_NUM 4
/* Size of RX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ RX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_RX_DESC_CACHE_SIZE_OFST 66
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_RX_DESC_CACHE_SIZE_LEN 1
/* Size of TX descriptor cache expressed as binary logarithm The actual size
 * equals (2 ^ TX_DESC_CACHE_SIZE)
 */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_TX_DESC_CACHE_SIZE_OFST 67
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_TX_DESC_CACHE_SIZE_LEN 1
/* Total number of available PIO buffers */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_NUM_PIO_BUFFS_OFST 68
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_NUM_PIO_BUFFS_LEN 2
/* Size of a single PIO buffer */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_SIZE_PIO_BUFF_OFST 70
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_SIZE_PIO_BUFF_LEN 2
/* On chips later than Medford the amount of address space assigned to each VI
 * is configurable. This is a global setting that the driver must query to
 * discover the VI to address mapping. Cut-through PIO (CTPIO) is not available
 * with 8k VI windows.
 */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_VI_WINDOW_MODE_OFST 72
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_VI_WINDOW_MODE_LEN 1
/* enum: Each VI occupies 8k as on Huntington and Medford. PIO is at offset 4k.
 * CTPIO is not mapped.
 */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_VI_WINDOW_MODE_8K 0x0
/* enum: Each VI occupies 16k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_VI_WINDOW_MODE_16K 0x1
/* enum: Each VI occupies 64k. PIO is at offset 4k. CTPIO is at offset 12k. */
#define          MC_CMD_GET_CAPABILITIES_V11_OUT_VI_WINDOW_MODE_64K 0x2
/* Number of vFIFOs per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_VFIFO_STUFFING_NUM_VFIFOS_OFST 73
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_VFIFO_STUFFING_NUM_VFIFOS_LEN 1
/* Number of buffers per adapter that can be used for VFIFO Stuffing
 * (SF-115995-SW) in the present configuration of firmware and port mode.
 */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_OFST 74
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_VFIFO_STUFFING_NUM_CP_BUFFERS_LEN 2
/* Entry count in the MAC stats array, including the final GENERATION_END
 * entry. For MAC stats DMA, drivers should allocate a buffer large enough to
 * hold at least this many 64-bit stats values, if they wish to receive all
 * available stats. If the buffer is shorter than MAC_STATS_NUM_STATS * 8, the
 * stats array returned will be truncated.
 */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_MAC_STATS_NUM_STATS_OFST 76
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_MAC_STATS_NUM_STATS_LEN 2
/* Maximum supported value for MC_CMD_FILTER_OP_V3/MATCH_MARK_VALUE. This field
 * will only be non-zero if MC_CMD_GET_CAPABILITIES/FILTER_ACTION_MARK is set.
 */
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_FILTER_ACTION_MARK_MAX_OFST 80
#define       MC_CMD_GET_CAPABILITIES_V11_OUT_FILTER_ACTION_MARK_MAX_LEN 4
/* On devices where the INIT_RXQ_WITH_BUFFER_SIZE flag (in
 * GET_CAPABILITIES_OUT_V2) is set, drivers have to specify a buffer size when
 * they create an RX queue. Due to hardware limitations, only a small number 