package picard.analysis;

import htsjdk.samtools.util.SequenceUtil;
import org.testng.Assert;
import org.testng.annotations.DataProvider;
import org.testng.annotations.Test;

public class AdapterUtilityTest {
    private static final AdapterUtility adapterUtility = new AdapterUtility(AdapterUtility.DEFAULT_ADAPTER_SEQUENCE);

    @DataProvider
    Object[][] testAdapterReadsData() {
        return new Object[][]{
                {"AGATCGGAAGAGCGTCGTGTAGGGAAAGAGTGTAAGAGGTTGTGGGTATGGGGGGGGGGGGGGGGGGGGTGGGGAAGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG", true},
                {"GATCGGAAGAGCACACGTCTGAACTCCAGTAACGCACATCTATCTCGTATTCCGTCTTCTTCTTGAACAGGGGTGGGTGGGGGGGGGGGGTGGGGGGGGGGGGGG", true},
                {"CTGAACTCCAGTCACGCACATCTATCGCGTATGCCGTCTTCTGCTTGAAAAGGGGGGGGGGGTGGGGGGGGGGG", true},
                {"GATCGGAAGAGCGTCGTGTAGGGAAAGAGTGTAAGAGGTTGTGGGGGTGAAGGGGGGGGGGGGGGG", true},
                {"AGATCGGAAGAGCTCGTATGCCGTCTTCTGCTTGGGGGGGGGGGGTGGGTCCA", true},
                {"AGATCGGAAGAGCTCGTATGCCGTCTTCTGCTTGGGGGGGGGGGGTGGGTCCA", true},
                {"CTCGTATGCCGTCTTCTGCTTGGGGGGGGGGGGTGGGTCCA", true},
                {"GAGCTCGTATGCCGTCTTCTGCTTGGGGGGGGGGGGTGGGTCCA", true},
                {"AGATCGGAAGAGCGTCGTGTAGGGaaaGAGTGTAAGAGGTTGTGGGTATGGGGGGGGGGGGGGGGGGGGTGGGGAAGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG", true},
                {"GATCggaAGAGCACACGTCTGAACTCCAGTAACGCACATCTATCTCGTATTCCGTCTTCTTCTTGAACAGGGGTGGGTGGGGGGGGGGGGTGGGGGGGGGGGGGG", true},
                {"agatcggaagagctcgtatgccgtcttctgcttggggggggggggtgggtcca", true},
                {AdapterUtility.DEFAULT_ADAPTER_SEQUENCE.get(0), true},
                {AdapterUtility.DEFAULT_ADAPTER_SEQUENCE.get(1), true},
                {AdapterUtility.DEFAULT_ADAPTER_SEQUENCE.get(2), true},
                {AdapterUtility.DEFAULT_ADAPTER_SEQUENCE.get(3), true},
                {AdapterUtility.DEFAULT_ADAPTER_SEQUENCE.get(4), true},
                {AdapterUtility.DEFAULT_ADAPTER_SEQUENCE.get(5), true},
                {"GAGGCCAAATATCATTCTGAGGGGCCACAGTAATTACAAACTTACTATCC", false},
                {"GCCATCCCATACATTGGGACAGACCTAGTTCAATGAATCTGAGGAGGCTA", false},
                {"CTCAGTAGACAGTCCCACCCTCACACGATTCTTTACCTTTCACTTCATCT", false},
                {"TACCCTTCATTATTGCAGCCCTAGCAGCACTCCACCTCCTATTCTTGCAC", false},
                {"GAAACGGGATCAAACAACCCCCTAGGAATCACCTCCCATTCCGATAAAAT", false},
                {"CACCTTCCACCCTTACTACACAATCAAAGACGCCCTCGGCTTACTTCTCT", false},
                {"TCCTTCTCTCCTTAATGACATTAACACTATTCTCACCAGACCTCCTAGGC", false},
                {"GACCCAGACAATTATACCCTAGCCAACCCCTTAAACACCCCTCCCCACAT", false},
                {"CAAGCCCGAATGATATTTCCTATTCGCCTACACAATTCTCCGATCCGTCC", false},
                {"CTAACAAACTAGGAGGCGTCCTTGCCCTATTACTATCCATCCTCATCCTA", false},
                {"GCAATAATCCCCATCCTCCATATATCCAAACAACAAAGCATAATATTTCG", false},
                {"CCCACTAAGCCAATCACTTTATTGACTCCTAGCCGCAGACCTCCTCATTC", false},
                {"TAACCTGAATCGGAGGACAACCAGTAAGCTACCCTTTTACCATCATTGGA", false},
                {"CAAGTAGCATCCGTACTATACTTCACAACAATCCTAATCCTAATACCAAC", false},
                {"TATCTCCCTAATTGAAAACAAAATACTCAAATGGGCCTGTCCTTGTAGTA", false},
                {"TAAACTAATACACCAGTCTTGTAAACCGGAGACGAAAACCTTTTTCCAAG", false},
                {"GACAAATCAGAGAAAAAGTCTTTAACTCCACCATTAGCACCCAAAGCTAA", false},
                {"GATTCTAATTTAAACTATTCTCTGTTCTTTCATGGGGAAGCAGATTTGGG", false},
                {"TACCACCCAAGTATTGACTCACCCATCAACAACCGCTATGTATTTCGTAC", false},
                {"ATTACTGCCAGCCACCATGAATATTGTACGGTACCATAAATACTTGACCA", false},
                {"CCTGTAGTACATAAAAACCCAACCCACATCAAACCCCCCCCCCCCATGCT", false},
                {"TACAAGCAAGTACAGCAATCAACCTTCAACTATCACACATCAACTGCAAC", false},
                {"TCCAAAGCCACCCCTCACCCACTAGGATACCAACAAACCTACCCACCCTT", false},
                {"AACAGTACATAGTACATAAAGTCATTTACCGTACATAGCACATTACAGTC", false},

        };
    }

    @Test(dataProvider = "testAdapterReadsData")
    public void testAdapterReads(final String readBases, final boolean isAdapter) {
        Assert.assertEquals(adapterUtility.isAdapterSequence(readBases.getBytes()), isAdapter);
    }

    @Test(dataProvider = "testAdapterReadsData")
    public void testAdapterReadsRevComp(final String readBases, final boolean isAdapter) {
        Assert.assertEquals(adapterUtility.isAdapterSequence(SequenceUtil.reverseComplement(readBases).getBytes(), true), isAdapter);
    }
}