import re
from typing import (
    Union,
    Optional,
    List,
)
from collections.abc import Sequence

from debputy.linting.lint_util import LintState, with_range_in_continuous_parts
from debputy.lsp.debputy_ls import DebputyLanguageServer
from debputy.lsp.lsp_debian_control_reference_data import (
    Dep5FileMetadata,
)
from debputy.lsp.lsp_features import (
    lint_diagnostics,
    lsp_completer,
    lsp_hover,
    lsp_standard_handler,
    lsp_folding_ranges,
    lsp_semantic_tokens_full,
    lsp_will_save_wait_until,
    lsp_format_document,
    SecondaryLanguage,
    LanguageDispatchRule,
    lsp_cli_reformat_document,
)
from debputy.lsp.lsp_generic_deb822 import (
    deb822_completer,
    deb822_hover,
    deb822_folding_ranges,
    deb822_semantic_tokens_full,
    deb822_format_file,
    scan_for_syntax_errors_and_token_level_diagnostics,
)
from debian._deb822_repro import (
    Deb822ParagraphElement,
)
from debputy.lsprotocol.types import (
    CompletionItem,
    CompletionList,
    CompletionParams,
    HoverParams,
    Hover,
    TEXT_DOCUMENT_CODE_ACTION,
    SemanticTokens,
    SemanticTokensParams,
    FoldingRangeParams,
    FoldingRange,
    WillSaveTextDocumentParams,
    TextEdit,
    DocumentFormattingParams,
)

try:
    from debian._deb822_repro.locatable import (
        Position as TEPosition,
        Range as TERange,
    )

    from pygls.server import LanguageServer
    from pygls.workspace import TextDocument
except ImportError:
    pass


_CONTAINS_SPACE_OR_COLON = re.compile(r"[\s:]")

_DISPATCH_RULE = LanguageDispatchRule.new_rule(
    "debian/copyright",
    None,
    "debian/copyright",
    [
        # emacs's name
        SecondaryLanguage("debian-copyright"),
        # vim's name
        SecondaryLanguage("debcopyright"),
    ],
)

_DEP5_FILE_METADATA = Dep5FileMetadata()

lsp_standard_handler(_DISPATCH_RULE, TEXT_DOCUMENT_CODE_ACTION)


@lsp_hover(_DISPATCH_RULE)
def _debian_copyright_hover(
    ls: "DebputyLanguageServer",
    params: HoverParams,
) -> Hover | None:
    return deb822_hover(ls, params, _DEP5_FILE_METADATA)


@lsp_completer(_DISPATCH_RULE)
def _debian_copyright_completions(
    ls: "DebputyLanguageServer",
    params: CompletionParams,
) -> CompletionList | Sequence[CompletionItem] | None:
    return deb822_completer(ls, params, _DEP5_FILE_METADATA)


@lsp_folding_ranges(_DISPATCH_RULE)
def _debian_copyright_folding_ranges(
    ls: "DebputyLanguageServer",
    params: FoldingRangeParams,
) -> Sequence[FoldingRange] | None:
    return deb822_folding_ranges(ls, params, _DEP5_FILE_METADATA)


@lint_diagnostics(_DISPATCH_RULE)
async def _lint_debian_copyright(lint_state: LintState) -> None:
    deb822_file = lint_state.parsed_deb822_file_content

    if not _DEP5_FILE_METADATA.file_metadata_applies_to_file(deb822_file):
        return

    first_error = await scan_for_syntax_errors_and_token_level_diagnostics(
        deb822_file,
        lint_state,
    )
    header_stanza, files_stanza, _ = _DEP5_FILE_METADATA.stanza_types()
    stanza_no = 0

    async for stanza_range, stanza in lint_state.slow_iter(
        with_range_in_continuous_parts(deb822_file.iter_parts())
    ):
        if not isinstance(stanza, Deb822ParagraphElement):
            continue
        stanza_position = stanza_range.start_pos
        if stanza_position.line_position >= first_error:
            break
        stanza_no += 1
        is_files_or_license_stanza = stanza_no != 1
        if is_files_or_license_stanza:
            stanza_metadata = _DEP5_FILE_METADATA.classify_stanza(
                stanza,
                stanza_no,
            )
            other_stanza_metadata = header_stanza
            other_stanza_name = "Header"
        elif "Format" in stanza:
            stanza_metadata = header_stanza
            other_stanza_metadata = files_stanza
            other_stanza_name = "Files/License"
        else:
            break

        await stanza_metadata.stanza_diagnostics(
            deb822_file,
            stanza,
            stanza_position,
            lint_state,
            confusable_with_stanza_name=other_stanza_name,
            confusable_with_stanza_metadata=other_stanza_metadata,
        )


@lsp_will_save_wait_until(_DISPATCH_RULE)
def _debian_copyright_on_save_formatting(
    ls: "DebputyLanguageServer",
    params: WillSaveTextDocumentParams,
) -> Sequence[TextEdit] | None:
    doc = ls.workspace.get_text_document(params.text_document.uri)
    lint_state = ls.lint_state(doc)
    return deb822_format_file(lint_state, _DEP5_FILE_METADATA)


@lsp_cli_reformat_document(_DISPATCH_RULE)
def _reformat_debian_copyright(
    lint_state: LintState,
) -> Sequence[TextEdit] | None:
    return deb822_format_file(lint_state, _DEP5_FILE_METADATA)


@lsp_format_document(_DISPATCH_RULE)
def _debian_copyright_format_document(
    ls: "DebputyLanguageServer",
    params: DocumentFormattingParams,
) -> Sequence[TextEdit] | None:
    doc = ls.workspace.get_text_document(params.text_document.uri)
    lint_state = ls.lint_state(doc)
    return deb822_format_file(lint_state, _DEP5_FILE_METADATA)


@lsp_semantic_tokens_full(_DISPATCH_RULE)
async def _debian_copyright_semantic_tokens_full(
    ls: "DebputyLanguageServer",
    request: SemanticTokensParams,
) -> SemanticTokens | None:
    return await deb822_semantic_tokens_full(
        ls,
        request,
        _DEP5_FILE_METADATA,
    )
