/*
 * Copyright 2021-2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.events.java;

import org.apiguardian.api.API;
import org.opentest4j.reporting.events.api.Appendable;
import org.opentest4j.reporting.events.api.ChildElement;
import org.opentest4j.reporting.events.api.Context;
import org.opentest4j.reporting.events.core.CoreFactory;
import org.opentest4j.reporting.events.core.Sources;
import org.opentest4j.reporting.schema.Namespace;
import org.opentest4j.reporting.schema.QualifiedName;

import java.util.Optional;

import static org.apiguardian.api.API.Status.EXPERIMENTAL;

/**
 * The {@code classSource} element of the Java namespace.
 *
 * @since 0.1.0
 */
@API(status = EXPERIMENTAL, since = "0.1.0")
public class ClassSource extends ChildElement<Sources, ClassSource> {

	private static final QualifiedName ELEMENT = QualifiedName.of(Namespace.REPORTING_JAVA, "classSource");
	private static final QualifiedName CLASS_NAME = QualifiedName.of(Namespace.REPORTING_JAVA, "className");

	ClassSource(Context context) {
		super(context, ClassSource.ELEMENT);
	}

	ClassSource withClassName(String className) {
		withAttribute(CLASS_NAME, className);
		return this;
	}

	/**
	 * Add a {@code filePosition} child element to this element.
	 *
	 * @param line   the line number
	 * @param column the column number
	 * @return this element
	 */
	public Appendable<ClassSource> addFilePosition(int line, Optional<Integer> column) {
		return append(CoreFactory.filePosition(line, column));
	}
}
