/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2021      Kunwar Maheep Singh <kunwar.maheep@students.iiit.ac.in>
 */

#define SIMDE_TEST_X86_AVX512_INSN rolv

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/rolv.h>

static int
test_simde_mm_rolv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t a[4];
    const int32_t b[4];
    const int32_t r[4];
  } test_vec[] = {
    { {  INT32_C(  1813796411),  INT32_C(  1991038441), -INT32_C(   617792082), -INT32_C(  1763795364) },
      {  INT32_C(   281980766),  INT32_C(  2137854004), -INT32_C(   402309189), -INT32_C(  1488232340) },
      { -INT32_C(   620292722),  INT32_C(  1586981581),  INT32_C(  1993959917), -INT32_C(   370816659) } },
    { { -INT32_C(  1609341002), -INT32_C(   350830531),  INT32_C(  1522943229),  INT32_C(  1089512674) },
      { -INT32_C(  2024685997), -INT32_C(  1979203890),  INT32_C(  1769082109),  INT32_C(   588299628) },
      {  INT32_C(  1035272347), -INT32_C(  1341162811), -INT32_C(  1420244833),  INT32_C(   172893199) } },
    { {  INT32_C(  1640178724), -INT32_C(   515057180),  INT32_C(     3871261),  INT32_C(   171977911) },
      { -INT32_C(  1181576726),  INT32_C(  1296275792),  INT32_C(   314029477), -INT32_C(  1774860430) },
      {  INT32_C(   210801031), -INT32_C(   639311540),  INT32_C(   123880352), -INT32_C(  1294194432) } },
    { { -INT32_C(   805766933), -INT32_C(   273595183),  INT32_C(   250604631),  INT32_C(    51916568) },
      {  INT32_C(   297577153), -INT32_C(   379650237),  INT32_C(   653989300), -INT32_C(   927190819) },
      { -INT32_C(  1611533865),  INT32_C(  2106205839), -INT32_C(   982454530),  INT32_C(     6489571) } },
    { { -INT32_C(   107432920),  INT32_C(  1357400569),  INT32_C(  1314838325), -INT32_C(   934185465) },
      {  INT32_C(  1691946273), -INT32_C(  1068681460),  INT32_C(   719734861), -INT32_C(  1594711432) },
      { -INT32_C(   214865839), -INT32_C(  2069916402), -INT32_C(   622417461),  INT32_C(   130568566) } },
    { {  INT32_C(  1352305239),  INT32_C(   161514195),  INT32_C(  1633156698), -INT32_C(  1792432012) },
      { -INT32_C(  1040579915), -INT32_C(  2038347975),  INT32_C(   145778576),  INT32_C(  1621730057) },
      {  INT32_C(  1256854353), -INT32_C(  1508687611), -INT32_C(    27631273),  INT32_C(  1397811498) } },
    { {  INT32_C(    28328749),  INT32_C(   520769733), -INT32_C(  1014996658), -INT32_C(  1101485815) },
      { -INT32_C(   444640597),  INT32_C(   711655578),  INT32_C(  1882332263), -INT32_C(   321791041) },
      { -INT32_C(  2112264179),  INT32_C(   343681347), -INT32_C(  1070553247), -INT32_C(   550742908) } },
    { { -INT32_C(   454196962),  INT32_C(   537130962),  INT32_C(  1642300504), -INT32_C(   652264403) },
      {  INT32_C(   683581070),  INT32_C(    89270686),  INT32_C(    74876229),  INT32_C(  2146518880) },
      {  INT32_C(  1615313211), -INT32_C(  2013200908),  INT32_C(  1014008588), -INT32_C(   652264403) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i a = simde_x_mm_loadu_epi32(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi32(test_vec[i].b);
    simde__m128i r = simde_mm_rolv_epi32(a, b);
    simde_test_x86_assert_equal_i32x4(r, simde_x_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m128i a = simde_test_x86_random_i32x4();
    simde__m128i b = simde_test_x86_random_i32x4();
    simde__m128i r = simde_mm_rolv_epi32(a, b);

    simde_test_x86_write_i32x4(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm_mask_rolv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t src[4];
    const simde__mmask8 k;
    const int32_t a[4];
    const int32_t b[4];
    const int32_t r[4];
  } test_vec[] = {
    { {  INT32_C(   317015480), -INT32_C(  1681945950),  INT32_C(  1916578424),  INT32_C(  1530204903) },
      UINT8_C( 69),
      { -INT32_C(   191638530), -INT32_C(  1500247421), -INT32_C(   601970242),  INT32_C(  1670654749) },
      { -INT32_C(  1710905840),  INT32_C(   269656165), -INT32_C(   268991011), -INT32_C(  1187753286) },
      { -INT32_C(   738265965), -INT32_C(  1681945950), -INT32_C(   612117193),  INT32_C(  1530204903) } },
    { { -INT32_C(  1448228826), -INT32_C(  1873853998),  INT32_C(   141323754), -INT32_C(  1469382504) },
      UINT8_C(167),
      {  INT32_C(   286016368), -INT32_C(   621929387),  INT32_C(  1721097491), -INT32_C(   645116399) },
      {  INT32_C(  1034696187), -INT32_C(   249086844), -INT32_C(  1450627159),  INT32_C(   189804954) },
      { -INT32_C(  2138545637), -INT32_C(  1360935587),  INT32_C(   733619917), -INT32_C(  1469382504) } },
    { { -INT32_C(   904110732), -INT32_C(  1918629255), -INT32_C(   118277657), -INT32_C(  2083356793) },
      UINT8_C(180),
      { -INT32_C(  1170685826),  INT32_C(   392374759), -INT32_C(   474935886), -INT32_C(  1168589731) },
      { -INT32_C(   499899688),  INT32_C(    13222342),  INT32_C(   864535220),  INT32_C(   317131412) },
      { -INT32_C(   904110732), -INT32_C(  1918629255), -INT32_C(   617727216), -INT32_C(  2083356793) } },
    { { -INT32_C(  1295245366), -INT32_C(    87478200), -INT32_C(  1696761283),  INT32_C(   257242423) },
      UINT8_C( 87),
      {  INT32_C(  1243542153),  INT32_C(  2113806011), -INT32_C(  1341050459),  INT32_C(   930751256) },
      {  INT32_C(   545205488),  INT32_C(  1885174261), -INT32_C(  1935149226), -INT32_C(   706496948) },
      { -INT32_C(   225883618), -INT32_C(   680542269),  INT32_C(  1768686668),  INT32_C(   257242423) } },
    { {  INT32_C(  1679753640), -INT32_C(   975102689),  INT32_C(  1719006030),  INT32_C(   111013654) },
      UINT8_C( 28),
      { -INT32_C(  1794038244),  INT32_C(  2079097219), -INT32_C(   540575704),  INT32_C(  1569168731) },
      { -INT32_C(   629347396), -INT32_C(  1071103539), -INT32_C(  1495887946),  INT32_C(  1237507372) },
      {  INT32_C(  1679753640), -INT32_C(   975102689),  INT32_C(   171438558),  INT32_C(  2044048856) } },
    { { -INT32_C(  2015440125),  INT32_C(  2097334869), -INT32_C(  1638086334),  INT32_C(   603710566) },
      UINT8_C(207),
      { -INT32_C(  1180893833), -INT32_C(  1267770331),  INT32_C(   283186483), -INT32_C(  1424807209) },
      { -INT32_C(   754935288),  INT32_C(  1712684444), -INT32_C(  1093880870),  INT32_C(   646836142) },
      { -INT32_C(  1661110343),  INT32_C(  1531377090), -INT32_C(   867990444), -INT32_C(   894047548) } },
    { {  INT32_C(   316615404), -INT32_C(  1178186106),  INT32_C(  1003071332), -INT32_C(   622404143) },
      UINT8_C(119),
      {  INT32_C(  1679076838),  INT32_C(  1983806146),  INT32_C(   908393542), -INT32_C(  1289598327) },
      {  INT32_C(  2017080361), -INT32_C(  1562577925), -INT32_C(  1703667779), -INT32_C(   468627714) },
      {  INT32_C(   693882056),  INT32_C(   330429398), -INT32_C(   960192632), -INT32_C(   622404143) } },
    { { -INT32_C(  1119345157), -INT32_C(   432830817),  INT32_C(   203184258), -INT32_C(   859881822) },
      UINT8_C(115),
      { -INT32_C(   311540487),  INT32_C(   933892128), -INT32_C(   768260988),  INT32_C(  2077039189) },
      { -INT32_C(   384136350),  INT32_C(   376111294),  INT32_C(  1538881308),  INT32_C(   818840887) },
      { -INT32_C(  1246161945),  INT32_C(   233473032),  INT32_C(   203184258), -INT32_C(   859881822) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i src = simde_x_mm_loadu_epi32(test_vec[i].src);
    simde__m128i a = simde_x_mm_loadu_epi32(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi32(test_vec[i].b);
    simde__m128i r = simde_mm_mask_rolv_epi32(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x4(r, simde_x_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m128i src = simde_test_x86_random_i32x4();
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m128i a = simde_test_x86_random_i32x4();
    simde__m128i b = simde_test_x86_random_i32x4();
    simde__m128i r = simde_mm_mask_rolv_epi32(src, k, a, b);

    simde_test_x86_write_i32x4(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm_maskz_rolv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask8 k;
    const int32_t a[4];
    const int32_t b[4];
    const int32_t r[4];
  } test_vec[] = {
    { UINT8_C(201),
      {  INT32_C(  1290345788),  INT32_C(   198254791),  INT32_C(  1885446998), -INT32_C(    53289871) },
      { -INT32_C(   138757130), -INT32_C(  1642868441),  INT32_C(   265645706),  INT32_C(  2027423292) },
      {  INT32_C(  1326660167),  INT32_C(           0),  INT32_C(           0),  INT32_C(   533540295) } },
    { UINT8_C( 35),
      { -INT32_C(   504642111),  INT32_C(   959968918),  INT32_C(   866821975),  INT32_C(   891987578) },
      {  INT32_C(   844898657), -INT32_C(  1547896012), -INT32_C(   689976368),  INT32_C(  1694128291) },
      { -INT32_C(  1009284221),  INT32_C(  1768133503),  INT32_C(           0),  INT32_C(           0) } },
    { UINT8_C( 29),
      { -INT32_C(   609008155),  INT32_C(   607382653),  INT32_C(  1033791127), -INT32_C(  1314990960) },
      {  INT32_C(   736546864),  INT32_C(  1476168076), -INT32_C(  1057303960), -INT32_C(  1310892340) },
      {  INT32_C(  1172691891),  INT32_C(           0), -INT32_C(  1637443779), -INT32_C(   313980135) } },
    { UINT8_C(164),
      {  INT32_C(  2099350928),  INT32_C(   638862784), -INT32_C(  1195945500), -INT32_C(  1058510608) },
      { -INT32_C(   682814386), -INT32_C(   499080176), -INT32_C(    38862689),  INT32_C(  1856069854) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(  1549510898),  INT32_C(           0) } },
    { UINT8_C(237),
      {  INT32_C(   145615810),  INT32_C(  1374475519), -INT32_C(   213801845), -INT32_C(  1606286964) },
      { -INT32_C(   206562994),  INT32_C(  1519555417), -INT32_C(  1573351614), -INT32_C(   208624079) },
      {  INT32_C(  2062582315),  INT32_C(           0), -INT32_C(   855207377),  INT32_C(    51986564) } },
    { UINT8_C(146),
      {  INT32_C(   294779709), -INT32_C(  1952652569),  INT32_C(   622366755), -INT32_C(   344737582) },
      { -INT32_C(    62560663), -INT32_C(  2009161736),  INT32_C(  2109333719), -INT32_C(  1374704528) },
      {  INT32_C(           0), -INT32_C(   410280734),  INT32_C(           0),  INT32_C(           0) } },
    { UINT8_C(168),
      { -INT32_C(  2087731295), -INT32_C(   324592804), -INT32_C(   339817421), -INT32_C(  1504401089) },
      { -INT32_C(  1919004434),  INT32_C(  1868834446), -INT32_C(  1931484449), -INT32_C(  1842049551) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1384074409) } },
    { UINT8_C( 77),
      { -INT32_C(   559343933), -INT32_C(  1995336259), -INT32_C(    53936813), -INT32_C(  1561629103) },
      {  INT32_C(   858814477), -INT32_C(  1089232931),  INT32_C(   212901759), -INT32_C(  1772534829) },
      {  INT32_C(   584612821),  INT32_C(           0), -INT32_C(    26968407),  INT32_C(  1921849179) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i a = simde_x_mm_loadu_epi32(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi32(test_vec[i].b);
    simde__m128i r = simde_mm_maskz_rolv_epi32(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x4(r, simde_x_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m128i a = simde_test_x86_random_i32x4();
    simde__m128i b = simde_test_x86_random_i32x4();
    simde__m128i r = simde_mm_maskz_rolv_epi32(k, a, b);

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x4(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_rolv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t a[8];
    const int32_t b[8];
    const int32_t r[8];
  } test_vec[] = {
    { {  INT32_C(  1530220639), -INT32_C(  1295417221), -INT32_C(   123704714),  INT32_C(   321380340), -INT32_C(   314430790), -INT32_C(    30044354), -INT32_C(   548461072),  INT32_C(   843814099) },
      {  INT32_C(  1603109348),  INT32_C(  1997624833), -INT32_C(  1250905407),  INT32_C(  1271437201), -INT32_C(    13104447), -INT32_C(  1963037286),  INT32_C(  1701399954), -INT32_C(   979847711) },
      { -INT32_C(  1286273547),  INT32_C(  1704132855), -INT32_C(   247409427), -INT32_C(  1075304881), -INT32_C(   628861579), -INT32_C(    67578308), -INT32_C(  1748796100),  INT32_C(  1687628198) } },
    { {  INT32_C(   925181238),  INT32_C(  1034827388),  INT32_C(  2045910760),  INT32_C(  1992670133),  INT32_C(  1601633733), -INT32_C(    34966421), -INT32_C(  1553771583),  INT32_C(  1047067400) },
      { -INT32_C(  1670017760), -INT32_C(  1395055420), -INT32_C(   131675070),  INT32_C(  1282337671),  INT32_C(  1403839720),  INT32_C(   441554520), -INT32_C(   222448407), -INT32_C(   818928209) },
      {  INT32_C(   925181238), -INT32_C(   622630973), -INT32_C(   406291551),  INT32_C(  1658706619),  INT32_C(  1996342623),  INT32_C(  1811802740), -INT32_C(   962100410),  INT32_C(  2105810740) } },
    { {  INT32_C(  2003609267),  INT32_C(   203638218), -INT32_C(  1744549615),  INT32_C(   501576500), -INT32_C(  1334800041),  INT32_C(   281723175),  INT32_C(   604145525),  INT32_C(  1609839532) },
      { -INT32_C(  1546231591), -INT32_C(  1213203803),  INT32_C(  2018489411),  INT32_C(  2123707431), -INT32_C(   332528187),  INT32_C(  1023277255),  INT32_C(   744619903),  INT32_C(   193680690) },
      {  INT32_C(  1726929229), -INT32_C(  2073511615), -INT32_C(  1071495028), -INT32_C(   222717426),  INT32_C(   236071670),  INT32_C(  1700828040), -INT32_C(  1845410886), -INT32_C(   827228208) } },
    { {  INT32_C(  1521377973), -INT32_C(  1626251684),  INT32_C(   957833490),  INT32_C(  1538763925),  INT32_C(  2034755250),  INT32_C(  1588937950),  INT32_C(  1988761156),  INT32_C(   545396075) },
      { -INT32_C(   746901385), -INT32_C(  1603040114), -INT32_C(  2082895123), -INT32_C(   371289801),  INT32_C(  1432495479), -INT32_C(  1363994774), -INT32_C(  1725678290), -INT32_C(   893737389) },
      {  INT32_C(  1521309489),  INT32_C(  1469523908), -INT32_C(   333297886),  INT32_C(  1252907990),  INT32_C(  1497146355), -INT32_C(   720144006), -INT32_C(  2054087262), -INT32_C(  1420229616) } },
    { {  INT32_C(  1704867287), -INT32_C(  1358556735), -INT32_C(   768417893), -INT32_C(   407170960), -INT32_C(  1623384779),  INT32_C(  1666052149), -INT32_C(  2114162130), -INT32_C(   280250856) },
      { -INT32_C(  1386878485), -INT32_C(  1772332038), -INT32_C(  1419211206), -INT32_C(   745397090),  INT32_C(  1987301185), -INT32_C(   287719232),  INT32_C(  1265620530),  INT32_C(  2017114764) },
      { -INT32_C(   240207060),  INT32_C(   112990279),  INT32_C(  1867041662),  INT32_C(   971949084),  INT32_C(  1048197739),  INT32_C(  1666052149), -INT32_C(   927332367), -INT32_C(  1151234316) } },
    { { -INT32_C(  1641705564),  INT32_C(   624198122), -INT32_C(  1378837233),  INT32_C(    41968321), -INT32_C(   226888910), -INT32_C(   421506380), -INT32_C(  1255059672), -INT32_C(  1389532151) },
      { -INT32_C(   431205637), -INT32_C(   502562605), -INT32_C(   560997603),  INT32_C(  1877020477), -INT32_C(  1218356733), -INT32_C(   711114324), -INT32_C(  1702178928),  INT32_C(   910669627) },
      {  INT32_C(   619785341),  INT32_C(   256977316), -INT32_C(   172354655),  INT32_C(   542116952), -INT32_C(  1815111273),  INT32_C(    86724206),  INT32_C(  1328067889),  INT32_C(  1298754400) } },
    { { -INT32_C(   602107127),  INT32_C(   834545428),  INT32_C(  1074810114),  INT32_C(  1605366108), -INT32_C(   149548725), -INT32_C(   473123758), -INT32_C(  1099082109),  INT32_C(   385139725) },
      {  INT32_C(  1827803480),  INT32_C(  1000190008),  INT32_C(  1518054909), -INT32_C(   373740898), -INT32_C(  1897803717),  INT32_C(   124890500),  INT32_C(   298184196),  INT32_C(   170441138) },
      {  INT32_C(   165420179),  INT32_C(   338804263),  INT32_C(  1208093088),  INT32_C(   401341527),  INT32_C(  1605939338),  INT32_C(  1019954478), -INT32_C(   405444549),  INT32_C(   271866835) } },
    { {  INT32_C(    58071754), -INT32_C(   935455541),  INT32_C(  1612888513),  INT32_C(   524934371),  INT32_C(   816655020), -INT32_C(   600367400), -INT32_C(  1091699700), -INT32_C(  2134305355) },
      { -INT32_C(    75284688),  INT32_C(   365150547),  INT32_C(  1568007802),  INT32_C(  1853669058), -INT32_C(  1046599191),  INT32_C(  1402852679), -INT32_C(  2045605167), -INT32_C(   804856928) },
      {  INT32_C(   449446774), -INT32_C(  1503772176),  INT32_C(    92310247),  INT32_C(  2099737484),  INT32_C(  1515542625),  INT32_C(   462384238), -INT32_C(   132547109), -INT32_C(  2134305355) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi32(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi32(test_vec[i].b);
    simde__m256i r = simde_mm256_rolv_epi32(a, b);
    simde_test_x86_assert_equal_i32x8(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m256i a = simde_test_x86_random_i32x8();
    simde__m256i b = simde_test_x86_random_i32x8();
    simde__m256i r = simde_mm256_rolv_epi32(a, b);

    simde_test_x86_write_i32x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_mask_rolv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t src[8];
    const simde__mmask8 k;
    const int32_t a[8];
    const int32_t b[8];
    const int32_t r[8];
  } test_vec[] = {
    { {  INT32_C(  1858832666), -INT32_C(   998011318),  INT32_C(   908261492), -INT32_C(  1616601418),  INT32_C(   257966792), -INT32_C(   396100329),  INT32_C(   678393224),  INT32_C(  1794667856) },
         UINT8_MAX,
      {  INT32_C(  1363794115),  INT32_C(  1405488731), -INT32_C(   838140880), -INT32_C(   476665439),  INT32_C(   133932554),  INT32_C(  2123358985), -INT32_C(   925976750),  INT32_C(  1925658799) },
      {  INT32_C(  1824788497),  INT32_C(  1321241118),  INT32_C(   656263814),  INT32_C(  2097918835),  INT32_C(  1652885081),  INT32_C(  1021318377),  INT32_C(  2063904460), -INT32_C(   118567961) },
      { -INT32_C(  1316576621), -INT32_C(   722369642), -INT32_C(  2101408717),  INT32_C(  1292836021),  INT32_C(   336590668),  INT32_C(   533074685), -INT32_C(   344642420),  INT32_C(  1671190457) } },
    { { -INT32_C(    94063909), -INT32_C(  1035459524),  INT32_C(  1659528686),  INT32_C(  1927279897), -INT32_C(   455777029),  INT32_C(  1159771513),  INT32_C(  1270883428), -INT32_C(   884756753) },
      UINT8_C( 96),
      { -INT32_C(   862141017),  INT32_C(  1941593869),  INT32_C(  1049369673),  INT32_C(  1614413564), -INT32_C(  1982259757),  INT32_C(  1659706942), -INT32_C(  1923991329),  INT32_C(   586030459) },
      { -INT32_C(   269579550),  INT32_C(   845326825), -INT32_C(  1049563451), -INT32_C(  1071470099),  INT32_C(    88734663), -INT32_C(   110676454), -INT32_C(   377046674), -INT32_C(  1207143210) },
      { -INT32_C(    94063909), -INT32_C(  1035459524),  INT32_C(  1659528686),  INT32_C(  1927279897), -INT32_C(   455777029), -INT32_C(   108284808), -INT32_C(  1908939948), -INT32_C(   884756753) } },
    { { -INT32_C(   391578881),  INT32_C(  1763379875), -INT32_C(   433419271), -INT32_C(    56210379),  INT32_C(  1644294216), -INT32_C(  1789171674), -INT32_C(   125902558),  INT32_C(  1437698646) },
      UINT8_C(133),
      {  INT32_C(  1663581529), -INT32_C(   463695528),  INT32_C(   135873212), -INT32_C(   649062935),  INT32_C(  2130686486), -INT32_C(   274689011), -INT32_C(  1656383213), -INT32_C(  1558013366) },
      {  INT32_C(   805718999), -INT32_C(  1743494180), -INT32_C(  1902039643),  INT32_C(  1483206978), -INT32_C(  1311348828),  INT32_C(   245397499),  INT32_C(  1521280272),  INT32_C(  1476251263) },
      { -INT32_C(  1397648354),  INT32_C(  1763379875),  INT32_C(    52975489), -INT32_C(    56210379),  INT32_C(  1644294216), -INT32_C(  1789171674), -INT32_C(   125902558),  INT32_C(  1368476965) } },
    { { -INT32_C(   158924007),  INT32_C(   210672486),  INT32_C(   177876936), -INT32_C(   983432671),  INT32_C(  1685469289), -INT32_C(  1082976337),  INT32_C(  2082021372),  INT32_C(   131274477) },
      UINT8_C( 26),
      { -INT32_C(   176095910), -INT32_C(  1145205621),  INT32_C(   702334759),  INT32_C(  1636999465),  INT32_C(   772929047), -INT32_C(  2010394519),  INT32_C(     7776234), -INT32_C(   736461446) },
      {  INT32_C(    97098618),  INT32_C(  1321240103),  INT32_C(  1987550285),  INT32_C(  1440155965),  INT32_C(  1770252288), -INT32_C(  1544376647), -INT32_C(   811374507),  INT32_C(  1604566757) },
      { -INT32_C(   158924007), -INT32_C(   557431331),  INT32_C(   177876936),  INT32_C(   741495845),  INT32_C(   772929047), -INT32_C(  1082976337),  INT32_C(  2082021372),  INT32_C(   131274477) } },
    { { -INT32_C(  2140902311),  INT32_C(  1070540274), -INT32_C(     4897087),  INT32_C(  1347718480),  INT32_C(   783931253), -INT32_C(   607016058), -INT32_C(   122981101), -INT32_C(  1940435405) },
      UINT8_C(187),
      { -INT32_C(   508752708),  INT32_C(   581103067),  INT32_C(   796041634), -INT32_C(   844774667),  INT32_C(   659805051),  INT32_C(   440020645),  INT32_C(   676148185),  INT32_C(  1189337482) },
      { -INT32_C(  1054371611),  INT32_C(   551799421),  INT32_C(  1632589163), -INT32_C(  1842416617),  INT32_C(  1840873928), -INT32_C(  1987578705), -INT32_C(  1313745881), -INT32_C(  1829202771) },
      {  INT32_C(   899782556),  INT32_C(  1683250619), -INT32_C(     4897087),  INT32_C(  2061947617),  INT32_C(  1406368551),  INT32_C(   391286045), -INT32_C(   122981101),  INT32_C(  2066827484) } },
    { { -INT32_C(  1571610844),  INT32_C(  1438791401), -INT32_C(  1565126261), -INT32_C(   835329018),  INT32_C(   339471973),  INT32_C(   161333986),  INT32_C(  1136348822),  INT32_C(   131445730) },
      UINT8_C(210),
      {  INT32_C(  1606199593),  INT32_C(  2095780203), -INT32_C(  1417507385), -INT32_C(  1324330814),  INT32_C(  1301488779),  INT32_C(   283352514),  INT32_C(   200484440),  INT32_C(   635304700) },
      {  INT32_C(   243571107),  INT32_C(  1904897962), -INT32_C(  1088680451), -INT32_C(   378524579),  INT32_C(   322307153), -INT32_C(   115139935),  INT32_C(  1006900800), -INT32_C(  1285431024) },
      { -INT32_C(  1571610844), -INT32_C(  1404719629), -INT32_C(  1565126261), -INT32_C(   835329018),  INT32_C(  1226218278),  INT32_C(   161333986),  INT32_C(   200484440), -INT32_C(    84138531) } },
    { {  INT32_C(   633464443),  INT32_C(  1385647189), -INT32_C(  1240288423),  INT32_C(   832537312),  INT32_C(   658822790),  INT32_C(   807430128), -INT32_C(  1905449858), -INT32_C(  2126328314) },
      UINT8_C(180),
      {  INT32_C(  1326097923), -INT32_C(   240624579), -INT32_C(   237936786), -INT32_C(   730397954), -INT32_C(  1379623098), -INT32_C(   466881089),  INT32_C(   820689506), -INT32_C(     1741828) },
      {  INT32_C(  1330638609), -INT32_C(  1170147509),  INT32_C(  1437274454),  INT32_C(  1495867923), -INT32_C(  2130252095),  INT32_C(  1164260067), -INT32_C(   394965012), -INT32_C(   874030406) },
      {  INT32_C(   633464443),  INT32_C(  1385647189), -INT32_C(   608406441),  INT32_C(   832537312),  INT32_C(  1535721101),  INT32_C(   559918591), -INT32_C(  1905449858), -INT32_C(   201353809) } },
    { { -INT32_C(  1793444279), -INT32_C(  2075174354),  INT32_C(  2128214635), -INT32_C(   556334308), -INT32_C(   731914511), -INT32_C(    65420272), -INT32_C(   840659437),  INT32_C(   848874473) },
      UINT8_C(  2),
      {  INT32_C(   221300659),  INT32_C(   276411414), -INT32_C(  1859258482), -INT32_C(  1383986225),  INT32_C(   784160362), -INT32_C(    46024337),  INT32_C(  1759907485),  INT32_C(  1500125606) },
      { -INT32_C(   144205088), -INT32_C(   586686385), -INT32_C(  1502726953), -INT32_C(  1454116801), -INT32_C(  1244196794),  INT32_C(  1722947785), -INT32_C(   842032858), -INT32_C(  1842988622) },
      { -INT32_C(  1793444279), -INT32_C(   636811204),  INT32_C(  2128214635), -INT32_C(   556334308), -INT32_C(   731914511), -INT32_C(    65420272), -INT32_C(   840659437),  INT32_C(   848874473) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i src = simde_x_mm256_loadu_epi32(test_vec[i].src);
    simde__mmask8 k = test_vec[i].k;
    simde__m256i a = simde_x_mm256_loadu_epi32(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi32(test_vec[i].b);
    simde__m256i r = simde_mm256_mask_rolv_epi32(src, k, a, b);
    simde_test_x86_assert_equal_i32x8(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m256i src = simde_test_x86_random_i32x8();
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m256i a = simde_test_x86_random_i32x8();
    simde__m256i b = simde_test_x86_random_i32x8();
    simde__m256i r = simde_mm256_mask_rolv_epi32(src, k, a, b);

    simde_test_x86_write_i32x8(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_maskz_rolv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask8 k;
    const int32_t a[8];
    const int32_t b[8];
    const int32_t r[8];
  } test_vec[] = {
    { UINT8_C(212),
      {  INT32_C(  1831045517), -INT32_C(   985333615),  INT32_C(  1577446254),  INT32_C(  1336192575), -INT32_C(  1642505850), -INT32_C(  1530560757), -INT32_C(  2007592370),  INT32_C(  1180494264) },
      {  INT32_C(    62095218), -INT32_C(   305530753),  INT32_C(   592170724),  INT32_C(    41086844),  INT32_C(  1403095880),  INT32_C(  1509451275), -INT32_C(  1327411464), -INT32_C(  1426703049) },
      {  INT32_C(           0),  INT32_C(           0), -INT32_C(   530663707),  INT32_C(           0),  INT32_C(   425297566),  INT32_C(           0),  INT32_C(  1317557906), -INT32_C(   601674124) } },
    { UINT8_C(188),
      { -INT32_C(  1573147222),  INT32_C(  1149642870),  INT32_C(  1656858995), -INT32_C(  1481981156), -INT32_C(   894239132),  INT32_C(  1153567990),  INT32_C(   729576429),  INT32_C(   333915753) },
      {  INT32_C(  1253450707), -INT32_C(  1081197493),  INT32_C(    35737574),  INT32_C(  2007616531), -INT32_C(  1069392694),  INT32_C(  1426326632), -INT32_C(   511672201),  INT32_C(  2062837926) },
      {  INT32_C(           0),  INT32_C(           0), -INT32_C(  1335206696),  INT32_C(   417676630), -INT32_C(   872836310), -INT32_C(  1039337916),  INT32_C(           0), -INT32_C(   104228284) } },
    { UINT8_C(139),
      { -INT32_C(   405355350), -INT32_C(  1563585198), -INT32_C(  2085236809), -INT32_C(   733139848),  INT32_C(  1933315438), -INT32_C(  1830120943),  INT32_C(  2050542354),  INT32_C(  1778758336) },
      { -INT32_C(   917382026),  INT32_C(   678108785),  INT32_C(  1722491117), -INT32_C(  1153763252),  INT32_C(   388921093),  INT32_C(   464066569), -INT32_C(  1533615644),  INT32_C(   168729492) },
      {  INT32_C(   716830129),  INT32_C(   715474331),  INT32_C(           0), -INT32_C(   758674108),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(   738631771) } },
    { UINT8_C(119),
      {  INT32_C(  2129253216),  INT32_C(  1584140606), -INT32_C(  1247030595), -INT32_C(  2084870644), -INT32_C(  1400057196),  INT32_C(  1552984187), -INT32_C(   638569410), -INT32_C(  1554908349) },
      {  INT32_C(   220281550),  INT32_C(   141266251),  INT32_C(  1824396895),  INT32_C(   284129660), -INT32_C(   960660405),  INT32_C(  1646415140), -INT32_C(   985984126), -INT32_C(   597128178) },
      {  INT32_C(  1960320954),  INT32_C(  1619653363), -INT32_C(   623515298),  INT32_C(           0),  INT32_C(  1721017700), -INT32_C(   922056779),  INT32_C(  1740689659),  INT32_C(           0) } },
    { UINT8_C(198),
      {  INT32_C(   387115401),  INT32_C(  1802902101),  INT32_C(  1374216920),  INT32_C(  1318910162),  INT32_C(    57828021), -INT32_C(  1736059771), -INT32_C(  1666823665),  INT32_C(   996311730) },
      { -INT32_C(  1051560852),  INT32_C(  1730988431),  INT32_C(  2109281451), -INT32_C(  1026861811),  INT32_C(  1036336568),  INT32_C(   567626257),  INT32_C(  1186823060),  INT32_C(  1770135549) },
      {  INT32_C(           0),  INT32_C(   220902843),  INT32_C(  1192673935),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1594242460),  INT32_C(  1198280790) } },
    { UINT8_C(148),
      { -INT32_C(  1658639660),  INT32_C(  1799981655), -INT32_C(  1720138173), -INT32_C(   816759918),  INT32_C(  1273073152), -INT32_C(   555810205), -INT32_C(   556063041),  INT32_C(  2087863720) },
      { -INT32_C(   971336337),  INT32_C(  1664246560), -INT32_C(  1141069271), -INT32_C(   427078171),  INT32_C(  1043426523),  INT32_C(   756813934), -INT32_C(   569640906), -INT32_C(  1403290051) },
      {  INT32_C(           0),  INT32_C(           0), -INT32_C(   242448590),  INT32_C(           0),  INT32_C(    39783536),  INT32_C(           0),  INT32_C(           0),  INT32_C(   260982965) } },
    { UINT8_C( 19),
      { -INT32_C(   667716747),  INT32_C(  1325504421), -INT32_C(   516571757), -INT32_C(  1262740664),  INT32_C(  1545796428),  INT32_C(   261312279), -INT32_C(   666078886),  INT32_C(  1106049484) },
      {  INT32_C(   286859116),  INT32_C(  1247812534),  INT32_C(   539727320), -INT32_C(    53155920), -INT32_C(    94767390), -INT32_C(  1626739643),  INT32_C(   712529501), -INT32_C(  1150589873) },
      {  INT32_C(   926375299), -INT32_C(   380387227),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1888218417),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
    { UINT8_C(131),
      { -INT32_C(  1606759291), -INT32_C(  1032289236), -INT32_C(  1770874705),  INT32_C(  1652125548), -INT32_C(  1264094264), -INT32_C(   770619780),  INT32_C(   572603326),  INT32_C(   749132967) },
      { -INT32_C(   707993432),  INT32_C(   328680548),  INT32_C(  1219037660),  INT32_C(  1084891768),  INT32_C(   318001557),  INT32_C(  1457784471), -INT32_C(   394787263), -INT32_C(  1961615646) },
      {  INT32_C(   986482080),  INT32_C(   663241420),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1298435428) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__mmask8 k = test_vec[i].k;
    simde__m256i a = simde_x_mm256_loadu_epi32(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi32(test_vec[i].b);
    simde__m256i r = simde_mm256_maskz_rolv_epi32(k, a, b);
    simde_test_x86_assert_equal_i32x8(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m256i a = simde_test_x86_random_i32x8();
    simde__m256i b = simde_test_x86_random_i32x8();
    simde__m256i r = simde_mm256_maskz_rolv_epi32(k, a, b);

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_rolv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { { -INT32_C(   797240390),  INT32_C(  1079064947),  INT32_C(   308249100), -INT32_C(  1672874639),  INT32_C(  1895926149), -INT32_C(   324616924),  INT32_C(  1513473617), -INT32_C(  1798755110),
        -INT32_C(  1285274561), -INT32_C(  1963739778), -INT32_C(  1449372873), -INT32_C(   834280120), -INT32_C(  1824569746),  INT32_C(  1417667843), -INT32_C(  2035305300),  INT32_C(   521828576) },
      {  INT32_C(   986873532),  INT32_C(  1791280691),  INT32_C(  1628659737), -INT32_C(  1271965626), -INT32_C(  1572376673),  INT32_C(    16238164),  INT32_C(  1518773882), -INT32_C(   629497826),
         INT32_C(  1360350238),  INT32_C(   733796626), -INT32_C(  2138190022), -INT32_C(   952845272), -INT32_C(  2140570581), -INT32_C(  1115660222),  INT32_C(   605488902), -INT32_C(   956329561) },
      { -INT32_C(  1392004805), -INT32_C(   879099255),  INT32_C(   405061380),  INT32_C(   310205543), -INT32_C(  1199520574), -INT32_C(   766588309),  INT32_C(  1164498713), -INT32_C(  1523430602),
        -INT32_C(   321318641), -INT32_C(   705025074), -INT32_C(   559517364),  INT32_C(  1172654286), -INT32_C(    97291111),  INT32_C(  1375704077), -INT32_C(  1410520287), -INT32_C(  1925418993) } },
    { { -INT32_C(   266922786),  INT32_C(   656200685), -INT32_C(   878204509), -INT32_C(  1852646299), -INT32_C(  1710097576),  INT32_C(  1649906012),  INT32_C(  1082617752), -INT32_C(   569997824),
        -INT32_C(  2016469606), -INT32_C(  1800410127), -INT32_C(   111192428), -INT32_C(  1970605774),  INT32_C(  1210424300), -INT32_C(   978617299), -INT32_C(   335203605),  INT32_C(  1388972984) },
      {  INT32_C(   450468137),  INT32_C(   414091396),  INT32_C(   286395871), -INT32_C(   358834946),  INT32_C(  1697890616),  INT32_C(   690675261), -INT32_C(   938135536),  INT32_C(  1679482683),
        -INT32_C(    42011783),  INT32_C(  1528179836),  INT32_C(   929835065), -INT32_C(    64878396),  INT32_C(   123819465),  INT32_C(  1127255091), -INT32_C(   150256196), -INT32_C(  1654905564) },
      {  INT32_C(   774487520),  INT32_C(  1909276370), -INT32_C(   439102255),  INT32_C(  1684322073),  INT32_C(  1486492155), -INT32_C(  1941245397),  INT32_C(  1872248967),  INT32_C(   116405296),
         INT32_C(   890215995),  INT32_C(   424345279),  INT32_C(   703774381), -INT32_C(  1464921304),  INT32_C(  1261951120), -INT32_C(   512873125), -INT32_C(  1094692050),  INT32_C(   748731269) } },
    { { -INT32_C(  1801725416),  INT32_C(  1089450247), -INT32_C(  1653056295),  INT32_C(   781884004),  INT32_C(   573963247),  INT32_C(  1130718599), -INT32_C(   834965334), -INT32_C(  1385392235),
         INT32_C(  2017593201), -INT32_C(  1850134088), -INT32_C(   231853683), -INT32_C(  1172256565),  INT32_C(  1272731076),  INT32_C(  1687110074),  INT32_C(  1177733809), -INT32_C(   755786143) },
      {  INT32_C(  1565210021), -INT32_C(   185727897),  INT32_C(    15080757), -INT32_C(  1447426331),  INT32_C(   368416347), -INT32_C(  2005302057), -INT32_C(  1345409970), -INT32_C(   259931830),
         INT32_C(  1582157047),  INT32_C(    89275600),  INT32_C(  1040529753), -INT32_C(  1679245505),  INT32_C(   749788501), -INT32_C(  1347147167),  INT32_C(   543064790),  INT32_C(   990961476) },
      { -INT32_C(  1820638446),  INT32_C(  2010678176), -INT32_C(  1691111669), -INT32_C(   749515643),  INT32_C(  2031202271), -INT32_C(  1012813134), -INT32_C(   601181298), -INT32_C(  1302440267),
        -INT32_C(  1195630333),  INT32_C(   834179513),  INT32_C(   467950691), -INT32_C(   586128283), -INT32_C(  1198949494), -INT32_C(   920747148), -INT32_C(  1403941710),  INT32_C(   792323613) } },
    { {  INT32_C(  2090426028), -INT32_C(   209523814),  INT32_C(  1680967460), -INT32_C(  1661003449),  INT32_C(  1472835574), -INT32_C(  1358529063),  INT32_C(  1154507776), -INT32_C(   260054972),
        -INT32_C(   664004546),  INT32_C(   684453379), -INT32_C(  1114833546),  INT32_C(   223972118),  INT32_C(   342106682), -INT32_C(  1597805664),  INT32_C(   333747151), -INT32_C(  1308400780) },
      {  INT32_C(  2123067515), -INT32_C(   727296418),  INT32_C(  1787900499), -INT32_C(   126358595), -INT32_C(  1391666419),  INT32_C(   374198086), -INT32_C(   685166237),  INT32_C(   260648340),
        -INT32_C(    74574691), -INT32_C(  1093651349),  INT32_C(   606626151),  INT32_C(  1511825228), -INT32_C(  1056495493),  INT32_C(  1540838904),  INT32_C(   456262022), -INT32_C(   886392018) },
      {  INT32_C(  1675938549), -INT32_C(  1126122778),  INT32_C(   958603660), -INT32_C(   207625432),  INT32_C(   905890553), -INT32_C(  1046514069),  INT32_C(   646127618),  INT32_C(    72288254),
        -INT32_C(   619871481),  INT32_C(  1601182022), -INT32_C(   964773026), -INT32_C(  1733205803), -INT32_C(   794615535), -INT32_C(  1600076949), -INT32_C(   115018812), -INT32_C(   656593792) } },
    { {  INT32_C(  1002879184),  INT32_C(  1425643245),  INT32_C(  1148723960),  INT32_C(  1017025985), -INT32_C(  1241602371), -INT32_C(  2129603077),  INT32_C(    77349590), -INT32_C(   825243650),
         INT32_C(  1812567679),  INT32_C(   616563244), -INT32_C(   429311708), -INT32_C(  1960704050), -INT32_C(  1472126803), -INT32_C(   869641738), -INT32_C(  1831745900),  INT32_C(   224436365) },
      {  INT32_C(  1668901174), -INT32_C(  1870185877),  INT32_C(  1098313843), -INT32_C(  1513318152), -INT32_C(  1353839175), -INT32_C(   210012065), -INT32_C(   863679426),  INT32_C(   601482733),
        -INT32_C(  1182379442), -INT32_C(    11989364), -INT32_C(   163528706),  INT32_C(   295439704),  INT32_C(  2042685978), -INT32_C(  1603519646),  INT32_C(  1970073992),  INT32_C(   613959126) },
      {  INT32_C(   873394606), -INT32_C(   860394841),  INT32_C(   398599105), -INT32_C(  1052991851),  INT32_C(  2070674765), -INT32_C(  1064801539), -INT32_C(  2128146251), -INT32_C(   117450279),
         INT32_C(  1704975106),  INT32_C(     2277964),  INT32_C(   966413897), -INT32_C(   829742585), -INT32_C(  1230961534),  INT32_C(   816400347), -INT32_C(   775514990),  INT32_C(   591616040) } },
    { {  INT32_C(   618536856),  INT32_C(   723789613),  INT32_C(  1059218663), -INT32_C(  1940865423),  INT32_C(   168104104), -INT32_C(   743804341),  INT32_C(   977803107), -INT32_C(   195108516),
         INT32_C(   756628480),  INT32_C(  1264074084),  INT32_C(   327842465), -INT32_C(   526394824),  INT32_C(   904570090),  INT32_C(  2047448342),  INT32_C(   146035116),  INT32_C(   855446066) },
      { -INT32_C(  1285614001), -INT32_C(   184633261),  INT32_C(  1795655730),  INT32_C(  1280026210),  INT32_C(  1635857995),  INT32_C(  2010876619),  INT32_C(   243240924), -INT32_C(   247366494),
        -INT32_C(   442195822), -INT32_C(  1948605864), -INT32_C(  1929977558), -INT32_C(   740802168),  INT32_C(  1110727031), -INT32_C(  1061613340),  INT32_C(  1104034207),  INT32_C(  1211240373) },
      {  INT32_C(   265032303),  INT32_C(   963205409), -INT32_C(  1818428279),  INT32_C(   826472902),  INT32_C(   679821392),  INT32_C(  1402101405),  INT32_C(   866419062), -INT32_C(   780434061),
        -INT32_C(   268389276),  INT32_C(  1682659389),  INT32_C(   703235150), -INT32_C(  1613088544),  INT32_C(  1964701010), -INT32_C(  1600564889),  INT32_C(    73017558),  INT32_C(  1179017122) } },
    { {  INT32_C(   137221808), -INT32_C(  1567422600),  INT32_C(  1898875369),  INT32_C(  1111754443),  INT32_C(  1149597792),  INT32_C(   671366792),  INT32_C(   761909879), -INT32_C(  1837785886),
        -INT32_C(   342187406), -INT32_C(  1819464022), -INT32_C(  2113618761),  INT32_C(   583354562),  INT32_C(  1231440576),  INT32_C(     7432840),  INT32_C(   523098685), -INT32_C(   374234506) },
      { -INT32_C(   288076988),  INT32_C(   830562681), -INT32_C(   525105891), -INT32_C(  1912440627),  INT32_C(  1272408514),  INT32_C(   273369299), -INT32_C(  1724942302),  INT32_C(  1585635354),
        -INT32_C(  1521658324), -INT32_C(   707342665),  INT32_C(   565545556), -INT32_C(   978340094), -INT32_C(   200243680), -INT32_C(   251372594), -INT32_C(   309709613),  INT32_C(  1078660116) },
      { -INT32_C(  2099418368), -INT32_C(   247126514),  INT32_C(   774230333), -INT32_C(  2133235640),  INT32_C(   303423873), -INT32_C(   197050335), -INT32_C(  1247327780), -INT32_C(  1974872461),
        -INT32_C(  1440272711),  INT32_C(  1430898327), -INT32_C(   881319861), -INT32_C(  1961549048),  INT32_C(  1231440576),  INT32_C(  1520566300), -INT32_C(   773260946),  INT32_C(   661560090) } },
    { {  INT32_C(   434542690), -INT32_C(  1141982105),  INT32_C(  1239196486),  INT32_C(  2064551003), -INT32_C(   512811502),  INT32_C(  1288860793), -INT32_C(  1137091416), -INT32_C(   889355160),
        -INT32_C(  2065439971), -INT32_C(   432024929), -INT32_C(   802219147), -INT32_C(  1169408601), -INT32_C(   727991461), -INT32_C(   685740753),  INT32_C(   831805897), -INT32_C(    84176419) },
      {  INT32_C(   327081588),  INT32_C(   653901232), -INT32_C(  2131351335), -INT32_C(  1069923739),  INT32_C(   764728830),  INT32_C(   184923202), -INT32_C(   348350195), -INT32_C(  1629145302),
        -INT32_C(   961453034), -INT32_C(    85152991),  INT32_C(   964420563),  INT32_C(   603567397), -INT32_C(   850293109),  INT32_C(  1356420674),  INT32_C(   423302639),  INT32_C(  1672945741) },
      { -INT32_C(  2044617111), -INT32_C(  1134052370), -INT32_C(  1936475834),  INT32_C(  1641122671), -INT32_C(  1201944700),  INT32_C(   860475877),  INT32_C(   731191175), -INT32_C(   166616277),
        -INT32_C(   949929736), -INT32_C(   864049857), -INT32_C(   609320584),  INT32_C(  1233630455), -INT32_C(   572858716),  INT32_C(  1552004287),  INT32_C(   753178826),  INT32_C(  1916518239) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_rolv_epi32(a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i a = simde_test_x86_random_i32x16();
    simde__m512i b = simde_test_x86_random_i32x16();
    simde__m512i r = simde_mm512_rolv_epi32(a, b);

    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_mask_rolv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t src[16];
    const simde__mmask16 k;
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { { -INT32_C(  1507235708), -INT32_C(   408938989),  INT32_C(   505420793),  INT32_C(  1547835857), -INT32_C(   366373977), -INT32_C(   667286807),  INT32_C(  1710322968),  INT32_C(   449357973),
         INT32_C(   616624400),  INT32_C(      745479),  INT32_C(  1293888380), -INT32_C(   324443836), -INT32_C(   573123852), -INT32_C(   306900779),  INT32_C(   441624197),  INT32_C(  1580472910) },
      UINT16_C(62475),
      { -INT32_C(  1923804286),  INT32_C(   850972947), -INT32_C(   946602722), -INT32_C(  1080457239),  INT32_C(   433024612), -INT32_C(  1379969957), -INT32_C(  1547235986), -INT32_C(   292039828),
        -INT32_C(   109253402), -INT32_C(   617859907),  INT32_C(   446873393), -INT32_C(  1411826617),  INT32_C(   113617066),  INT32_C(  1806927100), -INT32_C(    32605294),  INT32_C(   887924046) },
      {  INT32_C(  1328375954), -INT32_C(   836019811),  INT32_C(  1625935129), -INT32_C(  1274297847),  INT32_C(  1740296299), -INT32_C(   405639851),  INT32_C(   921034984),  INT32_C(   409653638),
        -INT32_C(   681076678),  INT32_C(   178623217),  INT32_C(  1768590944), -INT32_C(  1138919855), -INT32_C(  1692149946),  INT32_C(   763557445),  INT32_C(  1550084054),  INT32_C(  1937034809) },
      {  INT32_C(  1309291860),  INT32_C(  1716984354),  INT32_C(   505420793),  INT32_C(   856675199), -INT32_C(   366373977), -INT32_C(   667286807),  INT32_C(  1710322968),  INT32_C(   449357973),
         INT32_C(   616624400),  INT32_C(      745479),  INT32_C(   446873393), -INT32_C(   324443836), -INT32_C(  1318442367),  INT32_C(  1987092365), -INT32_C(   457210978), -INT32_C(  1670784694) } },
    { {  INT32_C(  1481301094), -INT32_C(   832376978), -INT32_C(   818360962), -INT32_C(  1987357373),  INT32_C(  1915072045),  INT32_C(  2074060708),  INT32_C(  1222050575),  INT32_C(   951798994),
        -INT32_C(  1768946392),  INT32_C(  1919284212),  INT32_C(    54631872),  INT32_C(   529321202),  INT32_C(   529641851),  INT32_C(  1754935641),  INT32_C(   112226868), -INT32_C(   432051266) },
      UINT16_C(53104),
      { -INT32_C(   507353988),  INT32_C(   410944215),  INT32_C(   317026693),  INT32_C(   583229584),  INT32_C(   441654399),  INT32_C(   881624964), -INT32_C(   845198706), -INT32_C(  1399058640),
         INT32_C(  1267621748),  INT32_C(  1734544609),  INT32_C(   242829437),  INT32_C(   674249896), -INT32_C(   582843559), -INT32_C(  1743598070),  INT32_C(  1214624024),  INT32_C(   905183937) },
      {  INT32_C(  1115718241),  INT32_C(   212460431), -INT32_C(   736484820), -INT32_C(  1191425441), -INT32_C(   661307699),  INT32_C(   611362572),  INT32_C(   443340377),  INT32_C(   961503704),
         INT32_C(  1920716771), -INT32_C(   562092622), -INT32_C(  1498179257), -INT32_C(  1319194652), -INT32_C(    91622162), -INT32_C(   199230821), -INT32_C(  1475441712), -INT32_C(   790536724) },
      {  INT32_C(  1481301094), -INT32_C(   832376978), -INT32_C(   818360962), -INT32_C(  1987357373),  INT32_C(  1670374218), -INT32_C(   931642552),  INT32_C(   496713365),  INT32_C(   951798994),
         INT32_C(  1551039394),  INT32_C(   864394636),  INT32_C(  1017396871), -INT32_C(  2096903550),  INT32_C(   529641851),  INT32_C(  1754935641), -INT32_C(  1323808667),  INT32_C(  1076630367) } },
    { { -INT32_C(   549299156), -INT32_C(   910311039),  INT32_C(  1064267867),  INT32_C(   250662432),  INT32_C(  1560836802),  INT32_C(  1146234740), -INT32_C(  1611898701), -INT32_C(   361771586),
        -INT32_C(  1412845015), -INT32_C(   831223949),  INT32_C(   386786295),  INT32_C(  1931869873), -INT32_C(   321835400),  INT32_C(   137372501),  INT32_C(  1101536387),  INT32_C(   304879593) },
      UINT16_C(62921),
      {  INT32_C(   830225853),  INT32_C(   420836363), -INT32_C(  1323841909),  INT32_C(   199200570), -INT32_C(  1423035269), -INT32_C(   439897795),  INT32_C(   536654067), -INT32_C(  2146122046),
         INT32_C(   246518019), -INT32_C(  1876441595), -INT32_C(   952025459),  INT32_C(  1221730509), -INT32_C(  1829568427), -INT32_C(  1535657295),  INT32_C(   767783786),  INT32_C(  1018026041) },
      {  INT32_C(  1850433129), -INT32_C(  1291947483),  INT32_C(  2121875633), -INT32_C(  1245230240), -INT32_C(    62408117), -INT32_C(   543048075),  INT32_C(  1795974194), -INT32_C(  1515669188),
         INT32_C(  1024717592),  INT32_C(   384766565), -INT32_C(  1298831278), -INT32_C(    26714957), -INT32_C(  1946505450), -INT32_C(  1603560594),  INT32_C(  1007384320),  INT32_C(  1239593777) },
      { -INT32_C(   126125470), -INT32_C(   910311039),  INT32_C(  1064267867),  INT32_C(   199200570),  INT32_C(  1560836802),  INT32_C(  1146234740), -INT32_C(  1010008078),  INT32_C(   671173740),
         INT32_C(    51294609), -INT32_C(   831223949), -INT32_C(    97051388),  INT32_C(  1931869873),  INT32_C(   358923456), -INT32_C(   290690787),  INT32_C(   767783786), -INT32_C(  1334675109) } },
    { {  INT32_C(   210171302),  INT32_C(  1495430407), -INT32_C(  1878214691),  INT32_C(   713978899), -INT32_C(  1850374109),  INT32_C(   607199268), -INT32_C(   916439656), -INT32_C(  1760410896),
         INT32_C(  1067685944), -INT32_C(   359021299), -INT32_C(  1871010435),  INT32_C(  1018824729), -INT32_C(  1261539184),  INT32_C(   685309840),  INT32_C(   770783548), -INT32_C(  1278999685) },
      UINT16_C(26523),
      { -INT32_C(  1943230221),  INT32_C(   204581266), -INT32_C(   199996870),  INT32_C(  1415882118),  INT32_C(   844428633),  INT32_C(   258707485), -INT32_C(  2129467715), -INT32_C(  1914130790),
        -INT32_C(   384232106), -INT32_C(   118142274),  INT32_C(   451742356),  INT32_C(   141513135),  INT32_C(  1681572678),  INT32_C(   292791635),  INT32_C(   647136652), -INT32_C(  1967949005) },
      {  INT32_C(  1316211856), -INT32_C(  1438160874),  INT32_C(   566506610), -INT32_C(   869649531),  INT32_C(  1244685558), -INT32_C(  1772379382),  INT32_C(  1555885352), -INT32_C(   119115928),
         INT32_C(  1363630395),  INT32_C(   888901313),  INT32_C(  1196801986), -INT32_C(   384598029), -INT32_C(   298630172),  INT32_C(   260345574), -INT32_C(   462733188), -INT32_C(   354594385) },
      { -INT32_C(  1460433876),  INT32_C(  1686309994), -INT32_C(  1878214691), -INT32_C(  1936412470),  INT32_C(  1447859517),  INT32_C(   607199268), -INT32_C(   916439656), -INT32_C(   391210355),
        -INT32_C(  1219966806), -INT32_C(   236284547),  INT32_C(  1806969424),  INT32_C(  1018824729), -INT32_C(  1261539184),  INT32_C(  1558795460), -INT32_C(  1033295784), -INT32_C(  1278999685) } },
    { {  INT32_C(  1799038122),  INT32_C(  1956591282), -INT32_C(   390334987),  INT32_C(  1490210676), -INT32_C(   112851694),  INT32_C(   269011604), -INT32_C(  1175162102),  INT32_C(  1856229828),
        -INT32_C(  1478893835),  INT32_C(   169572628), -INT32_C(   487401618), -INT32_C(  1204042586),  INT32_C(  1588691402),  INT32_C(  1450096972), -INT32_C(   267361748),  INT32_C(   694072116) },
      UINT16_C(14482),
      { -INT32_C(   323901743), -INT32_C(  1547493456),  INT32_C(  1013410049), -INT32_C(   742510302), -INT32_C(    41088369),  INT32_C(  1885387104),  INT32_C(   136615082), -INT32_C(  1908361539),
         INT32_C(   226226524),  INT32_C(   313540112), -INT32_C(   900851800), -INT32_C(   660796343),  INT32_C(  2010524182), -INT32_C(  1897449756), -INT32_C(  2020209718),  INT32_C(   504813505) },
      { -INT32_C(   651456056),  INT32_C(  2011945935),  INT32_C(   994130418),  INT32_C(  1544871750), -INT32_C(   304879095), -INT32_C(   360989920), -INT32_C(  2005855546), -INT32_C(  1314486295),
        -INT32_C(   393555688), -INT32_C(  1637911124), -INT32_C(   170221137), -INT32_C(  1991119232), -INT32_C(   126409256), -INT32_C(  1495076128), -INT32_C(   315731196), -INT32_C(   207629093) },
      {  INT32_C(  1799038122), -INT32_C(  1881648671), -INT32_C(   390334987),  INT32_C(  1490210676),  INT32_C(   437592059),  INT32_C(   269011604), -INT32_C(  1175162102), -INT32_C(  2123531492),
        -INT32_C(  1478893835),  INT32_C(   169572628), -INT32_C(   487401618), -INT32_C(   660796343),  INT32_C(   376952362), -INT32_C(  1897449756), -INT32_C(   267361748),  INT32_C(   694072116) } },
    { {  INT32_C(  1373317541),  INT32_C(  1324366751),  INT32_C(  1547946716), -INT32_C(  1864002376), -INT32_C(  1719116615),  INT32_C(  1396730447), -INT32_C(  1740542275), -INT32_C(   393420733),
        -INT32_C(  1472567543),  INT32_C(  2130061986), -INT32_C(  1394984460), -INT32_C(  2026061618),  INT32_C(  1797375004), -INT32_C(   339779282),  INT32_C(   310640847), -INT32_C(   369422112) },
      UINT16_C(13687),
      { -INT32_C(  2007033198),  INT32_C(  1942115224),  INT32_C(   993235199),  INT32_C(   956256023),  INT32_C(  2056859067), -INT32_C(  1669699304),  INT32_C(  1991006843), -INT32_C(   710204349),
        -INT32_C(   698480066),  INT32_C(  1548296285), -INT32_C(   946373456), -INT32_C(  2030004532), -INT32_C(   603874877),  INT32_C(  2121825026),  INT32_C(   418653653), -INT32_C(  2014404535) },
      {  INT32_C(   140332202),  INT32_C(   476358508), -INT32_C(   270271453),  INT32_C(  1450632082),  INT32_C(  2134013821), -INT32_C(   939676942),  INT32_C(   417395407),  INT32_C(  1017106066),
        -INT32_C(  2042299110), -INT32_C(   945641052),  INT32_C(   934774181), -INT32_C(   426955415), -INT32_C(  1754939483),  INT32_C(   962552682), -INT32_C(   414040235),  INT32_C(   656666893) },
      {  INT32_C(  2087340577),  INT32_C(   624527164), -INT32_C(   644052999), -INT32_C(  1864002376),  INT32_C(  1867720119), -INT32_C(  1537052183),  INT32_C(   759020374), -INT32_C(   393420733),
        -INT32_C(    78022616),  INT32_C(  2130061986), -INT32_C(   219179496), -INT32_C(  2026061618), -INT32_C(  2144126853), -INT32_C(   504624647),  INT32_C(   310640847), -INT32_C(   369422112) } },
    { { -INT32_C(  1834129170), -INT32_C(  1235595503),  INT32_C(  1038946772), -INT32_C(   484213954), -INT32_C(  1535407814),  INT32_C(  1105124076),  INT32_C(   640167961),  INT32_C(   273500961),
        -INT32_C(   995951949),  INT32_C(   494599241),  INT32_C(  1281058829),  INT32_C(   489651939), -INT32_C(   205346297), -INT32_C(  1657495420), -INT32_C(   238854960),  INT32_C(  1526796456) },
      UINT16_C(41994),
      { -INT32_C(  1700768992), -INT32_C(   872239503),  INT32_C(   692774394), -INT32_C(   975941117),  INT32_C(  2053462085), -INT32_C(  1177143818),  INT32_C(   684359206), -INT32_C(    87239462),
        -INT32_C(  1718326232),  INT32_C(   342202138),  INT32_C(  2134814588),  INT32_C(  1195643393),  INT32_C(  1623304554),  INT32_C(    85628894), -INT32_C(   265427946), -INT32_C(   521471560) },
      { -INT32_C(  2139521435), -INT32_C(  1835737579), -INT32_C(  1894657394),  INT32_C(  1339447012), -INT32_C(   558917633),  INT32_C(  1172556079),  INT32_C(  1714753709),  INT32_C(  1850089225),
        -INT32_C(  1276198755),  INT32_C(   742753182),  INT32_C(   985421397), -INT32_C(  1400270164),  INT32_C(  1502230570), -INT32_C(  1348571902), -INT32_C(  2045389955), -INT32_C(  1879810830) },
      { -INT32_C(  1834129170), -INT32_C(   835092395),  INT32_C(  1038946772),  INT32_C(  1564811324), -INT32_C(  1535407814),  INT32_C(  1105124076),  INT32_C(   640167961),  INT32_C(   273500961),
        -INT32_C(   995951949),  INT32_C(   494599241), -INT32_C(   275781675),  INT32_C(   489651939), -INT32_C(   205346297),  INT32_C(   342515576), -INT32_C(   238854960), -INT32_C(   421297237) } },
    { { -INT32_C(  1170021604), -INT32_C(  1142454426), -INT32_C(  1963613218),  INT32_C(  1597406773), -INT32_C(  1179074377), -INT32_C(  1436002771),  INT32_C(   456162857), -INT32_C(   156555814),
         INT32_C(  1857088776),  INT32_C(  1395234677),  INT32_C(  1893539642),  INT32_C(  1422857373),  INT32_C(    34441428),  INT32_C(   145520350), -INT32_C(   836510220),  INT32_C(   180669954) },
      UINT16_C(29884),
      { -INT32_C(  1593036424),  INT32_C(  1639990916), -INT32_C(  2039128394), -INT32_C(  1072805198), -INT32_C(   130618292),  INT32_C(   416623348), -INT32_C(  1125722120),  INT32_C(  1496359649),
         INT32_C(  1476082899),  INT32_C(   968473475), -INT32_C(   859886055), -INT32_C(   997405320), -INT32_C(  1363295558), -INT32_C(   456748308),  INT32_C(  1268821354),  INT32_C(   598069327) },
      { -INT32_C(  1871011827),  INT32_C(  1976120155), -INT32_C(   633239198),  INT32_C(   278908246),  INT32_C(  2076138639),  INT32_C(  1482655214), -INT32_C(  2103115982), -INT32_C(   593147441),
         INT32_C(  1147936745), -INT32_C(  1246153133),  INT32_C(   361757375),  INT32_C(  1445342919),  INT32_C(  2043798922), -INT32_C(  1647234966), -INT32_C(    14715601), -INT32_C(  1478769474) },
      { -INT32_C(  1170021604), -INT32_C(  1142454426),  INT32_C(   433421018), -INT32_C(  1397750894),  INT32_C(  1982233627),  INT32_C(  1253901877),  INT32_C(   456162857),  INT32_C(  1366338712),
         INT32_C(  1857088776),  INT32_C(  1395234677), -INT32_C(   429943028),  INT32_C(  1422857373), -INT32_C(   150279494),  INT32_C(   441168787),  INT32_C(  1454712272),  INT32_C(   180669954) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi32(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_rolv_epi32(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i src = simde_test_x86_random_i32x16();
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512i a = simde_test_x86_random_i32x16();
    simde__m512i b = simde_test_x86_random_i32x16();
    simde__m512i r = simde_mm512_mask_rolv_epi32(src, k, a, b);

    simde_test_x86_write_i32x16(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_rolv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask16 k;
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { UINT16_C(18659),
      { -INT32_C(  1518455060),  INT32_C(  2074099179),  INT32_C(  2024367954), -INT32_C(  1889717058),  INT32_C(  2126563501), -INT32_C(  2064322715), -INT32_C(   901205266),  INT32_C(  1175595866),
         INT32_C(  1307283554),  INT32_C(   533236685), -INT32_C(  1315409165),  INT32_C(  1396766118),  INT32_C(   584122557), -INT32_C(   559495696), -INT32_C(   760680840),  INT32_C(  2081995290) },
      {  INT32_C(   399049546), -INT32_C(  2110287217), -INT32_C(  1439445244), -INT32_C(  2097318972),  INT32_C(  1688522356),  INT32_C(   205671316),  INT32_C(  1407118137), -INT32_C(   254806363),
        -INT32_C(  2012767751),  INT32_C(   789200427), -INT32_C(   757514995),  INT32_C(   626316977),  INT32_C(   948566180),  INT32_C(  2101660483),  INT32_C(  1557144246),  INT32_C(   307011608) },
      { -INT32_C(   119819627),  INT32_C(   519421392),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   161984690), -INT32_C(   577466011), -INT32_C(  1035637944),
         INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   347298175),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1640715717),  INT32_C(           0) } },
    { UINT16_C(21305),
      { -INT32_C(  1517132646),  INT32_C(  1826791315), -INT32_C(   985492623),  INT32_C(  1103029944),  INT32_C(  1661796639),  INT32_C(  1334231934),  INT32_C(  1810865695),  INT32_C(  1270818992),
         INT32_C(   535843212),  INT32_C(  1653330672),  INT32_C(   505924966), -INT32_C(   765401677),  INT32_C(  1698065895),  INT32_C(  1354021936),  INT32_C(   196846426),  INT32_C(  1465285323) },
      { -INT32_C(  1133099317),  INT32_C(  2115895576), -INT32_C(  2103622194),  INT32_C(   307559723), -INT32_C(  1703441814), -INT32_C(  1578488762), -INT32_C(  1683183921), -INT32_C(   319684064),
         INT32_C(  1621649736),  INT32_C(   970966634),  INT32_C(   918256651), -INT32_C(   481816711), -INT32_C(   528629606), -INT32_C(  1149146900),  INT32_C(   777399566),  INT32_C(  1998211375) },
      { -INT32_C(  1826302676),  INT32_C(           0),  INT32_C(           0), -INT32_C(   147471859),  INT32_C(   872709516), -INT32_C(   508502125),  INT32_C(           0),  INT32_C(           0),
        -INT32_C(   263091169),  INT32_C(   793493898),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1651189321),  INT32_C(           0), -INT32_C(   388594962),  INT32_C(           0) } },
    { UINT16_C(49842),
      { -INT32_C(  1215816488),  INT32_C(   271815765),  INT32_C(   320908490),  INT32_C(   215202447),  INT32_C(   477413274), -INT32_C(   783711622), -INT32_C(   887457359), -INT32_C(   930165520),
         INT32_C(  1048516328),  INT32_C(  1968091818), -INT32_C(   309825954), -INT32_C(  1007002840), -INT32_C(  1797296614), -INT32_C(  1570428687), -INT32_C(  1838317663),  INT32_C(   878377803) },
      { -INT32_C(  1133323758), -INT32_C(   349060980),  INT32_C(  1473820975),  INT32_C(   773509652),  INT32_C(   851704129), -INT32_C(  1009506270), -INT32_C(   212516440),  INT32_C(  1328001085),
         INT32_C(   369858954), -INT32_C(  1996407462),  INT32_C(   199285494), -INT32_C(   314967124),  INT32_C(   371195124), -INT32_C(   841288923),  INT32_C(  1925197621),  INT32_C(  1774315743) },
      {  INT32_C(           0),  INT32_C(   960844035),  INT32_C(           0),  INT32_C(           0),  INT32_C(   954826548),  INT32_C(  1160120811),  INT32_C(           0),  INT32_C(   420600222),
         INT32_C(           0), -INT32_C(  1445643574),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   198029905), -INT32_C(  1708294747) } },
    { UINT16_C(52353),
      { -INT32_C(  2130060416),  INT32_C(  1163591780),  INT32_C(  1145047051),  INT32_C(   356594933),  INT32_C(   621307467), -INT32_C(   212583117),  INT32_C(  1893414063),  INT32_C(   490560925),
        -INT32_C(  1650506184),  INT32_C(  1373829702), -INT32_C(   124378622), -INT32_C(  1592928426),  INT32_C(  1892029757),  INT32_C(    40049491), -INT32_C(   327991473), -INT32_C(   737496932) },
      {  INT32_C(  1030859254), -INT32_C(  1500621917), -INT32_C(   862051211),  INT32_C(   946711547),  INT32_C(   329790656), -INT32_C(  1642722225), -INT32_C(   410351285),  INT32_C(   800822329),
        -INT32_C(   513004227), -INT32_C(   158860416),  INT32_C(   465708319), -INT32_C(  1873596464), -INT32_C(  1297810333),  INT32_C(  1397799432),  INT32_C(  2084232003), -INT32_C(  1381239441) },
      { -INT32_C(   534756746),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(   976911035),
         INT32_C(           0),  INT32_C(           0),  INT32_C(  2085294337), -INT32_C(   682188531),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1671035519),  INT32_C(  1481533957) } },
    { UINT16_C( 6178),
      {  INT32_C(   353608590),  INT32_C(  1530540697), -INT32_C(  1584788915),  INT32_C(  1050537626), -INT32_C(   235362912),  INT32_C(   852245496),  INT32_C(  1663515575),  INT32_C(  1987791592),
        -INT32_C(  2037674259),  INT32_C(   249677248),  INT32_C(  1789881295), -INT32_C(    89633703), -INT32_C(   370433551), -INT32_C(  1810122788), -INT32_C(   604552462),  INT32_C(  2035380876) },
      { -INT32_C(  1056973568),  INT32_C(  1926160546), -INT32_C(  1512276660), -INT32_C(  1163950903),  INT32_C(    44272165),  INT32_C(   882294337), -INT32_C(  1928360703), -INT32_C(    16293633),
        -INT32_C(   524351939),  INT32_C(   861048551), -INT32_C(   707252725), -INT32_C(   661686094),  INT32_C(  1155150594), -INT32_C(   226987791), -INT32_C(    58685443),  INT32_C(   637241319) },
      {  INT32_C(           0),  INT32_C(  1827195493),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1704490992),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(   828893857), -INT32_C(  1481734201),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(47757),
      {  INT32_C(  1464366085), -INT32_C(  2138745689), -INT32_C(  1191692503), -INT32_C(   370410993), -INT32_C(  1235624642),  INT32_C(  1329419983),  INT32_C(  1305879890),  INT32_C(  1325950026),
        -INT32_C(  2136583976), -INT32_C(   855626844),  INT32_C(  1921448034),  INT32_C(   828076530),  INT32_C(   518501710),  INT32_C(  1584211211), -INT32_C(  1800715190), -INT32_C(  2132560984) },
      { -INT32_C(  1476359932),  INT32_C(   376832180), -INT32_C(   343343880), -INT32_C(  1138957203), -INT32_C(  1529216103),  INT32_C(  1912751912),  INT32_C(   856075915),  INT32_C(  1706355041),
         INT32_C(   638497906), -INT32_C(  1388477259), -INT32_C(   308755072),  INT32_C(  1118418089), -INT32_C(   521698376),  INT32_C(  1448339915), -INT32_C(   108373609), -INT32_C(  1252114877) },
      {  INT32_C(  1955020885),  INT32_C(           0),  INT32_C(   699988023),  INT32_C(  2135031101),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1643067244),
         INT32_C(           0),  INT32_C(  1956225029),  INT32_C(           0), -INT32_C(  1226578846),  INT32_C(  1310648245),  INT32_C(  1764252403),  INT32_C(           0),  INT32_C(   119381316) } },
    { UINT16_C(27890),
      {  INT32_C(   435202012), -INT32_C(   304189355),  INT32_C(   128092254), -INT32_C(  1316267318), -INT32_C(  1919199686),  INT32_C(   921121452), -INT32_C(  1972098517), -INT32_C(  1141479457),
         INT32_C(  1674897166), -INT32_C(  1252936873),  INT32_C(    96334651), -INT32_C(  2018031539),  INT32_C(  1242911390), -INT32_C(  1333658492),  INT32_C(    87749926),  INT32_C(  1807822940) },
      {  INT32_C(  1875809559), -INT32_C(  2094784696),  INT32_C(  1602871570), -INT32_C(   941146071),  INT32_C(   370343058),  INT32_C(   516330488), -INT32_C(   450690936),  INT32_C(  1213260849),
        -INT32_C(  1028186502),  INT32_C(  1346821182), -INT32_C(   407842883), -INT32_C(  1582393585), -INT32_C(  1950891885), -INT32_C(   592871853), -INT32_C(  1346253698),  INT32_C(   737677745) },
      {  INT32_C(           0), -INT32_C(   563063315),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1491744365), -INT32_C(  1405688014),  INT32_C(  1948920714), -INT32_C(   809535508),
         INT32_C(           0),  INT32_C(           0),  INT32_C(  1622654567), -INT32_C(  1540963365),  INT32_C(           0), -INT32_C(   467303409), -INT32_C(  2125546167),  INT32_C(           0) } },
    { UINT16_C(44847),
      {  INT32_C(   864775661),  INT32_C(  1845643710), -INT32_C(   570093264),  INT32_C(  1788844210), -INT32_C(   857148893), -INT32_C(  1885837618),  INT32_C(   245385494),  INT32_C(  1656606836),
        -INT32_C(    74102467), -INT32_C(  1033267054),  INT32_C(  1554017961),  INT32_C(   700858118),  INT32_C(    16100914),  INT32_C(   714116628),  INT32_C(  1278750935),  INT32_C(  1051653632) },
      { -INT32_C(   784776385), -INT32_C(  2053921829),  INT32_C(   417411857), -INT32_C(  1522423949),  INT32_C(  1772435285), -INT32_C(  1668008507),  INT32_C(  1726532710),  INT32_C(    27563714),
        -INT32_C(  1244471590), -INT32_C(  1841666687),  INT32_C(   212474777),  INT32_C(   397536450), -INT32_C(   394176989), -INT32_C(   226224756), -INT32_C(  1537708575), -INT32_C(   576324349) },
      { -INT32_C(  1715095818), -INT32_C(   210759091),  INT32_C(   576830474), -INT32_C(   980200196),  INT32_C(           0), -INT32_C(   217261615),  INT32_C(           0),  INT32_C(           0),
        -INT32_C(   135375580), -INT32_C(  2066534107),  INT32_C(  1387872477), -INT32_C(  1491534824),  INT32_C(           0),  INT32_C(   148980393),  INT32_C(           0), -INT32_C(   176705535) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_rolv_epi32(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512i a = simde_test_x86_random_i32x16();
    simde__m512i b = simde_test_x86_random_i32x16();
    simde__m512i r = simde_mm512_maskz_rolv_epi32(k, a, b);

    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm_rolv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t a[2];
    const int64_t b[2];
    const int64_t r[2];
  } test_vec[] = {
    { {  INT64_C( 5057515482088796707),  INT64_C(  571657017102954936) },
      {  INT64_C( 6632879945960180690), -INT64_C( 2126296340786075475) },
      { -INT64_C( 7051528967369713473), -INT64_C(  128632974007539784) } },
    { {  INT64_C(  982079754496470482),  INT64_C( 2957801001677646078) },
      {  INT64_C( 3930320351409590275),  INT64_C( 8761212831322785525) },
      {  INT64_C( 7856638035971763856),  INT64_C( 2289272849474562381) } },
    { {  INT64_C( 9143280864712749953),  INT64_C( 7664439809984900829) },
      { -INT64_C( 2296795214303879584),  INT64_C( 5478957471170344682) },
      { -INT64_C( 2966619254513502848),  INT64_C(  917456319793699026) } },
    { {  INT64_C( 3430623965912161694),  INT64_C( 1010144072584863540) },
      { -INT64_C( 6113046325753242274), -INT64_C( 8654434239321948466) },
      { -INT64_C( 8680486626399944300),  INT64_C( 3471051112936440705) } },
    { {  INT64_C( 4730258509686703907),  INT64_C( 2818656095946147936) },
      { -INT64_C(  293970732022426242), -INT64_C( 8085798150086580391) },
      { -INT64_C( 3429121391005711928), -INT64_C( 7917970187762123839) } },
    { {  INT64_C( 1139598376703454849),  INT64_C(  766521836130184370) },
      {  INT64_C(  610296777346258000),  INT64_C(  529195525334568284) },
      { -INT64_C( 6147538812357505072), -INT64_C( 4057203631136820296) } },
    { { -INT64_C( 5097399647130460797), -INT64_C( 1465402040846557166) },
      { -INT64_C( 8889014774488566804), -INT64_C( 8756224715748741395) },
      {  INT64_C( 7338681100285276969), -INT64_C( 6052172191148837946) } },
    { { -INT64_C( 8519972155877224191), -INT64_C( 4908692902732302502) },
      {  INT64_C( 1617066224680061124),  INT64_C( 7093457905718574312) },
      { -INT64_C( 7192345978068725736),  INT64_C( 2951927843741880309) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i a = simde_x_mm_loadu_epi64(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi64(test_vec[i].b);
    simde__m128i r = simde_mm_rolv_epi64(a, b);
    simde_test_x86_assert_equal_i64x2(r, simde_x_mm_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m128i a = simde_test_x86_random_i64x2();
    simde__m128i b = simde_test_x86_random_i64x2();
    simde__m128i r = simde_mm_rolv_epi64(a, b);

    simde_test_x86_write_i64x2(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x2(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm_mask_rolv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t src[2];
    const simde__mmask8 k;
    const int64_t a[2];
    const int64_t b[2];
    const int64_t r[2];
  } test_vec[] = {
    { {  INT64_C( 9086061883268213015), -INT64_C( 3049407461607587804) },
      UINT8_C(157),
      { -INT64_C( 1232336815746793212),  INT64_C( 7747512153549033575) },
      {  INT64_C( 7812915412421514012),  INT64_C( 1329497638197682305) },
      {  INT64_C( 2295828727351631299), -INT64_C( 3049407461607587804) } },
    { {  INT64_C( 8950047831210604821),  INT64_C( 5868874438462243782) },
      UINT8_C(129),
      {  INT64_C( 7606608392553863937),  INT64_C( 1297189361245709976) },
      { -INT64_C(   47951370427807846), -INT64_C( 3351375419357766258) },
      {  INT64_C( 2323495032837259368),  INT64_C( 5868874438462243782) } },
    { { -INT64_C( 2480797002370110688),  INT64_C( 3297700988328610263) },
      UINT8_C( 16),
      {  INT64_C( 8006707524856954347), -INT64_C( 4439405090809374274) },
      {  INT64_C( 2970093302680358384), -INT64_C( 7600410773982222164) },
      { -INT64_C( 2480797002370110688),  INT64_C( 3297700988328610263) } },
    { {  INT64_C( 6496798273563857319),  INT64_C( 7544939469235710044) },
      UINT8_C(144),
      { -INT64_C( 1707471186496075588),  INT64_C( 6423867309845365397) },
      {  INT64_C( 8137527375420514126),  INT64_C( 6016616240685572588) },
      {  INT64_C( 6496798273563857319),  INT64_C( 7544939469235710044) } },
    { { -INT64_C( 7205406430060073447),  INT64_C(  838867711862477827) },
      UINT8_C( 32),
      {  INT64_C( 7842061677316644249),  INT64_C( 8136693096277853058) },
      { -INT64_C( 7381660349684579253),  INT64_C( 1770022953024936750) },
      { -INT64_C( 7205406430060073447),  INT64_C(  838867711862477827) } },
    { { -INT64_C( 3101057842430297601), -INT64_C( 7185050850251244905) },
      UINT8_C(151),
      { -INT64_C( 7925430756068022079), -INT64_C( 4905218227064114548) },
      {  INT64_C( 7536178517811033898),  INT64_C( 8769447148664745609) },
      {  INT64_C( 8116337859974848338), -INT64_C( 2714538232327628425) } },
    { {  INT64_C( 1353138746086477693),  INT64_C( 3351221297697839555) },
      UINT8_C(101),
      {  INT64_C( 3529480439294274703),  INT64_C( 1857279417788143604) },
      { -INT64_C( 7223015401278407761), -INT64_C( 7492358063356213071) },
      { -INT64_C( 7617952579386787607),  INT64_C( 3351221297697839555) } },
    { {  INT64_C(  642611858914269433), -INT64_C( 1945809894241020901) },
      UINT8_C( 19),
      {  INT64_C( 3314773185350984862),  INT64_C( 9067100261058060303) },
      {  INT64_C( 5762807054986219481),  INT64_C( 8869955864090697452) },
      {  INT64_C( 5521320548968956129),  INT64_C( 8791299002478517683) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i src = simde_x_mm_loadu_epi64(test_vec[i].src);
    simde__m128i a = simde_x_mm_loadu_epi64(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi64(test_vec[i].b);
    simde__m128i r = simde_mm_mask_rolv_epi64(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x2(r, simde_x_mm_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m128i src = simde_test_x86_random_i64x2();
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m128i a = simde_test_x86_random_i64x2();
    simde__m128i b = simde_test_x86_random_i64x2();
    simde__m128i r = simde_mm_mask_rolv_epi64(src, k, a, b);

    simde_test_x86_write_i64x2(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x2(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x2(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm_maskz_rolv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask8 k;
    const int64_t a[2];
    const int64_t b[2];
    const int64_t r[2];
  } test_vec[] = {
    { UINT8_C( 77),
      {  INT64_C( 7740881591386491629), -INT64_C( 6052119792850024359) },
      { -INT64_C( 5213630320420069601), -INT64_C( 5786825623392179717) },
      { -INT64_C( 5802354346872630939),  INT64_C(                   0) } },
    { UINT8_C(218),
      { -INT64_C( 4458545511592030812),  INT64_C( 3084275576502962464) },
      { -INT64_C( 3159914868979300685), -INT64_C( 2433838006619698866) },
      {  INT64_C(                   0),  INT64_C( 7139027534075136691) } },
    { UINT8_C( 82),
      {  INT64_C( 1013185421964518878),  INT64_C( 1201761308506701166) },
      { -INT64_C( 8822814469400045690),  INT64_C(  377631833770159526) },
      {  INT64_C(                   0), -INT64_C( 7606760622383897922) } },
    { UINT8_C( 67),
      { -INT64_C( 4538196844831247351), -INT64_C( 8900981286356910225) },
      {  INT64_C( 3912790560372537751),  INT64_C( 9126370326712432838) },
      {  INT64_C(  129342673912889995),  INT64_C( 2186263958153845729) } },
    { UINT8_C( 85),
      {  INT64_C( 1960220444390397615),  INT64_C( 6438261391337465445) },
      { -INT64_C( 8747442592639024947),  INT64_C( 6119119370317528628) },
      { -INT64_C( 8988207754882194586),  INT64_C(                   0) } },
    { UINT8_C(144),
      {  INT64_C( 8341730069935036740), -INT64_C( 2402269221811419761) },
      {  INT64_C( 1520443264424770829), -INT64_C( 2469645394666410321) },
      {  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C( 39),
      { -INT64_C( 8881969281916191898), -INT64_C( 4795137057902913542) },
      {  INT64_C( 2364365005779130733),  INT64_C( 1790347645621565603) },
      { -INT64_C( 8688340231511874130),  INT64_C( 3680777720970740227) } },
    { UINT8_C(211),
      { -INT64_C( 7408889773796735668),  INT64_C( 7090239257450223032) },
      {  INT64_C( 5618509178199115115), -INT64_C( 3523758285624726866) },
      { -INT64_C( 7914402578774881882),  INT64_C( 7308806292385201718) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i a = simde_x_mm_loadu_epi64(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi64(test_vec[i].b);
    simde__m128i r = simde_mm_maskz_rolv_epi64(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x2(r, simde_x_mm_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m128i a = simde_test_x86_random_i64x2();
    simde__m128i b = simde_test_x86_random_i64x2();
    simde__m128i r = simde_mm_maskz_rolv_epi64(k, a, b);

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x2(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x2(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_rolv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t a[4];
    const int64_t b[4];
    const int64_t r[4];
  } test_vec[] = {
    { {  INT64_C( 6205232311378897248), -INT64_C( 5992885019796330250), -INT64_C( 6030299675446116550),  INT64_C( 1004750959582923402) },
      { -INT64_C( 3116421817156143743),  INT64_C( 4568382877079963623), -INT64_C( 1965342563017537369),  INT64_C( 9058841102555440637) },
      { -INT64_C( 6036279450951757120), -INT64_C( 6641547939380500033), -INT64_C( 1098986790690671983),  INT64_C( 4737279888375253329) } },
    { { -INT64_C(  171234544481478420), -INT64_C( 4327081712644721746),  INT64_C(  478939606089225574), -INT64_C( 8113120774993204242) },
      { -INT64_C( 6533603793700237505), -INT64_C( 8013508847851729940),  INT64_C( 1274789327587613345),  INT64_C(  456100271454183458) },
      {  INT64_C( 9137754764614036598), -INT64_C( 8125922821472666821), -INT64_C( 8961975465094213103), -INT64_C( 6142716677300576625) } },
    { { -INT64_C( 7831567716856385869), -INT64_C( 7605623238938613319), -INT64_C( 2578890529353710592),  INT64_C( 5196580665262420304) },
      { -INT64_C( 2170942659295479388), -INT64_C( 2272286047985173780), -INT64_C(  463245514423158533), -INT64_C( 3582844220658928791) },
      { -INT64_C( 6504628080847689094), -INT64_C( 5000234292691902919),  INT64_C(  495870423261120032),  INT64_C( 4783562295562021423) } },
    { {  INT64_C( 7615579813630851751), -INT64_C( 6542391421359721612), -INT64_C( 2802385277239478407),  INT64_C(    6789867809286944) },
      { -INT64_C( 6971714971393641895), -INT64_C( 4304835092398503438), -INT64_C( 6149663595469300385), -INT64_C(  538729226120691668) },
      { -INT64_C(  316320195384483854), -INT64_C( 3615663916754317015),  INT64_C( 1237816708321638072), -INT64_C( 9029154296449100627) } },
    { { -INT64_C( 4176791973433289410), -INT64_C( 8960575405276594794), -INT64_C( 5154656141353776319), -INT64_C( 5398754412938237317) },
      { -INT64_C( 8740314866350300646), -INT64_C(  158174938364700890), -INT64_C( 7504336067797442409), -INT64_C(  444231426109203799) },
      {  INT64_C( 5884860375242384439),  INT64_C( 6394941737674541368), -INT64_C( 4150725365035418755),  INT64_C( 8995086391067733558) } },
    { {  INT64_C( 4908250824115287463),  INT64_C( 2840747240380597427),  INT64_C( 2030883135373849725),  INT64_C( 5627267143863951679) },
      {  INT64_C( 5793883790269284358), -INT64_C( 3441546904377231482), -INT64_C( 1373992963974527137),  INT64_C( 4787083670077516410) },
      {  INT64_C(  533403490316020177), -INT64_C( 2659617352737280823), -INT64_C( 3736551459859754311), -INT64_C(  200304327028837500) } },
    { { -INT64_C( 2445508096858565341),  INT64_C( 6479047059406376646),  INT64_C( 7373266617564353465), -INT64_C( 6574139994963222972) },
      { -INT64_C( 4575248230406477237),  INT64_C( 7031461140774942017),  INT64_C( 2191593688059592472),  INT64_C( 3719236516231180664) },
      {  INT64_C( 9113805682656222960), -INT64_C( 5488649954896798324), -INT64_C( 1412393794541300966),  INT64_C( 4946293754261702494) } },
    { { -INT64_C( 5384801621478342261), -INT64_C(  300187687695933492),  INT64_C( 3403430178611764292), -INT64_C( 4074132330071336358) },
      {  INT64_C( 7994330939524926407), -INT64_C( 6156500514396511824),  INT64_C( 5925052623643709012), -INT64_C( 1018084914356788259) },
      { -INT64_C( 6725076821974399526),  INT64_C( 2291483204900844784), -INT64_C( 7245764061629844554),  INT64_C( 7513476748138166471) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi64(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi64(test_vec[i].b);
    simde__m256i r = simde_mm256_rolv_epi64(a, b);
    simde_test_x86_assert_equal_i64x4(r, simde_x_mm256_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m256i a = simde_test_x86_random_i64x4();
    simde__m256i b = simde_test_x86_random_i64x4();
    simde__m256i r = simde_mm256_rolv_epi64(a, b);

    simde_test_x86_write_i64x4(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_mask_rolv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t src[4];
    const simde__mmask8 k;
    const int64_t a[4];
    const int64_t b[4];
    const int64_t r[4];
  } test_vec[] = {
    { {  INT64_C( 7810682925676833620),  INT64_C( 5985160756872628223), -INT64_C( 5230762904762001635),  INT64_C( 6491849828833688828) },
      UINT8_C(218),
      { -INT64_C(  188584152523124412), -INT64_C( 5859095327906412797),  INT64_C( 3045065761745323437), -INT64_C( 4280519164576110578) },
      {  INT64_C( 3931510750482673627), -INT64_C( 8586824231851357360), -INT64_C( 6154995051722285762),  INT64_C( 3151394187017504120) },
      {  INT64_C( 7810682925676833620),  INT64_C( 5753228663357419184), -INT64_C( 5230762904762001635),  INT64_C( 1064143132582293608) } },
    { { -INT64_C( 4929387243048672721),  INT64_C( 1602076233802342615), -INT64_C( 7900977192870213356), -INT64_C( 1056016312816712995) },
      UINT8_C( 41),
      {  INT64_C( 5348393900557226773),  INT64_C( 3529759047217834801), -INT64_C( 3913386154312160069),  INT64_C( 5464692867359038823) },
      { -INT64_C( 1583607108087289136),  INT64_C( 2100874671356520657), -INT64_C( 5383314145520094854),  INT64_C( 2521234393740268925) },
      {  INT64_C( 5758522363223558713),  INT64_C( 1602076233802342615), -INT64_C( 7900977192870213356), -INT64_C( 1622756400793814100) } },
    { {  INT64_C( 6937762856370547363),  INT64_C( 7172123042797591050), -INT64_C( 7641431986054858511),  INT64_C( 3708790775069824515) },
      UINT8_C(239),
      { -INT64_C( 1062321921640656811), -INT64_C( 5861367614095577801),  INT64_C( 3554005152954771233),  INT64_C( 8564144592914589860) },
      {  INT64_C( 8020329532457113732), -INT64_C( 8826828337228404836),  INT64_C( 5098398732816500405), -INT64_C(  403624710162578737) },
      {  INT64_C( 1449593327459042655),  INT64_C( 6592613254817285075), -INT64_C( 2006870079728650020), -INT64_C(  427572718128186516) } },
    { { -INT64_C( 6915379462673132447), -INT64_C( 5713956341573079606),  INT64_C( 8944714755642263353), -INT64_C( 4830722633271324028) },
      UINT8_C(231),
      { -INT64_C( 8522907964025383861),  INT64_C( 2317723893533894398),  INT64_C( 8789982106111313270),  INT64_C( 4067191288403130295) },
      { -INT64_C( 1744335712739172891), -INT64_C(  324256401308758074),  INT64_C( 5740241899482205704), -INT64_C(  897259700828406425) },
      { -INT64_C( 2492028324447941731),  INT64_C(  760376596492828552), -INT64_C(  267357828069099911), -INT64_C( 4830722633271324028) } },
    { {  INT64_C( 6447182776330662824),  INT64_C( 6615286835339350350),  INT64_C( 1688344841358623110),  INT64_C( 4301773907547605169) },
      UINT8_C(171),
      { -INT64_C( 7695641016086377242),  INT64_C( 1517507279067874896), -INT64_C( 1966422000129462577),  INT64_C( 1003372698651817032) },
      {  INT64_C( 1321848009074551422),  INT64_C( 6130711390037098795), -INT64_C(   47271614993540393),  INT64_C(  123616454472945794) },
      { -INT64_C( 6535596272448982215),  INT64_C( 5202361980658558023),  INT64_C( 1688344841358623110),  INT64_C( 4013490794607268128) } },
    { { -INT64_C(  420491939406370474),  INT64_C( 5060091525778294110),  INT64_C( 2566790793708344040),  INT64_C( 7018441154279301090) },
      UINT8_C( 45),
      {  INT64_C( 7612480455589001770),  INT64_C( 6117892441670977776),  INT64_C(  418016895845107672), -INT64_C( 4425506159426784540) },
      {  INT64_C( 8717189351500976619), -INT64_C( 4085786377189863115),  INT64_C( 4400156856366023141),  INT64_C( 9010359879460004017) },
      { -INT64_C( 2030750366123613641),  INT64_C( 5060091525778294110),  INT64_C( 6823229415999341789),  INT64_C( 7334539887396311072) } },
    { { -INT64_C(  663427997498534965), -INT64_C( 8750712188229041448),  INT64_C( 8147472149956870522), -INT64_C( 4840589009170549054) },
      UINT8_C(217),
      {  INT64_C( 2071532523948150751),  INT64_C( 6354934314030696317), -INT64_C( 3730798991519513388),  INT64_C( 4603656693211827524) },
      { -INT64_C( 8538574151640318839),  INT64_C( 6114323141352177003), -INT64_C( 4515416643441266059),  INT64_C( 1598969792386765713) },
      {  INT64_C( 9160240060008742457), -INT64_C( 8750712188229041448),  INT64_C( 8147472149956870522), -INT64_C(  955302452485652538) } },
    { { -INT64_C( 6503205308482366166), -INT64_C( 2014514829009092810),  INT64_C( 7724172377897031412), -INT64_C( 3063418983029080871) },
      UINT8_C(226),
      { -INT64_C( 3880482422275699723), -INT64_C( 4765337374295466775),  INT64_C( 9019305428904267661),  INT64_C(  274152289049481567) },
      { -INT64_C( 7738002557989885688), -INT64_C( 9108708415114389773), -INT64_C( 8022600674513244500), -INT64_C( 3053077924904447599) },
      { -INT64_C( 6503205308482366166), -INT64_C( 1779503559033794098),  INT64_C( 7724172377897031412), -INT64_C( 3063418983029080871) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i src = simde_x_mm256_loadu_epi64(test_vec[i].src);
    simde__m256i a = simde_x_mm256_loadu_epi64(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi64(test_vec[i].b);
    simde__m256i r = simde_mm256_mask_rolv_epi64(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x4(r, simde_x_mm256_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m256i src = simde_test_x86_random_i64x4();
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m256i a = simde_test_x86_random_i64x4();
    simde__m256i b = simde_test_x86_random_i64x4();
    simde__m256i r = simde_mm256_mask_rolv_epi64(src, k, a, b);

    simde_test_x86_write_i64x4(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x4(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_maskz_rolv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask8 k;
    const int64_t a[4];
    const int64_t b[4];
    const int64_t r[4];
  } test_vec[] = {
    { UINT8_C( 50),
      {  INT64_C( 2062210223432378515),  INT64_C( 4609876292300470768), -INT64_C( 6848285292610181786), -INT64_C( 3821511856297765509) },
      {  INT64_C( 3912804843453273323),  INT64_C( 6186957356681878924), -INT64_C( 4102584318667567967), -INT64_C( 6032708308781203898) },
      {  INT64_C(                   0), -INT64_C( 7412638215852588033),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C( 87),
      { -INT64_C( 8620783405868655172),  INT64_C( 2410778018792864086),  INT64_C( 8322710339215811720),  INT64_C(   43875205180095936) },
      {  INT64_C( 7093216747554027710),  INT64_C( 4403018184743315492),  INT64_C( 9004374319500265357), -INT64_C( 4019644042961737229) },
      {  INT64_C( 2456490166960224111),  INT64_C( 2846157738051689821),  INT64_C(  477002425426841200),  INT64_C(                   0) } },
    { UINT8_C(147),
      {  INT64_C(  673009010921680903), -INT64_C(  977849103318888393),  INT64_C( 1106526397753389169), -INT64_C( 3325378590960675861) },
      { -INT64_C( 5567755927487743599), -INT64_C( 1870683296135743061), -INT64_C( 3610574209537886225),  INT64_C( 8059453071036775740) },
      {  INT64_C(  306919047483495086), -INT64_C(  972285404577571067),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C(102),
      {  INT64_C( 6333068079347606236), -INT64_C( 5038207460355344091), -INT64_C( 3826534876834895361), -INT64_C( 6786898501781885827) },
      {  INT64_C( 7597216622064546105), -INT64_C( 1552999788192056907), -INT64_C(  710872233382048117),  INT64_C( 2003030777421417061) },
      {  INT64_C(                   0),  INT64_C( 2645656518657350015),  INT64_C( 3122803568693739095),  INT64_C(                   0) } },
    { UINT8_C(226),
      { -INT64_C( 2803894699492429147),  INT64_C( 5713853244896497182), -INT64_C( 8133010795097775294), -INT64_C( 4885970088975316194) },
      {  INT64_C(  519942458895587518), -INT64_C( 1327977313823294462),  INT64_C( 3822652385955240375), -INT64_C( 4041500139565318891) },
      {  INT64_C(                   0),  INT64_C( 4408668905876437113),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C( 14),
      {  INT64_C( 6358517921013507335),  INT64_C( 7823299090988630214), -INT64_C( 7603844076638368358), -INT64_C( 7494158482206120388) },
      { -INT64_C( 4953942041260001653), -INT64_C( 3526839446381706096), -INT64_C( 8798718341147291340),  INT64_C( 8465209200806953811) },
      {  INT64_C(                   0), -INT64_C( 1075557652407882607), -INT64_C( 2762562978823387088),  INT64_C( 3785149030919880699) } },
    { UINT8_C(109),
      { -INT64_C( 2095548635635622580),  INT64_C( 3318562061773409971), -INT64_C( 4460536980572775655), -INT64_C( 4515102211571909772) },
      { -INT64_C( 8431010257414801025),  INT64_C( 1540637875545249430),  INT64_C( 5318174831346964684),  INT64_C( 1956404591951379183) },
      {  INT64_C( 8175597719036964518),  INT64_C(                   0), -INT64_C( 8082839453632979935),  INT64_C( 9059659890401408342) } },
    { UINT8_C( 69),
      {  INT64_C( 5756346034441435459),  INT64_C( 5640302744813901065), -INT64_C( 7413367955902934724), -INT64_C( 1408975045875367007) },
      {  INT64_C( 4400326025305099913), -INT64_C(  435724648458986450),  INT64_C( 2740519007381165904),  INT64_C( 1684281996975750602) },
      { -INT64_C( 4229882159513303393),  INT64_C(                   0),  INT64_C( 7863055307440429342),  INT64_C(                   0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi64(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi64(test_vec[i].b);
    simde__m256i r = simde_mm256_maskz_rolv_epi64(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x4(r, simde_x_mm256_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m256i a = simde_test_x86_random_i64x4();
    simde__m256i b = simde_test_x86_random_i64x4();
    simde__m256i r = simde_mm256_maskz_rolv_epi64(k, a, b);

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x4(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_rolv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t a[8];
    const int64_t b[8];
    const int64_t r[8];
  } test_vec[] = {
    { {  INT64_C( 1149393879454812352),  INT64_C(  220153112429291259), -INT64_C( 3739642002311641347),  INT64_C( 2351198276940937382),
        -INT64_C(  261536427234063702),  INT64_C(  745039031827163453),  INT64_C( 3577746561953393520),  INT64_C(  463588583565834733) },
      {  INT64_C( 7315731631410860425), -INT64_C( 4921574249653050980), -INT64_C( 1138221021198528903),  INT64_C( 1933922832266203764),
         INT64_C( 8829918252527528957), -INT64_C( 3921836682791140724), -INT64_C( 3091719627398633692), -INT64_C(   53906846865074540) },
      { -INT64_C( 1806144077841727457),  INT64_C( 7919780535544635969), -INT64_C(  317446329294771443), -INT64_C( 8475200475694442189),
         INT64_C( 6884836974236823893),  INT64_C( 7967102201985487013), -INT64_C( 7370502516189210451), -INT64_C(  737229265383168273) } },
    { { -INT64_C( 7205566922582793049),  INT64_C( 2237170786822726898),  INT64_C( 7954587134626027821),  INT64_C( 3294053848226990516),
         INT64_C( 8963486567028218808),  INT64_C(  788635563350546391),  INT64_C( 5399250351097984932),  INT64_C(  989023171233546502) },
      { -INT64_C(  605389869114102638), -INT64_C( 4036990927468144623), -INT64_C( 7650997224917855503),  INT64_C( 2024045368926393267),
         INT64_C( 3917517209441200966), -INT64_C( 3381326046037793239),  INT64_C( 4599348996498610607),  INT64_C( 1631532115235960318) },
      { -INT64_C( 4882437906744053758),  INT64_C( 1005574741427502616), -INT64_C( 5594916832320086899),  INT64_C( 5593872843376800928),
         INT64_C( 1814074004809903647),  INT64_C( 9200763972777130917), -INT64_C( 7218666059710382260), -INT64_C( 8976116244046389183) } },
    { { -INT64_C( 2907960116173284279), -INT64_C( 7660280026454767361),  INT64_C( 3700736464885385872),  INT64_C( 5297463517140198045),
        -INT64_C( 2755820501191950277), -INT64_C( 5935909115196218077), -INT64_C( 5126182574623198348),  INT64_C( 3535232209372223430) },
      {  INT64_C( 4998674672936273578), -INT64_C(  753528845611690047),  INT64_C( 4482342182734156640), -INT64_C( 5892938104884143599),
        -INT64_C( 1134465995839941670), -INT64_C( 2513474247430598744),  INT64_C( 5143547513094651052),  INT64_C( 6096051433114592999) },
      { -INT64_C(  904898762971366983),  INT64_C( 3126184020800016895),  INT64_C( 2077296942577854517), -INT64_C( 4755559901194185976),
         INT64_C( 7757129304766874991), -INT64_C( 5051592178415297527), -INT64_C( 7550483178041787748),  INT64_C( 1390454604471565155) } },
    { {  INT64_C(  430816543141579994), -INT64_C( 3712289764860311519),  INT64_C( 5225672420742976520),  INT64_C( 3784938324107711415),
        -INT64_C( 8503861072418187623), -INT64_C( 9156765309826814977), -INT64_C( 1936191396797031305), -INT64_C( 4925357850502933481) },
      {  INT64_C( 1901836373035508595), -INT64_C( 6498370806945772583), -INT64_C( 1506842395403251965),  INT64_C( 4525839217478872563),
         INT64_C(  655693969662509541),  INT64_C( 9069171553532007177), -INT64_C( 1478934430789772613), -INT64_C( 4757261526589450510) },
      {  INT64_C( 7408473976934517382),  INT64_C( 4700734985258727064),  INT64_C( 4911891218524708930), -INT64_C( 7081448385436087906),
        -INT64_C( 1748853296807351010), -INT64_C( 2790843909103157503), -INT64_C( 4672191999577295133),  INT64_C( 5791328500812465245) } },
    { { -INT64_C( 7238737526314494895), -INT64_C( 3809852798752529111), -INT64_C( 8796141868831551991),  INT64_C( 1026324549702118825),
        -INT64_C(  397132802898789552),  INT64_C(  710088798735585534), -INT64_C( 5645832744546388851),  INT64_C( 8264511184782485222) },
      {  INT64_C( 6357062159051276396),  INT64_C( 2390358706226714146),  INT64_C( 1671088557705475958),  INT64_C( 4711447821993796557),
        -INT64_C( 6097954675095587156),  INT64_C( 9170165604601939596),  INT64_C( 3996994604225102897),  INT64_C( 5451195716668437392) },
      { -INT64_C( 4520178370022397925), -INT64_C( 8120108157398500286), -INT64_C( 9051818034053736781), -INT64_C( 4064586451798122041),
        -INT64_C(   21093409802609112), -INT64_C( 6061844025150807907),  INT64_C( 7573756376188469270),  INT64_C( 8152257718806540977) } },
    { {  INT64_C( 7454497365745083455),  INT64_C( 7782162133244461175), -INT64_C( 4976120063459525553), -INT64_C( 2278041165827729288),
        -INT64_C( 2509600768430394134), -INT64_C( 3530438925533465252), -INT64_C( 8485493002489770664),  INT64_C(  870043697937472657) },
      {  INT64_C( 3745866615465556920),  INT64_C(  865195478726826162), -INT64_C( 7695301547150833001), -INT64_C( 7901887865082620376),
        -INT64_C( 2909137182038939754), -INT64_C( 3835928665870669565),  INT64_C( 5129222978805191573),  INT64_C( 7416277192949298559) },
      {  INT64_C( 4568747554724401700),  INT64_C(  134457074396037665), -INT64_C( 7684013434691815262),  INT64_C( 2240673719741948183),
        -INT64_C( 2778322752619721978),  INT64_C( 8649976743151381222),  INT64_C( 2472565366112470959), -INT64_C( 8788350187886039480) } },
    { {  INT64_C( 5407824002248055824), -INT64_C( 5198906445239544100),  INT64_C( 8538964493761269273),  INT64_C(  768632270479102580),
        -INT64_C( 3681326624067846826), -INT64_C( 3801965306581357726),  INT64_C( 9182703352189062146),  INT64_C( 5736299469029457558) },
      { -INT64_C( 8527893382009333014),  INT64_C( 4053038683585872322),  INT64_C( 6728555723140187094),  INT64_C( 8802436726903415775),
        -INT64_C( 5903098792712596125),  INT64_C( 4677095323873714299), -INT64_C( 7292874004600120667), -INT64_C( 6580151301186537308) },
      {  INT64_C( 5652089190419864791), -INT64_C( 2348881707248624782), -INT64_C( 4769852881247559649),  INT64_C( 2479101002181336445),
        -INT64_C( 1009955622276667516),  INT64_C( 1610570841079603035),  INT64_C( 2346797961619472767),  INT64_C(  879302972162891673) } },
    { {  INT64_C( 8595755391997838000),  INT64_C( 8290019494456683618),  INT64_C( 5944656601798356266),  INT64_C( 4487613336619774561),
         INT64_C( 1849050590774719798),  INT64_C(  779951631472382097), -INT64_C( 4819292111168884194),  INT64_C( 1021656141142140001) },
      { -INT64_C( 4689116733267507428), -INT64_C( 2259416394433179207),  INT64_C( 7258335867950634146), -INT64_C( 2570287924019783058),
         INT64_C( 3923369189597746312),  INT64_C( 2454028040365183084), -INT64_C(  977040858465453559), -INT64_C( 3706026791639923468) },
      { -INT64_C( 3908315051190868961), -INT64_C( 4258689864975233320), -INT64_C( 7530945837686413637), -INT64_C( 4857044710488577887),
        -INT64_C( 6258394678120073703),  INT64_C( 6703907856323577342),  INT64_C( 4386144958611209594),  INT64_C( 5048784610051159546) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_rolv_epi64(a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i a = simde_test_x86_random_i64x8();
    simde__m512i b = simde_test_x86_random_i64x8();
    simde__m512i r = simde_mm512_rolv_epi64(a, b);

    simde_test_x86_write_i64x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_mask_rolv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t src[8];
    const simde__mmask8 k;
    const int64_t a[8];
    const int64_t b[8];
    const int64_t r[8];
  } test_vec[] = {
    { {  INT64_C( 5248087594654318624),  INT64_C( 2460519113906693251),  INT64_C( 1420779166242850112), -INT64_C( 7173011014080270493),
         INT64_C( 7835435318256530841), -INT64_C(  365189890244135758),  INT64_C( 2237292569400581594), -INT64_C( 4216149711708040347) },
      UINT8_C( 47),
      {  INT64_C( 3204457334572595895),  INT64_C( 5720380984363730163),  INT64_C( 4285503142626024350), -INT64_C( 8116844600747402511),
        -INT64_C( 1546479634230250650), -INT64_C( 1255706809672175856), -INT64_C( 5056561855132950991), -INT64_C( 3730317056614778155) },
      {  INT64_C( 7896133257260601523),  INT64_C( 1035901433174558499), -INT64_C( 5592772694788119840),  INT64_C( 4244531078890473396),
        -INT64_C( 1660568473297481430), -INT64_C( 4076958439570758528),  INT64_C( 4660612692354011677), -INT64_C( 3324708037660570598) },
      { -INT64_C( 3046293978807916351),  INT64_C( 5130206469971968404),  INT64_C( 1444348840663132775), -INT64_C( 5829639569331276171),
         INT64_C( 7835435318256530841), -INT64_C( 1255706809672175856),  INT64_C( 2237292569400581594), -INT64_C( 4216149711708040347) } },
    { { -INT64_C( 5043666299556151212),  INT64_C( 3791612074708655344), -INT64_C(  404798536787489131),  INT64_C( 6080554390725115369),
        -INT64_C(  636011369590769096),  INT64_C( 6563834583380280267),  INT64_C(  960918338953561722), -INT64_C( 6705860491083910757) },
      UINT8_C(186),
      { -INT64_C( 8964212469811571619), -INT64_C( 2851159463513392249),  INT64_C( 1196560150467234082), -INT64_C( 8980147808456067418),
         INT64_C( 6986183467554840879),  INT64_C(  346428501388442979),  INT64_C( 8771217480703330955),  INT64_C( 9114862673041765333) },
      {  INT64_C( 1935106064260124320),  INT64_C( 8864453989434863133),  INT64_C( 9148181558281439655),  INT64_C( 3737696953114891025),
        -INT64_C( 3921077626165408827),  INT64_C( 3036216159357216182),  INT64_C( 4034548132311484375),  INT64_C( 3635096961320187564) },
      { -INT64_C( 5043666299556151212),  INT64_C( 6803056058786763954), -INT64_C(  404798536787489131),  INT64_C( 3912305305400968896),
         INT64_C( 2196942077240288748),  INT64_C( 6395449779949491471),  INT64_C(  960918338953561722), -INT64_C(  919482095373062223) } },
    { {  INT64_C( 6824716678147743809),  INT64_C( 2708115068201304296),  INT64_C( 9137138244342004079),  INT64_C( 1997399050348749848),
        -INT64_C( 7901918854676715953), -INT64_C(  886766557911491778),  INT64_C( 1160897330254730956),  INT64_C( 6390413294533269946) },
      UINT8_C(181),
      {  INT64_C(  122989882298552247),  INT64_C( 5210233702369738991), -INT64_C( 2229475293804827648),  INT64_C(  714736787232760807),
         INT64_C( 3186071599388248977),  INT64_C( 3723909764584750195),  INT64_C( 2725126645626826216), -INT64_C( 8243761067367613636) },
      { -INT64_C( 4756932282671687791), -INT64_C( 8205892055694333513),  INT64_C( 8944518574978869182),  INT64_C( 6094956624958094347),
         INT64_C(  919048049922300345),  INT64_C( 8285240903673046215), -INT64_C( 2992202371117974851), -INT64_C( 2296145149977263093) },
      { -INT64_C( 1924467786307992727),  INT64_C( 2708115068201304296),  INT64_C( 4054317194976180992),  INT64_C( 1997399050348749848),
         INT64_C( 2474849381659770519), -INT64_C( 2954896049600317031),  INT64_C( 1160897330254730956), -INT64_C( 4451838524632996756) } },
    { {  INT64_C( 2967771420859516109), -INT64_C(  382927755385887096), -INT64_C( 5067269014255362548), -INT64_C( 3604101258852987659),
         INT64_C(  898502774502776813), -INT64_C( 6017661633809543611), -INT64_C( 2775008036282731754), -INT64_C( 7310766416092396356) },
      UINT8_C(207),
      {  INT64_C( 8205100448199469690),  INT64_C( 5816326199048233128), -INT64_C(  275547542002566336), -INT64_C( 1314132500973754109),
         INT64_C( 2090063257686906413),  INT64_C( 1149109290070250181),  INT64_C( 4027251835021502191), -INT64_C(  146385885591794538) },
      { -INT64_C( 6298689925477991164), -INT64_C( 5224671686144928864),  INT64_C( 4669813532829679202), -INT64_C( 1451212515780658929),
        -INT64_C( 7401108566840153410),  INT64_C( 3588673801581794126), -INT64_C(  112422937129620347), -INT64_C( 5088018664282801520) },
      {  INT64_C( 2154398655224653735), -INT64_C( 6912637677012009508), -INT64_C( 4552597147115832351), -INT64_C( 6793123869881141535),
         INT64_C(  898502774502776813), -INT64_C( 6017661633809543611), -INT64_C(  255149795278791194), -INT64_C( 1238479814879937033) } },
    { {  INT64_C( 1204048529486170147), -INT64_C( 3947417975385572042),  INT64_C( 2832035834993704800),  INT64_C(  564923457047833123),
         INT64_C(  909121111506236106),  INT64_C( 1775624111860597910),  INT64_C( 2955967771544427848),  INT64_C( 8199681252459521248) },
      UINT8_C( 80),
      {  INT64_C( 6943761849412681455),  INT64_C( 8778695297453275203),  INT64_C( 6642432009962500760), -INT64_C( 6026249796547513617),
        -INT64_C(  472928855977065357),  INT64_C( 5791164821504624393),  INT64_C( 8367912982555843852),  INT64_C(  876197077347083454) },
      {  INT64_C( 8580355020071672932), -INT64_C(  136308712236279533),  INT64_C( 5013600975374695065),  INT64_C( 6461796806543086855),
         INT64_C( 7452197707851221126),  INT64_C( 2094259056363620383), -INT64_C( 4768240296989568249),  INT64_C(  229975492078179555) },
      {  INT64_C( 1204048529486170147), -INT64_C( 3947417975385572042),  INT64_C( 2832035834993704800),  INT64_C(  564923457047833123),
         INT64_C( 6626041364886920446),  INT64_C( 1775624111860597910),  INT64_C( 1181705491994019386),  INT64_C( 8199681252459521248) } },
    { {  INT64_C( 5372058967511377106),  INT64_C( 1498692677256658731), -INT64_C( 4993517995702482880), -INT64_C( 1474150314099470327),
         INT64_C( 1215583653993742696), -INT64_C( 1765819353204311459), -INT64_C( 7051352145137950572), -INT64_C( 2043035009515472992) },
      UINT8_C( 48),
      { -INT64_C( 8092084156536424521), -INT64_C( 4970814435758760476),  INT64_C( 5345190650240331516),  INT64_C( 4895879592428300050),
         INT64_C( 1010015961666356511),  INT64_C( 6178800532614691689),  INT64_C( 1138583862308376560),  INT64_C( 6541127970144162550) },
      {  INT64_C( 1513160735269543879), -INT64_C( 1171542836100170482), -INT64_C( 6561110587240898586),  INT64_C( 5433853002339763155),
         INT64_C( 1732261924944230219), -INT64_C( 2330826139406163272),  INT64_C( 3255738981115108060), -INT64_C( 8338234106764897160) },
      {  INT64_C( 5372058967511377106),  INT64_C( 1498692677256658731), -INT64_C( 4993517995702482880), -INT64_C( 1474150314099470327),
         INT64_C( 2477353237228353648),  INT64_C( 7590183313562959419), -INT64_C( 7051352145137950572), -INT64_C( 2043035009515472992) } },
    { {  INT64_C( 8718561788276475099), -INT64_C( 7527248101175689552),  INT64_C( 8681056619412409329),  INT64_C( 3779009751845024932),
         INT64_C( 3637246595698455617),  INT64_C( 8280541371477764759), -INT64_C(   60457352557687888), -INT64_C( 7172170031094784277) },
      UINT8_C( 74),
      { -INT64_C( 4448995574967132527), -INT64_C( 8507663096536132647),  INT64_C( 7319725741072227954), -INT64_C( 2277029967223007289),
         INT64_C(  780753517765341947),  INT64_C( 4568117654829144159), -INT64_C(  936330716463478316),  INT64_C( 1910500545187070548) },
      {  INT64_C( 4099123220512050857),  INT64_C( 3901940688070454042), -INT64_C( 2313491710567379004),  INT64_C( 1427085187287807125),
         INT64_C( 1668250501024447252),  INT64_C( 8760387161061997564), -INT64_C( 3558039170667879116),  INT64_C( 4305530474606043458) },
      {  INT64_C( 8718561788276475099), -INT64_C(  794588125075096875),  INT64_C( 8681056619412409329), -INT64_C( 6204948619972571957),
         INT64_C( 3637246595698455617),  INT64_C( 8280541371477764759), -INT64_C( 7111412407984337276), -INT64_C( 7172170031094784277) } },
    { {  INT64_C( 8746640825840676173), -INT64_C( 5063899185467137910), -INT64_C( 5487890747158354992), -INT64_C(  460458236670158279),
        -INT64_C( 7006575088694475316), -INT64_C( 3078888720160613882),  INT64_C( 1143551286862331578),  INT64_C( 3005323022584266000) },
      UINT8_C(210),
      { -INT64_C( 3340398549666401288),  INT64_C( 3653733716617211747), -INT64_C(  338024245944066167), -INT64_C( 5158730709013623928),
         INT64_C( 4106714224415625205),  INT64_C(  707601151774872618),  INT64_C( 8826940858231740870),  INT64_C( 5555884311800960930) },
      { -INT64_C( 7650640037918707258), -INT64_C( 5489037006271675061),  INT64_C( 6073867926576241630), -INT64_C( 7551461450737886835),
         INT64_C( 7945199818399460418),  INT64_C( 6533047940317542155),  INT64_C( 2607430904819933480), -INT64_C(  561193248999414322) },
      {  INT64_C( 8746640825840676173), -INT64_C( 6531442294028297835), -INT64_C( 5487890747158354992), -INT64_C(  460458236670158279),
        -INT64_C( 2019887176047050796), -INT64_C( 3078888720160613882),  INT64_C( 5249445070105997700), -INT64_C( 7073439209693342906) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi64(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_rolv_epi64(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i src = simde_test_x86_random_i64x8();
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m512i a = simde_test_x86_random_i64x8();
    simde__m512i b = simde_test_x86_random_i64x8();
    simde__m512i r = simde_mm512_mask_rolv_epi64(src, k, a, b);

    simde_test_x86_write_i64x8(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_rolv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask8 k;
    const int64_t a[8];
    const int64_t b[8];
    const int64_t r[8];
  } test_vec[] = {
    { UINT8_C(233),
      {  INT64_C( 6951723742327055977), -INT64_C(  150529619695734348), -INT64_C( 2836218887696759967), -INT64_C( 2095995197794139453),
         INT64_C( 2771625212042606125), -INT64_C( 1633732955813227973),  INT64_C( 7232326497493575828), -INT64_C( 6845386272481677915) },
      {  INT64_C( 6186869580858085444),  INT64_C( 2950420242153391231), -INT64_C( 5193761897011236633), -INT64_C( 2157270820046155526),
         INT64_C( 8256915554940012070),  INT64_C( 3023440655213543992), -INT64_C( 5988205641411222501), -INT64_C( 5301252911054210322) },
      {  INT64_C(  547115434975585942),  INT64_C(                   0),  INT64_C(                   0),  INT64_C( 1120171579641313547),
         INT64_C(                   0),  INT64_C( 4317073872917030738),  INT64_C(  879889972480635669),  INT64_C( 4281067545204426649) } },
    { UINT8_C(241),
      { -INT64_C( 3552399420179548666),  INT64_C( 1898048947936592081),  INT64_C( 2627702401767605482),  INT64_C( 8696286446534001556),
        -INT64_C( 4847095942182890400), -INT64_C( 1173100821681222873), -INT64_C( 7970827094043687725), -INT64_C( 2315695185766377998) },
      { -INT64_C( 7045168620257881965),  INT64_C( 6145149052938629715), -INT64_C( 6987348370497814920), -INT64_C( 1556311054189635832),
         INT64_C( 1064436911141630895),  INT64_C( 7902882715373273558),  INT64_C( 4159332851996900972),  INT64_C( 5369460200690465602) },
      { -INT64_C( 4871805010331667046),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),
        -INT64_C( 3877495522084915463), -INT64_C( 4528512143695548909),  INT64_C( 8668647624929170391),  INT64_C( 9183963330644039627) } },
    { UINT8_C( 90),
      {  INT64_C( 3919859238630158573), -INT64_C( 6222961578624696694),  INT64_C( 3434580156014574716), -INT64_C( 5452428924706733100),
         INT64_C( 1797124880819645060),  INT64_C( 3566274442223885417),  INT64_C( 4635122568948963247), -INT64_C( 5331539367901323898) },
      {  INT64_C( 6674722721838942203), -INT64_C( 7826220531850350212), -INT64_C( 4201846085234721787),  INT64_C( 8651439611816700968),
         INT64_C( 6750231600308584541), -INT64_C( 8066807251202024044), -INT64_C(  580205902917745683),  INT64_C( 8760170994222564765) },
      {  INT64_C(                   0), -INT64_C( 6153542621698278424),  INT64_C(                   0), -INT64_C( 3872017383621867525),
        -INT64_C( 2933904595923102351),  INT64_C(                   0),  INT64_C( 3095635450286400659),  INT64_C(                   0) } },
    { UINT8_C(225),
      {  INT64_C( 6110292287542118804),  INT64_C( 1480680773846517165), -INT64_C( 5711168328204830609), -INT64_C(  113359191195003093),
        -INT64_C(  371543337792557554), -INT64_C( 7589370048845138104), -INT64_C( 7694553437446958883),  INT64_C( 6175717427664454475) },
      {  INT64_C( 3820375115076135613), -INT64_C( 7050810995801940886), -INT64_C( 7202279099599561460),  INT64_C( 8790189455027875424),
        -INT64_C(  931251671722108944),  INT64_C( 7675165514523796004),  INT64_C( 7311011278986782855), -INT64_C( 4804765973241542884) },
      { -INT64_C( 8459585500912010958),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),
         INT64_C(                   0), -INT64_C( 7382679030592290199), -INT64_C( 7225404086604501302),  INT64_C( 8886182833322412260) } },
    { UINT8_C(225),
      { -INT64_C( 7961978558934770731), -INT64_C( 7992664579049036062),  INT64_C( 2902003029150002235),  INT64_C( 5040851172592000265),
        -INT64_C( 6949577876075028658),  INT64_C( 1297951312178806153), -INT64_C( 6866159193481831892), -INT64_C( 5957514235147408483) },
      { -INT64_C( 6889600230700157945), -INT64_C(  387567066056323146), -INT64_C(  511371475215131984),  INT64_C( 5473376872916425705),
        -INT64_C( 8141817803137048757), -INT64_C( 2752899859418967038),  INT64_C( 8928677159147426710), -INT64_C( 8055892698055053661) },
      { -INT64_C( 4562331489625314616),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),
         INT64_C(                   0),  INT64_C( 5191805248715224612),  INT64_C( 2730112551200763296),  INT64_C( 7900823296056574704) } },
    { UINT8_C( 87),
      {  INT64_C(  628744680744481347), -INT64_C( 6256050370134308133), -INT64_C(  659859773562675413),  INT64_C( 3785499723561680504),
         INT64_C( 5960976415794641085),  INT64_C( 2390312944986155566), -INT64_C( 7695196987971049673), -INT64_C(  571872233190493974) },
      { -INT64_C( 3616563538624382203),  INT64_C(  604465112712001868), -INT64_C( 3632860210000316444), -INT64_C( 5899793954472698900),
         INT64_C( 8511795780916005043), -INT64_C( 3347551191088729934),  INT64_C( 4216327263921230547), -INT64_C( 7559150256015725630) },
      {  INT64_C( 1673085710113851489), -INT64_C( 2254797687558915438),  INT64_C( 3720521671174213763),  INT64_C(                   0),
        -INT64_C( 4185368195386994851),  INT64_C(                   0),  INT64_C( 6404675724052834729),  INT64_C(                   0) } },
    { UINT8_C( 72),
      {  INT64_C( 8786497623901268547), -INT64_C( 3418170549935377759), -INT64_C(  850518657915206135), -INT64_C(  539182131682173765),
        -INT64_C( 9089069030614698406), -INT64_C( 8906767132868590698), -INT64_C( 7435118775458561260),  INT64_C( 8390020402005650114) },
      {  INT64_C( 6750531659035550798), -INT64_C( 3533260023839724296), -INT64_C( 3220499168490689575), -INT64_C( 4056130859428011481),
         INT64_C( 4144335582332816478), -INT64_C( 6622081683545098864),  INT64_C( 7526264026601079950), -INT64_C( 3022937379214606901) },
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0), -INT64_C( 1419093494550996323),
         INT64_C(                   0),  INT64_C(                   0),  INT64_C( 5311845664811198004),  INT64_C(                   0) } },
    { UINT8_C(235),
      { -INT64_C(  777603757450134527),  INT64_C( 6720044201020522257),  INT64_C( 1239623852269429183), -INT64_C( 8648495105020692048),
        -INT64_C( 7146823880338037156), -INT64_C( 5343922324828548639),  INT64_C( 3231141953514927475),  INT64_C( 2078751803137810636) },
      {  INT64_C( 5224959965797025371), -INT64_C( 2953067899045883302),  INT64_C( 4735101205907120653), -INT64_C( 9029035385638620714),
        -INT64_C( 3665108263908038030), -INT64_C( 8432390530970884930), -INT64_C( 2019360797627213757),  INT64_C( 7738639886952533179) },
      {  INT64_C(  371942656267889452),  INT64_C( 7121037618811570578),  INT64_C(                   0), -INT64_C( 2196663298057699689),
         INT64_C(                   0),  INT64_C( 7887391455647638648),  INT64_C( 7402391554409868185),  INT64_C( 6982490021489138438) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_rolv_epi64(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m512i a = simde_test_x86_random_i64x8();
    simde__m512i b = simde_test_x86_random_i64x8();
    simde__m512i r = simde_mm512_maskz_rolv_epi64(k, a, b);

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_rolv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_mask_rolv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_maskz_rolv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_rolv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_mask_rolv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_maskz_rolv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_rolv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_rolv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_rolv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_rolv_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_mask_rolv_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_maskz_rolv_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_rolv_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_mask_rolv_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_maskz_rolv_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_rolv_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_rolv_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_rolv_epi64)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
