# gophian -- tools to help with Debianizing Go software
# Copyright (C) 2024-2025 Maytham Alsudany <maytham@debian.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.

import re
from typing import List, Optional, Union

import click
import requests
from treelib import Tree

from gophian.__about__ import BUGS_URL
from gophian.colour import parse_colour
from gophian.error import ExecutionError, GophianError
from gophian.packages import DebianGolangPackages
from gophian.session import Session
from gophian.trace import trace
from gophian.vcs import package_from_import_path


@click.command()
@click.argument("importpath")
@click.option("--quiet/--no-quiet", default=False, help="Don't print warnings.")
@click.option(
    "--warn-packaged/--no-warn-packaged",
    default=True,
    help="Warn if the given Go package has already been packaged for Debian.",
)
@click.option(
    "--include-packaged-deps/--no-include-packaged-deps",
    default=True,
    help="Include packaged dependencies in the estimate.",
)
@click.option(
    "--include-version-conflicts/--no-include-version-conflicts",
    default=True,
    help=(
        "Include any version conflicts between the package's go.mod and the"
        "version in Debian unstable. Ignored if --no-include-packaged-deps is"
        "passed."
    ),
)
@click.option(
    "--max-depth",
    required=False,
    help="Maximum recursion depth of unpackaged dependencies.",
    type=int,
)
@click.option(
    "--packaged-unstable-color",
    "packaged_unstable_color_str",
    default="bright_green",
    help="Colour to denote dependencies packaged in Debian unstable.",
    type=str,
    envvar="GOPHIAN_PACKAGED_UNSTABLE_COLOR",
)
@click.option(
    "--packaged-experimental-color",
    "packaged_experimental_color_str",
    default="green",
    help="Colour to denote dependencies packaged in Debian experimental.",
    type=str,
    envvar="GOPHIAN_PACKAGED_EXPERIMENTAL_COLOR",
)
@click.option(
    "--packaged-new-color",
    "packaged_new_color_str",
    default="blue",
    help="Colour to denote dependencies pending in the Debian NEW queue.",
    type=str,
    envvar="GOPHIAN_PACKAGED_NEW_COLOR",
)
@click.option(
    "--seen-color",
    "seen_color_str",
    default="bright_black",
    help="Colour to denote dependencies that already appear elsewhere in the output tree.",
    type=str,
    envvar="GOPHIAN_SEEN_COLOR",
)
@click.option(
    "--unpackaged-color",
    "unpackaged_color_str",
    default="reset",
    help="Colour to denote unpackaged dependencies.",
    type=str,
    envvar="GOPHIAN_UNPACKAGED_COLOR",
)
@click.option(
    "--version-conflicts-color",
    "version_conflicts_color_str",
    default="reset",
    help="Colour to denote version conflicts.",
    type=str,
    envvar="GOPHIAN_VERSION_CONFLICTS_COLOR",
)
@click.pass_context
def estimate(
    context: click.Context,
    importpath: str,
    quiet: bool,
    warn_packaged: bool,
    include_packaged_deps: bool,
    include_version_conflicts: bool,
    max_depth: Optional[int],
    packaged_unstable_color_str: str,
    packaged_experimental_color_str: str,
    packaged_new_color_str: str,
    seen_color_str: str,
    unpackaged_color_str: str,
    version_conflicts_color_str: str,
) -> None:
    """
    Estimate work required to Debianize a given Go package.
    """

    requests_session: requests.Session = context.obj

    if not quiet:
        click.secho("gophian is experimental software!", bold=True, fg="yellow")
        click.secho("Please report any problems to:", fg="yellow")
        click.secho(BUGS_URL, fg="yellow")

    colours: dict[str, str | int | tuple[int, int, int]] = {
        "packaged_unstable_color": parse_colour(
            "GOPHIAN_PACKAGED_UNSTABLE_COLOR", packaged_unstable_color_str
        ),
        "packaged_experimental_color": parse_colour(
            "GOPHIAN_PACKAGED_EXPERIMENTAL_COLOR", packaged_experimental_color_str
        ),
        "packaged_new_color": parse_colour(
            "GOPHIAN_PACKAGED_NEW_COLOR", packaged_new_color_str
        ),
        "seen_color": parse_colour("GOPHIAN_SEEN_COLOR", seen_color_str),
        "unpackaged_color": parse_colour(
            "GOPHIAN_UNPACKAGED_COLOR", unpackaged_color_str
        ),
        "version_conflicts_color": parse_colour(
            "GOPHIAN_VERSION_CONFLICTS_COLOR", version_conflicts_color_str
        ),
    }

    try:
        package, _ = package_from_import_path(requests_session, importpath)
    except Exception as error:
        click.echo(error)
        click.secho("Did you specify a Go package import path?", fg="yellow")
        context.exit(1)

    if package != importpath:
        click.echo(f"Continuing with repo root {package} instead of {importpath}")

    debian_packages = DebianGolangPackages(requests_session)

    if debian_packages._check_for_package(package, quiet) and warn_packaged:
        if not quiet:
            click.secho("To ignore, pass the --no-warn-packaged flag.", fg="cyan")
        return

    with Session(requests_session) as session:
        seen: List[str] = []
        errors: List[ExecutionError | GophianError] = []

        tree = Tree()
        tree.create_node(package, package)
        dependency_tree(
            session,
            seen,
            errors,
            package,
            include_packaged_deps,
            include_version_conflicts,
            colours,
            debian_packages,
            tree,
            max_depth,
        )
        click.echo(tree.show(stdout=False))

        for error in errors:
            click.echo(error)


def dependency_tree(
    session: Session,
    seen: List[str],
    errors: List[Union[ExecutionError, GophianError]],
    package: str,
    include_packaged_deps: bool,
    include_version_conflicts: bool,
    colours: dict[str, str | int | tuple[int, int, int]],
    debian_packages: DebianGolangPackages,
    tree: Tree,
    max_depth: Optional[int],
    level: int = 1,
):
    try:
        trace("Fetching package: " + package)
        go_package = session.go_get(package)
        trace("Finding dependencies: " + package)
        dep_packages = go_package.find_dependencies()
        trace("Found dependencies: " + package)
        trace(str(dep_packages))
    except ExecutionError as error:
        trace("Execution error: " + package)
        errors.append(error)
        tree.create_node(
            click.style("Error getting dependencies", fg="red"),
            f"{package}_error",
            parent=package,
        )
        return
    except GophianError as error:
        trace("Gophian error: " + package)
        errors.append(error)
        tree.create_node(
            click.style("Error getting dependencies", fg="red"),
            f"{package}_error",
            parent=package,
        )
        return
    for dep_package, _ in dep_packages:
        result = debian_packages.library_is_packaged(dep_package)
        if result is None:
            try:
                dep_go_package = session.go_get(dep_package)
            except ExecutionError as error:
                trace("Execution error: " + package)
                errors.append(error)
                tree.create_node(
                    dep_package,
                    f"{package}_{dep_package}",
                    parent=package,
                )
                tree.create_node(
                    click.style("Error fetching package", fg="red"),
                    f"{package}_{dep_package}_error",
                    parent=f"{package}_{dep_package}",
                )
                continue
            except GophianError as error:
                trace("Gophian error: " + package)
                errors.append(error)
                tree.create_node(
                    dep_package,
                    f"{package}_{dep_package}",
                    parent=package,
                )
                tree.create_node(
                    click.style("Error fetching package", fg="red"),
                    f"{package}_{dep_package}_error",
                    parent=f"{package}_{dep_package}",
                )
                continue
            # This handles the case where the module has a version suffix
            # e.g. github/go-git/go-billy/v5
            if (
                dep_go_package.module != dep_package
                and dep_go_package.module.startswith(dep_package + "/")
            ):
                result = debian_packages.library_is_packaged(dep_go_package.module)
        if result is not None:
            trace(dep_package + " is packaged in Debian")
            debian_package, debian_package_version, suite = result
            if include_packaged_deps:
                trace(dep_package + " will be included in output")
                if suite == "experimental":
                    trace(dep_package + " is in experimental")
                    node_text = click.style(
                        f"{dep_package} ({debian_package}) [experimental]",
                        fg=colours["packaged_experimental_color"],
                    )
                elif suite == "NEW":
                    trace(dep_package + " is in NEW")
                    node_text = click.style(
                        f"{dep_package} ({debian_package}) [NEW]",
                        fg=colours["packaged_new_color"],
                    )
                else:
                    trace(dep_package + " is in unstable")
                    node_text = click.style(
                        f"{dep_package} ({debian_package})",
                        fg=colours["packaged_unstable_color"],
                    )
                if include_version_conflicts and dep_package in go_package.deps:
                    trace(dep_package + " will be checked for version conflicts")
                    debian_version = re.sub(
                        r"^[0-9]+:", "", debian_package_version.vstring
                    )
                    [go_mod_major, go_mod_minor] = go_package.deps[
                        dep_package
                    ].vstring.split(".")[:2]
                    [debian_major, debian_minor] = debian_version.split(".")[:2]
                    if (
                        (
                            go_package.deps[dep_package].commit
                            and go_package.deps[dep_package].vstring != debian_version
                        )
                        or (go_mod_major != debian_major)
                        or (go_mod_major == "0" and go_mod_minor != debian_minor)
                    ):
                        trace(
                            dep_package
                            + " has a version conflict: "
                            + f"[{debian_version} ≠ {go_package.deps[dep_package]}]"
                        )
                        node_text += click.style(
                            f" [{debian_version} ≠ {go_package.deps[dep_package]}]",
                            fg=colours["version_conflicts_color"],
                        )
                tree.create_node(
                    node_text,
                    f"{package}_{dep_package}",
                    parent=package,
                )
        elif dep_package in seen:
            trace(dep_package + " has already been seen")
            tree.create_node(
                click.style(dep_package, fg=colours["seen_color"]),
                f"{package}_{dep_package}",
                parent=package,
            )
        else:
            trace(dep_package + " is not in Debian")
            trace("Depth reached: " + str(level) + " < " + str(max_depth))
            tree.create_node(
                click.style(dep_package, fg=colours["unpackaged_color"]),
                dep_package,
                parent=package,
            )
            seen.append(dep_package)
            if max_depth is None or level < max_depth:
                dependency_tree(
                    session,
                    seen,
                    errors,
                    dep_package,
                    include_packaged_deps,
                    include_version_conflicts,
                    colours,
                    debian_packages,
                    tree,
                    max_depth,
                    level + 1,
                )
