/* SPDX-FileCopyrightText: 2010 - Jesse van den Kieboom
 * SPDX-FileCopyrightText: 2014, 2025 - Sébastien Wilmet
 * SPDX-License-Identifier: LGPL-2.1-or-later
 */

#include <gtksourceview/gtksource.h>
#include <locale.h>
#include <string.h>

typedef struct
{
	const gchar *expected_buffer_content;
	gint newline_type;
} LoaderTestData;

static void
set_file_content (GFile       *file,
		  const gchar *content)
{
	GError *error = NULL;

	g_file_replace_contents (file,
				 content,
				 strlen (content),
				 NULL,
				 FALSE,
				 G_FILE_CREATE_REPLACE_DESTINATION,
				 NULL,
				 NULL,
				 &error);
	g_assert_no_error (error);
}

static void
delete_file (GFile *location)
{
	GError *error = NULL;

	g_file_delete (location, NULL, &error);
	g_assert_no_error (error);
}

static void
load_file_cb (GObject      *source_object,
	      GAsyncResult *result,
	      gpointer      user_data)
{
	GtkSourceFileLoader *loader = GTK_SOURCE_FILE_LOADER (source_object);
	LoaderTestData *data = user_data;
	GError *error = NULL;

	gtk_source_file_loader_load_finish (loader, result, &error);
	g_assert_no_error (error);

	if (data->expected_buffer_content != NULL)
	{
		GtkSourceBuffer *buffer;
		GtkTextIter start;
		GtkTextIter end;
		gchar *buffer_content;

		buffer = gtk_source_file_loader_get_buffer (loader);

		gtk_text_buffer_get_bounds (GTK_TEXT_BUFFER (buffer), &start, &end);
		buffer_content = gtk_text_iter_get_slice (&start, &end);

		g_assert_cmpstr (buffer_content, ==, data->expected_buffer_content);

		g_free (buffer_content);
	}

	if (data->newline_type != -1)
	{
		g_assert_cmpint (gtk_source_file_loader_get_newline_type (loader),
				 ==,
				 data->newline_type);
	}

	/* Finished. */
	gtk_main_quit ();
}

static void
test_loader (const gchar *content,
             const gchar *expected_buffer_content,
             gint         newline_type)
{
	GFile *location;
	GtkSourceBuffer *buffer;
	GtkSourceFile *file;
	GtkSourceFileLoader *loader;
	GSList *candidate_encodings;
	LoaderTestData *data;

	location = g_file_new_build_filename (g_get_tmp_dir (),
					      "libgedit-gtksourceview-test-file-loader.txt",
					      NULL);
	set_file_content (location, content);

	buffer = gtk_source_buffer_new (NULL);
	file = gtk_source_file_new ();
	gtk_source_file_set_location (file, location);
	loader = gtk_source_file_loader_new (buffer, file);

	candidate_encodings = g_slist_prepend (NULL, gtk_source_encoding_new_utf8 ());
	gtk_source_file_loader_set_candidate_encodings (loader, candidate_encodings);

	data = g_new0 (LoaderTestData, 1);
	data->expected_buffer_content = expected_buffer_content;
	data->newline_type = newline_type;

	gtk_source_file_loader_load_async (loader,
					   G_PRIORITY_DEFAULT,
					   NULL, NULL, NULL, NULL,
					   load_file_cb,
					   data);

	gtk_main ();

	g_free (data);
	delete_file (location);
	g_object_unref (location);
	g_object_unref (buffer);
	g_object_unref (file);
	g_object_unref (loader);
	g_slist_free_full (candidate_encodings, (GDestroyNotify) gtk_source_encoding_free);
}

static void
test_end_line_stripping (void)
{
	test_loader ("hello world\n",
		     "hello world",
		     -1);

	test_loader ("hello world",
		     "hello world",
		     -1);

	test_loader ("\nhello world",
		     "\nhello world",
		     -1);

	test_loader ("\nhello world\n",
		     "\nhello world",
		     -1);

	test_loader ("hello world\n\n",
		     "hello world\n",
		     -1);

	test_loader ("hello world\r\n",
		     "hello world",
		     -1);

	test_loader ("hello world\r\n\r\n",
		     "hello world\r\n",
		     -1);

	test_loader ("\n",
		     "",
		     -1);

	test_loader ("\r\n",
		     "",
		     -1);

	test_loader ("\n\n",
		     "\n",
		     -1);

	test_loader ("\r\n\r\n",
		     "\r\n",
		     -1);
}

static void
test_end_new_line_detection (void)
{
	test_loader ("hello world\n",
		     NULL,
		     GTK_SOURCE_NEWLINE_TYPE_LF);

	test_loader ("hello world\r\n",
		     NULL,
		     GTK_SOURCE_NEWLINE_TYPE_CR_LF);

	test_loader ("hello world\r",
		     NULL,
		     GTK_SOURCE_NEWLINE_TYPE_CR);
}

static void
test_begin_new_line_detection (void)
{
	test_loader ("\nhello world",
		     NULL,
		     GTK_SOURCE_NEWLINE_TYPE_LF);

	test_loader ("\r\nhello world",
		     NULL,
		     GTK_SOURCE_NEWLINE_TYPE_CR_LF);

	test_loader ("\rhello world",
		     NULL,
		     GTK_SOURCE_NEWLINE_TYPE_CR);
}

static void
max_size_load_cb (GObject      *source_object,
		  GAsyncResult *result,
		  gpointer      user_data)
{
	GtkSourceFileLoader *loader = GTK_SOURCE_FILE_LOADER (source_object);
	GtkSourceBuffer *buffer;
	GtkTextIter start_iter;
	GtkTextIter end_iter;
	GError *error = NULL;

	gtk_source_file_loader_load_finish (loader, result, &error);
	g_assert_error (error, GTK_SOURCE_FILE_LOADER_ERROR, GTK_SOURCE_FILE_LOADER_ERROR_TOO_BIG);
	g_clear_error (&error);

	/* The buffer must be empty. */
	buffer = gtk_source_file_loader_get_buffer (loader);
	g_assert_nonnull (buffer);
	gtk_text_buffer_get_bounds (GTK_TEXT_BUFFER (buffer), &start_iter, &end_iter);
	g_assert_true (gtk_text_iter_equal (&start_iter, &end_iter));

	/* Finished. */
	gtk_main_quit ();
}

static void
test_max_size_from_file (void)
{
	GFile *location;
	GtkSourceBuffer *buffer;
	GtkSourceFile *file;
	GtkSourceFileLoader *loader;

	location = g_file_new_build_filename (g_get_tmp_dir (),
					      "libgedit-gtksourceview-test-file-loader.txt",
					      NULL);
	set_file_content (location, "Moon");

	buffer = gtk_source_buffer_new (NULL);
	file = gtk_source_file_new ();
	gtk_source_file_set_location (file, location);

	loader = gtk_source_file_loader_new (buffer, file);
	gtk_source_file_loader_set_max_size (loader, 3);

	gtk_source_file_loader_load_async (loader,
					   G_PRIORITY_DEFAULT,
					   NULL, NULL, NULL, NULL,
					   max_size_load_cb,
					   NULL);
	gtk_main ();

	delete_file (location);
	g_object_unref (location);
	g_object_unref (buffer);
	g_object_unref (file);
	g_object_unref (loader);
}

static void
test_max_size_from_stream (void)
{
	GInputStream *input_stream;
	GtkSourceBuffer *buffer;
	GtkSourceFile *file;
	GtkSourceFileLoader *loader;

	input_stream = g_memory_input_stream_new_from_data ("Sun", -1, NULL);

	buffer = gtk_source_buffer_new (NULL);
	file = gtk_source_file_new ();

	loader = gtk_source_file_loader_new_from_stream (buffer, file, input_stream);
	gtk_source_file_loader_set_max_size (loader, 2);

	gtk_source_file_loader_load_async (loader,
					   G_PRIORITY_DEFAULT,
					   NULL, NULL, NULL, NULL,
					   max_size_load_cb,
					   NULL);
	gtk_main ();

	g_object_unref (input_stream);
	g_object_unref (buffer);
	g_object_unref (file);
	g_object_unref (loader);
}

int
main (int    argc,
      char **argv)
{
	int ret;

	/* To better setup the locale encoding (usually, to get UTF-8). */
	setlocale (LC_ALL, "");

	gtk_source_init ();
	g_test_init (&argc, &argv, NULL);

	g_test_add_func ("/file-loader/end-line-stripping", test_end_line_stripping);
	g_test_add_func ("/file-loader/end-new-line-detection", test_end_new_line_detection);
	g_test_add_func ("/file-loader/begin-new-line-detection", test_begin_new_line_detection);
	g_test_add_func ("/file-loader/max-size/from_file", test_max_size_from_file);
	g_test_add_func ("/file-loader/max-size/from_stream", test_max_size_from_stream);

	ret = g_test_run ();
	gtk_source_finalize ();

	return ret;
}
