#!/usr/bin/env python3
# Copyright 2025 The Chromium Authors
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.
"""
tast_control.py -- populating tast_control.gni for legacy tast runs.
"""

import argparse
import sys


def PopulateData(out, data, indent=2):
  """Populates data to out file.

  out: the out file stream
  data: the data file stream
  indent: indent spaces, default 2
  """
  out.write(' ' * indent)
  out.write('# Autogenerated from ')
  out.write(data.name)
  out.write('\n')
  for line in data:
    stripped = line.strip()
    if stripped == '':
      continue
    if stripped.startswith('#'):
      continue
    stripped = stripped.removeprefix('tast.')
    out.write(' ' * indent)
    out.write('"%s",\n' % stripped)


def main(argv):
  parser = argparse.ArgumentParser(usage="tast_control.py [options]")
  parser.add_argument("-o",
                      "--output",
                      metavar="FILE",
                      help=("Output tast_control.gni file."))
  parser.add_argument("-t",
                      "--template",
                      metavar="FILE",
                      help=("Template tast_control.gni file."))
  parser.add_argument("-i",
                      "--input",
                      metavar="FILE",
                      help=("Input disabled tests file."))
  parser.add_argument(
      "--input-public",
      metavar="FILE",
      help=("Input disabled tests file for public builders only."))

  args, extras = parser.parse_known_args(argv[1:])

  out_file = args.output
  tmpl_file = args.template
  in_file = args.input
  in_public_file = args.input_public

  with open(out_file, 'w') as out, open(tmpl_file, 'r') as tmpl:
    for line in tmpl:
      if line.strip() == 'TAST_CONTROL_DISABLED_TESTS':
        with open(in_file, 'r') as data:
          PopulateData(out, data)
          continue
      if line.strip() == 'TAST_CONTROL_DISABLED_TESTS_PUBLIC_BUILDERS':
        with open(in_public_file, 'r') as data:
          PopulateData(out, data, indent=4)
          continue
      out.write(line)

  return 0


if __name__ == '__main__':
  sys.exit(main(sys.argv))
