/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright 2009--2026 by Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


#pragma once


/////////////////////// Stdlib includes


/////////////////////// Qt includes
#include <QDomElement>


/////////////////////// Local includes
#include "MsXpS/export-import-config.h"

#include "MsXpS/libXpertMassCore/globals.hpp"
#include "MsXpS/libXpertMassCore/PropListHolder.hpp"
#include "MsXpS/libXpertMassCore/IsotopicData.hpp"
#include "MsXpS/libXpertMassCore/Formula.hpp"
#include "MsXpS/libXpertMassCore/Utils.hpp"
#include "MsXpS/libXpertMassCore/PolChemDef.hpp"

namespace MsXpS
{
namespace libXpertMassCore
{


class Modif;

typedef std::unique_ptr<Modif> ModifUPtr;
typedef std::unique_ptr<const Modif> ModifCstUPtr;
typedef std::shared_ptr<Modif> ModifSPtr;
typedef std::shared_ptr<const Modif> ModifCstSPtr;
typedef std::weak_ptr<Modif> ModifWPtr;
typedef std::weak_ptr<const Modif> ModifCstWPtr;

using UuidModifCstWPtrPair = std::pair<QString, ModifCstWPtr>;
using UuidModifWPtrPair    = std::pair<QString, ModifWPtr>;

class Monomer;
using MonomerCstSPtr = std::shared_ptr<const Monomer>;

class DECLSPEC Modif: public PropListHolder
{
  public:
  Modif(PolChemDefCstSPtr pol_chem_def_csp,
        const QDomElement &element,
        int version);
  Modif(PolChemDefCstSPtr pol_chem_def_csp = nullptr,
        const QString &modif_name          = QString(),
        const QString &formula_string      = QString(),
        double mono                        = 0.0,
        double avg                         = 0.0);
  Modif(const Modif &other);
  virtual ~Modif();

  //////////////// THE POLCHEMDEF /////////////////////
  void setPolChemDefCstSPtr(PolChemDefCstSPtr pol_chem_def_csp);
  const PolChemDefCstSPtr &getPolChemDefCstSPtr() const;

  //////////////// THE NAME /////////////////////
  void setName(const QString &name);
  QString getName() const;

  //////////////// THE FORMULA /////////////////////
  void setFormula(const QString &formula_string);
  const QString &getFormula() const;
  QString formula(bool with_title) const;

  //////////////// THE TARGETS /////////////////////
  void setTargets(const QString &targets);
  QString getTargets() const;
  bool doesTargetMonomer(const QString &code) const;

  //////////////// THE MAX MODIF COUNT /////////////////////
  void setMaxCount(int);
  int getMaxCount() const;

  //////////////// OPERATORS /////////////////////
  using PropListHolder::operator=;
  Modif &operator=(const Modif &modif);
  bool operator==(const Modif &modif) const;
  bool operator!=(const Modif &modif) const;

  //////////////// VALIDATIONS /////////////////////
  ModifCstSPtr getFromPolChemDefByName() const;
  Enums::PolChemDefEntityStatus isKnownByNameInPolChemDef() const;

  QString
  validateTargets(const QString &targets_string, bool simplify, bool &ok) const;
  QString validateTargets(bool simplify, bool &ok);
  bool validate(ErrorList *error_list_p) const;
  bool isValid() const;

  //////////////// MASS OPERATIONS /////////////////////
  bool calculateMasses(const IsotopicDataCstSPtr &isotopic_data_csp,
                       double &mono,
                       double &avg) const;
  bool calculateMasses(const IsotopicDataCstSPtr &isotopic_data_csp);
  Modif &calculateMasses(bool &ok,
                         const IsotopicDataCstSPtr &isotopic_data_csp);
  const Modif &accountMasses(double *mono_p = nullptr,
                             double *avg_p  = nullptr,
                             int times      = 1) const;
  const Modif &accountMasses(double &mono, double &avg, int times = 1) const;
  double getMass(Enums::MassType mass_type);

  //////////////// XML /////////////////////
  bool renderXmlMdfElement(const QDomElement &element, int version);
  QString
  formatXmlMdfElement(int offset,
                      const QString &indent = Utils::xmlIndentationToken) const;

  //////////////// UTILS /////////////////////
  void clear();
  QString toString() const;

  protected:
  PolChemDefCstSPtr mcsp_polChemDef = nullptr;
  QString m_name                    = "";
  QString m_formula                 = "";
  QString m_targets                 = "";
  int m_maxCount                    = -1;

  double m_mono = 0.0;
  double m_avg  = 0.0;

  mutable bool m_isValid = false;
};

} // namespace libXpertMassCore
} // namespace MsXpS

Q_DECLARE_METATYPE(MsXpS::libXpertMassCore::Modif);
extern int modifMetaTypeId;

Q_DECLARE_METATYPE(MsXpS::libXpertMassCore::Modif *);
extern int modifPtrMetaTypeId;
