/*
 * Copyright 2021-2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.tooling.core.htmlreport;

import org.intellij.lang.annotations.Language;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;

import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.jupiter.api.Assertions.assertDoesNotThrow;

class DefaultHtmlReportWriterTests {

	@Test // https://github.com/ota4j-team/open-test-reporting/issues/293
	void canHandleWhitespaceInSources(@TempDir Path tempDir) throws Exception {
		@Language("xml")
		var xml = """
				<?xml version="1.0" encoding="utf-8"?>
				<e:events xmlns:c="https://schemas.opentest4j.org/reporting/core/0.2.0" xmlns:e="https://schemas.opentest4j.org/reporting/events/0.2.0">
				    <e:started id="1" name="SomeTest" time="2025-01-31T13:52:41.404Z">
				        <c:sources>
				            <c:fileSource path="./some/file"/>
				        </c:sources>
				    </e:started>
				    <e:finished id="1" time="2025-01-31T13:52:41.608Z">
				        <c:result status="SUCCESSFUL"/>
				    </e:finished>
				</e:events>
				""";
		var xmlFile = Files.writeString(tempDir.resolve("report.xml"), xml);

		var writer = new DefaultHtmlReportWriter();
		assertDoesNotThrow(() -> writer.writeHtmlReport(List.of(xmlFile), tempDir.resolve("report.html")));
	}

	@Test // https://github.com/ota4j-team/open-test-reporting/issues/294
	void canHandleWhitespaceInAttachments(@TempDir Path tempDir) throws Exception {
		@Language("xml")
		var xml = """
				<?xml version="1.0" encoding="utf-8"?>
				<e:events xmlns:c="https://schemas.opentest4j.org/reporting/core/0.2.0" xmlns:e="https://schemas.opentest4j.org/reporting/events/0.2.0">
				    <e:started id="1" name="SomeTest" time="2025-01-31T13:52:41.404Z"/>
				    <e:reported id="1" time="2025-01-31T14:22:15.608Z">
				        <c:attachments>
				            <c:output source="stdout" time="2025-01-31T14:22:15.608Z">someOutput</c:output>
				        </c:attachments>
				    </e:reported>
				    <e:finished id="1" time="2025-01-31T13:52:41.608Z">
				        <c:result status="SUCCESSFUL"/>
				    </e:finished>
				</e:events>
				""";
		var xmlFile = Files.writeString(tempDir.resolve("report.xml"), xml);

		var writer = new DefaultHtmlReportWriter();
		assertDoesNotThrow(() -> writer.writeHtmlReport(List.of(xmlFile), tempDir.resolve("report.html")));
	}

	@Test
	void handlesRelativePath() {
		var xmlFile = Path.of("junit-platform-events-8096972057150721349.xml");
		var relativePath = DefaultHtmlReportWriter.tryRelativize(Path.of("open-test-report.html"), xmlFile);
		assertThat(relativePath).isEqualTo(xmlFile);
	}

	@Test
	void avoidsHtmlInclusionProblems(@TempDir Path tempDir) throws Exception {
		@Language("xml")
		var xml = """
				<?xml version="1.0" encoding="utf-8"?>
				<e:events xmlns:c="https://schemas.opentest4j.org/reporting/core/0.2.0" xmlns:e="https://schemas.opentest4j.org/reporting/events/0.2.0">
				    <e:started id="1" name="&lt;/script&gt;" time="2025-01-31T13:52:41.404Z"/>
				    <e:finished id="1" time="2025-01-31T13:52:41.608Z">
				        <c:result status="SUCCESSFUL"/>
				    </e:finished>
				</e:events>
				""";
		var xmlFile = Files.writeString(tempDir.resolve("report.xml"), xml);
		var htmlFile = tempDir.resolve("report.html");

		var writer = new DefaultHtmlReportWriter();
		writer.writeHtmlReport(List.of(xmlFile), htmlFile);

		assertThat(Files.readString(htmlFile)).contains("\"name\":\"\\u003c/script\\u003e\"");
	}

}
