/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */


#include "common/file.h"
#include "common/system.h"
#include "graphics/cursorman.h"
#include "sci/graphics/drivers/gfxdriver_intern.h"

namespace Sci {

class SCI0_CGABWDriver final : public SCI0_DOSPreVGADriver {
public:
	SCI0_CGABWDriver(uint32 monochromeColor, bool rgbRendering);
	~SCI0_CGABWDriver() override;
	void copyRectToScreen(const byte *src, int srcX, int srcY, int pitch, int destX, int destY, int w, int h, const PaletteMod*, const byte*) override;
	void replaceCursor(const void *cursor, uint w, uint h, int hotspotX, int hotspotY, uint32 keycolor) override;
	Common::Point getMousePos() const override;
	void setMousePos(const Common::Point &pos) const override;
	void setShakePos(int shakeXOffset, int shakeYOffset) const override;
	void clearRect(const Common::Rect &r) const override;
	Common::Point getRealCoords(Common::Point &pos) const override;
	static bool validateMode(Common::Platform p) { return (p == Common::kPlatformDOS) && checkDriver(_driverFiles, 2); }
private:
	void setupRenderProc() override;
	byte _monochromePalette[6];
	const byte *_monochromePatterns;
	bool _earlyVersion;
	typedef void (*LineProc)(byte*&, const byte*, int, int, int, const byte*, const byte*);
	LineProc _renderLine;
	static const char *_driverFiles[2];
};

SCI0_CGABWDriver::SCI0_CGABWDriver(uint32 monochromeColor, bool rgbRendering) : SCI0_DOSPreVGADriver(2, 640, 400, rgbRendering), _monochromePatterns(nullptr), _earlyVersion(false), _renderLine(nullptr) {
	_monochromePalette[0] = _monochromePalette[1] = _monochromePalette[2] = 0;
	_monochromePalette[3] = (monochromeColor >> 16) & 0xff;
	_monochromePalette[4] = (monochromeColor >> 8) & 0xff;
	_monochromePalette[5] = monochromeColor & 0xff;
	assignPalette(_monochromePalette);

	if (!(_monochromePatterns = SciGfxDrvInternal::monochrInit(_driverFiles[0], _earlyVersion)) && !(_monochromePatterns = SciGfxDrvInternal::monochrInit(_driverFiles[1], _earlyVersion)))
		error("Failed to open '%s' or '%s'", _driverFiles[0], _driverFiles[1]);
}

SCI0_CGABWDriver::~SCI0_CGABWDriver() {
	delete[] _monochromePatterns;
}

void SCI0_CGABWDriver::copyRectToScreen(const byte *src, int srcX, int srcY, int pitch, int destX, int destY, int w, int h, const PaletteMod*, const byte*) {
	GFXDRV_ASSERT_READY;

	byte *dst = _compositeBuffer;
	int ty = destY & 7;

	if (_earlyVersion) {
		++ty;
		byte diff = srcX & 1;
		srcX &= ~1;
		destX &= ~1;
		w = (w + diff + 1) & ~1;
	}

	src += (srcY * pitch + srcX);

	for (int i = 0; i < h; ++i) {
		_renderLine(dst, src, w, srcX & 3, ty, _monochromePatterns, _internalPalette);
		ty = (ty + 1) & 7;
		src += pitch;
	}

	g_system->copyRectToScreen(_compositeBuffer, (w << 1) * _pixelSize, destX << 1, destY << 1, w << 1, h << 1);
}

void SCI0_CGABWDriver::replaceCursor(const void *cursor, uint w, uint h, int hotspotX, int hotspotY, uint32 keycolor) {
	GFXDRV_ASSERT_READY;
	// Instead of implementing the original cursor rendering code, we rely on the 8 bit cursor that
	// has already been generated by the engine. We simply convert the colors as needed and scale the cursor...
	assert(keycolor == 1);
	keycolor = 0x0f;
	w <<= 1;
	const byte *s = reinterpret_cast<const byte*>(cursor);
	byte *d0 = _compositeBuffer;
	byte *d1 = _compositeBuffer + w;

	for (uint i = 0; i < h; ++i) {
		for (uint ii = 0; ii < w; ++ii) {
			*d0++ = *d1++ = *s ? (*s ^ 0x0e) : 0;
			if (ii & 1)
				++s;
		}
		d0 += w;
		d1 += w;
	}

	CursorMan.replaceCursor(_compositeBuffer, w, h << 1, hotspotX << 1, hotspotY << 1, keycolor);
}

Common::Point SCI0_CGABWDriver::getMousePos() const {
	Common::Point res = GfxDriver::getMousePos();
	res.x >>= 1;
	res.y >>= 1;
	return res;
}

void SCI0_CGABWDriver::setMousePos(const Common::Point &pos) const {
	g_system->warpMouse(pos.x << 1, pos.y << 1);
}

void SCI0_CGABWDriver::setShakePos(int shakeXOffset, int shakeYOffset) const {
	g_system->setShakePos(shakeXOffset << 1, shakeYOffset << 1);
}

void SCI0_CGABWDriver::clearRect(const Common::Rect &r) const {
	Common::Rect r2(r.left << 1, r.top << 1, r.right << 1, r.bottom << 1);
	GfxDriver::clearRect(r2);
}

Common::Point SCI0_CGABWDriver::getRealCoords(Common::Point &pos) const {
	return Common::Point(pos.x << 1, pos.y << 1);
}

template <typename T> void cgabwRenderLine_v1(byte *&dst, const byte *src, int w, int tx, int ty, const byte *patterns, const byte *pal) {
	const T *p = reinterpret_cast<const T*>(pal);
	const uint16 *patterns16 = reinterpret_cast<const uint16*>(patterns);
	T *d1 = reinterpret_cast<T*>(dst);
	T *d2 = d1 + (w << 1);
	w >>= 1;

	for (int i = 0; i < w; ++i) {
		uint16 pt = patterns16[((src[0] & 0x0f) << 4) | (src[1] & 0x0f)];
		src += 2;
		byte sh = (ty & 3) << 1;
		byte lo = ((pt & 0xff) >> sh) | ((pt & 0xff) << (8 - sh));
		byte hi = (pt >> (8 + sh)) | ((pt >> 8) << (8 - sh));
		if (sizeof(T) == 1) {
			*d1++ = *d2++ = ((lo >> (6 - (tx << 1))) >> 1) & 1;
			*d1++ = *d2++ = (lo >> (6 - (tx << 1))) & 1;
			*d1++ = *d2++ = ((hi >> (4 - (tx << 1))) >> 1) & 1;
			*d1++ = *d2++ = (hi >> (4 - (tx << 1))) & 1;
		} else {
			*d1++ = *d2++ = p[((lo >> (6 - (tx << 1))) >> 1) & 1];
			*d1++ = *d2++ = p[(lo >> (6 - (tx << 1))) & 1];
			*d1++ = *d2++ = p[((hi >> (4 - (tx << 1))) >> 1) & 1];
			*d1++ = *d2++ = p[(hi >> (4 - (tx << 1))) & 1];
		}
		tx ^= 2;
	}

	dst = reinterpret_cast<byte*>(d2);
}

template <typename T> void cgabwRenderLine_v2(byte *&dst, const byte *src, int w, int tx, int ty, const byte *patterns, const byte *pal) {
	const T *p = reinterpret_cast<const T*>(pal);
	T *d1 = reinterpret_cast<T*>(dst);
	T *d2 = d1 + (w << 1);

	for (int i = 0; i < w; ++i) {
		byte pt = patterns[((*src++ & 0x0f) << 3) + ty] >> (6 - (tx << 1));
		if (sizeof(T) == 1) {
			*d1++ = *d2++ = (pt >> 1) & 1;
			*d1++ = *d2++ = pt & 1;
		} else {
			*d1++ = *d2++ = p[(pt >> 1) & 1];
			*d1++ = *d2++ = p[pt & 1];
		}
		tx = (tx + 1) & 3;
	}

	dst = reinterpret_cast<byte*>(d2);
}

void SCI0_CGABWDriver::setupRenderProc() {
	static const LineProc lineProcs[] = {
		&cgabwRenderLine_v1<byte>,
		&cgabwRenderLine_v1<uint16>,
		&cgabwRenderLine_v1<uint32>,
		&cgabwRenderLine_v2<byte>,
		&cgabwRenderLine_v2<uint16>,
		&cgabwRenderLine_v2<uint32>
	};

	int t = _pixelSize >> 1;
	if (!_earlyVersion)
		t += 3;

	assert(t < ARRAYSIZE(lineProcs));
	_renderLine = lineProcs[t];
}

const char *SCI0_CGABWDriver::_driverFiles[2] = { "CGA320BW.DRV", "CGA320M.DRV" };

SCI_GFXDRV_VALIDATE_IMPL(SCI0_CGABW)

GfxDriver *SCI0_CGABWDriver_create(int rgbRendering, ...) {
	static const uint32 monochromeColors[] = { 0xffbf66, 0x66ff66, 0xffffff };
	va_list args;
	va_start(args, rgbRendering);
	int config = CLIP<int>(va_arg(args, int), 0, ARRAYSIZE(monochromeColors) - 1);
	va_end(args);

	return new SCI0_CGABWDriver(monochromeColors[config], rgbRendering != 0);
}

} // End of namespace Sci
