/* SPDX-License-Identifier: LGPL-2.1-or-later */
#pragma once

#if HAVE_BZIP2
#include <bzlib.h>
#endif
#include <lzma.h>
#include <zlib.h>
#if HAVE_ZSTD
#include <zstd.h>
#endif

#include "shared-forward.h"

typedef enum ImportCompressType {
        IMPORT_COMPRESS_UNKNOWN,
        IMPORT_COMPRESS_UNCOMPRESSED,
        IMPORT_COMPRESS_XZ,
        IMPORT_COMPRESS_GZIP,
        IMPORT_COMPRESS_BZIP2,
        IMPORT_COMPRESS_ZSTD,
        _IMPORT_COMPRESS_TYPE_MAX,
        _IMPORT_COMPRESS_TYPE_INVALID = -EINVAL,
} ImportCompressType;

typedef struct ImportCompress {
        ImportCompressType type;
        bool encoding;
        union {
                lzma_stream xz;
                z_stream gzip;
#if HAVE_BZIP2
                bz_stream bzip2;
#endif
#if HAVE_ZSTD
                ZSTD_CCtx *c_zstd;
                ZSTD_DCtx *d_zstd;
#endif
        };
} ImportCompress;

typedef int (*ImportCompressCallback)(const void *data, size_t size, void *userdata);

void import_compress_free(ImportCompress *c);

int import_uncompress_detect(ImportCompress *c, const void *data, size_t size);
void import_uncompress_force_off(ImportCompress *c);
int import_uncompress(ImportCompress *c, const void *data, size_t size, ImportCompressCallback callback, void *userdata);

int import_compress_init(ImportCompress *c, ImportCompressType t);
int import_compress(ImportCompress *c, const void *data, size_t size, void **buffer, size_t *buffer_size, size_t *buffer_allocated);
int import_compress_finish(ImportCompress *c, void **buffer, size_t *buffer_size, size_t *buffer_allocated);

DECLARE_STRING_TABLE_LOOKUP(import_compress_type, ImportCompressType);
