/*********************************************************
 * Copyright (c) 2011-2016,2023 VMware, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation version 2.1 and no later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the Lesser GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA.
 *
 *********************************************************/

#ifndef _VGAUTHPROTO_H_
#define _VGAUTHPROTO_H_

#include "VGAuthError.h"

/* clang-format off */

/*
 * @file VGAuthProto.h
 *
 * Defines the wire protocol between the VGAuth client library
 * and the keystore/ticket service.
 */

/*
 * All requests generate replies.
 *
 * Per-operation comments below document the sucess case; the failure
 * case would have the same request data, but the same fixed error
 * format is returned for any operation.  Other common data (sequence number,
 * the request name) is also implied.
 *
 * If a request causes an error, the error code and an error message
 * are returned.
 *
 * If a request is successful, any response is returned.
 *
 * All requests contain a sequnce number, generated by the client.
 * The response or error returned from the service will contain this
 * sequence number.
 *
 * Note that 'superuser' below means root/system to distinguish
 * from a root certificate.
 *
 */


/*
 * The name of the public pipe used to start the initial handshake.
 */
#ifdef _WIN32
#define  SERVICE_PUBLIC_PIPE_NAME   "\\\\.\\pipe\\vgauth-service"
#else
#define  SERVICE_PUBLIC_PIPE_NAME   "/var/run/vmware/guestServicePipe"
#endif

#ifdef _WIN32
#define  SUPERUSER_NAME "system"
#else
#define  SUPERUSER_NAME "root"
#endif


/*
 * XXX For readbility/commentary purposes, all element names are
 * on the long/descriptive side.  If space becomes an issue, they
 * can be squashed.
 *
 * Format strings are #define here to avoid a bunch of duplicate
 * 'static char *' if this header is included in many places.
 * Of course, that means if they get used in multiple spots, we have
 * waste again.  TBD which makes the most sense.  Another benefit of
 * #defines is that they can easily be combined.
 *
 */

#define VGAUTH_XML_PREAMBLE "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>"

/*
 * Common tags.
 */
#define VGAUTH_PROTOCOL_VERSION "1"

#define VGAUTH_REQUESTNAME_ELEMENT_NAME "requestName"
#define VGAUTH_SEQUENCENO_ELEMENT_NAME "sequenceNumber"
#define VGAUTH_USERNAME_ELEMENT_NAME "userName"
#define VGAUTH_ERRORCODE_ELEMENT_NAME "errorCode"
#define VGAUTH_ERRORMSG_ELEMENT_NAME "errorMsg"
#define VGAUTH_TICKET_ELEMENT_NAME "ticket"
#define VGAUTH_PID_ELEMENT_NAME "pid"
#define VGAUTH_TOKEN_ELEMENT_NAME "token"
#define VGAUTH_SIGALGORITHM_ELEMENT_NAME "sigAlg"
#define VGAUTH_DATA_ELEMENT_NAME "data"
#define VGAUTH_SAMLTOKEN_ELEMENT_NAME "samlToken"
#define VGAUTH_CHALLENGE_EVENT_ELEMENT_NAME "challengeEvent"
#define VGAUTH_ALIASINFO_ELEMENT_NAME "aliasInfo"
#define VGAUTH_MAPPEDALIASES_ELEMENT_NAME "mappedAliases"
#define VGAUTH_SUBJECTS_ELEMENT_NAME "subjects"
#define VGAUTH_SUBJECT_ELEMENT_NAME "subject"
#define VGAUTH_ANYSUBJECT_ELEMENT_NAME "anySubject"
#define VGAUTH_COMMENT_ELEMENT_NAME "comment"
#define VGAUTH_ALIAS_ELEMENT_NAME "alias"
#define VGAUTH_VALIDATE_ONLY_ELEMENT_NAME "validateOnly"
#define VGAUTH_HOST_VERIFIED_ELEMENT_NAME "hostVerified"

/*
 * Complex types
 */


#define  VGAUTH_PEMCERT_ELEMENT_NAME "pemCert"

#define VGAUTH_USERNAME_FORMAT \
       "<"VGAUTH_USERNAME_ELEMENT_NAME">%s</"VGAUTH_USERNAME_ELEMENT_NAME">"

#define VGAUTH_PEMCERT_FORMAT \
       "<"VGAUTH_PEMCERT_ELEMENT_NAME">%s</"VGAUTH_PEMCERT_ELEMENT_NAME">"

#define VGAUTH_SUBJECT_FORMAT \
   "<"VGAUTH_SUBJECT_ELEMENT_NAME">%s</"VGAUTH_SUBJECT_ELEMENT_NAME">"

#define VGAUTH_ANYSUBJECT_FORMAT \
   "<"VGAUTH_ANYSUBJECT_ELEMENT_NAME"/>"

#define VGAUTH_COMMENT_FORMAT \
   "<"VGAUTH_COMMENT_ELEMENT_NAME">%s</"VGAUTH_COMMENT_ELEMENT_NAME">"

/*
 * A NamedAliasInfo contains a subject and a comment.
 */
#define VGAUTH_NAMEDALIASINFO_FORMAT \
   "<"VGAUTH_ALIASINFO_ELEMENT_NAME">" \
      VGAUTH_SUBJECT_FORMAT \
      VGAUTH_COMMENT_FORMAT \
   "</"VGAUTH_ALIASINFO_ELEMENT_NAME">"

/*
 * An AnyAliasInfo contains an anySubject tag and a comment.
 */
#define VGAUTH_ANYALIASINFO_FORMAT \
   "<"VGAUTH_ALIASINFO_ELEMENT_NAME">" \
      VGAUTH_ANYSUBJECT_FORMAT \
      VGAUTH_COMMENT_FORMAT \
   "</"VGAUTH_ALIASINFO_ELEMENT_NAME">"

/*
 * An Alias contains a pemCert and 1 or more of a mix of
 * AnyAliasInfo and NamedAliasInfo
 */
#define VGAUTH_ALIAS_FORMAT_START \
   "<"VGAUTH_ALIAS_ELEMENT_NAME">" \
   VGAUTH_PEMCERT_FORMAT

#define VGAUTH_ALIAS_FORMAT_END \
   "</"VGAUTH_ALIAS_ELEMENT_NAME">" \

/*
 * A MappedAlias contains a userName, pemCert and an array of a
 * NamedSubject or AnySubject
 */
#define VGAUTH_MAPPEDALIASES_FORMAT_START \
   "<"VGAUTH_MAPPEDALIASES_ELEMENT_NAME">" \
   VGAUTH_USERNAME_FORMAT \
   VGAUTH_PEMCERT_FORMAT \
   "<"VGAUTH_SUBJECTS_ELEMENT_NAME">"

#define VGAUTH_MAPPEDALIASES_FORMAT_END \
   "</"VGAUTH_SUBJECTS_ELEMENT_NAME">" \
   "</"VGAUTH_MAPPEDALIASES_ELEMENT_NAME">" \

/*
 * A UserHandleInfo contains a type, and optional data determined by that type.
 */
#define VGAUTH_USERHANDLEINFO_ELEMENT_NAME "userHandleInfo"
#define VGAUTH_USERHANDLETYPE_ELEMENT_NAME "userHandleType"
#define VGAUTH_USERHANDLESAMLINFO_ELEMENT_NAME "samlInfo"
#define VGAUTH_USERHANDLESAMLSUBJECT_ELEMENT_NAME "samlSubject"

/*
 * Possible userHandle types
 */
#define VGAUTH_USERHANDLE_TYPE_NAMEPASSWORD "namePassword"
#define VGAUTH_USERHANDLE_TYPE_SSPI         "sspi"
#define VGAUTH_USERHANDLE_TYPE_SAML         "saml"
#define VGAUTH_USERHANDLE_TYPE_SAML_INFO_ONLY    "samlInfoOnly"

#define VGAUTH_USERHANDLESAMLINFO_FORMAT_START \
   "<"VGAUTH_USERHANDLESAMLINFO_ELEMENT_NAME">"\
   "<"VGAUTH_USERHANDLESAMLSUBJECT_ELEMENT_NAME">" \
      "%s" \
   "</"VGAUTH_USERHANDLESAMLSUBJECT_ELEMENT_NAME">"
/*
 * Followed by one of VGAUTH_NAMEDALIASINFO_FORMAT or
 * VGAUTH_ANYALIASINFO_FORMAT.
 */
#define VGAUTH_USERHANDLESAMLINFO_FORMAT_END \
   "</"VGAUTH_USERHANDLESAMLINFO_ELEMENT_NAME">"\


#define VGAUTH_USERHANDLEINFO_FORMAT_START \
   "<"VGAUTH_USERHANDLEINFO_ELEMENT_NAME">"\
   "<"VGAUTH_USERHANDLETYPE_ELEMENT_NAME">" \
   "%s" \
   "</"VGAUTH_USERHANDLETYPE_ELEMENT_NAME">"
/*
 * Followed by VGAUTH_USERHANDLESAMLINFO_FORMAT_{START,END} if the type is
 * VGAUTH_USERHANDLE_TYPE_SAML.
 */

#define VGAUTH_USERHANDLEINFO_FORMAT_END \
   "</"VGAUTH_USERHANDLEINFO_ELEMENT_NAME">"\

/** Requests */

#define VGAUTH_REQUEST_ELEMENT_NAME "request"
#define VGAUTH_REPLY_ELEMENT_NAME "reply"

/*
 * Request envelope
 */
#define VGAUTH_REQUEST_FORMAT_START \
   VGAUTH_XML_PREAMBLE "<"VGAUTH_REQUEST_ELEMENT_NAME">" \
   "<"VGAUTH_SEQUENCENO_ELEMENT_NAME">%d</"VGAUTH_SEQUENCENO_ELEMENT_NAME">"

#define VGAUTH_REQUEST_FORMAT_END "</"VGAUTH_REQUEST_ELEMENT_NAME">"

/*
 * Reply envelope
 */
#define VGAUTH_REPLY_FORMAT_START \
   VGAUTH_XML_PREAMBLE "<"VGAUTH_REPLY_ELEMENT_NAME">" \
   "<"VGAUTH_SEQUENCENO_ELEMENT_NAME">%d</"VGAUTH_SEQUENCENO_ELEMENT_NAME">"

#define VGAUTH_REPLY_FORMAT_END "</"VGAUTH_REPLY_ELEMENT_NAME">"


/*
 * Error response.
 */
#define VGAUTH_ERROR_FORMAT \
   VGAUTH_XML_PREAMBLE \
   "<"VGAUTH_REPLY_ELEMENT_NAME">" \
       "<"VGAUTH_SEQUENCENO_ELEMENT_NAME">%d</"VGAUTH_SEQUENCENO_ELEMENT_NAME">" \
       "<"VGAUTH_ERRORCODE_ELEMENT_NAME">"VGAUTHERR_FMT64"</"VGAUTH_ERRORCODE_ELEMENT_NAME">" \
       "<"VGAUTH_ERRORMSG_ELEMENT_NAME">%s</"VGAUTH_ERRORMSG_ELEMENT_NAME">" \
   "</"VGAUTH_REPLY_ELEMENT_NAME">"


/*
 * SessionRequest should be sent as the first request upon
 * connection.  It exchanges the version number of the client and service,
 * and request that the service set up a private and secure connection
 * for the supplied user.
 *
 * Permissions: any user
 *
 * Request:
 *    version #
 *    username
 * =>
 *    version #
 *    pipe name
 *
 */
#define VGAUTH_REQUESTSESSION_ELEMENT_NAME   "SessionRequest"
#define VGAUTH_VERSION_ELEMENT_NAME "version"

#define VGAUTH_SESSION_REQUEST_FORMAT \
   VGAUTH_REQUEST_FORMAT_START \
       "<"VGAUTH_REQUESTNAME_ELEMENT_NAME">"VGAUTH_REQUESTSESSION_ELEMENT_NAME"</"VGAUTH_REQUESTNAME_ELEMENT_NAME">" \
       "<"VGAUTH_VERSION_ELEMENT_NAME">"VGAUTH_PROTOCOL_VERSION"</"VGAUTH_VERSION_ELEMENT_NAME">" \
       "<"VGAUTH_USERNAME_ELEMENT_NAME">%s</"VGAUTH_USERNAME_ELEMENT_NAME">" \
   VGAUTH_REQUEST_FORMAT_END

#define  VGAUTH_PIPENAME_ELEMENT_NAME "pipeName"

#define VGAUTH_SESSION_REPLY_FORMAT \
   VGAUTH_REPLY_FORMAT_START \
       "<"VGAUTH_VERSION_ELEMENT_NAME">"VGAUTH_PROTOCOL_VERSION"</"VGAUTH_VERSION_ELEMENT_NAME">" \
       "<"VGAUTH_PIPENAME_ELEMENT_NAME">%s</"VGAUTH_PIPENAME_ELEMENT_NAME">" \
    VGAUTH_REPLY_FORMAT_END


/*
 * Connection
 *
 * Simple handshake for the secure channel that deals with a specified
 * user.  The first request sent after a connection to the namedPipe
 * coming back from SessionRequest.
 *
 * Permissions: expected to be done by the user owning the pipe
 *
 * Request:
 *   pid - The client process ID on Windows. On other platforms, it can be
 *         an empty string.
 *   On Windows vista and later version Windows, the server can call
 *   GetNamedPipeClientProcessId() to retrieve the client PID.
 *   However, the function is not available on an earlier Windows system
 *   such as XP.
 *   Therefore, the client sends the pid in this request, and the service shall
 *   verify the it by impersonating the named pipe client and opening the
 *   process identified by the pid.
 * =>
 *   (empty)
 */
#define VGAUTH_REQUESTCONNECT_ELEMENT_NAME   "Connect"

#define VGAUTH_CONNECT_REQUEST_FORMAT \
   VGAUTH_REQUEST_FORMAT_START \
       "<"VGAUTH_REQUESTNAME_ELEMENT_NAME">"VGAUTH_REQUESTCONNECT_ELEMENT_NAME"</"VGAUTH_REQUESTNAME_ELEMENT_NAME">" \
       "<"VGAUTH_PID_ELEMENT_NAME">%s</"VGAUTH_PID_ELEMENT_NAME">" \
    VGAUTH_REQUEST_FORMAT_END

#define VGAUTH_CONNECT_REPLY_FORMAT \
    VGAUTH_REPLY_FORMAT_START \
   "<"VGAUTH_CHALLENGE_EVENT_ELEMENT_NAME">%s</"VGAUTH_CHALLENGE_EVENT_ELEMENT_NAME">" \
    VGAUTH_REPLY_FORMAT_END


/*
 * AddAlias
 *
 * Adds a certificate and subject to the specified user's alias store.
 *
 * This will fail with a permission error if the request doesn't
 * come over a namedPipe owned by superuser or 'user'.
 *
 * Request:
 *   userName
 *   addMappedLink (boolean)
 *   pemCert
 *   either a NamedAliasInfo or an AnyAliasInfo
 * =>
 *   (empty)
 */

#define VGAUTH_REQUESTADDALIAS_ELEMENT_NAME  "AddAlias"

#define VGAUTH_ADDMAPPEDLINK_ELEMENT_NAME "addMappedLink"

#define VGAUTH_ADDALIAS_REQUEST_FORMAT_START \
   VGAUTH_REQUEST_FORMAT_START \
       "<"VGAUTH_REQUESTNAME_ELEMENT_NAME">"VGAUTH_REQUESTADDALIAS_ELEMENT_NAME"</"VGAUTH_REQUESTNAME_ELEMENT_NAME">" \
       "<"VGAUTH_USERNAME_ELEMENT_NAME">%s</"VGAUTH_USERNAME_ELEMENT_NAME">" \
       "<"VGAUTH_ADDMAPPEDLINK_ELEMENT_NAME">%d</"VGAUTH_ADDMAPPEDLINK_ELEMENT_NAME">" \
       VGAUTH_PEMCERT_FORMAT

/*
 * Followed by either VGAUTH_NAMEDALIASINFO_FORMAT or VGAUTH_ANYALIASINFO_FORMAT
 */

#define VGAUTH_ADDALIAS_REQUEST_FORMAT_END \
   VGAUTH_REQUEST_FORMAT_END


#define VGAUTH_ADDALIAS_REPLY_FORMAT \
   VGAUTH_REPLY_FORMAT_START \
   VGAUTH_REPLY_FORMAT_END


/*
 * RemoveAlias
 *
 * Remove a Subject from the pemCert in the specified user's alias store.
 * Any mapped link is also removed.  If no Subject is specified,
 * entries for all Subjects associated with the cert will be removed.
 *
 * This will fail with a permission error if the request doesn't
 * come over a namedPipe owned by superuser or the owner of the ticket.
 *
 * Request:
 *   userName
 *   pemCertificate
 *   an optional AnySubject or NamedSubject.  If neither is specified, all
 *       Subjects and the pemCert are removed.
 * =>
 *   (empty)
 */

#define VGAUTH_REQUESTREMOVEALIAS_ELEMENT_NAME  "RemoveAlias"

#define VGAUTH_REMOVEALIAS_REQUEST_FORMAT_START \
   VGAUTH_REQUEST_FORMAT_START \
       "<"VGAUTH_REQUESTNAME_ELEMENT_NAME">"VGAUTH_REQUESTREMOVEALIAS_ELEMENT_NAME"</"VGAUTH_REQUESTNAME_ELEMENT_NAME">" \
       VGAUTH_USERNAME_FORMAT \
       VGAUTH_PEMCERT_FORMAT
/*
 * followed by either
 * VGAUTH_NAMEDSUBJECT_FORMAT or VGAUTH_ANYSUBJECT_FORMAT or nothing.
 */

#define VGAUTH_REMOVEALIAS_REQUEST_FORMAT_END \
   VGAUTH_REQUEST_FORMAT_END


#define VGAUTH_REMOVEALIAS_REPLY_FORMAT \
   VGAUTH_REPLY_FORMAT_START \
   VGAUTH_REPLY_FORMAT_END


/*
 * QueryUserAliases
 *
 * Queries the subjects and certificates from the specified user's alias store.
 *
 * This will fail with a permission error if the request doesn't
 * come over a namedPipe owned by superuser or 'user'.
 *
 * Request:
 *   userName
 * =>
 *   array of Alias
 */

#define VGAUTH_REQUESTQUERYALIASES_ELEMENT_NAME  "QueryAliases"

#define VGAUTH_QUERYALIASES_REQUEST_FORMAT \
   VGAUTH_REQUEST_FORMAT_START \
       "<"VGAUTH_REQUESTNAME_ELEMENT_NAME">"VGAUTH_REQUESTQUERYALIASES_ELEMENT_NAME"</"VGAUTH_REQUESTNAME_ELEMENT_NAME">" \
       "<"VGAUTH_USERNAME_ELEMENT_NAME">%s</"VGAUTH_USERNAME_ELEMENT_NAME">" \
   VGAUTH_REQUEST_FORMAT_END


#define VGAUTH_QUERYALIASES_REPLY_FORMAT_START \
   VGAUTH_REPLY_FORMAT_START

/*
 * Followed by 0 or more VGAUTH_ALIAS_FORMAT_START,
 * with 1 or more
 * VGAUTH_NAMEDALIASINFO_FORMAT or VGAUTH_ANYALIASINFO_FORMAT
 * Closed by VGAUTH_ALIAS_FORMAT_END
 */

#define VGAUTH_QUERYALIASES_REPLY_FORMAT_END \
   VGAUTH_REPLY_FORMAT_END


/*
 * QueryMappedAliases
 *
 * Queries all the certificate/subject pairs and associated users
 * from the mapping file.
 *
 * Permissions: any user
 *
 * Request:
 * =>
 *   array of pemCerts
 *   array of userName
 *   array of NameSubject or AnySubject
 */

#define VGAUTH_REQUESTQUERYMAPPEDALIASES_ELEMENT_NAME  "QueryMappedAliases"

#define VGAUTH_QUERYMAPPEDALIASES_REQUEST_FORMAT \
   VGAUTH_REQUEST_FORMAT_START \
       "<"VGAUTH_REQUESTNAME_ELEMENT_NAME">"VGAUTH_REQUESTQUERYMAPPEDALIASES_ELEMENT_NAME"</"VGAUTH_REQUESTNAME_ELEMENT_NAME">" \
   VGAUTH_REQUEST_FORMAT_END


#define VGAUTH_QUERYMAPPEDALIASES_REPLY_FORMAT_START \
   VGAUTH_REPLY_FORMAT_START

/*
 * Followed by an array of 0 or more VGAUTH_MAPPEDALIASES_FORMAT_START, which
 * contains a userName and pemCert, and an array of VGAUTH_SUBJECT_FORMAT
 * or VGAUTH_ANYSUBJECT_FORMAT, closed by VGAUTH_MAPPEDALIASES_FORMAT_END.
 */

#define VGAUTH_QUERYMAPPEDALIASES_REPLY_FORMAT_END \
   VGAUTH_REPLY_FORMAT_END


/*
 * CreateTicket
 *
 * Creates a ticket for the specified user.
 *
 * This will fail with a permission error if the request doesn't
 * come over a namedPipe owned by superuser or 'user'.
 *
 * Request:
 *   userName
 *   token - The access token that the service shall duplicate on Windows.
 *           On other platform, it can be an empty string.
 *           On Windows, the service duplicates the access token when creating a
 *           ticket object. Later the access token can be returned to the
 *           client upon a ValidateTicket request.
 *   userHandleInfo
 * =>
 *   Ticket
 */

#define VGAUTH_REQUESTCREATETICKET_ELEMENT_NAME  "CreateTicket"

#define VGAUTH_CREATETICKET_REQUEST_FORMAT_START \
   VGAUTH_REQUEST_FORMAT_START \
       "<"VGAUTH_REQUESTNAME_ELEMENT_NAME">"VGAUTH_REQUESTCREATETICKET_ELEMENT_NAME"</"VGAUTH_REQUESTNAME_ELEMENT_NAME">" \
       "<"VGAUTH_USERNAME_ELEMENT_NAME">%s</"VGAUTH_USERNAME_ELEMENT_NAME">" \
       "<"VGAUTH_TOKEN_ELEMENT_NAME">%s</"VGAUTH_TOKEN_ELEMENT_NAME">" \
       VGAUTH_USERHANDLEINFO_FORMAT_START \
/*
 * Followed by optional VGAUTH_USERHANDLESAMLINFO_FORMAT_START.
 */

#define VGAUTH_CREATETICKET_REQUEST_FORMAT_END \
   VGAUTH_USERHANDLEINFO_FORMAT_END \
   VGAUTH_REQUEST_FORMAT_END

#define VGAUTH_CREATETICKET_REPLY_FORMAT \
   VGAUTH_REPLY_FORMAT_START \
   "<"VGAUTH_TICKET_ELEMENT_NAME">%s</"VGAUTH_TICKET_ELEMENT_NAME">" \
   VGAUTH_REPLY_FORMAT_END


/*
 * ValidateTicket
 *
 * Validates a ticket.
 *
 * This will fail with a permission error if the ticket in the request doesn't
 * come over a namedPipe owned by superuser or 'user'.
 *
 * Request:
 *   Ticket
 * =>
 *   userName
 *   token - On Windows, the impersonation HANDLE, converted as a base 10
 *           number to a string, that represents the user that the ticket has
 *           authenticated the caller to. The service duplicates the
 *           impersonation HANDLE to the client process. The HANDLE value is
 *           only meaningful in the client process address space.
 *           On other platform, it is an empty string.
 *   userHandleInfo
 */

#define VGAUTH_REQUESTVALIDATETICKET_ELEMENT_NAME  "ValidateTicket"

#define VGAUTH_VALIDATETICKET_REQUEST_FORMAT \
   VGAUTH_REQUEST_FORMAT_START \
       "<"VGAUTH_REQUESTNAME_ELEMENT_NAME">"VGAUTH_REQUESTVALIDATETICKET_ELEMENT_NAME"</"VGAUTH_REQUESTNAME_ELEMENT_NAME">" \
      "<"VGAUTH_TICKET_ELEMENT_NAME">%s</"VGAUTH_TICKET_ELEMENT_NAME">" \
   VGAUTH_REQUEST_FORMAT_END

#define VGAUTH_VALIDATETICKET_REPLY_FORMAT_START \
   VGAUTH_REPLY_FORMAT_START \
       "<"VGAUTH_USERNAME_ELEMENT_NAME">%s</"VGAUTH_USERNAME_ELEMENT_NAME">" \
       "<"VGAUTH_TOKEN_ELEMENT_NAME">%s</"VGAUTH_TOKEN_ELEMENT_NAME">" \
       VGAUTH_USERHANDLEINFO_FORMAT_START
/*
 * Followed by an optional VGAUTH_USERHANDLESAMLINFO_FORMAT_{START,END}.
 */

#define VGAUTH_VALIDATETICKET_REPLY_FORMAT_END \
   VGAUTH_USERHANDLEINFO_FORMAT_END \
   VGAUTH_REPLY_FORMAT_END


/*
 * RevokeTicket
 *
 * Revokes a ticket.
 *
 * This will fail with a permission error if the ticket in the request doesn't
 * come over a namedPipe owned by superuser or 'user'.
 *
 * Request:
 *   Ticket
 * =>
 *   (empty)
 */

#define VGAUTH_REQUESTREVOKETICKET_ELEMENT_NAME  "RevokeTicket"

#define VGAUTH_REVOKETICKET_REQUEST_FORMAT \
   VGAUTH_REQUEST_FORMAT_START \
       "<"VGAUTH_REQUESTNAME_ELEMENT_NAME">"VGAUTH_REQUESTREVOKETICKET_ELEMENT_NAME"</"VGAUTH_REQUESTNAME_ELEMENT_NAME">" \
      "<"VGAUTH_TICKET_ELEMENT_NAME">%s</"VGAUTH_TICKET_ELEMENT_NAME">" \
   VGAUTH_REQUEST_FORMAT_END

#define VGAUTH_REVOKETICKET_REPLY_FORMAT \
   VGAUTH_REPLY_FORMAT_START \
   VGAUTH_REPLY_FORMAT_END


/*
 * ValidateSamlToken
 *
 * Validates a SAML Bearer token against the certificate store.
 *
 * Returns the user info for doing impersonation, as well as any attributes
 * associated with the certificates in the certstore used for validation.
 *
 * This will fail with a permission error if the ticket in the request doesn't
 * come over a namedPipe owned by superuser.
 *
 * If valdiateOnly is set, then on Windows the service only validates the
 * SAML token, and does not create an access token on Windows.  This
 * flag is ignored on *ix.
 *
 * If hostVerified is set, then the service will skip the signature
 * check in the SAML token.
 *
 * Request:
 *   SAML token
 *   user
 *   validateOnly (bool)
 *   hostVerified (bool)
 * =>
 *   user
 *   token (empty for non-Windows)
 *   userHandleInfo
 */

#define VGAUTH_REQUESTVALIDATESAMLBEARERTOKEN_ELEMENT_NAME "ValidateSamlBToken"

#define VGAUTH_VALIDATESAMLBEARERTOKEN_REQUEST_FORMAT \
   VGAUTH_REQUEST_FORMAT_START \
      "<"VGAUTH_REQUESTNAME_ELEMENT_NAME">"VGAUTH_REQUESTVALIDATESAMLBEARERTOKEN_ELEMENT_NAME"</"VGAUTH_REQUESTNAME_ELEMENT_NAME">" \
      "<"VGAUTH_SAMLTOKEN_ELEMENT_NAME">%s</"VGAUTH_SAMLTOKEN_ELEMENT_NAME">" \
      "<"VGAUTH_USERNAME_ELEMENT_NAME">%s</"VGAUTH_USERNAME_ELEMENT_NAME">" \
      "<"VGAUTH_VALIDATE_ONLY_ELEMENT_NAME">%s</"VGAUTH_VALIDATE_ONLY_ELEMENT_NAME">" \
      "<"VGAUTH_HOST_VERIFIED_ELEMENT_NAME">%s</"VGAUTH_HOST_VERIFIED_ELEMENT_NAME">" \
   VGAUTH_REQUEST_FORMAT_END

#define VGAUTH_VALIDATESAMLBEARERTOKEN_REPLY_FORMAT_START \
   VGAUTH_REPLY_FORMAT_START \
       "<"VGAUTH_USERNAME_ELEMENT_NAME">%s</"VGAUTH_USERNAME_ELEMENT_NAME">" \
       "<"VGAUTH_TOKEN_ELEMENT_NAME">%s</"VGAUTH_TOKEN_ELEMENT_NAME">" \
       VGAUTH_USERHANDLESAMLINFO_FORMAT_START
/*
 * Followed by the VGAUTH_NAMEDALIASINFO_FORMAT or
 * VGAUTH_ANYALIASINFO_FORMAT.
 */

#define VGAUTH_VALIDATESAMLBEARERTOKEN_REPLY_FORMAT_END \
   VGAUTH_USERHANDLESAMLINFO_FORMAT_END \
   VGAUTH_REPLY_FORMAT_END

/* clang-format on */
#endif   // _VGAUTHPROTO_H_
