/**********************************************************************
 * $read_timeval example -- PLI application using VPI routines
 *
 * C source to read time variable values of Verilog objects.
 *
 * Usage: $read_timeval(<time_variable_name>);
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *********************************************************************/

#include <stdlib.h>    /* ANSI C standard library */
#include <stdio.h>     /* ANSI C standard input/output library */
#include "vpi_user.h"  /* IEEE 1364 PLI VPI routine library  */
#include "veriuser.h"  /* IEEE 1364 PLI TF routine library    
                          (using TF routines for simulation control) */

/* prototypes of routines in this PLI application */
int  PLIbook_ReadTimeVal_calltf(), PLIbook_ReadTimeVal_compiletf();
char decode_bit_value();

/**********************************************************************
 * VPI Registration Data
 *********************************************************************/
void PLIbook_ReadTimeVal_register()
{
  s_vpi_systf_data tf_data;
  tf_data.type      = vpiSysTask;
  tf_data.tfname    = "$read_timeval";
  tf_data.calltf    = PLIbook_ReadTimeVal_calltf;
  tf_data.compiletf = PLIbook_ReadTimeVal_compiletf;
  tf_data.sizetf    = NULL;

  vpi_register_systf(&tf_data);
}
/*********************************************************************/

/* include utility routines to work with tfargs */
#include "vpi_utilities.c"

/**********************************************************************
 * compiletf application
 *********************************************************************/
int PLIbook_ReadTimeVal_compiletf(char *user_data)
{
  vpiHandle systf_h, arg_itr, arg_h;
  int       arg_type;
  
  /* obtain a handle to the system task instance */
  systf_h = vpi_handle(vpiSysTfCall, NULL);
  if (systf_h == NULL) {
    vpi_printf("ERROR: $read_timeval failed to obtain systf handle\n");
    tf_dofinish(); /* abort simulation */
    return(0);
  }

  /* obtain handles to system task arguments */
  arg_itr = vpi_iterate(vpiArgument, systf_h);
  if (arg_itr == NULL) {
    vpi_printf("ERROR: $read_timeval requires 1 argument\n");
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  
  /* check the type of object in system task arguments */
  arg_h = vpi_scan(arg_itr);
  arg_type = vpi_get(vpiType, arg_h);
  if (arg_type != vpiTimeVar) {
    vpi_printf("ERROR: $read_timeval arg must be a time variable\n");
    vpi_free_object(arg_itr); /* free iterator memory */
    tf_dofinish(); /* abort simulation */
    return(0);
  }

  /* check that there are no more system task arguments */
  arg_h = vpi_scan(arg_itr);
  if (arg_h != NULL) {
    vpi_printf("ERROR: $read_timeval can only have 1 argument\n");
    vpi_free_object(arg_itr); /* free iterator memory */
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  return(0);
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
int PLIbook_ReadTimeVal_calltf(char *user_data) 
{
  vpiHandle   systf_h, arg_itr, arg_h, timevar_h;
  s_vpi_value timevar_val;    /* structure to receive variable value */

  /* obtain a handle to the system task instance */
  systf_h = vpi_handle(vpiSysTfCall, NULL);

  /* obtain handle to system task argument
     compiletf has already verified only 1 arg with correct type */
  arg_itr = vpi_iterate(vpiArgument, systf_h);
  timevar_h = vpi_scan(arg_itr);
  vpi_free_object(arg_itr);  /* free iterator memory */

  timevar_val.format = vpiTimeVal;         /* set value format field */

  vpi_get_value(timevar_h, &timevar_val);   /* read variable's value */

  vpi_printf("\nTime Variable %s:  ", vpi_get_str(vpiName, timevar_h));
  vpi_printf("  hi-word/lo-word = %d/%d\n\n",
             timevar_val.value.time->high,
             timevar_val.value.time->low);
  return(0);
}
/*********************************************************************/
