use hir::HasSource;
use syntax::{
    Edition,
    ast::{self, AstNode, make},
    syntax_editor::{Position, SyntaxEditor},
};

use crate::{
    AssistId,
    assist_context::{AssistContext, Assists},
    utils::{
        DefaultMethods, IgnoreAssocItems, add_trait_assoc_items_to_impl, filter_assoc_items,
        gen_trait_fn_body,
    },
};

// Assist: add_impl_missing_members
//
// Adds scaffold for required impl members.
//
// ```
// trait Trait<T> {
//     type X;
//     fn foo(&self) -> T;
//     fn bar(&self) {}
// }
//
// impl Trait<u32> for () {$0
//
// }
// ```
// ->
// ```
// trait Trait<T> {
//     type X;
//     fn foo(&self) -> T;
//     fn bar(&self) {}
// }
//
// impl Trait<u32> for () {
//     $0type X;
//
//     fn foo(&self) -> u32 {
//         todo!()
//     }
// }
// ```
pub(crate) fn add_missing_impl_members(acc: &mut Assists, ctx: &AssistContext<'_>) -> Option<()> {
    add_missing_impl_members_inner(
        acc,
        ctx,
        DefaultMethods::No,
        IgnoreAssocItems::DocHiddenAttrPresent,
        "add_impl_missing_members",
        "Implement missing members",
    )
}

// Assist: add_impl_default_members
//
// Adds scaffold for overriding default impl members.
//
// ```
// trait Trait {
//     type X;
//     fn foo(&self);
//     fn bar(&self) {}
// }
//
// impl Trait for () {
//     type X = ();
//     fn foo(&self) {}$0
// }
// ```
// ->
// ```
// trait Trait {
//     type X;
//     fn foo(&self);
//     fn bar(&self) {}
// }
//
// impl Trait for () {
//     type X = ();
//     fn foo(&self) {}
//
//     $0fn bar(&self) {}
// }
// ```
pub(crate) fn add_missing_default_members(
    acc: &mut Assists,
    ctx: &AssistContext<'_>,
) -> Option<()> {
    add_missing_impl_members_inner(
        acc,
        ctx,
        DefaultMethods::Only,
        IgnoreAssocItems::DocHiddenAttrPresent,
        "add_impl_default_members",
        "Implement default members",
    )
}

fn add_missing_impl_members_inner(
    acc: &mut Assists,
    ctx: &AssistContext<'_>,
    mode: DefaultMethods,
    ignore_items: IgnoreAssocItems,
    assist_id: &'static str,
    label: &'static str,
) -> Option<()> {
    let _p = tracing::info_span!("add_missing_impl_members_inner").entered();
    let impl_def = ctx.find_node_at_offset::<ast::Impl>()?;
    let impl_ = ctx.sema.to_def(&impl_def)?;

    if ctx.token_at_offset().all(|t| {
        t.parent_ancestors()
            .take_while(|node| node != impl_def.syntax())
            .any(|s| ast::BlockExpr::can_cast(s.kind()) || ast::ParamList::can_cast(s.kind()))
    }) {
        return None;
    }

    let target_scope = ctx.sema.scope(impl_def.syntax())?;
    let trait_ref = impl_.trait_ref(ctx.db())?;
    let trait_ = trait_ref.trait_();

    let mut ign_item = ignore_items;

    if let IgnoreAssocItems::DocHiddenAttrPresent = ignore_items {
        // Relax condition for local crates.
        let db = ctx.db();
        if trait_.module(db).krate().origin(db).is_local() {
            ign_item = IgnoreAssocItems::No;
        }
    }

    let missing_items = filter_assoc_items(
        &ctx.sema,
        &ide_db::traits::get_missing_assoc_items(&ctx.sema, &impl_def),
        mode,
        ign_item,
    );

    if missing_items.is_empty() {
        return None;
    }

    let target = impl_def.syntax().text_range();
    acc.add(AssistId::quick_fix(assist_id), label, target, |edit| {
        let new_item = add_trait_assoc_items_to_impl(
            &ctx.sema,
            ctx.config,
            &missing_items,
            trait_,
            &impl_def,
            &target_scope,
        );

        let Some((first_new_item, other_items)) = new_item.split_first() else {
            return;
        };

        let mut first_new_item = if let DefaultMethods::No = mode
            && let ast::AssocItem::Fn(func) = &first_new_item
            && let Some(body) = try_gen_trait_body(
                ctx,
                func,
                trait_ref,
                &impl_def,
                target_scope.krate().edition(ctx.sema.db),
            )
            && let Some(func_body) = func.body()
        {
            let mut func_editor = SyntaxEditor::new(first_new_item.syntax().clone_subtree());
            func_editor.replace(func_body.syntax(), body.syntax());
            ast::AssocItem::cast(func_editor.finish().new_root().clone())
        } else {
            Some(first_new_item.clone())
        };

        let new_assoc_items = first_new_item
            .clone()
            .into_iter()
            .chain(other_items.iter().cloned())
            .map(either::Either::Right)
            .collect::<Vec<_>>();

        let mut editor = edit.make_editor(impl_def.syntax());
        if let Some(assoc_item_list) = impl_def.assoc_item_list() {
            let items = new_assoc_items.into_iter().filter_map(either::Either::right).collect();
            assoc_item_list.add_items(&mut editor, items);
        } else {
            let assoc_item_list = make::assoc_item_list(Some(new_assoc_items)).clone_for_update();
            editor.insert_all(
                Position::after(impl_def.syntax()),
                vec![make::tokens::whitespace(" ").into(), assoc_item_list.syntax().clone().into()],
            );
            first_new_item = assoc_item_list.assoc_items().next();
        }

        if let Some(cap) = ctx.config.snippet_cap {
            let mut placeholder = None;
            if let DefaultMethods::No = mode {
                if let Some(ast::AssocItem::Fn(func)) = &first_new_item {
                    if let Some(m) = func.syntax().descendants().find_map(ast::MacroCall::cast)
                        && m.syntax().text() == "todo!()"
                    {
                        placeholder = Some(m);
                    }
                }
            }

            if let Some(macro_call) = placeholder {
                let placeholder = edit.make_placeholder_snippet(cap);
                editor.add_annotation(macro_call.syntax(), placeholder);
            } else if let Some(first_new_item) = first_new_item {
                let tabstop = edit.make_tabstop_before(cap);
                editor.add_annotation(first_new_item.syntax(), tabstop);
            };
        };
        edit.add_file_edits(ctx.vfs_file_id(), editor);
    })
}

fn try_gen_trait_body(
    ctx: &AssistContext<'_>,
    func: &ast::Fn,
    trait_ref: hir::TraitRef<'_>,
    impl_def: &ast::Impl,
    edition: Edition,
) -> Option<ast::BlockExpr> {
    let trait_path = make::ext::ident_path(
        &trait_ref.trait_().name(ctx.db()).display(ctx.db(), edition).to_string(),
    );
    let hir_ty = ctx.sema.resolve_type(&impl_def.self_ty()?)?;
    let adt = hir_ty.as_adt()?.source(ctx.db())?;
    gen_trait_fn_body(func, &trait_path, &adt.value, Some(trait_ref))
}

#[cfg(test)]
mod tests {
    use crate::tests::{check_assist, check_assist_not_applicable};

    use super::*;

    #[test]
    fn test_add_missing_impl_members() {
        check_assist(
            add_missing_impl_members,
            r#"
trait Foo {
    type Output;

    const CONST: usize = 42;
    const CONST_2: i32;

    fn foo(&self);
    fn bar(&self);
    fn baz(&self);
}

struct S;

impl Foo for S {
    fn bar(&self) {}
$0
}"#,
            r#"
trait Foo {
    type Output;

    const CONST: usize = 42;
    const CONST_2: i32;

    fn foo(&self);
    fn bar(&self);
    fn baz(&self);
}

struct S;

impl Foo for S {
    fn bar(&self) {}

    $0type Output;

    const CONST_2: i32;

    fn foo(&self) {
        todo!()
    }

    fn baz(&self) {
        todo!()
    }

}"#,
        );
    }

    #[test]
    fn test_copied_overridden_members() {
        check_assist(
            add_missing_impl_members,
            r#"
trait Foo {
    fn foo(&self);
    fn bar(&self) -> bool { true }
    fn baz(&self) -> u32 { 42 }
}

struct S;

impl Foo for S {
    fn bar(&self) {}
$0
}"#,
            r#"
trait Foo {
    fn foo(&self);
    fn bar(&self) -> bool { true }
    fn baz(&self) -> u32 { 42 }
}

struct S;

impl Foo for S {
    fn bar(&self) {}

    fn foo(&self) {
        ${0:todo!()}
    }

}"#,
        );
    }

    #[test]
    fn test_empty_impl_def() {
        check_assist(
            add_missing_impl_members,
            r#"
trait Foo { fn foo(&self); }
struct S;
impl Foo for S { $0 }"#,
            r#"
trait Foo { fn foo(&self); }
struct S;
impl Foo for S {
    fn foo(&self) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_impl_def_without_braces_macro() {
        check_assist(
            add_missing_impl_members,
            r#"
trait Foo { fn foo(&self); }
struct S;
impl Foo for S$0"#,
            r#"
trait Foo { fn foo(&self); }
struct S;
impl Foo for S {
    fn foo(&self) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_impl_def_without_braces_tabstop_first_item() {
        check_assist(
            add_missing_impl_members,
            r#"
trait Foo {
    type Output;
    fn foo(&self);
}
struct S;
impl Foo for S { $0 }"#,
            r#"
trait Foo {
    type Output;
    fn foo(&self);
}
struct S;
impl Foo for S {
    $0type Output;

    fn foo(&self) {
        todo!()
    }
}"#,
        );
    }

    #[test]
    fn fill_in_type_params_1() {
        check_assist(
            add_missing_impl_members,
            r#"
trait Foo<T> { fn foo(&self, t: T) -> &T; }
struct S;
impl Foo<u32> for S { $0 }"#,
            r#"
trait Foo<T> { fn foo(&self, t: T) -> &T; }
struct S;
impl Foo<u32> for S {
    fn foo(&self, t: u32) -> &u32 {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn fill_in_type_params_2() {
        check_assist(
            add_missing_impl_members,
            r#"
trait Foo<T> { fn foo(&self, t: T) -> &T; }
struct S;
impl<U> Foo<U> for S { $0 }"#,
            r#"
trait Foo<T> { fn foo(&self, t: T) -> &T; }
struct S;
impl<U> Foo<U> for S {
    fn foo(&self, t: U) -> &U {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_lifetime_substitution() {
        check_assist(
            add_missing_impl_members,
            r#"
pub trait Trait<'a, 'b, A, B, C> {
    fn foo(&self, one: &'a A, another: &'b B) -> &'a C;
}

impl<'x, 'y, T, V, U> Trait<'x, 'y, T, V, U> for () {$0}"#,
            r#"
pub trait Trait<'a, 'b, A, B, C> {
    fn foo(&self, one: &'a A, another: &'b B) -> &'a C;
}

impl<'x, 'y, T, V, U> Trait<'x, 'y, T, V, U> for () {
    fn foo(&self, one: &'x T, another: &'y V) -> &'x U {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_lifetime_substitution_with_body() {
        check_assist(
            add_missing_default_members,
            r#"
pub trait Trait<'a, 'b, A, B, C: Default> {
    fn foo(&self, _one: &'a A, _another: &'b B) -> (C, &'a i32) {
        let value: &'a i32 = &0;
        (C::default(), value)
    }
}

impl<'x, 'y, T, V, U: Default> Trait<'x, 'y, T, V, U> for () {$0}"#,
            r#"
pub trait Trait<'a, 'b, A, B, C: Default> {
    fn foo(&self, _one: &'a A, _another: &'b B) -> (C, &'a i32) {
        let value: &'a i32 = &0;
        (C::default(), value)
    }
}

impl<'x, 'y, T, V, U: Default> Trait<'x, 'y, T, V, U> for () {
    $0fn foo(&self, _one: &'x T, _another: &'y V) -> (U, &'x i32) {
        let value: &'x i32 = &0;
        (<U>::default(), value)
    }
}"#,
        );
    }

    #[test]
    fn test_const_substitution() {
        check_assist(
            add_missing_default_members,
            r#"
struct Bar<const N: usize> {
    bar: [i32, N]
}

trait Foo<const N: usize, T> {
    fn get_n_sq(&self, arg: &T) -> usize { N * N }
    fn get_array(&self, arg: Bar<N>) -> [i32; N] { [1; N] }
}

struct S<T> {
    wrapped: T
}

impl<const X: usize, Y, Z> Foo<X, Z> for S<Y> {
    $0
}"#,
            r#"
struct Bar<const N: usize> {
    bar: [i32, N]
}

trait Foo<const N: usize, T> {
    fn get_n_sq(&self, arg: &T) -> usize { N * N }
    fn get_array(&self, arg: Bar<N>) -> [i32; N] { [1; N] }
}

struct S<T> {
    wrapped: T
}

impl<const X: usize, Y, Z> Foo<X, Z> for S<Y> {
    $0fn get_n_sq(&self, arg: &Z) -> usize { X * X }

    fn get_array(&self, arg: Bar<X>) -> [i32; X] { [1; X] }
}"#,
        )
    }

    #[test]
    fn test_const_substitution_2() {
        check_assist(
            add_missing_default_members,
            r#"
trait Foo<const N: usize, const M: usize, T> {
    fn get_sum(&self, arg: &T) -> usize { N + M }
}

impl<X> Foo<42, {20 + 22}, X> for () {
    $0
}"#,
            r#"
trait Foo<const N: usize, const M: usize, T> {
    fn get_sum(&self, arg: &T) -> usize { N + M }
}

impl<X> Foo<42, {20 + 22}, X> for () {
    $0fn get_sum(&self, arg: &X) -> usize { 42 + {20 + 22} }
}"#,
        )
    }

    #[test]
    fn test_const_substitution_with_defaults() {
        check_assist(
            add_missing_default_members,
            r#"
trait Foo<T, const N: usize = 42, const M: bool = false, const P: char = 'a'> {
    fn get_n(&self) -> usize { N }
    fn get_m(&self) -> bool { M }
    fn get_p(&self) -> char { P }
    fn get_array(&self, arg: &T) -> [bool; N] { [M; N] }
}

impl<X> Foo<X> for () {
    $0
}"#,
            r#"
trait Foo<T, const N: usize = 42, const M: bool = false, const P: char = 'a'> {
    fn get_n(&self) -> usize { N }
    fn get_m(&self) -> bool { M }
    fn get_p(&self) -> char { P }
    fn get_array(&self, arg: &T) -> [bool; N] { [M; N] }
}

impl<X> Foo<X> for () {
    $0fn get_n(&self) -> usize { 42 }

    fn get_m(&self) -> bool { false }

    fn get_p(&self) -> char { 'a' }

    fn get_array(&self, arg: &X) -> [bool; 42] { [false; 42] }
}"#,
        );
    }

    #[test]
    fn test_const_substitution_with_defaults_2() {
        check_assist(
            add_missing_impl_members,
            r#"
mod m {
    pub const LEN: usize = 42;
    pub trait Foo<const M: usize = LEN, const N: usize = M, T = [bool; N]> {
        fn get_t(&self) -> T;
    }
}

impl m::Foo for () {
    $0
}"#,
            r#"
mod m {
    pub const LEN: usize = 42;
    pub trait Foo<const M: usize = LEN, const N: usize = M, T = [bool; N]> {
        fn get_t(&self) -> T;
    }
}

impl m::Foo for () {
    fn get_t(&self) -> [bool; m::LEN] {
        ${0:todo!()}
    }
}"#,
        )
    }

    #[test]
    fn test_const_substitution_with_defaults_3() {
        check_assist(
            add_missing_default_members,
            r#"
mod m {
    pub const VAL: usize = 0;

    pub trait Foo<const N: usize = {40 + 2}, const M: usize = {VAL + 1}> {
        fn get_n(&self) -> usize { N }
        fn get_m(&self) -> usize { M }
    }
}

impl m::Foo for () {
    $0
}"#,
            r#"
mod m {
    pub const VAL: usize = 0;

    pub trait Foo<const N: usize = {40 + 2}, const M: usize = {VAL + 1}> {
        fn get_n(&self) -> usize { N }
        fn get_m(&self) -> usize { M }
    }
}

impl m::Foo for () {
    $0fn get_n(&self) -> usize { N }

    fn get_m(&self) -> usize { M }
}"#,
        )
    }

    #[test]
    fn test_cursor_after_empty_impl_def() {
        check_assist(
            add_missing_impl_members,
            r#"
trait Foo { fn foo(&self); }
struct S;
impl Foo for S {}$0"#,
            r#"
trait Foo { fn foo(&self); }
struct S;
impl Foo for S {
    fn foo(&self) {
        ${0:todo!()}
    }
}"#,
        )
    }

    #[test]
    fn test_qualify_path_1() {
        check_assist(
            add_missing_impl_members,
            r#"
mod foo {
    pub struct Bar;
    pub trait Foo { fn foo(&self, bar: Bar); }
}
struct S;
impl foo::Foo for S { $0 }"#,
            r#"
mod foo {
    pub struct Bar;
    pub trait Foo { fn foo(&self, bar: Bar); }
}
struct S;
impl foo::Foo for S {
    fn foo(&self, bar: foo::Bar) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_qualify_path_2() {
        check_assist(
            add_missing_impl_members,
            r#"
mod foo {
    pub mod bar {
        pub struct Bar;
        pub trait Foo { fn foo(&self, bar: Bar); }
    }
}

use foo::bar;

struct S;
impl bar::Foo for S { $0 }"#,
            r#"
mod foo {
    pub mod bar {
        pub struct Bar;
        pub trait Foo { fn foo(&self, bar: Bar); }
    }
}

use foo::bar;

struct S;
impl bar::Foo for S {
    fn foo(&self, bar: bar::Bar) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_qualify_path_generic() {
        check_assist(
            add_missing_impl_members,
            r#"
mod foo {
    pub struct Bar<T>;
    pub trait Foo { fn foo(&self, bar: Bar<u32>); }
}
struct S;
impl foo::Foo for S { $0 }"#,
            r#"
mod foo {
    pub struct Bar<T>;
    pub trait Foo { fn foo(&self, bar: Bar<u32>); }
}
struct S;
impl foo::Foo for S {
    fn foo(&self, bar: foo::Bar<u32>) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_qualify_path_and_substitute_param() {
        check_assist(
            add_missing_impl_members,
            r#"
mod foo {
    pub struct Bar<T>;
    pub trait Foo<T> { fn foo(&self, bar: Bar<T>); }
}
struct S;
impl foo::Foo<u32> for S { $0 }"#,
            r#"
mod foo {
    pub struct Bar<T>;
    pub trait Foo<T> { fn foo(&self, bar: Bar<T>); }
}
struct S;
impl foo::Foo<u32> for S {
    fn foo(&self, bar: foo::Bar<u32>) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_substitute_param_no_qualify() {
        // when substituting params, the substituted param should not be qualified!
        check_assist(
            add_missing_impl_members,
            r#"
mod foo {
    pub trait Foo<T> { fn foo(&self, bar: T); }
    pub struct Param;
}
struct Param;
struct S;
impl foo::Foo<Param> for S { $0 }"#,
            r#"
mod foo {
    pub trait Foo<T> { fn foo(&self, bar: T); }
    pub struct Param;
}
struct Param;
struct S;
impl foo::Foo<Param> for S {
    fn foo(&self, bar: Param) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_qualify_path_associated_item() {
        check_assist(
            add_missing_impl_members,
            r#"
mod foo {
    pub struct Bar<T>;
    impl Bar<T> { type Assoc = u32; }
    pub trait Foo { fn foo(&self, bar: Bar<u32>::Assoc); }
}
struct S;
impl foo::Foo for S { $0 }"#,
            r#"
mod foo {
    pub struct Bar<T>;
    impl Bar<T> { type Assoc = u32; }
    pub trait Foo { fn foo(&self, bar: Bar<u32>::Assoc); }
}
struct S;
impl foo::Foo for S {
    fn foo(&self, bar: foo::Bar<u32>::Assoc) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_qualify_path_nested() {
        check_assist(
            add_missing_impl_members,
            r#"
mod foo {
    pub struct Bar<T>;
    pub struct Baz;
    pub trait Foo { fn foo(&self, bar: Bar<Baz>); }
}
struct S;
impl foo::Foo for S { $0 }"#,
            r#"
mod foo {
    pub struct Bar<T>;
    pub struct Baz;
    pub trait Foo { fn foo(&self, bar: Bar<Baz>); }
}
struct S;
impl foo::Foo for S {
    fn foo(&self, bar: foo::Bar<foo::Baz>) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_qualify_path_fn_trait_notation() {
        check_assist(
            add_missing_impl_members,
            r#"
mod foo {
    pub trait Fn<Args> { type Output; }
    pub trait Foo { fn foo(&self, bar: dyn Fn(u32) -> i32); }
}
struct S;
impl foo::Foo for S { $0 }"#,
            r#"
mod foo {
    pub trait Fn<Args> { type Output; }
    pub trait Foo { fn foo(&self, bar: dyn Fn(u32) -> i32); }
}
struct S;
impl foo::Foo for S {
    fn foo(&self, bar: dyn Fn(u32) -> i32) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_empty_trait() {
        check_assist_not_applicable(
            add_missing_impl_members,
            r#"
trait Foo;
struct S;
impl Foo for S { $0 }"#,
        )
    }

    #[test]
    fn test_ignore_unnamed_trait_members_and_default_methods() {
        check_assist_not_applicable(
            add_missing_impl_members,
            r#"
trait Foo {
    fn (arg: u32);
    fn valid(some: u32) -> bool { false }
}
struct S;
impl Foo for S { $0 }"#,
        )
    }

    #[test]
    fn test_with_docstring_and_attrs() {
        check_assist(
            add_missing_impl_members,
            r#"
#[doc(alias = "test alias")]
trait Foo {
    /// doc string
    type Output;

    #[must_use]
    fn foo(&self);
}
struct S;
impl Foo for S {}$0"#,
            r#"
#[doc(alias = "test alias")]
trait Foo {
    /// doc string
    type Output;

    #[must_use]
    fn foo(&self);
}
struct S;
impl Foo for S {
    $0type Output;

    fn foo(&self) {
        todo!()
    }
}"#,
        )
    }

    #[test]
    fn test_default_methods() {
        check_assist(
            add_missing_default_members,
            r#"
trait Foo {
    type Output;

    const CONST: usize = 42;
    const CONST_2: i32;

    fn valid(some: u32) -> bool { false }
    fn foo(some: u32) -> bool;
}
struct S;
impl Foo for S { $0 }"#,
            r#"
trait Foo {
    type Output;

    const CONST: usize = 42;
    const CONST_2: i32;

    fn valid(some: u32) -> bool { false }
    fn foo(some: u32) -> bool;
}
struct S;
impl Foo for S {
    $0const CONST: usize = 42;

    fn valid(some: u32) -> bool { false }
}"#,
        )
    }

    #[test]
    fn test_generic_single_default_parameter() {
        check_assist(
            add_missing_impl_members,
            r#"
trait Foo<T = Self> {
    fn bar(&self, other: &T);
}

struct S;
impl Foo for S { $0 }"#,
            r#"
trait Foo<T = Self> {
    fn bar(&self, other: &T);
}

struct S;
impl Foo for S {
    fn bar(&self, other: &Self) {
        ${0:todo!()}
    }
}"#,
        )
    }

    #[test]
    fn test_generic_default_parameter_is_second() {
        check_assist(
            add_missing_impl_members,
            r#"
trait Foo<T1, T2 = Self> {
    fn bar(&self, this: &T1, that: &T2);
}

struct S<T>;
impl Foo<T> for S<T> { $0 }"#,
            r#"
trait Foo<T1, T2 = Self> {
    fn bar(&self, this: &T1, that: &T2);
}

struct S<T>;
impl Foo<T> for S<T> {
    fn bar(&self, this: &T, that: &Self) {
        ${0:todo!()}
    }
}"#,
        )
    }

    #[test]
    fn test_qualify_generic_default_parameter() {
        check_assist(
            add_missing_impl_members,
            r#"
mod m {
    pub struct S;
    pub trait Foo<T = S> {
        fn bar(&self, other: &T);
    }
}

struct S;
impl m::Foo for S { $0 }"#,
            r#"
mod m {
    pub struct S;
    pub trait Foo<T = S> {
        fn bar(&self, other: &T);
    }
}

struct S;
impl m::Foo for S {
    fn bar(&self, other: &m::S) {
        ${0:todo!()}
    }
}"#,
        )
    }

    #[test]
    fn test_qualify_generic_default_parameter_2() {
        check_assist(
            add_missing_impl_members,
            r#"
mod m {
    pub struct Wrapper<T, V> {
        one: T,
        another: V
    };
    pub struct S;
    pub trait Foo<T = Wrapper<S, bool>> {
        fn bar(&self, other: &T);
    }
}

struct S;
impl m::Foo for S { $0 }"#,
            r#"
mod m {
    pub struct Wrapper<T, V> {
        one: T,
        another: V
    };
    pub struct S;
    pub trait Foo<T = Wrapper<S, bool>> {
        fn bar(&self, other: &T);
    }
}

struct S;
impl m::Foo for S {
    fn bar(&self, other: &m::Wrapper<m::S, bool>) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_qualify_generic_default_parameter_3() {
        check_assist(
            add_missing_impl_members,
            r#"
mod m {
    pub struct Wrapper<T, V> {
        one: T,
        another: V
    };
    pub struct S;
    pub trait Foo<T = S, V = Wrapper<T, S>> {
        fn bar(&self, other: &V);
    }
}

struct S;
impl m::Foo for S { $0 }"#,
            r#"
mod m {
    pub struct Wrapper<T, V> {
        one: T,
        another: V
    };
    pub struct S;
    pub trait Foo<T = S, V = Wrapper<T, S>> {
        fn bar(&self, other: &V);
    }
}

struct S;
impl m::Foo for S {
    fn bar(&self, other: &m::Wrapper<m::S, m::S>) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_assoc_type_bounds_are_removed() {
        check_assist(
            add_missing_impl_members,
            r#"
trait Tr {
    type Ty: Copy + 'static;
}

impl Tr for ()$0 {
}"#,
            r#"
trait Tr {
    type Ty: Copy + 'static;
}

impl Tr for () {
    $0type Ty;
}"#,
        )
    }

    #[test]
    fn test_whitespace_fixup_preserves_bad_tokens() {
        check_assist(
            add_missing_impl_members,
            r#"
trait Tr {
    fn foo();
}

impl Tr for ()$0 {
    +++
}"#,
            r#"
trait Tr {
    fn foo();
}

impl Tr for () {
    fn foo() {
        ${0:todo!()}
    }
    +++
}"#,
        )
    }

    #[test]
    fn test_whitespace_fixup_preserves_comments() {
        check_assist(
            add_missing_impl_members,
            r#"
trait Tr {
    fn foo();
}

impl Tr for ()$0 {
    // very important
}"#,
            r#"
trait Tr {
    fn foo();
}

impl Tr for () {
    fn foo() {
        ${0:todo!()}
    }
    // very important
}"#,
        )
    }

    #[test]
    fn weird_path() {
        check_assist(
            add_missing_impl_members,
            r#"
trait Test {
    fn foo(&self, x: crate)
}
impl Test for () {
    $0
}
"#,
            r#"
trait Test {
    fn foo(&self, x: crate)
}
impl Test for () {
    fn foo(&self, x: crate) {
        ${0:todo!()}
    }
}
"#,
        )
    }

    #[test]
    fn missing_generic_type() {
        check_assist(
            add_missing_impl_members,
            r#"
trait Foo<BAR> {
    fn foo(&self, bar: BAR);
}
impl Foo for () {
    $0
}
"#,
            r#"
trait Foo<BAR> {
    fn foo(&self, bar: BAR);
}
impl Foo for () {
    fn foo(&self, bar: BAR) {
        ${0:todo!()}
    }
}
"#,
        )
    }

    #[test]
    fn does_not_requalify_self_as_crate() {
        check_assist(
            add_missing_default_members,
            r"
struct Wrapper<T>(T);

trait T {
    fn f(self) -> Wrapper<Self> {
        Wrapper(self)
    }
}

impl T for () {
    $0
}
",
            r"
struct Wrapper<T>(T);

trait T {
    fn f(self) -> Wrapper<Self> {
        Wrapper(self)
    }
}

impl T for () {
    $0fn f(self) -> Wrapper<Self> {
        Wrapper(self)
    }
}
",
        );
    }

    #[test]
    fn test_default_body_generation() {
        check_assist(
            add_missing_impl_members,
            r#"
//- minicore: default
struct Foo(usize);

impl Default for Foo {
    $0
}
"#,
            r#"
struct Foo(usize);

impl Default for Foo {
    $0fn default() -> Self {
        Self(Default::default())
    }
}
"#,
        )
    }

    #[test]
    fn test_from_macro() {
        check_assist(
            add_missing_default_members,
            r#"
macro_rules! foo {
    () => {
        trait FooB {
            fn foo<'lt>(&'lt self) {}
        }
    }
}
foo!();
struct Foo(usize);

impl FooB for Foo {
    $0
}
"#,
            r#"
macro_rules! foo {
    () => {
        trait FooB {
            fn foo<'lt>(&'lt self) {}
        }
    }
}
foo!();
struct Foo(usize);

impl FooB for Foo {
    $0fn foo<'lt>(&'lt self){}
}
"#,
        )
    }

    #[test]
    fn test_assoc_type_when_trait_with_same_name_in_scope() {
        check_assist(
            add_missing_impl_members,
            r#"
pub trait Foo {}

pub trait Types {
    type Foo;
}

pub trait Behavior<T: Types> {
    fn reproduce(&self, foo: T::Foo);
}

pub struct Impl;

impl<T: Types> Behavior<T> for Impl { $0 }"#,
            r#"
pub trait Foo {}

pub trait Types {
    type Foo;
}

pub trait Behavior<T: Types> {
    fn reproduce(&self, foo: T::Foo);
}

pub struct Impl;

impl<T: Types> Behavior<T> for Impl {
    fn reproduce(&self, foo: <T as Types>::Foo) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_assoc_type_on_concrete_type() {
        check_assist(
            add_missing_impl_members,
            r#"
pub trait Types {
    type Foo;
}

impl Types for u32 {
    type Foo = bool;
}

pub trait Behavior<T: Types> {
    fn reproduce(&self, foo: T::Foo);
}

pub struct Impl;

impl Behavior<u32> for Impl { $0 }"#,
            r#"
pub trait Types {
    type Foo;
}

impl Types for u32 {
    type Foo = bool;
}

pub trait Behavior<T: Types> {
    fn reproduce(&self, foo: T::Foo);
}

pub struct Impl;

impl Behavior<u32> for Impl {
    fn reproduce(&self, foo: <u32 as Types>::Foo) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_assoc_type_on_concrete_type_qualified() {
        check_assist(
            add_missing_impl_members,
            r#"
pub trait Types {
    type Foo;
}

impl Types for std::string::String {
    type Foo = bool;
}

pub trait Behavior<T: Types> {
    fn reproduce(&self, foo: T::Foo);
}

pub struct Impl;

impl Behavior<std::string::String> for Impl { $0 }"#,
            r#"
pub trait Types {
    type Foo;
}

impl Types for std::string::String {
    type Foo = bool;
}

pub trait Behavior<T: Types> {
    fn reproduce(&self, foo: T::Foo);
}

pub struct Impl;

impl Behavior<std::string::String> for Impl {
    fn reproduce(&self, foo: <std::string::String as Types>::Foo) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_assoc_type_on_concrete_type_multi_option_ambiguous() {
        check_assist(
            add_missing_impl_members,
            r#"
pub trait Types {
    type Foo;
}

pub trait Types2 {
    type Foo;
}

impl Types for u32 {
    type Foo = bool;
}

impl Types2 for u32 {
    type Foo = String;
}

pub trait Behavior<T: Types + Types2> {
    fn reproduce(&self, foo: <T as Types2>::Foo);
}

pub struct Impl;

impl Behavior<u32> for Impl { $0 }"#,
            r#"
pub trait Types {
    type Foo;
}

pub trait Types2 {
    type Foo;
}

impl Types for u32 {
    type Foo = bool;
}

impl Types2 for u32 {
    type Foo = String;
}

pub trait Behavior<T: Types + Types2> {
    fn reproduce(&self, foo: <T as Types2>::Foo);
}

pub struct Impl;

impl Behavior<u32> for Impl {
    fn reproduce(&self, foo: <u32 as Types2>::Foo) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_assoc_type_on_concrete_type_multi_option() {
        check_assist(
            add_missing_impl_members,
            r#"
pub trait Types {
    type Foo;
}

pub trait Types2 {
    type Bar;
}

impl Types for u32 {
    type Foo = bool;
}

impl Types2 for u32 {
    type Bar = String;
}

pub trait Behavior<T: Types + Types2> {
    fn reproduce(&self, foo: T::Bar);
}

pub struct Impl;

impl Behavior<u32> for Impl { $0 }"#,
            r#"
pub trait Types {
    type Foo;
}

pub trait Types2 {
    type Bar;
}

impl Types for u32 {
    type Foo = bool;
}

impl Types2 for u32 {
    type Bar = String;
}

pub trait Behavior<T: Types + Types2> {
    fn reproduce(&self, foo: T::Bar);
}

pub struct Impl;

impl Behavior<u32> for Impl {
    fn reproduce(&self, foo: <u32 as Types2>::Bar) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_assoc_type_on_concrete_type_multi_option_foreign() {
        check_assist(
            add_missing_impl_members,
            r#"
mod bar {
    pub trait Types2 {
        type Bar;
    }
}

pub trait Types {
    type Foo;
}

impl Types for u32 {
    type Foo = bool;
}

impl bar::Types2 for u32 {
    type Bar = String;
}

pub trait Behavior<T: Types + bar::Types2> {
    fn reproduce(&self, foo: T::Bar);
}

pub struct Impl;

impl Behavior<u32> for Impl { $0 }"#,
            r#"
mod bar {
    pub trait Types2 {
        type Bar;
    }
}

pub trait Types {
    type Foo;
}

impl Types for u32 {
    type Foo = bool;
}

impl bar::Types2 for u32 {
    type Bar = String;
}

pub trait Behavior<T: Types + bar::Types2> {
    fn reproduce(&self, foo: T::Bar);
}

pub struct Impl;

impl Behavior<u32> for Impl {
    fn reproduce(&self, foo: <u32 as bar::Types2>::Bar) {
        ${0:todo!()}
    }
}"#,
        );
    }

    #[test]
    fn test_transform_path_in_path_expr() {
        check_assist(
            add_missing_default_members,
            r#"
pub trait Const {
    const FOO: u32;
}

pub trait Trait<T: Const> {
    fn foo() -> bool {
        match T::FOO {
            0 => true,
            _ => false,
        }
    }
}

impl Const for u32 {
    const FOO: u32 = 1;
}

struct Impl;

impl Trait<u32> for Impl { $0 }"#,
            r#"
pub trait Const {
    const FOO: u32;
}

pub trait Trait<T: Const> {
    fn foo() -> bool {
        match T::FOO {
            0 => true,
            _ => false,
        }
    }
}

impl Const for u32 {
    const FOO: u32 = 1;
}

struct Impl;

impl Trait<u32> for Impl {
    $0fn foo() -> bool {
        match <u32 as Const>::FOO {
            0 => true,
            _ => false,
        }
    }
}"#,
        );
    }

    #[test]
    fn test_default_partial_eq() {
        check_assist(
            add_missing_default_members,
            r#"
//- minicore: eq
struct SomeStruct {
    data: usize,
    field: (usize, usize),
}
impl PartialEq for SomeStruct {$0}
"#,
            r#"
struct SomeStruct {
    data: usize,
    field: (usize, usize),
}
impl PartialEq for SomeStruct {
    $0fn ne(&self, other: &Self) -> bool {
        !self.eq(other)
    }
}
"#,
        );
    }

    #[test]
    fn test_partial_eq_body_when_types_semantically_match() {
        check_assist(
            add_missing_impl_members,
            r#"
//- minicore: eq
struct S<T, U>(T, U);
type Alias<T> = S<T, T>;
impl<T> PartialEq<Alias<T>> for S<T, T> {$0}
"#,
            r#"
struct S<T, U>(T, U);
type Alias<T> = S<T, T>;
impl<T> PartialEq<Alias<T>> for S<T, T> {
    $0fn eq(&self, other: &Alias<T>) -> bool {
        self.0 == other.0 && self.1 == other.1
    }
}
"#,
        );
    }

    #[test]
    fn test_partial_eq_body_when_types_dont_match() {
        check_assist(
            add_missing_impl_members,
            r#"
//- minicore: eq
struct S<T, U>(T, U);
type Alias<T> = S<T, T>;
impl<T> PartialEq<Alias<T>> for S<T, i32> {$0}
"#,
            r#"
struct S<T, U>(T, U);
type Alias<T> = S<T, T>;
impl<T> PartialEq<Alias<T>> for S<T, i32> {
    fn eq(&self, other: &Alias<T>) -> bool {
        ${0:todo!()}
    }
}
"#,
        );
    }

    #[test]
    fn test_ignore_function_body() {
        check_assist_not_applicable(
            add_missing_default_members,
            r#"
trait Trait {
    type X;
    fn foo(&self);
    fn bar(&self) {}
}

impl Trait for () {
    type X = u8;
    fn foo(&self) {$0
        let x = 5;
    }
}"#,
        )
    }

    #[test]
    fn test_ignore_param_list() {
        check_assist_not_applicable(
            add_missing_impl_members,
            r#"
trait Trait {
    type X;
    fn foo(&self);
    fn bar(&self);
}

impl Trait for () {
    type X = u8;
    fn foo(&self$0) {
        let x = 5;
    }
}"#,
        )
    }

    #[test]
    fn test_ignore_scope_inside_function() {
        check_assist_not_applicable(
            add_missing_impl_members,
            r#"
trait Trait {
    type X;
    fn foo(&self);
    fn bar(&self);
}

impl Trait for () {
    type X = u8;
    fn foo(&self) {
        let x = async {$0 5 };
    }
}"#,
        )
    }

    #[test]
    fn test_apply_outside_function() {
        check_assist(
            add_missing_default_members,
            r#"
trait Trait {
    type X;
    fn foo(&self);
    fn bar(&self) {}
}

impl Trait for () {
    type X = u8;
    fn foo(&self)$0 {}
}"#,
            r#"
trait Trait {
    type X;
    fn foo(&self);
    fn bar(&self) {}
}

impl Trait for () {
    type X = u8;
    fn foo(&self) {}

    $0fn bar(&self) {}
}"#,
        )
    }

    #[test]
    fn test_works_inside_function() {
        check_assist(
            add_missing_impl_members,
            r#"
trait Tr {
    fn method();
}
fn main() {
    struct S;
    impl Tr for S {
        $0
    }
}
"#,
            r#"
trait Tr {
    fn method();
}
fn main() {
    struct S;
    impl Tr for S {
        fn method() {
            ${0:todo!()}
        }
    }
}
"#,
        );
    }

    #[test]
    fn test_add_missing_preserves_indentation() {
        // in different modules
        check_assist(
            add_missing_impl_members,
            r#"
mod m {
    pub trait Foo {
        const CONST_MULTILINE: (
            i32,
            i32
        );

        fn foo(&self);
    }
}
struct S;
impl m::Foo for S { $0 }"#,
            r#"
mod m {
    pub trait Foo {
        const CONST_MULTILINE: (
            i32,
            i32
        );

        fn foo(&self);
    }
}
struct S;
impl m::Foo for S {
    $0const CONST_MULTILINE: (
        i32,
        i32
    );

    fn foo(&self) {
        todo!()
    }
}"#,
        );
        // in the same module
        check_assist(
            add_missing_impl_members,
            r#"
mod m {
    trait Foo {
        type Output;

        const CONST: usize = 42;
        const CONST_2: i32;
        const CONST_MULTILINE: (
            i32,
            i32
        );

        fn foo(&self);
        fn bar(&self);
        fn baz(&self);
    }

    struct S;

    impl Foo for S {
        fn bar(&self) {}
$0
    }
}"#,
            r#"
mod m {
    trait Foo {
        type Output;

        const CONST: usize = 42;
        const CONST_2: i32;
        const CONST_MULTILINE: (
            i32,
            i32
        );

        fn foo(&self);
        fn bar(&self);
        fn baz(&self);
    }

    struct S;

    impl Foo for S {
        fn bar(&self) {}

        $0type Output;

        const CONST_2: i32;

        const CONST_MULTILINE: (
            i32,
            i32
        );

        fn foo(&self) {
            todo!()
        }

        fn baz(&self) {
            todo!()
        }

    }
}"#,
        );
    }

    #[test]
    fn test_add_default_preserves_indentation() {
        check_assist(
            add_missing_default_members,
            r#"
mod m {
    pub trait Foo {
        type Output;

        const CONST: usize = 42;
        const CONST_2: i32;
        const CONST_MULTILINE: = (
            i32,
            i32,
        ) = (3, 14);

        fn valid(some: u32) -> bool { false }
        fn foo(some: u32) -> bool;
    }
}
struct S;
impl m::Foo for S { $0 }"#,
            r#"
mod m {
    pub trait Foo {
        type Output;

        const CONST: usize = 42;
        const CONST_2: i32;
        const CONST_MULTILINE: = (
            i32,
            i32,
        ) = (3, 14);

        fn valid(some: u32) -> bool { false }
        fn foo(some: u32) -> bool;
    }
}
struct S;
impl m::Foo for S {
    $0const CONST: usize = 42;

    const CONST_MULTILINE: = (
        i32,
        i32,
    ) = (3, 14);

    fn valid(some: u32) -> bool { false }
}"#,
        )
    }

    #[test]
    fn nested_macro_should_not_cause_crash() {
        check_assist(
            add_missing_impl_members,
            r#"
macro_rules! ty { () => { i32 } }
trait SomeTrait { type Output; }
impl SomeTrait for i32 { type Output = i64; }
macro_rules! define_method {
    () => {
        fn method(&mut self, params: <ty!() as SomeTrait>::Output);
    };
}
trait AnotherTrait { define_method!(); }
impl $0AnotherTrait for () {
}
"#,
            r#"
macro_rules! ty { () => { i32 } }
trait SomeTrait { type Output; }
impl SomeTrait for i32 { type Output = i64; }
macro_rules! define_method {
    () => {
        fn method(&mut self, params: <ty!() as SomeTrait>::Output);
    };
}
trait AnotherTrait { define_method!(); }
impl AnotherTrait for () {
    $0fn method(&mut self,params: <ty!()as SomeTrait>::Output) {
        todo!()
    }
}
"#,
        );
    }

    // FIXME: `T` in `ty!(T)` should be replaced by `PathTransform`.
    #[test]
    fn paths_in_nested_macro_should_get_transformed() {
        check_assist(
            add_missing_impl_members,
            r#"
macro_rules! ty { ($me:ty) => { $me } }
trait SomeTrait { type Output; }
impl SomeTrait for i32 { type Output = i64; }
macro_rules! define_method {
    ($t:ty) => {
        fn method(&mut self, params: <ty!($t) as SomeTrait>::Output);
    };
}
trait AnotherTrait<T: SomeTrait> { define_method!(T); }
impl $0AnotherTrait<i32> for () {
}
"#,
            r#"
macro_rules! ty { ($me:ty) => { $me } }
trait SomeTrait { type Output; }
impl SomeTrait for i32 { type Output = i64; }
macro_rules! define_method {
    ($t:ty) => {
        fn method(&mut self, params: <ty!($t) as SomeTrait>::Output);
    };
}
trait AnotherTrait<T: SomeTrait> { define_method!(T); }
impl AnotherTrait<i32> for () {
    $0fn method(&mut self,params: <ty!(T)as SomeTrait>::Output) {
        todo!()
    }
}
"#,
        );
    }

    #[test]
    fn doc_hidden_default_impls_ignored() {
        // doc(hidden) attr is ignored trait and impl both belong to the local crate.
        check_assist(
            add_missing_default_members,
            r#"
struct Foo;
trait Trait {
    #[doc(hidden)]
    fn func_with_default_impl() -> u32 {
        42
    }
    fn another_default_impl() -> u32 {
        43
    }
}
impl Tra$0it for Foo {}"#,
            r#"
struct Foo;
trait Trait {
    #[doc(hidden)]
    fn func_with_default_impl() -> u32 {
        42
    }
    fn another_default_impl() -> u32 {
        43
    }
}
impl Trait for Foo {
    $0fn func_with_default_impl() -> u32 {
        42
    }

    fn another_default_impl() -> u32 {
        43
    }
}"#,
        )
    }

    #[test]
    fn doc_hidden_default_impls_lang_crates() {
        // Not applicable because Eq has a single method and this has a #[doc(hidden)] attr set.
        check_assist_not_applicable(
            add_missing_default_members,
            r#"
//- minicore: eq
use core::cmp::Eq;
struct Foo;
impl E$0q for Foo { /* $0 */ }
"#,
        )
    }

    #[test]
    fn doc_hidden_default_impls_lib_crates() {
        check_assist(
            add_missing_default_members,
            r#"
    //- /main.rs crate:a deps:b
    struct B;
    impl b::Exte$0rnTrait for B {}
    //- /lib.rs crate:b new_source_root:library
    pub trait ExternTrait {
        #[doc(hidden)]
        fn hidden_default() -> Option<()> {
            todo!()
        }

        fn unhidden_default() -> Option<()> {
            todo!()
        }

        fn unhidden_nondefault() -> Option<()>;
    }
                "#,
            r#"
    struct B;
    impl b::ExternTrait for B {
        $0fn unhidden_default() -> Option<()> {
            todo!()
        }
    }
    "#,
        )
    }

    #[test]
    fn doc_hidden_default_impls_local_crates() {
        check_assist(
            add_missing_default_members,
            r#"
trait LocalTrait {
    #[doc(hidden)]
    fn no_skip_default() -> Option<()> {
        todo!()
    }
    fn no_skip_default_2() -> Option<()> {
        todo!()
    }
}

struct B;
impl Loc$0alTrait for B {}
            "#,
            r#"
trait LocalTrait {
    #[doc(hidden)]
    fn no_skip_default() -> Option<()> {
        todo!()
    }
    fn no_skip_default_2() -> Option<()> {
        todo!()
    }
}

struct B;
impl LocalTrait for B {
    $0fn no_skip_default() -> Option<()> {
        todo!()
    }

    fn no_skip_default_2() -> Option<()> {
        todo!()
    }
}
            "#,
        )
    }

    #[test]
    fn doc_hidden_default_impls_workspace_crates() {
        check_assist(
            add_missing_default_members,
            r#"
//- /lib.rs crate:b new_source_root:local
trait LocalTrait {
    #[doc(hidden)]
    fn no_skip_default() -> Option<()> {
        todo!()
    }
    fn no_skip_default_2() -> Option<()> {
        todo!()
    }
}

//- /main.rs crate:a deps:b
struct B;
impl b::Loc$0alTrait for B {}
            "#,
            r#"
struct B;
impl b::LocalTrait for B {
    $0fn no_skip_default() -> Option<()> {
        todo!()
    }

    fn no_skip_default_2() -> Option<()> {
        todo!()
    }
}
            "#,
        )
    }

    #[test]
    fn doc_hidden_nondefault_member() {
        check_assist(
            add_missing_impl_members,
            r#"
//- /lib.rs crate:b new_source_root:local
trait LocalTrait {
    #[doc(hidden)]
    fn no_skip_non_default() -> Option<()>;

    #[doc(hidden)]
    fn skip_default() -> Option<()> {
        todo!()
    }
}

//- /main.rs crate:a deps:b
struct B;
impl b::Loc$0alTrait for B {}
            "#,
            r#"
struct B;
impl b::LocalTrait for B {
    fn no_skip_non_default() -> Option<()> {
        ${0:todo!()}
    }
}
            "#,
        )
    }

    #[test]
    fn impl_with_type_param_with_former_param_as_default() {
        check_assist(
            add_missing_impl_members,
            r#"
pub trait Test<'a, T, U = T> {
    fn test(item: &'a T) -> U;
}
impl<'a> Test<'a, i32> for bool {
    $0
}
"#,
            r#"
pub trait Test<'a, T, U = T> {
    fn test(item: &'a T) -> U;
}
impl<'a> Test<'a, i32> for bool {
    fn test(item: &'a i32) -> i32 {
        ${0:todo!()}
    }
}
"#,
        );
    }

    #[test]
    fn issue_17321() {
        check_assist(
            add_missing_impl_members,
            r#"
fn main() {}

mod other_file_1 {
    pub const SOME_CONSTANT: usize = 8;
}

mod other_file_2 {
    use crate::other_file_1::SOME_CONSTANT;

    pub trait Trait {
        type Iter: Iterator<Item = [u8; SOME_CONSTANT]>;
    }
}

pub struct MyStruct;

impl other_file_2::Trait for MyStruct$0 {}"#,
            r#"
fn main() {}

mod other_file_1 {
    pub const SOME_CONSTANT: usize = 8;
}

mod other_file_2 {
    use crate::other_file_1::SOME_CONSTANT;

    pub trait Trait {
        type Iter: Iterator<Item = [u8; SOME_CONSTANT]>;
    }
}

pub struct MyStruct;

impl other_file_2::Trait for MyStruct {
    $0type Iter;
}"#,
        );
    }
}
