unit PGPMEMORY;

interface
uses pgpBase;
{+//____________________________________________________________________________ }
{-Copyright (C) 1997 Pretty Good Privacy, Inc. }
{-All rights reserved. }

{-Contains the definition of the PGPMemoryMgr object. }

{-$Id: Pgpmemory.pas,v 1.5.12.2 1999/08/27 22:00:48 build Exp $ }
{=____________________________________________________________________________ }


{+//____________________________________________________________________________ }
{-Mini-tutorial: }

{-A PGPMemoryMgr is an object which implements memory management, including }
{-allocation, reallocation, deallocation, and secure versions of the same. }

{-*** Using it *** }
{-A typical sequence of calls is as follows: }
{-PGPNewMemoryMgr }
{-... }
{-PGPNewData or PGPNewSecureData }
{-PGPFreeData }
{-... }
{-PGPFreeMemoryMgr }

{-Typically, a program will create one PGPMemoryMgr per thread at }
{-thread creation time and use that memory mgr until the thread dies. }
{-Generally, an individual PGPMemoryMgr instance is not thread-safe; }
{-you must either synchronize or use one PGPMemoryMgr per thread. }


{-*** Custom Allocators *** }

{-Default allocators are supplied, but the client can create a custom }
{-PGPMemoryMgr using PGPNewMemoryMgrCustom() which uses client-supplied }
{-routines. }
{-Custom routines need only concern themselves with the actual }
{-allocation and deallocation. }
{-The following should be kept in mind for user supplied routines: }
{-- they can ignore the allocation flags passed }
{-- leaks, memory clearing, etc is done by the PGPMemoryMgr }
{-- secure allocator must set 'isNonPageable' to TRUE only if the }
{-memory really can't be paged. }
{-- the user value is not interpreted by the PGPMemoryMgr. Typically, }
{-it would be a pointer to some data the allocation routines use }
{-to store state. }


{-*** Secure memory allocation *** }

{-Blocks can be allocated as "Secure" blocks. Secure blocks are guaranteed }
{-to be wiped when they are deallocated. Additionally, if the operating }
{-system and the current conditions allow, the block will be allocated }
{-in non-pageable memory. You can determine the attributes of a block using }
{-PGPGetMemoryMgrDataInfo(). }


{-*** Leaks tracking *** }

{-Leaks tracking is implemented when debugging is on, }
{-but currently reporting is limited to reporting the number of leaks }
{-outstanding when the PGPMemoryMgr is disposed. }


{-*** Debugging *** }

{-For debugging purposes, blocks may be larger in debug mode to accomodate }
{-various schemes to detect stray pointers, etc. }
{=____________________________________________________________________________ }


const
kPGPMemoryMgrFlags_None = 0;
           kPGPMemoryMgrFlags_Clear = 1 ;

Type PGPMemoryMgrFlags = PGPFLAGS;


Type PGPMemoryMgrRef = pointer;
type PGPMemoryMgrAllocationProc=function( mgr: PGPMemoryMgrRef;
						 userValue:PGPUserValue;
						 requestSize:PGPSize;
                                                  flags: PGPMemoryMgrFlags ):pointer;

{+// realloc not be implemented using PGPNewData() */ }
Type PGPMemoryMgrReallocationProc=function( mgr:PGPMemoryMgrRef;
						userValue:PGPUserValue ;
						var allocation:pointer;
                                                newAllocationSize:PGPSize ;
						 flags:PGPMemoryMgrFlags;
                                                 existingSize:PGPSize ):PGPERROR;

type PGPMemoryMgrDeallocationProc=function( mgr:PGPMemoryMgrRef;
						userValue:PGPUserValue ;
						var allocation:pointer;
                                                  allocationSize:PGPSize ):PGPERROR;



type PGPMemoryMgrSecureAllocationProc=function( mgr: PGPMemoryMgrRef;
						 userValue:PGPUserValue;
						 requestSize:PGPSize;
                                                  flags: PGPMemoryMgrFlags ;
						var isNonPageable:PGPBoolean ):pointer;


// deallocation proc need not clear the memory upon deallocation since
  //	PGPFreeData() does it automatically */
type PGPMemoryMgrSecureDeallocationProc=function( mgr:PGPMemoryMgrRef;
						userValue:PGPUserValue ;
						var allocation:pointer;
                                                  allocationSize:PGPSize;
						wasLocked:PGPBoolean ):PGPERROR;


TYPE PGPNEWMEMORYMGRSTRUCT=record
{+// sizeofStruct must be inited to sizeof( PGPNewMemoryMgrStruct ) */ }
          sizeofStruct: PGPUINT32;
          reservedFlags: PGPFLAGS;
          allocProc: PGPMEMORYMGRALLOCATIONPROC;
          reallocProc: PGPMEMORYMGRREALLOCATIONPROC;
          deallocProc: PGPMEMORYMGRDEALLOCATIONPROC;
          secureAllocProc: PGPMEMORYMGRSECUREALLOCATIONPROC;
          reserved: Pointer;
          secureDeallocProc: PGPMEMORYMGRSECUREDEALLOCATIONPROC;
          customValue: PGPUSERVALUE;
          PAD:array[0..7]of pointer;
        end ;
PPGPNEWMEMORYMGRSTRUCT=^PGPNEWMEMORYMGRSTRUCT;


{+//____________________________________________________________________________ }
{-Memory Mgr routines }
{=____________________________________________________________________________ }

function PGPMemoryMgrIsValid(mgr: PGPMEMORYMGRREF): PGPBOOLEAN; cdecl; far;

function PGPNewMemoryMgr(reserved: PGPFLAGS;
                         var newMemoryMgr: PGPMEMORYMGRREF): PGPERROR; cdecl; far;


function PGPNewMemoryMgrCustom(const custom: PPGPNEWMEMORYMGRSTRUCT;
                               var newMemoryMgr: PGPMEMORYMGRREF): PGPERROR; cdecl; far;


function PGPFreeMemoryMgr(mgr: PGPMEMORYMGRREF): PGPERROR; cdecl; far;


function PGPGetMemoryMgrCustomValue(mgr: PGPMEMORYMGRREF;
                                    var customValue: PGPUSERVALUE): PGPERROR; cdecl; far;

function PGPSetMemoryMgrCustomValue(mgr: PGPMEMORYMGRREF;
                                    customValue: PGPUSERVALUE): PGPERROR; cdecl; far;

{+// allocate a block of the specified size */ }

function PGPNewData(mgr: PGPMEMORYMGRREF;
                     requestSize: PGPSIZE;
                     flags: PGPMEMORYMGRFLAGS):pointer; cdecl; far;

{+// allocate a block of the specified size in non-pageable memory */ }
{+// *isSecure is TRUE if the block definitely can't be paged */ }

function PGPNewSecureData(mgr: PGPMEMORYMGRREF;
                           requestSize: PGPSIZE;
                           flags: PGPMEMORYMGRFLAGS):pointer; cdecl; far;

{+// properly reallocs secure or non-secure blocks */ }
{+// WARNING: the block may move, even if its size is being reduced */ }

function PGPReallocData(mgr: PGPMEMORYMGRREF;
                        var allocation: Pointer;
                        newAllocationSize: PGPSIZE;
                        flags: PGPMEMORYMGRFLAGS): PGPERROR; cdecl; far;

{+// properly frees secure or non-secure blocks */ }

function PGPFreeData(allocation: Pointer): PGPERROR; cdecl; far;


{+//____________________________________________________________________________ }
{-Block Info: }
{-kPGPMemoryMgrBlockInfo_Valid it's a valid block }
{-kPGPMemoryMgrBlockInfo_Secure block is a secure allocation }
{-kPGPMemoryMgrBlockInfo_NonPageable block cannot be paged by VM }

{-Secure blocks are always wiped before being disposed, }
{-but may or may not be pageable, depending on the OS facilities. Some }
{-OSs may not provide the ability to make blocks non-pageable. }

{-You should check these flags if the information matters to you. }
{=____________________________________________________________________________ }
Const kPGPMemoryMgrBlockInfo_Valid = 1 shl 0 ;
Const kPGPMemoryMgrBlockInfo_Secure = 1 shl 1 ;
Const kPGPMemoryMgrBlockInfo_NonPageable = 1 shl 2 ;

function PGPGetMemoryMgrDataInfo(allocation: Pointer): PGPFLAGS; cdecl; far;


{+//____________________________________________________________________________ }
{-Debug-only routines: }
{=____________________________________________________________________________ }
{+// validates data structures (debug only) */ }

procedure PGPCheckMemoryMgrIntegrity(mgr: PGPMEMORYMGRREF); cdecl; far;


implementation

function PGPValidateMemoryMgr(mgr: PGPMEMORYMGRREF):PGPBoolean;
begin
	result:=PGPMemoryMgrIsValid(mgr);
end;

function PGPMemoryMgrIsValid(mgr: PGPMEMORYMGRREF): PGPBOOLEAN; external 'PGP_SDK.DLL';

function PGPNewMemoryMgr; external 'PGP_SDK.DLL';

function PGPNewMemoryMgrCustom; external 'PGP_SDK.DLL';

function PGPFreeMemoryMgr; external 'PGP_SDK.DLL';

function PGPGetMemoryMgrCustomValue; external 'PGP_SDK.DLL';

function PGPSetMemoryMgrCustomValue; external 'PGP_SDK.DLL';

function PGPNewData; external 'PGP_SDK.DLL';

function PGPNewSecureData; external 'PGP_SDK.DLL';

function PGPReallocData; external 'PGP_SDK.DLL';

function PGPFreeData; external 'PGP_SDK.DLL';

function PGPGetMemoryMgrDataInfo; external 'PGP_SDK.DLL';

procedure PGPCheckMemoryMgrIntegrity; external 'PGP_SDK.DLL';

end.
