/*____________________________________________________________________________
	Copyright (C) 2000 Network Associates Inc. and affiliated companies.
	All rights reserved.
	

	$Id: ItemHandling.c,v 1.7.12.8 2000/08/09 01:23:06 build Exp $
____________________________________________________________________________*/

/*::: MODULE OVERVIEW :::::::::::::
Provides common Notes item-handling & -manipulation facilities.

--- suggested enhancements ----
2/21/00 PR: see suggestions in text-list item-matching procedures' 
	documentation
9/12/99 PR: For text & text-list item content copying, allow for the 
	possibility that embedded null characters may be in the content. This 
	means we need a length output on a number of functions here,  la in 
	eus_AttemptGetTextItemContent() (though still not fully effective there 
	until this described enhancement is fulfilled)

--- revision history ----------
2/21/00 Version 1.1.5: Paul Ryan
+ added position-found and count outputs to ef_TextListContainsEntry(), thus 
  changing its signature
+ added eus_IsRtf(), ef_TextListEntryMatches()
+ added eus_ItemScan() utility stub
+ signature change to ef_GetNumberListEntry() to foster consistency with 
														text-list procedures
+ documentation improvement, sundry minor logic enhancements

9/12/99 Version 1.1: Paul Ryan
+ logic shortening
+ documentation addition
+ signature change to eus_CreateTextListEntryCopy()
+ removed eus_GetItemContents() as not useful

12/13/98 Version 1.0: Paul Ryan
::::::::::::::::::::::::::::::::::::*/

#include "ItemHandling.h"


/** eus_CreateTextListEntryCopy( ***
Purpose is to return a copy of the textual content of a specified text-list 
item entry.

--- parameters & return ----
us_ENTRY: zero-based index to element in the text-list whose content should 
	be copied
puc_LIST: pointer to the text-list from which to extract the specified entry
f_ITEM_TYPE_PRESENT: flag specifying whether the two-byte Notes item-type 
	specifier precedes the text-list structure
ppc: Output. Pointer to the pointer variable this procedure should set to 
	point to the string buffer this procedure will allocate and populate with 
	a copy of the textual content requested by the caller. CALLER IS 
	RESPONSIBLE for freeing the allocated buffer to the operating system. If 
	the textual content is not text, or if the textual content is the null 
	string, or if the procedure is unsuccessful, no buffer will be allocated 
	and the pointer variable is guaranteed to be NULL upon return.
RETURN:
	eus_SUCCESS if no error occured
	!eus_SUCCESS if invalid parameters were passed
	ERR_MEMORY if required memory allocation failed
	the Notes API error code otherwise

--- revision history -------
2/21/00 PR: minor documentation adjustment

9/12/99 PR
+ if entry is the null string, return a null pointer instead of allocating 
  resources for a null-string return
+ shortened logic

12/2/98 PR: created			*/
STATUS eus_CreateTextListEntryCopy( const WORD  us_ENTRY, 
									BYTE *const  puc_LIST, 
									const BOOL  f_ITEM_TYPE_PRESENT, 
									char * *const  ppc)	{
	BYTE * puc;
	WORD  us;
	STATUS  us_err;

	if (!( puc_LIST && ppc))
		return !eus_SUCCESS;

	*ppc = NULL;

	if (f_ITEM_TYPE_PRESENT && *(WORD *) puc_LIST != TYPE_TEXT_LIST)
		return !eus_SUCCESS;

	if (us_err = ListGetText( puc_LIST, f_ITEM_TYPE_PRESENT, us_ENTRY, &puc, 
																		&us))
		return us_err;

	if (us)	{
		if (!( *ppc = malloc( us + 1)))
			return ERR_MEMORY;
		memcpy( *ppc, puc, us);
		(*ppc)[ us] = NULL;
	}

	return eus_SUCCESS;
} //eus_CreateTextListEntryCopy(


/** ef_GetNumberListEntry( ***
Obtains a copy of the specified entry in a given number-list.

--- parameters & return ----
puc: address of the number-list to check
f_ITEM_TYPE_PRESENT: tells whether a TYPE_NUMBER_RANGE item-type header 
	precedes the structure of the number-list
US: zero-based index of the entry to copy
RETURN: TRUE if number-list entry was successfully copied; FALSE if not or if 
	the search could not be conducted due to an invalid argument

--- revision history -------
2/21/00 PR
+ signature change to foster consistency with text-list procedures
+ logic improvement
+ completed standard documentation

9/12/99 PR: logic optimization
12/2/98 PR: created			*/
BOOL ef_GetNumberListEntry( const BYTE * puc, 
							const BOOL  f_ITEM_TYPE_PRESENT, 
							const WORD  US, 
							double *const  pdbl)	{
	if (!( puc && pdbl))
		return FALSE;

	//if the purported number-list is obviously incompatible with the task 
	//	at hand, short-circuit that the requested entry could not be delivered
	if (f_ITEM_TYPE_PRESENT)	{
		if (*(WORD *) puc != TYPE_NUMBER_RANGE)
			return FALSE;
		puc += sizeof( WORD);
	}
	if (!( ((RANGE *) puc)->ListEntries > US && 
											!((RANGE *) puc)->RangeEntries))
		return FALSE;

	//copy the specified entry into the output variable
	*pdbl = *( (double *) (puc += sizeof( RANGE)) + US);

	return TRUE;
} //ef_GetNumberListEntry(


/** eus_RemoveItem( ***
Remove (delete) a specified item or items from an open note. If specified 
item is not present on the note, procedure returns success. If item-name is 
specified, all items going by that name are removed.

--- parameters & return ----
h_NOTE: handle to the open note containing the named rich-text item
pc_ITMNM: Optional. Address of the name of the item(s) to remove. If null, 
	procedure will use the item BLOCKID instead.
bid_ITEM: Optional. Copy of the item BLOCKID. Parameter ignored if the name 
	of the rich-text item is provided. A blank BLOCKID structure is provided 
	by the constant ebid_NULLBLOCKID.
RETURN:
	eus_SUCCESS if no error occured
	!eus_SUCCESS if invalid parameters were passed
	the Notes API error code otherwise

--- revision history -------
2/21/00 PR
+ improved safety by means of better context-checking
+ completed standard documentation

9/12/99 PR: logic optimization
12/13/98 PR: created		*/
STATUS eus_RemoveItem( NOTEHANDLE  h_NOTE, 
						char  pc_ITMNM[], 
						const BLOCKID  bid_ITEM)	{
	WORD  us;

	if (!( h_NOTE && (pc_ITMNM && *pc_ITMNM || !ISNULLBLOCKID( bid_ITEM))))
		return !eus_SUCCESS;

	//delete the item from the note by the method implied by the passed-in 
	//	arguments
	if (pc_ITMNM && *pc_ITMNM)
		if (!NSFItemIsPresent( h_NOTE, pc_ITMNM, us = strlen( pc_ITMNM)))
			return eus_SUCCESS;
		else
			return NSFItemDelete( h_NOTE, pc_ITMNM, us);
	else
		return NSFItemDeleteByBLOCKID( h_NOTE, bid_ITEM);

	return NULL;	//should never get here; line included to avoid compiler 
					//	warning
} //eus_RemoveItem(


/** ef_TextListContainsEntry( ***
Tells whether a given Notes text-list contains a given string. A containment 
test that cannot be conducted because the inputs are clearly wrong is 
treated as though the string is not contained in the text-list.

--- parameters & return ------
PV: address of the text-list to check
f_ITEM_TYPE_PRESENT: tells whether a TYPE_TEXT_LIST item-type header precedes 
	the LIST structure of the text-list
PC: address of the string to check for in the text-list
pus: Optional Output. Address of variable in which to store the zero-based 
	position in the list at which a match was found. If null pointer 
	provided, procedure ignores this functionality.
pus_count: Optional Output. Address of variable in which to store the 
	one-based count of the numbe of items in the list. If null pointer 
	provided, procedure ignores this functionality.
RETURN: TRUE if string was found in the text-list; FALSE if not or if the 
	search could not be conducted due to an invalid argument

--- suggested enhancement ----
2/21/00 PR: allow for case-insensitive testing (will require signature 
	adjustment)

--- revision history ---------
2/21/00 PR
+ added optional position-found and count outputs, thus changing procedure's 
  signature
+ standard documentation
+ improved error handling

12/2/98 PR: created			*/
BOOL ef_TextListContainsEntry( void *const  PV, 
								const BOOL  f_ITEM_TYPE_PRESENT, 
								const char  PC[], 
								WORD *const  pus, 
								WORD *const  pus_count)	{
	const UINT  UI = PC ? strlen( PC) : NULL;
	const WORD  us_LEN_STR = (WORD) UI;

	WORD  us_count, us;

	//if any parameters are obviously bad or if the string passed in is too 
	//	long, short-circuit with general failure
	if (!( PV && PC && UI == us_LEN_STR && (f_ITEM_TYPE_PRESENT ? 
									*(WORD *) PV == TYPE_TEXT_LIST : TRUE)))
		return FALSE;

	if (pus)
		*pus = NULL;
	if (pus_count)
		*pus_count = NULL;

	//for each entry in the list...
	us_count = ListGetNumEntries( PV, f_ITEM_TYPE_PRESENT);
	for (us = 0; us < us_count; us++)	{
		//if the entry matches the specified string, set outputs as needed 
		//	and return that the list does contain the string
		if (f_TextListEntryMatches( PV, f_ITEM_TYPE_PRESENT, us, PC, 
															us_LEN_STR))	{
			if (pus)
				*pus = us;
			if (pus_count)
				*pus_count = us_count;
			return TRUE;
		}
	} //for (us = 0; us < us_count

	//no entry matched the specified string, so return that the list does not 
	//	contain the string
	return FALSE;
} //ef_TextListContainsEntry(


/** ef_TextListEntryMatches( ***
Tells whether the specified entry matches a given string. A match test that 
cannot be conducted because the inputs are clearly wrong is treated as though 
the string and "entry" do not match.

--- parameters & return ------
PV: address of the text-list to check
f_ITEM_TYPE_PRESENT: tells whether a TYPE_TEXT_LIST item-type header precedes 
	the LIST structure of the text-list
US: zero-based index of list entry to test for match
PC: address of the string to test aginst the text-list entry
RETURN: TRUE if string was found in the text-list; FALSE if not or if the 
	search could not be conducted due to an invalid argument

--- suggested enhancement ----
2/21/00 PR: allow for case-insensitive testing (will require signature 
	adjustment)

--- revision history ---------
2/21/00 PR: created			*/
BOOL ef_TextListEntryMatches( void *const  PV, 
								const BOOL  f_ITEM_TYPE_PRESENT, 
								const WORD  US, 
								const char  PC[])	{
	const UINT  UI = PC ? strlen( PC) : NULL;
	const WORD  us_LEN_STR = (WORD) UI;

	if (!( PV && PC && UI == us_LEN_STR && (f_ITEM_TYPE_PRESENT ? 
									*(WORD *) PV == TYPE_TEXT_LIST : TRUE)))
		return FALSE;

	return f_TextListEntryMatches( PV, f_ITEM_TYPE_PRESENT, US, PC, 
																us_LEN_STR);
} //ef_TextListEntryMatches(


/** f_TextListEntryMatches( ***
Tells whether the specified entry matches a given string. Inlined for speed.

--- parameters & return ------
PV: address of the text-list to check
f_ITEM_TYPE_PRESENT: tells whether a TYPE_TEXT_LIST item-type header precedes 
	the LIST structure of the text-list
US: zero-based index of list entry to test for match
PC: address of the string to test aginst the text-list entry
us_LEN: Optional Input. Length of the given string to use in the test. If 
	zero, procedure will evaluate length itself.
RETURN: TRUE if string was found in the text-list; FALSE if not or if the 
	search could not be conducted due to an invalid argument

--- suggested enhancement ----
2/21/00 PR: allow for case-insensitive testing (will require signature 
	adjustment)

--- revision history ---------
2/21/00 PR: created			*/
_inline BOOL f_TextListEntryMatches( void *const  PV, 
										const BOOL  f_ITEM_TYPE_PRESENT, 
										const WORD  US, 
										const char  PC[], 
										const WORD  us_LEN)	{
	const UINT  UI = us_LEN ? us_LEN : PC ? strlen( PC) : NULL;
	const WORD  us_LEN_STR = (WORD) UI;

	BYTE * puc;
	WORD  us_len;

	_ASSERTE( PV && PC && UI == us_LEN_STR && (f_ITEM_TYPE_PRESENT ? 
									*(WORD *) PV == TYPE_TEXT_LIST : TRUE));

	if (eus_SUCCESS != ListGetText( PV, f_ITEM_TYPE_PRESENT, US, &puc, 
																	&us_len))
		return FALSE;

	return us_len == us_LEN_STR && memcmp( puc, PC, us_LEN_STR) == ei_SAME;
} //f_TextListEntryMatches(


/** eus_TestTextListItemHasEntry( ***
Tests whether a purported text-list item contains a given entry. If the given 
item is not a text-list, procedure returns success, but entry not found.

--- parameters & return -------
h_NOTE: handle the the open note containing the named text-list item
pc_ITMNM: pointer to the name of the text-list item
pc_ENTRY: address of the string to check for in the text list
pf_found: Output. Address of the flag variable in which to store whether the 
	given entry was found in the purported text-list item.
RETURN:
	eus_SUCCESS if no error occured
	!eus_SUCCESS if invalid parameters were passed
	the Notes API error code otherwise

--- suggested enhancements ----
2/21/00 PR
+ Change the pf_found output so that it optionally outputs the zero-based 
  position at which the given entry was found, or ei_NOT_FOUND if not found 
  at all. Also so that it optionally outputs the one-based count of the 
  number of items in the list. This will of course require checking that 
  dependent code is updated to support this behavioral change.
+ allow for case-insensitive testing (will require more signature adjustment)

--- revision history ----------
2/21/00 PR
+ standard documentation
+ minor logic adjustment, including support of signature change to 
  ef_TextListContainsEntry()

9/12/99 PR: minor logic adjustment
1/16/99 PR: created			*/
STATUS eus_TestTextListItemHasEntry( NOTEHANDLE  h_NOTE, 
										char  pc_ITMNM[], 
										const char  pc_ENTRY[], 
										BOOL *const  pf_found)	{
	WORD  us_type;
	BLOCKID  bid;
	STATUS  us_err;

	if (!( h_NOTE && pc_ITMNM && *pc_ITMNM && pf_found))
		return !eus_SUCCESS;

	*pf_found = FALSE;

	if (us_err = NSFItemInfo( h_NOTE, pc_ITMNM, (WORD) strlen( pc_ITMNM), 
												NULL, &us_type, &bid, NULL))
		return ERR( us_err) == ERR_ITEM_NOT_FOUND ? eus_SUCCESS : us_err;
	if (us_type != TYPE_TEXT_LIST)
		return eus_SUCCESS;

	*pf_found = ef_TextListContainsEntry( OSLockBlock( void, bid), TRUE, 
													pc_ENTRY, NULL, NULL);
	OSUnlockBlock( bid);

	return eus_SUCCESS;
} //eus_TestTextListItemHasEntry(


/** eus_AttemptCopyTextListItemEntry( ***
Attempts to copy the string at the specified element of the named text-list 
item. If the copy cannot be made for obvious reasons (e.g. not a text-list 
item, no such element, null string), procedure is successful, but no copying 
occurs.

--- parameters & return ----
h_NOTE: handle the the open note containing the named text-list item
pc_ITMNM: pointer to the name of the text-list item
us_ENTRY: the zero-based index of the element in the text-list item to be 
	retrieved
ppc: Output. Pointer to the variable in which to store a pointer to a copy 
	of the string at the specified text-list element. CALLER IS RESPONSIBLE 
	for freeing the resources associated with the copied string. Variable is 
	guaranteed to be NULL if no copying (and thus resource allocation) 
	occurred.
RETURN:
	eus_SUCCESS if no error occured
	!eus_SUCCESS if invalid parameters were passed
	ERR_MEMORY if required memory allocation failed
	the Notes API error code otherwise

--- revision history -------
2/21/00 PR: minor logic adjustment

9/12/99 PR
+ minor logic adjustment; handled not-found error
+ completed standard documentation

1/22/99 PR: created			*/
STATUS eus_AttemptCopyTextListItemEntry( NOTEHANDLE  h_NOTE, 
											char  pc_ITMNM[], 
											const WORD  us_ENTRY, 
											char * *const  ppc)	{
	WORD  us_type;
	BLOCKID  bid;
	STATUS  us_err;

	if (!( h_NOTE && pc_ITMNM && *pc_ITMNM && ppc))
		return !eus_SUCCESS;

	*ppc = NULL;

	if (us_err = NSFItemInfo( h_NOTE, pc_ITMNM, (WORD) strlen( pc_ITMNM), 
												NULL, &us_type, &bid, NULL))
		return ERR( us_err) == ERR_ITEM_NOT_FOUND ? eus_SUCCESS : us_err;
	if (us_type != TYPE_TEXT_LIST)
		return eus_SUCCESS;

	us_err = eus_CreateTextListEntryCopy( us_ENTRY, OSLockBlock( void, bid), 
																TRUE, ppc);
	if (us_err == ERR_ODS_NO_SUCH_ENTRY)
		us_err = eus_SUCCESS;
	OSUnlockBlock( bid);

	return us_err;
} //eus_AttemptCopyTextListItemEntry(


/** eus_IsRtf( ***
Determines whether the specified item is a Notes Rich-Text item. If item 
doesn't exist on the note, procedure returns success.

--- parameters & return ----
h_NOTE: handle to the open note containing the named rich-text item
pc_ITMNM: Optional. Pointer to the name of the rich-text item. If null, 
	procedure will use the item BLOCKID instead.
bid_CONTENTS: Optional. Copy of the item-contents BLOCKID. Parameter ignored 
	if the name of the rich-text item is provided.
pf:	Output. Address of a boolean variable in which to store whether the 
	specified item is a rich-text item (TRUE) or not (FALSE). Guaranteed to 
	be FALSE if a Notes API error is encountered.
RETURN:
	eus_SUCCESS if no error occured
	!eus_SUCCESS if invalid parameters were passed
	the Notes API error code otherwise

--- revision history -------
2/21/00 PR: created			*/
STATUS eus_TestIsRtf( NOTEHANDLE  h_NOTE, 
						char pc_ITMNM[], 
						const BLOCKID  bid_CONTENTS, 
						BOOL *const  pf)	{
	STATUS  us_err;

	if (!( h_NOTE && (pc_ITMNM && *pc_ITMNM || !ISNULLBLOCKID( 
														bid_CONTENTS)) && pf))
		return !eus_SUCCESS;

	*pf = FALSE;

	//if the name of the item was provided...
	if (pc_ITMNM && *pc_ITMNM)	{
		WORD  us;

		//determine whether it's a rich-text item
		if (us_err = NSFItemInfo( h_NOTE, pc_ITMNM, (WORD) strlen( 
										pc_ITMNM), NULL, &us, NULL, NULL))	{
			if (ERR( us_err) == ERR_ITEM_NOT_FOUND)
				return us_err;
		}else
			*pf = us == TYPE_COMPOSITE;
	//else determine whether the content specified by the content BLOCKID 
	//	is a rich-text item
	}else	{
		*pf = *OSLockBlock( WORD, bid_CONTENTS) == TYPE_COMPOSITE;
		OSUnlockBlock( bid_CONTENTS);
	} //if (pc_ITMNM && *pc_ITMNM)

	return eus_SUCCESS;
} //eus_IsRtf(



/** eus_AttemptCopyTextItemContent( ***
Attempts to copy the text content of the specified text or text-list item. 
Content contained in elements following the first element of a text-list 
item is copied only if a delimiting string is provided. If the copy cannot be 
made for obvious reasons (e.g. not a text-list item, no such element, null 
string), procedure is successful, but no copying occurs.

--- implementation NOTE -----
9/12/99 PR: "implosion" of text-list content not yet implemented, for lack of 
	time and immediate need
	2/21/00 PR: consider NSFItemConvertToText when ready to implement this

--- suggested enhancement ---
9/12/99 PR: length (pul) output will be fully effective only once 9/12/99 
	module enhancement is implemented

--- parameters & return -----
h_NOTE: handle to the open note containing the named text-list item
pc_ITMNM: pointer to the name of the text-list item
pc_DELIMIT: Optional. Pointer to a string of characters specifying how a 
	text-list's elements' content should be delimited in the return content. 
	If NULL is passed, only the first element's content of a text-list will 
	be copied to the return content.
ppuc: Output. Pointer to the variable in which to store a pointer to the 
	return content. The return content is guaranteed to be null terminated 
	(although the first null-terminator is not guaranteed to be the last).
pul: Optional Output. Pointer to the variable in which to store the length 
	of the return content. If NULL is passed, this output is suppressed.
RETURN:
	eus_SUCCESS if no error occured
	!eus_SUCCESS if invalid parameters were passed
	ERR_MEMORY if required memory allocation failed
	the Notes API error code otherwise

--- revision history --------
2/21/00 PR: minor documentation correction & logic adjustment
9/12/99 PR: created			*/
STATUS eus_AttemptCopyTextItemContent( NOTEHANDLE  h_NOTE, 
										char  pc_ITMNM[], 
										char  pc_DELIMIT[], 
										BYTE * *const  ppuc, 
										DWORD *const  pul)	{
	WORD  us_type;
	DWORD  ul_len;
	BLOCKID  bid;
	STATUS  us_err;

	if (!( h_NOTE && pc_ITMNM && ppuc))
		return !eus_SUCCESS;

	*ppuc = NULL;
	if (pul)
		*pul = NULL;

	if (us_err = NSFItemInfo( h_NOTE, pc_ITMNM, (WORD) strlen( pc_ITMNM), 
											NULL, &us_type, &bid, &ul_len))
		return ERR( us_err) == ERR_ITEM_NOT_FOUND ? eus_SUCCESS : us_err;

	//if a simple text item...
	if (us_type == TYPE_TEXT)	{
		//copy content if not blank
		if (!( *ppuc = malloc( (ul_len -= sizeof( WORD)) + 1)))
			return ERR_MEMORY;
		memcpy( *ppuc, OSLockBlock( void, bid), ul_len);
		OSUnlockBlock( bid);
		(*ppuc)[ ul_len++] = NULL;

if (pul)
*pul = ul_len;
	//else if a text-list item...
	}else if (us_type == TYPE_TEXT_LIST)	{
		WORD  us_entries;
		void * pv = OSLockBlock( void, bid);

		//if we're not delimiting the text-list element content or if there's 
		//	only one element...
		if (!pc_DELIMIT || (us_entries = ListGetNumEntries( pv, TRUE)) == 
																		1)	{
			//copy the first element's content if not blank
			if (us_err = eus_CreateTextListEntryCopy( 0, pv, TRUE, ppuc))	{
				OSUnlockBlock( bid);
				return us_err;
			}
		//else implode the text-list element content
		}else if (us_entries)	{
			//allocate space for the return string
			//	ul_len - sizeof( LIST) - 
			//	sizeof( WORD) * (entries + 1) + 
			//	strlen( pc_DELIMIT) * entries + 1

			//copy the first element's content

			//for each of the remaining elements...
				//append delimiting string

				//append the element's content

			//null terminate the return content
		//else we've got a blank list, so just return that the _attempt_ was 
		//	successful
		}else
			return eus_SUCCESS;

		OSUnlockBlock( bid);
	//else the item doesn't contain text, so just return that the _attempt_ 
	//	was successful
	}else
		return eus_SUCCESS;

	//if requested, set the length output
//	if (pul)
//		*pul = us_len;

	return eus_SUCCESS;
} //eus_AttemptCopyTextItemContent(


STATUS eus_ItemScan( WORD  us_spare, 
						WORD  us_flags, 
						char * pc_nm, 
						WORD  us_lenNm, 
						void * pv_val, 
						DWORD  ul_lenVal, 
						void * pv_hNote)	{
	return eus_SUCCESS;
} //eus_ItemScan(


