/*____________________________________________________________________________
	Copyright (C) 1999 Pretty Good Privacy, Inc.
	All rights reserved.
	

	$Id: RevealNoteHandle.c,v 1.13.8.1.2.5 2000/08/09 01:23:09 build Exp $
____________________________________________________________________________*/

/*::: MODULE OVERVIEW :::::::::::::
Purpose is to encapsulate the guts of the C code component of the Lotus Notes 
RevealNoteHandle abstract data type ("object") that allows a current document 
open in the user interface to learn its active note handle in a legitimate 
programmatic way. The Notes API NSF Hook and Extension Manager facilities are 
used to this end. Both facilities provide a means of having Notes invoke 
callback functions when events of interest occur. In our case, the events of 
interest are the creation and opening of Notes documents. The NSF Hook 
facility is used to hook the open event; Extension Manager is used to hook 
the creation event. (Unfortunately, neither facility can alone provide 
complete coverage of these events because they behave differently based on 
whether the database being accessed is local or on a server). See the Notes 
API User Guide and Reference databases for more information on these 
facilities. The Extension Manager facility is necessarily implemented by a 
separate DLL (nPGPext.dll in the case of the PGP E-Mail Plug-In) so that the 
entry point function may be set properly.

In essence, the design of the RevealNoteHandle object is for the API 
facilities to log information about each note that is created or opened via 
the UI.  While we cannot be sure through the API facilities that a given 
create or open event was caused via the UI, we can weed out NSFNoteCreate 
invocations that aren't attibuted to a real database (Notes does a lot of 
internal state configuration using regular, temporary notes belonging to no 
particular database).  So, for every invocation associated with a real 
database, we (1) set and log the note's "last-accessed" date-time with the 
time component adjusted so that hundredths of seconds are preserved, (2) log 
the note handle and (3), in the case of the open event, the note's UNID into 
a list mechanism.  If the note handle to the current session of a document is 
needed, a call requesting it is made to this DLL via LotusScript (PostOpen is 
a good event for this; QueryOpen will work for opened notes, but not created 
notes).  The DLL services the request by looking through its list of note 
information for a match.  If a match is found, the corresponding note handle 
is returned.

Of course, only those forms which implement this RevealNoteHandle 
functionality will ever call back requesting a note handle (e.g. mail 
databases in the case of the Notes PGP E-Mail Plug-In), meaning that this 
information is being stored needlessly in many cases because we can't be 
fully sure of anything about database context (like "is this a Notes Mail 
database?") from a just-opened or -created note.  Remember, the NSF Hook and 
Extension Manager facilities cause the callback functions to  be called at 
every note-creation and -open call in the current Notes session.  That's why 
we have to depend on the document itself to call back up and look for it's 
note handle.  Obviously a side-effect of this design is that we endlessly 
collect information about many notes, most of which aren't needed. This 
problem is addressed by calling a clean-up routine regularly to remove aged 
content (see RemoveOldListNodes).

Currently, matches are made according to note last-accessed date and, if 
possible, UNID (no UNID is available at the note creation event).  These 
values are  available to all of Notes' programming interfaces and so can be 
passed on as lookup keys to this DLL.

Note that a small chance exists whereby a request for the note handle will be 
filled with the wrong handle, leading most likely to a Notes PANIC halt. The 
chance owes itself to the fact that the LotusScript LastAccessed property of 
the NotesDocument class is specific to the second, not the hundredth of a 
second that is actually stored in in the note's last-accessed time stamp. 
This module therefore adjusts the last-accessed stamp to preserve the crucial 
second hundreths, but in doing so, for technical reasons, some hour 
information is lost. The chance exists therefore that two different entries, 
especially those created via a NSFNoteCreate() call since then no UNID 
information is stored, could have the same last-accessed time stamp. A couple 
precautions are provided to minimize the problem:

+ In Notes 4.6+, the undocumented LotusScript Handle property of the 
  NotesDocument class may be passed into xul_RequestNoteHandle() to ensure 
  that no non-match is returned. Before Notes 4.6, this property is not 
  available, however.
+ If it seems that the handle being requested is associated with a create 
  event because no "match-me" UNID was provided by the LotusScript call, the 
  list will be searched in an attempt to find an entry that does not contain 
  a UNID member.

--- revision history --------
10/27/99 Version 1.01: Paul Ryan
+ minor documentation adjustment

1/6/99 Version 1.0: Paul Ryan
::::::::::::::::::::::::::::::::::::*/

#include "RevealNoteHandle.h"


#define mui_HEX_DIGITS_IN_UNID  sizeof( UNID) * 2

static NoteInfoNode * mpt_ListHead;


/** xus_NsfHookInit( ***


--- parameters & return ----

RETURN: eus_SUCCESS

--- revision history -------
10/27/99 PR
+ added debug-build protection code
+ documentation adjustment

12/30/98 PR: created		*/
//DOC!!
STATUS LNPUBLIC xus_NsfHookInit( DBHOOKVEC * pt)	{
	_ASSERTE( pt);

	pt->Init = xus_NsfHookInit;
	pt->NoteOpen = us_PostNoteOpen;
	pt->Term = NULL;
	pt->NoteUpdate = NULL;
	pt->DbStampNotes = NULL;

	return eus_SUCCESS;
} //xus_NsfHookInit(


/** us_PostNoteOpen( ***


--- parameters & return ----

RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
12/30/98 PR: created		*/
//DOC!!
static STATUS LNPUBLIC us_PostNoteOpen( DBHOOKVEC * pt, 
										char * pc_USERNM, 
										LIST * pc_GROUPS, 
										DBHANDLE  h_DB, 
										NOTEID  NID, 
										NOTEHANDLE  h_NOTE, 
										const WORD us_flags)	{
	UNID * punid;
	NoteInfo  t_NoteInfo;
	TIMEDATE  td;
	ORIGINATORID  oid;
	STATUS us_error;

	_ASSERTE( pt && pc_USERNM && pc_GROUPS && h_NOTE);

	NSFNoteGetInfo( h_NOTE, _NOTE_OID, &oid);
	punid = (UNID *) &oid;

	//fill the bulk of the NoteInfo structure to be added to the current list
	OSCurrentTIMEDATE( &td);
	t_NoteInfo.td_inserted = td;
	t_NoteInfo.punid = malloc( sizeof( UNID));
	memcpy( t_NoteInfo.punid, punid, sizeof( UNID));
	t_NoteInfo.h_note = h_NOTE;

	//Create a last-accessed timestamp for the note and manipulate it so we 
	//	can later get good matches when LotusScript calls back up to this DLL 
	//	via xul_RequestNoteHandle(). The manipulation is to shift up the ticks
	//	member of the timestamp by seven bits (128 ticks) so that hundredths 
	//	of seconds are boosted to the level of "seconds." The need to do this 
	//	is driven by the fact that LotusScript's LastAccessed property in the 
	//	NotesDocument class is specific only to the second. Our manipulation 
	//	of course kills some hour information, but that info is a lot less 
	//	important to us than the hundredths of a second info. The last-
	//	accessed timestamp is designed to be date-only specific anyway, and 
	//	the manipulated stamp will not persist if the document is not saved.
	t_NoteInfo.td_testStamp = *ptd_PreserveHundredthTicks( &td);

	//add the NoteInfo instance to the current list
	if (us_error = us_AddNoteInfoNode( &t_NoteInfo, &mpt_ListHead, ecs))
		return us_error;

	//take this opportunity to remove timed-out entries from the list
	RemoveOldListNodes( &mpt_ListHead, ecs);

	//Set the last-accessed stamp.  This will persist only if the note is 
	//	saved somewhere down the line.
	NSFNoteSetInfo( h_NOTE, _NOTE_ACCESSED, &td);

	return eus_SUCCESS;
} //us_PostNoteOpen(


/** us_AddNoteInfoNode( ***
Purpose is to add an item into a linked list of NoteInfo instances, 
in a manner that is thread-safe.

--- parameters & return ---
pt_NoteInfo: the NoteInfo instance to add to the list
ppt_ListHead: pointer to the start of the list to which a node will be added
cs: the CriticalSection to use for thread safety
RETURN: eus_SUCCESS if no out-of-memory errors occur, ERR_MEMORY otherwise

--- revision history ------
12/30/98 PR: created		*/
static STATUS us_AddNoteInfoNode( const NoteInfo *const  pt_NoteInfo, 
									NoteInfoNode * *const  ppt_ListHead, 
									CRITICAL_SECTION  cs)	{
	NoteInfoNode * pt_Node = *ppt_ListHead;

	_ASSERTE( (ppt_ListHead && pt_NoteInfo && pt_NoteInfo->h_note));

	//make sure current thread has exclusive access to the possibly shared 
	//	list
	EnterCriticalSection( &cs);

	//if the first node exists, allocate space for another node
	if (pt_Node)	{
		if (!( pt_Node = malloc( sizeof( NoteInfoNode))))
			goto errJump;

		pt_Node->pt_next = *ppt_ListHead;
		*ppt_ListHead = pt_Node;
	//otherwise allocate space for the first entry in the list
	}else	{
		if (!( pt_Node = *ppt_ListHead = malloc( sizeof( NoteInfoNode))))
			goto errJump;

		pt_Node->pt_next = NULL;
	} //if (pt_Node)

	//copy in the node information
	memcpy( &pt_Node->t_NoteInfo, pt_NoteInfo, sizeof( NoteInfo));

errJump:
	//allow other threads access to the list
	LeaveCriticalSection( &cs);

	//return success or failure depending on whether we ran out of memory
	return pt_Node ? eus_SUCCESS : ERR_MEMORY;
} //us_AddNoteInfoNode(


/** ptd_PreserveHundredthTicks( ***


--- parameter & return ---

--- revision history -----
12/1/98 PR: created		*/
//DOC!!
static __inline TIMEDATE * ptd_PreserveHundredthTicks( 
													TIMEDATE *const  ptd)	{
	const UINT  i_FORCE_SECONDS_ONLY = 7, 
				i_MASK_EXCESS = 0x003fffff;
	
	DWORD  ul_date, ul_ticks;

	_ASSERTE( ptd);

	ul_date = TimeExtractDate( ptd);
	ul_ticks = TimeExtractTicks( ptd) << i_FORCE_SECONDS_ONLY & i_MASK_EXCESS;

	TimeConstruct( ul_date, ul_ticks, ptd);

	return ptd;
} //ptd_PreserveHundredthTicks(


/** RemoveOldListNodes( ***
Purpose is to remove nodes in the linked list of NoteInfo instances which are 
older than a specified timeout period.  This prevents the list from getting 
overly large.

--- parameter ----------
ppt_ListHead: pointer to the start of the list from which old notes will 
	be deleted

--- revision history ---
10/9/98 PR: created		*/
static void RemoveOldListNodes( NoteInfoNode * *const  ppt_ListHead, 
									CRITICAL_SECTION  cs)	{
	//use a 5 minute timeout
	const int i_TIMEOUT = 5 * 60;

	NoteInfoNode * pt_Node, 
				* pt_Prev = NULL;
	TIMEDATE  td_now;

	_ASSERTE( ppt_ListHead);

	//if there aren't any notes in the list, short circuit
	if (!( pt_Node = *ppt_ListHead))
		return;

	//get current time
	OSCurrentTIMEDATE( &td_now);

	//make sure current thread has exclusive access to the 
	//	probably shared list
	EnterCriticalSection( &cs);

	//for each node in the list of NoteInfo instances...
	do	{
		//if the created date falls beyond the timeout, remove the node
		if (TimeDateDifference( &td_now, &pt_Node->t_NoteInfo.td_inserted) > 
																i_TIMEOUT)	{
			//if we're deleting the head node, move the pointer 
			//	to the "new" head node
			if (!pt_Prev)
				*ppt_ListHead = (*ppt_ListHead)->pt_next;
			//else link the previous and next nodes together 
			//	to keep the list intact
			else
				pt_Prev->pt_next = pt_Node->pt_next;

			FreeNoteInfoNode( &pt_Node);
		}else
			pt_Prev = pt_Node;
	} while (pt_Node = pt_Prev ? pt_Prev->pt_next : *ppt_ListHead);

	//allow other threads access to the list
	LeaveCriticalSection( &cs);
} //RemoveOldListNodes(


/** FreeNoteInfoNode( ***
Purpose is to free previously allocated memory associated with the passed-in 
NoteInfoNode structure.

--- parameter ----------
ppt: Input & Output. Two-level pointer to the NoteInfo structure whose 
	memory needs to be freed.  All pointers associated with the structure 
	(internal punid member and the pointer to the structure itself) will 
	be set to null as a result of the call.

--- revision history ---
10/14/98 PR: created	*/
static void FreeNoteInfoNode( NoteInfoNode * *const  ppt)	{
	NoteInfoNode * pt;

	//if input doesn't point to anything, short-circuit
	if (!( ppt && (pt = *ppt)))
		return;

	//if necessary, free & nullify the NoteInfoNode's UNID member pointer
	if (pt->t_NoteInfo.punid)	{
		free( pt->t_NoteInfo.punid);
		pt->t_NoteInfo.punid = NULL;
	}

	//free & nullify the pointer to the NoteInfoNode structure
	free( pt);
	*ppt = NULL;
} //FreeNoteInfoNode(


/** xul_RequestNoteHandle( ***
Function provides a means by which LotusScript (or VBA, for that matter) may 
obtain the handle to a note's current "session."  The handle should have 
already been logged by this DLL during an Extension Manager callback event 
associated with NSFNoteCreate, NSFNoteOpen or NSFNoteOpenByUNID.  This 
function matches passed-in parameters against its log entries and, if a match 
is found, returns the corresponding handle to the caller.

--- parameter ----------
d_LastAccessed: Double value representing the date-time stored in the note's 
	last-accessed property (LastAccessed in LotusScript).  The contents 
	correspond to a LotusScript variant of type Date; see LotusScript help 
	for a full description.  Basically the integer component of the double 
	value represents the day, and the fractional component represents the 
	time of day, specific only to the second, not hundredth of a second.
pc_UNID: String-representation of the UNID associated with the note whose 
	handle is being requested. Parameter will be ignored when compared 
	against note entries logged via the create event.
h_NOTE_CHECK: Optional. Possible handle to the note whose handle is being 
	requested. Confused? This "possible" handle is derived from the 
	undocumented LotusScript Handle property of a NotesDocument object. We 
	use this handle as a protection check against a Notes PANIC halt caused 
	when a note is opened more than once within a given second. This is a 
	concern because the LotusScript LastAccessed property of a NotesDocument 
	object is specific only to the second. This functionality check is 
	skipped if the parameter if passed in null.

	Probably a useless parameter now, but it can't hurt, and may provide 
	some protection in weird circumstances...
RETURN: if a match was found, the handle to the active note; otherwise null

--- revision history ---
12/5/98 PR: created		*/
unsigned long xul_RequestNoteHandle( 
									double  d_LastAccessed, 
									const char *const  pc_UNID, 
									const NOTEHANDLE  h_NOTE_CHECK)	{
	const DWORD  ul_TICKS_PER_DAY = 60 * 60 * 24 * 100, 
					ul_JULIAN_OFFSET = 2415019;
	const float  fl_TINY_BIT = (float) 0.0001;

	TIMEDATE  td;
	TIME  time;
	UNID  unid;
	BOOL  f_unidIsNull;
	NOTEHANDLE  h = NULL;

	//if the last-accessed parameter is null, short-circuit with failure
	if (!d_LastAccessed)
		return FALSE;

	//Create a corresponding TIMEDATE from the last-accessed parameter.  
	//	The "Julian offset" constant accounts for the day difference between 
	//	variant dates in LotusScript and the Julian system Notes uses 
	//	internally.  The "tiny bit" ensures that we don't slice off a 
	//	hundredth of a second when getting the correct ticks value.
	TimeConstruct( (DWORD) d_LastAccessed + ul_JULIAN_OFFSET, 
							(DWORD) (fl_TINY_BIT + ul_TICKS_PER_DAY * 
							(d_LastAccessed - (DWORD) d_LastAccessed)), &td);

	//incorporate into the resulting GMT TIMEDATE the time zone 
	//	in use here, the current location
	time.GM = td;
	if (TimeGMToLocal( &time) != eus_SUCCESS)
		return FALSE;
	OSCurrentTimeZone( &time.zone, &time.dst);
	if (TimeLocalToGM( &time) != eus_SUCCESS)
		return FALSE;
	td = time.GM;

	//Fill a UNID structure by converting the string UNID parameter.  If the 
	//	parameter is non-null and can't be converted, failure is returned.
	if (!f_ConstructUnidFromStr( pc_UNID, &unid, &f_unidIsNull))
		return FALSE;

	//for all the nodes in the current list of NoteInfo structures...
	if (mpt_ListHead)	{
		NoteInfoNode * pt_Node;
		UNID * punid;

		EnterCriticalSection( &ecs);
		pt_Node = mpt_ListHead;
		do	{
			//if the NoteInfo structure matches the passed-in parameters 
			//	closely enough...
			punid = pt_Node->t_NoteInfo.punid;
			if (!TimeDateDifference( &td, &pt_Node->t_NoteInfo.td_testStamp) 
										&& (f_unidIsNull || !punid || 
										memcmp( punid, &unid, sizeof( UNID)) 
										== ei_SAME))	{
				//store the corresponding note handle
				h = pt_Node->t_NoteInfo.h_note;

				//If we seem to be dealing with entries logged via the 
				//	create, not open, event or if a passed-in "check" handle 
				//	matches the handle we just found, search no further. 
				//	Otherwise, look for a later match since it's more likely 
				//	that the latest match in the list is the one that's 
				//	required. (This measure lessens the possibility of a 
				//	PANIC halt).
				if (h_NOTE_CHECK)	{
					if (h_NOTE_CHECK == h)
						break;
					else
						h = NULL;
				}else if (f_unidIsNull && !punid || !f_unidIsNull)
					break;
			} //if (!TimeDateDifference(
		} while (pt_Node = pt_Node->pt_next);
		LeaveCriticalSection( &ecs);
	} //if (mpt_ListHead)

	//take this opportunity to remove any too-old nodes from the  
	//	current list of NoteInfo structures
	RemoveOldListNodes( &mpt_ListHead, ecs);

	//return to the caller the handle discovered, if any
	return (long) h;
} //xul_RequestNoteHandle(


/** f_ConstructUnidFromStr( ***
Purpose is to construct a valid UNID instance from a 
string-representation of a UNID (16 bytes).

--- parameters & return ---
pc_UNID: string containing the string-representation 
	of the UNID to be constructed
punid: pointer to the UNID structure to be populated
pf_IsNull: pointer to the flag notifying whether the passed-in 
	UNID was blank or non-existent
RETURN: TRUE if construction was successful, FALSE otherwise

--- revision history ------
10/14/98 PR: created	*/
static BOOL f_ConstructUnidFromStr( const char  pc_UNID[], 
									UNID *const  punid, 
									BOOL *const  pf_IsNull)	{
	static char  pc_ZeroUnid[ mui_HEX_DIGITS_IN_UNID + 1];
	static BOOL  f_firstTimeThru = TRUE;

	//initialize pointer to start of UNID string
	const char * pc = pc_UNID;
	DWORD  ul;

	_ASSERTE( pc_UNID && punid && pf_IsNull);

	//the first time through, set a buffer to all zeros
	if (f_firstTimeThru)	{
		memset( pc_ZeroUnid, '0', mui_HEX_DIGITS_IN_UNID);
		f_firstTimeThru = FALSE;
	}

	//if no UNID string was passed in, or if it consists of all zeros, 
	//	return that the UNID is, for all intents and purposes, null
	if (!pc_UNID || !*pc_UNID || strcmp( pc_UNID, pc_ZeroUnid) == ei_SAME)	{
		*pf_IsNull = TRUE;
		return TRUE;
	}

	//if UNID string has an incorrect length, short-circuit with failure
	if (strlen( pc_UNID) != mui_HEX_DIGITS_IN_UNID)
		return FALSE;

	//fill the UNID structure with DWORDS converted from 8-byte 
	//	hex-numerical strings
	if (!f_aToHexUl( pc_UNID, &ul))
		return FALSE;
	punid->File.Innards[ 1] = ul;

	if (!f_aToHexUl( pc_UNID + 0x08, &ul))
		return FALSE;
	punid->File.Innards[ 0] = ul;

	if (!f_aToHexUl( pc_UNID + 0x10, &ul))
		return FALSE;
	punid->Note.Innards[ 1] = ul;

	if (!f_aToHexUl( pc_UNID + 0x18, &ul))
		return FALSE;
	punid->Note.Innards[ 0] = ul;

	*pf_IsNull = FALSE;
	return TRUE;
} //f_ConstructUnidFromStr(


/** f_aToHexUl( ***
Purpose is to convert the first 8 characters of the passed-in string 
of hex-numeric characters into a corresponding unsigned long number.

--- parameters & return ---
PC: String of hex-numeric characters.  The first 8 characters will be 
	converted to an unsigned long.  Null-termination is ignored.
pul: pointer to the unsigned long in which to store an unsigned long 
	representation of the string's first 8 characters
RETURN: TRUE if conversion was successful, FALSE otherwise

--- revision history ------
10/14/98 PR: created	*/
static BOOL f_aToHexUl( const char *const  PC, 
						unsigned long *const  pul)	{
	//static for speed
	static const UINT  ui_BITS_PER_HEX_DIGIT = 4, 
						ui_HEX_DIGITS_IN_UL = sizeof( unsigned long) * 2;

	UINT  ui, ui_c;
	BOOL  f_btwn0n9, f_btwnAnF;
	unsigned long  ul = (unsigned long) NULL;

	_ASSERTE( PC && pul);

	//for each of the first eight bytes in string...
	for (ui = 0; ui < ui_HEX_DIGITS_IN_UL; ui++)	{
		//if byte cannot be interpreted as a hex digit, 
		//	short-circuit with failure
		ui_c = PC[ ui];
		if (!( (f_btwn0n9 = ('0' <= ui_c && ui_c <= '9')) || 
								(f_btwnAnF = ('A' <= ui_c && ui_c <= 'F')) || 
								'a' <= ui_c && ui_c <= 'f'))
			return FALSE;

		//get the hex value if the hex-string digit
		ui_c = f_btwn0n9 ? ui_c - '0' : 0xA + (f_btwnAnF ? ui_c - 'A' : 
																ui_c - 'a');

		//increment local variable by the interpreted digit times the 
		//	order-of-magintude of its "place" in the ultimate unsigned long
		ul += (unsigned long) ui_c << ui_BITS_PER_HEX_DIGIT * 
										(ui_HEX_DIGITS_IN_UL * 2 - (ui + 1));
	} //for (ui = 0; ui < sizeof( unsigned long)

	//copy the fully-formed local variable to the unsigned long 
	//	to be returned to caller
	*pul = ul;
	return TRUE;
} //f_aToHexUl(


/** e_ClearRevealNoteHandleContext( ***


--- parameter ----------

--- revision history ---
12/30/98 PR: created			*/
//DOC!!
void e_ClearRevealNoteHandleContext( CRITICAL_SECTION  cs)	{
	NoteInfoNode * pt_Node, * pt_Next;

	//make sure current thread has exclusive access to the possibly shared 
	//	list
	EnterCriticalSection( &cs);

	//if necessary, free any left-over NoteInfo instances
	if (pt_Node = mpt_ListHead)
		do	{
			pt_Next = pt_Node->pt_next;
			FreeNoteInfoNode( &pt_Node);
		} while (pt_Node = pt_Next);

	//allow other threads access to the list
	LeaveCriticalSection( &cs);
} //e_ClearRevealNoteHandleContext(


/** xus_PostNoteCreate( ***


--- parameter & return ---

--- revision history -----
12/30/98 PR: created		*/
//DOC!!
STATUS xus_PostNoteCreate( NOTEHANDLE  h_NOTE)	{
	NoteInfo  t_NoteInfo;
	TIMEDATE  td;
    STATUS us_error;

	if( !h_NOTE)
		return !eus_SUCCESS;

	//fill the NoteInfo structure to be added to the current list
	NSFNoteGetInfo( h_NOTE, _NOTE_ACCESSED, &td);
	t_NoteInfo.td_inserted = td;
	t_NoteInfo.punid = NULL;
	t_NoteInfo.h_note = h_NOTE;

	//Get the note's last-accessed timestamp and manipulate it so we can 
	//	later get good matches when LotusScript calls back up to this DLL via 
	//	xul_RequestNoteHandle(). The manipulation is to shift up the ticks 
	//	member of the timestamp by seven bits (128 ticks) so that we are 
	//	dealing in seconds, not hundredths of seconds. The need to do this is 
	//	driven by the fact that LotusScript's LastAccessed property in the 
	//	NotesDocument class is specific only to the second. Our manipulation 
	//	of course kills some hour information, but that info is a lot less 
	//	important to us than the hundredths of a second info. The last-
	//	accessed timestamp is designed to be date-only specific anyway, and 
	//	the manipulated stamp will not persist if the document is not saved.
	t_NoteInfo.td_testStamp = *ptd_PreserveHundredthTicks( &td);

	//add the NoteInfo instance to the current list
	if (us_error = us_AddNoteInfoNode( &t_NoteInfo, &mpt_ListHead, ecs))
		return us_error;

	//take this opportunity to expunge any too-old NoteInfo nodes
	RemoveOldListNodes( &mpt_ListHead, ecs);

	//Set the last-accessed stamp.  This will persist only if the note is 
	//	saved somewhere down the line.
	NSFNoteSetInfo( h_NOTE, _NOTE_ACCESSED, &td);

	return ERR_EM_CONTINUE;
} //xus_PostNoteCreate(


/** xul_R45getHandleIllegit( ***


--- parameter & return ----


--- revision history ------
12/13/98 PR: created		*/
//DOC!!
unsigned long xul_R45getHandleIllegit( 
									  const BYTE *const  puc_NotesDocument)	{
	const int  i_45_OFFSET_TO_HANDLE = 0xEC;

	return *(unsigned long *) (puc_NotesDocument + i_45_OFFSET_TO_HANDLE);
} //xul_R45getHandleIllegit(


