/*____________________________________________________________________________
	Copyright (C) 2000 Pretty Good Privacy, Inc.
	All rights reserved.


	$Id: nPGPNts.c,v 1.15.8.1.2.10 2000/08/09 01:23:14 build Exp $
____________________________________________________________________________*/

/*::: MODULE OVERVIEW :::::::::::::
Provides central management services for the back-end infrastructure of the 
Lotus Notes PGP Plug-In.

--- revision history --------
2/27/00: Version 1.1.1: Paul Ryan
+ Notes Mail R5-template compatibility
+ documentation adjustments, logic enhancements
+ Bug fix of behavior where a pre-existing encoded rich-text attachment was 
  not being removed during a new encoding. Fix relates primarily to message 
  forward and "reply with history" operations.

9/12/99: Version 1.1: Paul Ryan
+ PGP 6.5.1 compatibility
+ logic enhancements
   - support for reading into memory rich-text content resident in a 
     compressed attachment
   - default handling of Notes addresses in the user-friendlier abbreviated 
     format
   - improved Domino Directory name-lookup functionality to emulate better 
     Notes' normal algorithm, and thereby enhanced the plug-in's group-
	 resolution capability
+ documentation enhancements

1/29/99 Version 1.0: Paul Ryan
::::::::::::::::::::::::::::::::::::*/

#include "nPGPNts.h"


//global-scope declarations
HINSTANCE  eh_Instance;
CRITICAL_SECTION  ecs;			//opaque storage needed for thread 
				//	synchronization under Win32

//module-scope declarations
#define mus_MAXBUF_EXTFILENM  255
static const char  mpc_PGP_ARMOR_EXT[] = ".asc", 
					mpc_ROOTFILENM_PGP_RTF[] = "pgp.rtf";

static char  mpc_FileNmPgpRtf[ sizeof( mpc_ROOTFILENM_PGP_RTF) + 
											sizeof( mpc_PGP_ARMOR_EXT) - 1];

ResettableRtfInfo  mt_RtfResettable;


/** xs_DecryptVerifyNote( ***
Purpose is to PGP decrypt/verify the PGP encrypted/signed content present in 
the specified note.

--- parameters & return ----
h_NOTE: handle to the note to undergo PGP decryption/verification
pc_ITMNM_RTF: pointer to a string telling the name of the rich-text field to 
	be used in the decryption/verification
pus_MsgChanged: pointer to the variable of VisualBasic integer type in which 
	to return whether message content was changed (ms_VB_TRUE) or not 
	(ms_VB_FALSE)
RETURN: eus_SUCCESS if no error occured. If the user canceled a PGP dialog, 
	ei_USER_ABORT. Otherwise !eus_SUCCESS or, if available, the Notes API 
	error code.

--- revision history -------
9/12/99 PR: documentation adjustment
2/2/99 PR: created			*/
short xs_DecryptVerifyNote( NOTEHANDLE  h_NOTE, 
							char  pc_ITMNM_RTF[], 
							short *const  pus_MsgChanged)	{
	RtfCursor  t_cursor;
	RtfContext  t_RtfContext;
	DWORD  ul_objId, ul_lenContent;
	BLOCKID  bid_item;
	STATUS  us_err;
	int  i_err, i_errNonFatal = NULL;

	if (!( h_NOTE && pc_ITMNM_RTF && pus_MsgChanged))
		return !eus_SUCCESS;

	//default the message-changed flag to say that no change has occurred
	*pus_MsgChanged = ms_VB_FALSE;

	//initialize resources associated with the rich-text handling 
	//	we're going to do
	if (us_err = eus_InitializeRtfContext( h_NOTE, pc_ITMNM_RTF, 
													&t_cursor, &t_RtfContext))
		return us_err;

	//if the special PGP-Armored file attachment containing Notes rich-text 
	//	is present on the note...
	if (us_err = us_LocatePgpRtfAttachment( t_cursor, &t_RtfContext, 
												h_NOTE, &ul_objId, 
												&ul_lenContent, &bid_item))
		i_errNonFatal = us_err;
	if (bid_item.pool && !i_errNonFatal)	{
		//PGP decode the attachment's rich-text content
		i_err = i_PgpDecodeUsingRichTextObj( ul_objId, ul_lenContent, 
											bid_item, h_NOTE, pc_ITMNM_RTF, 
											&t_RtfContext, &i_errNonFatal);
		if (i_err && !i_errNonFatal)
			goto errJump;

		//if no error occurred, inform the caller that the message content 
		//	changed
		if (!i_errNonFatal)
			*pus_MsgChanged = ms_VB_TRUE;
	} //if (bid_item.pool && !i_errNonFatal)

	//if no the special rich-text version doesn't exist or if a non-fatal 
	//	error occurred in decoding that version, we will decode the message's 
	//	largest PGP ASCII block
	if (!bid_item.pool || i_errNonFatal)	{
		BOOL  f_MsgChanged;
		if (i_err = i_PgpDecodeUsingAsciiText( h_NOTE, pc_ITMNM_RTF, 
									t_cursor, &t_RtfContext, &f_MsgChanged))
			goto errJump;

		//if it's true, inform the caller that the message content did change
		if (f_MsgChanged)
			*pus_MsgChanged = ms_VB_TRUE;
	} //if (!bid_item.pool || i_errNonFatal)

	if (i_errNonFatal)	{
		const char *const  ppc_MSG_PART[] = {"An error occurred (", ") "
								"during the attempt to\nPGP decode a "
								"rich-text version of this message.\nThe "
								"plain-text version will be used instead."};
		char  pc_msg[ 255];

		sprintf( pc_msg, "%s%i%s", ppc_MSG_PART[0], i_errNonFatal, 
															ppc_MSG_PART[1]);
		MessageBox( eh_mainWnd, pc_msg, epc_APPNM, MB_OK | 
														MB_ICONINFORMATION);
		i_err = us_err = eus_SUCCESS;
	} //if (i_errNonFatal)

errJump:
	ef_FreeRtfContext( &t_RtfContext);

	//if no errors occurred, return that procedure was successful; otherwise 
	//	return an appropriate error code
	return (short) (us_err + i_err);
} //xs_DecryptVerifyNote(


/** i_PgpDecodeUsingAsciiText( ***


--- parameters & return ----

RETURN: eus_SUCCESS if no error occured; ... otherwise

--- revision history -------
9/12/99 PR: code adjustment associated with an improvement to handling of 
	public-key decoding
1/24/99 PR: created			*/
//DOC!!
static int i_PgpDecodeUsingAsciiText( NOTEHANDLE  h_NOTE, 
										char  pc_ITMNM[], 
										const RtfCursor  t_CURSOR_START, 
										RtfContext *const  pt_rtfContext, 
										BOOL *const  pf_MsgChanged)	{
	DWORD  ul_lenContent, ul_lenBlock;
	char * pc_block, * pc_output = NULL, 
			* pc_input = NULL;
	STATUS  us_err = eus_SUCCESS;
	BOOL  f_failure;
	int  i_err = NULL;

	//Get the textual content of the rich-text field along with its length. 
	//	If no content exists, short-circuit with success. Remember, 
	//	eul_GetRtfText() returns the size of the buffer allocated, not the 
	//	size of the null-terminated string.
	if (f_failure = eul_ERR_FAILURE == (ul_lenContent = eul_GetRtfText( 
									pt_rtfContext, NULL, TRUE, &pc_input)))
		return !eus_SUCCESS;
	if (ul_lenContent <= 1)
		goto errJump;

	//making use of the regular-text version of the content, PGP decode 
	//	just the portion containing the PGP-encoded block
	if (i_err = ei_FindAndDecodePgpBlock( &pc_input, ul_lenContent, 
								&pc_block, &ul_lenBlock, &pc_output, NULL))
		goto errJump;

	//if output was generated by the decoding (none will be if the 
	//	PGP-encoded block is a key block)...
	if (FALSE || pc_output)	{
		//replace the PGP-encoded block in the rich-text field with the 
		//	decoded output
		if (us_err = us_ReplacePgpAsciiRichText( pc_input, pc_block, 
											ul_lenBlock, pc_output, 
											t_CURSOR_START, pt_rtfContext))
			goto errJump;

		//commit to note the changes made to the rich-text field
		if (us_err = eus_CommitChangedRtf( h_NOTE, pc_ITMNM, pt_rtfContext))
			goto errJump;

		//inform the caller that the message content did change
		*pf_MsgChanged = TRUE;
	} //if (pc_output)

errJump:
	//free any memory allocation made by eul_GetRtfText() and free all 
	//	resources devoted to this instance of rich-text handling
	if (pc_input)
		free( pc_input);
	if (pc_output)
		e_FreePgpMem( pc_output);

	return us_err + f_failure + i_err;
} //i_PgpDecodeUsingAsciiText(


/** i_PgpDecodeUsingRichTextObj( ***


--- parameters & return ----
ul_OBJID: Optional.
RETURN: 

--- revision history -------
9/12/99 PR: accommodation of compressed objects (forced into this because the 
	Domino SMTP MTA may decide to compress)
12/15/98 PR: created		*/
//DOC!!
static int i_PgpDecodeUsingRichTextObj( const DWORD  ul_OBJID, 
										const DWORD  ul_LEN_OBJ, 
										const BLOCKID  bid_OBJ_ITEM, 
										NOTEHANDLE  h_NOTE, 
										char  pc_ITMNM_RTF[], 
										RtfContext *const  pt_rtfContext, 
										int *const  pi_errNonFatal)	{
	HANDLE  h;
	BYTE * puc;
	DWORD  ul;
	char * pc_output = NULL;
	STATUS  us_err;
	int  i_err;

	_ASSERTE( h_NOTE && bid_OBJ_ITEM.pool && pc_ITMNM_RTF && pt_rtfContext);

	if (pi_errNonFatal)
		*pi_errNonFatal = FALSE;

	//read the special attachment into a memory buffer
	if (us_err = eus_getObjectContentsInBuffer( ul_OBJID, ul_OBJID ? 
											ebid_NULLBLOCKID : bid_OBJ_ITEM, 
											ul_LEN_OBJ, h_NOTE, NULL, &h))	{
		if (pi_errNonFatal)
			*pi_errNonFatal = us_err;
		return us_err;
	}

	//PGP decode the buffer, then, our work with it done, clear the buffer
	puc = OSLockObject( h);
	i_err = ei_PgpDecodeBuffer( puc, TRUE, &pc_output, &ul);
	OSUnlockObject( h);
	OSMemFree( h);
	if (i_err)	{
		if (pi_errNonFatal && i_err != ei_USER_ABORT)
			*pi_errNonFatal = i_err;
		goto errJump;
	}

	//Replace the current rich-text field with the CD records in the 
	//	decrypted buffer. From here on all errors are fatal because we 
	//	will no longer know the state of the rich-text tracking context.
	if (us_err = eus_ReplaceRtfWithCdStream( pc_output, ul, pt_rtfContext))
		goto errJump;

	//commit to note the changes made to the rich-text field
	if (us_err = eus_CommitChangedRtf( h_NOTE, pc_ITMNM_RTF, pt_rtfContext))
		goto errJump;

	//remove the item used by the note to point to the special attachment
	us_err = NSFNoteDetachFile( h_NOTE, bid_OBJ_ITEM);

errJump:
	if (pc_output)
		e_FreePgpMem( pc_output);

	return us_err + i_err;
} //i_PgpDecodeUsingRichTextObj(


/** us_ReplacePgpAsciiRichText( ***


--- parameters & return ----

RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- side effect ------------


--- revision history -------
7/16/99 PR: changed default nullification of end-cursor member of the RtfSpan 
	structure passed to eus_ReplaceRtfSpanText() to use the standard 
	null-cursor flag
11/23/98 PR: created		*/
//DOC!!
static STATUS us_ReplacePgpAsciiRichText( 
										const char *const  pc_INPUT, 
										const char *const  pc_BLOCK, 
										const DWORD  ul_LEN_BLOCK, 
										const char *const  pc_OUTPUT, 
										RtfCursor  t_cursor, 
										RtfContext *const pt_RtfContext)	{
	long  l;
	const char * pc;
	char * pc_cmp = NULL;
	WORD  us_offset, us = NULL;
	RtfSpan  t_rtfSpan;
	STATUS  us_err;
	BOOL  f_failure;
	int  i_err = NULL;

	_ASSERTE( pc_INPUT && pc_BLOCK && ul_LEN_BLOCK && pc_OUTPUT && 
									t_cursor.puc_location && pt_RtfContext);

	//Copy the first line of the PGP-encoded block into a string buffer. The 
	//	string will be used to find the start of the PGP-encoded block within 
	//	the rich-text field.
	if (f_failure = !(BOOL) (pc_cmp = malloc( l = strstr( pc_BLOCK, 
												epc_CRLF) - pc_BLOCK + 1)))
		goto errJump;
	memcpy( pc_cmp, pc_BLOCK, l - 1);
	pc_cmp[ l - 1] = NULL;

	//starting from the beginning of the rich-text field, loop to the 
	//	start of the PGP-encoded block it contains
	pc = pc_INPUT - 1;
	do	{
		//locate the next instance of the sought-for string in the 
		//	rich-text field
		if (f_failure = !ef_CursorToStringStart( pc_cmp, TRUE, t_cursor, 
									us, pt_RtfContext, &t_cursor, &us_offset))
			goto errJump;
		if (f_failure = !(BOOL) t_cursor.puc_location)
			goto errJump;

		//move to the corresponding line in the regular-text version of 
		//	the content
		if (f_failure = !(BOOL) (pc = strstr( pc + 1, pc_cmp)))
			goto errJump;

		//in case the line we found isn't the actual beginning of the 
		//	PGP-encoded content, prepare for the next pass by nudging 
		//	ahead the parameter for the start of the next search
		us = us_offset + 1;
	} while (pc < pc_BLOCK);

	//set up the span variable to be used in replacing the PGP-encoded 
	//	block in the rich-text field
	t_rtfSpan.t_cursorBegin = t_cursor;
	t_rtfSpan.us_offsetBegin = us_offset;
	t_rtfSpan.t_cursorEnd.puc_location = NULL;

	//replace the corresponding span of rich-text with the PGP-decoded 
	//	content
	us_err = eus_ReplaceRtfSpanText( &t_rtfSpan, ul_LEN_BLOCK, TRUE, 
													pc_OUTPUT, pt_RtfContext);

errJump:
	if (pc_cmp)
		free( pc_cmp);

	return us_err ? us_err : i_err || f_failure ? !eus_SUCCESS : 
																eus_SUCCESS;
} //us_ReplacePgpAsciiRichText(


/** xus_TestNoteForPgpAscii( ***
This exported interface allows the note passed in to be inspected for 
PGP-encoded ASCII text in the message body (rich-text field).

--- parameters & return ---
h_NOTE: the handle to the note to be inspected for PGP-encoded ASCII
pc_ITMNM_RTF: the name of the message-body rich-text field
pus_hasPgpAscii: flag telling whether encoded ASCII was found (TRUE) 
	or not (FALSE)
RETURN: eus_SUCCESS if no errors occurred; otherwise !eus_SUCCESS or a 
	Notes API error code

--- revision history ------
11/18/98 PR: created		*/
STATUS xus_TestNoteForPgpAscii( NOTEHANDLE  h_NOTE, 
								char  pc_ITMNM_RTF[], 
								short *const  ps_hasPgpAscii)	{
	RtfCursor  t_cursorBegin;
	RtfContext  t_RtfContext;
	char * pc = NULL;
	BOOL  f_hasPgpAscii, f_failure;
	STATUS  us_err;
	int  i_err;

	//initialize resources associated with the rich-text handling 
	//	we're going to do
	if (us_err = eus_InitializeRtfContext( h_NOTE, pc_ITMNM_RTF, 
											&t_cursorBegin, &t_RtfContext))
		return us_err;

	//set default that PGP-encoded ASCII content was not found
	*ps_hasPgpAscii = ms_VB_FALSE;

	//get the textual content of the rich-text field
	if (f_failure = eul_ERR_FAILURE == eul_GetRtfText( &t_RtfContext, NULL, 
																TRUE, &pc))
		goto errJump;

	//determine whether PGP-encoded ASCII is included in the textual content
	i_err = ei_TestForPgpAscii( pc, &f_hasPgpAscii);

	//if PGP-encoded ASCII is included, indicate it to caller by setting 
	//	the output flag
	if (f_hasPgpAscii)
		*ps_hasPgpAscii = ms_VB_TRUE;

errJump:
	//as necessary, free resources allocated during this procedure
	if (pc)
		free( pc);
	ef_FreeRtfContext( &t_RtfContext);

	//if no errors occurred, return that procedure was successful; otherwise 
	//	return an appropriate error code
	return f_failure || i_err ? !eus_SUCCESS : eus_SUCCESS;
} //xus_TestNoteForPgpAscii(


/** xus_TestNoteForPgpArmor( ***
Purpose is to determine whether PGP-Armored file attachments occur in the 
specified rich-text field of the specified note.

--- parameters & return ----
h_NOTE: the handle to the note to be processed
pc_ITMNM_RTF: pointer to a string telling the name of the rich-text field to 
	be processed
ps_hasPgpArmor: Output. Pointer to the Visual-Basic type flag in which to 
	return whether any PGP-Armored file attachemnts were found.
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
11/12/98 PR: created		*/
STATUS xus_TestNoteForPgpArmor( const NOTEHANDLE  h_NOTE, 
								char  pc_ITMNM_RTF[], 
								short *const ps_hasPgpArmor)	{
	DWORD  ul_objId;
	RtfContext  t_RtfContext;
	RtfCursor  t_cursor;
	const char * pc_FileNm;
	STATUS  us_err;
	BOOL  f_failure;

	if (!( h_NOTE && pc_ITMNM_RTF && ps_hasPgpArmor))
		return !eus_SUCCESS;

	//default to FALSE the flag telling whether a PGP-Armored file attachment 
	//	was found
	*ps_hasPgpArmor = ms_VB_FALSE;

	//If the special PGP-encoded rich-text object is attached, there's no 
	//	need to run through the rich-text field looking for hotspots. 
	//	Instead, short-circuit with success.
	if (!*mpc_FileNmPgpRtf)
		strcat( strcpy( mpc_FileNmPgpRtf, mpc_ROOTFILENM_PGP_RTF), 
														mpc_PGP_ARMOR_EXT);
	if (us_err = eus_getAttachmentInfo( h_NOTE, NULL, mpc_FileNmPgpRtf, 
										&ul_objId, NULL, NULL, NULL, NULL))
		return us_err;
	if (ul_objId)	{
		*ps_hasPgpArmor = ms_VB_TRUE;
		return eus_SUCCESS;
	}

	//initialize contextual information about the specified rich-text field
	if (us_err = eus_InitializeRtfContext( h_NOTE, pc_ITMNM_RTF, 
													&t_cursor, &t_RtfContext))
		return us_err;

	//advance the rich-text cursor to the next file-attachment hotspot, which 
	//	(though unlikely) may be the first CD record 
	if (f_failure = !ef_CursorToAttachmentHotspot( &t_cursor, &t_RtfContext, 
											FALSE, NULL, NULL, &pc_FileNm))
		goto errJump;

	//loop until either no further file-attachment hotspots exist in the 
	//	rich-text field or a PGP-Armored attachment is found
	while (t_cursor.puc_location)	{
		//a file-attachment hotspot must have been found, so determine 
		//	whether the "original" name of the attachment as stored in the 
		//	hotspot has the PGP-Armored file extension. If it does, set the 
		//	flag telling the caller whether a PGP-Armored file attachment was 
		//	found to TRUE, then break out of this loop
		if (stricmp( pc_FileNm + strlen( pc_FileNm) - 
											(sizeof( mpc_PGP_ARMOR_EXT) - 1), 
											mpc_PGP_ARMOR_EXT) == ei_SAME)	{
			*ps_hasPgpArmor = ms_VB_TRUE;
			break;
		}

		//advance the rich-text cursor to the next file-attachment hotspot, 
		//	first skipping past the current CD record
		if (f_failure = !ef_CursorToAttachmentHotspot( &t_cursor, 
													&t_RtfContext, TRUE, 
													NULL, NULL, &pc_FileNm))
			goto errJump;
	} //while (t_cursor.puc_location)

errJump:
	//free the contextual information about the rich-text field
	ef_FreeRtfContext( &t_RtfContext);

	//if no errors occurred, return that procedure was successful; 
	//	otherwise return an appropriate error code
	return f_failure ? !eus_SUCCESS : eus_SUCCESS;
} //xus_TestNoteForPgpArmor(


/** us_LocatePgpRtfAttachment( ***


--- parameters & return ----

RETURN: 

--- revision history -------
9/12/99 PR: added smart differentiation between compressed & non-compressed 
	files
1/30/99 PR: created				*/
//DOC!!
static STATUS us_LocatePgpRtfAttachment( 
										RtfCursor  t_cursor, 
										const RtfContext *const  pt_CONTEXT, 
										NOTEHANDLE  h_NOTE, 
										DWORD *const  pul_objId, 
										DWORD *const  pul_size, 
										BLOCKID *const  pbid_item)	{
	const char * pc_UniqueNm, * pc_FileNm;
	BOOL  f_Match = FALSE, f_compressed, f_failure;
	DWORD  ul_objId;
	STATUS  us_err;

	_ASSERTE( t_cursor.puc_location && t_cursor.us_recLength && pul_objId && 
													pul_size && pbid_item);

	*pul_objId = *pul_size = NULL;
	*pbid_item = ebid_NULLBLOCKID;

	if (!NSFNoteHasObjects( h_NOTE, NULL))
		return eus_SUCCESS;

	//advance the rich-text cursor to the next file-attachment hotspot, which 
	//	(though unlikely) may be the first CD record 
	if (f_failure = !ef_CursorToAttachmentHotspot( &t_cursor, pt_CONTEXT, 
									FALSE, NULL, &pc_UniqueNm, &pc_FileNm))
		return !eus_SUCCESS;

	//Loop either until no further file-attachment hotspots exist in the 
	//	rich-text field or until the PGP rich-text attachment is found. We 
	//	search for the special attachment using this technique first in case 
	//	more than one version of the PGP rich-text attachment exists on the 
	//	document. If there is more than one, we should default to the one 
	//	actually present in the RTF because that one must be the most recent 
	//	version.
	if (!*mpc_FileNmPgpRtf)
		strcat( strcpy( mpc_FileNmPgpRtf, mpc_ROOTFILENM_PGP_RTF), 
														mpc_PGP_ARMOR_EXT);
	while (t_cursor.puc_location)	{
		//A file-attachment hotspot must have been found, so determine 
		//	whether the "original" name of the attachment as stored in the 
		//	hotspot matches the filename we're after. If we have found a 
		//	match, break out of this loop to move on to the next stage
		if (f_Match = stricmp( pc_FileNm, mpc_FileNmPgpRtf) == ei_SAME)
			break;

		//advance the rich-text cursor to the next file-attachment hotspot, 
		//	first nudging past the current CD record
		if (f_failure = !ef_CursorToAttachmentHotspot( &t_cursor, pt_CONTEXT, 
										TRUE, NULL, &pc_UniqueNm, &pc_FileNm))
			return !eus_SUCCESS;
	} //while (t_cursor.puc_location)

	//Get the necessary location info for the caller to use in accessing the 
	//	attachment. If we didn't find a corresponding attachment hotspot in 
	//	the rich-text field, we will accept a properly named attachment not 
	//	incorporated.
	if (!f_Match)
		pc_UniqueNm = mpc_FileNmPgpRtf;
	if (us_err = eus_getAttachmentInfo( h_NOTE, NULL, pc_UniqueNm, 
											&ul_objId, pul_size, 
											&f_compressed, pbid_item, NULL))
		return us_err;

	//if the file is not compressed, set the object ID return variable, thus 
	//	signifying that the file is uncompressed (important later to 
	//	eus_getObjectContentsInBuffer()
	if (!f_compressed)
		*pul_objId = ul_objId;

	return us_err;
} //us_LocatePgpRtfAttachment(


/** xus_ConvertPgpArmoredHotspots( ***
Purpose is to convert each attachment hotspot associated with a PGP-Armored 
file into a custom hotspot that will behave according to PGP Plug-In 
standards.  The custom hotspot is included as a resource in this DLL. 

NOT YET DONE: SHOULD CHANGE NAME to something like SetupPgpArmor since we're 
also dealing with setting the host type of the special RTF attachment so that 
it doesn't get copied when message is forwarded or replied to with history.

--- parameter & return ----
h_NOTE: the handle to the note containing the hotspots to be converted
pc_ITMNM_RTF: the name of the message-body rich-text field
RETURN: eus_SUCCESS if no errors occurred; the Notea API error code otherwise

--- revision history ------
1/29/99 PR: created			*/
STATUS xus_ConvertPgpArmoredHotspots( const NOTEHANDLE  h_NOTE, 
										char  pc_ITMNM_RTF[])	{
	RtfContext  t_RtfContext, t_AttachParagraph;
	RtfCursor  t_cursor;
	RtfSpan  t_span;
	const char * pc_FileNm;
	char * pc_UniqueNm, 
			pc_UniqueNmRtfAttachment[ mi_MAXLEN_UNIQUE_ATTCNM] = {NULL};
	BOOL  f_ChangeHappened = FALSE, f_failure;
	BLOCKID  bid_contents;
	STATUS  us_err;

	if (!( h_NOTE && pc_ITMNM_RTF))
		return !eus_SUCCESS;

	//if there aren't any attachments, short-circuit with success
	if (!NSFNoteHasObjects( h_NOTE, NULL))
		return !eus_SUCCESS;

	//initialize resources associated with the rich-text handling we're going 
	//	to do
	if (us_err = eus_InitializeRtfContext( h_NOTE, pc_ITMNM_RTF, &t_cursor, 
															&t_RtfContext))
		return us_err;

	//advance the rich-text cursor to the next file-attachment hotspot, which 
	//	(though unlikely) may be the first CD record
	if (f_failure = !ef_CursorToAttachmentHotspot( &t_cursor, &t_RtfContext, 
									FALSE, &t_span, &pc_UniqueNm, &pc_FileNm))
		goto errJump;

	//loop until no further file-attachment hotspots exist in the rich-text 
	//	field
	if (!*mpc_FileNmPgpRtf)
		strcat( strcpy( mpc_FileNmPgpRtf, mpc_ROOTFILENM_PGP_RTF), 
														mpc_PGP_ARMOR_EXT);
	while (t_cursor.puc_location)	{
		//a file-attachment hotspot must have been found, so determine 
		//	whether the "original" name of the attachment as stored in the 
		//	hotspot has the PGP-Armored file extension
		if (stricmp( pc_FileNm + strlen( pc_FileNm) - 
											(sizeof( mpc_PGP_ARMOR_EXT) - 1), 
											mpc_PGP_ARMOR_EXT) == ei_SAME)	{
			//determine whether the "original" name of the attachment as 
			//	stored in the hotspot matches the special filename we use for 
			//	the PGP-encoded rich-text message replacement
			if (stricmp( pc_FileNm, mpc_FileNmPgpRtf) == ei_SAME)	{
				//if we've already processed a special-attachment instance, 
				//	short-circuit with failure
				if (f_failure = (BOOL) *pc_UniqueNmRtfAttachment)
					goto errJump;

				//remove current attachment hotspot
				if (us_err = eus_RemoveRtSpan( &t_span, &t_RtfContext))
					goto errJump;

				//initialize the rich-text hidden paragraph we will use to 
				//	store the special object
				if (us_err = us_SetupRtfAttachParagraph( h_NOTE, 
											&t_AttachParagraph, pc_UniqueNm))
					goto errJump;
				strcpy( pc_UniqueNmRtfAttachment, pc_UniqueNm);

				f_ChangeHappened = TRUE;
			//else if the cursor has alighted on a file-attachment hotspot 
			//	and it's PGP Armored...
			}else	{
				//with internal resource components and the filename Notes 
				//	has associated with the PGP-Armored attachment, construct 
				//	& compile an @DbCommand @Formula

				//with internal resource components, the @Formula just 
				//	compiled, and a rich-text text record with the formatted 
				//	original filename associated with the PGP-Armored 
				//	attachment, construct a replacement @Formula hotspot

				//replace the file-attachment hotspot with the newly 
				//	constructed @Formula hotspot
			} //if (stricmp( pc_FileNm, mpc_FileNmPgpRtf) == ei_SAME)
		} //if (stricmp( pc_FileNm + strlen( pc_FileNm)

		//advance the rich-text cursor to the next file-attachment hotspot, 
		//	first skipping past the current CD record
		if (f_failure = !ef_CursorToAttachmentHotspot( &t_cursor, 
												&t_RtfContext, TRUE, NULL, 
												&pc_UniqueNm, &pc_FileNm))
			goto errJump;
	} //while (t_cursor.puc_location)

	//Eliminate the possibility of the special PGP rich-text attachment being 
	//	inherited into documents created via the @Command( [MailForward]) 
	//	mechanism or via forms having the "Inherit entire selected document 
	//	into the rich-text field ... as rich-text" setting. First, if a 
	//	corresponding rich-text hotspot was found for the special rich-text 
	//	attachment...
	if (*pc_UniqueNmRtfAttachment)	{
		//append the special ending hidden-paragraph item to the new virtual 
		//	rich-text field. Since both above-mentioned mechanisms ignore 
		//	whatever's in hidden paragraphs, this method avoids the 
		//	inheritance issue and is cosmetically attractive to boot.
		if (us_err = eus_AppendItemsToRtf( &t_AttachParagraph, 
															&t_RtfContext))
			goto errJump;
	//Else if the special PGP rich-text attachment is present without a 
	//	corresponding hotspot, the message probably came into the system via 
	//	the client POP3/IMAP4 "Direct to Internet" mechanism. In this case we 
	//	change the "host type" of the special attachment to a type that Notes 
	//	will not inherit into documents created via the mechanisms described 
	//	above. This is an acceptable thing to do for _received_ messages, 
	//	since this message won't be going through a SMTP MTA anymore, so we 
	//	don't care about the fact that the MTA will not forward attachments 
	//	with the STREAM host type. Another nice by-product of doing this is 
	//	that the attachment will not show up "below the line" on the message.
	}else if (us_err = eus_getAttachmentInfo( h_NOTE, NULL, 
												mpc_FileNmPgpRtf, NULL, NULL, 
												NULL, NULL, &bid_contents))
		goto errJump;
	else if (bid_contents.pool)	{
		FILEOBJECT *const  ptfo = (FILEOBJECT *) (OSLockBlock( WORD, 
														bid_contents) + 1);

		ptfo->HostType = HOST_STREAM;
		OSUnlockBlock( bid_contents);
	} //if (us_err = eus_getAttachmentInfo(

	//if necessary, commit the updated rich-text field to the note, replacing 
	//	the current actual rich-text content
	if (f_ChangeHappened)
		us_err = eus_CommitChangedRtf( h_NOTE, pc_ITMNM_RTF, &t_RtfContext);

errJump:
	if ((us_err || f_failure) && *pc_UniqueNmRtfAttachment)
		ef_FreeRtfContext( &t_AttachParagraph);
	ef_FreeRtfContext( &t_RtfContext);

	//if no errors occurred, return that procedure was successful; 
	//	otherwise return an appropriate error code
	return us_err + f_failure;
} //xus_ConvertPgpArmoredHotspots(


/** DllMain( ***
Win32 entry point for loading and unloading this DLL.  We use the unload 
event to free statically allocated resources.

--- parameters & return ---
h_DllInstance: the overall handle to this DLL's runtime information, settings
ul_reason: parameter tells why the function was called
pv_reserved: just that, reserved

--- revision history ------
2/27/00 PR: support of rich-text reset functionality, part of changes needed 
	for Notes Mail R5-template compatibility 
11/18/98 PR: created		*/
BOOL WINAPI DllMain( HINSTANCE  h_DllInstance, 
						DWORD  ul_reason, 
						LPVOID  pv_reserved)	{
	int  i_err;

	switch (ul_reason)	{
		case DLL_PROCESS_ATTACH:
			//Set up debugging aids and the memory-leak detector in 
			//	particular.  Lines are preprocessed out in Release builds.
			_CrtSetReportMode( _CRT_WARN, _CRTDBG_MODE_DEBUG | 
														_CRTDBG_MODE_WNDW);
			_CrtSetDbgFlag( _CRTDBG_LEAK_CHECK_DF | 
										_CrtSetDbgFlag( _CRTDBG_REPORT_FLAG));

			//keep ahold of the instance handle of the process controlling 
			//	this DLL
			eh_Instance = h_DllInstance;

			//set up mechanism for enforcing DLL thread-safety
			InitializeCriticalSection( &ecs);

			if (i_err = ei_InitializePgpContext())
				return FALSE;

			break;

		case DLL_PROCESS_DETACH:
			//free any resources still allocated to a copy of rich-text 
			//	content stored in memory
			if (mt_RtfResettable.h_content)
				OSMemFree( mt_RtfResettable.h_content);

			e_ClearRevealNoteHandleContext( ecs);
			DeleteCriticalSection( &ecs);

			e_ReleasePgpContext();
	} //switch (ul_reason)

	return TRUE;
} //DllMain(


/** es_PreMailSend( ***
Purpose is to take any PGP action necessary before the message gets sent on 
to MAIL.BOX as the current mail-send event is carried through.

Although it probably would have been more elegant to place this procedure in 
the ndbPGP DLL, the debugging infrastructure doesn't like one DLL freeing 
memory allocated in another DLL, even if the DLLs are in the same process. I 
don't know whether this is a limitation of the Windows debugging 
infrastructure or if its just a no-no in general for reasons unknown to me.

--- parameter & return ----
ppv_ARGV: pointer to the series of arguments provided by the PGP @DbCommand 
	used to call this DLL
RETURN: eus_SUCCESS if successful; an appropriate error code otherwise

--- revision history ------
2/27/00 PR
+ recast to support original 6.5.1 release configuration (needed once 
  decision made that R5-compatible version would release with PGP 7.0

9/12/99 PR
+ logic adjustment to support enhancement of better emulation of 
  Lotus Notes' name lookup behavior
+ documentation adjustment, logic shortening

1/23/99 PR: created			*/
short xs_PreMailSend( void * *const  ppv_ARGV)	{
	const int  i_ARG_HANDLE = 0, i_ARG_PGP_FLAGS = 1, 
				i_ARG_RECIPIENT_NMS = 2, i_ARG_LOOKUP_EXHAUST = 3, 
				i_ARG_LOOKUP_HOW = 4, i_ARG_ATTACHMENTS = 5, 
				i_ARG_RTF_FLDNM = 6;
	const char  pc_PGPFLG_SIGN[] = "SI", pc_PGPFLG_ENCRYPT[] = "EN";

	double  d;
	NOTEHANDLE  h_note;
	char * pc_ItmNmRtf = NULL, * pc_LookupHow = NULL, 
			* pc_RecipientsList = NULL;
	BOOL  f_PgpSign, f_PgpEncrypt, f_lookupExhaust = FALSE, 
			f_failure = FALSE;
	int  i_Attachments = 0;
	STATUS  us_err;
	short  s_error;

	if (!( ppv_ARGV && *ppv_ARGV))
		return !eus_SUCCESS;

	//recover the handle to the calling note's session
	if (!ef_GetNumberListEntry( ppv_ARGV[ i_ARG_HANDLE], TRUE, 0, &d))
		return !eus_SUCCESS;
	h_note = (NOTEHANDLE) (DWORD) d;

	//determine what sort of PGP encoding is to be performed
	f_PgpSign = ef_TextListContainsEntry( ppv_ARGV[ i_ARG_PGP_FLAGS], TRUE, 
												pc_PGPFLG_SIGN, NULL, NULL);
	f_PgpEncrypt = ef_TextListContainsEntry( ppv_ARGV[ i_ARG_PGP_FLAGS], 
													TRUE, pc_PGPFLG_ENCRYPT, 
													NULL, NULL);

	//get the recipient-list string
	if (us_err = eus_CreateTextListEntryCopy( 0, 
											ppv_ARGV[ i_ARG_RECIPIENT_NMS], 
											TRUE, &pc_RecipientsList))
		return us_err;
	if (!pc_RecipientsList && f_PgpEncrypt)
		return !eus_SUCCESS;

	//if PGP security has been specified to be performed and if the list of 
	//	recipient names contains someone...
	if ((f_PgpSign || f_PgpEncrypt) && pc_RecipientsList)	{
		//get the name of the rich-text field to process
		if ((us_err = eus_CreateTextListEntryCopy( 0, 
											ppv_ARGV[ i_ARG_RTF_FLDNM], TRUE, 
											&pc_ItmNmRtf)) || 
											(f_failure = !pc_ItmNmRtf))
			goto errJump;

		//if we're to encrypt...
		if (f_PgpEncrypt)	{
			//get the number of attachments associated with the note
			if (!ef_GetNumberListEntry( ppv_ARGV[ i_ARG_ATTACHMENTS], TRUE, 
																	0, &d))
				return !eus_SUCCESS;
			i_Attachments = (int) d;

			//get the name of the user's mail (home) server
			if (us_err = eus_CreateTextListEntryCopy( 0, 
												ppv_ARGV[ i_ARG_LOOKUP_HOW], 
												TRUE, &pc_LookupHow))
				goto errJump;
			if (!pc_LookupHow && !(pc_LookupHow = calloc( 1, sizeof( char))))
				goto errJump;

			//determine whether caller wants name lookups to be "exhaustive"
			if (f_failure = !ef_GetNumberListEntry( ppv_ARGV[ 
													i_ARG_LOOKUP_EXHAUST], 
													TRUE, 0, &d))
				goto errJump;
			f_lookupExhaust = (BOOL) d;
		} //if (f_PgpEncrypt)

		//Initialize the information structure we use to hold onto a copy of 
		//	the PGP-encoded rich-text field we will be constructing. We may 
		//	need to reset the note's rich-text field to the PGP-encoded 
		//	version if the encoding was initiated by a user press of the 
		//	"Send and File" action button.
		if (mt_RtfResettable.h_note)	{
			OSMemFree( mt_RtfResettable.h_content);
			mt_RtfResettable.h_content = NULL;
		}
		mt_RtfResettable.h_note = h_note;

		//Prepare the message for sending in a PGP-secured manner. If failure 
		//	is encountered, flag that the structure used to hold a copy of 
		//	the PGP-encoded rich-text field is empty.
		if (s_error = xs_PgpSecureMessage( h_note, f_PgpSign, f_PgpEncrypt, 
										pc_RecipientsList, f_lookupExhaust, 
										pc_LookupHow, i_Attachments, 
										pc_ItmNmRtf, &mt_RtfResettable))
			mt_RtfResettable.h_note = NULL;
	} //if ((f_PgpSign || f_PgpEncrypt) && 

errJump:
	//free resources allocated via this procedure
	if (pc_RecipientsList)
		free( pc_RecipientsList);
	if (pc_ItmNmRtf)
		free( pc_ItmNmRtf);
	if (pc_LookupHow)
		free( pc_LookupHow);

    return (short) us_err + s_error + f_failure;
} //xs_PreMailSend(


/** xs_PgpSecureMessage( ***
PGP encode the specified Notes Mail message (meaning its Notes rich-text 
field and any associated file attachments) according to caller instructions.

--- parameters & return ----
h_NOTE: handle to the note to undergo PGP encoding
f_PgpSign: flag telling whether the message should be PGP signed
f_PgpEncrypt: flag telling whether the message should be PGP encrypted
pc_RecipientsList: Optional. Pointer to a string of user addresses, delimited 
	by semicolon, to be used in the encoding process. The "addresses" will be 
	fully resolved as needed against the Domino Directory. Required only if 
	PGP encryption is specified.
f_LOOKUP_EXHAUST: Optional. Flag telling whether the addresses provided will 
	be checked exhaustively for ambiguity against the Domino Directory 
	configuration currently employed by the caller. Required only if PGP 
	encryption is specified.
pc_LookupHow: Optional. Pointer to the name of the server holding the Domino 
	Directory to be used potentially in the resolution process. Required only 
	if PGP encryption is specified.
i_ATTACHMENTS: Optional. The number of attachments included in the message. 
	Ignored if PGP encryption is not specified.
pc_ITMNM_RTF: pointer to a string telling the name of the rich-text field to 
	be used in the encoding
pt_RtfResettable: Optional Input & Output. Pointer to the information 
	structure used in later resetting the message's rich-text field to the 
	encoded version to be created in this procedure. The need for this 
	functionality is driven by peculiarities of the "Send and File" action 
	button in the R5 version of the Notes Mail template. The h_note property 
	of the information structure is used as a safety check on the 
	execution of this functionality. In the context of this procedure, it 
	is used to ensure the caller is aware of what it's doing in using this 
	parameter by checking that its value equals that specified in the 
	primary h_NOTE parameter described above. The t_RtfContext property of 
	the structure will be filled upon successful completion of this 
	procedure. If parameter is null, the procedure understands that the 
	rich-text field reset functionality has not been engaged by the caller.
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
2/27/00 PR
+ support of rich-text reset functionality, part of changes needed for 
  Notes Mail R5-template compatibility
+ enhancement to delete the PGP-encoded rich-text attachment if error is 
  encountered after the attachment has been accomplished
+ completed standard documentation

9/12/99 PR: logic adjustments to support enhancement of better emulation of 
	Lotus Notes' name-lookup behavior, including removal of recipient-count 
	parameter

1/19/99 PR: created			*/
//					static FILE * Log;
STATUS xs_PgpSecureMessage( NOTEHANDLE  h_NOTE, 
							BOOL  f_PgpSign, 
							BOOL  f_PgpEncrypt, 
							char  pc_RecipientsList[], 
							const BOOL  f_LOOKUP_EXHAUST, 
							char  pc_LookupHow[], 
							const int  i_ATTACHMENTS, 
							char  pc_ITMNM_RTF[], 
							ResettableRtfInfo *const  pt_RtfResettable)	{
	const BOOL  f_RESETTABLE = pt_RtfResettable && pt_RtfResettable->h_note;

	RtfCursor  t_cursor;
	RtfContext  t_RtfContext, t_AttachParagraph;
	char * * ppc_recipients = NULL, * pc_output = NULL;
	UINT  ui_recipients;
	PgpEncodeContext  t_EncodeContext;
	BOOL  f_PgpEncodeContextSetUp = FALSE, f_RtfAttachParagraph = FALSE, 
			f_PgpSyncUnknownKeys, f_ambiguous, f_failure = FALSE;
	BLOCKID  bid_ItmAttach;
	STATUS  us_err;
	int  i_err = NULL;

	_ASSERTE( h_NOTE && (f_PgpSign || f_PgpEncrypt) && (f_PgpEncrypt ? 
								(BOOL) pc_RecipientsList : TRUE) && 
								pc_ITMNM_RTF && (f_RESETTABLE ? 
								pt_RtfResettable->h_note == h_NOTE : TRUE));
/*if (!( Log = fopen( "nPGPLog.txt", "wtc")))
	return !eus_SUCCESS;
if (setvbuf( Log, NULL, _IONBF, NULL) != eus_SUCCESS)
	return !eus_SUCCESS;
fprintf( Log, "Started message log\n");
fclose( Log);
Log = fopen( "nPGPLog.txt", "atc");
setvbuf( Log, NULL, _IONBF, NULL);
*/
	//initialize resources associated with the rich-text handling we're going 
	//	to do
	if (us_err = eus_InitializeRtfContext( h_NOTE, pc_ITMNM_RTF, &t_cursor, 
															&t_RtfContext))
		return us_err;

	//initialize the PGP context for encoding content
	if (i_err = ei_SetUpPgpEncodeContext( f_PgpSign = !!f_PgpSign, 
									f_PgpEncrypt = !!f_PgpEncrypt, 
									&t_EncodeContext, &f_PgpSyncUnknownKeys))
		goto errJump;
	f_PgpEncodeContextSetUp = TRUE;

	//if message is to be encrypted...
	if (f_PgpEncrypt)	{
		//as much as possible, resolve recipient addressses and create a 
		//	character-string array of the addresses, the format expected by 
		//	the PGP infrastructure
		if (us_err = us_CreateRecipientsArray( pc_RecipientsList, 
											!!f_LOOKUP_EXHAUST, pc_LookupHow, 
											&ppc_recipients, 
											&ui_recipients, &f_ambiguous))
			goto errJump;

		//adjust as needed any canonicalized Notes names to the more 
		//	user-friendly abbreviated format (the PGP default, starting with 
		//	PGP 6.5.1)
		if (i_err = i_PgpAdjustNotesNames( f_PgpSyncUnknownKeys, 
											ui_recipients, ppc_recipients, 
											&t_EncodeContext))
			goto errJump;

		//obtain the final list of encryption keys for use in the ensuing 
		//	encoding, using the standard PGP client infrastructure
		if (i_err = ei_ResolveEncryptionKeys( ppc_recipients, ui_recipients, 
											f_ambiguous, &t_EncodeContext))
			goto errJump;
	} //if (f_PgpEncrypt)

	//PGP-encode all the textual content of the rich-text field into a buffer
	if (i_err = i_PgpEncodeRtfText( &t_EncodeContext, &t_RtfContext, 
																&pc_output))
		goto errJump;

	//if necessary, PGP Armor each attachment in the message
	if (i_ATTACHMENTS && f_PgpEncrypt)
		if (i_err = i_PgpEncodeAttachments( &t_EncodeContext, h_NOTE, 
													t_cursor, &t_RtfContext))
			goto errJump;

	//all manipulation done, PGP-encode the rich-text field as a 
	//	PGP-Armored file and attach it to the note and reference it in an 
	//	ending, hidden paragraph
	f_RtfAttachParagraph = TRUE;
	if (i_err = i_PgpEncodeRtfAsAttachment( &t_EncodeContext, h_NOTE, 
									&t_RtfContext, ept_InitUtilityRtfContext( 
									&t_AttachParagraph), &bid_ItmAttach))
		goto errJump;

	//completely replace the original rich-text field with the PGP-encoded 
	//	textual content, in Courier 10 font
	if (us_err = eus_ReplaceRtfWithStandardText( pc_output, TRUE, 
															&t_RtfContext))
		goto errJump;

	//append the special ending hidden-paragraph item to the new virtual 
	//	rich-text field
	if (us_err = eus_AppendItemsToRtf( &t_AttachParagraph, &t_RtfContext))
		goto errJump;

	//Commit the updated rich-text field to the note, replacing the current 
	//	actual rich-text content
	if (us_err = eus_CommitChangedRtf( h_NOTE, pc_ITMNM_RTF, &t_RtfContext))
		ef_UnappendRtfItems( &t_AttachParagraph, &t_RtfContext);

errJump:
	if (pc_output)
		e_FreePgpMem( pc_output);

	//if an error occurred and context for PGP's hidden rich-text paragraph 
	//	has been initialized...
	if ((i_err || us_err) && f_RtfAttachParagraph)	{
		//if the encoded version of the rich-text field has been encapsulated 
		//	as an attachment to the note, remove that attachment
		if (us_err)
			f_failure = NSFNoteDetachFile( h_NOTE, bid_ItmAttach);

		//free any resources allocated to PGP's hidden rich-text paragraph
		ef_FreeRtfContext( &t_AttachParagraph);
	} //if ((i_err || us_err) && f_RtfAttachParagraph)

	if (f_PgpEncodeContextSetUp)
		f_failure = !ef_FreePgpEncodeContext( &t_EncodeContext);
	if (ppc_recipients)
		e_FreeStringArray( &ppc_recipients, ui_recipients);

	_ASSERTE( !f_failure);

	//if caller wants a copy of the encoded version of the rich-text field 
	//	available for a potential reset of the field later, copy the version 
	//	content into memory
	if (f_RESETTABLE && !(us_err || i_err))
		us_err = eus_CopyRtfIntoBuffer( &t_RtfContext, 
											&pt_RtfResettable->h_content, 
											&pt_RtfResettable->ul_len, NULL);

	//free all resources associated with the encoded rich-text field 
	//	constructed by this procedure
	ef_FreeRtfContext( &t_RtfContext);

	return us_err + (short) i_err;
} //xs_PgpSecureMessage(


/** i_PgpEncodeAttachments( ***


--- parameters & return ----

RETURN: 

--- revision history -------
2/27/00 PR: enhancement to prevent attachments already PGP encoded from being 
	encoded again; associated documentation adjustments, logic shortening
9/12/99 PR: documentation adjustment
1/30/99 PR: created			*/
//DOC!!
static int i_PgpEncodeAttachments( 
								PgpEncodeContext *const  pt_EncodeContext, 
								NOTEHANDLE  h_NOTE, 
								RtfCursor  t_cursor, 
								RtfContext *const  pt_rtfContext)	{
	static const char  pc_ENCRYPTED_TAG[] = "<PGP_encrypted> ", 
						pc_ARMOR_EXT_PGP[] = ".pgp";
	static const UINT  ui_LEN_ARMOR_EXT_ASC = sizeof( mpc_PGP_ARMOR_EXT), 
						ui_LEN_ARMOR_EXT_PGP = sizeof( pc_ARMOR_EXT_PGP);

	static char  pc_dirNmTemp[ mus_MAXBUF_EXTFILENM - 2];

	const WORD  us_LEN_ITEMNM_STANDARD_ATTACH = (WORD) strlen( 
												epc_ITEMNM_STANDARD_ATTACH);

	char * pc_AttachNm, * pc_FileNm, * pc_fileNmNew, 
			pc_extFileNmTmp[ mus_MAXBUF_EXTFILENM], 
			pc_extFileNmNew[ mus_MAXBUF_EXTFILENM];
	RtfSpan  t_rtSpan;
	UINT  ui_lenFileNm;
	BLOCKID  bid_item;
	STATUS  us_err;
	BOOL  f_failure;
	int  i_err, i_errRemove;

	_ASSERTE( pt_EncodeContext && pt_EncodeContext->f_Encrypt && h_NOTE && 
										pt_rtfContext && NSFItemIsPresent( 
										h_NOTE, epc_ITEMNM_STANDARD_ATTACH, 
										us_LEN_ITEMNM_STANDARD_ATTACH));

	//if this is the first time through, initialize the path to the user's 
	//	temporary directory
	if (!*pc_dirNmTemp)
		if (!epc_getTempDirNm( pc_dirNmTemp, mus_MAXBUF_EXTFILENM - 2))
			return !eus_SUCCESS;

	//advance the rich-text cursor to the next file-attachment hotspot, which 
	//	may be the current CD record 
	if (f_failure = !ef_CursorToAttachmentHotspot( &t_cursor, pt_rtfContext, 
													FALSE, &t_rtSpan, 
													&pc_AttachNm, &pc_FileNm))
		return !eus_SUCCESS;

	//loop until no further file-attachment hotspots exist in the rich-text 
	//	field
	while (t_cursor.puc_location)	{
		//if the attachment hasn't already been PGP encoded, get the 
		//	attachment item that corresponds to the attachment referenced in 
		//	the hotspot
		bid_item.pool = NULL;
		if (!( stricmp( pc_FileNm + (ui_lenFileNm = strlen( pc_FileNm)) - 
								ui_LEN_ARMOR_EXT_ASC, mpc_PGP_ARMOR_EXT) == 
								ei_SAME || stricmp( pc_FileNm + 
								ui_lenFileNm - ui_LEN_ARMOR_EXT_PGP, 
								pc_ARMOR_EXT_PGP) == ei_SAME))
			if (us_err = eus_getAttachmentInfo( h_NOTE, 
									epc_ITEMNM_STANDARD_ATTACH, pc_AttachNm, 
									NULL, NULL, NULL, &bid_item, NULL))
										
				return us_err;

		//if the attachment should be encoded...
		if (bid_item.pool)	{
			if (strlen( pc_FileNm) + ui_LEN_ARMOR_EXT_ASC > 
														mus_MAXBUF_EXTFILENM)
				return !eus_SUCCESS;

			//Extract the attachment to a file in the user's temporary 
			//	directory. Need to use the real file name so PGP will 
			//	default to this name when the file is ultimately decoded from 
			//	the file system by the user.
			if (us_err = NSFNoteExtractFile( h_NOTE, bid_item, 
										strcat( strcpy( pc_extFileNmTmp, 
										pc_dirNmTemp), pc_FileNm), NULL))	{
				_ASSERTE( !(ERR( us_err) == ERR_NOEXTRACT_ENCRYPTED || ERR( 
										us_err) == ERR_NOTE_BADATTSIGN));
				return us_err;
			}

			//PGP encode the file
			if (i_err = ei_PgpEncodeFile( pt_EncodeContext, 
										pc_extFileNmTmp, strcat( strcpy( 
										pc_extFileNmNew, pc_extFileNmTmp), 
										mpc_PGP_ARMOR_EXT)))
				return i_err;

			//Delete the original file copied to the temporary directory. 
			//	Ignore any failure of this operation.
			i_errRemove = remove( pc_extFileNmTmp);
			_ASSERTE( i_errRemove == eus_SUCCESS);

			//write the encoded file back into the mail database as a file 
			//	object, replacing the file's unencoded counterpart
			if (us_err = NSFNoteDetachFile( h_NOTE, bid_item))
				return us_err;
			if (us_err = NSFNoteAttachFile( h_NOTE, 
											epc_ITEMNM_STANDARD_ATTACH, 
											us_LEN_ITEMNM_STANDARD_ATTACH, 
											pc_extFileNmNew, pc_fileNmNew = 
											strrchr( pc_extFileNmNew, 
											ec_PATH_SPECIFIER) + 1, NULL))
				return us_err;

			//Delete the encoded file written to the temporary directory. 
			//	Ignore any failure of this operation.
			i_errRemove = remove( pc_extFileNmNew);
			_ASSERTE( i_errRemove == eus_SUCCESS);

//PGP SHORTCUT: not checking on unique attachment name!!
			if (us_err = eus_ResetAttachHotspotNames( pc_fileNmNew, 
										pc_fileNmNew, t_rtSpan.t_cursorBegin, 
										pt_rtfContext))
				return us_err;

			//place the tag telling that the file is encrypted just after the 
			//	attachment icon, but before the end of the hotspot. 
			if (us_err = eus_InsertTextAtHotspotEnd( pc_ENCRYPTED_TAG, 
										t_rtSpan.t_cursorEnd, pt_rtfContext))
				return us_err;
		} //if (bid_item.pool)

		//advance the rich-text cursor to the next file-attachment hotspot, 
		//	first skipping past the current CD record
		if (f_failure = !ef_CursorToAttachmentHotspot( &t_cursor, 
											pt_rtfContext, TRUE, &t_rtSpan, 
											&pc_AttachNm, &pc_FileNm))
			return !eus_SUCCESS;
	} //while (t_cursor.puc_location)

	return eus_SUCCESS;
} //i_PgpEncodeAttachments(


/** i_PgpEncodeRtfAsAttachment( ***
PGP-encode a copy of the given rich-text field and attach it as an object to 
the specified note. So that the attachment may be hidden from the user, 
the caller is also provided a rich-text hidden paragraph which references the 
attachment and so inhibits its display "below the line" in the Notes 
user-interface.

--- parameters & return ----
pt_EncodeContext: Input & Output. Pointer to the PGP information structure 
	being used in the current round of encoding. The opaque structure may be 
	updated if user adjusts presented defaults.
h_NOTE: handle to the note to which the encoded rich-text field will be 
	attached as an attachment
pt_RTF_CONTEXT: pointer to the rich-text context structure describing the 
	rich-text field to be copied, encoded and encapsulated as an attachment
pt_AttachParagraph: Output. Address of the rich-text context structure by 
	which a hidden rich-text paragraph will be constructed which includes a 
	reference to the generated attachment. Caller is responsible for having 
	initialized the structure prior to calling this procedure. If procedure 
	is unsuccessful, it is possible that the structure will have undergone 
	change even so.
pbid_item: Output. Address of the structure to fill with the Notes BLOCKID 
	associated with the $FILE item that describes the attachment created by 
	this procedure.
RETURN: eus_SUCCESS if successful; the Notes API or PGP SDK error code if 
	unsuccessful

--- revision history -------
2/27/00 PR
+ adjustments associated with a change to the eus_CopyRtfIntoBuffer() 
  signature
+ adjusted signature to provide the ultimate attachment name to caller and to 
  allow a PGP error-code return as well as a Notes error-code return
+ documentation adjustment

1/29/99 PR: created			*/
static int i_PgpEncodeRtfAsAttachment( 
								PgpEncodeContext *const  pt_EncodeContext, 
								NOTEHANDLE  h_NOTE, 
								const RtfContext *const  pt_RTF_CONTEXT, 
								RtfContext *const  pt_AttachParagraph, 
								BLOCKID *const  pbid_item)	{
	DWORD  ul, ul_lenOutput;
	BYTE * puc_input, * puc_output = NULL;
	char  pc_UniqueNm[ mi_MAXLEN_UNIQUE_ATTCNM] = {NULL};
	HANDLE  h;
	BOOL  f_RtfAttachParagraph = FALSE;
	int  i_err;
	STATUS  us_err;

	_ASSERTE( pt_EncodeContext && h_NOTE && pt_RTF_CONTEXT && 
											pt_AttachParagraph && pbid_item);

	//stream the now-complete virtual rich-text content into a buffer
	if (us_err = eus_CopyRtfIntoBuffer( pt_RTF_CONTEXT, &h, &ul, &puc_input))
		return us_err;

	i_err = ei_PgpEncodeBuffer( puc_input, (long) ul, pt_EncodeContext, 
											TRUE, &ul_lenOutput, &puc_output);
	OSUnlockObject( h);
	OSMemFree( h);
	if (i_err)
		return i_err;

	//initialize the rich-text hidden paragraph we will use to store the 
	//	special object
	if (us_err = us_SetupRtfAttachParagraph( h_NOTE, pt_AttachParagraph, 
																pc_UniqueNm))
		goto errJump;

	//Attach the appropriate buffer content (encrypted or not) to the note as 
	//	a file object. The MS-DOS host format is used here at encoding-time 
	//	to ensure that the SMTP MTA will include it if the message goes 
	//	through it. The STREAM format is for some reason ignored and dropped 
	//	by the MTA (in Domino R4 at least).
	if (us_err = eus_AttachBufferAsObject( puc_output, ul_lenOutput, NULL, 
										h_NOTE, epc_ITEMNM_STANDARD_ATTACH, 
										pc_UniqueNm, HOST_MSDOS))
		goto errJump;

	//retrieve for the caller the item-BLOCKID of the attachment just created
	us_err = eus_getAttachmentInfo( h_NOTE, NULL, pc_UniqueNm, 
										NULL, NULL, NULL, pbid_item, NULL);

errJump:
	e_FreePgpMem( puc_output);

	return us_err;
} //i_PgpEncodeRtfAsAttachment(


/** us_SetupRtfAttachParagraph( ***


--- parameters & return ----

RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
1/29/99 PR: created			*/
//DOC!!
static STATUS us_SetupRtfAttachParagraph( 
									 NOTEHANDLE  h_NOTE, 
									 RtfContext *const  pt_AttachParagraph, 
									 char *const  pc_UniqueNm)	{
	STATUS  us_err;
//NOTE: might be able the make the module-scope variables in this function 
//	local
	_ASSERTE( pc_UniqueNm && h_NOTE && pt_AttachParagraph);

	if (us_err = eus_AppendRtParagraph( 
0xCBA9, 
												&mt_HIDDEN_PARAGRAPH_DEF, 
												ept_InitUtilityRtfContext( 
												pt_AttachParagraph), NULL))
		return us_err;

	//Include an attachment hotspot for the object in the special 
	//	hidden-paragraph rich-text item. This will keep the attachment from 
	//	showing up "below-the-line," as it would if not referenced by an 
	//	attachment hotspot in a rich-text field on the document. Since the 
	//	hotspot always remains hidden, there's no need to include a graphic 
	//	in the hotspot.
	if (!*mpc_FileNmPgpRtf)
		strcat( strcpy( mpc_FileNmPgpRtf, mpc_ROOTFILENM_PGP_RTF), 
														mpc_PGP_ARMOR_EXT);
	us_err = eus_AppendAttachmentHotspot( h_NOTE, mpc_FileNmPgpRtf, 
													pc_UniqueNm, NULL, NULL, 
													pt_AttachParagraph);

	if (us_err)
		ef_FreeRtfContext( pt_AttachParagraph);

	return us_err;
} //us_SetupRtfAttachParagraph(


/** us_CreateRecipientsArray( ***
Purpose is to convert a string containing the current message recipients into 
a string array, the format expected by the PGP infrastructure. In doing this, 
the procedure resolves the recipients against the Domino Directory using the 
specified lookup configuration. All members of an ambiguous address 
resolution are included in the ultimate list, as are recipients that do not 
resolve.

--- parameters & return -----
pc_RecipientsList: Input & Output. Pointer to the string containing the 
	message recipients as currently understood, delimited by semi-colons. 
	Output is undefined due to use of strtok() to parse the string.
f_LOOKUP_EXHAUST: Flag telling whether the directories should be searched 
	exhaustively for each name so as to be sure all matches are accounted for.
pc_LookupHow: pointer to the name of the server holding the Domino Directory 
	to be used potentially in the resolution process
pppc_recipients: Output. Pointer to the string array variable to populate 
	with the list of recipients PGP should try to match encryption keys to. 
	CALLER IS RESPONSIBLE for freeing the resources allocated to the array.
pui_recipients: Output. Pointer to the variable in which to store the number 
	of elements placed in the string array (pppc_recipients).
pf_ambiguous: Output. Pointer to the flag variable to set according to 
	whether ambiguous resolutions were encountered.
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history --------
9/12/99 PR
+ logic overhaul to support enhancement of better emulation of Lotus Notes' 
  name lookup behavior
+ documentation adjustment

12/20/98 PR: created		*/
static STATUS us_CreateRecipientsArray( char  pc_RecipientsList[], 
										const BOOL  f_LOOKUP_EXHAUST, 
										char  pc_LookupHow[], 
										char * * *const  pppc_recipients, 
										UINT *const  pui_recipients, 
										BOOL *const  pf_ambiguous)	{
	const char  pc_LIST_DELIMITER[] = ";";

	UINT  ui, ui_entries = 0;
	char * pc_recip, * pc, * * ppc = NULL;
	NameFoundNode * pt_unresolved = NULL, * pt;
	StringNode * pt_resolved = NULL, * pt_nd;
	STATUS  us_err = eus_SUCCESS;
	BOOL  f_failure = FALSE;

	_ASSERTE( pc_RecipientsList && pppc_recipients && pf_ambiguous);

	*pppc_recipients = (char * *) *pui_recipients = NULL;
	*pf_ambiguous = FALSE;

	//for each entry in the text list...
	pc_recip = strtok( pc_RecipientsList, pc_LIST_DELIMITER);
	_ASSERTE( pc_recip);
	do	{
		//allocate space for a copy of the entry, and copy the entry into it
		if (f_failure = !(pc = malloc( ui = strlen( pc_recip) + 1)))
			goto errJump;
		strcpy( pc, pc_recip);

		//string on a new entry to the unresolved list, transferring 
		//	ownership of the name string to the entry
		pt = pt_unresolved;
		if (f_failure = !ef_AddNameFoundNodeFifo( pc, FALSE, TRUE, &pt))
			goto errJump;
		if (pt && !pt_unresolved)
			pt_unresolved = pt;
	} while (pc_recip = strtok( NULL, pc_LIST_DELIMITER));
	pc = NULL;

	//if there are any unresolved recipient names...
	if (pt_unresolved)	{
		char * pc = epc_strtokStrict( pc_LookupHow, pc_LIST_DELIMITER);

		//attempt to resolve the names against the default Domino Directory 
		//	set
		if (us_err = eus_ResolveMailAddresses( pt_unresolved, 
												f_LOOKUP_EXHAUST, pc, NULL, 
												&pt_resolved, &ui_entries, 
												pf_ambiguous))
			goto errJump;

		//if a further server to resolve against has been specified...
		if ((pc = epc_strtokStrict( NULL, pc_LIST_DELIMITER)) && *pc)	{
			//attempt to resolve the names against the server's Domino 
			//	Directory set, if necessary
			if (!f_LOOKUP_EXHAUST)	{
				pt = pt_unresolved;
				do
					if (!pt->pt_name->f_found)
						break;
				while (pt = pt->pt_next);
			}
			if (f_LOOKUP_EXHAUST || pt)	{
				if (us_err = eus_ResolveMailAddresses( pt_unresolved, 
										f_LOOKUP_EXHAUST, pc, NULL, 
										&pt_resolved, &ui, 
										*pf_ambiguous ? NULL : pf_ambiguous))
					goto errJump;
				ui_entries += ui;
			}
		} //if (pc = strtok( NULL,

		//Tack any addresses still unresolved onto the "resolved" list so PGP 
		//	may match keys against everything. Internet addresses will often 
		//	not be resolved by the Domino Directory.
		pt = pt_unresolved;
		pt_nd = pt_resolved;
		do	{
			if (pt->pt_name->f_found)
				continue;

			if (f_failure = !ef_AddStringNodeFifo( pt->pt_name->pc_nm, FALSE, 
																	&pt_nd))
				goto errJump;
			pt->pt_name->pc_nm = NULL;
			if (!pt_resolved)
				pt_resolved = pt_nd;
			ui_entries++;
		} while (pt = pt->pt_next);
	} //if (pt_unresolved)

	//transfer control of the resolved list to a string array
	if (ui_entries)	{
		if (f_failure = !( ppc = calloc( ui_entries, sizeof( char *))))
			goto errJump;
		pt_nd = pt_resolved;
		for (ui = 0; ui < ui_entries; ui++)	{
			ppc[ ui] = pt_nd->pc;
			pt_nd = pt_nd->pt_next;
		}
		_ASSERTE( !pt_nd);

		*pppc_recipients = ppc;
		*pui_recipients = ui_entries;
	} //if (ui_entries)

errJump:
	//Free allocated resources no longer needed. If we've been successful, 
	//	the string array now owns the strings in the resolved list.
	if (pt_unresolved)
		e_FreeNameFoundList( &pt_unresolved);
	if (pt_resolved)
		e_FreeList( &pt_resolved, us_err || f_failure ? TRUE : FALSE);
	if (pc)
		free( pc);

	return us_err + f_failure;
} //us_CreateRecipientsArray(


/** i_PgpEncodeRtfText( ***


--- parameters & return ----

RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
12/15/98 PR: created		*/
//DOC!!
static __inline int i_PgpEncodeRtfText( 
								PgpEncodeContext *const  pt_EncodeContext, 
								RtfContext *const  pt_context, 
								char * *const  ppc_output)	{
	char * pc = NULL;
	int  i_err;

	_ASSERTE( pt_EncodeContext && (pt_EncodeContext->f_Encrypt || 
												pt_EncodeContext->f_Sign) && 
												pt_context && ppc_output);

	//get a copy of the rich-text field's textual content
	if (eul_ERR_FAILURE == eul_GetRtfText( pt_context, NULL, TRUE, &pc))
		return !eus_SUCCESS;

	i_err = ei_PgpEncodeBuffer( pc, strlen( pc), pt_EncodeContext, FALSE, 
															NULL, ppc_output);

	if (pc)
		free( pc);

	return i_err;
} //i_PgpEncodeRtfText(


/** i_PgpAdjustNotesNames( ***
To accommodate the intuitive UI desired in the context of PGP keys, the 
purpose of this procedure is to default any Notes names to the user-friendly 
abbreviated format, albeit while maintaining backward compatibility with the 
canonicalized format used prior to PGP 6.5.1.

--- parameters & return ----
f_PGP_SYNC_UNKNOWN_KEYS: flag telling whether names not located on the user's 
	default keyrings should be sought on the PGP certificate servers she's
	specified
ui_RECIPIENTS: the number of members in the list of recipients
ppc_recipients: Input & Output. Pointer to the string-array list of 
	recipients to check for Notes names which may be switched safely to 
	abbreviaed format. Such names will be switched. A Notes name will remain 
	in canonicalized format only if the canonicalized version is the only 
	one for which a matching key can be found within the PGP infrastructure.
pt_EncodeContext: Input & Output. Pointer to the PGP information structure 
	being used in the current round of encoding. If f_PGP_SYNC_UNKNOWN_KEYS 
	is TRUE, the opaque structure may be updated if names are located via a 
	PGP certificate-server search.
RETURN: the PGP error that occurred, if any; out-of-memory condition causes a 
	!eus_SUCCESS return; otherwise eus_SUCCESS is returned

--- revision history -------
9/12/99 PR: created			*/
static int i_PgpAdjustNotesNames( 
								const BOOL  f_PGP_SYNC_UNKNOWN_KEYS, 
								const UINT  ui_RECIPIENTS, 
								char *const *const  ppc_recipients, 
								PgpEncodeContext *const  pt_EncodeContext)	{
	NameFoundNode * pt_abbrevd = NULL, * pt_canonical = NULL, 
					* pt_node = NULL, * pt_nd = NULL;
	NameFoundInfoEx * pt, * pt_nm;
	UINT  ui;
	char * pc = NULL, * pc_src;
	BOOL  f_failure;
	int  i_err;

	_ASSERTE( ppc_recipients && pt_EncodeContext);

	//for each name in the full list...
	for (ui = 0; ui < ui_RECIPIENTS; ui++)	{
		//if the name obviously isn't a canonicalized Notes name, iterate to 
		//	the next name in the full list
		if (memcmp( ppc_recipients[ ui], mpc_TKN_NOTES_CANONICAL, 
									mui_LEN_TKN_NOTES_CANONICAL) != ei_SAME)
			continue;

		//Create an abbreviated version of the name. However if it turns out 
		//	that the name is not really a canonicalized name, iterate to the 
		//	next name in the full list.
		if (f_failure = !( pc = malloc( strlen( (pc_src = ppc_recipients[ 
										ui]) + mui_LEN_TKN_NOTES_CANONICAL))))
			goto errJump;
		if (f_failure = !epc_NotesNmAbbreviate( pc_src, pc))	{
			free( pc);
			pc = NULL;
			continue;
		}

		//create a list node to hold the name information, transferring 
		//	ownership of the abbreviated version to it
		if (f_failure = !( pt = calloc( 1, sizeof( NameFoundInfoEx))))
			goto errJump;
		pt->pc_parent = pc_src;
		pt->t.pc_nm = pc;
		pc = NULL;

		//add the new node to the abbreviated-names list
		if (f_failure = !ef_AddListNodeFifo( pt, &pt_node))	{
			free( pt);
			goto errJump;
		}
		if (!pt_abbrevd)
			pt_abbrevd = pt_node;
	} //for (ui = 0; ui < ui_RECIPIENTS

	//if no Notes names were present, short-circuit with success
	if (!pt_abbrevd)
		return eus_SUCCESS;

	//check locally within PGP and mark accordingly the abbreviated names it 
	//	knows of
	if (i_err = ei_PgpLookupEmailLocal( pt_EncodeContext, pt_abbrevd))
		goto errJump;

	//for each name in the list of abbreviated names...
	pt_node = pt_abbrevd;
	do	{
		//If the name has already been found, reset the corresponding name in 
		//	the full list, and then iterate to the next name. Since an 
		//	abbreviated Notes name is always shorter than it's canonicalized 
		//	counterpart, the simple strcpy() is safe.
		if ((pt_nm = (NameFoundInfoEx *) pt_node->pt_name)->t.f_found)	{
			strcpy( pt_nm->pc_parent, pt_nm->t.pc_nm);
			*pt_nm->t.pc_nm = NULL;
			continue;
		}

		//create a list node to hold the name information where the 
		//	canonicalized version is primary
		if (f_failure = !( pt = calloc( 1, sizeof( NameFoundInfoEx))))
			goto errJump;
		pt->pc_parent = pt_nm->t.pc_nm;
		pt->t.pc_nm = pt_nm->pc_parent;
		if (f_failure = !ef_AddListNodeFifo( pt, &pt_nd))	{
			free( pt);
			goto errJump;
		}
		if (!pt_canonical)
			pt_canonical = pt_nd;
	} while (pt_node = pt_node->pt_next);

	//if a list was populated...
	if (pt_canonical)	{
		//check locally within PGP and mark accordingly the canonicalized 
		//	names it knows of
		if (i_err = ei_PgpLookupEmailLocal( pt_EncodeContext, pt_canonical))
			goto errJump;

		//for each of the names found this way, nullify the corresponding 
		//	name entry in the abbreviated list
		pt_nd = pt_canonical;
		do
			if (pt_nd->pt_name->f_found)
				*((NameFoundInfoEx *) pt_nd->pt_name)->pc_parent = NULL;
		while (pt_nd = pt_nd->pt_next);
	} //if (pt_canonical)

	//if the user has specified that unknown names should be sought via 
	//	PGP certificate servers she's specified...
	if (f_PGP_SYNC_UNKNOWN_KEYS)	{
		//determine whether any name in the abbreviated list remains not found
		pt_nd = pt_abbrevd;
		do
			if (!( !*pt_nd->pt_name->pc_nm || pt_nd->pt_name->f_found))
				break;
		while (pt_nd = pt_nd->pt_next);

		//if any name remains not found...
		if (pt_nd)	{
			//check servers within PGP and mark accordingly the abbreviated 
			//	names it knows of
			if (i_err = ei_PgpLookupEmailViaServers( pt_EncodeContext, 
																pt_abbrevd))
				goto errJump;

			//for each node in the abbreviated list...
			pt_node = pt_abbrevd;
			pt_nd = pt_canonical;
			do	{
				//if the name wasn't just found, iterate to the next list node
				if (!( *(pt = (NameFoundInfoEx *) 
								pt_node->pt_name)->t.pc_nm && pt->t.f_found))
					continue;

				//flag the associated node in the canonical list that the 
				//	name has been found
				strcpy( pt->pc_parent, pt->t.pc_nm);
				do
					if (pt->t.pc_nm == ((NameFoundInfoEx *) 
												pt_nd->pt_name)->pc_parent)	{
						pt_nd->pt_name->f_found = TRUE;
						break;
					}
				while (pt_nd = pt_nd->pt_next);
				*pt->t.pc_nm = NULL;
			} while (pt_node = pt_node->pt_next);
		} //if (pt_nd)

		//determine whether any name in the canonicalized list remains not 
		//	found
		pt_nd = pt_canonical;
		do
			if (!pt_nd->pt_name->f_found)
				break;
		while (pt_nd = pt_nd->pt_next);

		//if any name remains not found...
		if (pt_nd)	{
			//check servers within PGP and mark accordingly the 
			//	canonicalized names it knows of
			if (i_err = ei_PgpLookupEmailViaServers( pt_EncodeContext, 
																pt_canonical))
				goto errJump;

			//For each of the canonicalized names marked as found, nullify 
			//	the corresponding name entry in the abbreviated list. It's 
			//	probably more process-efficient to needlessly reset the 
			//	pc_parent member than to test whether it needs resetting.
			pt_nd = pt_canonical;
			do
				if (pt_nd->pt_name->f_found)
					*((NameFoundInfoEx *) pt_nd->pt_name)->pc_parent = NULL;
			while (pt_nd = pt_nd->pt_next);
		} //if (pt_nd)
	} //if (f_PGP_SYNC_UNKNOWN_KEYS)

	//for each remaining name in the list of abbreviated names still not 
	//	found, replace the corresponding entry in the full list with the 
	//	abbreviated entry
	pt_nd = pt_abbrevd;
	do
		if (*pt_nd->pt_name->pc_nm)	{
			strcpy( ((NameFoundInfoEx *) pt_nd->pt_name)->pc_parent, 
													pt_nd->pt_name->pc_nm);
			_ASSERTE( !pt_nd->pt_name->f_found);
		}
	while (pt_nd = pt_nd->pt_next);

errJump:
	if (pt_canonical)
		e_FreeList( &pt_canonical, TRUE);
	if (pt_abbrevd)
		e_FreeNameFoundList( &pt_abbrevd);
	if (pc)
		free( pc);

	return f_failure + i_err;
} //i_PgpAdjustNotesNames(


/** xus_ResetToPgpEncodedRtf( ***
Resets the specified note's rich-text field to the PGP-encoded version saved 
in memory via a prior call to es_PreMailSend. This is necessary when the 
Notes R5 mail template is in use. More specifically, the R5 template 
implements the "Save and File" action differently from the R4 template, 
driving the need for the reset functionality. For more specificis, see the 
documentation in the PGP subform concerning this issue.

--- parameters & return ----
h_NOTE: handle to the note whose rich-text field is to be reset to the 
	PGP-encoded version
pc_ITMNM_RTF: pointer to a string telling the name of the rich-text field to 
	be used in the encoding
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
2/27/00 PR: created		*/
unsigned short xus_ResetToPgpEncodedRtf( const NOTEHANDLE  h_NOTE, 
											char  pc_ITMNM_RTF[])	{
	RtfContext  t_RtfContext;
	BYTE * puc;
	STATUS  us_err;

	if (!( h_NOTE && pc_ITMNM_RTF))
		return !eus_SUCCESS;

	//if no PGP-encoded rich-text field is ready to be written to the note, 
	//	short-circuit with a general failure
	if (mt_RtfResettable.h_note != h_NOTE)
		return !eus_SUCCESS;
	_ASSERTE( mt_RtfResettable.h_content);

	//initialize resources associated with the rich-text handling we're 
	//	going to do
	if (us_err = eus_InitializeRtfContext( h_NOTE, pc_ITMNM_RTF, NULL, 
															&t_RtfContext))
		return us_err;

	//replace the current rich-text field with the PGP-encoded version 
	//	stored during the "PreMailSend" that must have just before taken 
	//	place
	puc = OSLockObject( mt_RtfResettable.h_content);
	us_err = eus_ReplaceRtfWithCdStream( puc, mt_RtfResettable.ul_len, 
															&t_RtfContext);
	OSUnlockObject( mt_RtfResettable.h_content);
	if (us_err)
		goto errJump;

	//commit the updated rich-text field to the note, overwriting the current 
	//	actual rich-text content
	us_err = eus_CommitChangedRtf( h_NOTE, pc_ITMNM_RTF, &t_RtfContext);

errJump:
	//free resources allocated to the stored version, and flag that this has 
	//	been done
	OSMemFree( mt_RtfResettable.h_content);
	memset( &mt_RtfResettable, NULL, sizeof( ResettableRtfInfo));

	ef_FreeRtfContext( &t_RtfContext);

	return us_err;
} //xus_ResetToPgpEncodedRtf(


/** x_ClearEncodedRtfCopy( ***
Free resources allocated to the copy of the PGP-encoded rich-text field that 
was stored in memory in case it was needed to reset the message's rich-text 
field with the encoded version (a functionality required due to how Lotus 
implemented its "Save and File" action in the R5 template).

--- parameter ----------
H: handle to the note initiating this procedure call

--- revision history ---
2/27/00 PR: created		*/
void x_ClearEncodedRtfCopy( const NOTEHANDLE  H)	{
	if (!( H && mt_RtfResettable.h_note == H))
		return;

	OSMemFree( mt_RtfResettable.h_content);
	memset( &mt_RtfResettable, NULL, sizeof( ResettableRtfInfo));
} //x_ClearEncodedRtfCopy(


