#
# ReplayCapture: This is the event capturing callback.  Every event calls
#    this procedure.  It:
#        1. records the action (by calling RecordAction
#        2. looks up the original action (saved in ReplayData)
#        3. executes the original binding (by calling ReplayAction).
#
proc ReplayCapture {subscript replaceList} {
    global ReplayData
    # Ignore mouse events when we are replaying a script.
    if {$ReplayData(playingOn)} {
        return
    }
    set action $ReplayData($subscript)
    RecordAction $subscript $replaceList
    ReplayAction $action $replaceList
}
#
# ReplayAction: this procedure executes the original binding for an action.
#    It is used while recording events to execute the action requested and
#        it is used when replaying events to replay the action.
#    Note: we must do the %-substitutions just like the event binding
#        expects.  This is done by SubstituteInAction.
#    Note: The action must be executed at the global level since all
#        event handlers expect to be executed at the global level.
#
proc ReplayAction {action replaceList} {
    global targetApp
    set isw [lookup Widget $replaceList]
    if {$isw != {}} {
        set args [lookup Args $replaceList]
        switch $isw \
            "Scale" {
                set w [lookup W  $replaceList]
                SendToApp "$w set $args"
            }
        if {$action == {}} {
            return
        }
        set subbedAction [concat $action $args]
    } else {
        set subbedAction [SubstituteInAction $action $replaceList]
    }
    SendToApp $subbedAction
}
#
# SubstituteInAction: this procedure takes an action script and does
#     all the %-replacements.  The parameters "replaceList: contains
#     all the necessary replacement strings.
#
proc SubstituteInAction {action replaceList} {
    set len [string length $action]
    set subbedAction ""
    #
    # go through each character of the script looking for %s
    #
    for {set i 0} {$i < $len} {incr i} {
        set ch [string index $action $i]
        if {[string compare $ch "%"]==0} {
            # we found a %, see what to substitute
            incr i
            set ch [string index $action $i]
            if {[string compare $ch "%"]==0} {
                # a %% is replaced by a %
                append subbedAction "%"
            } else {
                # replace the %? which the appropriate string
                append subbedAction [lookup $ch $replaceList]
            }
        } else {
            # other characters are just copied over
            append subbedAction $ch
        }
    }
    return $subbedAction
}
#
# lookup: lookup a %-character in the association list.
#     The format of "alist" is:
#         { {A a-string} {B b-string} ...}
#
proc lookup {item alist} {
    foreach pair $alist {
        if {[string compare $item [lindex $pair 0]]==0} {
            return [lindex $pair 1]
        }
    }
    return ""
}
#
# RecordAction: records an action in the replay list (if recording is on)
#
proc RecordAction {subscript replaceList} {
    global ReplayData
    #
    # Only record events when the recording switch is on.
    #
    if {!$ReplayData(recordingOn)} {
        return
    }
    #
    # do not rebind the widgets in the replay windows
    #
    set w [lookup W $replaceList]
    if [string match .replay.* $w] {
        return
    }
    #
    # Compute the delay between this event and the last event
    # and call InsertAction to insert it into the replay list.
    #
    set currentTime $ReplayData(Timer)
    set delay [expr $currentTime-$ReplayData(LastEventAt)]
    set ReplayData(LastEventAt) $currentTime
    InsertAction $delay $subscript $replaceList
}
#
# InsertAction: insert an action in the replay script.
#     delay: the time between this event and the last event
#     subscript: the subscript of the event which records everything
#         we need to know about it, in particular, where to find
#         the script of the original binding
#     replaceList: the details of the event so we can replay it exactly
#          as it happened.
#
proc InsertAction {delay subscript replaceList {scriptInsert 1}} {
    global ReplayData
    set item [list $delay $subscript $replaceList]
    if $scriptInsert {
        lappend ReplayData(script) $item
    }
    if $ReplayData(showEventsRecording) {
        .replay.events.times insert end $delay
        .replay.events.subs insert end $subscript
        .replay.events.reps insert end $replaceList
        # show the last five items in the list box
        set size [expr [.replay.events.times size]-5]
        if {$size < 0} {set size 0}
        foreach list {times subs reps} {
           .replay.events.$list yview $size
        }
    }
}
#
# ReplayTimerTick: this is called every 100ms while recording is on.
#     It is used to record the delay between events so we can replay
#     them at the same speed they were recorded at.
#
proc ReplayTimerTick {} {
    global ReplayData
    # has the timer been turned off?
    if {$ReplayData(Timer) < 0} {
        return
    }
    incr ReplayData(Timer)
    if {$ReplayData(recordingOn)} {
        after 100 ReplayTimerTick
    }
}




















