#
# NeoWebScript - Server Side Programming based on Safe Tcl
#
# Copyright (C) NeoSoft, All Rights Reserved.  See NeoWebScript LICENSE
# files with this distribution for conditions on use and redistribution.
#
# $Id: image.tcl,v 1.1.1.1 1999/03/31 20:53:26 damon Exp $
#

proc send_image_request {safeInterp} {
    global webenv imageInfo
    if [getImageCacheFile cache_file] {
#echo "Returning cached image $cache_file"
	return [open $cache_file]
    }
    setup_data_access
    set filename [md5 $webenv(SCRIPT_FILENAME)]
    db_name_check $filename data dataFileName
    if ![file isdirectory [file dirname $dataFileName]] {
	create_user_dir data
    }
    set tmpname [file rootname $dataFileName].[id process]
    set imageInfo(fp) [open $tmpname "CREAT RDWR" 0660]
    interp share "" $imageInfo(fp) $safeInterp
    load {} Gd $safeInterp
    set user_code [read_file $webenv(SCRIPT_FILENAME)]
    $safeInterp alias image_expire image_expire
    $safeInterp eval "
	rename include_virtual {}
	rename include_file {}
	set imageFile $imageInfo(fp)
    "
    if [catch {$safeInterp eval $user_code}] {
	set_header Content-type text/plain -force
	global errorInfo
	puts $imageInfo(fp) $errorInfo
    }
    flush $imageInfo(fp)
    if [fstat $imageInfo(fp) size] {
	catch {unlink $dataFileName}
	seek $imageInfo(fp) 0
	if [info exists imageInfo(expiration)] {
	    frename $tmpname $dataFileName
	    set db [open_image_db rwL]
	    db put $db $webenv(SCRIPT_FILENAME) [list $dataFileName \
		$imageInfo(expiration) [file mtime $webenv(SCRIPT_FILENAME)]]
	    db close $db
#echo "Saving and returning new image"
	    return $imageInfo(fp)
	} else {
	    unlink $tmpname
#echo "No expiration"
	    return $imageInfo(fp)
	}
    } else {
	unlink $tmpname
#echo "No image generated"
	return; # no image generated
    }
}

#
# handle_cached_location:
#   This is called if an image generation is requested, but there is
#   no file actually present: on a cached file generated somewhere else.
#
proc handle_cached_location {} {
    if [getImageCacheFile cachedname] {
	return [open $cachedname]
    }
    return
}

proc image_expire {seconds} {
    global imageInfo
    set imageInfo(expiration) [expr [clock seconds] + $seconds]
#echo Expires: $imageInfo(expiration), [gm_timestr_822 $imageInfo(expiration)]
    set_header Expires [gm_timestr_822 $imageInfo(expiration)]
}

#
# getImageCacheFile:
#    Stuff pathname to cached file in $filenamevar if it exists, has not
#     expired, and the script file has not been modified.
#     1 == valid cache file exists, 0 it does not.
proc getImageCacheFile {filenamevar} {
    global webenv
    set list [getImageCacheInfo]
#echo cache check: found = $found, list = $list
    if ![lempty $list] {
	lassign $list dataFileName expiration scriptLastMod
#echo expire times: [expr $expiration - [clock seconds]]
	if {$scriptLastMod == [file mtime $webenv(SCRIPT_FILENAME)] &&
		[clock seconds] < $expiration} {
	    set_header Expires [gm_timestr_822 $expiration]
	    uplevel set $filenamevar $dataFileName
	    return 1
	}
	delImageCacheInfo
    }
    return 0; # no cache available
}

#
# getImageCacheInfo: retrieve cache data from the database, if the file
#                      exists.
# input: name of image file (or empty string for current URI).
# output: list from the database (or empty string if does not exist).
#
proc getImageCacheInfo {{imagename {}}} {
    global webenv
    if [lempty $imagename] {
	set filename $webenv(SCRIPT_FILENAME)
    } else {
	set dir [file dirname $webenv(SCRIPT_FILENAME)]
	set filename [file join $dir $imagename]
    }
    set db [open_image_db r]
    set found [db get $db $filename list]
    db close $db
    if !$found return
    if [file exists [lindex $list 0]] {
	return $list
    } else {
	delImageCacheInfo $imagename
    }
    return
}

SAFE_proc_and_alias image_open {safeInterp {imagename {}}} {
    set info [getImageCacheInfo $imagename]
    if [lempty $info] return
    set dataFileName [lindex $info 0]
    set fp [open $dataFileName]
    interp transfer "" $fp $safeInterp
    return $fp
}

SAFE_proc_and_alias image_create {safeInterp imagename {expire 60}} {
    global webenv
    setup_data_access
    set filename [md5 $webenv(SCRIPT_FILENAME)]
    db_name_check $filename data dataFileName
    if ![file isdirectory [file dirname $dataFileName]] {
	create_user_dir data
    }
    set db [open_image_db rwL]
    db put $db $webenv(SCRIPT_FILENAME) [list $dataFileName \
	$imageInfo(expiration) [file mtime $webenv(SCRIPT_FILENAME)]]
    db close $db
    if {$mode == "w"} {
	global webenv
	set db [open_image_db rwL]
	db put $db $webenv(SCRIPT_FILENAME) [list $dataFileName \
	    $imageInfo(expiration) [file mtime $webenv(SCRIPT_FILENAME)]]
	db close $db
    }
    set fp [open $dataFileName w]
    interp transfer "" $fp $safeInterp
    return $fp
}

proc delImageCacheInfo {{imagename {}}} {
    global webenv
    if [lempty $imagename] {
	set filename $webenv(SCRIPT_FILENAME)
    } else {
	set dir [file dirname $webenv(SCRIPT_FILENAME)]
	set filename [file join $dir $imagename]
    }
    set db [open_image_db rwL]
    db del $db $filename
    db close $db
}

proc open_image_db {mode} {
    global parallelSystemBase
    set dbFileName $parallelSystemBase/images.db
    if {[catch {set db [db open $dbFileName hash $mode 0644]}] == 1} {
	set db [db open $dbFileName hash ctwL 0664]
    }
    return $db
}

package provide Image 1.0
