#
# NeoWebScript - Server Side Programming based on Safe Tcl
#
# Copyright (C) NeoSoft, All Rights Reserved.  See NeoWebScript LICENSE
# files with this distribution for conditions on use and redistribution.
#
# $Id: init.tcl,v 1.3 1999/07/09 20:49:08 damon Exp $
#

#
# If we're debugging, we load devel.tcl every time we need to
# create a safe interpreter.
#
# If we're not debugging, we load devel.tcl at startup.
#
# devel_setup is executed every time a safe interpreter is
# created, in either case.
#

set debugging 1
catch {rename copyfile ""}
catch {rename unsupported0 copyfile}

# generic paths for neowebscript files.
set binPath [file join $server(SERVER_ROOT) bin]
set nwsPath [file join $server(SERVER_ROOT) neowebscript]
set logPath [file join $server(SERVER_ROOT) logs]

set parallelUserBase [file join $server(SERVER_ROOT) neowebscript neoscript-data users]
set parallelSystemBase [file join $server(SERVER_ROOT) neowebscript neoscript-data system]

set nwsLocalPath [file join $nwsPath nwslocal]

# safe_and_trusted_proc: for procs that have identical args and code
# in either interp.  Autoloaded.
proc safe_and_trusted_proc {name arglist body} {
    global safe_proc_cache
    set safe_proc_cache($name) [list proc $name $arglist $body]
    proc $name $arglist $body
}

# Declare as alias_proc if the routine will be called exactly the
# same in either trusted or safe interp.  Command in trusted interp
# is the same as the alias name, and safeInterp is not passed.
proc safe_and_trusted_alias {name arglist body} {
    global safe_alias_cache
    proc $name $arglist $body
    set safe_alias_cache($name) $name
}

# Declare as safe_proc to make available only for safe interp.  Autoloaded.
proc safe_proc {name arglist body} {
    global safe_proc_cache
    set safe_proc_cache($name) [list proc $name $arglist $body]
}

# Safe_alias declare an alias the safe interp.  Autoloaded.
proc safe_alias {name args} {
    global safe_alias_cache
    set safe_alias_cache($name) $args
}

# Declare as SAFE_proc_and_alias for situation where SAFE_$procname is
# the alias and it takes the interpreter as its first argument.
# Arglist must allow for safeInterp name to be passed as first arg.
proc SAFE_proc_and_alias {name arglist body} {
    global safe_alias_cache
    proc SAFE_$name $arglist $body
    set safe_alias_cache($name) "SAFE_$name \$safeInterp"
}

# load in shared base functions
source [file join $nwsPath common.tcl]

# load in the db commands
source [file join $nwsPath db.tcl]

# load in the neo forms 1 package
source [file join $nwsPath neoform1.tcl]

# load in client.tcl
source [file join $nwsPath client.tcl]

# Load in Postgres if the configuration says we have it
if ![info exists NeoWebServerConf(HavePostgres)] {
    set NeoWebServerConf(HavePostgres) 0
}
if $NeoWebServerConf(HavePostgres) {
    source [file join $nwsPath postgres.tcl]
}

#
# Check path to executables defined by NeoWebServerConf.  If not defined,
# or path isn't an executable, then try to guess one.
#
foreach {command paths} [list \
    SendMail [list /usr/sbin/sendmail /usr/lib/sendmail] \
    Du [list /usr/bin/du] \
    WebUnpack [list [file join $binPath webunpack]] \
] {
    if {![info exists NeoWebServerConf($command)] \
      || ![file executable $NeoWebServerConf($command)]} {
        foreach path $paths {
	    if [file executable $path] {
	        set NeoWebServerConf($command) $path
		break
	    }
	}
    }
}

#
# explicitly load devel.tcl into the master interp if debugging is off.
# else, it is loaded into the safe interp when the latter is always created.
#
if !$debugging {
    source [file join $nwsPath devel.tcl]
}

#
# Log a message to the log directory we keep for the users.
#
SAFE_proc_and_alias log_message {safeInterp logfile string} {
    global parallelDir errorCode errorInfo webenv

    setup_data_access

    db_name_check $logfile log logFileName

    if {[catch {set fp [open $logFileName "CREAT WRONLY APPEND" 0660]} result] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX ENOENT"} {
	    error "$logFileName: $errorCode"
	}
	create_user_dir log
	set fp [open $logFileName "CREAT WRONLY APPEND" 0660]
    }
    puts $fp [list [clock seconds] $webenv(REMOTE_HOST) $string]
    close $fp
    return
}

#
# send email using sendmail
#
SAFE_proc_and_alias open_outbound_mail {safeInterp args} {
    global webpageOwnerName webenv NeoWebServerConf

    if ![info exists NeoWebServerConf(SendMail)] {
        error "server not configured to send email"
    }

    if {[cindex $args 0] != "-"} {
	lassign $args data(subject) data(to) data(from)
    } else {
	import_keyvalue_pairs data $args
    }

    if ![info exists data(subject)] {
        error "-subject not specified"
    }

    setup_data_access

    regsub {^www\.} $webenv(SERVER_NAME) {} server_name

    if {![info exists data(from)] || [lempty $data(from)]} {
        set data(from) "$webpageOwnerName@$server_name"
    }
    if {![info exists data(to)]} {
        set data(to) "$webpageOwnerName@$server_name"
    }
    regsub -all " " $data(from) "" data(from)


    set fp [open "|$NeoWebServerConf(SendMail) -t -f $data(from)" w]
    puts $fp "From: $data(from)"
    puts $fp "To: $data(to)"
    puts $fp "Subject: $data(subject)"
    foreach header [lmatch [array names data] {[A-Z]*}] {
	if [info exists data($header)] {
	    puts $fp "$header: $data($header)"
	}
    }
    puts $fp "X-Mailer: $webenv(SERVER_SOFTWARE)"
    puts $fp "X-Webserver-Host: $webenv(SERVER_NAME)"
    puts $fp "X-Sender-URI: $webenv(DOCUMENT_URI)"
    puts $fp "X-Visitor-Host: $webenv(REMOTE_HOST) at [clock format [clock seconds]]"
    puts $fp ""

    interp transfer "" $fp $safeInterp
    return $fp
}

#
# Post news via NNTP
#
SAFE_proc_and_alias open_post_news {safeInterp args} {
    global webpageOwnerName webenv

    import_keyvalue_pairs data $args

    if ![info exists data(newsgroups)] {
        error "-newsgroups not specified"
    }

    if ![info exists data(subject)] {
        error "-subject not specified"
    }

    if ![info exists data(distribution)] {
        set data(distribution) ""
    }

    setup_data_access

    if {[info exists data(from)]} {
        set from $data(from)
    } else {
	set from "$webpageOwnerName@[string trimleft $webenv(SERVER_NAME) www.]"
    }

    if {[catch {set fp [socket news 119]} result] == 1} {
        error "news server unavailable: $result"
    }
    fconfigure $fp -translation lf
    gets $fp line
    if {[string index $line 0] != 2} {
        error "news server not ready: $line"
    }
    puts $fp "post"
    flush $fp
    gets $fp line
    if {[string index $line 0] != "3"} {
        error "news server post command failed: $line"
    }

    puts $fp "From: $from"
    puts $fp "Newsgroups: $data(newsgroups)"
    puts $fp "Subject: $data(subject)"
    puts $fp "Distribution: $data(distribution)"
    if [info exists data(approved)] {
	puts $fp "Approved: $data(approved)"
    }
    puts $fp "X-News-Posting-Program: $webenv(SERVER_SOFTWARE)"
    puts $fp "X-Webserver-Host: $webenv(SERVER_NAME)"
    puts $fp "X-Sender-URI: $webenv(DOCUMENT_URI)"
    puts $fp "X-Visitor-Host: $webenv(REMOTE_HOST) at [clock format [clock seconds]]"

    puts $fp ""
    interp transfer "" $fp $safeInterp
    return $fp
}

#
# invokes unix 'du' command
#
SAFE_proc_and_alias directory_listing {safeInterp {pattern *}} {
    global NeoWebServerConf

    if ![info exists NeoWebServerConf(Du)] {
        error "server not configured to show directory listings"
    }

    set duFP [open "|$NeoWebServerConf(Du) -sk" r]
    gets $duFP line
    close $duFP

    set total [lindex $line 0]
    html "Total space used from this level down is $total Kbytes."

    html "<table border>"
    html "<tr><th>name</th><th>type</th><th>size</th><th>last mod</th></tr>"
    html "<tr><td><a href=..>..</a></td><td>directory</td><td>.</td><td>.</td>"
    foreach file [lsort [glob -nocomplain $pattern]] {
        if {[catch {file stat $file x} result] == 1} {
	    html "<tr><td>$file</td><td>$result</td></tr>"
	} else {
	    set date [clock format $x(mtime)]
	    html [format "<tr><td><a href=%s>%s</a></td><td>%s</td><td align=right>%s</td><td>%s</td></tr>\n" $file $file $x(type) $x(size) $date]
	}
    }
    html "</table>"
}

#
# guess at the number of hits we're getting
# per hour by calculating the rate of growth
# of the log file.
#
# assumes logs are in SERVER_ROOT/logs/neolog
#
safe_and_trusted_alias estimate_hits_per_hour {} {
    global logPath

    set hitsInInterval 1000
    set bytesPerLogEntry 131

    set fp [open [file join $logPath access_log]]
    set size [fstat $fp size]
    set offset [expr $size - $hitsInInterval * $bytesPerLogEntry]
    if {$offset > 0} {
	seek $fp $offset
	gets $fp
    } else {
	set hitsInInterval [expr $size / $bytesPerLogEntry]
    }

    set result [gets $fp line]
    close $fp
    if {$result < 0} {return 0}

    set ET [expr [clock seconds] - [lindex $line 0]]
    return [expr ($hitsInInterval * 3600) / $ET]
}

#
# Return the hostname of the machine that is fetching the current
# page.  If we don't have the name because they're running the
# server with minimal DNS, perform a DNS lookup for the hostname
# and return that.
#
# If we can't figure out the hostname, we return the ip address.
#
safe_and_trusted_alias remote_hostname {{ip 0}} {
    global webenv

    if {$ip != 0} {
	if {[catch {host_info official_name $ip} res] == 1} {
	   return $ip
	}
    } else {
	if {$webenv(REMOTE_HOST) != $webenv(REMOTE_ADDR)} {
	   return $webenv(REMOTE_HOST)
	}
	if {[catch {host_info official_name $webenv(REMOTE_ADDR)} res] == 1} {
	   return $webenv(REMOTE_ADDR)
	}
    }
    return $res
}

#
# (invoked by Apache) destroy the safe interpreter if it already exists,
# then create a new one and install our services (exported procs) into it.
#
proc setup_safe_interpreter {} {
    global debugging parallelDir errorInfo webenv \
           NeoWebServerConf NeoWebDirConf nwsPath
    if [info exists parallelDir] {
        unset parallelDir
        set errorInfo ""
    }
    if {[info exists NeoWebDirConf(Supervisor)] && $NeoWebDirConf(Supervisor)} {
	set safeInterp [interp create]
	load {} Neo $safeInterp
    } else {
	set safeInterp [::safe::interpCreate]
	::safe::setLogCmd log_safe
    }
    load {} Tclx $safeInterp
    init_setup $safeInterp
    extend_slave $safeInterp
    if $debugging {
	global server
	uplevel #0 source [file join $nwsPath devel.tcl]
    }
    if $NeoWebServerConf(HavePostgres) {
	postgres_setup $safeInterp
    }
    if [catch {local_setup $safeInterp} errorMsg] {
    }
    
    if [catch {devel_setup $safeInterp} errorMsg] {
	puts stderr {devel_setup error}
	global errorInfo
	puts stderr $errorInfo
    }
    setup_slave_interp_unknown $safeInterp
    return $safeInterp
}

# set tracefp [open /tmp/safetrace a]

proc log_safe {args} {
    return
    global tracefp
    puts $tracefp $args
    flush $tracefp
}

#
# create aliases to mod_neoscript functions, and to safe procs in init.tcl
#
proc init_setup {safeInterp} {
    $safeInterp alias abort_page abort_page
    $safeInterp alias flush_page flush_page
    $safeInterp alias html html
    $safeInterp alias set_header set_header

    $safeInterp alias unquote_string www_unescape_string
    $safeInterp alias quote_string www_escape_string
    $safeInterp alias escape_sgml_chars www_escape_sgml_chars

    $safeInterp alias include_file SAFE_include_file $safeInterp
    $safeInterp alias include_virtual SAFE_include_virtual $safeInterp
    $safeInterp alias load_file SAFE_load_file $safeInterp
    $safeInterp alias load_virtual SAFE_load_virtual $safeInterp

    $safeInterp alias estimate_hits_per_hour estimate_hits_per_hour
    $safeInterp alias remote_hostname remote_hostname
    $safeInterp alias gm_timestr_822 gm_timestr_822

    if [interp issafe $safeInterp] {
	$safeInterp alias file SAFE_file $safeInterp
	$safeInterp alias source SAFE_source $safeInterp
    }

    global webenv
    $safeInterp eval array set webenv [list [array get webenv]]
}

#
# load procs in neowebscript/nwslocal/*.tcl
#
proc local_setup {safeInterp} {
    global nwsLocalPath
    foreach lfile [lsort [glob -nocomplain [file join $nwsLocalPath *.tcl]]] {
	source $lfile
    }
}

#
# Experimental implementation of unknown
#
proc setup_slave_interp_unknown {safeInterp} {
    global webenv auto_path NeoWebDirConf UnknownTcl tcl_platform tcl_library
    global nwsPath
    set local_lib_path [file dirname [www_simplify_pathname $webenv(SCRIPT_FILENAME)]]
    if [interp issafe $safeInterp] {
	::safe::interpAddToAccessPath $safeInterp $local_lib_path
	$safeInterp alias open SAFE_open $safeInterp
	$safeInterp alias glob SAFE_glob $safeInterp
	$safeInterp alias read_file SAFE_read_file $safeInterp
    }
    lappend auto_path $local_lib_path
    $safeInterp eval "set auto_path [list $auto_path]"
    $safeInterp eval "set tcl_library [list $tcl_library]
		array set tcl_platform [list [array get tcl_platform]]"
    $safeInterp alias SAFE_autoload SAFE_autoload $safeInterp
    $safeInterp eval {
	rename auto_load slave_auto_load
	proc auto_load {cmd {namespace {}}} {
	    if [SAFE_autoload $cmd] {return 1}
	    slave_auto_load $cmd $namespace
	}
    }
}

proc handle_subst_request {safeInterp handle} {
    set text [read $handle]
    $safeInterp eval {
	rename include_file ""
	rename include_virtual ""
	rename load_file ""
	rename load_virtual ""
	rename html ""
    }
    set html [$safeInterp eval [list subst $text]]
    html $html
    close $handle
}

#
# The code called from the interface from Apache
#
# safeInterp is the interpreter we created for
# our mod_neoscript C module when it asked us to
# (setup_safe_interpreter, above)
#
# tag and tag_val are the key-value pairs in the server
# side include
#
proc handle_neowebscript_request {safeInterp tag tag_val} {
    global debugging errorInfo

    switch $tag {
	"return" {
	    handle_server_side_return $safeInterp $tag_val
	}
	"code" {
	    handle_server_side_eval $safeInterp $tag_val
	}
	"eval" {
	    handle_server_side_eval $safeInterp $tag_val
	}
	"var" {
	    handle_server_side_variable $safeInterp $tag_val
	}
	"expr" {
	    handle_server_side_expr $safeInterp $tag_val
	}
	default {
	    html "\[Error -- unrecognized tag '$tag' in NeoWebScript directive\]"
	}
    }

    return
}

#
# The code called from handle_neowebscript_request to do an eval=
#
proc handle_server_side_eval {safeInterp ssiCode} {
    global debugging errorInfo

    if {[catch {$safeInterp eval $ssiCode} result] == 1} {
	html "\{NEOWEBSCRIPT ERROR: $result\}"
	if {$debugging} {
	    html "\n<pre>$errorInfo</pre>"
	}
    }
    return
}

#
# The code called from handle_neowebscript_request to do a return=
#
proc handle_server_side_return {safeInterp ssiCode} {
    global debugging errorInfo

    if {[catch {$safeInterp eval $ssiCode} result] == 1} {
	html "\{NEOWEBSCRIPT ERROR: $result\}"
	if {$debugging} {
	    html "\n<pre>$errorInfo</pre>"
	}
    } else {
	if {$result != ""} {
	    html $result
	}
    }
    return
}

#
# The code called from handle_neowebscript_request to do a var=
#
proc handle_server_side_variable {safeInterp ssiVar} {
    global debugging

    if {[catch {$safeInterp eval set $ssiVar} result] == 1} {
	html "\{NEOWEBSCRIPT ERROR: $result\}"
	if {$debugging} {
	    html "\n<pre>$errorInfo</pre>"
	}
    } else {
	if {$result != ""} {
	    html $result
	}
    }
    return
}

#
# The code called from handle_neowebscript_request to do a var=
#
proc handle_server_side_expr {safeInterp ssiExpr} {
    global debugging

    if {[catch {$safeInterp eval expr $ssiExpr} result] == 1} {
	html "\{NEOWEBSCRIPT ERROR: $result\}"
	if {$debugging} {
	    html "\n<pre>$errorInfo</pre>"
	}
    } else {
	if {$result != ""} {
	    html $result
	}
    }
    return
}

#
# The code called when generating an image is requested
#
proc handle_image_request {safeInterp} {
    package require Image
    return [send_image_request $safeInterp]
}

package ifneeded Image 1.0 {
    source [file join $nwsPath image.tcl]
}

#
# autoload cached procs into the safe interp
#
proc SAFE_autoload {safeInterp cmd} {
    global safe_proc_cache safe_alias_cache
    if [info exists safe_proc_cache($cmd)] {
	if [catch {$safeInterp eval $safe_proc_cache($cmd)} result] {
	    global tracefp errorInfo
	    puts tracefp $errorInfo
	    return -code error $result
	}
	return 1
    }
    if [info exists safe_alias_cache($cmd)] {
	eval $safeInterp alias $cmd [subst $safe_alias_cache($cmd)]
	return 1
    }
    return 0
}

lappend auto_path [file join $nwsPath neoscript-tcl]

proc isfile_safe {fileName} {
    global auto_path tcl_platform
    if {[lsearch $auto_path [file dirname $fileName]] > -1} {return 1}
    if {[cindex $fileName 0] == "/" || [string match *..* $fileName] \
	|| [string match */.* $fileName] || [string match *~* $fileName]} {
	return 0
    }
    return 1
}

proc SAFE_open {safeInterp filename args} {
    if ![isfile_safe $filename] {
	return -code error "$filename: Invalid pathname requested"
    }
    set channel [$safeInterp invokehidden open $filename r]
    return $channel
}

proc SAFE_source {safeInterp filename} {
    if ![isfile_safe $filename] {
	return -code error "$filename: Invalid pathname requested"
    }
    $safeInterp eval [read_file $filename]
}

proc SAFE_glob {safeInterp args} {
    foreach pattern $args {
	if ![isfile_safe $pattern] {
            return -code error "Illegal filename: $pattern"
        }
    }
    return [eval glob $args]
}

proc SAFE_read_file {safeInterp filename} {
    if ![isfile_safe $filename] {
        return -code error "Illegal filename: $filename"
    }
    return [read_file $filename]
}

proc SAFE_file {safeInterp args} {
    lassign $args command

    set setVar 0
    set err "not allowed to invoke subcommand $command of file"
    switch $command {
	attributes { return -code error $err }
	copy { return -code error $err }
	delete { return -code error $err }
	mkdir { return -code error $err }
	rename { return -code error $err }

	lstat { set setVar 1 }

	stat { set setVar 1 }
    }

    set res [eval file $args]
    if $setVar {
	lassign $args command file varName
	$safeInterp eval array set $varName [list [array get $varName]]
    }
    return $res
}

#
# handle_neowebscript_mime_upload
#
# Called from C to setup for MIME uploads.  Is expected to create
# a unique file to receive the data in, open it for write, write
# out the content type and content length, and return the name
# of the file it used.
#
# The C code then takes this filename, opens it for append, and copies
# in all of the incoming data.  Once it's in, the C code calls
# finish_neowebscript_mime_upload
#
proc handle_neowebscript_mime_upload {} {
    global webenv mimeFilename

    set mimeFilename "/tmp/neowebscript.mime.[clock seconds].[id process]"
    set fp [open $mimeFilename w]
    puts $fp "Content-Type: $webenv(CONTENT_TYPE)"
    puts $fp "Content-Length: $webenv(CONTENT_LENGTH)"
    puts $fp ""
    close $fp
    return $mimeFilename
}

#
# finish_neowebscript_mime_upload
#
# We've received a MIME upload, we now use our webunpack tool, which
# is a customized version of the CMU mpack code, to crack it out into
# a directory.  Set up the directory, dig the names of the fields out
# of the "andTheRest" part.  Put the results in the mime response and
# mime uploaded files arrays.
#
proc finish_neowebscript_mime_upload {} {
    global mimeFilename mimeResponse mimeUploadedFiles mimeUnpackDir \
    	NeoWebServerConf

    set mimeUnpackDir $mimeFilename.dir
    mkdir $mimeUnpackDir

    set unpackInfo [exec $NeoWebServerConf(WebUnpack) $mimeUnpackDir <$mimeFilename]

    foreach line [split $unpackInfo "\n"] {
	set andTheRest [join [lassign $line tempName contentType fieldType requestedFileName]]

	if ![regexp {[Nn][Aa][Mm][Ee]="([^"]*)"} $andTheRest dummy name] {
	    error "MIME upload failed to find name field in '$andTheRest'"
	}

	if [string match *\* $requestedFileName] {
	    set requestedFileName [lindex [split $requestedFileName \\] end]
	}

	set fieldTempFileName $mimeUnpackDir/$tempName
	if {$fieldType == "field"} {
	    set mimeResponse($name) [read_file $fieldTempFileName]
	    unlink $fieldTempFileName
	} elseif {$fieldType == "file"} {
	    set mimeUploadedFiles($name) [list $tempName $requestedFileName [file size $fieldTempFileName] $contentType]
	} else {
	    error "unrecognized field type '$fieldType' processing MIME file $mimeFilename"
	}
    }
}

#
# cleanup_neowebscript_mime_upload
#
# Remove the upload file and the unpack contents.
#
proc cleanup_neowebscript_mime_upload {} {
    global mimeFilename mimeUnpackDir
    exec rm -rf $mimeFilename $mimeUnpackDir
}

#
# load_mime_response
#
# Load the mime response variables into the safe interpreter and returns a
# list of input field names (also stored in the global array "filesUploaded"
# as each array element).  Each field may refer to a distinct uploaded file.
#
SAFE_proc_and_alias load_mime_response {safeInterp {responseVar "response"} {fileInfoVar "filesUploaded"}} {
    global mimeResponse mimeUploadedFiles

    $safeInterp eval array set $responseVar [list [array get mimeResponse]]
    $safeInterp eval array set $fileInfoVar [list [array get mimeUploadedFiles]]

    return [array names mimeUploadedFiles]
}

#
# save_mime_file
#
# (deprecated, see open_mime_file)
#
SAFE_proc_and_alias save_mime_file {safeInterp fieldName dataFileName} {
    global mimeUploadedFiles mimeUnpackDir

    if {![info exists mimeUploadedFiles($fieldName)]} {
	error "MIME upload file for field name '$fieldName' failed -- no such field"
    }

    setup_data_access
    set tempName [lindex $mimeUploadedFiles($fieldName) 0]
    ul_name_check $dataFileName data mimeSaveFileName
    create_user_dir data
    exec cp $mimeUnpackDir/$tempName $mimeSaveFileName
}

#
# open_mime_file
#
# Takes an input field name, and returns a channel identifier
# (i.e. a file handle) to the MIME-uploaded file.
#
SAFE_proc_and_alias open_mime_file {safeInterp fieldName} {
    global mimeUploadedFiles mimeUnpackDir parallelDir errorCode

    if ![info exists mimeUploadedFiles($fieldName)] {
        error "MIME upload file for field name '$fieldName' failed -- no such file"
    }

    setup_data_access
    set tempName [lindex $mimeUploadedFiles($fieldName) 0]
    set mimeSaveFileName $mimeUnpackDir/$tempName

    if {[catch {set fp [open $mimeSaveFileName "CREAT RDWR" 0660]} res] == 1} {
        #error "unable to open MIME upload file: $errorCode"
        error "unable to open MIME upload file: $res"
    }
    interp transfer "" $fp $safeInterp
    return $fp
}

